/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <ibtk/version.h>

#include "xutils.h"
#include "gatos.h"
#include "cc.h"
#include "version.h"

/* for debugging */
/* 
#define REPORT_DEPTH 
*/

/* fix later */
/* do_alloc always returns a valid chunk of memory */
#define do_alloc(a,b)  (calloc((a),(b)))

XImage *proto_image = NULL; /* this contains image parameters and should be 
			       initialized when the program first opens the 
			       window 
			     */
#ifdef HASH_CHARS			     
XImage *font_bitmaps[256];
XImage *scaled_font_bitmaps[256];
u8 font_attr[256];
long font_fg_color[256];
long font_bg_color[256];
long last_height = -1;
/* counters */
long unscaled_char_requests = 0;
long unscaled_char_rendered = 0;
long scaled_char_requests = 0;
long scaled_char_rendered = 0;
#endif HASH_CHARS

/* ------------------------------------------------------------------------ */

void createProtoImage(Display *display, Drawable d)  {
  int i;
  if(proto_image != NULL) {
    XFreeImage(proto_image);
    }
  XFlush(display);	
  proto_image = XGetImage(display, d, 1, 1, 4, 4, ~0, ZPixmap);

#ifdef HASH_CHARS
  for(i = 0; i < 256; i++) {
    font_bitmaps[i] = NULL;
    scaled_font_bitmaps[i] = NULL;
    font_attr[i] = 255;
    }
#endif HASH_CHARS

#ifdef REPORT_DEPTH
  fprintf(stderr,"createProtoImage: depth                 =%d\n",
	  proto_image->depth);
  fprintf(stderr,"createProtoImage: 8*bytes_per_line/width=%d\n",
	  8*proto_image->bytes_per_line/proto_image->width);
  fflush(stderr);
#endif REPORT_DEPTH

  }

/* ------------------------------------------------------------------------ */

int get_actual_depth(void)  {
  if(proto_image == NULL) {
    fprintf(stderr,
	    "get_actual_depth: proto_image==NULL, internal malfunction\n");
    fflush(stderr);
    return 0;
    }
  return 8*proto_image->bytes_per_line/proto_image->width;
  }

/* ------------------------------------------------------------------------ */

void reset_xutils_stats(void)  {

#ifdef HASH_CHARS
  unscaled_char_requests = 0;
  unscaled_char_rendered = 0;
  scaled_char_requests = 0;
  scaled_char_rendered = 0;
#endif HASH_CHARS

  }

/* ------------------------------------------------------------------------ */

void print_xutils_stats(void)  {

#ifdef HASH_CHARS
  fprintf(stderr, "unscaled_char_requests=%ld\n", unscaled_char_requests);
  fprintf(stderr, "unscaled_char_rendered=%ld\n", unscaled_char_rendered);
  if(unscaled_char_requests)
    fprintf(stderr,"unscaled_char_hash effectiveness=%ld%%\n",
	    100-(100*unscaled_char_rendered)/unscaled_char_requests);

  fprintf(stderr, "scaled_char_requests=%ld\n", scaled_char_requests);
  fprintf(stderr, "scaled_char_rendered=%ld\n", scaled_char_rendered);
  if(scaled_char_requests)
    fprintf(stderr,"scaled_char_hash effectiveness=%ld%%\n",
	    100-(100*scaled_char_rendered)/scaled_char_requests);
  fflush(stderr);
#endif HASH_CHARS

  }

/* ------------------------------------------------------------------------ */

void fillPalette(Display *d,Drawable dw, GC gc, PALETTE *p)  {
  XColor g;

  XFlush(d);

  g.red = 0xFFFF; g.green = 0xFFFF; g.blue = 0;
  XAllocColor(d, DefaultColormap(d, 0), &g);
  p->yellow_col = g.pixel;

  g.red = 0xFFFF; g.green = 0; g.blue = 0;
  XAllocColor(d, DefaultColormap(d, 0), &g);
  p->red_col = g.pixel;

  g.red = 0x0; g.green = 0xFFFF; g.blue = 0;
  XAllocColor(d, DefaultColormap(d, 0), &g);
  p->green_col = g.pixel;

  g.red = 0; g.green = 0x0; g.blue = 0xFFFF;
  XAllocColor(d, DefaultColormap(d, 0), &g);
  p->blue_col = g.pixel;

  g.red = 0xFFFF; g.green = 0xFFFF; g.blue = 0xFFFF;
  XAllocColor(d, DefaultColormap(d, 0), &g);
  p->white_col = g.pixel;

  /* fix this one later */
  g.red = 0xFFFF; g.green = 0; g.blue = 0xFFFF;
  XAllocColor(d, DefaultColormap(d, 0), &g);
  p->magenta_col = g.pixel; 

  /* fix this one later */
  g.red = 0x0; g.green = 0xFFFF; g.blue = 0xFFFF;
  XAllocColor(d, DefaultColormap(d, 0), &g);
  p->cyan_col = g.pixel; 

  g.red = 0x1; g.green = 0x1; g.blue = 0x1;
  XAllocColor(d, DefaultColormap(d,0 ), &g);
  p->black_col = g.pixel; 
  }

/* ------------------------------------------------------------------------ */
/*
XImage *printString1(Display *display, Drawable d, char *string, char *font,
		     long height,PALETTE *pal)  {
  long startx, starty;
  long length, size;
  int dir_r, fa_r, fd_r;
  long depth;
  XCharStruct ov_r;
  XImage *r, *r1;
  Pixmap p;
  Font f;
  GC gc;
  XFontStruct *fs;

  depth = DefaultDepthOfScreen((DefaultScreenOfDisplay(display)));
  f = XLoadFont(display, font);
  gc = XCreateGC(display, d, None, None);
  XSetFont(display, gc, f);
  fs = XQueryFont(display, f);
  XTextExtents(fs, string, strlen(string), &dir_r, &fa_r, &fd_r, &ov_r);
  size = fa_r+fd_r+2;
  length = XTextWidth(fs, string, strlen(string))+4;

  if(!dir_r) startx = 0;
  else startx = length;
  starty = fa_r+1;

  p = XCreatePixmap(display, d, length, size, depth);
  XSetForeground(display, gc, pal->black_col);
  XFillRectangle(display, p, gc, 0, 0, length, size);
  XSetForeground(display, gc, pal->yellow_col);
  XDrawString(display, p, gc, startx, starty, string, strlen(string));
  r = XGetImage(display, p, 0, 0, length, size, ~0, ZPixmap);
  XFreeGC(display, gc);
  XFreePixmap(display, p);
  XFreeFontInfo(NULL, fs, 0);
  XUnloadFont(display, f);
  r1 = XScaleImage(r, (height*1.0)/r->height, pal->black_col, 0);
  XFreeImage(r);
  return r1;
  }
*/
/* ------------------------------------------------------------------------ */

void ImagePutImage(XImage *dest, int x, int y, XImage *source)  {
  DGAPutImage(dest->data, dest->width, x, y, source);
  }

/* ------------------------------------------------------------------------ */

void ImageDrawRectangle(XImage *dest, int x, int y, int width,
			int height, long color)  {
  DGADrawRectangle(dest->data, 8*dest->bytes_per_line/dest->width,
		   dest->width, x, y, width, height, color);
 }

/* ------------------------------------------------------------------------ */
/* allow to override in Makefile */
/*
#ifndef FONTNAME
#define FONTNAME 	"/usr/lib/kbd/consolefonts/cp850-8x16"
#endif
*/
#include "cp850_8x16.inc"
#include "iso01mod_8x16.inc"

#ifndef SCANLINES
#define SCANLINES 	16
#endif SCANLINES

#define NORMAL_SLANT	0.0
#define ITALIC_SLANT	0.2
int font_file = -1;

struct FONT_DATA { 
  char scan_line[SCANLINES];
  } *font_chars = (struct FONT_DATA *)iso01mod_8x16;

/* ------------------------------------------------------------------------ */
/* you should not free the resulting XImage structure */
XImage *charToImage(char c, long fg_color, long bg_color)  {
  XImage *im;
  int i, j, k;
  long color;
  int depth;

#ifdef HASH_CHARS
  unscaled_char_requests++;
  if((font_bitmaps[(u8)c] != NULL)
      && (font_fg_color[(u8)c] == fg_color)
      && (font_bg_color[(u8)c] == bg_color)) 
    return font_bitmaps[(u8)c];
#endif HASH_CHARS

  depth = 8*proto_image->bytes_per_line/proto_image->width;

#ifdef REPORT_DEPTH
  printf("charToImage: depth=%d\n", depth); fflush(stdout);
#endif REPORT_DEPTH

  im = do_alloc(1, sizeof(XImage));
  im->width = 8;
  im->height = SCANLINES;
  im->bytes_per_line = (im->width*depth)/8;
  im->bits_per_pixel = depth;
  im->xoffset = 0;
  im->bitmap_bit_order = proto_image->bitmap_bit_order;
  im->bitmap_pad = proto_image->bitmap_pad;
  im->bitmap_unit = proto_image->bitmap_unit;
  im->format = ZPixmap;
  im->byte_order = proto_image->byte_order;
  im->depth = proto_image->depth;
  im->data = do_alloc(im->height*im->bytes_per_line, sizeof(char));

  XInitImage(im);
  for(j = 0; j < 16; j++)
    for(i = 0; i < 8; i++){
      if(font_chars[(u8)c].scan_line[j] & (0x80 >> i)) color = fg_color;
      else color = bg_color;
      for(k = 0; k < depth/8; k++)				
	im->data[i*depth/8+j*im->bytes_per_line+k] = (u8)((color>>(8*k))&0xFF);
      }

#ifdef HASH_CHARS		
  unscaled_char_rendered++;
  if(font_bitmaps[(u8)c] != NULL) XFreeImage(font_bitmaps[(u8)c]);
  font_bitmaps[(u8)c] = im;		
  font_bg_color[(u8)c] = bg_color;
  font_fg_color[(u8)c] = fg_color;
#endif HASH_CHARS

  return im;		
  }

/* ------------------------------------------------------------------------ */

XImage *getScaledChar(char c, long height, long fg_color, long bg_color, 
		      u8 attribute)  {
  XImage *r1, *r2;
  double slant;
  int i;

#ifdef HASH_CHARS
  scaled_char_requests++;
  /* check if height has changed - this invalidates everything */
  if(height != last_height) {
    for(i = 0; i < 256; i++) 
      font_attr[i]=255; /* this is enough to cause faults */
    last_height = height;
    }
		
/* check if character is in hash */
  if((font_fg_color[(u8)c] == fg_color)
      && (font_bg_color[(u8)c] == bg_color)
      && (font_attr[(u8)c] == attribute)
      && (font_bitmaps[(u8)c] != NULL)
                                /* this one because it's easy to check and 
				   if it's invalid than all the other data 
				   is invalid as well */
      && (scaled_font_bitmaps[(u8)c] != NULL))
    return scaled_font_bitmaps[(u8)c];
#endif HASH_CHARS

  if(attribute & ITALIC_ATTR) slant = ITALIC_SLANT;
  else slant = NORMAL_SLANT;
  r1 = charToImage(c, fg_color, bg_color); 
  r2 = XScaleImage(r1, (height*1.0)/r1->height, bg_color, slant);

  if(attribute & UNDERLINE_ATTR) {
    ImageDrawRectangle(r2, 0, r2->height-2, r2->width, 2, fg_color);
    }

#ifdef HASH_CHARS
  scaled_char_rendered++;
  if(scaled_font_bitmaps[(u8)c] != NULL) 
    XFreeImage(scaled_font_bitmaps[(u8)c]);
  scaled_font_bitmaps[(u8)c] = r2;
  font_attr[(u8)c] = attribute; 
/* other parameters have been taken care of by charToImage() */
#else
  XFreeImage(r1);
#endif HASH_CHARS

  return r2;
  }

/* ------------------------------------------------------------------------ */

XImage *getCCChar(char c, long height, int attr, PALETTE *pal)  {
  long fg_color, bg_color;
  u8  a;

  bg_color=pal->bg_col;
  switch(attr & 0xF0) {
    case CC_ATTR_WHITE:
      fg_color =pal->yellow_col;
      break; 
    case CC_ATTR_GREEN:
      fg_color = pal->green_col;
      break;
    case CC_ATTR_BLUE:
      fg_color = pal->blue_col;
      break;
    case CC_ATTR_CYAN:
      fg_color = pal->cyan_col;
      break;
    case CC_ATTR_RED:
      fg_color = pal->red_col;
      break;
    case CC_ATTR_YELLOW:
      fg_color = pal->white_col;
      break;
    case CC_ATTR_MAGENTA:
      fg_color = pal->magenta_col;
      break;
    case CC_ATTR_BACK:
      fg_color = pal->black_col;
      break;
    default:
      fg_color = pal->black_col;
      break;
    }
  a = 0;
  if(c == ' ') fg_color = pal->black_col;
  if(attr & CC_ATTR_ITALICS) a |= ITALIC_ATTR;
  if(attr & CC_ATTR_UNDERLINE) a |= UNDERLINE_ATTR;
  if(attr & CC_ATTR_FLASH) {
    bg_color = pal->white_col;
    }
  return getScaledChar(c, height, fg_color, bg_color, a);
  }

/* ------------------------------------------------------------------------ */

int load_font(char *font_name)  {
  if(font_file < 0) {
    font_file = open(font_name, O_RDONLY);
    if(font_file < 0) {
      perror("printString: opening font:");
      return -1;
      }
    font_chars = mmap(NULL, 256*SCANLINES, PROT_READ, MAP_SHARED, 
		      font_file, 0);
    if(font_chars == NULL) {
      perror("printString: opening font:");
      close(font_file);
      font_file = -1;
      return -1;
      }
    }	
  return 0;
  }

/* ------------------------------------------------------------------------ */

XImage *printString(char *string, char *font, long height, PALETTE *pal)  {
  long color;
  long depth;
  long char_width = 0;
  long offset;
  XImage *r = NULL;
  XImage *r2;
  char *c;
  u8 attr;

  if(proto_image == NULL) {
    fprintf(stderr, "Warning: proto_image was not created,\n");
    fprintf(stderr, "         internal program malfunction.\n");
    fflush(stderr);
    return NULL;
    }
  char_width = (height*10)/SCANLINES;
  depth = 8*proto_image->bytes_per_line/proto_image->width;

  r2 = getScaledChar('X', height, pal->yellow_col, pal->black_col, 0);

  r = do_alloc(1, sizeof(XImage));	
  r->width = (strlen(string)+1)*r2->width;
  r->height = r2->height;

#ifndef HASH_CHARS
  XFreeImage(r2);
#endif HASH_CHARS

  r->format = proto_image->format;
  r->byte_order = proto_image->byte_order;
  r->depth = proto_image->depth;
  r->bits_per_pixel = proto_image->bits_per_pixel;
  r->xoffset = 0;
  r->bitmap_unit = proto_image->bitmap_unit;
  r->bitmap_bit_order = proto_image->bitmap_bit_order;
  r->bitmap_pad = proto_image->bitmap_pad;
  r->data = NULL;
  r->bytes_per_line = r->width*depth/8;	
  r->data = do_alloc(r->height*r->bytes_per_line, sizeof(char));

  ImageDrawRectangle(r, 0, 0, r->width, r->height, pal->bg_col);
  color = pal->yellow_col;
  offset = 2;
  attr = NORMAL_ATTR;
  for(c = string; *c;) {
    if(*c == '\\') { /* got control symbol */
      c++;
      switch(*c) {
        case 'y':
	  color = pal->yellow_col;
	  break;
        case 'g':
	  color = pal->green_col;
	  break;
        case 'r':
	  color = pal->red_col;
	  break;
        case 'b':
	  color = pal->blue_col;
	  break;
        case 'u':
	  attr |= UNDERLINE_ATTR;
	  break;
        case 'i':
	  attr |= ITALIC_ATTR;
	  break;
        case 'f':
	  attr = NORMAL_ATTR;				
	  break;
        case '\\':
	  c--; /* display symbol */
	  break;
        default:
	  c--; /* display symbol */
	  break;
        }
      c++; continue;
      }
    r2 = getScaledChar(*c, height, color, pal->bg_col, attr);
    ImagePutImage(r, offset, 0, r2);
    offset += r2->width;
    c++;

#ifndef HASH_CHARS
    XFreeImage(r2);
#endif HASH_CHARS

    }
  return r;
  }

/* ------------------------------------------------------------------------ */

void DGADrawRectangle(u8 *frame, int depth, int ysize, int x, int y,
		      int width, int height, long color)  {
  long xstart, yinc, bytes_per_line;
  int i, j;
  u8 *ptr;

  bytes_per_line = width*depth/8;

#ifdef REPORT_DEPTH
  fprintf(stderr, "DGADrawRectangle: depth=%d\n", depth); fflush(stderr);
#endif REPORT_DEPTH

  color = color & 0xFFFFFF; /* zero the top 8 bytes */
  switch(depth) { 
    case 8:
      xstart = x+y*ysize;
      yinc = ysize;
      for(j = 0; j < height; j++)
	for(i = 0; i < width; i++) {
	  memcpy(frame+xstart+i+j*yinc, &color, 1);
	  }
      break;
    case 16:
      xstart = x*2+y*ysize*2;
      yinc = ysize*2;
      for(j = 0; j < height; j++)
	for(i = 0; i < width; i++) {
	  memcpy(frame + xstart+i*2+j*yinc, &color, 2);
	  }
      break;
    case 24:/* This is probably for "packed" 24 bit mode.
	       It is interesting that xserver reports both 32 and 24bit
	       mode as 24bit.. perhaps we are wrong bc we look at the 
	       depth of the visual */
      /*  Just fallback to 32bit depth code */
      /*
	xstart = x*3+y*ysize*3;
	yinc = ysize*3;
	for(i = 0;i < width;i++)
	for(j = 0;j < height;j++){
	frame[xstart+i*3+j*yinc+2] = (u8)((color>>16)&0xFF);
	frame[xstart+i*3+j*yinc+1] = (u8)((color>>8)&0xFF);
	frame[xstart+i*3+j*yinc] = (u8)(color&0xFF);
	}
	break;
      */
    case 32:
      xstart = x*4+y*ysize*4;
      yinc = ysize*4;
      ptr = frame+xstart;
      for(j = 0; j < height; j++) {
	for(i = 0; i < width; i++) {
	  memcpy(frame + xstart+i*4+j*yinc, &color, 4);
	  }
        }
      break;
    default:
      fprintf(stderr, "DGADrawRectangle: unimplemented depth %d\n", depth);
      fflush(stderr);
      break;
    }
  }

/* ------------------------------------------------------------------------ */

void DGAPutImage(u8 *frame, long ysize, int x, int y, XImage *im)  {
  int j;
  long yinc, xstart, xlimit, ylimit;
  int actual_depth; /* actual depth is divided by 8 */

  actual_depth = im->bytes_per_line/im->width;
  yinc = ysize*actual_depth;
  xstart = x*actual_depth+y*yinc;
  /* Horizontal clipping */
  xlimit = im->bytes_per_line>
    yinc-x*actual_depth ? yinc-x*actual_depth : im->bytes_per_line;
  ylimit = im->height;
  for(j = 0; j < ylimit; j++)
    memcpy(&(frame[xstart+j*yinc]), &(im->data[j*im->bytes_per_line]), xlimit);
  }

/* ------------------------------------------------------------------------ */
/* this last trick bc gatos doesn't always use black color for transparency */
XImage * XScaleImage(XImage *im, double ratio, long black, double slant)  {
  int i, j;
  XImage *r;
  long color1, color2, color3, color4, color;
  double source_x, source_y, delta1, delta2;
  int x1, y1, actual_depth; /* here actual depth is divided by 8 */

  r = NULL;
  while(r == NULL) {
    r = (XImage *)calloc(1, sizeof(XImage));
    if(r == NULL) sleep(1);
    }

  r->width = (int)(ratio*(im->width+2));
  r->height = (int)(ratio*(im->height+2));
  r->format = im->format;
  r->byte_order = im->byte_order;
  r->depth = im->depth;
  r->bits_per_pixel = im->bits_per_pixel;
  r->xoffset = 0;
  r->bitmap_unit = im->bitmap_unit;
  r->bitmap_bit_order = im->bitmap_bit_order;
  r->bitmap_pad = im->bitmap_pad;
  r->data = NULL;
  actual_depth = im->bytes_per_line/im->width;
  r->bytes_per_line = r->width*actual_depth;	

  while(r->data == NULL) {
    r->data = (char *)calloc(r->height*r->bytes_per_line, sizeof(char));
    if(r->data == NULL) sleep(1);
    }
  switch(actual_depth) {
    case 1:
      for(i = 0; i < r->width; i++)
	for(j = 0; j < r->height; j++) {
	  /* no antialiasing in 8bpp */
	  source_y = j/ratio;
	  source_x = i/ratio+slant*source_y;
	  x1 = (int)source_x;
	  y1 = (int)source_y;				
	  source_x -= 2;
	  source_y -= 2;
	  x1 -= 2;
	  y1 -= 2;
	  if((x1 < 0) 
	      || (y1 < 0) 
	      || (x1 >= im->width) 
	      || (y1 >= im->height)) 
	    color = black;
	  else color = (u8)im->data[x1*actual_depth+y1*im->bytes_per_line];
	  r->data[i*actual_depth+j*r->bytes_per_line] = (u8)(color & 0xFF);
	  }
      break;
    case 2:
      for(i = 0; i < r->width; i++)
	for(j = 0; j < r->height; j++) {
	  source_y = j/ratio;
	  source_x = i/ratio+slant*source_y;
	  x1 = (int)source_x;
	  y1 = (int)source_y;				
	  delta1 = source_x-x1;
	  delta2 = source_y-y1;
	  source_x -= 2;
	  source_y -= 2;
	  x1 -= 2;
	  y1 -= 2;
	  if((x1 < 0) 
	      || (y1 < 0) 
	      || (x1 >= im->width) 
	      || (y1 >= im->height)) 
	    color1=black;
	  else color1 = (u8)im->data[x1*actual_depth+y1*im->bytes_per_line]+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+1])<<8);
	  if(color1 == black) color1 = 0;
	  x1++;
	  if((x1 < 0) 
	      || (y1 < 0) 
	      || (x1 >= im->width) 
	      || (y1 >= im->height)) 
	    color2=black;
	  else color2 = (u8)im->data[x1*actual_depth+y1*im->bytes_per_line]+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+1])<<8);
	  if(color2 == black) color2 = 0;
	  y1++;
	  if((x1 < 0) 
	      || (y1 < 0) 
	      || (x1 >= im->width) 
	      || (y1 >= im->height)) 
	    color4=black;
	  else color4 = (u8)im->data[x1*actual_depth+y1*im->bytes_per_line]+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+1])<<8);
	  if(color4 == black) color4 = 0;
	  x1--;
	  if((x1 < 0) 
	      || (y1 < 0) 
	      || (x1 >= im->width) 
	      || (y1 >= im->height)) 
	    color3=black;
	  else color3 = (u8)im->data[x1*actual_depth+y1*im->bytes_per_line]+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+1])<<8);
	  if(color3 == black) color3 = 0;
	  
#define BLUE(u)		((u)&0x1F)
#define GREEN(u)	(((u)>>5)&0x3F)
#define RED(u)		(((u)>>11)&0x1F)
	  
	  color = (long)(BLUE(color4)*delta1*delta2+
			 BLUE(color2)*delta1*(1-delta2)+
			 BLUE(color3)*(1-delta1)*delta2+      
			 BLUE(color1)*(1-delta1)*(1-delta2))
	    | (((long)(GREEN(color4)*delta1*delta2+
		       GREEN(color2)*delta1*(1-delta2)+
		       GREEN(color3)*(1-delta1)*delta2+      
		       GREEN(color1)*(1-delta1)*(1-delta2)))<<5) 
	    | (((long)(RED(color4)*delta1*delta2+
		       RED(color2)*delta1*(1-delta2)+
		       RED(color3)*(1-delta1)*delta2+      
		       RED(color1)*(1-delta1)*(1-delta2)))<<11);
	  if(color == 0) color = black;
#undef BLUE
#undef GREEN
#undef RED
	  r->data[i*actual_depth+j*r->bytes_per_line] = 
	    (u8)(color & 0xFF);
	  r->data[i*actual_depth+j*r->bytes_per_line+1] =
	    (u8)((color >> 8)&0xFF);
	  }
      break;
    case 3:/* This is probably for "packed" 24 bit mode.
	      It is interesting that xserver reports both 32 and 24bit
	      mode as 24bit.. perhaps we are wrong bc we look at the 
	      depth of the visual */
      /*  Just fallback to 32bit depth code */
      /*
	xstart = x*3+y*ysize*3;
	yinc = ysize*3;
	for(i = 0; i < im->byte_per_line; i++)
	for(j = 0; j < height; j++){
	frame[xstart+i+j*yinc] = (u8)im->data[i+j*im->bytes_per_line];
	}
	break;
      */
    case 4:
      for(i = 0; i < r->width; i++)
	for(j = 0; j < r->height; j++) {
	  source_y = j/ratio;
	  source_x = i/ratio+slant*source_y;
	  x1 = (int)source_x;
	  y1 = (int)source_y;				
	  delta1 = source_x-x1;
	  delta2 = source_y-y1;
	  source_x -= 2;
	  source_y -= 2;
	  x1 -= 2;
	  y1 -= 2;
	  if((x1 < 0) 
	      || (y1 < 0) 
	      || (x1 >= im->width) 
	      || (y1 >= im->height)) 
	    color1 = black;
	  else color1 = (u8)im->data[x1*actual_depth+y1*im->bytes_per_line]+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+1])<<8)+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+2])<<16);
	  if(color1 == black) color1 = 0;
	  x1++;
	  if((x1 < 0) 
	      || (y1 < 0) 
	      || (x1 >= im->width) 
	      || (y1 >= im->height)) 
	    color2=black;
	  else color2 = (u8)im->data[x1*actual_depth+y1*im->bytes_per_line]+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+1])<<8)+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+2])<<16);
	  if(color2 == black) color2 = 0;
	  y1++;
	  if((x1 < 0) 
	      || (y1 < 0) 
	      || (x1 >= im->width) 
	      || (y1 >= im->height)) 
	    color4 = black;
	  else color4 = (u8)im->data[x1*actual_depth+y1*im->bytes_per_line]+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+1])<<8)+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+2])<<16);
	  if(color4 == black) color4 = 0;
	  x1--;
	  if((x1 < 0) 
	      || (y1 < 0) 
	      || (x1 >= im->width) 
	      || (y1 >= im->height)) 
	    color3 = black;
	  else color3 = (u8)im->data[x1*actual_depth+y1*im->bytes_per_line]+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+1])<<8)+
		 (((u8)im->data[x1*actual_depth+y1*im->bytes_per_line+2])<<16);
	  if(color3 == black) color3 = 0;
	  
#define BLUE(u)		((u)&0xFF)
#define GREEN(u)	(((u)>>8)&0xFF)
#define RED(u)		(((u)>>16)&0xFF)
	  
	  color = ((long)(BLUE(color4)*delta1*delta2+
			  BLUE(color2)*delta1*(1-delta2)+
			  BLUE(color3)*(1-delta1)*delta2+      
			  BLUE(color1)*(1-delta1)*(1-delta2)))
	    | (((long)(GREEN(color4)*delta1*delta2+
		       GREEN(color2)*delta1*(1-delta2)+
		       GREEN(color3)*(1-delta1)*delta2+      
		       GREEN(color1)*(1-delta1)*(1-delta2)))<<8) 
	    | (((long)(RED(color4)*delta1*delta2+
		       RED(color2)*delta1*(1-delta2)+
		       RED(color3)*(1-delta1)*delta2+      
		       RED(color1)*(1-delta1)*(1-delta2)))<<16);
	  if(color == 0) color = black;
#undef BLUE
#undef GREEN
#undef RED
	  r->data[i*actual_depth+j*r->bytes_per_line] = 
	    (u8)(color & 0xFF);
	  r->data[i*actual_depth+j*r->bytes_per_line+1] =
	    (u8)((color >> 8)&0xFF);
	  r->data[i*actual_depth+j*r->bytes_per_line+2] =
	    (u8)((color >> 16)&0xFF);
	  }
      break;
    default:
      fprintf(stderr, "XScaleImage: unimplemented depth %d\n", actual_depth);
      fflush(stderr);
      break;
    }
  XInitImage(r);	
  return r;
  }

/* ------------------------------------------------------------------------ */

void XFreeImage(XImage *im)  {
  free(im->data);
  free(im);
  }

/* ------------------------------------------------------------------------ */

int GetWindowXSize(Display *display, Drawable d)  {
  Window wintmp;
  unsigned rootxsize, rootysize, utmp;
  int itmp;

  XGetGeometry(display, d, &wintmp, &itmp, &itmp, &rootxsize, 
	       &rootysize, &utmp, &utmp);

  return rootxsize;
  }

/* ------------------------------------------------------------------------ */

int GetWindowYSize(Display *display, Drawable d)  {
  Window wintmp;
  unsigned rootxsize, rootysize, utmp;
  int itmp;

  XGetGeometry(display, d, &wintmp, &itmp, &itmp, &rootxsize, 
	       &rootysize, &utmp, &utmp);
  
  return rootysize;
  }

/* ------------------------------------------------------------------------ */
/* 
 * Check if ibtk version is equal/greater 
 * or less than required
 */
void check_ibtk_version(void)  {
  char ctmp[8], *preq = &IBTK_REQUIRED_VERSION[0];
  int imaj = 0, imin = 0, isub = 0, ipre = 0, i;
   
  /* 
   * Extract major/minor/sub/pre versions
   * from IBTK_REQUIRED_VERSION string.
   */
  for(i = 0; i < 8; i += 2) {
    strncpy(ctmp, preq+i, 2);
    switch(i) {
      case 0: 
	imaj = atoi(ctmp); 
	break;
      case 2: 
	imin = atoi(ctmp); 
	break;
      case 4: 
	isub = atoi(ctmp); 
	break;
      case 6: 
	ipre = atoi(ctmp); 
	break;
      }
    }
  
  if(!ibtk_required_version(IBTK_REQUIRED_VERSION)) {
    perror(_("xatitv: Bad IBTK library version."));
    fprintf(stderr, _("Expected IBTK version %d.%d.%d"), imaj, imin, isub);
    if(ipre > 0) fprintf(stderr, _("-pre%d"), ipre);
    fprintf(stderr, _(", and %s found.\n"), get_ibtk_full_version_string());
    exit(1);
    }
  }

/* ------------------------------------------------------------------------ */
/*
 * Check the DISPLAY environment variable,
 * return 1 if DISPLAY is set and not match 
 * with the local display, else return 0.
 */
int exported_display(void)  {
  char myhn[256], myfqdn[256], myipaddr[16] = "127.0.0.1", displayname[512];
  char *disp_env, *p;
  struct hostent *hp = NULL;

  /* Get the DISPLAY environment variable */
  disp_env = getenv("DISPLAY");

  if(disp_env != NULL && strlen(disp_env) > 0) {
    if(!gethostname(myfqdn, 255) 
        && (hp = gethostbyname(myfqdn)) != NULL) {

      /* Get the local IP address */
      while(hp->h_addr_list[0])
	sprintf(myipaddr, "%s", 
		inet_ntoa(*(struct in_addr *) *hp->h_addr_list++)); 

      /* Get the hostname without domain name */
      gethostname(myhn, 255);
      strncpy(myfqdn, hp->h_name, 255);

      /* Get the fqdn */
      if(strchr(myfqdn, '.') == NULL) {
	char ** alias;
	alias = hp->h_aliases;
	while(alias && *alias)
	  if(strchr(*alias, '.') && (strlen(*alias) > strlen(myfqdn)))
	    strncpy(myfqdn, *alias, 255);
	  else
	    alias++;
        }
      } 
    else {
      *myhn = '\0';
      *myfqdn = '\0';
      }

    /* Remove the ':' and follow chars */
    if((p=strchr(disp_env, ':'))) 
      memcpy(displayname, disp_env, (strlen(disp_env)-strlen(p)));

    if((strncasecmp(displayname, "localhost", 9) == 0)
        || (strcmp(displayname, "unix") == 0)
        || (strncasecmp(displayname, myhn, strlen(myhn)) == 0)
        || (strncasecmp(displayname, myfqdn, strlen(myfqdn)) == 0)
        || (strcmp(displayname, myipaddr) == 0)
        || (strcmp(displayname, "127.0.0.1") == 0)
        || (strncmp(disp_env,":", 1) == 0)) {
      return 0;
      } 
    else {
      return 1;
      }
    }
  return 0;
  }

/* ------------------------------------------------------------------------ */
/* 
 * Return the Vendor name and release,
 * Protocol version/revision, available screen(s) and color depth.
 */
char *GetXVendorRelease(Display *display) {
  static char buf[256];

  sprintf(buf, "XServer Vendor: %s. Release: %d,\n", 
	  XServerVendor(display), XVendorRelease(display));
  sprintf(buf, "%s        Protocol Version: %d, Revision: %d,\n", buf,
	  XProtocolVersion(display), XProtocolRevision(display));
  sprintf(buf, "%s        Available Screen(s): %d,\n", buf,
	  XScreenCount(display));
  sprintf(buf, "%s        Depth: %d.\n", buf, 
	  XDisplayPlanes(display, XDefaultScreen(display)));
  return buf;
  }

/* ------------------------------------------------------------------------ */

