#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright: (c) 2019, Dag Wieers (@dagwieers) <dag@wieers.com>
# GNU General Public License v3.0+ (see LICENSE or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function

__metaclass__ = type

ANSIBLE_METADATA = {"metadata_version": "1.1", "status": ["preview"], "supported_by": "community"}

DOCUMENTATION = r"""
---
module: mso_schema_site_anp_epg_staticleaf
short_description: Manage site-local EPG static leafs in schema template
description:
- Manage site-local EPG static leafs in schema template on Cisco ACI Multi-Site.
author:
- Dag Wieers (@dagwieers)
options:
  schema:
    description:
    - The name of the schema.
    type: str
    required: true
  site:
    description:
    - The name of the site.
    type: str
    required: true
  template:
    description:
    - The name of the template.
    type: str
    required: true
  anp:
    description:
    - The name of the ANP.
    type: str
    required: true
  epg:
    description:
    - The name of the EPG.
    type: str
    required: true
  pod:
    description:
    - The pod of the static leaf.
    type: str
  leaf:
    description:
    - The path of the static leaf.
    type: str
    aliases: [ name ]
  vlan:
    description:
    - The VLAN id of the static leaf.
    type: int
  state:
    description:
    - Use C(present) or C(absent) for adding or removing.
    - Use C(query) for listing an object or multiple objects.
    type: str
    choices: [ absent, present, query ]
    default: present
notes:
- The ACI MultiSite PATCH API has a deficiency requiring some objects to be referenced by index.
  This can cause silent corruption on concurrent access when changing/removing on object as
  the wrong object may be referenced. This module is affected by this deficiency.
seealso:
- module: cisco.mso.mso_schema_site_anp_epg
- module: cisco.mso.mso_schema_template_anp_epg
extends_documentation_fragment: cisco.mso.modules
"""

EXAMPLES = r"""
- name: Add a new static leaf to a site EPG
  cisco.mso.mso_schema_site_anp_epg_staticleaf:
    host: mso_host
    username: admin
    password: SomeSecretPassword
    schema: Schema1
    site: Site1
    template: Template1
    anp: ANP1
    epg: EPG1
    leaf: Leaf1
    vlan: 123
    state: present
  delegate_to: localhost

- name: Remove a static leaf from a site EPG
  cisco.mso.mso_schema_site_anp_epg_staticleaf:
    host: mso_host
    username: admin
    password: SomeSecretPassword
    schema: Schema1
    site: Site1
    template: Template1
    anp: ANP1
    epg: EPG1
    leaf: Leaf1
    state: absent
  delegate_to: localhost

- name: Query a specific site EPG static leaf
  cisco.mso.mso_schema_site_anp_epg_staticleaf:
    host: mso_host
    username: admin
    password: SomeSecretPassword
    schema: Schema1
    site: Site1
    template: Template1
    anp: ANP1
    epg: EPG1
    leaf: Leaf1
    state: query
  delegate_to: localhost
  register: query_result

- name: Query all site EPG static leafs
  cisco.mso.mso_schema_site_anp_epg_staticleaf:
    host: mso_host
    username: admin
    password: SomeSecretPassword
    schema: Schema1
    site: Site1
    template: Template1
    anp: ANP1
    state: query
  delegate_to: localhost
  register: query_result
"""

RETURN = r"""
"""

from ansible.module_utils.basic import AnsibleModule
from ansible_collections.cisco.mso.plugins.module_utils.mso import MSOModule, mso_argument_spec


def main():
    argument_spec = mso_argument_spec()
    argument_spec.update(
        schema=dict(type="str", required=True),
        site=dict(type="str", required=True),
        template=dict(type="str", required=True),
        anp=dict(type="str", required=True),
        epg=dict(type="str", required=True),
        pod=dict(type="str"),  # This parameter is not required for querying all objects
        leaf=dict(type="str", aliases=["name"]),
        vlan=dict(type="int"),
        state=dict(type="str", default="present", choices=["absent", "present", "query"]),
    )

    module = AnsibleModule(
        argument_spec=argument_spec,
        supports_check_mode=True,
        required_if=[
            ["state", "absent", ["pod", "leaf", "vlan"]],
            ["state", "present", ["pod", "leaf", "vlan"]],
        ],
    )

    schema = module.params.get("schema")
    site = module.params.get("site")
    template = module.params.get("template").replace(" ", "")
    anp = module.params.get("anp")
    epg = module.params.get("epg")
    pod = module.params.get("pod")
    leaf = module.params.get("leaf")
    vlan = module.params.get("vlan")
    state = module.params.get("state")

    leafpath = "topology/{0}/node-{1}".format(pod, leaf)

    mso = MSOModule(module)

    # Get schema objects
    schema_id, schema_path, schema_obj = mso.query_schema(schema)

    # Get site
    site_id = mso.lookup_site(site)

    # Get site_idx
    if not schema_obj.get("sites"):
        mso.fail_json(msg="No site associated with template '{0}'. Associate the site with the template using mso_schema_site.".format(template))
    sites = [(s.get("siteId"), s.get("templateName")) for s in schema_obj.get("sites")]
    if (site_id, template) not in sites:
        mso.fail_json(msg="Provided site/template '{0}-{1}' does not exist. Existing sites/templates: {2}".format(site, template, ", ".join(sites)))

    # Schema-access uses indexes
    site_idx = sites.index((site_id, template))
    # Path-based access uses site_id-template
    site_template = "{0}-{1}".format(site_id, template)

    # Get ANP
    anp_ref = mso.anp_ref(schema_id=schema_id, template=template, anp=anp)
    anps = [a.get("anpRef") for a in schema_obj.get("sites")[site_idx]["anps"]]
    if anp_ref not in anps:
        mso.fail_json(msg="Provided anp '{0}' does not exist. Existing anps: {1}".format(anp, ", ".join(anps)))
    anp_idx = anps.index(anp_ref)

    # Get EPG
    epg_ref = mso.epg_ref(schema_id=schema_id, template=template, anp=anp, epg=epg)
    epgs = [e.get("epgRef") for e in schema_obj.get("sites")[site_idx]["anps"][anp_idx]["epgs"]]
    if epg_ref not in epgs:
        mso.fail_json(msg="Provided epg '{0}' does not exist. Existing epgs: {1}".format(epg, ", ".join(epgs)))
    epg_idx = epgs.index(epg_ref)

    # Get Leaf
    leafs = [(leaf.get("path"), leaf.get("portEncapVlan")) for leaf in schema_obj.get("sites")[site_idx]["anps"][anp_idx]["epgs"][epg_idx]["staticLeafs"]]
    if (leafpath, vlan) in leafs:
        leaf_idx = leafs.index((leafpath, vlan))
        # FIXME: Changes based on index are DANGEROUS
        leaf_path = "/sites/{0}/anps/{1}/epgs/{2}/staticLeafs/{3}".format(site_template, anp, epg, leaf_idx)
        mso.existing = schema_obj.get("sites")[site_idx]["anps"][anp_idx]["epgs"][epg_idx]["staticLeafs"][leaf_idx]

    if state == "query":
        if leaf is None or vlan is None:
            mso.existing = schema_obj.get("sites")[site_idx]["anps"][anp_idx]["epgs"][epg_idx]["staticLeafs"]
        elif not mso.existing:
            mso.fail_json(msg="Static leaf '{leaf}/{vlan}' not found".format(leaf=leaf, vlan=vlan))
        mso.exit_json()

    leafs_path = "/sites/{0}/anps/{1}/epgs/{2}/staticLeafs".format(site_template, anp, epg)
    ops = []

    mso.previous = mso.existing
    if state == "absent":
        if mso.existing:
            mso.sent = mso.existing = {}
            ops.append(dict(op="remove", path=leaf_path))

    elif state == "present":
        payload = dict(
            path=leafpath,
            portEncapVlan=vlan,
        )

        mso.sanitize(payload, collate=True)

        if mso.existing:
            ops.append(dict(op="replace", path=leaf_path, value=mso.sent))
        else:
            ops.append(dict(op="add", path=leafs_path + "/-", value=mso.sent))

        mso.existing = mso.proposed

    if not module.check_mode:
        mso.request(schema_path, method="PATCH", data=ops)

    mso.exit_json()


if __name__ == "__main__":
    main()
