/*	Stream_Monitor

PIRL CVS ID: Stream_Monitor.java,v 1.17 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import	PIRL.Utilities.Styled_Writer;
import	PIRL.Strings.String_Utilities;

import	javax.swing.JPanel;
import	javax.swing.JScrollPane;
import	javax.swing.JViewport;
import	javax.swing.JScrollBar;
import	javax.swing.JTextPane;
import	javax.swing.text.DefaultStyledDocument;
import	javax.swing.text.Caret;
import	javax.swing.text.DefaultCaret;
import	javax.swing.text.BadLocationException;
import	javax.swing.JMenu;
import	javax.swing.JMenuItem;
import	javax.swing.JFileChooser;
import	javax.swing.text.AttributeSet;
import	javax.swing.text.SimpleAttributeSet;
import	javax.swing.text.StyleConstants;
import	javax.swing.JOptionPane;
import	javax.swing.SwingUtilities;
import	java.awt.GridBagLayout;
import	java.awt.GridBagConstraints;
import	java.awt.event.ActionListener;
import	java.awt.event.ActionEvent;
import	java.awt.event.MouseListener;
import	java.awt.event.MouseEvent;
import	java.awt.event.MouseWheelListener;
import	java.awt.event.MouseWheelEvent;
import	java.awt.Color;

import	java.io.File;
import	java.io.Writer;
import	java.io.BufferedWriter;
import	java.io.FileWriter;
import	java.io.OutputStreamWriter;
import	java.io.OutputStream;
import	java.io.IOException;

import	java.util.Hashtable;
import	java.util.Enumeration;


/**	A <i>Stream_Monitor</i> provides a scrolling display of styled text
	written to it.
<p>
	Text may be provided directly to the monitor; in this case it acts
	like the JTextPane it employs. Or text may be written to the stream
	the monitor offers. Adding text to the monitor is thread safe. Text
	is added to the bottom of the current content. The capacity of the
	monitor - the amount of text that will be retained - may be
	controlled. When the capacity has been reached text is removed from
	the top of the current content in whole line increments. The text
	content may be edited by the user, saved to a disk file, or cleared.
<p>
	The monitor JTextPane is contained within a JScrollPane initialized
	with the default scroll bar policies. The vertical scroll bar is
	actively managed such that when the thumb is located at the bottom of
	the scroll bar it will stay there, and text added to the monitor will
	scroll up keeping the end of the last text added visible. However, if
	the thumb is moved to any other location the thumb will remain there
	and the visible text will not move unless it must scroll up when the
	capacity is reached and excess text is removed from the top.
<p>
	Text written to the monitor may be auto-styled with a table of
	text-to-style associations. The text-to-style associations may be
	managed as desired.
<p>
	File and View menus configured for text file saving, capacity control
	and content clearing are provided for use in an application menu bar.
	These menus may be modified to suit the needs of the application.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.17
*/
public class Stream_Monitor
	extends JPanel
	implements Styled_Writer
{
/**	Class name and identification.
*/
public static final String
	ID = "Stream_Monitor (1.17 2012/04/16 06:22:59)";


/**	Text style attributes suitable for "stdout" {@link #Auto_Style(String
	AttributeSet) auto-styling}.
<p>
	The style is a bold Monospaced font.
*/
public static final SimpleAttributeSet
	stdout_STYLE			= new SimpleAttributeSet ();

/**	Text style attributes suitable for "stderr" {@link #Auto_Style(String
	AttributeSet) auto-styling}.
<p>
	The style is a bold Monospaced font with a yellow background.
*/
public static final SimpleAttributeSet
	stderr_STYLE			= new SimpleAttributeSet ();

/**	A text style that will be noticed.
<p>
	The style is bold red text.
*/
public static final SimpleAttributeSet
	NOTICE_STYLE			= new SimpleAttributeSet ();

/**	A highlighted text style.
<p>
	The style is blue text.
*/
public static final SimpleAttributeSet
	HIGHLIGHT_STYLE			= new SimpleAttributeSet ();

/**	A marker text style.
<p>
	The style is a bold Monospaced font.
*/
public static final SimpleAttributeSet
	MARKER_STYLE			= new SimpleAttributeSet ();
static
	{
	StyleConstants.setFontFamily	(stdout_STYLE, "Monospaced");
	StyleConstants.setBold			(stdout_STYLE, true);

	StyleConstants.setFontFamily	(stderr_STYLE, "Monospaced");
	StyleConstants.setBold			(stderr_STYLE, true);
	StyleConstants.setBackground	(stderr_STYLE, Color.YELLOW);

	StyleConstants.setBold			(NOTICE_STYLE, true);
	StyleConstants.setForeground	(NOTICE_STYLE, Color.RED);

	StyleConstants.setForeground	(HIGHLIGHT_STYLE, Color.BLUE);

	StyleConstants.setFontFamily	(MARKER_STYLE, "Monospaced");
	StyleConstants.setBold			(MARKER_STYLE, true);
	}

private boolean
	Auto_Style				= false;
private Hashtable
	Auto_Style_Table		= new Hashtable ();

private DefaultStyledDocument
	Monitor_Document;

private JTextPane
	Monitor_Text_Pane;

private JScrollPane
	Scroll_Pane;

//	Auto-scroll controls.
private DefaultCaret
	Text_Caret				= null;
private JScrollBar
	Scroll_Bar				= null;
private int
	Default_Update_Policy;
private static final int
	SCROLL_AT_BOTTOM_MARGIN	= 10;

/**	The lower limit on the {@link #Capacity(int) capacity} value.
*/
public static final int
	MIN_CAPACITY			= 8 * 1024;

private static int
	Default_Capacity		= 1024 * 1024;

private volatile int
	Capacity				= Default_Capacity;

public static final int
	Stream_Buffer_Capacity	= 1024;
	
private Monitor_Writer
	Character_Writer		= new Monitor_Writer ();

private Monitor_OutputStream
	Byte_Stream				= new Monitor_OutputStream ();

private String
	Monitor_Text_Filename	= null,
	CWD						= System.getProperty ("user.dir");

private JMenu
	File_Menu,
	View_Menu;

private static String
	NL						= System.getProperty ("line.separator");
private static int
	NL_length				= NL.length ();


//  DEBUG control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTORS			= 1 << 0,
	DEBUG_WRITE					= 1 << 1,
	DEBUG_AUTO_SCROLL			= 1 << 2,
	DEBUG_ALL					= -1,

	DEBUG		    			= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Stream_Monitor.
*/
public Stream_Monitor ()
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> Stream_Monitor");
Menus ();
Panels ();
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< Stream_Monitor");
}

/*==============================================================================
	Accessors
*/
/**	Get an OutputStream for writing to the monitor display.
<p>
	The stream is line-oriented: Each byte written is buffered until an
	end-of-line sequence, or a maximum of  is encountered at which point the buffer is
	{@link #Write(String) written} to the monitor display. An end-of-line
	line sequence is any one of a line feed ('\n', 0x0A), a carriage
	return ('\r', 0x0D), or a carriage return followed immediately by a
	linefeed.
<p>
	<b>N.B.</b>: Because both the OutputStream returned by this method
	and the {@link #Writer()} are buffered output to the same monitor
	display it is recommended that only one be used with any
	Stream_Monitor object; otherwise unpredictable interleaved text
	may be displayed.
<p>
	@return	An {@link OutputStream} that can be used to write text to
		the monitor display.
*/
public Monitor_OutputStream Stream ()
{return Byte_Stream;}

/**	Get a Writer for writing to the monitor display.
<p>
	The stream is line-oriented: Each character written is buffered until an
	end-of-line sequence is encountered at which point the buffer is
	{@link #Write(String) written} to the monitor display. An end-of-line
	line sequence is any one of a line feed ('\n', 0x0A), a carriage
	return ('\r', 0x0D), or a carriage return followed immediately by a
	linefeed.
<p>
	<b>N.B.</b>: Because both the Writer returned by this method
	and the {@link #Stream()} are buffered output to the same monitor
	display it is recommended that only one be used with any
	Stream_Monitor object; otherwise unpredictable interleaved text
	may be displayed.
<p>
	@return	A {@link Writer} that can be used to write text to the
		monitor display.
*/
public Monitor_Writer Writer ()
{return Character_Writer;}

/**	Set the maximum number of characters to be retained for scrolling.
<p>
	@param	capacity	The maximum number of characters to be retained.
		This value will be limited to not less than {@link #MIN_CAPACITY}.
*/
public Stream_Monitor Capacity
	(
	int		capacity
	)
{
if (capacity < MIN_CAPACITY)
	capacity = MIN_CAPACITY;
Capacity = capacity;
Remove_Excess_Capacity ();
return this;
}

/**	Get the maximum number of characters to be retained for scrolling.
<p>
	@return	The maximum number of characters to be retained.
	@see	#Capacity(int)
*/
public int Capacity ()
{return Capacity;}


private void Remove_Excess_Capacity
	(
	int		additional
	)
{
synchronized (Monitor_Document)
{
if ((Monitor_Document.getLength () + additional) > Capacity)
	{
	/*	Remove the excess capacity.

		Starting at the position offset from the beginning of the document
		by the additonal amount minus the length of a new-line (NL) sequence
		- in case the excess ends with a NL - search for the next NL before
		the end of the document. Then remove text from the beginning of the
		document up to and including the NL that was found.

		If the additional amount is greater than the length of the document
		- which is very unlikely - remove all document text.
	*/
	int
		position = additional - NL_length,		//	Search start position.
		end = Monitor_Document.getLength (),	//	End of the document.
		last = end - NL_length;					//	Last search position.

	/*	If any text is selected, unselect it.  This was causing Stream_Monitor
		to freeze when removing excess capacity.
	*/
	if (Monitor_Text_Pane.getSelectionStart () 
		!= Monitor_Text_Pane.getSelectionEnd ())
		Monitor_Text_Pane.setSelectionEnd 
			(Monitor_Text_Pane.getSelectionStart ());

	if ((DEBUG & DEBUG_WRITE) != 0)
		System.out.println
			("    Removing excess capacity -" + NL
			+"      capacity = " + Capacity + NL
			+"      document = " + end + NL
			+"      addition = " + additional + NL
			+"      begin position = " + position);
	if (additional < end &&
		last >= 0)
		{
		if (position < 0)
			//	New text length is less than NL length.
			position = 0;
		try
			{
			//	Find the next NL.
			while (position < last &&
					! Monitor_Document.getText (position, NL_length)
						.equals (NL))
				++position;
			position += NL_length;
			}
		catch (BadLocationException e)
			{
			// Shouldn't happen.
			if ((DEBUG & DEBUG_WRITE) != 0)
				System.out.println
					("      Invalid search position " + position);
			position = end;
			}
		}
	else
		//	New text length is greater than the document length.
		position = end;

	if ((DEBUG & DEBUG_WRITE) != 0)
		System.out.println
			("      final position = " + position);

	try {Monitor_Document.remove (0, position);}
	catch (BadLocationException exception)
		{
		//	Shouldn't happen: The remove position is within the document.
		if ((DEBUG & DEBUG_WRITE) != 0)
			System.out.println
				("    Stream_Monitor.Write: Failed to remove "
					+ position + " characters to first NL");
		}
	}
}	//	Monitor_Document synchronized.
}


private void Remove_Excess_Capacity ()
{Remove_Excess_Capacity (0);}

/**	Set the default character capacity.
<p>
	When a new Stream_Monitor is constructed its initial capacity
	will be set to the default capacity.
<p>
	@param	capacity	The initial capacity for a new Stream_Monitor.
		This value will be limited to not less than {@link #MIN_CAPACITY}.
*/
public static void Default_Capacity
	(
	int		capacity
	)
{
if (capacity < MIN_CAPACITY)
	capacity = MIN_CAPACITY;
Default_Capacity = capacity;
}

/**	Get the default character capacity.
<p>
	@return	The initial capacity for a new Stream_Monitor.
	@see	#Default_Capacity(int)
*/
public static int Default_Capacity ()
{return Default_Capacity;}

/**	Get the number of characters in the monitor.
<p>
	<b>N.B.</b>: The amount of capacity used may be larger than the
	{@link #Capacity(int) capacity} set for the monitor.
<p>
	@return	The amount of used character space in the monitor.
	@see	#Write(String, AttributeSet)
*/
public int Used ()
{return Monitor_Document.getLength ();}

/**	Enable or disable auto-styling.
<p>
	If auto-styling is enabled, and a style is not specified, all text
	{@link #Write(String, AttributeSet) written} to the monitor display
	is first examined for an initial sequence that matches one of the
	text-to-style keys. If a match is found the associated style is
	applied to the displayed text.
<p>
	@param	enable	If true, auto-styling is enabled; false disables
		auto-styling.
	@return	This Stream_Monitor.
*/
public Stream_Monitor Auto_Style
	(
	boolean	enable
	)
{Auto_Style = enable; return this;}

/**	Test if auto-styling mode is enabled
<p>
	@return	true, auto-styling is enabled; false otherwise.
	@see	#Auto_Style(boolean)
*/
public boolean Auto_Style ()
{return Auto_Style;}

/**	Add an auto-style text-to-style association.
<p>
	When auto-style is enabled each time text is {@link #Write(String)
	written} or the {@link #Stream() output stream} is flushed to the
	monitor display the text is checked for the specified prefix text
	and, if found, the corresponding style is applied.
<p>
	@param	text	The prefix String to be associated with a style.
		If null no association is made.
	@param	style	The AttributeSet to be associated with the text.
		If null no association is made.
	@return	This Stream_Monitor.
	@see	#Auto_Style(boolean)
	@see	#Write(String, AttributeSet)
*/
public Stream_Monitor Auto_Style
	(
	String			text,
	AttributeSet	style
	)
{
synchronized (Auto_Style_Table)
	{
	try {Auto_Style_Table.put (text, style);}
	catch (NullPointerException e) {}
	}
return this;
}

/**	Get the style associated with a text prefix.
<p>
	@param	text	The prefix text that is associated with a style.
	@return	The AttributeSet associated with the text. This will be
		null if there is no text-to-style association.
	@see	#Auto_Style(boolean)
*/
public AttributeSet Auto_Style
	(
	String			text
	)
{
synchronized (Auto_Style_Table)
	{
	try {return (AttributeSet)Auto_Style_Table.get (text);}
	catch (NullPointerException e)
		{return null;}
	}
}

/**	Get the text-to-style association table.
<p>
	@return	The Hastable containing the text-to-style associations.
	@see	#Auto_Style(boolean)
*/
public Hashtable Auto_Style_Table ()
{return Auto_Style_Table;}

/**	Get the monitor's "File" menu.
<p>
	The File menu has "Save" and "Save As ..." menu items, in that order,
	that have action listeners that will {@link
	#Save_Monitor_Text(boolean) save the monitor text}. They have no
	mnemonics or accelerators.
<p>
	@return	A JMenu.
*/
public JMenu File_Menu ()
{return File_Menu;}

/**	Get the monitor's "View" menu.
<p>
	The View menu has "Capacity ..." and "Clear" menu items, in that
	order, that have action listeners that will interactively set the
	monitor character {@link #Capacity(int) capacity} and {@link #Clear()
	clear} the monitor contents, respectively. They have no mnemonics or
	accelerators.
<p>
	@return	A JMenu.
*/
public JMenu View_Menu ()
{return View_Menu;}

/**	Get the JScrollPane that contains the JTextPane where the
	monitor text is displayed.
<p>
	The StyledDocument managing the text can be obtained by:
<p><code>
	Stream_Monitor
	  monitor = new Stream_Monitor();
	...
	StyledDocument
	  document = (JTextPane)(monitor.Scroll_Pane().getViewport().getView())
	    .getStyledDocument();
</code><p>
	@return	The JScrollPane that contains the JTextPane where the
		monitor text is displayed.
*/
public JScrollPane Scroll_Pane ()
{return Scroll_Pane;}

/*==============================================================================
	GUI
*/
private void Menus ()
{
JMenuItem
	menu_item;

File_Menu = new JMenu ("File");

menu_item = new JMenuItem ("Save");
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Save_Monitor_Text (false);}});
File_Menu.add (menu_item);

menu_item = new JMenuItem ("Save As ...");
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Save_Monitor_Text (true);}});
File_Menu.add (menu_item);

View_Menu = new JMenu ("View");

menu_item = new JMenuItem ("Capacity ...");
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Set_Capacity ();}});
View_Menu.add (menu_item);

menu_item = new JMenuItem ("Clear");
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Clear ();}});
View_Menu.add (menu_item);
}


private void Panels ()
{
//	Text pane.
Monitor_Document = new DefaultStyledDocument ();
Monitor_Text_Pane = new JTextPane (Monitor_Document);
Scroll_Pane = new JScrollPane (Monitor_Text_Pane);

setLayout (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
add (Scroll_Pane, location);

//	Auto-scroll controls.
Caret
	caret = Monitor_Text_Pane.getCaret ();
if (caret instanceof DefaultCaret)
	{
	Text_Caret = (DefaultCaret)caret;
	Default_Update_Policy = Text_Caret.getUpdatePolicy ();
	Scroll_Bar = Scroll_Pane.getVerticalScrollBar ();

	//	Add a MouseListener to the Scroll_Bar.
	Scroll_Bar.addMouseListener (new MouseListener ()
		{
		public void mousePressed (MouseEvent event) {}
		public void mouseReleased (MouseEvent event)
			{Page_Scrolled ();}
		public void mouseEntered (MouseEvent event) {}
		public void mouseExited (MouseEvent event) {}
		public void mouseClicked (MouseEvent event) {}
		});

	//	Add a MouseWheelListener to the Scroll_Pane.
	Scroll_Pane.addMouseWheelListener (new MouseWheelListener ()
		{public void mouseWheelMoved (MouseWheelEvent event)
			{Page_Scrolled ();}});
	}
}

/*------------------------------------------------------------------------------
	Actions
*/
/**	Save the current monitor text to a file.
<p>
	If interactive is true a JFileChooser will be used to select the
	file where the text will be saved. Otherwise the text will be
	saved to the previously selected file. If no file has been
	previously selected interactive mode will be used.
<p>
	The current monitor contents are written as plain text.
<p>
	If there was a problem writing the text to the file an
	{@link Dialog_Box#Error(String, Component) error dialog} will
	display the exception message.
<p>
	@param	interactive	If interactive file selection is to be used.
*/
public void Save_Monitor_Text
	(
	boolean		interactive
	)
{
File
	file;
if (! interactive &&
	Monitor_Text_Filename == null)
	//	No filename; force interactive operation.
	interactive = true;
if (interactive)
	{
	JFileChooser
		file_chooser = new JFileChooser (CWD);
	file_chooser.setFileSelectionMode (JFileChooser.FILES_ONLY);
	if (file_chooser.showSaveDialog (this)
			!= JFileChooser.APPROVE_OPTION)
		return;
	file = file_chooser.getSelectedFile ();
	CWD = file.getParent ();
	Monitor_Text_Filename = file.getPath ();
	}
else
	file = new File (Monitor_Text_Filename);

//	Write the monitor text to the file.
synchronized (Monitor_Document)
	{
	try
		{
		BufferedWriter
			writer = new BufferedWriter (new FileWriter (file));
		Monitor_Text_Pane.write (writer);
		writer.close ();
		}
	catch (IOException exception)
		{
		Dialog_Box.Error
			(ID + '\n'
			+ "Unable to save the monitor text to "
				+ file.getPath () + NL + NL
				+ exception.getMessage (),
			this);
		}
	}
}

/**	Save the current monitor text to a file.
<p>
	@param	pathname	The pathname to the file where the text is to
		be save. If null, interactive file selection will be used.
	@see	#Save_Monitor_Text(boolean)
*/
public void Save_Monitor_Text
	(
	String		pathname
	)
{
Monitor_Text_Filename = pathname;
Save_Monitor_Text (false);
}

/**	Save the current monitor text to a file.
<p>
	If a file has previously been selected for saving the monitor text
	then it is overwritten. Otherwise interactive file selection is used.
<p>
	@see	#Save_Monitor_Text(boolean)
*/
public void Save_Monitor_Text ()
{Save_Monitor_Text (false);}


private void Set_Capacity ()
{
int
	used = Used ();
double
	capacity;
String
	value,
	capacity_value = String_Utilities.Amount_Magnitude (Capacity),
	message =
		"Current content - " + String_Utilities.Amount_Magnitude (used);
if (used >= 1024)
	message += " (" + used + ')';
message += NL + "Monitor capacity (characters):";
		
while ((value = JOptionPane.showInputDialog (this, message, capacity_value))
		!= null)
	{
	if (value.length () == 0)
		break;
	
	if (! Character.isDigit (value.charAt (value.length () - 1)) &&
		value.charAt (value.length () - 1) != '.')
		{
		switch (Character.toUpperCase (value.charAt (value.length () - 1)))
			{
			case 'K':	used = 1024;			break;
			case 'M':	used = 1024 * 1024;	break;
			default:
				Dialog_Box.Notice ("Please enter a numeric value.", this);
				continue;
			}
		value = value.substring (0, value.length () - 1);
		}
	else
		used = 1;

	try {capacity = Double.parseDouble (value);}
	catch (NumberFormatException exception)
		{
		Dialog_Box.Notice ("Please enter a numeric value.", this);
		continue;
		}
	capacity *= used;
	if (capacity < 0)
		{
		Dialog_Box.Notice
			("A negative capacity can not be used.",
			this);
		continue;
		}
	if (capacity < MIN_CAPACITY)
		{
		Dialog_Box.Notice
			("The minimum capacity is "
				+ MIN_CAPACITY + " characters.",
			this);
		continue;
		}
	if (capacity >= Integer.MAX_VALUE)
		{
		Dialog_Box.Notice
			("The value is too large.",
			this);
		continue;
		}
	Capacity ((int)capacity);
	break;
	}
}

/**	Manage page scrolling event.

	The page has been scrolled, either by the scroll bar or by the mouse
	wheel.  Check the scroll bar position and current document caret
	position, and if the scroll bar was moved to the end, move the caret
	to the end.  Otherwise the scroll bar was moved away from the end.
	If the  caret was at the end, move it away from the end. Otherwise,
	leave the caret where it is.
*/
private void Page_Scrolled ()
{
if (Scroll_Bar.getMinimum () +
	Scroll_Bar.getValue () +
	Scroll_Bar.getVisibleAmount () +
	SCROLL_AT_BOTTOM_MARGIN
	>= Scroll_Bar.getMaximum ())
	{
	//	Scroll bar moved to end, set caret to end.
	Monitor_Text_Pane.setCaretPosition
		(Monitor_Text_Pane.getDocument ().getLength ());
	Text_Caret.setVisible (false);
	}
else if (Text_Caret.getMark () == Monitor_Text_Pane.getDocument ().getLength ())
	{
	//	Scroll bar moved away from end, if caret is at end, move it.
	Monitor_Text_Pane.setCaretPosition
		(Monitor_Text_Pane.getDocument ().getLength () - 1);
	Text_Caret.setVisible (false);
	}
}

/*==============================================================================
	Manipulators
*/
/**	Clear the monitor of all text.
*/
public Stream_Monitor Clear ()
{
synchronized (Monitor_Document)
{
try {Monitor_Document.remove (0, Monitor_Document.getLength ());}
catch (BadLocationException exception) {}
}
Monitor_Text_Pane.setCaretPosition (0);
return this;
}

/**	Write text to the monitor.
<p>
	This method {@link #Write(String, AttributeSet) write}s the text
	with a null style.
<p>
	@param	text	A text String to be added to the end of the monitor
		display.
	@see	#Write(String, AttributeSet)
*/
public Stream_Monitor Write
	(
	String			text
	)
{Character_Writer.Write (text, null); return this;}

/**	Write styled text to the monitor.
<p>
	If the style is null and {@link #Auto_Style(boolean) auto-styling}
	is enabled the initial portion of the text will be checked for an
	auto-style key. If one is found the associated style will be applied.
	<b>N.B.</b>: The style is applied to all of the text specified;
	auto-styling is not limited to the single line which begins with
	an auto-style prefix.
<p>
	If the amount of space {@link #Used() used} is greater than the
	{@link #Capacity() capacity} plus a {@link #MIN_CAPACITY} margin,
	monitor text is removed from the top (FIFO) up to and including
	the end-of-line characters to bring the amount used below the
	capacity.
<p>
	The text is appended to the bottom of the monitor with the style,
	if any, applied. The scroll pane is moved up so that the end of
	the text appears at the bottom of the display.
<p>
	A style is assembled using combinations of the numerous
	StyleConstants setXXX functions applied to a SimpleAttributeSet. For
	example:
<p>
<code><pre>
SimpleAttributeSet bold_red_text = new SimpleAttributeSet ();
StyleConstants.setBold       (bold_red_text, true);
StyleConstants.setForeground (bold_red_text, Color.RED);
StyleConstants.setFontFamily (bold_red_text, "Monospaced");
</pre></code>
<p>
	assembles a style that will produce text in a bold, red, monospaced
	font.
<p>
	@param	text	The text String to be appended to the monitor.
	@param	style	The AttributeSet to be applied to the text. This
		may be null if plain text is to be displayed.
	@see	StyleConstants
*/
public Stream_Monitor Write
	(
	String			text,
	AttributeSet	style
	)
{Character_Writer.Write (text, style); return this;}


private void Write_to_Monitor
	(
	String			text,
	AttributeSet	style
	)
{
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		(">>> Stream_Monitor.Write_to_Monitor:\n"
		+ text);

if (text == null ||
	text.length () == 0)
	{
	if ((DEBUG & DEBUG_WRITE) != 0)
		System.out.println
			("<<< Stream_Monitor.Write_to_Monitor");
	return;
	}

if (Auto_Style &&
	style == null)
	{
	synchronized (Auto_Style_Table)
		{
		Enumeration
			keys = Auto_Style_Table.keys ();
		while (keys.hasMoreElements ())
			{
			String
				key = (String)keys.nextElement ();
			if (text.startsWith (key))
				{
				style = (AttributeSet)Auto_Style_Table.get (key);
				break;
				}
			}
		}
	}
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		("    Style " + style);

Remove_Excess_Capacity (text.length ());

//	Append the text on the event dispatcher thread.
final String
	string = text;
final AttributeSet
	attributes = style;

SwingUtilities.invokeLater (new Runnable ()
	{
	public void run ()
	{
	/*
	if (Text_Caret != null &&
		Scroll_Pane.getViewport ().getViewRect ().height != 0)
		{
		//	Auto-scroll control.
		if ((DEBUG & DEBUG_AUTO_SCROLL) != 0)
			System.out.println
				("--> Auto-scroll control:" + NL
				+"     Viewport = "
					+ Scroll_Pane.getViewport ().getViewRect () + NL
				+"    ScrollBar -" + NL
				+"      minimum = " + Scroll_Bar.getMinimum () + NL
				+"        value = " + Scroll_Bar.getValue () + NL
				+"       extent = " + Scroll_Bar.getVisibleAmount () + NL
				+"      maximum = " + Scroll_Bar.getMaximum ());

		if ((Scroll_Bar.getMinimum ()
			+Scroll_Bar.getValue ()
			+Scroll_Bar.getVisibleAmount ()
			+SCROLL_AT_BOTTOM_MARGIN)
				>= Scroll_Bar.getMaximum ())
			{
		if ((DEBUG & DEBUG_AUTO_SCROLL) != 0)
			System.out.println
				("    ScrollBar thumb at bottom");
			//	The scroll bar thumb is at the bottom position.
			Text_Caret.setUpdatePolicy (Default_Update_Policy);
			}
		else
			{
			if ((DEBUG & DEBUG_AUTO_SCROLL) != 0)
				System.out.println
					("    ScrollBar thumb not at bottom");
			Text_Caret.setUpdatePolicy (DefaultCaret.NEVER_UPDATE);
			}
		}
	*/

	//	Add the new text.
	try {Monitor_Document.insertString
			(Monitor_Document.getLength (), string, attributes);}
	catch (BadLocationException e)
		{
		if ((DEBUG & DEBUG_WRITE) != 0)
			System.out.println
				("    Stream_Monitor.Write: Failed to append "
					+ string.length () + " characters");
		}
	/*
	if (Text_Caret != null)
		{
		if (Text_Caret.getUpdatePolicy () != DefaultCaret.NEVER_UPDATE)
			{
	*/
			//	Reposition at the end of the text pane.
			/* 
			Monitor_Text_Pane.setCaretPosition (Monitor_Document.getLength ());
			Scroll_Bar.setValue
				(Scroll_Bar.getMaximum () - Scroll_Bar.getVisibleAmount ());
			*/
	/*
			}
		if ((DEBUG & DEBUG_AUTO_SCROLL) != 0)
			System.out.println
				("<-- After insert:" + NL
				+"     Viewport = "
					+ Scroll_Pane.getViewport ().getViewRect () + NL
				+"    ScrollBar -" + NL
				+"      minimum = " + Scroll_Bar.getMinimum () + NL
				+"        value = " + Scroll_Bar.getValue () + NL
				+"       extent = " + Scroll_Bar.getVisibleAmount () + NL
				+"      maximum = " + Scroll_Bar.getMaximum ());
		}
	*/
	}});
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		("<<< Stream_Monitor.Write");
}

/*==============================================================================
	OutputStream implementation
*/
public class Monitor_Writer
	extends Writer
	implements Styled_Writer
{
private char[]
	Character_Buffer	= new char[Stream_Buffer_Capacity];
private int
	Character_Count		= 0;


/**	Write a character.
<p>
	Each character written is buffered until an end-of-line sequence is
	encountered at which point the buffer is {@link #flush() flushed} to
	the display. An end-of-line line sequence is any one of a line feed
	('\n', 0x0A), a carriage return ('\r', 0x0D), or a carriage return
	followed immediately by a linefeed.
<p>
	@param	character	An integer value of which only the least significant
		16-bits are used.
*/
public void write
	(
	int		character
	)
{put_character ((char)(character & 0xFFFF));}


private void put_character
	(
	char	character
	)
{
if (character != 0x0D &&
	Character_Count > 0 &&
	Character_Buffer[Character_Count - 1] == 0x0D)
	//	CR not followed by a LF.
	flush ();

Character_Buffer[Character_Count++] = character;
if (character == 0x0A ||	//	LF
	Character_Count == Character_Buffer.length)
	flush ();
}

/**	Write an array of characters.
<p>
	The characters are buffered until an end-of-line sequence is
	encountered or the buffer is full at which point the buffer is
	{@link #flush() flushed} to the display. An end-of-line line
	sequence is any one of a line feed ('\n', 0x0A), a carriage return
	('\r', 0x0D), or a carriage return followed immediately by a
	linefeed.
<p>
	@param	characters	The char array containing the characters to be
		written. If null or empty nothing is done.
	@param	offset	Array offset from which to start writing characters.
	@param	amount	The number of characters to write. If zero nothing is
		done.
	@throws	IndexOutOfBoundsException	If the offset or amount are
		negative or the offset plus the amount is greater than the length
		of the array.
*/
public void write
	(
	char[]	characters,
	int		offset,
	int		amount
	)
{
if (characters == null ||
	characters.length == 0 ||
	amount == 0)
	return;

if (offset < 0 ||
	amount < 0 ||
	(offset + amount) > characters.length)
	throw new IndexOutOfBoundsException (ID + NL
		+ "Invalid values for " + characters.length
			+ " character array content -" + NL
		+ "  Array offset " + offset + " and amount " + amount + '.');
amount += offset;
while (offset < amount)
	put_character (characters[offset++]);
}

/**	Write styled text to the monitor.
<p>
	Any currently buffered characters are flushed before the text
	is written to the monitor.
<p>
	@param	text	The text String to be appended to the monitor.
	@param	style	The AttributeSet to be applied to the text. This
		may be null if plain text is to be displayed.
	@see	Stream_Monitor#Write(String, AttributeSet)
*/
public Monitor_Writer Write
	(
	String			text,
	AttributeSet	style
	)
{
Write_to_Monitor (text, style);
return this;
}

/**	Write text to the monitor.
<p>
	Any currently buffered characters are flushed before the text
	is written to the monitor.
<p>
	@param	text	The text String to be appended to the monitor.
	@see	Stream_Monitor#Write(String)
*/
public Monitor_Writer Write
	(
	String			text
	)
{
Write_to_Monitor (text, null);
return this;
}

/**	Flush the currently buffered characters to the monitor display.
<p>
	The currently buffered characters are converted to a String that is
	{@link #Write(String, AttributeSet) written} to the monitor display.
	The text is eligible for {@link #Auto_Style(boolean) auto-style}
	processing, if enabled.
*/
public void flush ()
{
if (Character_Count > 0)
	{
	Write_to_Monitor (new String (Character_Buffer, 0, Character_Count), null);
	Character_Count = 0;
	}
}

/**	The currently buffered characters are flushed, but the writer
	can still be written to.
*/
public void close ()
{flush ();}

};	//	End of Monitor_Writer class.

/*==============================================================================
	Writer implementation
*/
public class Monitor_OutputStream
	extends OutputStream
{
private byte[]
	Byte_Buffer	= new byte[Stream_Buffer_Capacity];
private int
	Byte_Count	= 0;


/**	Write a byte to the monitor.
<p>
	Each byte written is buffered until an end-of-line sequence is
	encountered or the buffer is full at which point the buffer is
	{@link #flush() flushed} to the display. An end-of-line line
	sequence is any one of a line feed ('\n', 0x0A), a carriage return
	('\r', 0x0D), or a carriage return followed immediately by a
	linefeed.
<p>
	@param	value	An integer value of which only the least significant
		byte is buffered.
*/
public synchronized void write
	(
	int		value
	)
{
if (value != 0x0D &&
	Byte_Count > 0 &&
	Byte_Buffer[Byte_Count - 1] == 0x0D)
	//	CR not followed by a LF.
	flush ();
	
Byte_Buffer[Byte_Count++] = (byte)(value & 0xFF);
if (value == 0x0A ||
	Byte_Count == Byte_Buffer.length)
	//	LF.
	flush ();
}

/**	Flush the currently buffered bytes to the monitor display.
<p>
	The currently buffered bytes are converted to a String using the
	default character encoding. This text is {@link #Write(String,
	AttributeSet) written} to the monitor display. The text is eligible
	for {@link #Auto_Style(boolean) auto-style} processing, if enabled.
*/
public void flush ()
{
if (Byte_Count > 0)
	{
	Write_to_Monitor (new String (Byte_Buffer, 0, Byte_Count), null);
	Byte_Count = 0;
	}
}

/**	The currently buffered characters are flushed, but the writer
	can still be written to.
*/
public void close ()
{flush ();}

};	//	End of Monitor_OutputStream class.

}
