/*	PVL_Exception

PIRL CVS ID: PVL_Exception.java,v 1.12 2012/04/16 06:14:23 castalia Exp

Copyright (C) 2001-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.PVL;

import	java.lang.Exception;


/**
A <I>PVL_Exception</I> is thrown by classes in the PVL package. To
differentiate the possible causes for a PVL_Exception, while avoiding a
profusion of subclasses, a local <I>Message</I> String is provided that
describes the reason for the exception. The <CODE>getMessage</CODE>
method of the Exception base class will always return a String that
includes this message along with any other description that was
provided when the exceptpion was created. In addition, a
<I>Location</I> is available which can indicate where a <I>Parser</I>
encountered an exception in an input stream.
<P>
@author		Bradford Castalia, UA/PIRL
@version	1.12 
*/
public class PVL_Exception
	extends Exception
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.PVL.PVL_Exception (1.12 2012/04/16 06:14:23)";

/**	The name of this exception.
*/
public static final String
	Exception_Name = "PVL Exception";


//	Hard errors:

/**	<CODE>Message</CODE>: Unexpected system error!
*/
public static final String
	SYSTEM_ERROR				= "Unexpected system error!";

/**	<CODE>Message</CODE>: Invalid argument.
*/
public static final String
	BAD_ARGUMENT				= "Invalid argument.";

/**	<CODE>Message</CODE>: Incompatible object class types.
*/
public static final String
	INCOMPATIBLE_TYPES			= "Incompatible object class types.";

/**	<CODE>Message</CODE>: Problem with file I/O.
*/
public static final String
	FILE_IO						= "Problem with file I/O.";

	//	Syntax errors:

/**	<CODE>Message</CODE>: Missing end of quoted string.
*/
public static final String
	MISSING_QUOTE_END			= "Missing end of quoted string.";

/**	<CODE>Message</CODE>: Missing end of comment string.
*/
public static final String
	MISSING_COMMENT_END			= "Missing end of comment string.";

/**	<CODE>Message</CODE>: Missing end of units string.
*/
public static final String
	MISSING_UNITS_END			= "Missing end of units string.";

/**	<CODE>Message</CODE>: Reserved character in symbol.
*/
public static final String
	RESERVED_CHARACTER			= "Reserved character in symbol.";

/**	<CODE>Message</CODE>: Illegal syntax.
*/
public static final String
	ILLEGAL_SYNTAX				= "Illegal syntax.";

/**	<CODE>Message</CODE>: Empty statement or list.
*/
public static final String
	EMPTY_STATEMENT				= "Empty statement or list.";

/**	<CODE>Message</CODE>: Numeric value out of range.
*/
public static final String
	VALUE_OVERFLOW				= "Numeric value out of range.";

/**	<CODE>Message</CODE>: Array enclosure characters do not match.
*/
public static final String
	ARRAY_CLOSURE_MISMATCH		= "Array enclosure characters do not match.";

	//	Abstract (high level) usage warnings:

/**	<CODE>Message</CODE>: Inappropriate aggregate identifier (not a string).
*/
public static final String
	GROUP_VALUE					= "Inappropriate aggregate identifier (not a string).";

/**	<CODE>Message</CODE>: Missing end of aggregate parameters.
*/
public static final String
	MISSING_AGGREGATE_END		= "Missing end of aggregate parameters.";

/**	<CODE>Message</CODE>: Aggregate enclosure parameters do not match.
*/
public static final String
	AGGREGATE_CLOSURE_MISMATCH	= "Aggregate enclosure parameters do not match.";

/**	<CODE>Message</CODE>: Binary sized records detected.
*/
public static final String
	SIZED_RECORDS				= "Binary sized records detected.";

/**	<CODE>Message</CODE>: Could not find the last location.
*/
public static final String
	NO_LAST_LOCATION			= "Could not find the last location.";

private String
	_Message_ = null;

private long
	_Location_ = -1;


/**	Creates a PVL_Exception with the PVL_Exception class identification
	String as the default <CODE>Message</CODE>.
*/
public PVL_Exception ()
{this (ID);}

/**	Creates a PVL_Exception with the specified message.
<P>
	@param	message	A String to set as the exception <CODE>Message</CODE>.
*/
public PVL_Exception
	(
	String	message
	)
{
super
	(
	Exception_Name + ":\n"
	+ message
	);
_Message_ = message;
}

/**	Creates a PVL_Exception with the specified message and
	explanation text.

	The expanation will follow the message after a newline in the
	<CODE>getMessage</CODE> String.
<P>
	@param	message	A String to set as the exception <CODE>Message</CODE>.
	@param	explanation	A String to append to the exception description.
*/
public PVL_Exception
	(
	String	message,
	String	explanation
	)
{
this
	(
	message + "\n"
	+ explanation
	);
_Message_ = message;
}

/**	Creates a PVL_Exception with the specified ID line, message,
	and explanation text.

	The ID line will precede the message, and the expanation will
	follow, in the <CODE>getMessage</CODE> String.
<P>
	@param	ID	A String to precede the message in the exception description.
	@param	message	A String to set as the exception <CODE>Message</CODE>.
	@param	explanation	A String to append to the exception description.
*/
public PVL_Exception
	(
	String	ID,
	String	message,
	String	explanation
	)
{
this
	(
	ID + "\n"
	+ message + "\n"
	+ explanation
	);
_Message_ = message;
}

/**	Creates a PVL_Exception with the specified ID line, message,
	explanation text, and location value.

	The ID line will precede the message, and the expanation will
	follow, in the <CODE>getMessage</CODE> String. A line providing
	the location value will also be appended.
<P>
	@param	ID	A String to precede the message in the exception description.
	@param	message	A String to set as the exception <CODE>Message</CODE>.
	@param	explanation	A String to append to the exception description.
	@param	location	A long to set as the exception <CODE>Location</CODE>.
*/
public PVL_Exception
	(
	String	ID,
	String	message,
	String	explanation,
	long	location
	)
{
this
	(
	ID + "\n"
	+ message + "\n"
	+ explanation + "\n"
	+ "At data input location " + location + "."
	);
_Message_ = message;
_Location_ = location;
}

/**	Gets a String composed of "PVL Exception: " followed by the
	<CODE>Message</CODE>.
<P>
	@return	A brief description of the exception.
*/
public String toString ()
{
return (Exception_Name + ": " + _Message_);
}

/**	Gets the exception's local message String.

	The message String identifies the specific cause for the
	exception. It might be used as follows:

<PRE>try
&nbsp;&nbsp;&nbsp;&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;// Some PVL operation.
&nbsp;&nbsp;&nbsp;&nbsp;}
catch (PVL_Exception exception)
&nbsp;&nbsp;&nbsp;&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;if (exception.Message ().equals (PVL_Exception.FILE_IO))
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// Do something with the file.
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;// ...
&nbsp;&nbsp;&nbsp;&nbsp;}
</PRE><P>
	@return	The exception's local message String.
*/
public String Message ()
{return _Message_;}

/**	Gets the exception's location value.

	The value will be -1 if no value was set.
<P>
	@return	The exception's location value.
*/
public long Location ()
{return _Location_;}



}	//	End of class


	
