---
subtitle: Application Development
---

The [Quick Start](quickstart.md) tutorial introduces how to setup a new application project,
and [Application Design](application.md) draw a high-level view of how it works.
This tutorial goes into more details about the development environment provided by the **Dome**
framework.


## HTML & JavaScript Environment

The Web environment for designing your application is based on the following
framework:

- **Electron** [v5.0+](https://electronjs.org) runtime environment (packaging Chrome and Node)
- **Node** [v12.0+](https://nodejs.org/dist/latest-v12.x/docs/api) provided by Electron
- **Chromium** [v73.0+](https://www.chromium.org/Home) HTML rendering engine provided by Electron

You will write your application in a an environment setup with the following
JavaScript features:

- **Webpack** for packaging the application from sources
- **Babel** to parse language extensions
- **Babel Presets Env** to use ECMA modules and JavaScript strict mode
- **Babel Presets React** to use JSX react notation in all your `*.js` files
- **Babel Object Rest Spread** to use object spread syntax `{ ...props }`
- **React v16** with its latest features (including Hooks)
- **React Hot Loader** for [live-editing](hotreload.md) your application
- **CSS Loader** to load and install `*.css` files

## Available Packages

The implementation and internals of the **Dome** framework rely on
high-quality and mature popular packages. They are automatically packed
into your dependencies by the provided template makefile. We list them here
since you may re-use them for your own purpose:

- [**Lodash**](https://lodash.com) convenient JavaScript utilities
- [**React-Virtualized**](http://bvaughn.github.io/react-virtualized)
for layout of large data set in list, tables and grids
- [**React-Draggable**](https://github.com/mzabriskie/react-draggable)
for straightforward drag-and-drop features
- [**CodeMirror**](https://codemirror.net) for rich text capabilities and edition
that scales on large document

## Project Files Organization

The main directory holding your application project with **Dome** consists of
several files and directories. Each is created and/or managed by various
entities : yourself the **Developer**, the **Dome** template makefile, the
**Yarn** JavaScript package manager and the **Electron** suite of
utilities. Parts of those files are generated and updated on demand and shall or
shall _not_ be tracked in your source version control system, be it based on `git`,
`svn` or anything else.

Most files are actually created and used by **Dome**, **Electron** and **Yarn**.
However, that are _in fine_ under the responsibility of the **Developer**
since you can edit and extend them to fit your needs.
Those generated files generally contains comments to indicate how you
can extend them. You shall put those files under version control only if you edit them.

As depicted in the [Quick Start](quickstart.md) tutorial, the only file you
need to create by yourself is the main `Makefile`, with a minimal content
limited to:

```makefile
DOME="..." ;; path to your dome installation
include $(DOME)/template/makefile
```

The following table lists the main files and directories of your project, with
the associated entity responsible for it, and whether they should be tracked by
your source version control system (VCS).

Optional files are tagged with « (Opt.) » in the corresponding `Entity` column,
and are _not_ generated by default. Similarly, files mainly used by the system
but that can still be configured by the developer are tagged with « (+Dev) ».
Files are tagged with « ✓(*) » in the `VCS` column shall be tracked only if you
actually modified them.

| File or Directory | Description | Entity  | VCS |
|:------------------|:------------|:-------:|:---:|
| `./Makefile` | Drives the build system | **Developer** | ✓ |
| `./src/main/` | Sources for the main process | **Developer** | ✓ |
| `./src/main/index.js` | Main process entry point | **Electron** (+Dev) | ✓ |
| `./src/renderer/` | Sources for the renderer process | **Developer** | ✓ |
| `./src/renderer/index.js` | Renderer process entry point | **Electron** (+Dev) | ✓ |
| `./src/renderer/Application.js` | Top React component of the main application window | **Developer** | ✓ |
| `./src/renderer/Preferences.js` | Top React component of the user preferences window | **Developer** | ✓ |
| `./src/common/` | Sources for _both_ processes | **Developer** (Opt.) | ✓ |
| `./static/` | Static resources | **Developer** (Opt.) | ✓ |
| `./package.json` | JavaScript packages configuration | **Yarn** (+Dev) | ✓ |
| `./node_modules` | Downloaded packages | **Yarn** | 𐄂 |
| `./yarn.lock` | Packages versions stamps | **Yarn** | ✓ |
| `./electron-webpack.json` | Electron configuration | **Electron** (+Dev) | ✓(*) |
| `./babelrc` | Babel configuration | **Electron** (+Dev) | ✓(*) |
| `./webpack.main.json` | Webpack main process config. | **Electron** (+Dev) | ✓(*) |
| `./webpack.renderer.json` | Webpack renderer process config. | **Electron** (+Dev) | ✓(*) |
| `./.dome-*` | Dome makefile stamps | **Dome** | 𐄂 |
| `$(DOME)/*` | Dome framework clone | **Dome** | (var.) |
| `./dist/` | Packaged application | **Dome** | 𐄂 |
| `./bin/` | Local command-lines | **Dome** | 𐄂 |

The main entry points of the two **Electron** processes running your application,
`src/main/index.js` and `src/renderer/index.js` shall be modified with extreme care,
since they are in charge with the correct bootstrap of your application.
The generated files from **Dome** templates contains explicit comments and guidelines.

You may want to modify the name of your application top-level **React** components,
namely `src/renderer/Application.js` and `src/render/Preferences.js` by default.
To do this, you shall hook into the
`src/renderer/index.js` file and follow the provided comments and guidelines,
and set the `DOME_CUSTOM_ENTRIES` variable into your own `./Makefile` to make
**Dome** aware about this.

If you need static resource files to be packed with your application, put them in the `static` directory.
You can then access to their actual path from your code by using the `System.getStatic(<path>)` utility function.

The location of the **Dome** resources actually depends on the developer
profile, as explained in more detailed in the [Live Editing](hotreload.md) tutorial.
To summarize:
- for normal developers, it can be a local copy of the dome framework, for instance in your `/usr/local/lib/dome` directory;
- for developers working with a specific or modified version of the dome framework, it can be in the local
`./dome/` directory of your project, with or without versioning control;
- for **Dome** developers, it is also possible to clone the Git repository inside the `./src/dome/` sub-directory of your
project, hence also enabling live-editing of **Dome** itself;
- for **Dome** maintainers, it is even possible to use the root of the **Dome** development repository as an application project
in itself, for demos, examples or testing purposes.

Most of the **Electron** configuration files depends on the definition of the
`DOME` environment variable to find the necessary resources for your application
during development. Commands initiated by **Dome** from `$(DOME)/template/makefile`
will explicitly copy the `$(DOME)` makefile variable into the `$DOME` shell environment
variable when calling the **Electron** builder utilities, and might use it to generate static
files from templates, if necessary.

If you ever _move_ the place where **Dome** is installed, you
probably have to inspect the generated files that you have modified to check
if they still behave as expected.

Notice, however, that distributed versions of
the application (those accessible from the `./dist/` directory) are
self-contained: they are _not_ tied to your local installation of **Dome**.

The command-line version of the application is accessible to the developer from the `./bin` directory.

# Common Development Tasks

Most development tasks shall be accessible from your `./Makefile`, and simply delegate to the
predefined **Dome** targets in `$(DOME)/template/makefile` ; however, for very specific needs,
you can borrow the content of the template makefile into your own one and modify it at your
convenience. The **Dome** environment will not target this template makefile explicitly.

The top target of the **Dome** template makefile is `make dome-help`,
which is thus activated by default unless you put your own targets _before_
its inclusion. The `dome-help` target simply echoes a short list of all the
`dome-xxx` predefined targets in the template. It also reports on
actual values of makefile variables used by the template.

```
$ make dome-help

[Dome] Builtin Targets:
  make dome-help    This help
  make dome-dev     Run application in development mode
  make dome-app     Compile application
  make dome-dist    Package application for production
  make dome-doc     Generate Dome documentation
  make dome-api     Generate Application documentation
  make dome-pkg     Install development packages
  make dome-templ   Update templates
  make dome-reboot  Rebuild templates from scratch
  make dome-clean   Remove dome temporary files
  make dome-plugins Package plugins for distribution

[Dome] Development:
  … (environment variables)

[Dome] Documentation:
  … (environment variables)

```

As depicted above, the **Dome** template also provides two extensible targets `help::` and `clean::`.
By default, the `make clean` would only remove the **Dome** stamp files, not the actually generated files,
since they can have been modified when developing the application. This can cause, however, an _update_ of
configuration files that can be managed incrementally, typically for the `package.json` and the installed
JavaScript packages.

The **Dome** framework can generate its own documentation locally, hence you may consult it offline. It
is also capable of generating internal **JSDoc** documentation of the application itself. The application
documentation does _not_ contain the framework documentation, though.

The environment variable can be redefined _via_ your own Makefile before the **Dome** template inclusion.
