// Copyright 2016 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
////////////////////////////////////////////////////////////////////////////////

#include "compact_enc_det/compact_enc_det.h"

#include <stdio.h>                      // for fprintf, stderr, FILE, etc
#include <string.h>                     // for strlen, NULL
#include <string>                       // for string


#include "gtest/gtest.h"
#include "util/basictypes.h"
#include "util/commandlineflags.h"
#include "util/varsetter.h"             // for VarSetter

using std::string;

DECLARE_bool(enc_detect_detail);
DECLARE_bool(ced_allow_utf8utf8);
DEFINE_int32(trackme, -1, "Track this encoding in --enc_detect_detail output");

static const char kDetailHead[] =
#include "compact_enc_det/detail_head_string.inc"  // IWYU pragma: keep
;

// escape PostScript string (no parens)
string PsStr(const char* str) {
  string res("");
  for (size_t i = 0; i < strlen(str); ++i) {
    char c = str[i];
    if (c == '(') {res.append("\\(");}
    else if (c == ')') {res.append("\\)");}
    else if (c == '\\') {res.append("\\\\");}
    else {res.append(1, c);}
  }
  return res;
}

void PsCopyHeader(FILE* psfile) {
  // Write all but the trailing NUL
  fwrite(kDetailHead, 1, sizeof(kDetailHead) - 1, psfile);
  if (FLAGS_trackme >= 0) {
    fprintf(psfile, "/track-me %d def\n\n", FLAGS_trackme);
  }
}

void PsEndfile(FILE* psfile, int pagenum) {
  fprintf(psfile, "showpage\n\n");
}

// Test strings. These are all included here to make the unit test self-contained.

const char* kTeststr00 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/pt_wikipedia_org_clean__ASCII.txt
  " a Al-Qaeda causa a morte a 202 pessoas. 2003 - Michael Schumacher, conq"
  "uista o Hexa na F\xF3""rmula 1. Nasceram neste dia... 1537 - Rei Eduardo"
  " VI de Inglaterra (m. 1553) 1798 - Imperador Pedro I do Brasil (Pedro IV"
  " de Portugal, foto). 1923 - Fernando Sabino, escritor e jornalista brasi"
  "leiro (m. 2004) 1935 - Luciano Pavarotti, cantor l\xED""rico italiano Fa"
  "leceram neste dia... 638 - Papa Hon\xF3""rio I 1965 - Paul Hermann M\xFC"""
  "ller, qu\xED""mico su\xED""\xE7""o (n. 1899) 1992 - Ulysses Guimar\xE3"""
  "es, pol\xED""tico brasileiro (n. 1916) Outros dias: 10 \xB7"" 11 \xB7"" "
  "12 \xB7"" 13 \xB7"" 14 | ver todos... Sabia que... Durante 400 anos, n\xE3"""
  "o choveu no deserto do Atacama? E que esse \xE9"" o maior per\xED""odo s"
  "em chuvas j\xE1"" registrado? Fala, o c\xE3""o do presidente Franklin Ro"
  "osevelt, tornou-se um soldado raso honor\xE1""rio do Ex\xE9""rcito Ameri"
  "cano ao \"contribuir\" com um d\xF3""lar para o esfor\xE7""o de guerra p"
  "ara cada dia do ano? O primeiro presidente da Academia Brasileira de Let"
  "ras foi Machado de Assis, aclamado em 15 de Dezembro de 1896? O visual d"
  "os personagens Princesa Jasmine e Aladdin, do filme Aladdin, cl\xE1""ssi"
  "co de anima\xE7""\xE3""o da Disney de 1992, foi inspirado no visual da a"
  "triz Jennifer Connelly e do ator Tom Cruise? Llanfairpwllgwyngyllgogeryc"
  "hwyrndrobwllllantysiliogogogoch \xE9"" a cidade com o maior n\xFA""mero "
  "de letras do mundo, com 58 letras? A escritora brasileira Clarice Lispec"
  "tor nasceu na Ucr\xE2""nia? arquivo Eventos recentes O escritor turco Or"
  "han Pamuk \xE9"" galardoado com o Pr\xE9""mio Nobel da Literatura. Avi\xE3"""
  "o particular do jogador de basebol Cory Lidle colide com pr\xE9""dio res"
  "idencial em Nova Iorque. Google adquire o maior servi\xE7""o de v\xED""d"
  "eos da internet, o YouTube, por 1,65 bilh\xE3""o de d\xF3""lares. Coreia"
  " do Norte realiza pela primeira vez um teste nuclear subterr\xE2""neo, a"
  "lvo de fortes cr\xED""ticas pela comunidade internacional. Erup\xE7""\xE3"""
  "o de vulc\xE3""o causa p\xE2""nico na Papua Nova Guin\xE9""; o calor e o"
  "s ventos causados foram respons\xE1""veis pelo estilha\xE7""amento das j"
  "anelas de resid\xEA""ncias num raio de at\xE9"" doze quil\xF3""metros. A"
  "nna Politkovskaia, jornalista russa conhecida pelas cr\xED""ticas ao gov"
  "erno de Vladimir Putin, foi assassinada em Moscovo. T\xEA""m in\xED""cio"
  " os primeiros Jogos da Lusofonia em Macau. OTAN assume comando das tropa"
  "s no Afeganist\xE3""o. Brasil decidir\xE1"" em segundo turno a"
;

// By design, this is the most sensitive test of the bunch, i.e. the most
// likely to fail when the encoding detector is changed.
// On a good day, Latin4 will be only about 2**2 more likely than Latin2.
// In real input, Latin4 is almost always accompanied by a charset= hint
//
const char* kTeststr03 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/lt_wikipedia_org_clean__Latin4.txt
// Hand-edit to boost Latin4 by small amounts. Bigrams here must differ or
// CED will ignore duplicates.
  ", kuria neteko vis\xF9"". savo va"
  ", kuria neteko vis\xF9""j savo va"
  "Kor\xEC""jai "
  "\xE7""vykiai "
  "kra\xB9""tus "
  "Pr\xFE""sija "
  "t\xFE""k "
// end hand-edit
  " yra interneto enciklopedija, kuri\xB1"" skaityti, tobulinti ir pildyti "
  "gali visi \xBE""mon\xEC""s, be joki\xF9"" apribojim\xF9"" ar mokes\xE8"""
  "i\xF9"". Vikipedij\xB1"" galima skaityti daugiau nei \xB9""imtu kalb\xF9"""
  "; lietuvi\xB9""kuosiuose puslapiuose apra\xB9""ytos daugiau nei 30 t\xFE"""
  "kstan\xE8""i\xF9"" tem\xF9"", skai\xE8""ius nuolat auga. Daugiau informa"
  "cijos apie Vikipedij\xB1"" Spalio 12 dienos \xE7""vykiai Lietuvoje infor"
  "macija ruo\xB9""iama Pasaulyje informacija ruo\xB9""iama Keisti M\xEC""n"
  "esio \xB9""alis \xA9""io m\xEC""nesio \xB9""alies projektas skirtas \xA9"""
  "iaur\xEC""s Kor\xEC""jai ir Tad\xBE""ikistanui. Labai pra\xB9""ome prisi"
  "d\xEC""ti prie straipsni\xF9"" apie \xB9""iuos kra\xB9""tus vystymo. Kei"
  "sti Savait\xEC""s iniciatyva \xA9""iaur\xEC""s karas \xE7""vyko tarp Rus"
  "ijos, Danijos-Norvegijos ir Saksonijos-Abiej\xF9"" Taut\xF9"" Respubliko"
  "s unijos koalicijos (nuo 1715 m. prie jos prisijung\xEC"" Pr\xFE""sija i"
  "r Hanoveris) ir \xA9""vedijos nuo 1700 iki 1721 m. Karo pasekm\xEC""s - "
  "\xA9""vedijos \xE7""taka Baltijos j\xFE""roje smarkiai suma\xBE""\xEC""j"
  "o, \xB9""vedams praradarus teritorijas j\xFE""ros pakrant\xEC""je (Livon"
  "ijoje, Estijoje, Vokietijoje), tuo tarpu Rusija tapo viena i\xB9"" did\xBE"""
  "i\xF9""j\xF9"" Europos valstybi\xF9"". 1721 m. \xA9""vedija buvo privers"
  "ta pasira\xB9""yti taik\xB1"", kuria neteko vis\xF9"" savo va"
;

const char* kTeststr04 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/av_wikipedia_org_clean__ISO-8859-5.txt
  "ed for the creation of a Wikipedia in the Avar (\xB0""\xD2""\xD0""\xE0"""
  ") language. There are currently 150 articles (mostly stubs). \xB2""\xE1"""
  "\xD5"" \xE1""\xE2""\xE0""\xD0""\xDD""\xD8""\xE6""\xEB"" / All pages \xB4"""
  "\xD0""\xD3""\xEA""\xD8""\xE1""\xE2""\xD0""\xDD"" \xDC""\xD0""\xD3""I\xD0"""
  "\xE0""\xE3""\xDB"" \xDC""\xD0""\xE6""I \xDC""\xD0""\xD3""I\xD0""\xE0""\xE3"""
  "\xDB""\xD0""\xDB"" \xB1""\xE3""\xD9""\xDD""\xD0""\xE5""\xEA""\xE1""\xDA"""
  " \xC1""\xE3""\xDB""\xD0""\xE5""\xEA"" \xC6""I\xDE""\xD1""\xDE""\xDA""\xEC"""
  " \xB1""\xD0""\xE2""\xD0""\xE1""\xE3""\xDD""\xD0"" \xB0""\xD7""\xD5""\xE0"""
  "\xD1""\xD0""\xD9""\xD4""\xD6""\xD0""\xDD"" \xB1""\xD0""\xDA""\xE3"" \xC0"""
  "\xDE""\xE1""\xE1""\xD8""\xEF""\xDB""\xEA""\xE3""\xDB"" \xC4""\xD5""\xD4"""
  "\xD5""\xE0""\xD0""\xE6""\xD8""\xEF"" \xD3""I\xE3""\xE0""\xE3""\xE1"" \xDC"""
  "\xD0""\xE6""I \xBD""\xE3""\xD6""\xD5""\xD4""\xD0""\xD3""\xD8"" \xDA""I\xDE"""
  "\xDB""\xD0"" \xDC""\xD0""\xD3""I\xD0""\xE0""\xE3""\xDB"" \xDC""\xD0""\xE6"""
  "I\xD0""\xDB""\xD4""\xD0"" \xB2""\xD8""\xDA""\xD8""\xDF""\xD5""\xD4""\xD8"""
  "\xEF"" \xE6""\xD5""\xD1""\xD5""\xE2""I\xD5""\xD7""\xD0""\xD1""\xD8""\xEF"""
  "\xDB""\xD4""\xD0"" \xD3""\xEA""\xDE""\xE0""\xDB""\xEA"" \xD3""I\xD0""\xE5"""
  "\xEC""\xD0""\xDB""\xDB""\xEA""\xD8""\xD7""\xD5""!\xC9""\xD8""\xD1""\xD0"""
  "\xD1"" \xD3""\xEC""\xE3""\xDC""\xD5""\xE0""\xD0""\xDB""\xD4""\xD0"" \xE2"""
  "I\xD0""\xD4"" \xD1""\xE3""\xD3""\xDE"" \"\xE5""\xD8""\xE1""\xD8""\xD7""\xD0"""
  "\xD1""\xD5""\" \xE1""\xE1""\xEB""\xDB""\xDA""\xD0"": \xDC""\xD0""\xDA""\xEA"""
  "\xD0""\xDB""\xD0""\xEF""\xDB""\xD4""\xD0"" \xE2""I\xD0""\xD4""\xD5"" \xD6"""
  "\xDE"" \xD6""\xE3""\xD1""\xD0""\xDB""\xD0"" \xD2""\xD0"" \xE6""I\xE3""\xDD"""
  "\xE3""\xDB""\xD0"", \xB3""\xEC""\xD5""\xDB""\xD4""\xD0""\xDB""\xEA""\xE3"""
  "\xDD"" \xE2""I\xE3""\xD1""\xD0""\xDB""\xD0"". \xB2""\xDE""\xE0""\xD5""\xE5"""
  "\xD0"" \xDD""\xE3""\xD6""\xD5""\xE0""\xD3""\xDE"" \xE6""I\xD0""\xE0"" \xD1"""
  "\xD8""\xE6""\xD8""\xDD""\xD5"" \xDA""I\xDE""\xE7""\xDE""\xDD""\xD3""\xD5"""
  "(\xE1""\xE1""\xEB""\xDB""\xDA""\xD0"" \xD1""\xE3""\xD3""\xDE"" \xE2""I\xD0"""
  "\xD4"" \xDA""\xD2""\xD0""\xE0""\xD0""\xDD""\xD8""\xD4""\xD0"")! \xB3""\xEC"""
  "\xD5""\xD1"" \xE5""I\xD0""\xD6""\xD0""\xDB""\xEA""\xE3""\xDB""\xD0"" \xDD"""
  "\xE3""\xD6""\xD5""\xE0""\xD3""\xE3""\xDD"" \xD1""\xE3""\xE5""\xEC""\xD5"""
  "\xDD"" \xD3""\xEC""\xD0""\xD1""\xD8""\xD7""\xD5""! \xBD""\xE3""\xD6""\xD5"""
  "\xD4""\xD0""\xD3""\xDE"" \xE5""\xEA""\xD2""\xD0""\xD2""\xE3""\xDB"" \xDC"""
  "\xD0""\xD3""I\xD0""\xE0""\xE3""\xDB"" \xDC""\xD0""\xE6""I \xDB""\xEA""\xD0"""
  "\xEF""\xDB""\xD4""\xD0"" \xE8""\xD0""\xDA""\xDB""\xEA""\xD8"" \xD1""\xD0"""
  "\xE2""\xD0""\xDD""\xD8"", \xE5""I\xD0""\xDB""\xD1""\xD8""\xE5""\xEC""\xD5"""
  " \"\xC6""I\xE3""\xDC""\xD0""\xD4""\xD0""\" \xE1""\xD0""\xD9""\xE2""\xD0"""
  "\xDB""\xEA""\xE3""\xDB"" \xDC""\xD0""\xE2""\xD5""\xE0""\xD8""\xD0""\xDB"""
  "\xD0""\xD7""\xD4""\xD0"" \xE2""I\xD0""\xD4"" \xE5""I\xD0""\xDB""\xE2""I\xD8"""
  " \xD3""\xEC""\xD0""\xD1""\xD8""\xD7""\xD5"". If you come across pages th"
  "at shouldn't be here, replace their content with {{db|reasonfordeletion}"
  "} so that when someone who speaks the language \"adopts\" the Avar Wikip"
  "edia t"
;

const char* kTeststr05 =      // aka ISO-8859-6
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/ar_wikipedia_org_clean__Arabic.txt
  " \xC8""\xC7""\xE4""\xE4""\xDA""\xC9"" \xC7""\xE4""\xC2""\xD1""\xC7""\xE5"""
  "\xEA""\xC9"". \xEA""\xD3""\xE8""\xD9"" \xC8""\xC7""\xE4""\xD9""\xC8""\xD1"""
  "\xEA""\xC9"" \xCA""\xE6""\xD7""\xE2"" \xEA""\xD4""\xE8""\xD9"" \xE8""\xE5"""
  "\xD9""\xE6""\xC7""\xE7""\xC7"" \xC7""\xE4""\xCD""\xD1""\xE1""\xEA"" \"\xEA"""
  "\xE7""\xE8""\xE7"" \xD4""\xE8""\xD9""\" \xC3""\xE9"" \"\xC7""\xE4""\xE4"""
  "\xE7"" \xEA""\xCE""\xE4""\xD5""\". \xC7""\xE4""\xE5""\xD3""\xEA""\xCD"" "
  "\xCD""\xD3""\xC8"" \xC7""\xE4""\xE3""\xCA""\xC7""\xC8"" \xC7""\xE4""\xE5"""
  "\xE2""\xCF""\xD3"" \xE7""\xE8"" \xC7""\xC8""\xE6"" \xC7""\xE4""\xE4""\xE7"""
  " . \xE4""\xC7""\xEA""\xC4""\xE5""\xE6"" \xC7""\xE4""\xEA""\xE7""\xE8""\xCF"""
  " \xC8""\xEA""\xD3""\xE8""\xD9"" \xE8""\xEA""\xD1""\xE1""\xD6""\xE8""\xE6"""
  " \xE1""\xE3""\xD1""\xC9"" \xCA""\xC3""\xE4""\xE7""\xE7"" \xE8"" \xC8""\xC3"""
  "\xE6""\xE7"" \xCC""\xD2""\xC1"" \xE5""\xE6"" \xCB""\xC7""\xE4""\xE8""\xCB"""
  " \xC7""\xE4""\xE7""\xEA"" \xAC"" \xE8""\xC7""\xE4""\xEA""\xE7""\xE8""\xCF"""
  "\xEA""\xC9"" \xC3""\xEA""\xD6""\xC7"" \xE4""\xC7""\xCA""\xD9""\xCA""\xD1"""
  "\xE1"" \xC8""\xE3""\xE8""\xE6"" \xEA""\xD3""\xE8""\xD9"" \xE7""\xE8"" \xC7"""
  "\xE4""\xE5""\xD3""\xEA""\xC7"" \xC3""\xE8"" \xC7""\xE4""\xE5""\xD3""\xEA"""
  "\xCD"" \xC7""\xE4""\xE5""\xE6""\xCA""\xD8""\xD1"" \xAC"" \xE4""\xC3""\xE6"""
  "\xE7"" \xE8""\xCD""\xD3""\xC8"" \xC7""\xD9""\xCA""\xE2""\xC7""\xCF""\xE7"""
  "\xE5"" \xE4""\xE5"" \xEA""\xCA""\xE5""\xE5"" \xC7""\xE4""\xE6""\xC8""\xE8"""
  "\xC7""\xCA"" \xC7""\xE4""\xCA""\xEA"" \xCA""\xCD""\xCF""\xCB""\xCA"" \xD9"""
  "\xE6"" \xC7""\xE4""\xE5""\xD3""\xEA""\xCD"" \xE8""\xD9""\xE6"" \xC7""\xE4"""
  "\xD9""\xD5""\xD1"" \xC7""\xE4""\xE5""\xD3""\xEA""\xCD""\xC7""\xE6""\xEA"""
  " \xC7""\xE4""\xD0""\xEA"" \xD3""\xEA""\xCC""\xE4""\xC8""\xE7"" \xE5""\xD9"""
  "\xE7"" . \xE1""\xEA"" \xCD""\xEA""\xE6"" \xEA""\xD5""\xE1"" \xC7""\xE4"""
  "\xE2""\xD1""\xC2""\xE6"" \xD9""\xEA""\xD3""\xE9"" \xC8""\xC3""\xE6""\xE7"""
  " \xE3""\xE4""\xE5""\xC9"" \xC7""\xE4""\xE4""\xE7"" \xC7""\xE4""\xCA""\xEA"""
  " \xC3""\xE4""\xE2""\xC7""\xE7""\xC7"" \xC5""\xE4""\xE9"" \xE5""\xD1""\xEA"""
  "\xE5"" \xC8""\xE6""\xCA"" \xD9""\xE5""\xD1""\xC7""\xE6"". \xEA""\xD0""\xE3"""
  "\xD1"" \xC7""\xE4""\xE2""\xD1""\xC7""\xE6"" \xC3""\xE6"" \xD9""\xEA""\xD3"""
  "\xE9"" \xC8""\xD4""\xD1"" \xE3""\xE3""\xE4"" \xC7""\xE4""\xC8""\xD4""\xD1"""
  " \xE8""\xC3""\xE6"" \xC7""\xE4""\xE4""\xE7"" \xCE""\xE4""\xE2""\xE7"" \xE3"""
  "\xE5""\xC7"" \xCE""\xE4""\xE2"" \xC2""\xCF""\xE5"" \xE5""\xE6"" \xCA""\xD1"""
  "\xC7""\xC8"" \xCB""\xE5"" \xE2""\xC7""\xE4"" \xE4""\xE7"" \xE3""\xE6"" \xE1"""
  "\xEA""\xE3""\xE8""\xE6""\xAC"" \xE8"" \xC3""\xE5""\xE7"" \xD5""\xCF""\xEA"""
  "\xE2""\xC9"" \xE8"" \xE7""\xEA"" \xC7""\xE4""\xD3""\xEA""\xCF""\xC9"" \xE5"""
  "\xD1""\xEA""\xE5"" \xC7""\xE4""\xD9""\xD0""\xD1""\xC7""\xC1"" \xC7""\xE4"""
  "\xCA""\xEA"" \xC7""\xCE""\xCA""\xC7""\xD1""\xE7""\xC7"" \xC7""\xE4""\xE4"""
  "\xE7"" \xE4""\xE5""\xD9""\xCC""\xD2""\xCA""\xE7"" \xC8""\xE8""\xE4""\xC7"""
  "\xCF""\xC9"" \xD9""\xEA""\xD3""\xE9"". \xE4""\xE4""\xE5""\xD2""\xEA""\xCF"""
  " | \xC7""\xE4""\xE5""\xE2""\xC7""\xE4"" \xC7""\xE4""\xE5""\xCE""\xCA""\xC7"""
  "\xD1"" \xC7""\xE4""\xD3""\xC7""\xC8""\xE2"" | \xE2""\xC7""\xC6""\xE5""\xC9"""
  " \xE5""\xE2""\xC7""\xE4""\xC7""\xCA"" \xE5""\xCE""\xCA""\xC7""\xD1""\xC9"""
  " \xE5""\xE2""\xC7""\xE4""\xC9"" \xC7""\xE4""\xC3""\xD3""\xC8""\xE8""\xD9"""
  " \xC8""\xEA""\xEA""\xD1"" \xCF""\xEA"" \xE3""\xE8""\xC8""\xEA""\xD1""\xCA"""
  "\xC7""\xE6"" (1 \xEA""\xE6""\xC7""\xEA""\xD1"" 1863 - 2 \xD3""\xC8""\xCA"""
  "\xE5""\xC8""\xD1"" 1937) \xEA""\xD9""\xD1""\xE1"" \xC3""\xEA""\xD6""\xC7"""
  " \xC8""\xE0""(\xC8""\xEA""\xEA""\xD1"" \xCF""\xEA"" \xE1""\xD1""\xEA""\xCF"""
  "\xEA"") \xE8"" \xE7""\xE8"" \xE5""\xE3""\xCA""\xD4""\xE1"" \xC7""\xE4""\xC3"""
  "\xE4""\xD9""\xC7""\xC8"" \xC7""\xE4""\xC3""\xE8""\xE4""\xE5""\xC8""\xEA"""
  "\xC9"" \xC7""\xE4""\xCD""\xCF""\xEA""\xCB""\xC9"". \xCF""\xD1""\xD3"" \xC7"""
  "\xE4""\xE5""\xCD""\xC7""\xE5""\xC7""\xC9"" \xCB""\xE5"" \xCA""\xD1""\xE3"""
  "\xE7""\xC7"" \xE8""\xCF""\xCE""\xE4"" \xC7""\xE4""\xE5""\xE4""\xE3""\xEA"""
  "\xC9"" \xC7""\xE4""\xD9""\xD3""\xE3""\xD1""\xEA""\xC9"" \xE1""\xEA"" (\xC7"""
  "\xE4""\xD3""\xC7""\xE6""\xD3""\xEA""\xD1"") \xE8""\xE5""\xE6""\xE7""\xC7"""
  " \xCA""\xCD""\xE8""\xE4"" \xC5""\xE4""\xE9"" \xC7""\xE4""\xCA""\xD1""\xC8"""
  "\xEA""\xC9"" \xE8""\xC7""\xE4""\xD1""\xEA""\xC7""\xD6""\xC9"" , \xE8""\xE5"""
  "\xD9"" \xD9""\xD4""\xE2""\xE7"" \xE4""\xE4""\xCA""\xC7""\xD1""\xEA""\xCE"""
  " \xC7""\xE4""\xEA""\xE8""\xE6""\xC7""\xE6""\xEA"" \xC7""\xE4""\xE2""\xCF"""
  "\xEA""\xE5"" \xE6""\xC7""\xCF""\xE9"" \xE1""\xEA"" 25 \xE6""\xE8""\xE1"""
  "\xE5""\xC8""\xD1"" 1892 \xC3""\xEA"" \xD9""\xE6""\xCF""\xE5""\xC7"" \xE3"""
  "\xC7""\xE6"" \xE1""\xEA"" \xD3""\xE6"" \xC7""\xE4""\xCA""\xC7""\xD3""\xD9"""
  "\xC9"" \xE8""\xC7""\xE4""\xD9""\xD4""\xD1""\xEA""\xE6"" \xC8""\xC5""\xCD"""
  "\xEA""\xC7""\xC1"" \xC7""\xE4""\xC3""\xE4""\xD9""\xC7""\xC8"" \xC7""\xE4"""
  "\xC3""\xE8""\xE4""\xE5""\xC8""\xEA""\xC9"" \xC7""\xE4""\xE2""\xCF""\xEA"""
  "\xE5""\xC9"" \xC7""\xE4""\xCA""\xEA"" \xC3""\xD7""\xE4""\xE2""\xE7""\xC7"""
  " (\xC5""\xEA""\xE1""\xEA""\xE6""\xE8""\xD3"") \xE5""\xE4""\xE3"" \xE4""\xEA"""
  "\xCF""\xEA""\xC7"" \xD9""\xC7""\xE5"" 776 \xE2""\xC8""\xE4"" \xC7""\xE4"""
  "\xE5""\xEA""\xE4""\xC7""\xCF"" \xE1""\xEA"" \xC3""\xE8""\xE4""\xE5""\xC8"""
  "\xEA""\xC7"" \xD9""\xE4""\xE9"" \xE5""\xD4""\xC7""\xD1""\xE1"" \xC3""\xCB"""
  "\xEA""\xE6""\xC7"" \xC8""\xD9""\xCF""\xE5""\xC7"" \xC7""\xD3""\xCA""\xD4"""
  "\xC7""\xD1"" \xC7""\xE4""\xC2""\xE4""\xE7""\xC9"" \xE8""\xE4""\xCA""\xE3"""
  "\xE8""\xE6"" \xE7""\xCF""\xE6""\xC9"" \xE4""\xE4""\xCD""\xD1""\xE8""\xC8"""
  " \xC7""\xE4""\xE5""\xD3""\xCA""\xE5""\xD1""\xC9"" \xC8""\xEA""\xE6"" \xC8"""
  "\xE4""\xC7""\xCF"" \xC7""\xE4""\xC5""\xDA""\xD1""\xEA""\xE2"" (\xC7""\xE4"""
  "\xEA""\xE8""\xE6""\xC7""\xE6""\xEA""\xE8""\xE6"" \xC7""\xE4""\xE2""\xCF"""
  "\xE5""\xC7""\xC1""). \xCF""\xEA"" \xE3""\xE8""\xC8""\xEA""\xD1""\xCA""\xC7"""
  "\xE6"" | \xE5""\xCE""\xCA""\xC7""\xD1""\xC7""\xCA"" \xC7""\xE4""\xC3""\xD3"""
  "\xC7""\xC8""\xEA""\xD9"" \xC7""\xE4""\xD3""\xC7""\xC8""\xE2""\xC9"" \xE5"""
  "\xCC""\xCA""\xE5""\xD9"" \xE8""\xEA""\xE3""\xEA""\xC8""\xEA""\xCF""\xEA"""
  "\xC7"" \xE8""\xEA""\xE3""\xEA""\xC8""\xEA""\xCF""\xEA""\xC7"" \xE5""\xD4"""
  "\xD1""\xE8""\xD9"" \xEA""\xD9""\xCA""\xE5""\xCF"" \xD9""\xE4""\xE9"" \xE6"""
  "\xD5""\xE8""\xD5"" \xCD""\xD1""\xC9"" \xC7""\xE4""\xCA""\xD9""\xCF""\xEA"""
  "\xE4"" \xE5""\xCA""\xC7""\xCD""\xC9"" \xE4""\xE4""\xCC""\xE5""\xEA""\xD9"""
  "\xAC"" \xD1""\xCC""\xC7""\xC1""\xEB"" \xE4""\xC7"" \xCA""\xE6""\xD3""\xCE"""
  " \xC3""\xEA"" \xE6""\xD5"" \xE5""\xE8""\xCC""\xE8""\xCF"" \xD9""\xE4""\xE9"""
  " \xC7""\xE4""\xE8""\xEA""\xC8"" \xC3""\xE8"" \xC3""\xEA"" \xD9""\xE5""\xE4"""
  " \xD0""\xEA"" \xCD""\xE2""\xE8""\xE2"" \xCA""\xC3""\xE4""\xEA""\xE1"" \xE8"""
  "\xE6""\xD4""\xD1"" \xE5""\xCD""\xE1""\xE8""\xD8""\xC9"". \xC5""\xD0""\xC7"""
  " \xE3""\xE6""\xCA"" \xCA""\xD1""\xDA""\xC8"" \xC8""\xC7""\xE4""\xE5""\xD3"""
  "\xC7""\xD9""\xCF""\xC9""\xAC"" \xC7""\xE4""\xE2""\xF0"" \xE6""\xD8""\xD1"""
  "\xC9"" \xD9""\xE4""\xE9"" \xE5""\xD4""\xD1""\xE8""\xD9"" \xEA""\xE8""\xE5"""
  " \xE8""\xEA""\xE3""\xEA""\xC8""\xEA""\xCF""\xEA""\xC7"" \xC7""\xE4""\xD9"""
  "\xD1""\xC8""\xEA""\xC9""\xAC"" \xE4""\xD7""\xE4""\xC8"" \xE5""\xD3""\xC7"""
  "\xD9""\xCF""\xC9"" \xE5""\xE6"" \xC7""\xE4""\xE5""\xD4""\xC7""\xD1""\xE3"""
  "\xEA""\xE6"" \xE4""\xC7"" \xCA""\xCA""\xD1""\xCF""\xCF""\xE8""\xC7"" \xC8"""
  "\xC7""\xE4""\xCA""\xE8""\xCC""\xE7"" \xC5""\xE4""\xE9"" \xE5""\xEA""\xCF"""
  "\xC7""\xE6"" \xE8""\xEA""\xE3""\xEA""\xC8""\xEA""\xCF""\xEA""\xC7"". \xE4"""
  "\xE5""\xD9""\xD1""\xE1""\xC9"" \xC2""\xCE""\xD1"" \xC3""\xCE""\xC8""\xC7"""
  "\xD1"" \xC7""\xE4""\xE5""\xE8""\xD3""\xE8""\xD9""\xC9""\xAC"" \xC7""\xE4"""
  "\xD1""\xCC""\xC7""\xC1"" \xC7""\xE4""\xD0""\xE7""\xC7""\xC8"" \xC5""\xE4"""
  "\xE9"" \xC8""\xE8""\xC7""\xC8""\xC9"" \xC7""\xE4""\xE5""\xCC""\xCA""\xE5"""
  "\xD9"". \xC5""\xD0""\xC7"" \xE3""\xC7""\xE6"" \xE4""\xCF""\xEA""\xE3"" \xD3"""
  "\xC4""\xC7""\xE4""\xC7""\xEB"" \xE4""\xE5"" \xCA""\xCC""\xCF"" \xCC""\xE8"""
  "\xC7""\xC8""\xE7"" \xE1""\xEA"" \xE8""\xEA""\xE3""\xEA""\xC8""\xEA""\xCF"""
  "\xEA""\xC7""\xAC"" \xE4""\xC7"" \xCA""\xCA""\xD1""\xCF""\xCF"" \xE1""\xEA"""
  " \xE3""\xCA""\xC7""\xC8""\xC9"" \xD3""\xC4""\xC7""\xE4""\xE3"" \xE7""\xE6"""
  "\xC7"". \xE8""\xEA""\xE3""\xEA""\xC8""\xEA""\xCF""\xEA""\xC7"" \xE4""\xEA"""
  "\xD3""\xCA"" \xE4""\xE6""\xD4""\xD1"" \xC7""\xE4""\xD9""\xE2""\xC7""\xC6"""
  "\xCF"" \xE8""\xC7""\xE4""\xE5""\xD9""\xCA""\xE2""\xCF""\xC7""\xCA""\xAC"""
  " \xE8""\xC7""\xE4""\xC2""\xD1""\xC7""\xC1"" \xC7""\xE4""\xD4""\xCE""\xD5"""
  "\xEA""\xC9""\xAC"" \xE8""\xC7""\xE4""\xD3""\xEA""\xD1"" \xC7""\xE4""\xD0"""
  "\xC7""\xCA""\xEA""\xC9""\xAC"" \xE8""\xD4""\xCC""\xD1""\xC7""\xCA"" \xC7"""
  "\xE4""\xD9""\xC7""\xC6""\xE4""\xC9"" \xE8""\xC7""\xE4""\xC3""\xE6""\xD3"""
  "\xC7""\xC8"". \xC7""\xE2""\xD1""\xC3"" \xE8""\xEA""\xE3""\xEA""\xC8""\xEA"""
  "\xCF""\xEA""\xC7"" \xE4""\xEA""\xD3""\xCA"". \xE4""\xC7"" \xCA""\xE6""\xD4"""
  "\xC6"" \xE5""\xE2""\xC7""\xE4""\xC7""\xEB"" \xE5""\xCA""\xCD""\xEA""\xD2"""
  "\xC7""\xEB"" \xD9""\xE6"" \xE6""\xD3""\xC8"" \xC3""\xE8"" \xD9""\xC7""\xC6"""
  "\xE4""\xC9"" \xC3""\xE8"" \xE2""\xC8""\xEA""\xE4""\xC9""\xAC"" \xD0""\xE3"""
  "\xD1"" \xC7""\xE4""\xD9""\xC7""\xC6""\xE4""\xC7""\xCA"" \xE8""\xC7""\xE4"""
  "\xE2""\xC8""\xC7""\xC6""\xE4"" \xEA""\xCA""\xE5"" \xE1""\xE2""\xD7"" \xD6"""
  "\xE5""\xE6"" \xD3""\xEA""\xC7""\xE2"" \xCA""\xC7""\xD1""\xEA""\xCE""\xEA"""
  " \xC3""\xE8"" \xD3""\xEA""\xC7""\xD3""\xEA"" \xE5""\xD9""\xEA""\xE6"". \xCD"""
  "\xE8""\xE4"" \xC7""\xE4""\xE5""\xD4""\xD1""\xE8""\xD9"": \xCD""\xE8""\xE4"""
  " \xE8""\xEA""\xE3""\xEA""\xC8""\xEA""\xCF""\xEA""\xC7"" - \xD3""\xEA""\xC7"""
  "\xD3""\xC7""\xCA"" \xE8""\xCA""\xD9""\xE4""\xEA""\xE5""\xC7""\xCA"" \xE8"""
  "\xEA""\xE3""\xEA""\xC8""\xEA""\xCF""\xEA""\xC7"" - \xC3""\xD3""\xC6""\xE4"""
  "\xC9"" \xE5""\xCA""\xE3""\xD1""\xD1""\xC9"" - \xC7""\xE4""\xE5""\xEA""\xCF"""
  "\xC7""\xE6"" - \xC8""\xC7""\xC8""\xE4"" - \xCD""\xE2""\xE8""\xE2"" \xC7"""
  "\xE4""\xCA""\xC3""\xE4""\xEA""\xE1"" \xE8""\xC7""\xE4""\xE6""\xD4""\xD1"""
  " - \xE3""\xEA""\xE1""\xEA""\xC9"" \xCA""\xCD""\xD1""\xEA""\xD1"" \xC7""\xE4"""
  "\xD5""\xE1""\xCD""\xC7""\xCA"" - \xCA""\xD3""\xE5""\xEA""\xC9"" \xC7""\xE4"""
  "\xE5""\xE8""\xC7""\xD6""\xEA""\xD9"" - \xCA""\xD5""\xE6""\xEA""\xE1"" \xC7"""
  "\xE4""\xE5""\xE8""\xC7""\xD6""\xEA""\xD9"" - \xE5""\xE8""\xC7""\xD6""\xEA"""
  "\xD9"" \xE5""\xE2""\xCA""\xD1""\xCD""\xC9"". \xEA""\xE5""\xE3""\xE6""\xE3"""
  " \xC3""\xE6"" \xCA""\xD3""\xC7""\xD9""\xCF"" \xE1""\xEA"" \xCA""\xCD""\xD3"""
  "\xEA""\xE6"" \xC7""\xE4""\xE5""\xE2""\xC7""\xE4""\xC7""\xCA"" \xC7""\xE4"""
  "\xE2""\xD5""\xEA""\xD1""\xC9"" \xE1""\xEA"" \xE8""\xEA""\xE3""\xEA""\xC8"""
  "\xEA""\xCF""\xEA""\xC7"". \xC7""\xE6""\xD8""\xD1"" \xC3""\xEA""\xD6""\xC7"""
  "\xEB"" \xC7""\xE4""\xCA""\xD5""\xE8""\xEA""\xCA""\xC7""\xCA"" \xE8""\xC7"""
  "\xE4""\xE5""\xD4""\xC7""\xD1""\xEA""\xD9"" \xC7""\xE4""\xE5""\xD7""\xD1"""
  "\xE8""\xCD""\xC9"". \xCA""\xD5""\xE8""\xEA""\xCA""\xC7""\xCA"" \xCC""\xC7"""
  "\xD1""\xEA""\xC9"" \xD3""\xC7""\xE7""\xE5"" \xC8""\xD5""\xE8""\xCA""\xE3"""
  " \xC3""\xE8"" \xD1""\xC3""\xEA""\xE3"" \xE1""\xEA"": \xE6""\xE2""\xC7""\xD4"""
  " \xCD""\xE8""\xE4"" \xE2""\xE8""\xC7""\xE4""\xC8"" \xC7""\xE4""\xC3""\xCF"""
  "\xEA""\xC7""\xE6"". \xCA""\xD5""\xE8""\xEA""\xCA"" \xE4""\xE4""\xCD""\xD0"""
  "\xE1"". \xCA""\xD1""\xD4""\xEA""\xCD""\xC7""\xCA"" \xC7""\xE4""\xE5""\xE2"""
  "\xC7""\xE4""\xC9"" \xC7""\xE4""\xE5""\xCE""\xCA""\xC7""\xD1""\xC9"". \xCA"""
  "\xD9""\xC7""\xE8""\xE6"" \xC7""\xE4""\xC3""\xD3""\xC8""\xE8""\xD9"". \xCA"""
  "\xD9""\xC7""\xE8""\xE6"" \xC7""\xE4""\xC3""\xD3""\xC8""\xE8""\xD9"" \xE4"""
  "\xE3""\xD1""\xC9"""
;

const char* kTeststr06 =      // aka ISO-8859-7 Greek
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/el_wikipedia_org_clean__CP1253.txt
  " \xE3""\xE9""\xE1"" \xF4""\xE7""\xED"" \xF0""\xE5""\xF1""\xE9""\xE3""\xF1"""
  "\xE1""\xF6""\xDE"" \xE3""\xE5""\xF9""\xE3""\xF1""\xE1""\xF6""\xE9""\xEA"""
  "\xFE""\xED"", \xF0""\xEF""\xEB""\xE9""\xF4""\xE9""\xEA""\xFE""\xED"" \xEA"""
  "\xE1""\xE9"" \xE9""\xF3""\xF4""\xEF""\xF1""\xE9""\xEA""\xFE""\xED"" \xF0"""
  "\xE5""\xF1""\xE9""\xEF""\xF7""\xFE""\xED"", \xE3""\xEB""\xF9""\xF3""\xF3"""
  "\xFE""\xED"" \xEA""\xE1""\xE9"" \xEB""\xE1""\xFE""\xED"". \xCF""\xE9"" \xE5"""
  "\xE8""\xED""\xFC""\xF4""\xE7""\xF4""\xE5""\xF2"" \xF0""\xEF""\xF5"" \xEA"""
  "\xE1""\xF4""\xEF""\xE9""\xEA""\xEF""\xFD""\xED"" \xF3""\xF4""\xE7""\xED"""
  " \xF0""\xE5""\xF1""\xE9""\xEF""\xF7""\xDE"" \xF7""\xF1""\xE7""\xF3""\xE9"""
  "\xEC""\xEF""\xF0""\xEF""\xE9""\xEF""\xFD""\xED"" \xE4""\xE9""\xE1""\xF6"""
  "\xEF""\xF1""\xE5""\xF4""\xE9""\xEA""\xDE"" \xEF""\xF1""\xEF""\xEB""\xEF"""
  "\xE3""\xDF""\xE1"" \xE3""\xE9""\xE1"" \xF4""\xE7""\xED"" \xDF""\xE4""\xE9"""
  "\xE1"" \xEF""\xED""\xF4""\xFC""\xF4""\xE7""\xF4""\xE1"", \xE5""\xDF""\xF4"""
  "\xE5"" \xDF""\xE4""\xE9""\xE1"" \xEF""\xF1""\xEF""\xEB""\xEF""\xE3""\xDF"""
  "\xE1"" \xE3""\xE9""\xE1"" \xE4""\xE9""\xE1""\xF6""\xEF""\xF1""\xE5""\xF4"""
  "\xE9""\xEA""\xDD""\xF2"" \xEF""\xED""\xF4""\xFC""\xF4""\xE7""\xF4""\xE5"""
  "\xF2"", \xF0""\xF1""\xDC""\xE3""\xEC""\xE1"" \xF3""\xF5""\xF7""\xED""\xDC"""
  " \xF0""\xE5""\xF1""\xDF""\xF0""\xEB""\xEF""\xEA""\xEF"" \xE3""\xE9""\xE1"""
  " \xF4""\xEF""\xF5""\xF2"" \xF5""\xF0""\xFC""\xEB""\xEF""\xE9""\xF0""\xEF"""
  "\xF5""\xF2"" \xEA""\xE1""\xF4""\xEF""\xDF""\xEA""\xEF""\xF5""\xF2"" \xF4"""
  "\xE7""\xF2"" \xF0""\xE5""\xF1""\xE9""\xEF""\xF7""\xDE""\xF2"" \xEA""\xE1"""
  "\xE9"" \xE1""\xEA""\xFC""\xEC""\xE7"" \xF0""\xE5""\xF1""\xE9""\xF3""\xF3"""
  "\xFC""\xF4""\xE5""\xF1""\xEF"" \xE3""\xE9""\xE1"" \xF4""\xEF""\xF5""\xF2"""
  " \xEE""\xDD""\xED""\xEF""\xF5""\xF2"". (\xF0""\xE5""\xF1""\xE9""\xF3""\xF3"""
  "\xFC""\xF4""\xE5""\xF1""\xE1""...) \xA2""\xEB""\xEB""\xE1"" \xE5""\xF0"""
  "\xE9""\xEB""\xE5""\xE3""\xEC""\xDD""\xED""\xE1"" \xDC""\xF1""\xE8""\xF1"""
  "\xE1""... \xCA""\xFD""\xEA""\xEB""\xEF""\xF2"" \xDC""\xF1""\xE8""\xF1""\xF9"""
  "\xED"" \xCF"" \xEA""\xFD""\xEA""\xEB""\xEF""\xF2"" \xDC""\xF1""\xE8""\xF1"""
  "\xF9""\xED"" \xF4""\xE7""\xF2"" \xC2""\xE9""\xEA""\xE9""\xF0""\xE1""\xDF"""
  "\xE4""\xE5""\xE9""\xE1""\xF2"" \xE5""\xDF""\xED""\xE1""\xE9"" \xEC""\xE9"""
  "\xE1"" \xF3""\xE5""\xE9""\xF1""\xDC"" \xE1""\xF0""\xFC"" \xEB""\xDE""\xEC"""
  "\xEC""\xE1""\xF4""\xE1"" \xF3""\xF5""\xE3""\xEA""\xE5""\xEA""\xF1""\xE9"""
  "\xEC""\xDD""\xED""\xE7""\xF2"" \xE8""\xE5""\xEC""\xE1""\xF4""\xEF""\xEB"""
  "\xEF""\xE3""\xDF""\xE1""\xF2"". \xCF"" \xF0""\xE1""\xF1""\xFE""\xED"" \xEA"""
  "\xFD""\xEA""\xEB""\xEF""\xF2"" \xE5""\xDF""\xED""\xE1""\xE9"" \xE1""\xF6"""
  "\xE9""\xE5""\xF1""\xF9""\xEC""\xDD""\xED""\xEF""\xF2"" \xF3""\xE5"" \xEC"""
  "\xEF""\xF5""\xF3""\xE9""\xEA""\xEF""\xFD""\xF2"" \xF4""\xE7""\xF2"" \xF4"""
  "\xE6""\xE1""\xE6"". \xC5""\xF0""\xE9""\xEB""\xE5""\xE3""\xEC""\xDD""\xED"""
  "\xE1"" \xDC""\xF1""\xE8""\xF1""\xE1"": \xCB""\xEF""\xFD""\xE9""\xF2"" \xA2"""
  "\xF1""\xEC""\xF3""\xF4""\xF1""\xEF""\xED""\xE3""\xEA"", \xCD""\xF4""\xDF"""
  "\xE6""\xE9"" \xC3""\xEA""\xE9""\xEB""\xDD""\xF3""\xF0""\xE9"", \xCD""\xF4"""
  "\xE9""\xEF""\xFD""\xEA"" \xB8""\xEB""\xE9""\xED""\xE3""\xEA""\xF4""\xEF"""
  "\xED"", \xD4""\xF3""\xDC""\xF1""\xEB""\xE9"" \xD0""\xDC""\xF1""\xEA""\xE5"""
  "\xF1"", \xCD""\xDF""\xED""\xE1"" \xD3""\xE9""\xEC""\xFC""\xED"", \xD4""\xE6"""
  "\xEF""\xED"" \xCA""\xEF""\xEB""\xF4""\xF1""\xDD""\xE9""\xED"", \xC5""\xF1"""
  "\xEB"" \xD7""\xDC""\xE9""\xED""\xF2"", \xCA""\xFC""\xEB""\xEC""\xE1""\xED"""
  " \xD7""\xFC""\xEA""\xE9""\xED""\xF2"", \xD6""\xEB""\xDD""\xF4""\xF3""\xE5"""
  "\xF1"" \xD7""\xDD""\xED""\xF4""\xE5""\xF1""\xF3""\xEF""\xED"" \xC7""\xEC"""
  "\xE5""\xF1""\xEF""\xEB""\xFC""\xE3""\xE9""\xEF"" 12 \xCF""\xEA""\xF4""\xF9"""
  "\xE2""\xF1""\xDF""\xEF""\xF5"" \xC5""\xF0""\xE9""\xEB""\xE5""\xE3""\xEC"""
  "\xDD""\xED""\xE1"" \xE3""\xE5""\xE3""\xEF""\xED""\xFC""\xF4""\xE1"": 199"
  "9 - \xD0""\xE5""\xE8""\xE1""\xDF""\xED""\xE5""\xE9"" \xEF"" Wilt Chamber"
  "lain, \xDD""\xED""\xE1""\xF2"" \xE1""\xF0""\xFC"" \xF4""\xEF""\xF5""\xF2"""
  " \xEA""\xE1""\xEB""\xFD""\xF4""\xE5""\xF1""\xEF""\xF5""\xF2"" \xEA""\xE1"""
  "\xEB""\xE1""\xE8""\xEF""\xF3""\xF6""\xE1""\xE9""\xF1""\xE9""\xF3""\xF4"""
  "\xDD""\xF2"" \xF0""\xEF""\xF5"" \xDD""\xE6""\xE7""\xF3""\xE1""\xED"" \xF0"""
  "\xEF""\xF4""\xDD"", \xE1""\xF0""\xFC"" \xEA""\xE1""\xF1""\xE4""\xE9""\xE1"""
  "\xEA""\xDE"" \xF0""\xF1""\xEF""\xF3""\xE2""\xEF""\xEB""\xDE"" \xF3""\xF4"""
  "\xEF"" \xF3""\xF0""\xDF""\xF4""\xE9"" \xF4""\xEF""\xF5"" \xF3""\xF4""\xEF"""
  " Bel Air, California. ..\xD0""\xE5""\xF1""\xE9""\xF3""\xF3""\xFC""\xF4"""
  "\xE5""\xF1""\xE1"" \xC5""\xE3""\xEA""\xF5""\xEA""\xEB""\xEF""\xF0""\xE1"""
  "\xDF""\xE4""\xE5""\xE9""\xE1"" \xC8""\xE5""\xEC""\xE1""\xF4""\xE9""\xEA"""
  "\xFC""\xF2"" \xEA""\xE1""\xF4""\xDC""\xEB""\xEF""\xE3""\xEF""\xF2"" \xD0"""
  "\xEF""\xEB""\xE9""\xF4""\xE9""\xF3""\xEC""\xFC""\xF2"" \xEA""\xE1""\xE9"""
  " \xD4""\xDD""\xF7""\xED""\xE5""\xF2"" \xC6""\xF9""\xE3""\xF1""\xE1""\xF6"""
  "\xE9""\xEA""\xDE"" \xB7"" \xC8""\xDD""\xE1""\xF4""\xF1""\xEF"" \xB7"" \xCA"""
  "\xE9""\xED""\xE7""\xEC""\xE1""\xF4""\xEF""\xE3""\xF1""\xDC""\xF6""\xEF"""
  "\xF2"" \xB7"" \xCB""\xEF""\xE3""\xEF""\xF4""\xE5""\xF7""\xED""\xDF""\xE1"""
  " \xB7"" \xCC""\xEF""\xF5""\xF3""\xE9""\xEA""\xDE"" \xB7"" \xC1""\xF1""\xF7"""
  "\xE9""\xF4""\xE5""\xEA""\xF4""\xEF""\xED""\xE9""\xEA""\xDE"" \xB7"" \xC3"""
  "\xEB""\xF5""\xF0""\xF4""\xE9""\xEA""\xDE"" \xB7"" \xC1""\xE8""\xEB""\xE7"""
  "\xF4""\xE9""\xF3""\xEC""\xFC""\xF2"" \xB7"" \xCC""\xF5""\xE8""\xEF""\xEB"""
  "\xEF""\xE3""\xDF""\xE1"" \xB7"" \xC8""\xF1""\xE7""\xF3""\xEA""\xE5""\xDF"""
  "\xE1"" \xC8""\xE5""\xF4""\xE9""\xEA""\xDD""\xF2"" \xC5""\xF0""\xE9""\xF3"""
  "\xF4""\xDE""\xEC""\xE5""\xF2"" \xC1""\xED""\xE8""\xF1""\xF9""\xF0""\xEF"""
  "\xEB""\xEF""\xE3""\xDF""\xE1"" \xB7"" \xC1""\xF3""\xF4""\xF1""\xEF""\xED"""
  "\xEF""\xEC""\xDF""\xE1"" \xB7"" \xC2""\xE9""\xEF""\xEB""\xEF""\xE3""\xDF"""
  "\xE1"" \xB7"" \xC3""\xE5""\xF9""\xEB""\xEF""\xE3""\xDF""\xE1"" \xB7"" \xC5"""
  "\xF0""\xE9""\xF3""\xF4""\xDE""\xEC""\xE7"" \xF5""\xF0""\xEF""\xEB""\xEF"""
  "\xE3""\xE9""\xF3""\xF4""\xFE""\xED""\xB7"" \xCC""\xE1""\xE8""\xE7""\xEC"""
  "\xE1""\xF4""\xE9""\xEA""\xDC"" \xB7"" \xD4""\xE5""\xF7""\xED""\xEF""\xEB"""
  "\xEF""\xE3""\xDF""\xE1"" \xB7"" \xD6""\xF5""\xF3""\xE9""\xEA""\xDE"" \xB7"""
  " \xD7""\xE7""\xEC""\xE5""\xDF""\xE1"" \xB7"" \xC9""\xE1""\xF4""\xF1""\xE9"""
  "\xEA""\xDE"" \xB7"" \xCC""\xE5""\xF4""\xE5""\xF9""\xF1""\xEF""\xEB""\xEF"""
  "\xE3""\xDF""\xE1"" \xC3""\xE5""\xF9""\xE3""\xF1""\xE1""\xF6""\xDF""\xE1"""
  " \xC1""\xF3""\xDF""\xE1"" \xB7"" \xC1""\xF6""\xF1""\xE9""\xEA""\xDE"" \xB7"""
  " \xC5""\xF5""\xF1""\xFE""\xF0""\xE7"""
;

const char* kTeststr07 =      // aka ISO-8859-8-I Hebrew in logical order
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/he_wikipedia_org_clean__CP1255.txt
  " \xF2""\xF8""\xEB""\xE9""\xED"" \xF2""\xE6""\xF8""\xE4"": \xF2""\xE6""\xF8"""
  "\xE4"" - \xF9""\xE0""\xEC""\xE5""\xFA"" \xE5""\xFA""\xF9""\xE5""\xE1""\xE5"""
  "\xFA"" - \xEB""\xEC"" \xE3""\xF4""\xE9"" \xE4""\xF2""\xE6""\xF8""\xE4"" "
  "- \xE0""\xF8""\xE2""\xE6"" \xE7""\xE5""\xEC"" \xEC""\xF0""\xF1""\xE9""\xE5"""
  "\xF0""\xE5""\xFA"" \xF2""\xF8""\xE9""\xEB""\xE4"" \xEE""\xFA""\xF7""\xF4"""
  "\xFA"" \xE0""\xE9""\xEB""\xE5""\xFA"" \xE5""\xE9""\xF7""\xE9""\xF4""\xE3"""
  "\xE9""\xE4"" \xE4""\xF2""\xE1""\xF8""\xE9""\xFA"" \xEE""\xF8""\xEB""\xE6"""
  "\xFA"" \xE1""\xE9""\xEE""\xE9""\xED"" \xE0""\xEC""\xE4"" \xEE""\xE0""\xEE"""
  "\xF5"" \xE1""\xEE""\xFA""\xF7""\xF4""\xFA"" \xE0""\xE9""\xEB""\xE5""\xFA"""
  " \xE1""\xF0""\xE5""\xF9""\xE0"" \xEC""\xE1""\xF0""\xE5""\xEF"". \xEB""\xEC"""
  " \xE0""\xE7""\xE3"" \xE9""\xEB""\xE5""\xEC"" \xEC""\xF1""\xE9""\xE9""\xF2"""
  ". \xF2""\xE6""\xF8""\xE5"" \xEC""\xE4""\xE1""\xE8""\xE9""\xE7"" \xE0""\xFA"""
  " \xE4""\xF6""\xEC""\xE7""\xFA""\xE4""! \xF2""\xF8""\xEA"" \xEE""\xE5""\xEE"""
  "\xEC""\xF5"" \xE4""\xE4""\xFA""\xF0""\xE2""\xE3""\xE5""\xFA"" \xE4""\xE2"""
  "\xF8""\xEE""\xF0""\xE9""\xFA"" \xEC""\xF0""\xE0""\xF6""\xE9""\xE6""\xED"""
  " \xE4""\xF7""\xE9""\xF4""\xE4"" \xE9""\xE7""\xE9""\xE3""\xE9""\xED"" \xE5"""
  "\xF7""\xE1""\xE5""\xF6""\xE5""\xFA"" \xE1""\xE2""\xF8""\xEE""\xF0""\xE9"""
  "\xE4"" \xE4""\xF0""\xE0""\xF6""\xE9""\xFA"" \xE0""\xF9""\xF8"" \xE4""\xFA"""
  "\xF0""\xE2""\xE3""\xE5"" \xEC""\xEE""\xF9""\xE8""\xF8""\xE5"" \xF9""\xEC"""
  " \xE0""\xE3""\xE5""\xEC""\xF3"" \xE4""\xE9""\xE8""\xEC""\xF8"" \xE5""\xEC"""
  "\xF9""\xEC""\xE8""\xE5""\xEF"" \xE4""\xE0""\xE9""\xEE""\xE9""\xED"" \xF9"""
  "\xEC"" \xE4""\xEE""\xF4""\xEC""\xE2""\xE4"" \xE4""\xF0""\xE0""\xF6""\xE9"""
  "\xFA"" \xE1""\xE9""\xEF"" \xE4""\xF9""\xF0""\xE9""\xED"" 1933 - 1945. \xE7"""
  "\xEC""\xF7"" \xEE""\xE0""\xF0""\xF9""\xE9"" \xE4""\xFA""\xF0""\xE5""\xF2"""
  "\xE4"" \xE4""\xE9""\xE5"" \xEE""\xF2""\xE5""\xF8""\xE1""\xE9""\xED"" \xE1"""
  "\xF0""\xE9""\xF1""\xE9""\xE5""\xF0""\xE5""\xFA"" \xEC""\xE4""\xE3""\xE9"""
  "\xE7"" \xE0""\xFA"" \xE4""\xE9""\xE8""\xEC""\xF8"" \xE5""\xFA""\xF0""\xE5"""
  "\xF2""\xFA""\xE5"" \xEE""\xE4""\xF9""\xEC""\xE8""\xE5""\xEF"", \xF0""\xE9"""
  "\xF1""\xE9""\xE5""\xF0""\xE5""\xFA"" \xF9""\xE4""\xE2""\xE9""\xF2""\xE5"""
  " \xEC""\xF9""\xE9""\xE0"" \xE1""\xF7""\xF9""\xF8"" \xE4""\xF2""\xF9""\xF8"""
  "\xE9""\xED"" \xE1""\xE9""\xE5""\xEC""\xE9"", \xF9""\xE1""\xE5"" \xF0""\xE9"""
  "\xF1""\xFA""\xE4"" \xF7""\xE1""\xE5""\xF6""\xE4"" \xEE""\xE0""\xE5""\xF8"""
  "\xE2""\xF0""\xFA"", \xF9""\xE4""\xFA""\xE1""\xF1""\xF1""\xE4"" \xF2""\xEC"""
  " \xE0""\xF0""\xF9""\xE9"" \xF6""\xE1""\xE0"" \xE1""\xEB""\xE9""\xF8""\xE9"""
  "\xED"" \xE1""\xE5""\xE5""\xF8""\xEE""\xE0""\xEB""\xE8"", \xEC""\xE4""\xFA"""
  "\xF0""\xF7""\xF9"" \xE1""\xE7""\xE9""\xE9"" \xE4""\xE9""\xE8""\xEC""\xF8"""
  ", \xE5""\xEC""\xE4""\xE7""\xEC""\xE9""\xF3"" \xE0""\xFA"" \xF9""\xEC""\xE8"""
  "\xE5""\xEF"" \xE4""\xEE""\xF4""\xEC""\xE2""\xE4"" \xE4""\xF0""\xE0""\xF6"""
  "\xE9""\xFA"" \xE1""\xF9""\xEC""\xE8""\xE5""\xEF"" \xF6""\xE1""\xE0""\xE9"""
  " \xE5""\xE0""\xE6""\xF8""\xE7""\xE9"" \xE1""\xF8""\xE0""\xF9""\xE5"" \xE9"""
  "\xF2""\xEE""\xE3""\xE5"" \xE4""\xE2""\xF0""\xF8""\xEC"" \xE1""\xE3""\xE9"""
  "\xEE""\xE5""\xF1"" \xEC""\xE5""\xE3""\xE5""\xE5""\xE9""\xE2"" \xE1""\xF7"""
  ", \xE5""\xF8""\xE0""\xF9"" \xF2""\xE9""\xF8""\xE9""\xFA"" \xEC""\xE9""\xE9"""
  "\xF4""\xF6""\xE9""\xE2"", \xF7""\xF8""\xEC"" \xF4""\xF8""\xE9""\xE3""\xF8"""
  "\xE9""\xEA"" \xE2""\xF8""\xE3""\xEC""\xF8"". \xEB""\xF9""\xEC""\xE5""\xF0"""
  "\xE5"" \xF9""\xEC"" \xE4""\xF7""\xF9""\xF8"" \xE4""\xE1""\xE9""\xE0"" \xEC"""
  "\xE2""\xEC"" \xEE""\xF2""\xF6""\xF8""\xE9""\xED"" \xE5""\xE4""\xE5""\xF6"""
  "\xE0""\xE5""\xFA"" \xEC""\xE4""\xE5""\xF8""\xE2"", \xE0""\xF9""\xF8"" \xE7"""
  "\xE9""\xF1""\xEC"", \xEC""\xEE""\xF2""\xF9""\xE4"", \xE0""\xFA"" \xE4""\xE4"""
  "\xFA""\xF0""\xE2""\xE3""\xE5""\xFA"" \xE4""\xEE""\xE0""\xE5""\xF8""\xE2"""
  "\xF0""\xFA"" \xEC""\xEE""\xF9""\xE8""\xF8""\xE5"" \xF9""\xEC"" \xE4""\xE9"""
  "\xE8""\xEC""\xF8"", \xF2""\xE3"" \xEC""\xF1""\xE5""\xF3"" \xE9""\xEE""\xE9"""
  " \xE4""\xF8""\xE9""\xE9""\xEA"" \xE4""\xF9""\xEC""\xE9""\xF9""\xE9"", \xEB"""
  "\xFA""\xF9""\xF2""\xE4"" \xE7""\xE5""\xE3""\xF9""\xE9""\xED"" \xEC""\xE0"""
  "\xE7""\xF8"" \xEE""\xEB""\xEF"". \xEC""\xF2""\xF8""\xEA"" \xE4""\xEE""\xEC"""
  "\xE0"" - \xEE""\xE5""\xEE""\xEC""\xF6""\xE9""\xED"" \xF0""\xE5""\xF1""\xF4"""
  "\xE9""\xED"" \xE4""\xE9""\xE3""\xF2""\xFA""? \xE1""\xF9""\xF0""\xFA"" 19"
  "61, \xF9""\xE1""\xE4"" \xE4""\xE9""\xE5"" \xE7""\xE9""\xEC""\xE5""\xF4"""
  "\xE9"" \xEE""\xF9""\xE8""\xF8"" \xE1""\xF1""\xE5""\xF8""\xE9""\xE4"" \xE5"""
  "\xE4""\xE0""\xE9""\xE7""\xE5""\xE3"" \xF2""\xED"" \xEE""\xF6""\xF8""\xE9"""
  "\xED"", \xE4""\xF8""\xF4""\xE5""\xE1""\xEC""\xE9""\xF7""\xE4"" \xE4""\xF2"""
  "\xF8""\xE1""\xE9""\xFA"" \xE4""\xEE""\xE0""\xE5""\xE7""\xE3""\xFA"", \xF4"""
  "\xE5""\xF8""\xF7"", \xF0""\xC6""\xE0""\xC1""\xF1""\xC8""\xF8"" \xE7""\xE0"""
  "\xF4""\xE6"" \xE0""\xEC""-\xE0""\xF1""\xE3"", \xEC""\xE9""\xEE""\xE9""\xED"""
  " \xF0""\xF9""\xE9""\xE0"" \xF1""\xE5""\xF8""\xE9""\xE4"", \xE5""\xF0""\xEB"""
  "\xEC""\xE0"" \xEC""\xEE""\xF9""\xEA"" \xE7""\xE5""\xE3""\xF9"" \xE1""\xEE"""
  "\xF6""\xF8""\xE9""\xED"", \xE5""\xEC""\xE0""\xE7""\xF8"" \xEE""\xEB""\xEF"""
  " \xF0""\xEE""\xEC""\xE8"" \xEC""\xEC""\xE1""\xF0""\xE5""\xEF"", \xE5""\xE2"""
  "\xED"" \xF9""\xED"" \xE4""\xE5""\xF9""\xEC""\xEA"" \xE0""\xEC"" \xE4""\xEB"""
  "\xEC""\xE0"". ... \xF8""\xE5""\xF6""\xE4"" \xEC""\xE3""\xF2""\xFA"" \xF2"""
  "\xE5""\xE3""? \xE4""\xE9""\xE5""\xED"" \xE1""\xE4""\xE9""\xF1""\xE8""\xE5"""
  "\xF8""\xE9""\xE4"" 1492 - \xEB""\xF8""\xE9""\xF1""\xE8""\xE5""\xF4""\xF8"""
  " \xF7""\xE5""\xEC""\xE5""\xEE""\xE1""\xE5""\xF1"" \xEE""\xE2""\xEC""\xE4"""
  " \xE0""\xFA"" \xE9""\xE1""\xF9""\xFA"" \xE0""\xEE""\xF8""\xE9""\xF7""\xE4"""
  " 1537 - \xE1""\xEF"" \xE6""\xEB""\xF8"" \xF0""\xE5""\xEC""\xE3"" \xEC""\xE4"""
  "\xF0""\xF8""\xE9"" \xE4""\xF9""\xEE""\xE9""\xF0""\xE9"" 1899 - \xF4""\xE5"""
  "\xF8""\xF6""\xFA"" \xEE""\xEC""\xE7""\xEE""\xFA"" \xE4""\xE1""\xE5""\xF8"""
  "\xE9""\xED"" \xE4""\xF9""\xF0""\xE9""\xE4"" \xEC""\xE0""\xE7""\xF8"" \xF9"""
  "\xE4""\xE1""\xE5""\xF8""\xE9""\xED"""
;


// This is just the above text brute-force byte reversed.
const char* kTeststr07v =      // aka ISO-8859-8 Hebrew in visual order
// Produced by stringify.cc on 2007-04-04 12:17 from file wikifiles/he_wikipedia_org_clean__visual.txt
  "\xED""\xE9""\xF8""\xE5""\xE1""\xE4""\xF9"" \xF8""\xE7""\xE0""\xEC"" \xE4"""
  "\xE9""\xF0""\xF9""\xE4"" \xED""\xE9""\xF8""\xE5""\xE1""\xE4"" \xFA""\xEE"""
  "\xE7""\xEC""\xEE"" \xFA""\xF6""\xF8""\xE5""\xF4"" - 9981 \xE9""\xF0""\xE9"""
  "\xEE""\xF9""\xE4"" \xE9""\xF8""\xF0""\xE4""\xEC"" \xE3""\xEC""\xE5""\xF0"""
  " \xF8""\xEB""\xE6"" \xEF""\xE1"" - 7351 \xE4""\xF7""\xE9""\xF8""\xEE""\xE0"""
  " \xFA""\xF9""\xE1""\xE9"" \xFA""\xE0"" \xE4""\xEC""\xE2""\xEE"" \xF1""\xE5"""
  "\xE1""\xEE""\xE5""\xEC""\xE5""\xF7"" \xF8""\xF4""\xE5""\xE8""\xF1""\xE9"""
  "\xF8""\xEB"" - 2941 \xE4""\xE9""\xF8""\xE5""\xE8""\xF1""\xE9""\xE4""\xE1"""
  " \xED""\xE5""\xE9""\xE4"" ?\xE3""\xE5""\xF2"" \xFA""\xF2""\xE3""\xEC"" \xE4"""
  "\xF6""\xE5""\xF8"" ... .\xE0""\xEC""\xEB""\xE4"" \xEC""\xE0"" \xEA""\xEC"""
  "\xF9""\xE5""\xE4"" \xED""\xF9"" \xED""\xE2""\xE5"" ,\xEF""\xE5""\xF0""\xE1"""
  "\xEC""\xEC"" \xE8""\xEC""\xEE""\xF0"" \xEF""\xEB""\xEE"" \xF8""\xE7""\xE0"""
  "\xEC""\xE5"" ,\xED""\xE9""\xF8""\xF6""\xEE""\xE1"" \xF9""\xE3""\xE5""\xE7"""
  " \xEA""\xF9""\xEE""\xEC"" \xE0""\xEC""\xEB""\xF0""\xE5"" ,\xE4""\xE9""\xF8"""
  "\xE5""\xF1"" \xE0""\xE9""\xF9""\xF0"" \xED""\xE9""\xEE""\xE9""\xEC"" ,\xE3"""
  "\xF1""\xE0""-\xEC""\xE0"" \xE6""\xF4""\xE0""\xE7"" \xF8""\xC8""\xF1""\xC1"""
  "\xE0""\xC6""\xF0"" ,\xF7""\xF8""\xE5""\xF4"" ,\xFA""\xE3""\xE7""\xE5""\xE0"""
  "\xEE""\xE4"" \xFA""\xE9""\xE1""\xF8""\xF2""\xE4"" \xE4""\xF7""\xE9""\xEC"""
  "\xE1""\xE5""\xF4""\xF8""\xE4"" ,\xED""\xE9""\xF8""\xF6""\xEE"" \xED""\xF2"""
  " \xE3""\xE5""\xE7""\xE9""\xE0""\xE4""\xE5"" \xE4""\xE9""\xF8""\xE5""\xF1"""
  "\xE1"" \xF8""\xE8""\xF9""\xEE"" \xE9""\xF4""\xE5""\xEC""\xE9""\xE7"" \xE5"""
  "\xE9""\xE4"" \xE4""\xE1""\xF9"" ,1691 \xFA""\xF0""\xF9""\xE1"" ?\xFA""\xF2"""
  "\xE3""\xE9""\xE4"" \xED""\xE9""\xF4""\xF1""\xE5""\xF0"" \xED""\xE9""\xF6"""
  "\xEC""\xEE""\xE5""\xEE"" - \xE0""\xEC""\xEE""\xE4"" \xEA""\xF8""\xF2""\xEC"""
  " .\xEF""\xEB""\xEE"" \xF8""\xE7""\xE0""\xEC"" \xED""\xE9""\xF9""\xE3""\xE5"""
  "\xE7"" \xE4""\xF2""\xF9""\xFA""\xEB"" ,\xE9""\xF9""\xE9""\xEC""\xF9""\xE4"""
  " \xEA""\xE9""\xE9""\xF8""\xE4"" \xE9""\xEE""\xE9"" \xF3""\xE5""\xF1""\xEC"""
  " \xE3""\xF2"" ,\xF8""\xEC""\xE8""\xE9""\xE4"" \xEC""\xF9"" \xE5""\xF8""\xE8"""
  "\xF9""\xEE""\xEC"" \xFA""\xF0""\xE2""\xF8""\xE5""\xE0""\xEE""\xE4"" \xFA"""
  "\xE5""\xE3""\xE2""\xF0""\xFA""\xE4""\xE4"" \xFA""\xE0"" ,\xE4""\xF9""\xF2"""
  "\xEE""\xEC"" ,\xEC""\xF1""\xE9""\xE7"" \xF8""\xF9""\xE0"" ,\xE2""\xF8""\xE5"""
  "\xE4""\xEC"" \xFA""\xE5""\xE0""\xF6""\xE5""\xE4""\xE5"" \xED""\xE9""\xF8"""
  "\xF6""\xF2""\xEE"" \xEC""\xE2""\xEC"" \xE0""\xE9""\xE1""\xE4"" \xF8""\xF9"""
  "\xF7""\xE4"" \xEC""\xF9"" \xE5""\xF0""\xE5""\xEC""\xF9""\xEB"" .\xF8""\xEC"""
  "\xE3""\xF8""\xE2"" \xEA""\xE9""\xF8""\xE3""\xE9""\xF8""\xF4"" \xEC""\xF8"""
  "\xF7"" ,\xE2""\xE9""\xF6""\xF4""\xE9""\xE9""\xEC"" \xFA""\xE9""\xF8""\xE9"""
  "\xF2"" \xF9""\xE0""\xF8""\xE5"" ,\xF7""\xE1"" \xE2""\xE9""\xE5""\xE5""\xE3"""
  "\xE5""\xEC"" \xF1""\xE5""\xEE""\xE9""\xE3""\xE1"" \xEC""\xF8""\xF0""\xE2"""
  "\xE4"" \xE5""\xE3""\xEE""\xF2""\xE9"" \xE5""\xF9""\xE0""\xF8""\xE1"" \xE9"""
  "\xE7""\xF8""\xE6""\xE0""\xE5"" \xE9""\xE0""\xE1""\xF6"" \xEF""\xE5""\xE8"""
  "\xEC""\xF9""\xE1"" \xFA""\xE9""\xF6""\xE0""\xF0""\xE4"" \xE4""\xE2""\xEC"""
  "\xF4""\xEE""\xE4"" \xEF""\xE5""\xE8""\xEC""\xF9"" \xFA""\xE0"" \xF3""\xE9"""
  "\xEC""\xE7""\xE4""\xEC""\xE5"" ,\xF8""\xEC""\xE8""\xE9""\xE4"" \xE9""\xE9"""
  "\xE7""\xE1"" \xF9""\xF7""\xF0""\xFA""\xE4""\xEC"" ,\xE8""\xEB""\xE0""\xEE"""
  "\xF8""\xE5""\xE5""\xE1"" \xED""\xE9""\xF8""\xE9""\xEB""\xE1"" \xE0""\xE1"""
  "\xF6"" \xE9""\xF9""\xF0""\xE0"" \xEC""\xF2"" \xE4""\xF1""\xF1""\xE1""\xFA"""
  "\xE4""\xF9"" ,\xFA""\xF0""\xE2""\xF8""\xE5""\xE0""\xEE"" \xE4""\xF6""\xE5"""
  "\xE1""\xF7"" \xE4""\xFA""\xF1""\xE9""\xF0"" \xE5""\xE1""\xF9"" ,\xE9""\xEC"""
  "\xE5""\xE9""\xE1"" \xED""\xE9""\xF8""\xF9""\xF2""\xE4"" \xF8""\xF9""\xF7"""
  "\xE1"" \xE0""\xE9""\xF9""\xEC"" \xE5""\xF2""\xE9""\xE2""\xE4""\xF9"" \xFA"""
  "\xE5""\xF0""\xE5""\xE9""\xF1""\xE9""\xF0"" ,\xEF""\xE5""\xE8""\xEC""\xF9"""
  "\xE4""\xEE"" \xE5""\xFA""\xF2""\xE5""\xF0""\xFA""\xE5"" \xF8""\xEC""\xE8"""
  "\xE9""\xE4"" \xFA""\xE0"" \xE7""\xE9""\xE3""\xE4""\xEC"" \xFA""\xE5""\xF0"""
  "\xE5""\xE9""\xF1""\xE9""\xF0""\xE1"" \xED""\xE9""\xE1""\xF8""\xE5""\xF2"""
  "\xEE"" \xE5""\xE9""\xE4"" \xE4""\xF2""\xE5""\xF0""\xFA""\xE4"" \xE9""\xF9"""
  "\xF0""\xE0""\xEE"" \xF7""\xEC""\xE7"" .5491 - 3391 \xED""\xE9""\xF0""\xF9"""
  "\xE4"" \xEF""\xE9""\xE1"" \xFA""\xE9""\xF6""\xE0""\xF0""\xE4"" \xE4""\xE2"""
  "\xEC""\xF4""\xEE""\xE4"" \xEC""\xF9"" \xED""\xE9""\xEE""\xE9""\xE0""\xE4"""
  " \xEF""\xE5""\xE8""\xEC""\xF9""\xEC""\xE5"" \xF8""\xEC""\xE8""\xE9""\xE4"""
  " \xF3""\xEC""\xE5""\xE3""\xE0"" \xEC""\xF9"" \xE5""\xF8""\xE8""\xF9""\xEE"""
  "\xEC"" \xE5""\xE3""\xE2""\xF0""\xFA""\xE4"" \xF8""\xF9""\xE0"" \xFA""\xE9"""
  "\xF6""\xE0""\xF0""\xE4"" \xE4""\xE9""\xF0""\xEE""\xF8""\xE2""\xE1"" \xFA"""
  "\xE5""\xF6""\xE5""\xE1""\xF7""\xE5"" \xED""\xE9""\xE3""\xE9""\xE7""\xE9"""
  " \xE4""\xF4""\xE9""\xF7""\xE4"" \xED""\xE6""\xE9""\xF6""\xE0""\xF0""\xEC"""
  " \xFA""\xE9""\xF0""\xEE""\xF8""\xE2""\xE4"" \xFA""\xE5""\xE3""\xE2""\xF0"""
  "\xFA""\xE4""\xE4"" \xF5""\xEC""\xEE""\xE5""\xEE"" \xEA""\xF8""\xF2"" !\xE4"""
  "\xFA""\xE7""\xEC""\xF6""\xE4"" \xFA""\xE0"" \xE7""\xE9""\xE8""\xE1""\xE4"""
  "\xEC"" \xE5""\xF8""\xE6""\xF2"" .\xF2""\xE9""\xE9""\xF1""\xEC"" \xEC""\xE5"""
  "\xEB""\xE9"" \xE3""\xE7""\xE0"" \xEC""\xEB"" .\xEF""\xE5""\xF0""\xE1""\xEC"""
  " \xE0""\xF9""\xE5""\xF0""\xE1"" \xFA""\xE5""\xEB""\xE9""\xE0"" \xFA""\xF4"""
  "\xF7""\xFA""\xEE""\xE1"" \xF5""\xEE""\xE0""\xEE"" \xE4""\xEC""\xE0"" \xED"""
  "\xE9""\xEE""\xE9""\xE1"" \xFA""\xE6""\xEB""\xF8""\xEE"" \xFA""\xE9""\xF8"""
  "\xE1""\xF2""\xE4"" \xE4""\xE9""\xE3""\xF4""\xE9""\xF7""\xE9""\xE5"" \xFA"""
  "\xE5""\xEB""\xE9""\xE0"" \xFA""\xF4""\xF7""\xFA""\xEE"" \xE4""\xEB""\xE9"""
  "\xF8""\xF2"" \xFA""\xE5""\xF0""\xE5""\xE9""\xF1""\xF0""\xEC"" \xEC""\xE5"""
  "\xE7"" \xE6""\xE2""\xF8""\xE0"" - \xE4""\xF8""\xE6""\xF2""\xE4"" \xE9""\xF4"""
  "\xE3"" \xEC""\xEB"" - \xFA""\xE5""\xE1""\xE5""\xF9""\xFA""\xE5"" \xFA""\xE5"""
  "\xEC""\xE0""\xF9"" - \xE4""\xF8""\xE6""\xF2"" :\xE4""\xF8""\xE6""\xF2"" "
  "\xED""\xE9""\xEB""\xF8""\xF2"" "
;

const char* kTeststr08 =      // aka ISO-8859-9
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/mus_wikipedia_org_clean__Latin5.txt
  " language and think it would be cool to have your own Encyclopedia then "
  "you can make it. Go ahead. Translate this page and start working on your"
  " Encyclopedia. For more information go to the main website Other wikis M"
  "eta-Wikipedia/Meta-Vicipaedia | Wikitonary/Victionaria | Wikibooks/Vicil"
  "ibraria | Wikiquote/Viciquotas | Wikisource | Wikitravel A\xFE""k Arkada"
  "\xFE""l\xFD""k Dostluk \xDE""iirler sevgili Av Videolar\xFD"" Av K\xF6"""
  "pekleri Sat\xFD""l\xFD""k Av T\xFC""fekleri Bal\xFD""k Av\xFD"" Tav\xFE"""
  "an Av\xFD"" Domuz Av\xFD"" Av Oyunlar\xFD"" A\xFE""k Arkada\xFE""l\xFD"""
  "k Dostluk \xDE""iirler sevgili Retrieved from \"http://mus.wikipedia.org"
  "/wiki/Main_Page\" Views Article Discussion Edit History Personal tools L"
  "og in / create account if (window.isMSIE55) fixalpha(); Navigation Main "
  "Page Community portal Current events Recent changes Random page Help Don"
  "ations Search Toolbo"
;


const char* kTeststr10 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/ja_wikipedia_org_clean__EUC-JP.txt
  " \xCB""\xCC""\xB3""\xA4""\xC6""\xBB""\xC6""\xFC""\xCB""\xDC""\xA5""\xCF"""
  "\xA5""\xE0""\xA5""\xD5""\xA5""\xA1""\xA5""\xA4""\xA5""\xBF""\xA1""\xBC"""
  "\xA5""\xBA""\xA4""\xAC""\xA5""\xBD""\xA5""\xD5""\xA5""\xC8""\xA5""\xD0"""
  "\xA5""\xF3""\xA5""\xAF""\xA4""\xC8""\xA4""\xCE""\xA5""\xD7""\xA5""\xEC"""
  "\xA1""\xBC""\xA5""\xAA""\xA5""\xD5""\xA4""\xF2""\xC0""\xA9""\xA4""\xB7"""
  "\xA1""\xA2""25\xC7""\xAF""\xA4""\xD6""\xA4""\xEA""3\xC5""\xD9""\xCC""\xDC"""
  "\xA4""\xCE""\xA5""\xD1""\xA5""\xB7""\xA5""\xD5""\xA5""\xA3""\xA5""\xC3"""
  "\xA5""\xAF""\xA1""\xA6""\xA5""\xEA""\xA1""\xBC""\xA5""\xB0""\xCD""\xA5"""
  "\xBE""\xA1""\xA1""\xA3"" \xCB""\xCC""\xC4""\xAB""\xC1""\xAF""\xB3""\xCB"""
  "\xBC""\xC2""\xB8""\xB3""\xA4""\xCB""\xC2""\xD0""\xA4""\xB9""\xA4""\xEB"""
  "\xB7""\xD0""\xBA""\xD1""\xC0""\xA9""\xBA""\xDB""\xA4""\xC7""\xC6""\xFC"""
  "\xCB""\xDC""\xC6""\xC8""\xBC""\xAB""\xA4""\xCE""\xC4""\xC9""\xB2""\xC3"""
  "\xC0""\xA9""\xBA""\xDB""\xA4""\xF2""\xB7""\xE8""\xC4""\xEA""\xA1""\xA3"""
  " \xA5""\xC7""\xA5""\xA3""\xA1""\xBC""\xA5""\xD7""\xA5""\xA4""\xA5""\xF3"""
  "\xA5""\xD1""\xA5""\xAF""\xA5""\xC8""\xA4""\xAC""\xC7""\xAF""\xC6""\xE2"""
  "\xA4""\xC7""\xB0""\xFA""\xC2""\xE0""\xA4""\xB9""\xA4""\xEB""\xA4""\xC8"""
  "\xC8""\xAF""\xC9""\xBD""\xA1""\xA3""\xA2""\xAA""\xA5""\xA6""\xA5""\xA3"""
  "\xA5""\xAD""\xA5""\xCB""\xA5""\xE5""\xA1""\xBC""\xA5""\xB9"" Google\xA4"""
  "\xAC""Google Docs & Spreadsheets\xA1""\xCA""Google\xC8""\xC7""\xA1""\xD6"""
  "Office\xA1""\xD7""\xA1""\xA2""\xA5""\xD9""\xA1""\xBC""\xA5""\xBF""\xC8"""
  "\xC7""\xA1""\xCB""\xA4""\xCE""\xC4""\xF3""\xB6""\xA1""\xA4""\xF2""\xB3"""
  "\xAB""\xBB""\xCF""\xA1""\xA3"" [\xA5""\xA6""\xA5""\xA3""\xA5""\xAD""\xA5"""
  "\xCB""\xA5""\xE5""\xA1""\xBC""\xA5""\xB9""] [\xBA""\xC7""\xB6""\xE1""\xA4"""
  "\xCE""\xBD""\xD0""\xCD""\xE8""\xBB""\xF6""\xC1""\xB4""\xCA""\xB8""] \xC5"""
  "\xEA""\xC9""\xBC""\xA1""\xA6""\xBF""\xE4""\xC1""\xA6"" \xB5""\xA8""\xC0"""
  "\xE1""\xA4""\xCE""\xCF""\xC3""\xC2""\xEA"" \xBD""\xA9"" - \xB1""\xBF""\xC6"""
  "\xB0""\xB2""\xF1"" - \xB0""\xF2""\xBC""\xD1""\xB2""\xF1"" - \xA5""\xB5"""
  "\xA5""\xF3""\xA5""\xDE"" - \xA5""\xD6""\xA5""\xC9""\xA5""\xA6"" - \xA5"""
  "\xAD""\xA5""\xF3""\xA5""\xE2""\xA5""\xAF""\xA5""\xBB""\xA5""\xA4"" - \xA4"""
  "\xC9""\xA4""\xF3""\xA4""\xB0""\xA4""\xEA"" - \xBD""\xE9""\xB4""\xA7""\xC0"""
  "\xE3"" - \xBD""\xA9""\xA4""\xCE""\xCD""\xBC""\xBE""\xC6""\xA4""\xB1""(*)"
  " - \xBE""\xBE""\xC2""\xFB"" \xBA""\xA3""\xC6""\xFC""\xA4""\xCE""\xA4""\xB3"""
  "\xA4""\xE8""\xA4""\xDF"" \xB5""\xEC""\xCE""\xF1"":8\xB7""\xEE""22\xC6""\xFC"""
  " - \xC2""\xE7""\xB0""\xC2"" - \xC6""\xFC""\xA4""\xCE""\xB4""\xB3""\xBB"""
  "\xD9"":\xB2""\xB5""\xB0""\xE7"" - \xB6""\xE5""\xC0""\xB1"":\xBB""\xCD""\xCE"""
  "\xD0""\xCC""\xDA""\xC0""\xB1"" - \xC6""\xF3""\xBD""\xBD""\xC8""\xAC""\xBD"""
  "\xC9"":\xD0""\xB6""\xBD""\xC9"" - \xBD""\xBD""\xC6""\xF3""\xC4""\xBE"":\xBD"""
  "\xFC"" - \xC0""\xE1""\xB5""\xA4"":\xB4""\xA8""\xCF""\xAA""\xA1""\xA2""\xC1"""
  "\xFA""\xB9""\xDF""\xA4""\xDE""\xA4""\xC7""10\xC6""\xFC"" - \xC1""\xAA""\xC6"""
  "\xFC"":\xC2""\xE7""\xC8""\xC8""\xC5""\xDA"" \xBA""\xA3""\xC6""\xFC""\xA4"""
  "\xCF""\xB2""\xBF""\xA4""\xCE""\xC6""\xFC"" \xA5""\xB0""\xA5""\xEA""\xA5"""
  "\xCB""\xA5""\xC3""\xA5""\xB8""\xC5""\xB7""\xCA""\xB8""\xC2""\xE6""\xA4"""
  "\xF2""\xC4""\xCC""\xA4""\xEB""\xB7""\xD0""\xC0""\xFE""\xA4""\xF2""0\xC5"""
  "\xD9""\xA4""\xC8""\xC4""\xEA""\xA4""\xE1""\xA4""\xEB""\xA1""\xCA""1884\xC7"""
  "\xAF""\xA1""\xCB"" 11\xB5""\xDC""\xB2""\xC8""51\xBF""\xCD""\xA4""\xCE""\xB9"""
  "\xC4""\xC2""\xB2""\xCE""\xA5""\xC3""\xA6""\xA4""\xAC""\xB7""\xE8""\xC4"""
  "\xEA""\xA1""\xCA""1947\xC7""\xAF""\xA1""\xCB"" \xA5""\xD6""\xA5""\xE9""\xA5"""
  "\xC3""\xA5""\xAF""\xA1""\xA6""\xA5""\xC1""\xA5""\xE5""\xA1""\xBC""\xA5"""
  "\xBA""\xA5""\xC7""\xA1""\xBC""\xA1""\xCA""1989\xC7""\xAF""\xA1""\xCB"" \xC2"""
  "\xE7""\xB9""\xBE""\xB7""\xF2""\xBB""\xB0""\xCF""\xBA""\xA4""\xCE""\xA5"""
  "\xCE""\xA1""\xBC""\xA5""\xD9""\xA5""\xEB""\xCA""\xB8""\xB3""\xD8""\xBE"""
  "\xDE""\xBC""\xF5""\xBE""\xDE""\xA4""\xAC""\xB7""\xE8""\xC4""\xEA""\xA1"""
  "\xCA""1994\xC7""\xAF""\xA1""\xCB"" \xEB""\xBE""\xCA""\xF3"" \xA5""\xB3"""
  "\xA5""\xEA""\xA1""\xBC""\xA1""\xA6""\xA5""\xE9""\xA5""\xA4""\xA5""\xC9"""
  "\xA5""\xEB""\xA1""\xCA""Cory Lidle,10\xB7""\xEE""11\xC6""\xFC""\xDD""\xC7"""
  "\xA1""\xCB"" - \xA5""\xCB""\xA5""\xE5""\xA1""\xBC""\xA5""\xE8""\xA1""\xBC"""
  "\xA5""\xAF""\xA1""\xA6""\xA5""\xE4""\xA5""\xF3""\xA5""\xAD""\xA1""\xBC"""
  "\xA5""\xB9""\xA4""\xCE""\xC5""\xEA""\xBC""\xEA"" \xBB""\xCD""\xC2""\xE5"""
  "\xCC""\xDC""\xCC""\xF8""\xB2""\xC8""\xBE""\xAE""\xA4""\xBB""\xA4""\xF3"""
  "\xA1""\xCA""10\xB7""\xEE""10\xC6""\xFC""\xDD""\xC7""\xA1""\xCB"" - \xCD"""
  "\xEE""\xB8""\xEC""\xB2""\xC8"" \xA5""\xDE""\xA5""\xEC""\xA5""\xAF""\xA1"""
  "\xA6""\xA5""\xB0""\xA5""\xEC""\xA5""\xD5""\xA5""\xBF""\xA1""\xCA""Marek "
  "Grechuta,10\xB7""\xEE""9\xC6""\xFC""\xDD""\xC7""\xA1""\xCB"" - \xA5""\xDD"""
  "\xA1""\xBC""\xA5""\xE9""\xA5""\xF3""\xA5""\xC9""\xA4""\xCE""\xB2""\xCE"""
  "\xBC""\xEA"" \xC0""\xBE""\xBB""\xB3""\xC5""\xD0""\xBB""\xD6""\xCD""\xBA"""
  "\xA1""\xCA""10\xB7""\xEE""9\xC6""\xFC""\xDD""\xC7""\xA1""\xCB"" - \xB8"""
  "\xB5""\xC5""\xEC""\xC9""\xF0""\xC6""\xB0""\xCA""\xAA""\xB8""\xF8""\xB1"""
  "\xE0""\xB1""\xE0""\xC4""\xB9""\xA1""\xCA""\xA1""\xD6""\xA5""\xAB""\xA5"""
  "\xD0""\xB1""\xE0""\xC4""\xB9""\xA1""\xD7""\xA1""\xCB"" \xA5""\xA4""\xA1"""
  "\xBC""\xA5""\xB4""\xA5""\xEA""\xA1""\xA6""\xA5""\xE9""\xA5""\xC6""\xA5"""
  "\xA3""\xA5""\xB7""\xA5""\xA7""\xA5""\xD5""\xA1""\xCA""10\xB7""\xEE""6\xC6"""
  "\xFC""\xDD""\xC7""\xA1""\xCB"" - \xA5""\xED""\xA5""\xB7""\xA5""\xA2""\xA4"""
  "\xCE""\xC6""\xFC""\xCB""\xDC""\xB8""\xA6""\xB5""\xE6""\xB2""\xC8""\xA1"""
  "\xA2""\xB8""\xB5""\xA5""\xBD""\xA5""\xD3""\xA5""\xA8""\xA5""\xC8""\xCF"""
  "\xA2""\xCB""\xAE""\xB6""\xA6""\xBB""\xBA""\xC5""\xDE""\xB5""\xA1""\xB4"""
  "\xD8""\xBB""\xE6""\xA5""\xD7""\xA5""\xE9""\xA5""\xA6""\xA5""\xC0""\xC5"""
  "\xEC""\xB5""\xFE""\xBB""\xD9""\xB6""\xC9""\xC4""\xB9"" \xC9""\xB4""\xB2"""
  "\xCA""\xBB""\xF6""\xC5""\xB5"" \xBC""\xD2""\xB2""\xF1"" \xCB""\xA1""\xCE"""
  "\xA7"" - \xCE""\xD1""\xCD""\xFD"" - \xC0""\xAF""\xBC""\xA3"" - \xB9""\xD4"""
  "\xC0""\xAF"" - \xB7""\xD0""\xBA""\xD1"" - \xBB""\xBA""\xB6""\xC8"" - \xBF"""
  "\xA6""\xB6""\xC8"" - \xB6""\xB5""\xB0""\xE9"" - \xCE""\xF2""\xBB""\xCB"""
  " - \xCA""\xA1""\xBB""\xE3"" - \xB0""\xE5""\xCE""\xC5"" - \xBB""\xD4""\xCC"""
  "\xB1""\xB3""\xE8""\xC6""\xB0"" - \xCA""\xBF""\xCF""\xC2"" - \xC0""\xEF"""
  "\xC1""\xE8"" - \xB7""\xB3""\xBB""\xF6"" \xB7""\xDD""\xBD""\xD1""\xA4""\xC8"""
  "\xCA""\xB8""\xB2""\xBD"" \xBF""\xA9"" - \xB8""\xC0""\xB8""\xEC"" - \xBD"""
  "\xA1""\xB6""\xB5"" - \xB7""\xDD""\xC7""\xBD"" - \xC0""\xB8""\xB3""\xE8"""
  " - \xB8""\xE4""\xB3""\xDA"" - \xCA""\xB8""\xB3""\xD8"" - \xB2""\xBB""\xB3"""
  "\xDA"" - \xC8""\xFE""\xBD""\xD1"" - \xC9""\xF1""\xC2""\xE6""\xB7""\xDD"""
  "\xBD""\xD1"" - \xB1""\xC7""\xB2""\xE8"" - \xCC""\xA1""\xB2""\xE8"" - \xA5"""
  "\xA2""\xA5""\xCB""\xA5""\xE1"" - \xA5""\xB9""\xA5""\xDD""\xA1""\xBC""\xA5"""
  "\xC4"" - \xA5""\xB2""\xA1""\xBC""\xA5""\xE0"" - \xA5""\xE1""\xA5""\xC7"""
  "\xA5""\xA3""\xA5""\xA2"" - \xA5""\xC6""\xA5""\xEC""\xA5""\xD3"" - \xA5"""
  "\xE9""\xA5""\xB8""\xA5""\xAA"" - \xBD""\xF1""\xCA""\xAA"" - \xC9""\xF7"""
  "\xC2""\xAF"" - \xC5""\xC1""\xBE""\xB5"" - \xC7""\xAF""\xC3""\xE6""\xB9"""
  "\xD4""\xBB""\xF6"" \xC0""\xA4""\xB3""\xA6"" \xA5""\xA2""\xA5""\xB8""\xA5"""
  "\xA2"" - \xA5""\xA2""\xA5""\xD5""\xA5""\xEA""\xA5""\xAB"" - \xA5""\xAA"""
  "\xA5""\xBB""\xA5""\xA2""\xA5""\xCB""\xA5""\xA2"" - \xCB""\xCC""\xA5""\xA2"""
  "\xA5""\xE1""\xA5""\xEA""\xA5""\xAB"" - \xC6""\xEE""\xA5""\xA2""\xA5""\xE1"""
  "\xA5""\xEA""\xA5""\xAB"" - \xA5""\xE8""\xA1""\xBC""\xA5""\xED""\xA5""\xC3"""
  "\xA5""\xD1"" \xC6""\xFC""\xCB""\xDC"" \xCB""\xCC""\xB3""\xA4""\xC6""\xBB"""
  " - \xC5""\xEC""\xCB""\xCC"" - \xB4""\xD8""\xC5""\xEC"" - \xC3""\xE6""\xC9"""
  "\xF4"" - \xB6""\xE1""\xB5""\xA6"" - \xC3""\xE6""\xB9""\xF1"" - \xBB""\xCD"""
  "\xB9""\xF1"" - \xB6""\xE5""\xBD""\xA3"" - \xB2""\xAD""\xC6""\xEC"" \xBC"""
  "\xAB""\xC1""\xB3"" \xCA""\xAA""\xBC""\xC1"" - \xA5""\xA8""\xA5""\xCD""\xA5"""
  "\xEB""\xA5""\xAE""\xA1""\xBC"" - \xB1""\xA7""\xC3""\xE8"" - \xC3""\xCF"""
  "\xB5""\xE5"" - \xB4""\xC4""\xB6""\xAD"" - \xB5""\xA4""\xBE""\xDD"" - \xBA"""
  "\xD2""\xB3""\xB2"" - \xB3""\xA4""\xCD""\xCE"" - \xC2""\xE7""\xCE""\xA6"""
  "\xA1""\xA2""\xC5""\xE7"" - \xC0""\xB8""\xA4""\xAD""\xCA""\xAA""\xA4""\xC8"""
  "\xBC""\xAB""\xC1""\xB3"" - \xBF""\xA2""\xCA""\xAA"" - \xC6""\xB0""\xCA"""
  "\xAA"" - \xBF""\xCD""\xB4""\xD6"" - \xB9""\xDB""\xCA""\xAA"" \xB5""\xBB"""
  "\xBD""\xD1"" \xA5""\xB3""\xA5""\xF3""\xA5""\xD4""\xA5""\xE5""\xA1""\xBC"""
  "\xA5""\xBF"" - \xA5""\xA4""\xA5""\xF3""\xA5""\xBF""\xA1""\xBC""\xA5""\xCD"""
  "\xA5""\xC3""\xA5""\xC8"" - \xA5""\xA8""\xA5""\xEC""\xA5""\xAF""\xA5""\xC8"""
  "\xA5""\xED""\xA5""\xCB""\xA5""\xAF""\xA5""\xB9"" - \xA5""\xD0""\xA5""\xA4"""
  "\xA5""\xAA""\xA5""\xC6""\xA5""\xAF""\xA5""\xCE""\xA5""\xED""\xA5""\xB8"""
  "\xA1""\xBC"" - \xB8""\xF2""\xC4""\xCC"" - \xC5""\xB4""\xC6""\xBB"" - \xBC"""
  "\xAB""\xC6""\xB0""\xBC""\xD6"" - \xB7""\xFA""\xC3""\xDB"" - \xBE""\xF0"""
  "\xCA""\xF3"" - \xC4""\xCC""\xBF""\xAE"" - \xBB""\xF1""\xB8""\xBB"" - \xB8"""
  "\xB6""\xBB""\xD2""\xCE""\xCF"" - \xBF""\xF4""\xB3""\xD8""\xA4""\xC8""\xBC"""
  "\xAB""\xC1""\xB3""\xB2""\xCA""\xB3""\xD8"" \xBF""\xF4""\xB3""\xD8"" - \xCA"""
  "\xAA""\xCD""\xFD""\xB3""\xD8"" - \xB2""\xBD""\xB3""\xD8"" - \xC0""\xB8"""
  "\xCA""\xAA""\xB3""\xD8"" - \xC0""\xB8""\xC2""\xD6""\xB3""\xD8"" - \xC3"""
  "\xCF""\xB5""\xE5""\xB2""\xCA""\xB3""\xD8"" - \xC5""\xB7""\xCA""\xB8""\xB3"""
  "\xD8"" \xBF""\xCD""\xCA""\xB8""\xB2""\xCA""\xB3""\xD8""\xA4""\xC8""\xBC"""
  "\xD2""\xB2""\xF1""\xB2""\xCA""\xB3""\xD8"" \xCA""\xB8""\xB3""\xD8"" - \xB8"""
  "\xC0""\xB8""\xEC""\xB3""\xD8"" - \xC5""\xAF""\xB3""\xD8"" - \xCF""\xC0"""
  "\xCD""\xFD""\xB3""\xD8"" - \xBD""\xA1""\xB6""\xB5""\xB3""\xD8"" - \xBF"""
  "\xB4""\xCD""\xFD""\xB3""\xD8"" - \xC8""\xFE""\xB3""\xD8"" - \xBC""\xD2"""
  "\xB2""\xF1""\xB3""\xD8"" - \xCB""\xA1""\xB3""\xD8"" - \xCE""\xF2""\xBB"""
  "\xCB""\xB3""\xD8"" - \xB9""\xCD""\xB8""\xC5""\xB3""\xD8"" - \xCC""\xB1"""
  "\xC2""\xAF""\xB3""\xD8"" - \xB6""\xB5""\xB0""\xE9""\xB3""\xD8"" - \xC0"""
  "\xAF""\xBC""\xA3""\xB3""\xD8"" - \xB7""\xD0""\xBA""\xD1""\xB3""\xD8"" - "
  "\xB7""\xD0""\xB1""\xC4""\xB3""\xD8"" \xB1""\xFE""\xCD""\xD1""\xB2""\xCA"""
  "\xB3""\xD8""\xA4""\xC8""\xC1""\xED""\xB9""\xE7""\xB2""\xCA""\xB3""\xD8"""
  " \xBE""\xF0""\xCA""\xF3""\xB3""\xD8"" - \xB4""\xC4""\xB6""\xAD""\xB3""\xD8"""
  " - \xC3""\xCF""\xCD""\xFD""\xB3""\xD8"" - \xBF""\xCD""\xCE""\xE0""\xB3"""
  "\xD8"" - \xB0""\xE5""\xB3""\xD8"" - \xBC""\xD2""\xB2""\xF1""\xCA""\xA1"""
  "\xBB""\xE3""\xB3""\xD8"" - \xCC""\xF4""\xB3""\xD8"" - \xC7""\xC0""\xB3"""
  "\xD8"" - \xB9""\xA9""\xB3""\xD8"" \xBB""\xF1""\xCE""\xC1"" \xBA""\xF7""\xB0"""
  "\xFA"" - \xC7""\xAF""\xC9""\xBD"" - \xCE""\xF1"" - 365\xC6""\xFC"" - \xC3"""
  "\xCF""\xBF""\xDE"" - \xBF""\xF4""\xA4""\xCE""\xB0""\xEC""\xCD""\xF7"" - "
  "\xBF""\xCD""\xCC""\xBE""\xB0""\xEC""\xCD""\xF7"" - \xB0""\xEC""\xCD""\xF7"""
  "\xA4""\xCE""\xB0""\xEC""\xCD""\xF7"" - \xC0""\xA4""\xB3""\xA6""\xB3""\xC6"""
  "\xB9""\xF1""\xB4""\xD8""\xB7""\xB8""\xB5""\xAD""\xBB""\xF6"" \xA5""\xA6"""
  "\xA5""\xA3""\xA5""\xAD""\xA5""\xDD""\xA1""\xBC""\xA5""\xBF""\xA5""\xEB"""
  " \xA5""\xA2""\xA5""\xCB""\xA5""\xE1"" - \xB0""\xE5""\xB3""\xD8""\xA4""\xC8"""
  "\xB0""\xE5""\xCE""\xC5"" - \xC0""\xB8""\xA4""\xAD""\xCA""\xAA""\xA4""\xC8"""
  "\xBC""\xAB""\xC1""\xB3"" - \xB1""\xC7""\xB2""\xE8"" - \xA5""\xA8""\xA5"""
  "\xEC""\xA5""\xAF""\xA5""\xC8""\xA5""\xED""\xA5""\xCB""\xA5""\xAF""\xA5"""
  "\xB9"" - \xB2""\xBB""\xB3""\xDA"" - \xB2""\xB9""\xC0""\xF4"" - \xB2""\xBD"""
  "\xB3""\xD8"" - \xB4""\xC4""\xB6""\xAD"" - \xCB""\xCC""\xA5""\xA2""\xA5"""
  "\xE1""\xA5""\xEA""\xA5""\xAB"" - \xB6""\xB5""\xB0""\xE9"" - \xB7""\xB3"""
  "\xBB""\xF6"" - \xA5""\xAF""\xA5""\xE9""\xA5""\xB7""\xA5""\xC3""\xA5""\xAF"""
  "\xB2""\xBB""\xB3""\xDA"" - \xB7""\xD0""\xBA""\xD1""\xB3""\xD8"" - \xA5"""
  "\xB2""\xA1""\xBC""\xA5""\xE0"" - \xB8""\xC0""\xB8""\xEC""\xB3""\xD8"" - "
  "\xA5""\xB3""\xA5""\xEA""\xA5""\xA2"" - \xA5""\xB3""\xA5""\xF3""\xA5""\xD4"""
  "\xA5""\xE5""\xA1""\xBC""\xA5""\xBF"" - \xBA""\xD2""\xB3""\xB2"" - \xBC"""
  "\xAB""\xC6""\xB0""\xBC""\xD6"" - \xBF""\xA9"" - \xBF""\xA2""\xCA""\xAA"""
  " - \xBF""\xC0""\xC6""\xBB"" - \xA5""\xB9""\xA5""\xDD""\xA1""\xBC""\xA5"""
  "\xC4"" - \xC0""\xB8""\xCA""\xAA""\xB3""\xD8"" - \xC0""\xA4""\xB3""\xA6"""
  "\xB0""\xE4""\xBB""\xBA"" - \xC0""\xEF""\xC1""\xE8"" - \xC2""\xE8""\xBB"""
  "\xB0""\xC4""\xEB""\xB9""\xF1"" - \xC2""\xE7""\xC5""\xEC""\xB0""\xA1""\xB6"""
  "\xA6""\xB1""\xC9""\xB7""\xF7"" - \xC3""\xCF""\xB5""\xE5""\xB2""\xCA""\xB3"""
  "\xD8"" - \xC3""\xE6""\xB9""\xF1"" - \xC3""\xCF""\xCD""\xFD""\xB3""\xD8"""
  " - \xC5""\xAF""\xB3""\xD8"" - \xC5""\xB4""\xC6""\xBB"" - \xA5""\xC6""\xA5"""
  "\xEC""\xA5""\xD3"" - \xC5""\xB7""\xCA""\xB8""\xB3""\xD8"" - \xC6""\xFC"""
  "\xCB""\xDC"" - \xC6""\xFC""\xCB""\xDC""\xA4""\xCE""\xC5""\xD4""\xC6""\xBB"""
  "\xC9""\xDC""\xB8""\xA9"" - \xA5""\xD0""\xA5""\xB9"" - \xC8""\xFE""\xBD"""
  "\xD1"" - \xC9""\xF1""\xC2""\xE6""\xB7""\xDD""\xBD""\xD1"" - \xCA""\xA9"""
  "\xB6""\xB5"" - \xCA""\xAA""\xCD""\xFD""\xB3""\xD8"" - \xA5""\xD5""\xA5"""
  "\xE9""\xA5""\xF3""\xA5""\xB9"" - \xCA""\xB8""\xB3""\xD8"" - \xCA""\xBF"""
  "\xCF""\xC2"" - \xCC""\xA1""\xB2""\xE8"" - \xA5""\xE8""\xA1""\xBC""\xA5"""
  "\xED""\xA5""\xC3""\xA5""\xD1"" - \xA5""\xE9""\xA5""\xB8""\xA5""\xAA"" - "
  "\xCE""\xF2""\xBB""\xCB"" - \xCE""\xF2""\xBB""\xCB""\xC7""\xA7""\xBC""\xB1"""
  "\xA4""\xC8""\xC0""\xEF""\xB8""\xE5""\xBD""\xE8""\xCD""\xFD"" \xA5""\xAB"""
  "\xA5""\xC6""\xA5""\xB4""\xA5""\xEA"" \xB0""\xEC""\xCD""\xF7"" - \xA5""\xA6"""
  "\xA5""\xA3""\xA5""\xAD""\xA5""\xDA""\xA5""\xC7""\xA5""\xA3""\xA5""\xA2"""
  " - \xA5""\xA6""\xA5""\xA3""\xA5""\xAD""\xA5""\xE1""\xA5""\xC7""\xA5""\xA3"""
  "\xA5""\xA2""\xA1""\xA6""\xA5""\xD7""\xA5""\xED""\xA5""\xB8""\xA5""\xA7"""
  "\xA5""\xAF""\xA5""\xC8"" -"
;

const char* kTeststr11 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/ja_wikipedia_org_clean__SJS.txt
  " \x96""k\x8A""C\x93""\xB9""\x93""\xFA""\x96""{\x83""n\x83""\x80""\x83""t"
  "\x83""@\x83""C\x83""^\x81""[\x83""Y\x82""\xAA""\x83""\\\x83""t\x83""g\x83"""
  "o\x83""\x93""\x83""N\x82""\xC6""\x82""\xCC""\x83""v\x83""\x8C""\x81""[\x83"""
  "I\x83""t\x82""\xF0""\x90""\xA7""\x82""\xB5""\x81""A25\x94""N\x82""\xD4"""
  "\x82""\xE8""3\x93""x\x96""\xDA""\x82""\xCC""\x83""p\x83""V\x83""t\x83""B"
  "\x83""b\x83""N\x81""E\x83""\x8A""\x81""[\x83""O\x97""D\x8F""\x9F""\x81"""
  "B \x96""k\x92""\xA9""\x91""N\x8A""j\x8E""\xC0""\x8C""\xB1""\x82""\xC9""\x91"""
  "\xCE""\x82""\xB7""\x82""\xE9""\x8C""o\x8D""\xCF""\x90""\xA7""\x8D""\xD9"""
  "\x82""\xC5""\x93""\xFA""\x96""{\x93""\xC6""\x8E""\xA9""\x82""\xCC""\x92"""
  "\xC7""\x89""\xC1""\x90""\xA7""\x8D""\xD9""\x82""\xF0""\x8C""\x88""\x92"""
  "\xE8""\x81""B \x83""f\x83""B\x81""[\x83""v\x83""C\x83""\x93""\x83""p\x83"""
  "N\x83""g\x82""\xAA""\x94""N\x93""\xE0""\x82""\xC5""\x88""\xF8""\x91""\xDE"""
  "\x82""\xB7""\x82""\xE9""\x82""\xC6""\x94""\xAD""\x95""\\\x81""B\x81""\xA8"""
  "\x83""E\x83""B\x83""L\x83""j\x83""\x85""\x81""[\x83""X Google\x82""\xAA"""
  "Google Docs & Spreadsheets\x81""iGoogle\x94""\xC5""\x81""uOffice\x81""v\x81"""
  "A\x83""x\x81""[\x83""^\x94""\xC5""\x81""j\x82""\xCC""\x92""\xF1""\x8B""\x9F"""
  "\x82""\xF0""\x8A""J\x8E""n\x81""B [\x83""E\x83""B\x83""L\x83""j\x83""\x85"""
  "\x81""[\x83""X] [\x8D""\xC5""\x8B""\xDF""\x82""\xCC""\x8F""o\x97""\x88"""
  "\x8E""\x96""\x91""S\x95""\xB6""] \x93""\x8A""\x95""[\x81""E\x90""\x84""\x91"""
  "E \x8B""G\x90""\xDF""\x82""\xCC""\x98""b\x91""\xE8"" \x8F""H - \x89""^\x93"""
  "\xAE""\x89""\xEF"" - \x88""\xF0""\x8E""\xCF""\x89""\xEF"" - \x83""T\x83"""
  "\x93""\x83""} - \x83""u\x83""h\x83""E - \x83""L\x83""\x93""\x83""\x82""\x83"""
  "N\x83""Z\x83""C - \x82""\xC7""\x82""\xF1""\x82""\xAE""\x82""\xE8"" - \x8F"""
  "\x89""\x8A""\xA5""\x90""\xE1"" - \x8F""H\x82""\xCC""\x97""[\x8F""\xC4""\x82"""
  "\xAF""(*) - \x8F""\xBC""\x91""\xF9"" \x8D""\xA1""\x93""\xFA""\x82""\xCC"""
  "\x82""\xB1""\x82""\xE6""\x82""\xDD"" \x8B""\x8C""\x97""\xEF"":8\x8C""\x8E"""
  "22\x93""\xFA"" - \x91""\xE5""\x88""\xC0"" - \x93""\xFA""\x82""\xCC""\x8A"""
  "\xB1""\x8E""x:\x89""\xB3""\x88""\xE5"" - \x8B""\xE3""\x90""\xAF"":\x8E"""
  "l\x97""\xCE""\x96""\xD8""\x90""\xAF"" - \x93""\xF1""\x8F""\\\x94""\xAA"""
  "\x8F""h:\x98""\xB4""\x8F""h - \x8F""\\\x93""\xF1""\x92""\xBC"":\x8F""\x9C"""
  " - \x90""\xDF""\x8B""C:\x8A""\xA6""\x98""I\x81""A\x91""\x9A""\x8D""~\x82"""
  "\xDC""\x82""\xC5""10\x93""\xFA"" - \x91""I\x93""\xFA"":\x91""\xE5""\x94"""
  "\xC6""\x93""y \x8D""\xA1""\x93""\xFA""\x82""\xCD""\x89""\xBD""\x82""\xCC"""
  "\x93""\xFA"" \x83""O\x83""\x8A""\x83""j\x83""b\x83""W\x93""V\x95""\xB6"""
  "\x91""\xE4""\x82""\xF0""\x92""\xCA""\x82""\xE9""\x8C""o\x90""\xFC""\x82"""
  "\xF0""0\x93""x\x82""\xC6""\x92""\xE8""\x82""\xDF""\x82""\xE9""\x81""i188"
  "4\x94""N\x81""j 11\x8B""{\x89""\xC6""51\x90""l\x82""\xCC""\x8D""c\x91""\xB0"""
  "\x97""\xA3""\x92""E\x82""\xAA""\x8C""\x88""\x92""\xE8""\x81""i1947\x94"""
  "N\x81""j \x83""u\x83""\x89""\x83""b\x83""N\x81""E\x83""`\x83""\x85""\x81"""
  "[\x83""Y\x83""f\x81""[\x81""i1989\x94""N\x81""j \x91""\xE5""\x8D""]\x8C"""
  "\x92""\x8E""O\x98""Y\x82""\xCC""\x83""m\x81""[\x83""x\x83""\x8B""\x95""\xB6"""
  "\x8A""w\x8F""\xDC""\x8E""\xF3""\x8F""\xDC""\x82""\xAA""\x8C""\x88""\x92"""
  "\xE8""\x81""i1994\x94""N\x81""j \xE6""]\x95""\xF1"" \x83""R\x83""\x8A""\x81"""
  "[\x81""E\x83""\x89""\x83""C\x83""h\x83""\x8B""\x81""iCory Lidle,10\x8C"""
  "\x8E""11\x93""\xFA""\x9F""f\x81""j - \x83""j\x83""\x85""\x81""[\x83""\x88"""
  "\x81""[\x83""N\x81""E\x83""\x84""\x83""\x93""\x83""L\x81""[\x83""X\x82"""
  "\xCC""\x93""\x8A""\x8E""\xE8"" \x8E""l\x91""\xE3""\x96""\xDA""\x96""\xF6"""
  "\x89""\xC6""\x8F""\xAC""\x82""\xB9""\x82""\xF1""\x81""i10\x8C""\x8E""10\x93"""
  "\xFA""\x9F""f\x81""j - \x97""\x8E""\x8C""\xEA""\x89""\xC6"" \x83""}\x83"""
  "\x8C""\x83""N\x81""E\x83""O\x83""\x8C""\x83""t\x83""^\x81""iMarek Grechu"
  "ta,10\x8C""\x8E""9\x93""\xFA""\x9F""f\x81""j - \x83""|\x81""[\x83""\x89"""
  "\x83""\x93""\x83""h\x82""\xCC""\x89""\xCC""\x8E""\xE8"" \x90""\xBC""\x8E"""
  "R\x93""o\x8E""u\x97""Y\x81""i10\x8C""\x8E""9\x93""\xFA""\x9F""f\x81""j -"
  " \x8C""\xB3""\x93""\x8C""\x95""\x90""\x93""\xAE""\x95""\xA8""\x8C""\xF6"""
  "\x89""\x80""\x89""\x80""\x92""\xB7""\x81""i\x81""u\x83""J\x83""o\x89""\x80"""
  "\x92""\xB7""\x81""v\x81""j \x83""C\x81""[\x83""S\x83""\x8A""\x81""E\x83"""
  "\x89""\x83""e\x83""B\x83""V\x83""F\x83""t\x81""i10\x8C""\x8E""6\x93""\xFA"""
  "\x9F""f\x81""j - \x83""\x8D""\x83""V\x83""A\x82""\xCC""\x93""\xFA""\x96"""
  "{\x8C""\xA4""\x8B""\x86""\x89""\xC6""\x81""A\x8C""\xB3""\x83""\\\x83""r\x83"""
  "G\x83""g\x98""A\x96""M\x8B""\xA4""\x8E""Y\x93""}\x8B""@\x8A""\xD6""\x8E"""
  "\x86""\x83""v\x83""\x89""\x83""E\x83""_\x93""\x8C""\x8B""\x9E""\x8E""x\x8B"""
  "\xC7""\x92""\xB7"" \x95""S\x89""\xC8""\x8E""\x96""\x93""T \x8E""\xD0""\x89"""
  "\xEF"" \x96""@\x97""\xA5"" - \x97""\xCF""\x97""\x9D"" - \x90""\xAD""\x8E"""
  "\xA1"" - \x8D""s\x90""\xAD"" - \x8C""o\x8D""\xCF"" - \x8E""Y\x8B""\xC6"""
  " - \x90""E\x8B""\xC6"" - \x8B""\xB3""\x88""\xE7"" - \x97""\xF0""\x8E""j "
  "- \x95""\x9F""\x8E""\x83"" - \x88""\xE3""\x97""\xC3"" - \x8E""s\x96""\xAF"""
  "\x8A""\x88""\x93""\xAE"" - \x95""\xBD""\x98""a - \x90""\xED""\x91""\x88"""
  " - \x8C""R\x8E""\x96"" \x8C""|\x8F""p\x82""\xC6""\x95""\xB6""\x89""\xBB"""
  " \x90""H - \x8C""\xBE""\x8C""\xEA"" - \x8F""@\x8B""\xB3"" - \x8C""|\x94"""
  "\\ - \x90""\xB6""\x8A""\x88"" - \x8C""\xE2""\x8A""y - \x95""\xB6""\x8A"""
  "w - \x89""\xB9""\x8A""y - \x94""\xFC""\x8F""p - \x95""\x91""\x91""\xE4"""
  "\x8C""|\x8F""p - \x89""f\x89""\xE6"" - \x96""\x9F""\x89""\xE6"" - \x83"""
  "A\x83""j\x83""\x81"" - \x83""X\x83""|\x81""[\x83""c - \x83""Q\x81""[\x83"""
  "\x80"" - \x83""\x81""\x83""f\x83""B\x83""A - \x83""e\x83""\x8C""\x83""r "
  "- \x83""\x89""\x83""W\x83""I - \x8F""\x91""\x95""\xA8"" - \x95""\x97""\x91"""
  "\xAD"" - \x93""`\x8F""\xB3"" - \x94""N\x92""\x86""\x8D""s\x8E""\x96"" \x90"""
  "\xA2""\x8A""E \x83""A\x83""W\x83""A - \x83""A\x83""t\x83""\x8A""\x83""J "
  "- \x83""I\x83""Z\x83""A\x83""j\x83""A - \x96""k\x83""A\x83""\x81""\x83"""
  "\x8A""\x83""J - \x93""\xEC""\x83""A\x83""\x81""\x83""\x8A""\x83""J - \x83"""
  "\x88""\x81""[\x83""\x8D""\x83""b\x83""p \x93""\xFA""\x96""{ \x96""k\x8A"""
  "C\x93""\xB9"" - \x93""\x8C""\x96""k - \x8A""\xD6""\x93""\x8C"" - \x92""\x86"""
  "\x95""\x94"" - \x8B""\xDF""\x8B""E - \x92""\x86""\x8D""\x91"" - \x8E""l\x8D"""
  "\x91"" - \x8B""\xE3""\x8F""B - \x89""\xAB""\x93""\xEA"" \x8E""\xA9""\x91"""
  "R \x95""\xA8""\x8E""\xBF"" - \x83""G\x83""l\x83""\x8B""\x83""M\x81""[ - "
  "\x89""F\x92""\x88"" - \x92""n\x8B""\x85"" - \x8A""\xC2""\x8B""\xAB"" - \x8B"""
  "C\x8F""\xDB"" - \x8D""\xD0""\x8A""Q - \x8A""C\x97""m - \x91""\xE5""\x97"""
  "\xA4""\x81""A\x93""\x87"" - \x90""\xB6""\x82""\xAB""\x95""\xA8""\x82""\xC6"""
  "\x8E""\xA9""\x91""R - \x90""A\x95""\xA8"" - \x93""\xAE""\x95""\xA8"" - \x90"""
  "l\x8A""\xD4"" - \x8D""z\x95""\xA8"" \x8B""Z\x8F""p \x83""R\x83""\x93""\x83"""
  "s\x83""\x85""\x81""[\x83""^ - \x83""C\x83""\x93""\x83""^\x81""[\x83""l\x83"""
  "b\x83""g - \x83""G\x83""\x8C""\x83""N\x83""g\x83""\x8D""\x83""j\x83""N\x83"""
  "X - \x83""o\x83""C\x83""I\x83""e\x83""N\x83""m\x83""\x8D""\x83""W\x81""["
  " - \x8C""\xF0""\x92""\xCA"" - \x93""S\x93""\xB9"" - \x8E""\xA9""\x93""\xAE"""
  "\x8E""\xD4"" - \x8C""\x9A""\x92""z - \x8F""\xEE""\x95""\xF1"" - \x92""\xCA"""
  "\x90""M - \x8E""\x91""\x8C""\xB9"" - \x8C""\xB4""\x8E""q\x97""\xCD"" - \x90"""
  "\x94""\x8A""w\x82""\xC6""\x8E""\xA9""\x91""R\x89""\xC8""\x8A""w \x90""\x94"""
  "\x8A""w - \x95""\xA8""\x97""\x9D""\x8A""w - \x89""\xBB""\x8A""w - \x90"""
  "\xB6""\x95""\xA8""\x8A""w - \x90""\xB6""\x91""\xD4""\x8A""w - \x92""n\x8B"""
  "\x85""\x89""\xC8""\x8A""w - \x93""V\x95""\xB6""\x8A""w \x90""l\x95""\xB6"""
  "\x89""\xC8""\x8A""w\x82""\xC6""\x8E""\xD0""\x89""\xEF""\x89""\xC8""\x8A"""
  "w \x95""\xB6""\x8A""w - \x8C""\xBE""\x8C""\xEA""\x8A""w - \x93""N\x8A""w"
  " - \x98""_\x97""\x9D""\x8A""w - \x8F""@\x8B""\xB3""\x8A""w - \x90""S\x97"""
  "\x9D""\x8A""w - \x94""\xFC""\x8A""w - \x8E""\xD0""\x89""\xEF""\x8A""w - "
  "\x96""@\x8A""w - \x97""\xF0""\x8E""j\x8A""w - \x8D""l\x8C""\xC3""\x8A""w"
  " - \x96""\xAF""\x91""\xAD""\x8A""w - \x8B""\xB3""\x88""\xE7""\x8A""w - \x90"""
  "\xAD""\x8E""\xA1""\x8A""w - \x8C""o\x8D""\xCF""\x8A""w - \x8C""o\x89""c\x8A"""
  "w \x89""\x9E""\x97""p\x89""\xC8""\x8A""w\x82""\xC6""\x91""\x8D""\x8D""\x87"""
  "\x89""\xC8""\x8A""w \x8F""\xEE""\x95""\xF1""\x8A""w - \x8A""\xC2""\x8B"""
  "\xAB""\x8A""w - \x92""n\x97""\x9D""\x8A""w - \x90""l\x97""\xDE""\x8A""w "
  "- \x88""\xE3""\x8A""w - \x8E""\xD0""\x89""\xEF""\x95""\x9F""\x8E""\x83"""
  "\x8A""w - \x96""\xF2""\x8A""w - \x94""_\x8A""w - \x8D""H\x8A""w \x8E""\x91"""
  "\x97""\xBF"" \x8D""\xF5""\x88""\xF8"" - \x94""N\x95""\\ - \x97""\xEF"" -"
  " 365\x93""\xFA"" - \x92""n\x90""} - \x90""\x94""\x82""\xCC""\x88""\xEA"""
  "\x97""\x97"" - \x90""l\x96""\xBC""\x88""\xEA""\x97""\x97"" - \x88""\xEA"""
  "\x97""\x97""\x82""\xCC""\x88""\xEA""\x97""\x97"" - \x90""\xA2""\x8A""E\x8A"""
  "e\x8D""\x91""\x8A""\xD6""\x8C""W\x8B""L\x8E""\x96"" \x83""E\x83""B\x83"""
  "L\x83""|\x81""[\x83""^\x83""\x8B"" \x83""A\x83""j\x83""\x81"" - \x88""\xE3"""
  "\x8A""w\x82""\xC6""\x88""\xE3""\x97""\xC3"" - \x90""\xB6""\x82""\xAB""\x95"""
  "\xA8""\x82""\xC6""\x8E""\xA9""\x91""R - \x89""f\x89""\xE6"" - \x83""G\x83"""
  "\x8C""\x83""N\x83""g\x83""\x8D""\x83""j\x83""N\x83""X - \x89""\xB9""\x8A"""
  "y - \x89""\xB7""\x90""\xF2"" - \x89""\xBB""\x8A""w - \x8A""\xC2""\x8B""\xAB"""
  " - \x96""k\x83""A\x83""\x81""\x83""\x8A""\x83""J - \x8B""\xB3""\x88""\xE7"""
  " - \x8C""R\x8E""\x96"" - \x83""N\x83""\x89""\x83""V\x83""b\x83""N\x89""\xB9"""
  "\x8A""y - \x8C""o\x8D""\xCF""\x8A""w - \x83""Q\x81""[\x83""\x80"" - \x8C"""
  "\xBE""\x8C""\xEA""\x8A""w - \x83""R\x83""\x8A""\x83""A - \x83""R\x83""\x93"""
  "\x83""s\x83""\x85""\x81""[\x83""^ - \x8D""\xD0""\x8A""Q - \x8E""\xA9""\x93"""
  "\xAE""\x8E""\xD4"" - \x90""H - \x90""A\x95""\xA8"" - \x90""_\x93""\xB9"""
  " - \x83""X\x83""|\x81""[\x83""c - \x90""\xB6""\x95""\xA8""\x8A""w - \x90"""
  "\xA2""\x8A""E\x88""\xE2""\x8E""Y - \x90""\xED""\x91""\x88"" - \x91""\xE6"""
  "\x8E""O\x92""\xE9""\x8D""\x91"" - \x91""\xE5""\x93""\x8C""\x88""\x9F""\x8B"""
  "\xA4""\x89""h\x8C""\x97"" - \x92""n\x8B""\x85""\x89""\xC8""\x8A""w - \x92"""
  "\x86""\x8D""\x91"" - \x92""n\x97""\x9D""\x8A""w - \x93""N\x8A""w - \x93"""
  "S\x93""\xB9"" - \x83""e\x83""\x8C""\x83""r - \x93""V\x95""\xB6""\x8A""w "
  "- \x93""\xFA""\x96""{ - \x93""\xFA""\x96""{\x82""\xCC""\x93""s\x93""\xB9"""
  "\x95""{\x8C""\xA7"" - \x83""o\x83""X - \x94""\xFC""\x8F""p - \x95""\x91"""
  "\x91""\xE4""\x8C""|\x8F""p - \x95""\xA7""\x8B""\xB3"" - \x95""\xA8""\x97"""
  "\x9D""\x8A""w - \x83""t\x83""\x89""\x83""\x93""\x83""X - \x95""\xB6""\x8A"""
  "w - \x95""\xBD""\x98""a - \x96""\x9F""\x89""\xE6"" - \x83""\x88""\x81""["
  "\x83""\x8D""\x83""b\x83""p - \x83""\x89""\x83""W\x83""I - \x97""\xF0""\x8E"""
  "j - \x97""\xF0""\x8E""j\x94""F\x8E""\xAF""\x82""\xC6""\x90""\xED""\x8C"""
  "\xE3""\x8F""\x88""\x97""\x9D"" \x83""J\x83""e\x83""S\x83""\x8A"" \x88""\xEA"""
  "\x97""\x97"" - \x83""E\x83""B\x83""L\x83""y\x83""f\x83""B\x83""A - \x83"""
  "E\x83""B\x83""L\x83""\x81""\x83""f\x83""B\x83""A\x81""E\x83""v\x83""\x8D"""
  "\x83""W\x83""F\x83""N\x83""g -"
;

const char* kTeststr12 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/ja_wikipedia_org_clean__JIS.txt
  " \x1B""$BKL3$F;F|K\\%O%`%U%!%$%?!<%:$,%=%U%H%P%s%/$H$N%W%l!<%*%U$r@)$7!\""
  "\x1B""(B25\x1B""$BG/$V$j\x1B""(B3\x1B""$BEYL\\$N%Q%7%U%#%C%/!&%j!<%0M%>!"
  "!#\x1B""(B \x1B""$BKLD+A/3K<B83$KBP$9$k7P:Q@):[$GF|K\\FH<+$NDI2C@):[$r7h"
  "Dj!#\x1B""(B \x1B""$B%G%#!<%W%$%s%Q%/%H$,G/Fb$G0zB`$9$k$HH/I=!#\"*%&%#%-"
  "%K%e!<%9\x1B""(B Google\x1B""$B$,\x1B""(BGoogle Docs & Spreadsheets\x1B"""
  "$B!J\x1B""(BGoogle\x1B""$BHG!V\x1B""(BOffice\x1B""$B!W!\"%Y!<%?HG!K$NDs6"
  "!$r3+;O!#\x1B""(B [\x1B""$B%&%#%-%K%e!<%9\x1B""(B] [\x1B""$B:G6a$N=PMh;v"
  "A4J8\x1B""(B] \x1B""$BEjI<!&?dA&\x1B""(B \x1B""$B5(@a$NOCBj\x1B""(B \x1B"""
  "$B=)\x1B""(B - \x1B""$B1?F02q\x1B""(B - \x1B""$B0r<Q2q\x1B""(B - \x1B""$"
  "B%5%s%^\x1B""(B - \x1B""$B%V%I%&\x1B""(B - \x1B""$B%-%s%b%/%;%$\x1B""(B "
  "- \x1B""$B$I$s$0$j\x1B""(B - \x1B""$B=i4'@c\x1B""(B - \x1B""$B=)$NM<>F$1"
  "\x1B""(B(*) - \x1B""$B>>B{\x1B""(B \x1B""$B:#F|$N$3$h$_\x1B""(B \x1B""$B"
  "5lNq\x1B""(B:8\x1B""$B7n\x1B""(B22\x1B""$BF|\x1B""(B - \x1B""$BBg0B\x1B"""
  "(B - \x1B""$BF|$N43;Y\x1B""(B:\x1B""$B250g\x1B""(B - \x1B""$B6e@1\x1B""("
  "B:\x1B""$B;MNPLZ@1\x1B""(B - \x1B""$BFs==H,=I\x1B""(B:\x1B""$BP6=I\x1B"""
  "(B - \x1B""$B==FsD>\x1B""(B:\x1B""$B=|\x1B""(B - \x1B""$B@a5$\x1B""(B:\x1B"""
  "$B4(O*!\"Az9_$^$G\x1B""(B10\x1B""$BF|\x1B""(B - \x1B""$BA*F|\x1B""(B:\x1B"""
  "$BBgHHEZ\x1B""(B \x1B""$B:#F|$O2?$NF|\x1B""(B \x1B""$B%0%j%K%C%8E7J8Bf$r"
  "DL$k7P@~$r\x1B""(B0\x1B""$BEY$HDj$a$k!J\x1B""(B1884\x1B""$BG/!K\x1B""(B "
  "11\x1B""$B5\\2H\x1B""(B51\x1B""$B?M$N9DB2N%C&$,7hDj!J\x1B""(B1947\x1B""$"
  "BG/!K\x1B""(B \x1B""$B%V%i%C%/!&%A%e!<%:%G!<!J\x1B""(B1989\x1B""$BG/!K\x1B"""
  "(B \x1B""$BBg9>7r;0O:$N%N!<%Y%kJ83X>^<u>^$,7hDj!J\x1B""(B1994\x1B""$BG/!"
  "K\x1B""(B \x1B""$Bk>Js\x1B""(B \x1B""$B%3%j!<!&%i%$%I%k!J\x1B""(BCory Li"
  "dle,10\x1B""$B7n\x1B""(B11\x1B""$BF|]G!K\x1B""(B - \x1B""$B%K%e!<%h!<%/!"
  "&%d%s%-!<%9$NEj<j\x1B""(B \x1B""$B;MBeL\\Lx2H>.$;$s!J\x1B""(B10\x1B""$B7"
  "n\x1B""(B10\x1B""$BF|]G!K\x1B""(B - \x1B""$BMn8l2H\x1B""(B \x1B""$B%^%l%"
  "/!&%0%l%U%?!J\x1B""(BMarek Grechuta,10\x1B""$B7n\x1B""(B9\x1B""$BF|]G!K\x1B"""
  "(B - \x1B""$B%]!<%i%s%I$N2N<j\x1B""(B \x1B""$B@>;3EP;VM:!J\x1B""(B10\x1B"""
  "$B7n\x1B""(B9\x1B""$BF|]G!K\x1B""(B - \x1B""$B85ElIpF0J*8x1`1`D9!J!V%+%P"
  "1`D9!W!K\x1B""(B \x1B""$B%$!<%4%j!&%i%F%#%7%'%U!J\x1B""(B10\x1B""$B7n\x1B"""
  "(B6\x1B""$BF|]G!K\x1B""(B - \x1B""$B%m%7%\"$NF|K\\8&5f2H!\"85%=%S%(%HO\""
  "K.6&;:E^5!4X;f%W%i%&%@El5~;Y6ID9\x1B""(B \x1B""$BI42J;vE5\x1B""(B \x1B"""
  "$B<R2q\x1B""(B \x1B""$BK!N'\x1B""(B - \x1B""$BNQM}\x1B""(B - \x1B""$B@/<"
  "#\x1B""(B - \x1B""$B9T@/\x1B""(B - \x1B""$B7P:Q\x1B""(B - \x1B""$B;:6H\x1B"""
  "(B - \x1B""$B?&6H\x1B""(B - \x1B""$B650i\x1B""(B - \x1B""$BNr;K\x1B""(B "
  "- \x1B""$BJ!;c\x1B""(B - \x1B""$B0eNE\x1B""(B - \x1B""$B;TL13hF0\x1B""(B"
  " - \x1B""$BJ?OB\x1B""(B - \x1B""$B@oAh\x1B""(B - \x1B""$B73;v\x1B""(B \x1B"""
  "$B7]=Q$HJ82=\x1B""(B \x1B""$B?)\x1B""(B - \x1B""$B8@8l\x1B""(B - \x1B""$"
  "B=!65\x1B""(B - \x1B""$B7]G=\x1B""(B - \x1B""$B@83h\x1B""(B - \x1B""$B8d"
  "3Z\x1B""(B - \x1B""$BJ83X\x1B""(B - \x1B""$B2;3Z\x1B""(B - \x1B""$BH~=Q\x1B"""
  "(B - \x1B""$BIqBf7]=Q\x1B""(B - \x1B""$B1G2h\x1B""(B - \x1B""$BL!2h\x1B"""
  "(B - \x1B""$B%\"%K%a\x1B""(B - \x1B""$B%9%]!<%D\x1B""(B - \x1B""$B%2!<%`"
  "\x1B""(B - \x1B""$B%a%G%#%\"\x1B""(B - \x1B""$B%F%l%S\x1B""(B - \x1B""$B"
  "%i%8%*\x1B""(B - \x1B""$B=qJ*\x1B""(B - \x1B""$BIwB/\x1B""(B - \x1B""$BE"
  "A>5\x1B""(B - \x1B""$BG/Cf9T;v\x1B""(B \x1B""$B@$3&\x1B""(B \x1B""$B%\"%"
  "8%\"\x1B""(B - \x1B""$B%\"%U%j%+\x1B""(B - \x1B""$B%*%;%\"%K%\"\x1B""(B "
  "- \x1B""$BKL%\"%a%j%+\x1B""(B - \x1B""$BFn%\"%a%j%+\x1B""(B - \x1B""$B%h"
  "!<%m%C%Q\x1B""(B \x1B""$BF|K\\\x1B""(B \x1B""$BKL3$F;\x1B""(B - \x1B""$B"
  "ElKL\x1B""(B - \x1B""$B4XEl\x1B""(B - \x1B""$BCfIt\x1B""(B - \x1B""$B6a5"
  "&\x1B""(B - \x1B""$BCf9q\x1B""(B - \x1B""$B;M9q\x1B""(B - \x1B""$B6e=#\x1B"""
  "(B - \x1B""$B2-Fl\x1B""(B \x1B""$B<+A3\x1B""(B \x1B""$BJ*<A\x1B""(B - \x1B"""
  "$B%(%M%k%.!<\x1B""(B - \x1B""$B1'Ch\x1B""(B - \x1B""$BCO5e\x1B""(B - \x1B"""
  "$B4D6-\x1B""(B - \x1B""$B5$>]\x1B""(B - \x1B""$B:R32\x1B""(B - \x1B""$B3"
  "$MN\x1B""(B - \x1B""$BBgN&!\"Eg\x1B""(B - \x1B""$B@8$-J*$H<+A3\x1B""(B -"
  " \x1B""$B?\"J*\x1B""(B - \x1B""$BF0J*\x1B""(B - \x1B""$B?M4V\x1B""(B - \x1B"""
  "$B9[J*\x1B""(B \x1B""$B5;=Q\x1B""(B \x1B""$B%3%s%T%e!<%?\x1B""(B - \x1B"""
  "$B%$%s%?!<%M%C%H\x1B""(B - \x1B""$B%(%l%/%H%m%K%/%9\x1B""(B - \x1B""$B%P"
  "%$%*%F%/%N%m%8!<\x1B""(B - \x1B""$B8rDL\x1B""(B - \x1B""$BE4F;\x1B""(B -"
  " \x1B""$B<+F0<V\x1B""(B - \x1B""$B7zC[\x1B""(B - \x1B""$B>pJs\x1B""(B - "
  "\x1B""$BDL?.\x1B""(B - \x1B""$B;q8;\x1B""(B - \x1B""$B86;RNO\x1B""(B - \x1B"""
  "$B?t3X$H<+A32J3X\x1B""(B \x1B""$B?t3X\x1B""(B - \x1B""$BJ*M}3X\x1B""(B -"
  " \x1B""$B2=3X\x1B""(B - \x1B""$B@8J*3X\x1B""(B - \x1B""$B@8BV3X\x1B""(B "
  "- \x1B""$BCO5e2J3X\x1B""(B - \x1B""$BE7J83X\x1B""(B \x1B""$B?MJ82J3X$H<R"
  "2q2J3X\x1B""(B \x1B""$BJ83X\x1B""(B - \x1B""$B8@8l3X\x1B""(B - \x1B""$BE"
  "/3X\x1B""(B - \x1B""$BO@M}3X\x1B""(B - \x1B""$B=!653X\x1B""(B - \x1B""$B"
  "?4M}3X\x1B""(B - \x1B""$BH~3X\x1B""(B - \x1B""$B<R2q3X\x1B""(B - \x1B""$"
  "BK!3X\x1B""(B - \x1B""$BNr;K3X\x1B""(B - \x1B""$B9M8E3X\x1B""(B - \x1B"""
  "$BL1B/3X\x1B""(B - \x1B""$B650i3X\x1B""(B - \x1B""$B@/<#3X\x1B""(B - \x1B"""
  "$B7P:Q3X\x1B""(B - \x1B""$B7P1D3X\x1B""(B \x1B""$B1~MQ2J3X$HAm9g2J3X\x1B"""
  "(B \x1B""$B>pJs3X\x1B""(B - \x1B""$B4D6-3X\x1B""(B - \x1B""$BCOM}3X\x1B"""
  "(B - \x1B""$B?MN`3X\x1B""(B - \x1B""$B0e3X\x1B""(B - \x1B""$B<R2qJ!;c3X\x1B"""
  "(B - \x1B""$BLt3X\x1B""(B - \x1B""$BG@3X\x1B""(B - \x1B""$B9)3X\x1B""(B "
  "\x1B""$B;qNA\x1B""(B \x1B""$B:w0z\x1B""(B - \x1B""$BG/I=\x1B""(B - \x1B"""
  "$BNq\x1B""(B - 365\x1B""$BF|\x1B""(B - \x1B""$BCO?^\x1B""(B - \x1B""$B?t"
  "$N0lMw\x1B""(B - \x1B""$B?ML>0lMw\x1B""(B - \x1B""$B0lMw$N0lMw\x1B""(B -"
  " \x1B""$B@$3&3F9q4X785-;v\x1B""(B \x1B""$B%&%#%-%]!<%?%k\x1B""(B \x1B""$"
  "B%\"%K%a\x1B""(B - \x1B""$B0e3X$H0eNE\x1B""(B - \x1B""$B@8$-J*$H<+A3\x1B"""
  "(B - \x1B""$B1G2h\x1B""(B - \x1B""$B%(%l%/%H%m%K%/%9\x1B""(B - \x1B""$B2"
  ";3Z\x1B""(B - \x1B""$B29@t\x1B""(B - \x1B""$B2=3X\x1B""(B - \x1B""$B4D6-"
  "\x1B""(B - \x1B""$BKL%\"%a%j%+\x1B""(B - \x1B""$B650i\x1B""(B - \x1B""$B"
  "73;v\x1B""(B - \x1B""$B%/%i%7%C%/2;3Z\x1B""(B - \x1B""$B7P:Q3X\x1B""(B -"
  " \x1B""$B%2!<%`\x1B""(B - \x1B""$B8@8l3X\x1B""(B - \x1B""$B%3%j%\"\x1B"""
  "(B - \x1B""$B%3%s%T%e!<%?\x1B""(B - \x1B""$B:R32\x1B""(B - \x1B""$B<+F0<"
  "V\x1B""(B - \x1B""$B?)\x1B""(B - \x1B""$B?\"J*\x1B""(B - \x1B""$B?@F;\x1B"""
  "(B - \x1B""$B%9%]!<%D\x1B""(B - \x1B""$B@8J*3X\x1B""(B - \x1B""$B@$3&0d;"
  ":\x1B""(B - \x1B""$B@oAh\x1B""(B - \x1B""$BBh;0Dk9q\x1B""(B - \x1B""$BBg"
  "El0!6&1I7w\x1B""(B - \x1B""$BCO5e2J3X\x1B""(B - \x1B""$BCf9q\x1B""(B - \x1B"""
  "$BCOM}3X\x1B""(B - \x1B""$BE/3X\x1B""(B - \x1B""$BE4F;\x1B""(B - \x1B""$"
  "B%F%l%S\x1B""(B - \x1B""$BE7J83X\x1B""(B - \x1B""$BF|K\\\x1B""(B - \x1B"""
  "$BF|K\\$NETF;I\\8)\x1B""(B - \x1B""$B%P%9\x1B""(B - \x1B""$BH~=Q\x1B""(B"
  " - \x1B""$BIqBf7]=Q\x1B""(B - \x1B""$BJ)65\x1B""(B - \x1B""$BJ*M}3X\x1B"""
  "(B - \x1B""$B%U%i%s%9\x1B""(B - \x1B""$BJ83X\x1B""(B - \x1B""$BJ?OB\x1B"""
  "(B - \x1B""$BL!2h\x1B""(B - \x1B""$B%h!<%m%C%Q\x1B""(B - \x1B""$B%i%8%*\x1B"""
  "(B - \x1B""$BNr;K\x1B""(B - \x1B""$BNr;KG'<1$H@o8e=hM}\x1B""(B \x1B""$B%"
  "+%F%4%j\x1B""(B \x1B""$B0lMw\x1B""(B - \x1B""$B%&%#%-%Z%G%#%\"\x1B""(B -"
  " \x1B""$B%&%#%-%a%G%#%\"!&%W%m%8%'%/%H\x1B""(B -"
;

const char* kTeststr13 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/zh-classical_wikipedia_org_clean__BIG5_HKSCS.txt
  " \xA1""E \xA4""V\xB6""q\xBC""\xC6""\xA6""r \xA1""E \xB8""U\xB0""\xEA""\xA8"""
  "\xF3""\xA7""@ \xA1""E \xAC""\xC9""\xAD""\xB1""\xC2""\xBD""\xC4""\xB6"" \xB5"""
  "\xB4""\xA7""\xAE""\xA6""n\xA4""\xE5"" \xB6""\xB5""\xC4""y\xAA""\xCC""\xA1"""
  "A\xA4""U\xAC""\xDB""\xA4""H\xA4""]\xA1""A\xA6""r\xA6""\xD0""\xA1""C\xAA"""
  "\xEC""\xB0""_\xAE""\xC9""\xA1""A\xA6""~\xA4""G\xA4""Q\xA5""|\xA1""C\xA8"""
  "\xE4""\xA9""u\xA4""\xF7""\xB6""\xB5""\xB1""\xE7""\xA1""A\xB1""\xE7""\xA4"""
  "\xF7""\xA7""Y\xB7""\xA1""\xB1""N\xB6""\xB5""\xBF""P\xA1""A\xAC""\xB0""\xAF"""
  "\xB3""\xB1""N\xA4""\xFD""\xE6""\xF9""\xA9""\xD2""\xBC""\xAE""\xAA""\xCC"""
  "\xA4""]\xA1""C\xB6""\xB5""\xA4""\xF3""\xA5""@\xA5""@\xAC""\xB0""\xB7""\xA1"""
  "\xB1""N\xA1""A\xAB""\xCA""\xA9""\xF3""\xB6""\xB5""\xA1""A\xAC""G\xA9""m\xB6"""
  "\xB5""\xA4""\xF3""\xA1""C \xA4""\xD3""\xA5""v\xA4""\xBD""\xA4""\xEA""\xA1"""
  "G\xA7""^\xBB""D\xA4""\xA7""\xA9""P\xA5""\xCD""\xA4""\xEA""\xA1""u\xB5""\xCF"""
  "\xA5""\xD8""\xBB""\\\xAD""\xAB""\xC0""\xFB""\xA4""l\xA1""v\xA1""A\xA4""S"
  "\xBB""D\xB6""\xB5""\xA6""\xD0""\xA5""\xE7""\xAD""\xAB""\xC0""\xFB""\xA4"""
  "l\xA1""C\xA6""\xD0""\xB0""Z\xA8""\xE4""\xAD""]\xB8""\xC7""\xA8""\xB8""\xA1"""
  "H\xA6""\xF3""\xBF""\xB3""\xA4""\xA7""\xBC""\xC9""\xA4""]\xA1""I\xA4""\xD2"""
  "\xAF""\xB3""\xA5""\xA2""\xA8""\xE4""\xAC""F\xA1""A\xB3""\xAF""\xAF""A\xAD"""
  "\xBA""\xC3""\xF8""\xA1""A\xBB""\xA8""\xB3""\xC7""\x95""|\xB0""_\xA1""A\xAC"""
  "\xDB""\xBB""P\xA8""\xC3""\xAA""\xA7""\xA1""A\xA4""\xA3""\xA5""i\xB3""\xD3"""
  "\xBC""\xC6""\xA1""C\xB5""M\xA6""\xD0""\xAB""D\xA6""\xB3""\xA4""\xD8""\xA4"""
  "o\xAD""\xBC""\xD4""\xB0""\xA1""A\xB0""_\xC3""\xF7""\xAF""a\xA4""\xA7""\xA4"""
  "\xA4""\xA1""A\xA4""T\xA6""~\xA1""A\xB9""E\xB1""N\xA4""\xAD""\xBD""\xD1"""
  "\xAB""J\xB7""\xC0""\xAF""\xB3""\xA1""A\xA4""\xC0""\xB5""\xF5""\xA4""\xD1"""
  "\xA4""U\xA1""A\xA6""\xD3""\xAB""\xCA""\xA4""\xFD""\xAB""J\xA1""A\xAC""F\xA5"""
  "\xD1""\xA6""\xD0""\xA5""X\xA1""A\xB8""\xB9""\xAC""\xB0""\xA1""u\xC5""Q\xA4"""
  "\xFD""\xA1""v\xA1""A\xA6""\xEC""\xC1""\xF6""\xA4""\xA3""\xB2""\xD7""\xA1"""
  "A\xAA""\xF1""\xA5""j\xA5""H\xA8""\xD3""\xA5""\xBC""\xB9""\xC1""\xA6""\xB3"""
  "\xA4""]\xA1""C\xA4""\xCE""\xA6""\xD0""\xAD""I\xC3""\xF6""\xC3""h\xB7""\xA1"""
  "\xA1""A\xA9""\xF1""\xB3""v\xB8""q\xAB""\xD2""\xA6""\xD3""\xA6""\xDB""\xA5"""
  "\xDF""\xA1""A\xAB""\xE8""\xA4""\xFD""\xAB""J\xAB""q\xA4""v\xA1""A\xC3""\xF8"""
  "\xA8""o\xA1""C\xA6""\xDB""\xAC""\xE1""\xA5""\\\xA5""\xEF""\xA1""A\xBE""\xC4"""
  "\xA8""\xE4""\xA8""p\xB4""\xBC""\xA6""\xD3""\xA4""\xA3""\xAE""v\xA5""j\xA1"""
  "A\xBF""\xD7""\xC5""Q\xA4""\xFD""\xA4""\xA7""\xB7""~\xA1""A\xB1""\xFD""\xA5"""
  "H\xA4""O\xA9""\xBA""\xB8""g\xC0""\xE7""\xA4""\xD1""\xA4""U\xA1""A\xA4""\xAD"""
  "\xA6""~\xA8""\xF2""\xA4""`\xA8""\xE4""\xB0""\xEA""\xA1""A\xA8""\xAD""\xA6"""
  "\xBA""\xAA""F\xAB""\xB0""\xA1""A\xA9""|\xA4""\xA3""\xC4""\xB1""\xB9""\xED"""
  "\xA6""\xD3""\xA4""\xA3""\xA6""\xDB""\xB3""d\xA1""A\xB9""L\xA8""o\xA1""C\xA4"""
  "D\xA4""\xDE""\xA1""u\xA4""\xD1""\xA4""`\xA7""\xDA""\xA1""A\xAB""D\xA5""\xCE"""
  "\xA7""L\xA4""\xA7""\xB8""o\xA4""]\xA1""v\xA1""A\xB0""Z\xA4""\xA3""\xC2"""
  "\xD5""\xAB""v\xA1""I \xA6""~\xA4""\xBA""\xB5""\xB4""\xA7""\xAE""\xA6""n\xA4"""
  "\xE5"" \xA1""E \xB5""\xB4""\xA7""\xAE""\xA6""n\xA4""\xE5""\xA4""@\xC4""\xFD"""
  " \xA6""\xBC""\xAA""\xBE""\xA4""\xA7""\xA5""G\xA1""H \xA6""\xBC""\xA5""i\xAA"""
  "\xBE""\xB1""E\xAC""\xB0""\xA4""\xA4""\xB5""\xD8""\xA5""\xC1""\xB1""\xDA"""
  "\xA9""l\xAF""\xAA""\xA1""H\xA1""]\xB9""\xCF""\xA1""^"
;

const char* kTeststr14 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/wuu_wikipedia_org_clean__GB.txt
  " \xD4""\xC2""\xB1""\xFD""\xB7""\xA2""\xD5""\xB9""\xB5""\xBD""\xBD""\xF1"""
  "\xB3""\xAF""\xA3""\xAC""\xC6""\xB7""\xD6""\xD6""\xB7""\xB1""\xB6""\xE0"""
  "\xA3""\xAC""\xB7""\xE7""\xCE""\xB6""\xD2""\xF2""\xB5""\xD8""\xB6""\xF8"""
  "\xD2""\xEC""\xA1""\xA3""\xEB""\xA1""\xC0""\xEF""\xCF""\xF2""\xBE""\xA9"""
  "\xCA""\xBD""\xA1""\xA2""\xCB""\xD5""\xCA""\xBD""\xA1""\xA2""\xB9""\xE3"""
  "\xCA""\xBD""\xA1""\xA2""\xB3""\xB1""\xCA""\xBD""\xB5""\xC8""\xCF""\xE0"""
  "\xB5""\xB1""\xD3""\xD0""\xCC""\xD8""\xC9""\xAB""\xA3""\xAC""\xB2""\xA6"""
  "\xC0""\xAD""\xB9""\xE3""\xB4""\xF3""\xC0""\xCF""\xB0""\xD9""\xD0""\xD5"""
  "\xCB""\xF9""\xBB""\xB6""\xCF""\xB2""\xA1""\xA3"" \x82""\x99""\xB0""\xA2"""
  "\xCF""\xFE""\xB5""\xC3""... \xCE""\xE2""\xD3""\xEF""\xC0""\xD5""\xC0""\xAD"""
  "\xCA""\xC0""\xBD""\xE7""\xC0""\xCB""\xC4""\xB8""\xD3""\xEF""\xC8""\xCB"""
  "\xBF""\xDA""\xC5""\xC5""\xB5""\xBD""\xB5""\xDA""\xBC""\xB8""\xA3""\xBF"""
  " \xCE""\xE2""\xD3""\xEF""\xB5""\xD8""\xB7""\xBD""\xCF""\xB7""\xCE""\xFD"""
  "\xBE""\xE7""\xCA""\xC7""\xC4""\xC4""\xBA""\xE0""\xB7""\xA2""\xD5""\xB9"""
  "\xB3""\xF6""\xC0""\xB4""\xB8""\xF6""\xA3""\xBF"" \xD7""\xF6""\xB9""\xFD"""
  "\xC7""\xB0""\xD6""\xD0""\xB9""\xFA""\xB9""\xFA""\xBC""\xD2""\xB8""\xB1"""
  "\xD6""\xF7""\xCF""\xAF""\xB8""\xF6""\xA1""\xB0""\xBA""\xEC""\xC9""\xAB"""
  "\xD7""\xCA""\xB1""\xBE""\xBC""\xD2""\xA1""\xB1""\xCA""\xC7""\xC9""\xB6"""
  "\xC8""\xCB""\xA3""\xBF"" \xC1""\xBA""\xB3""\xAF""\xCE""\xB0""\xA1""\xA2"""
  "\xC1""\xF5""\xBC""\xCE""\xC1""\xE1""\xB5""\xC8""\xD6""\xF7""\xD1""\xDD"""
  "\xB8""\xF6""\xCE""\xE2""\xD3""\xEF""\xB5""\xE7""\xD3""\xB0""\xA1""\xB0"""
  "\xBA""\xA3""\xC9""\xCF""\xBB""\xA8""\xA1""\xB1""\xBD""\xB2""\xD1""\xDB"""
  "\xC9""\xB6""\xA3""\xBF"" \xC0""\xD5""1983\xB0""\xE6""\xB5""\xE7""\xCA""\xD3"""
  "\xC1""\xAC""\xD0""\xF8""\xBE""\xE7""\xA1""\xB0""\xC9""\xE4""\xB5""\xF1"""
  "\xD3""\xA2""\xD0""\xDB""\xB4""\xAB""\xA1""\xB1""\xC0""\xEF""\xCF""\xF2"""
  "\xB3""\xF6""\xD1""\xDD""\xC5""\xAE""\xD6""\xF7""\xBD""\xC7""\xB8""\xF6"""
  "\xCA""\xC7""\xC9""\xB6""\xC8""\xCB""\xA3""\xBF"" \xD0""\xC2""\xCE""\xC5"""
  "\xC0""\xCB""\xCF""\xF2"" Roger D. Kornberg \xD2""\xF2""\xCE""\xAA""\xD9"""
  "\xB5""\xC0""\xD5""\xD2""\xC5""\xB4""\xAB""\xD7""\xAA""\xC2""\xBC""\xC9"""
  "\xCF""\xCD""\xB7""\xB8""\xF6""\xB9""\xA4""\xD7""\xF7""\xBB""\xF1""\xB5"""
  "\xC3""2006\xC4""\xEA""\xB6""\xC8""\xB8""\xF6""\xC5""\xB5""\xB1""\xB4""\xB6"""
  "\xFB""\xBB""\xAF""\xD1""\xA7""\xBD""\xB1""\xA1""\xA3"" \xB3""\xAF""\xCF"""
  "\xCA""\xD0""\xFB""\xB2""\xBC""\xD2""\xAA""\xBD""\xF8""\xD0""\xD0""\xBA"""
  "\xCB""\xCA""\xD4""\xD1""\xE9""\xA3""\xAC""\xB2""\xA2""\xC9""\xF9""\xD1"""
  "\xD4""\xB8""\xA5""\xBB""\xE1""\xCA""\xD7""\xCF""\xC8""\xCA""\xB9""\xD3"""
  "\xC3""\xBA""\xCB""\xCE""\xE4""\xC6""\xF7""\xA1""\xA3"" \xCE""\xE2""\xD3"""
  "\xEF""\xCE""\xAC""\xBB""\xF9""\xB0""\xD9""\xBF""\xC6""\xC0""\xD5""2006\xC4"""
  "\xEA""10\xD4""\xC2""1\xBA""\xC5""\xD5""\xFD""\xCA""\xBD""\xB3""\xC9""\xC1"""
  "\xA2""\xA1""\xA3"" \xC5""\xB5""\xB1""\xB4""\xB6""\xFB""\xBD""\xB1""\xCE"""
  "\xAF""\xD4""\xB1""\xBB""\xE1""\xD0""\xFB""\xB2""\xBC""\xC4""\xC3""2006\xC4"""
  "\xEA""\xB6""\xC8""\xB8""\xF6""\xC5""\xB5""\xB1""\xB4""\xB6""\xFB""\xCE"""
  "\xEF""\xC0""\xED""\xD1""\xA7""\xBD""\xB1""\xCA""\xDA""\xB2""\xA6""\xC1"""
  "\xBD""\xCE""\xBB""\xC3""\xC0""\xB9""\xFA""\xBF""\xC6""\xD1""\xA7""\xBC"""
  "\xD2"" John "
;

const char* kTeststr22 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/se_wikipedia_org_clean__UTF8.txt
  "tnegirjji gosa gii beare s\xC3""\xA1""htt\xC3""\xA1"" \xC4""\x8D""\xC3"""
  "\xA1""llit artihkkaliid. Maid don s\xC3""\xA1""ht\xC3""\xA1""t dievasmah"
  "ttit ja divvut juo \xC4""\x8D""\xC3""\xA1""llojuvvon artihkkaliid dahje "
  "\xC4""\x8D""\xC3""\xA1""lligoahttit aibbas o\xC4""\x91""\xC4""\x91""a ar"
  "tihkkala muhtun f\xC3""\xA1""tt\xC3""\xA1""s. S\xC3""\xA1""megielat Wiki"
  "pedias leat d\xC3""\xA1""l 1,042 artihkkala. Wikipedia v\xC3""\xA1""ldos"
  "iiddus leat eambbo die\xC4""\x91""ut Mo don s\xC3""\xA1""ht\xC3""\xA1""t"
  " veahkehit Wikipedia d\xC3""\xA1""rbba\xC5""\xA1""a du veahkki! Rukses l"
  "i\xC5""\x8B""ka mearkka\xC5""\xA1""a dan, ahte f\xC3""\xA1""tt\xC3""\xA1"""
  "s ii leat vel \xC3""\xA1""lgg\xC3""\xA1""huvvon artihkal. Don s\xC3""\xA1"""
  "ht\xC3""\xA1""t veahkehit omd. nu ahte \xC4""\x8D""\xC3""\xA1""l\xC3""\xA1"""
  "t artihkkala, muhtun siidus\xC3""\xA1""valdagain. \xC3""\x81""lgg\xC3""\xA1"""
  "t o\xC4""\x91""\xC4""\x91""a artihkkala : Siida | Luossa | Durkkagiella "
  "| Deatnu | M\xC3""\xA1""notbadji | Euro | Skandin\xC3""\xA1""vala\xC5""\xA1"""
  " gielat | Rainier III | Ovttastuvvan N\xC3""\xA1""\xC5""\xA1""uvnnat | A"
  "rthur Miller | John F. Kennedy Lasi artihkals\xC3""\xA1""valdagat \xC3"""
  "\x81""igeguovdil Template:O\xC4""\x91""\xC4""\x91""asiiguin Loga lasi o\xC4"""
  "\x91""\xC4""\x91""asiid golggotm\xC3""\xA1""nu 13. beaivi Template:Golgg"
  "otm\xC3""\xA1""nu 13. Geah\xC4""\x8D""a ear\xC3""\xA1"" beivviid Dihtetg"
  "o, ahte... Template:Dihtetgo ahte Lasi unna die\xC4""\x91""ut Wikipediij"
  "a ear\xC3""\xA1"" gielain S\xC3""\xA1""meguovllu riikkaid gielain Girjed"
  "\xC3""\xA1""rog"
;

const char* kTeststr25 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/ru-sib_wikipedia_org_clean__KOI8R.txt
  ": \xE1""\xCD""\xC9""\xD4""\xC1""\xCE""\xCF""\xDA""\xCE""\xC1""\xCA""\xD3"""
  "\xD7""\xCF"" - \xE4""\xC1""\xCC""\xC1""\xC5""\xDA""\xCE""\xC1""\xCA""\xD3"""
  "\xD7""\xCF"" - \xE4""\xCF""\xCD""\xCF""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7"""
  "\xCF"" - \xF6""\xD9""\xD3""\xD4""\xC5""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7"""
  "\xCF"" - \xFA""\xD7""\xD8""\xCF""\xDA""\xC4""\xCF""\xDA""\xCE""\xC1""\xCA"""
  "\xD3""\xD7""\xCF"" - \xFA""\xC5""\xCD""\xC5""\xDA""\xCE""\xC1""\xCA""\xD3"""
  "\xD7""\xCF"" - \xED""\xC1""\xD4""\xC5""\xCD""\xC1""\xD4""\xC9""\xCB""\xC1"""
  " - \xED""\xC9""\xD2""\xCF""\xCB""\xCF""\xCC""\xCF""\xDA""\xCE""\xC1""\xCA"""
  "\xD3""\xD7""\xCF"" - \xF0""\xD2""\xC9""\xD2""\xCF""\xC4""\xCF""\xDA""\xCE"""
  "\xC1""\xCA""\xD3""\xD7""\xCF"" - \xF3""\xD4""\xC1""\xD4""\xC9""\xD3""\xD4"""
  "\xC9""\xCB""\xC1"" - \xF4""\xD2""\xC1""\xD7""\xCF""\xDA""\xCE""\xC1""\xCA"""
  "\xD3""\xD7""\xCF"" - \xE8""\xC9""\xCD""\xC9""\xD1"" \xED""\xD5""\xC4""\xD2"""
  "\xD8""\xCF""\xCE""\xCF""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7""\xCF"" \xC4"""
  "\xC1"" \xE1""\xD2""\xD4""\xC5""\xCC""\xD8""\xCE""\xD9"" \xD7""\xC5""\xC4"""
  "\xD9"": \xE1""\xD2""\xD4""\xC5""\xCC""\xC5""\xDA""\xCE""\xC1""\xCA""\xD3"""
  "\xD7""\xCF"" - \xE2""\xC1""\xDB""\xCC""\xD9""\xCB""\xCF""\xDA""\xCE""\xC1"""
  "\xCA""\xD3""\xD7""\xCF"" - \xF7""\xC5""\xD2""\xC1"" - \xF7""\xD8""\xCF"""
  "\xD4""\xCF""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7""\xCF"" - \xE7""\xCF""\xD7"""
  "\xCF""\xD2""\xCF""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7""\xCF"" - \xE4""\xC5"""
  "\xD1""\xCE""\xCE""\xC9"" - \xE4""\xCF""\xCD""\xCF""\xD7""\xDB""\xD9""\xCE"""
  "\xC1"" - \xE4""\xD5""\xDB""\xC5""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7""\xCF"""
  " - \xFA""\xC5""\xCD""\xC5""\xD0""\xC9""\xD3""\xC1""\xCE""\xCE""\xC5"" - "
  "\xEB""\xCF""\xCE""\xCF""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7""\xCF"" - \xEC"""
  "\xC1""\xC4"" - \xEC""\xC0""\xC4""\xC5""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7"""
  "\xCF"" - \xEC""\xC0""\xC4""\xCE""\xC9""\xCB""\xC9"" - \xED""\xD5""\xC4"""
  "\xD2""\xD8""\xCF""\xCE""\xCF""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7""\xCF"""
  " - \xEE""\xC1""\xD2""\xCF""\xC4""\xCF""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7"""
  "\xCF"" - \xF0""\xD2""\xC1""\xD7""\xC9""\xCC""\xD8""\xCE""\xD1"" - \xF0"""
  "\xD2""\xCF""\xCD""\xD9""\xD3""\xC5""\xCC"" \xF2""\xCF""\xC2""\xCF""\xD4"""
  "\xCE""\xD9"" \xD7""\xC5""\xC4""\xD9"": \xF7""\xC5""\xD3""\xD3""\xC5""\xDA"""
  "\xCE""\xC1""\xCA""\xD3""\xD7""\xCF"" - \xF7""\xD2""\xC1""\xDE""\xC5""\xC2"""
  "\xD3""\xD7""\xCF"" - \xE4""\xC5""\xD2""\xD8""\xCF""\xD7""\xC5""\xCE""\xDB"""
  "\xD9""\xCE""\xC1"" - \xED""\xC5""\xD6""\xD5""\xC7""\xC9""\xCD""\xC7""\xC1"""
  " (\xE9""\xCE""\xD4""\xC5""\xD2""\xCE""\xC5""\xD4"") - \xEE""\xC1""\xDE"""
  "\xCF""\xD4""\xCE""\xC9""\xDE""\xC5""\xD3""\xD7""\xCF"" - \xF0""\xCF""\xD7"""
  "\xD1""\xDA""\xCB""\xC9"" - \xF2""\xC1""\xC4""\xC9""\xD7""\xCF"" - \xF4"""
  "\xC5""\xC8""\xCE""\xCF""\xDA""\xCE""\xC1""\xCA""\xD3""\xD7""\xCF"" - \xF4"""
  "\xCF""\xD2""\xC7""\xCF""\xD7""\xCC""\xD1"" - \xF5""\xD3""\xD4""\xD2""\xCF"""
  "\xCA""\xDB""\xD9""\xD3""\xD7""\xCF"" \xF5""\xDA""\xCF""\xD2""\xCF""\xDE"""
  "\xDE""\xC5"" \xC4""\xC1"" \xF6""\xD9""\xDA""\xCE""\xC5""\xD7""\xD2""\xD1"""
  "\xC4"": \xE2""\xC1""\xD3""\xCE""\xC5""\xD0""\xC9""\xD3""\xC1""\xCE""\xCE"""
  "\xC5"" - \xE7""\xD5""\xCC""\xD8""\xC2""\xC1"" - \xFA""\xCF""\xC4""\xDE"""
  "\xC5""\xD3""\xD7""\xCF"" - \xEB""\xD2""\xC1""\xD3""\xCE""\xCF""\xD3""\xCC"""
  "\xCF""\xD7""\xD7""\xC5"" - \xEC""\xC5""\xD0""\xCF""\xCC""\xC1""\xC4"" - "
  "\xEE""\xC1""\xD7""\xCF""\xC2""\xD9""\xCB""\xC9"" - \xF0""\xC1""\xD7""\xCE"""
  "\xC9""\xC3""\xC1"" - \xF0""\xCC""\xD1""\xD3""\xCF""\xDE""\xCB""\xC9"" - "
  "\xF0""\xCF""\xDA""\xD9""\xD2""\xC9""\xDB""\xDB""\xC5"" - \xF0""\xCF""\xD3"""
  "\xCB""\xD5""\xCC""\xCB""\xC9"" - \xF5""\xDA""\xCF""\xD2""\xCF""\xDE""\xDE"""
  "\xC5"" \xE4""\xC5""\xD1""\xCE""\xD3""\xCB""\xC1"" \xDE""\xC5""\xD2""\xC5"""
  "\xC4""\xC1"" \xC7""\xCF""\xC4""\xCF""\xD7"" - \xF0""\xC5""\xD3""\xD4""\xC5"""
  "\xC7""\xC1"" \xC4""\xC5""\xD1""\xCE""\xD3""\xCB""\xC9""\xC8"" \xC4""\xC1"""
  "\xCE""\xCF""\xCB"" - \xF0""\xC5""\xD3""\xD4""\xC5""\xC7""\xC1"" \xCB""\xD2"""
  "\xC1""\xCA""\xCF""\xD7"" \xF2""\xCF""\xC2""\xCF""\xD4""\xC1"" \xD3""\xC5"""
  "\xC4""\xD8""\xCD""\xC9""\xC3""\xD9"" \xEE""\xC1"" \xC5""\xD4""\xCF""\xCA"""
  " \xD3""\xC5""\xC4""\xD8""\xCD""\xC9""\xC3""\xC5"" \xD2""\xCF""\xC2""\xC9"""
  "\xCD"" \xE2""\xC9""\xC2""\xCC""\xC9""\xC0"" \xEE""\xCF""\xD7""\xCF""\xD3"""
  "\xC9"" 9 \xD3""\xC5""\xD2""\xD8""\xD0""\xCE""\xD1"" \xE9""\xDA""\xC4""\xC5"""
  "\xD1""\xCE""\xC1"" \xCE""\xCF""\xD7""\xC1"" \xCB""\xD2""\xC1""\xD3""\xCF"""
  "\xD4""\xC9""\xDB""\xDB""\xC1"" \xC7""\xCF""\xCC""\xCF""\xD7""\xCE""\xCF"""
  "\xCA"""
;

const char* kTeststr26 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/bg_wikipedia_org_clean__CP1251.txt
  " \x96"" \xD4""\xE8""\xEB""\xEE""\xF1""\xEE""\xF4""\xE8""\xFF"" \x96"" \xEE"""
  "\xF9""\xE5""... \xD2""\xE5""\xF5""\xED""\xE8""\xF7""\xE5""\xF1""\xEA""\xE8"""
  " \xE8"" \xEF""\xF0""\xE8""\xEB""\xEE""\xE6""\xED""\xE8"" \xEE""\xE1""\xEB"""
  "\xE0""\xF1""\xF2""\xE8"" \xC7""\xE5""\xEC""\xE5""\xE4""\xE5""\xEB""\xE8"""
  "\xE5"" \x96"" \xC8""\xED""\xF2""\xE5""\xF0""\xED""\xE5""\xF2"" \x96"" \xCA"""
  "\xEE""\xEC""\xF3""\xED""\xE8""\xEA""\xE0""\xF6""\xE8""\xE8"" \x96"" \xCC"""
  "\xE5""\xE4""\xE8""\xF6""\xE8""\xED""\xE0"" \x96"" \xCE""\xE1""\xF0""\xE0"""
  "\xE7""\xEE""\xE2""\xE0""\xED""\xE8""\xE5"" \x96"" \xCF""\xF0""\xE0""\xE2"""
  "\xEE"" \x96"" \xCF""\xF0""\xEE""\xEC""\xE8""\xF8""\xEB""\xE5""\xED""\xEE"""
  "\xF1""\xF2"" \x96"" \xD1""\xF2""\xF0""\xEE""\xE8""\xF2""\xE5""\xEB""\xF1"""
  "\xF2""\xE2""\xEE"" \x96"" \xD2""\xE5""\xF5""\xED""\xE8""\xEA""\xE0"" \x96"""
  " \xD2""\xF0""\xE0""\xED""\xF1""\xEF""\xEE""\xF0""\xF2"" \x96"" \xEE""\xF9"""
  "\xE5""... \xCA""\xF3""\xEB""\xF2""\xF3""\xF0""\xE0"" \xE8"" \xE8""\xE7"""
  "\xEA""\xF3""\xF1""\xF2""\xE2""\xEE"" \xCF""\xEE"" \xF1""\xE2""\xE5""\xF2"""
  "\xE0"" \x96"" \xCB""\xE8""\xF2""\xE5""\xF0""\xE0""\xF2""\xF3""\xF0""\xE0"""
  " \x96"" \xCA""\xE8""\xED""\xEE"" \x96"" \xCC""\xF3""\xE7""\xE8""\xEA""\xE0"""
  " \x96"" \xCC""\xE5""\xE4""\xE8""\xE8"" \x96"" \xC0""\xF0""\xF5""\xE8""\xF2"""
  "\xE5""\xEA""\xF2""\xF3""\xF0""\xE0"" \x96"" \xC6""\xE8""\xE2""\xEE""\xEF"""
  "\xE8""\xF1"" \x96"" \xD2""\xE5""\xE0""\xF2""\xFA""\xF0"" \x96"" \xD0""\xE5"""
  "\xEB""\xE8""\xE3""\xE8""\xFF"" \x96"" \xEE""\xF9""\xE5""... \xCE""\xE1"""
  "\xF9""\xE5""\xF1""\xF2""\xE2""\xEE"" \xCE""\xF0""\xE3""\xE0""\xED""\xE8"""
  "\xE7""\xE0""\xF6""\xE8""\xE8"" \x96"" \xCA""\xF3""\xEB""\xF2""\xF3""\xF0"""
  "\xE0"" \x96"" \xD0""\xE5""\xEB""\xE8""\xE3""\xE8""\xFF"" \x96"" \xCF""\xF0"""
  "\xE5""\xE4""\xF0""\xE0""\xE7""\xF1""\xFA""\xE4""\xFA""\xF6""\xE8"" \x96"""
  " \xD1""\xE8""\xE3""\xF3""\xF0""\xED""\xEE""\xF1""\xF2"" \x96"" \xCF""\xEE"""
  "\xEB""\xE8""\xF2""\xE8""\xEA""\xE0"" \x96"" \xCD""\xE0""\xF3""\xEA""\xE0"""
  " \x96"" \xEE""\xF9""\xE5""... \xC6""\xE8""\xE2""\xEE""\xF2"" \xE8"" \xE7"""
  "\xE0""\xE1""\xE0""\xE2""\xEB""\xE5""\xED""\xE8""\xFF"" \xC3""\xEE""\xF2"""
  "\xE2""\xE0""\xF0""\xF1""\xF2""\xE2""\xEE"" \x96"" \xC7""\xE4""\xF0""\xE0"""
  "\xE2""\xE5"" \x96"" \xC8""\xE3""\xF0""\xE8"" \x96"" \xD1""\xE5""\xEA""\xF1"""
  "\xF3""\xE0""\xEB""\xED""\xEE""\xF1""\xF2"" \x96"" \xD1""\xEF""\xEE""\xF0"""
  "\xF2"" \x96"" \xD2""\xF3""\xF0""\xE8""\xE7""\xFA""\xEC"" \x96"" \xEE""\xF9"""
  "\xE5""... \xC4""\xF0""\xF3""\xE3""\xE8"" \xEA""\xE0""\xF2""\xE5""\xE3""\xEE"""
  "\xF0""\xE8""\xE8"" \xC2""\xF1""\xE8""\xF7""\xEA""\xE8"" \xF1""\xF2""\xE0"""
  "\xF2""\xE8""\xE8"" \x96"" \xCD""\xEE""\xE2""\xE8"" \xF1""\xF2""\xE0""\xF2"""
  "\xE8""\xE8"" \x96"" \xD1""\xEF""\xE8""\xF1""\xFA""\xEA"" \xED""\xE0"" \xF1"""
  "\xEF""\xE8""\xF1""\xFA""\xF6""\xE8""\xF2""\xE5"" \x96"" \xD1""\xEF""\xE8"""
  "\xF1""\xFA""\xEA"" \xED""\xE0"" \xF1""\xF2""\xF0""\xE0""\xED""\xE8""\xF2"""
  "\xE5"" \xEF""\xEE"" \xF1""\xE2""\xE5""\xF2""\xE0"" \x96"" \xCA""\xE0""\xEB"""
  "\xE5""\xED""\xE4""\xE0""\xF0"" \x96"" \xC1""\xE8""\xEE""\xE3""\xF0""\xE0"""
  "\xF4""\xE8""\xE8"" \x96"" \xCA""\xE0""\xF0""\xF2""\xE8""\xED""\xEA""\xE8"""
  " \xCF""\xF0""\xE5""\xE3""\xEB""\xE5""\xE4"" \xED""\xE0"" \xF1""\xF2""\xE0"""
  "\xF2""\xE8""\xE8""\xF2""\xE5"": \xEF""\xEE"" \xE0""\xE7""\xE1""\xF3""\xF7"""
  "\xE5""\xED"" \xF0""\xE5""\xE4"" | \xEF""\xEE"" \xEA""\xE0""\xF2""\xE5""\xE3"""
  "\xEE""\xF0""\xE8""\xE8"" | \xEF""\xEE""\xF0""\xF2""\xE0""\xEB""\xED""\xE8"""
  " \xF1""\xF2""\xF0""\xE0""\xED""\xE8""\xF6""\xE8"" \xD3""\xE8""\xEA""\xE8"""
  "\xEF""\xE5""\xE4""\xE8""\xFF"" \xED""\xE0"" \xE4""\xF0""\xF3""\xE3""\xE8"""
  " \xE5""\xE7""\xE8""\xF6""\xE8"" \xCF""\xEE""\xEC""\xEE""\xF9"" \xCF""\xEE"""
  "\xEC""\xEE""\xF9"": \xC2""\xFA""\xE2""\xE5""\xE4""\xE5""\xED""\xE8""\xE5"""
  " \x96"" \xCD""\xE0""\xF0""\xFA""\xF7""\xED""\xE8""\xEA"" \x96"" \xCF""\xFF"""
  "\xF1""\xFA""\xF7""\xED""\xE8""\xEA"" \x96"" \xC1""\xFA""\xF0""\xE7""\xE0"""
  " \xEF""\xEE""\xEC""\xEE""\xF9"" \x96"" \xCD""\xE0""\xF1""\xF2""\xF0""\xEE"""
  "\xE9""\xEA""\xE8"" \x96"" \xCF""\xEE""\xEB""\xE5""\xE7""\xED""\xE8"" \xF1"""
  "\xFA""\xE2""\xE5""\xF2""\xE8"" \x96"" \xCA""\xE0""\xEA"" \xF1""\xE5"" \xF0"""
  "\xE5""\xE4""\xE0""\xEA""\xF2""\xE8""\xF0""\xE0"" \x96"" \xD7""\xC7""\xC2"""
  " \x96"" \xD0""\xE0""\xE7""\xE3""\xEE""\xE2""\xEE""\xF0""\xE8"" \x96"" \xEE"""
  "\xF9""\xE5"" \xBB"" \xCF""\xEE""\xF0""\xF2""\xE0""\xEB""\xED""\xE8"" \xF1"""
  "\xF2""\xF0""\xE0""\xED""\xE8""\xF6""\xE8"": \xC0""\xF1""\xF2""\xF0""\xEE"""
  "\xED""\xEE""\xEC""\xE8""\xFF"" \x96"" \xC1""\xE8""\xEE""\xEB""\xEE""\xE3"""
  "\xE8""\xFF"" \x96"" \xC5""\xE3""\xE8""\xEF""\xF2""\xEE""\xEF""\xE5""\xE4"""
  "\xE8""\xFF"" \x96"" \xCB""\xE8""\xF2""\xE5""\xF0""\xE0""\xF2""\xF3""\xF0"""
  "\xE0"" \x96"" \xD1""\xE5""\xEA""\xF1""\xF3""\xE0""\xEB""\xED""\xEE""\xF1"""
  "\xF2"" \x96"" \xD1""\xEB""\xE0""\xE2""\xFF""\xED""\xE8"" \x96"" \xD1""\xF2"""
  "\xF0""\xE0""\xED""\xE8""\xF2""\xE5"" \xE2"" \xF1""\xE2""\xE5""\xF2""\xE0"""
  " \x96"" \xD4""\xE5""\xED""\xF2""\xFA""\xE7""\xE8"" \x96"" \xD4""\xE8""\xEB"""
  "\xEC""\xE8"" \x96"" \xD4""\xEE""\xF0""\xEC""\xF3""\xEB""\xE0"" 1 \x96"" "
  "\xEE""\xF9""\xE5"" \xBB"" \xC8""\xE7""\xE1""\xF0""\xE0""\xED""\xE0"" \xF1"""
  "\xF2""\xE0""\xF2""\xE8""\xFF"" \xCC""\xE5""\xE6""\xE4""\xF3""\xED""\xE0"""
  "\xF0""\xEE""\xE4""\xED""\xE0""\xF2""\xE0"" \xF1""\xE8""\xF1""\xF2""\xE5"""
  "\xEC""\xE0"" \xE5""\xE4""\xE8""\xED""\xE8""\xF6""\xE8"", \xF1""\xFA""\xEA"""
  "\xF0""\xE0""\xF9""\xE0""\xE2""\xE0""\xED""\xE0"" SI, \xE5"" \xED""\xE0"""
  "\xE9""-\xF8""\xE8""\xF0""\xEE""\xEA""\xEE"" \xE8""\xE7""\xEF""\xEE""\xEB"""
  "\xE7""\xE2""\xE0""\xED""\xE0""\xF2""\xE0"" \xF1""\xE8""\xF1""\xF2""\xE5"""
  "\xEC""\xE0"" \xEE""\xF2"" \xE5""\xE4""\xE8""\xED""\xE8""\xF6""\xE8"" \xE2"""
  " \xF6""\xE5""\xEB""\xE8""\xFF"" \xF1""\xE2""\xFF""\xF2"". \xC7""\xE0""\xE5"""
  "\xE4""\xED""\xEE"" \xF1"" \xEF""\xEE""-\xF1""\xF2""\xE0""\xF0""\xE0""\xF2"""
  "\xE0"" \xF1""\xE8""\xF1""\xF2""\xE5""\xEC""\xE0"" \xF1""\xE0""\xED""\xF2"""
  "\xE8""\xEC""\xE5""\xF2""\xFA""\xF0""-\xE3""\xF0""\xE0""\xEC""-\xF1""\xE5"""
  "\xEA""\xF3""\xED""\xE4""\xE0"", SI \xE5"" \xED""\xE0""\xF0""\xE8""\xF7"""
  "\xE0""\xED""\xE0"" \xEF""\xEE""\xED""\xFF""\xEA""\xEE""\xE3""\xE0"" \"\xEC"""
  "\xE5""\xF2""\xF0""\xE8""\xF7""\xED""\xE0"" \xF1""\xE8""\xF1""\xF2""\xE5"""
  "\xEC""\xE0""\". \xC5""\xE4""\xE8""\xED""\xE8""\xF6""\xE8""\xF2""\xE5"" \xED"""
  "\xE0"" \xF1""\xE8""\xF1""\xF2""\xE5""\xEC""\xE0""\xF2""\xE0"" SI \xF1""\xE0"""
  " \xEE""\xEF""\xF0""\xE5""\xE4""\xE5""\xEB""\xE5""\xED""\xE8"" \xED""\xE0"""
  " \xEC""\xE5""\xE6""\xE4""\xF3""\xED""\xE0""\xF0""\xEE""\xE4""\xED""\xE8"""
  " \xEA""\xEE""\xED""\xF4""\xE5""\xF0""\xE5""\xED""\xF6""\xE8""\xE8"", \xEE"""
  "\xF0""\xE3""\xE0""\xED""\xE8""\xE7""\xE8""\xF0""\xE0""\xED""\xE8"" \xEE"""
  "\xF2"" \xCC""\xE5""\xE6""\xE4""\xF3""\xED""\xE0""\xF0""\xEE""\xE4""\xED"""
  "\xEE""\xF2""\xEE"" \xE1""\xFE""\xF0""\xEE"" \xE7""\xE0"" \xEC""\xE5""\xF0"""
  "\xEA""\xE8"" \xE8"" \xF2""\xE5""\xE3""\xEB""\xE8""\xEB""\xEA""\xE8"" (Bu"
  "reau International des Poids et Mesures, BIPM). \xC8""\xEC""\xE5""\xF2"""
  "\xEE"" \xED""\xE0"" \xF1""\xE8""\xF1""\xF2""\xE5""\xEC""\xE0""\xF2""\xE0"""
  " \xE5"" \xE4""\xE0""\xE4""\xE5""\xED""\xEE"" \xEF""\xF0""\xE5""\xE7"" 19"
  "60 \xE3"". \xCC""\xE5""\xE6""\xE4""\xF3""\xED""\xE0""\xF0""\xEE""\xE4""\xED"""
  "\xE0""\xF2""\xE0"" \xF1""\xE8""\xF1""\xF2""\xE5""\xEC""\xE0"" SI \xE5"" "
  "\xE8""\xE7""\xE3""\xF0""\xE0""\xE4""\xE5""\xED""\xE0"" \xE2""\xFA""\xF0"""
  "\xF5""\xF3"" \xF1""\xE5""\xE4""\xE5""\xEC"" \xEE""\xF1""\xED""\xEE""\xE2"""
  "\xED""\xE8"" \xE5""\xE4""\xE8""\xED""\xE8""\xF6""\xE8"" \xEA""\xE0""\xF2"""
  "\xEE"" \xEA""\xE8""\xEB""\xEE""\xE3""\xF0""\xE0""\xEC""\xE0"" \xE8"" \xEC"""
  "\xE5""\xF2""\xFA""\xF0""\xE0"". \xD7""\xF0""\xE5""\xE7"" \xF2""\xFF""\xF5"""
  " \xF1""\xE5"" \xEE""\xEF""\xF0""\xE5""\xE4""\xE5""\xEB""\xFF""\xF2"" \xF0"""
  "\xE0""\xE7""\xEB""\xE8""\xF7""\xED""\xE8"" \xEF""\xF0""\xEE""\xE8""\xE7"""
  "\xE2""\xEE""\xE4""\xED""\xE8"" \xE5""\xE4""\xE8""\xED""\xE8""\xF6""\xE8"""
  ". \xD1""\xE8""\xF1""\xF2""\xE5""\xEC""\xE0""\xF2""\xE0"" \xF1""\xFA""\xF9"""
  "\xEE"" \xEE""\xEF""\xF0""\xE5""\xE4""\xE5""\xEB""\xFF"" \xED""\xFF""\xEA"""
  "\xEE""\xE8"" \xEF""\xF0""\xE5""\xE4""\xF1""\xF2""\xE0""\xE2""\xEA""\xE8"""
  ", \xF7""\xF0""\xE5""\xE7"" \xEA""\xEE""\xE8""\xF2""\xEE"" \xF1""\xE5"" \xEE"""
  "\xE1""\xF0""\xE0""\xE7""\xF3""\xE2""\xE0""\xF2"" \xEA""\xF0""\xE0""\xF2"""
  "\xED""\xE8"" \xE8""\xEB""\xE8"" \xE4""\xF0""\xEE""\xE1""\xED""\xE8"" \xED"""
  "\xE0"" \xE5""\xE4""\xE8""\xED""\xE8""\xF6""\xE8""\xF2""\xE5""."
;

const char* kTeststr27 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/de_wikipedia_org_clean__CP1252.txt
  " auf der Computermesse Systems in M\xFC""nchen vertreten (Halle A3.542)."
  " Artikel des Tages Das Weinbaugebiet Bordeaux, auf franz\xF6""sisch Bord"
  "elais, ist das gr\xF6""\xDF""te zusammenh\xE4""ngende Anbaugebiet der We"
  "lt f\xFC""r Qualit\xE4""tswein. Es gibt etwa 4000 Ch\xE2""teaux genannte"
  " Weing\xFC""ter, die die weltber\xFC""hmten Weine erzeugen. Ein differen"
  "ziertes System von subregionalen und kommunalen Appellationen und Klassi"
  "fikationen schafft unter ihnen eine qualitative Hierarchie. Die einzelne"
  "n Lagen spielen demgegen\xFC""ber eine untergeordnete Rolle. Ihre Stelle"
  " nimmt das Ch\xE2""teau ein, zu dem sie geh\xF6""ren. Im Jahr 2002 wurde"
  "n auf gut 120.000 Hektar Anbaufl\xE4""che insgesamt 5,74 Millionen Hekto"
  "liter Qualit\xE4""tswein erzeugt. mehr Fr\xFC""here Artikel des Tages \xB7"""
  " Weitere exzellente Artikel Was geschah am 13. Oktober? 1781 \x96"" Jose"
  "phinismus: Mit seinem Toleranzpatent hebt Joseph II. im Erzherzogtum \xD6"""
  "sterreich das Glaubensmonopol der Katholischen Kirche auf. 1806 \x96"" I"
  "n Berlin wird die Preu\xDF""isch-K\xF6""nigliche Blindenanstalt unter Le"
  "itung von Johann August Zeune er\xF6""ffnet \x96"" Deutschlands erste Bl"
  "indenschule. 1806 \x96"" Otto Unverdorben, der 1826 Anilin, ein fl\xFC"""
  "ssiges Zersetzungsprodukt von Indigo, entdecken wird, wird geboren. 1906"
  " \x96"" Karl Holzamer, Gr\xFC""ndungsintendant des ZDF, wird"
;

const char* kTeststr28 =
// http://www.mon.gov.ua/books/ann.php?id=193
// Produced by stringify.cc on 2007-04-10 14:21 from file koi8u.html
  "<html>\n<head>\n <meta http-equiv=\"Content-Type\" content=\"text/html; "
  "charset=koi8-u\">\n <title>\xE1""\xCE""\xCF""\xD4""\xC1""\xC3""\xA6""\xD1"""
  " | \xED""i\xCE""i\xD3""\xD4""\xC5""\xD2""\xD3""\xD4""\xD7""\xCF"" \xCF"""
  "\xD3""\xD7""i\xD4""\xC9"" \xA6"" \xCE""\xC1""\xD5""\xCB""\xC9"" \xF5""\xCB"""
  "\xD2""\xC1""\xA7""\xCE""\xC9""</title>\n<head>\n<body>\n\xF0""\xCF""\xCC"""
  "\xA6""\xDD""\xD5""\xCB"" \xF7"". \xF0"".<br><b>\xE2""\xC4""\xD6""\xA6""\xCC"""
  "\xD8""\xCE""\xC9""\xC3""\xD4""\xD7""\xCF""</b><br><a\nonClick=\"window.o"
  "pen('/books/pub.php?id=5',,\n'directories=no,height=200,location=no,menu"
  "bar=no,resizable=no,scrollbars=yes,status=no,toolbar=no,width=350');retu"
  "rn false;\\\ntarget=\"_blank\" href=\"/books/pub.php?id=5\">\xF7""\xC9"""
  "\xC4""\xC1""\xD7""\xCE""\xC9""\xC3""\xD4""\xD7""\xCF"" \"\xF7""\xC9""\xDD"""
  "\xC1"" \xDB""\xCB""\xCF""\xCC""\xC1""\"</a>, 2001 - 287c.<br><br>\xF7""\xC9"""
  "\xD3""\xD7""\xA6""\xD4""\xCC""\xC5""\xCE""\xCF"" \xC7""\xCF""\xD3""\xD0"""
  "\xCF""\xC4""\xC1""\xD2""\xD3""\xD8""\xCB""\xC5"" \xDA""\xCE""\xC1""\xDE"""
  "\xC5""\xCE""\xCE""\xD1"" \xC2""\xC4""\xD6""\xA6""\xCC""\xD8""\xCE""\xC9"""
  "\xC3""\xD4""\xD7""\xC1"", \xCA""\xCF""\xC7""\xCF"" \xD2""\xCF""\xDA""\xD7"""
  "\xC9""\xD4""\xCF""\xCB"" \xA6"" \xD3""\xD5""\xDE""\xC1""\xD3""\xCE""\xC9"""
  "\xCA"" \xD3""\xD4""\xC1""\xCE"", \xC2""\xA6""\xCF""\xCC""\xCF""\xC7""\xA6"""
  "\xC0"" \xC2""\xC4""\xD6""\xCF""\xCC""\xC9""\xCE""\xCF""\xA7"" \xD3""\xA6"""
  "\xCD""'\xA7"", \xD0""\xC9""\xD4""\xC1""\xCE""\xCE""\xD1"" \xC7""\xCF""\xC4"""
  "\xA6""\xD7""\xCC""\xA6"", \xD5""\xD4""\xD2""\xC9""\xCD""\xC1""\xCE""\xCE"""
  "\xD1"" \xA6"" \xD2""\xCF""\xDA""\xD7""\xC5""\xC4""\xC5""\xCE""\xCE""\xD1"""
  " \xC2""\xC4""\xD6""\xA6""\xCC"". \xEF""\xD0""\xC9""\xD3""\xC1""\xCE""\xCF"""
  " \xD4""\xC9""\xD0""\xC9"" \xD7""\xD5""\xCC""\xC9""\xCB""\xA6""\xD7"" \xA6"""
  " \xD0""\xC1""\xD3""\xA6""\xDE""\xCE""\xC9""\xC8"" \xC2""\xD5""\xC4""\xA6"""
  "\xD7""\xC5""\xCC""\xD8"", \xD0""\xC1""\xD3""\xA6""\xDE""\xCE""\xC9""\xC3"""
  "\xD8""\xCB""\xC9""\xCA"" \xA6""\xCE""\xD7""\xC5""\xCE""\xD4""\xC1""\xD2"""
  " \xD4""\xC1"" \xCD""\xC5""\xC8""\xC1""\xCE""\xA6""\xDA""\xC1""\xC3""\xA6"""
  "\xC0"" \xD7""\xC9""\xD2""\xCF""\xC2""\xCE""\xC9""\xDE""\xC9""\xC8"" \xD0"""
  "\xD2""\xCF""\xC3""\xC5""\xD3""\xA6""\xD7"", \xCD""\xC5""\xC4""\xCF""\xCE"""
  "\xCF""\xD3""\xCE""\xD5"" \xC2""\xC1""\xDA""\xD5"" \xA6"" \xDA""\xC1""\xD0"""
  "\xC9""\xCC""\xC5""\xCE""\xCE""\xD1"" \xC5""\xCE""\xD4""\xCF""\xCD""\xCF"""
  "\xC6""\xA6""\xCC""\xD8""\xCE""\xC9""\xC8"" \xCB""\xD5""\xCC""\xD8""\xD4"""
  "\xD5""\xD2"", \xD4""\xC5""\xC8""\xCE""\xCF""\xCC""\xCF""\xC7""\xA6""\xC0"""
  " \xD7""\xC9""\xD2""\xCF""\xC2""\xCE""\xC9""\xC3""\xD4""\xD7""\xC1"" \xD0"""
  "\xD2""\xCF""\xC4""\xD5""\xCB""\xD4""\xA6""\xD7"" \xC2""\xC4""\xD6""\xA6"""
  "\xCC""\xD8""\xCE""\xC9""\xC3""\xD4""\xD7""\xC1"", \xC8""\xD7""\xCF""\xD2"""
  "\xCF""\xC2""\xC9"" \xA6"" \xDB""\xCB""\xA6""\xC4""\xCE""\xC9""\xCB""\xC9"""
  " \xC2""\xC4""\xD6""\xA6""\xCC"", \xC5""\xCB""\xCF""\xCE""\xCF""\xCD""\xA6"""
  "\xCB""\xD5"" \xD4""\xC1"" \xCF""\xD2""\xC7""\xC1""\xCE""\xA6""\xDA""\xC1"""
  "\xC3""\xA6""\xC0"" \xC7""\xC1""\xCC""\xD5""\xDA""\xA6"", \xCE""\xC1""\xD7"""
  "\xC5""\xC4""\xC5""\xCE""\xCF"" \xCC""\xC1""\xC2""\xCF""\xD2""\xC1""\xD4"""
  "\xCF""\xD2""\xCE""\xCF""-\xD0""\xD2""\xC1""\xCB""\xD4""\xC9""\xDE""\xCE"""
  "\xA6"" \xD2""\xCF""\xC2""\xCF""\xD4""\xC9"".\r\n\xE4""\xCC""\xD1"" \xD5"""
  "\xDE""\xCE""\xA6""\xD7"" \xD0""\xD2""\xCF""\xC6""\xC5""\xD3""\xA6""\xCA"""
  "\xCE""\xCF""-\xD4""\xC5""\xC8""\xCE""\xA6""\xDE""\xCE""\xC9""\xC8"" \xCE"""
  "\xC1""\xD7""\xDE""\xC1""\xCC""\xD8""\xCE""\xC9""\xC8"" \xDA""\xC1""\xCB"""
  "\xCC""\xC1""\xC4""\xA6""\xD7"".<p style=\"text-align: right; font-weight"
  ": bold;\">18\n\xC7""\xD2""\xCE"".</p></body>\n</html>"
;

const char* kTeststr29 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/pl_wikipedia_org_clean__CP1250.txt
  "kroekonomicznej\". ( Wikinews) Bie\xBF""\xB9""ce wydarzenia: w Wikipedii"
  ", w Wikinews Aktualno\x9C""ci sportowe: w Wikipedii, w Wikinews Czy wies"
  "z, \xBF""e... Z najnowszych artyku\xB3""\xF3""w Wikipedii: ... zdobywca "
  "pierwszego polskiego medalu olimpijskiego Adam Kr\xF3""likiewicz zmar\xB3"""
  " po wypadku na planie filmu Andrzeja Wajdy Popio\xB3""y? ... wydany w 19"
  "99 album Tori Amos To Venus and Back by\xB3"" pierwszym albumem tej arty"
  "stki od 1992, kt\xF3""ry nie dosta\xB3"" si\xEA"" do pierwszej dziesi\xB9"""
  "tki na brytyjskiej li\x9C""cie najpopularniejszych p\xB3""yt? ... emisja"
  " audycji radiowej W Jezioranach rozpocz\xEA""\xB3""a si\xEA"" w 1960? .."
  ". w czasie bitwy w Sundzie holenderski 28-dzia\xB3""owy okr\xEA""t Breda"
  " by\xB3"" zdobyty dwukrotnie? ... tramwaje Tatra T6A2 u\xBF""ywane s\xB9"""
  " mi\xEA""dzy innymi w w\xEA""gierskim mie\x9C""cie Szeged? Archiwum | Na"
  "pisz nowy artyku\xB3"" Wybrane rocznice 12 pa\x9F""dziernika: imieniny o"
  "bchodz\xB9"" Maksymilian, Serafin, Witolda 1492 - Krzysztof Kolumb dop\xB3"""
  "yn\xB9""\xB3"" do archipelagu Baham\xF3""w; oficjalna data odkrycia Amer"
  "yki 1891 - urodzi\xB3""a si\xEA"" Edyta Stein, znana jako \x9C""w. Teres"
  "a Benedykta od Krzy\xBF""a, filozof, \x9C""wi\xEA""ta Ko\x9C""cio\xB3""a"
  " rzymskokatolickiego 1920 - wojna polsko-bolszewicka: podpisanie rozejmu"
  "; gen. Lucjan \xAF""eligowski zaj\xB9""\xB3"" Wilno i proklamowa\xB3"" u"
  "tworzenie Litwy \x8C""rodkowej. 1924 - zmar\xB3"" Anatole France, pisarz"
  " francuski 1943 - rozpocz\xEA""\xB3""a si\xEA"" bitwa pod Lenino Inne ro"
  "cznice: 11 pa\x9F""dziernika - 12 pa\x9F""dziernika - 13 pa\x9F""dzierni"
  "ka Archiwum | Kalendarium Artyku\xB3"" na medal Wielka w\xEA""dr\xF3""wk"
  "a lud\xF3""w \x96"" okres masowych migracji plemion barbarzy\xF1""skich,"
  " w szczeg\xF3""lno\x9C""ci Hun\xF3""w i German\xF3""w, na ziemie cesarst"
  "wa rzymskiego u schy\xB3""ku staro\xBF""ytno\x9C""ci i w pocz\xB9""tkach"
  " \x9C""redniowiecza (IV-VI wiek). Proces ten radykalnie odmieni\xB3"" ob"
  "raz kontynentu europejskiego \x96"" doprowadzi\xB3"" do zmian etnicznych"
  " na du\xBF""ych obszarach, wyz"
;

const char* kTeststr32 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/lv_wikipedia_org_clean__CP1257.txt
  " neaizvietojamu materi\xE2""lu ar \xEF""oti pla\xF0""u pielietojumu. Dim"
  "ants ir tradicion\xE2""li visv\xE7""rt\xEE""g\xE2""kais d\xE2""rgakmens."
  " Izcil\xE2""s fizik\xE2""l\xE2""s \xEE""pa\xF0""\xEE""bas padara dimantu"
  " par lielisku abraz\xEE""vu \x96"" vien\xEE""g\xE2""s vielas, ar kur\xE2"""
  "m var ieskr\xE2""p\xE7""t dimantu, ir cits dimants, borazons, ultracieta"
  "is fuller\xEE""ts vai dimanta nanostien\xEE""\xF0""i. Tas noz\xEE""m\xE7"""
  " ar\xEE"" to, ka dimants lieliski saglab\xE2"" virsmas pul\xE7""jumu \x96"""
  " savu izcilo sp\xEE""dumu. Las\xEE""t vair\xE2""k... Iepriek\xF0""\xE7"""
  "jie raksti: Bizantijas imp\xE7""rija, Delartisk\xE2"" kom\xE7""dija, Iro"
  "nija, Naudas v\xE7""sture, 1905. gada revol\xFB""cija Vai tu zin\xE2""ji"
  "... No latvie\xF0""u Vikip\xE7""dijas jaun\xE2""kajiem rakstiem: ... ka "
  "viens no liel\xE2""kajiem v\xE2""rsm\xE2""s rakst\xEE""tajiem darbiem li"
  "terat\xFB""ras v\xE7""stur\xE7"" ir senindie\xF0""u eposs \"Mah\xE2""bh\xE2"""
  "rata\"? ... ka v\xE2""rda \"ebrejs\" v\xE7""sturisk\xE2"" noz\xEE""me ir"
  " \"tie, kas klejo no vietas uz vietu\"? ... ka feni\xED""ie\xF0""i bija "
  "pirmie, kas jau 598-595. g. p.m.\xE7"". apku\xEC""oju\xF0""i apk\xE2""rt"
  " \xC2""frikai, dom\xE2""dami gan, ka pieveiku\xF0""i tikai L\xEE""biju? "
  "... ka nosaukums \"jaunlatvie\xF0""i\" pirm\xE2""s atmodas laik\xE2"" ra"
  "dies no V\xE2""cij\xE2"" izveidoju\xF0""\xE2""s liter\xE2""tu liber\xE2"""
  "listisk\xE2""s kust\xEE""bas \"Junges Deutschland\", kuras pamatidejas b"
  "ija l\xEE""dz\xEE""gas jaunlatvie\xF0""u kust\xEE""bai? ... ka m\xFB""ra"
  " \xE7""kas ar koka p\xE2""rsegumu Rietumeirop\xE2"" s\xE2""ka b\xFB""v\xE7"""
  "t Karolingu renesanses laik\xE2""? Vikip\xE7""dija cit\xE2""s valod\xE2"""
  "s Vikip\xE7""dijas, kur\xE2""s i"
;

const char* kTeststr33 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/th_wikipedia_org_clean__ISO-8859-11.txt
  " \xE0""\xBE""\xD7""\xE8""\xCD""\xB6""\xC7""\xD2""\xC2""\xCA""\xD1""\xB5"""
  "\xC2""\xEC""\xBB""\xAF""\xD4""\xAD""\xD2""\xB3"" \xA1""\xE8""\xCD""\xB9"""
  "\xE0""\xA2""\xE9""\xD2""\xC3""\xD1""\xBA""\xCB""\xB9""\xE9""\xD2""\xB7"""
  "\xD5""\xE8"" \xE0""\xA1""\xD2""\xCB""\xC5""\xD5""\xE0""\xCB""\xB9""\xD7"""
  "\xCD""\xBB""\xC3""\xD0""\xA1""\xD2""\xC8""\xC7""\xE8""\xD2""\xB5""\xB9"""
  "\xBB""\xC3""\xD0""\xCA""\xBA""\xA4""\xC7""\xD2""\xC1""\xCA""\xD3""\xE0"""
  "\xC3""\xE7""\xA8"" \xE3""\xB9""\xA1""\xD2""\xC3""\xB7""\xB4""\xC5""\xCD"""
  "\xA7""\xC3""\xD0""\xE0""\xBA""\xD4""\xB4""\xB9""\xD4""\xC7""\xE0""\xA4"""
  "\xC5""\xD5""\xC2""\xC3""\xEC""\xE3""\xB5""\xE9""\xB4""\xD4""\xB9"" \xC1"""
  "\xD2""\xE0""\xC5""\xE0""\xAB""\xD5""\xC2"" \xBB""\xC3""\xD0""\xA1""\xD2"""
  "\xC8""\xE0""\xB5""\xD7""\xCD""\xB9""\xCD""\xD1""\xB9""\xB5""\xC3""\xD2"""
  "\xC2""\xCA""\xD3""\xCB""\xC3""\xD1""\xBA""\xA1""\xD2""\xC3""\xE0""\xB4"""
  "\xD4""\xB9""\xE0""\xC3""\xD7""\xCD""\xE3""\xB9""\xAA""\xE8""\xCD""\xA7"""
  "\xE1""\xA4""\xBA""\xC1""\xD0""\xC5""\xD0""\xA1""\xD2"" \xA2""\xB3""\xD0"""
  "\xB7""\xD5""\xE8""\xCB""\xC1""\xCD""\xA1""\xA4""\xC7""\xD1""\xB9""\xA8"""
  "\xD2""\xA1""\xE4""\xBF""\xBB""\xE8""\xD2""\xE3""\xB9""\xCD""\xD4""\xB9"""
  "\xE2""\xB4""\xB9""\xD5""\xE0""\xAB""\xD5""\xC2"" \xE0""\xA2""\xE9""\xD2"""
  "\xBB""\xA1""\xA4""\xC5""\xD8""\xC1""\xCB""\xC5""\xD2""\xC2""\xBE""\xD7"""
  "\xE9""\xB9""\xB7""\xD5""\xE8""\xE3""\xB9""\xE0""\xCD""\xE0""\xAA""\xD5"""
  "\xC2""\xB5""\xD0""\xC7""\xD1""\xB9""\xCD""\xCD""\xA1""\xE0""\xA9""\xD5"""
  "\xC2""\xA7""\xE3""\xB5""\xE9"" \xCA""\xB6""\xD2""\xBA""\xD1""\xB9""\xB7"""
  "\xD5""\xE8""\xB7""\xD3""\xCB""\xB9""\xE9""\xD2""\xB7""\xD5""\xE8""\xCA"""
  "\xC3""\xC3""\xCB""\xD2""\xBC""\xD9""\xE9""\xE4""\xB4""\xE9""\xC3""\xD1"""
  "\xBA""\xC3""\xD2""\xA7""\xC7""\xD1""\xC5""\xE2""\xB9""\xE0""\xBA""\xC5"""
  " \xB7""\xC2""\xCD""\xC2""\xBB""\xC3""\xD0""\xA1""\xD2""\xC8""\xC3""\xD2"""
  "\xC2""\xAA""\xD7""\xE8""\xCD""\xBC""\xD9""\xE9""\xE4""\xB4""\xE9""\xC3"""
  "\xD1""\xBA""\xC3""\xD2""\xA7""\xC7""\xD1""\xC5""\xBB""\xC3""\xD0""\xA8"""
  "\xD3""\xBB""\xD5"" \xBE"".\xC8"". 2549 \xE0""\xCB""\xB5""\xD8""\xA1""\xD2"""
  "\xC3""\xB3""\xEC""\xBB""\xD1""\xA8""\xA8""\xD8""\xBA""\xD1""\xB9"" \xC3"""
  "\xD9""\xE9""\xE4""\xCB""\xC1""\xC7""\xE8""\xD2""\x85"" \xA8""\xD2""\xA1"""
  "\xBA""\xB7""\xA4""\xC7""\xD2""\xC1""\xC5""\xE8""\xD2""\xCA""\xD8""\xB4"""
  "\xA2""\xCD""\xA7""\xC7""\xD4""\xA1""\xD4""\xBE""\xD5""\xE0""\xB4""\xD5"""
  "\xC2"" : ...\xBE""\xC3""\xD0""\xE0""\xA8""\xE9""\xD2""\xE2""\xCD""\xC1"""
  "\xD9""\xA1""\xD2""\xC1""\xD2""\xE2""\xCD""\xE2""\xC2"" \xCD""\xD6""\xB9"""
  "\xC2""\xD4""\xC1""\xBA""\xD2"" \xA1""\xD2""\xBA""\xD2""\xC1""\xBA""\xD2"""
  " \xCD""\xD4""\xA1""\xD9""\xC3""\xD9"" \xC3""\xD9""\xA1""\xD4""\xB4""\xD5"""
  "\xB7""\xD5""\xE8"" 4 \xE1""\xCB""\xE8""\xA7""\xE2""\xB5""\xE2""\xC3""\xE3"""
  "\xB9""\xC2""\xD9""\xA1""\xD1""\xB9""\xB4""\xD2"" \xB7""\xC3""\xA7""\xE0"""
  "\xBB""\xE7""\xB9""\xBE""\xC3""\xD0""\xC1""\xCB""\xD2""\xA1""\xC9""\xD1"""
  "\xB5""\xC3""\xD4""\xC2""\xEC""\xB7""\xD5""\xE8""\xC1""\xD5""\xBE""\xC3"""
  "\xD0""\xAA""\xB9""\xC1""\xD2""\xC2""\xD8""\xB9""\xE9""\xCD""\xC2""\xB7"""
  "\xD5""\xE8""\xCA""\xD8""\xB4""\xE3""\xB9""\xE2""\xC5""\xA1"" ...\xBE""\xC3"""
  "\xD0""\xBB""\xC3""\xD0""\xC1""\xD8""\xA2""\xE1""\xCB""\xE8""\xA7""\xC3"""
  "\xD1""\xB0""\xC1""\xD2""\xC5""\xD5""\xE0""\xCD""\xB5""\xD1""\xC7"" \xB5"""
  "\xD2""\xB9""\xD8""\xC1""\xD2""\xBF""\xD4""\xC5""\xD4""\xB7""\xD5""\xE8"""
  " 2 \xE1""\xCB""\xE8""\xA7""\xC3""\xD1""\xB0""\xE0""\xCD""\xA1""\xC3""\xD2"""
  "\xAA""\xAB""\xD2""\xC1""\xD1""\xC7"" \xB7""\xC3""\xA7""\xE0""\xBB""\xE7"""
  "\xB9""\xBE""\xC3""\xD0""\xC1""\xCB""\xD2""\xA1""\xC9""\xD1""\xB5""\xC3"""
  "\xD4""\xC2""\xB7""\xD5""\xE8""\xB7""\xC3""\xA7""\xC1""\xD5""\xBE""\xC3"""
  "\xD0""\xAA""\xB9""\xC1""\xD2""\xC2""\xD8""\xC1""\xD2""\xA1""\xB7""\xD5"""
  "\xE8""\xCA""\xD8""\xB4"" \xE3""\xB9""\xBA""\xC3""\xC3""\xB4""\xD2""\xBC"""
  "\xD9""\xE9""\xB9""\xD3""\xBB""\xC3""\xD0""\xE0""\xB7""\xC8""\xB7""\xD1"""
  "\xE8""\xC7""\xE2""\xC5""\xA1"" \xE1""\xC5""\xD0""\xB7""\xC3""\xA7""\xA4"""
  "\xC3""\xCD""\xA7""\xC3""\xD2""\xAA""\xC2""\xEC""\xE3""\xB9""\xB0""\xD2"""
  "\xB9""\xD0""\xE4""\xC1""\xE8""\xE3""\xAA""\xE8""\xBB""\xC3""\xD0""\xC1"""
  "\xD8""\xA2""\xE1""\xCB""\xE8""\xA7""\xC3""\xD1""\xB0"" \xB7""\xD5""\xE8"""
  "\xC2""\xD2""\xC7""\xB9""\xD2""\xB9""\xB7""\xD5""\xE8""\xCA""\xD8""\xB4"""
  "\xE3""\xB9""\xE2""\xC5""\xA1"" ...\xBF""\xD2""\xE2""\xC3""\xCB""\xEC""\xE0"""
  "\xBB""\xBB""\xD4""\xB7""\xD5""\xE8"" 2 \xE0""\xB9""\xE0""\xBF""\xCD""\xC3"""
  "\xEC""\xA1""\xD2""\xE0""\xC3"" \xE1""\xCB""\xE8""\xA7""\xCD""\xD5""\xC2"""
  "\xD4""\xBB""\xB5""\xEC""\xE2""\xBA""\xC3""\xD2""\xB3"" \xB7""\xC3""\xA7"""
  "\xE0""\xBB""\xE7""\xB9""\xBE""\xC3""\xD0""\xC1""\xCB""\xD2""\xA1""\xC9"""
  "\xD1""\xB5""\xC3""\xD4""\xC2""\xEC""\xB7""\xD5""\xE8""\xB7""\xC3""\xA7"""
  "\xA4""\xC3""\xCD""\xA7""\xC3""\xD2""\xAA""\xC2""\xEC""\xC2""\xD2""\xC7"""
  "\xB9""\xD2""\xB9""\xB7""\xD5""\xE8""\xCA""\xD8""\xB4"" \xE3""\xB9""\xBB"""
  "\xC3""\xD0""\xC7""\xD1""\xB5""\xD4""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"""
  "\xE2""\xC5""\xA1"" ...\xBF""\xD8""\xB5""\xBA""\xCD""\xC5""\xE3""\xB9""\xBB"""
  "\xC3""\xD0""\xE0""\xB7""\xC8""\xE4""\xB7""\xC2""\xC1""\xD5""\xC5""\xD5"""
  "\xA1""\xBF""\xD8""\xB5""\xBA""\xCD""\xC5""\xCD""\xD2""\xAA""\xD5""\xBE"""
  " 2 \xC5""\xD5""\xA1"" \xA4""\xD7""\xCD"" \xE4""\xB7""\xC2""\xC5""\xD5""\xA1"""
  "\xE1""\xC5""\xD0""\xE2""\xBB""\xC3""\xC5""\xD5""\xA1"" \xB7""\xD5""\xE8"""
  "\xE0""\xA1""\xE7""\xBA""\xB6""\xD2""\xC7""\xC3"" \x96"" \xCA""\xC3""\xE9"""
  "\xD2""\xA7""\xBA""\xB7""\xA4""\xC7""\xD2""\xC1""\xE3""\xCB""\xC1""\xE8"""
  " \xCA""\xD2""\xC3""\xD2""\xB9""\xD8""\xA1""\xC3""\xC1"" \xC7""\xD4""\xB7"""
  "\xC2""\xD2""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC""\xB8""\xC3""\xC3""\xC1"""
  "\xAA""\xD2""\xB5""\xD4"" \xE1""\xC5""\xD0""\xA4""\xB3""\xD4""\xB5""\xC8"""
  "\xD2""\xCA""\xB5""\xC3""\xEC"" \xBF""\xD4""\xCA""\xD4""\xA1""\xCA""\xEC"""
  " \x96"" \xB4""\xD2""\xC3""\xD2""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"" \x96"""
  " \xC7""\xD4""\xB7""\xC2""\xD2""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC""\xCA"""
  "\xD8""\xA2""\xC0""\xD2""\xBE"" \x96"" \xAA""\xD5""\xC7""\xC7""\xD4""\xB7"""
  "\xC2""\xD2"" \x96"" \xE0""\xA4""\xC1""\xD5"" \x96"" \xAA""\xD5""\xC7""\xE0"""
  "\xA4""\xC1""\xD5"" \x96"" \xC7""\xD4""\xB7""\xC2""\xD2""\xC8""\xD2""\xCA"""
  "\xB5""\xC3""\xEC""\xE2""\xC5""\xA1"" \x96"" \xB8""\xC3""\xB3""\xD5""\xC7"""
  "\xD4""\xB7""\xC2""\xD2"" \x96"" \xB9""\xD4""\xE0""\xC7""\xC8""\xC7""\xD4"""
  "\xB7""\xC2""\xD2"" \x96"" \xA4""\xB3""\xD4""\xB5""\xC8""\xD2""\xCA""\xB5"""
  "\xC3""\xEC"" \x96"" \xCA""\xB6""\xD4""\xB5""\xD4""\xC8""\xD2""\xCA""\xB5"""
  "\xC3""\xEC"" \xC7""\xD4""\xB7""\xC2""\xD2""\xA1""\xD2""\xC3"" \xC7""\xD4"""
  "\xB7""\xC2""\xD2""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC""\xBB""\xC3""\xD0"""
  "\xC2""\xD8""\xA1""\xB5""\xEC"" \xE1""\xC5""\xD0""\xC7""\xD4""\xC8""\xC7"""
  "\xA1""\xC3""\xC3""\xC1"" \xA1""\xD2""\xC3""\xA2""\xB9""\xCA""\xE8""\xA7"""
  " \x96"" \xA1""\xD2""\xC3""\xCA""\xD7""\xE8""\xCD""\xCA""\xD2""\xC3"" \x96"""
  " \xA1""\xD2""\xC3""\xE0""\xC1""\xD7""\xCD""\xA7"" \x96"" \xA1""\xD2""\xC3"""
  "\xBB""\xA1""\xA4""\xC3""\xCD""\xA7"" \x96"" \xB9""\xD4""\xB5""\xD4""\xC8"""
  "\xD2""\xCA""\xB5""\xC3""\xEC"" \x96"" \xC3""\xD1""\xB0""\xA1""\xD4""\xA8"""
  " \x96"" \xB8""\xD8""\xC3""\xA1""\xD4""\xA8"" \x96"" \xE0""\xA1""\xC9""\xB5"""
  "\xC3""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"" \x96"" \xE0""\xB7""\xA4""\xE2"""
  "\xB9""\xE2""\xC5""\xC2""\xD5""\xAA""\xD5""\xC7""\xC0""\xD2""\xBE"" \x96"""
  " \xE1""\xBE""\xB7""\xC2""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"" \x96"" \xBE"""
  "\xC2""\xD2""\xBA""\xD2""\xC5""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"" \x96"""
  " \xC8""\xD6""\xA1""\xC9""\xD2""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"" \x96"""
  " \xBA""\xC3""\xC3""\xB3""\xD2""\xC3""\xD1""\xA1""\xC9""\xC8""\xD2""\xCA"""
  "\xB5""\xC3""\xEC""\xE1""\xC5""\xD0""\xC7""\xD4""\xB7""\xC2""\xD2""\xA1"""
  "\xD2""\xC3""\xCA""\xD2""\xC3""\xCA""\xB9""\xE0""\xB7""\xC8"" \x96"" \xC7"""
  "\xD4""\xB7""\xC2""\xD2""\xA1""\xD2""\xC3""\xA4""\xCD""\xC1""\xBE""\xD4"""
  "\xC7""\xE0""\xB5""\xCD""\xC3""\xEC"" \x96"" \xC7""\xD4""\xC8""\xC7""\xA1"""
  "\xC3""\xC3""\xC1""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"" \xCB""\xC1""\xC7"""
  "\xB4""\xCB""\xC1""\xD9""\xE8"" \xA4""\xE9""\xB9""\xCB""\xD2"" \x97"" \xCA"""
  "\xB6""\xD2""\xB9""\xD5""\xC2""\xE8""\xCD""\xC2"" \x97"" \xCB""\xC1""\xC7"""
  "\xB4""\xCB""\xC1""\xD9""\xE8""\xCB""\xC5""\xD1""\xA1"" \x96"" \xCB""\xC1"""
  "\xC7""\xB4""\xCB""\xC1""\xD9""\xE8""\xB7""\xD1""\xE9""\xA7""\xCB""\xC1"""
  "\xB4"" \x96"" \xB7""\xD8""\xA1""\xCB""\xD1""\xC7""\xA2""\xE9""\xCD"" \x97"""
  " \xB5""\xD2""\xC1""\xE0""\xC7""\xC5""\xD2"" \x96"" \xC8""\xB5""\xC7""\xC3"""
  "\xC3""\xC9"" \x96"" \xBB""\xAF""\xD4""\xB7""\xD4""\xB9"" \x97"" \xCD""\xE9"""
  "\xD2""\xA7""\xCD""\xD4""\xA7"" \x96"" \xB4""\xD1""\xAA""\xB9""\xD5""\xC3"""
  "\xD2""\xC2""\xAA""\xD7""\xE8""\xCD"" \x97"" \xBA""\xD8""\xA4""\xA4""\xC5"""
  "\xCA""\xD3""\xA4""\xD1""\xAD""\xA2""\xCD""\xA7""\xE4""\xB7""\xC2"", \xA2"""
  "\xCD""\xA7""\xE2""\xC5""\xA1"" \x97"" \xBB""\xC3""\xD0""\xE0""\xB7""\xC8"""
  " \x96"" \xA8""\xD1""\xA7""\xCB""\xC7""\xD1""\xB4""\xE3""\xB9""\xBB""\xC3"""
  "\xD0""\xE0""\xB7""\xC8""\xE4""\xB7""\xC2"" \xC7""\xD1""\xB2""\xB9""\xB8"""
  "\xC3""\xC3""\xC1"" \xC7""\xD4""\xA8""\xD4""\xB5""\xC3""\xC8""\xD4""\xC5"""
  "\xBB""\xEC"" \xE1""\xC5""\xD0""\xAA""\xD5""\xC7""\xD4""\xB5""\xBB""\xC3"""
  "\xD0""\xA8""\xD3""\xC7""\xD1""\xB9"" \xA1""\xD2""\xC3""\xB7""\xD3""\xCD"""
  "\xD2""\xCB""\xD2""\xC3"" \x96"" \xA1""\xD2""\xC3""\xA8""\xD1""\xB4""\xCA"""
  "\xC7""\xB9"" \x96"" \xE4""\xC1""\xE9""\xB4""\xCD""\xA1""\xE4""\xC1""\xE9"""
  "\xBB""\xC3""\xD0""\xB4""\xD1""\xBA"" \x96"" \xCB""\xD1""\xB5""\xB6""\xA1"""
  "\xC3""\xC3""\xC1"" \x96"" \xA7""\xD2""\xB9""\xCD""\xB4""\xD4""\xE0""\xC3"""
  "\xA1"" \x96"" \xC0""\xD2""\xBE""\xC2""\xB9""\xB5""\xC3""\xEC"" \x96"" \xC7"""
  "\xD4""\xB7""\xC2""\xD8"" \x96"" \xE2""\xB7""\xC3""\xB7""\xD1""\xC8""\xB9"""
  "\xEC"" \x96"" \xCD""\xD4""\xB9""\xE0""\xB7""\xCD""\xC3""\xEC""\xE0""\xB9"""
  "\xE7""\xB5"" \x96"" \xA4""\xCD""\xC1""\xBE""\xD4""\xC7""\xE0""\xB5""\xCD"""
  "\xC3""\xEC"" \x96"" \xA1""\xD5""\xCC""\xD2"" \x96"" \xE0""\xA1""\xC1"" \x96"""
  " \xB9""\xD1""\xB9""\xB7""\xB9""\xD2""\xA1""\xD2""\xC3"" \x96"" \xA1""\xD2"""
  "\xC3""\xBA""\xD1""\xB9""\xE0""\xB7""\xD4""\xA7"" \x96"" \xB4""\xB9""\xB5"""
  "\xC3""\xD5"" \x96"" \xCD""\xD8""\xBB""\xC3""\xD2""\xA1""\xC3"" \x96"" \xB9"""
  "\xD2""\xAF""\xC8""\xD4""\xC5""\xBB""\xEC"" \x96"" \xC7""\xC3""\xC3""\xB3"""
  "\xA1""\xC3""\xC3""\xC1"" \x96"" \xA8""\xD4""\xB5""\xC3""\xA1""\xC3""\xC3"""
  "\xC1"" \x96"" \xB7""\xD1""\xC8""\xB9""\xC8""\xD4""\xC5""\xBB""\xEC""\xE1"""
  "\xC5""\xD0""\xA1""\xD2""\xC3""\xCD""\xCD""\xA1""\xE1""\xBA""\xBA"" \x96"""
  " \xA1""\xD2""\xC3""\xB6""\xE8""\xD2""\xC2""\xC0""\xD2""\xBE"" \x96"" \xC7"""
  "\xD1""\xB4""\xE4""\xB7""\xC2"" \x96"" \xCA""\xB6""\xD2""\xBB""\xD1""\xB5"""
  "\xC2""\xA1""\xC3""\xC3""\xC1""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"" \x96"""
  " \xA1""\xD2""\xC3""\xB7""\xE8""\xCD""\xA7""\xE0""\xB7""\xD5""\xE8""\xC2"""
  "\xC7"" \xC1""\xB9""\xD8""\xC9""\xC2""\xEC"" \xCA""\xD1""\xA7""\xA4""\xC1"""
  " \xE1""\xC5""\xD0""\xBB""\xC3""\xD1""\xAA""\xAD""\xD2"" \xBB""\xC3""\xD1"""
  "\xAA""\xAD""\xD2"" \x96"" \xA8""\xD4""\xB5""\xC7""\xD4""\xB7""\xC2""\xD2"""
  " \x96"" \xCA""\xD1""\xA7""\xA4""\xC1""\xC7""\xD4""\xB7""\xC2""\xD2"" \x96"""
  " \xC1""\xD2""\xB9""\xD8""\xC9""\xC2""\xC7""\xD4""\xB7""\xC2""\xD2"" \x96"""
  " \xE2""\xBA""\xC3""\xD2""\xB3""\xA4""\xB4""\xD5"" \x96"" \xBB""\xD8""\xC3"""
  "\xD2""\xB3""\xC7""\xD4""\xB7""\xC2""\xD2"" \x96"" \xBB""\xC3""\xD0""\xC7"""
  "\xD1""\xB5""\xD4""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"" \x96"" \xBB""\xC3"""
  "\xD0""\xC7""\xD1""\xB5""\xD4""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC""\xA2"""
  "\xCD""\xA7""\xC7""\xD4""\xB7""\xC2""\xD2""\xC8""\xD2""\xCA""\xB5""\xC3"""
  "\xEC""\xE1""\xC5""\xD0""\xE0""\xB7""\xA4""\xE2""\xB9""\xE2""\xC5""\xC2"""
  "\xD5"" \x96"" \xC0""\xD9""\xC1""\xD4""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"""
  " \x96"" \xC0""\xD2""\xC9""\xD2"" \x96"" \xC0""\xD2""\xC9""\xD2""\xE4""\xB7"""
  "\xC2"" \x96"" \xC0""\xD2""\xC9""\xD2""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"""
  " \x96"" \xE0""\xC8""\xC3""\xC9""\xB0""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"""
  " \x96"" \xC3""\xD1""\xB0""\xC8""\xD2""\xCA""\xB5""\xC3""\xEC"" \x96"" \xC8"""
  "\xD2""\xCA""\xB9""\xD2"" \x96"" \xE4""\xCA""\xC2""\xC8""\xD2""\xCA""\xB5"""
  "\xC3""\xEC"" \xCB""\xD1""\xC7""\xA2""\xE9""\xCD""\xBE""\xD4""\xE0""\xC8"""
  "\xC9"" \xB7""\xE8""\xD2""\xCD""\xD2""\xA1""\xD2""\xC8""\xC2""\xD2""\xB9"""
  "\xCA""\xD8""\xC7""\xC3""\xC3""\xB3""\xC0""\xD9""\xC1""\xD4"" \x97"" \xBE"""
  "\xC3""\xD0""\xC1""\xCB""\xD2""\xA1""\xC9""\xD1""\xB5""\xC3""\xD4""\xC2"""
  "\xEC""\xE4""\xB7""\xC2"" \x97"" \xC3""\xD2""\xC2""\xAA""\xD7""\xE8""\xCD"""
  "\xC2""\xD5""\xE8""\xCB""\xE9""\xCD""\xE4""\xB7""\xC2"""
;

const char* kTeststr35 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/so_wikipedia_org_clean__CP1256.txt
  " \xE3""\xCE""\xD5""\xE6""\xD5"" \xE1""\xDF""\xCA""\xC7""\xC8""\xC9"" \xC7"""
  "\xE1""\xE3""\xE6""\xD3""\xE6""\xDA""\xC9"" \xC7""\xE1""\xCD""\xD1""\xC9"""
  " \"\xE6""\xED""\xDF""\xED""\xC8""\xED""\xCF""\xED""\xC7""\" \xC8""\xC7"""
  "\xE1""\xE1""\xDB""\xC9"" \xC7""\xE1""\xD5""\xE6""\xE3""\xC7""\xE1""\xED"""
  "\xC9"". \xDF""\xE1"" \xDE""\xC7""\xD1""\xC6"" \xED""\xD3""\xCA""\xD8""\xED"""
  "\xDA"" \xC3""\xE4"" \xED""\xDF""\xCA""\xC8"" \xC3""\xED"" \xE3""\xDE""\xC7"""
  "\xE1"": \xDD""\xC7""\xE1""\xED""\xD6""\xDB""\xD8"" \xDA""\xE1""\xEC"" \""
  "edit\" \xE6""\xED""\xDF""\xCA""\xC8""! \xE1""\xE1""\xE3""\xD3""\xC7""\xDA"""
  "\xCF""\xC9"" \xC5""\xDE""\xD1""\xC3"" \xC7""\xE1""\xE6""\xED""\xDF""\xED"""
  "\xC8""\xED""\xCF""\xED""\xC7"" \xC7""\xE1""\xDA""\xD1""\xC8""\xED""\xC9"""
  ". Questa \xE8"" la pagina principale di Wikipedia in lingua somala, un'e"
  "nciclopedia libera in cui tutti possono scrivere articoli. Per vedere co"
;

const char* kTeststr42 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/ru-sib_wikipedia_org_clean__CP866.txt
  ": \x80""\xAC""\xA8""\xE2""\xA0""\xAD""\xAE""\xA7""\xAD""\xA0""\xA9""\xE1"""
  "\xA2""\xAE"" - \x84""\xA0""\xAB""\xA0""\xA5""\xA7""\xAD""\xA0""\xA9""\xE1"""
  "\xA2""\xAE"" - \x84""\xAE""\xAC""\xAE""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2"""
  "\xAE"" - \x86""\xEB""\xE1""\xE2""\xA5""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2"""
  "\xAE"" - \x87""\xA2""\xEC""\xAE""\xA7""\xA4""\xAE""\xA7""\xAD""\xA0""\xA9"""
  "\xE1""\xA2""\xAE"" - \x87""\xA5""\xAC""\xA5""\xA7""\xAD""\xA0""\xA9""\xE1"""
  "\xA2""\xAE"" - \x8C""\xA0""\xE2""\xA5""\xAC""\xA0""\xE2""\xA8""\xAA""\xA0"""
  " - \x8C""\xA8""\xE0""\xAE""\xAA""\xAE""\xAB""\xAE""\xA7""\xAD""\xA0""\xA9"""
  "\xE1""\xA2""\xAE"" - \x8F""\xE0""\xA8""\xE0""\xAE""\xA4""\xAE""\xA7""\xAD"""
  "\xA0""\xA9""\xE1""\xA2""\xAE"" - \x91""\xE2""\xA0""\xE2""\xA8""\xE1""\xE2"""
  "\xA8""\xAA""\xA0"" - \x92""\xE0""\xA0""\xA2""\xAE""\xA7""\xAD""\xA0""\xA9"""
  "\xE1""\xA2""\xAE"" - \x95""\xA8""\xAC""\xA8""\xEF"" \x8C""\xE3""\xA4""\xE0"""
  "\xEC""\xAE""\xAD""\xAE""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2""\xAE"" \xA4"""
  "\xA0"" \x80""\xE0""\xE2""\xA5""\xAB""\xEC""\xAD""\xEB"" \xA2""\xA5""\xA4"""
  "\xEB"": \x80""\xE0""\xE2""\xA5""\xAB""\xA5""\xA7""\xAD""\xA0""\xA9""\xE1"""
  "\xA2""\xAE"" - \x81""\xA0""\xE8""\xAB""\xEB""\xAA""\xAE""\xA7""\xAD""\xA0"""
  "\xA9""\xE1""\xA2""\xAE"" - \x82""\xA5""\xE0""\xA0"" - \x82""\xEC""\xAE"""
  "\xE2""\xAE""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2""\xAE"" - \x83""\xAE""\xA2"""
  "\xAE""\xE0""\xAE""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2""\xAE"" - \x84""\xA5"""
  "\xEF""\xAD""\xAD""\xA8"" - \x84""\xAE""\xAC""\xAE""\xA2""\xE8""\xEB""\xAD"""
  "\xA0"" - \x84""\xE3""\xE8""\xA5""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2""\xAE"""
  " - \x87""\xA5""\xAC""\xA5""\xAF""\xA8""\xE1""\xA0""\xAD""\xAD""\xA5"" - "
  "\x8A""\xAE""\xAD""\xAE""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2""\xAE"" - \x8B"""
  "\xA0""\xA4"" - \x8B""\xEE""\xA4""\xA5""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2"""
  "\xAE"" - \x8B""\xEE""\xA4""\xAD""\xA8""\xAA""\xA8"" - \x8C""\xE3""\xA4"""
  "\xE0""\xEC""\xAE""\xAD""\xAE""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2""\xAE"""
  " - \x8D""\xA0""\xE0""\xAE""\xA4""\xAE""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2"""
  "\xAE"" - \x8F""\xE0""\xA0""\xA2""\xA8""\xAB""\xEC""\xAD""\xEF"" - \x8F"""
  "\xE0""\xAE""\xAC""\xEB""\xE1""\xA5""\xAB"" \x90""\xAE""\xA1""\xAE""\xE2"""
  "\xAD""\xEB"" \xA2""\xA5""\xA4""\xEB"": \x82""\xA5""\xE1""\xE1""\xA5""\xA7"""
  "\xAD""\xA0""\xA9""\xE1""\xA2""\xAE"" - \x82""\xE0""\xA0""\xE7""\xA5""\xA1"""
  "\xE1""\xA2""\xAE"" - \x84""\xA5""\xE0""\xEC""\xAE""\xA2""\xA5""\xAD""\xE8"""
  "\xEB""\xAD""\xA0"" - \x8C""\xA5""\xA6""\xE3""\xA3""\xA8""\xAC""\xA3""\xA0"""
  " (\x88""\xAD""\xE2""\xA5""\xE0""\xAD""\xA5""\xE2"") - \x8D""\xA0""\xE7"""
  "\xAE""\xE2""\xAD""\xA8""\xE7""\xA5""\xE1""\xA2""\xAE"" - \x8F""\xAE""\xA2"""
  "\xEF""\xA7""\xAA""\xA8"" - \x90""\xA0""\xA4""\xA8""\xA2""\xAE"" - \x92"""
  "\xA5""\xE5""\xAD""\xAE""\xA7""\xAD""\xA0""\xA9""\xE1""\xA2""\xAE"" - \x92"""
  "\xAE""\xE0""\xA3""\xAE""\xA2""\xAB""\xEF"" - \x93""\xE1""\xE2""\xE0""\xAE"""
  "\xA9""\xE8""\xEB""\xE1""\xA2""\xAE"" \x93""\xA7""\xAE""\xE0""\xAE""\xE7"""
  "\xE7""\xA5"" \xA4""\xA0"" \x86""\xEB""\xA7""\xAD""\xA5""\xA2""\xE0""\xEF"""
  "\xA4"": \x81""\xA0""\xE1""\xAD""\xA5""\xAF""\xA8""\xE1""\xA0""\xAD""\xAD"""
  "\xA5"" - \x83""\xE3""\xAB""\xEC""\xA1""\xA0"" - \x87""\xAE""\xA4""\xE7"""
  "\xA5""\xE1""\xA2""\xAE"" - \x8A""\xE0""\xA0""\xE1""\xAD""\xAE""\xE1""\xAB"""
  "\xAE""\xA2""\xA2""\xA5"" - \x8B""\xA5""\xAF""\xAE""\xAB""\xA0""\xA4"" - "
  "\x8D""\xA0""\xA2""\xAE""\xA1""\xEB""\xAA""\xA8"" - \x8F""\xA0""\xA2""\xAD"""
  "\xA8""\xE6""\xA0"" - \x8F""\xAB""\xEF""\xE1""\xAE""\xE7""\xAA""\xA8"" - "
  "\x8F""\xAE""\xA7""\xEB""\xE0""\xA8""\xE8""\xE8""\xA5"" - \x8F""\xAE""\xE1"""
  "\xAA""\xE3""\xAB""\xAA""\xA8"" - \x93""\xA7""\xAE""\xE0""\xAE""\xE7""\xE7"""
  "\xA5"" \x84""\xA5""\xEF""\xAD""\xE1""\xAA""\xA0"" \xE7""\xA5""\xE0""\xA5"""
  "\xA4""\xA0"" \xA3""\xAE""\xA4""\xAE""\xA2"" - \x8F""\xA5""\xE1""\xE2""\xA5"""
  "\xA3""\xA0"" \xA4""\xA5""\xEF""\xAD""\xE1""\xAA""\xA8""\xE5"" \xA4""\xA0"""
  "\xAD""\xAE""\xAA"" - \x8F""\xA5""\xE1""\xE2""\xA5""\xA3""\xA0"" \xAA""\xE0"""
  "\xA0""\xA9""\xAE""\xA2"" \x90""\xAE""\xA1""\xAE""\xE2""\xA0"" \xE1""\xA5"""
  "\xA4""\xEC""\xAC""\xA8""\xE6""\xEB"" \x8D""\xA0"" \xA5""\xE2""\xAE""\xA9"""
  " \xE1""\xA5""\xA4""\xEC""\xAC""\xA8""\xE6""\xA5"" \xE0""\xAE""\xA1""\xA8"""
  "\xAC"" \x81""\xA8""\xA1""\xAB""\xA8""\xEE"" \x8D""\xAE""\xA2""\xAE""\xE1"""
  "\xA8"" 9 \xE1""\xA5""\xE0""\xEC""\xAF""\xAD""\xEF"" \x88""\xA7""\xA4""\xA5"""
  "\xEF""\xAD""\xA0"" \xAD""\xAE""\xA2""\xA0"" \xAA""\xE0""\xA0""\xE1""\xAE"""
  "\xE2""\xA8""\xE8""\xE8""\xA0"" \xA3""\xAE""\xAB""\xAE""\xA2""\xAD""\xAE"""
  "\xA9"""
;

const char* kTeststr44 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/av_wikipedia_org_clean__ISO-2022-KR.txt
  "\x1B""$)Ced for the creation of a Wikipedia in the Avar (\x0E"",!,S,Q,b\x0F"""
  ") language. There are currently 150 articles (mostly stubs). \x0E"",#,c,"
  "V\x0F"" \x0E"",c,d,b,Q,_,Z,h,m\x0F"" / All pages \x0E"",%,Q,T,l,Z,c,d,Q,"
  "_\x0F"" \x0E"",^,Q,T\x0F""I\x0E"",Q,b,e,]\x0F"" \x0E"",^,Q,h\x0F""I \x0E"""
  ",^,Q,T\x0F""I\x0E"",Q,b,e,],Q,]\x0F"" \x0E"",\",e,[,_,Q,g,l,c,\\\x0F"" \x0E"""
  ",3,e,],Q,g,l\x0F"" \x0E"",8\x0F""I\x0E"",`,R,`,\\,n\x0F"" \x0E"",\",Q,d,"
  "Q,c,e,_,Q\x0F"" \x0E"",!,Y,V,b,R,Q,[,U,X,Q,_\x0F"" \x0E"",\",Q,\\,e\x0F"""
  " \x0E"",2,`,c,c,Z,q,],l,e,]\x0F"" \x0E"",6,V,U,V,b,Q,h,Z,q\x0F"" \x0E"","
  "T\x0F""I\x0E"",e,b,e,c\x0F"" \x0E"",^,Q,h\x0F""I \x0E"",/,e,X,V,U,Q,T,Z\x0F"""
  " \x0E"",\\\x0F""I\x0E"",`,],Q\x0F"" \x0E"",^,Q,T\x0F""I\x0E"",Q,b,e,]\x0F"""
  " \x0E"",^,Q,h\x0F""I\x0E"",Q,],U,Q\x0F"" \x0E"",#,Z,\\,Z,a,V,U,Z,q\x0F"""
  " \x0E"",h,V,R,V,d\x0F""I\x0E"",V,Y,Q,R,Z,q,],U,Q\x0F"" \x0E"",T,l,`,b,],"
  "l\x0F"" \x0E"",T\x0F""I\x0E"",Q,g,n,Q,],],l,Z,Y,V\x0F""!\x0E"",;,Z,R,Q,R"
  "\x0F"" \x0E"",T,n,e,^,V,b,Q,],U,Q\x0F"" \x0E"",d\x0F""I\x0E"",Q,U\x0F"" "
  "\x0E"",R,e,T,`\x0F"" \"\x0E"",g,Z,c,Z,Y,Q,R,V\x0F""\" \x0E"",c,c,m,],\\,"
  "Q\x0F"": \x0E"",^,Q,\\,l,Q,],Q,q,],U,Q\x0F"" \x0E"",d\x0F""I\x0E"",Q,U,V"
  "\x0F"" \x0E"",X,`\x0F"" \x0E"",X,e,R,Q,],Q\x0F"" \x0E"",S,Q\x0F"" \x0E"""
  ",h\x0F""I\x0E"",e,_,e,],Q\x0F"", \x0E"",$,n,V,],U,Q,],l,e,_\x0F"" \x0E"""
  ",d\x0F""I\x0E"",e,R,Q,],Q\x0F"". \x0E"",#,`,b,V,g,Q\x0F"" \x0E"",_,e,X,V"
  ",b,T,`\x0F"" \x0E"",h\x0F""I\x0E"",Q,b\x0F"" \x0E"",R,Z,h,Z,_,V\x0F"" \x0E"""
  ",\\\x0F""I\x0E"",`,i,`,_,T,V\x0F""(\x0E"",c,c,m,],\\,Q\x0F"" \x0E"",R,e,"
  "T,`\x0F"" \x0E"",d\x0F""I\x0E"",Q,U\x0F"" \x0E"",\\,S,Q,b,Q,_,Z,U,Q\x0F"""
  ")! \x0E"",$,n,V,R\x0F"" \x0E"",g\x0F""I\x0E"",Q,X,Q,],l,e,],Q\x0F"" \x0E"""
  ",_,e,X,V,b,T,e,_\x0F"" \x0E"",R,e,g,n,V,_\x0F"" \x0E"",T,n,Q,R,Z,Y,V\x0F"""
  "! \x0E"",/,e,X,V,U,Q,T,`\x0F"" \x0E"",g,l,S,Q,S,e,]\x0F"" \x0E"",^,Q,T\x0F"""
  "I\x0E"",Q,b,e,]\x0F"" \x0E"",^,Q,h\x0F""I \x0E"",],l,Q,q,],U,Q\x0F"" \x0E"""
  ",j,Q,\\,],l,Z\x0F"" \x0E"",R,Q,d,Q,_,Z\x0F"", \x0E"",g\x0F""I\x0E"",Q,],"
  "R,Z,g,n,V\x0F"" \"\x0E"",8\x0F""I\x0E"",e,^,Q,U,Q\x0F""\" \x0E"",c,Q,[,d"
  ",Q,],l,e,]\x0F"" \x0E"",^,Q,d,V,b,Z,Q,],Q,Y,U,Q\x0F"" \x0E"",d\x0F""I\x0E"""
  ",Q,U\x0F"" \x0E"",g\x0F""I\x0E"",Q,],d\x0F""I\x0E"",Z\x0F"" \x0E"",T,n,Q"
  ",R,Z,Y,V\x0F"". If you come across pages that shouldn't be here, replace"
  " their content with {{db|reasonfordeletion}} so that when someone who sp"
  "eaks the language \"adopts\" the Avar Wikipedia t"
;

const char* kTeststr46 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/ksh_wikipedia_org_clean__GB18030.txt
  "nanie-du-Nord (Allemagne), dans certaines r\xA8""\xA6""gions de la Belgi"
  "que et dans le sud-ouest des Pays-Bas. Hinweis f\xA8""\xB9""r alle, die "
  "keine ripuarische Sprache sprechen: Dies ist die ripuarische Ausgabe der"
  " Wikipedia. Ripuarisch wird in \xA8""\xB9""ber 100 Dialekten im weiteren"
  " Umkreis von Aachen, Bonn, K\x81""0\x8B""2ln, D\xA8""\xB9""sseldorf, in "
  "Belgien, den Niederlanden und im deutschen Rheinland von etwa einer Mill"
  "ion Menschen gesprochen. Informatie voor iedereen die geen Ripuarisch sp"
  "reekt: Dit is de Ripuarische versie van Wikipedia. Er zijn meer dan 100 "
  "Ripuarische dialecten die worden gesproken door bijna twee miljoen mense"
  "n in de omgeving van Aken, Bonn, Keulen en D\xA8""\xB9""sseldorf, in Bel"
  "gi\x81""0\x8A""5, Nederland en het Duitse Rijnland. Informazione per chi"
  " non parla un dialetto ripuario: Questa \xA8""\xA8"" la versione in ripu"
  "ario di Wikipedia. Pi\xA8""\xB4"" di 100 dialetti ripuari sono parlati d"
  "a oltre 1 milione di persone nelle seguenti regioni: Aquisgrana, Bonn, C"
  "olonia, D\xA8""\xB9""sseldorf, nel Belgio orientale, nei Paesi Bassi sud"
  "-orientali e nella Renania. \xA5""\xEA""\xA5""\xD7""\xA5""\xA2""\xA9""`\xA5"""
  "\xEA""\xD5""Z\xA4""\xF2""\xD4""\x92""\xA4""\xB5""\xA4""\xCA""\xA4""\xA4"""
  "\xB7""\xBD""\xA4""\xD8""\xA4""\xCE""\xA4""\xB4""\xB0""\xB8""\xC4""\xDA"""
  ": \xA4""\xB3""\xA4""\xEC""\xA4""\xCF""\xA5""\xEA""\xA5""\xD7""\xA5""\xA2"""
  "\xA9""`\xA5""\xEA""\xD5""Z\xB0""\xE6""\xA4""\xCE""\xA5""\xA6""\xA5""\xA3"""
  "\xA5""\xAD""\xA5""\xDA""\xA5""\xC7""\xA5""\xA3""\xA5""\xA2""\xA4""\xC7"""
  "\xA4""\xB9""\xA1""\xA3"" \xA5""\xA2""\xA9""`\xA5""\xD8""\xA5""\xF3""\xA1"""
  "\xA2""\xA5""\xDC""\xA5""\xF3""\xA1""\xA2""\xA5""\xB1""\xA5""\xEB""\xA5"""
  "\xF3""\xA1""\xA2""\xA5""\xC7""\xA5""\xE5""\xA5""\xC3""\xA5""\xBB""\xA5"""
  "\xEB""\xA5""\xC9""\xA5""\xEB""\xA5""\xD5""\xB8""\xF7""\xCA""\xD0""\xA4"""
  "\xC8""\xA1""\xA2""\xA4""\xBD""\xA4""\xCE""\xD6""\xDC""\xDE""x\xA1""\xA2"""
  "\xBC""\xB0""\xA4""\xD3""\xA1""\xA2""\xA5""\xD9""\xA5""\xEB""\xA5""\xAE"""
  "\xA9""`\xA1""\xA2""\xA5""\xAA""\xA5""\xE9""\xA5""\xF3""\xA5""\xC0""\xA1"""
  "\xA2""\xA5""\xC9""\xA5""\xA4""\xA5""\xC4""\xA4""\xCE""\xA5""\xE9""\xA5"""
  "\xA4""\xA5""\xF3""\xA5""\xE9""\xA5""\xF3""\xA5""\xC8""\xA4""\xCB""\xA4"""
  "\xAA""\xA4""\xA4""\xA4""\xC6""\xA1""\xA2""\xA4""\xAA""\xA4""\xE8""\xA4"""
  "\xBD""200\xCD""\xF2""\xC8""\xCB""\xA4""\xCE""\xC8""\xCB""\xA4""\xCB""\xA4"""
  "\xE8""\xA4""\xC3""\xA4""\xC6""\xA1""\xA2""100\xB7""N\xD2""\xD4""\xC9""\xCF"""
  "\xA4""\xCE""\xA5""\xEA""\xA5""\xD7""\xA5""\xA2""\xA9""`\xA5""\xEA""\xB7"""
  "\xBD""\xD1""\xD4""\xA4""\xAC""\xD4""\x92""\xA4""\xB5""\xA4""\xEC""\xA4"""
  "\xC6""\xA4""\xA4""\xA4""\xDE""\xA4""\xB9""\xA1""\xA3"" \xA7""\xB3""\xA7"""
  "\xD3""\xA7""\xD6""\xA7""\xD5""\xA7""\xD6""\xA7""\xDF""\xA7""\xDA""\xA7"""
  "\xD6"" \xA7""\xD5""\xA7""\xDD""\xA7""\xF1"" \xA7""\xDC""\xA7""\xE4""\xA7"""
  "\xE0"" \xA7""\xDF""\xA7""\xD6"" \xA7""\xD4""\xA7""\xE0""\xA7""\xD3""\xA7"""
  "\xE0""\xA7""\xE2""\xA7""\xF0""\xA7""\xE4"" \xA7""\xE2""\xA7""\xDA""\xA7"""
  "\xE1""\xA7""\xE5""\xA7""\xD1""\xA7""\xE2""\xA7""\xE3""\xA7""\xDC""\xA7"""
  "\xDA""\xA7""\xDB"" \xA7""\xF1""\xA7""\xD9""\xA7""\xED""\xA7""\xDC"" \xA7"""
  "\xDF""\xA7""\xD6""\xA7""\xE4"": \xA7""\xBF""\xA7""\xE4""\xA7""\xE0"" \xA7"""
  "\xE2""\xA7""\xDA""\xA7""\xE1""\xA7""\xE5""\xA7""\xD1""\xA7""\xE2""\xA7"""
  "\xE3""\xA7""\xDC""\xA7""\xD1""\xA7""\xF1"" \xA7""\xA3""\xA7""\xDA""\xA7"""
  "\xDC""\xA7""\xDA""\xA7""\xE1""\xA7""\xD6""\xA7""\xD5""\xA7""\xDA""\xA7"""
  "\xF1"". \xA7""\xAE""\xA7""\xDA""\xA7""\xDD""\xA7""\xDD""\xA7""\xDA""\xA7"""
  "\xE0""\xA7""\xDF"" \xA7""\xE9""\xA7""\xD6""\xA7""\xDD""\xA7""\xE0""\xA7"""
  "\xD3""\xA7""\xD6""\xA7""\xDC""\xA7""\xDA"" \xA7""\xD4""\xA7""\xE0""\xA7"""
  "\xD3""\xA7""\xE0""\xA7""\xE2""\xA7""\xF0""\xA7""\xE4"" \xA7""\xD2""\xA7"""
  "\xE0""\xA7""\xDD""\xA7""\xEA""\xA7""\xD6"" \xA7""\xE3""\xA7""\xE4""\xA7"""
  "\xE0"" \xA7""\xE2""\xA7""\xDA""\xA7""\xE1""\xA7""\xE5""\xA7""\xD1""\xA7"""
  "\xE2""\xA7""\xE3""\xA7""\xDC""\xA7""\xDA""\xA7""\xE7"" \xA7""\xF1""\xA7"""
  "\xD9""\xA7""\xED""\xA7""\xDC""\xA7""\xE0""\xA7""\xD3"", \xA7""\xDA""\xA7"""
  "\xDD""\xA7""\xDA"" \xA7""\xD5""\xA7""\xDA""\xA7""\xD1""\xA7""\xDD""\xA7"""
  "\xD6""\xA7""\xDC""\xA7""\xE4""\xA7""\xE0""\xA7""\xD3"", \xA7""\xD3"" \xA7"""
  "\xD9""\xA7""\xD1""\xA7""\xE1""\xA7""\xD1""\xA7""\xD5""\xA7""\xDF""\xA7"""
  "\xED""\xA7""\xDE"" \xA7""\xD4""\xA7""\xD6""\xA7""\xE2""\xA7""\xDE""\xA7"""
  "\xD1""\xA7""\xDF""\xA7""\xE3""\xA7""\xDC""\xA7""\xDA""\xA7""\xDE"" \xA7"""
  "\xE3""\xA7""\xD6""\xA7""\xD3""\xA7""\xD6""\xA7""\xE2""\xA7""\xDF""\xA7"""
  "\xED""\xA7""\xDE"" \xA7""\xB2""\xA7""\xD6""\xA7""\xDB""\xA7""\xDF""-\xA7"""
  "\xA3""\xA7""\xD6""\xA7""\xE3""\xA7""\xE4""\xA7""\xE6""\xA7""\xD1""\xA7"""
  "\xDD""\xA7""\xDA""\xA7""\xF1"", \xA7""\xE3""\xA7""\xE2""\xA7""\xD6""\xA7"""
  "\xD5""\xA7""\xE0""\xA7""\xDB"" \xA7""\xA1""\xA7""\xD1""\xA7""\xE7""\xA7"""
  "\xD6""\xA7""\xDF""\xA7""\xD1"", \xA7""\xA2""\xA7""\xE0""\xA7""\xDF""\xA7"""
  "\xDF""\xA7""\xD1"", \xA7""\xAC""\xA7""\xD7""\xA7""\xDD""\xA7""\xEE""\xA7"""
;

const char* kTeststr48 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/co_wikipedia_org_clean__ISO_2022_CN.txt
  "formatica - Ingenieria - Management - Farmacia - Robotica - Telecommunic"
  "azione Vita quotidiana \x1B""$)A\x0E""((\x0F"" distrazzione Automobile -"
  " Bricolage - Cucina - Divertimentu - Giardinu - Ghjocu - Nutrizione - Sa"
  "lute - Sessualit\x0E""($\x0F"" - Sport - Televisione - Turisimu - Traspo"
  "rtu Ricerca \x0E""((\x0F"" navigazione Articuli di qualit\x0E""($\x0F"" "
  "- Indice alfabeticu - Indice tematicu - Pagine nuvelle - Radica di e cat"
  "egurie Aiutate Wikipedia in lingua corsa! Par aiut\x0E""($\x0F"" Wikiped"
  "ia pudete: Scrive un articulu, o ancu s\x0E""((\x0F"" voi truvete un sba"
  "gliu, currighjitelu puru ! Scrivete in suttanacciu o in supranacciu, cum"
  "e voi vulete: diversit\x0E""($\x0F"" face ricchezza. Fate a sperienza: P"
  "ruv\x0E""($\x0F"" \x0E""($\x0F"" scrive un articulu. Videte i cunsigli p"
  "er scrive un articulu. U pi\x0E""(4\x0F"" simplice per cuminci\x0E""($\x0F"""
  " h\x0E""((\x0F"" di cumplitt\x0E""($\x0F"" un articulu. Videte l'articul"
  "i da cumplitt\x0E""($\x0F"". Si p\x0E""(0\x0F"" din\x0E""(4\x0F"" traduc"
  "e un articulu, per esempiu da a wikipedia in sicilianu, talianu o france"
  "se. Videte Articuli da traduce. Identific\x0E""($\x0F"". S\x0E""((\x0F"""
  " voi cunniscite u nomu di parechji pianti, fiori, arburi, o animali... c"
  "hi mancanu, parmittar\x0E""($\x0F"" dopu di cre\x0E""($\x0F"" l'intrata "
  "currispundenti. Ch\x0E""(,\x0F"" sar\x0E""($\x0F"" issu funzu, issu fior"
  "i, issa pianta? Incaric\x0E""($\x0F"" imagini o fot\x0E""(0\x0F"". U meg"
  "liu h\x0E""((\x0F"" d'incaric\x0E""($\x0F"" li annantu \x0E""($\x0F"" Wi"
  "kiCommons, parch\x0E""(,\x0F"" pudarani serva dopu \x0E""($\x0F"" tutti "
  "i prugetti Wikipedia, \x0E""($\x0F"" u Wikizziunariu,... T'avemu bisognu"
  " d'una fot\x0E""(0\x0F"" di tutti i paesi di Corsica (\x0E""((\x0F"" din"
  "\x0E""(0\x0F"" di i pianti \x0E""((\x0F"" di l'animali). Da leghja A ran"
  "edda A Ranedda (o ranella) (Emys orbicularis) h\x0E""((\x0F"" una cuppul"
  "ata d'acqua d'Auropa. Si trova in i stagna, in i paduli, \x0E""((\x0F"" "
  "in i lava d'Auropa, d'Asia \x0E""((\x0F"" di u Norduvestu di l'Africa. C"
  "lassificazioni Esistini parechji 13 sottuspezii di ranedda, ch\x0E""(,\x0F"""
  " s\x0E""(0\x0F"" divisi in 5 gruppi: Emys orbicularis occidentalis Emys "
  "orbicularis occidentalis (Fritz, 1994) Emys orbicularis hispanica (Fritz"
  ", Keller & Budde, 1996) Emys orbicularis fritzjuergenobsti (Fritz, 1993)"
  " Emys orbicularis galloitalica Emys orbicularis galloitalica (Fritz, 199"
  "5) Emys orbicularis lanzai (Fritz, 1995) Emys orbicularis capolongoi (Fr"
  "itz, 1995) Emys orbicularis hellenica Emys orbicularis hellenica (Valenc"
  "iennes, 1832) Emys orbicularis kurae (Fritz, 1994) Emys orbicularis orie"
  "ntalis (Fritz"
;

const char* kTeststr53 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/oc_wikipedia_org_clean__MACINTOSH.txt
  "Sci\x8E""ncias exactas & naturalas Astronomia - Bioquimia - Biologia - B"
  "otanica - Quimia - Criptologia - Matematicas - Medecina - Farmacia - Fis"
  "ica - Zoologia Tecnologias Astronautica - Electricitat e Electronica - E"
  "nergia - Industria - Informatica - Nanotecnologia - Robotica - Telecomun"
  "icacions Mancam de fotografias per illustrar los articles. Pod\x8F""tz n"
  "os ajudar, e nos senhalar d'imatges utils de melhorar. Proj\x8F""cte col"
  "laboratiu de la setmana Lo subj\x8F""cte per aquesta setmana encara es p"
  "as estat definit. Pod\x8F""tz crear lo penjador. Modificar O sabiatz ? P"
  "ensatz, se o av\x8F""tz pas encara fach, de pla\x8D""ar los mod\x8F""ls "
  "de Babel dins v\x98""stra pagina d'utilisator! Atal, v\x98""stres interl"
  "ocutors eventuals saupr\x88""n quina lenga utilizar se vos v\x98""lon pa"
  "rlar! Tanben, ser\x88"" possible a un nov\x8F""l vengut de trobar de mon"
  "de que parle sa lenga dins las categorias de locutors de las diferentas "
  "lengas... Per aqu\x98"", cal metre {{Babel-1}} se sab\x8F""tz pas qu'una"
  " lenga, {{Babel-2}} se ne sab\x8F""tz doas...etc... seguit del(s) c\x98"""
  "di(s) de la(s) lenga(s) e del niv\x8F""l qu'av\x8F""tz per l'(las)utiliz"
  "ar (de 1 a 4 : del niv\x8F""l basic al niv\x8F""l exp\x8F""rt). Per exem"
  "ple se v\x98""stra lenga mairala es l'occitan e que parlatz perfi\x8F""c"
  "hament lo franc\x8E""s e pro corr\x8F""ctament lo catalan, metretz {{Bab"
  "el-3|oc|fr 4|ca 3}}. Lo rendut dins v\x98""stra pagina de presentacion s"
  "er\x88"" aiceste: oc Aqueste utilisator parla occitan coma primi\x8F""ra"
  " lenga. fr-4 Cet utilisateur parle fran\x8D""ais \x88"" un niveau compar"
  "able \x88"" la langue maternelle . ca-3 Aquest usuari pot contribuir amb"
  " un nivell avan\x8D""at de catal\x88"". Per mai d'entre-senhas, veire: W"
  "ikip\x8F""dia:Babel Archius | Modificar Actualitats e eveniments Efemeri"
  "d del 12 d'octobre de 2006 12 d'octobre - Lo turc Orhan Pamuk es anoncia"
  "t coma lo laurejat del Pr\x8F""mi Nobel de literatura del 2006 9 d'octob"
  "re - La Cor\x8F""a del N\x98""rd an\x97""ncia qu'a efectuat un t\x8F""st"
  " nuclear amb succ\x8F""s. 1 d'octobre, Brasil : Al primi\x8F""r torn de "
  "l'eleccion presidenciala, Luiz In\x87""cio Lula da Silva, del Partit del"
  "s Trabalhadors (PT) obten 48,6% dels sufragis, contra 41,6% de Geraldo A"
  "lckmin, del Partit Social Democrata Brasilian (PSDB). A las eleccions re"
  "gionalas, pel PT 4 governadors d'estat son elegits e pel PSDB 4 autres ("
  "9 elegits d'autres partits). Lo segond torn presidencial, e tanben lo se"
  "gond torn d'eleccion de governador en 10 estats, tendr\x88"" lu\x98""c l"
  "o 29 d'octobre. 29 de setembre, Brasil : gr\x8F""va collision d'un avion"
  " Boeing 737-800 amb un jet executive mod\x8F""l Leg"
;

const char* kTeststr54 =
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/am_wikipedia_org_clean__UTF7.txt
  " +EsgS8A +EhgTSBJwEpsSzQ +EmYScw +EhgSOxMIEi0 +Eu0SfRILEgkTYg +EugScBIzE"
  "nUTThLOEnUSlQ +EhgTABIYEi0S6w +EhgTIxMlE00 +EqgSpRKVEw0SChLdEpsSzQ +EsoS"
  "qhNUEvUS6w +EmASGBJwEi0TEBId +EhsSRRIoEmU +Eu0SQBINEs4SdQ +Eu0SBhKTEg0TY"
  "g +EqUTBRMN +EmUS2Q +EnUSLRMVEh0 +EhgTIxMlE04SfQ +EmISlhIpEpU +EggSoBIbE"
  "i0SmxLN +EsoSqhNSEvIS6w +EhsS8BMN +EhgSDRKrEh0 +EhgSlRMIEvU +Eu0TIBItEws"
  "SCRNi +EtsSLA +EhASGRI1E2M +EyUSRRIdEnU 2 +EkASlQ 1999 +EtMSGBJw +Eh0ShR"
  "IoEnU +EpASzRNi +EugS2xIs +EkASlQ (+EqUSlRLw +EqASzRIuEzMSzRLrEpU +EqASR"
  "hMjEyASLQ) 12 October, 2006 +EpASyRNi [+EggSGxI1EnASqxKoEg0] To download"
  " an Amharic unicode font for correct display: Please see the font sectio"
  "n for more information. [+EggSGxI1EnASqxKoEg0] +EmASrhIdE1ISzRJwEi0Szg +"
  "EgsS7Q +EmASoBIbEi0Smw +EmASQBILEgk +EggSGxJAEpASYxJgEi0: +EggScBMoEhsSK"
  "g +EhgSKBMD +EugSKBLzEnESlRNhEwgTPQ +EqUSYxKtEs4 +Eu0SGBIIEqgScQ! +EjUSC"
  "A +Eh0SlQ +EggSGxLIEkU +Eu0TSBINEwk +EpASYBIt? +EgASIxJlEs4SlQ +EmAS2hIF"
  " +EusSRRItEmETYg +EkASDRL2En0SlQ +EusSzRJDEgk? +EqUS2hIF +EgsS7Q +Eu0TKB"
  "IdEik! +Eh0SLRMl +E0USERNOEn0 +EugTCRMNEg0 +EtMSLRIb \"+EwkTDRIN (Google"
  ") +EmA-1998 +EqU.+EqQ.+EqA. +EugTDRIIEjASZQ +EvUSLRMFEnU +EgYSlg +EpASzQ"
  " +EugTABIYEigSzRNi +EugTCRMNEg0 +E00SCBML +EswSZRIzEu0SdRKV +EugSMBIrEs0"
  " +EqUSkw +EugSGhLrEjUScBLzEvUSKBLN +Eu0SBBLN +EvUSLRMFEnU +EpASzRNi +Eug"
  "TCRMNEg0 +EtMSCxIb +EugS0xIIEh0SlQ +EhgSKBMD +EggSGxIwEpMS8xJ1 +E2M +Egg"
  "SGxJFEigSZRKT +EyASQxIa +EhsS9RIoEw0 +EpASzRNi +EtwSkw +EqUSKBNNEnU: +En"
  "MSKhKt +EmAS2xIsEs0 +EqUSCBJ1 (October 12, 2006 +EqU.+EqQ.+EqA.): Wikipe"
  "dia:+EnMSKhKrEso +EhsSNRJzEsgSOxLOEn0-/+EyUSRRIdEnU 2 +EhgS3RMIEmA +EtUS"
  "zRJAEnU +EnUSHRIFEi0ScBNhEgISMxJlEpM +EqoSkBNhEyUSYBJlE2Y +EnUSHRIFEi0Sc"
  "A +EgISMxJl - +EugS3RItEt0SLQ +EgISMxJl (+EqUSNRJzEnISNRJyEq0SNQ) - +Eug"
  "SrhIdE1ISzRJwEi0TYRMlEpMSdQ - +EugSNhNNEnUSzBIt +EqASIBIrEi0 - +EugScBNI"
  "EyUSLhNhEhUTDRMLEnU +EyUSkxJ1 (+E0oS2hKtEjU) - +EugSFRKVEzs +EyUSYBJl - "
  "+EugSLRI7 +EnATDRJjEi0 - +EhgSABKVEvISNRKQEnU - +EhsTExMTEt0 - +Eh0TIxKU"
  "E2ESABJlEnU - +EpUTDRL1 - +EhgTCBKTEps +EiUSkBNhE00TJRIoEnUSkw +EjMS7RKV"
  "EjUTZg +EiUSkBNhE00TJRIoEnU - +EiUSkBNhEhUS7RLIEnU (+EmMS7hIOEwI) - +EnU"
  "SHRIFEi0ScBNhEyQSkw - +EhUSrRIdEpM - +EugTJRKVEnATYRKVEyUSLQ +EyUSkxJ1 ("
  "+EqwSGhI1EnUSKg) - +EiUSkBNhE0gSCBKt (+EqASNRJ1Ei4SlhIa) - +EugSGBIsEnU "
  "+EyUSkxJ1 (+EwIS7hIOEwI) - +EhgSDRKtEtATYRIdEvUSLQ (+EwISzhMNEisTSg) - +"
  "EnQSrRKWEs4SDhMC - +EugSMxLtEpUSNRKT +EugSGBKqEpMSzhJ9 +EnMSKhKt +EugShR"
  "JlEigScBIgEmU +EyUSkxJ1EpM +E00SDRI1E00SkxNm +EugSMBLN +Eg0TBQ +EyUSkxJ1"
  " - +EugSYxIVEg0 +EyUSkxJ1 (+EjYSMhLuEg4TAg) - +EnMSKhKt - +EugSSxKVEks +"
  "EyUSkxJ1 - +EjUSyxIwEs0 - +EqASLRKsEuwSDhMC (+EiUSkBNhEkUSLRI1) - +EnUSH"
  "RIFEi0SdQ - +E00SDRI1E00Skw - +EgMS7RIbEpYSdQ - +EugSJRKQE2ESDRJhEpM +En"
  "USHRIFEi0SdQ - +EhUS3RJjEso +EwkS8xLuEn0 - +EugTVhIIEnISqw +EyUSkxJ1 - +"
  "EhgSlRMNEiUSdQ - +EugSFRMN +EyUSZRJFEpM - +EugSZBJwEiASZRKT +EugSPRIbEn0"
  " +EyUSkxJ2En0 - +EugSGBM7EhUTTRJ1 +EmQSdRKT +EugSGBIoEwM +EyUSkxJ1 +EmMS"
  "FRINEpM +EiUSkA +EyUSYBJlE2Y +EqASlRL1Eh0Scw - +EqASYBIzEjASDQ - +Ey0TSB"
  "Ir - +EhsTKxLIEns - +E0oSDRId - +EygSyxJzEs4SfQ - +EugTCBKQEnU +EqASIBIr"
  "Ei0 - +EqUTAA +EyUSYBJl - +EwoS3A +EhsSMxIIE0oS6w - +EugSYBLTEgsSdQ +EkA"
  "SlhJ9 - +EvUSKA +EwgTPQ +EhgSKBJl - +EiUSkA +Ez0SERNN - +EhkS2hJD - +Etg"
  "TSBKV - +EjUS1RIN - +EqATSBNhEnMSKhKt"
;



// NOTE: Unicode handled differently because of embedded NUL


const char kTeststr57[] =
// This is UTF-16 little endian (note leading space)
// Produced by stringify.cc on 2006-10-26 11:18 from file testfiles/new_wikipedia_org_clean__Unicode.txt
  " \x00""&\t,\tA\t \x00""\x14""  \x00""5\t?\t\x15""\t?\t*\t?\t!\t?\t/\t>\t"
  " \x00""(\t>\t*\t \x00""8\tM\t5\t>\t*\tB\t \x00""&\tA\t\x17""\tA\t \x00"""
  "(\tM\t9\tM\t/\tG\t8\t \x00""(\tM\t/\tG\t(\t>\t \x00""&\t?\t8\t\x02""\t|\x00"""
  " \x00""\x17""\tM\t5\t>\t2\t?\t \x00""&\t,\tA\t \x00""\x14""  \x00""5\t?\t"
  "\x15""\t?\t*\t?\t!\t?\t/\t>\t/\t>\t\x17""\tA\t \x00""8\tM\t5\t/\t.\t8\tG"
  "\t5\t\x15""\t$\tG\t8\t\x02""\t \x00""8\t+\tA\t'\tA\t\x15""\tA\t.\tM\t9\t"
  "?\t \x00""(\x00""2\t>\t\x07""\t,\tM\t0\t0\t?\t/\t(\t)\x00""$\tG\t8\t\x02"""
  "\t \x00""%\tG\t\x02""\t \x00""\x1B""\t?\t$\t \x00""\x1B""\t?\t\x17""\tA\t"
  " \x00""(\tM\t9\tM\t/\tG\t8\t/\t>\t\x17""\tA\t \x00""2\t?\t8\t:\x00"" \x00"""
  ".\t>\t2\t>\t \x00""&\t?\t|\x00"" \x00""%\t \x00""&\t,\tA\t \x00""\x14"" "
  " \x00""5\t?\t\x15""\t?\t*\t?\t!\t?\t/\t>\t/\tA\t \x00""%\t\x17""\tA\t \x00"""
  ",\t>\t0\tG\t/\tG\t \x00""\x16""\t\x02""\t2\t>\t,\t\x02""\t2\t>\t \x00""/"
  "\t>\t/\tG\t\x17""\tA\t \x00""%\t>\t/\t|\x00"" \x00""%\t(\t \x00""5\t?\t\x15"""
  "\t?\t*\t?\t!\t?\t/\t>\t/\t>\t\x17""\tA\t \x00""*\tM\t0\t>\t5\t?\t'\t?\t\x15"""
  "\t \x00""5\t \x00""(\t?\t/\t.\t/\t>\t\x17""\tA\t \x00"",\t>\t0\tG\t2\tG\t"
  " \x00""(\t\x02""\t \x00""\x16""\t\x02""\t2\t>\t,\t\x02""\t2\t>\t \x00""\x1C"""
  "\tA\t\x08""\t|\x00"" \x00""\x17""\tA\t%\t?\t \x00""&\t,\tA\t \x00""\x14"""
  "  \x00""5\t?\t\x15""\t?\t*\t?\t!\t?\t/\t>\t/\t>\t\x17""\tA\t \x00""/\tG\t"
  "\x15""\tM\t5\t \x00""\x16""\tM\t/\t:\x00"" \x00""(\t>\t*\t \x00""8\tM\t5"
  "\t>\t*\tB\t \x00""&\tA\t\x17""\tA\t \x00"",\tA\t2\tG\t\x1F""\t?\t(\t \x00"""
  ",\tK\t0\tM\t!\t,\x00"" \x00""*\t0\t?\t/\tK\t\x1C""\t(\t>\t$\t,\x00"" \x00"""
  "6\tM\t0\tK\t$\t \x00""5\t \x00""\x1C""\tM\t/\t>\t\x16""\t\x02""\t|\x00"""
  " \x00""%\t>\t/\t \x00"",\tA\t\x16""\t\x02""\t \x00""\x14""  \x00""5\t?\t"
  "\x15""\t?\t*\t?\t!\t?\t/\t>\t \x00""5\t \x00""5\t?\t\x15""\t?\t.\t@\t!\t"
  "?\t/\t>\t \x00""+\t>\t\t\t(\tM\t!\tG\t8\t(\t(\t>\t*\t \x00""8\tM\t5\t>\t"
  "*\tB\t \x00""&\tA\t\x17""\tA\t \x00""\x18""\tK\t7\t#\t>\t,\x00"" \x00""\x05"""
  "\t*\t!\tG\t\x1F""\t,\x00"" \x00""2\tG\t\x16""\t \x00""5\t \x00""*\tM\t0\t"
  "G\t8\t \x00""5\t?\t\x1C""\tM\t\x1E""\t*\tM\t$\t?\t|\x00"" \x00""5\t?\t\x15"""
  "\t?\t*\t?\t!\t?\t/\t>\t/\t>\t\x17""\tA\t \x00""+\tA\t\x15""\t@\t \x00""\x1C"""
  "\tM\t/\t>\t\x1D""\tM\t5\t \x00""5\t?\t\x15""\t?\t*\t?\t!\t?\t/\t>\t/\t>\t"
  "$\t \x00""5\t?\t\x15""\t?\t.\t?\t!\t?\t/\t>\t \x00""+\t>\t\t\t(\tM\t!\tG"
  "\t8\t(\t \x00""(\t\x02""\t \x00""9\tK\t8\tM\t\x1F""\t \x00""/\t>\t(\t>\t"
  " \x00""$\t\x17""\tA\t \x00""&\tA\t|\x00"" \x00""%\tM\t5\t \x00""+\t>\t\t"
  "\t(\tM\t!\tG\t8\t(\t \x00""\x1B""\t\x17""\tA\t \x00""(\t(\t-\x00""*\tM\t"
  "0\tK\t+\t?\t\x1F""\t \x00""\x17""\tA\t%\t?\t \x00""\x16""\t:\x00""|\x00"""
  " \x00""%\tM\t5\t \x00""\x17""\tA\t%\t?\t \x00""(\t\x02""\t \x00"".\tG\t."
  "\tG\t\x17""\tA\t \x00""*\t0\t?\t/\tK\t\x1C""\t(\t>\t \x00""/\t>\t$\t \x00"""
  "(\t\x02""\t \x00""9\tK\t8\tM\t\x1F""\t \x00""/\t>\t(\t>\t \x00""$\t\x17"""
  "\tA\t \x00""&\tA\t \x00"":\x00"" \x00""5\t?\t\x15""\tM\t8\tM\t(\t0\t@\t"
;

const char kTeststr58[] =
// This is UTF-16 big endian (note leading 00)
  "\x00 \x00<\x00h\x00t\x00m\x00l\x00>\x00""f\x00u\x00""b\x00""a\x00r"
;

const char kTeststr59[] =
// This is UTF-32 little endian (note leading space)
  " \x00\x00\x00<\x00\x00\x00h\x00\x00\x00t\x00\x00\x00m\x00\x00\x00l\x00\x00\x00"
  ">\x00\x00\x00""f\x00\x00\x00u\x00\x00\x00""b\x00\x00\x00""a\x00\x00\x00r\x00\x00\x00"
;

const char kTeststr60[] =
// This is UTF-32 big endian (note leading 00)
  "\x00\x00\x00 \x00\x00\x00<\x00\x00\x00h\x00\x00\x00t\x00\x00\x00m\x00\x00\x00l"
  "\x00\x00\x00>\x00\x00\x00""f\x00\x00\x00u\x00\x00\x00""b\x00\x00\x00""a\x00\x00\x00r"
;

const char kTeststr61[] =
// This is binary pseudo-JPEG
  "\xff\xd8\xff\xe0\x00\x10JFIF foo bar baz\xff\xe1\x00\xa5"
  "\x01\xd7\xff\x01\x57\x33\x44\x55\x66\x77\xed\xcb\xa9\x87"
  "\xff\xd7\xff\xe0\x00\x10JFIF foo bar baz\xff\xe1\x00\xa5"
  "\x87\x01\xd7\xff\x01\x57\x33\x44\x55\x66\x77\xed\xcb\xa9"
;

const char kTeststr62[] =
// This is pseudo-HZ
  " ~{\x54\x42\x31\x7D\x37\x22\x55\x39\x35\x3D\x3D\x71~} abc"
;

const char kTeststr63[] =
// This is pseudo-UTF8UTF8
  "Ice cream a l\xc3\x83\xc2\xa0 mode."
;

// This one is really JAGRAN, and "www.amarujala.com" matches JAGRAN
const char* kTestUrl52 = "http://www.amarujala.com/today/default.asp";
const char kTeststr52[] =
// Produced by stringify.cc on 2007-04-10 12:07 from file amarujala_com.html
  "\r<html>\r<meta name='pregma' content='no-cache'>\r<meta name='descripti"
  "on' content= 'Amar Ujala online: Hindi news updates on business, politic"
  "s, sports, fashion, hindi films, national and international news reports"
  ". News from northern India: Uttaranchal, UP, Punjab, Delhi, Haryana, Cha"
  "ndigarh, Jammu and Kashmir, Himachal Pradesh'>\r<meta name='keywords' co"
  "ntent= 'hindi, news in hindi, hindi news, hindi newspaper, india, hindi "
  "font, north india, regional news, India, Indian news, news from india, u"
  "ttar\r,indian newspapers, U.P., uttaranchal, uttar pradesh, Kashmir conf"
  "lict, Pak, Pakistan, \ramar ujala, amarujala, amar ujala online, India P"
  "akistan news, bollywood, cricket, \rbusiness, karobar, amar ujala karoba"
  "r, India national news, Kashmir,\rIndo-Pak, delhi, , fashion, movies, ci"
  "nema, career, films, haryana,\rchandigarh, Punjab, kashmir, jammu and  k"
  "ashmir, \rweather, election, india election, india pakistan news, \relec"
  "toral polls, free speech, astrology, horoscope, Congress, BJP, RSS, Jant"
  "a Dal, union, legislative assembly, vidhan sabha, , MLA, Parliament, new"
  " delhi, stocks, BSE Sensex, Sensex, NSE, stock market, arts, , cyber new"
  "s, union budget, railway budget, himachal pradesh, hindi samachar, headl"
  "ine news, breaking news, Hurriyat, Musharraf, almora, Almoda, haridwar, "
  "dehra doon, dehra dun, dehradun, garhwal, gurdaspur, amritsar, hoshiyarp"
  "ur\r,kapurthala, jalandhar, firozpur, faridkot, moga, ludhiana, mohali, "
  "gurgaon, faridabad, noida, NOIDA, ghaziabad, ambala, shimla'>\r\r<meta h"
  "ttp-equiv='Content-Type' content='text/html; charset=iso-8859-1'>\r\r<he"
  "ad>\r<title>Amar Ujala - India's Leading Hindi Daily Newspaper</title>\r"
  "<style>\ra {text-decoration :none}\r</style>\r<STYLE type=text/css>\r<!-"
  "- /* $WEFT -- Created by: Amarujala infotech on 05/12/01 --*/\r@font-fac"
  "e {\rfont-family: au;\rfont-style:  normal;\rfont-weight: normal;\rsrc: "
  "url(../DBFNT0.eot);\r}\r-->\r</STYLE>\r<STYLE type=text/css>\rTABLE.wind"
  "ow{\rBORDER-LEFT: #000000 1px solid;\rBORDER-RIGHT: #000000 1px solid;\r"
  "BORDER-TOP: #000000 1px solid;\rBORDER-BOTTOM: #000000 1px solid\r}\r</S"
  "TYLE>\r<script language='javascript'>\rvar timeoutSecs = 20000\rfunction"
  " clearDiv()\r{\rpopwin.style.visibility=\"hidden\";\r}\rfunction setClea"
  "rTimeout()\r{\rsetTimeout('clearDiv()',timeoutSecs);\r}\r<!--\rfunction "
  "sel()\r{\ri=document.amar.day.selectedIndex;\rdd=document.amar.day(i).va"
  "lue;\rif (i==0)\r{\ralert('Please select a day');\rreturn(0);\r}\ri=docu"
  "ment.amar.month.selectedIndex;\rmm=document.amar.month(i).value;\ryy=doc"
  "ument.amar.year.value;\rif (yy == '2007'){\rif (mm < '04')\r{\rlocation."
  "href = \"../archive/archive.asp?dd=\"+dd+\"&mm=\"+mm+\"&yy=\"+yy+\"\"\rr"
  "eturn(0);\r}\relse {\rv='../'+dd+mm+yy+'/default.asp';\rsdate=dd + '/' +"
  " mm + '/' + yy;\rcdate=\"10/04/2007\";\rif (sdate==cdate)\r{\rv='../toda"
  "y/default.asp';\r}\rwindow.open(v);\r} }\relse {\ri=document.amar.day.se"
  "lectedIndex;\rdd=document.amar.day(i).value;\ri=document.amar.month.sele"
  "ctedIndex;\rmm=document.amar.month(i).value;\ryy=document.amar.year.valu"
  "e;\rlocation.href = \"../archive/archive.asp?dd=\"+dd+\"&mm=\"+mm+\"&yy="
  "\"+yy+\"\"\rreturn(0);\r}}\r-->\r</script>\r<style>\ra {text-decoration:"
  "none}\r</style>\r</head>\r<body bgcolor='#FFFFFF' leftmargin='0' topmarg"
  "in='0' marginwidth='0' marginheight='0'>\r<table width='100%' border='0'"
  " cellspacing='0' cellpadding='0'>\r<tr  align='left' valign='top'>\r<td>"
  "\r<table width='760' border='0' cellspacing='0' cellpadding='0'>\r<tr>\r"
  "<td width='100%' align='center'>\r<table width='100%' border='0' cellspa"
  "cing='0' cellpadding='3'>\r<tr>\r<td align=\"left\" width='25%'>\r<font "
  "face=\"au\" size=\"4\" color=\"blue\">\r\xDF""\xF0""S\xC5""U\xA7""\xA2"""
  "\xC7""U\xE8""\xC1""-87/3(14\xA5""\xE6""\xF0""\xDF""\xDA""U)UU<br>\r\n\xDC"""
  "\xE6""\xDA""U\xE6""-10,\xE2""\xDA""U\xDF""\xD9""-8<br>\r\n\xC1""\xE8""\xCC"""
  " \xB7""\xA4""\xE6"" \xDC""\xFF""\xD8""-357\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r"
  "\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\n\n\r"
  "\n\r\n\r\n\r\n\r\n\n\r\n\n</font>\r</td>\r<td align=\"center\" width='50"
  "%'><img src='' id=\"logo\" alt=\"Downloading\"><br><font face = 'au' siz"
  "e = '4'>\xD7""\xA2""\xBB""\xDC""\xDF""\xE6""\xDA""</font>,&nbsp;<font fa"
  "ce='verdana' size='2'>10</font>&nbsp;<font face = 'au' size = '4'>\xA5"""
  "\xC2""\xFD""\xF1""\xDC""</font>&nbsp;<font face='verdana' size='2'>2007<"
  "/font></td>\r<td align=\"right\" width='25%'>\r<a href='../Worldcup07/de"
  "fault.asp' target=\"_new\"><img border='0' id=\"topright\" alt=\"Downloa"
  "ding\" style=\"border: 1 solid #000000\"></a>\r</td>\r</tr>\r</table>\r<"
  "/td>\r</tr>\r</table>\r</td>\r</tr>\r</table>\r<script language='javascr"
  "ipt'>\r</script>\r<table border = '0' cellspacing = '0' cellpadding = '0"
  "' width='760'>\r<tr align='left'>\r<td width = '100' valign='top' align="
  "'left'>\r<b><font face='AU' color='#2C4FA0' size='4'>&nbsp;&nbsp;\xCC""\xE6"""
  "\xC1""\xE6"" \xE2""\xD7""\xE6""\xBF""\xE6""\xDA""</font></b>\r</td>\r<td"
  " width='550'>\r<SCRIPT language=JavaScript1.2 src='newticker1.js'></SCRI"
  "PT><br>\r</td>\r<td width = '100' align='right' valign='top'>\r<a href=\""
  "../aufont/aufont.asp\" target=\"new\"><img border='0' src='../images/fon"
  "t.gif' border = '0'></a>\r</td>\r</tr>\r</table>\r<table width='760' bor"
  "der='0' cellspacing='1' cellpadding='0' >\r<tr align='left'>\r<td>&nbsp;"
  "</td>\r<td>&nbsp;</td>\r</tr>\r</table>\r<table width='760' border='0' c"
  "ellspacing='0' cellpadding='0'>\r<tr align='left'>\r<td width='24%' bgco"
  "lor='#DE000D'></td>\r<td width='1%'><img src='../images/spacer.gif' heig"
  "ht='3'></td>\r<td width='50%' bgcolor='#000086'></td>\r<td width='1%'></"
  "td>\r<td width='24%' bgcolor='#FEAD00'></td>\r<tr>\r<table>\r<div align="
  "'left'>\r<table border='0' width='760' cellspacing='0' cellpadding='0'>\r"
  "<tr align='left'>\r<td align='left' width='130' bgcolor='#E5ECFD' valign"
  " = 'top'>\r<table border = '0' cellpadding = 0 cellspacing = 0>\r<tr><td"
  " width = '130' bgcolor = '#2C4FA0' align = 'center'><font face='au' colo"
  "r='#ffffff' size='3'>\xD7""\xE9""\x81""\xD8"" \xBF""\xF1""\xD9""\xDC""</"
  "font></td></tr>\r<tr align='center'>\r<td width='123' valign = 'top'>\r\r"
  "<table border='0' cellspacing='0' cellpadding='0' width='100%'>\r<tr>\r<"
  "td height='5'></td>\r</tr>\r<tr>\r<tr align='center'>\r<td >\r<table bor"
  "der='0' cellspacing='0' cellpadding='0' width='100%'>\r<tr>\r<td height="
  "'1'></td>\r</tr>\r<tr>\r<td width='15%'>&nbsp;&nbsp;<a href='../NRISpeci"
  "al/default.asp' target='new'><font face='arial' size='2' color='black'>N"
  "RI Special</a></font></td>\r</tr>\r<tr>\r<td height='1'></td>\r</tr>\r<t"
  "r>\r<td width='15%'>&nbsp;&nbsp;<a href='../Cinema/default.asp' target='"
  "new'><font face='au' size='4' color='black'>\xE7""\xE2""\xD9""\xF0""\xD7"""
  "\xE6""</a></font></td>\r</tr>\r<tr>\r<td height='1'></td>\r</tr>\r<tr>\r"
  "<td width='15%'>&nbsp;&nbsp;<a href='../Hotspot/default.asp' target='new"
  "'><font face='au' size='4' color='black'>\xE3""\xE6""\xF2""\xC5"" S\xC2"""
  "\xE6""\xF2""\xC5"" </a></font></td>\r</tr>\r<tr>\r<td height='1'></td>\r"
  "</tr>\r<tr>\r<td width='15%'>&nbsp;&nbsp;<a href='../newcareer/default.a"
  "sp' target='new'><font face='au' size='4' color='black'>\xB7""\xF1""\xE7"""
  "\xDA""\xD8""\xDA""</a></font></td>\r</tr>\r<tr>\r<td height='1'></td>\r<"
  "/tr>\r<tr>\r<td width='15%'>&nbsp;&nbsp;<a href='../Dharam/default.asp' "
  "target='new'><font face='au' size='4' color='black'>\xCF""\xD7""\xFC""-\xB7"""
  "\xA4""\xD7""\xFC""</a></font></td>\r</tr>\r<tr>\r<td height='1'></td>\r<"
  "/tr>\r<tr>\r<td width='15%'>&nbsp;&nbsp;<a href='../Tourism/default.asp'"
  " target='new'><font face='au' size='4' color='black'>\xE2""\xF1""\xDA"" "
  "\xE2""\xC2""\xE6""\xC5""\xE6""</a></font></td>\r</tr>\r<tr>\r<td height="
  "'1'></td>\r</tr>\r<tr>\r<td width='15%'>&nbsp;&nbsp;<a href='../Greeting"
  "/default.asp' target='new'><font face='au' size='4' color='black'>\xBB"""
  "\xFD""\xE8""\xE7""\xC5""\xA2""\x82""\xE2""</a></font></td>\r</tr>\r<tr>\r"
  "<td height='1'></td>\r</tr>\r<tr>\r<td width='15%'>&nbsp;&nbsp;<a href='"
  "../Ghar/default.asp' target='new'><font face='au' size='4' color='black'"
  ">\x83""\xE6""\xDA""-\xC2""\xE7""\xDA""\xDF""\xE6""\xDA""</a></font></td>"
  "\r</tr>\r<tr>\r<td height='1'></td>\r</tr>\r<tr>\r<td width='15%'>&nbsp;"
  "&nbsp;<a href='../Result/default.asp' target='new'><font face='au' size="
  "'4' color='black'>\xE7""\xDA""\xC1""\xCB""\xC5""</a></font></td>\r</tr>\r"
  "<tr>\r<td height='1'></td>\r</tr>\r</table>\r</td></tr>\r</table>\r</td>"
  "</tr>\r</table>\r</td>\r<td Valign='top' align = 'center'>\r<table width"
  " = '98%' border = '0' cellpadding = '0'>\r<tr><td width = '62%' valign ="
  " 'top'>\r<table width = '98%' border = '0' cellpadding = '0'>\r<tr><td v"
  "align='top' align='left' width='50' rowspan='2'>\r<img border='0' src='s"
  "sc.jpg' width='50' height='60'>\r</td>\r<td valign='top' align='left' wi"
  "dth='9'>\r<img border='0' src='../images/homepagenew/arrow.gif' width='8"
  "' height='7' align='middle'>&nbsp;\r</td><td valign='top' align='left' w"
  "idth='413'>\r<font face='AU' size='4'>\r<p align='justify'><a href =home"
  "news.asp?home=10am2.asp>\xE2""\xE8""\xE7""\xDC""\xA2""\xBB"" \xD7""\xF0"""
  "\xB4"" \xE7""\xC9""\xDC""\xE6""\xA7""\xFC"" \xE2""\xF0"" \xB7""\xE6""\xF0"""
  "\xC5""\xFC"" \xD9""\xE6""\xDA""\xE6""\xC1""</a>\r</td></tr>\r<tr>\r<td v"
  "align='top' align='left' width='9'>&nbsp;</td>\r<td valign='top' align='"
  "left'><p align = 'justify'>\r<font face='Wingdings' color='#C60000'>\xA7"""
  "</font>&nbsp;<font face='au' size='4'>\xE7""\xD9""\xCE""\xF0""\xFC""\xE0"""
  "\xE6""\xE6""\xF0""\xB4"" \xB7""\xE6"" \xC2""\xE6""\xDC""\xD9"" \xD9"" \xB7"""
  "\xDA""\xD9""\xF0"" \xC2""\xDA"" \xB0""\xF0""\xCC""\xDA""\xE6""\xC1""</fo"
  "nt><br>\r<font face='Wingdings' color='#C60000'>\xA7""</font>&nbsp;<font"
  " face='au' size='4'>\xE7""\xCE""\xCB""\xDC""\xE8"" \xB7""\xF0"" \xC2""\xE9"""
  "\xE7""\xDC""\xE2"" \xB7""\xE7""\xD7""\xE0""\xD9""\xDA"" \xCC""\xDC""\xD5"""
  "</font><br>\r</font></p></td>\r</tr>\r<tr>\r<td valign='top' align='left"
  "' width='98%' colspan='3' background='../images/homepagenew/line.jpg'> <"
  "/td>\r</tr>\r</table>\r<table width = '98%' border = '0' cellpadding = '"
  "0'>\r<tr><td valign='top' align='left' width='50' rowspan='2'>\r<img bor"
  "der='0' src='mah.jpg' width='50' height='60'>\r</td>\r<td valign='top' a"
  "lign='left' width='9'>\r<img border='0' src='../images/homepagenew/arrow"
  ".gif' width='8' height='7' align='middle'>&nbsp;\r</td><td valign='top' "
  "align='left' width='413'>\r<font face='AU' size='4'>\r<p align='justify'"
  "><a href =homenews.asp?home=10rm1.asp>\xA5""\xA2""\xD5""\xF0""\xC7""\xB7"""
  "\xDA"" \xB7""\xE8"" \xD7""\xEA""\xE7""\xCC""\xFC"" \xE2""\xF0"" \xC0""\xF0"""
  "\xC7""\xB8""\xC0""\xE6""\xC7""\xB8""</a>\r</td></tr>\r<tr>\r<td valign='"
  "top' align='left' width='9'>&nbsp;</td>\r<td valign='top' align='left'><"
  "p align = 'justify'>\r<font face='Wingdings' color='#C60000'>\xA7""</fon"
  "t>&nbsp;<font face='au' size='4'>\xD7""\xE3""\xE6""\xDA""\xE6""C\xFE"" \xB7"""
  "\xF0"" \xC1""\xE6""\xDC""\xD9""\xE6"" \xB7""\xE6"" \xD7""\xE6""\xD7""\xDC"""
  "\xE6""</font><br>\r<font face='Wingdings' color='#C60000'>\xA7""</font>&"
  "nbsp;<font face='au' size='4'>\xA9""\xC2""\xBC""\xFD""\xDF"" \xB7""\xF0"""
  " \xD5""\xE6""\xCE"" \xDC""\xBB""\xE6"" \xB7""\x8D""\xD8""\xEA""\xFC""</f"
  "ont><br>\r</font></p></td>\r</tr>\r<tr>\r<td valign='top' align='left' w"
  "idth='98%' colspan='3' background='../images/homepagenew/line.jpg'> </td"
  ">\r</tr>\r</table>\r<table width = '99%' border = '1' cellpadding = '0' "
  " cellspacing = '0' bgcolor='#FFFFFF' bordercolor='#BBDAFD'><tr><td>\r<ta"
  "ble width = '100%' border = '0' cellpadding = '1' >\r<tr><td valign='top"
  "' align='left' width='51' rowspan='2'>\r<a href='../Vishesh/default.asp'"
  " target=\"new\"><img border='0' src='10vish.jpg' width='50' height='60'>"
  "</a></td>\r<td valign='top' align='left' width='1'></td>\r<td valign='to"
  "p' bgcolor='#E7EEFC' width='208'><a href =homenews.asp?home=10homt2k.asp"
  "><font color='#C60000' size='4' face='AU'>\xE7""\xC2""\xCC""\xE6"" \xD5"""
  "\xD9""\xD9""\xF0"" \xB7""\xE8"" \xFF""\xE6""\xD7""\xCC""\xE6"" \xB9""\xE6"""
  "\xF0"" \xDA""\xE3""\xF0"" \xC2""\xE9""L\xE1""</font></a>\r</td>\r<td val"
  "ign='top' align='center' bgcolor='#2C4FA0' width='8' rowspan='2'><a href"
  "=\"../Vishesh/Default.asp\" target=\"new\"><img  src='../images/homepage"
  "new/visheshtd.gif' width='17' height='60' border='0'></a>\r</td></tr>\r<"
  "tr>\r<td valign='top' align='left'></td>\r<td valign='top' align='left' "
  "width='208'><p align = 'justify'><font face='au' size='4'>\xC1""\xE8""\xDF"""
  "\xD9"" \xE0""\xE6""\xF1""\xDC""\xE8"" \xD7""\xF0""\xB4"" \xE7""\xCE""\xD9"""
  "\xE6""\xF0""\xB4""\xE7""\xCE""\xD9"" \xE3""\xE6""\xF0"" \xDA""\xE3""\xF0"""
  " \xD5""\xCE""\xDC""\xE6""\xDF"" \xB7""\xE8"" \xDF""\xC1""\xE3"" \xE2""\xF0"""
  " \xC2""\xE9""L\xE1""\xE6""\xF0""\xB4"" ...</font></P></td>\r</tr>\r</tab"
  "le>\r</td></tr></table>\r<table width = '98%' border = '0' cellpadding ="
  " '0'>\r<tr>\r<td valign='top' align='left' width='98%' colspan='3'> </td"
  ">\r</tr>\r</table>\r</td>\r<td width = '1%' style='border-left: 1 double"
  " #808080' valign='top' align='right'>&nbsp;</td>\r<td width = '37%' alig"
  "n = 'left' valign = 'top'>\r<div align='left'>\r<table border = '0' cell"
  "padding = '1' cellspacing='0' width = '98%'>\r<tr><td align='left' width"
  "='9'>\r<img border='0' src='../images/homepagenew/arrow.gif' width='8' h"
  "eight='7' align='middle'>\r</td><td valign='top'>\r<table cellSpacing='0"
  "' cellPadding='1' width='99%' border='0'>\r<tr>\r<td vAlign='top' width="
  "'518' colSpan='4'>\r<p align='justify'><a href =homenews.asp?home=10rm5."
  "asp><font face='au' size='4'>\xC1""\xF0""\xC5"" \xD9""\xF0"" \xB0""\xD8"""
  "\xDA"" \xE2""\xE3""\xE6""\xDA""\xE6"" \xB7""\xE6""\xF0""\xB9""\xDA""\xE8"""
  "\xCE""\xE6""</font></a></p>\r</td>\r</tr>\r</table>\r</td></tr>\r<tr>\r<"
  "td>\r</td><td valign='top'>\r<font face='Wingdings' color='#C60000'>\xA7"""
  "</font>&nbsp;<font face='AU' size='4'>\xE2""\xE6""\xF1""\xCE""\xE6"" 180"
  "0 \xB7""\xDA""\xE6""\xF0""\xC7""\xB8"" L\xC2""\xB0"" \xD7""\xF0""\xB4""<"
  "/font><br>\r<font face='Wingdings' color='#C60000'>\xA7""</font>&nbsp;<f"
  "ont face='AU' size='4'>\xA5""\x8B""\xD8"" \xE0""\xE6""\xCC""\xE6""\xF0"""
  "Z \xB7""\xE6"" \xB9""\xE9""\xDC""\xE6""\xE2""\xE6"" \xD9""\xE3""\xE8""\xB4"""
  "</font><br>\r</td>\r</tr>\r<tr>\r<td valign='top' align='left' width='98"
  "%' colspan='3' background='../images/homepagenew/line.jpg'> </td>\r</tr>"
  "\r<tr>\r<td colspan='2' valign='top' align='left' height='5'>\r</td>\r</"
  "tr>\r<tr>\r<td>\r<img border='0' src='../images/homepagenew/arrow.gif' w"
  "idth='8' height='7' align='middle'>\r</td><td valign='top'>\r<a href =ho"
  "menews.asp?home=10am4.asp><font face='au' size='4'>\xB7""\xF0""\xB7"" \xC2"""
  "\xDA"" \xD2""\xE7""\xCC""\xDA""\xA2""\xBB""\xE6""\xD3"" \xB7""\xE6""\xC5"""
  "\xE6""</font></a>\r</td>\r</tr>\r<tr>\r<td>\r</td><td valign='top'>\r<fo"
  "nt face='Wingdings' color='#C60000'>\xA7""</font>&nbsp;<font face='au' s"
  "ize='4'>\xCC""\xF0""\xB4""\xCE""\xE9""\xDC""\xB7""\xA4""\xDA""U \xC2""\xDA"""
  "U \xA5""\xE6""\xDA""\xE6""\xF0""\xC2""</font><br>\r<font face='Wingdings"
  "' color='#C60000'>\xA7""</font>&nbsp;<font face='au' size='4'>\xE2""\xDA"""
  "U\xB7""\xA4""\xE6""\xDA""U \xB7""\xA4""\xDA""U\xE6""\xB0""\xBB""\xE8"" \xC1"""
  "\xE6""\xA2""\xBF""</font><br>\r</td>\r</tr>\r</tr>\r<tr>\r<td valign='to"
  "p' align='left' width='98%' colspan='3' background='../images/homepagene"
  "w/line.jpg'> </td>\r</tr>\r<tr>\r<td colspan='2' valign='top' align='lef"
  "t' height='4'>\r</td>\r</tr>\r<tr>\r<td>\r<img border='0' src='../images"
  "/homepagenew/arrow.gif' width='8' height='7' align='middle'>\r</td><td>\r"
  "<a href =homenews.asp?home=10am9.asp><font face='au' color='blue' size='"
  "4'>\xA5""\xE6""\xC6"" \xA9""\xCB""\xC8""\xE6"" \xA5""\xE6""\xCC""\xA2""\xB7"""
  "\xE8"" \xC9""\xF0""\xDA""</font></a>\r</td>\r</tr>\r<tr>\r<td>\r</td><td"
  ">\r<font face='Wingdings' color='#C60000'>\xA7""</font>&nbsp;<font face="
  "'au' size='4'>\xA5""L\x87""\xE6""\xE6""\xBF""\xDC"" \xC2""\xFD""\xCE""\xF0"""
  "\xE0""\xE6"" \xB7""\xE8"" \x83""\xE6""\xC5""\xD9""\xE6""</font><br>\r<fo"
  "nt face='Wingdings' color='#C60000'>\xA7""</font>&nbsp;<font face='au' s"
  "ize='4'>\xCE""\xE6""\xF0"" \xD7""\xE7""\xE3""\xDC""\xE6"" \xA5""\xE6""\xCC"""
  "\xA2""\xB7""\xE8"" \xE0""\xE6""\xE6""\xE7""\xD7""\xDC""</font><br>\r</td"
  ">\r</tr>\r<tr>\r<td colspan='2' height='3'>\r</td>\r</tr>\r</table>\r</d"
  "iv>\r</table>\r</td>\r<td align='left' width='130' valign = 'top'>\r<tab"
  "le border = 0 cellpadding = 0 cellspacing = 1>\r<tr><td width = 130 bgco"
  "lor = '#2C4FA0' align = center><a href=\"../news_gallery/default.asp\" t"
  "arget=\"new\"><font face='AU' color='#ffffff' size='3'>\xE7""\xBF""\x98"""
  "\xE6"" \xCE""\xE8""\x83""\xE6""\xE6""\xFC""</font></a></td></tr>\r<tr>\r"
  "<td width='123' valign = 'top' >\r<!--for caption-->\r<a href=# onclick="
  "\"javascript:window.open('phome.asp','win2','top=50,left=250,width=370,h"
  "eight=460,scrollbars=yes');\"><img border=0 src='' id=\"phome\" width='1"
  "27' height='145'></a><br>\r<div align='justify'><font face='au' color='#"
  "cb2828'>\xE2""\xE6""\xF1""\xB4""\xCE""\xD8""\xFC""\xD0"" \xD7""\xE9""\xA2"""
  "\xD5""\xA7""\xFC"" \xD7""\xF0""\xB4"" \xC2""\xFD""\xF0""\xE2"" \xB7""\xE6"""
  "\xA2""\xC8""\xFD""\xF0""\xE2"" \xB7""\xF0"" ...</font>&nbsp;<a href=\"#\""
  " onclick=\"javascript:window.open('phome.asp','win2','top=50,left=250,wi"
  "dth=370,height=460,scrollbars=yes');\"><font face='Wingdings' color='#cb"
  "2828'>\xF0""</font></a></div>\r</td></tr>\r</table>\r</td>\r</tr>\r</tab"
  "le>\r<table width='760' border='0' cellspacing='0' cellpadding='0'>\r<tr"
  " align='left'>\r<td width='130' valign='top' rowspan='2' bgcolor='#F6F6F"
  "7'>\r<table width='100%' border='0' cellspacing='0' cellpadding='0' >\r<"
  "tr><td height='1'></td></tr>\r<tr><td> \r<table width='100%' border='0' "
  "cellspacing='1' cellpadding='1' bgcolor = '#2C4FA0'>\r<tr>\r<td bgcolor="
  "'#6807D5' align='center' ><a href=\"edit.asp\"><font face='AU' size='3' "
  "color='#FFFFFF'>\xE2""\xA2""\xC2""\xE6""\xCE""\xB7""\xE8""\xD8""</font><"
  "/a></td>\r</tr>\r<tr>\r<td bgcolor='#F3F3F4' align='right' height='0'></"
  "td>\r</tr>\r<tr>\r<td bgcolor='#E4E5FF' valign='top'>\r<table width='100"
  "%' border='0' cellspacing='0' cellpadding='0' height='160'>\r<tr>\r<td w"
  "idth='1%' valign='top' >\r&nbsp;<img src='../images/arrow.gif'>&nbsp;</t"
  "d><td  width='99%' >\r<a href=\"editnews.asp?edit=9c3.asp\"><font face='"
  "AU' size='3' color='#000000'>\xC5""\xB7""\xDA""\xE6""\xDF"" \xE2""\xF0"""
  " \xD5""\xE7""\xBF""\xB0""</font></a>\r</td></tr><tr>\r<td width='1%' val"
  "ign='top' >\r&nbsp;<img src='../images/arrow.gif'>&nbsp;</td><td  width="
  "'99%' >\r<a href=\"editnews.asp?edit=9c3a.asp\"><font face='AU' size='3'"
  " color='#000000'>\xD5""\xE6""\xC2"" \xD5""\xC7""\xB8""\xE6"" \xE7""\xB7"""
  " \xC2""\xF1""\xE2""\xE6""</font></a>\r</td></tr><tr>\r<td width='1%' val"
  "ign='top' >\r&nbsp;<img src='../images/arrow.gif'>&nbsp;</td><td  width="
  "'99%' >\r<a href=\"editnews.asp?edit=8.asp\"><font face='AU' size='3' co"
  "lor='#000000'>\xE3""\xDA"" \xD5""\xE6""\xDA"" v\xD8""\xF4""\xB4"" \xE3"""
  "\xE6""\xDA"" \xC1""\xE6""\xCC""\xE6"" \xE3""\xF1"" \xC1""{\xD7""\xEA""-\xB7"""
  "\xE0""\xD7""\xE8""\xDA""?</font></a>\r</td></tr><tr>\r<td width='1%' val"
  "ign='top' >\r&nbsp;<img src='../images/arrow.gif'>&nbsp;</td><td  width="
  "'99%' >\r<a href=\"editnews.asp?edit=8c.asp\"><font face='AU' size='3' c"
  "olor='#000000'>\xA5""\xE6""\xE7""\xB9""\xDA"" \xE7""\xB7""\xCF""\xDA"" \xC1"""
  "\xE6""\xB0"" \xD7""\xE9""\xE2""\xE7""\xDC""\xD7"" \xD7""\xCC""\xCE""\xE6"""
  "\xCC""\xE6""</font></a>\r</td></tr><tr>\r<td width='1%' valign='top' >\r"
  "&nbsp;<img src='../images/arrow.gif'>&nbsp;</td><td  width='99%' >\r<a h"
  "ref=\"editnews.asp?edit=8c1.asp\"><font face='AU' size='3' color='#00000"
  "0'>\xE2""\xE6""\xE2"", \xD5""\xE3""\xEA"" \xA5""\xE6""\xF1""\xDA"" \xE7"""
  "\xB7""\xFD""\xB7""\xF0""\xC5""</font></a>\r</td></tr><tr>\r</td>\r</tr>\r"
  "</table>\r</td>\r</tr>\r</table>\r</td></tr><tr><td height='2'>\r</td></"
  "tr><tr><td>\r<table border='0' bgcolor='#F6F3E0'>\r<tr><td>\r<table widt"
  "h='100%' border='0' cellspacing='1' cellpadding='1' bgcolor='#FFBC00'>\r"
  "<tr>\r<td bgcolor='brown' align='center'><font face='AU' size='4' color="
  "'#ffFFFF'><b>\xD6""\xE7""\xDF""c\xD8""\xC8""\xDC"" \xC1""\xE6""\xD9""\xF0"""
  "\xB4""</b></font></a></td>\r</tr>\r<tr>\r<td >\r<table width='100%' bord"
  "er='0' cellspacing='0' cellpadding='0'>\r<tr><td height='10'></td></tr>\r"
  "<tr>\r<td width='1%' valign='top' ALIGN='center' bgcolor='#FFBC00'>\r<sc"
  "ript>\rfunction formmonthsubmit()\r{\rselect_futurename = document.mform"
  ".select_fname.value\rmfuture = document.mform.mfuture.value\rwindow.open"
  "('../future/monthlyfuture1.asp?fm='+ mfuture + '&nm=' + select_futurenam"
  "e,'win','width=400 height=300 scrollbars=yes')\r}\rfunction dothis()\r{\r"
  "mform.mfuture.value=0;\r}\r</script>\r<form name=\"mform\" method=\"post"
  "\" action=\"\">\r<select name=\"select_fname\" style=\"FONT-SIZE: 17px; "
  " WIDTH: 90px;  FONT-FAMILY: au\" onchange=\"dothis();\">\r<option value="
  "'wb' selected>\xE2""\xE6""\x8C""\xCC""\xE6""\xE7""\xE3""\xB7""</option>\r"
  "<option value='yb'>\xDF""\xE6""\xE7""\xE1""\xFC""\xB7"" </option>\r</sel"
  "ect>\r<select name=\"mfuture\" style=\"FONT-SIZE: 17px; WIDTH: 90px; FON"
  "T-FAMILY: au\" onchange=\"formmonthsubmit();\">\r<option value='0' selec"
  "ted>\xDA""\xE6""\xE7""\xE0""\xE6""\xD8""\xE6""\xA2""</option>\r<option v"
  "alue='1'>\xD7""\xF0""\xE1""</option>\r<option value='2'>\xDF""\xEB""\xE1"""
  " </option>\r<option value='3'>\xE7""\xD7""\xCD""\xE9""\xD9"" </option>\r"
  "<option value='4'>\xB7""\xB7""\xFC"" </option>\r<option value='5'>\xE7"""
  "\xE2""\xA2""\xE3"" </option>\r<option value='6'>\xB7""\x8B""\xD8""\xE6"""
  " </option>\r<option value='7'>\xCC""\xE9""\xDC""\xE6"" </option>\r<optio"
  "n value='8'>\xDF""\xEB""\xE7""\xE0""\xBF""\xB7""\xA4"" </option>\r<optio"
  "n value='9'>\xCF""\xD9""\xE9"" </option>\r<option value='10'>\xD7""\xB7"""
  "\xDA"" </option>\r<option value='11'>\xB7""\xE9""\xA2""\xD6"" </option>\r"
  "<option value='12'>\xD7""\xE8""\xD9"" </option>\r</select>\r</FORM>\r</t"
  "d>\r</tr>\r</table>\r</td>\r</tr>\r</table>\r</td></tr>\r</table>\r</td>"
  "</tr><tr><td height='2'>\r</td></tr><tr><td>\r<table width='100%' border"
  "='0' cellspacing='1' cellpadding='1' bgcolor = '#2C4FA0'>\r<tr>\r<td bgc"
  "olor='#2C4FA0' align='center'><font face='AU' size='4' color='#FFFFFF'>\xE7"""
  "\xC1""\x99""\xE6""\xE6""\xE2""\xE6""</font></td>\r</tr>\r<tr>\r<td bgcol"
  "or='#F3F3F4' align='right' height='0'></td>\r</tr>\r<tr>\r<td bgcolor='#"
  "EDF2FD' >\r<table width='100%' border='0' cellspacing='0' cellpadding='0"
  "' align='center' >\r<tr>\r<td valign='top' width='1%' >&nbsp;<img src='."
  "./images/arrow.gif' border='0'>&nbsp;</TD>\r<td width='98%'><font face='"
  "AU' size='3' color='#04135E'><a href=\"../future/fullmain.asp?id=113\">\xD5"""
  "\xE3""\xE9""\xCC"" \xBF""\xD7""\x88""\xB7""\xA4""\xE6""\xDA""\xE8"" \xE3"""
  "\xF4""\xCC""\xF0"" \xE3""\xF1""\xB4""\xE7""\xC2""\xDA""\xE6""\xE7""\xD7"""
  "\xC7""</a></font></td>\r</tr>\r<tr>\r<td valign='top' width='1%' >&nbsp;"
  "<img src='../images/arrow.gif' border='0'>&nbsp;</TD>\r<td width='98%'><"
  "font face='AU' size='3' color='#04135E'><a href=\"../future/fullmain.asp"
  "?id=111\">\xB7""\xE9""\xA4""\xDC""\xBB""\xE9""L\xA4"" \xB7""\xA4""\xE6"""
  " \xE2""{\xD7""\xE6""\xD9"" \xB7""\xA4""\xDA""\xF0""\xB4"", \xDC""\xE6""\xD6"""
  " \xE3""\xF4""\xBB""..</a></font></td>\r</tr>\r<tr>\r<td valign='top' wid"
  "th='1%' >&nbsp;<img src='../images/arrow.gif' border='0'>&nbsp;</TD>\r<t"
  "d width='98%'><font face='AU' size='3' color='#04135E'><a href=\"../futu"
  "re/fullmain.asp?id=97\">\xDF""\xE6""S\xCC""\xE9"" \xD7""\xF0""\xB4"" \xC2"""
  "\xA2""\xBF"" \xCC""\x88""\xDF""\xF4""\xB4"" \xB7""\xA4""\xE6"" \xD7""\xE3"""
  "\x88""\xDF""</a></font></td>\r</tr>\r</table>\r</td>\r</tr>\r</table>\r<"
  "/td></tr><tr><td height='3'>\r</td></tr><tr><td>\r<table width='100%' bo"
  "rder='0' cellspacing='1' cellpadding='1' bgcolor = '#2C4FA0'>\r<tr>\r<td"
  " bgcolor='#2C4FA0' align='center'><font face='AU' size='4' color='#FFFFF"
  "F'>\xC8""\xF1""\xE0""\xE6""\xD9"" \xBB""\xF1""\xDC""\xDA""\xE8""</font><"
  "/td>\r</tr>\r<tr>\r<td bgcolor='#F3F3F4' align='right' height='0'></td>\r"
  "</tr>\r<tr>\r<td bgcolor='#FFFFFF' align='left'>\r<table width='98%' bor"
  "der='0' cellspacing='2' cellpadding='6' align='center'>\r<tr>\r<td><div "
  "align='justify'>\r<a href=\"../news_gallery/detail.asp?sectionid=61&cate"
  "id=181&id=2297&fldname=20070410&imgname=MDF87887.jpg&wpaper=1&capt=\xDF"""
  "\xF0""\xB4""\xC7""\xDC"" \xDA""\xE6""\xE7""\xC7""\xB7"" \xB7""\xF0"" \xC2"""
  "\xE7""\xDA""\xCF""\xE6""\xD9""\xE6""\xF0""\xB4"" \xB7""\xE6"" \xC2""\xFD"""
  "\xCE""\xE0""\xE6""\xFC""\xD9"" \xB7""\xDA""\xCC""\xE8"" \xD7""\xE6""\xF2"""
  "\xC7""\xDC""\xD0""\" target=\"new\"> \r<img src=\"../news_gallery/200704"
  "10/MDF87887.jpg\" align='center' border='0'>\r</a>\r</td>\r</tr>\r</tabl"
  "e>\r</td>\r</tr>\r</table>\r</td></tr><tr><td height='3'>\r</td></tr><tr"
  "><td>\r<table width='100%' border='0' cellspacing='1' cellpadding='1' bg"
  "color = '#2C4FA0'>\r<tr>\r<td bgcolor='#9148fb' align='center'><a href=\""
  "../gaurtalab/default.asp\" target=\"new\"><font face='AU' size='4' color"
  "='#FFFFFF'>\xBB""\xF5""\xDA""\xCC""\xDC""\xD5""</font></td>\r</tr>\r<tr>"
  "\r<td bgcolor='#F3F3F4' align='right' height='0'></td>\r</tr>\r<tr>\r<td"
  " bgcolor='#f5f0fd' align='left'>\r<table width='99%' border='0' cellspac"
  "ing='1' cellpadding='0'>\r<tr>\r<td width='1%' valign='top'>&nbsp;<img s"
  "rc='../images/arrow.gif' border='0'>&nbsp;</td>\r<td width='98%'><a href"
  "=\"../gaurtalab/detail.asp?id=505\" target=\"new\" ><font face='au' colo"
  "r='black'><div align='justify'>\xBB""\xCE""\xDA"" \xB7""\xF0"" \xC2""\xE9"""
  "\xDA""\xF4""\xCF""\xE6""</div></font></a>\r</td></tr>\r<tr>\r<td width='"
  "1%' valign='top'>&nbsp;<img src='../images/arrow.gif' border='0'>&nbsp;<"
  "/td>\r<td width='98%'><a href=\"../gaurtalab/detail.asp?id=506\" target="
  "\"new\" ><font face='au' color='black'><div align='justify'>\xBB""\xFD"""
  "\xE6""\xD7""\xE8""\x87""\xE6""\xE6""\xF0""\xB4"" \xB7""\xE8"" S\xD7""\xEB"""
  "\xE7""\xCC"" \xD7""\xF0""\xB4"" \xD7""\xA2""\xBB""\xDC"" \xC2""\xE6""\xA2"""
  "\xC7""\xF0""</div></font></a>\r</td></tr>\r<tr>\r<td width='1%' valign='"
  "top'>&nbsp;<img src='../images/arrow.gif' border='0'>&nbsp;</td>\r<td wi"
  "dth='98%'><a href=\"../gaurtalab/detail.asp?id=507\" target=\"new\" ><fo"
  "nt face='au' color='black'><div align='justify'>\xB7""\xE6""\xDA""\xCC"""
  "\xEA""\xE2"" \xCC""\xF4"" \xD7""\xE3""\xC1"" \xB0""\xB7"" \xD5""\xE3""\xE6"""
  "\xD9""\xE6"" \xCD""\xE6""</div></font></a>\r</td></tr>\r</table>\r</td>\r"
  "</tr>\r</table>\r</td></tr><tr><td height='4'>\r</td></tr><tr><td align="
  "'center'>\r<table width='100%' border='0' cellspacing='0' cellpadding='0"
  "' align='center'>\r<tr><td align='center'><a href=\"../future/ques.asp\""
  " onclick=\"window.open('../future/form.asp','win','top=50 left=50 width="
  "500 height=320 scrollbars=yes')\" target=\"new\"><img src=\"../future/im"
  "age/futureques2.gif\" border=\"0\"></a></td></tr>\r<tr><td height='2'></"
  "td></tr>\r<tr>\r<td width='100%' align='center'><a href=\"../HCjudgment0"
  "7/Default.asp\" target=\"new\" ><img src=\"../image/HCjudgment07.gif\" a"
  "lign='center' style='border : 1 solid #000000'height='60' width='125'></"
  "a>\r</td></tr>\r<tr><td height='2'></td></tr>\r</table>\r</td></tr></tab"
  "le>\r</td>\r<td align='center' valign='top'>\r<table width='99%' border="
  "'0' cellspacing='0' cellpadding='0'>\r<tr>\r<td align='center' backgroun"
  "d='images/line.jpg'> </td>\r</tr>\r<tr>\r<td align='center'></td>\r</tr>"
  "\r</table>\r<table width='99%' border='0' cellspacing='0' cellpadding='0"
  "'>\r<tr><td colspan=3 align='center'>\r</td></tr>\r<tr>\r<td height='4'>"
  "<img src='../images/spacer.gif' width='1' height='1'></td>\r</tr>\r</tab"
  "le>\r<table width='98%' border='0' cellspacing='0' cellpadding='0' >\r<t"
  "r>\r<td valign='top' width='49%'>\r<table border='0' cellspacing='0' cel"
  "lpadding='0' width='100%'>\r<tr><td colspan='2'>\r<table border = \"0\" "
  "width = \"100%\" cellspacing = \"0\" cellpadding = \"0\">\r             "
  "   <tr><td width = \"30%\" align = \"left\"><a href=\"National.asp\"><im"
  "g src='../images/homepagenew/deshb.gif' border='0'></a></td>\r          "
  "          <td width = \"70%\" align = \"right\"><a href=\"../UPElection0"
  "7/default.asp\" target = \"new\"><img src='../image/tlink_UPElection07.g"
  "if' border='0'></a></td>\r                </tr>\r            </table>\r<"
  "/td></tr>\r<tr>\r<td width='7%'><img src='../images/arrow.gif' width='4'"
  " height='7'><br></td>\r<td><font face='au' size = '3'> <a href =natnews."
  "asp?nat=10desh1.asp >\xC0""}\xE6""\xE8""\xE2""\xBB""\xC9""\xB8"" \xD7""\xF0"""
  "\xB4"" \xDC""\xEA"" \xB7""\xE6"" \xB7""\xE3""\xDA"" \xE0""\xE6""\xE9""M<"
  "/a></font>\r</td>\r</tr>\r<tr>\r<td width='7%'><img src='../images/arrow"
  ".gif' width='4' height='7'><br></td>\r<td><font face='au' size = '3'> <a"
  " href =natnews.asp?nat=10desh1a.asp >\xB7""\xF0""\xDA""\xDC"" \xD1"" \xC2"""
  "\xF0""\x8C""\xE2""\xE8"" \xB7""\xE6""\xF0"" \xA9""\x88""\xC2""\xE6""\xCE"""
  "\xD9"" \xB7""\xE8"" \xA5""\xD9""\xE9""\xD7""\xE7""\xCC""</a></font>\r</t"
  "d>\r</tr>\r<tr>\r<td width='7%'><img src='../images/arrow.gif' width='4'"
  " height='7'><br></td>\r<td><font face='au' size = '3'> <a href =natnews."
  "asp?nat=10desh1b.asp >\xC2""\xFD""\x87""\xE6""\xDF"" \xD7""\xE9""\xB9""\xC1"""
  "\xE8""\xFC"" \xB7""\xE6""\xF0"" \xA5""S\xC2""\xCC""\xE6""\xDC"" \xE2""\xF0"""
  " \xC0""\xE9""^\xE8""</a></font>\r</td>\r</tr>\r<tr>\r<td width='7%'><img"
  " src='../images/arrow.gif' width='4' height='7'><br></td>\r<td><font fac"
  "e='au' size = '3'> <a href =natnews.asp?nat=10desh1c.asp >\xE7""\xD5""\xE3"""
  "\xE6""\xDA"" \xD1"" \xD2""\x8C""\xD8""\xE6""\xE2""\xF0""\xD3"" \xBB""\xE6"""
  "\xA2""\xDF"" \xD7""\xF0""\xB4"" \xE3""\xF1""\xB4""\xC7"" \xC2""\xA2""\xC2"""
  " \xBB""\xC7""\xB8""\xE6""</a></font>\r</td>\r</tr>\r<tr>\r<td colspan=\""
  "2\" height = '22'> </td>\r</tr>\r<tr>\r<td colspan='2'>\r<table border ="
  " \"0\" width = \"100%\" cellspacing = \"0\" cellpadding = \"0\">\r      "
  "          <tr><td width = \"30%\" align = \"left\"><a href=\"Sports.asp\""
  "><img src='../images/homepagenew/sportb.gif' border='0'></a></td>\r     "
  "               <td width = \"70%\" align = \"right\"><a href=\"#\" oncli"
  "ck=\"javascript:window.open('../Games/game.htm','Games','toolbar=no,dire"
  "ctories=no, resize=no, menubar=no,location=no,scrollbars=no,width=440,he"
  "ight=380,maximize=no,minimize=no,left=100,top=50')\"><img src='../image/"
  "tlink_game.gif' border='0'></a></td>\r                </tr>\r           "
  " </table>\r</td></tr>\r<tr>\r<td width='7%'><img src='../images/arrow.gi"
  "f' width='4' height='7'></td>\r<td><font face='au' size='3'> <a href =sp"
  "tnews.asp?spt=10am7a.asp >\xA7""\xA2""\xC1""\xD7""\xE6""\xD7"" \xDF"" \xC2"""
  "\xE8""\xE2""\xE8""\xD5""\xE8"" \xE2""\xF0"" \xC1""\xE6""\xA2""\xBF"" \xE2"""
  "\xE7""\xD7""\xE7""\xCC"" \xE2""\xA2""\xCC""\xE9""C \xD9""\xE3""\xE8""\xB4"""
  "<br></a></font></td>\r</tr>\r<tr>\r<td width='7%'><img src='../images/ar"
  "row.gif' width='4' height='7'></td>\r<td><font face='au' size='3'> <a hr"
  "ef =sptnews.asp?spt=10am7.asp >\xA5""\xE6""\xF2""S\xC5""\xF0""\xFE""\xE7"""
  "\xDC""\xD8""\xE6"" \xE2""\xF0"" \xE2""\xE8""\xB9"" \xDC""\xF0"" \xA7""\xA2"""
  "\x82""\xDC""\xF1""\xB4""\xC7"" \xD1"" x\xDC""\xF0""\xB4""\xBF""\xDA""<br"
  "></a></font></td>\r</tr>\r<tr>\r<td width='7%'><img src='../images/arrow"
  ".gif' width='4' height='7'></td>\r<td><font face='au' size='3'> <a href "
  "=sptnews.asp?spt=9sport1.asp >\xE7""\xB7""\xE7""\xDF""\xD8""\xE6""\xF0"""
  "\xB4"" \xD9""\xF0"" \xA5""\xE6""\xD8""\xDA""\xDC""\xF1""\xB4""\xC7"" \xB7"""
  "\xE6""\xF0"" \xE3""\xDA""\xE6""\xD8""\xE6""<br></a></font></td>\r</tr>\r"
  "<tr>\r<td width='7%'><img src='../images/arrow.gif' width='4' height='7'"
  "></td>\r<td><font face='au' size='3'> <a href =sptnews.asp?spt=09sportc."
  "asp >\xE2""\xE9""\xC2""\xDA""-4\xD1"" \xB7""\xE6""\xF1""\xD9"" \xE3""\xE6"""
  "\xF0""\xBB""\xE8"" \xBF""\xF5""\xCD""\xE8"" \xC5""\xE8""\xD7""?<br></a><"
  "/font></td>\r</tr>\r</table>\r</td>\r<td width='2%' valign='top'>&nbsp;<"
  "/td>\r<td valign='top' width='49%'>\r<table width='100%' border='0' cell"
  "spacing='0' cellpadding='0'>\r<tr><td colspan='2'>\r<table border = \"0\""
  " width = \"100%\" cellspacing = \"0\" cellpadding = \"0\">\r            "
  "    <tr><td width = \"30%\" align = \"left\"><a href=\"Int.asp\"><img sr"
  "c='../images/homepagenew/videshb.gif' border='0'></a></td>\r            "
  "        <td width = \"70%\" align = \"right\"><a href=\"../Takraar07/def"
  "ault.asp\" target = \"new\"><img src='../image/tlink_Takraar07.gif' bord"
  "er='0'></a></td>\r                </tr>\r            </table>\r</td></tr"
  ">\r<tr>\r<td width='7%'><img src='../images/arrow.gif' width='4' height="
  "'7'><br></td>\r<td><font face='au' size = '3'> <a href =intnews.asp?int="
  "10am6c.asp >\xD6""\xE6""\xDA""\xCC""-\xC2""\xE6""\xB7"" \xB7""\xF0"" \xA5"""
  "\xE6""\xE7""\xCD""\xFC""\xB7"" \xE7""\xDA""\xE0""\xCC""\xF0"" \xD7""\xF0"""
  "\xB4"" \xD7""\xC1""\xD5""\xEA""\xCC""\xE8""</a></font>\r</td>\r</tr>\r<t"
  "r>\r<td width='7%'><img src='../images/arrow.gif' width='4' height='7'><"
  "br></td>\r<td><font face='au' size = '3'> <a href =intnews.asp?int=10am6"
  "b.asp >\xD2""\xE2""\xE9""\xDA""\xFF""\xE6""\xE6"" \xC2""\xE7""\xDA""\xE1"""
  "\xCE"" \xB7""\xF0"" \xC2""\xFD""S\xCC""\xE6""\xDF"" \xB7""\xE6"" \xC2""\xE6"""
  "\xDC""\xD9"" \xB7""\xDA""\xF0"" \xA7""\xFC""\xDA""\xE6""\xD9""\xD3""</a>"
  "</font>\r</td>\r</tr>\r<tr>\r<td width='7%'><img src='../images/arrow.gi"
  "f' width='4' height='7'><br></td>\r<td><font face='au' size = '3'> <a hr"
  "ef =intnews.asp?int=10am6a.asp >\xD9""\xF0""\xC2""\xE6""\xDC""\xE8"" \xE2"""
  "\xF0""\xD9""\xE6"" \xD9""\xF0"" \xE3""\xE7""\xCD""\xD8""\xE6""\xDA"" \xC1"""
  "\xD7""\xE6"" \xB7""\xDA""\xE6""\xB0""</a></font>\r</td>\r</tr>\r<tr>\r<t"
  "d width='7%'><img src='../images/arrow.gif' width='4' height='7'><br></t"
  "d>\r<td><font face='au' size = '3'> <a href =intnews.asp?int=10am6.asp >"
  "\xB0""\xC5""\xD7""\xE8"" \xE7""\xDF""\xDF""\xE6""\xCE"" \xB7""\xE6"" \xE3"""
  "\xDC"" \xDF""\xE6""\xCC""\xE6""\xFC"" \xE2""\xF0"" \xE2""\xA2""\xD6""\xDF"""
  " \xD1"" \xA5""\xD7""\xF0""\xE7""\xDA""\xB7""\xE6""</a></font>\r</td>\r</"
  "tr>\r<tr>\r<td colspan=\"2\" height = '22'> </td>\r</tr>\r<tr>\r<td cols"
  "pan='2' width='231' >\r<table border = \"0\" width = \"100%\" cellspacin"
  "g = \"0\" cellpadding = \"0\">\r                <tr><td width = \"30%\" "
  "align = \"left\"><a href=\"Business.asp\"><img src='../images/homepagene"
  "w/businessb.gif' border='0'></a></td>\r                    <td width = \""
  "70%\" align = \"right\"><a href=\"../Budget07/default.asp\" target=\"new"
  "\"><img src='../image/tlink_Budget.gif' border='0'></a></td>\r          "
  "      </tr>\r            </table>\r</td></tr>\r<tr>\r<td width='7%'><img"
  " src='../images/arrow.gif' width='4' height='7'></td>\r<td>\r<font face="
  "'au' size='3'><a href =businews.asp?busi=10am8a.asp>\xA5""\xE6""\xA7""\xFC"""
  "\xE2""\xE8""\xA5""\xE6""\xA7""\xFC""\xE2""\xE8""\xA5""\xE6""\xA7""\xFC"""
  " \xD5""\xF1""\xB4""\xB7"" \xB7""\xE6"" \xB7""\xE6""\xF0""-\xD5""\xFD""\xE6"""
  "\xA2""\xC7""\xF0""\xC7"" \xB7""\xE6""\xC7""\xFC""</a></font></td>\r</tr>"
  "\r<tr>\r<td width='7%'><img src='../images/arrow.gif' width='4' height='"
  "7'></td>\r<td>\r<font face='au' size='3'><a href =businews.asp?busi=10am"
  "8.asp>\xD5""\xF1""\xB4""\xB7""\xE6""\xF0""\xB4"" \xB7""\xE6""\xF0"" \xDC"""
  "\xF0""\xD9""\xE6"" \xE3""\xE6""\xF0""\xBB""\xE6"" L\xC2""\xB0"" \xB7""\xF0"""
  " \xB9""\xA2""\xC7"" \xDF""\xE6""\xDC""\xF0"" \xBF""\xF0""\xB7""</a></fon"
  "t></td>\r</tr>\r<tr>\r<td width='7%'><img src='../images/arrow.gif' widt"
  "h='4' height='7'></td>\r<td>\r<font face='au' size='3'><a href =businews"
  ".asp?busi=10mobi2.asp>\xC2""\xA2""\xBC""\xFD""\xE3"" \xE2""\xE6""\xF1"" "
  "\xD7""\xF0""\xB4"" \xE7""\xD7""\xDC""\xF0""\xB4""\xBB""\xF0"" \xB7""\xF1"""
  "\xD7""\xDA""\xE6"" \xDF""\xE6""\xDC""\xF0"" \xD7""\xE6""\xF0""\xD5""\xE6"""
  "\xA7""\xDC""</a></font></td>\r</tr>\r<tr>\r<td width='7%'><img src='../i"
  "mages/arrow.gif' width='4' height='7'></td>\r<td>\r<font face='au' size="
  "'3'><a href =businews.asp?busi=10busi2d.asp>\xA5""\xE6""\xA7""\xFC""\xC5"""
  "\xE8"" \xE7""\xD9""\xDF""\xF0""\xE0""\xE6""\xB7""\xF4""\xB4"" \xB7""\xF4"""
  " \xD9""\xE9""\xB7""\xE2""\xE6""\xD9"" \xB7""\xE8"" \xA5""\xE6""\xE0""\xE6"""
  "\xA2""\xB7""\xE6""</a></font></td>\r</tr>\r</table>\r</td>\r</tr>\r</tab"
  "le><BR>\r<table border='0' cellspacing='0' cellpadding='1' width='98%' a"
  "lign='center' valign='top'>\r<tr><td align='center'>\r<a href='gkp.asp'>"
  "<img src=\"../image/gorakhpur.gif\" width=\"468\" height=\"49\" border=\""
  "0\"></a>\r</td></tr>\r<tr><td><img border='0' src='../images/homepagenew"
  "/citynews.gif'></td></tr>\r<tr bgcolor='ffffff'>\r<td valign='center'>\r"
  "<table border=0  cellspacing=0 cellpadding=0 width='100%' align=center v"
  "align='top' style='border-bottom: 1 solid #0058B0' height='117'>\r<tr>\r"
  "<td width='25%' height='24' valign='top'>\r<font face='AU' size='4' colo"
  "r='#B90000'>\xD9""\xA7""\xFC"" \xE7""\xCE""\xCB""\xDC""\xE8""</font>\r</"
  "td><td width='2%' align='center' height='24' valign='top'>\r<b><font col"
  "or='#B90000'>:</font></b>\r</td><td width='73%' height='24' valign='top'"
  ">\r<a href =delnews.asp?city=10uenhyy5.asp><font face='au' size='4'>\xC2"""
  "\xE9""\xE7""\xDC""\xE2"" \xE2""\xE9""\xCF""\xE6""\xDA"" \xB7""\xF0""\xA5"""
  "\xE6""\xCE""\xF0""\xE0""\xE6"" \xD9"" \xD7""\xE6""\xD9""\xD9""\xF0"" \xC2"""
  "\xDA"" \xA5""\xC7""\xB8""\xF0"" \xC2""\xFD""\xCE""\xF0""\xE0""\xE6""</fo"
  "nt></a>\r</td>\r</tr>\r<td width='25%' height='24' valign='top'>\r<font "
  "face='AU' size='4' color='#B90000'>\xD9""\xF4""\xB0""\xC7""\xE6""</font>"
  "\r</td><td width='2%' align='center' height='24' valign='top'>\r<b><font"
  " color='#B90000'>:</font></b>\r</td><td width='73%' height='24' valign='"
  "top'>\r<a href =bulnews.asp?city=10buld2.asp><font face='au' size='4'>\xDA"""
  "\xF1""\xDC""\xE8"" \xB7""\xF0"" \xE7""\xDC""\xB0"" \xD9""\xE3""\xE8""\xB4"""
  " \xE7""\xD7""\xDC"" \xDA""\xE3""\xF0"" \xE3""\xF1""\xB4"" \xDC""\xE6""\xF0"""
  "\xBB""</font></a>\r</td>\r</tr>\r<td width='25%' height='24' valign='top"
  "'>\r<font face='AU' size='4' color='#B90000'>\xBB""\xE6""\xF0""\xDA""\xB9"""
  "\xC2""\xE9""\xDA""</font>\r</td><td width='2%' align='center' height='24"
  "' valign='top'>\r<b><font color='#B90000'>:</font></b>\r</td><td width='"
  "73%' height='24' valign='top'>\r<a href =gkpnews.asp?city=10ilaha4.asp><"
  "font face='au' size='4'>\xE3""\xDA"" \xE2""\xE6""\xA2""\xE2"" \xB7""\xF0"""
  " \xE2""\xE6""\xCD"" \xDC""\xF0""\xCC""\xF0"" \xE3""\xF1""\xB4"" \xC1""\xE3"""
  "\xDA""</font></a>\r</td>\r</tr>\r<td width='25%' height='24' valign='top"
  "'>\r<font face='AU' size='4' color='#B90000'>\x9F""\xE6""\xE8""\xD9""\xBB"""
  "\xDA""</font>\r</td><td width='2%' align='center' height='24' valign='to"
  "p'>\r<b><font color='#B90000'>:</font></b>\r</td><td width='73%' height="
  "'24' valign='top'>\r<a href =srinews.asp?city=10juda3.asp><font face='au"
  "' size='4'>\xB7""\xE0""\xD7""\xE8""\xDA"" \xB7""\xF0"" \xB7""\xE9""\xC0"""
  " \xA5""\xD9""\xC0""\xE9""\xB0"" S\xCD""\xDC"" \xE3""\xF4""\xB4""\xBB""\xF0"""
  " \xC2""\xD8""\xFC""\xC5""\xD9"" \xB7""\xF0""\xB4""\xBC""\xFD""</font></a"
  ">\r</td>\r</tr>\r<td width='25%' height='24' valign='top'>\r<font face='"
  "AU' size='4' color='#B90000'>\xE2""\xE6""\xE7""\xE3""\xD5""\xE6""\xD5""\xE6"""
  "\xCE""</font>\r</td><td width='2%' align='center' height='24' valign='to"
  "p'>\r<b><font color='#B90000'>:</font></b>\r</td><td width='73%' height="
  "'24' valign='top'>\r<a href =gbdnews.asp?city=10kioui2.asp><font face='a"
  "u' size='4'>\xB0""\xB7"" \xE3""\xE8"" \xE7""\xCE""\xD9"" \xE7""\xD9""\xB7"""
  "\xE6""\xE3"", \xCC""\xDC""\xE6""\xB7"" \xA5""\xE6""\xF1""\xDA"" \xE7""\xC8"""
  "\xDA"" \xE7""\xD9""\xB7""\xE6""\xE3""</font></a>\r</td>\r</tr>\r</table>"
  "\r</td>\r</tr>\r</table>\r<table border='0' width = '98%' cellspacing='0"
  "' cellpadding='0' align='center' valign='top'>\r<tr>\r<td width='50%'>\r"
  "        <table border='0' width = '100%' cellspacing='2' cellpadding='1'"
  " align='center' valign='top'>\r        <tr>\r        <td width='49%' ali"
  "gn='center'>\r            <table border='0' width = '100%' cellspacing='"
  "2' cellpadding='0' align='center' valign='top'>\r                <tr><td"
  " align='center'><img src='../image/future_article.jpg' border='0'></td><"
  "/tr>\r                <tr><td>\r<a href='../future/ques.asp?id=4095' tar"
  "get='new'><font face='au' color='black'>&nbsp;<img src='../images/arrow."
  "gif' border='0'>&nbsp;\xD7""\xF0""\xDA""U\xE6"" \xE7""\xDF""\xDF""\xE6"""
  "\xE3""U \xB7""\xA4""\xD5"" \xCC""\xB7""\xA4"" \xE3""U\xE6""\xF0""\xBB""\xE6"""
  "? </font></a><font face='au' color='brown'>-\xD9""\xDA""\xF0""U\xE0""\xE6"""
  "</font><br>\r<a href='../future/ques.asp?id=4094' target='new'><font fac"
  "e='au' color='black'>&nbsp;<img src='../images/arrow.gif' border='0'>&nb"
  "sp;\xD7""\xE9""\xDB""\xE6""\xF0"" \xD9""\xE6""\xF1""\xB7""\xA4""\xDA""U\xE8"""
  " \xB7""\xA4""\xD5"" \xCC""\xB7""\xA4"" \xE7""\xD7""\xDC""\xF0""\xBB""\xE8"""
  "? </font></a><font face='au' color='brown'>-\xE0""\xE6""\xE6""\xCE""\xE6"""
  "\xD5""</font><br>\r<a href='../future/ques.asp?id=4093' target='new'><fo"
  "nt face='au' color='black'>&nbsp;<img src='../images/arrow.gif' border='"
  "0'>&nbsp;\xD7""\xE9""\xDB""\xE6""\xF0"" \xC1""\xE6""\xF2""\xD5"" \xB7""\xA4"""
  "\xD5"" \xCC""\xB7""\xA4"" \xE7""\xD7""\xDC""\xF0""\xBB""\xE8""? </font><"
  "/a><font face='au' color='brown'>-\xE7""\xD9""\xCC""\xF0""\xE0""\xE6""</"
  "font><br>\r<a href='../future/ques.asp?id=4092' target='new'><font face="
  "'au' color='black'>&nbsp;<img src='../images/arrow.gif' border='0'>&nbsp"
  ";\xD7""\xE9""\xDB""\xE6""\xF0"" \xCE""\xEA""\xE2""\xDA""U\xE8"" \xD9""\xE6"""
  "\xF1""\xB7""\xA4""\xDA""U\xE8"" \xB7""\xA4""\xD5"" \xE7""\xD7""\xDC""\xF0"""
  "\xBB""\xE8""? </font></a><font face='au' color='brown'>-\xD9""\xDF""\xE8"""
  "\xD9""</font><br>\r<a href='../future/ques.asp?id=4091' target='new'><fo"
  "nt face='au' color='black'>&nbsp;<img src='../images/arrow.gif' border='"
  "0'>&nbsp;\xE7""\xDF""\xCE""\xF0""\xE0""\xE6"" \xD8""\xE6""\x98""\xE6""\xE6"""
  " \xB7""\xA4""\xE6"" \xD8""\xE6""\xF0""\xBB"" \xB7""\xA4""\xD5"" \xCC""\xB7"""
  "\xA4""? </font></a><font face='au' color='brown'>-\xE2""\xA2""\xC1""\xD8"""
  "</font><br>\r<a href='../future/ques.asp?id=4089' target='new'><font fac"
  "e='au' color='black'>&nbsp;<img src='../images/arrow.gif' border='0'>&nb"
  "sp;\xDF""\xF1""\xDF""\xE6""\xE7""\xE3""U\xB7""\xA4"" \xC1""\xE8""\xDF""\xD9"""
  " \xB7""\xF1""\xA4""\xE2""\xE6"" \xE3""U\xE6""\xF0""\xBB""\xE6""? </font>"
  "</a><font face='au' color='brown'>-\xC2""\xE9""\xD9""\xE8""\xCC""</font>"
  "<br>\r<a href='../future/ques.asp?id=4088' target='new'><font face='au' "
  "color='black'>&nbsp;<img src='../images/arrow.gif' border='0'>&nbsp;\xD7"""
  "\xF0""\xDA""U\xE6"" \xE7""\xDF""\xDF""\xE6""\xE3""U \xB7""\xA4""\xD5"" \xCC"""
  "\xB7""\xA4"" \xE3""U\xE6""\xF0""\xBB""\xE6""? </font></a><font face='au'"
  " color='brown'>-\xDA""U\xE6""\xE3""\xE9""U\xDC""</font><br>\r           "
  "    <br> </td></tr>\r             </table>\r        </td>\r        <td w"
  "idth='2%' valign='top'><img src='../image/line.jpg' border='0'></td>\r  "
  "      <td width='49%' valign='top'>\r            <table border='0' width"
  " = '100%' cellspacing='0' cellpadding='0' align='center' valign='top'>\r"
  "                <tr><td>\r<table border = \"0\" width = \"100%\" cellspa"
  "cing = \"0\" cellpadding = \"0\">\r                <tr><td width = \"30%"
  "\" align = \"left\"><a href=\"../Jokes/default.asp\" target=\"new\"><img"
  " src='../images/homepagenew/jokesb.gif' border='0'></a></td>\r          "
  "          <td width = \"70%\" align = \"right\"><a href=\"../yearlyfutur"
  "e07/default.asp\" target=\"new\" ><img src='../image/tlink_yearlyfuture0"
  "7.gif' border='0'></a>'\r</td>\r                </tr>\r            </tab"
  "le>\r                </td></tr>\r                <tr><td>\r<a href='../j"
  "okes/default.asp'  onclick=\"javascript:window.open('../jokes/detail.asp"
  "?id=5845','','toolbars=no,height=430,width=510,top=0,left=100,resizable="
  "yes,scrollbars=yes')\"><font face='au' size='4' color='red'>\xBF""\xD7"""
  "\x88""\xB7""\xE6""\xDA""...</font></a><br>\r<img src='../jokes/images/10"
  "042007/homelatifa.jpg' border='0' align='left'>\r<font face='au' color='"
  "black' size='3'>\xB0""\xB7"" \xBF""\xD7""\x88""\xB7""\xE6""\xDA""\xE8"" "
  "\xD5""\xE6""\xD5""\xE6"" \xD9""\xF0"" \xE3""\xDF""\xE6"" \xD7""\xF0""\xB4"""
  " \xE3""\xE6""\xCD"" \xDC""\xE3""\xDA""\xE6""\xD8""\xE6"" \xA5""\xE6""\xF1"""
  "\xDA"" \xCC""\xE6""\xD5""\xE8""\xC1"" \xE7""\xD9""\xB7""\xE6""\xDC""\xB7"""
  "\xDA"" \xD6""v\xCC"" \xB7""\xF0"" \xE3""\xE6""\xCD""\xE6""\xA2""\xF0"" \xD7"""
  "\xF0""\xB4"" \xDA""\xB9""\xCC""\xF0"" \xE3""\xE9""\xB0"" \xB7""\xE3""\xE6"""
  "- \xD2""\xD5""\xF0""\xC5""\xE6"", \xA7""\xE2""\xF0"" \xB7""\xE6""\xDC""\xF0"""
  " \xCF""\xE6""\xBB""\xF0"" \xD7""\xF0""\xB4"" \xC2""\xE3""\xD9"" \xDC""\xF0"""
  "\xD9""\xE6"", \xCC""\xF0""\xDA""\xE6"" \xB7""\xCB""\xD8""\xE6""\x87""\xE6"""
  " \xE3""\xE6""\xF0""\xBB""\xE6""\xD0""\xD3""\xD6""v\xCC"" \xD5""\xE6""\xF0"""
  "\xDC""\xE6""- \xD2""\xD5""\xE6""\xD5""\xE6"", \xA7""\xE2""\xE2""\xF0"" \xCC"""
  "\xE6""\xF0"" \xA5""\x91""\xC0""\xE6"" \xE3""\xE6""\xF0""\xCC""\xE6"" \xE7"""
  "\xB7"" \xA5""\xE6""\xC2"" 500 L\xC2""\xB0"" \xB7""\xE6"" \xB0""\xB7"" \xD9"""
  "\xE6""\xF0""\xC5"" \xE7""\xD9""\xB7""\xE6""&nbsp;...</font>\r</td></tr>\r"
  "            </table>\r        </td>\r        </tr>\r        </table>\r</"
  "td>\r</tr>\r</table>\r<table border='0' width = '98%' cellspacing='0' ce"
  "llpadding='0' align='center' valign='top'>\r<tr>\r<td width='50%'>\r    "
  "    <table border='0' width = '100%' cellspacing='0' cellpadding='0' ali"
  "gn='center' valign='top'>\r        <tr>\r        <td width='49%' align='"
  "center' valign='top'>\r            <table border='0' width = '100%' cell"
  "spacing='0' cellpadding='0' align='center' valign='top'>\r              "
  "  <tr><td align='center'><a href=\"../ghar/advise.asp\" target=\"new\"><"
  "img src=\"../image/gharmash.jpg\" border='0'></a></td></tr>\r           "
  "     <tr><td valign='top'>\r<a href=\"../ghar/advise.asp#201\" target=\""
  "new\"><font face='au' color='red'>\xC2""\xFD""\xE0""\xD9""-</font><font "
  "face='au' color='#000000'>\xD7""\xF1""\xB4""\xD9""\xF0"" \xC8""\xEA""\xC7"""
  " \xC5""\xF0""v\xD9""\xF4""\xDC""\xE6""\xF2""\xC1""\xE8"" \xD7""\xF0""\xB4"""
  " \xB0""\xD7""\xB0""\xE2""\xE2""\xE8"" \xE7""\xB7""\xD8""\xE6"" \xE3""\xF1"""
  "\xD0"" \xA5""\xD5"" \xD7""\xF1""\xB4"" \xC8""\xEA""\xC7"" \xC5""\xF0""v\xD9"""
  "\xF4""\xDC""\xE6""\xF2""\xC1""\xE8"" \xD7""\xF0""\xA2""\xB4"" \xB0""\xD7"""
  "\xC5""\xF0""\xB7"" &nbsp;..</font></a>\r<font face='au' color='black'>\r"
  "</font><br>\r<a href=\"../ghar/advise.asp#200\" target=\"new\"><font fac"
  "e='au' color='red'>\xC2""\xFD""\xE0""\xD9""-</font><font face='au' color"
  "='#000000'>\xD7""\xF1""\xB4"" M\xDA""\xDC"" \xC7""\xF0""\xDF""\xDC""\xC2"""
  "\xD7""\xF0""\xB4""\xC5"" \xB7""\xE6""\xF0""\xE2""\xF0""\xFC"" \xB7""\xF0"""
  "  \xD5""\xE6""\xDA""\xF0"" \xD7""\xF0""\xB4"" \xC1""\xE6""\xD9""\xD9""\xE6"""
  " \xBF""\xE6""\xE3""\xCC""\xE6"" \xE3""\xEA""\xA2""\xD0"" \xA7""\xE2"" \xB7"""
  "\xE6""\xF0""\xE2""\xFC"" \xD7""\xF0""\xB4"" \xA5""\xE6""\xDF""\xF0""\xCE"""
  "\xD9"" &nbsp;..</font></a>\r<font face='au' color='black'>\r</font><br>\r"
  "<a href=\"../ghar/advise.asp#199\" target=\"new\"><font face='au' color="
  "'red'>\xC2""\xFD""\xE0""\xD9""-</font><font face='au' color='#000000'>v\xD8"""
  "\xE6"" \xDA""v\xCC""\xCE""\xE6""\xD9"" \xB7""\xDA""\xD9""\xF0"" \xE2""\xF0"""
  " \xE2""\xA2""\xB7""\xFD""\xE6""\xD7""\xB7""  \xD5""\xE8""\xD7""\xE6""\xE7"""
  "\xDA""\xD8""\xE6""\xA2"" \xE3""\xF4""\xD9""\xF0"" \xB7""\xE6"" \xB9""\xCC"""
  "\xDA""\xE6"" \xD5""\xC9""\xB8"" \xC1""\xE6""\xCC""\xE6"" \xE3""\xF1""\xD0"""
  " \xA7""\xE2"" \xD5""\xE6""\xDA""\xF0"" &nbsp;..</font></a>\r<font face='"
  "au' color='black'>\r</font><br>\r                </td></tr>\r           "
  "  </table>\r        </td>\r        <td width='2%' valign='top'><img src="
  "'../image/line.jpg' border='0'></td>\r        <td width='49%' valign='to"
  "p'>\r            <table border='0' width = '100%' cellspacing='0' cellpa"
  "dding='0' align='center' valign='top'>\r                <tr><td>\r<table"
  " border = \"0\" width = \"100%\" cellspacing = \"0\" cellpadding = \"0\""
  ">\r                <tr><td width = \"30%\" align = \"left\"><a href=\".."
  "/newcareer/default.asp\" target='new'><img border='0' src='../images/hom"
  "epagenew/career.gif'></a></td>\r                    <td width = \"70%\" "
  "align = \"center\">\r<a href=\"../result/default.asp\" target=\"new\"><i"
  "mg src='../image/tlink_result.gif' border='0'></a>\r                    "
  " </td>\r                </tr>\r            </table>\r                </t"
  "d></tr>\r                <tr><td>\r<a href='../newcareer/detail.asp?sec_"
  "id=1 & 505' ><font face='AU' size='4' color='#FF0000'>\xC2""\xFD""\xE0"""
  "\xD9"" \xD7""\xF0""\xB4"" \xE3""\xE8"" \xE7""\xC0""\xC2""\xE6"" \xE3""\xF1"""
  " \xE3""\xDC""</font></a><br>\r<img src='../newcareer/image/20070405/queh"
  "ome.jpg' border='0' align='left' width='50' height='60'>\r<font face='au"
  "' color='black' size='3'>\xA7""\xA2""\xC1""\xE8""\xE7""\xD9""\xD8""\xE7"""
  "\xDA""\xA2""\xBB"" \xB0""\xB7"" \xB0""\xF0""\xE2""\xE6"" \xC8""\xE8""\xCB"""
  "\xC7"" \xE3""\xF1"", \xE7""\xC1""\xE2""\xD7""\xF0""\xB4"" \xE2""\xC8""\xDC"""
  "\xCC""\xE6"" \xB7""\xF4"" \xDC""\xF0""\xB7""\xDA"" \xE0""\xE6""\xE6""\xD8"""
  "\xCE"" \xE3""\xE8"" \xE7""\xB7""\xE2""\xE8"" \xB7""\xF4"" \xE2""\xA2""\xCE"""
  "\xF0""\xE3"" \xE3""\xF4""\xBB""\xE6""\xD0"" \xD8""\xE3""\xE8"" \xB7""\xE6"""
  "\xDA""\x87""\xE6"" \xE3""\xF1"" \xE7""\xB7"" \xE3""\xDA"" \xE2""\xE6""\xDC"""
  " \xCE""\xF0""\xE0""\xE6"" \xB7""\xF0"" \xE7""\xDF""\xE0""\xDF""S\xCC""\xDA"""
  "\xE8""\xD8"" \xA5""\xE6""\xA7""\xFC""\xA5""\xE6""\xA7""\xFC""\xC5""\xE8"""
  " \xE2""\xA2""S\xCD""\xE6""\xD9""\xF4""\xB4"" \xD7""\xF0""\xB4"" \xCE""\xE6"""
  "\xE7""\xB9""\xDC""\xF0"" \xB7""\xE6"" \xD7""\xE6""\xBB""\xFC"" \xC2""\xFD"""
  "\xE0""\xE6""S\xCC"" \xB7""\xDA""\xD9""\xF0""</font><font face='arial'>.."
  ".</font>\r                </td></tr>\r            </table>\r        </td"
  ">\r        </tr>\r        </table>\r</td>\r</tr>\r</table>\r</td>\r<td w"
  "idth='130' valign='top' rowspan='2'>\r<table border='0' width='100%'><tr"
  "><td>\r<table width='100%' border='0' cellspacing='1' cellpadding='1' bg"
  "color = '#2C4FA0' >\r<tr>\r<td bgcolor = '#0b61c2' align='center' height"
  "='10'><a href=\"../shubh/section.asp?sec=3\" target=\"new\"><font face='"
  "AU' size='4' color='#FFFFFF'>\xD9""\xB0"" \xA9""\x88""\xC2""\xE6""\xCE"""
  "</font></a></td>\r</tr>\r<tr>\r<td bgcolor='#F3F3F4' align='right' heigh"
  "t='0'></td>\r</tr>\r<tr>\r<td bgcolor='#DDECFD' align='left' valign='top"
  "'>\r<table width='99%' border='0' cellspacing='0' cellpadding='0' >\r<tr"
  ">\r<td valign='top' width='1%'>&nbsp;<img src='../image/arrow.gif' borde"
  "r='0'>&nbsp;</td><td width='98%' align='left'><a href=\"../shubh/fullmai"
  "n.asp?sid=1&filename=new&foldername=20070408&sec=3\" target=\"new\"><fon"
  "t face='AU' size='3' color='#04135E'>\xD7""\xF1""v\xE2"" \xC8""\xF1""\xA4"""
  "v\xC5""U\xDA""U \xB0""\xC1"" \xE7""\xDA""U\x8B""\xD8""\xEA""</font></a>\r"
  "</td>\r</tr>\r<tr>\r<td valign='top' width='1%'>&nbsp;<img src='../image"
  "/arrow.gif' border='0'>&nbsp;</td><td width='98%' align='left'><a href=\""
  "../shubh/fullmain.asp?sid=1&filename=new&foldername=20070401&sec=3\" tar"
  "get=\"new\"><font face='AU' size='3' color='#04135E'>\xC5""U\xE6""\xA7"""
  "\xD7""\xF0""v\xE2"" \xB7""\xA4""\xE8"" \xE7""\xCE""\xE0""\xE6""\xE6"" \x99"""
  "\xE6""\xE6""\xD9"" \xCE""\xF0""\xD9""\xF0"" \xDF""\xE6""\xDC""\xE8"" ..<"
  "/font></a>\r</td>\r</tr>\r<tr>\r<td valign='top' width='1%'>&nbsp;<img s"
  "rc='../image/arrow.gif' border='0'>&nbsp;</td><td width='98%' align='lef"
  "t'><a href=\"../shubh/fullmain.asp?sid=1&filename=new&foldername=2007031"
  "8&sec=3\" target=\"new\"><font face='AU' size='3' color='#04135E'>\xB0"""
  "v\xE0""\xE6""\xD9"" \xB7""\xF0""\xA4"" \xE3""U\xCB""\xB7""\xF0""\xA4"" \xE2"""
  "\xF1""\xB4""\xC7""U\xDC""</font></a>\r</td>\r</tr>\r<tr>\r<td valign='to"
  "p' width='1%'>&nbsp;<img src='../image/arrow.gif' border='0'>&nbsp;</td>"
  "<td width='98%' align='left'><a href=\"../shubh/fullmain.asp?sid=1&filen"
  "ame=new&foldername=20070311&sec=3\" target=\"new\"><font face='AU' size="
  "'3' color='#04135E'>\xE2""\xE6""\xF0""\xD9""\xE8"" \xB7""\xA4""\xE6"" \xC2"""
  "\xF1""\xD9""\xDC"" \xC2""\xE8""\xE2""\xE8""</font></a>\r</td>\r</tr>\r</"
  "table>\r</td>\r</tr>\r</table>\r</td></tr>\r<tr><td valign='top'>\r<tabl"
  "e width='100%' border='0' cellspacing='1' cellpadding='1' bgcolor='#FFBC"
  "00'>\r<tr>\r<td bgcolor='brown' align='center'><a href=\"../Wallpaper/De"
  "fault.asp\" Target=\"new\"><font face='AU' size='4' color='#FFFFFF'><b>\xDF"""
  "\xE6""\xF2""\xDC""\xC2""\xF0""\xC2""\xDA""</b></font></a></td>\r</tr>\r<"
  "tr>\r<td >\r<table width='100%' border='0' cellspacing='0' cellpadding='"
  "0' >\r<tr>\r<td  valign='top' ALIGN='center' bgcolor='#FFBC00' height=96"
  ">\r<a href='../wallpaper/default.asp' target='new'><img border='0' src='"
  "../image/wall.gif' align='center' width=124 height=100></a></td>\r</tr>\r"
  "</td>\r</tr>\r</table>\r</td>\r</tr>\r</table>\r</td></tr>\r<tr><td vali"
  "gn='top'>\r<table width='100%' border='0' cellspacing='1' cellpadding='1"
  "' bgcolor = '#2C4FA0' height='150' align='top'>\r<tr>\r<td bgcolor='#316"
  "5CE' align='center'><a href=\"../Jaika/default.asp\" target=\"new\"><fon"
  "t face='AU' size='4' color='#FFFFFF'>\xC1""\xE6""\xD8""\xB7""\xE6""</fon"
  "t></a></td>\r</tr>\r<tr>\r<td bgcolor='#F3F3F4' align='right' height='0'"
  "></td>\r</tr>\r<tr>\r<td bgcolor='#F3F3F4' align='left' valign='top'>\r<"
  "table width='98%' border='0' cellspacing='2' cellpadding='0' align='cent"
  "er' bgcolor='#F3F3F4'>\r<tr>\r<td>\r<a href=\"../jaika/detail.asp?folder"
  "name=20041106&heading=\xB7""\xE6""\xC1""\xEA""-\xD7""\xB9""\xE6""\xD9""\xF0"""
  " \xDF""\xE6""\xDC""\xE6"" \xD9""\xD7""\xB7""\xE8""\xD9""&sid=3\" target="
  "\"new\"><font face='AU' size='3' color='blue'>\xB7""\xE6""\xC1""\xEA""-\xD7"""
  "\xB9""\xE6""\xD9""\xF0"" \xDF""\xE6""\xDC""\xE6"" \xD9""\xD7""\xB7""\xE8"""
  "\xD9""</font></a>\r<div align='justify'><font face='AU' size='3' color='"
  "#04135E'>\xD7""\xF1""\xCE""\xE6"" \xD7""\xF0""\xB4"" \xBB""\xDA""\xD7"" "
  "\x83""\xE6""\xE8"" \xB7""\xE6"" \xD7""\xE6""\xF0""\xD8""\xD9"" \xC7""\xE6"""
  "\xDC""\xF0""\xB4""\xD0"" \xA9""\xE2""\xD7""\xF0""\xB4"" \xE2""\xEA""\xC1"""
  "\xE8"", \xC2""\xE9""\xCE""\xE8""\xD9""\xE6"", \xD9""\xD7""\xB7"", \xA5"""
  "\xC1""\xDF""\xE6""\xD8""\xD9"" \xE2""\xD6""\xE8"" \xA5""\x91""\xC0""\xE8"""
  " \xCC""\xDA""\xE3"" \xE7""\xD7""\xDC""\xE6""\xB0""\xA2"" \xA5""\xE6""\xF1"""
  "\xDA"" \xC2""\xEA""\xDA""\xE8"" \xD5""\xF0""\xDC""\xD9""\xF0"" </font>\r"
  "</div></td>\r</tr>\r</table>\r</td>\r</tr>\r</table>\r</td></tr>\r<tr><t"
  "d valign='top'>\r<table width='100%' border='0' cellspacing='1' cellpadd"
  "ing='1' bgcolor = '#2C4FA0' >\r<tr>\r<td bgcolor='#0286bd' align='center"
  "' height='10'><a href=\"../Aakhar/default.asp\" target=\"new\"><font fac"
  "e='AU' size='4' color='#FFFFFF'>\xA5""\xE6""\xB9""\xDA""</font></a></td>"
  "\r</tr>\r<tr>\r<td bgcolor='#F3F3F4' align='right' height='0'></td>\r</t"
  "r>\r<tr>\r<td bgcolor='#eaf8fe' align='left' valign='top'>\r<table width"
  "='99%' border='0' cellspacing='2' cellpadding='0' align='center' >\r<tr>"
  "\r<td width='98%'>\r<a href=\"../Aakhar/detail.asp?section_id=1&id=362\""
  " target=\"new\"><font face='AU' size='3' color='blue'>\xE7""\xC8""\xDA"""
  " \xD9""\xE6""\xDA""\xE8"" \xB7""\xE6"" \xCE""\xCE""\xFC"" \xA9""\xB7""\xF0"""
  "\xDA""\xD9""\xF0"" \xD7""\xF0""\xB4"" \xC1""\xE9""\xC5""\xE8""\xA2"" </f"
  "ont></a><br>\r<div align='justify'><font face='AU' size='3' color='#0000"
  "00'> \xB7""\xC5""\xF7""\xC5""\xDA""\xC2""\xA2""\xE7""\xCD""\xD8""\xF4""\xB4"""
  " \xC2""\xDA"" \xE7""\xC5""\x8C""\xC2""\x87""\xE6""\xE8"" \xB7""\xE8"" \xDF"""
  "\xC1""\xE3"" \xE2""\xF0"" \xD5""\xE6""\xA2""\x82""\xDC""\xE6""\xCE""\xF0"""
  "\xE0""\xE6"" \xE2""\xF0"" \xE7""\xD9""\xDF""\xE6""\xFC""\xE2""\xD9"" \xDB"""
  "\xE6""\xF0""\xDC"" \xDA""\xE3""\xE8"" \xDC""\xF0""\xE7""\xB9""\xB7""\xE6"""
  " </font></div>\r</td>\r</tr>\r</table>\r</td>\r</tr>\r</table>\r</td></t"
  "r>\r<tr><td valign='top'>\r</td>\r</tr>\r</table>\r<table width='100%' b"
  "order='0' cellspacing='0' cellpadding='0'>\r<tr align='center' >\r<td bg"
  "color = '#2C4FA0' height='5'><font face='AU' color='#FFFFFF' size='3'>\xA5"""
  "\xD7""\xDA"" \xA9""\xC1""\xE6""\xDC""\xE6"" \xC1""\xD9""\xD7""\xCC""</fo"
  "nt></td>\r</tr>\r<tr>\r<td>\r<TABLE border=0 cellPadding=0 cellSpacing=0"
  " width=128  bgcolor='#F3F3F4' colspan=3>\r<tr>\r<td height='5' bgcolor='"
  "#ffffff'></td>\r</tr>\r<form method=post action='http://www.amarujala.co"
  "m/poll/pollhindi.asp' TARGET='win1'>\r<tr><td width='2' height='2' bgcol"
  "or='#ffffff'></td></tr>\r<tr><TD align=center vAlign=center colspan=2><t"
  "able width=100%>\r<tr><td colspan=2>\r\n<TABLE width=\"100%\">\r\n\t<TR>"
  "\r\n\t\t<TD colspan=\"4\" valign=\"top\"><div align=\"justify\"><font fa"
  "ce=\"au\">v\xD8""\xE6"" \xE2""\xE7""\xBF""\xD9"" \xCC""\xF0""\xB4""\xCE"""
  "\xE9""\xDC""\xB7""\xA4""\xDA"" m\xE6""\xDA""\xE6"" \xB7""\xF0""\xA4""\xB7"""
  "\xA4""\xA4""\xC2""\xDA"" \xD5""\xD9""\xE6"" \xE7""\xCC""\xDA""\xA2""\xBB"""
  "\xE6"" \xB7""\xA4""\xE6""\xC5""\xD9""\xE6"" \xB7""\xA4""\xDA""\xE6""\xF0"""
  "\xC7""\xB8""\xE6""\xF0""\xB4"" \xD6""\xE6""\xDA""\xCC""\xE8""\xD8""\xE6"""
  "\xF0""\xB4"" \xB7""\xA4""\xE6"" \xA5""\xC2""\xD7""\xE6""\xD9"" \xB7""\xA4"""
  "\xE3""\xE6"" \xC1""\xE6""\xD9""\xE6"" \xBF""\xE6""\xE7""\xE3""\xB0""?</f"
  "ont></div>\r\n\t\t</TD>\r\n\t</TR>\r\n\t<tr>\r\n\t\t\r\n\t\t<td bgcolor="
  "\"#dddddd\" align=\"center\" valign=\"top\"><input type=\"radio\" name=\""
  "ans\" value=\"1\"><br>\r\n\t\t\t<font face=\"au\">\r\n\t\t\t\t\xE3""\xE6"""
  "\xA1""\r\n\t\t\t</font>\r\n\t\t</td>\r\n\t\t\r\n\t\t<td bgcolor=\"#ddddd"
  "d\" align=\"center\" valign=\"top\"><input type=\"radio\" name=\"ans\" v"
  "alue=\"2\"><br>\r\n\t\t\t<font face=\"au\">\r\n\t\t\t\t\xD9""\xE3""\xE8"""
  "\xB4""\r\n\t\t\t</font>\r\n\t\t</td>\r\n\t\t\r\n\t\t<td bgcolor=\"#ddddd"
  "d\" align=\"center\" valign=\"top\"><input type=\"radio\" name=\"ans\" v"
  "alue=\"3\"><br>\r\n\t\t\t<font face=\"au\">\r\n\t\t\t\t\xC2""\xCC""\xE6"""
  " \xD9""\xE3""\xE8""\r\n\t\t\t</font>\r\n\t\t</td>\r\n\t\t\r\n\t</tr>\r\n"
  "\t<tr>\r\n\t\t\r\n\t</tr>\r\n</TABLE>\r\n</td></tr>\r<tr><td align=cente"
  "r colspan=2>\r<input name=submit onclick=\" newwin=window.open('','win1'"
  ",'toolbar=no,directories=no, resize=no, menubar=no,location=no,scrollbar"
  "s=no,width=510,height=400,maximize=no,minimize=no');\" style=\"color:#ff"
  "ffff;background:#3366CC;FONT-FAMILY: au; font-size: small;\" type=submit"
  " value=\"\xA5""\xE6""\xC2""\xB7""\xE6"" \xD7""\xCC""\"></td></tr></form>"
  "\r<tr><td align=right>\r<a href='#' onclick=\"javascript:window.open('ht"
  "tp://www.amarujala.com/poll/previoushindi.asp','win1','toolbar=no,direct"
  "ories=no, resize=no, menubar=no,location=no,scrollbars=no,width=320,heig"
  "ht=350,maximize=no,minimize=no');\"><font color='#0058B0' face='au' size"
  "='3'><p align='center'>\xE7""\xC2""\xC0""\xDC""\xE6"" \xE2""\xDF""\xF0"""
  "\xFC""\xFF""\xE6""\x87""\xE6""</p></font></a></td></tr>\r</table>\r</td>"
  "</tr>\r</table>\r</table>\r</td>\r</tr>\r<tr><td>\r<script language='jav"
  "ascript'>\rvar sURL='default.asp';\rfunction ticker()\r{\rsetTimeout('ti"
  "ckerupdate()',600*1000);\r}\rfunction tickerupdate()\r{\rwindow.location"
  ".href=sURL;\r}\r</script>\r<script language='javascript'>\rticker();\r</"
  "script>\r<table border='0' cellspacing='1' cellpadding='1' width='98%' a"
  "lign='center' valign='top'>\r<tr>\r<td align='center'>\r<html>\r\n\r\n<h"
  "ead>\r\n<meta http-equiv=\"Content-Language\" content=\"en-us\">\r\n<met"
  "a http-equiv=\"Content-Type\" content=\"text/html; charset=windows-1252\""
  ">\r\n<meta name=\"GENERATOR\" content=\"Microsoft FrontPage 4.0\">\r\n<m"
  "eta name=\"ProgId\" content=\"FrontPage.Editor.Document\">\r\n<title>New"
  " Page 1</title>\r\n</head>\r\n\r\n<body topmargin=\"0\" leftmargin=\"0\""
  ">\r\n\r\n\r\n\r\n\r\n<div align=\"center\">\r\n  <table border=\"0\" wid"
  "th=\"468\" cellspacing=\"0\" cellpadding=\"0\">\r\n    <tr>\r\n      <td"
  " valign=\"top\" align=\"left\"><map name=\"FPMap0\"><area href=\"#\" onc"
  "lick=\"javascript:window.open('../smsivr/sms.htm','sms','top=50,left=150"
  ",width=520,height=450,scrollbars=yes')\" style=\"cursor:hand\" shape=\"r"
  "ect\" coords=\"359, 48, 386, 59\"><area href=\"#\" onclick=\"javascript:"
  "window.open('../smsivr/ivr.htm','ivr','top=50,left=150,width=520,height="
  "450,scrollbars=yes')\" style=\"cursor:hand\" shape=\"rect\" coords=\"394"
  ", 49, 440, 59\"></map><img border=\"0\" src=\"../smsivr/banner_468x60.gi"
  "f\" usemap=\"#FPMap0\" width=\"468\" height=\"60\" style=\"border: 1 sol"
  "id #000000\"></td>\r\n    </tr>\r\n  </table>\r\n</div>\r\n\r\n\r\n\r\n\r"
  "\n</body>\r\n\r\n</html>\r\n\r</td>\r</tr>\r</table>\r<A href = 'archive"
  "'>\r<table border='0' cellspacing='1' cellpadding='1' width='98%' align="
  "'center' valign='top'>\r<tr bgcolor='ffffff'>\r<td valign='center' bgcol"
  "or='#9977ff'>\r<table border=0 cellspacing=0 cellpadding=4 width='100%' "
  "align=center valign=top>\r<form name='amar'>\r<tr>\r<TD bgcolor='#3366CC"
  "' align=center width='20%'><font face=au size=3 color='#ffffff'>\xE7""\xC2"""
  "\xC0""\xDC""\xF0"" \xA5""\xA2""\xB7""\xA4""</font></td>\r<TD bgcolor='#f"
  "fffeb' align=center width='20'><font face=au size=3>&nbsp;&nbsp;\xCC""\xE6"""
  "\xDA""\xE8""\xB9""</font></td>\r<TD bgcolor='#ffffeb' align=center width"
  "='20%'><font face='arial' size=1><select name=day style='font-family: au"
  ";font-size=16'>\r<option value=''>\xCC""\xE6""\xDA""\xE8""\xB9"" &nbsp;&"
  "nbsp;&nbsp;&nbsp;</option>\r<option value= 10 selected>10</option>\r<opt"
  "ion value=01 >1</option>\r<option value=02 >2</option>\r<option value=03"
  " >3</option>\r<option value=04 >4</option>\r<option value=05 >5</option>"
  "\r<option value=06 >6</option>\r<option value=07 >7</option>\r<option va"
  "lue=08 >8</option>\r<option value=09 >9</option>\r<option value=10 >10</"
  "option>\r<option value=11 >11</option>\r<option value=12 >12</option>\r<"
  "option value=13 >13</option>\r<option value=14 >14</option>\r<option val"
  "ue=15 >15</option>\r<option value=16 >16</option>\r<option value=17 >17<"
  "/option>\r<option value=18 >18</option>\r<option value=19 >19</option>\r"
  "<option value=20 >20</option>\r<option value=21 >21</option>\r<option va"
  "lue=22 >22</option>\r<option value=23 >23</option>\r<option value=24 >24"
  "</option>\r<option value=25 >25</option>\r<option value=26 >26</option>\r"
  "<option value=27 >27</option>\r<option value=28 >28</option>\r<option va"
  "lue=29 >29</option>\r<option value=30 >30</option>\r<option value=31 >31"
  "</option>\r</select></font></td>\r<TD bgcolor='#ffffeb' valign=middle wi"
  "dth='10%'><font face=au size=3>\xD7""\xE6""\xE3""</font></td>\r<TD bgcol"
  "or='#ffffeb' valign=middle width='15%'><font face='arial' size=1>\r<sele"
  "ct name='month' style='font-family:au; font-size=16'>\r<option value='12"
  "'>\xE7""\xCE""\xE2""{\xD5""\xDA""</option>\r<option value='11'>\xD9""\xDF"""
  "{\xD5""\xDA""</option>\r<option value='10'>\xA5""v\xCC""\xEA""\xD5""\xDA"""
  "</option>\r<option value='09'>\xE7""\xE2""\xCC""{\xD5""\xDA""</option>\r"
  "<option value='08'>\xA5""\xBB""S\xCC""</option>\r<option value='07'>\xC1"""
  "\xE9""\xDC""\xE6""\xA7""\xFC""</option>\r<option value='06'>\xC1""\xEA"""
  "\xD9""</option>\r<option value='05'>\xD7""\xA7""\xFC""</option>\r<option"
  " value='04'Selected>\xA5""\xC2""\xFD""\xF1""\xDC""</option>\r<option val"
  "ue='03'>\xD7""\xE6""\xBF""\xFC""</option>\r<option value='02'>\xC8""\xDA"""
  "\xDF""\xDA""\xE8""</option>\r<option value='01'>\xC1""\xD9""\xDF""\xDA"""
  "\xE8""</option>\r</select></font></td>\r<TD bgcolor='#ffffeb' align=cent"
  "er width='10%'><font face=au size=3>\xDF""\xE1""\xFC""</font></td>\r<TD "
  "bgcolor='#ffffeb' align=center width='20%'><font face='arial' size=1>\r<"
  "select name='year'>\r<option value='2007' Selected>2007&nbsp;</option>\r"
  "<option value='2006'>2006&nbsp;</option>\r<option value='2005'>2005&nbsp"
  ";</option>\r<option value='2004'>2004&nbsp;</option>\r<option value='200"
  "3'>2003&nbsp;</option>\r<option value='2002'>2002&nbsp;</option>\r<optio"
  "n value='2001'>2001&nbsp;</option>\r</font>\r</select></td>\r<TD bgcolor"
  "='#ffffeb' valign=center width='10%'><img src='../image/go1.gif' onclick"
  "='sel()'></td>\r</tr>\r</form>\r</table>\r</td></tr>\r</table>\r</a>\r</"
  "td></tr>\r</table>\r<table border='0' cellpadding='0' width='760' cellsp"
  "acing='0' >\r<tr align='left'>\r<td height='2'></td>\r</tr>\r<tr>\r<td h"
  "eight='2'></td>\r</tr>\r<tr>\r<td bgcolor='#949c9c'align='center'> <a hr"
  "ef='http://www.amarujala.com/aboutus/aboutus.htm'><font size='1' face='A"
  "rial' color='#ffffff'>About us</font></a><font size='1' face='Arial' col"
  "or='#ffffff'> | </font><a href='http://www.amarujala.com/aboutus/contact"
  "us.htm'><font size='1' face='Arial' color='#ffffff'>Contactus</font></a>"
  "<font size='1' face='Arial' color='#ffffff'> | </font><a href='http://ww"
  "w.amarujala.com/aboutus/mediakit.htm'><font size='1' face='Arial' color="
  "'#ffffff'>Mediakit</font></a><font size='1' face='Arial' color='#ffffff'"
  "> | </font><a href='http://www.amarujala.com/sitemap/default.asp'><font "
  "size='1' face='Arial' color='#ffffff'>Site Map</font></a>\r</tr>\r<tr>\r"
  "<td height='2'></td>\r</tr>\r<tr>\r<td bgcolor='#cb2828'align='center'> "
  "<font color='#ffffff' size='1' face='Arial'>\xA9""AmarUjala.com\r2007</f"
  "ont></td>\r</tr>\r</table>\r  </center>\r<script language=\"javascript\""
  ">\rlogo.src = \"../image/au_main.gif\";\rtopright.src=\"../image/worldcu"
  "p07.gif\";\rphome.src = \"10home.jpg\";\r</script>\r<script type='text/j"
  "avascript' src='10e_var6_home.js'></script>\r<script type='text/javascri"
  "pt' src='../menu_com.js'></script>\r<script language='javascript'>\rwind"
  "ow.open('../bharatmatrimony/bm_amarujala-genu-may17_300x300.html','bhara"
  "tmatri','width=300,height=300,top=280,left=5,resize=yes')\r</script>\r<s"
  "cript language='javascript'>\rwindow.open('../shopping/Current/index.htm"
  "','shopping_Nov','width=425,height=189,top=280,left=320,resize=yes')\r</"
  "script>\r</body>\r</html>\r"
;



// This one is really UTF-8, even though "www.jagran.com" matches JAGRAN
const char* kTestUrl52b = "http://www.jagran.com/default.aspx";
const char kTeststr52b[] =
// Produced by stringify.cc on 2007-04-10 12:49 from file jagran_com.html
  "\r\n\r\n<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\""
  " \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">\r\n\r\n<ht"
  "ml xmlns=\"http://www.w3.org/1999/xhtml\" >\r\n<head><title>\r\n\tJagran"
  ".com - World's Largest Hindi Portal\r\n</title>\r\n<STYLE TYPE=\"text/cs"
  "s\">\r\n<!-- /* $WEFT -- Created by: Webmaster (webmaster@jagran.com) on"
  " 9/5/2006 -- */\r\n  @font-face {\r\n    font-family: Jagran;\r\n    fon"
  "t-style:  normal;\r\n    font-weight: 700;\r\n    src: url(JAGRAN9.eot);"
  "\r\n  }\r\n  @font-face {\r\n    font-family: Jagran;\r\n    font-style:"
  "  normal;\r\n    font-weight: normal;\r\n    src: url(JAGRAN10.eot);\r\n"
  "  }\r\n-->\r\n</STYLE>\r\n\r\n<script type=\"text/javascript\" language="
  "\"JavaScript1.2\" src=\"menu/stmenu.js\"></script>\r\n<script type=\"tex"
  "t/javascript\" language=\"javascript\">\r\n\r\nfunction editor()\r\n{\r\n"
  "window.open(\"http://www3.jagran.com/editor/editor.asp\",\"editor\",\"to"
  "p=0,left=20,height=560,width=510,status=no,toolbar=no,menubar=no,locatio"
  "n=no\",\"1\");\r\n}\r\nfunction jagranjobs()\r\n{\r\nwindow.open(\"http:"
  "//www.jagran.com/jagranjobs.htm\",\"jagranjobs\",\"top=0,left=20,height="
  "460,width=560,status=no,toolbar=no,menubar=no,location=no,scrollbars=yes"
  "\",\"1\");\r\n}\r\nfunction banner()\r\n{\r\nwindow.open(\"\",\"bannerji"
  "\",\"width=700,height=500,toolbar=no,location=no,directories=no,menubar="
  "no,scrollbars=yes,status=no,resizable=yes\",\"1\");\r\n}\r\n\tfunction f"
  "riend()\r\n\t{\r\n\twindow.open(\"\",\"abc\",\"height=380,width=400,stat"
  "us=no,toolbar=no,menubar=no,location=no\",\"1\");\r\n\t}\r\n\tfunction M"
  "M_preloadImages() {\r\n\tvar d=document; if(d.images){ if(!d.MM_p) d.MM_"
  "p=new Array();\r\n\tvar i,j=d.MM_p.length,a=MM_preloadImages.arguments; "
  "for(i=0; i<a.length; i++)\r\n\tif (a[i].indexOf(\"#\")!=0){ d.MM_p[j]=ne"
  "w Image; d.MM_p[j++].src=a[i];}}\r\n\tinitGroup('sample')\r\n\t}\r\n\tva"
  "r t1=new Image(7,7); t1.src=\"images/tag1r.gif\";\r\n\tfunction pic(str)"
  "\r\n\t{\r\n\t\twindow.open(str,\"pic\",\"top=10,left=10,width=700,toolba"
  "r=no,location=no,directories=no,menubar=no,scrollbars=yes,status=no,resi"
  "zable=yes\",\"1\");\r\n\t}\r\n\t\tfunction pic1(str)\r\n\t{\r\n\t\twindo"
  "w.open(str,\"pic\",\"top=10,left=10,height=250,width=600,toolbar=no,loca"
  "tion=no,directories=no,menubar=no,scrollbars=yes,status=no,resizable=yes"
  "\",\"1\");\r\n\t}\r\n\tfunction openwindow()\r\n\t{\r\n\twindow.open(\"\""
  ",\"abc\",\"top=0,left=0,height=300,width=300,status=no,toolbar=no,menuba"
  "r=no,location=no\",\"1\");\r\n\t}\r\n-->\r\n  </script>\r\n<script langu"
  "age=javascript>\r\nfunction changestyle()\r\n{\r\nvar lt = document.getE"
  "lementsByTagName('link');\r\nvar agt=navigator.userAgent.toLowerCase();\r"
  "\nvar is_ie     = ((agt.indexOf(\"msie\") != -1) && (agt.indexOf(\"opera"
  "\") == -1));\r\n    if (is_ie==false)\r\n    {\r\n        lt[0].href=\"n"
  "ews/news.css\"\r\n    }\r\n}\r\n</script> \r\n\r\n    <link type=\"text/"
  "css\" rel=\"Stylesheet\" href=\"news/Jagran.css\" /></head>\r\n<body bac"
  "kground=\"images/bg.gif\" leftmargin=0 topmargin=0 onload=\"changestyle("
  ")\">\r\n<table width=100% cellpadding=0 cellspacing=0 border=0>\r\n<tr><"
  "td width=\"24%\">\r\n<table border=\"0\" cellPadding=\"0\" cellSpacing=\""
  "0\" width=\"100%\" align=\"left\">\r\n  <tr>\r\n    <td vAlign=\"top\">\r"
  "\n      <table align=\"left\" background=\"images/logobg.gif\" style=\"b"
  "ackground-repeat:no-repeat\" border=\"0\" cellPadding=\"0\" cellSpacing="
  "\"0\" width=\"100%\">\r\n        <tr><td align=\"left\" vAlign=\"top\"><"
  "img align=absmiddle border=\"0\" src=\"images/logoimg.gif\" width=\"143\""
  " height=\"75\" alt=\"\">\r\n        <font face=arial></font><font face=\""
  "Jagran\" size=\"3\" color=darkblue><b><br />&nbsp;&nbsp;\xE0""\xA4""\x86"""
  "\xE0""\xA4""\xAA""\xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\x85"""
  "\xE0""\xA4""\xAA""\xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4""\x86"""
  "\xE0""\xA4""\xAA""\xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\xAA"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xB8""</b></font>\r\n        </td></tr>\r\n"
  "      </table>\r\n    </td>\r\n  </tr>\r\n</table>\r\n</td><td valign=to"
  "p height=\"75\" width=60%>\r\n<!-- Banner & Ticker Table -->    \r\n<tab"
  "le border=\"0\" cellPadding=\"0\" cellSpacing=\"0\" width=\"100%\" align"
  "=\"left\">\r\n  <tr><td align=\"left\" vAlign=\"top\"><img alt=\"\" bord"
  "er=\"0\" height=\"2\" src=\"images/spacer.gif\" width=\"8\"></td></tr>\r"
  "\n  <tr>\r\n    <td valign=bottom align=\"left\">\r\n<a href=\"http://sh"
  "opping.jagran.com/\" target=_blank><img alt=\"\" src=\"images/shopbane1."
  "gif\" border=0 >    </td>\r\n  </tr>\r\n  <tr><td align=\"left\" vAlign="
  "\"top\"><img alt=\"\" border=\"0\" height=\"2\" src=\"images/spacer.gif\""
  " width=\"8\"></td></tr>\r\n  <tr>\r\n\t<th align=\"center\" valign=botto"
  "m>\r\n\t\r\n   <img alt=\"\" border=\"0\" src=\"images/news.gif\" width="
  "\"62\" height=\"15\">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<font face=\"Ja"
  "gran\" color=\"blue\" size=\"3\"><marquee bgcolor=\"white\" align=\"midd"
  "le\" width=\"82%\" scrollamount=\"3\">\r\n        \xE0""\xA4""\xAA""\xE0"""
  "\xA5""\x82""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\xB5""\xE0"""
  "\xA5""\x8B""\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0"""
  "\xA4""\xB0""\xC2""\xA0""\xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4"""
  "\xB5""\xE0""\xA4""\xBF""\xE0""\xA4""\x95""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\xB8"" \xE0""\xA4""\xAA""\xE0""\xA4""\xB0"" \xE0""\xA4""\xA7""\xE0""\xA5"""
  "\x8D""\xE0""\xA4""\xAF""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8"" \xE0""\xA4"""
  "\xA6""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF""\xE0""\xA4""\xBE"" \xE0""\xA4"""
  "\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x8F"": \xE0""\xA4""\xB6""\xE0""\xA5"""
  "\x87""\xE0""\xA4""\x96""\xE0""\xA4""\xBE""\xE0""\xA4""\xB5""\xE0""\xA4"""
  "\xA4""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<img src='images/bulet1."
  "gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\r\n "
  "       \r\n        \xE0""\xA4""\xAA""\xE0""\xA4""\xBE""\xE0""\xA4""\x95"""
  " \xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4""\xAA""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\x87""\xE0""\xA4""\xAA""\xE0""\xA4""\xB2""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\x87""\xE0""\xA4""\xA8"" \xE0""\xA4""\x97""\xE0""\xA5""\x88"""
  "\xE0""\xA4""\xB8"" \xE0""\xA4""\xAC""\xE0""\xA4""\x82""\xE0""\xA4""\x9F"""
  "\xE0""\xA4""\xB5""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\x95""\xE0""\xA5""\x8B"" \xE0""\xA4""\xAE""\xE0""\xA4""\x82"""
  "\xE0""\xA4""\x9C""\xE0""\xA5""\x82""\xE0""\xA4""\xB0""\xE0""\xA5""\x80"""
  " \xE0""\xA4""\xA6""\xE0""\xA5""\x80""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp"
  ";&nbsp;<img src='images/bulet1.gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbs"
  "p;&nbsp;&nbsp;&nbsp;&nbsp;\r\n        \r\n        \xE0""\xA4""\x9A""\xE0"""
  "\xA5""\x81""\xE0""\xA4""\xA8""\xE0""\xA4""\xBE""\xE0""\xA4""\xB5"" \xE0"""
  "\xA4""\x86""\xE0""\xA4""\xAF""\xE0""\xA5""\x8B""\xE0""\xA4""\x97"" \xE0"""
  "\xA4""\xB8""\xE0""\xA5""\x87"" \xE0""\xA4""\x9F""\xE0""\xA4""\x95""\xE0"""
  "\xA4""\xB0""\xE0""\xA4""\xBE""\xE0""\xA4""\xB5"" \xE0""\xA4""\x95""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\xB0""\xE0""\xA4""\xBE""\xE0""\xA4""\xB8""\xE0"""
  "\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA5""\x87"" \xE0""\xA4""\xAA""\xE0"""
  "\xA4""\xB0"" \xE0""\xA4""\xA8""\xE0""\xA4""\xB9""\xE0""\xA5""\x80""\xE0"""
  "\xA4""\x82"": \xE0""\xA4""\xAD""\xE0""\xA4""\xBE""\xE0""\xA4""\x9C""\xE0"""
  "\xA4""\xAA""\xE0""\xA4""\xBE""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;"
  "<img src='images/bulet1.gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbsp;&nbsp"
  ";&nbsp;&nbsp;&nbsp;\r\n        \r\n        \xE0""\xA4""\x86""\xE0""\xA4"""
  "\xB0""\xE0""\xA4""\x95""\xE0""\xA5""\x8D""\xE0""\xA4""\xB7""\xE0""\xA4"""
  "\xA3"" \xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0""\xA4""\xAE""\xE0""\xA4"""
  "\xB2""\xE0""\xA5""\x87"" \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4"""
  "\x82"" \xE0""\xA4""\xB8""\xE0""\xA4""\xAD""\xE0""\xA5""\x80"" \xE0""\xA4"""
  "\xB5""\xE0""\xA4""\xBF""\xE0""\xA4""\x95""\xE0""\xA4""\xB2""\xE0""\xA5"""
  "\x8D""\xE0""\xA4""\xAA""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"" \xE0""\xA4"""
  "\xAA""\xE0""\xA4""\xB0"" \xE0""\xA4""\xB9""\xE0""\xA5""\x8B""\xE0""\xA4"""
  "\x97""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB5""\xE0""\xA4""\xBF""\xE0""\xA4"""
  "\x9A""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0"": \xE0""\xA4""\x85""\xE0""\xA4"""
  "\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\x9C""\xE0""\xA5""\x81""\xE0""\xA4"""
  "\xA8""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<img src='images/bulet1."
  "gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\r\n "
  "       \r\n        \xE0""\xA4""\xA8""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xAF""\xE0""\xA4""\xAE""\xE0""\xA5""\x82"""
  "\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA4""\xBF"""
  " \xE0""\xA4""\xB8""\xE0""\xA5""\x87""\xE0""\xA4""\xA0""\xE0""\xA4""\xA8"""
  "\xE0""\xA4""\xBE""\xC2""\xA0""\xE0""\xA4""\x95""\xE0""\xA4""\xBE"" \xE0"""
  "\xA4""\x97""\xE0""\xA5""\x81""\xE0""\xA4""\x9C""\xE0""\xA4""\xB0""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\xA4"" \xE0""\xA4""\xB9""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\x88""\xE0""\xA4""\x95""\xE0""\xA5""\x8B""\xE0""\xA4""\xB0""\xE0"""
  "\xA5""\x8D""\xE0""\xA4""\x9F"" \xE0""\xA4""\xB8""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\x87""\xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0"""
  "\xA5""\x80""\xE0""\xA4""\xAB""\xE0""\xA4""\xBE""&nbsp;&nbsp;&nbsp;&nbsp;"
  "&nbsp;&nbsp;&nbsp;<img src='images/bulet1.gif' align=bottom >&nbsp;&nbsp"
  ";&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\r\n        \r\n        \xE0""\xA4"""
  "\x95""\xE0""\xA5""\x87""\xE0""\xA4""\x82""\xE0""\xA4""\xA6""\xE0""\xA5"""
  "\x8D""\xE0""\xA4""\xB0"" \xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4"""
  "\xA8""\xE0""\xA4""\x95""\xE0""\xA5""\x8D""\xE0""\xA4""\xB8""\xE0""\xA4"""
  "\xB2""\xE0""\xA5""\x80"" \xE0""\xA4""\xB9""\xE0""\xA4""\xBF""\xE0""\xA4"""
  "\x82""\xE0""\xA4""\xB8""\xE0""\xA4""\xBE"" \xE0""\xA4""\xAA""\xE0""\xA4"""
  "\xB0"" 24\xC2""\xA0""\xE0""\xA4""\x85""\xE0""\xA4""\xAA""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xB0""\xE0""\xA5""\x88""\xE0""\xA4""\xB2"" \xE0""\xA4""\x95"""
  "\xE0""\xA5""\x8B"" \xE0""\xA4""\xAC""\xE0""\xA5""\x88""\xE0""\xA4""\xA0"""
  "\xE0""\xA4""\x95"" \xE0""\xA4""\xAC""\xE0""\xA5""\x81""\xE0""\xA4""\xB2"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x88""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;"
  "&nbsp;<img src='images/bulet1.gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbsp"
  ";&nbsp;&nbsp;&nbsp;&nbsp;\r\n        \r\n        \xE0""\xA4""\x95""\xE0"""
  "\xA4""\xB6""\xE0""\xA5""\x8D""\xE0""\xA4""\xAE""\xE0""\xA5""\x80""\xE0"""
  "\xA4""\xB0"" \xE0""\xA4""\xB8""\xE0""\xA5""\x87"" \xE0""\xA4""\xB8""\xE0"""
  "\xA5""\x87""\xE0""\xA4""\xA8""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9""\xE0"""
  "\xA4""\x9F""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\xAA""\xE0""\xA4""\xB0"" \xE0""\xA4""\xB5""\xE0""\xA4""\xBF""\xE0"""
  "\xA4""\x9A""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0"" \xE0""\xA4""\x95""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\xB2""\xE0""\xA4""\xBF""\xE0""\xA4""\x8F"" \xE0"""
  "\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xA7""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\xA8""\xE0""\xA4""\xAE""\xE0""\xA4""\x82""\xE0"""
  "\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x80"" \xE0"""
  "\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4""\xAC""\xE0""\xA5""\x88""\xE0"""
  "\xA4""\xA0""\xE0""\xA4""\x95"" \xE0""\xA4""\x95""\xE0""\xA5""\x80""&nbsp"
  ";&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<img src='images/bulet1.gif' align="
  "bottom >&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\r\n        \r\n"
  "        \xE0""\xA4""\xAE""\xE0""\xA4""\xB2""\xE0""\xA5""\x87""\xE0""\xA4"""
  "\xB0""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF""\xE0""\xA4""\xBE"" \xE0""\xA4"""
  "\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xAD""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xB5""\xE0""\xA4""\xBF""\xE0""\xA4""\xA4"" 45\xC2""\xA0"""
  "\xE0""\xA4""\x9C""\xE0""\xA4""\xBF""\xE0""\xA4""\xB2""\xE0""\xA5""\x8B"""
  "\xE0""\xA4""\x82"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\x95"""
  "\xE0""\xA4""\xB2""\xE0""\xA5""\x87""\xE0""\xA4""\x95""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\x9F""\xE0""\xA4""\xB0""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"""
  " \xE0""\xA4""\x95""\xE0""\xA5""\x80"" \xE0""\xA4""\xAC""\xE0""\xA5""\x88"""
  "\xE0""\xA4""\xA0""\xE0""\xA4""\x95"" \xE0""\xA4""\xA6""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xB2""\xE0""\xA5""\x8D""\xE0""\xA4""\xB2""\xE0""\xA5""\x80"""
  " \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\x86"""
  "\xE0""\xA4""\x9C""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<img src='im"
  "ages/bulet1.gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp"
  ";&nbsp;\r\n        \r\n        \xE0""\xA4""\xA6""\xE0""\xA4""\xBF""\xE0"""
  "\xA4""\xB2""\xE0""\xA5""\x8D""\xE0""\xA4""\xB2""\xE0""\xA5""\x80"" \xE0"""
  "\xA4""\xA8""\xE0""\xA4""\x97""\xE0""\xA4""\xB0"" \xE0""\xA4""\xA8""\xE0"""
  "\xA4""\xBF""\xE0""\xA4""\x97""\xE0""\xA4""\xAE"" \xE0""\xA4""\x95""\xE0"""
  "\xA4""\xBE"" \xE0""\xA4""\xB8""\xE0""\xA5""\x80""\xE0""\xA4""\xB2""\xE0"""
  "\xA4""\xBF""\xE0""\xA4""\x82""\xE0""\xA4""\x97"" \xE0""\xA4""\x85""\xE0"""
  "\xA4""\xAD""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xA8"" \xE0""\xA4""\xAB""\xE0""\xA4""\xBF""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\x9A""\xE0""\xA4""\xBE""\xE0""\xA4""\xB2""\xE0""\xA5""\x82""&nbsp;"
  "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<img src='images/bulet1.gif' align=b"
  "ottom >&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\r\n        \r\n "
  "       \xE0""\xA4""\xB0""\xE0""\xA4""\xA3""\xE0""\xA4""\x9C""\xE0""\xA5"""
  "\x80"" \xE0""\xA4""\x95""\xE0""\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xA4""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8""\xE0""\xA5""\x8B""\xE0""\xA4"""
  "\x82"" \xE0""\xA4""\x94""\xE0""\xA4""\xB0"" \xE0""\xA4""\x95""\xE0""\xA5"""
  "\x8B""\xE0""\xA4""\x9A""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"" \xE0""\xA4"""
  "\x95""\xE0""\xA5""\x80"" \xE0""\xA4""\xAC""\xE0""\xA5""\x88""\xE0""\xA4"""
  "\xA0""\xE0""\xA4""\x95"" \xE0""\xA4""\xAE""\xE0""\xA5""\x81""\xE0""\xA4"""
  "\x82""\xE0""\xA4""\xAC""\xE0""\xA4""\x88"" \xE0""\xA4""\xAE""\xE0""\xA5"""
  "\x87""\xE0""\xA4""\x82"" 20\xC2""\xA0""\xE0""\xA4""\x85""\xE0""\xA4""\xAA"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x88""\xE0""\xA4""\xB2"""
  " \xE0""\xA4""\x95""\xE0""\xA5""\x8B""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp"
  ";&nbsp;<img src='images/bulet1.gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbs"
  "p;&nbsp;&nbsp;&nbsp;&nbsp;\r\n        \r\n        \xE0""\xA4""\xA6""\xE0"""
  "\xA5""\x87""\xE0""\xA4""\xB6""\xE0""\xA4""\xAD""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\xAA""\xE0"""
  "\xA5""\x87""\xE0""\xA4""\x9F""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0"""
  "\xA5""\x8B""\xE0""\xA4""\xB2"" \xE0""\xA4""\xA1""\xE0""\xA5""\x80""\xE0"""
  "\xA4""\xB2""\xE0""\xA4""\xB0""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"" \xE0"""
  "\xA4""\x95""\xE0""\xA5""\x80"" \xE0""\xA4""\xA4""\xE0""\xA5""\x87""\xE0"""
  "\xA4""\xB0""\xE0""\xA4""\xB9"" \xE0""\xA4""\x85""\xE0""\xA4""\xAA""\xE0"""
  "\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x88""\xE0""\xA4""\xB2"" \xE0"""
  "\xA4""\x95""\xE0""\xA5""\x8B"" \xE0""\xA4""\xB8""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\x82""\xE0""\xA4""\x95""\xE0""\xA5""\x87""\xE0""\xA4""\xA4""\xE0"""
  "\xA4""\xBF""\xE0""\xA4""\x95"" \xE0""\xA4""\xB9""\xE0""\xA5""\x9C""\xE0"""
  "\xA4""\xA4""\xE0""\xA4""\xBE""\xE0""\xA4""\xB2""&nbsp;&nbsp;&nbsp;&nbsp;"
  "&nbsp;&nbsp;&nbsp;<img src='images/bulet1.gif' align=bottom >&nbsp;&nbsp"
  ";&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\r\n        \r\n        \xE0""\xA4"""
  "\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xB5""\xE0""\xA5"""
  "\x80""\xE0""\xA4""\xA3"" \xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4"""
  "\x8F""\xE0""\xA4""\x95"" \xE0""\xA4""\x95""\xE0""\xA4""\xB0""\xE0""\xA5"""
  "\x8B""\xE0""\xA5""\x9C"" \xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\x82""\xE0""\xA4""\x97""\xE0""\xA5""\x87"" \xE0""\xA4""\xA5""\xE0""\xA5"""
  "\x87"": \xE0""\xA4""\xB0""\xE0""\xA5""\x87""\xE0""\xA4""\x96""\xE0""\xA4"""
  "\xBE"" \xE0""\xA4""\xAE""\xE0""\xA4""\xB9""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\x9C""\xE0""\xA4""\xA8""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<img s"
  "rc='images/bulet1.gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp"
  ";&nbsp;&nbsp;\r\n        \r\n        \xE0""\xA4""\xB6""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xB0""\xE0""\xA5""\x80""\xE0""\xA4""\xB2""\xE0""\xA4""\x82"""
  "\xE0""\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xAE""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\x82"" \xE0""\xA4""\xB8""\xE0""\xA5""\x9C""\xE0""\xA4""\x95"""
  " \xE0""\xA4""\xA6""\xE0""\xA5""\x81""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\x98""\xE0""\xA4""\x9F""\xE0""\xA4""\xA8""\xE0""\xA4""\xBE"""
  " \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82"" 23\xC2""\xA0""\xE0"""
  "\xA4""\x95""\xE0""\xA5""\x80"" \xE0""\xA4""\xAE""\xE0""\xA5""\x8C""\xE0"""
  "\xA4""\xA4""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<img src='images/b"
  "ulet1.gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp"
  ";\r\n        \r\n        \xE0""\xA4""\x85""\xE0""\xA4""\xB0""\xE0""\xA5"""
  "\x81""\xE0""\xA4""\xA3""\xE0""\xA4""\xBE""\xE0""\xA4""\x9A""\xE0""\xA4"""
  "\xB2"" \xE0""\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4"""
  "\xA6""\xE0""\xA5""\x87""\xE0""\xA4""\xB6"" \xE0""\xA4""\xAD""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xB0""\xE0""\xA4""\xA4"" \xE0""\xA4""\x95""\xE0""\xA4"""
  "\xBE"" \xE0""\xA4""\x85""\xE0""\xA4""\xB5""\xE0""\xA4""\xBF""\xE0""\xA4"""
  "\xAD""\xE0""\xA4""\xBE""\xE0""\xA4""\x9C""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xAF"" \xE0""\xA4""\x85""\xE0""\xA4""\x82""\xE0""\xA4""\x97"": \xE0""\xA4"""
  "\x96""\xE0""\xA4""\xBE""\xE0""\xA4""\x82""\xE0""\xA4""\xA6""\xE0""\xA5"""
  "\x82""&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<img src='images/bulet1."
  "gif' align=bottom >&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\r\n "
  "       \r\n</marquee></font></th>\r\n  </tr>\r\n</table></td>\r\n<td val"
  "ign=top width=16%>\r\n<table border=\"0\" cellPadding=\"0\" cellSpacing="
  "\"0\" width=\"100%\" align=\"left\">\r\n  <tr><td height=3></td></tr>\r\n"
  "  <tr>\r\n    <td valign=middle align=right>\r\n    <a href=\"http://www"
  "1.jagran.com/ir\" target=_blank><img alt=\"\" border=\"0\" src=\"http://"
  "www1.jagran.com/investor.gif\"></a>\r\n    </td></tr>    \r\n</table>\r\n"
  "</td></tr>\r\n<tr><td colspan=3><!--font face=arial>Menu Bar</font-->\r\n"
  "<script type=\"text/javascript\" language=\"JavaScript1.2\" src=\"menu/m"
  "enu.js\"></script>\r\n</td></tr>\r\n</table>\r\n<script language=\"JavaS"
  "cript\">\r\nfunction winClose()\r\n{\r\ndocument.all.Bannerc7.style.visi"
  "bility=\"hidden\";\r\n\r\n}\r\n</script>\r\n<div id='Bannerc7' style='po"
  "sition:absolute;visibility:visible;background-color:transparent; width: "
  "230; height: 40; top: 140px; left: 420px'> \r\n  <object classid='clsid:"
  "D27CDB6E-AE6D-11cf-96B8-444553540000'   codebase=\"'http://download.macr"
  "omedia.com/pub/shockwave/cabs/flash/swflash.cab#version=5,0,0,0'\"  widt"
  "h=230 height=40 id=OBJECT1 VIEWASTEXT>\r\n    <param name=movie value='h"
  "ttp://www1.jagran.com/images/shadi230x40.swf'>\r\n    <param name=menu v"
  "alue=false>\r\n    <param name=quality value=autolow>\r\n    <param name"
  "=bgcolor value=transparent>\r\n    <param name=wmode value=transparent>\r"
  "\n  </object>\r\n</div>\r\n<script language=\"Javascript\">setTimeout('w"
  "inClose()',10000);</script>\r\n<table width=100% align=center border=0 c"
  "ellspacing=0 cellpadding=0><tr>\r\n<td width=\"18%\" valign=TOP>\r\n\r\n"
  "<!--table width=100% border=0 cellpadding=0 cellspacing=0>\r\n<tr><td wi"
  "dth=18>&nbsp;</td><td align=\"left\" vAlign=\"top\"><img alt=\"\" border"
  "=\"0\" height=\"1\" src=\"images/spacer.gif\" width=\"2\"></td></tr>\r\n"
  "<tr><td width=18>&nbsp;</td><td align=\"left\" vAlign=\"top\">\r\n\r\n</"
  "td></tr>\r\n<tr><td width=18>&nbsp;</td><td align=\"left\" vAlign=\"top\""
  "><img alt=\"\" border=\"0\" height=\"1\" src=\"images/spacer.gif\" width"
  "=\"2\"></td></tr>\r\n</table-->\r\n<table width=100% border=0 cellpaddin"
  "g=0 cellspacing=0>\r\n<tr><td width=10>&nbsp;</td><td colspan=2><font fa"
  "ce=\"Jagran\" size=\"4\">&nbsp;<b>\xE0""\xA4""\x9A""\xE0""\xA5""\x88""\xE0"""
  "\xA4""\xA8""\xE0""\xA4""\xB2""\xE0""\xA5""\x8D""\xE0""\xA4""\xB8""</b></"
  "font></td></tr>\r\n<tr><td width=10>&nbsp;</td><td colspan=2 vAlign=\"to"
  "p\" height=\"1\"><img alt=\"\" border=\"0\" src=\"images/line.gif\" heig"
  "ht=\"1\"></td></tr>\r\n\r\n\r\n    \r\n    \r\n            <tr>\r\n<td w"
  "idth=10>&nbsp;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=l"
  "eft><font face=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl00_HyperLink"
  "1\" title=\"Home Page\" href=\"/default.aspx\">\xE0""\xA4""\xAE""\xE0""\xA5"""
  "\x81""\xE0""\xA4""\x96""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF"" \xE0""\xA4"""
  "\xAA""\xE0""\xA5""\x83""\xE0""\xA4""\xB7""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xA0""</a></font></th></tr>\r\n            \r\n            <tr>\r\n<td w"
  "idth=10>&nbsp;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=l"
  "eft><font face=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl01_HyperLink"
  "1\" title=\"News\" href=\"/news\">\xE0""\xA4""\xB8""\xE0""\xA4""\xAE""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\x9A""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""</a></"
  "font></th></tr>\r\n            \r\n            <tr>\r\n<td width=10>&nbs"
  "p;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=left><font fa"
  "ce=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl02_HyperLink1\" title=\""
  "Panchang\" href=\"/rashi2007\">\xE0""\xA4""\xAA""\xE0""\xA4""\x82""\xE0"""
  "\xA4""\x9A""\xE0""\xA4""\xBE""\xE0""\xA4""\x82""\xE0""\xA4""\x97"" 2007<"
  "/a></font></th></tr>\r\n            \r\n            <tr>\r\n<td width=10"
  ">&nbsp;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=left><fo"
  "nt face=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl03_HyperLink1\" tit"
  "le=\"Religion\" href=\"/religion\">\xE0""\xA4""\xA7""\xE0""\xA4""\xB0""\xE0"""
  "\xA5""\x8D""\xE0""\xA4""\xAE"" \xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\x97""</a></font></th></tr>\r\n"
  "            \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img"
  " src=images/bulet1.gif /></th><th align=left><font face=jagran size=3>\r"
  "\n<a id=\"BottomNavRepeat_ctl04_HyperLink1\" title=\"Junior Jagran\" hre"
  "f=\"/junior\">\xE0""\xA4""\x9C""\xE0""\xA5""\x82""\xE0""\xA4""\xA8""\xE0"""
  "\xA4""\xBF""\xE0""\xA4""\xAF""\xE0""\xA4""\xB0"" \xE0""\xA4""\x9C""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3""</a></"
  "font></th></tr>\r\n            \r\n            <tr>\r\n<td width=10>&nbs"
  "p;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=left><font fa"
  "ce=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl05_HyperLink1\" title=\""
  "Jagran Mobile Services\" href=\"/7272\">7272</a></font></th></tr>\r\n   "
  "         \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img sr"
  "c=images/bulet1.gif /></th><th align=left><font face=jagran size=3>\r\n<"
  "a id=\"BottomNavRepeat_ctl06_HyperLink1\" title=\"News with difference\""
  " href=\"/hatke\">\xE0""\xA4""\x9C""\xE0""\xA4""\xB0""\xE0""\xA4""\xBE"" "
  "\xE0""\xA4""\xB9""\xE0""\xA4""\x9F"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"""
  "</a></font></th></tr>\r\n            \r\n            <tr>\r\n<td width=1"
  "0>&nbsp;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=left><f"
  "ont face=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl07_HyperLink1\" ti"
  "tle=\"Jagran Greetings\" href=\"/navigation.aspx?id=1\">\xE0""\xA4""\x9C"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"""
  " \xE0""\xA4""\x97""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x80"""
  "\xE0""\xA4""\x9F""\xE0""\xA4""\xBF""\xE0""\xA4""\x82""\xE0""\xA4""\x97"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xB8""</a></font></th></tr>\r\n           "
  " \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img src=images"
  "/bulet1.gif /></th><th align=left><font face=jagran size=3>\r\n<a id=\"B"
  "ottomNavRepeat_ctl08_HyperLink1\" title=\"Jagran Sahitya\" href=\"/sahit"
  "ya\">\xE0""\xA4""\xB8""\xE0""\xA4""\xBE""\xE0""\xA4""\xB9""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF""</a></font></th></"
  "tr>\r\n            \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<"
  "th><img src=images/bulet1.gif /></th><th align=left><font face=jagran si"
  "ze=3>\r\n<a id=\"BottomNavRepeat_ctl09_HyperLink1\" title=\"Jagran Radio"
  "\" href=\"/radio\">\xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x97"""
  "\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"" \xE0""\xA4""\xB0""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\xA1""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF""\xE0""\xA5""\x8B"""
  "</a></font></th></tr>\r\n            \r\n            <tr>\r\n<td width=1"
  "0>&nbsp;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=left><f"
  "ont face=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl10_HyperLink1\" ti"
  "tle=\"Epaper\" href=\"/navigation.aspx?id=2\">\xE0""\xA4""\x88""-\xE0""\xA4"""
  "\xAA""\xE0""\xA5""\x87""\xE0""\xA4""\xAA""\xE0""\xA4""\xB0""</a></font><"
  "/th></tr>\r\n            \r\n            <tr>\r\n<td width=10>&nbsp;</td"
  ">\r\n<th><img src=images/bulet1.gif /></th><th align=left><font face=jag"
  "ran size=3>\r\n<a id=\"BottomNavRepeat_ctl11_HyperLink1\" title=\"Jagran"
  " Sakhi\" href=\"/sakhi\">\xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"" \xE0""\xA4""\xB8""\xE0""\xA4"""
  "\x96""\xE0""\xA5""\x80""</a></font></th></tr>\r\n            \r\n       "
  "     <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img src=images/bulet1.gif "
  "/></th><th align=left><font face=jagran size=3>\r\n<a id=\"BottomNavRepe"
  "at_ctl12_HyperLink1\" title=\"Jagran Forum\" href=\"/navigation.aspx?id="
  "3\">\xE0""\xA4""\x86""\xE0""\xA4""\xAA""\xE0""\xA4""\x95""\xE0""\xA5""\x80"""
  " \xE0""\xA4""\xAC""\xE0""\xA4""\xBE""\xE0""\xA4""\xA4""</a></font></th><"
  "/tr>\r\n            \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n"
  "<th><img src=images/bulet1.gif /></th><th align=left><font face=jagran s"
  "ize=3>\r\n<a id=\"BottomNavRepeat_ctl13_HyperLink1\" title=\"Jagran Yatr"
  "a\" href=\"/yatra\">\xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x97"""
  "\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"" \xE0""\xA4""\xAF""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xBE"""
  "</a></font></th></tr>\r\n            \r\n            <tr>\r\n<td width=1"
  "0>&nbsp;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=left><f"
  "ont face=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl14_HyperLink1\" ti"
  "tle=\"Jagran Cricket\" href=\"/cricket\">\xE0""\xA4""\x9C""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"" \xE0""\xA4""\x95"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xBF""\xE0""\xA4""\x95"""
  "\xE0""\xA5""\x87""\xE0""\xA4""\x9F""</a></font></th></tr>\r\n           "
  " \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img src=images"
  "/bulet1.gif /></th><th align=left><font face=jagran size=3>\r\n<a id=\"B"
  "ottomNavRepeat_ctl15_HyperLink1\" title=\"Cinema\" href=\"/cinemazaa\">\xE0"""
  "\xA4""\xB8""\xE0""\xA4""\xBF""\xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\xAE""\xE0""\xA5""\x9B""\xE0""\xA4""\xBE""</a></font></th></tr>\r\n"
  "            \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img"
  " src=images/bulet1.gif /></th><th align=left><font face=jagran size=3>\r"
  "\n<a id=\"BottomNavRepeat_ctl16_HyperLink1\" title=\"My Jagran\" href=\""
  "/navigation.aspx?id=4\">\xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4"""
  "\xB0""\xE0""\xA4""\xBE"" \xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3""</a></font></th></tr>\r\n     "
  "       \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img src="
  "images/bulet1.gif /></th><th align=left><font face=jagran size=3>\r\n<a "
  "id=\"BottomNavRepeat_ctl17_HyperLink1\" title=\"Jagran Matrimony Classif"
  "ieds\" href=\"/navigation.aspx?id=5\">\xE0""\xA4""\xB5""\xE0""\xA4""\xB0"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\x97""\xE0""\xA5""\x80""\xE0""\xA4""\x95"""
  "\xE0""\xA5""\x83""\xE0""\xA4""\xA4""</a></font></th></tr>\r\n           "
  " \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img src=images"
  "/bulet1.gif /></th><th align=left><font face=jagran size=3>\r\n<a id=\"B"
  "ottomNavRepeat_ctl18_HyperLink1\" title=\"Jagran Josh\" href=\"/josh\">\xE0"""
  "\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0""\xE0"""
  "\xA4""\xA3"" \xE0""\xA4""\x9C""\xE0""\xA5""\x8B""\xE0""\xA4""\xB6""</a><"
  "/font></th></tr>\r\n            \r\n            <tr>\r\n<td width=10>&nb"
  "sp;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=left><font f"
  "ace=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl19_HyperLink1\" title=\""
  "Rashiphal\" href=\"/news/rashi.aspx\">\xE0""\xA4""\xB0""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xB6""\xE0""\xA4""\xBF""\xE0""\xA4""\xAB""\xE0""\xA4""\xB2"""
  "</a></font></th></tr>\r\n            \r\n            <tr>\r\n<td width=1"
  "0>&nbsp;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=left><f"
  "ont face=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl20_HyperLink1\" ti"
  "tle=\"Humour\" href=\"/gudgudee\">\xE0""\xA4""\x97""\xE0""\xA5""\x81""\xE0"""
  "\xA4""\xA6""\xE0""\xA4""\x97""\xE0""\xA5""\x81""\xE0""\xA4""\xA6""\xE0"""
  "\xA5""\x80""</a></font></th></tr>\r\n            \r\n            <tr>\r\n"
  "<td width=10>&nbsp;</td>\r\n<th><img src=images/bulet1.gif /></th><th al"
  "ign=left><font face=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl21_Hype"
  "rLink1\" title=\"Khanakhazana\" href=\"/khanakhazana\">\xE0""\xA4""\x96"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xA8""\xE0""\xA4""\xBE"" \xE0""\xA4""\x96"""
  "\xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8""\xE0""\xA4""\xBE"""
  "</a></font></th></tr>\r\n            \r\n            <tr>\r\n<td width=1"
  "0>&nbsp;</td>\r\n<th><img src=images/bulet1.gif /></th><th align=left><f"
  "ont face=jagran size=3>\r\n<a id=\"BottomNavRepeat_ctl22_HyperLink1\" ti"
  "tle=\"Jagran Shopping\" href=\"/navigation.aspx?id=6\">\xE0""\xA4""\x9C"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"""
  " \xE0""\xA4""\xB6""\xE0""\xA5""\x89""\xE0""\xA4""\xAA""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\x82""\xE0""\xA4""\x97""</a></font></th></tr>\r\n           "
  " \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img src=images"
  "/bulet1.gif /></th><th align=left><font face=jagran size=3>\r\n<a id=\"B"
  "ottomNavRepeat_ctl23_HyperLink1\" title=\"Jagran Special\" href=\"/speci"
  "al\">\xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xA3"" \xE0""\xA4""\xB5""\xE0""\xA4""\xBF""\xE0""\xA4""\xB6"""
  "\xE0""\xA5""\x87""\xE0""\xA4""\xB7""</a></font></th></tr>\r\n           "
  " \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img src=images"
  "/bulet1.gif /></th><th align=left><font face=jagran size=3>\r\n<a id=\"B"
  "ottomNavRepeat_ctl24_HyperLink1\" title=\"Photogallery\" href=\"/jagrani"
  "mage\">\xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4"""
  "\xB0""\xE0""\xA4""\xA3"" \xE0""\xA4""\x87""\xE0""\xA4""\xAE""\xE0""\xA5"""
  "\x87""\xE0""\xA4""\x9C""</a></font></th></tr>\r\n            \r\n       "
  "     <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img src=images/bulet1.gif "
  "/></th><th align=left><font face=jagran size=3>\r\n<a id=\"BottomNavRepe"
  "at_ctl25_HyperLink1\" title=\"Mandi\" href=\"/news/mandi.aspx\">\xE0""\xA4"""
  "\xAE""\xE0""\xA4""\x82""\xE0""\xA4""\xA1""\xE0""\xA4""\xBF""\xE0""\xA4"""
  "\xAF""\xE0""\xA4""\xBE""\xE0""\xA4""\x82""</a></font></th></tr>\r\n     "
  "       \r\n            <tr>\r\n<td width=10>&nbsp;</td>\r\n<th><img src="
  "images/bulet1.gif /></th><th align=left><font face=jagran size=3>\r\n<a "
  "id=\"BottomNavRepeat_ctl26_HyperLink1\" title=\"Academic Results\" href="
  "\"/navigation.aspx?id=7\">\xE0""\xA4""\xB0""\xE0""\xA4""\xBF""\xE0""\xA4"""
  "\x9C""\xE0""\xA4""\xB2""\xE0""\xA5""\x8D""\xE0""\xA4""\x9F""</a></font><"
  "/th></tr>\r\n            \r\n            <tr>\r\n<td width=10>&nbsp;</td"
  ">\r\n<th><img src=images/bulet1.gif /></th><th align=left><font face=jag"
  "ran size=3>\r\n<a id=\"BottomNavRepeat_ctl27_HyperLink1\" title=\"Jagran"
  " Matrimony\" href=\"/navigation.aspx?id=8\">\xE0""\xA4""\x9C""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"" \xE0""\xA4"""
  "\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x9F""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xB0""\xE0""\xA5""\x80""\xE0""\xA4""\xAE""\xE0""\xA5""\x89""\xE0""\xA4"""
  "\xA8""\xE0""\xA5""\x80""</a></font></th></tr>\r\n            \r\n<tr><td"
  " width=10>&nbsp;</td><th valign=\"top\"><img src=images/bulet1.gif align"
  "=\"absMiddle\" valign=\"top\" /></th><td><font color=\"black\" face=\"Ja"
  "gran\" ><b>\xE0""\xA4""\xB8""\xE0""\xA4""\xBF""\xE0""\xA4""\x9F""\xE0""\xA5"""
  "\x80"" \xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4"""
  "\xB0""\xE0""\xA4""\xA3""</b></font><br><a target=\"_parent\" href=\"http"
  "://city.jagran.com/kanpur.asp\" style=\"TEXT-DECORATION: none\"><font co"
  "lor=\"black\" face=\"Jagran\" >&nbsp;(\xE0""\xA4""\x95""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xA8""\xE0""\xA4""\xAA""\xE0""\xA5""\x81""\xE0""\xA4""\xB0"""
  ")</font></a><br><a target=\"_parent\" href=\"http://city.jagran.com/delh"
  "i.asp\" style=\"TEXT-DECORATION: none\"><font color=\"black\" face=\"Jag"
  "ran\" >&nbsp;(\xE0""\xA4""\xA8""\xE0""\xA4""\x88"" \xE0""\xA4""\xA6""\xE0"""
  "\xA4""\xBF""\xE0""\xA4""\xB2""\xE0""\xA5""\x8D""\xE0""\xA4""\xB2""\xE0"""
  "\xA5""\x80"")</font></a><br><a target=\"_parent\" href=\"http://city.jag"
  "ran.com/lucknow/default.asp\" style=\"TEXT-DECORATION: none\"><font colo"
  "r=\"black\" face=\"Jagran\" >&nbsp;(\xE0""\xA4""\xB2""\xE0""\xA4""\x96"""
  "\xE0""\xA4""\xA8""\xE0""\xA4""\x8A"")</font></a></td></tr>\r\n\t\t<tr><t"
  "d align=\"left\" vAlign=\"top\"><img alt=\"\" border=\"0\" height=\"8\" "
  "src=\"images/spacer.gif\" width=\"8\"></td></tr>\r\n</table><br />\r\n  "
  "    <table border=\"0\" cellPadding=\"0\" cellSpacing=\"0\" width=\"95%\""
  " align=left>\r\n        <tr>\r\n          <td colSpan=\"3\">&nbsp;&nbsp;"
  "&nbsp;&nbsp;<img alt=\"\" align=\"absMiddle\" border=\"0\" height=\"31\""
  " src=\"images/weathericon.gif\" width=\"34\"><img alt=\"\" border=\"0\" "
  "height=\"17\" src=\"images/mausam.gif\" width=\"35\"></td></tr>\r\n     "
  "   <tr>\r\n          <td align=\"left\" vAlign=\"bottom\">&nbsp;&nbsp;<i"
  "mg alt=\"\" border=\"0\" height=\"12\" src=\"images/cty.gif\" width=\"25"
  "\"></td>\r\n          <td align=\"center\" vAlign=\"top\"><img alt=\"\" "
  "border=\"0\" height=\"30\" src=\"images/maxtemicon.gif\" width=\"20\"></"
  "td>\r\n          <td align=\"center\" vAlign=\"top\"><img alt=\"\" borde"
  "r=\"0\" height=\"30\" src=\"images/mintemicon.gif\" width=\"20\"></td>\r"
  "\n        </tr>\r\n        \r\n        \r\n        <tr>\r\n          <td"
  " align=\"left\" vAlign=\"bottom\">&nbsp;&nbsp;<font face=jagran size=3>\xE0"""
  "\xA4""\x95""\xE0""\xA5""\x8B""\xE0""\xA4""\xB2""\xE0""\xA4""\x95""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE""</font></td>\r\n        "
  "  <td align=\"center\" vAlign=\"bottom\"><font face=arial size=1>34.2</f"
  "ont></td>\r\n          <td align=\"center\" vAlign=\"bottom\"><font face"
  "=arial size=1>26.3</font></td>\r\n        </tr>\r\n        \r\n        <"
  "tr>\r\n          <td align=\"left\" vAlign=\"bottom\">&nbsp;&nbsp;<font "
  "face=jagran size=3>\xE0""\xA4""\x9A""\xE0""\xA5""\x87""\xE0""\xA4""\xA8"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xA8""\xE0""\xA4""\x88""</font></td>\r\n  "
  "        <td align=\"center\" vAlign=\"bottom\"><font face=arial size=1>3"
  "4.2</font></td>\r\n          <td align=\"center\" vAlign=\"bottom\"><fon"
  "t face=arial size=1>21.2</font></td>\r\n        </tr>\r\n        \r\n   "
  "     <tr>\r\n          <td align=\"left\" vAlign=\"bottom\">&nbsp;&nbsp;"
  "<font face=jagran size=3>\xE0""\xA4""\xA8""\xE0""\xA4""\x88"" \xE0""\xA4"""
  "\xA6""\xE0""\xA4""\xBF""\xE0""\xA4""\xB2""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xB2""\xE0""\xA5""\x80""</font></td>\r\n          <td align=\"center\" v"
  "Align=\"bottom\"><font face=arial size=1>35.2</font></td>\r\n          <"
  "td align=\"center\" vAlign=\"bottom\"><font face=arial size=1>21.3</font"
  "></td>\r\n        </tr>\r\n        \r\n        <tr>\r\n          <td ali"
  "gn=\"left\" vAlign=\"bottom\">&nbsp;&nbsp;<font face=jagran size=3>\xE0"""
  "\xA4""\xAE""\xE0""\xA5""\x81""\xE0""\xA4""\x82""\xE0""\xA4""\xAC""\xE0"""
  "\xA4""\x88""</font></td>\r\n          <td align=\"center\" vAlign=\"bott"
  "om\"><font face=arial size=1>33.2</font></td>\r\n          <td align=\"c"
  "enter\" vAlign=\"bottom\"><font face=arial size=1>23.2</font></td>\r\n  "
  "      </tr>\r\n        \r\n<tr>\r\n    <td align=\"left\" vAlign=\"top\""
  " colspan=\"3\"><img alt=\"\" border=\"0\" height=\"8\" src=\"images/spac"
  "er.gif\" width=\"8\"></td></tr>\r\n        <tr>\r\n      <td colSpan=\"3"
  "\">&nbsp;&nbsp;&nbsp;&nbsp;<img alt=\"\" align=\"absMiddle\" border=\"0\""
  " height=\"31\" src=\"images/stockicon.gif\" width=\"34\"><img alt=\"\" b"
  "order=\"0\" height=\"17\" src=\"images/stock.gif\" width=\"68\"></td></t"
  "r>\r\n    <tr>\r\n      <td align=\"left\" vAlign=\"bottom\">&nbsp;&nbsp"
  ";</td>\r\n      <td align=\"center\" vAlign=\"bottom\"><img alt=\"\" bor"
  "der=\"0\" height=\"24\" src=\"images/stcloseicon.gif\" width=\"24\"></td"
  ">\r\n      <td align=\"center\" vAlign=\"bottom\"><img alt=\"\" border=\""
  "0\" height=\"24\" src=\"images/stchicon.gif\" width=\"22\"></td>\r\n    "
  "</tr>\r\n        \r\n        \r\n        <tr>\r\n          <td align=\"l"
  "eft\" vAlign=\"bottom\">&nbsp;&nbsp;<font face=jagran size=3>\xE0""\xA4"""
  "\x8F""\xE0""\xA4""\xA8""\xE0""\xA4""\x8F""\xE0""\xA4""\xB8""\xE0""\xA4"""
  "\x88""</font></td>\r\n          <td align=\"center\" vAlign=\"bottom\"><"
  "font face=arial size=1>3848.15        </font></td>\r\n          <td alig"
  "n=\"center\" vAlign=\"bottom\"><font face=arial size=1>+4.65          </"
  "font></td>\r\n        </tr>\r\n        \r\n        <tr>\r\n          <td"
  " align=\"left\" vAlign=\"bottom\">&nbsp;&nbsp;<font face=jagran size=3>\xE0"""
  "\xA4""\xAC""\xE0""\xA5""\x80""\xE0""\xA4""\x8F""\xE0""\xA4""\xB8""\xE0"""
  "\xA4""\x88""</font></td>\r\n          <td align=\"center\" vAlign=\"bott"
  "om\"><font face=arial size=1>13182.49       </font></td>\r\n          <t"
  "d align=\"center\" vAlign=\"bottom\"><font face=arial size=1>+4.75      "
  "    </font></td>\r\n        </tr>\r\n        \r\n   <tr>\r\n    <td alig"
  "n=\"left\" vAlign=\"top\" colspan=\"3\"><img alt=\"\" border=\"0\" heigh"
  "t=\"8\" src=\"images/spacer.gif\" width=\"8\"></td></tr>\r\n        <tr>"
  "\r\n      <td colSpan=\"3\">&nbsp;&nbsp;&nbsp;&nbsp;<img alt=\"\" align="
  "\"absMiddle\" border=\"0\"  src=\"images/money.gif\"><img alt=\"\" borde"
  "r=\"0\"  src=\"images/mudra.gif\"></td></tr>\r\n    <tr>\r\n      <td al"
  "ign=\"left\" vAlign=\"middle\">&nbsp;</td>\r\n      <td align=\"center\""
  " vAlign=\"top\"><img alt=\"\" border=\"0\" src=\"images/buy.gif\" ></td>"
  "\r\n      <td align=\"center\" vAlign=\"top\"><img alt=\"\" border=\"0\""
  " src=\"images/sell.gif\"></td>\r\n    </tr>\r\n         \r\n        \r\n"
  "        <tr>\r\n          <td align=\"left\" vAlign=\"bottom\">&nbsp;&nb"
  "sp;<font face=jagran size=3>\xE0""\xA4""\xA1""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\xB2""\xE0""\xA4""\xB0""</font></td>\r\n          <td align=\"center\" v"
  "Align=\"top\"><font face=arial size=1>42.72</font></td>\r\n          <td"
  " align=\"center\" vAlign=\"top\"><font face=arial size=1>43.11</font></t"
  "d>\r\n        </tr>\r\n        \r\n        <tr>\r\n          <td align=\""
  "left\" vAlign=\"bottom\">&nbsp;&nbsp;<font face=jagran size=3>\xE0""\xA4"""
  "\xAA""\xE0""\xA5""\x8C""\xE0""\xA4""\x82""\xE0""\xA4""\xA1""</font></td>"
  "\r\n          <td align=\"center\" vAlign=\"top\"><font face=arial size="
  "1>83.94</font></td>\r\n          <td align=\"center\" vAlign=\"top\"><fo"
  "nt face=arial size=1>84.93</font></td>\r\n        </tr>\r\n        \r\n "
  "       <tr>\r\n          <td align=\"left\" vAlign=\"bottom\">&nbsp;&nbs"
  "p;<font face=jagran size=3>\xE0""\xA4""\xAF""\xE0""\xA5""\x87""\xE0""\xA4"""
  "\xA8""</font></td>\r\n          <td align=\"center\" vAlign=\"top\"><fon"
  "t face=arial size=1>35.81</font></td>\r\n          <td align=\"center\" "
  "vAlign=\"top\"><font face=arial size=1>36.60</font></td>\r\n        </tr"
  ">\r\n        \r\n        <tr>\r\n          <td align=\"left\" vAlign=\"b"
  "ottom\">&nbsp;&nbsp;<font face=jagran size=3>\xE0""\xA4""\xAF""\xE0""\xA5"""
  "\x82""\xE0""\xA4""\xB0""\xE0""\xA5""\x8B""</font></td>\r\n          <td "
  "align=\"center\" vAlign=\"top\"><font face=arial size=1>57.24</font></td"
  ">\r\n          <td align=\"center\" vAlign=\"top\"><font face=arial size"
  "=1>57.98</font></td>\r\n        </tr>\r\n        \r\n<tr>\r\n<td colspan"
  "=3>&nbsp;&nbsp;&nbsp;&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;<a href=\"http://"
  "ind.jagran.com/up07\" target=_blank><img src=\"http://ind.jagran.com/up0"
  "7.gif \" border=0></a></td></tr>   \r\n    </table>\r\n</td>\r\n<td widt"
  "h=\"66%\"><!--font face=arial>Center Table</font-->\r\n<!-- Section 1 --"
  ">\r\n<table width=100%>\r\n<tr><td><table align=\"left\" border=\"0\" ce"
  "llPadding=\"0\" cellSpacing=\"0\" width=\"62%\">\r\n\t        <tr><td ba"
  "ckground=\"images/boxbg.gif\" colspan=\"2\" valign=\"top\"><img alt=\"\""
  " align=\"top\" border=\"0\" height=\"20\" src=\"images/boxcurve1.gif\" w"
  "idth=\"20\"><b><font color=\"#000000\" face=\"Jagran\" size=\"4\"><a tar"
  "get=\"_parent\" href=\"news\" style=\"TEXT-DECORATION: none\">\xE0""\xA4"""
  "\xB8""\xE0""\xA5""\x81""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\x96""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\x82""</a></font></b> <font face=\"arial\" size=\"1\">&nbsp;&nbsp;&nbsp;"
  "&nbsp;Last Updated : 4/11/2007 1:17:10 AM</font>\r\n\t       \r\n\t     "
  "   </td></tr>\r\n\t        <!--<tr><td colspan=2><img src=\"images/space"
  "r.gif\" width=1 height=\"30\"></td></tr>-->\r\n        \r\n        \r\n "
  "                   <tr><td vAlign=\"top\" width=\"3%\"><font color=\"tom"
  "ato\" face=\"verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td>"
  "\r\n            <th align=left><font face=jagran size=3><a id=\"Repeater"
  "1_ctl00_HyperLink1\" href=\"news/details.aspx?id=3271941\">\xE0""\xA4""\xAD"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA4""\xA4""-\xE0""\xA4""\xAA"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x95"" \xE0""\xA4""\xB0""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xB6""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA5""\x8B"""
  "\xE0""\xA4""\x82"" \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82"""
  " \xE0""\xA4""\x86""\xE0""\xA4""\xA4""\xE0""\xA4""\x82""\xE0""\xA4""\x95"""
  "\xE0""\xA4""\xB5""\xE0""\xA4""\xBE""\xE0""\xA4""\xA6"" \xE0""\xA4""\x9A"""
  "\xE0""\xA4""\xBF""\xE0""\xA4""\x82""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"""
  " \xE0""\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB5""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xB7""\xE0""\xA4""\xAF""</a></font></th></tr>\r\n        \r\n"
  "                    <tr><td vAlign=\"top\" width=\"3%\"><font color=\"to"
  "mato\" face=\"verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td"
  ">\r\n            <th align=left><font face=jagran size=3><a id=\"Repeate"
  "r1_ctl01_HyperLink1\" href=\"news/details.aspx?id=3271560\">\xE0""\xA4"""
  "\xA8""\xE0""\xA5""\x87""\xE0""\xA4""\xAA""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\xB2"" \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4"""
  "\xAE""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA4""\xB5""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xA1""\xE0""\xA4""\xBC""\xE0""\xA5""\x80"" \xE0""\xA4"""
  "\x95""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA5""\x8B""\xE0""\xA4"""
  "\xAC""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA5""\x80"" \xE0""\xA4"""
  "\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\x85""\xE0""\xA4""\xAA""\xE0""\xA4"""
  "\xB9""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3""</a></font></th></tr>\r\n     "
  "   \r\n                    <tr><td vAlign=\"top\" width=\"3%\"><font col"
  "or=\"tomato\" face=\"verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></fo"
  "nt></td>\r\n            <th align=left><font face=jagran size=3><a id=\""
  "Repeater1_ctl02_HyperLink1\" href=\"news/details.aspx?id=3271429\">\xE0"""
  "\xA4""\xAE""\xE0""\xA5""\x81""\xE0""\xA4""\xA0""\xE0""\xA4""\xAD""\xE0"""
  "\xA5""\x87""\xE0""\xA5""\x9C"" \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0"""
  "\xA4""\x82"" \xE0""\xA4""\x86""\xE0""\xA4""\xA0"" \xE0""\xA4""\x95""\xE0"""
  "\xA4""\x9F""\xE0""\xA5""\x8D""\xE0""\xA4""\x9F""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\x89""\xE0""\xA4""\xB2""\xE0""\xA5""\x8D""\xE0""\xA4""\xAB""\xE0"""
  "\xA4""\xBE"" \xE0""\xA4""\x89""\xE0""\xA4""\x97""\xE0""\xA5""\x8D""\xE0"""
  "\xA4""\xB0""\xE0""\xA4""\xB5""\xE0""\xA4""\xBE""\xE0""\xA4""\xA6""\xE0"""
  "\xA5""\x80"" \xE0""\xA4""\xA2""\xE0""\xA5""\x87""\xE0""\xA4""\xB0""</a><"
  "/font></th></tr>\r\n        \r\n                    <tr><td vAlign=\"top"
  "\" width=\"3%\"><font color=\"tomato\" face=\"verdana\" size=\"3\"><b>&n"
  "bsp;&nbsp;O&nbsp;</b></font></td>\r\n            <th align=left><font fa"
  "ce=jagran size=3><a id=\"Repeater1_ctl03_HyperLink1\" href=\"news/detail"
  "s.aspx?id=3271427\">\xE0""\xA4""\x9A""\xE0""\xA5""\x81""\xE0""\xA4""\xA8"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xB5"" \xE0""\xA4""\x86""\xE0""\xA4""\xAF"""
  "\xE0""\xA5""\x8B""\xE0""\xA4""\x97"" \xE0""\xA4""\xB8""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\x9F""\xE0""\xA4""\x95""\xE0""\xA4""\xB0""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xB5"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\xAA""\xE0""\xA4""\xB0"" \xE0""\xA4""\xA8"""
  "\xE0""\xA4""\xB9""\xE0""\xA5""\x80""\xE0""\xA4""\x82"": \xE0""\xA4""\xAD"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x9C""\xE0""\xA4""\xAA""\xE0""\xA4""\xBE"""
  "</a></font></th></tr>\r\n        \r\n                    <tr><td vAlign="
  "\"top\" width=\"3%\"><font color=\"tomato\" face=\"verdana\" size=\"3\">"
  "<b>&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n            <th align=left><fo"
  "nt face=jagran size=3><a id=\"Repeater1_ctl04_HyperLink1\" href=\"news/d"
  "etails.aspx?id=3271420\">\xE0""\xA4""\xAA""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\x95"" \xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4""\xAA""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\x87""\xE0""\xA4""\xAA""\xE0""\xA4""\xB2""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\x87""\xE0""\xA4""\xA8"" \xE0""\xA4""\x97""\xE0""\xA5"""
  "\x88""\xE0""\xA4""\xB8"" \xE0""\xA4""\xAC""\xE0""\xA4""\x82""\xE0""\xA4"""
  "\x9F""\xE0""\xA4""\xB5""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA5"""
  "\x87"" \xE0""\xA4""\x95""\xE0""\xA5""\x8B"" \xE0""\xA4""\xAE""\xE0""\xA4"""
  "\x82""\xE0""\xA4""\x9C""\xE0""\xA5""\x82""\xE0""\xA4""\xB0""\xE0""\xA5"""
  "\x80"" \xE0""\xA4""\xA6""\xE0""\xA5""\x80""</a></font></th></tr>\r\n    "
  "    \r\n\r\n        \r\n        \r\n                    <tr><td vAlign=\""
  "top\" width=\"3%\"><font color=\"tomato\" face=\"verdana\" size=\"3\"><b"
  ">&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n            <th align=left><font"
  " face=jagran size=3 color=blue><a id=\"Repeater2_ctl00_HyperLink1\" href"
  "=\"news/details.aspx?id=3271376\" style=\"color:Blue;\">\xE0""\xA4""\xB8"""
  "\xE0""\xA5""\x80""\xE0""\xA4""\xB2""\xE0""\xA4""\xBF""\xE0""\xA4""\x82"""
  "\xE0""\xA4""\x97"" \xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0""\xA4""\xAE"""
  "\xE0""\xA4""\xB2""\xE0""\xA5""\x87"" \xE0""\xA4""\xAE""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\x82"" \xE0""\xA4""\xA6""\xE0""\xA4""\xBF""\xE0""\xA4""\xB2"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xB2""\xE0""\xA5""\x80"" \xE0""\xA4""\xAA"""
  "\xE0""\xA5""\x81""\xE0""\xA4""\xB2""\xE0""\xA4""\xBF""\xE0""\xA4""\xB8"""
  " \xE0""\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xAE"""
  "\xE0""\xA5""\x81""\xE0""\xA4""\x96"" \xE0""\xA4""\xA4""\xE0""\xA4""\xB2"""
  "\xE0""\xA4""\xAC""</a></th></tr>\r\n        \r\n                    <tr>"
  "<td vAlign=\"top\" width=\"3%\"><font color=\"tomato\" face=\"verdana\" "
  "size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n            <th ali"
  "gn=left><font face=jagran size=3 color=blue><a id=\"Repeater2_ctl01_Hype"
  "rLink1\" href=\"news/details.aspx?id=3271375\" style=\"color:Blue;\">\xE0"""
  "\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xB5""\xE0"""
  "\xA5""\x80""\xE0""\xA4""\xA3"" \xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\x8F""\xE0""\xA4""\x95"" \xE0""\xA4""\x95""\xE0""\xA4""\xB0""\xE0"""
  "\xA5""\x8B""\xE0""\xA5""\x9C"" \xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\x82""\xE0""\xA4""\x97""\xE0""\xA5""\x87"" \xE0""\xA4""\xA5""\xE0"""
  "\xA5""\x87"": \xE0""\xA4""\xB0""\xE0""\xA5""\x87""\xE0""\xA4""\x96""\xE0"""
  "\xA4""\xBE"" \xE0""\xA4""\xAE""\xE0""\xA4""\xB9""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\x9C""\xE0""\xA4""\xA8""</a></th></tr>\r\n        \r\n            "
  "        <tr><td vAlign=\"top\" width=\"3%\"><font color=\"tomato\" face="
  "\"verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n       "
  "     <th align=left><font face=jagran size=3 color=blue><a id=\"Repeater"
  "2_ctl02_HyperLink1\" href=\"news/details.aspx?id=3271354\" style=\"color"
  ":Blue;\">\xE0""\xA4""\xAA""\xE0""\xA5""\x80""\xE0""\xA4""\x8F""\xE0""\xA4"""
  "\xB8""\xE0""\xA4""\x8F""\xE0""\xA4""\xB2""\xE0""\xA4""\xB5""\xE0""\xA5"""
  "\x80"" \xE0""\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB5""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xA3""\xE0""\xA4""\xBF""\xE0""\xA4""\x9C""\xE0""\xA5"""
  "\x8D""\xE0""\xA4""\xAF""\xE0""\xA4""\xBF""\xE0""\xA4""\x95"" \xE0""\xA4"""
  "\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\x95""\xE0""\xA5"""
  "\x8D""\xE0""\xA4""\xB7""\xE0""\xA5""\x87""\xE0""\xA4""\xAA""\xE0""\xA4"""
  "\xA3"" 23 \xE0""\xA4""\x95""\xE0""\xA5""\x8B""</a></th></tr>\r\n        "
  "\r\n<tr><td vAlign=\"top\" width=\"3%\">&nbsp;</td><td align=\"left\">\r"
  "\n</td></tr>\r\n\r\n\t\t</table>\r\n\r\n \r\n    \r\n        \t\t\r\n   "
  "   <table border=\"0\" cellPadding=\"0\" cellSpacing=\"0\" width=\"38%\""
  " align=\"left\">\r\n      <tr><td>&nbsp;</td><td align=\"center\" backgr"
  "ound=\"images/boxbg.gif\" vAlign=\"top\" width=\"100%\"><b><font color=\""
  "#000000\" face=\"Jagran\" size=\"4\">&nbsp;&nbsp;&nbsp;<a href=\"jagrani"
  "mage\" style=\"TEXT-DECORATION: none\" target=\"_blank\">\xE0""\xA4""\x9C"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"""
  " \xE0""\xA4""\x87""\xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x9C"""
  "</a></font></b></td></tr>\r\n      <tr><td align=\"center\" vAlign=\"top"
  "\" colspan=2>\r\n     \r\n        <a href='jagranimage/inner.aspx?subcat"
  "gid=23&count=1&mainphotoid=36239'>\r\n        <img border=1 src=http://i"
  "mg.jagran.com/jagranimage/pimages/11aprct307t.jpg ><br /><b>\r\n        "
  "\xE0""\xA4""\xAA""\xE0""\xA5""\x82""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xB5""\xE0""\xA5""\x8B""\xE0""\xA4""\xA4""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xA4""\xE0""\xA4""\xB0"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\xB5""\xE0""\xA4""\xBF""\xE0""\xA4""\x95""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xB8"" \xE0""\xA4""\xAA""\xE0""\xA4""\xB0"" \xE0""\xA4""\x9C"""
  "\xE0""\xA5""\x8B""\xE0""\xA4""\xB0"" </b>\r\n        </a>\r\n        \r\n"
  "       </td></tr></table>\t\t\r\n</td></tr> </table>       \r\n\r\n  <!-"
  "- Banners-->      \r\n <table width=100%>\r\n\r\n<tr><td align=\"left\" "
  "bgColor=\"#ffe8d9\" vAlign=\"top\"><img alt=\"\" border=\"0\" height=\"1"
  "\" src=\"images/spacer.gif\" \r\nwidth=\"2\"></td></tr>\r\n\r\n<!--tr><t"
  "d align=\"center\">\r\n<table border=0 width=100%>\r\n<tr>\r\n<td align="
  "\"center\" width=\"33%\"><a class=\"pg\" href=\"#\" onclick=\"javascript"
  ":pic1('http://www1.jagran.com/flash/default.asp')\"><img src=\"images/an"
  "im.gif\" border=\"0\"></td>\r\n<td align=\"center\" width=\"33%\"><a hre"
  "f=\"http://www.shaadi.com/matrimonials/hindi-matrimonial?ptnr=jntxt\" ta"
  "rget=_blank><font face=Jagran size=4 color=\"blue\"><b>\xE0""\xA4""\x9C"""
  "\xE0""\xA5""\x80""\xE0""\xA4""\xB5""\xE0""\xA4""\xA8""\xE0""\xA4""\xB8"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xA5""\xE0""\xA5""\x80"" \xE0""\xA4""\x9A"""
  "\xE0""\xA5""\x81""\xE0""\xA4""\xA8""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF"""
  "\xE0""\xA5""\x87""</b></font></a></td>\r\n<td align=\"center\" width=\"3"
  "3%\"><a href=\"http://bmser.bharatmatrimony.com/cgi-bin/tracking.php?sec"
  "tion=Innerpage_468x60&siteurlsite=Jagran&domain=15&landing=partnerlandin"
  "g-basketball.shtml&creative=bm_basket_sept05_468x60&referredby=80510800\""
  " target=_blank><font face=Jagran size=4 color=\"blue\"><b>\xE0""\xA4""\xAD"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA4""\xA4""\xE0""\xA4""\xAE"""
  "\xE0""\xA5""\x87""\xE0""\xA4""\x9F""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0"""
  "\xE0""\xA5""\x80""\xE0""\xA4""\xAE""\xE0""\xA5""\x89""\xE0""\xA4""\xA8"""
  "\xE0""\xA5""\x80"".\xE0""\xA4""\x95""\xE0""\xA5""\x89""\xE0""\xA4""\xAE"""
  "</b></font></a></td>\r\n</tr>\r\n</table> \r\n</td></tr>\r\n<tr><td alig"
  "n=\"left\" bgColor=\"#ffe8d9\" vAlign=\"top\"><img alt=\"\" border=\"0\""
  " height=\"1\" src=\"images/spacer.gif\" width=\"17\"></td></tr-->\r\n<tr"
  "><td align=\"center\">\r\n\r\n        \r\n            <a href='http://ww"
  "w4.jagran.com/cricketnew/livescore.htm' target=_blank>\r\n            <f"
  "ont color=\"red\" face=\"Jagran\" size=\"4\"><b>\xE0""\xA4""\xB5""\xE0"""
  "\xA5""\x87""\xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\x9F""\xE0"""
  "\xA4""\x87""\xE0""\xA4""\x82""\xE0""\xA4""\xA1""\xE0""\xA5""\x80""\xE0"""
  "\xA4""\x9C""-\xE0""\xA4""\xA6""\xE0""\xA4""\x95""\xE0""\xA5""\x8D""\xE0"""
  "\xA4""\xB7""\xE0""\xA4""\xBF""\xE0""\xA4""\xA3"" \xE0""\xA4""\x85""\xE0"""
  "\xA4""\xAB""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x80""\xE0"""
  "\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB5""\xE0""\xA4""\xA8""\xE0"""
  "\xA4""\xA1""\xE0""\xA5""\x87""\xC2""\xA0""\xE0""\xA4""\x95""\xE0""\xA5"""
  "\x87"" \xE0""\xA4""\xB2""\xE0""\xA4""\xBE""\xE0""\xA4""\x87""\xE0""\xA4"""
  "\xB5""\xC2""\xA0""\xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\x95"""
  "\xE0""\xA5""\x8B""\xE0""\xA4""\xB0"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\xB2""\xE0""\xA4""\xBF""\xE0""\xA4""\x8F"" \xE0""\xA4""\xAF"""
  "\xE0""\xA4""\xB9""\xE0""\xA4""\xBE""\xE0""\xA4""\x82"" \xE0""\xA4""\x95"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xB2""\xE0""\xA4""\xBF""\xE0""\xA4""\x95"""
  "\xC2""\xA0""\xE0""\xA4""\x95""\xE0""\xA4""\xB0""\xE0""\xA5""\x87""\xE0"""
  "\xA4""\x82""</b></font> </a>\r\n        \r\n          </td></tr>\r\n    "
  "   \r\n<tr><td align=\"center\" width=\"75%\"><a href=\"http://www1.jagr"
  "an.com/forum\" target=_blank><font face=Jagran size=4 color=blue><b>\xE0"""
  "\xA4""\x95""\xE0""\xA4""\xBF""\xE0""\xA4""\xA4""\xE0""\xA4""\xA8""\xE0"""
  "\xA4""\xBE"" \xE0""\xA4""\xA8""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\xAF""\xE0""\xA5""\x8B""\xE0""\xA4""\x9A""\xE0"""
  "\xA4""\xBF""\xE0""\xA4""\xA4"" \xE0""\xA4""\xB9""\xE0""\xA5""\x88"" \xE0"""
  "\xA4""\xA8""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xAF""\xE0""\xA4""\xBE""\xE0""\xA4""\xB2""\xE0""\xA4""\xAF"" \xE0"""
  "\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xAB""\xE0""\xA5""\x88""\xE0"""
  "\xA4""\xB8""\xE0""\xA4""\xB2""\xE0""\xA4""\xBE""?</b></font></a></td></t"
  "r>\r\n\r\n<!-- Section 2 -->\r\n</table>\r\n\r\n<table width=100% align="
  "left>\r\n<tr><td align=\"left\" bgColor=\"#ffe8d9\" vAlign=\"top\"><img "
  "alt=\"\" border=\"0\" height=\"1\" src=\"images/spacer.gif\" width=\"17\""
  "></td></tr>\r\n<tr>\r\n    <td align=\"left\" vAlign=\"top\">\r\n    \r\n"
  "      <table border=\"0\" cellPadding=\"0\" cellSpacing=\"0\" width=\"49"
  "%\" align=\"left\">\r\n          <tr><td align=\"left\" background=\"ima"
  "ges/boxbg.gif\" vAlign=\"top\" width=\"50%\"><b><font color=\"#000000\" "
  "face=\"Jagran\" size=\"4\">&nbsp;<a target=\"_parent\" href=\"cricket\">"
  "\xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xA3"" \xE0""\xA4""\x95""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xBF""\xE0""\xA4""\x95""\xE0""\xA5""\x87""\xE0""\xA4""\x9F"""
  "</a></font></b></td></tr>\r\n          <tr><td align=\"left\" vAlign=\"t"
  "op\">\r\n         \r\n        \r\n            <a href='cricket/cinner.as"
  "px?newsid=4853'>\r\n            <table id=\"Repeater11_ctl00_Table1\" ce"
  "llpadding=\"2\" align=\"left\">\r\n\t<tr>\r\n\t\t<td><img id=\"Repeater1"
  "1_ctl00_Image1\" src=\"http://img.jagran.com/cricket/images/shastri2t.jp"
  "g\" align=\"left\" style=\"border-width:3px;border-style:Outset;height:1"
  "00px;width:100px;\" /></td>\r\n\t</tr>\r\n</table>\r\n\r\n            <f"
  "ont color=\"black\" face=\"Jagran\" size=\"4\"><b>\xE0""\xA4""\x95""\xE0"""
  "\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xA8"" \xE0""\xA4""\xB9""\xE0""\xA5""\x80"" \xE0""\xA4""\xB9""\xE0"""
  "\xA5""\x8B""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9""\xE0"""
  "\xA5""\x88"" \xE0""\xA4""\x9F""\xE0""\xA5""\x80""\xE0""\xA4""\xAE"" \xE0"""
  "\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xAC""\xE0""\xA5""\x89""\xE0"""
  "\xA4""\xB8"":\xC2""\xA0""\xE0""\xA4""\xB6""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xB0""\xE0""\xA5""\x80"" </b><br /><b>\xE0""\xA4""\xA8""\xE0""\xA4""\x88"""
  " \xE0""\xA4""\xA6""\xE0""\xA4""\xBF""\xE0""\xA4""\xB2""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xB2""\xE0""\xA5""\x80"", 10 \xE0""\xA4""\x85""\xE0""\xA4""\xAA"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x88""\xE0""\xA4""\xB2"""
  "\xE0""\xA5""\xA4""</b> \xE0""\xA4""\xAA""\xE0""\xA4""\xBF""\xE0""\xA4""\x9B"""
  "\xE0""\xA4""\xB2""\xE0""\xA5""\x87"" \xE0""\xA4""\xA6""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xA8""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"" \xE0""\xA4""\xAC"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x82""\xE0""\xA4""\x97""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xB2""\xE0""\xA4""\xBE""\xE0""\xA4""\xA6""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\xB6"" \xE0""\xA4""\xA6""\xE0""\xA5""\x8C""\xE0""\xA4""\xB0"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\x95""\xE0""\xA5""\x87""  \xE0""\xA4""\xB2"""
  "\xE0""\xA4""\xBF""\xE0""\xA4""\x8F"" \xE0""\xA4""\xAD""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xB0""\xE0""\xA4""\xA4""\xE0""\xA5""\x80""\xE0""\xA4""\xAF"""
  " \xE0""\xA4""\x95""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\x95""\xE0""\xA5""\x87""\xE0""\xA4""\x9F"" \xE0""\xA4""\x9F"""
  "\xE0""\xA5""\x80""\xE0""\xA4""\xAE"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\xAE""\xE0""\xA5""\x88""\xE0""\xA4""\xA8""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\x9C""\xE0""\xA4""\xB0"" \xE0""\xA4""\xA8""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xAF""\xE0""\xA5""\x81""\xE0""\xA4""\x95""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xA4"" \xE0""\xA4""\x95""\xE0""\xA4""\xBF""\xE0""\xA4""\x8F"""
  " \xE0""\xA4""\x97""\xE0""\xA4""\x8F"" \xE0""\xA4""\xAA""\xE0""\xA5""\x82"""
  "\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\xB5"" \xE0""\xA4""\xB9"""
  "\xE0""\xA4""\xB0""\xE0""\xA4""\xAB""\xE0""\xA4""\xA8""\xE0""\xA4""\xAE"""
  "\xE0""\xA5""\x8C""\xE0""\xA4""\xB2""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xB5""\xE0""\xA4""\xBF"" \xE0""\xA4""\xB6""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xB0""\xE0""\xA5""\x80"" \xE0""\xA4""\xA8""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\x95""\xE0""\xA4""\xB9""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9"""
  "\xE0""\xA5""\x88"" \xE0""\xA4""\x95""\xE0""\xA4""\xBF"" \xE0""\xA4""\x95"""
  "\xE0""\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xA8"" \xE0""\xA4""\xB9""\xE0""\xA5""\x80"" \xE0""\xA4""\x9F"""
  "\xE0""\xA5""\x80""\xE0""\xA4""\xAE"" \xE0""\xA4""\x95""\xE0""\xA4""\xBE"""
  " \xE0""\xA4""\xAC""\xE0""\xA5""\x89""\xE0""\xA4""\xB8""\xC2""\xA0""\xE0"""
  "\xA4""\xB9""\xE0""\xA5""\x8B""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"" \xE0"""
  "\xA4""\xB9""\xE0""\xA5""\x88""\xE0""\xA5""\xA4"" <br>\xC2""\xA0""\xC2""\xA0"""
  "\xC2""\xA0""\xC2""\xA0""\xE0""\xA4""\xB6""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xB0""\xE0""\xA5""\x80"" \xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4"""
  "\x9C""\xE0""\xA5""\x8B""\xE0""\xA4""\xB0"" \xE0""\xA4""\xA6""\xE0""\xA5"""
  "\x87""\xE0""\xA4""\x95""\xE0""\xA4""\xB0"" \xE0""\xA4""\x95""\xE0""\xA4"""
  "\xB9""\xE0""\xA4""\xBE"", '\xE0""\xA4""\x9F""\xE0""\xA5""\x80""\xE0""\xA4"""
  "\xAE"" \xE0""\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xAC""\xE0""\xA5"""
  "\x89""\xE0""\xA4""\xB8""\xC2""\xA0""\xE0""\xA4""\x95""\xE0""\xA4""\xAA"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8"""
  " \xE0""\xA4""\xB9""\xE0""\xA5""\x80"" \xE0""\xA4""\xB9""\xE0""\xA5""\x8B"""
  "\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9""\xE0""\xA5""\x88"""
  "\xE0""\xA5""\xA4""' \xE0""\xA4""\xB6""\xE0""\xA4""\xBE""\xE0""\xA4""\xB8"""
  "\xE0""\xA5""\x8D""&nbsp;<img alt=\"\" align=\"absmiddle\" src=\"images/m"
  "ore.gif\" border=\"0\" WIDTH=\"15\" HEIGHT=\"6\"></font> </a>\r\n       "
  " \r\n\t\t\t</td></tr></table>\r\n<!-- Spacer Table -->\r\n\t\t\t<table b"
  "order=\"0\" cellPadding=\"0\" cellSpacing=\"0\" width=\"1%\" align=\"lef"
  "t\">\r\n\t\t\t<tr><td><img alt=\"\" border=\"0\" height=\"1\" src=\"imag"
  "es/spacer.gif\" width=\"1\"></td></tr>\r\n\t\t\t</table>\r\n<!-- Spacer "
  "Table Ends -->\r\n      <table border=\"0\" cellPadding=\"0\" cellSpacin"
  "g=\"0\" width=\"49%\" align=\"left\">\r\n        <tr><td align=\"center\""
  " background=\"images/boxbg.gif\" vAlign=\"top\" width=\"95%\"><b><font c"
  "olor=\"#000000\" face=\"Jagran\" size=\"4\">&nbsp;&nbsp;&nbsp;<a  href=\""
  "gudgudee\" style=\"TEXT-DECORATION: none\" target=\"_blank\">\xE0""\xA4"""
  "\x97""\xE0""\xA5""\x81""\xE0""\xA4""\xA6""\xE0""\xA4""\x97""\xE0""\xA5"""
  "\x81""\xE0""\xA4""\xA6""\xE0""\xA5""\x80""</a></font></b></td><td><img a"
  "lt=\"\" align=\"top\" border=\"0\" src=\"images/boxcurve2.gif\" WIDTH=\""
  "20\" HEIGHT=\"20\"></td></tr>\r\n        <tr><td colspan=2>\r\n         "
  "\r\n        \r\n            <table cellpadding=2 align=left><tr><td><img"
  " id=\"Repeater10_ctl00_Image2\" src=\"http://img.jagran.com/gudgudee/gim"
  "ages/joke2.jpg\" align=\"left\" style=\"border-width:3px;border-style:Ou"
  "tset;height:100px;width:100px;\" /></td></tr></table>\r\n            <a "
  "href='gudgudee'>\r\n            <font color=\"black\" face=\"Jagran\" si"
  "ze=\"4\"><b>\xE0""\xA4""\xA6""\xE0""\xA4""\xBE""\xE0""\xA4""\xA6""\xE0"""
  "\xA4""\xBE""</b> (\xE0""\xA4""\xB8""\xE0""\xA5""\x8B""\xE0""\xA4""\xB9"""
  "\xE0""\xA4""\xA8"" \xE0""\xA4""\xB8""\xE0""\xA5""\x87"")-  \xE0""\xA4""\xAC"""
  "\xE0""\xA5""\x87""\xE0""\xA4""\x9F""\xE0""\xA4""\xBE"", \xE0""\xA4""\x85"""
  "\xE0""\xA4""\x82""\xE0""\xA4""\xA6""\xE0""\xA4""\xB0"" \xE0""\xA4""\xB8"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\xB0"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\xA6""\xE0""\xA4""\xBE""\xE0""\xA4""\x82"""
  "\xE0""\xA4""\xA4"" \xE0""\xA4""\xA4""\xE0""\xA5""\x8B"" \xE0""\xA4""\xB2"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\x86""\xE0""\xA5""\xA4""<br>\r\n<b>\xE0"""
  "\xA4""\xB8""\xE0""\xA5""\x8B""\xE0""\xA4""\xB9""\xE0""\xA4""\xA8""</b> ("
  "\xE0""\xA4""\xA6""\xE0""\xA4""\xBE""\xE0""\xA4""\xA6""\xE0""\xA4""\xBE"""
  " \xE0""\xA4""\xB8""\xE0""\xA5""\x87"")-\xE0""\xA4""\xAA""\xE0""\xA4""\xB0"""
  " \xE0""\xA4""\xA6""\xE0""\xA4""\xBE""\xE0""\xA4""\xA6""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\x9C""\xE0""\xA5""\x80""\xC2""\xA0""\xE0""\xA4""\x85""\xE0"""
  "\xA4""\xAD""\xE0""\xA5""\x80"" \xE0""\xA4""\xB0""\xE0""\xA5""\x8B""\xE0"""
  "\xA4""\x9F""\xE0""\xA5""\x80"" \xE0""\xA4""\xA4""\xE0""\xA5""\x8B"" \xE0"""
  "\xA4""\xAC""\xE0""\xA4""\xA8""\xE0""\xA5""\x80"" \xE0""\xA4""\xA8""\xE0"""
  "\xA4""\xB9""\xE0""\xA5""\x80""\xE0""\xA4""\x82""\xE0""\xA5""\xA4""<br>\r"
  "\n<b>\xE0""\xA4""\xA6""\xE0""\xA4""\xBE""\xE0""\xA4""\xA6""\xE0""\xA4""\xBE"""
  "</b>- \xE0""\xA4""\x93"" \xE0""\xA4""\xB0""\xE0""\xA5""\x8B""\xE0""\xA4"""
  "\x9F""\xE0""\xA5""\x80"" \xE0""\xA4""\xA8""\xE0""\xA4""\xB9""\xE0""\xA5"""
  "\x80""\xE0""\xA4""\x82"" \xE0""\xA4""\x96""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\xA8""\xE0""\xA5""\x80"" \xE0""\xA4""\xB8""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\xAE""\xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4""\xB5""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xB2""\xE0""\xA5""\x80"" \xE0""\xA4""\xAC""\xE0""\xA5"""
  "\x81""\xE0""\xA4""\xA2""\xE0""\xA4""\xBF""\xE0""\xA4""\xBC""\xE0""\xA4"""
  "\xAF""\xE0""\xA4""\xBE"" \xE0""\xA4""\x95""\xE0""\xA5""\x8B"" \xE0""\xA4"""
  "\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\x87""\xE0""\xA4""\xB2""\xC2""\xA0""\xE0""\xA4""\xA6""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\xA8""\xE0""\xA5""\x80"" \xE0""\xA4""\xB9""\xE0""\xA5""\x88"""
  "\xE0""\xA5""\xA4"" &nbsp;<img alt=\"\" align=\"absmiddle\" src=\"images/"
  "more.gif\" border=\"0\" WIDTH=\"15\" HEIGHT=\"6\"></font> </a>\r\n      "
  "  \r\n          </td></tr></table>\r\n    </td></tr>\r\n<tr><td align=\""
  "left\" bgColor=\"#ffe8d9\" vAlign=\"top\"><img alt=\"\" border=\"0\" hei"
  "ght=\"1\" src=\"images/spacer.gif\" width=\"17\"></td></tr>\t\t   \r\n\r"
  "\n<tr>\r\n    <td align=\"left\" vAlign=\"top\">\r\n      <table border="
  "\"0\" cellPadding=\"0\" cellSpacing=\"0\" width=\"50%\" align=\"left\">\r"
  "\n        <tr><td align=\"left\" background=\"images/boxbg.gif\" vAlign="
  "\"top\"><b><font color=\"#000000\" face=\"Jagran\" size=\"4\">&nbsp;<a h"
  "ref=\"sakhi\" style=\"TEXT-DECORATION: none\" target=\"_blank\">\xE0""\xA4"""
  "\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0""\xE0""\xA4"""
  "\xA3"" \xE0""\xA4""\xB8""\xE0""\xA4""\x96""\xE0""\xA5""\x80""</a></font>"
  "</b></td><td align=\"left\" background=\"images/boxbg.gif\" vAlign=\"top"
  "\">&nbsp;</td></tr>\r\n        <tr><td align=\"left\" vAlign=\"top\" wid"
  "th=\"95%\">\r\n        \r\n       \r\n        \r\n            <table id="
  "\"Repeater4_ctl00_Table2\" cellpadding=\"2\" align=\"left\">\r\n\t<tr>\r"
  "\n\t\t<td><img id=\"Repeater4_ctl00_Image1\" src=\"http://img.jagran.com"
  "/sakhi/images/sak3498T.jpg\" align=\"left\" style=\"border-width:3px;bor"
  "der-style:Outset;\" /></td>\r\n\t</tr>\r\n</table>\r\n\r\n            <a"
  " href='sakhi/sinner.aspx?idsection=1&amp;idarticle=3498&amp;idedition=74"
  "'><font color=\"black\" face=\"Jagran\" size=\"4\"><b><b>\xE0""\xA4""\x95"""
  "\xE0""\xA4""\xB5""\xE0""\xA4""\xB0"" \xE0""\xA4""\xB8""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\x9F""\xE0""\xA5""\x8B""\xE0""\xA4""\xB0""\xE0""\xA5""\x80"""
  ":</b> \xE0""\xA4""\xA8""\xE0""\xA4""\x88"" \xE0""\xA4""\xAA""\xE0""\xA4"""
  "\xB9""\xE0""\xA4""\x9A""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8"" \xE0""\xA4"""
  "\xA4""\xE0""\xA4""\xB2""\xE0""\xA4""\xBE""\xE0""\xA4""\xB6""\xE0""\xA4"""
  "\xA4""\xE0""\xA5""\x80"" \xE0""\xA4""\xAF""\xE0""\xA5""\x81""\xE0""\xA4"""
  "\xB5""\xE0""\xA4""\xBE"" \xE0""\xA4""\xAA""\xE0""\xA5""\x80""\xE0""\xA4"""
  "\xA2""\xE0""\xA4""\xBC""\xE0""\xA5""\x80"" </b><br>\xE0""\xA4""\x89""\xE0"""
  "\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xB8""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xB9"", \xE0""\xA4""\x8A""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0"""
  "\xA4""\x9C""\xE0""\xA4""\xBE"" \xE0""\xA4""\x94""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\xB9""\xE0""\xA4""\xB0"" \xE0""\xA4""\xAA""\xE0""\xA4""\xB2"" \xE0"""
  "\xA4""\xA8""\xE0""\xA4""\x88"" \xE0""\xA4""\x9A""\xE0""\xA5""\x81""\xE0"""
  "\xA4""\xA8""\xE0""\xA5""\x8C""\xE0""\xA4""\xA4""\xE0""\xA4""\xBF""\xE0"""
  "\xA4""\xAF""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"" \xE0""\xA4""\x95""\xE0"""
  "\xA5""\x8B"" \xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xB5""\xE0"""
  "\xA5""\x80""\xE0""\xA4""\x95""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0"""
  "\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\xB9""\xE0""\xA5""\x8C""\xE0""\xA4""\xB8""\xE0""\xA4""\xB2""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\xB8""\xE0""\xA5""\x87"" \xE0""\xA4""\xAD""\xE0"""
  "\xA4""\xB0""\xE0""\xA4""\xAA""\xE0""\xA5""\x82""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\xB9""\xE0""\xA5""\x88"" \xE0""\xA4""\x86""\xE0""\xA4""\x9C"" \xE0"""
  "\xA4""\x95""\xE0""\xA5""\x80"" \xE0""\xA4""\xAF""\xE0""\xA5""\x81""\xE0"""
  "\xA4""\xB5""\xE0""\xA4""\xBE"" \xE0""\xA4""\xAA""\xE0""\xA5""\x80""\xE0"""
  "\xA4""\xA2""\xE0""\xA4""\xBC""\xE0""\xA5""\x80""\xE0""\xA5""\xA4"" \xE0"""
  "\xA4""\x85""\xE0""\xA4""\xAA""\xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\xB2""\xE0""\xA4""\x95""\xE0""\xA5""\x8D""\xE0""\xA4""\xB7""\xE0"""
  "\xA5""\x8D""\xE0""\xA4""\xAF"" \xE0""\xA4""\x95""\xE0""\xA5""\x80"" \xE0"""
  "\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA4""\xBF"" \xE0"""
  "\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\x97"" \xE0""\xA4""\xAE""\xE0"""
  "\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\x86""\xE0""\xA4""\xA8""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\xB5""\xE0""\xA4""\xBE""\xE0""\xA4""\xB2""\xE0"""
  "\xA5""\x80"" \xE0""\xA4""\xAC""\xE0""\xA4""\xBE""\xE0""\xA4""\xA7""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\x93""\xE0""\xA4""\x82"" \xE0""\xA4""\xB8""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\x9C""\xE0""\xA5""\x82""\xE0""\xA4""\x9D""\xE0"""
  "\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4""\x94""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\xB9""\xE0""\xA4""\xB0"" \xE0""\xA4""\xA4""\xE0""\xA4""\xB0""\xE0"""
  "\xA4""\xB9"" \xE0""\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\x9C""\xE0"""
  "\xA5""\x8B""\xE0""\xA4""\x96""\xE0""\xA4""\xBF""\xE0""\xA4""\xAE"" \xE0"""
  "\xA4""\x89""\xE0""\xA4""\xA0""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\x95""\xE0""\xA5""\x80"" \xE0""\xA4""\xB9""\xE0"""
  "\xA4""\xBF""\xE0""\xA4""\xAE""\xE0""\xA5""\x8D""\xE0""\xA4""\xAE""\xE0"""
  "\xA4""\xA4"" \xE0""\xA4""\xB9""\xE0""\xA5""\x88"" \xE0""\xA4""\x89""\xE0"""
  "\xA4""\xB8""\xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82""\xE0"""
  "\xA5""\xA4"" \xE0""\xA4""\x86""\xE0""\xA4""\x9C"" \xE0""\xA4""\x95""\xE0"""
  "\xA4""\xBE"" \xE0""\xA4""\xAF""\xE0""\xA5""\x81""\xE0""\xA4""\xB5""\xE0"""
  "\xA4""\xBE"" \xE0""\xA4""\x9C""\xE0""\xA5""\x80""\xE0""\xA4""\xB5""\xE0"""
  "\xA4""\xA8"" \xE0""\xA4""\xB8""\xE0""\xA5""\x87"" \xE0""\xA4""\x9C""\xE0"""
  "\xA5""\x81""\xE0""\xA4""\xA1""\xE0""\xA4""\xBC""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\xB9""\xE0""\xA4""\xB0"" \xE0""\xA4""\xAA""\xE0""\xA4""\xB9""\xE0"""
  "\xA4""\xB2""\xE0""\xA5""\x82"" \xE0""\xA4""\xAA""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\xA4""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0"""
  "\xA4""\x95""\xE0""\xA4""\xBF""\xE0""\xA4""\x95"" \xE0""\xA4""\xA2""\xE0"""
  "\xA4""\x82""\xE0""\xA4""\x97"" \xE0""\xA4""\xB8""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\xB8""\xE0""\xA5""\x8B""\xE0""\xA4""\x9A""\xE0""\xA4""\xA4""\xE0"""
  "\xA4""\xBE"" \xE0""\xA4""\xB9""\xE0""\xA5""\x88"", \xE0""\xA4""\xAC""\xE0"""
  "\xA4""\xB9""\xE0""\xA4""\xB8"" \xE0""\xA4""\x95""\xE0""\xA4""\xB0""\xE0"""
  "\xA4""\xA4""\xE0""\xA4""\xBE"" \xE0""\xA4""\x94""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\xB8""\xE0""\xA4""\x9F""\xE0""\xA5""\x80""\xE0""\xA4""\x95"" \xE0"""
  "\xA4""\xB5""\xE0""\xA4""\xBF""\xE0""\xA4""\xB6""\xE0""\xA5""\x8D""\xE0"""
  "\xA4""\xB2""\xE0""\xA5""\x87""\xE0""\xA4""\xB7""\xE0""\xA4""\xA3"" \xE0"""
  "\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\xAC""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xA6"" \xE0""\xA4""\xB9""\xE0""\xA5""\x80"" \xE0""\xA4""\xB8""\xE0"""
  "\xA4""\xB9""\xE0""\xA5""\x80"" \xE0""\xA4""\xA8""\xE0""\xA4""\xBF""\xE0"""
  "\xA4""\xB7""\xE0""\xA5""\x8D""&nbsp;<img alt=\"\" align=\"absmiddle\" sr"
  "c=\"images/more.gif\" border=\"0\" WIDTH=\"15\" HEIGHT=\"6\"></font></a>"
  "</a>\r\n        \r\n</td><td>&nbsp;</td>\r\n         </tr></table>\r\n\t"
  "\t<table align=\"left\" border=\"0\" cellPadding=\"0\" cellSpacing=\"0\""
  " width=\"50%\">\r\n        <tr><td>&nbsp;</td><td align=\"center\" backg"
  "round=\"images/boxbg.gif\" vAlign=\"top\" width=\"95%\"><!--<img alt=\"\""
  " align=\"top\" border=\"0\" height=\"20\" src=\"images/boxcurve1.gif\" w"
  "idth=\"20\">--><b><font color=\"#000000\" face=\"Jagran\" size=\"4\">&nb"
  "sp;&nbsp;&nbsp;<a  href=\"cinemazaa\" style=\"TEXT-DECORATION: none\" ta"
  "rget=\"_blank\">\xE0""\xA4""\xB8""\xE0""\xA4""\xBF""\xE0""\xA4""\xA8""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\xAE""\xE0""\xA4""\x9C""\xE0""\xA4""\xBE""</a><"
  "/font></b></td><td background=\"images/boxbg.gif\">&nbsp;&nbsp;</td></tr"
  ">\r\n        <tr><td>&nbsp;</td><td align=\"left\" vAlign=\"top\" colspa"
  "n=2>\r\n        \r\n               \r\n        \r\n            <table id"
  "=\"Repeater5_ctl00_Table3\" cellpadding=\"2\" align=\"left\">\r\n\t<tr>\r"
  "\n\t\t<td><img id=\"Repeater5_ctl00_Image1\" src=\"http://img.jagran.com"
  "/cinemazaa/images/bg0005t.jpg\" align=\"left\" style=\"border-width:3px;"
  "border-style:Outset;\" /></td>\r\n\t</tr>\r\n</table>\r\n\r\n           "
  " <a href='cinemazaa/BollywoodBuzzDesc.aspx?BuzzCode=BB1023'>\r\n        "
  "    <font face=\"Jagran\" size=\"4\"><b>'\xE0""\xA4""\xAE""\xE0""\xA5""\x88"""
  "\xE0""\xA4""\xA8""\xE0""\xA4""\x87""\xE0""\xA4""\x9F""\xE0""\xA4""\xB0"""
  "' \xE0""\xA4""\xB9""\xE0""\xA5""\x88""\xE0""\xA4""\x82"" \xE0""\xA4""\xAA"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x80""\xE0""\xA4""\xA4"""
  "\xE0""\xA4""\xBF"" \xE0""\xA4""\x9C""\xE0""\xA4""\xBF""\xE0""\xA4""\x82"""
  "\xE0""\xA4""\x9F""\xE0""\xA4""\xBE""!\r\n </b></font><br><font face=\"Ja"
  "gran\" size=\"4\">\xE0""\xA4""\xAA""\xE0""\xA4""\xBF""\xE0""\xA4""\x9B"""
  "\xE0""\xA4""\xB2""\xE0""\xA5""\x87"" 27 \xE0""\xA4""\xAB""\xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xB5""\xE0""\xA4""\xB0""\xE0""\xA5""\x80"" \xE0""\xA4""\xB8"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\xAA""\xE0""\xA4""\xA4""\xE0""\xA4""\xBF"""
  " \xE0""\xA4""\xB6""\xE0""\xA5""\x87""\xE0""\xA4""\x96""\xE0""\xA4""\xB0"""
  " \xE0""\xA4""\x95""\xE0""\xA4""\xAA""\xE0""\xA5""\x82""\xE0""\xA4""\xB0"""
  " \xE0""\xA4""\xB8""\xE0""\xA5""\x87"" \xE0""\xA4""\x85""\xE0""\xA4""\xB2"""
  "\xE0""\xA4""\x97"" \xE0""\xA4""\xB9""\xE0""\xA5""\x8B"" \xE0""\xA4""\x9A"""
  "\xE0""\xA5""\x81""\xE0""\xA4""\x95""\xE0""\xA5""\x80""\xE0""\xA4""\x82"""
  " \xE0""\xA4""\xB8""\xE0""\xA5""\x81""\xE0""\xA4""\x9A""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xBE"""
  " \xE0""\xA4""\x95""\xE0""\xA5""\x83""\xE0""\xA4""\xB7""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xA3""\xE0""\xA4""\xAE""\xE0""\xA5""\x82""\xE0""\xA4""\xB0"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA4""\xBF"" \xE0""\xA4""\xA8"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\x86""\xE0""\xA4""\x96""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xB0"" \xE0""\xA4""\xAC""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"""
  " \xE0""\xA4""\xB9""\xE0""\xA5""\x80"" \xE0""\xA4""\xA6""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xAF""\xE0""\xA4""\xBE"" \xE0""\xA4""\x95""\xE0""\xA4""\xBF"""
  " \xE0""\xA4""\x89""\xE0""\xA4""\xA8""\xE0""\xA4""\x95""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\xA6""\xE0""\xA4""\xBE""\xE0""\xA4""\x82""\xE0""\xA4""\xAA"""
  "\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF"" \xE0""\xA4""\x95"""
  "\xE0""\xA5""\x8B"" \xE0""\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0"""
  "\xE0""\xA5""\x80""\xE0""\xA4""\xA4""\xE0""\xA4""\xBF"" \xE0""\xA4""\x9C"""
  "\xE0""\xA4""\xBF""\xE0""\xA4""\x82""\xE0""\xA4""\x9F""\xE0""\xA4""\xBE"""
  " \xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0""\xA4""\xA4""\xE0""\xA5""\x8B"""
  "\xE0""\xA4""\xA1""\xE0""\xA4""\xBC""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9"""
  "\xE0""\xA5""\x88"", \xE0""\xA4""\x87""\xE0""\xA4""\xB8""\xE0""\xA4""\xB2"""
  "\xE0""\xA4""\xBF""\xE0""\xA4""\x8F"" \xE0""\xA4""\x85""\xE0""\xA4""\xAC"""
  " \xE0""\xA4""\x90""\xE0""\xA4""\xB8""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB2"""
  "\xE0""\xA4""\x97""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9"""
  "\xE0""\xA5""\x88"" \xE0""\xA4""\x95""\xE0""\xA4""\xBF"" \xE0""\xA4""\x9A"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xB0"" \xE0""\xA4""\xB8""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xB2"" \xE0""\xA4""\xAA""\xE0""\xA4""\xB9""\xE0""\xA4""\xB2"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\xAD""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xA4"" \xE0""\xA4""\xB2""\xE0""\xA5""\x8C""\xE0""\xA4""\x9F"""
  " \xE0""\xA4""\x86""\xE0""\xA4""\x88"" \xE0""\xA4""\xB8""\xE0""\xA5""\x81"""
  "\xE0""\xA4""\x9A""\xE0""\xA4""\xBF""\xE0""\xA4""\xA4""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xB0""\xE0""\xA4""\xBE"" \xE0""\xA4""\x95""\xE0""\xA5""\x83"""
  "\xE0""\xA4""\xB7""\xE0""\xA5""\x8D""\xE0""\xA4""\xA3""\xE0""\xA4""\xAE"""
  "\xE0""\xA5""\x82""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4"""
  "\xE0""\xA4""\xBF"" \xE0""\xA4""\x94""\xE0""\xA4""\xB0"" \xE0""\xA4""\xB6"""
  "\xE0""\xA5""\x87""\xE0""\xA4""\x96""\xE0""\xA4""\xB0"" \xE0""\xA4""\x95"""
  "\xE0""\xA4""\xAA""\xE0""\xA5""\x82""\xE0""\xA4""\xB0"" \xE0""\xA4""\x95"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\xB8""\xE0""\xA4""\x82""\xE0""\xA4""\xAC"""
  "\xE0""\xA4""\x82""\xE0""\xA4""\xA7""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"""
  " \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\x96"""
  "\xE0""\xA4""\x9F""\xE0""\xA4""\xBE""\xE0""\xA4""\xB8"" \xE0""\xA4""\xAA"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x80""\xE0""\xA4""\xA4"""
  "\xE0""\xA4""\xBF"" \xE0""\xA4""\x95""\xE0""\xA5""\x80"" \xE0""\xA4""\xB5"""
  "\xE0""\xA4""\x9C""\xE0""\xA4""\xB9"" \xE0""\xA4""\xB8""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\xB9""\xE0""\xA5""\x80"" \xE0""\xA4""\x86""\xE0""\xA4""\x88"""
  " \xE0""\xA4""\xA5""\xE0""\xA5""\x80""\xE0""\xA5""\xA4"" <br> \xE0""\xA4"""
  "\xB8""\xE0""\xA5""\x81""\xE0""\xA4""\x9A""\xE0""\xA4""\xBF""\xE0""\xA4"""
  "\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xBE"" \xE0""\xA4"""
  "\xA8""\xE0""\xA5""\x87"" \xE0""\xA4""\xB6""\xE0""\xA5""\x87""\xE0""\xA4"""
  "\x96""\xE0""\xA4""\xB0"" \xE0""\xA4""\x95""\xE0""\xA5""\x80"" \xE0""\xA4"""
  "\xB5""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF""\xE0""\xA4""\xB8""\xE0""\xA5"""
  "\x8D""\xE0""\xA4""\xA4""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"" \xE0""\xA4"""
  "\xA4""\xE0""\xA5""\x8B""&nbsp;<img alt=\"\" align=\"absmiddle\" src=\"im"
  "ages/more.gif\" border=\"0\" WIDTH=\"15\" HEIGHT=\"6\"></font></a>\r\n  "
  "      \r\n          </td></tr></table>\r\n          </td></tr>\r\n\r\n  "
  "         <tr><td align=\"left\" bgColor=\"#ffe8d9\" vAlign=\"top\"><img "
  "alt=\"\" border=\"0\" height=\"1\" src=\"images/spacer.gif\" width=\"17\""
  "></td></tr>\r\n  <tr>\r\n    <td align=\"left\" vAlign=\"top\">\r\n\r\n "
  "     <table border=\"0\" cellPadding=\"0\" cellSpacing=\"0\" width=\"50%"
  "\" align=\"left\">\r\n        <tr><td align=\"left\" background=\"images"
  "/boxbg.gif\" vAlign=\"top\"><b><font color=\"#000000\" face=\"Jagran\" s"
  "ize=\"4\">&nbsp;<a href=\"khanakhazana\" style=\"TEXT-DECORATION: none\""
  " target=\"_blank\">\xE0""\xA4""\x96""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8"""
  "\xE0""\xA4""\xBE"" \xE0""\xA4""\x96""\xE0""\xA4""\x9C""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xA8""\xE0""\xA4""\xBE""</a></font></b></td><td align=\"left"
  "\" background=\"images/boxbg.gif\" vAlign=\"top\">&nbsp;</td></tr>\r\n  "
  "      <tr><td align=\"left\" vAlign=\"top\" width=\"95%\">\r\n        \r"
  "\n               \r\n        \r\n            <table id=\"Repeater6_ctl00"
  "_Table4\" cellpadding=\"2\" align=\"left\">\r\n\t<tr>\r\n\t\t<td><img id"
  "=\"Repeater6_ctl00_Image1\" src=\"http://img.jagran.com/khanakhazana/ima"
  "ges/kk255t.jpg\" align=\"left\" style=\"border-width:3px;border-style:Ou"
  "tset;\" /></td>\r\n\t</tr>\r\n</table>\r\n\r\n            <a href='khana"
  "khazana'>\r\n            <font face=\"Jagran\" size=\"4\"><b>\xE0""\xA4"""
  "\xB8""\xE0""\xA4""\xAE""\xE0""\xA4""\xB0"" \xE0""\xA4""\xAE""\xE0""\xA4"""
  "\xBF""\xE0""\xA4""\xB2""\xE0""\xA5""\x8D""\xE0""\xA4""\x95"" \xE0""\xA4"""
  "\xB6""\xE0""\xA5""\x87""\xE0""\xA4""\x95"" </b></font><br><font face=\"J"
  "agran\" size=\"4\">\xE0""\xA4""\x95""\xE0""\xA5""\x87""\xE0""\xA4""\xB2"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\xB5"" \xE0""\xA4""\x9A""\xE0""\xA5""\x80"""
  "\xE0""\xA4""\x95""\xE0""\xA5""\x82"" \xE0""\xA4""\x95""\xE0""\xA5""\x8B"""
  " \xE0""\xA4""\x9B""\xE0""\xA5""\x80""\xE0""\xA4""\xB2""\xE0""\xA4""\x95"""
  "\xE0""\xA4""\xB0"" \xE0""\xA4""\x9F""\xE0""\xA5""\x81""\xE0""\xA4""\x95"""
  "\xE0""\xA4""\xA1""\xE0""\xA4""\xBC""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"""
  " \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\x95"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x9F"" \xE0""\xA4""\xB2""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\x82""\xE0""\xA5""\xA4"" <br> \r\n\xE0""\xA4""\xB8""\xE0""\xA4"""
  "\x82""\xE0""\xA4""\xA4""\xE0""\xA4""\xB0""\xE0""\xA5""\x87"" \xE0""\xA4"""
  "\x95""\xE0""\xA5""\x8B"" \xE0""\xA4""\x9B""\xE0""\xA5""\x80""\xE0""\xA4"""
  "\xB2""\xE0""\xA4""\x95""\xE0""\xA4""\xB0"" \xE0""\xA4""\x89""\xE0""\xA4"""
  "\xB8""\xE0""\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\x9C""\xE0""\xA5"""
  "\x82""\xE0""\xA4""\xB8"" \xE0""\xA4""\xA8""\xE0""\xA4""\xBF""\xE0""\xA4"""
  "\x95""\xE0""\xA4""\xBE""\xE0""\xA4""\xB2""\xE0""\xA4""\x95""\xE0""\xA4"""
  "\xB0"" \xE0""\xA4""\x95""\xE0""\xA4""\xAA"" \xE0""\xA4""\xAE""\xE0""\xA5"""
  "\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\xB0""\xE0""\xA4""\x96"" \xE0""\xA4"""
  "\xB2""\xE0""\xA5""\x87""\xE0""\xA4""\x82""\xE0""\xA5""\xA4"" <br> \r\n\xE0"""
  "\xA4""\x85""\xE0""\xA4""\xAC"" \xE0""\xA4""\xAE""\xE0""\xA4""\xBF""\xE0"""
  "\xA4""\x95""\xE0""\xA5""\x8D""\xE0""\xA4""\xB8""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\x95""\xE0"""
  "\xA5""\x87""\xE0""\xA4""\xB2""\xE0""\xA4""\xBE"", \xE0""\xA4""\x9A""\xE0"""
  "\xA5""\x80""\xE0""\xA4""\x95""\xE0""\xA5""\x82"", \xE0""\xA4""\xB8""\xE0"""
  "\xA4""\x82""\xE0""\xA4""\xA4""\xE0""\xA4""\xB0""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB0""\xE0""\xA4""\xB8"", \xE0"""
  "\xA4""\xA6""\xE0""\xA5""\x82""\xE0""\xA4""\xA7"" \xE0""\xA4""\x94""\xE0"""
  "\xA4""\xB0"" \xE0""\xA4""\x9A""\xE0""\xA5""\x80""\xE0""\xA4""\xA8""\xE0"""
  "\xA5""\x80"" \xE0""\xA4""\xA1""\xE0""\xA4""\xBE""\xE0""\xA4""\xB2""\xE0"""
  "\xA4""\x95""\xE0""\xA4""\xB0"" \xE0""\xA4""\xB6""\xE0""\xA5""\x87""\xE0"""
  "\xA4""\x95"" \xE0""\xA4""\xAC""\xE0""\xA4""\xA8""\xE0""\xA4""\xBE"" \xE0"""
  "\xA4""\xB2""\xE0""\xA5""\x87""\xE0""\xA4""\x82""\xE0""\xA5""\xA4"" <br> "
  "\r\n\xE0""\xA4""\xA5""\xE0""\xA5""\x8B""\xE0""\xA4""\xA1""\xE0""\xA4""\xBC"""
  "\xE0""\xA5""\x80"" \xE0""\xA4""\xB8""\xE0""\xA5""\x80"" \xE0""\xA4""\xAC"""
  "\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\xAB"" \xE0""\xA4""\xA1"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xB2""\xE0""\xA4""\x95""\xE0""\xA4""\xB0"""
  " \xE0""\xA4""\xA0""\xE0""\xA4""\x82""\xE0""\xA4""\xA1""\xE0""\xA4""\xBE"""
  " \xE0""\xA4""\x95""\xE0""\xA4""\xB0""\xE0""\xA5""\x87""\xE0""\xA4""\x82"""
  " \xE0""\xA4""\x94""\xE0""\xA4""\xB0"" \xE0""\xA4""\xA0""\xE0""\xA4""\x82"""
  "\xE0""\xA4""\xA1""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB8""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xAE""\xE0""\xA4""\xB0"" \xE0""\xA4""\xAE""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xB2""\xE0""\xA5""\x8D""\xE0""\xA4""\x95"" \xE0""\xA4""\xB6"""
  "\xE0""\xA5""\x87""\xE0""\xA4""\x95"" \xE0""\xA4""\x97""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xB2""\xE0""\xA4""\xBE""\xE0""\xA4""\xB8"" \xE0""\xA4""\xAE"""
  "\xE0""\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\xA1""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xB2""\xE0""\xA4""\x95""\xE0""\xA4""\xB0"" \xE0""\xA4""\xA4"""
  "\xE0""\xA5""\x81""\xE0""\xA4""\xB0""\xE0""\xA4""\x82""\xE0""\xA4""\xA4"""
  " \xE0""\xA4""\xB8""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\xB5"""
  " \xE0""\xA4""\x95""\xE0""\xA4""\xB0""\xE0""\xA5""\x87""\xE0""\xA4""\x82"""
  "\xE0""\xA5""\xA4"" &nbsp;<img alt=\"\" align=\"absmiddle\" src=\"images/"
  "more.gif\" border=\"0\" WIDTH=\"15\" HEIGHT=\"6\"></font></a>\r\n       "
  " \r\n</td><td>&nbsp;</td>\r\n         </tr></table>\r\n         \r\n    "
  "    <table border=\"0\" cellPadding=\"0\" cellSpacing=\"0\" width=\"50%\""
  " align=\"left\">\r\n          <tr><td colspan=2 align=\"center\" backgro"
  "und=\"images/boxbg.gif\" vAlign=\"top\"><b><font color=\"#000000\" face="
  "\"Jagran\" size=\"4\"><a target=\"_parent\" href=\"josh\">\xE0""\xA4""\x9C"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"""
  " \xE0""\xA4""\x9C""\xE0""\xA5""\x8B""\xE0""\xA4""\xB6""</a></font></b></"
  "td></tr>\r\n          <tr><td align=\"left\" vAlign=\"top\" width=\"95%\""
  ">\r\n               \r\n        \r\n            <table id=\"Repeater7_ct"
  "l00_Table5\" cellpadding=\"2\" align=\"left\">\r\n\t<tr>\r\n\t\t<td><img"
  " id=\"Repeater7_ctl00_Image1\" src=\"http://img.jagran.com/josh/images/1"
  "1apriallead07t.jpg\" align=\"left\" style=\"border-width:3px;border-styl"
  "e:Outset;\" /></td>\r\n\t</tr>\r\n</table>\r\n\r\n            <a href='j"
  "osh/inner1new.aspx?idarticle=2588&idedition=217&idcategory=1'>\r\n      "
  "      <font face=\"Jagran\" size=\"4\"><b>\xE0""\xA4""\x96""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xA6""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF"" \xE0""\xA4"""
  "\xB8""\xE0""\xA4""\x82""\xE0""\xA4""\xB0""\xE0""\xA4""\x95""\xE0""\xA5"""
  "\x8D""\xE0""\xA4""\xB7""\xE0""\xA4""\xA3"" \xE0""\xA4""\x95""\xE0""\xA5"""
  "\x80"" '\xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\x95""\xE0""\xA4"""
  "\xBF""\xE0""\xA4""\xB2""' \xE0""\xA4""\x85""\xE0""\xA4""\xAA""\xE0""\xA4"""
  "\xA8""\xE0""\xA4""\xBE""\xE0""\xA4""\x8F""\xE0""\xA4""\x82"" \xE0""\xA4"""
  "\x95""\xE0""\xA4""\xB0""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF""\xE0""\xA4"""
  "\xB0"" \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4"""
  "\x86""\xE0""\xA4""\x97""\xE0""\xA5""\x87"" \xE0""\xA4""\xAC""\xE0""\xA4"""
  "\xA2""\xE0""\xA4""\xBC""\xE0""\xA4""\xA4""\xE0""\xA5""\x87"" \xE0""\xA4"""
  "\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\x8F""\xE0""\xA4""\x82"" </b></font>"
  "<br><font face=\"Jagran\" size=\"4\"><b>\xE0""\xA4""\x85""\xE0""\xA4""\xAA"""
  "\xE0""\xA4""\xA8""\xE0""\xA5""\x80""</b> \xE0""\xA4""\xAA""\xE0""\xA4""\xB8"""
  "\xE0""\xA4""\x82""\xE0""\xA4""\xA6""\xE0""\xA5""\x80""\xE0""\xA4""\xA6"""
  "\xE0""\xA4""\xBE"" \xE0""\xA4""\xAB""\xE0""\xA4""\xB2""\xE0""\xA5""\x8B"""
  "\xE0""\xA4""\x82"" \xE0""\xA4""\xB5"" \xE0""\xA4""\xB8""\xE0""\xA4""\xAC"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\x9C""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF"""
  "\xE0""\xA5""\x8B""\xE0""\xA4""\x82"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xB5""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xA6"" \xE0""\xA4""\x95""\xE0""\xA5""\x80"" \xE0""\xA4""\xAF"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xA6"" \xE0""\xA4""\x86""\xE0""\xA4""\xA4"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\xB9""\xE0""\xA5""\x80"" \xE0""\xA4""\xB5"""
  "\xE0""\xA4""\xBF""\xE0""\xA4""\xA8""\xE0""\xA4""\xBE""\xE0""\xA4""\xAF"""
  "\xE0""\xA4""\x95"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\xAE"""
  "\xE0""\xA5""\x81""\xE0""\xA4""\x82""\xE0""\xA4""\xB9"" \xE0""\xA4""\xAE"""
  "\xE0""\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\xAA""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xA8""\xE0""\xA5""\x80"" \xE0""\xA4""\xAD""\xE0""\xA4""\xB0"""
  " \xE0""\xA4""\x86""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9"""
  "\xE0""\xA5""\x88""\xE0""\xA5""\xA4"" \xE0""\xA4""\xB5""\xE0""\xA4""\xB9"""
  " \xE0""\xA4""\xB8""\xE0""\xA5""\x8B""\xE0""\xA4""\x9A""\xE0""\xA4""\xA4"""
  "\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9""\xE0""\xA5""\x88"" \xE0""\xA4""\x95"""
  "\xE0""\xA4""\xBF"" \xE0""\xA4""\x95""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF"""
  "\xE0""\xA4""\xBE"" \xE0""\xA4""\x87""\xE0""\xA4""\xA8"" \xE0""\xA4""\xB8"""
  "\xE0""\xA4""\xAD""\xE0""\xA5""\x80"" \xE0""\xA4""\x95""\xE0""\xA4""\xBE"""
  " \xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xB5""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\xA6"" \xE0""\xA4""\x89""\xE0""\xA4""\xB8""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\xB8""\xE0""\xA4""\xBE""\xE0""\xA4""\xB2"" \xE0""\xA4""\x95"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\xAC""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xB9""\xE0""\xA5""\x8B""\xC2""\xA0""\xE0""\xA4""\xAE""\xE0"""
  "\xA4""\xB9""\xE0""\xA5""\x80""\xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\xA8""\xE0""\xA4""\xB9""\xE0""\xA5""\x80""\xE0""\xA4""\x82"" \xE0"""
  "\xA4""\xAE""\xE0""\xA4""\xBF""\xE0""\xA4""\xB2"" \xE0""\xA4""\xB8""\xE0"""
  "\xA4""\x95""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE""! \xE0""\xA4""\xB9""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\xB2""\xE0""\xA4""\xBE""\xE0""\xA4""\x82""\xE0"""
  "\xA4""\x95""\xE0""\xA4""\xBF"", \xE0""\xA4""\x96""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xA8""\xE0""\xA5""\x87""-\xE0""\xA4""\x96""\xE0""\xA4""\xBF""\xE0"""
  "\xA4""\xB2""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8""\xE0""\xA5""\x87"" \xE0"""
  "\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\xB6""\xE0""\xA5""\x8C""\xE0"""
  "\xA4""\x95""\xE0""\xA5""\x80""\xE0""\xA4""\xA8"" \xE0""\xA4""\xB2""\xE0"""
  "\xA5""\x8B""\xE0""\xA4""\x97"" \xE0""\xA4""\xAA""\xE0""\xA5""\x8D""\xE0"""
  "\xA4""\xB0""\xE0""\xA4""\xBE""\xE0""\xA4""\x9A""\xE0""\xA5""\x80""\xE0"""
  "\xA4""\xA8"" \xE0""\xA4""\x95""\xE0""\xA4""\xBE""\xE0""\xA4""\xB2"" \xE0"""
  "\xA4""\xB8""\xE0""\xA5""\x87"" \xE0""\xA4""\xB9""\xE0""\xA5""\x80"" \xE0"""
  "\xA4""\x85""\xE0""\xA4""\x9A""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0"" \xE0"""
  "\xA4""\x94""\xE0""\xA4""\xB0"" \xE0""\xA4""\xAE""\xE0""\xA5""\x81""\xE0"""
  "\xA4""\xB0""\xE0""\xA4""\xAC""\xE0""\xA5""\x8D""\xE0""\xA4""\xAC""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\xB0""\xE0"""
  "\xA5""\x82""\xE0""\xA4""\xAA"" \xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0"""
  "\xA4""\x82"" \xE0""\xA4""\xAB""\xE0""\xA4""\xB2""\xE0""\xA5""\x8B""\xE0"""
  "\xA4""\x82""-\xE0""\xA4""\xB8""\xE0""\xA4""\xAC""\xE0""\xA5""\x8D""\xE0"""
  "\xA4""\x9C""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF""\xE0""\xA5""\x8B""\xE0"""
  "\xA4""\x82"" \xE0""\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB8""\xE0"""
  "\xA5""\x8D""\xE0""\xA4""\xB5""\xE0""\xA4""\xBE""\xE0""\xA4""\xA6"" \xE0"""
  "\xA4""\x9A""\xE0""\xA4""\x9F""\xE0""\xA4""\x95""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xB0""\xE0""\xA5""\x87"" \xE0""\xA4""\xB2""\xE0""\xA5""\x87""\xE0"""
  "\xA4""\x95""\xE0""\xA4""\xB0"" \xE0""\xA4""\xB2""\xE0""\xA5""\x87""\xE0"""
  "\xA4""\xA4""\xE0""\xA5""\x87"" \xE0""\xA4""\xB0""\xE0""\xA4""\xB9""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\xB9""\xE0""\xA5""\x88""\xE0""\xA4""\x82""\xE0"""
  "\xA5""\xA4"" \xE0""\xA4""\x86""\xE0""\xA4""\x9C"" \xE0""\xA4""\x95""&nbs"
  "p;<img alt=\"\" align=\"absmiddle\" src=\"images/more.gif\" border=\"0\""
  " WIDTH=\"15\" HEIGHT=\"6\"></font></a>\r\n        </td><td>&nbsp;</td>\r"
  "\n         </tr>\r\n\t\t\t</table>\r\n\t\t\t</td></tr>\r\n           <tr"
  "><td align=\"left\" bgColor=\"#ffe8d9\" vAlign=\"top\"><img alt=\"\" bor"
  "der=\"0\" height=\"1\" src=\"images/spacer.gif\" width=\"17\"></td></tr>"
  "\r\n           \t\t\t<tr><td align=center><script type=\"text/javascript"
  "\"><!--\r\ngoogle_ad_client = \"pub-2902991337334079\";\r\ngoogle_ad_wid"
  "th = 468;\r\ngoogle_ad_height = 60;\r\ngoogle_ad_format = \"468x60_as\";"
  "\r\ngoogle_ad_type = \"text_image\";\r\ngoogle_ad_channel =\"\";\r\n//--"
  "></script>\r\n<script type=\"text/javascript\"\r\n  src=\"http://pagead2"
  ".googlesyndication.com/pagead/show_ads.js\">\r\n</script></td></tr>\r\n<"
  "!-- josh & hatke -->\r\n<tr><td align=\"left\" bgColor=\"#ffe8d9\" vAlig"
  "n=\"top\"><img alt=\"\" border=\"0\" height=\"1\" src=\"images/spacer.gi"
  "f\" width=\"17\"></td></tr>\r\n  <tr>\r\n    <td align=\"left\" vAlign=\""
  "top\">\r\n\r\n      <table border=\"0\" cellPadding=\"0\" cellSpacing=\""
  "0\" width=\"50%\" align=\"left\">\r\n        <tr><td align=\"left\" back"
  "ground=\"images/boxbg.gif\" vAlign=\"top\"><b><font color=\"#000000\" fa"
  "ce=\"Jagran\" size=\"4\">&nbsp;<a target=\"_parent\" href=\"hatke\">\xE0"""
  "\xA4""\x9C""\xE0""\xA4""\xB0""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9""\xE0"""
  "\xA4""\x9F"" \xE0""\xA4""\x95""\xE0""\xA5""\x87""</a></font></b></td><td"
  " align=\"left\" background=\"images/boxbg.gif\" vAlign=\"top\">&nbsp;</t"
  "d></tr>\r\n        <tr><td colspan=2>\r\n        <table border=0 width=1"
  "00% cellspacing=\"0\" cellpadding=\"0\">\r\n                       \r\n "
  "       \r\n        <tr><td valign=baseline width=\"3%\"><font color=\"to"
  "mato\" face=\"verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td"
  ">\r\n        <th align=left><a href='hatke/InnerStory.aspx?StoryId=398'>"
  "<font face=\"Jagran\" size=\"3\">..\xE0""\xA4""\x85""\xE0""\xA4""\xAC"" "
  "\xE0""\xA4""\x97""\xE0""\xA4""\xA7""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"""
  " \xE0""\xA4""\x95""\xE0""\xA5""\x8B"" \xE0""\xA4""\xAD""\xE0""\xA5""\x80"""
  " \xE0""\xA4""\xAE""\xE0""\xA4""\xBF""\xE0""\xA4""\xB2""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\x97""\xE0""\xA4""\xBE"" \xE0""\xA4""\x85""\xE0""\xA4""\xB5"""
  "\xE0""\xA4""\x95""\xE0""\xA4""\xBE""\xE0""\xA4""\xB6"" </font></a></th><"
  "/tr>\r\n        \r\n        <tr><td valign=baseline width=\"3%\"><font c"
  "olor=\"tomato\" face=\"verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></"
  "font></td>\r\n        <th align=left><a href='hatke/InnerStory.aspx?Stor"
  "yId=397'><font face=\"Jagran\" size=\"3\">..\xE0""\xA4""\x94""\xE0""\xA4"""
  "\xB0"" \xE0""\xA4""\xAA""\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xA8""\xE0""\xA5""\x80"" \xE0""\xA4""\x95""\xE0""\xA5""\x8B"" \xE0""\xA4"""
  "\x95""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF""\xE0""\xA4""\xBE"" \xE0""\xA4"""
  "\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x87""\xE0""\xA4"""
  "\xAE""\xE0""\xA5""\x80"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4"""
  "\xB9""\xE0""\xA4""\xB5""\xE0""\xA4""\xBE""\xE0""\xA4""\xB2""\xE0""\xA5"""
  "\x87"" </font></a></th></tr>\r\n        \r\n        <tr><td valign=basel"
  "ine width=\"3%\"><font color=\"tomato\" face=\"verdana\" size=\"3\"><b>&"
  "nbsp;&nbsp;O&nbsp;</b></font></td>\r\n        <th align=left><a href='ha"
  "tke/InnerStory.aspx?StoryId=396'><font face=\"Jagran\" size=\"3\">\xE0"""
  "\xA4""\x89""\xE0""\xA4""\xAE""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0"" 120\xC2"""
  "\xA0""\xE0""\xA4""\xB5""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xB7"", 48\xC2""\xA0""\xE0""\xA4""\xB5""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xB7""\xE0""\xA5""\x8B"" \xE0""\xA4""\xB8""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\xA8""\xE0""\xA4""\xB9""\xE0""\xA4""\xBE""\xE0""\xA4""\xAF"""
  "\xE0""\xA4""\xBE"" \xE0""\xA4""\xB9""\xE0""\xA5""\x80"" \xE0""\xA4""\xA8"""
  "\xE0""\xA4""\xB9""\xE0""\xA5""\x80""\xE0""\xA4""\x82"" </font></a></th><"
  "/tr>\r\n        \r\n        <tr><td valign=baseline width=\"3%\"><font c"
  "olor=\"tomato\" face=\"verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></"
  "font></td>\r\n        <th align=left><a href='hatke/InnerStory.aspx?Stor"
  "yId=395'><font face=\"Jagran\" size=\"3\">\xE0""\xA4""\xAE""\xE0""\xA5"""
  "\x8B""\xE0""\xA4""\x9F""\xE0""\xA5""\x80"" \xE0""\xA4""\xB0""\xE0""\xA4"""
  "\x95""\xE0""\xA4""\xAE"" \xE0""\xA4""\xA6""\xE0""\xA5""\x87""\xE0""\xA4"""
  "\x95""\xE0""\xA4""\xB0"" \xE0""\xA4""\x85""\xE0""\xA4""\xA8""\xE0""\xA5"""
  "\x8B""\xE0""\xA4""\x96""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB5""\xE0""\xA5"""
  "\x8D""\xE0""\xA4""\xAF""\xE0""\xA4""\xBE""\xE0""\xA4""\xAA""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xB0"" </font></a></th></tr>\r\n        \r\n        <t"
  "r><td valign=baseline width=\"3%\"><font color=\"tomato\" face=\"verdana"
  "\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n        <th alig"
  "n=left><a href='hatke/InnerStory.aspx?StoryId=394'><font face=\"Jagran\""
  " size=\"3\">\xE0""\xA4""\xA4""\xE0""\xA4""\xA8""\xE0""\xA5""\x8D""\xE0"""
  "\xA4""\xB9""\xE0""\xA4""\xBE""\xE0""\xA4""\x88"" \xE0""\xA4""\xAE""\xE0"""
  "\xA5""\x87""\xE0""\xA4""\x82"" \xE0""\xA4""\x87""\xE0""\xA4""\x82""\xE0"""
  "\xA4""\x9F""\xE0""\xA4""\xB0""\xE0""\xA4""\xA8""\xE0""\xA5""\x87""\xE0"""
  "\xA4""\x9F"" \xE0""\xA4""\xAC""\xE0""\xA4""\xA8""\xE0""\xA4""\xBE"" \xE0"""
  "\xA4""\xB5""\xE0""\xA4""\xB0""\xE0""\xA4""\xA6""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\xA8"" </font></a></th></tr>\r\n        \r\n        <tr><td valign"
  "=baseline width=\"3%\"><font color=\"tomato\" face=\"verdana\" size=\"3\""
  "><b>&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n        <th align=left><a hre"
  "f='hatke/InnerStory.aspx?StoryId=393'><font face=\"Jagran\" size=\"3\">\xE0"""
  "\xA4""\xB8""\xE0""\xA5""\x8B""\xE0""\xA4""\x8A""\xE0""\xA4""\x82""\xE0"""
  "\xA4""\x97""\xE0""\xA4""\xBE"", \xE0""\xA4""\xAE""\xE0""\xA4""\x97""\xE0"""
  "\xA4""\xB0"" \xE0""\xA4""\x85""\xE0""\xA4""\xAA""\xE0""\xA4""\xA8""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\xA8""\xE0""\xA4""\xB9""\xE0""\xA5""\x80""\xE0"""
  "\xA4""\x82"" \xE0""\xA4""\x94""\xE0""\xA4""\xB0""\xE0""\xA5""\x8B""\xE0"""
  "\xA4""\x82"" \xE0""\xA4""\x95""\xE0""\xA5""\x87"" \xE0""\xA4""\x98""\xE0"""
  "\xA4""\xB0""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"" \xE0""\xA4""\xAE""\xE0"""
  "\xA5""\x87""\xE0""\xA4""\x82"" </font></a></th></tr>\r\n        \r\n    "
  "        </table>\r\n            </td></tr>\r\n        </table>\r\n      "
  "   \r\n        <table border=\"0\" cellPadding=\"0\" cellSpacing=\"0\" w"
  "idth=\"50%\" align=\"left\">\r\n          <tr><td colspan=2 align=\"cent"
  "er\" background=\"images/boxbg.gif\" vAlign=\"top\"><b><font color=\"#00"
  "0000\" face=\"Jagran\" size=\"4\"><a href=\"radio\" style=\"TEXT-DECORAT"
  "ION: none\" target=\"_blank\">\xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0"""
  "\xA4""\x97""\xE0""\xA4""\xB0""\xE0""\xA4""\xA3"" \xE0""\xA4""\xB0""\xE0"""
  "\xA5""\x87""\xE0""\xA4""\xA1""\xE0""\xA4""\xBF""\xE0""\xA4""\xAF""\xE0"""
  "\xA5""\x8B"" </a></font></b></td></tr>\r\n\t\t                       \r\n"
  "        \r\n        <tr><td valign=baseline width=\"3%\"><font color=\"t"
  "omato\" face=\"verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></t"
  "d>\r\n        <th align=left ><a href='/radio'><font face=\"Jagran\" siz"
  "e=\"3\">\xE0""\xA4""\xB9""\xE0""\xA5""\x8B""\xE0""\xA4""\xB2""\xE0""\xA5"""
  "\x80"" \xE0""\xA4""\x86""\xE0""\xA4""\x88"" \xE0""\xA4""\xB0""\xE0""\xA5"""
  "\x87""...</font></a></th></tr>\r\n        \r\n        <tr><td valign=bas"
  "eline width=\"3%\"><font color=\"tomato\" face=\"verdana\" size=\"3\"><b"
  ">&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n        <th align=left ><a href="
  "'/radio'><font face=\"Jagran\" size=\"3\">\xE0""\xA4""\xAD""\xE0""\xA4"""
  "\x95""\xE0""\xA5""\x8D""\xE0""\xA4""\xA4""\xE0""\xA4""\xBF"" \xE0""\xA4"""
  "\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\xAA""\xE0""\xA4""\xB0""\xE0""\xA5"""
  "\x8D""\xE0""\xA4""\xB5"" \xE0""\xA4""\xAE""\xE0""\xA4""\xB9""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xB6""\xE0""\xA4""\xBF""\xE0""\xA4""\xB5""\xE0""\xA4"""
  "\xB0""\xE0""\xA4""\xBE""\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4"""
  "\xB0""\xE0""\xA4""\xBF""</font></a></th></tr>\r\n        \r\n        <tr"
  "><td valign=baseline width=\"3%\"><font color=\"tomato\" face=\"verdana\""
  " size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n        <th align="
  "left ><a href='/radio'><font face=\"Jagran\" size=\"3\">\xE0""\xA4""\x9A"""
  "\xE0""\xA4""\xB2""\xE0""\xA5""\x8B"" \xE0""\xA4""\xA5""\xE0""\xA5""\x8B"""
  "\xE0""\xA4""\xA1""\xE0""\xA4""\xBC""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB0"""
  "\xE0""\xA5""\x82""\xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8"""
  "\xE0""\xA5""\x80"" \xE0""\xA4""\xB9""\xE0""\xA5""\x8B"" \xE0""\xA4""\x9C"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\x8F""..</font></a></th></tr>\r\n        \r"
  "\n        <tr><td valign=baseline width=\"3%\"><font color=\"tomato\" fa"
  "ce=\"verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n    "
  "    <th align=left ><a href='/radio'><font face=\"Jagran\" size=\"3\">\xE0"""
  "\xA4""\xB5""\xE0""\xA4""\xB8""\xE0""\xA4""\x82""\xE0""\xA4""\xA4"" \xE0"""
  "\xA4""\x86""\xE0""\xA4""\xAF""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"" \xE0"""
  "\xA4""\xB0""\xE0""\xA5""\x87""..</font></a></th></tr>\r\n        \r\n   "
  "     <tr><td valign=baseline width=\"3%\"><font color=\"tomato\" face=\""
  "verdana\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n        <"
  "th align=left ><a href='/radio'><font face=\"Jagran\" size=\"3\">\xE0""\xA4"""
  "\xAF""\xE0""\xA5""\x87"" \xE0""\xA4""\xA6""\xE0""\xA5""\x87""\xE0""\xA4"""
  "\xB6"" \xE0""\xA4""\xB9""\xE0""\xA5""\x88"" \xE0""\xA4""\xB5""\xE0""\xA5"""
  "\x80""\xE0""\xA4""\xB0"" \xE0""\xA4""\x9C""\xE0""\xA4""\xB5""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xA8""\xE0""\xA5""\x8B""\xE0""\xA4""\x82"" \xE0""\xA4"""
  "\x95""\xE0""\xA4""\xBE"".</font></a></th></tr>\r\n        \r\n        <t"
  "r><td valign=baseline width=\"3%\"><font color=\"tomato\" face=\"verdana"
  "\" size=\"3\"><b>&nbsp;&nbsp;O&nbsp;</b></font></td>\r\n        <th alig"
  "n=left ><a href='/radio'><font face=\"Jagran\" size=\"3\">\xE0""\xA4""\x9D"""
  "\xE0""\xA4""\xB2""\xE0""\xA4""\x95"" 2006</font></a></th></tr>\r\n      "
  "  \r\n\t\t\t</table>\r\n          </td></tr>\r\n<!-- josh and hatke -->\t"
  "\t\t\r\n<!-- Bottom Banner -->\r\n <tr><td align=\"left\" bgColor=\"#ffe"
  "8d9\" vAlign=\"top\"><img alt=\"\" border=\"0\" height=\"1\" src=\"image"
  "s/spacer.gif\" width=\"6\"></td></tr>\r\n<tr align=center>\r\n<td bgcolo"
  "r=\"#FFFFFF\" valign=top align=center>\r\n </td></tr>\r\n\r\n <!--tr><td"
  " align=\"center\" vAlign=\"top\">\r\n<a href=\"worldcup/tez.htm\" target"
  "=_blank><img src=\"images/tez.gif\" border=0></a>\r\n</td></tr-->  \r\n<"
  "tr><td align=\"left\" vAlign=\"top\"><img alt=\"\" border=\"0\" height=\""
  "9\" src=\"images/spacer.gif\" width=\"6\"></td></tr>\r\n<tr><td align=\""
  "center\">\r\n  <map name=\"FPMap1\">\r\n  <area alt=\"\" href=\"http://w"
  "ww.jagran.com/about/about.htm\" shape=\"rect\" coords=\"15,3,49,15\">\r\n"
  "  <area alt=\"\" href=\"mailto:webmaster@jagran.com?subject=feedback/sug"
  "gestions\" shape=\"rect\" coords=\"64,5,133,17\">\r\n  <area alt=\"\" hr"
  "ef=\"mailto:webmaster@jagran.com?subject=feedback/suggestions\" shape=\""
  "rect\" coords=\"152,4,208,16\">\r\n  <area alt=\"\" href=\"http://www.ja"
  "gran.com/friend/sendpage.asp\" onclick=\"javascript:friend()\" target=ab"
  "c shape=\"rect\" coords=\"224,2,275,16\">\r\n  </map><img alt=\"\" borde"
  "r=\"0\" height=\"19\" src=\"images/channel1.jpg\" useMap=\"#FPMap1\" wid"
  "th=\"292\"></td></tr>\r\n  <tr><td align=\"center\" vAlign=\"top\"><font"
  " face=\"Verdana\" size=\"1\">\xC2""\xA9"" Jagran Infotech Ltd. 2006-08</"
  "font></td></tr></table>       \r\n\r\n</td>\r\n<td width=\"16%\" valign="
  "top align=center>\r\n<a href=\"news/font.zip\"><img src=news/images/font"
  "down.gif border=0 /></a>\r\n<br />\r\n\r\n<table width=100% border=0 ali"
  "gn=center>\r\n\r\n<tr><td align=\"left\" vAlign=\"top\" bgColor=\"#fffff"
  "f\"><img alt=\"\" border=\"0\" height=\"2\" src=\"images/spacer.gif\" wi"
  "dth=\"2\"></td></tr>\r\n<tr bgColor=\"#ffffff\"><td><a target=\"_blank\""
  " href=\"http://www.shaadi.com/ptnr.php?ptnr=zl6uv\"><img alt=\"\" src=\""
  "images/shadinew125x125.gif\" border=0></a></td></tr>\r\n<tr><td align=\""
  "left\" vAlign=\"top\" bgColor=\"#ffffff\"><img alt=\"\" border=\"0\" hei"
  "ght=\"2\" src=\"images/spacer.gif\" width=\"2\"></td></tr>\r\n\r\n<tr><t"
  "d align=\"left\" vAlign=\"top\" bgcolor=\"#ffffff\"  align=center>\r\n<!"
  "-- Shaadi -->\r\n<table width=125 border=0 cellpadding=0 cellspacing=0 a"
  "lign=center>\r\n<form method=\"post\" action=\"http://www.shaadi.com/par"
  "tner_search/matrimonial_search/searchresults.php?ptnr=jnbnr\" target=\"_"
  "blank\">\r\n<input type=\"hidden\" name=\"photograph\" value=\"Yes\">\r\n"
  "<input type=\"hidden\" name=\"mothertongue\" value=\"Hindi\">\r\n\t<tr>\r"
  "\n\t\t<td>\r\n\t\t\t<object classid=\"clsid:d27cdb6e-ae6d-11cf-96b8-4445"
  "53540000\" codebase=\"http://download.macromedia.com/pub/shockwave/cabs/"
  "flash/swflash.cab#version=7,0,0,0\" width=\"125\" height=\"148\" id=\"12"
  "5\" align=\"middle\">\r\n<param name=\"allowScriptAccess\" value=\"sameD"
  "omain\" />\r\n<param name=\"movie\" value=\"images/125.swf\" />\r\n<para"
  "m name=\"quality\" value=\"high\" />\r\n<param name=\"bgcolor\" value=\""
  "#ffffff\" />\r\n<embed src=\"images/125.swf\" quality=\"high\" bgcolor=\""
  "#ffffff\" width=\"125\" height=\"148\" name=\"125\" align=\"middle\" all"
  "owScriptAccess=\"sameDomain\" type=\"application/x-shockwave-flash\" plu"
  "ginspage=\"http://www.macromedia.com/go/getflashplayer\" />\r\n</object>"
  "</td>\r\n\t</tr>\r\n\t<tr>\r\n\t\t<td background=\"images/searchbg.gif\""
  " width=125 height=73 align=center><b style=\"font: bold 12px arial;\" al"
  "ign=center>Search now</b><br>\r\n\t\t\t<select name=\"gender\" style=\"f"
  "ont: normal 11px tahoma; color: #000000; width: 85px; margin-top:5px;\">"
  "\r\n\t\t\t\t<option value=\"female\" selected>Bride</option>\r\n\t\t\t\t"
  "<option value=\"male\">Groom</option>\r\n\t\t\t</select><br><select name"
  "=\"age\" style=\"font:normal 11px tahoma; color: #000000; width: 85px; m"
  "argin-top:8px;\">\r\n\t\t\t\t<option value=\"18 - 24\" selected>18 - 24<"
  "/option>\r\n\t\t\t\t<option value=\"25 - 30\">25 - 30</option>\r\n\t\t\t"
  "\t<option value=\"31 - 35\">31 - 35</option>\r\n\t\t\t\t<option value=\""
  "36 - 45\">36 - 45</option>\r\n\t\t\t\t<option value=\"46 - 50\">46 - 50<"
  "/option>\r\n\t\t\t\t<option value=\"51 - 99\">50+</option>\r\n\t\t\t\t</"
  "select></td>\r\n\t</tr>\r\n\t<tr>\r\n\t\t<td><input type=\"image\" src=\""
  "images/search.gif\" width=125 height=29 border=0></td>\r\n\t</tr></form>"
  "\r\n</table><!--shaadi -->\r\n</td></tr>\r\n<tr><td align=\"left\" vAlig"
  "n=\"top\" bgColor=\"#ffffff\"><img alt=\"\" border=\"0\" height=\"2\" sr"
  "c=\"images/spacer.gif\" width=\"2\"></td></tr>\r\n<tr><td align=\"center"
  "\" vAlign=\"top\" bgcolor=\"#ffffff\">\r\n<img src=\"images/textshopping"
  ".gif\" width=\"120\" height=\"120\" usemap=\"#Maptext\" border=\"0\"> \r"
  "\n<map name=\"Maptext\">\r\n  <area shape=\"rect\" coords=\"0,3,118,28\""
  " href=\"http://shopping.jagran.com/Product_Details.aspx?ItemId=10873\" t"
  "arget=_blank>\r\n  <area shape=\"rect\" coords=\"10,43,109,79\" href=\"h"
  "ttp://shopping.jagran.com/Product_Details.aspx?ItemId=9997\" target=_bla"
  "nk>\r\n  <area shape=\"rect\" coords=\"12,91,108,113\" href=\"http://sho"
  "pping.jagran.com/product_price.aspx?div=193&divname=Home%20Appliances\" "
  "target=_blank>\r\n</map>\r\n<br>\r\n<a href=\"http://ind.jagran.com/irel"
  "and/\" target=_blank><img alt=\"\" border=\"0\" src=\"http://ind.jagran."
  "com/images/irelandban.gif\"></a>\r\n<br>\r\n<a href=\"http://www.jagranc"
  "ityplus.com\" target=_blank><img alt=\"\" border=\"0\" src=\"images/City"
  "Plusban3.gif\"></a>\r\n</td></tr>\r\n<tr><td align=\"left\" vAlign=\"top"
  "\" bgColor=\"#ffffff\"><img alt=\"\" border=\"0\" height=\"2\" src=\"ima"
  "ges/spacer.gif\" width=\"2\"></td></tr>\r\n<tr><td align=\"left\" vAlign"
  "=\"top\" bgColor=\"#ffffff\"><img alt=\"\" border=\"0\" height=\"2\" src"
  "=\"images/spacer.gif\" width=\"2\"></td></tr>\r\n<tr><td align=\"center\""
  " vAlign=\"top\" bgColor=\"#ffffff\"><a href=\"http://www1.jagran.com/kat"
  "hputli/default.asp\" target=_blank><img alt=\"\" src=\"images/kathputli."
  "gif\" border=0></a></td></tr>\r\n<tr><td align=\"left\" vAlign=\"top\" b"
  "gColor=\"#ffffff\"><img alt=\"\" border=\"0\" height=\"2\" src=\"images/"
  "spacer.gif\" width=\"2\"></td></tr>\r\n<tr><td align=\"center\" vAlign=\""
  "top\" bgcolor=\"#ffffff\"><a href=\"http://www6.jagran.com/jagranimage/D"
  "efault.aspx?subcatgid=62\" target=_blank><img alt=\"\" border=\"0\" src="
  "\"images/art.gif\"></a></td></tr>\t\t\r\n<tr><td align=\"left\" vAlign=\""
  "top\" bgColor=\"#ffffff\"><img alt=\"\" border=\"0\" height=\"2\" src=\""
  "images/spacer.gif\" width=\"2\"></td></tr>\r\n<!-- Link-->\r\n\r\n\t\t<t"
  "r><td align=left bgcolor=\"#ffffff\" valign=\"top\">\r\n\t\t<table borde"
  "r=\"0\" cellpadding=\"0\" cellspacing=\"0\" width=\"100%\" background=\""
  "images/line11.gif\">\r\n\t\t<tr>\r\n          <td bgcolor=\"#FFB780\"><i"
  "mg alt=\"\" border=\"0\" src=\"images/spacer.gif\" width=\"1\" height=\""
  "2\"></td>\r\n        </tr>\r\n        <tr><th width=\"100%\" align=\"cen"
  "ter\"><a href=\"javascript:editor()\"><font face=\"Jagran\" color=\"blue"
  "\" size=\"4\"><b>\xE0""\xA4""\xB8""\xE0""\xA4""\x82""\xE0""\xA4""\xAA""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\xA6""\xE0""\xA4""\x95"" \xE0""\xA4""\x95""\xE0"""
  "\xA5""\x87"" \xE0""\xA4""\xA8""\xE0""\xA4""\xBE""\xE0""\xA4""\xAE"" \xE0"""
  "\xA4""\xAA""\xE0""\xA4""\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""</b></"
  "font></a></th></tr>\r\n        <tr><td bgcolor=\"#FFB780\"><img alt=\"\""
  " border=\"0\" src=\"images/spacer.gif\" width=\"1\" height=\"2\"></td></"
  "tr>\r\n        <tr><th width=\"100%\" align=\"center\"><a href=\"japan/d"
  "efault.aspx\" target=_blank><font face=\"Jagran\" color=\"blue\" size=\""
  "4\"><b>\xE0""\xA4""\xAE""\xE0""\xA5""\x87""\xE0""\xA4""\xB0""\xE0""\xA5"""
  "\x80"" \xE0""\xA4""\x9C""\xE0""\xA4""\xBE""\xE0""\xA4""\xAA""\xE0""\xA4"""
  "\xBE""\xE0""\xA4""\xA8"" \xE0""\xA4""\xAF""\xE0""\xA4""\xBE""\xE0""\xA4"""
  "\xA4""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA4""\xBE""</b></font><"
  "/a></th></tr>\r\n        <tr><td bgcolor=\"#FFB780\"><img alt=\"\" borde"
  "r=\"0\" src=\"images/spacer.gif\" width=\"1\" height=\"2\"></td></tr>\r\n"
  "        <tr><th width=\"100%\" align=\"center\"><a href=\"http://www1.ja"
  "gran.com/newsletter/registration.asp\" onclick=\"javascript:friend()\" t"
  "arget=abc><font face=\"Jagran\" color=\"blue\" size=\"4\"><b>\xE0""\xA4"""
  "\xA8""\xE0""\xA5""\x8D""\xE0""\xA4""\xAF""\xE0""\xA5""\x82""\xE0""\xA4"""
  "\x9C"" \xE0""\xA4""\xB2""\xE0""\xA5""\x87""\xE0""\xA4""\x9F""\xE0""\xA4"""
  "\xB0""</b></font></a></th></tr>\r\n        <tr><td bgcolor=\"#FFB780\"><"
  "img alt=\"\" border=\"0\" src=\"images/spacer.gif\" width=\"1\" height=\""
  "2\"></td></tr>\r\n        <tr><th width=\"100%\" align=\"center\"><a hre"
  "f=\"nri/default.aspx\" target=\"_blank\"><font face=\"Jagran\" color=\"b"
  "lue\" size=\"4\"><b>\xE0""\xA4""\xAA""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xB5""\xE0""\xA4""\xBE""\xE0""\xA4""\xB8""\xE0""\xA5""\x80"""
  " \xE0""\xA4""\xAD""\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA4""\xA4"""
  "\xE0""\xA5""\x80""\xE0""\xA4""\xAF""</b></font></a></th></tr>\r\n       "
  " <tr><td bgcolor=\"#FFB780\"><img alt=\"\" border=\"0\" src=\"images/spa"
  "cer.gif\" width=\"1\" height=\"2\"></td></tr>\r\n        <tr><th width=\""
  "100%\" align=\"center\"><a href=\"opgupta/default.aspx\" target=\"_blank"
  "\"><font face=\"Jagran\" color=\"blue\" size=\"4\"><b>\xE0""\xA4""\xA7"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xB0""\xE0""\xA5""\x8D""\xE0""\xA4""\xAE"""
  "\xE0""\xA4""\xBF""\xE0""\xA4""\x95"" \xE0""\xA4""\x97""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xB0""\xE0""\xA4""\x82""\xE0""\xA4""\xA5"" \xE0""\xA4""\x94"""
  "\xE0""\xA4""\xB0"" \xE0""\xA4""\xB9""\xE0""\xA4""\xBF""\xE0""\xA4""\xA8"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xA6""\xE0""\xA5""\x82"" \xE0""\xA4""\xB8"""
  "\xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0""\xA4""\x9C""</b></font></a></t"
  "h></tr>\r\n        <tr><td bgcolor=\"#FFB780\"><img alt=\"\" border=\"0\""
  " src=\"images/spacer.gif\" width=\"1\" height=\"2\"></td></tr>\r\n      "
  "  <tr><th width=\"100%\" align=\"center\"><a href=\"http://www1.jagran.c"
  "om/nmg/default.htm\" target=\"_blank\"><font face=\"Jagran\" color=\"blu"
  "e\" size=\"4\"><b>\xE0""\xA4""\xA8""\xE0""\xA4""\xB0""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\x82""\xE0""\xA4""\xA6""\xE0""\xA5""\x8D""\xE0""\xA4""\xB0"""
  " \xE0""\xA4""\xAE""\xE0""\xA5""\x8B""\xE0""\xA4""\xB9""\xE0""\xA4""\xA8"""
  " - \xE0""\xA4""\xB8""\xE0""\xA5""\x8D""\xE0""\xA4""\xAE""\xE0""\xA5""\x83"""
  "\xE0""\xA4""\xA4""\xE0""\xA4""\xBF"" \xE0""\xA4""\xB6""\xE0""\xA5""\x87"""
  "\xE0""\xA4""\xB7""</b></font></a></th></tr>\r\n        <tr><td bgcolor=\""
  "#FFB780\"><img alt=\"\" border=\"0\" src=\"images/spacer.gif\" width=\"1"
  "\" height=\"2\"></td></tr>\r\n        </table>\r\n\t\t</td></tr>\r\n\t\t"
  "<!--Link-->\r\n<tr><td>\r\n<form action=\"http://www6.jagran.com/news/ja"
  "nmat/poll.asp\" id=\"form2\" method=\"post\" name=\"form2\" target=\"_bl"
  "ank\">\r\n<table border=0  bgColor=\"#ffe8d9\" width=100%> \r\n<tr>\r\n<"
  "td align=\"left\" bgColor=\"#ff6601\" vAlign=\"top\">\r\n<font face=\"Ja"
  "gran\" size=\"4\">&nbsp;\xE0""\xA4""\x9C""\xE0""\xA4""\xA8""\xE0""\xA4"""
  "\xAE""\xE0""\xA4""\xA4""</font> </td></tr>\r\n\t        \r\n\r\n\r\n\t<t"
  "r><td><font face=\"Jagran\" size=\"4\">\xE0""\xA4""\x95""\xE0""\xA5""\x8D"""
  "\xE0""\xA4""\xAF""\xE0""\xA4""\xBE"" \xE0""\xA4""\xA4""\xE0""\xA4""\xBF"""
  "\xE0""\xA4""\xB0""\xE0""\xA4""\x82""\xE0""\xA4""\x97""\xE0""\xA5""\x87"""
  " \xE0""\xA4""\xB8""\xE0""\xA4""\xB0""\xE0""\xA5""\x80""\xE0""\xA4""\x96"""
  "\xE0""\xA5""\x87"" \xE0""\xA4""\x95""\xE0""\xA5""\x87""\xE0""\xA4""\x95"""
  " \xE0""\xA4""\x95""\xE0""\xA5""\x8B"" \xE0""\xA4""\x95""\xE0""\xA4""\xBE"""
  "\xE0""\xA4""\x9F""\xE0""\xA4""\xA8""\xE0""\xA4""\xBE"" \xE0""\xA4""\xB0"""
  "\xE0""\xA4""\xBE""\xE0""\xA4""\xB7""\xE0""\xA5""\x8D""\xE0""\xA4""\x9F"""
  "\xE0""\xA5""\x8D""\xE0""\xA4""\xB0""\xE0""\xA5""\x80""\xE0""\xA4""\xAF"""
  " \xE0""\xA4""\xA7""\xE0""\xA5""\x8D""\xE0""\xA4""\xB5""\xE0""\xA4""\x9C"""
  " \xE0""\xA4""\x95""\xE0""\xA4""\xBE"" \xE0""\xA4""\x85""\xE0""\xA4""\xAA"""
  "\xE0""\xA4""\xAE""\xE0""\xA4""\xBE""\xE0""\xA4""\xA8"" \xE0""\xA4""\xB9"""
  "\xE0""\xA5""\x88""?</font></td></tr>\r\n\t\t\t<tr><td align=\"left\">&nb"
  "sp;<input type=\"radio\" value=\"Y\" style=\"width:auto\" checked name=\""
  "vote\">&nbsp;&nbsp;<font face=\"Jagran\" size=\"4\">\xE0""\xA4""\xB9""\xE0"""
  "\xA4""\xBE""\xE0""\xA4""\x82""</font></td></tr>\r\n\t\t\t<tr><td align=\""
  "left\">&nbsp;<input type=\"radio\" value=\"N\" style=\"width:auto\" name"
  "=\"vote\">&nbsp;&nbsp;<font face=\"Jagran\" size=\"4\">\xE0""\xA4""\xA8"""
  "\xE0""\xA4""\xB9""\xE0""\xA5""\x80""\xE0""\xA4""\x82""</font></td></tr>\r"
  "\n\t\t\t<tr><td align=\"left\">&nbsp;<input type=\"radio\" value=\"C\" s"
  "tyle=\"width:auto\" name=\"vote\">&nbsp;&nbsp;<font face=\"Jagran\" size"
  "=\"4\">\xE0""\xA4""\xAA""\xE0""\xA4""\xA4""\xE0""\xA4""\xBE"" \xE0""\xA4"""
  "\xA8""\xE0""\xA4""\xB9""\xE0""\xA5""\x80""\xE0""\xA4""\x82""</font></td>"
  "</tr>\r\n\t\t\t<tr><td valign=\"middle\" align=\"center\"><input type=\""
  "hidden\" name=\"qid\" value='2717'>\r\n\t\t\t<input style=\"width:auto\""
  " type=\"image\" src=\"images/vote.gif\" name=\"Vote\"></td></tr>\r\n    "
  "    \r\n</table>\r\n</form>\r\n</td></tr>\r\n</table>\r\n\r\n</td>\r\n</"
  "tr>\r\n</table>\r\n</body>\r\n</html>\r\n"
;


// This one is a mixture of UTF-8, CP1252, and UTF8UTF8
const char* kTeststr99 =
// 1252
  " auf der Computermesse Systems in M\xFC""nchen vertreten (Halle A3.542)."
  " Artikel des Tages Das Weinbaugebiet Bordeaux, auf franz\xF6""sisch Bord"
  "elais, ist das gr\xF6""\xDF""te zusammenh\xE4""ngende Anbaugebiet der We"
  "lt f\xFC""r Qualit\xE4""tswein. Es gibt etwa 4000 Ch\xE2""teaux genannte"
// UTF8
  "tnegirjji gosa gii beare s\xC3""\xA1""htt\xC3""\xA1"" \xC4""\x8D""\xC3"""
  "\xA1""llit artihkkaliid. Maid don s\xC3""\xA1""ht\xC3""\xA1""t dievasmah"
  "ttit ja divvut juo \xC4""\x8D""\xC3""\xA1""llojuvvon artihkkaliid dahje "
  "\xC4""\x8D""\xC3""\xA1""lligoahttit aibbas o\xC4""\x91""\xC4""\x91""a ar"
// UTF8UTF8
  "Ice cream a l\xc3\x83\xc2\xa0 mode."
// 1252
  " Weing\xFC""ter, die die weltber\xFC""hmten Weine erzeugen. Ein differen"
  "ziertes System von subregionalen und kommunalen Appellationen und Klassi"
  "fikationen schafft unter ihnen eine qualitative Hierarchie. Die einzelne"
  "n Lagen spielen demgegen\xFC""ber eine untergeordnete Rolle. Ihre Stelle"
// UTF8UTF8
  "Ice cream \xC3\x83\xC2\xA9 mode."
  "Ice cream \xC3\xA2\xE2\x82\xAC\xCB\x9C mode."
// UTF8
  "tnegirjji gosa gii beare s\xC3""\xA1""htt\xC3""\xA1"" \xC4""\x8D""\xC3"""
  "\xA1""llit artihkkaliid. Maid don s\xC3""\xA1""ht\xC3""\xA1""t dievasmah"
  "ttit ja divvut juo \xC4""\x8D""\xC3""\xA1""llojuvvon artihkkaliid dahje "
  "\xC4""\x8D""\xC3""\xA1""lligoahttit aibbas o\xC4""\x91""\xC4""\x91""a ar"
// 1252
  " nimmt das Ch\xE2""teau ein, zu dem sie geh\xF6""ren. Im Jahr 2002 wurde"
  "n auf gut 120.000 Hektar Anbaufl\xE4""che insgesamt 5,74 Millionen Hekto"
  "liter Qualit\xE4""tswein erzeugt. mehr Fr\xFC""here Artikel des Tages \xB7"""
  " Weitere exzellente Artikel Was geschah am 13. Oktober? 1781 \x96"" Jose"
;

const char kTestStrNoUTF8UTF8[] =
   "&quot;If management is an art, then surely Jack Welch has proved "
    "himself a master painter.&quot; - BusinessWeek&lt;p&gt;Boardroom "
    "legend Jack Welch is widely regarded as one of the most effective CEOs "
    "in business history. Welch’s groundbreaking programs—including Six "
    "Sigma and Work-Out—along with his numerous strategies on business "
    "leadership have helped transform GE into the global benchmark for "
    "maximized productivity and labor efficiency. &lt;p&gt;Now, The GE Way "
    "Fieldbook explains how you can implement the same programs that helped "
    "turn GE into a $100 billion juggernaut. Drawing from his unprecedented "
    "access to GE’s top-level corridors of power—including a "
    "never-before-published full-length interview with Jack Welch—veteran "
    "business author Robert Slater packs innovative strategies, easy-to-use "
    "diagnostic exercises, detailed questionnaires, and more into the "
    "most hands-on, applications-oriented book ever written on General "
    "Electric. Only in The GE Way Fieldbook will you find:&lt;br&gt; "
    "*&quot;The Boca Raton Speeches&quot;—Never-before-seen excerpts "
    "taken from Jack Welch’s internal speeches to GE employees &lt;br&gt; "
    "*More than 100 exercises, overheads, and exhibits from the files "
    "of Jack Welch and GE &lt;br&gt; *The most complete treatment of "
    "GE’s Six Sigma program ever published&lt;br&gt; *Step-by-step "
    "action plans that are blueprints for implementing Six Sigma and "
    "Work-Out—and creating the boundaryless organization&lt;p&gt;The "
    "fieldbook has become one of today’s most popular, effective teaching "
    "tools—but never before has one focused on the inner workings and "
    "strategies of a specific company. Let The GE Way Fieldbook give you "
    "an inside look at the stunningly successful Jack Welch era at GE, "
    "provide the techniques and tools you need to focus every worker in "
    "your organization on progress and growth, and outline a strategic "
    "roadmap for implementing GE’s business practices—and removing "
    "the boundaries to success—within your own organization.";

const char kUTF16LEChomsky[] =
// generated by stringify.cc
  """\xff""""\xfe""<""\x00""h""\x00""t""\x00""m""\x00""l""\x00"" ""\x00"""
  "x""\x00""m""\x00""l""\x00""n""\x00""s""\x00"":""\x00""v""\x00""="
  """\x00""""\x22""""\x00""u""\x00""r""\x00""n""\x00"":""\x00""s""\x00""c"
  """\x00""h""\x00""e""\x00""m""\x00""a""\x00""s""\x00""-""\x00""m"
  """\x00""i""\x00""c""\x00""r""\x00""o""\x00""s""\x00""o""\x00""f"
  """\x00""t""\x00""-""\x00""c""\x00""o""\x00""m""\x00"":""\x00""v"
  """\x00""m""\x00""l""\x00""""\x22""""\x00""""\x0a""""\x00""x""\x00""m"
  """\x00""l""\x00""n""\x00""s""\x00"":""\x00""o""\x00""=""\x00""""\x22"""
  """\x00""u""\x00""r""\x00""n""\x00"":""\x00""s""\x00""c""\x00""h"
  """\x00""e""\x00""m""\x00""a""\x00""s""\x00""-""\x00""m""\x00""i"
  """\x00""c""\x00""r""\x00""o""\x00""s""\x00""o""\x00""f""\x00""t"
  """\x00""-""\x00""c""\x00""o""\x00""m""\x00"":""\x00""o""\x00""f"
  """\x00""f""\x00""i""\x00""c""\x00""e""\x00"":""\x00""o""\x00""f"
  """\x00""f""\x00""i""\x00""c""\x00""e""\x00""""\x22""""\x00""""\x0a"""
  """\x00""x""\x00""m""\x00""l""\x00""n""\x00""s""\x00"":""\x00""w"
  """\x00""=""\x00""""\x22""""\x00""u""\x00""r""\x00""n""\x00"":""\x00""s"
  """\x00""c""\x00""h""\x00""e""\x00""m""\x00""a""\x00""s""\x00""-"
  """\x00""m""\x00""i""\x00""c""\x00""r""\x00""o""\x00""s""\x00""o"
  """\x00""f""\x00""t""\x00""-""\x00""c""\x00""o""\x00""m""\x00"":"
  """\x00""o""\x00""f""\x00""f""\x00""i""\x00""c""\x00""e""\x00"":"
  """\x00""w""\x00""o""\x00""r""\x00""d""\x00""""\x22""""\x00""""\x0a"""
  """\x00""x""\x00""m""\x00""l""\x00""n""\x00""s""\x00"":""\x00""m"
  """\x00""=""\x00""""\x22""""\x00""h""\x00""t""\x00""t""\x00""p""\x00"":"
  """\x00""/""\x00""/""\x00""s""\x00""c""\x00""h""\x00""e""\x00""m"
  """\x00""a""\x00""s""\x00"".""\x00""m""\x00""i""\x00""c""\x00""r"
  """\x00""o""\x00""s""\x00""o""\x00""f""\x00""t""\x00"".""\x00""c"
  """\x00""o""\x00""m""\x00""/""\x00""o""\x00""f""\x00""f""\x00""i"
  """\x00""c""\x00""e""\x00""/""\x00""2""\x00""0""\x00""0""\x00""4"
  """\x00""/""\x00""1""\x00""2""\x00""/""\x00""o""\x00""m""\x00""m"
  """\x00""l""\x00""""\x22""""\x00""""\x0a""""\x00""x""\x00""m""\x00""l"
  """\x00""n""\x00""s""\x00"":""\x00""m""\x00""v""\x00""=""\x00""""\x22"""
  """\x00""h""\x00""t""\x00""t""\x00""p""\x00"":""\x00""/""\x00""/"
  """\x00""m""\x00""a""\x00""c""\x00""V""\x00""m""\x00""l""\x00""S"
  """\x00""c""\x00""h""\x00""e""\x00""m""\x00""a""\x00""U""\x00""r"
  """\x00""i""\x00""""\x22""""\x00"" ""\x00""x""\x00""m""\x00""l""\x00""n"
  """\x00""s""\x00""=""\x00""""\x22""""\x00""h""\x00""t""\x00""t""\x00""p"
  """\x00"":""\x00""/""\x00""/""\x00""w""\x00""w""\x00""w""\x00""."
  """\x00""w""\x00""3""\x00"".""\x00""o""\x00""r""\x00""g""\x00""/"
  """\x00""T""\x00""R""\x00""/""\x00""R""\x00""E""\x00""C""\x00""-"
  """\x00""h""\x00""t""\x00""m""\x00""l""\x00""4""\x00""0""\x00""""\x22"""
  """\x00"">""\x00""""\x0a""""\x00""""\x0a""""\x00""<""\x00""h""\x00""e"
  """\x00""a""\x00""d""\x00"">""\x00""""\x0a""""\x00""<""\x00""m""\x00""e"
  """\x00""t""\x00""a""\x00"" ""\x00""n""\x00""a""\x00""m""\x00""e"
  """\x00""=""\x00""T""\x00""i""\x00""t""\x00""l""\x00""e""\x00"" "
  """\x00""c""\x00""o""\x00""n""\x00""t""\x00""e""\x00""n""\x00""t"
  """\x00""=""\x00""""\x22""""\x00""P""\x00""a""\x00""r""\x00""t""\x00""n"
  """\x00""e""\x00""r""\x00""s""\x00"" ""\x00""i""\x00""n""\x00"" "
  """\x00""H""\x00""a""\x00""t""\x00""e""\x00"":""\x00"" ""\x00""C"
  """\x00""h""\x00""o""\x00""m""\x00""s""\x00""k""\x00""y""\x00"" "
  """\x00""a""\x00""n""\x00""d""\x00"" ""\x00""t""\x00""h""\x00""e"
  """\x00"" ""\x00""H""\x00""o""\x00""l""\x00""o""\x00""c""\x00""a"
  """\x00""u""\x00""s""\x00""t""\x00"" ""\x00""D""\x00""e""\x00""n"
  """\x00""i""\x00""e""\x00""r""\x00""s""\x00""""\x22""""\x00"">""\x00"""
  """\x0a""""\x00""<""\x00""m""\x00""e""\x00""t""\x00""a""\x00"" ""\x00"""
  "n""\x00""a""\x00""m""\x00""e""\x00""=""\x00""K""\x00""e""\x00""y"
  """\x00""w""\x00""o""\x00""r""\x00""d""\x00""s""\x00"" ""\x00""c"
  """\x00""-""\x00""-""\x00""[""\x00""i""\x00""f""\x00"" ""\x00""!"
  """\x00""m""\x00""s""\x00""o""\x00""]""\x00"">""\x00""""\x0a""""\x00""<"
  """\x00""s""\x00""t""\x00""y""\x00""l""\x00""e""\x00"">""\x00""""\x0a"""
  """\x00""v""\x00""""\x5c""""\x00"":""\x00""*""\x00"" ""\x00""{""\x00""b"
  """\x00""e""\x00""h""\x00""a""\x00""v""\x00""i""\x00""o""\x00""r"
  """\x00"":""\x00""u""\x00""r""\x00""l""\x00""(""\x00""#""\x00""d"
  """\x00""e""\x00""f""\x00""a""\x00""u""\x00""l""\x00""t""\x00""#"
  """\x00""V""\x00""M""\x00""L""\x00"")""\x00"";""\x00""}""\x00""""\x0a"""
  """\x00""o""\x00""""\x5c""""\x00"":""\x00""*""\x00"" ""\x00""{""\x00""b"
  """\x00""e""\x00""h""\x00""a""\x00""v""\x00""i""\x00""o""\x00""r"
  """\x00"":""\x00""u""\x00""r""\x00""l""\x00""(""\x00""#""\x00""d"
  """\x00""e""\x00""f""\x00""a""\x00""u""\x00""l""\x00""t""\x00""#";

const char kUTF16LEFltrs[] =
  """\xff""""\xfe""<""\x00""h""\x00""t""\x00""m""\x00""l""\x00"" ""\x00"""
  "x""\x00""m""\x00""l""\x00""n""\x00""s""\x00"":""\x00""v""\x00""="
  """\x00""""\x22""""\x00""u""\x00""r""\x00""n""\x00"":""\x00""s""\x00""c"
  """\x00""h""\x00""e""\x00""m""\x00""a""\x00""s""\x00""-""\x00""m"
  """\x00""i""\x00""c""\x00""r""\x00""o""\x00""s""\x00""o""\x00""f"
  """\x00""t""\x00""-""\x00""c""\x00""o""\x00""m""\x00"":""\x00""v"
  """\x00""m""\x00""l""\x00""""\x22""""\x00""""\x0d""""\x00""""\x0a"""
  """\x00""x""\x00""m""\x00""l""\x00""n""\x00""s""\x00"":""\x00""o"
  """\x00""=""\x00""""\x22""""\x00""u""\x00""r""\x00""n""\x00"":""\x00""s"
  """\x00""c""\x00""h""\x00""e""\x00""m""\x00""a""\x00""s""\x00""-"
  """\x00""m""\x00""i""\x00""c""\x00""r""\x00""o""\x00""s""\x00""o"
  """\x00""f""\x00""t""\x00""-""\x00""c""\x00""o""\x00""m""\x00"":"
  """\x00""o""\x00""f""\x00""f""\x00""i""\x00""c""\x00""e""\x00"":"
  """\x00""o""\x00""f""\x00""f""\x00""i""\x00""c""\x00""e""\x00""""\x22"""
  """\x00""""\x0d""""\x00""""\x0a""""\x00""x""\x00""m""\x00""l""\x00""n"
  """\x00""s""\x00"":""\x00""w""\x00""=""\x00""""\x22""""\x00""u""\x00""r"
  """\x00""n""\x00"":""\x00""s""\x00""c""\x00""h""\x00""e""\x00""m"
  """\x00""a""\x00""s""\x00""-""\x00""m""\x00""i""\x00""c""\x00""r"
  """\x00""o""\x00""s""\x00""o""\x00""f""\x00""t""\x00""-""\x00""c"
  """\x00""o""\x00""m""\x00"":""\x00""o""\x00""f""\x00""f""\x00""i"
  """\x00""c""\x00""e""\x00"":""\x00""w""\x00""o""\x00""r""\x00""d"
  """\x00""""\x22""""\x00""""\x0d""""\x00""""\x0a""""\x00""x""\x00""m"
  """\x00""l""\x00""n""\x00""s""\x00"":""\x00""m""\x00""=""\x00""""\x22"""
  """\x00""h""\x00""t""\x00""t""\x00""p""\x00"":""\x00""/""\x00""/"
  """\x00""s""\x00""c""\x00""h""\x00""e""\x00""m""\x00""a""\x00""s"
  """\x00"".""\x00""m""\x00""i""\x00""c""\x00""r""\x00""o""\x00""s"
  """\x00""o""\x00""f""\x00""t""\x00"".""\x00""c""\x00""o""\x00""m"
  """\x00""/""\x00""o""\x00""f""\x00""f""\x00""i""\x00""c""\x00""e"
  """\x00""/""\x00""2""\x00""0""\x00""0""\x00""4""\x00""/""\x00""1"
  """\x00""2""\x00""/""\x00""o""\x00""m""\x00""m""\x00""l""\x00""""\x22"""
  """\x00""""\x0d""""\x00""""\x0a""""\x00""x""\x00""m""\x00""l""\x00""n"
  """\x00""s""\x00""=""\x00""""\x22""""\x00""h""\x00""t""\x00""t""\x00""p"
  """\x00"":""\x00""/""\x00""/""\x00""w""\x00""w""\x00""w""\x00""."
  """\x00""w""\x00""3""\x00"".""\x00""o""\x00""r""\x00""g""\x00""/"
  """\x00""T""\x00""R""\x00""/""\x00""R""\x00""E""\x00""C""\x00""-"
  """\x00""h""\x00""t""\x00""m""\x00""l""\x00""4""\x00""0""\x00""""\x22"""
  """\x00"" ""\x00""x""\x00""m""\x00""l""\x00"":""\x00""l""\x00""a"
  """\x00""n""\x00""g""\x00""=""\x00""""\x22""""\x00""e""\x00""n""\x00"""
  """\x22""""\x00"">""\x00""""\x0d""""\x00""""\x0a""""\x00""""\x0d"""
  """\x00""""\x0a""""\x00""<""\x00""h""\x00""e""\x00""a""\x00""d""\x00"">"
  """\x00""""\x0d""""\x00""""\x0a""""\x00""<""\x00""m""\x00""e""\x00""t"
  """\x00""a""\x00"" ""\x00""h""\x00""t""\x00""t""\x00""p""\x00""-"
  """\x00""e""\x00""q""\x00""u""\x00""i""\x00""v""\x00""=""\x00""C"
  """\x00""o""\x00""n""\x00""t""\x00""e""\x00""n""\x00""t""\x00""-"
  """\x00""T""\x00""y""\x00""p""\x00""e""\x00"" ""\x00""c""\x00""o"
  """\x00""n""\x00""t""\x00""e""\x00""n""\x00""t""\x00""=""\x00""""\x22"""
  """\x00""t""\x00""e""\x00""x""\x00""t""\x00""/""\x00""h""\x00""t"
  """\x00""m""\x00""l""\x00"";""\x00"" ""\x00""c""\x00""h""\x00""a"
  """\x00""r""\x00""s""\x00""e""\x00""t""\x00""=""\x00""u""\x00""n"
  """\x00""i""\x00""c""\x00""o""\x00""d""\x00""e""\x00""""\x22""""\x00"">"
  """\x00""""\x0d""""\x00""""\x0a""""\x00""<""\x00""m""\x00""e""\x00""t"
  """\x00""a""\x00"" ""\x00""n""\x00""a""\x00""m""\x00""e""\x00""="
  """\x00""P""\x00""r""\x00""o""\x00""g""\x00""I""\x00""d""\x00"" "
  """\x00""c""\x00""o""\x00""n""\x00""t""\x00""e""\x00""n""\x00""t"
  """\x00""=""\x00""W""\x00""o""\x00""r""\x00""d""\x00"".""\x00""D";

namespace {

class CompactEncDetTest : public testing::Test {
 protected:
  // Objects declared here can be used by all tests in the test case for Foo.

  // Convert str to UTF-8, returning as function result
  // Name is just for debug output
  Encoding TestCompactEncDetWithURL(const char* str, int len,
                                    const char* url, const char* name,
                                    bool* is_reliable) {
    int bytes_consumed;

    if (FLAGS_enc_detect_detail) {
      fprintf(stderr, "(Unit test %s) start-detail\n", PsStr(name).c_str());
    }

    Encoding enc = CompactEncDet::DetectEncoding(
        str, len,
        url,                                // url hint
        NULL,                               // http hint
        NULL,                               // meta hint
        UNKNOWN_ENCODING,                   // enc hint
        UNKNOWN_LANGUAGE,  // lang hint
        CompactEncDet::WEB_CORPUS,
        false,  // Include 7-bit encodings
        &bytes_consumed, is_reliable);

    if (FLAGS_enc_detect_detail) {
      fprintf(stderr, "() end-detail\n\n");
    }
    return enc;
  }

  // Name is just for debug output
  // Same as above with is_reliable supplied and then ignored
  Encoding TestCompactEncDetWithURL(const char* str, int len,
                                    const char* url, const char* name) {
    bool is_reliable;
    return TestCompactEncDetWithURL(str, len, NULL, name, &is_reliable);
  }

  // Name is just for debug output
  Encoding TestCompactEncDetWithReliable(const char* str, int len,
                                         const char* name, bool* is_reliable) {
    return TestCompactEncDetWithURL(str, len, NULL, name, is_reliable);
  }

  // Name is just for debug output
  Encoding TestCompactEncDet(const char* str, int len, const char* name) {
    return TestCompactEncDetWithURL(str, len, NULL, name);
  }

  // Name is just for debug output
  Encoding TestCompactEncDet(const char* str, const char* name) {
    return TestCompactEncDetWithURL(str, strlen(str), NULL, name);
  }
  // Every string that is detected as UTF8UTF8 when --ced_allow_utf8utf=true
  // should be detected as some other encoding when --ced_allow_utf8utf8=false.
  void TestUTF8UTF8(const char* str, Encoding other, const char* name) {
    Encoding encoding;
#if defined(HTML5_MODE)
    encoding = ASCII_7BIT;
#else
    encoding = UTF8UTF8;
#endif
    {
      VarSetter<bool> utf8utf8(&FLAGS_ced_allow_utf8utf8, true);
      EXPECT_EQ(encoding, TestCompactEncDet(str, name));
    }
    {
      VarSetter<bool> noutf8utf8(&FLAGS_ced_allow_utf8utf8, false);
      EXPECT_EQ(other, TestCompactEncDet(str, name));
    }
  }
};    // end class CompactEncDetTest


TEST_F(CompactEncDetTest, ZeroLength) {
  // The spec (.h file) says that DetectEncoding returns ASCII
  // (ISO_8859_1) when the text_length is 0. It doesn't require that
  // the text actually *have* a length of 0. In particular, we want to
  // allow the case where the text is actually NULL.  This shows up
  // in a unit test, caribou/medley/internal/parser_impl_test, which
  // uses StringPiece() and calls the .data() method, which returns
  // NULL, and the .size() method, which returns 0. Without the test
  // for for length == 0, we segfaulted.
  //
  // For completeness, test all three cases:
  // Empty string
  EXPECT_EQ(ISO_8859_1, TestCompactEncDet("", 0, "ISO_8859_1"));
  // Non-empty string
  EXPECT_EQ(ISO_8859_1, TestCompactEncDet("abcdef", 0, "ISO_8859_1"));
  // Null string
  EXPECT_EQ(ISO_8859_1, TestCompactEncDet(NULL, 0, "ISO_8859_1"));
}

// NOTE: strlen will not work for UTF-16 and UTF-32 strings with embedded NUL

TEST_F(CompactEncDetTest, EasyTests) {
  // One-byte
  EXPECT_EQ(ISO_8859_1,
            TestCompactEncDet(kTeststr00, strlen(kTeststr00), "ISO_8859_1"));
  // By design, Latin4 is the most sensitive test of the bunch,
  // most likely to fail with minor changes in the detector probabilities
  EXPECT_EQ(ISO_8859_4,
            TestCompactEncDet(kTeststr03, strlen(kTeststr03), "ISO_8859_4"));
  EXPECT_EQ(ISO_8859_5,
            TestCompactEncDet(kTeststr04, strlen(kTeststr04), "ISO_8859_5"));
  EXPECT_EQ(ISO_8859_6,
            TestCompactEncDet(kTeststr05, strlen(kTeststr05), "ISO_8859_6"));
  // This text is in fact CP1253, previously mis-detected as Greek.
  // 0xA2 = U+0386 Alpha with Tonos
  EXPECT_EQ(MSFT_CP1253,
            TestCompactEncDet(kTeststr06, strlen(kTeststr06), "ISO_8859_7 => 1253"));
  EXPECT_EQ(MSFT_CP1255,    // Logical (modern browsers) order
            TestCompactEncDet(kTeststr07, strlen(kTeststr07), "ISO_8859_8_I"));
  EXPECT_EQ(ISO_8859_8,     // Visual (original 1994 browsers) order
            TestCompactEncDet(kTeststr07v, strlen(kTeststr07v), "ISO_8859_8"));
  EXPECT_EQ(ISO_8859_9,
            TestCompactEncDet(kTeststr08, strlen(kTeststr08), "ISO_8859_9"));

  // Two-byte Japanese
  EXPECT_EQ(JAPANESE_SHIFT_JIS,
            TestCompactEncDet(kTeststr11, strlen(kTeststr11), "JAPANESE_SHIFT_JIS"));

  // Two-byte Chinese
  EXPECT_EQ(CHINESE_GB,
            TestCompactEncDet(kTeststr14, strlen(kTeststr14), "CHINESE_GB"));
  EXPECT_EQ(GB18030,
            TestCompactEncDet(kTeststr46, strlen(kTeststr46), "GB18030"));

  // Two-byte Chinese-T
  EXPECT_EQ(CHINESE_BIG5,
            TestCompactEncDet(kTeststr13, strlen(kTeststr13), "CHINESE_BIG5"));

  // Two-byte Korean
  //KOREAN_EUC_KR,    // 16: Teragram KSC

  // UTF-8
  EXPECT_EQ(UTF8,
            TestCompactEncDet(kTeststr22, strlen(kTeststr22), "UTF8"));

  // More one-byte
  EXPECT_EQ(ISO_8859_11,
            TestCompactEncDet(kTeststr33, strlen(kTeststr33), "ISO_8859_11"));
  EXPECT_EQ(RUSSIAN_KOI8_R,
            TestCompactEncDet(kTeststr25, strlen(kTeststr25), "RUSSIAN_KOI8_R"));
  EXPECT_EQ(RUSSIAN_KOI8_RU,
            TestCompactEncDet(kTeststr28, strlen(kTeststr28), "RUSSIAN_KOI8_RU"));
  EXPECT_EQ(RUSSIAN_CP1251,
            TestCompactEncDet(kTeststr26, strlen(kTeststr26), "RUSSIAN_CP1251"));
  EXPECT_EQ(MSFT_CP1252,
            TestCompactEncDet(kTeststr27, strlen(kTeststr27), "MSFT_CP1252"));
  EXPECT_EQ(MSFT_CP1250,
            TestCompactEncDet(kTeststr29, strlen(kTeststr29), "MSFT_CP1250"));
  EXPECT_EQ(MSFT_CP1257,
            TestCompactEncDet(kTeststr32, strlen(kTeststr32), "MSFT_CP1257"));
  EXPECT_EQ(MSFT_CP1256,
            TestCompactEncDet(kTeststr35, strlen(kTeststr35), "MSFT_CP1256"));
  EXPECT_EQ(RUSSIAN_CP866,
            TestCompactEncDet(kTeststr42, strlen(kTeststr42), "RUSSIAN_CP866"));
  EXPECT_EQ(MACINTOSH_ROMAN,
            TestCompactEncDet(kTeststr53, strlen(kTeststr53), "MACINTOSH_ROMAN"));

  // Ascii7 and seven-bit CJK and Unicode encodings
  EXPECT_EQ(JAPANESE_JIS,
            TestCompactEncDet(kTeststr12, strlen(kTeststr12), "JAPANESE_JIS"));
#if defined(HTML5_MODE)
  EXPECT_EQ(ASCII_7BIT,
            TestCompactEncDet(kTeststr44, strlen(kTeststr44), "ASCII_7BIT"));
  EXPECT_EQ(ASCII_7BIT,
            TestCompactEncDet(kTeststr48, strlen(kTeststr48), "ASCII_7BIT"));
  EXPECT_EQ(ASCII_7BIT,
            TestCompactEncDet(kTeststr54, strlen(kTeststr54), "ASCII_7BIT"));
  EXPECT_EQ(ASCII_7BIT,
            TestCompactEncDet(kTeststr62, strlen(kTeststr62), "ASCII_7BIT"));
#else
  EXPECT_EQ(ISO_2022_KR,
            TestCompactEncDet(kTeststr44, strlen(kTeststr44), "ISO_2022_KR"));
  EXPECT_EQ(ISO_2022_CN,
            TestCompactEncDet(kTeststr48, strlen(kTeststr48), "ISO_2022_CN"));
  EXPECT_EQ(UTF7,
            TestCompactEncDet(kTeststr54, strlen(kTeststr54), "UTF7"));
  EXPECT_EQ(HZ_GB_2312,
            TestCompactEncDet(kTeststr62, strlen(kTeststr62), "HZ_GB_2312"));
#endif

  TestUTF8UTF8(kTeststr63, UTF8, "UTF8UTF8");

  // Unicode and other embedded NUL bytes
  Encoding encoding_UTF16LE;
  Encoding encoding_UTF16BE;
#if defined(HTML5_MODE)
    encoding_UTF16LE = ASCII_7BIT;
    encoding_UTF16BE = ASCII_7BIT;
#else
    encoding_UTF16LE = UTF16LE;
    encoding_UTF16BE = UTF16BE;
#endif

  EXPECT_EQ(encoding_UTF16LE,
            TestCompactEncDet(kTeststr57, sizeof(kTeststr57), "UTF16LE"));
  EXPECT_EQ(encoding_UTF16LE,
            TestCompactEncDet(kUTF16LEChomsky,
                              sizeof(kUTF16LEChomsky),
                              "UTF16LE"));
  EXPECT_EQ(encoding_UTF16LE,
            TestCompactEncDet(kUTF16LEFltrs,
                              sizeof(kUTF16LEFltrs),
                              "UTF16LE"));
  EXPECT_EQ(encoding_UTF16BE,
            TestCompactEncDet(kTeststr58, sizeof(kTeststr58), "UTF16BE"));
  EXPECT_EQ(UTF32LE,
            TestCompactEncDet(kTeststr59, sizeof(kTeststr59), "UTF32LE"));
  EXPECT_EQ(UTF32BE,
            TestCompactEncDet(kTeststr60, sizeof(kTeststr60), "UTF32BE"));
#if defined(HTML5_MODE)
  EXPECT_EQ(ASCII_7BIT,
            TestCompactEncDet(kTeststr61, sizeof(kTeststr61), "BINARYENC"));
#else
  EXPECT_EQ(BINARYENC,
            TestCompactEncDet(kTeststr61, sizeof(kTeststr61), "ASCII_7BIT"));
#endif
  // Indic. Detection requires a full URL hint
  // dsites 2007.10.10 NO LONGER DETECTED. Will return some LatinX result
  EXPECT_EQ(ISO_8859_10, TestCompactEncDetWithURL(kTeststr52,
                                                  strlen(kTeststr52),
                                                  kTestUrl52, "Indic JAGRAN"));
  EXPECT_EQ(UTF8, TestCompactEncDetWithURL(kTeststr52b, strlen(kTeststr52b),
                                           kTestUrl52b, "Indic UTF8"));

  // dsites 2011.11.07 remove mixed encoding UTF8CP1252 -- it was a bad idea
  // Mixed UTF-8 and CP1252 and UTF8UTF8
  //// TestUTF8UTF8(kTeststr99, MSFT_CP1252, "UTF8CP1252");

}

TEST_F(CompactEncDetTest, LangHintTests) {
  // Make sure the Tier1 language hints are plausible
  EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfLangHint("English"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfLangHint("Spanish"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfLangHint("German"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfLangHint("French"));
  EXPECT_EQ(JAPANESE_SHIFT_JIS, CompactEncDet::TopEncodingOfLangHint("Japanese"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfLangHint("Portuguese_B"));
  EXPECT_EQ(CHINESE_GB, CompactEncDet::TopEncodingOfLangHint("Chinese"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfLangHint("Italian"));
  EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfLangHint("Dutch"));
  EXPECT_EQ(ISO_8859_9, CompactEncDet::TopEncodingOfLangHint("Turkish"));
  EXPECT_EQ(ISO_8859_2, CompactEncDet::TopEncodingOfLangHint("Polish"));
  EXPECT_EQ(ISO_8859_11, CompactEncDet::TopEncodingOfLangHint("Thai"));
  EXPECT_EQ(CHINESE_BIG5, CompactEncDet::TopEncodingOfLangHint("ChineseT"));
  EXPECT_EQ(MSFT_CP1256, CompactEncDet::TopEncodingOfLangHint("Arabic"));
  EXPECT_EQ(RUSSIAN_CP1251, CompactEncDet::TopEncodingOfLangHint("Russian"));
  EXPECT_EQ(KOREAN_EUC_KR, CompactEncDet::TopEncodingOfLangHint("Korean"));
}


TEST_F(CompactEncDetTest, TLDHintTests) {
  // Make sure the Tier1 domain hints are plausible
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfTLDHint("de"));
  EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfTLDHint("au"));
  EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfTLDHint("ca"));
  EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfTLDHint("in"));
  EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfTLDHint("uk"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfTLDHint("es"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfTLDHint("mx"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfTLDHint("fr"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfTLDHint("it"));
  EXPECT_EQ(JAPANESE_SHIFT_JIS, CompactEncDet::TopEncodingOfTLDHint("jp"));
  EXPECT_EQ(KOREAN_EUC_KR, CompactEncDet::TopEncodingOfTLDHint("kr"));
  EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfTLDHint("nl"));
  EXPECT_EQ(ISO_8859_2, CompactEncDet::TopEncodingOfTLDHint("pl"));
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfTLDHint("br"));
  EXPECT_EQ(RUSSIAN_CP1251, CompactEncDet::TopEncodingOfTLDHint("ru"));
  EXPECT_EQ(ISO_8859_11, CompactEncDet::TopEncodingOfTLDHint("th"));
  EXPECT_EQ(ISO_8859_9, CompactEncDet::TopEncodingOfTLDHint("tr"));
  EXPECT_EQ(CHINESE_GB, CompactEncDet::TopEncodingOfTLDHint("cn"));
  EXPECT_EQ(CHINESE_BIG5, CompactEncDet::TopEncodingOfTLDHint("tw"));
}



TEST_F(CompactEncDetTest, CharsetHintTests) {
  // Make sure official encoding names give that encoding, or sub/superset
  EXPECT_EQ(ISO_8859_1, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-1"));   // ASCII_7BIT ok
  EXPECT_EQ(ISO_8859_2, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-2"));
  EXPECT_EQ(ISO_8859_3, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-3"));   // ASCII_7BIT bug
  EXPECT_EQ(ISO_8859_4, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-4"));   // MSFT_CP1257 bug
  EXPECT_EQ(ISO_8859_5, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-5"));
  EXPECT_EQ(ISO_8859_6, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-6"));   // ASCII_7BIT bug
  EXPECT_EQ(ISO_8859_7, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-7"));
  EXPECT_EQ(ISO_8859_8, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-8"));
  EXPECT_EQ(ISO_8859_9, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-9"));
  EXPECT_EQ(ISO_8859_10, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-10")); // ASCII_7BIT  bug

  EXPECT_EQ(JAPANESE_EUC_JP, CompactEncDet::TopEncodingOfCharsetHint("EUC-JP"));
  EXPECT_EQ(JAPANESE_SHIFT_JIS, CompactEncDet::TopEncodingOfCharsetHint("Shift-JIS"));
  EXPECT_EQ(JAPANESE_JIS, CompactEncDet::TopEncodingOfCharsetHint("JIS"));        // JAPANESE_SHIFT_JIS ok
  EXPECT_EQ(CHINESE_BIG5, CompactEncDet::TopEncodingOfCharsetHint("Big5"));
  EXPECT_EQ(CHINESE_GB, CompactEncDet::TopEncodingOfCharsetHint("GB"));
  EXPECT_EQ(CHINESE_EUC_CN, CompactEncDet::TopEncodingOfCharsetHint("EUC-CN"));   // CHINESE_GB ok
  EXPECT_EQ(KOREAN_EUC_KR, CompactEncDet::TopEncodingOfCharsetHint("EUC-KR"));
  // Never return UNICODE; return UTF-16BE or lE instead
  // EXPECT_EQ(UNICODE, CompactEncDet::TopEncodingOfCharsetHint("Unicode"));         // ASCII_7BIT
  EXPECT_EQ(CHINESE_EUC_DEC, CompactEncDet::TopEncodingOfCharsetHint("EUC-DEC")); // UNKNOWN_ENCODING ok
  EXPECT_EQ(CHINESE_CNS, CompactEncDet::TopEncodingOfCharsetHint("CNS"));         // UNKNOWN_ENCODING ok

  EXPECT_EQ(CHINESE_BIG5_CP950, CompactEncDet::TopEncodingOfCharsetHint("windows-950"));// CHINESE_BIG5 ok
  EXPECT_EQ(JAPANESE_CP932, CompactEncDet::TopEncodingOfCharsetHint("windows-932"));    // JAPANESE_EUC_JP ok
  EXPECT_EQ(UTF8, CompactEncDet::TopEncodingOfCharsetHint("UTF8"));
  // Never return "unknown"
  // EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfCharsetHint("unknown"));// JAPANESE_SHIFT_JIS bug
  EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfCharsetHint("ASCII"));        // UNKNOWN_ENCODING bug
  EXPECT_EQ(ASCII_7BIT, CompactEncDet::TopEncodingOfCharsetHint("US-ASCII"));     // UNKNOWN_ENCODING bug
  EXPECT_EQ(RUSSIAN_KOI8_R, CompactEncDet::TopEncodingOfCharsetHint("KOI8R"));
  EXPECT_EQ(RUSSIAN_CP1251, CompactEncDet::TopEncodingOfCharsetHint("windows-1251"));
  EXPECT_EQ(MSFT_CP1252, CompactEncDet::TopEncodingOfCharsetHint("windows-1252"));      // ASCII_7BIT ok
  EXPECT_EQ(RUSSIAN_KOI8_RU, CompactEncDet::TopEncodingOfCharsetHint("KOI8U"));
  EXPECT_EQ(MSFT_CP1250, CompactEncDet::TopEncodingOfCharsetHint("windows-1250"));

  EXPECT_EQ(ISO_8859_15, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-15"));
  EXPECT_EQ(MSFT_CP1254, CompactEncDet::TopEncodingOfCharsetHint("windows-1254"));      // ISO_8859_9 ok
  EXPECT_EQ(MSFT_CP1257, CompactEncDet::TopEncodingOfCharsetHint("windows-1257"));
  EXPECT_EQ(ISO_8859_11, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-11"));
  EXPECT_EQ(MSFT_CP874, CompactEncDet::TopEncodingOfCharsetHint("win-874"));        // UNKNOWN_ENCODING bug
  EXPECT_EQ(MSFT_CP1256, CompactEncDet::TopEncodingOfCharsetHint("windows-1256"));
  EXPECT_EQ(MSFT_CP1255, CompactEncDet::TopEncodingOfCharsetHint("windows-1255"));  // UNKNOWN_ENCODING
  // Always map ISO-8859-8-I to MSFT_CP1255
  EXPECT_EQ(MSFT_CP1255, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-8-I")); // MSFT_CP1255
  // Always map Visual Hebrew to ISO_8859_8
  EXPECT_EQ(ISO_8859_8, CompactEncDet::TopEncodingOfCharsetHint("Visual"));       // ISO_8859_8 ok
  EXPECT_EQ(CZECH_CP852, CompactEncDet::TopEncodingOfCharsetHint("windows-852"));

  EXPECT_EQ(CZECH_CSN_369103, CompactEncDet::TopEncodingOfCharsetHint("CSN-369103"));
  EXPECT_EQ(MSFT_CP1253, CompactEncDet::TopEncodingOfCharsetHint("windows-1253"));      // UTF8
  EXPECT_EQ(RUSSIAN_CP866, CompactEncDet::TopEncodingOfCharsetHint("windows-866"));
  EXPECT_EQ(ISO_8859_13, CompactEncDet::TopEncodingOfCharsetHint("ISO-8859-13")); // MSFT_CP1257 ok
  EXPECT_EQ(ISO_2022_KR, CompactEncDet::TopEncodingOfCharsetHint("ISO-2022-KR")); // KOREAN_EUC_KR ok
  EXPECT_EQ(GBK, CompactEncDet::TopEncodingOfCharsetHint("GBK"));                 // CHINESE_GB ok
  EXPECT_EQ(GB18030, CompactEncDet::TopEncodingOfCharsetHint("GB18030"));         // CHINESE_GB  ok
  EXPECT_EQ(BIG5_HKSCS, CompactEncDet::TopEncodingOfCharsetHint("BIG5-HKSCS"));   // CHINESE_BIG5 ok
  EXPECT_EQ(ISO_2022_CN, CompactEncDet::TopEncodingOfCharsetHint("ISO-2022-CN")); // CHINESE_GB  ok
  EXPECT_EQ(TSCII, CompactEncDet::TopEncodingOfCharsetHint("TSCII"));             // UNKNOWN_ENCODING

  EXPECT_EQ(TAMIL_MONO, CompactEncDet::TopEncodingOfCharsetHint("TAM"));          // UNKNOWN_ENCODING
  EXPECT_EQ(TAMIL_BI, CompactEncDet::TopEncodingOfCharsetHint("TAB"));            // UNKNOWN_ENCODING
  EXPECT_EQ(JAGRAN, CompactEncDet::TopEncodingOfCharsetHint("JAGRAN"));           // UNKNOWN_ENCODING
  EXPECT_EQ(MACINTOSH_ROMAN, CompactEncDet::TopEncodingOfCharsetHint("MacRoman"));// ASCII_7BIT
  EXPECT_EQ(UTF7, CompactEncDet::TopEncodingOfCharsetHint("UTF-7"));              // ASCII_7BIT
  EXPECT_EQ(BHASKAR, CompactEncDet::TopEncodingOfCharsetHint("BHASKAR"));         // UNKNOWN_ENCODING
  EXPECT_EQ(HTCHANAKYA, CompactEncDet::TopEncodingOfCharsetHint("HTCHANAKYA"));   // UNKNOWN_ENCODING
  EXPECT_EQ(UTF16BE, CompactEncDet::TopEncodingOfCharsetHint("UTF-16BE"));        // ASCII_7BIT
  EXPECT_EQ(UTF16LE, CompactEncDet::TopEncodingOfCharsetHint("UTF-16LE"));        // UNKNOWN_ENCODING
  EXPECT_EQ(UTF32BE, CompactEncDet::TopEncodingOfCharsetHint("UTF-32BE"));        // UNKNOWN_ENCODING

  EXPECT_EQ(UTF32LE, CompactEncDet::TopEncodingOfCharsetHint("UTF-32LE"));        // UNKNOWN_ENCODING
  EXPECT_EQ(BINARYENC, CompactEncDet::TopEncodingOfCharsetHint("binary"));        // UNKNOWN_ENCODING
  EXPECT_EQ(HZ_GB_2312, CompactEncDet::TopEncodingOfCharsetHint("HZ-GB-2312"));   // GBK
  EXPECT_EQ(UTF8UTF8, CompactEncDet::TopEncodingOfCharsetHint("utf8utf8"));       // UNKNOWN_ENCODING
}

TEST_F(CompactEncDetTest, UTF8UTF8Tests) {
  // Present subset of all 128 possible UTF8-UTF8 double-conversions and
  // make sure they are properly detected.
  TestUTF8UTF8("\xC3\xA2\xE2\x80\x9A\xC2\xAC", UTF8, "UTF8UTF8 80");
  TestUTF8UTF8("\xC3\x82\xC2\x81", UTF8, "UTF8UTF8 81");
  TestUTF8UTF8("\xC3\x86\xE2\x80\x99", UTF8, "UTF8UTF8 83");
  TestUTF8UTF8("\xC3\xA2\xE2\x82\xAC\xE2\x84\xA2", UTF8, "UTF8UTF8 92");
  TestUTF8UTF8("\xC3\x8B\xC5\x93", UTF8, "UTF8UTF8 98");
  TestUTF8UTF8("\xC3\x83\xC6\x92", UTF8, "UTF8UTF8 C3");
  TestUTF8UTF8("\xC3\x83\xCB\x86", UTF8, "UTF8UTF8 C8");
  TestUTF8UTF8("\xC3\x83\xC2\xBF", UTF8, "UTF8UTF8 FF");

  // These are bare-byte extra conversions to UTF-8
  TestUTF8UTF8("\xc3\x85\xc2\x8f:\xc3\x8c\xc2\x95:"
               "\xc3\x85\xc2\x8f:\xc3\x8a\xc2\xbe:"
               "\xc3\x85\xc2\x8f:\xc3\x85\xc2\xad:"
               "\xc3\x8a\xc2\xbb:\xc3\x85\xc2\x8f:"
               "\xc3\xa7\xc2\x92\xc2\xb0",
               UTF8,
               "UTF8UTF8 bytes1");
  TestUTF8UTF8("\x20\x20\xc3\xa7\xc2\x92\xc2\xb0"
               "\xc3\xa5\xc2\xa2\xc2\x83\xc3\xa3"
               "\xc2\x83\xc2\xbb\xc3\xa5\xc2\x85"
               "\xc2\xac\xc3\xa5\xc2\xae\xc2\xb3"
               "\xc3\xa3\xc2\x80\xc2\x80\xc3\xa4"
               "\xc2\xba\xc2\x8b\xc3\xa5\xc2\x85"
               "\xc2\xb8",
               UTF8,
               "UTF8UTF8 bytes2");
}

TEST_F(CompactEncDetTest, NoUTF8UTF8) {
  VarSetter<bool> utf8utf8(&FLAGS_ced_allow_utf8utf8, true);

  if (FLAGS_enc_detect_detail) {
    const char* name = "NoUTF8UTF8";
    fprintf(stderr, "(Unit test %s) start-detail\n", PsStr(name).c_str());
  }

  int bytes_consumed = 0;
  bool confidence = false;
  Encoding encoding = CompactEncDet::DetectEncoding(
      kTestStrNoUTF8UTF8, strlen(kTestStrNoUTF8UTF8),
      NULL,              // No url hint.
      NULL,              // No charset_hint,
      NULL,              // No meta_charset hint.
      UNKNOWN_ENCODING,  // No encoding hint (setting this to UTF8
                         // decreased the detection accuracy although
                         // the content of 'text' is always UTF8
                         // encoded).
      UNKNOWN_LANGUAGE,  // Just like encoding, this too lowers
                                          // the accuracy.
      CompactEncDet::QUERY_CORPUS, true, &bytes_consumed, &confidence);

  if (FLAGS_enc_detect_detail) {
    fprintf(stderr, "() end-detail\n\n");
  }

  EXPECT_EQ(UTF8, encoding);
}

const char kTestShiftJISNoHint[] =
  "\x82\xa0\x82\xaf\x82\xdc\x82\xb5\x82\xc4\x82\xa8\x82\xdf\x82\xc5"
  "\x82\xc6\x82\xa4\x82\xb2\x82\xb4\x82\xa2\x82\xdc\x82\xb7\x82\xb1"
  "\x82\xc6\x82\xb5\x82\xe0\x82\xe6\x82\xeb\x82\xb5\x82\xad\x82\xa8"
  "\x82\xcb\x82\xaa\x82\xa2\x82\xa2\x82\xbd\x82\xb5\x82\xa0\x82\xaf"
  "\x82\xdc\x82\xb5\x82\xc4\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa4"
  "\x82\xb2\x82\xb4\x82\xa2\x82\xdc\x82\xb7\x82\xb1\x82\xc6\x82\xb5"
  "\x82\xe0\x82\xe6\x82\xeb\x82\xb5\x82\xad\x82\xa8\x82\xcb\x82\xaa"
  "\x82\xa2\x82\xa2\x82\xbd\x82\xb5\x82\xa0\x82\xaf\x82\xdc\x82\xb5"
  "\x82\xc4\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa4\x82\xb2\x82\xb4"
  "\x82\xa2\x82\xdc\x82\xb7\x82\xb1\x82\xc6\x82\xb5\x82\xe0\x82\xe6"
  "\x82\xeb\x82\xb5\x82\xad\x82\xa8\x82\xcb\x82\xaa\x82\xa2\x82\xa2"
  "\x82\xbd\x82\xb5\x82\xa0\x82\xaf\x82\xdc\x82\xb5\x82\xc4\x82\xa8"
  "\x82\xdf\x82\xc5\x82\xc6\x82\xa4\x82\xb2\x82\xb4\x82\xa2\x82\xdc"
  "\x82\xb7\x82\xb1\x82\xc6\x82\xb5\x82\xe0\x82\xe6\x82\xeb\x82\xb5"
  "\x82\xad\x82\xa8\x82\xcb\x82\xaa\x82\xa2\x82\xa2\x82\xbd\x82\xb5"
  "\x82\xa0\x82\xaf\x82\xdc\x82\xb5\x82\xc4\x82\xa8\x82\xdf\x82\xc5"
  "\x82\xc6\x82\xa4\x82\xb2\x82\xb4\x82\xa2\x82\xdc\x82\xb7\x82\xb1"
  "\x82\xc6\x82\xb5\x82\xe0\x82\xe6\x82\xeb\x82\xb5\x82\xad\x82\xa8"
  "\x82\xcb\x82\xaa\x82\xa2\x82\xa2\x82\xbd\x82\xb5\x82\xa0\x82\xaf"
  "\x82\xdc\x82\xb5\x82\xc4\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa4"
  "\x82\xb2\x82\xb4\x82\xa2\x82\xdc\x82\xb7\x82\xb1\x82\xc6\x82\xb5"
  "\x82\xe0\x82\xe6\x82\xeb\x82\xb5\x82\xad\x82\xa8\x82\xcb\x82\xaa"
  "\x82\xa2\x82\xa2\x82\xbd\x82\xb5\x82\xa0\x82\xaf\x82\xdc\x82\xb5"
  "\x82\xc4\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa4\x82\xb2\x82\xb4"
  "\x82\xa2\x82\xdc\x82\xb7\x82\xb1\x82\xc6\x82\xb5\x82\xe0\x82\xe6"
  "\x82\xeb\x82\xb5\x82\xad\x82\xa8\x82\xcb\x82\xaa\x82\xa2\x82\xa2"
  "\x82\xbd\x82\xb5\x82\xa0\x82\xaf\x82\xdc\x82\xb5\x82\xc4\x82\xa8"
  "\x82\xdf\x82\xc5\x82\xc6\x82\xa4\x82\xb2\x82\xb4\x82\xa2\x82\xdc"
  "\x82\xb7\x82\xb1\x82\xc6\x82\xb5\x82\xe0\x82\xe6\x82\xeb\x82\xb5"
  "\x82\xad\x82\xa8\x82\xcb\x82\xaa\x82\xa2\x82\xa2\x82\xbd\x82\xb5"
  "\x82\xa0\x82\xaf\x82\xdc\x82\xb5\x82\xc4\x82\xa8\x82\xdf\x82\xc5"
  "\x82\xc6\x82\xa4\x82\xb2\x82\xb4\x82\xa2\x82\xdc\x82\xb7\x82\xb1"
  "\x82\xc6\x82\xb5\x82\xe0\x82\xe6\x82\xeb\x82\xb5\x82\xad\x82\xa8"
  "\x82\xcb\x82\xaa\x82\xa2\x82\xa2\x82\xbd\x82\xb5\x82\xa0\x82\xaf"
  "\x82\xdc\x82\xb5\x82\xc4\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa4"
  "\x82\xb2\x82\xb4\x82\xa2\x82\xdc\x82\xb7\x82\xb1\x82\xc6\x82\xb5"
  "\x82\xe0\x82\xe6\x82\xeb\x82\xb5\x82\xad\x82\xa8\x82\xcb\x82\xaa"
  "\x82\xa2\x82\xa2\x82\xbd\x82\xb5\x82\xa0\x82\xaf\x82\xdc\x82\xb5"
  "\x82\xc4\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa0\x82\xaf\x82\xdc"
  "\x82\xb5\x82\xc4\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa4\x82\xb2"
  "\x82\xb4\x82\xa2\x82\xdc\x82\xb7\x82\xb1\x82\xc6\x82\xb5\x82\xe0"
  "\x82\xe6\x82\xeb\x82\xb5\x82\xad\x82\xa8\x82\xcb\x82\xaa\x82\xa2"
  "\x82\xa2\x82\xbd\x82\xb5\x82\xa0\x82\xaf\x82\xdc\x82\xb5\x82\xc4"
  "\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa0\x82\xaf\x82\xdc\x82\xb5"
  "\x82\xc4\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa4\x82\xb2\x82\xb4"
  "\x82\xa2\x82\xdc\x82\xb7\x82\xb1\x82\xc6\x82\xb5\x82\xe0\x82\xe6"
  "\x82\xeb\x82\xb5\x82\xad\x82\xa8\x82\xcb\x82\xaa\x82\xa2\x82\xa2"
  "\x82\xbd\x82\xb5\x82\xa0\x82\xaf\x82\xdc\x82\xb5\x82\xc4\x82\xa8"
  "\x82\xdf\x82\xc5\x82\xc6\x82\xa0\x82\xaf\x82\xdc\x82\xb5\x82\xc4"
  "\x82\xa8\x82\xdf\x82\xc5\x82\xc6\x82\xa4\x82\xb2\x82\xb4\x82\xa2"
  "\x82\xdc\x82\xb7\x82\xb1\x82\xc6\x82\xa8\x82\xdf\x82\xc5\x82\xc6"
  "\x82\xa4\x82\xb2\x82\xb4\x82\xa2\x82\xdc\x82\xb7\x82\xb1\x82\xc6"
  "\x0a";

TEST_F(CompactEncDetTest, ShiftJISNoHintTest) {
  bool is_reliable;
  EXPECT_EQ(JAPANESE_SHIFT_JIS,
            TestCompactEncDetWithReliable(kTestShiftJISNoHint,
                                          strlen(kTestShiftJISNoHint),
                                          "JAPANESE_SHIFT_JIS",
                                          &is_reliable));
  EXPECT_EQ(is_reliable, true);
}

#if 0
CP1252 => UTF8 => UTF8UTF8
80 => E282AC => C3A2E2809AC2AC
81 => C281 => C382C281
82 => E2809A => C3A2E282ACC5A1
83 => C692 => C386E28099
84 => E2809E => C3A2E282ACC5BE
85 => E280A6 => C3A2E282ACC2A6
86 => E280A0 => C3A2E282ACC2A0
87 => E280A1 => C3A2E282ACC2A1
88 => CB86 => C38BE280A0
89 => E280B0 => C3A2E282ACC2B0
8A => C5A0 => C385C2A0
8B => E280B9 => C3A2E282ACC2B9
8C => C592 => C385E28099
8D => C28D => C382C28D
8E => C5BD => C385C2BD
8F => C28F => C382C28F
90 => C290 => C382C290
91 => E28098 => C3A2E282ACCB9C
92 => E28099 => C3A2E282ACE284A2
93 => E2809C => C3A2E282ACC593
94 => E2809D => C3A2E282ACC29D
95 => E280A2 => C3A2E282ACC2A2
96 => E28093 => C3A2E282ACE2809C
97 => E28094 => C3A2E282ACE2809D
98 => CB9C => C38BC593
99 => E284A2 => C3A2E2809EC2A2
9A => C5A1 => C385C2A1
9B => E280BA => C3A2E282ACC2BA
9C => C593 => C385E2809C
9D => C29D => C382C29D
9E => C5BE => C385C2BE
9F => C5B8 => C385C2B8
A0 => C2A0 => C382C2A0
A1 => C2A1 => C382C2A1
A2 => C2A2 => C382C2A2
A3 => C2A3 => C382C2A3
A4 => C2A4 => C382C2A4
A5 => C2A5 => C382C2A5
A6 => C2A6 => C382C2A6
A7 => C2A7 => C382C2A7
A8 => C2A8 => C382C2A8
A9 => C2A9 => C382C2A9
AA => C2AA => C382C2AA
AB => C2AB => C382C2AB
AC => C2AC => C382C2AC
AD => C2AD => C382C2AD
AE => C2AE => C382C2AE
AF => C2AF => C382C2AF
B0 => C2B0 => C382C2B0
B1 => C2B1 => C382C2B1
B2 => C2B2 => C382C2B2
B3 => C2B3 => C382C2B3
B4 => C2B4 => C382C2B4
B5 => C2B5 => C382C2B5
B6 => C2B6 => C382C2B6
B7 => C2B7 => C382C2B7
B8 => C2B8 => C382C2B8
B9 => C2B9 => C382C2B9
BA => C2BA => C382C2BA
BB => C2BB => C382C2BB
BC => C2BC => C382C2BC
BD => C2BD => C382C2BD
BE => C2BE => C382C2BE
BF => C2BF => C382C2BF
C0 => C380 => C383E282AC
C1 => C381 => C383C281
C2 => C382 => C383E2809A
C3 => C383 => C383C692
C4 => C384 => C383E2809E
C5 => C385 => C383E280A6
C6 => C386 => C383E280A0
C7 => C387 => C383E280A1
C8 => C388 => C383CB86
C9 => C389 => C383E280B0
CA => C38A => C383C5A0
CB => C38B => C383E280B9
CC => C38C => C383C592
CD => C38D => C383C28D
CE => C38E => C383C5BD
CF => C38F => C383C28F
D0 => C390 => C383C290
D1 => C391 => C383E28098
D2 => C392 => C383E28099
D3 => C393 => C383E2809C
D4 => C394 => C383E2809D
D5 => C395 => C383E280A2
D6 => C396 => C383E28093
D7 => C397 => C383E28094
D8 => C398 => C383CB9C
D9 => C399 => C383E284A2
DA => C39A => C383C5A1
DB => C39B => C383E280BA
DC => C39C => C383C593
DD => C39D => C383C29D
DE => C39E => C383C5BE
DF => C39F => C383C5B8
E0 => C3A0 => C383C2A0
E1 => C3A1 => C383C2A1
E2 => C3A2 => C383C2A2
E3 => C3A3 => C383C2A3
E4 => C3A4 => C383C2A4
E5 => C3A5 => C383C2A5
E6 => C3A6 => C383C2A6
E7 => C3A7 => C383C2A7
E8 => C3A8 => C383C2A8
E9 => C3A9 => C383C2A9
EA => C3AA => C383C2AA
EB => C3AB => C383C2AB
EC => C3AC => C383C2AC
ED => C3AD => C383C2AD
EE => C3AE => C383C2AE
EF => C3AF => C383C2AF
F0 => C3B0 => C383C2B0
F1 => C3B1 => C383C2B1
F2 => C3B2 => C383C2B2
F3 => C3B3 => C383C2B3
F4 => C3B4 => C383C2B4
F5 => C3B5 => C383C2B5
F6 => C3B6 => C383C2B6
F7 => C3B7 => C383C2B7
F8 => C3B8 => C383C2B8
F9 => C3B9 => C383C2B9
FA => C3BA => C383C2BA
FB => C3BB => C383C2BB
FC => C3BC => C383C2BC
FD => C3BD => C383C2BD
FE => C3BE => C383C2BE
FF => C3BF => C383C2BF
#endif
}  // namespace
