//
//  Lynkeos
//  $Id: MyImageListWindow.m,v 1.23 2005/02/05 20:50:19 j-etienne Exp $
//
//  Created by Jean-Etienne LAMIAUD on Fri Nov 28 2003.
//  Copyright (c) 2003-2005. Jean-Etienne LAMIAUD
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#include "MyDocument.h"
#include "MyImageListWindow.h"

#include "MyImageListWindowPrivate.h"

#include "MyUserPrefsController.h"

#define K_LIST_TAB_IDENT        @"list"
#define K_ALIGN_TAB_IDENT       @"align"
#define K_ANALYZE_TAB_IDENT     @"analyze"
#define K_STACK_TAB_IDENT       @"stack"
#define K_PROCESS_TAB_IDENT     @"process"

static u_short adjustFFTside( u_short n )
{
   int v, i2, i3, i5, i7, inf, sup;
   
   inf = 0;
   sup = INT_MAX;

   for ( i7 = 1 ; ; i7 *= 7 )
   {
      for( i5 = 1; ; i5 *= 5 )
      {
         for( i3 = 1; ; i3 *= 3 )
         {
            for( i2 = 1; ; i2 *= 2 )
            {
               v = i2*i3*i5*i7;
               if ( v >= n && v < sup )
                  sup = v;
               if ( v <= n && v > inf )
                  inf =v;
               
               if ( v >= n )
                  break;
            }
            if ( i3*i5*i7 >= n || v == n )
               break;
         }
         if ( i5*i7 >= n || v == n )
            break;
      }
      if ( i7 >= n || v == n )
         break;
   }
   
   return( n >= (sup+inf)/2 ? sup : inf );
}

static void adjustFFTrect( MyIntegerRect *r )
{
   MyIntegerSize oldSize = r->size;

   r->size.width = adjustFFTside( r->size.width );
   r->size.height = adjustFFTside( r->size.height );
   r->origin.x += (oldSize.width - r->size.width)/2;
   r->origin.y += (oldSize.height - r->size.height)/2;
}

@implementation MyImageListWindow

- (id) init
{
   _windowMode = ImageMode;
   _windowState = ListMode;
   _highlightedItem = nil;
   _sideMenuLimit = 0;
   _fillSideArmed = NO;

   return( [super initWithWindowNibName:@"ImageListWindow"] );
}

- (void)windowDidLoad
{
   MyIntegerRect empty = {{0,0},{0,0}};
   MyObjectImageList* list;

   // Restore the window frame
   NSString* wframe = [(MyDocument*)[self document] windowFrame];
   if (wframe != nil )
      [[self window] setFrameFromString:wframe];

   // Get the document contents
   _currentList = (MyImageList*)[(MyDocument*)[self document] imageList];
   list = (MyObjectImageList*)_currentList;     // At init we work in imageList

   NSTableColumn *tableColumn = nil;
   NSButtonCell *buttonCell = nil;

   // Initialize the buttons in the outline first column
   tableColumn = [_textView tableColumnWithIdentifier: @"select"];
   buttonCell = [[[NSButtonCell alloc] initTextCell: @""] autorelease];
   [buttonCell setEditable: YES];
   [buttonCell setButtonType: NSSwitchButton];
   [buttonCell setAllowsMixedState: YES];
   [tableColumn setDataCell:buttonCell];
   [_textView reloadData];

   // Initialize dragging
   [_textView registerForDraggedTypes:
      [NSArray arrayWithObject:NSFilenamesPboardType]];

   // Initialize image view
   [_imageView setSelection:empty resizable:NO];

   // Update each panel (at its initial value)
   [self updateListControls];
   [self fillSidePopup];
   [self updateAlignControls];
   [self updateAnalyzeControls];
   [self updateSelectThresholdSlide];
   [self updateStackControls];
   [self updateProcessControls];
}

- (void)keyDown:(NSEvent *)theEvent
{
   unichar c = [[theEvent characters] characterAtIndex:0];

   if ( _windowState == ListMode || _windowState == AlignMode ||
        _windowState == AnalyzeMode )
   {
      BOOL nextEnabled = [[_currentList imageArray] count] != 0;

      switch( c )
      {
	 case NSLeftArrowFunctionKey:
            if ( nextEnabled )
               [self highlightPrevious:nil];
	    break;
	 case NSRightArrowFunctionKey :
            if ( nextEnabled )
               [self highlightNext:nil];
	    break;
	 case NSHomeFunctionKey :
            if ( nextEnabled )
               [self highlightItem:[_currentList firstItem]];
	    break;
	 case NSEndFunctionKey :
            if ( nextEnabled )
               [self highlightItem:[_currentList lastItem]];
	    break;
	 case '\r' :
	 case ' ' :
            if ( _highlightedItem != nil )
               [self toggleEntrySelection:nil];
	    break;
	 case NSDeleteFunctionKey :
	 case '\b' :
	 case 127 : // Delete char
            if ( _highlightedItem != nil )
               [self deleteAction:nil];
	    break;
	 default:
	    [super keyDown:theEvent];
	    break;
      }
   }
   else
      [super keyDown:theEvent];
}

// Accessors
- (MyImageListWindowState) windowMode { return( _windowMode ); }

- (MyImageListWindowState) windowState { return( _windowState ); }

- (MyImageListItem*) highlightedItem { return( _highlightedItem ); }

   // Actions
- (void) setWindowState :(MyImageListWindowState)state
{
   MyImageListWindowState oldState = _windowState;
   _windowState = state;

   switch ( _windowState )
   {
      case AlignMode :
         [_listMenu setEnabled:NO];
	 [_alignButton setTitle:NSLocalizedString(@"Align",@"Align button")];
         [self updateAlignControls];
	 break;
      case Aligning :
         _processImageUpdate = [MyUserPrefsController alignImageUpdating];
         [_listMenu setEnabled:NO];
	 [_alignButton setTitle:NSLocalizedString(@"Stop",@"Stop button")];
         [self updateAlignControls];
	 break;
      case AnalyzeMode :
         [_listMenu setEnabled:NO];
         [_analyzeButton setTitle:NSLocalizedString(@"Analyze",
                                                    @"Analyze button")];
         [self updateAnalyzeControls];
         [self updateSelectThresholdSlide];
	 break;
      case Analyzing :
         _processImageUpdate = [MyUserPrefsController analysisImageUpdating];
         [_listMenu setEnabled:NO];
	 [_analyzeButton setTitle:NSLocalizedString(@"Stop",@"Stop button")];
         [self updateAnalyzeControls];
	 break;
      case StackMode :
         [_listMenu setEnabled:YES];
	 [_stackButton setTitle:NSLocalizedString(@"Stack",@"Stack button")];
         [self updateStackControls];
	 break;
      case Stacking :
         _processImageUpdate = [MyUserPrefsController stackImageUpdating];
         [_listMenu setEnabled:NO];
	 [_stackButton setTitle:NSLocalizedString(@"Stop",@"Stop button")];
         [self updateStackControls];
	 break;
      case ListMode :
      case ProcessMode :
         [_listMenu setEnabled:YES];
         break;
      default:
         _windowState = oldState;
         NSAssert1( NO, @"setWindowState with bad state : %d", state);
	 break;
   }
}

- (void) refreshOutline
{
   [_textView reloadData];
}

- (void) highlightItem :(MyImageListItem*)item
{
   if ( item == nil )
      [_textView deselectAll:self];

   else
   {
      int row;

      // Expand father (won't do anything if already expanded)
      if ( [item isMemberOfClass:[MyMovieImage class]] )
	 [_textView expandItem:[(MyMovieImage*)item getParent]];

      row = [_textView rowForItem:item];

      // Don't change highlight if item is not in the list
      if ( row < 0 )
	 return;

      // Set the hilight
      [_textView selectRow:row byExtendingSelection:NO];
      [_textView scrollRowToVisible:row];
   }
}

- (void) highlightNext :(id)sender
{
   [self highlightOther:YES];
}

- (void) highlightPrevious :(id)sender
{
   [self highlightOther:NO];
}

// User select or deselect a line
- (BOOL) outlineView:(NSOutlineView*)outlineView shouldSelectItem:(id)item
{
   return ( _windowState == ListMode || _windowState == AlignMode || 
            _windowState == AnalyzeMode );
}

- (void) outlineViewSelectionDidChange :(NSNotification*)aNotification
{
   // Don't use aNotification because it's also called from this class (ugly...)
   MyImageListItem *item = nil;
   int row = [_textView selectedRow];
   MyIntegerRect selRect = {{0,0},{0,0}};
   NSPoint offset = {0,0};
   bool resize = NO;
   bool should_update_image = NO;
   NSImage *image = nil;

   // Determine if image frame shall be updated
   switch ( _windowState )
   {
      case ListMode :
      case AlignMode :
      case AnalyzeMode :
      case StackMode :
         should_update_image = YES;
         break;
      case Aligning :
      case Analyzing :
      case Stacking :
         should_update_image = _processImageUpdate;
         break;
      default :
         NSAssert1( NO, @"Selection change in state %@", 
                    [self windowStateName] );
         break;
   }
   
   _highlightedItem = nil;

   if ( row != -1 )
   {
      // Selection is valid
      item = [_textView itemAtRow:row];
      _highlightedItem = item;

      if ( ! [item isMemberOfClass:[MyMovie class]] )
      {
         if ( should_update_image )
            image = [item getImage];

	 offset = [item alignOffset];

	 // Update the selection rectangle
	 switch ( _windowState )
	 {
	    case Aligning :
	       if ( ! _processImageUpdate )
		  break;
	       /* else fall through */
	    case AlignMode :
	       if ( [item getSelectionState] != 0  )
	       {
		  if ( [item hasSearchSquare] )
		     selRect.origin = [item searchSquareOrigin];
		  else
		     selRect.origin = 
                          [(MyObjectImageList*)_currentList searchSquareOrigin];
		  selRect.size.width = 
                            [(MyObjectImageList*)_currentList searchSquareSide];
		  selRect.size.height = selRect.size.width;
	       }
               break;
	    case Analyzing :
	       if ( ! _processImageUpdate )
		  break;
	       /* else fall through */
	    case AnalyzeMode :
	       selRect.origin = 
                         [(MyObjectImageList*)_currentList analyzeSquareOrigin];
	       selRect.size.width = 
                           [(MyObjectImageList*)_currentList analyzeSquareSide];
	       selRect.size.height = selRect.size.width;
	       break;
	    case Stacking :
	       if ( ! _processImageUpdate )
		  break;
            case StackMode :
               if ( _windowMode == ImageMode )
               {
                  selRect = [(MyObjectImageList*)_currentList cropRectangle];
                  resize = (_windowState == StackMode);
               }
               else
               {
                  selRect.origin = MyMakeIntegerPoint(0,0);
                  selRect.size = MyIntegerSizeFromNSSize(
                                          [[_currentList firstItem] imageSize]);
               }
	       break;
	    default :
	       break;
	 }
      }
   }

   if ( should_update_image )
   {
      [_imageView setImage:image offset:offset];
      [_imageView setSelection :selRect resizable:resize];
   }

   if ( _windowState == ListMode )
      [self updateListControls];
   if ( _windowState == AlignMode || _windowState == Aligning )
      [self updateAlignControls];
   if ( _windowState == AnalyzeMode || _windowState == Analyzing )
      [self updateAnalyzeControls];
   if ( _windowState == StackMode || _windowState == Stacking )
      [self updateStackControls];
}

- (void)outlineView:(NSOutlineView *)outlineView willDisplayCell:(id)cell
     forTableColumn:(NSTableColumn *)tableColumn item:(id)item
{
   NSString* column = [tableColumn identifier];
   if ( [column isEqual:@"index"] || [column isEqual:@"name"] )
   {
      NSColor *color;
      if ( ![item isMemberOfClass:[MyMovie class]] && [item isAligned] )
	 color = [NSColor greenColor];
      else
	 color = [NSColor textColor];
      [cell setTextColor:color];
   }

   [cell setEditable: (_windowState == ListMode || _windowState == AlignMode)];
}

- (id) outlineView:(NSOutlineView *)outlineView child:(int)index 
            ofItem:(id)item
{
   if ( item == nil )
      return( [[_currentList imageArray] objectAtIndex: index] );

   else
   {
      NSAssert( [item isMemberOfClass: [MyMovie class]], @"item has no child" );
      return( [item getMovieImageAtIndex: index] );
   }
}

- (BOOL) outlineView:(NSOutlineView *)outlineView isItemExpandable:(id)item
{
   return ( [item isMemberOfClass: [MyMovie class]] );
}

- (int) outlineView:(NSOutlineView *)outlineView numberOfChildrenOfItem:(id)item
{
  if ( item == nil ){
      return( [[_currentList imageArray] count] );
  }
   else
   {
#ifdef GNUSTEP
     if([item isMemberOfClass: [MyMovie class]]){
       return( [item numberOfImages] );
     }
     else{
       return 0;
     }
#else
     NSAssert([item isMemberOfClass: [MyMovie class]], @"item has no child" );
     return( [item numberOfImages] );
#endif
   }
}

- (id) outlineView:(NSOutlineView *)outlineView 
       objectValueForTableColumn:(NSTableColumn *)tableColumn
            byItem:(id)item
{
   id columnId = [tableColumn identifier];

   if ( [columnId isEqual:@"select"] )
      return( [NSNumber numberWithInt: [item getSelectionState]] );

   else if ( [columnId isEqual:@"name"] )
      return( [item getName] );

   else if ( [columnId isEqual:@"index"] )
      return( [item getIndex] );

   else if ( [columnId isEqual:@"dx"] )
   {
      if ( _windowMode == ImageMode &&
           ( [item isMemberOfClass:[MyImage class]] || 
             [item isMemberOfClass:[MyMovieImage class]] )
	   && [item getSelectionState] == NSOnState && [item isAligned] )
	 return( [NSString stringWithFormat:@"%.1f",[item alignOffset].x] );
   }

   else if ( [columnId isEqual:@"dy"] )
   {
      if ( _windowMode == ImageMode &&
           ([item isMemberOfClass:[MyImage class]] || 
            [item isMemberOfClass:[MyMovieImage class]])
	   && [item getSelectionState] == NSOnState && [item isAligned] )
	 return( [NSString stringWithFormat:@"%.1f",[item alignOffset].y] );
   }
   else if ( [columnId isEqual:@"quality"] )
   {
      if ( _windowMode == ImageMode &&
           ([item isMemberOfClass:[MyImage class]] || 
            [item isMemberOfClass:[MyMovieImage class]]) )
      {
	 double q = [item getQuality];
	 if ( q >= 0 )
	    return( [NSString stringWithFormat:@"%.2f",q] );
      }
   }

   return( nil );
}

- (void)outlineView:(NSOutlineView *)outlineView
     setObjectValue:(id)object forTableColumn:(NSTableColumn *)tableColumn 
             byItem:(id)item
{
   if ( [[tableColumn identifier] isEqual:@"select"] )
   {
      [(MyDocument*)[self document] changeEntrySelection :item 
                                                   value :[object boolValue]];

      // Redraw checkbox if needed
      if ( _windowMode == ImageMode )
         [_refCheckBox setState: 
            ([(MyObjectImageList*)_currentList referenceItem] == item)];

      // Redraw other affected lines
      if ( [item isMemberOfClass:[MyMovie class]] )
	 [_textView reloadItem:item reloadChildren:YES];
      else if ( [item isMemberOfClass: [MyMovieImage class]] )
	 [_textView reloadItem:[(MyMovieImage*)item getParent] 
                    reloadChildren:NO];
   }
}

// Tab view management
- (BOOL)tabView:(NSTabView *)tabView 
        shouldSelectTabViewItem:(NSTabViewItem *)tabViewItem
{
   id ident = [tabViewItem identifier];

   // Any tab change is forbidden in these states
   if ( _windowState == Aligning ||
        _windowState == Analyzing || 
        _windowState == Stacking )
      return( NO );

   // Align and Analyze tabs are authorized only in ImageList mode
   if ( _windowMode != ImageMode && 
        ( [ident isEqual: K_ALIGN_TAB_IDENT] || 
          [ident isEqual: K_ANALYZE_TAB_IDENT] ) )
      return( NO );

   // What is not forbidden is authorized
   return( YES );
}

- (void)tabView :(NSTabView *)tabView 
            didSelectTabViewItem:(NSTabViewItem *)tabViewItem
{
   NSPoint nowhere = {0,0};
   MyIntegerRect selRect = { {0,0}, {0,0} };
   bool resize = NO;
   id ident = [tabViewItem identifier];

   if ( [ident isEqual: K_ALIGN_TAB_IDENT] )
   {
      [self setWindowState: AlignMode];
      [self outlineViewSelectionDidChange:nil];
   }
   else if ( [ident isEqual: K_ANALYZE_TAB_IDENT] )
   {
      [self setWindowState: AnalyzeMode];
      [self outlineViewSelectionDidChange:nil];
   }
   else if ( [ident isEqual: K_STACK_TAB_IDENT] )
   {
      MyImageListItem* ref;

      if ( _windowMode == ImageMode )
         ref = [(MyObjectImageList*)_currentList referenceItem];
      else
         ref = [_currentList firstItem];

      [self setWindowState: StackMode];

      [self highlightItem:nil];
      [self highlightItem:ref];
   }
   else if ( [ident isEqual: K_LIST_TAB_IDENT] )
   {
      [self setWindowState: ListMode];
      [self outlineViewSelectionDidChange:nil];
   }
   else if ( [ident isEqual: K_PROCESS_TAB_IDENT] )
   {
      [self setWindowState: ProcessMode];
      [self highlightItem:nil];
      [self updateProcessControls];

      if ( [(MyDocument*)[self document] stackedImage] == nil )
	 [self postProcessAction:nil];
      else
	 [_imageView setImage:[(MyDocument*)[self document] stackedImage] 
                     offset:nowhere];
   }

   if ( _windowState != AlignMode && _windowState != AnalyzeMode 
        && _windowState != StackMode )
      [_imageView setSelection :selRect resizable:resize];
}

// Image view management
- (void) myImageView :(MyImageView*)imageView 
         selectionRectDidChange :(MyIntegerRect)rect :(unsigned int)modifier
{
   MyObjectImageList *list = (MyObjectImageList*)_currentList;

   NSAssert1( _windowMode == ImageMode, 
              @"Selection rect change in state : %@", [self windowModeName] );

   switch( _windowState )
   {
      case AlignMode :
	 NSAssert( rect.size.width == [list searchSquareSide] &&
		   rect.size.height == [list searchSquareSide],
		   @"Invalid selection rectangle size in Align state" );
	 [_searchFieldX setFloatValue:rect.origin.x];
	 [_searchFieldY setFloatValue:rect.origin.y];
	 if ( (modifier & NSAlternateKeyMask) == 0 )
	    // Standard drag : update the default search square
	    [(MyDocument*)[self document] changeSearchSquareOrigin:rect.origin];
	 else
	 { // Option drag : update the image search square
	    NSAssert( _highlightedItem != nil, 
                      @"Image search square without image" );
	    [_highlightedItem setSearchSquareOrigin:rect.origin];
	 }
         break;
      case AnalyzeMode :
	 NSAssert( rect.size.width == [list analyzeSquareSide] &&
		   rect.size.height == [list analyzeSquareSide],
		   @"Invalid selection rectangle size in Analyze state" );
	 [_analyzeFieldX setFloatValue:rect.origin.x];
	 [_analyzeFieldY setFloatValue:rect.origin.y];
	 [(MyDocument*)[self document] changeAnalyzeSquareOrigin:rect.origin];
	 break;
      case StackMode :
         if ( [MyUserPrefsController adjustFFTSizes] )
         {
            // Adjust the rectangle size to optimize the FFT calculations
            adjustFFTrect( &rect );
            [_imageView setSelection :rect resizable:YES];
         }

	 [(MyDocument*)[self document] changeCropRectangle:rect];
         [self updateStackControls];
	 break;
      default :
	 NSAssert1( NO, @"Selection change in invalid state %d", _windowState );
   }
}

- (void) updateSidePopup
{
   // Schedule an update to be done (once) after this event is processed
   if ( ! _fillSideArmed )
   {
      [[NSRunLoop currentRunLoop] performSelector:@selector(fillSidePopup)
                                           target:self
                                         argument:nil
                                            order:0
                                            modes:
                                [NSArray arrayWithObject:NSDefaultRunLoopMode]];
      _fillSideArmed = YES;
   }
}

- (void) searchSquareChange :(id)sender
{
   MyIntegerPoint o;
   MyIntegerRect selRect;

   NSAssert1( _windowMode == ImageMode, @"Search square change in mode : %@", 
              [self windowModeName] );
   NSAssert1( _windowState == AlignMode, 
              @"Align tab control change while in state %@", 
               [self windowStateName] );

   if ( sender == _searchSideMenu )
   {
      int sel = [_searchSideMenu indexOfSelectedItem];
      u_short side;
      BOOL alignEnabled;

      if ( sel == -1 )
      {
	 side = 0;
	 alignEnabled = NO;
      }
      else
      {
	 side = [[_searchSideMenu titleOfSelectedItem] doubleValue];
	 alignEnabled = YES;
      }

      [(MyDocument*)[self document] changeSearchSquareSide:side];
      [self updateAlignControls];
   }

   o.x = (int)[_searchFieldX floatValue];
   o.y = (int)[_searchFieldY floatValue];
   if ( _highlightedItem != nil && [_highlightedItem hasSearchSquare] )
      [_highlightedItem setSearchSquareOrigin:o];
   else
      [(MyObjectImageList*)_currentList setSearchSquareOrigin:o];

   selRect.origin = o;
   selRect.size.width = [(MyObjectImageList*)_currentList searchSquareSide];
   selRect.size.height = selRect.size.width;
   [_imageView setSelection :selRect resizable:NO];
}

- (void) analyzeSquareChange :(id)sender
{
   MyIntegerPoint o;
   MyIntegerRect selRect;

   NSAssert1( _windowMode == ImageMode, @"Analyze square change in mode : %@", 
              [self windowModeName] );
   NSAssert1( _windowState == AnalyzeMode, 
              @"Analyze tab control change while in state %@", 
                 [self windowStateName] );

   if ( sender == _analyzeSideMenu )
   {
      int sel = [_analyzeSideMenu indexOfSelectedItem];
      u_short side;
      BOOL analyzeEnabled;

      if ( sel == -1 )
      {
	 side = 0;
	 analyzeEnabled = NO;
      }
      else
      {
	 side = [[_analyzeSideMenu titleOfSelectedItem] doubleValue];
	 analyzeEnabled = YES;
      }

      [(MyDocument*)[self document] changeAnalyzeSquareSide:side];
      [self updateAnalyzeControls];
   }

   o.x = (int)[_analyzeFieldX floatValue];
   o.y = (int)[_analyzeFieldY floatValue];
   [(MyDocument*)[self document] changeAnalyzeSquareOrigin:o];

   selRect.origin = o;
   selRect.size.width = [(MyObjectImageList*)_currentList analyzeSquareSide];
   selRect.size.height = selRect.size.width;
   [_imageView setSelection :selRect resizable:NO];
}

- (void) analyzeMethodChange :(id)sender 
{
   [(MyDocument*)[self document] changeAnalysisMethod:
                                                [sender indexOfSelectedItem]];
}

- (void) cropRectangleChange :(id)sender
{
   MyIntegerRect r;

   NSAssert( _windowMode == ImageMode, 
              @"Crop rectangle change in a non image mode" );

   r = [(MyObjectImageList*)_currentList cropRectangle];

   if ( sender == _cropX )
      r.origin.x = (int)[_cropX floatValue];
   else if ( sender == _cropY )
      r.origin.y = (int)[_cropY floatValue];
   else if ( sender == _cropW )
      r.size.width = (int)[_cropW floatValue];
   else if ( sender == _cropH )
      r.size.height = (int)[_cropH floatValue];

   if ( [MyUserPrefsController adjustFFTSizes] )
      // Adjust the rectangle size to optimize the FFT calculations
      adjustFFTrect( &r );

   [(MyDocument*)[self document] changeCropRectangle:r];

   [_imageView setSelection :r resizable:YES];
   [self updateStackControls];
}

// Buttons or menu actions
- (void) modeMenuAction :(id)sender
{
   _windowMode = [_listMenu indexOfSelectedItem];

   // Set the window controller in the requested mode
   // and update the image list shortcut
   switch ( _windowMode )
   {
      case ImageMode :
         _currentList = [(MyDocument*)[self document] imageList];
         break;
      case DarkFrameMode :
         _currentList = [(MyDocument*)[self document] darkFrameList];
         break;
      case FlatFieldMode :
         _currentList = [(MyDocument*)[self document] flatFieldList];
         break;
   }

   // Redisplay the outline view
   [_textView reloadData];

   // Forget about the stacked image
   [(MyDocument*)[self document] invalidateStackedImage];

   // Restore an item highlight consistent with the current state
   // Update the active panel
   [self highlightItem:nil];
   switch ( _windowState )
   {
      case ListMode :
         break;
      case AlignMode :
         [self updateAlignControls];
         break;
      case AnalyzeMode :
         [self updateAnalyzeControls];
         [self updateSelectThresholdSlide];
         break;
      case StackMode :
         if ( _windowMode == ImageMode )
            [self highlightItem:
                           [(MyObjectImageList*)_currentList referenceItem]];
         else
            [self highlightItem:[_currentList firstItem]];
         [self updateStackControls];
         break;
      case ProcessMode :
         [self updateProcessControls];
         // Regenerate and display the stacked image (if any)
         [self postProcessAction:nil];
         break;
      default :
         NSAssert2( NO, @"Mode change to %@ in state %@",
                    [self windowModeName], [self windowStateName] );
   }
}

- (void) addAction :(id)sender
{
   // Ask the user to choose some images/movies
   NSOpenPanel* panel = [NSOpenPanel openPanel];
   NSArray *files, *image_types, *movie_types;
   NSMutableArray *types;

   image_types = [NSImage imageFileTypes];
   movie_types = [NSMovie movieUnfilteredFileTypes];
   types = [NSMutableArray arrayWithArray : image_types];
   [types addObjectsFromArray : movie_types];

   [panel setAllowsMultipleSelection:YES];
   [panel runModalForTypes: types];
   files = [panel filenames];

   // And add their objects to the document
   [self addFiles:files];
}

- (void) deleteAction :(id)sender
{
   int sel = [_textView selectedRow];
   id item = [_textView itemAtRow:sel];

   [(MyDocument*)[self document] deleteEntry:item];

   [self updateListControls];
}

- (void) referenceAction :(id)sender
{
   NSAssert1( _windowMode == ImageMode, @"Reference item change in mode : %@", 
              [self windowModeName] );

   [(MyDocument*)[self document] changeReferenceEntry:
                     ([sender state] == NSOnState ? _highlightedItem : nil)];

   // Reconcile button and reference if user action was not possible
   if ( [(MyObjectImageList*)_currentList referenceItem] == _highlightedItem && 
        [sender state] != NSOnState )
      [sender setState : NSOnState];
}

- (void) toggleEntrySelection :(id)sender
{
   if( (_windowState == ListMode || _windowState == AlignMode || 
        _windowState == AnalyzeMode) &&
       _highlightedItem != nil )
   {
      [(MyDocument*)[self document] changeEntrySelection :_highlightedItem
                    value:([_highlightedItem getSelectionState] != NSOnState)];

      // Redraw checkbox if needed
      if ( _windowMode == ImageMode )
         [_refCheckBox setState: 
            ([(MyObjectImageList*)_currentList referenceItem] == 
                                                             _highlightedItem)];

      // Redraw item and other affected lines
      [_textView reloadItem:_highlightedItem reloadChildren:
                            [_highlightedItem isMemberOfClass:[MyMovie class]]];
      if ( [_highlightedItem isMemberOfClass: [MyMovieImage class]] )
	 [_textView reloadItem:[(MyMovieImage*)_highlightedItem getParent] 
                    reloadChildren:NO];
   }
}

- (void) alignAction :(id)sender
{
   [sender setEnabled:NO];
   [(MyDocument*)[self document] align];
}

- (void) autoSelectAction :(id)sender
{
   [(MyDocument*)[self document] autoChangeSelection : [sender doubleValue]];
   [_textView reloadData];
}

- (void) analyzeAction :(id)sender
{
   [sender setEnabled:NO];
   [(MyDocument*)[self document] analyzeQuality];
}

- (void) doubleButtonAction :(id)sender
{
   [(MyDocument*)[self document] setDoubleSize : [sender state]];
}

- (void) monochromeButtonAction :(id)sender
{
   [(MyDocument*)[self document] setMonochromeFlat : [sender state]];
}

- (void) stackAction :(id)sender
{
   [sender setEnabled:NO];

   // Ensure the process tab controls are up to date,
   // because they will be used when the stacking ends
   [self updateProcessControls];

   [(MyDocument*)[self document] stack];
}

- (void) postProcessAction :(id)sender
{
   // Reconcile controls
   if ( sender == _deconvRadius )
      [_deconvTextRadius setDoubleValue:[_deconvRadius doubleValue]];
   else if ( sender == _deconvTextRadius )
      [_deconvRadius setDoubleValue:[_deconvTextRadius doubleValue]];
   else if ( sender == _deconvThreshold )
      [_deconvTextThreshold setDoubleValue:[_deconvThreshold doubleValue]];
   else if ( sender == _deconvTextThreshold )
      [_deconvThreshold setDoubleValue:[_deconvTextThreshold doubleValue]];
   else if ( sender == _unsharpRadius )
      [_unsharpTextRadius setDoubleValue:[_unsharpRadius doubleValue]];
   else if ( sender == _unsharpTextRadius )
      [_unsharpRadius setDoubleValue:[_unsharpTextRadius doubleValue]];
   else if ( sender == _unsharpGain )
      [_unsharpTextGain setDoubleValue:[_unsharpGain doubleValue]];
   else if ( sender == _unsharpTextGain )
      [_unsharpGain setDoubleValue:[_unsharpTextGain doubleValue]];

   // Process according to these values
   [(MyDocument*)[self document] postProcess : [_deconvRadius doubleValue]
					     : [_deconvThreshold doubleValue]
					     : [_unsharpRadius doubleValue]
					     : [_unsharpGain doubleValue] ];
   [self updateProcessControls];	// Levels may have changed
   [(MyDocument*)[self document] adjustLevels :[_blackTextLevel doubleValue] 
                                              :[_whiteTextLevel doubleValue]];

   if ( _windowState == ProcessMode )
      [_imageView setImage:[(MyDocument*)[self document] stackedImage] 
                  offset:NSMakePoint(0,0)];
}

- (void) changeLevelsAction : (id)sender
{
   double black = [_blackLevel doubleValue],
   white = [_whiteLevel doubleValue];

   // Reconcile controls and ensure consistency
   if ( sender == _blackLevel || sender == _blackTextLevel )
   {
      black = [sender doubleValue];
      if ( black > white )
	 black = white-1.0/HUGE;
      [_blackLevel setDoubleValue:black];
      [_blackTextLevel setDoubleValue:black];
   }
   else if ( sender == _whiteLevel || sender == _whiteTextLevel )
   {
      white = [sender doubleValue];
      if ( white < black )
	 white = black+1.0/HUGE;
      [_whiteLevel setDoubleValue:white];
      [_whiteTextLevel setDoubleValue:white];
   }

   [(MyDocument*)[self document] adjustLevels :black :white];

   [_imageView setImage:[(MyDocument*)[self document] stackedImage] 
               offset:NSMakePoint(0,0)];
}

- (void) addFiles :(NSArray*)files
{
   NSArray *image_types, *movie_types;
   NSEnumerator* list;
   NSString* file;
      
   image_types = [NSImage imageFileTypes];
   movie_types = [NSMovie movieUnfilteredFileTypes];
      
   // And add their objects to the document
   list = [files objectEnumerator];
   while ( (file = [list nextObject]) != nil )
   {
      NSURL *url = [NSURL fileURLWithPath:file];

#ifdef GNUSTEP
      // a little ugly, but don't found "NSHFSTypeOfFile" on GNUstep.
      NSString* url_type = [file pathExtension];
#else
      NSString* url_type = NSHFSTypeOfFile(file);
#endif
      NSString* ext = [file pathExtension];
      MyImageListItem *item = nil;

      if ( [image_types containsObject :url_type] || 
            [image_types containsObject :ext] )
         item = [MyImage imageListItemWithURL: url];

      else if ( [movie_types containsObject :url_type] || 
                  [movie_types containsObject :ext] )
         item = [MyMovie imageListItemWithURL: url];

      if ( item != nil )
         [(MyDocument*)[self document] addEntry: item];
      else
      {
         NSMutableString* message = [NSMutableString stringWithFormat:
            NSLocalizedString(
                              @"BadFile",
                              @"Message of bad file alert message"),
            [url absoluteString]];

         NSRunAlertPanel(NSLocalizedString(@"BadFileTitle",
                                          @"Bad format file alert panel title"),
                           message, nil, nil, nil );
      }
   }

   [self updateListControls];
}

- (void) updateAlignControls
{
   MyObjectImageList *list = (MyObjectImageList*)_currentList;

   NSAssert1( _windowMode == ImageMode, 
              @"Align controls updating in mode %@", [self windowModeName] );

   if ( _windowState == AlignMode )
   {
      BOOL menuEnabled = ( _highlightedItem != nil
                           && ! [_highlightedItem isMemberOfClass:
                                                            [MyMovie class]] );
      BOOL controlsEnabled = ( menuEnabled 
                               && [_searchSideMenu indexOfSelectedItem] >= 0 );
      MyIntegerPoint o;

      if ( menuEnabled && [_highlightedItem hasSearchSquare] )
         o = [_highlightedItem searchSquareOrigin];
      else
         o = [list searchSquareOrigin];

      [_searchFieldX setFloatValue: o.x];
      [_searchFieldX setEnabled: controlsEnabled];
      [_searchFieldY setFloatValue: o.y];
      [_searchFieldY setEnabled: controlsEnabled];
      [_searchSideMenu setEnabled: menuEnabled];
      [_refCheckBox setState: 
                       ( _highlightedItem == [list referenceItem] ? 
                         NSOnState : NSOffState )];
      [_refCheckBox setEnabled: controlsEnabled];

      [_alignButton setEnabled:controlsEnabled];
   }
   else
   {
      [_searchFieldX setEnabled: NO];
      [_searchFieldY setEnabled: NO];
      [_searchSideMenu setEnabled: NO];
      [_refCheckBox setEnabled: NO];

      [_alignButton setEnabled:_windowState == Aligning];
   }
}

@end
