/***********************************************************************************

    Copyright (C) 2007-2018 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <cairomm/context.h>

#include "widget_chart.hpp"
#include "strings.hpp"

#include "diarydata.hpp"


using namespace LIFEO;

// WIDGETCHART =====================================================================================
WidgetChart::WidgetChart( BaseObjectType* cobject, const Glib::RefPtr< Gtk::Builder >& )
:   DrawingArea( cobject )
{
    set_events( Gdk::BUTTON_PRESS_MASK | Gdk::BUTTON_RELEASE_MASK |
                Gdk::POINTER_MOTION_MASK | Gdk::LEAVE_NOTIFY_MASK | Gdk::SCROLL_MASK );
}

bool
WidgetChart::on_scroll_event( GdkEventScroll* event )
{
    if( event->direction == GDK_SCROLL_UP )
        scroll( -1 );
    else
    if( event->direction == GDK_SCROLL_DOWN )
        scroll( 1 );

    return true;
}

bool
WidgetChart::on_button_press_event( GdkEventButton* event )
{
    if( m_points && event->button == 1 && event->y > m_height - m_ov_height )
    {
        int col_start{ int( ( event->x / m_width ) * m_span - m_step_count / 2 ) };
        const unsigned int col_start_max{ m_span - m_step_count };
        if( col_start > int( col_start_max ) )
            col_start = col_start_max;
        else
        if( col_start < 0 )
            col_start = 0;
        m_step_start = col_start;

        update();

        m_flag_button_pressed = true;
    }

    return true;
}

bool
WidgetChart::on_button_release_event( GdkEventButton* event )
{
    if( event->button == 1 )
    {
        m_flag_button_pressed = false;
    }

    return true;
}

bool
WidgetChart::on_motion_notify_event( GdkEventMotion* event )
{
    if( m_points )
    {
        if( m_flag_button_pressed )
        {
            int col_start = ( event->x / m_width ) * m_span - m_step_count / 2;
            if( col_start > int( m_span - m_step_count ) )
                col_start = m_span - m_step_count;
            else
            if( col_start < 0 )
                col_start = 0;

            if( col_start != ( int ) m_step_start )
            {
                m_step_start = col_start;
                update();
            }
        }
        int hovered_step{ int( round( ( event->x - s_x_min ) / m_step_x ) ) };

        if( hovered_step < 0 )
            hovered_step = 0;
        else if( hovered_step >= int( m_step_count ) )
            hovered_step = m_step_count - 1;

        bool flag_pointer_hovered = ( event->y > m_height - m_ov_height );
        if( flag_pointer_hovered != m_flag_pointer_hovered )
        {
            m_flag_pointer_hovered = flag_pointer_hovered;
            update();  // TODO: limit to scrollbar only
        }
        else if( hovered_step != m_hovered_step )
        {
            m_hovered_step = hovered_step;
            update();  // TODO: limit to tooltip only
        }
    }

    return Gtk::DrawingArea::on_motion_notify_event( event );
}

bool
WidgetChart::on_leave_notify_event( GdkEventCrossing* event )
{
    if( m_points )
    {
        m_flag_pointer_hovered = false;
        update();  // TODO: limit to scrollbar only
    }

    return true;
}

void
WidgetChart::on_size_allocate( Gtk::Allocation& allocation )
{
    Gtk::Widget::on_size_allocate( allocation );
    resize( allocation.get_width(), allocation.get_height() );
}

bool
WidgetChart::on_draw( const Cairo::RefPtr< Cairo::Context >& cr )
{
    return( Chart::draw( cr ) );
}

//bool
//WidgetChart::handle_query_tooltip( int x, int y, bool keyboard_mode,
//                                   const Glib::RefPtr< Gtk::Tooltip >& tooltip )
//{
//    if( !m_points || keyboard_mode || m_span < 2 )
//        return false;
//
//    if( y > m_y_max ) // skip the label bars and overview area
//        return false;
//
//    int i{ int( round( ( x - s_x_min ) / m_step_x ) + m_step_start ) };
//
//    if( i < 0 )
//        i = 0;
//    else if( i >= int( m_span ) )
//        i = m_span - 1;
//
//    if( m_points->unit.empty() )
//        tooltip->set_text( STR::compose( m_points->values[ i ] ) );
//    else
//        tooltip->set_text( STR::compose( m_points->values[ i ], " ", m_points->unit ) );
//
//    return true;
//}
