/**
 * @file    export.c
 * @brief   Routines for exporting plots as images and animations.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <sys/stat.h>

#include "ygraph.h"

/**
 * @brief    Export the current frame to an image file.
 *
 *           This routine exports the
 *           screen impage displayed in the plot_area widget of the
 *           currently opened window to a file. The type of file is
 *           for the moment hardcoded to be png.
 *
 * @param    plot      The current displayed plot.
 * @param    filename  The name of the output file.
 * @todo     It would be useful to be able to specify the image properties
 *           via a widget rather than set them to defaults.
 */
void
export_to_file(Plot* plot, gchar* filename)
{
  GtkWidget* plot_area;
  GdkPixbuf* img;
  gboolean retval;
  GError *error = NULL;

  plot_area = plot->plot_area;

  img = gdk_pixbuf_get_from_drawable(NULL, plot->pixmap, NULL,
                                             0, 0, 0, 0,
                                             plot_area->allocation.width,
                                             plot_area->allocation.height);

  retval = gdk_pixbuf_save(img, filename, "png", &error, NULL);

  if (!retval)
    message_dialog ("Couldn't export file.");
}

/**
 * @brief    Export a sequence of frames to a directory.
 *
 *           This routine exports each of the frames in a data set
 *           as an image in a seperate file.
 *           The individual files are output in the PNG format, and given
 *           a file name corresponding to their iteration number (hard
 *           maximum of six digits).
 *
 * @param    plot     The current displayed plot.
 * @param    dirname  The name of the output file.
 */
void
export_to_dir(Plot* plot, gchar* dirname)
{
  GtkWidget* plot_area;
  GdkPixbuf* img;
  gchar err_msg[MAX_LINE_LENGTH];
  gchar filename[FILE_STR_SIZE];
  gboolean retval;
  GError *error = NULL;

  mkdir(dirname, S_IRWXU);

  plot_area = plot->plot_area;

  global_display_mode = ANIMATE_MODE;
  global_current_frame = FIRST_FRAME - 1;

  while (frame_draw_next(NULL))
    {
       g_snprintf(filename, FILE_STR_SIZE, "%s/%s_%06d.png", dirname, 
                  SAVE_FILE_BASE_NAME, global_current_frame);

       img = gdk_pixbuf_get_from_drawable(NULL, plot->pixmap, NULL,
                                          0, 0, 0, 0,
                                          plot_area->allocation.width,
                                          plot_area->allocation.height);

        retval = gdk_pixbuf_save(img, filename, "png", &error, NULL);

        if (!retval)
        {
          g_snprintf(err_msg, MAX_LINE_LENGTH, "Couldn't export file:\n%s",
                     filename);
          message_dialog (err_msg);
          return;
        }
    }
}
