/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#ifndef __AMT_MOF_OBJECT_H
#define __AMT_MOF_OBJECT_H

#include <memory>
#include <openwsman/cpp/WsmanClient.h>
#include "EndpointReference.h"

using namespace std;
using namespace WsmanClientNamespace;
using namespace CimXMLUtilsNamespace;
using namespace CimXMLTypesNamespace;
using namespace WsmanExceptionNamespace;


namespace CimClassNamespace
{
	template<class T>
	class CimClassContainer;

	// CIMClass Exception class
	class CIMClassException : public GeneralWsmanException
	{
	public:
		CIMClassException(const char* msg, unsigned int err = WSMAN_GENERAL_ERROR)
			:GeneralWsmanException(msg, err){}
	};

	class CimClass : public Serializable, public DeSerializable
	{
	protected:
		bool isCreated;
		string resourceUri;
		ResourceCreated creationData;
		static const WsmanClient* defaultProvider;
	public:
		CimClass(const string& objName,
			const string& uri,
			const string& ns,
			const string& nsPrefix = "");
		virtual ~CimClass();
		string GetObjectName() const;
		string GetResourceUri() const;
		virtual NameValuePairs GetSelectorsList() const = 0;
		EndpointReference ToEndpointReference() const;
		void Get(const NameValuePairs *selectors = NULL,
			const WsmanClient *cl = defaultProvider);
		void Put(const WsmanClient *cl = defaultProvider);
		void Create(const WsmanClient *cl = defaultProvider);
		void Delete(const WsmanClient *cl = defaultProvider);
		void Invoke(const string& functionName,
			const Serializable* input, DeSerializable* output,
			const WsmanClient *cl = defaultProvider) const;

		// Enumerates the CIM objects of type T. Subclasses will
		// be enumerated but they will be returned as instances of
		// T. (Subclass properties will not be recorded).
		template<class T>
		static void Enumerate(vector<T> &vec,
			const NameValuePairs *selectors = NULL,
			const WsmanClient *cl = defaultProvider);

		// Enumerates the CIM objects of type T. Subclasses will
		// be enumerated and subclass properties will be recorded.
		// Pointer to the created instances of T are entered to the
		// given container.
		template<class T>
		static void Enumerate(CimClassContainer<T> &container,
			const NameValuePairs *selectors = NULL,
			const WsmanClient *cl = defaultProvider);
		static void RegisterDefaultWsmanProvider(const WsmanClient *cl);
	};

	// maps CimClass names to ids, used by the factory class
	class CimClassMap : public map<string, unsigned int>
	{
	public:
		CimClassMap();
		~CimClassMap(){}
	};

	// map entry
	typedef pair<string,unsigned int> map_entry;

	// factory class
	class CimClassFactory
	{
	private:
		static const CimClassMap cimMap;
		// factory function. The implementation for this is autogenerated
		// in CimClassFactory.cpp
	public:
		static CimClass* CreateCimClass(const string& xml);
	};

	template<class T>
	class CimClassContainer
	{
	private:
		vector<T*> vec;
	public:
		CimClassContainer(){}

		~CimClassContainer()
		{
			Clear();
		}

		void AddCimClass(const string& xml)
		{
			T* t = static_cast<T*>(CimClassFactory::CreateCimClass(xml));
			if(!t)
			{
				throw CIMClassException("An unidentified CIM object was received.", WSMAN_RESPONSE_UNKNOWN);
			}
			vec.push_back(t);
		}

		T* operator [](unsigned int i) const
		{
			return vec[i];
		}

		unsigned int Size() const
		{
			return vec.size();
		}

		bool Empty() const
		{
			return vec.empty();
		}

		void Clear()
		{
			for(unsigned int i = 0; i < vec.size(); i++)
			{
				delete vec[i];
				vec[i] = NULL;
			}
			vec.empty();
		}
	};


} // namespace CimClassNamespace

using namespace CimClassNamespace;

template<class T>
void CimClass::Enumerate(vector<T> &vec,
						 const NameValuePairs *selectors,
						 const WsmanClient *cl)
{
	if(!cl)
	{
		throw CIMClassException("A NULL WsmanClient pointer was provided.", WSMAN_MISSING_INPUT);
	}
	vector<string > tmp;
	cl->Enumerate(T::CLASS_URI, tmp, selectors);
	for(unsigned int i = 0; i < tmp.size(); i++)
	{
		T t;
		t.Deserialize(tmp[i]);
		vec.push_back(t);
	}
}

template<class T>
void CimClass::Enumerate(CimClassContainer<T> &container,
						 const NameValuePairs *selectors,
						 const WsmanClient *cl)
{
	if(!cl)
	{
		throw CIMClassException("A NULL WsmanClient pointer was provided.", WSMAN_MISSING_INPUT);
	}
	vector<string > tmp;
	cl->Enumerate(T::CLASS_URI, tmp, selectors);
	for(unsigned int i = 0; i < tmp.size(); i++)
	{
		container.AddCimClass(tmp[i]);
	}
}
#endif
