/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "boot.h"

#include <QDebug>

Boot::Boot() : mFirstLoad(true)
{
    pluginName = tr("Boot");
    pluginType = CURRENCY;
}

Boot::~Boot()
{
}

QString Boot::plugini18nName()
{
    return pluginName;
}

int Boot::pluginTypes()
{
    return pluginType;
}

QWidget *Boot::pluginUi()
{
    if (mFirstLoad) {
        mFirstLoad = false;
        pluginWidget = new QWidget;
        pluginWidget->setAttribute(Qt::WA_DeleteOnClose);

        initUI(pluginWidget);

        mSystemDbus = new QDBusInterface("com.control.center.qt.systemdbus",
                                          "/",
                                          "com.control.center.interface",
                                          QDBusConnection::systemBus(), this);
        if (!mSystemDbus->isValid()){
            qCritical() << "Create Client Interface Failed:" << QDBusConnection::systemBus().lastError();
        }

        initConnection();

    }
    return pluginWidget;
}

const QString Boot::name() const
{
    return QStringLiteral("Boot");
}

bool Boot::isShowOnHomePage() const
{
    return true;
}

QIcon Boot::icon() const
{
    return QIcon::fromTheme("ukui-bootmenu-symbolic");
}

bool Boot::isEnable() const
{
    return true;
}

void Boot::initUI(QWidget *widget)
{
    mVlayout = new QVBoxLayout(widget);
    mVlayout->setContentsMargins(0, 0, 0, 0);

    mBootTitleLabel = new TitleLabel(pluginWidget);
    mBootTitleLabel->setText(tr("boot"));
    mBootTitleLabel->setContentsMargins(14,0,0,0);

    mBootFrame = new QFrame(pluginWidget);
    mBootFrame->setFrameShape(QFrame::Shape::Box);
    mBootFrame->setMinimumSize(550, 80);
    mBootFrame->setMaximumSize(16777215, 80);

    QHBoxLayout *bootHLayout = new QHBoxLayout();
    QVBoxLayout *leftVlayout = new QVBoxLayout();

    //~ contents_path /Boot/Grub verify
    mGrubLabel = new QLabel(tr("Grub verify"));
    mGrubLabel->setAlignment(Qt::AlignBottom);
    mGrubLabel->setMinimumWidth(140);
    mNeedPasswdLabel = new LightLabel(tr("Password required for Grub editing after enabling"), pluginWidget);
    mNeedPasswdLabel->setAlignment(Qt::AlignTop);
    mResetPasswdBtn = new QPushButton(tr("Reset password"));
    mGrubBtn = new SwitchButton();

    leftVlayout->addWidget(mGrubLabel);
    leftVlayout->addWidget(mNeedPasswdLabel);
    leftVlayout->setContentsMargins(0, 0, 0, 0);

    bootHLayout->addLayout(leftVlayout);
    bootHLayout->addStretch();
    bootHLayout->addWidget(mResetPasswdBtn);
    bootHLayout->addSpacing(7);
    bootHLayout->addWidget(mGrubBtn);

    bootHLayout->setContentsMargins(12, 0, 14, 0);

    mBootFrame->setLayout(bootHLayout);

    mVlayout->addWidget(mBootTitleLabel);
    mVlayout->addWidget(mBootFrame);
    mVlayout->addStretch();
}

void Boot::initConnection()
{

    initEnableStatus();

    connect(mGrubBtn, &SwitchButton::checkedChanged, this, &Boot::bootSlot);

    connect(mResetPasswdBtn, &QPushButton::clicked, this, &Boot::resetPasswdSlot);

}

void Boot::initEnableStatus()
{
    if (mSystemDbus != nullptr) {
        QDBusReply<bool> ret = mSystemDbus->call("getGrupPasswdStatus");
        mGrubBtn->blockSignals(true);
        mGrubBtn->setChecked(ret);
        mGrubBtn->blockSignals(false);
    }

    mResetPasswdBtn->setVisible(mGrubBtn->isChecked());
}

void Boot::bootSlot(bool checked)
{
    if (checked) {
        GrubVerify dia(pluginWidget);
        if (dia.exec() != QDialog::Accepted) {
            mGrubBtn->blockSignals(true);
            mGrubBtn->setChecked(!checked);
            mGrubBtn->blockSignals(false);
        }
    } else {
        if (mSystemDbus != nullptr) {
            QDBusReply<bool> ret = mSystemDbus->call("setGrupPasswd", "", "", false);
            if (ret == false) {
                mGrubBtn->blockSignals(true);
                mGrubBtn->setChecked(!checked);
                mGrubBtn->blockSignals(false);
                qDebug() << "call setGrupPasswd to close grub password failed!";
            }
        }
    }

    mResetPasswdBtn->setVisible(mGrubBtn->isChecked());
}

void Boot::resetPasswdSlot()
{
    GrubVerify dia(pluginWidget);
    if (dia.exec() != QDialog::Accepted) {
        qDebug() << "reset passwd failed!" << __FUNCTION__;
    }
}

