// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "annulus.h"
#include "fitsimage.h"

Annulus::Annulus(const Annulus& a) : BaseEllipse(a) {}

Annulus::Annulus(Base* p, const Vector& ctr, 
		 double r1, double r2, int rn,
		 const char* clr, int* dsh, 
		 int wth, const char* fnt, const char* txt, 
		 unsigned short prop, const char* cmt, 
		 const List<Tag>& tg, const List<CallBack>& cb)
  : BaseEllipse(p, ctr, 0, clr, dsh, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = rn+1;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++) {
    double r = i*(r2-r1)/rn+r1;
    annuli_[i] = Vector(r,r);
  }

  strcpy(type_, "annulus");
  numHandle = 4 + numAnnuli_;

  updateBBox();
}

Annulus::Annulus(Base* p, const Vector& ctr, 
		 int rn, double* r,
		 const char* clr, int* dsh, 
		 int wth, const char* fnt, const char* txt, 
		 unsigned short prop, const char* cmt, 
		 const List<Tag>& tg, const List<CallBack>& cb)
  : BaseEllipse(p, ctr, 0, clr, dsh, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = rn;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = Vector(r[i],r[i]);
  sortAnnuli();

  strcpy(type_, "annulus");
  numHandle = 4 + numAnnuli_;

  updateBBox();
}

void Annulus::edit(const Vector& v, int h)
{
  Matrix mm = bckMatrix();

  if (h<5) {
    // calc dist between edge of circle and handle
    double d = annuli_[numAnnuli_-1].length() - annuli_[numAnnuli_-1][0];

    for (int i=0; i<numAnnuli_; i++) {
      double r = ((v * mm).length() - d)/annuli_[numAnnuli_-1][0];
      annuli_[i] *= r;
    }
  }
  else {
    double d = (v * mm).length();
    annuli_[h-5] = Vector(d,d);
  }

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Annulus::editEnd()
{
  sortAnnuli();

  updateBBox();
  doCallBack(CallBack::EDITENDCB);
}

int Annulus::addAnnuli(const Vector& v)
{
  Matrix mm = bckMatrix();
  double l = (v * mm).length();

  return insertAnnuli(Vector(l,l));
}

void Annulus::analysis(AnalysisTask mm, int which)
{
  switch (mm) {
  case RADIAL:
    if (!analysisRadial_ && which) {
      addCallBack(CallBack::MOVECB, analysisRadialCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::EDITCB, analysisRadialCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::EDITENDCB, analysisRadialCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::UPDATECB, analysisRadialCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::DELETECB, analysisRadialCB_[1], 
		  parent->options->cmdName);
    }
    if (analysisRadial_ && !which) {
      deleteCallBack(CallBack::MOVECB, analysisRadialCB_[0]);
      deleteCallBack(CallBack::EDITCB, analysisRadialCB_[0]);
      deleteCallBack(CallBack::EDITENDCB, analysisRadialCB_[0]);
      deleteCallBack(CallBack::UPDATECB, analysisRadialCB_[0]);
      deleteCallBack(CallBack::DELETECB, analysisRadialCB_[1]);
    }

    analysisRadial_ = which;
    break;
  case STATS:
    if (!analysisStats_ && which) {
      addCallBack(CallBack::MOVECB, analysisStatsCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::EDITCB, analysisStatsCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::EDITENDCB, analysisStatsCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::UPDATECB, analysisStatsCB_[0], 
		  parent->options->cmdName);
      addCallBack(CallBack::DELETECB, analysisStatsCB_[1], 
		  parent->options->cmdName);
    }
    if (analysisStats_ && !which) {
      deleteCallBack(CallBack::MOVECB, analysisStatsCB_[0]);
      deleteCallBack(CallBack::EDITCB, analysisStatsCB_[0]);
      deleteCallBack(CallBack::EDITENDCB, analysisStatsCB_[0]);
      deleteCallBack(CallBack::UPDATECB, analysisStatsCB_[0]);
      deleteCallBack(CallBack::DELETECB, analysisStatsCB_[1]);
    }

    analysisStats_ = which;
    break;
  }
}

void Annulus::analysisRadial(char* xname, char* yname, char* ename,
			     Coord::CoordSystem sys)
{
  double* xx;
  double* yy;
  double* ee;

  BBox* bb = new BBox[numAnnuli_];
  for (int ii=0; ii<numAnnuli_; ii++) {
    Vector ll = -annuli_[ii] * Translate(center);
    Vector ur =  annuli_[ii] * Translate(center);
    bb[ii] = BBox(ll,ur) ;
  }

  int num = parent->markerAnalysisRadial(this, &xx, &yy, &ee, 
					 numAnnuli_-1, annuli_, 
					 bb, sys);
  analysisRadialResult(xname, yname, ename, xx, yy, ee, num);
}

void Annulus::analysisStats(Coord::CoordSystem sys)
{
  ostringstream str;
  BBox* bb = new BBox[numAnnuli_];
  for (int ii=0; ii<numAnnuli_; ii++) {
    Vector ll = -annuli_[ii] * Translate(center);
    Vector ur =  annuli_[ii] * Translate(center);
    bb[ii] = BBox(ll,ur) ;
  }
  parent->markerAnalysisStats(this, str, numAnnuli_-1, bb, sys);
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}

// list

void Annulus::list(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
		   Coord::SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(sys,center);
  listPre(str, sys, sky, ptr, strip, 0);

  switch (sys) {
  case Coord::IMAGE:
  case Coord::PHYSICAL:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] << ',';
      for (int i=0; i<numAnnuli_; i++) {
	double r = ptr->mapLenFromRef(annuli_[i][0],sys);
	str << r;
	if (i!=numAnnuli_-1)
	  str << ',';
      }
      str << ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSCel(sys)) {
	switch (format) {
	case Coord::DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] <<',';
	    for (int i=0; i<numAnnuli_; i++) {
	      double r = ptr->mapLenFromRef(annuli_[i][0],sys,Coord::ARCSEC);
	      str << r << "\"";
	      if (i!=numAnnuli_-1)
		str << ',';
	    }
	    str << ')';
	  }
	  break;
	case Coord::SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;

	    str << type_ << '(' << ra << ',' << dec << ',' ;
	    for (int i=0; i<numAnnuli_; i++) {
	      double r = ptr->mapLenFromRef(annuli_[i][0],sys,Coord::ARCSEC);
	      str << r << "\"";
	      if (i!=numAnnuli_-1)
		str << ',';
	    }
	    str << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] << ',';
	for (int i=0; i<numAnnuli_; i++) {
	  double r = ptr->mapLenFromRef(annuli_[i][0],sys);
	  str << r;
	  if (i!=numAnnuli_-1)
	    str << ',';
	}
	str << ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Annulus::listXML(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky, 
		      Coord::SkyFormat format)
{
  FitsImage* ptr = parent->findFits(sys,center);

  XMLRowInit();
  XMLRow(XMLSHAPE,type_);

  XMLRowCenter(ptr,sys,sky,format);
  XMLRowRadiusX(ptr,sys,annuli_,numAnnuli_);

  XMLRowProps(ptr,sys);
  XMLRowEnd(str);
}

void Annulus::listCiao(ostream& str, Coord::CoordSystem sys, int strip)
{
  FitsImage* ptr = parent->findFits();

  switch (sys) {
  case Coord::IMAGE:
  case Coord::PHYSICAL:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,Coord::PHYSICAL);
      for (int i=0; i<numAnnuli_-1; i++) {
	listCiaoPre(str);

	str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	    << ptr->mapLenFromRef(annuli_[i][0],Coord::PHYSICAL) << ','
	    << ptr->mapLenFromRef(annuli_[i+1][0],Coord::PHYSICAL) << ')';

	listCiaoPost(str, strip);
      }
    }
    break;
  default:
    if (ptr->hasWCSCel(sys)) {

      char buf[64];
      ptr->mapFromRef(center,sys,Coord::FK5,Coord::SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
      string x(buf);
      istringstream wcs(x);
      wcs >> ra >> dec;

      for (int i=0; i<numAnnuli_-1; i++) {
	listCiaoPre(str);

	str << type_ << '(' << ra << ',' << dec << ','
	    << ptr->mapLenFromRef(annuli_[i][0],sys,Coord::ARCMIN) << '\'' << ','
	    << ptr->mapLenFromRef(annuli_[i+1][0],sys,Coord::ARCMIN) << '\'' << ')';

	listCiaoPost(str, strip);
      }
    }
  }
}

void Annulus::listPros(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
		       Coord::SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits();

  switch (sys) {
  case Coord::IMAGE:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    sys = Coord::IMAGE;
  case Coord::PHYSICAL:
    {
      Vector v = ptr->mapFromRef(center,sys);
      coord.listProsCoordSystem(str,sys,sky);
      str << "; " << type_ << ' ' << setprecision(8) << v;
      for (int i=0; i<numAnnuli_; i++) {
	double r = ptr->mapLenFromRef(annuli_[i][0],Coord::IMAGE);
	str << r << ' ';
      }
    }
    break;
  default:
    if (ptr->hasWCSCel(sys)) {
      switch (format) {
      case Coord::DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  coord.listProsCoordSystem(str,sys,sky);
	  str << "; " << type_ << ' ' << setprecision(8) 
	      << v[0] << "d " << v[1] <<"d";
	  for (int i=0; i<numAnnuli_; i++) {
	    double r = ptr->mapLenFromRef(annuli_[i][0],sys,Coord::ARCSEC);
	    str << ' ' << r << '"';
	  }
	}
	break;
      case Coord::SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char decc[16];
	  char *dec = decc;
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    dec++;

	  coord.listProsCoordSystem(str,sys,sky);
	  str << "; " << type_ << ' ' << ra << ' ' << dec;
	  for (int i=0; i<numAnnuli_; i++) {
	    double r = ptr->mapLenFromRef(annuli_[i][0],sys,Coord::ARCSEC);
	    str << ' ' << r << '"';
	  }
	}
	break;
      }
    }
  }

  listProsPost(str, strip);
}

void Annulus::listSAOimage(ostream& str, int strip)
{
  FitsImage* ptr = parent->findFits();
  listSAOimagePre(str);

  Vector v = ptr->mapFromRef(center,Coord::IMAGE);
  str << type_ << '(' << setprecision(8) << v[0] << ',' << v[1] << ',';

  for (int i=0; i<numAnnuli_; i++) {
    double r = ptr->mapLenFromRef(annuli_[i][0],Coord::IMAGE);
    str << r;
    if (i!=numAnnuli_-1)
      str << ',';
  }

  str << ')';
  listSAOimagePost(str, strip);
}

// special composite funtionallity

void Annulus::setComposite(const Matrix& mx, double aa)
{
  center *= mx;
  updateBBox();
}
