/*
*  RAL -- Rubrica Addressbook Library
*  file: abook.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <glib-object.h>

#include "form.h"

struct _RubricaFormPrivate {
  gboolean dispose_has_run;
};


static void rubrica_form_class_init (RubricaFormClass* klass);
static void rubrica_form_init       (RubricaForm* obj);

static void rubrica_form_dispose    (RubricaForm* obj);
static void rubrica_form_finalize   (RubricaForm* obj);


GType
rubrica_form_get_type()
{
  static GType rubrica_form_type = 0;
  
  if (!rubrica_form_type)
    {
      static const GTypeInfo rubrica_form_info =
	{
	  sizeof(RubricaFormClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_form_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaForm),
	  0,
	  (GInstanceInitFunc) rubrica_form_init
	};

      rubrica_form_type = g_type_register_static (G_TYPE_OBJECT, 
						   "RubricaForm",
						   &rubrica_form_info, 0);
    }
  
  return rubrica_form_type;
}


static void
rubrica_form_class_init(RubricaFormClass* klass)
{
  GObjectClass *class;
 
  class  = G_OBJECT_CLASS (klass);

  class->dispose      = (GObjectFinalizeFunc) rubrica_form_dispose;
  class->finalize     = (GObjectFinalizeFunc) rubrica_form_finalize;
  
  klass->extract_data = NULL;
  klass->display_data = NULL;
  klass->clean_form   = NULL;
}


static void
rubrica_form_init(RubricaForm* self)
{
  self->private = g_new(RubricaFormPrivate, 1);

  self->private->dispose_has_run = FALSE;
}

 

static void 
rubrica_form_dispose (RubricaForm* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  if (self->private->dispose_has_run)
    return;

  self->private->dispose_has_run = TRUE;
}


static void 
rubrica_form_finalize (RubricaForm* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  g_free(self->private);
  self->private = NULL;
}





/*   ***************************** Public *****************************
*/


/**
 * rubrica_form_new
 *
 * create a #RubricaForm
 *
 * Returns: a new #RubricaForm
 */
RubricaForm*
rubrica_form_new(void)
{
  RubricaForm* form;

  form = g_object_new(rubrica_form_get_type(), NULL);
  
  return form;
}


/**
 * rubrica_form_free
 * @form: a #RubricaForm
 * 
 * free the #RubricaForm
 */
void
rubrica_form_free(RubricaForm* form)
{
  g_return_if_fail(IS_RUBRICA_FORM(form));

  g_object_unref(form);   
}



/**
 * rubrica_form_clean_form
 * @form: a #RubricaForm
 *
 * clean the form
 */
void 
rubrica_form_clean_form(RubricaForm *form)
{
  RubricaFormClass* class;
  
  g_return_if_fail(IS_RUBRICA_FORM(form));

  class = RUBRICA_FORM_GET_CLASS(form);
  
  if (class->clean_form)
    class->clean_form(form); 
}



/**
 * rubrica_form_extract_data
 * @form: a #RubricaForm
 * @card: a #RubricaCard
 * 
 * extract data from form and fill the card
 */
void
rubrica_form_extract_data (RubricaForm* form, RCard* card)
{
  RubricaFormClass* class;

  g_return_if_fail(IS_RUBRICA_FORM(form));

  class = RUBRICA_FORM_GET_CLASS(form);
  
  if (class->extract_data)
    class->extract_data(form, card); 
}


/**
 * rubrica_form_display_data
 * @form: a #RubricaForm
 * @card: a #RubricaCard
 * 
 * dispaly the cars's data into the form
 */
void 
rubrica_form_display_data (RubricaForm* form, RCard* card) 
{
  RubricaFormClass* class;

  g_return_if_fail(IS_RUBRICA_FORM(form));

  class = RUBRICA_FORM_GET_CLASS(form);
  
  if (class->display_data)
    class->display_data(form, card); 
}


