#ifndef _RHEO_MASS_H
#define _RHEO_MASS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:mass
NAME: @code{mass} -- L2 scalar product
@bfindex mass
@bfindex inv_mass
@apindex P0
@apindex P1
@apindex P2
@apindex P1d
@apindex bubble
@clindex domain
@clindex space
@cindex  scalar product
@cindex  lumped mass form
@cindex  boundary conditions
@cindex  Neumann boundary conditions
@cindex  Robin boundary conditions
SYNOPSIS:
    @example
        form(const space& V, const space& V, "mass");
  	form(const space& M, const space& V, "mass");
        form (const space& V, const space& V, "mass", const domain& gamma);
        form_diag(const space& V, "mass");
    @end example
DESCRIPTION:       
    @noindent
    Assembly the matrix associated to the
    L2 scalar product of the finite element space V.
@iftex
@tex
    $$
   	m(u,v) = \int_\Omega  u v \, dx
    $$
@end tex
@end iftex
@ifnottex
    @example
                 /
                 |
   	m(u,v) = | u v dx
                 |
                 / Omega
    @end example
@end ifnottex
    
    @noindent
    The V space may be either a 
    @code{P0}, @code{P1}, @code{P2},
    @code{bubble}, @code{P1d}
    and
    @code{P1d}
    finite 
    element spaces for building a form
    @pxref{form class}.

    @noindent
    The use of quadrature formulae is sometime usefull
    for building diagonal matrix.
    These approximate matrix are eay to invert.
    This procedure is available
    for @code{P0} and @code{P1} approximations.
    
    @noindent
    Notes that when dealing with discontinuous finite
    element space, i.e.
    @code{P0} and @code{P1d}, the corresponding 
    mass matrix is block diagonal, and the
    @code{inv_mass} form may be usefull.

    When two different space M and V are supplied,
    assembly the matrix associated to the
    projection operator from one
    finite element space M to space V.
@iftex
  $$
  	m(q,v) = \int_\Omega  q v \, dx
  $$
  for all $q \in M$ and  $v \in V$.
@end iftex
@ifnottex
  @example
                 /
                 |
   	m(q,v) = | q v dx
                 |
                 / Omega
  @end example
  for all q in M and v in V.
@end ifnottex
  
  @noindent
  This form is usefull for instance to convert discontinuous gradient
  components to a continuous approximation.
  The transpose operator may also be usefull to performs
  the opposite operation.

  @noindent
  The following $V$ and $M$ space approximation
  combinations are supported for the @code{mass} form:
    P0-P1,
    P0-P1d,
    P1d-P2,
    P1-P1d
    and
    P1-P2.

EXAMPLE:
    @noindent
    The following piece of code build the mass matrix
    associated to the P1 approximation:
    @example
	geo g("square");
   	space V(g, "P1");
   	form m(V, V, "mass");
    @end example
    @noindent
    The use of lumped mass form write also:
    @example
   	form_diag md(V, "mass");
    @end example
  @noindent
  The following piece of code build the projection form:
  @example
  	geo g("square");
  	space V(g, "P1");
  	space M(g, "P0");
  	form m(M, V, "mass");
  @end example

SCALAR PRODUCT ON THE BOUNDARY:
    @noindent
    Assembly the matrix associated to the
    L2 scalar product related to a boundary domain
    of a mesh and a specified polynomial approximation.
    These forms are usefull when defining non-homogeneous Neumann or
    Robin boundary conditions.

    @noindent
    Let W be a space of functions defined on Gamma, a subset of the
    boundary of the whole domain Omega.
@iftex
    $$
   	m(u,v) = \int_\Gamma  u \, v \, ds, \ \ \forall u, v \in W
    $$
@end iftex
@ifnottex
    @example
                 /
                 |
   	m(u,v) = | u v dx
                 |
                 / Gamma
    @end example
    for all u, v in W.
@end ifnottex
    Let V a space of functions defined on Omega
    and gamma the trace operator from V into W.
@iftex
    $$
   	mb(u,v) = \int_\Gamma  u \, \gamma v \, ds, \ \ \forall u \in W, v \in V
    $$
    $$
   	ab(u,v) = \int_\Gamma  \gamma u \, \gamma v \, ds, \ \ \forall u, v \in V
    $$
@end iftex
@ifnottex
    For all u in W and v in V:
    @example
                  /
                  |
   	mb(u,v) = | u gamma(v) dx
                  |
                  / Gamma
    @end example
    For all u and v in V:
    @example
                  /
                  |
   	ab(u,v) = | gamma(u) gamma(v) dx
                  |
                  / Gamma
    @end example
@end ifnottex

EXAMPLE:
    @noindent
    The following piece of code build forms for the P1 approximation,
    assuming that the mesh contains a domain named @code{boundary}:
    @example
	geo omega ("square");
        domain gamma = omega.boundary();
   	space V  (omega, "P1");
   	space W  (omega, gamma, "P1");
   	form  m  (W, W, "mass");
   	form  mb (W, V, "mass");
   	form  ab (V, V, "mass", gamma);
    @end example
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
DATE:
    27 november 1997
End:
*/

#include "rheolef/form_element_rep.h"
namespace rheolef { 

class mass: public form_element_rep {
public:
// allocator:

    mass() : form_element_rep() {}

// virtual accessor:

    void operator() (const geo_element& K, ublas::matrix<Float>& m) const;
    void check_after_initialize () const;
};
}// namespace rheolef
#endif // _RHEO_MASS_H
