#ifndef _RHEOLEF_BANDED_LEVEL_SET_H
#define _RHEOLEF_BANDED_LEVEL_SET_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/// Banded level set routines 
///
/// Authors: Lara Aborm, Jocelyn Etienne, Pierre Saramito
///
#include "rheolef/field.h"

namespace rheolef {
class quadruplet {
public:
    quadruplet (size_t a=0, size_t b=0, size_t c=0, size_t d=0) {
        q[0]=a;
        q[1]=b;
        q[2]=c;
        q[3]=d;
    }
    size_t  operator[] (size_t i) const {
        return q[i%4];
    }
    size_t& operator[] (size_t i)       {
        return q[i%4];
    }
    friend std::ostream& operator<< (std::ostream& out, const quadruplet& q) {
        out << "((" << q[0] << "," << q[1] << "), (" << q[2] << "," << q[3] << "))";
        return out;
    }
protected:
    size_t q[4];
};

// given a level set function, returns the space on band 
void banded_level_set (const field& phi, geo& geo_band, domain& dom_band);
// given a level set function, returns the band geometry
geo banded_level_set (const field& phi);

// given a level set function, returns the zero level set geometry
geo zero_level_set (const field& phi);
geo zero_level_set (const field& phi, std::vector<size_t>& element2face);

// for direct solver assembly (see helmholtz_band.cc example):
size_t build_vertex_connex_component (geo& band);

// used for projection/extension between Gamma_h and beta_h fields
class band {
public:

// allocator:

   band (field phi_h_lambda);

// accessors:

   const field& get_phi         () const { return _phi_h_beta; }
   const geo&   get_geo         () const { return _phi_h_beta.get_geo(); }
   const space& get_space       () const { return _phi_h_beta.get_space(); }
   const field& get_bgd_phi     () const { return _phi_h_lambda; }
   const geo&   get_bgd_geo     () const { return _phi_h_lambda.get_geo(); }
   const space& get_bgd_space   () const { return _phi_h_lambda.get_space(); }
   const geo&   get_surface_geo () const { return _gamma_h; }
   size_t element2face (size_t K_idx) const;

// data:
protected:
   field _phi_h_lambda;
   field _phi_h_beta;
   geo   _gamma_h;
   std::vector<size_t> _element2face;
};
inline
size_t
band::element2face (size_t K_idx) const
{
    check_macro (K_idx < _element2face.size(),
	"element2face: element index " << K_idx << " out of range [0:"<< _element2face.size());
    return _element2face[K_idx];
}

} // namespace
#endif // _RHEOLEF_BANDED_LEVEL_SET_H
