/*
 *  This file is part of RawTherapee.
 *
 *  Copyright (c) 2004-2010 Gabor Horvath <hgabor@rawtherapee.com>
 *
 *  RawTherapee is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  RawTherapee is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with RawTherapee.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <glibmm/ustring.h>

#include "colortemp.h"
#include "iccmatrices.h"
#include "rt_math.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "sleef.h"
#include "settings.h"
#include "iccstore.h"


namespace rtengine
{

static const color_match_type cie_colour_match_jd2 = {//350nm to 830nm   5 nm J.Desmis 2° Standard Observer.
    {0.0000000, 0.000000, 0.000000}, {0.0000000, 0.000000, 0.000000}, {0.0001299, 0.0003917, 0.0006061},
    {0.0002321, 0.000006965, 0.001086}, {0.0004149, 0.00001239, 0.001946}, {0.0007416, 0.00002202, 0.003846},
    {0.001368, 0.000039, 0.006450001}, {0.002236, 0.000064, 0.01054999}, {0.004243, 0.000120, 0.02005001},
    {0.007650, 0.000217, 0.036210}, {0.014310, 0.000396, 0.06785001}, {0.023190, 0.000640, 0.110200},
    {0.043510, 0.001210, 0.207400}, {0.077630, 0.002180, 0.371300}, {0.134380, 0.004000, 0.645600},
    {0.214770, 0.007300, 1.0390501}, {0.283900, 0.011600, 1.385600}, {0.328500, 0.016840, 1.622960},
    {0.348280, 0.023000, 1.747060}, {0.348060, 0.029800, 1.782600}, {0.336200, 0.038000, 1.772110},
    {0.318700, 0.048000, 1.744100}, {0.290800, 0.060000, 1.669200}, {0.251100, 0.073900, 1.528100},
    {0.195360, 0.090980, 1.287640}, {0.142100, 0.112600, 1.041900}, {0.095640, 0.139020, 0.8129501},
    {0.05795001, 0.169300, 0.616200}, {0.032010, 0.208020, 0.465180}, {0.014700, 0.258600, 0.353300},
    {0.004900, 0.323000, 0.272000}, {0.002400, 0.407300, 0.212300}, {0.009300, 0.503000, 0.158200},
    {0.029100, 0.608200, 0.111700}, {0.063270, 0.710000, 0.07824999}, {0.109600, 0.793200, 0.05725001},
    {0.165500, 0.862000, 0.042160}, {0.2257499, 0.9148501, 0.029840}, {0.290400, 0.954000, 0.020300},
    {0.359700, 0.980300, 0.013400}, {0.43344990, 0.9949501, 0.008749999}, {0.5120501, 1.000000, 0.005749999},
    {0.594500, 0.995000, 0.003900}, {0.678400, 0.978600, 0.002749999}, {0.762100, 0.952000, 0.002100},
    {0.842500, 0.915400, 0.001800}, {0.916300, 0.870000, 0.001650001}, {0.978600, 0.816300, 0.001400},
    {1.026300, 0.757000, 0.001100}, {1.056700, 0.694900, 0.001000}, {1.062200, 0.631000, 0.000800},
    {1.045600, 0.566800, 0.000600}, {1.002600, 0.503000, 0.000340}, {0.938400, 0.441200, 0.000240},
    {0.8544499, 0.381000, 0.000190}, {0.751400, 0.321000, 0.000100}, {0.642400, 0.265000, 0.00004999999},
    {0.541900, 0.217000, 0.000030}, {0.447900, 0.175000, 0.000020}, {0.360800, 0.138200, 0.000010},
    {0.283500, 0.107000, 0.000000}, {0.218700, 0.081600, 0.000000}, {0.164900, 0.061000, 0.000000},
    {0.121200, 0.044580, 0.000000}, {0.087400, 0.032000, 0.000000}, {0.063600, 0.023200, 0.000000},
    {0.046770, 0.017000, 0.000000}, {0.032900, 0.011920, 0.000000}, {0.022700, 0.008210, 0.000000},
    {0.015840, 0.005723, 0.000000}, {0.01135916, 0.004102, 0.000000}, {0.008110916, 0.002929, 0.000000},
    {0.005790346, 0.002091, 0.000000}, {0.004109457, 0.001484, 0.000000}, {0.002899327, 0.001047, 0.000000},
    {0.00204919, 0.000740, 0.000000}, {0.001439971, 0.000520, 0.000000}, {0.0009999493, 0.0003611, 0.000000},
    {0.0006900786, 0.0002492, 0.000000}, {0.0004760213, 0.0001719, 0.000000}, {0.0003323011, 0.000120, 0.000000},
    {0.0002348261, 0.0000848, 0.000000}, {0.0001661505, 0.000060, 0.000000}, {0.000117413, 0.0000424, 0.000000},
    {0.00008307527, 0.000030, 0.000000}, {0.00005870652, 0.0000212, 0.000000}, {0.00004150994, 0.00001499, 0.000000},
    {0.00002935326, 0.0000106, 0.000000}, {0.00002067383, 0.0000074657, 0.000000}, {0.00001455977, 0.0000052578, 0.000000},
    {0.00001025398, 0.0000037029, 0.000000}, {0.000007221456, 0.00000260778, 0.000000}, {0.000005085868, 0.0000018366, 0.000000},
    {0.000003581652, 0.0000012934, 0.000000}, {0.000002522525, 0.00000091093, 0.000000}, {0.000001776509, 0.00000064153, 0.000000},
    {0.000001251141, 0.00000045181, 0.000000}
};


static const color_match_type cie_colour_match_jd = {//350nm to 830nm   5 nm J.Desmis 10° Standard Observer.
    {0.000000000000, 0.000000000000, 0.000000000000},
    {0.000000000000, 0.000000000000, 0.000000000000},
    {0.000000122200, 0.000000013398, 0.000000535027},
    {0.000000919270, 0.000000100650, 0.000004028300},
    {0.000005958600, 0.000000651100, 0.000026143700},
    {0.000033266000, 0.000003625000, 0.000146220000},
    {0.000159952000, 0.000017364000, 0.000704776000},
    {0.000662440000, 0.000071560000, 0.002927800000},
    {0.002361600000, 0.000253400000, 0.010482200000},
    {0.007242300000, 0.000768500000, 0.032344000000},
    {0.019109700000, 0.002004400000, 0.086010900000},
    {0.043400000000, 0.004509000000, 0.197120000000},
    {0.084736000000, 0.008756000000, 0.389366000000},
    {0.140638000000, 0.014456000000, 0.656760000000},
    {0.204492000000, 0.021391000000, 0.972542000000},
    {0.264737000000, 0.029497000000, 1.282500000000},
    {0.314679000000, 0.038676000000, 1.553480000000},
    {0.357719000000, 0.049602000000, 1.798500000000},
    {0.383734000000, 0.062077000000, 1.967280000000},
    {0.386726000000, 0.074704000000, 2.027300000000},
    {0.370702000000, 0.089456000000, 1.994800000000},
    {0.342957000000, 0.106256000000, 1.900700000000},
    {0.302273000000, 0.128201000000, 1.745370000000},
    {0.254085000000, 0.152761000000, 1.554900000000},
    {0.195618000000, 0.185190000000, 1.317560000000},
    {0.132349000000, 0.219940000000, 1.030200000000},
    {0.080507000000, 0.253589000000, 0.772125000000},
    {0.041072000000, 0.297665000000, 0.570060000000},
    {0.016172000000, 0.339133000000, 0.415254000000},
    {0.005132000000, 0.395379000000, 0.302356000000},
    {0.003816000000, 0.460777000000, 0.218502000000},
    {0.015444000000, 0.531360000000, 0.159249000000},
    {0.037465000000, 0.606741000000, 0.112044000000},
    {0.071358000000, 0.685660000000, 0.082248000000},
    {0.117749000000, 0.761757000000, 0.060709000000},
    {0.172953000000, 0.823330000000, 0.043050000000},
    {0.236491000000, 0.875211000000, 0.030451000000},
    {0.304213000000, 0.923810000000, 0.020584000000},
    {0.376772000000, 0.961988000000, 0.013676000000},
    {0.451584000000, 0.982200000000, 0.007918000000},
    {0.529826000000, 0.991761000000, 0.003988000000},
    {0.616053000000, 0.999110000000, 0.001091000000},
    {0.793832000000, 0.982380000000, 0.000000000000},
    {0.878655000000, 0.955552000000, 0.000000000000},
    {0.951162000000, 0.915175000000, 0.000000000000},
    {1.014160000000, 0.868934000000, 0.000000000000},
    {1.074300000000, 0.825623000000, 0.000000000000},
    {1.118520000000, 0.777405000000, 0.000000000000},
    {1.134300000000, 0.720353000000, 0.000000000000},
    {1.123990000000, 0.658341000000, 0.000000000000},
    {1.089100000000, 0.593878000000, 0.000000000000},
    {1.030480000000, 0.527963000000, 0.000000000000},
    {0.950740000000, 0.461834000000, 0.000000000000},
    {0.856297000000, 0.398057000000, 0.000000000000},
    {0.754930000000, 0.339554000000, 0.000000000000},
    {0.647467000000, 0.283493000000, 0.000000000000},
    {0.535110000000, 0.228254000000, 0.000000000000},
    {0.431567000000, 0.179828000000, 0.000000000000},
    {0.343690000000, 0.140211000000, 0.000000000000},
    {0.268329000000, 0.107633000000, 0.000000000000},
    {0.204300000000, 0.081187000000, 0.000000000000},
    {0.152568000000, 0.060281000000, 0.000000000000},
    {0.112210000000, 0.044096000000, 0.000000000000},
    {0.081260600000, 0.031800400000, 0.000000000000},
    {0.057930000000, 0.022601700000, 0.000000000000},
    {0.040850800000, 0.015905100000, 0.000000000000},
    {0.028623000000, 0.011130300000, 0.000000000000},
    {0.019941300000, 0.007748800000, 0.000000000000},
    {0.013842000000, 0.005375100000, 0.000000000000},
    {0.009576880000, 0.003717740000, 0.000000000000},
    {0.006605200000, 0.002564560000, 0.000000000000},
    {0.004552630000, 0.001768470000, 0.000000000000},
    {0.003144700000, 0.001222390000, 0.000000000000},
    {0.002174960000, 0.000846190000, 0.000000000000},
    {0.001505700000, 0.000586440000, 0.000000000000},
    {0.001044760000, 0.000407410000, 0.000000000000},
    {0.000727450000, 0.000284041000, 0.000000000000},
    {0.000508258000, 0.000198730000, 0.000000000000},
    {0.000356380000, 0.000139550000, 0.000000000000},
    {0.000250969000, 0.000098428000, 0.000000000000},
    {0.000177730000, 0.000069819000, 0.000000000000},
    {0.000126390000, 0.000049737000, 0.000000000000},
    {0.000090151000, 0.000035540500, 0.000000000000},
    {0.000064525800, 0.000025486000, 0.000000000000},
    {0.000046339000, 0.000018338400, 0.000000000000},
    {0.000033411700, 0.000013249000, 0.000000000000},
    {0.000024209000, 0.000009619600, 0.000000000000},
    {0.000017611500, 0.000007012800, 0.000000000000},
    {0.000012855000, 0.000005129800, 0.000000000000},
    {0.000009413630, 0.000003764730, 0.000000000000},
    {0.000006913000, 0.000002770810, 0.000000000000},
    {0.000005093470, 0.000002046130, 0.000000000000},
    {0.000003767100, 0.000001516770, 0.000000000000},
    {0.000002795310, 0.000001128090, 0.000000000000},
    {0.000002082000, 0.000000842160, 0.000000000000},
    {0.000001553140, 0.000000629700, 0.000000000000}
};



ColorTemp::ColorTemp(double t, double g, double e, const std::string &m, StandardObserver o) : temp(t), green(g), equal(e), method(m), observer(o)
{
    clip(temp, green, equal);
}

void ColorTemp::clip(double &temp, double &green)
{
    temp = rtengine::LIM(temp, MINTEMP, MAXTEMP);
    green = rtengine::LIM(green, MINGREEN, MAXGREEN);
}

void ColorTemp::clip(double &temp, double &green, double &equal)
{
    temp = rtengine::LIM(temp, MINTEMP, MAXTEMP);
    green = rtengine::LIM(green, MINGREEN, MAXGREEN);
    equal = rtengine::LIM(equal, MINEQUAL, MAXEQUAL);
}

ColorTemp::ColorTemp(double mulr, double mulg, double mulb, double e, StandardObserver observer) : equal(e), method("Custom"), observer(observer)
{
    mul2temp(mulr, mulg, mulb, equal, observer, temp, green);
}

ColorTemp ColorTemp::convertObserver(StandardObserver observer) const
{
    if (observer == this->observer) {
        return *this;
    }

    double r;
    double g;
    double b;
    getMultipliers(r, g, b);
    return ColorTemp(r, g, b, equal, observer);
}

void ColorTemp::mul2temp(const double rmul, const double gmul, const double bmul, const double equal, StandardObserver observer, double& temp, double& green) const
{

    double maxtemp = MAXTEMP, mintemp = MINTEMP;
    double tmpr, tmpg, tmpb;
    temp = (maxtemp + mintemp) / 2;

    while (maxtemp - mintemp > 1) {
        temp2mul(temp, 1.0, equal, observer, tmpr, tmpg, tmpb);

        if (tmpb / tmpr > bmul / rmul) {
            maxtemp = temp;
        } else {
            mintemp = temp;
        }

        temp = (maxtemp + mintemp) / 2;
    }

    green = (tmpg / tmpr) / (gmul / rmul);
    clip(temp, green);
}


// spectral data for Daylight direct Sun: I have choose 5300K because Nikon=5200K, Olympus=5300K, Panasonic=5500K, Leica=5400K, Minolta=5100K
const double ColorTemp::Daylight5300_spect[97] = {
    24.82, 26.27, 27.72, 28.97, 30.22, 29.71, 29.19, 31.95, 34.71, 45.49, 56.26, 59.97, 63.68, 65.30, 66.92, 65.39, 63.86, 72.59, 81.32, 87.53, 93.73, 95.15, 96.56, 96.55, 96.54, 98.13, 99.73, 97.70, 95.66, 97.19, 98.72,
    98.90, 99.08, 98.98, 98.87, 101.13, 103.39, 102.48, 101.57, 102.14, 102.71, 101.36, 100.00, 98.71, 97.42, 97.81, 98.21, 95.20, 92.20, 93.92, 95.63, 96.15, 96.67, 96.34, 96.01, 94.21, 92.41, 93.58, 94.74, 93.05, 91.36, 92.29,
    93.21, 95.25, 97.28, 95.30, 93.32, 87.92, 82.51, 84.29, 86.06, 86.94, 87.81, 80.24, 72.68, 77.32, 81.96, 84.88, 87.79, 81.01, 74.22, 64.41, 54.60, 66.55, 78.51, 76.35, 74.20, 74.79, 75.38, 72.48, 69.58, 65.11, 60.64,
    63.88, 67.13, 68.85, 70.57
};

//spectral data for Daylight Cloudy: I have choose 6200K because Nikon=6000K, Olympus=6000K, Panasonic=6200K, Leica=6400K, Minolta=6500K
const double ColorTemp::Cloudy6200_spect[97] = {
    39.50, 40.57, 41.63, 43.85, 46.08, 45.38, 44.69, 47.20, 49.71, 63.06, 76.41, 80.59, 84.77, 85.91, 87.05, 84.14, 81.23, 90.29, 99.35, 105.47, 111.58, 112.23, 112.87, 111.74, 110.62, 111.41, 112.20, 108.98, 105.76, 106.32,
    106.89, 106.34, 105.79, 104.62, 103.45, 105.09, 106.72, 105.24, 103.76, 103.75, 103.75, 101.87, 100.00, 98.29, 96.58, 96.46, 96.34, 92.85, 89.37, 90.25, 91.12, 91.06, 90.99, 90.17, 89.35, 87.22, 85.10, 85.48, 85.85,
    84.03, 82.20, 82.45, 82.69, 83.92, 85.15, 83.14, 81.13, 76.65, 72.17, 73.27, 74.36, 75.65, 76.95, 70.34, 63.74, 67.98, 72.22, 74.88, 77.54, 71.59, 65.65, 56.82, 47.99, 58.53, 69.06, 67.27, 65.47, 65.96, 66.44, 63.92, 61.41, 57.52,
    53.63, 56.47, 59.31, 60.80, 62.29
};

//spectral data for Daylight Shade: I have choose 7600K because Nikon=8000K, Olympus=7500K, Panasonic=7500K, Leica=7500K, Minolta=7500K
const double ColorTemp::Shade7600_spect[97] = {
    64.42, 64.46, 64.51, 68.35, 72.20, 70.22, 68.24, 69.79, 71.35, 87.49, 103.64, 108.68, 113.72, 114.12, 114.53, 109.54, 104.55, 113.59, 122.63, 128.52, 134.41, 134.02, 133.63, 131.02, 128.41, 128.08, 127.75, 123.16,
    118.57, 117.89, 117.22, 115.72, 114.22, 111.60, 108.99, 109.84, 110.68, 108.57, 106.45, 105.71, 104.98, 102.49, 100.00, 97.78, 95.55, 94.82, 94.08, 90.47, 86.87, 86.94, 87.01, 86.45, 85.88, 84.57, 83.27, 80.83, 78.40, 78.21,
    78.03, 76.22, 74.42, 74.15, 73.89, 74.41, 74.92, 73.01, 71.09, 67.26, 63.42, 64.01, 64.60, 66.10, 67.60, 61.83, 56.06, 59.94, 63.82, 66.27, 68.71, 63.49, 58.26, 50.30, 42.34, 51.64, 60.95, 59.45, 57.95, 58.35, 58.76, 56.57,
    54.38, 51.00, 47.62, 50.10, 52.58, 53.88, 55.19
};

//spectral data for tungsten - incandescent 2856K
const double ColorTemp::A2856_spect[97] = {
    4.75, 5.42, 6.15, 6.95, 7.83, 8.78, 9.80, 10.91, 12.09, 13.36, 14.72, 16.16, 17.69, 19.30, 21.01, 22.80, 24.68, 26.65, 28.71, 30.86, 33.10, 35.42, 37.82, 40.31, 42.88, 45.53, 48.25, 51.05, 53.92, 56.87, 59.87, 62.94, 66.07, 69.26, 72.50,
    75.80, 79.14, 82.53, 85.95, 89.42, 92.91, 96.44, 100.00, 103.58, 107.18, 110.80, 114.43, 118.07, 121.72, 125.38, 129.03, 132.68, 136.33, 139.97, 143.60, 147.21, 150.81, 154.39, 157.95, 161.48, 164.99, 168.47, 171.92, 175.34,
    178.72, 182.07, 185.38, 188.65, 191.88, 195.06, 198.20, 201.30, 204.34, 207.34, 210.29, 213.19, 216.04, 218.84, 221.58, 224.28, 226.91, 229.49, 232.02, 234.49, 236.91, 239.27, 241.57, 243.82, 246.01, 248.14, 250.21, 252.23, 254.19,
    256.10, 257.95, 259.74, 261.47
};

//spectral data for fluo  F1 Daylight 6430K
const double ColorTemp::FluoF1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.87, 2.36, 2.94, 3.47, 5.17, 19.49, 6.13, 6.24, 7.01, 7.79, 8.56, 43.67, 16.94, 10.72, 11.35, 11.89, 12.37, 12.75, 13.00, 13.15, 13.23, 13.17, 13.13, 12.85, 12.52,
    12.20, 11.83, 11.50, 11.22, 11.05, 11.03, 11.18, 11.53, 27.74, 17.05, 13.55, 14.33, 15.01, 15.52, 18.29, 19.55, 15.48, 14.91, 14.15, 13.22, 12.19, 11.12, 10.03, 8.95, 7.96, 7.02, 6.20, 5.42, 4.73, 4.15, 3.64, 3.20, 2.81,
    2.47, 2.18, 1.93, 1.72, 1.67, 1.43, 1.29, 1.19, 1.08, 0.96, 0.88, 0.81, 0.77, 0.75, 0.73, 0.68, 0.69, 0.64, 0.68, 0.69, 0.61, 0.52, 0.43, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo  F2 Cool white 4230K
const double ColorTemp::FluoF2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.18, 1.48, 1.84, 2.15, 3.44, 15.69, 3.85, 3.74, 4.19, 4.62, 5.06, 34.98, 11.81, 6.27, 6.63, 6.93, 7.19, 7.40, 7.54, 7.62, 7.65, 7.62, 7.62, 7.45, 7.28, 7.15, 7.05, 7.04, 7.16, 7.47, 8.04, 8.88, 10.01, 24.88, 16.64, 14.59, 16.16, 17.60, 18.62, 21.47, 22.79, 19.29, 18.66, 17.73, 16.54, 15.21, 13.80, 12.36, 10.95, 9.65, 8.40, 7.32, 6.31, 5.43, 4.68, 4.02, 3.45,
    2.96, 2.55, 2.19, 1.89, 1.64, 1.53, 1.27, 1.10, 0.99, 0.88, 0.76, 0.68, 0.61, 0.56, 0.54, 0.51, 0.47, 0.47, 0.43, 0.46, 0.47, 0.40, 0.33, 0.27, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo  F3 White 3450K
const double ColorTemp::FluoF3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.82, 1.02, 1.26, 1.44, 2.57, 14.36, 2.70, 2.45, 2.73, 3.00, 3.28, 31.85, 9.47, 4.02, 4.25, 4.44, 4.59, 4.72, 4.80, 4.86, 4.87, 4.85, 4.88, 4.77, 4.67, 4.62, 4.62, 4.73, 4.99, 5.48, 6.25,
    7.34, 8.78, 23.82, 16.14, 14.59, 16.63, 18.49, 19.95, 23.11, 24.69, 21.41, 20.85, 19.93, 18.67, 17.22, 15.65, 14.04, 12.45, 10.95, 9.51, 8.27, 7.11, 6.09, 5.22, 4.45, 3.80, 3.23, 2.75, 2.33, 1.99, 1.70, 1.55,
    1.27, 1.09, 0.96, 0.83, 0.71, 0.62, 0.54, 0.49, 0.46, 0.43, 0.39, 0.39, 0.35, 0.38, 0.39, 0.33, 0.28, 0.21, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F4 Warm white 2940K
const double ColorTemp::FluoF4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.57, 0.70, 0.87, 0.98, 2.01, 13.75, 1.95, 1.59, 1.76, 1.93, 2.10, 30.28, 8.03, 2.55, 2.70, 2.82, 2.91, 2.99, 3.04, 3.08, 3.09, 3.09, 3.14, 3.06, 3.00, 2.98, 3.01,
    3.14, 3.41, 3.90, 4.69, 5.81, 7.32, 22.59, 15.11, 13.88, 16.33, 18.68, 20.64, 24.28, 26.26, 23.28, 22.94, 22.14, 20.91, 19.43, 17.74, 16.00, 14.42, 12.56, 10.93, 9.52, 8.18, 7.01, 6.00, 5.11, 4.36, 3.69, 3.13, 2.64,
    2.24, 1.91, 1.70, 1.39, 1.18, 1.03, 0.88, 0.74, 0.64, 0.54, 0.49, 0.46, 0.42, 0.37, 0.37, 0.33, 0.35, 0.36, 0.31, 0.26, 0.19, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F5 Daylight 6350K
const double ColorTemp::FluoF5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.87, 2.35, 2.92, 3.45, 5.10, 18.91, 6.00, 6.11, 6.85, 7.58, 8.31, 40.76, 16.06, 10.32, 10.91, 11.40, 11.83, 12.17, 12.40, 12.54, 12.58, 12.52, 12.47, 12.20, 11.89,
    11.61, 11.33, 11.10, 10.96, 10.97, 11.16, 11.54, 12.12, 27.78, 17.73, 14.47, 15.20, 15.77, 16.10, 18.54, 19.50, 15.39, 14.64, 13.72, 12.69, 11.57, 10.45, 9.35, 8.29, 7.32, 6.41, 5.63, 4.90, 4.26,
    3.72, 3.25, 2.83, 2.49, 2.19, 1.93, 1.71, 1.52, 1.43, 1.26, 1.13, 1.05, 0.96, 0.85, 0.78, 0.72, 0.68, 0.67, 0.65, 0.61, 0.62, 0.59, 0.62, 0.64, 0.55, 0.47, 0.40, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F6 Lite white 4150K
const double ColorTemp::FluoF6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.05, 1.31, 1.63, 1.90, 3.11, 14.8, 3.43, 3.30, 3.68, 4.07, 4.45, 32.61, 10.74, 5.48, 5.78, 6.03, 6.25, 6.41, 6.52, 6.58, 6.59, 6.56, 6.56, 6.42, 6.28, 6.20, 6.19, 6.30, 6.60, 7.12, 7.94, 9.07, 10.49, 25.22, 17.46, 15.63, 17.22, 18.53,
    19.43, 21.97, 23.01, 19.41, 18.56, 17.42, 16.09, 14.64, 13.15, 11.68, 10.25, 8.96, 7.74, 6.69, 5.71, 4.87, 4.16, 3.55, 3.02, 2.57, 2.20, 1.87, 1.60, 1.37, 1.29, 1.05, 0.91, 0.81, 0.71, 0.61, 0.54, 0.48, 0.44,
    0.43, 0.40, 0.37, 0.38, 0.35, 0.39, 0.41, 0.33, 0.26, 0.21, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F7 D65 Daylight simulator 6500K
const double ColorTemp::FluoF7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.56, 3.18, 3.84, 4.53, 6.15, 19.37, 7.37, 7.05, 7.71, 8.41, 9.15, 44.14, 17.52, 11.35, 12.00, 12.58, 13.08, 13.45, 13.71, 13.88, 13.95, 13.93, 13.82, 13.64, 13.43, 13.25, 13.08, 12.93, 12.78, 12.60,
    12.44, 12.33, 12.26, 29.52, 17.05, 12.44, 12.58, 12.72, 12.83, 15.46, 16.75, 12.83, 12.67, 12.43, 12.19, 11.89, 11.60, 11.35, 11.12, 10.95, 10.76, 10.42, 10.11, 10.04, 10.02, 10.11, 9.87, 8.65, 7.27, 6.44, 5.83, 5.41,
    5.04, 4.57, 4.12, 3.77, 3.46, 3.08, 2.73, 2.47, 2.25, 2.06, 1.90, 1.75, 1.62, 1.54, 1.45, 1.32, 1.17, 0.99, 0.81, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F8 D50 simulator Sylvania F40 Design 5000K
const double ColorTemp::FluoF8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.21, 1.5, 1.81, 2.13, 3.17, 13.08, 3.83, 3.45, 3.86, 4.42, 5.09, 34.1, 12.42, 7.68, 8.60, 9.46, 10.24, 10.84, 11.33, 11.71, 11.98, 12.17, 12.28, 12.32, 12.35, 12.44, 12.55, 12.68, 12.77, 12.72,
    12.60, 12.43, 12.22, 28.96, 16.51, 11.79, 11.76, 11.77, 11.84, 14.61, 16.11, 12.34, 13.61, 13.87, 14.07, 14.20, 14.16, 14.13, 14.34, 14.50, 14.46, 14.00, 12.58, 10.99, 9.98, 9.22, 8.62, 8.07, 7.39, 6.71, 6.16, 5.63, 5.03, 4.46, 4.02, 3.66,
    3.36, 3.09, 2.85, 2.65, 2.51, 2.37, 2.15, 1.89, 1.61, 1.32, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F9 Cool white deluxe 4150K
const double ColorTemp::FluoF9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.9, 1.12, 1.36, 1.60, 2.59, 12.8, 3.05, 2.56, 2.86, 3.30, 3.82, 32.62, 10.77, 5.84, 6.57, 7.25, 7.86, 8.35, 8.75, 9.06, 9.31, 9.48, 9.61, 9.68, 10.04, 10.26, 10.48, 10.63, 10.76, 10.96,
    11.18, 27.71, 16.29, 12.28, 12.74, 13.21, 13.65, 16.57, 18.14, 14.55, 14.65, 14.66, 14.61, 14.50, 14.39, 14.40, 14.47, 14.62, 14.72, 14.55, 14.4, 14.58, 14.88, 15.51, 15.47, 13.20, 10.57, 9.18, 8.25, 7.57, 7.03,
    6.35, 5.72, 5.25, 4.80, 4.29, 3.80, 3.43, 3.12, 2.86, 2.64, 2.43, 2.26, 2.14, 2.02, 1.83, 1.61, 1.38, 1.12, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F10 Philips TL85 - 5000K
const double ColorTemp::FluoF10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.11, 0.63, 0.62, 0.57, 1.48, 12.16, 2.12, 2.70, 3.74, 5.14, 6.75, 34.39, 14.86, 10.4, 10.76, 10.11, 9.27, 8.29, 7.29, 7.91, 16.64, 16.73, 10.44, 5.94, 3.34, 2.35, 1.88, 1.59, 1.47,
    1.80, 5.71, 40.98, 73.69, 33.61, 8.24, 3.38, 2.47, 4.86, 11.45, 14.79, 12.16, 8.97, 6.52, 8.81, 44.12, 34.55, 12.09, 12.15, 10.52, 4.43, 1.95, 2.19, 3.19, 2.77, 2.29, 2.00, 1.52, 1.35, 1.47, 1.79, 1.74, 1.02, 1.14,
    3.32, 4.49, 2.05, 0.49, 0.24, 0.21, 0.21, 0.24, 0.24, 0.21, 0.17, 0.21, 0.22, 0.17, 0.12, 0.09, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F11 Philips TL84 4150K
const double ColorTemp::FluoF11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.91, 0.63, 0.46, 0.37, 1.29, 12.68, 1.59, 1.79, 2.46, 3.38, 4.49, 33.94, 12.13, 6.95, 7.19, 7.12, 6.72, 6.13, 5.46, 4.79, 5.66, 14.29, 14.96, 8.97, 4.72, 2.33, 1.47, 1.10, 0.89, 0.83, 1.18, 4.90, 39.59,
    72.84, 32.61, 7.52, 2.83, 1.96, 1.67, 4.43, 11.28, 14.76, 12.73, 9.74, 7.33, 9.72, 55.27, 42.58, 13.18, 13.16, 12.26, 5.11, 2.07, 2.34, 3.58, 3.01, 2.48, 2.14, 1.54, 1.33, 1.46, 1.94, 2.00, 1.20, 1.35, 4.10, 5.58,
    2.51, 0.57, 0.27, 0.23, 0.21, 0.24, 0.24, 0.20, 0.24, 0.32, 0.26, 0.16, 0.12, 0.09, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for fluo F12 Philips TL83 3000K
const double ColorTemp::FluoF12_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.96, 0.64, 0.45, 0.33, 1.19, 12.48, 1.12, 0.94, 1.08, 1.37, 1.78, 29.05, 7.90, 2.65, 2.71, 2.65, 2.49, 2.33, 2.10, 1.91, 3.01, 10.83, 11.88, 6.88, 3.43, 1.49, 0.92, 0.71, 0.60, 0.63, 1.10, 4.56, 34.4, 65.40, 29.48,
    7.16, 3.08, 2.47, 2.27, 5.09, 11.96, 15.32, 14.27, 11.86, 9.28, 12.31, 68.53, 53.02, 14.67, 14.38, 14.71, 6.46, 2.57, 2.75, 4.18, 3.44, 2.81, 2.42, 1.64, 1.36, 1.49, 2.14, 2.34, 1.42, 1.61, 5.04, 6.98, 3.19, 0.71, 0.30, 0.26, 0.23, 0.28, 0.28, 0.21,
    0.17, 0.21, 0.19, 0.15, 0.10, 0.05, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for HMI lamp studio "Osram" 4800K (for film, spectacle, studio...)
const double ColorTemp::HMI_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 9.66, 11.45, 13.24, 14.93, 16.63, 17.90, 19.20, 20.12, 21.03, 23.84, 26.65, 26.38, 26.12, 26.52, 27.92, 31.15, 34.37, 34.98, 35.61, 35.71, 35.81, 34.90, 34.02, 34.06, 34.08, 34.68, 35.28, 34.72, 34.20, 33.63,
    33.05, 34.70, 36.35, 38.01, 39.48, 37.29, 35.10, 36.22, 37.28, 38.76, 40.24, 39.56, 38.90, 39.35, 39.79, 38.90, 38.01, 38.05, 38.10, 37.45, 36.64, 35.82, 35.00, 35.06, 33.13, 33.85, 34.55, 35.26, 35.77, 34.92,
    34.09, 33.40, 32.72, 32.08, 31.45, 26.83, 22.23, 21.50, 20.79, 21.41, 22.03, 11.01, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for GTI lamp : Graphiclite & ColorMatch for Photography 5000K
const double ColorTemp::GTI_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3.26, 4.71, 6.17, 12.71, 19.27, 20.53, 21.80, 19.15, 16.53, 28.25, 39.97, 48.52, 57.06, 43.66, 30.27, 30.22, 30.16, 31.48, 32.98, 34.01, 35.04, 35.83, 36.62, 37.12, 37.62, 37.99, 38.19, 38.29, 38.48,
    38.82, 39.16, 45.40, 51.63, 51.83, 62.04, 52.41, 42.80, 42.95, 43.09, 45.64, 48.20, 46.23, 44.27, 43.74, 43.22, 43.30, 43.41, 43.10, 42.78, 42.03, 41.29, 40.29, 39.29, 37.89, 36.58, 34.92, 33.27, 31.47, 29.68, 27.90,
    26.13, 24.55, 22.98, 21.42, 19.86, 18.40, 16.92, 14.46, 13.99, 12.36, 11.73, 5.86, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for JudgeIII Lamp D50
const double ColorTemp::JudgeIII_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.08, 4.25, 4.43, 6.90, 9.40, 9.75, 10.11, 9.30, 8.54, 14.90, 21.16, 26.01, 30.83, 24.90, 19.00, 19.00, 19.00, 19.56, 20.13, 20.28, 20.44, 20.64, 20.85, 21.05, 21.24, 21.65, 22.11, 22.85, 23.58, 24.00, 24.43,
    27.75, 31.27, 33.90, 36.59, 30.90, 25.32, 25.05, 24.76, 26.03, 27.31, 25.90, 24.48, 23.85, 23.29, 23.10, 22.94, 23.24, 23.53, 24.02, 24.52, 23.80, 23.13, 24.51, 25.76, 27.90, 29.15, 24.70, 20.25, 16.60, 12.98, 11.63, 10.27, 9.30, 8.34,
    7.60, 6.91, 6.25, 5.67, 5.15, 4.68, 2.34, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data for Solux lamp : 3500K
const double ColorTemp::Solux3500_spect[97] = {
    0.5268, 0.93, 1.3278, 1.51, 1.6987, 2.65, 3.6100, 3.80, 3.9927, 6.08, 8.1680, 11.02, 13.863, 15.66, 17.4600, 18.78, 20.130, 21.43, 22.749, 24.02, 25.290, 27.40, 29.504, 31.77, 34.031, 36.35, 38.672, 40.55, 42.426, 44.15, 45.865, 47.37, 48.879,
    49.71, 50.531, 51.2, 51.872, 51.9, 51.928, 52.97, 54.015, 55.93, 57.846, 60.25, 62.650, 64.36, 66.065, 66.72, 67.369, 68.81, 70.260, 71.37, 72.487, 72.53, 72.578, 72.51, 72.447, 72.46, 72.471,
    72.76, 73.047, 74.25, 75.449, 76.5, 77.543, 78.79, 80.040, 80.72, 81.394, 82.12, 82.840, 83.23, 83.614, 83.36, 83.100, 82.36, 81.615, 80.11, 78.606, 75.91, 73.221, 69.61, 66.006, 62.43, 58.844, 56.07, 53.292,
    51.07, 48.839, 46.93, 45.013, 43.54, 42.070, 40.61, 39.150, 37.79, 36.425
};

//spectral data for Solux lamp : 4100K
const double ColorTemp::Solux4100_spect[97] = {
    0.5717, 0.70, 0.8286, 1.16, 1.522, 2.01, 2.384, 3.45, 4.57, 6.46, 8.4548, 11.31, 14.205, 16.10, 17.949, 19.51, 21.068, 22.60, 24.197, 25.37, 26.566, 28.15, 29.742, 30.90, 32.060, 33.26, 34.481, 34.80, 35.130, 35.42, 35.697, 36.20, 36.763,
    37.90, 39.004, 40.26, 41.494, 43.10, 44.690, 45.80, 46.900, 47.45, 47.885, 47.75, 47.635, 47.00, 46.410, 46.22, 46.058, 46.70, 47.344, 48.65, 50.005, 51.02, 52.045, 53.55, 55.075, 55.98, 56.823,
    56.85, 56.884, 56.15, 55.523, 54.60, 53.732, 52.55, 51.425, 50.30, 49.1830, 48.76, 48.273, 48.22, 48.169, 49.92, 49.915, 51.90, 53.099, 54.95, 56.852, 58.45, 60.090, 61.67, 63.2530, 63.55, 63.834, 63.55, 63.468,
    62.40, 61.373, 59.75, 58.1810, 56.25, 54.395, 51.90, 49.496, 47.05, 44.620
};

//spectral data for Solux lamp : near Daylight (for example  "Musée d'Orsay") - 4700K
const double ColorTemp::Solux4700_spect[97] = {
    0.4590, 0.83, 1.2011, 1.53, 1.8647, 2.15, 2.5338, 3.06, 3.5809, 3.99, 4.4137, 4.82, 5.2228, 5.63, 6.0387, 6.53, 6.9944, 7.55, 8.0266, 8.475, 8.9276, 8.90, 9.7840, 10.20, 10.6390, 11.00, 11.3600, 11.75, 12.1340, 12.36, 12.5880, 12.74, 12.8790,
    13.07, 13.2560, 13.38, 13.5220, 13.41, 13.3070, 13.35, 13.3990, 13.37, 13.3420, 13.39, 13.4220, 13.65, 13.2710, 13.25, 13.2330, 13.12, 13.0110, 12.93, 12.8470, 12.805, 12.7630, 12.66, 12.5760, 12.563, 12.5490,
    12.59, 12.6330, 12.617, 12.6010, 12.616, 12.6310, 12.6275, 12.6240, 12.70, 12.7710, 12.776, 12.7810, 12.786, 12.7950, 12.74, 12.6850, 12.64, 12.5950, 12.55, 12.5420, 12.43, 12.3180, 12.07, 11.8340, 11.72, 11.6190, 11.55, 11.5020,
    11.32, 11.1510, 11.05, 10.9530, 10.80, 10.6550, 10.495, 10.4390, 10.31, 10.1790
};

//spectral data for Solux lamp : near Daylight 4400K - test National Gallery
const double ColorTemp::NG_Solux4700_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 139, 152, 170, 185, 202, 223, 240, 257, 270, 282, 293, 305, 317, 329, 342, 355, 367, 378, 387, 395, 401, 405, 408, 411, 414, 415, 416, 415, 414, 414, 416, 419, 423, 427, 432, 437, 442, 447, 452,
    456, 461, 467, 475, 483, 488, 490, 491, 490, 487, 485, 481, 477, 472, 466, 461, 455, 449, 442, 434, 427, 419, 411, 403, 395, 386, 377, 367, 359, 351, 343, 335, 327, 322, 316, 312, 306, 305, 301, 299, 299, 298,
    0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0
};

//spectral data for LED LSI Lumelex 2040 - test National Gallery
const double ColorTemp::NG_LEDLSI2040_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.5, 1.2, 0.5, 0.7, 0.6, 1.6, 1.7, 7.0, 16.6, 35.5, 64, 106, 162.5, 230.5, 272.2, 249, 213.4, 214, 227.6, 231.9, 233, 235.2, 241.4, 253.7, 270.3, 288.5, 306.2, 322.3, 337.6, 352.5, 367.2, 381.7, 395.9, 409.6, 416.2, 423.2, 429.7, 435.8, 442.8,
    451.7, 464.2, 480.3, 501, 526.3, 555.9, 587.5, 625.4, 655.1, 681.7, 705.3, 721.5, 728.5, 729, 719.8, 702.5, 676.7, 646.2, 611.5, 571.7, 530.3, 488.3, 445.9, 404, 365.2, 326.8, 290.8, 257.6, 226.9, 199.8, 175.2, 154.2, 133.8, 116.4, 101.5, 88.5, 76.6, 67.3, 57.9, 50.7, 44.2, 38.2,
    0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0
};

//spectral data for LED CRS SP12 WWMR16 - test National Gallery
const double ColorTemp::NG_CRSSP12WWMR16_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0., 0., 0., 0., 0., 0.14, 0.33, 1.31, 3.34, 7.9, 17.4, 36.5, 72.6, 145.4, 260.5, 359.2, 365.3, 303.1, 256.1, 221.7, 193.6, 185.8, 191.4, 207.3, 232.8, 257.5, 285.1, 310.5, 333.4, 351.5, 368.8, 383.7, 398.8, 411.6, 424.7, 435.6, 447.9, 459.7, 471.7,
    484.6, 497.9, 512.3, 531.1, 548.9, 567.9, 587.5, 608.6, 625.3, 640.1, 648.6, 654.0, 654.3, 647.2, 633.9, 616.1, 590.5, 561.5, 526.5, 494.8, 457.9, 420.8, 382.4, 347.3, 309.9, 280.5, 249.2, 220.0, 194.9, 170.8, 149.1, 130.0, 112.3, 97.5, 84.9, 73.2, 63.1, 54.1, 45.6, 39.0, 32.6, 27.4,
    0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.0
};


//Data for flash :
// in theory, should be the spectral data of each owner flash (Nikon, Canon, Sony ...) or studio flash (Profoto. ..)
// but: 1) I did not, 2) the data vary depending on the power used ... so by default, although this is not true, I chose the values "Daylight" for temp...
// CRI for flash near of 95 - 97 !!

//spectral data for Flash daylight 5500K (Leica...)
const double ColorTemp::Flash5500_spect[97] = {
    27.77, 29.17, 30.58, 32.02, 33.47, 33.00, 32.53, 35.28, 38.04, 49.46, 60.88, 64.68, 68.48, 69.99, 71.51, 69.68, 67.85, 76.70, 85.54, 91.74, 97.93, 99.17, 100.41, 100.14, 99.86, 101.28, 102.70, 100.37, 98.04, 99.35, 100.65,
    100.66, 100.67, 100.32, 99.97, 102.08, 104.20, 103.15, 102.09, 102.53, 102.96, 101.48, 100.00, 98.61, 97.22, 97.49, 97.76, 94.60, 91.44, 92.94, 94.44, 94.80, 95.16, 94.70, 94.25, 92.36, 90.48, 91.42, 92.36, 90.63,
    88.89, 89.62, 90.36, 92.18, 94.00, 92.00, 90.00, 84.86, 79.72, 81.30, 82.88, 83.88, 84.89, 77.58, 70.27, 74.80, 82.19, 85.03, 78.47, 71.91, 62.37, 52.82, 64.39, 75.96, 73.91, 71.85, 72.41, 72.97, 70.17, 67.38, 63.07,
    58.75, 61.89, 65.02, 66.68, 68.34
};

//spectral data for Flash daylight 6000K (Canon, Pentax, Olympus,...Standard)
const double ColorTemp::Flash6000_spect[97] = {
    36.00, 37.18, 38.36, 40.36, 42.35, 41.77, 41.19, 43.79, 46.40, 59.24, 72.09, 76.15, 80.22, 81.47, 82.71, 80.12, 77.52, 86.54, 95.56, 101.70, 107.85, 108.66, 109.46, 108.57, 107.68, 108.65, 109.61, 106.63, 103.65, 104.42,
    105.19, 104.79, 104.39, 103.45, 102.51, 104.28, 106.05, 104.68, 103.31, 103.42, 103.54, 101.77, 100.00, 98.38, 96.75, 96.74, 96.72, 93.30, 89.89, 90.92, 91.95, 91.99, 92.03, 91.30, 90.57, 88.51, 86.45, 86.96,
    87.47, 85.66, 83.85, 84.21, 84.57, 85.95, 87.32, 85.31, 83.30, 78.66, 74.03, 75.24, 76.45, 77.68, 78.91, 72.13, 65.35, 69.67, 73.98, 76.68, 79.39, 73.29, 67.19, 58.19, 49.19, 59.98, 70.77, 68.91, 67.05, 67.55, 68.05,
    65.47, 62.88, 58.89, 54.90, 57.81, 60.72, 62.25, 63.78
};

//spectral data for Flash daylight 6500K (Nikon, Minolta, Panasonic, Sony...)
const double ColorTemp::Flash6500_spect[97] = {
    44.86, 45.72, 46.59, 49.16, 51.74, 50.83, 49.92, 52.26, 54.60, 68.65, 82.69, 87.06, 91.42, 92.39, 93.37, 90.00, 86.63, 95.72, 104.81, 110.88, 116.96, 117.36, 117.76, 116.29, 114.82, 115.35, 115.89, 112.33, 108.78, 109.06,
    109.33, 108.56, 107.78, 106.28, 104.78, 106.23, 107.68, 106.04, 104.40, 104.22, 104.04, 102.02, 100.00, 98.17, 96.34, 96.06, 95.79, 92.24, 88.69, 89.35, 90.02, 89.81, 89.61, 88.66, 87.71, 85.51, 83.30, 83.51, 83.72,
    81.88, 80.05, 80.14, 80.24, 81.27, 82.30, 80.31, 78.31, 74.03, 69.74, 70.69, 71.63, 73.00, 74.37, 68.00, 61.62, 65.76, 69.91, 72.51, 75.11, 69.36, 63.61, 55.02, 46.43, 56.63, 66.83, 65.11, 63.40, 63.86, 64.32, 61.90, 59.47,
    55.72, 51.97, 54.72, 57.46, 58.89, 60.33
};

const std::map<std::string, const double *> ColorTemp::spectMap = {
    {"Daylight", Daylight5300_spect},
    {"Cloudy", Cloudy6200_spect},
    {"Shade", Shade7600_spect},
    {"Tungsten", A2856_spect},
    {"Fluo F1", FluoF1_spect},
    {"Fluo F2", FluoF2_spect},
    {"Fluo F3", FluoF3_spect},
    {"Fluo F4", FluoF4_spect},
    {"Fluo F5", FluoF5_spect},
    {"Fluo F6", FluoF6_spect},
    {"Fluo F7", FluoF7_spect},
    {"Fluo F8", FluoF8_spect},
    {"Fluo F9", FluoF9_spect},
    {"Fluo F10", FluoF10_spect},
    {"Fluo F11", FluoF11_spect},
    {"Fluo F12", FluoF12_spect},
    {"HMI Lamp", HMI_spect},
    {"GTI Lamp", GTI_spect},
    {"JudgeIII Lamp", JudgeIII_spect},
    {"Solux Lamp 3500K", Solux3500_spect},
    {"Solux Lamp 4100K", Solux4100_spect},
    {"Solux Lamp 4700K", Solux4700_spect},
    {"NG Solux Lamp 4700K", NG_Solux4700_spect},
    {"LED LSI Lumelex 2040", NG_LEDLSI2040_spect},
    {"LED CRS SP12 WWMR16", NG_CRSSP12WWMR16_spect},
    {"Flash 5500K", Flash5500_spect},
    {"Flash 6000K", Flash6000_spect},
    {"Flash 6500K", Flash6500_spect}
};

// Data for Color ==> CRI (Color Rendering Index and Palette
// actually 20 color that must be good enough for CRI


//spectral data Colorchecker24 : Red C3
const double  ColorTemp::ColorchechredC3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0478, 0.0476, 0.0474, 0.0472, 0.0470, 0.0466, 0.0461, 0.0460, 0.0459, 0.0456, 0.0453, 0.0451, 0.0449, 0.0448, 0.0447, 0.0446, 0.0445, 0.0441, 0.0437, 0.0432, 0.0427, 0.0424, 0.0421, 0.0419,
    0.0417, 0.0415, 0.0412, 0.0412, 0.0412, 0.0413, 0.0413, 0.0415, 0.0416, 0.0421, 0.0426, 0.0436, 0.0446, 0.0469, 0.0491, 0.0549, 0.0607, 0.0773, 0.0939, 0.1376, 0.1812, 0.2568, 0.3323, 0.4070, 0.4816, 0.5308,
    0.5800, 0.6059, 0.6317, 0.6447, 0.6577, 0.6653, 0.6728, 0.6761, 0.6793, 0.6820, 0.6847, 0.6878, 0.6909, 0.6945, 0.6980, 0.7013, 0.7046, 0.7065, 0.7084, 0.7107, 0.7129, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : Orange A2
const double ColorTemp::ColorchechOraA2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0520, 0.0530, 0.0540, 0.0535, 0.0530, 0.0532, 0.0534, 0.0532, 0.0529, 0.0529, 0.0528, 0.0530, 0.0532, 0.0537, 0.0542, 0.0550, 0.0557, 0.0565, 0.0573, 0.0585, 0.0597, 0.0613, 0.0628, 0.0656, 0.0683, 0.0793,
    0.0902, 0.1085, 0.1268, 0.1414, 0.1559, 0.1645, 0.1730, 0.1837, 0.1944, 0.2184, 0.2424, 0.2877, 0.3329, 0.3923, 0.4517, 0.5021, 0.5525, 0.5739, 0.5952, 0.5967, 0.5982, 0.5962, 0.5942, 0.5932, 0.5922, 0.5927,
    0.5932, 0.5938, 0.5944, 0.5988, 0.6032, 0.6105, 0.6178, 0.6284, 0.6389, 0.6498, 0.6607, 0.6699, 0.6791, 0.6839, 0.6886, 0.6879, 0.6871, 0.6886, 0.6901, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 :yellow D3
const double ColorTemp::ColorchechYelD3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0476, 0.0482, 0.0488, 0.0492, 0.0496, 0.0498, 0.0499, 0.0498, 0.0496, 0.0501, 0.0506, 0.0516, 0.0526, 0.0547, 0.0567, 0.0610, 0.0652, 0.0733, 0.0813, 0.0962, 0.1110, 0.1333, 0.1556, 0.1884, 0.2211,
    0.2782, 0.3353, 0.4023, 0.4692, 0.5198, 0.5703, 0.5976, 0.6249, 0.6400, 0.6551, 0.6667, 0.6783, 0.6901, 0.7018, 0.7095, 0.7171, 0.7231, 0.7290, 0.7329, 0.7367, 0.7395, 0.7423, 0.7447, 0.7471, 0.7490, 0.7508,
    0.7533, 0.7558, 0.7578, 0.7598, 0.7623, 0.7647, 0.7654, 0.7661, 0.7677, 0.7693, 0.7720, 0.7746, 0.7780, 0.7814, 0.7845, 0.7876, 0.7889, 0.7902, 0.7920, 0.7938,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

//spectral data Colorchecker24 : Green E2
const double ColorTemp::ColorchechGreE2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0560, 0.0583, 0.0605, 0.0626, 0.0646, 0.0650, 0.0653, 0.0657, 0.0661, 0.0669, 0.0676, 0.0692, 0.0708, 0.0737, 0.0765, 0.0816, 0.0867, 0.0956, 0.1044, 0.1194, 0.1344, 0.1581, 0.1818, 0.2196, 0.2574, 0.3166, 0.3757,
    0.4297, 0.4837, 0.5142, 0.5446, 0.5541, 0.5636, 0.5608, 0.5579, 0.5480, 0.5381, 0.5258, 0.5135, 0.4959, 0.4783, 0.4570, 0.4356, 0.4124, 0.3891, 0.3710, 0.3529, 0.3425, 0.3320, 0.3266, 0.3211, 0.3180, 0.3149,
    0.3129, 0.3108, 0.3123, 0.3137, 0.3193, 0.3248, 0.3335, 0.3422, 0.3518, 0.3613, 0.3693, 0.3772, 0.3810, 0.3847, 0.3838, 0.3829, 0.3838, 0.3847, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::ColorGreenM25_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0758, 0.0614, 0.0662, 0.0591, 0.0648, 0.0633, 0.0671, 0.0613, 0.0621, 0.0610, 0.0680, 0.0690, 0.0784, 0.0830, 0.0920, 0.1070, 0.1231, 0.1423, 0.1607, 0.1731, 0.1816, 0.1911, 0.1951, 0.1986, 0.1915, 0.1889,
    0.1758, 0.1673, 0.1606, 0.1505, 0.1384, 0.1317, 0.1230, 0.1149, 0.1081, 0.0992, 0.0882, 0.0785, 0.0709, 0.0629, 0.0550, 0.0502, 0.0486, 0.0474, 0.0445, 0.0434, 0.0429, 0.0423, 0.0411, 0.0405, 0.0397, 0.0387, 0.0399, 0.0398, 0.0398, 0.0407, 0.0408, 0.0426, 0.0445, 0.0443, 0.0468,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorYellowkeltano_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0658, 0.0045,  0.0,  0.0, 0.0131,  0.0098,  0.0084, 0.0123, 0.0104, 0.0131, 0.0112, 0.0121, 0.0121, 0.0139,
    0.0128, 0.0148, 0.0199, 0.0296, 0.0389, 0.0476, 0.0689, 0.0945, 0.1132, 0.1326, 0.1490,  0.1674, 0.1741, 0.1775, 0.1868,
    0.1914, 0.1928, 0.1961, 0.1972, 0.1992, 0.2045, 0.2064, 0.2053, 0.2048, 0.2072, 0.2086, 0.2081, 0.2069, 0.2056,  0.2073,
    0.2096, 0.2114, 0.2067, 0.2089, 0.2100, 0.2061, 0.2019, 0.1983, 0.1971, 0.1961, 0.2016, 0.1956, 0.1946, 0.1922, 0.1983, 0.1991,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenalsi_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1732,  0.0292,  0.0315,  0.0533,  0.0469,  0.0359,  0.0404,  0.0356,  0.0345,  0.0374,  0.0348,  0.0336,  0.0332,  0.0333,  0.0342,
    0.0330,  0.0350,  0.0343,  0.0354,  0.0371,  0.0399,  0.0426,  0.0497,  0.0541,  0.0616,  0.0701,  0.0750,  0.0764,  0.0783,  0.0794,
    0.0784,  0.0732,  0.0708,  0.0652,  0.0612,  0.0595,  0.0570,  0.0531,  0.0507,  0.0501,  0.0488,  0.0458,  0.0437,  0.0420,  0.0436,
    0.0424,  0.0417,  0.0389,  0.0380,  0.0378,  0.0371,  0.0350,  0.0333,  0.0350,  0.0394,  0.0379,  0.0446,  0.0491,  0.0575,  0.0734,  0.0953,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorRedpetunia_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1890, 0.1097,  0.0855,  0.0899,  0.0987,  0.0881,  0.0807,  0.0804,  0.0787,  0.0691,  0.0643,  0.0549,  0.0465,  0.0404,  0.0385,
    0.0302,  0.0244,  0.0195,  0.0165,  0.0159,  0.0123,  0.0129,  0.0108,  0.0111,  0.0114,  0.0126,  0.0126,  0.0134,  0.0162,  0.0170,
    0.0213,  0.0248,  0.0279,  0.0351,  0.0412,  0.0566,  0.0752,  0.0909, 0.1069, 0.1270, 0.1526, 0.1707, 0.1858, 0.1999, 0.2112,
    0.2293, 0.2422, 0.2471, 0.2611, 0.2718, 0.2710, 0.2778, 0.2807, 0.2825, 0.2856, 0.2909, 0.2901, 0.2974, 0.3042, 0.3044, 0.3075,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::JDC468_B14_75Redspect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1588, 0.1780, 0.1980, 0.2310, 0.2789, 0.3213, 0.4585, 0.5324, 0.6059, 0.6334, 0.6609, 0.6630, 0.6674, 0.6640, 0.6599, 0.6600, 0.6551, 0.6532, 0.6472, 0.6450,
    0.6423, 0.6467, 0.6485, 0.6500, 0.6515, 0.6456, 0.6379, 0.6285, 0.6193, 0.6150, 0.6121, 0.6056, 0.6026, 0.5812, 0.5678, 0.5490, 0.5310, 0.5307, 0.5245, 0.5300,
    0.5305, 0.5310, 0.5324, 0.5298, 0.5262, 0.5230, 0.5219, 0.5231, 0.5247, 0.5280, 0.5312, 0.5364, 0.5436, 0.5550, 0.5634, 0.5731, 0.5832, 0.5901, 0.5943, 0.5950,
    0.5953, 0.5926, 0.5902, 0.5850, 0.5805, 0.5770, 0.5754, 0.6012, 0.6262,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::ColorRedkurttu_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0979, 0.1536,  0.0745, 0.1145,  0.0937,  0.0996,  0.0972,  0.0924,  0.0875,  0.0878,  0.0826,  0.0766,  0.0658,  0.0599,  0.0545,
    0.0509,  0.0443,  0.0381,  0.0356,  0.0327,  0.0318,  0.0304,  0.0295,  0.0288,  0.0285,  0.0271,  0.0281,  0.0282,  0.0278,  0.0280,
    0.0296,  0.0309,  0.0324,  0.0342,  0.0343,  0.0376,  0.0419,  0.0464,  0.0520,  0.0634,  0.0747,  0.0905, 0.1093, 0.1167, 0.1264,
    0.1439, 0.1650, 0.1928, 0.2183, 0.2380, 0.2537, 0.2754, 0.2893, 0.3009, 0.3115, 0.3213, 0.3173, 0.3222, 0.3237, 0.3192, 0.3210,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorRedlupiini_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1792, 0.1553, 0.1684, 0.1882, 0.1909, 0.1847, 0.2053, 0.2084, 0.2045, 0.2052, 0.2039, 0.2084, 0.2041, 0.2044, 0.2007, 0.1984, 0.1906, 0.1876,
    0.1886, 0.1855, 0.1859, 0.1875, 0.1816, 0.1800, 0.1811, 0.1780, 0.1802, 0.1816, 0.1838, 0.1915, 0.1973, 0.2018, 0.2083, 0.2114, 0.2133, 0.2226,
    0.2304, 0.2385, 0.2458, 0.2494, 0.2571, 0.2689, 0.2738, 0.2774, 0.2734, 0.2759, 0.2781, 0.2831, 0.2844, 0.2857, 0.2878, 0.2876, 0.2884, 0.2920,
    0.2932, 0.2860, 0.2894, 0.2934, 0.2925, 0.2928, 0.2921,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::ColorRedhevosminttu_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1280,  0.0706,  0.0612,  0.0462,  0.0391,  0.0339,  0.0253,  0.0285,  0.0261,  0.0264,  0.0239,  0.0208,  0.0201,  0.0186,  0.0185,  0.0161,  0.0156,  0.0149,
    0.0146,  0.0148,  0.0161,  0.0144,  0.0143,  0.0151,  0.0147,  0.0146,  0.0139,  0.0148,  0.0173,  0.0185,  0.0185,  0.0197,  0.0222,  0.0238,  0.0283,  0.0322,
    0.0384,  0.0439,  0.0519,  0.0633,  0.0792,  0.0922,  0.1061, 0.1186, 0.1235, 0.1342, 0.1538, 0.1691, 0.1839, 0.1974, 0.2024, 0.2098, 0.2128, 0.2187, 0.2204,
    0.2217, 0.2267, 0.2299, 0.2339, 0.2331, 0.2322,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::ColorRedneilikka_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0,    0.0,  0.0394,    0.0,  0.0245,   0.0095,  0.0174,  0.0149,  0.0194,  0.0171,  0.0181,  0.0175,  0.0172,  0.0167,  0.0147,  0.0137,  0.0107,  0.0108,  0.0100,
    0.0087,   0.0093,   0.0087,   0.0083,   0.0077,   0.0080,   0.0067,   0.0072,   0.0064,   0.0083,   0.0084,   0.0088,   0.0090,   0.0091,   0.0094,  0.0114,  0.0133,
    0.0178,  0.0241,  0.0309,  0.0419,  0.0612,  0.0823,  0.0992, 0.1153, 0.1222, 0.1366, 0.1503, 0.1658, 0.1767, 0.1841, 0.1884, 0.1992, 0.2035, 0.2007, 0.2009,
    0.2045, 0.2065, 0.2229, 0.2290, 0.2395, 0.2449,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorRedpelagornia_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2529, 0.2048, 0.2087, 0.2698, 0.2452, 0.2372, 0.2531, 0.2475, 0.2296, 0.2294, 0.2159, 0.2111, 0.1986, 0.1898, 0.1854, 0.1729, 0.1586, 0.1501, 0.1392, 0.1332, 0.1343,
    0.1255, 0.1217, 0.1182, 0.1183, 0.1203, 0.1230, 0.1277, 0.1381, 0.1474, 0.1615, 0.1762, 0.1876, 0.2028, 0.2214, 0.2464, 0.2657, 0.2919, 0.3051, 0.3172, 0.3293, 0.3421,
    0.3395, 0.3494, 0.3438, 0.3495, 0.3506, 0.3490, 0.3454, 0.3487, 0.3431, 0.3452, 0.3484, 0.3438, 0.3422, 0.3368, 0.3325, 0.3441, 0.3356, 0.3432, 0.3320,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorRedtalvio_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3131, 0.2199, 0.2559, 0.2540, 0.2844, 0.2530, 0.2694, 0.2765, 0.2594, 0.2673, 0.2617, 0.2629, 0.2491, 0.2384, 0.2308, 0.2256, 0.2081, 0.1973, 0.1857, 0.1752, 0.1719,
    0.1652, 0.1527, 0.1477, 0.1459, 0.1386, 0.1341, 0.1283, 0.1318, 0.1334, 0.1354, 0.1424, 0.1495, 0.1543, 0.1634, 0.1773, 0.1950, 0.2129, 0.2272, 0.2431, 0.2642, 0.2827,
    0.2941, 0.3045, 0.3082, 0.3158, 0.3216, 0.3307, 0.3364, 0.3388, 0.3387, 0.3517, 0.3573, 0.3501, 0.3499, 0.3523, 0.3495, 0.3606, 0.3493, 0.3518, 0.3522,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0

};

const double ColorTemp::ColorBrownpoimulehti_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0964,  0.0520,  0.0223,  0.0244,  0.0261,  0.0247,  0.0196,  0.0199,  0.0200,  0.0207,  0.0202,  0.0198,  0.0209,  0.0204,  0.0207,  0.0222,  0.0205,  0.0218, 0.0213,
    0.0212,  0.0224,  0.0218,  0.0230,  0.0235,  0.0251,  0.0250,  0.0245,  0.0250,  0.0263,  0.0273,  0.0271,  0.0275,  0.0281,  0.0264,  0.0274,  0.0288, 0.0287,  0.0307,
    0.0303,  0.0307,  0.0323,  0.0304,  0.0335,  0.0335,  0.0346,  0.0345,  0.0347,  0.0348,  0.0370,  0.0364,  0.0380,  0.0393,  0.0384,  0.0407,
    0.0419,  0.0421,  0.0419,  0.0433,  0.0431,  0.0461,  0.0465,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorOrangetuntematon_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0241,  0.0195,  0.0223,  0.0489,  0.0574,  0.0565,  0.0634,  0.0605,  0.0574,  0.0613,  0.0645,  0.0636,  0.0644,  0.0628,  0.0621,  0.0603,  0.0614,  0.0654,  0.0676,
    0.0719,  0.0776,  0.0795,  0.0862,  0.0879,  0.0918,  0.0918,  0.0955,  0.0980, 0.1013, 0.1055, 0.1132, 0.1225, 0.1258, 0.1362, 0.1427, 0.1579, 0.1796, 0.1936, 0.2079,
    0.2258, 0.2440, 0.2597, 0.2728, 0.2790, 0.2777, 0.2857, 0.2923, 0.2991, 0.3031, 0.3040, 0.3037, 0.3094, 0.3066, 0.3023, 0.3093, 0.3044, 0.3082, 0.3085, 0.3147, 0.3226, 0.3192,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorOrangetlehmus_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1062,  0.0866,  0.0443,  0.0544,  0.0496,  0.0485,  0.0492,  0.0458,  0.0450,  0.0425,  0.0458,  0.0477,  0.0497,  0.0461,  0.0451,  0.0481,  0.0454,  0.0500,  0.0515,
    0.0538,  0.0529,  0.0593,  0.0638,  0.0670,  0.0686,  0.0711,  0.0718,  0.0729,  0.0741,  0.0760,  0.0796,  0.0833,  0.0895,  0.0958, 0.1016, 0.1128, 0.1246, 0.1344, 0.1450,
    0.1505, 0.1596, 0.1636, 0.1621, 0.1631, 0.1627, 0.1628, 0.1658, 0.1583, 0.1486, 0.1415, 0.1322, 0.1265, 0.1159, 0.1062,  0.0975,  0.0974, 0.1063, 0.1326, 0.1736, 0.2141, 0.2568,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorOrangvaahtera_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1517,  0.0551,  0.0664,  0.0659,  0.0521,  0.0585,  0.0460,  0.0385,  0.0424,  0.0389,  0.0375,  0.0374,  0.0359,  0.0380,  0.0371,  0.0373,  0.0379,  0.0387,  0.0378,
    0.0394,  0.0405,  0.0416,  0.0463,  0.0496,  0.0536,  0.0542,  0.0577,  0.0579,  0.0619,  0.0642,  0.0678,  0.0710,  0.0777,  0.0829,  0.0894, 0.1035, 0.1174, 0.1334, 0.1484,
    0.1611, 0.1798, 0.1941, 0.2012, 0.2065, 0.2135, 0.2229, 0.2286, 0.2317, 0.2332, 0.2357, 0.2323, 0.2330, 0.2292, 0.2236, 0.2137, 0.2093, 0.2180, 0.2240, 0.2368, 0.2487, 0.2528,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBrownlehmus_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0758,  0.0236,  0.0148,  0.0430,  0.0347,  0.0435,  0.0438,  0.0495,  0.0439,  0.0454,  0.0472,  0.0471,  0.0461,  0.0459,  0.0458,  0.0479,  0.0492,  0.0482,  0.0499,  0.0513,
    0.0520,  0.0545,  0.0567,  0.0594,  0.0623,  0.0647,  0.0698,  0.0717,  0.0744,  0.0792,  0.0803,  0.0834,  0.0864,  0.0876,  0.0916,  0.0932,  0.0963, 0.1013, 0.1025, 0.1060,
    0.1099, 0.1118, 0.1153, 0.1175, 0.1207, 0.1242, 0.1268, 0.1266, 0.1284, 0.1305, 0.1305, 0.1304, 0.1353, 0.1360, 0.1330, 0.1332, 0.1413, 0.1502, 0.1610, 0.1682, 0.1737,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBrownuotiosammal_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0482,  0.0260,  0.0178, 0.0092,  0.0104, 0.0088, 0.0092, 0.0040, 0.0043, 0.0052, 0.0058, 0.0064, 0.0070, 0.0063, 0.0067, 0.0067, 0.0062, 0.0076, 0.0082, 0.0082, 0.0091, 0.0096,
    0.0104,  0.0116,  0.0135,  0.0141, 0.0142, 0.0155,  0.0168,  0.0179,  0.0198,  0.0199,  0.0193,  0.0201,  0.0212,  0.0218,  0.0226,  0.0240,  0.0242,  0.0238,  0.0255,  0.0265,
    0.0277,  0.0266,  0.0265,  0.0283,  0.0289,  0.0275,  0.0289,  0.0277,  0.0291,  0.0288,  0.0277,  0.0252,  0.0262,  0.0260,  0.0264,  0.0299,  0.0375,  0.0411,  0.0446,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlacksoil_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0,    0.0,   0.0089,  0.0122, 0.0052, 0.0053, 0.0104, 0.0127, 0.0130, 0.0134, 0.0137, 0.0137, 0.0134, 0.0136, 0.0138, 0.0139, 0.0134, 0.0140, 0.0142, 0.0148, 0.0154, 0.0153,
    0.0152,  0.0150, 0.0151, 0.0156, 0.0153, 0.0166, 0.0154, 0.0171, 0.0163, 0.0163, 0.0166, 0.0166, 0.0169, 0.0169, 0.0166, 0.0174, 0.0174, 0.0170, 0.0170, 0.0168, 0.0176, 0.0177,
    0.0176,  0.0174,  0.0179,  0.0180,  0.0180, 0.0183, 0.0177, 0.0193, 0.0178, 0.0187, 0.0194, 0.0193, 0.0182, 0.0196, 0.0184, 0.0195, 0.0195,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlackpihlaja[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0079, 0.0111, 0.0172, 0.0174, 0.0201, 0.0214, 0.0211, 0.0207, 0.0207, 0.0191, 0.0200, 0.0196, 0.0206, 0.0196, 0.0194,  0.0203, 0.0207, 0.0204, 0.0208, 0.0210, 0.0212,
    0.0211, 0.0208, 0.0209, 0.0219, 0.0222, 0.0224, 0.0231, 0.0241, 0.0232, 0.0244, 0.0249, 0.0250, 0.0267, 0.0264, 0.0262, 0.0269, 0.0282, 0.0277, 0.0289, 0.0284, 0.0279, 0.0302,
    0.0289, 0.0308, 0.0313, 0.0315, 0.0310, 0.0325, 0.0313, 0.0319, 0.0356, 0.0340, 0.0331, 0.0347, 0.0356, 0.0352, 0.0364, 0.0373, 0.0352,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGraynahjajaekaelae_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1204,  0.0585, 0.1113,  0.0733,  0.0600,  0.0653,  0.0715,  0.0685,  0.0726,  0.0682,  0.0713,  0.0691,  0.0719,  0.0691,  0.0683,  0.0693,  0.0711,  0.0715,  0.0701,  0.0700,
    0.0720,  0.0697,  0.0706,  0.0696,  0.0723,  0.0714,  0.0726,  0.0738,  0.0729,  0.0735,  0.0737,  0.0739,  0.0742,  0.0746,  0.0746,  0.0761,  0.0743,  0.0735,  0.0722,  0.0717,
    0.0728,  0.0749,  0.0721,  0.0712,  0.0705,  0.0737,  0.0733,  0.0758,  0.0780,  0.0785,  0.0775,  0.0771,  0.0755,  0.0744,  0.0743,  0.0742,  0.0755,  0.0779,  0.0849,  0.0940, 0.1042,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreennuotisammal_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0120, 0.0065, 0.0134, 0.0031, 0.0209, 0.0124, 0.0104, 0.0096,  0.0097, 0.0095, 0.0076, 0.0079, 0.0083, 0.0093, 0.0083, 0.0095, 0.0095, 0.0104, 0.0117, 0.0127, 0.0140, 0.0161,
    0.0214,  0.0252,  0.0290,  0.0310,  0.0328,  0.0343, 0.0347,  0.0373, 0.0365, 0.0351,  0.0347,  0.0343,  0.0311,  0.0301,  0.0285,  0.0283,  0.0263,  0.0256,  0.0255,  0.0251,
    0.0257,  0.0235,  0.0227,  0.0224,  0.0233,  0.0208,  0.0194,  0.0186,  0.0165,  0.0160,  0.0151,  0.0149,  0.0157,  0.0161,  0.0185,  0.0243,  0.0309,  0.0425,  0.0543,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenleskenlehti_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0525,  0.0273, 0.0, 0.0378,  0.0318, 0.0164, 0.0224, 0.0276, 0.0316, 0.0266, 0.0303, 0.0290, 0.0305, 0.0286, 0.0290, 0.0303, 0.0323, 0.0323, 0.0352, 0.0383,  0.0405, 0.0482,
    0.0614,  0.0743,  0.0920, 0.1015, 0.1139, 0.1192, 0.1175, 0.1216, 0.1195, 0.1145, 0.1116, 0.1009,  0.0947,  0.0867,  0.0802,  0.0754,  0.0741,  0.0709,  0.0675,  0.0625,
    0.0574,  0.0579,  0.0561,  0.0565,  0.0557,  0.0511,  0.0471,  0.0419,  0.0399,  0.0372,  0.0365,  0.0395,  0.0375,  0.0382,  0.0458,  0.0555,  0.0716, 0.1002, 0.1407,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenlinnunkaali_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0602, 0.0, 0.0267,  0.0306,  0.0339,  0.0335,  0.0300,  0.0247,  0.0292, 0.0289,  0.0295,  0.0298,  0.0292,  0.0318,  0.0312,  0.0289,  0.0299,  0.0307, 0.0310, 0.0320,
    0.0350,  0.0375,  0.0446,  0.0499,  0.0574,  0.0634,  0.0698,  0.0725,  0.0736,  0.0754,  0.0736,  0.0702,  0.0668, 0.0633, 0.0590,  0.0551,  0.0514,  0.0499,  0.0467,  0.0460,
    0.0445,  0.0424,  0.0415,  0.0409,  0.0399,  0.0412,  0.0393,  0.0380,  0.0370,  0.0362,  0.0366, 0.0343,  0.0342,  0.0350,  0.0333, 0.0350, 0.0364, 0.0418, 0.0494,  0.0670,  0.0914,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenpelto_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0366, 0.0360, 0.0233, 0.0173, 0.0179, 0.0157, 0.0175, 0.0206, 0.0205, 0.0180, 0.0179, 0.0173, 0.0178, 0.0187, 0.0189,  0.0184, 0.0171,  0.0195, 0.0204, 0.0193, 0.0219,
    0.0253, 0.0297,  0.0365,  0.0431, 0.0467,  0.0489,  0.0493,  0.0516,  0.0500,  0.0466, 0.0426, 0.0406, 0.0380, 0.0343, 0.0316, 0.0295, 0.0276, 0.0282, 0.0265,  0.0253,  0.0239,
    0.0228, 0.0226, 0.0229,  0.0238,  0.0237,  0.0216,  0.0221,  0.0219,  0.0217,  0.0212,  0.0219,  0.0229,  0.0258,  0.0284,  0.0309,  0.0375,  0.0487,  0.0732,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenrodvoikukka[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2205, 0.1755, 0.1710, 0.1365, 0.1159, 0.1207, 0.1024, 0.1118, 0.1127, 0.1141, 0.1134, 0.1125, 0.1149, 0.1140, 0.1120, 0.1128, 0.1139, 0.1156, 0.1212, 0.1273, 0.1262, 0.1359,
    0.1461, 0.1519, 0.1568, 0.1599, 0.1660, 0.1668, 0.1680, 0.1718, 0.1697, 0.1690, 0.1672, 0.1675, 0.1663, 0.1644, 0.1642, 0.1652, 0.1626, 0.1623, 0.1653, 0.1621, 0.1614, 0.1590,
    0.1625, 0.1609, 0.1615, 0.1576, 0.1509, 0.1483, 0.1418, 0.1391, 0.1324, 0.1294, 0.1267, 0.1220, 0.1315, 0.1417, 0.1650, 0.1861, 0.2006,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenlehmus[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2362, 0.1024,  0.0945,  0.0666,  0.0617,  0.0604, 0.0591, 0.0580,  0.0648,  0.0631,  0.0656,  0.0607,  0.0616,  0.0653,  0.0643,  0.0626,  0.0643,  0.0656,  0.0710,  0.0753,
    0.0801,  0.0929, 0.1105, 0.1277, 0.1437, 0.1601, 0.1742, 0.1774, 0.1798, 0.1848, 0.1832, 0.1820, 0.1787, 0.1730, 0.1663, 0.1593, 0.1541, 0.1461, 0.1446, 0.1419, 0.1335, 0.1298,
    0.1247, 0.1192, 0.1197, 0.1199, 0.1156, 0.1072, 0.1007,  0.0942, 0.0899, 0.0832, 0.0824, 0.0793, 0.0755, 0.0801, 0.0860, 0.1031, 0.1305, 0.1809, 0.2260,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenkoriste[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0945, 0.0292, 0.0315, 0.0433, 0.0321, 0.0294, 0.0295, 0.0321, 0.0278,  0.0261, 0.0282, 0.0272, 0.0270,  0.0278, 0.0285, 0.0274, 0.0277, 0.0268, 0.0269, 0.0283,
    0.0275, 0.0309, 0.0325, 0.0389, 0.0450, 0.0493, 0.0551, 0.0557, 0.0587, 0.0585, 0.0567, 0.0554, 0.0515, 0.0487, 0.0460, 0.0424, 0.0409, 0.0387, 0.0353, 0.0349,
    0.0353, 0.0333, 0.0309, 0.0309, 0.0312, 0.0315, 0.0321, 0.0298, 0.0304, 0.0304, 0.0281, 0.0273, 0.0293, 0.0311, 0.0314, 0.0333, 0.0355, 0.0392, 0.0439, 0.0595, 0.0811,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenpoimulehti[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1102, 0.0146, 0.0630, 0.0266, 0.0247, 0.0261, 0.0285, 0.0238, 0.0273, 0.0281, 0.0272, 0.0260, 0.0262, 0.0254, 0.0274, 0.0263, 0.0273, 0.0278, 0.0296, 0.0309,  0.0322,
    0.0388, 0.0493, 0.0607, 0.0712, 0.0840, 0.0953, 0.0986, 0.1006, 0.1034, 0.0999, 0.0981, 0.0918, 0.0855, 0.0794, 0.0711, 0.0649, 0.0627, 0.0604, 0.0563, 0.0531, 0.0515,
    0.0467, 0.0450, 0.0448, 0.0466, 0.0445, 0.0421, 0.0402, 0.0385, 0.0369, 0.0345, 0.0346, 0.0319, 0.0330, 0.0359, 0.0378, 0.0439, 0.0578, 0.0835, 0.1177,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenhopeapaju[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0787, 0.0512, 0.1260, 0.1032, 0.0765, 0.0881, 0.0994, 0.0908, 0.0983, 0.0985, 0.0941, 0.0985, 0.0971, 0.0967, 0.0964, 0.0937, 0.0928, 0.0959, 0.0973, 0.0992, 0.1004,
    0.1017, 0.1053, 0.1102, 0.1180, 0.1227, 0.1281, 0.1309, 0.1317, 0.1328, 0.1318, 0.1271, 0.1238, 0.1222, 0.1179, 0.1152, 0.1131, 0.1092, 0.1086, 0.1078, 0.1083, 0.1020,
    0.1015, 0.1000, 0.1027, 0.1037, 0.1028, 0.0970, 0.0962, 0.0977,  0.0952, 0.0963, 0.0955, 0.0935, 0.0980, 0.0979, 0.0963, 0.1028, 0.1059, 0.1228, 0.1401,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorReduuden[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2288, 0.1861, 0.2364, 0.2229, 0.2783, 0.2842, 0.2842, 0.2923, 0.2902, 0.2990, 0.2828, 0.2871, 0.2772, 0.2723, 0.2639, 0.2558, 0.2424, 0.2315, 0.2169, 0.2094, 0.2064,
    0.1964, 0.1865, 0.1739, 0.1680, 0.1624, 0.1548, 0.1457, 0.1424, 0.1408, 0.1434, 0.1451, 0.1492, 0.1528, 0.1597, 0.1755, 0.1951, 0.2147, 0.2367, 0.2648, 0.2986, 0.3236,
    0.3393, 0.3596, 0.3665, 0.3786, 0.3879, 0.3915, 0.3926, 0.3994, 0.3987, 0.4017, 0.4026, 0.4112, 0.4067, 0.4125, 0.4139, 0.4121, 0.4050, 0.4040, 0.4095,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorRedpajuan[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0445, 0.1024, 0.0605, 0.0833, 0.0937, 0.0959, 0.1052, 0.1028, 0.1049, 0.1029, 0.1017, 0.0975, 0.0948, 0.0882, 0.0865, 0.0812, 0.0757, 0.0718, 0.0658, 0.0638, 0.0628, 0.0597,
    0.0554, 0.0523, 0.0509, 0.0509, 0.0485, 0.0475, 0.0469, 0.0492, 0.0479, 0.0477, 0.0490, 0.0525, 0.0555, 0.0597, 0.0641, 0.0704, 0.0756, 0.0846, 0.0948, 0.1055, 0.1164, 0.1221,
    0.1266, 0.1339, 0.1393, 0.1491, 0.1553, 0.1604, 0.1608, 0.1650, 0.1643, 0.1652, 0.1655, 0.1658, 0.1651, 0.1739, 0.1813, 0.1818, 0.1938,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorRedjaloan[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0120, 0.0152, 0.0512, 0.0635, 0.0662, 0.0538, 0.0749, 0.0713, 0.0743, 0.0792, 0.0777, 0.0785, 0.0733, 0.0726, 0.0728, 0.0749, 0.0709, 0.0674, 0.0661, 0.0657, 0.0645, 0.0635,
    0.0598, 0.0570, 0.0553, 0.0544, 0.0545, 0.0538, 0.0546, 0.0514, 0.0540, 0.0567, 0.0585, 0.0577, 0.0602, 0.0651, 0.0690, 0.0765, 0.0836, 0.0907, 0.0980, 0.1089, 0.1147, 0.1188,
    0.1212, 0.1253, 0.1318, 0.1371, 0.1412, 0.1473, 0.1459, 0.1478, 0.1548, 0.1582, 0.1564, 0.1590, 0.1595, 0.1714, 0.1728, 0.1814, 0.1837,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueukon[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0801, 0.0682, 0.1070, 0.1319, 0.1311, 0.1420, 0.1453, 0.1394, 0.1318, 0.1292, 0.1268, 0.1179, 0.1132, 0.1054, 0.1015, 0.0948, 0.0846, 0.0780, 0.0731, 0.0709, 0.0705, 0.0667,
    0.0621, 0.0598, 0.0555, 0.0522, 0.0505, 0.0493, 0.0498, 0.0500, 0.0494, 0.0471, 0.0479, 0.0463, 0.0450, 0.0461, 0.0487, 0.0515, 0.0546, 0.0574, 0.0555, 0.0562, 0.0539, 0.0558,
    0.0546, 0.0552, 0.0567, 0.0626, 0.0715, 0.0807, 0.0862, 0.0978, 0.1086, 0.1199, 0.1313, 0.1323, 0.1350, 0.1366, 0.1358, 0.1320, 0.1365,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueorvokki[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0292, 0.0528, 0.0645, 0.1000, 0.0932, 0.1439, 0.1752, 0.1947, 0.2077, 0.2158, 0.2169, 0.2153, 0.2164, 0.2132, 0.2091, 0.1993, 0.1916, 0.1876, 0.1803, 0.1702, 0.1659, 0.1554,
    0.1503, 0.1425, 0.1330, 0.1229, 0.1186, 0.1134, 0.1065, 0.1031, 0.1014, 0.0993, 0.0989, 0.0980, 0.0939, 0.0936, 0.0945, 0.0995, 0.1055, 0.1104, 0.1180, 0.1247, 0.1284, 0.1343,
    0.1349, 0.1403, 0.1458, 0.1538, 0.1634, 0.1790, 0.1880, 0.2006, 0.2218, 0.2396, 0.2556, 0.2612, 0.2735, 0.2811, 0.2765, 0.2840, 0.2877,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBluemalvikki[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1062, 0.0528, 0.0749, 0.0571, 0.0512, 0.0538, 0.0455, 0.0445, 0.0431, 0.0384, 0.0353, 0.0299, 0.0249, 0.0212, 0.0190, 0.0162, 0.0123, 0.0105, 0.0090, 0.0081, 0.0083, 0.0075,
    0.0078, 0.0072, 0.0059, 0.0056, 0.0061, 0.0054, 0.0071, 0.0069, 0.0070, 0.0062, 0.0063, 0.0065, 0.0070, 0.0074, 0.0078, 0.0073, 0.0076, 0.0087, 0.0090, 0.0104, 0.0119, 0.0119,
    0.0131, 0.0145, 0.0156, 0.0184, 0.0225, 0.0255, 0.0314, 0.0414, 0.0538, 0.0669, 0.0849, 0.1068, 0.1247, 0.1467, 0.1701, 0.1885, 0.2032,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlackmaito[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0256,  0.0, 0.0172, 0.0356, 0.0213, 0.0270, 0.0203, 0.0203, 0.0195, 0.0208, 0.0202, 0.0201, 0.0210, 0.0210, 0.0203, 0.0204, 0.0209, 0.0203, 0.0209, 0.0201, 0.0205, 0.0201,
    0.0194, 0.0210, 0.0206, 0.0197, 0.0203, 0.0198, 0.0207, 0.0201, 0.0204, 0.0202, 0.0198, 0.0200, 0.0198, 0.0197, 0.0186, 0.0203, 0.0202, 0.0198, 0.0200, 0.0208, 0.0206, 0.0231,
    0.0235, 0.0223, 0.0244, 0.0254, 0.0278, 0.0289, 0.0297, 0.0309, 0.0338, 0.0335, 0.0338, 0.0368, 0.0412, 0.0524, 0.0686, 0.0926, 0.1185,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorOrangpihlaja[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0945, 0.0731, 0.0585, 0.0433, 0.0247, 0.0408, 0.0266, 0.0314, 0.0293, 0.0305, 0.0289, 0.0288, 0.0280, 0.0297, 0.0262, 0.0298, 0.0277, 0.0274, 0.0291, 0.0293, 0.0285, 0.0303,
    0.0300, 0.0310, 0.0324, 0.0336, 0.0364, 0.0377, 0.0426, 0.0465, 0.0499, 0.0561, 0.0602, 0.0667, 0.0741, 0.0890, 0.1028, 0.1164, 0.1275, 0.1465, 0.1602, 0.1640, 0.1695, 0.1744,
    0.1812, 0.1837, 0.1859, 0.1805, 0.1791, 0.1822, 0.1796, 0.1751, 0.1715, 0.1655, 0.1575, 0.1600, 0.1560, 0.1618, 0.1666, 0.1740, 0.1838,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorGreenlinden[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2677, 0.1682, 0.1170, 0.1032, 0.1085, 0.0816, 0.0728, 0.0755, 0.0833, 0.0832, 0.0813, 0.0845, 0.0857, 0.0884, 0.0855, 0.0882, 0.0914, 0.0997, 0.1084, 0.1179, 0.1231, 0.1437,
    0.1661, 0.1873, 0.2048, 0.2209, 0.2378, 0.2408, 0.2442, 0.2509, 0.2503, 0.2452, 0.2457, 0.2418, 0.2383, 0.2348, 0.2277, 0.2213, 0.2221, 0.2169, 0.2146, 0.2048, 0.1977, 0.1960,
    0.2000, 0.1993, 0.1961, 0.1899, 0.1784, 0.1748, 0.1625, 0.1517, 0.1389, 0.1260, 0.1165, 0.1143, 0.1244, 0.1522, 0.1870, 0.2324, 0.2586,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorYellowlehmus[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1890, 0.1097,  0.0900, 0.0832, 0.0814, 0.0799, 0.0758, 0.0853,  0.0803,  0.0808,  0.0833,  0.0862,  0.0916,  0.0943,  0.0960,  0.0969, 0.1039, 0.1162, 0.1283, 0.1370, 0.1427,
    0.1529, 0.1689, 0.1781, 0.1894, 0.1950, 0.2105, 0.2118, 0.2140, 0.2185, 0.2191, 0.2199, 0.2234, 0.2266, 0.2263, 0.2297, 0.2328, 0.2312, 0.2298, 0.2332, 0.2344, 0.2312, 0.2288,
    0.2347, 0.2384, 0.2390, 0.2358, 0.2280, 0.2306, 0.2315, 0.2310, 0.2253, 0.2274, 0.2271, 0.2242, 0.2292, 0.2254, 0.2208, 0.2319, 0.2314, 0.2264,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorYellowsuikeroalpi[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2048, 0.1666, 0.1140, 0.1210, 0.0699, 0.0680, 0.0615, 0.0566, 0.0567, 0.0561, 0.0609, 0.0585, 0.0614, 0.0572, 0.0599, 0.0575, 0.0636, 0.0730, 0.0982, 0.1194, 0.1360, 0.1766,
    0.2222, 0.2558, 0.2849, 0.3048, 0.3201, 0.3395, 0.3395, 0.3484, 0.3576, 0.3623, 0.3606, 0.3672, 0.3651, 0.3634, 0.3647, 0.3669, 0.3715, 0.3660, 0.3720, 0.3692, 0.3704, 0.3784,
    0.3683, 0.3731, 0.3681, 0.3697, 0.3635, 0.3694, 0.3617, 0.3610, 0.3632, 0.3663, 0.3616, 0.3595, 0.3599, 0.3584, 0.3588, 0.3613, 0.3527,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorYellowpensashanhikki1[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0356, 0.1365, 0.1024,  0.0902, 0.0535, 0.0387, 0.0355, 0.0247, 0.0365, 0.0307, 0.0321, 0.0330,  0.0319,  0.0332,  0.0317,  0.0336, 0.0408,  0.0487, 0.0709,  0.0963, 0.1235,
    0.1631, 0.2111, 0.2436, 0.2718, 0.2950, 0.3151, 0.3262, 0.3313, 0.3420, 0.3448, 0.3475, 0.3491, 0.3534, 0.3520, 0.3565, 0.3622, 0.3631, 0.3626, 0.3657, 0.3640, 0.3607,
    0.3641, 0.3627, 0.3601, 0.3591, 0.3588, 0.3667, 0.3618, 0.3601, 0.3630, 0.3613, 0.3592, 0.3609, 0.3569, 0.3590, 0.3568, 0.3563, 0.3588, 0.3480, 0.3471,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorYellowpensashanhikki2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1068, 0.0427, 0.0326, 0.0416, 0.0428, 0.0590, 0.0503, 0.0470, 0.0539, 0.0526, 0.0546, 0.0540, 0.0539, 0.0526, 0.0497, 0.0546, 0.0555, 0.0603, 0.0753, 0.0903, 0.1010, 0.1268,
    0.1563, 0.1868, 0.2068, 0.2226, 0.2429, 0.2495, 0.2560, 0.2625, 0.2636, 0.2610, 0.2655, 0.2667, 0.2635, 0.2630, 0.2612, 0.2560, 0.2597, 0.2588, 0.2543, 0.2478, 0.2499, 0.2472,
    0.2438, 0.2431, 0.2379, 0.2406, 0.2361, 0.2319, 0.2264, 0.2174, 0.2128, 0.2010, 0.1942, 0.1912, 0.1930, 0.2148, 0.2334, 0.2585, 0.2764,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBluehiidenvirna[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0315, 0.0512, 0.0675, 0.0832, 0.0765, 0.0865, 0.0807, 0.0867, 0.0911, 0.0904, 0.0852, 0.0826, 0.0780, 0.0753, 0.0711, 0.0661, 0.0595, 0.0528, 0.0513, 0.0476, 0.0431, 0.0391,
    0.0361, 0.0331, 0.0305, 0.0276, 0.0240, 0.0229, 0.0237, 0.0223, 0.0212, 0.0208, 0.0215, 0.0205, 0.0203, 0.0195, 0.0209, 0.0212, 0.0222, 0.0266, 0.0296, 0.0322, 0.0356,  0.0352,
    0.0388, 0.0391, 0.0411, 0.0425, 0.0473, 0.0532, 0.0550, 0.0630, 0.0669, 0.0748, 0.0823, 0.0879, 0.0904, 0.0917, 0.0930, 0.0950, 0.0942,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBluekurkkuyrtti[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2687, 0.1553, 0.2181, 0.2246, 0.2209, 0.2263, 0.2442, 0.2347, 0.2261, 0.2353, 0.2292, 0.2230, 0.2095, 0.2008, 0.1896, 0.1782, 0.1569, 0.1443, 0.1333, 0.1223, 0.1177, 0.1074,
    0.0992, 0.0902, 0.0813, 0.0755, 0.0701, 0.0626, 0.0577, 0.0548, 0.0525, 0.0498, 0.0469, 0.0445, 0.0456, 0.0448, 0.0428, 0.0441, 0.0448, 0.0447, 0.0455, 0.0467, 0.0496, 0.0534,
    0.0527, 0.0586, 0.0668, 0.0798, 0.0966, 0.1126, 0.1289, 0.1469, 0.1679, 0.1870, 0.2013, 0.2040, 0.2060, 0.2077, 0.2104, 0.2155, 0.2119,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorPinksiankaersaemoe[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0585, 0.0859, 0.0758, 0.1094, 0.0780, 0.1012, 0.0987, 0.1067, 0.1059, 0.1034, 0.1098, 0.1110, 0.1097, 0.1040, 0.1058, 0.1048, 0.1028, 0.1014, 0.1068, 0.1024, 0.1023, 0.1025,
    0.1032, 0.1029, 0.1011, 0.1007, 0.0986, 0.0973, 0.0946, 0.0906, 0.0949, 0.0923, 0.0943, 0.0949, 0.0956, 0.0998, 0.1051, 0.1107, 0.1166, 0.1242, 0.1284, 0.1355, 0.1394, 0.1438,
    0.1451, 0.1543, 0.1589, 0.1588, 0.1612, 0.1616, 0.1562, 0.1534, 0.1562, 0.1541, 0.1494, 0.1492, 0.1518, 0.1650, 0.1749, 0.1907, 0.1991,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorVioletharakankello[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2520, 0.1462, 0.1890, 0.1898, 0.1751, 0.1713, 0.1555, 0.1516, 0.1471, 0.1403, 0.1282, 0.1209, 0.1144, 0.1135, 0.1069, 0.0976, 0.0895, 0.0823, 0.0782, 0.0762, 0.0713,
    0.0685, 0.0661, 0.0635, 0.0603, 0.0559, 0.0551, 0.0550, 0.0541, 0.0567, 0.0562, 0.0574, 0.0580, 0.0589, 0.0586, 0.0620, 0.0670, 0.0690, 0.0718, 0.0801, 0.0786, 0.0769,
    0.0773, 0.0739, 0.0800, 0.0806, 0.0837, 0.0845, 0.0971, 0.1043, 0.1102, 0.1241, 0.1359, 0.1502, 0.1611, 0.1726, 0.1793, 0.1859, 0.1909, 0.1969, 0.2014,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorVioletalsikeapila[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1260, 0.0585, 0.0765, 0.1065, 0.0937, 0.0881, 0.0847, 0.0943, 0.1075, 0.1053, 0.1020, 0.0994, 0.1008, 0.1026, 0.1015,  0.0980,  0.0962,  0.0949,  0.0925, 0.0908, 0.0880, 0.0864,
    0.0843, 0.0814, 0.0802, 0.0749, 0.0698, 0.0691, 0.0677, 0.0660, 0.0653, 0.0660, 0.0631, 0.0633, 0.0644, 0.0692, 0.0743, 0.0809, 0.0889, 0.1005, 0.1160, 0.1325, 0.1396, 0.1450,
    0.1526, 0.1583, 0.1655, 0.1674, 0.1689, 0.1707, 0.1675, 0.1674, 0.1624, 0.1576, 0.1564, 0.1591, 0.1613, 0.1717, 0.1851, 0.1962, 0.2033,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorVioletakilleija[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0843, 0.1340, 0.1393, 0.1254, 0.1290, 0.1452, 0.1508, 0.1519, 0.1454, 0.1384, 0.1301, 0.1256, 0.1178, 0.1113, 0.1056, 0.0985, 0.0884, 0.0827, 0.0743, 0.0720, 0.0691, 0.0664,
    0.0605, 0.0578, 0.0540, 0.0507, 0.0499, 0.0475, 0.0485, 0.0494, 0.0492, 0.0479, 0.0487, 0.0493, 0.0471, 0.0495, 0.0559, 0.0595, 0.0645, 0.0689, 0.0720, 0.0732, 0.0716, 0.0723,
    0.0734, 0.0750, 0.0804, 0.0849, 0.0948, 0.1041, 0.1169, 0.1362, 0.1525, 0.1693, 0.1761, 0.1935, 0.2071, 0.2235, 0.2376, 0.2493, 0.2604,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorOrangekehaekukka[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0102, 0.0058, 0.0128, 0.0125, 0.0137, 0.0122, 0.0122, 0.0134, 0.0123, 0.0136, 0.0159, 0.0138, 0.0163, 0.0151, 0.0167, 0.0178, 0.0192,
    0.0177, 0.0206, 0.0226, 0.0315, 0.0451, 0.0707, 0.1045, 0.1446, 0.1707, 0.1944, 0.2131, 0.2276, 0.2524, 0.2719, 0.2841, 0.2968, 0.3052, 0.3199, 0.3264, 0.3282, 0.3429,
    0.3451, 0.3454, 0.3477, 0.3556, 0.3478, 0.3565, 0.3595, 0.3569, 0.3582, 0.3582, 0.3559, 0.3610, 0.3626, 0.3668, 0.3733, 0.3692, 0.3722,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorRedpihlaja[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0,  0.0,  0.0,  0.0,  0.0025,  0.0084, 0.0128,  0.0087, 0.0165, 0.0130, 0.0167, 0.0147, 0.0155, 0.0146, 0.0148, 0.0165, 0.0158, 0.0159, 0.0164, 0.0160, 0.0158, 0.0158,
    0.0157, 0.0157, 0.0173, 0.0173, 0.0179, 0.0195, 0.0210, 0.0234, 0.0264, 0.0302, 0.0349, 0.0386, 0.0461, 0.0572, 0.0735, 0.0886, 0.1038, 0.1216, 0.1376, 0.1521, 0.1607, 0.1691,
    0.1728, 0.1769, 0.1842, 0.1843, 0.1865, 0.1910, 0.1881, 0.1920, 0.1909, 0.1909, 0.1891, 0.1879, 0.1915, 0.1879, 0.1878, 0.1843, 0.1832,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorVioletpetunia[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0292, 0.0066, 0.0227, 0.0313, 0.0325, 0.0332, 0.0310, 0.0319, 0.0300, 0.0268, 0.0229, 0.0193, 0.0164, 0.0137, 0.0127, 0.0104, 0.0067, 0.0050, 0.0049, 0.0037, 0.0034, 0.0034,
    0.0044, 0.0032, 0.0033, 0.0031, 0.0038, 0.0041, 0.0033, 0.0034, 0.0045, 0.0044, 0.0037, 0.0042, 0.0044, 0.0049, 0.0049, 0.0067, 0.0080, 0.0089, 0.0110, 0.0130, 0.0137, 0.0145,
    0.0153, 0.0171, 0.0194, 0.0223, 0.0275, 0.0321, 0.0391, 0.0464, 0.0580, 0.0720, 0.0907, 0.1055, 0.1230, 0.1436, 0.1548, 0.1777, 0.1933,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorVioletorvokki[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0195,  0.0, 0.0152, 0.0031, 0.0022, 0.0032, 0.0077, 0.0069, 0.0045, 0.0020, 0.0027, 0.0026, 0.0013, 0.0012, 0.0014, 0.0011, 0.0015, 0.0023, 0.0016, 0.0018, 0.0016, 0.0012, 0.0016,
    0.0010, 0.0016, 0.0015, 0.0013, 0.0015, 0.0015, 0.0016, 0.0014, 0.0020, 0.0014, 0.0017, 0.0015, 0.0017, 0.0015, 0.0017, 0.0017, 0.0017, 0.0023, 0.0024, 0.0029, 0.0038, 0.0036,
    0.0038, 0.0037, 0.0043, 0.0058, 0.0065, 0.0070, 0.0086, 0.0113, 0.0155, 0.0222, 0.0285, 0.0405, 0.0506, 0.0645, 0.0817, 0.1035,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBluesinisievikki[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0801, 0.1109, 0.1861, 0.2325, 0.2329, 0.2380, 0.2562, 0.2565, 0.2558, 0.2611, 0.2517, 0.2567, 0.2475, 0.2397, 0.2337, 0.2294, 0.2195, 0.2001, 0.1881, 0.1892, 0.1854, 0.1746,
    0.1668, 0.1580, 0.1491, 0.1362, 0.1229, 0.1178, 0.1110, 0.1094, 0.1072, 0.1019,  0.0994, 0.0960, 0.0928, 0.0879, 0.0836, 0.0859, 0.0863, 0.0951, 0.1046, 0.1102, 0.1154, 0.1193,
    0.1174, 0.1166, 0.1153, 0.1199, 0.1275, 0.1316, 0.1376, 0.1550, 0.1739, 0.1918, 0.2104, 0.2228, 0.2364, 0.2377, 0.2423, 0.2394, 0.2334,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueiisoppi[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0623, 0.0085, 0.0605, 0.0833, 0.0776, 0.0756, 0.0755, 0.0781, 0.0774, 0.0775, 0.0697, 0.0724, 0.0697, 0.0654, 0.0617, 0.0575, 0.0536, 0.0494, 0.0460, 0.0469, 0.0442, 0.0436,
    0.0400, 0.0393, 0.0380, 0.0358, 0.0369, 0.0352, 0.0342, 0.0368, 0.0357, 0.0360, 0.0342, 0.0342, 0.0341, 0.0335, 0.0355, 0.0353, 0.0365, 0.0376, 0.0382, 0.0392, 0.0412, 0.0412,
    0.0407, 0.0414, 0.0420, 0.0449, 0.0487, 0.0504, 0.0517, 0.0571, 0.0651, 0.0734, 0.0806, 0.0885, 0.0968, 0.1088, 0.1210, 0.1296, 0.1411,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorWhiteojaka[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1732, 0.0951, 0.1800, 0.1365, 0.1801, 0.1697, 0.1762, 0.2103, 0.2243, 0.2218, 0.2200, 0.2206, 0.2255, 0.2254, 0.2269, 0.2261, 0.2272, 0.2251, 0.2254, 0.2260, 0.2256, 0.2266,
    0.2247, 0.2269, 0.2310, 0.2273, 0.2345, 0.2312, 0.2301, 0.2323, 0.2302, 0.2314, 0.2362, 0.2355, 0.2348, 0.2362, 0.2396, 0.2374, 0.2362, 0.2381, 0.2396, 0.2440, 0.2383, 0.2347,
    0.2422, 0.2419, 0.2472, 0.2423, 0.2406, 0.2425, 0.2377, 0.2381, 0.2380, 0.2398, 0.2390, 0.2404, 0.2370, 0.2375, 0.2364, 0.2411, 0.2417,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorWhitepetunia[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4095, 0.4022, 0.4410, 0.4095, 0.4095, 0.4095, 0.4193, 0.4207, 0.4388, 0.4328, 0.4223, 0.4168, 0.4221, 0.4304, 0.4245, 0.4210, 0.4212, 0.4192, 0.4181, 0.4233, 0.4207, 0.4224,
    0.4197, 0.4262, 0.4243, 0.4241, 0.4274, 0.4257, 0.4204, 0.4285, 0.4265, 0.4241, 0.4267, 0.4275, 0.4245, 0.4276, 0.4260, 0.4217, 0.4217, 0.4244, 0.4240, 0.4186, 0.4160, 0.4156,
    0.4227, 0.4286, 0.4237, 0.4137, 0.4202, 0.4187, 0.4100, 0.4112, 0.4103, 0.4090, 0.4125, 0.4115, 0.4098, 0.4036, 0.4047, 0.4105, 0.4050,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::ColorBluelobelia[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0, 0.0660, 0.1277, 0.1544, 0.1612, 0.1961, 0.1909, 0.1950, 0.1901, 0.1907, 0.1809, 0.1785, 0.1685, 0.1622, 0.1522, 0.1377, 0.1178, 0.1054, 0.0931, 0.0898, 0.0850, 0.0732,
    0.0610, 0.0508, 0.0434, 0.0370, 0.0343, 0.0329, 0.0303, 0.0265, 0.0232, 0.0199, 0.0183, 0.0169, 0.0172, 0.0177, 0.0200, 0.0233, 0.0214, 0.0214, 0.0199, 0.0186, 0.0199,
    0.0228, 0.0249, 0.0321, 0.0435, 0.0684, 0.1006, 0.1345, 0.1703, 0.2082, 0.2432, 0.2661, 0.2843, 0.2936, 0.3079, 0.3015, 0.3003, 0.3045, 0.3038,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorWhitepelargonia[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3493, 0.2882, 0.2284, 0.2730, 0.2869, 0.2609, 0.2781, 0.2869, 0.2861, 0.2869, 0.2795, 0.2810, 0.2740, 0.2716, 0.2650, 0.2631, 0.2539, 0.2554, 0.2450, 0.2453, 0.2447,
    0.2451, 0.2343, 0.2408, 0.2404, 0.2367, 0.2343, 0.2401, 0.2474, 0.2549, 0.2668, 0.2759, 0.2843, 0.2883, 0.2989, 0.3106, 0.3209, 0.3344, 0.3383, 0.3404, 0.3453, 0.3521,
    0.3495, 0.3571, 0.3521, 0.3548, 0.3582, 0.3557, 0.3581, 0.3539, 0.3563, 0.3589, 0.3597, 0.3579, 0.3502, 0.3546, 0.3507, 0.3554, 0.3490, 0.3561, 0.3518,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorWhitepaeivaen[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2168, 0.1365, 0.1969, 0.2095, 0.2231, 0.2530, 0.2944, 0.3092, 0.3107, 0.3148, 0.3188, 0.3207, 0.3195, 0.3216, 0.3225, 0.3261, 0.3211, 0.3228, 0.3260, 0.3237, 0.3258,
    0.3276, 0.3265, 0.3316, 0.3327, 0.3291, 0.3315, 0.3324, 0.3355, 0.3255, 0.3264, 0.3308, 0.3324, 0.3328, 0.3282, 0.3253, 0.3220, 0.3257, 0.3289, 0.3265, 0.3245, 0.3297,
    0.3284, 0.3292, 0.3228, 0.3312, 0.3290, 0.3277, 0.3278, 0.3284, 0.3182, 0.3244, 0.3273, 0.3291, 0.3212, 0.3256, 0.3154, 0.3243, 0.3306, 0.3234, 0.3155,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechGreB3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0531, 0.0545, 0.0559, 0.0563, 0.0566, 0.0571, 0.0576, 0.0576, 0.0575, 0.0581, 0.0586, 0.0596, 0.0606, 0.0629, 0.0652, 0.0699, 0.0745, 0.0839, 0.0932, 0.1101, 0.1270, 0.1521, 0.1771, 0.2098, 0.2424,
    0.2789, 0.3154, 0.3312, 0.3470, 0.3431, 0.3392, 0.3303, 0.3213, 0.3089, 0.2964, 0.2788, 0.2612, 0.2442, 0.2271, 0.2117, 0.1962, 0.1815, 0.1667, 0.1527, 0.1386, 0.1284, 0.1182, 0.1124, 0.1066, 0.1035, 0.1003,
    0.0987, 0.0971, 0.0961, 0.0950, 0.0950, 0.0950, 0.0962, 0.0973, 0.0994, 0.1014, 0.1045, 0.1075, 0.1106, 0.1137, 0.1157, 0.1176, 0.1175, 0.1173, 0.1173, 0.1173,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechCyaF3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0813, 0.1048, 0.1282, 0.1611, 0.1940, 0.2198, 0.2456, 0.2575, 0.2693, 0.2807, 0.2921, 0.3079, 0.3237, 0.3424, 0.3611, 0.3820, 0.4029, 0.4234, 0.4439, 0.4547, 0.4654, 0.4638, 0.4621, 0.4482, 0.4342, 0.4119, 0.3895,
    0.3656, 0.3417, 0.3160, 0.2903, 0.2654, 0.2404, 0.2167, 0.1929, 0.1720, 0.1510, 0.1368, 0.1226, 0.1138, 0.1049, 0.0993, 0.0936, 0.0890, 0.0844, 0.0810, 0.0776, 0.0759, 0.0742, 0.0733, 0.0724, 0.0723, 0.0722, 0.0727,
    0.0732, 0.0745, 0.0757, 0.0763, 0.0768, 0.0764, 0.0759, 0.0748, 0.0736, 0.0723, 0.0710, 0.0703, 0.0696, 0.0707, 0.0718, 0.0756, 0.0793, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechCyaF3_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0813, 0.1048, 0.1382, 0.1711, 0.1940, 0.2298, 0.2456, 0.2575, 0.2693, 0.2907, 0.3021, 0.3079, 0.3237, 0.3624, 0.3611, 0.3820, 0.4029, 0.4234, 0.4439, 0.4547, 0.4654, 0.4638, 0.4621, 0.4482, 0.4342, 0.4119, 0.3895,
    0.3656, 0.3417, 0.3160, 0.2903, 0.2654, 0.2404, 0.2167, 0.1929, 0.1720, 0.1510, 0.1368, 0.1226, 0.1138, 0.1049, 0.0993, 0.0936, 0.0890, 0.0844, 0.0810, 0.0776, 0.0759, 0.0742, 0.0733, 0.0724, 0.0723, 0.0722, 0.0727,
    0.0732, 0.0745, 0.0757, 0.0763, 0.0768, 0.0764, 0.0759, 0.0748, 0.0736, 0.0723, 0.0710, 0.0703, 0.0696, 0.0707, 0.0718, 0.0756, 0.0793, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechCyaF3_spect3[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0813, 0.1048, 0.1282, 0.1411, 0.1840, 0.2098, 0.2356, 0.2575, 0.2593, 0.2807, 0.2821, 0.3079, 0.3137, 0.3424, 0.3611, 0.3820, 0.4029, 0.4234, 0.4439, 0.4547, 0.4654, 0.4638, 0.4621, 0.4482, 0.4342, 0.4119, 0.3895,
    0.3656, 0.3417, 0.3160, 0.2903, 0.2654, 0.2404, 0.2167, 0.1929, 0.1720, 0.1510, 0.1368, 0.1226, 0.1138, 0.1049, 0.0993, 0.0936, 0.0890, 0.0844, 0.0810, 0.0776, 0.0759, 0.0742, 0.0733, 0.0724, 0.0723, 0.0722, 0.0727,
    0.0732, 0.0745, 0.0757, 0.0763, 0.0768, 0.0764, 0.0759, 0.0748, 0.0736, 0.0723, 0.0710, 0.0703, 0.0696, 0.0707, 0.0718, 0.0756, 0.0793, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechPurD2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0854, 0.1047, 0.1240, 0.1468, 0.1696, 0.1826, 0.1955, 0.1963, 0.1970, 0.1910, 0.1849, 0.1750, 0.1651, 0.1541, 0.1430, 0.1322, 0.1213, 0.1117, 0.1020, 0.0944, 0.0868, 0.0809, 0.0750, 0.0703, 0.0655,
    0.0627, 0.0599, 0.0583, 0.0567, 0.0550, 0.0533, 0.0525, 0.0517, 0.0518, 0.0519, 0.0523, 0.0526, 0.0525, 0.0524, 0.0520, 0.0516, 0.0523, 0.0529, 0.0560, 0.0591, 0.0662, 0.0732, 0.0828, 0.0924, 0.1021,
    0.1117, 0.1222, 0.1327, 0.1469, 0.1610, 0.1796, 0.1981, 0.2173, 0.2365, 0.2532, 0.2698, 0.2826, 0.2953, 0.3022, 0.3090, 0.3126, 0.3161, 0.3238, 0.3314, 0.3504, 0.3694, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechMagE3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1112, 0.1438, 0.1763, 0.2294, 0.2824, 0.3188, 0.3552, 0.3623, 0.3693, 0.3653, 0.3612, 0.3510, 0.3407, 0.3269, 0.3130, 0.2981, 0.2832, 0.2686, 0.2539, 0.2385, 0.2230, 0.2083, 0.1935, 0.1818, 0.1700, 0.1600, 0.1499,
    0.1394, 0.1288, 0.1188, 0.1088, 0.1051, 0.1014, 0.1026, 0.1038, 0.1041, 0.1043, 0.1064, 0.1085, 0.1225, 0.1364, 0.1701, 0.2037, 0.2532, 0.3027, 0.3587, 0.4147, 0.4683, 0.5219, 0.5672, 0.6124, 0.6455, 0.6785, 0.7009,
    0.7232, 0.7391, 0.7550, 0.7629, 0.7707, 0.7737, 0.7766, 0.7778, 0.7790, 0.7803, 0.7815, 0.7835, 0.7854, 0.7896, 0.7937, 0.8026, 0.8114, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechSkiA138_13_14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0479, 0.051, 0.0553, 0.058, 0.0610, 0.062, 0.0626, 0.0622, 0.0619, 0.0617, 0.0616, 0.0615, 0.0614, 0.0614, 0.0615, 0.0617, 0.0618, 0.0618, 0.0619, 0.0618, 0.0618, 0.062, 0.0622, 0.063, 0.0638, 0.066, 0.0696,
    0.073, 0.0767, 0.078, 0.0801, 0.0807, 0.0817, 0.0831, 0.0845, 0.0870, 0.0902, 0.0955, 0.1017, 0.1096, 0.1175, 0.1250, 0.1336, 0.1385, 0.1435, 0.1455, 0.1479, 0.1490, 0.1514, 0.1547, 0.1580, 0.1625, 0.1675,
    0.173, 0.1772, 0.181, 0.1842, 0.1846, 0.1853, 0.1831, 0.1811, 0.1788, 0.1765, 0.1769, 0.1773, 0.181, 0.1834, 0.1874, 0.1914, 0.1965, 0.2018, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechGraC4_67_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1074, 0.1380, 0.1704, 0.22, 0.2705, 0.305, 0.3409, 0.35, 0.3601, 0.3628, 0.3655, 0.3675, 0.3698, 0.371, 0.3724, 0.373, 0.3733, 0.3725, 0.3715, 0.3705, 0.3692,
    0.369, 0.3689, 0.368, 0.3673, 0.3678, 0.3684, 0.37, 0.3711, 0.3712, 0.3714, 0.3714, 0.3714, 0.371, 0.3707, 0.37, 0.3694, 0.3697, 0.3703, 0.3697, 0.3692, 0.3688, 0.3685, 0.3675, 0.3669, 0.3657, 0.3647, 0.3635, 0.3625, 0.361,
    0.3596, 0.3585, 0.3579, 0.357, 0.3560, 0.3555, 0.3548, 0.3535, 0.3526, 0.3513, 0.3500, 0.349, 0.3475, 0.3467, 0.3460, 0.3452, 0.3444, 0.3431, 0.3421, 0.3411, 0.3403, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Fictif_61greyspect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
    0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
    0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::JDC468_K15_87greyspect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1984, 0.2223, 0.2448, 0.2934, 0.3415, 0.4425, 0.5707, 0.6609, 0.7619, 0.7956, 0.8275, 0.8280, 0.8292, 0.8223, 0.8156, 0.8112, 0.8076, 0.8040, 0.7982, 0.7970, 0.7954, 0.8013, 0.8083, 0.8141, 0.8184, 0.8167,
    0.8137, 0.8080, 0.8026, 0.8013, 0.7988, 0.7963, 0.7942, 0.7855, 0.7765, 0.7680, 0.7603, 0.7640, 0.7681, 0.7750, 0.7827, 0.7876, 0.7923, 0.7935, 0.7945, 0.7955, 0.7964, 0.7975, 0.7982, 0.8000, 0.8017, 0.8051,
    0.8090, 0.8145, 0.8191, 0.8234, 0.8269, 0.8300, 0.8327, 0.8342, 0.8359, 0.8375, 0.8390, 0.8405, 0.8421, 0.8436, 0.8452, 0.8480, 0.8504, 0.8564, 0.8611,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0

};

const double ColorTemp::ColorchechSkiB166_18_18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0962, 0.114, 0.1328, 0.152, 0.1706, 0.1755, 0.1877, 0.189, 0.1903, 0.1913, 0.1923, 0.1946, 0.1971, 0.2015, 0.2064, 0.215, 0.2245, 0.239, 0.2535, 0.273, 0.2922, 0.31, 0.3274, 0.337, 0.3473,
    0.348, 0.3489, 0.335, 0.3224, 0.303, 0.2835, 0.275, 0.2671, 0.27, 0.2728, 0.2735, 0.2741, 0.279, 0.2836, 0.308, 0.3334, 0.375, 0.4183, 0.457, 0.4950, 0.516, 0.5409, 0.5515, 0.5625, 0.568, 0.5731, 0.5786,
    0.5820, 0.586, 0.5902, 0.596, 0.6025, 0.611, 0.6174, 0.627, 0.6375, 0.65, 0.6626, 0.677, 0.6910, 0.704, 0.7171, 0.723, 0.7339, 0.741, 0.7475,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechBluC150_m5_m22_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1053, 0.134, 0.1633, 0.2075, 0.2518, 0.283, 0.3163, 0.324, 0.3325, 0.334, 0.3355, 0.3352, 0.3349, 0.332, 0.3294, 0.325, 0.3199, 0.3127, 0.3055, 0.2955, 0.2863, 0.28, 0.2737, 0.267, 0.2612, 0.249, 0.2378, 0.228, 0.2199,
    0.218, 0.2173, 0.2146, 0.2118, 0.20, 0.1884, 0.178, 0.1682, 0.166, 0.1639, 0.162, 0.1613, 0.158, 0.1550, 0.1504, 0.1458, 0.1415, 0.1375, 0.135, 0.1327, 0.1316, 0.1305, 0.1304, 0.1302, 0.131, 0.1322, 0.1342, 0.1362,
    0.1367, 0.1372, 0.1356, 0.1340, 0.1311, 0.1288, 0.1253, 0.1227, 0.1205, 0.1187, 0.1195, 0.1205, 0.1255, 0.1303, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechDCBluN881_m7_m14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1371, 0.17, 0.2029, 0.291, 0.3790, 0.495, 0.6100, 0.67, 0.7249, 0.737, 0.7501, 0.7545, 0.7597, 0.764, 0.7677, 0.7685, 0.7693, 0.7677, 0.7662, 0.763, 0.7593, 0.753, 0.7471, 0.737, 0.7289, 0.718, 0.7077, 0.705, 0.6819, 0.666, 0.6515, 0.636, 0.6244,
    0.61, 0.5948, 0.577, 0.5581, 0.544, 0.5293, 0.522, 0.5147, 0.512, 0.5091, 0.506, 0.5029, 0.499, 0.4950, 0.494, 0.4931, 0.497, 0.5007, 0.508, 0.5176, 0.527, 0.5359, 0.542, 0.5487, 0.549, 0.5494, 0.544,
    0.5375, 0.531, 0.5244, 0.522, 0.5207, 0.524, 0.5264, 0.532, 0.5369, 0.542, 0.5505, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechSGSkiF763_14_26_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0508, 0.064, 0.0776, 0.903, 0.1099, 0.1128, 0.1256, 0.128, 0.1307, 0.133, 0.1357, 0.139, 0.1425, 0.148, 0.1523, 0.159, 0.1669, 0.177, 0.1871, 0.20, 0.2118, 0.2235, 0.2355, 0.2445, 0.2537, 0.259, 0.2655, 0.268,
    0.2700, 0.2708, 0.2716, 0.2743, 0.2770, 0.2803, 0.2827, 0.283, 0.2832, 0.283, 0.2828, 0.295, 0.3079, 0.344, 0.3803, 0.4105, 0.4409, 0.455, 0.4694, 0.477, 0.4851, 0.4896, 0.4962, 0.501, 0.5066, 0.511, 0.5160, 0.521,
    0.5256, 0.529, 0.5318, 0.535, 0.5383, 0.541, 0.5451, 0.549, 0.5524, 0.556, 0.5597, 0.562, 0.5650, 0.568, 0.5709, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechSGSkiK285_11_17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1122, 0.149, 0.1866, 0.259, 0.3318, 0.393, 0.4547, 0.469, 0.4846, 0.4845, 0.4844, 0.4838, 0.4834, 0.4837, 0.4840, 0.4847, 0.4854, 0.4852, 0.4849, 0.4842, 0.4835, 0.4832, 0.4828, 0.485,
    0.4874, 0.501, 0.5150, 0.536, 0.5572, 0.5685, 0.5798, 0.586, 0.5932, 0.5987, 0.6142, 0.6342, 0.6541, 0.683, 0.7119, 0.734, 0.7571, 0.769, 0.7829, 0.788, 0.7932, 0.795, 0.7968, 0.7973,
    0.7977, 0.7974, 0.7969, 0.797, 0.7972, 0.7973, 0.7975, 0.7983, 0.7990, 0.7978, 0.7965, 0.7957, 0.7949, 0.7944, 0.7940, 0.794, 0.7941, 0.7943, 0.7946, 0.7938, 0.7930, 0.7929, 0.7928,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechWhiA496_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1267, 0.172, 0.2179, 0.317, 0.4164, 0.505, 0.6780, 0.758, 0.8397, 0.865, 0.8911, 0.897, 0.9035, 0.9062, 0.9092, 0.9124, 0.9154, 0.9167, 0.9180, 0.9187, 0.9194, 0.92, 0.9225, 0.9217, 0.9209, 0.921,
    0.9212, 0.9227, 0.9242, 0.9235, 0.9227, 0.9232, 0.9238, 0.9243, 0.9248, 0.9237, 0.9227, 0.9239, 0.9252, 0.924, 0.9233, 0.9238, 0.9242, 0.924, 0.9239, 0.9239, 0.9239, 0.924, 0.9242, 0.9239,
    0.9237, 0.925, 0.9264, 0.9276, 0.9288, 0.9298, 0.9308, 0.9296, 0.9284, 0.928, 0.9276, 0.928, 0.9284, 0.9294, 0.9304, 0.9316, 0.9328, 0.9328, 0.9328, 0.9337, 0.9345,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechGreD1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0477, 0.0492, 0.0507, 0.0517, 0.0527, 0.0532, 0.0537, 0.054, 0.0544, 0.0554, 0.0563, 0.0573, 0.0584, 0.0592, 0.0601, 0.0607, 0.0611, 0.0613, 0.0619, 0.626, 0.0634, 0.0646, 0.0659, 0.069,
    0.0721, 0.0837, 0.0958, 0.117, 0.1386, 0.156, 0.1748, 0.1802, 0.1855, 0.1795, 0.1742, 0.1636, 0.1529, 0.144, 0.1351, 0.13, 0.1239, 0.1202, 0.1171, 0.1138, 0.1106, 0.108, 0.1048, 0.1035,
    0.1022, 0.1025, 0.1030, 0.1041, 0.1052, 0.106, 0.1068, 0.107, 0.1073, 0.1066, 0.1060, 0.1047, 0.1035, 0.1028, 0.1021, 0.1029, 0.1034, 0.105, 0.1069, 0.1086, 0.1104, 0.1127, 0.1150,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorchechSGBlaN3_6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0066, 0.0069, 0.0071, 0.0072, 0.0074, 0.0073, 0.0072, 0.0073, 0.0074, 0.0074, 0.0074, 0.0074, 0.0074, 0.0073, 0.0073, 0.0073, 0.0073, 0.0072, 0.0072, 0.0072, 0.0072, 0.0071, 0.0071, 0.0071,
    0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0070, 0.0070, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0071, 0.0070,
    0.0070, 0.0070, 0.0070, 0.0069, 0.0069, 0.0069, 0.0068, 0.0068, 0.0068, 0.0068, 0.0068, 0.0068, 0.0067, 0.0067, 0.0067, 0.0067, 0.0066, 0.0066, 0.0066, 0.0066, 0.0066, 0.0066, 0.0066, 0.0067,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_GraK14_44_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.04240, 0.0485, 0.05500, 0.0624, 0.06930, 0.084, 0.09820, 0.109, 0.12160, 0.127, 0.13300, 0.13490, 0.13690, 0.1379, 0.13890, 0.1396, 0.14060, 0.1407, 0.14080, 0.1423, 0.14380, 0.1488, 0.15370, 0.157, 0.16040,
    0.157, 0.15360, 0.1482, 0.14290, 0.1436, 0.14470, 0.1454, 0.14620, 0.137, 0.12870, 0.1205, 0.11250, 0.116, 0.11930, 0.1261, 0.13350, 0.1367, 0.13990, 0.139, 0.13810, 0.1371, 0.13610, 0.1372, 0.13820,
    0.1408, 0.14330, 0.1475, 0.15170, 0.1583, 0.16500, 0.172, 0.17940, 0.1836, 0.18780, 0.188, 0.18820, 0.186, 0.18430, 0.1801, 0.17620, 0.1741, 0.17210, 0.179, 0.18420, 0.1991, 0.21430,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::JDC468_BluM5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1510, 0.1802, 0.2069, 0.2550,  0.3047, 0.4055, 0.5069, 0.590, 0.6747, 0.701,   0.7351, 0.7345, 0.7338, 0.7195, 0.7063, 0.693,  0.6732, 0.6490, 0.6261, 0.5993, 0.5723, 0.5560,
    0.5401, 0.526, 0.5106, 0.4805,  0.4504, 0.42, 0.3907, 0.385, 0.3799, 0.3750,    0.3695, 0.3340,  0.3005, 0.2692, 0.2382, 0.2387, 0.2389, 0.2501, 0.2610, 0.2635, 0.2662, 0.2601,  0.2541,
    0.2450, 0.2426, 0.2430, 0.2434, 0.2490, 0.2523, 0.2612, 0.2692, 0.2694, 0.2996, 0.3145, 0.3329, 0.3413, 0.3498, 0.3467, 0.3442, 0.3355, 0.3266, 0.3131, 0.2996, 0.2911, 0.2831, 0.2950, 0.3070, 0.3430, 0.3799,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_RedG21va_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1207, 0.141, 0.1585, 0.1810, 0.2073, 0.2529,  0.2959,  0.3210, 0.3476, 0.3350, 0.3232, 0.2845,  0.2564, 0.2140,   0.1823, 0.1523, 0.1266, 0.1001, 0.0792, 0.061,  0.0439, 0.0349,  0.0295, 0.0260,  0.0222,
    0.0180, 0.0135, 0.0111,  0.0087, 0.0090,  0.0094, 0.0101,  0.0109, 0.0093,  0.0086, 0.0090,  0.0091, 0.0061,  0.0321, 0.0086,  0.1368, 0.2312,  0.3256, 0.4112,  0.4958, 0.5444,  0.5884, 0.6002,  0.6264,  0.6323, 0.6473,
    0.6546, 0.6659, 0.6775,  0.6881, 0.6982,  0.7081, 0.7150,  0.7201, 0.7217,  0.7232, 0.7222,  0.7215,  0.7187, 0.7157, 0.7144,  0.7131, 0.7196,  0.7269, 0.7303, 0.7599,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_RedI9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0258, 0.023,  0.0220, 0.0205,  0.0189, 0.0183,  0.0174, 0.0168,  0.0162, 0.0152,  0.0148, 0.0145,  0.0139, 0.0136,  0.0133, 0.0130,  0.0127, 0.0130,  0.0133, 0.0151,  0.0168, 0.0218,  0.0268, 0.0317,  0.0367, 0.0330,
    0.0313, 0.0270,  0.0227, 0.0240,  0.0255, 0.0280,  0.0302, 0.0280,  0.0225, 0.0215,  0.0209, 0.0424,  0.0639, 0.1401,  0.2131, 0.3250,  0.4369, 0.5210,  0.6265, 0.6795,  0.7336, 0.7551,  0.7784, 0.7890,  0.7994, 0.8070,
    0.8146, 0.8210,  0.8277, 0.8321,  0.8362, 0.8398,  0.8439, 0.8470,  0.8504, 0.8530,  0.8572,  0.8612, 0.8653, 0.8689,  0.8715, 0.8730,  0.8747, 0.8766,  0.8788,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_YelN10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0531, 0.0520,  0.0504, 0.0510,  0.0518, 0.0608,  0.0628, 0.0669,  0.0699, 0.0705,  0.0716, 0.0720,  0.0735, 0.0755,  0.0775, 0.0800,  0.0825, 0.0896,  0.0969, 0.1260, 0.1563, 0.2312,  0.3096, 0.4132,  0.5177, 0.5905,  0.6637,
    0.7251, 0.7350,  0.7458, 0.7480,  0.7507, 0.7460,  0.7414, 0.7356,  0.7301, 0.7320,  0.7347, 0.7390,  0.7438, 0.7472,  0.7500, 0.7508,  0.7515, 0.7528,  0.7538, 0.7550,  0.7563, 0.7581,  0.7607, 0.7642,  0.7686, 0.7710,
    0.7791, 0.7840,  0.7872, 0.7902,  0.7935, 0.7955,  0.7979, 0.7995,  0.8021, 0.8035,  0.8058, 0.8072,  0.8090, 0.8110,  0.8143, 0.8198,  0.8259,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_GreN7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0112, 0.0102,  0.0094, 0.0096,  0.0099, 0.0100,  0.0100, 0.0100,  0.0100, 0.0099,  0.0099, 0.0099,  0.0099, 0.0099,  0.0099, 0.0100,  0.0100, 0.0103,  0.0107, 0.0129,  0.0151, 0.0312,  0.0462, 0.1015,  0.1571, 0.2270,  0.2977,
    0.3558, 0.3441,  0.3321, 0.3020,  0.2710, 0.2312,  0.1954, 0.1602,  0.1251, 0.1003,  0.0794, 0.0672,  0.0563, 0.0513,  0.0452,  0.0418, 0.0378, 0.0356,  0.0337, 0.0336,  0.0335, 0.0345,  0.0358, 0.0383,  0.0405, 0.0445,  0.0497,
    0.0612, 0.0647,  0.0670, 0.0660,  0.0644, 0.0620,  0.0574, 0.0525,  0.0483, 0.0460,  0.0436, 0.0484,  0.0532, 0.0690,  0.0870,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_GreA10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0958, 0.1010,  0.1090, 0.1172,  0.1352, 0.1954,  0.1957, 0.2178,  0.2402, 0.2477,  0.2553, 0.2594,  0.2622, 0.2667,  0.2707, 0.2760,  0.2805, 0.2913,  0.3023, 0.3376,  0.3715, 0.4345,  0.5030, 0.5702,  0.6376, 0.6724,  0.7072,
    0.7216, 0.7160,  0.7110, 0.6990,  0.6865, 0.6667,  0.6446, 0.6174,  0.5921, 0.5727,  0.5511, 0.5386,  0.5238, 0.5134,  0.5070, 0.4980,  0.4918, 0.4867,  0.4830, 0.4834,  0.4838, 0.4889,  0.4906, 0.4976,  0.5046, 0.5162,  0.5279,
    0.5519, 0.5589,  0.5649, 0.5645,  0.5639, 0.5576,  0.5552, 0.5480,  0.5407, 0.5377,  0.5326, 0.5387,  0.5498, 0.5732,  0.5966,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_GreQ7_spect[97] = { //468 Q7
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0149, 0.0139, 0.0133, 0.0128, 0.0123, 0.01185, 0.0114, 0.011, 0.0106, 0.01045,    0.0103, 0.0104, 0.0105, 0.01065,    0.0108, 0.0109, 0.0110, 0.0117, 0.0124, 0.0205, 0.0283, 0.071,  0.1122, 0.2051,
    0.3017, 0.385, 0.4690, 0.4880,   0.5069, 0.4811, 0.4561, 0.411,  0.3687, 0.3185, 0.2673, 0.2190, 0.1703, 0.1371, 0.1042, 0.086,  0.0695, 0.598,  0.0527, 0.0475, 0.04210, 0.039, 0.03600, 0.0357,    0.0355, 0.037,
    0.0381, 0.0405, 0.0438, 0.0499, 0.0555, 0.0635, 0.0708, 0.074,  0.0789, 0.077,  0.0751, 0.071,  0.0658, 0.059,  0.0530, 0.049,  0.0458, 0.052,  0.0570, 0.077,  0.0982,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_GreK7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0114, 0.0111,  0.0109, 0.0107,  0.0105, 0.0106,  0.0108, 0.0107,  0.0106, 0.0105,  0.0104, 0.0103,  0.0103, 0.0106,  0.0109, 0.0112,  0.0118, 0.0135,  0.0153, 0.0244,  0.0334, 0.0666,  0.0984, 0.1534,  0.2082, 0.2412,  0.2835,
    0.2959, 0.2843,  0.2735, 0.2516,  0.2305, 0.2012,  0.1728, 0.1435,  0.1156, 0.0964,  0.0772, 0.0671,  0.0570, 0.0518,  0.0468, 0.0436,  0.0397, 0.0380,  0.0354, 0.0354,  0.0355, 0.0367,  0.0380, 0.0402,  0.0426, 0.0481,  0.0523,
    0.0643, 0.0678,  0.0704, 0.0693,  0.0676, 0.0639,  0.0609, 0.0567,  0.0514, 0.0487,  0.0468, 0.0518,  0.0567, 0.0730,  0.0902,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_PurE24_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0677, 0.901,  0.1043, 0.1298,  0.1534, 0.1913,  0.2297, 0.2553,  0.2756, 0.2789,  0.2620, 0.2380,  0.2135, 0.1837,  0.1536, 0.1312,  0.1068, 0.0867,  0.0663, 0.0517,  0.0368, 0.0309,  0.0247, 0.0214,  0.0186, 0.0151,  0.0116,
    0.0077, 0.0079,  0.0079, 0.0083,  0.0086, 0.0077,  0.0071, 0.0071,  0.0072, 0.0107,  0.0147, 0.0298,  0.0440, 0.0661,  0.0880, 0.1010,  0.1152, 0.1193,  0.1236, 0.1260,  0.1287, 0.1326,  0.1366, 0.1428,  0.1489, 0.1596,  0.1697,
    0.1936, 0.1996,  0.2057, 0.2036,  0.2015, 0.1954, 0.1890, 0.1798,  0.1706, 0.1651,  0.1603, 0.1692,  0.1788, 0.2075,  0.2363,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_BluH10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.01590, 0.028, 0.03970, 0.0697, 0.09970, 0.1526, 0.20550, 0.253, 0.30110, 0.3412, 0.38180, 0.423, 0.46610, 0.4683, 0.51030, 0.5005, 0.49950, 0.4785, 0.45810, 0.429, 0.39950, 0.374, 0.35010, 0.3135, 0.29630,
    0.2587, 0.22070, 0.182, 0.14450, 0.1125, 0.09060, 0.072, 0.04810, 0.033, 0.01740, 0.0113, 0.00520, 0.004, 0.00290, 0.0028, 0.00270, 0.0027, 0.00270, 0.0027, 0.00280, 0.0027, 0.00270, 0.0028, 0.00280,
    0.0029, 0.00300, 0.0029, 0.00290, 0.0029, 0.0029, 0.00290, 0.0029, 0.00290, 0.0029, 0.00290, 0.0029, 0.00290, 0.0029, 0.00290, 0.0029, 0.0031, 0.00320, 0.0035, 0.00380, 0.0047, 0.00560,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_BluD6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1127, 0.143, 0.1773, 0.223,   0.2813, 0.3987, 0.4782, 0.5665, 0.6470, 0.6870, 0.7270, 0.7403, 0.7593, 0.7592, 0.7591, 0.7480, 0.7402, 0.7234, 0.7054, 0.6876, 0.6617, 0.6512, 0.6302, 0.6124, 0.5962, 0.5660,
    0.5352, 0.5009, 0.4655, 0.4356, 0.4191, 0.3923, 0.3619, 0.3145, 0.2653, 0.2245, 0.1744, 0.1499, 0.1255, 0.1124, 0.1014, 0.0972, 0.0855, 0.0786, 0.0715, 0.0659, 0.0626, 0.0625, 0.0624, 0.0645, 0.0670, 0.0714,
    0.0769, 0.0865, 0.0964, 0.1086, 0.1200, 0.123,  0.1327, 0.1309, 0.1281, 0.1214, 0.1146, 0.1023, 0.0950, 0.0901, 0.0839, 0.0918, 0.1009, 0.1260, 0.1597,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_BluF4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0180, 0.0270, 0.0324, 0.0453, 0.0611, 0.0845, 0.1066, 0.1234, 0.1446, 0.1567, 0.1718, 0.1867, 0.1954, 0.2024, 0.2083, 0.2090, 0.2096, 0.2060, 0.2036, 0.1990, 0.1947, 0.1920, 0.1901, 0.1856, 0.1794, 0.1667, 0.1516, 0.1321,
    0.1167, 0.1032, 0.0876, 0.0730, 0.0584, 0.0445, 0.0296, 0.0212, 0.0125, 0.0099, 0.0069, 0.0060, 0.0053, 0.0050, 0.0049, 0.0047, 0.0046, 0.0045, 0.0044, 0.0043, 0.0043, 0.0043, 0.0043, 0.0046, 0.0049, 0.0050, 0.0052, 0.0057,
    0.0063, 0.0066, 0.0069, 0.0067, 0.0066, 0.0063, 0.0059, 0.0056, 0.0053, 0.0054, 0.0055, 0.0062, 0.0069, 0.0099, 0.0122,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::JDC468_GreI8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0230, 0.0232, 0.0234, 0.0254, 0.0263, 0.0298, 0.0329, 0.0367, 0.0377, 0.0388, 0.0399, 0.0410, 0.0421, 0.0440, 0.0460, 0.0481, 0.0496, 0.0523, 0.0559, 0.0645, 0.0727, 0.0878, 0.1020, 0.1156, 0.1288, 0.1334, 0.1394, 0.1398,
    0.1402, 0.1407, 0.1413, 0.1409, 0.1396, 0.1334, 0.1276, 0.1200, 0.1129, 0.1095, 0.1064, 0.1053, 0.1043, 0.1031, 0.1021, 0.1001, 0.0980, 0.0970, 0.0952, 0.0963, 0.0967, 0.0990, 0.1009, 0.1042, 0.1078, 0.1130, 0.1188, 0.1251,
    0.1307, 0.1335, 0.1374, 0.1376, 0.1378, 0.1362, 0.1345, 0.1312, 0.1278, 0.1257, 0.1240, 0.1290, 0.1345, 0.1476, 0.1615,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_OraO18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0826, 0.0830, 0.0832, 0.0861, 0.0892, 0.0993, 0.1108, 0.1180, 0.1248, 0.1253, 0.1263, 0.1261, 0.1259, 0.1267, 0.1289, 0.1304, 0.1319, 0.1370, 0.1419, 0.1631, 0.1851, 0.2311, 0.2743, 0.3131, 0.3536, 0.3551, 0.3585, 0.3488, 0.3322,
    0.3470, 0.3575, 0.3680, 0.3498, 0.3316, 0.3224, 0.3129, 0.3578, 0.4013, 0.4734, 0.5454, 0.5978, 0.6502, 0.6745, 0.6982, 0.7080, 0.7182, 0.7273, 0.7269, 0.7308, 0.7342, 0.7393, 0.7436, 0.7498, 0.7550, 0.7597, 0.7640, 0.7680, 0.7713,
    0.7766, 0.7786, 0.7816, 0.7841, 0.7863, 0.7889, 0.7902, 0.7931, 0.7957, 0.7997, 0.8068,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_OraD17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0462, 0.0442,  0.0422, 0.0401,  0.0383, 0.0390,  0.0396, 0.0396,  0.0395, 0.0388,  0.0380, 0.0378,  0.0376, 0.0381,  0.0384, 0.0391,  0.0399, 0.0421,  0.0451, 0.0561,  0.0676,  0.0934, 0.1189, 0.1432,  0.1671, 0.1650,  0.1632, 0.1512,  0.1402, 0.1456,
    0.1521, 0.1613,  0.1696, 0.1552,  0.1409,  0.1342, 0.1283, 0.1689,  0.2084, 0.2845,  0.3575, 0.4183,  0.4797, 0.5090,  0.5389, 0.5498,  0.5617, 0.5667,  0.5728, 0.5788,  0.5822, 0.5889,  0.5938, 0.6011,  0.6081, 0.6145,  0.6212, 0.6267,  0.6304, 0.6331,
    0.6352, 0.6361,  0.6373, 0.6372,  0.6370, 0.6376,  0.6384, 0.6413,  0.6483, 0.6523,  0.6668,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin35_15_17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0211, 0.022,   0.0225, 0.0234,  0.0244, 0.0294,  0.0349, 0.038,   0.0411, 0.0425,  0.0441, 0.0455,  0.0472, 0.0473,  0.0475, 0.0463,  0.0452, 0.0435,  0.0417, 0.0397,  0.0377,
    0.0375, 0.0372, 0.0412,  0.0452, 0.052,   0.0603, 0.0655,  0.0707, 0.0722,  0.0736, 0.0738,  0.0741, 0.0737,  0.0731, 0.0721,  0.0711, 0.0707,  0.0704, 0.071,   0.0717,
    0.0782, 0.0846, 0.099,   0.1159, 0.1298,  0.1432, 0.1497,  0.1581, 0.1603,  0.1644, 0.1659,  0.1673, 0.1679,  0.1690, 0.1696,  0.1702, 0.1704,  0.1705, 0.1706,  0.1707, 0.1707,  0.1707, 0.1707,  0.1706, 0.1706,  0.1707, 0.1707,  0.1706, 0.1712,  0.1719,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin57_22_18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0647, 0.0677,  0.0709, 0.0754,  0.0797, 0.099,   0.1181, 0.1296,  0.1409, 0.1469,  0.1529, 0.1594,  0.1657, 0.1672,  0.1683, 0.1648,  0.1615, 0.1561,  0.1506, 0.144,   0.1375, 0.136,   0.1339,
    0.1437, 0.1531, 0.172,   0.1911, 0.2032,  0.2153, 0.2171,  0.2190, 0.2176,  0.2164, 0.213,   0.2095, 0.2048,  0.2005, 0.1985,  0.1965, 0.198,   0.1997, 0.2196,  0.2396, 0.288,   0.3362, 0.378,
    0.4209, 0.444,   0.4671, 0.477,   0.4865, 0.491,   0.4955, 0.498,   0.5007, 0.5027,  0.5048, 0.5055,  0.5061, 0.5063,  0.5066, 0.5065,  0.5063, 0.506,   0.5057, 0.5056,  0.5055, 0.5056,  0.5057, 0.5078,  0.5099,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin40_17_17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0296, 0.0306,  0.0317, 0.0332,  0.0346, 0.042,   0.0498, 0.0543,  0.0588, 0.061,   0.0632, 0.0624,  0.0678, 0.068,   0.0682, 0.0663,  0.0649, 0.0625,  0.0598, 0.057,   0.0540, 0.0535,  0.0529, 0.057,
    0.0631, 0.072,   0.0827, 0.089,   0.0959, 0.0975,  0.0994, 0.0996,  0.0997, 0.0988,  0.0980, 0.0966,  0.0951, 0.0945,  0.0939, 0.0948,  0.0957, 0.105,   0.1143, 0.136,   0.1589, 0.178,   0.1980,
    0.2095, 0.2194, 0.224,   0.2283, 0.2302,  0.2325, 0.2337,  0.2348, 0.2357,  0.2366, 0.2369,  0.2371, 0.2372,  0.2373, 0.2373,  0.2373, 0.2373,  0.2372, 0.2372,  0.2372, 0.2372,  0.2372, 0.238,   0.2389,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin91_4_14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1430, 0.16,    0.1778, 0.202,   0.2303, 0.301,   0.3813, 0.4245,  0.4692, 0.499,   0.5287, 0.5635,  0.5977, 0.6175,  0.6372, 0.6394,  0.6418, 0.638,   0.6341, 0.6228,  0.6117, 0.6121,  0.6125,
    0.646,  0.6807, 0.742,   0.8032, 0.8355,  0.8687, 0.8595,  0.8510, 0.828,   0.8059, 0.778,   0.7490, 0.721,   0.6914, 0.676,   0.6608, 0.657,   0.6530, 0.6565,  0.7001, 0.7609,  0.8219,
    0.876,  0.9297, 0.9598,  0.9901, 1.003,   1.0156, 1.021,   1.0289, 1.0346,  1.0396, 1.045,   1.0513, 1.0538,  1.0561, 1.0559,  1.0557, 1.054,   1.0523, 1.049,   1.0466, 1.045,   1.0436, 1.0445,  1.0455, 1.053,   1.0605,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin87_8_8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1433, 0.161,   0.1780, 0.204,   0.2305, 0.306,   0.3828, 0.428,   0.4722, 0.502,   0.5317, 0.5645,  0.5997, 0.618,   0.6366, 0.6368,  0.6370, 0.631,   0.6251, 0.6120,  0.5994, 0.596,
    0.5931, 0.618,   0.6420, 0.705,   0.7347, 0.757,   0.7785, 0.765,   0.7532, 0.71,    0.7062, 0.677,   0.6491, 0.62,    0.5922, 0.577,   0.5619, 0.5586,  0.5556, 0.579,   0.6121,
    0.684,  0.7563, 0.82,    0.8837, 0.918,   0.9545, 0.969,   0.9843, 0.992,   0.9991, 1.005,   1.0104, 1.016,   1.0223, 1.0248,  1.0273, 1.0272,  1.0271, 1.025,   1.0238, 1.02,    1.0182, 1.017,   1.0151, 1.016,   1.0171, 1.024,   1.0321,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin89_8_21_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1394, 0.152,   0.1659, 0.1855,  0.2052, 0.266,   0.3277, 0.363,   0.3988, 0.422,   0.4450, 0.472,   0.4984, 0.512,   0.5270, 0.5274,  0.5278, 0.522,   0.5177, 0.5065,  0.4960, 0.4975,
    0.4995, 0.535,   0.5721, 0.637,   0.7016, 0.7395,  0.7786, 0.7777,  0.7767, 0.763,   0.7485, 0.728,   0.7081, 0.687,   0.6649, 0.653,   0.6425, 0.641,   0.6391, 0.665,   0.6925, 0.76,
    0.8266, 0.885,   0.9447, 0.975,   1.0106, 1.025,   1.0383, 1.045,   1.0525, 1.058,   1.0628, 1.068,   1.0730, 1.075,   1.0768, 1.0768,  1.0769, 1.0755,  1.0744, 1.0724,  1.0704, 1.069,   1.0685, 1.0691,  1.0697, 1.075,   1.0823,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin75_8_4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1030, 0.116,   0.1294, 0.1495,  0.1696, 0.227,   0.2847, 0.319,   0.3524, 0.375,   0.3977, 0.423,   0.4492, 0.462,   0.4770, 0.4768,  0.4767, 0.471,   0.4675, 0.458,   0.4480, 0.444,   0.4408,
    0.455,  0.4688, 0.496,   0.5243, 0.535,   0.5465, 0.534,   0.5228, 0.503,   0.4851, 0.463,   0.4408, 0.419,   0.3974, 0.385,   0.3741, 0.371,   0.3692, 0.391,   0.4110, 0.464,   0.5180,
    0.565,  0.6126, 0.638,   0.6651, 0.676,   0.6871, 0.692,   0.6979, 0.702,   0.7062, 0.71,    0.7151, 0.717,   0.7189, 0.7188,  0.7187, 0.717,   0.7162, 0.714,   0.7119, 0.7105,  0.7095, 0.7097,  0.7110, 0.716,   0.7223,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin75_10_33_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0873, 0.091,   0.0967, 0.103,   0.1097, 0.135,   0.1617, 0.177,   0.1913, 0.198,   0.2086, 0.218,   0.2289, 0.234,   0.2383, 0.2375,  0.2370, 0.2335,  0.2299, 0.223,   0.2180, 0.222,
    0.2259, 0.256,   0.2860, 0.338,   0.3905, 0.426,   0.4613, 0.47,    0.4786, 0.478,   0.4772, 0.471,   0.4668, 0.459,   0.4522, 0.449,   0.4454, 0.446,   0.4467, 0.464,   0.4834, 0.527,
    0.5727, 0.609,   0.6511, 0.673,   0.6946, 0.703,   0.7130, 0.718,   0.7224, 0.725,   0.7285, 0.731,   0.7337, 0.7345,  0.7351, 0.7353,  0.7355, 0.735,   0.7348, 0.7342,  0.7337, 0.7336,  0.7335, 0.7335,  0.7336, 0.737,   0.7395,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin65_33_11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1067, 0.113,   0.1182, 0.126,   0.1346, 0.165,   0.2033, 0.224,   0.2448, 0.259,   0.2666, 0.277,   0.2891, 0.291,   0.2927, 0.285,   0.2783, 0.268,   0.2569, 0.244,   0.2323, 0.225,   0.2195,
    0.225,  0.2323, 0.248,   0.2655, 0.275,   0.2832, 0.281,   0.2797, 0.275,   0.2708, 0.264,   0.2567, 0.248,   0.2403, 0.236,   0.2326, 0.235,   0.2386, 0.274,   0.3116, 0.40,    0.4885, 0.56,
    0.6435, 0.688,   0.7279, 0.745,   0.7633, 0.772,   0.7791, 0.783,   0.7883, 0.792,   0.7955, 0.7965,  0.7978, 0.7982,  0.7989, 0.7985,  0.7983, 0.7975,  0.7971, 0.7968,  0.7966, 0.7968,  0.7970, 0.801,   0.8043,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin65_7_24_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0619, 0.066,   0.0710, 0.077,   0.0840, 0.106,   0.1288, 0.142,   0.1546, 0.163,   0.1706, 0.179,   0.1893, 0.194,   0.1989, 0.1988,  0.1987, 0.196,   0.1941, 0.189,   0.1853, 0.188,   0.1894,
    0.209,  0.2282, 0.262,   0.2962, 0.318,   0.3402, 0.343,   0.3469, 0.343,   0.3407, 0.334,   0.3285, 0.321,   0.3140, 0.31,    0.3069, 0.308,   0.3066, 0.319,   0.3311, 0.362,   0.3918, 0.418,
    0.4451, 0.459,   0.4747, 0.481,   0.4873, 0.491,   0.4937, 0.496,   0.4981, 0.501,   0.5022, 0.503,   0.5035, 0.5035,  0.5036, 0.5032,  0.5029, 0.5022,  0.5016, 0.5013,  0.5011, 0.5013,  0.5014, 0.504,   0.5063,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin57_19_6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0662, 0.071,   0.0773, 0.085,   0.0939, 0.115,   0.1491, 0.165,   0.1821, 0.192,   0.2019, 0.214,   0.2236, 0.228,   0.2321, 0.2298,  0.2266, 0.221,   0.2161, 0.208,   0.2019, 0.199,
    0.1951, 0.201,   0.2066, 0.219,   0.2325, 0.239,   0.2443, 0.241,   0.2366, 0.23,    0.2235, 0.215,   0.2068, 0.199,   0.1895, 0.185,   0.1806, 0.1811,  0.1816, 0.20,    0.2197, 0.267,   0.3135,
    0.355,  0.3960, 0.418,   0.4411, 0.449,   0.4600, 0.4643,  0.4687, 0.471,   0.4743, 0.477,   0.4792, 0.48,    0.4811, 0.4813,  0.4815, 0.481,   0.4806, 0.4798,  0.4790, 0.4786,  0.4782, 0.4786,  0.4788, 0.481,   0.4844,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin57_4_19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0430, 0.047,   0.0505, 0.056,   0.0614, 0.077,   0.0963, 0.1063,  0.1164, 0.123,   0.1294, 0.137,   0.1448, 0.149,   0.1533, 0.154,   0.1544, 0.153,   0.1521, 0.149,   0.1463, 0.148,
    0.1496, 0.164,   0.1780, 0.202,   0.2273, 0.242,   0.2585, 0.26,    0.2616, 0.258,   0.2550, 0.2495,  0.2442, 0.238,   0.2320, 0.229,   0.2258, 0.2253,  0.2247, 0.232,   0.2394,
    0.258,  0.2763, 0.292,   0.3087, 0.318,   0.3269, 0.331,   0.3346, 0.337,   0.3387, 0.341,   0.3417, 0.343,   0.3447, 0.345,   0.3457, 0.3457,  0.3457, 0.3455,  0.3451, 0.3445,  0.3439, 0.3437,  0.3435, 0.3437,  0.3438, 0.346,   0.3475,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin57_10_28_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0463, 0.048,   0.0505, 0.053,   0.0563, 0.069,   0.0816, 0.088,   0.0961, 0.102,   0.1041, 0.1085,  0.1135, 0.1155,  0.1174, 0.1168,  0.1161, 0.114,   0.1118, 0.1085,  0.1054, 0.1074,  0.1094, 0.124,
    0.1406, 0.168,   0.1951, 0.223,   0.2325, 0.238,   0.2426, 0.243,   0.2432, 0.242,   0.2391, 0.239,   0.2326, 0.231,   0.2297, 0.234,   0.2309, 0.242,   0.2516, 0.277,   0.3017, 0.324,
    0.3456, 0.358,   0.3700, 0.375,   0.3802, 0.3827,  0.3854, 0.387,   0.3887, 0.39,    0.3913, 0.3916,  0.3920, 0.3921,  0.3923, 0.3921,  0.3920, 0.3918,  0.3916, 0.3916,  0.3915, 0.3915,  0.3915, 0.393,   0.3945,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin40_7_19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0215, 0.023,   0.0240, 0.026,   0.0275, 0.033,   0.0409, 0.044,   0.0487, 0.051,   0.0532, 0.056,   0.0585, 0.0595,  0.0608, 0.0605,  0.0602, 0.059,   0.0581, 0.057,   0.0549, 0.0555,  0.0562,
    0.061,  0.0692, 0.08,    0.0922, 0.099,   0.1075, 0.109,   0.1107, 0.11,    0.1098, 0.1082,  0.1069, 0.1045,  0.1031, 0.102,   0.1013, 0.1015,  0.1016, 0.106,   0.1112, 0.123,   0.1348, 0.145,
    0.1554, 0.161,   0.1668, 0.169,   0.1716, 0.1728,  0.1741, 0.175,   0.1756, 0.1763,  0.1769, 0.1771,  0.1773, 0.1773,  0.1774, 0.1773,  0.1772, 0.177,   0.1769, 0.1769,  0.1769, 0.1769,  0.1769, 0.1777,  0.1784,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin40_17_6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0314, 0.033,   0.0359, 0.039,   0.0427, 0.054,   0.0668, 0.074,   0.0812, 0.085,   0.0895, 0.094,   0.0985, 0.10,    0.1015, 0.0991,  0.0984, 0.096,   0.0930, 0.089,   0.0861, 0.085,   0.0828,
    0.085,  0.0878, 0.094,   0.0995, 0.103,   0.1052, 0.1035,  0.1026, 0.10,    0.0976, 0.094,   0.0911, 0.088,   0.0840, 0.083,   0.0805, 0.081,   0.0814, 0.09,    0.1006, 0.124,
    0.1474, 0.1685,  0.1885, 0.1995,  0.2110, 0.216,   0.2204, 0.223,   0.2247, 0.226,   0.2273, 0.2284,  0.2296, 0.230,   0.2304, 0.2305,  0.2306, 0.2305,  0.2303, 0.23,    0.2297, 0.2296,  0.2294, 0.2295,  0.2296, 0.231,   0.2321,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin40_4_11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0209, 0.023,   0.0250, 0.028,   0.0310, 0.039,   0.0497, 0.056,   0.0605, 0.064,   0.0675, 0.072,   0.0758, 0.078,   0.0802, 0.0803,  0.0804, 0.0797,  0.0790, 0.078,   0.0758, 0.076,   0.0764, 0.082,
    0.0875, 0.098,   0.1072, 0.113,   0.1189, 0.1187,  0.1185, 0.116,   0.1141, 0.111,   0.1078, 0.104,   0.1012, 0.099,   0.0977, 0.098,   0.0971, 0.101,   0.1049, 0.115,   0.1245, 0.133,
    0.1417, 0.147,   0.1513, 0.153,   0.1554, 0.1564,  0.1575, 0.158,   0.1590, 0.1598,  0.1606, 0.1608,  0.1611, 0.1611,  0.1611, 0.1609,  0.1608, 0.1604,  0.1601, 0.160,   0.1598, 0.1599,  0.1600, 0.1609,  0.1619,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin33_6_15_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0143, 0.015,   0.0162, 0.0175,  0.0189, 0.023,   0.0286, 0.031,   0.0342, 0.036,   0.0376, 0.039,   0.0415, 0.0425,  0.0434, 0.0432,  0.0431, 0.0425,  0.0418, 0.041,   0.0396, 0.04,
    0.0404, 0.0444,  0.0488, 0.056,   0.0638, 0.0689,  0.0735, 0.074,   0.0752, 0.0745,  0.0741, 0.073,   0.0717, 0.070,   0.0688, 0.0681,  0.0673, 0.0673,  0.0674, 0.0710,  0.0737, 0.0810,
    0.0889, 0.0960,  0.1023, 0.1065,  0.1098, 0.1120,  0.1129, 0.1135,  0.1145, 0.1150,  0.1155, 0.1160,  0.1164, 0.1165,  0.1167, 0.1167,  0.1168, 0.1167,  0.1166, 0.1165,  0.1164, 0.1164,  0.1163, 0.1163,  0.1164, 0.1170,  0.1174,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin33_15_5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0212, 0.023,   0.0243, 0.0265,  0.0289, 0.037,   0.0451, 0.051,   0.0549, 0.058,   0.0605, 0.063,   0.0666, 0.0675,  0.0686, 0.0672,  0.0664, 0.065,   0.0627, 0.0061,  0.0580, 0.0565,  0.0557,
    0.057,  0.0590, 0.063,   0.0666, 0.069,   0.0703, 0.0694,  0.0684, 0.0666,  0.0651, 0.063,   0.0607, 0.0585,  0.0559, 0.0545,  0.0535, 0.0540,  0.0542, 0.0610,  0.0672, 0.0832,
    0.0992, 0.1132,  0.1272, 0.1345,  0.1425, 0.1455,  0.1489, 0.1505,  0.1518, 0.1527,  0.1536, 0.1545,  0.1552, 0.1555,  0.1557, 0.1558,  0.1559, 0.1558,  0.1557, 0.1155,  0.1552, 0.1551,  0.1550, 0.1551,  0.1552, 0.1560,  0.1569,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin33_10_15_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0166, 0.0175,  0.0183, 0.0194,  0.0207, 0.0260,  0.0306, 0.033,   0.0364, 0.0380,  0.0396, 0.0415,  0.0431, 0.0437,  0.0443, 0.0438,  0.0432, 0.0420,  0.0409, 0.0395,  0.0380, 0.0380,
    0.0381, 0.0415,  0.0456, 0.0525,  0.0595, 0.0645,  0.0686, 0.0695,  0.0705, 0.0702,  0.0700, 0.0690,  0.0681, 0.0667,  0.0655, 0.065,   0.0644, 0.0648,  0.0650, 0.0695,  0.0739, 0.0852,
    0.0955, 0.1040,  0.1145, 0.1196,  0.1249, 0.1271,  0.1293, 0.1305,  0.1314, 0.1322,  0.1327, 0.1332,  0.1337, 0.1338,  0.1340, 0.1340,  0.1341, 0.1340,  0.1340, 0.1339,  0.1338, 0.1338,  0.1338, 0.1338,  0.1338, 0.1345,  0.1349,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin24_5_6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0086, 0.0095,  0.0102, 0.0112,  0.0127, 0.0167,  0.0203, 0.0225,  0.0248, 0.0265,  0.0277, 0.0295,  0.0309, 0.0315,  0.0325, 0.0324,  0.0323, 0.0319,  0.0315, 0.0307,  0.0299, 0.0298,
    0.0297, 0.0315,  0.0330, 0.0365,  0.0392, 0.0412,  0.0427, 0.0424,  0.0421, 0.0410,  0.0402, 0.0390,  0.0377, 0.0365,  0.0351, 0.0345,  0.0337, 0.0337,  0.0336, 0.0356,  0.0374, 0.0415,
    0.0470, 0.0512,  0.0554, 0.0575,  0.0601, 0.0610,  0.0620, 0.0625,  0.0630, 0.0634,  0.0637, 0.0640,  0.0643, 0.0645,  0.0646, 0.0646,  0.0646, 0.0645,  0.0644, 0.0643,  0.0642, 0.0642,  0.0641, 0.0641,  0.0641, 0.0645,  0.0649,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin26_18_18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0135, 0.0137,  0.0138, 0.0139,  0.0140, 0.0163,  0.0187, 0.0202,  0.0215, 0.0220,  0.0224, 0.0228,  0.0231, 0.0227,  0.0222, 0.0212,  0.0202, 0.0189,  0.0174, 0.0161,  0.0146, 0.0143,
    0.0140, 0.0163,  0.0184, 0.0224,  0.0268, 0.0291,  0.0331, 0.0348,  0.0358, 0.0366,  0.0374, 0.0378,  0.0380, 0.0380,  0.0379, 0.0380,  0.0381, 0.0388,  0.0394, 0.0440,  0.0490,
    0.0605, 0.0720, 0.0821,  0.0921, 0.0976,  0.1030, 0.1056,  0.1076, 0.1087,  0.1097, 0.1103,  0.1108, 0.1111,  0.1115, 0.1115,  0.1116, 0.1117,  0.1118, 0.1118,  0.1119, 0.1119,  0.1120, 0.1120,  0.1121, 0.1121,  0.1120, 0.1123,  0.1126,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin24_7_5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0093, 0.0105,  0.0111, 0.0125,  0.0137, 0.0180,  0.0221, 0.0245,  0.0270, 0.0285,  0.0301, 0.0316,  0.0336, 0.0345,  0.0353, 0.0350,  0.0349, 0.0343,  0.0338, 0.0329,  0.0320, 0.0317,  0.0315,
    0.0328, 0.0342, 0.0368,  0.0397, 0.0412,  0.0424, 0.0420,  0.0415, 0.0404,  0.0393, 0.0379,  0.0366, 0.0352,  0.0337, 0.0330,  0.0323, 0.0322,  0.0322, 0.0344,  0.0367, 0.0422,  0.0479, 0.0529,
    0.0578, 0.0606,  0.0633, 0.0644,  0.0656, 0.0662,  0.0667, 0.0670,  0.0674, 0.0678,  0.0681, 0.0683,  0.0684, 0.0684,  0.0684, 0.0683,  0.0683, 0.0682,  0.0680, 0.0679,  0.0678, 0.0678,  0.0679, 0.0683,  0.0688,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin20_4_2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0064, 0.0074,  0.0080, 0.00903, 0.0104, 0.0139,  0.0174, 0.0189,  0.0216, 0.0222,  0.0243, 0.0258,  0.0274, 0.0282,  0.0291, 0.0290,  0.0290, 0.0288,  0.0284, 0.0278,  0.0272, 0.0270,  0.0267, 0.0276,
    0.0285, 0.0302,  0.0320, 0.0327,  0.0335, 0.0328,  0.0321, 0.0311,  0.0299, 0.0280,  0.0272, 0.0259,  0.0246, 0.0239,  0.0232, 0.0230,  0.0229, 0.0243,  0.0256, 0.0291,  0.0324, 0.0354,
    0.0385, 0.0401,  0.0418, 0.0425,  0.0432, 0.0435,  0.0439, 0.0442,  0.0444, 0.0447,  0.0450, 0.0451,  0.0452, 0.0452,  0.0452, 0.0451,  0.0450, 0.0449,  0.0448, 0.0447,  0.0446, 0.0447,  0.0447, 0.0450,  0.0454,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin98_m2_10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1627, 0.1870,  0.2115, 0.2480,  0.2860, 0.3870,  0.4878, 0.5460,  0.6050, 0.6460,  0.6874, 0.7355,  0.7836, 0.8130,  0.8424, 0.8494,  0.8543, 0.8520,  0.8508, 0.8390,  0.8267, 0.8274,  0.8280,
    0.8680, 0.9076, 0.9600,  1.0497, 1.089,   1.1190, 1.10,    1.0836, 1.045,   1.0140, 0.975,   0.9305, 0.884,   0.8486, 0.826,   0.8042, 0.7980,  0.7895, 0.8093,  0.8292, 0.884,   0.9376, 0.987,
    1.0341, 1.059,   1.0892, 1.104,   1.1125, 1.1253,  1.1255, 1.131,   1.1375, 1.145,   1.1520, 1.155,   1.1585, 1.158,   1.1574, 1.1548,  1.1523, 1.148,   1.1439, 1.141,   1.1394, 1.141,   1.1423, 1.151,   1.1619,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin90_m1_20_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1228, 0.138,   0.1532, 0.175,   0.1987, 0.261,   0.3279, 0.365,   0.4022, 0.428,   0.4537, 0.4842,  0.5147, 0.5337,  0.5521, 0.557,   0.5611, 0.5602,  0.5593, 0.551,   0.5438, 0.548,
    0.5527, 0.593,   0.6334, 0.703,   0.7732, 0.8135,  0.8543, 0.851,   0.8474, 0.829,   0.8105, 0.786,   0.7613, 0.736,   0.7105, 0.697,   0.6835, 0.679,   0.6750, 0.6895,  0.7045, 0.743,
    0.7832, 0.818,   0.8530, 0.873,   0.8929, 0.899,   0.9099, 0.914,   0.9197, 0.924,   0.9282, 0.933,   0.9380, 0.9395,  0.9419, 0.9416,  0.9413, 0.9398,  0.9382, 0.9357,  0.9332, 0.9322,  0.9306, 0.9315,  0.9322, 0.939,   0.9452,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin95_0_4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1614, 0.1865,  0.2118, 0.2495,  0.2889, 0.392,   0.4969, 0.557,   0.6185, 0.6605,  0.7035, 0.749,   0.8018, 0.832,   0.8605, 0.865,   0.8696, 0.866,   0.8633, 0.849,   0.8365, 0.834,
    0.8308, 0.861,   0.8911, 0.946,   1.0030, 1.025,   1.0490, 1.025,   1.0030, 0.964,   0.9278, 0.884,   0.8407, 0.7985,  0.7565, 0.734,   0.7107, 0.6985,  0.6962, 0.718,   0.7416, 0.803,
    0.8642, 0.919,   0.9733, 1.001,   1.0349, 1.051,   1.0609, 1.068,   1.0747, 1.081,   1.0872, 1.095,   1.1021, 1.105,   1.1089, 1.1085,  1.1079, 1.1055,  1.1027, 1.098,   1.0940, 1.091,   1.0892, 1.0905,  1.0923, 1.102,   1.1123,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin81_2_14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1029, 0.116,   0.1285, 0.148,   0.1672, 0.222,   0.2774, 0.311,   0.3412, 0.362,   0.3849, 0.410,   0.4359, 0.451,   0.4659, 0.468,   0.4706, 0.4685,  0.4664, 0.4685,  0.4512, 0.4525,  0.4536,
    0.461,  0.5076, 0.551,   0.6035, 0.6295,  0.6559, 0.6495,  0.6442, 0.627,   0.6112, 0.5905,  0.5691, 0.537,   0.5266, 0.515,   0.5039, 0.501,   0.4975, 0.5125,  0.5283, 0.568,   0.6087, 0.643,
    0.6799, 0.700,   0.7200, 0.729,   0.7370, 0.7415,  0.7461, 0.750,   0.7536, 0.759,   0.7620, 0.764,   0.7655, 0.7653,  0.7651, 0.764,   0.7626, 0.761,   0.7583, 0.7572,  0.7561, 0.7567,  0.7575, 0.758,   0.7685,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin87_3_10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1295, 0.146,   0.1639, 0.190,   0.2160, 0.291,   0.3626, 0.405,   0.4480, 0.476,   0.5066, 0.541,   0.5743, 0.593,   0.6136, 0.616,   0.6186, 0.614,   0.6119, 0.601,   0.5911, 0.5905,
    0.5897, 0.623,   0.6460, 0.697,   0.7483, 0.773,   0.7992, 0.788,   0.7759, 0.752,   0.7287, 0.699,   0.6712, 0.642,   0.6141, 0.598,   0.5835, 0.579,   0.5750, 0.596,   0.6162, 0.669,
    0.7239, 0.772,   0.8193, 0.845,   0.8728, 0.888,   0.8954, 0.901,   0.9073, 0.912,   0.9171, 0.922,   0.9281, 0.931,   0.9328, 0.9325,  0.9323, 0.931,   0.9289, 0.926,   0.9232, 0.9215,  0.9201, 0.921,   0.9220, 0.929,   0.9364,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin77_12_21_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1039, 0.111,   0.1205, 0.132,   0.1448, 0.185,   0.2261, 0.249,   0.2734, 0.287,   0.3028, 0.318,   0.3364, 0.345,   0.3525, 0.351,   0.3499, 0.345,   0.3397, 0.3295,  0.3224, 0.329,   0.3234,
    0.349,  0.3729, 0.418,   0.4625, 0.490,   0.5173, 0.5185,  0.5196, 0.511,   0.5045, 0.492,   0.4807, 0.467,   0.4543, 0.447,   0.4410, 0.4409,  0.4407, 0.464,   0.4872, 0.544,   0.6020, 0.6522,
    0.7029, 0.731,   0.7588, 0.771,   0.7823, 0.787,   0.7939, 0.798,   0.8017, 0.805,   0.8090, 0.8103,  0.8115, 0.8117,  0.8118, 0.8111,  0.8104, 0.8193,  0.8081, 0.8076,  0.8070, 0.8073,  0.8077, 0.812,   0.8162,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSkin70_7_32_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0695, 0.074,   0.0777, 0.084,   0.0890, 0.104,   0.1321, 0.144,   0.1565, 0.164,   0.1713, 0.1795,  0.1889, 0.194,   0.1978, 0.198,   0.1983, 0.196,   0.1939, 0.189,   0.1853, 0.189,
    0.1933, 0.219,   0.2458, 0.291,   0.3362, 0.367,   0.3974, 0.405,   0.4120, 0.411,   0.4101, 0.406,   0.4007, 0.394,   0.3877, 0.385,   0.3816, 0.3817,  0.3819, 0.395,   0.4081, 0.440,
    0.4721, 0.498,   0.5284, 0.544,   0.5598, 0.566,   0.5730, 0.577,   0.5801, 0.5825,  0.5848, 0.587,   0.5890, 0.5895,  0.5901, 0.5903,  0.5903, 0.59,    0.5897, 0.5892,  0.5887, 0.5885,  0.5884, 0.5885,  0.5886, 0.5896,  0.5934,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSky60_0_m31_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0752, 0.094,   0.1121, 0.141,   0.1699, 0.243,   0.3150, 0.357,   0.4015, 0.432,   0.4631, 0.497,   0.5325, 0.553,   0.5730, 0.574,   0.5758, 0.572,   0.5695, 0.559,   0.5503, 0.539,
    0.5284, 0.5175,  0.5066, 0.493,   0.4800, 0.459,   0.4336, 0.401,   0.3684, 0.333,   0.3003, 0.265,   0.2313, 0.199,   0.1695, 0.167,   0.1349, 0.129,   0.1234, 0.136,   0.1489, 0.184,
    0.2212, 0.253,   0.2858, 0.303,   0.3218, 0.329,   0.3370, 0.341,   0.3440, 0.348,   0.3512, 0.355,   0.3606, 0.363,   0.3658, 0.3653,  0.3649, 0.3625,  0.3611, 0.358,   0.3544, 0.352,   0.3502, 0.3512,  0.3529, 0.359,   0.3660,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColabSky42_0_m24_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0336, 0.041,   0.0501, 0.063,   0.0761, 0.103,   0.1412, 0.151,   0.1799, 0.193,   0.2076, 0.223,   0.2387, 0.248,   0.2569, 0.2575,  0.2581, 0.256,   0.2553, 0.250,   0.2466, 0.2411,
    0.2368, 0.2318,  0.2268, 0.2205,  0.2145, 0.204,   0.1935, 0.179,   0.1641, 0.149,   0.1335, 0.118,   0.1025, 0.087,   0.0748, 0.067,   0.0593, 0.056,   0.0541, 0.059,   0.0655, 0.081,
    0.0979, 0.112,   0.1269, 0.134,   0.1430, 0.147,   0.1497, 0.151,   0.1529, 0.1545,  0.1561, 0.158,   0.1603, 0.1616,  0.1627, 0.1625,  0.1623, 0.1614,  0.1605, 0.159,   0.1575, 0.1567,  0.1557, 0.1563,  0.1569, 0.159,   0.1627,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorblue_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0316, 0.0480, 0.0658, 0.0925, 0.1199, 0.1812, 0.2424, 0.2770, 0.3145, 0.3430, 0.3702, 0.4022, 0.4346, 0.4560, 0.4778, 0.4843, 0.4902, 0.4940, 0.4960, 0.4920, 0.4889, 0.4820,
    0.4764, 0.4685, 0.4606, 0.4486, 0.4379, 0.4160, 0.3955, 0.3640, 0.3330, 0.2990, 0.2660, 0.2291, 0.1991, 0.1705, 0.1403, 0.1220, 0.1067, 0.0967, 0.0907, 0.0846, 0.0785, 0.0698,
    0.0601, 0.0525, 0.0455, 0.0423, 0.0386, 0.0370, 0.0358, 0.0354, 0.0351, 0.0368, 0.0382, 0.0413, 0.0449, 0.0474, 0.0492, 0.0484, 0.0477, 0.0460, 0.0437, 0.0402, 0.0371, 0.0349,
    0.0329, 0.0341, 0.0356, 0.0410, 0.0462,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorViolA1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1571, 0.185, 0.2150, 0.255, 0.3040, 0.332, 0.3684, 0.381, 0.3952, 0.3956, 0.3965, 0.387, 0.3782, 0.362, 0.3418, 0.322, 0.2995, 0.275, 0.2543, 0.229, 0.2043, 0.185, 0.1686,
    0.155, 0.1420, 0.131, 0.1070, 0.093, 0.0785, 0.075, 0.0725, 0.074, 0.0755, 0.072, 0.0695, 0.069, 0.0680, 0.083, 0.0914, 0.115, 0.1379, 0.162, 0.1833, 0.193, 0.2038, 0.205,
    0.2065, 0.207, 0.2079, 0.209, 0.2110, 0.214, 0.2176, 0.226, 0.2319, 0.242, 0.2518, 0.258, 0.2632, 0.263, 0.2616, 0.256, 0.2522, 0.246, 0.2380, 0.233, 0.2290, 0.235, 0.2432, 0.265, 0.2901,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorViolA4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2270, 0231, 0.2413, 0.286, 0.3287, 0.367, 0.4079, 0.427, 0.4469, 0.452, 0.4594, 0.456, 0.4535, 0.432, 0.4268, 0.403, 0.3886, 0.368, 0.3427, 0.314, 0.2866, 0.265, 0.2433, 0.222,
    0.2087, 0.183, 0.1604, 0.130, 0.1181, 0.112, 0.1069, 0.108, 0.1098, 0.103, 0.0985, 0.094, 0.0916, 0.104, 0.1130, 0.131, 0.1496, 0.152, 0.1746, 0.176, 0.1783, 0.176, 0.1742, 0.174,
    0.1738, 0.175, 0.1763, 0.181, 0.1831, 0.192, 0.1975, 0.206, 0.2169, 0.222, 0.2274, 0.226, 0.2247, 0.219, 0.2140, 0.206, 0.1990, 0.195, 0.1897, 0.196, 0.2039, 0.221, 0.2508,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorViolA6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1426, 0.203, 0.2660, 0.315, 0.3556, 0.392, 0.4259, 0.435, 0.4459, 0.437, 0.4317, 0.417, 0.3942, 0.365, 0.3425, 0.317, 0.2917, 0.266, 0.2413, 0.218, 0.1885, 0.172, 0.1524,  0.141,
    0.1267, 0.112, 0.0948, 0.083, 0.0700, 0.068, 0.0661, 0.068, 0.0708, 0.069, 0.0671, 0.068, 0.0699, 0.900, 0.1092, 0.159, 0.2099, 0.284, 0.3582, 0.441, 0.4857, 0.525, 0.5583, 0.567,
    0.5950, 0.605, 0.6146, 0.623, 0.6307, 0.638, 0.6495, 0.661, 0.6720, 0.679, 0.6825, 0.681, 0.6809, 0.675, 0.6718, 0.667, 0.6593, 0.655, 0.6517, 0.658, 0.6649, 0.681, 0.7066,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyK3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4939, 0.435, 0.3859, 0.403, 0.4198, 0.446, 0.4780, 0.505, 0.5328, 0.552, 0.5672, 0.578, 0.5880, 0.595, 0.5994, 0.602, 0.6029, 0.600, 0.5981, 0.588, 0.5808, 0.571, 0.5618, 0.551,
    0.5369, 0.503, 0.4819, 0.452, 0.4190, 0.404, 0.3921, 0.386, 0.3815, 0.364, 0.3400, 0.321, 0.2991, 0.298, 0.2977, 0.304, 0.3090, 0.309, 0.3088, 0.302, 0.2930, 0.284, 0.2753, 0.271,
    0.2660, 0.265, 0.2636, 0.266, 0.2678, 0.275, 0.2811, 0.290, 0.2995, 0.306, 0.3125, 0.314, 0.3153, 0.313, 0.3111, 0.307, 0.3006, 0.298, .2952, 0.306, 0.3116, 0.325, 0.3584,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyK9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4058, 0.441, 0.4734, 0.502, 0.5372, 0.585, 0.6051, 0.643, 0.6698, 0.685, 0.6992, 0.705, 0.7118, 0.712, 0.7135, 0.711, 0.7071, 0.702, 0.6938, 0.681, 0.6702, 0.663, 0.6511, 0.642,
    0.6282, 0.604, 0.5732, 0.542, 0.5103, 0.499, 0.4913, 0.492, 0.4926, 0.475, 0.4604, 0.452, 0.4341, 0.453, 0.4648, 0.496, 0.5111, 0.525, 0.5335, 0.531, 0.5283, 0.522, 0.5154, 0.512,
    0.5098, 0.509, 0.5093, 0.513, 0.5151, 0.523, 0.5309, 0.544, 0.5520, 0.562, 0.5642, 0.565, 0.5657, 0.562, 0.5598, 0.554, 0.5489, 0.546, 0.5430, 0.553, 0.5601, 0.576, 0.6067,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyC4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3280, 0.2950, 0.2611, 0.304, 0.3781, 0.423, 0.4646, 0.498, 0.5292, 0.555, 0.5732, 0.591, 0.6112, 0.6221, 0.6307, 0.631, 0.6310, 0.625, 0.6181, 0.607, 0.5847, 0.563, 0.5488, 0.524,
    0.5066, 0.465, 0.4358, 0.398, 0.3585, 0.336, 0.3151, 0.302, 0.2855, 0.254, 0.2309, 0.203, 0.1786, 0.166, 0.1546, 0.149, 0.1443, 0.143, 0.1359, 0.131, 0.1245, 0.123, 0.115, 0.114,
    0.1120, 0.112, 0.1127, 0.114, 0.1169, 0.122, 0.1275, 0.133, 0.1421, 0.147, 0.1504, 0.149, 0.1488, 0.145, 0.1416, 0.136, 0.1303, 0.127, 0.1241, 0.132, 0.1355, 0.155, 0.1739,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyC14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5697, 0.511, 0.4660, 0.481, 0.5000, 0.528, 0.5560, 0.583, 0.6072, 0.622, 0.6402, 0.653, 0.6632, 0.674, 0.6850, 0.699, 0.7069, 0.717, 0.7292, 0.735, 0.7488, 0.757, 0.7678, 0.773,
    0.7786, 0.776, 0.7721, 0.765, 0.7544, 0.746, 0.7394, 0.731, 0.7232, 0.704, 0.6889, 0.674, 0.6446, 0.631, 0.6171, 0.606, 0.5966, 0.585, 0.5743, 0.5570, 0.5425, 0.529, 0.5093, 0.498,
    0.4884, 0.482, 0.4784, 0.478, 0.4774, 0.481, 0.4822, 0.487, 0.4944, 0.503, 0.5076, 0.512, 0.5186, 0.522, 0.5268, 0.529, 0.5303, 0.532, 0.5332, 0.539, 0.5454, 0.565, 0.5760,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyE4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1483, 0.161, 0.1756, 0.213, 0.2536, 0.283, 0.3084, 0.331, 0.3665, 0.387, 0.4189, 0.445, 0.4746, 0.496, 0.5127, 0.519, 0.5239, 0.522, 0.5193, 0.508, 0.4917, 0.476, 0.4569, 0.431,
    0.4123, 0.375, 0.3422, 0.309, 0.2672, 0.242, 0.2179, 0.208, 0.1820, 0.162, 0.1356, 0.113, 0.0972, 0.091, 0.0784, 0.073, 0.0698, 0.066, 0.0646, 0.062, 0.0592, 0.057, 0.0556, 0.055,
    0.0546, 0.055, 0.0551, 0.056, 0.0571, 0.059, 0.0611, 0.064, 0.0670, 0.069, 0.0701, 0.070, 0.0692, 0.067, 0.0661, 0.065, 0.0620, 0.061, 0.0606, 0.063, 0.0663, 0.072, 0.0834,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyM1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3100, 0.303, 0.2922, 0.322, 0.3514, 0.376, 0.4042, 0.424, 0.4443, 0.457, 0.4769, 0.497, 0.5002, 0.507, 0.5133, 0.516, 0.5187, 0.518, 0.5179, 0.511, 0.5057, 0.497, 0.4928, 0.483,
    0.4729, 0.454, 0.4235, 0.398, 0.3643, 0.346, 0.3371, 0.329, 0.3234, 0.301, 0.2827, 0.263, 0.2418, 0.235, 0.2338, 0.235, 0.2370, 0.236, 0.2329, 0.226, 0.2184, 0.213, 0.2028, 0.198,
    0.1958, 0.194, 0.1937, 0.196, 0.1973, 0.203, 0.2084, 0.212, 0.2244, 0.233, 0.2351, 0.236, 0.2372, 0.234, 0.2331, 0.229, 0.2239, 0.222, 0.2178, 0.224, 0.2319, 0.251, 0.2731,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSky2B1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5277, 0.485, 0.4431, 0.476, 0.4972, 0.539, 0.5820, 0.607, 0.6387, 0.653, 0.6750, 0.691, 0.7001, 0.707, 0.7140, 0.718, 0.7202, 0.720, 0.7193, 0.713, 0.7053, 0.695, 0.6891, 0.674,
    0.6657, 0.632, 0.6181, 0.587, 0.5614, 0.543, 0.5312, 0.521, 0.5101, 0.483, 0.4589, 0.431, 0.4045, 0.398, 0.3857, 0.385, 0.3826, 0.376, 0.3751, 0.364, 0.3574, 0.346, 0.3393, 0.335,
    0.3314, 0.331, 0.3304, 0.333, 0.3368, 0.346, 0.3523, 0.363, 0.3742, 0.382, 0.3874, 0.385, 0.3883, 0.384, 0.3818, 0.375, 0.3693, 0.364, 0.3616, 0.374, 0.3800, 0.396, 0.4324,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyT7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1943, 0.256, 0.3199, 0.376, 0.4536, 0.494, 0.5443, 0.572, 0.6043, 0.631, 0.6499, 0.664, 0.6839, 0.698, 0.7125, 0.726, 0.7329, 0.741, 0.7482, 0.751, 0.7527, 0.752, 0.7514, 0.745,
    0.7383, 0.721, 0.7028, 0.675, 0.6526, 0.631, 0.6034, 0.589, 0.5500, 0.512, 0.4708, 0.432, 0.3848, 0.342, 0.3268, 0.311, 0.2929, 0.282, 0.2712, 0.261, 0.2493, 0.236, 0.2316, 0.227,
    0.2243, 0.223, 0.2234, 0.229, 0.2288, 0.235, 0.2436, 0.255, 0.2640, 0.268, 0.2762, 0.277, 0.2767, 0.272, 0.2693, 0.263, 0.2566, 0.254, 0.2489, 0.255, 0.2665, 0.275, 0.3165,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyU19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5829, 0.534, 0.4865, 0.489, 0.4927, 0.532, 0.5690, 0.593, 0.6221, 0.641, 0.6532, 0.662, 0.6728, 0.674, 0.6832, 0.687, 0.6889, 0.688, 0.6884, 0.683, 0.6771, 0.671, 0.6648, 0.665,
    0.6465, 0.622, 0.6038, 0.583, 0.5524, 0.542, 0.5297, 0.523, 0.5194, 0.492, 0.4797, 0.451, 0.4387, 0.436, 0.4356, 0.442, 0.4455, 0.445, 0.4444, 0.432, 0.4282, 0.413, 0.4094, 0.404,
    0.4009, 0.400, 0.3992, 0.402, 0.4046, 0.411, 0.4185, 0.426, 0.4385, 0.446, 0.4515, 0.452, 0.4545, 0.452, 0.4505, 0.446, 0.4411, 0.438, 0.4368, 0.443, 0.4539, 0.467, 0.5013,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyU2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3594, 0.345, 0.3425, 0.333, 0.3214, 0.346, 0.3654, 0.387, 0.4097, 0.424, 0.4360, 0.446, 0.4590, 0.467, 0.4793, 0.494, 0.5002, 0.517, 0.5234, 0.538, 0.5476, 0.564, 0.5745, 0.583,
    0.5940, 0.593, 0.5901, 0.580, 0.5703, 0.563, 0.5545, 0.546, 0.5384, 0.521, 0.5029, 0.478, 0.4592, 0.444, 0.4334, 0.421, 0.4149, 0.408, 0.3947, 0.378, 0.3657, 0.352, 0.3363, 0.324,
    0.3177, 0.313, 0.3087, 0.308, 0.3077, 0.310, 0.3123, 0.317, 0.3231, 0.329, 0.3351, 0.339, 0.3454, 0.348, 0.3520, 0.353, 0.3545, 0.355, 0.3562, 0.359, 0.3674, 0.375, 0.3976,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyT17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4213, 0.376, 0.3332, 0.387, 0.4205, 0.467, 0.5078, 0.532, 0.5650, 0.587, 0.6025, 0.611, 0.6223, 0.624, 0.6279, 0.623, 0.6195, 0.607, 0.5981, 0.576, 0.5578, 0.534, 0.5197, 0.499,
    0.4785, 0.435, 0.4074, 0.376, 0.3325, 0.317, 0.3030, 0.295, 0.2918, 0.272, 0.2511, 0.235, 0.2125, 0.212, 0.2105, 0.214, 0.2198, 0.219, 0.2199, 0.215, 0.2083, 0.202, 0.1945, 0.191,
    0.1895, 0.189, 0.1898, 0.193, 0.1954, 0.202, 0.2094, 0.215, 0.2288, 0.236, 0.2406, 0.240, 0.2399, 0.236, 0.2317, 0.225, 0.2189, 0.214, 0.2108, 0.218, 0.2261, 0.245, 0.2755,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlackM8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0593, 0.040, 0.0128, 0.023, 0.0031, 0.024, 0.0149, 0.017, 0.0197, 0.021, 0.0221, 0.022, 0.0223, 0.023, 0.0230, 0.023, 0.0232, 0.023, 0.0237, 0.024, 0.0243, 0.024, 0.0247, 0.024,
    0.0249, 0.025, 0.0249, 0.025, 0.0248, 0.025, 0.0247, 0.024, 0.0244, 0.024, 0.0240, 0.024, 0.0240, 0.024, 0.0242, 0.024, 0.0244, 0.024, 0.0247, 0.025, 0.0252, 0.025, 0.0254, 0.026,
    0.0262, 0.027, 0.0269, 0.027, 0.0271, 0.027, 0.0273, 0.027, 0.0278, 0.028, 0.0280, 0.028, 0.0276, 0.028, 0.0275, 0.028, 0.0282, 0.028, 0.0288, 0.029, 0.0295, 0.030, 0.0303,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlackM12_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0377, 0.022, 0.0146, 0.015, 0.0167, 0.020, 0.0222, 0.023, 0.0257, 0.026, 0.0261, 0.027, 0.0270, 0.027, 0.0271, 0.027, 0.0274, 0.028, 0.0281, 0.028, 0.0287, 0.029, 0.0299, 0.030,
    0.0306, 0.030, 0.0304, 0.030, 0.0297, 0.029, 0.0285, 0.028, 0.0277, 0.027, 0.0267, 0.026, 0.0257, 0.026, 0.0257, 0.026, 0.0258, 0.026, 0.0259, 0.026, 0.0263, 0.027, 0.0276, 0.028,
    0.0279, 0.028, 0.0288, 0.029, 0.0297, 0.030, 0.0302, 0.030, 0.0304, 0.030, 0.0303, 0.030, 0.0302, 0.030, 0.0303, 0.031, 0.0310, 0.031, 0.0321, 0.033, 0.0337, 0.035, 0.0356,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlackM13_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2961, 0.100, 0.0487, 0.035, 0.0261, 0.030, 0.0330, 0.031, 0.0296, 0.030, 0.0305, 0.030, 0.0305, 0.031, 0.0314, 0.032, 0.0320, 0.032, 0.0322, 0.033, 0.0335, 0.034, 0.0348, 0.035,
    0.0352, 0.035, 0.0347, 0.034, 0.0336, 0.033, 0.0325, 0.032, 0.0314, 0.030, 0.0299, 0.029, 0.0284, 0.028, 0.0284, 0.029, 0.0286, 0.029, 0.0288, 0.029, 0.0290, 0.030, 0.0299, 0.030,
    0.0306, 0.031, 0.0314, 0.032, 0.0320, 0.032, 0.0328, 0.033, 0.0332, 0.033, 0.0331, 0.033, 0.0330, 0.033, 0.0328, 0.033, 0.0337, 0.034, 0.0350, 0.036, 0.0366, 0.037, 0.0385,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorWhite2B12_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3848, 0.405, 0.4127, 0.435, 0.4780, 0.501, 0.5375, 0.572, 0.5917, 0.606, 0.6194, 0.627, 0.6353, 0.642, 0.6485, 0.653, 0.6625, 0.669, 0.6784, 0.692, 0.7010, 0.072, 0.7367, 0.0760,
    0.7706, 0.775, 0.7847, 0.784, 0.7843, 0.784, 0.7826, 0.781, 0.7799, 0.775, 0.7689, 0.762, 0.7521, 0.748, 0.7463, 0.743, 0.7401, 0.731, 0.7314, 0.723, 0.7168, 0.705, 0.7008, 0.701,
    0.6904, 0.689, 0.6860, 0.686, 0.6861, 0.687, 0.6896, 0.691, 0.6986, 0.701, 0.7064, 0.711, 0.7138, 0.715, 0.7185, 0.721, 0.7233, 0.725, 0.7281, 0.731, 0.7338, 0.741, 0.7498,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorWhite2B14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.6608, 0.631, 0.5900, 0.605, 0.6114, 0.634, 0.6747, 0.698, 0.7329, 0.745, 0.7599, 0.765, 0.7706, 0.775, 0.7776, 0.781, 0.7831, 0.786, 0.7890, 0.792, 0.7952, 0.797, 0.8074, 0.810,
    0.8185, 0.821, 0.8214, 0.821, 0.8201, 0.821, 0.8218, 0.822, 0.8244, 0.823, 0.8229, 0.822, 0.8192, 0.824, 0.8281, 0.831, 0.8344, 0.836, 0.8388, 0.840, 0.8415, 0.843, 0.8445, 0.847,
    0.8495, 0.851, 0.8521, 0.853, 0.8556, 0.858, 0.8604, 0.863, 0.8666, 0.867, 0.8687, 0.869, 0.8687, 0.867, 0.8672, 0.868, 0.8682, 0.870, 0.8703, 0.871, 0.8720, 0.874, 0.8803,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blackred97_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0031, 0.0035, 0.0047, 0.0050, 0.0056, 0.0060, 0.0064, 0.0065, 0.0070, 0.007, 0.0070, 0.007, 0.0071, 0.007, 0.0066, 0.007, 0.0065, 0.006, 0.0065, 0.006, 0.0064, 0.006, 0.0063,
    0.006, 0.0063, 0.006, 0.0058, 0.005, 0.0052, 0.005, 0.0052, 0.005, 0.0053, 0.005, 0.0047, 0.005, 0.0044, 0.005, 0.0052, 0.006, 0.0066, 0.007, 0.0077, 0.008, 0.0081, 0.008, 0.008,
    0.008, 0.0084, 0.009, 0.0089, 0.009, 0.0096, 0.01, 0.0106, 0.011, 0.0121, 0.012, 0.0127, 0.012, 0.0128, 0.012, 0.0124, 0.012, 0.0115, 0.012, 0.0117, 0.013, 0.0133, 0.015, 0.0169,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blackredbl443_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0067, 0.0067, 0.0068, 0.0068, 0.0067, 0.0070, 0.0074, 0.008, 0.0083, 0.0085, 0.0088, 0.009, 0.0092, 0.0092, 0.0093, 0.0096, 0.0098, 0.0098, 0.0099, 0.01, 0.0101, 0.0104, 0.0106,
    0.0106, 0.0109, 0.0105, 0.0104, 0.01, 0.0094, 0.09, 0.0086, 0.08, 0.0075, 0.06, 0.0058, 0.05, 0.0044, 0.04, 0.0039, 0.0038, 0.0037, 0.0037, 0.0038, 0.0037, 0.0036, 0.0036,
    0.0035, 0.0036, 0.0036, 0.0034, 0.0033, 0.0033, 0.0033, 0.0035, 0.0036, 0.0037, 0.0038, 0.004, 0.0042, 0.004, 0.0041, 0.004, 0.0036, 0.0034, 0.0033, 0.0034, 0.0035, 0.004, 0.0042, 0.005, 0.0062,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blackbl27_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0060, 0.0062, 0.0064, 0.0066, 0.0070, 0.0073, 0.0076, 0.008, 0.0086, 0.0086, 0.0086, 0.0087, 0.0088, 0.0088, 0.0089, 0.0089, 0.0089, 0.0085, 0.0083, 0.008, 0.0079, 0.0078, 0.0078,
    0.0078, 0.0077, 0.007, 0.0067, 0.006, 0.0059, 0.0057, 0.0057, 0.0057, 0.0056, 0.005, 0.0046, 0.0045, 0.0041, 0.0043, 0.0045, 0.0047, 0.0050, 0.0053, 0.0056, 0.0055, 0.0054, 0.0055,
    0.0053, 0.0053, 0.0054, 0.0054, 0.0055, 0.0056, 0.0057, 0.006, 0.0065, 0.007, 0.0073, 0.0075, 0.0079, 0.008, 0.0080, 0.008, 0.0075, 0.007, 0.0067, 0.0067, 0.0068, 0.007, 0.0081, 0.01, 0.0114,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blackbl28_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0092, 0.01, 0.0118, 0.012, 0.0155, 0.016, 0.0218, 0.025, 0.0267, 0.028, 0.0296, 0.03, 0.0312, 0.031, 0.0306, 0.03, 0.0282, 0.026, 0.0244, 0.022, 0.0205, 0.02, 0.0186, 0.017, 0.0167,
    0.015, 0.0126, 0.01, 0.0091, 0.0085, 0.0080, 0.0075, 0.0071, 0.006, 0.0050, 0.05, 0.0042, 0.0042, 0.0042, 0.0043, 0.0044, 0.0044, 0.0045, 0.0045, 0.0045, 0.0043, 0.0042, 0.0042,
    0.0041, 0.004, 0.0039, 0.004, 0.0040, 0.004, 0.0043, 0.0044, 0.0048, 0.005, 0.0050, 0.0048, 0.0047, 0.0045, 0.0042, 0.0041, 0.0041, 0.0041, 0.0042, 0.0045, 0.0049, 0.006, 0.0074,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blackgr214_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0072, 0.007, 0.0062, 0.0062, 0.0061, 0.0063, 0.0064, 0.0067, 0.0069, 0.007, 0.0071, 0.0072, 0.0075, 0.008, 0.0081, 0.008, 0.0082, 0.0085, 0.0088, 0.009, 0.0103, 0.011, 0.0126,
    0.012, 0.0145, 0.0145, 0.0146, 0.014, 0.0140, 0.014, 0.0133, 0.012, 0.0118, 0.01, 0.0094, 0.008, 0.0070, 0.006, 0.0059, 0.0055, 0.0051, 0.005, 0.0046, 0.0045, 0.0045, 0.0044,
    0.0043, 0.0044, 0.0045, 0.0045, 0.0045, 0.0046, 0.0048, 0.005, 0.0056, 0.006, 0.0065, 0.007, 0.0072, 0.007, 0.0067, 0.0066, 0.0065, 0.006, 0.0057, 0.0056, 0.0058, 0.006, 0.0068, 0.008, 0.0103,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blackbl436_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0095, 0.01, 0.0123, 0.013, 0.0173, 0.02, 0.0242, 0.026, 0.0293, 0.03, 0.0317, 0.031, 0.0310, 0.03, 0.0283, 0.025, 0.0246, 0.02, 0.0197, 0.018, 0.0152, 0.014, 0.0129, 0.012,
    0.0112, 0.01, 0.0083, 0.007, 0.0063, 0.006, 0.0059, 0.0058, 0.0056, 0.005, 0.0045, 0.0045, 0.0043, 0.005, 0.0050, 0.0055, 0.0059, 0.006, 0.0064, 0.0063, 0.0062, 0.006, 0.0060,
    0.006, 0.0060, 0.006, 0.0062, 0.0064, 0.0066, 0.007, 0.0075, 0.008, 0.0086, 0.009, 0.0093, 0.009, 0.0089, 0.0086, 0.0082, 0.008, 0.0073, 0.0072, 0.0072, 0.008, 0.0088, 0.01, 0.0139,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Whitebl455_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1560, 0.18, 0.1943, 0.25, 0.2724, 0.40, 0.4469, 0.50, 0.5896, 0.60, 0.6393, 0.639, 0.6390, 0.625, 0.6239, 0.61, 0.6096, 0.60, 0.5895, 0.57, 0.5698, 0.567, 0.5662, 0.56,
    0.5603, 0.55, 0.5304, 0.51, 0.4977, 0.498, 0.4975, 0.498, 0.4998, 0.47, 0.4601, 0.44, 0.4261, 0.43, 0.4520, 0.47, 0.4986, 0.51, 0.5237, 0.525, 0.5270, 0.526, 0.5260, 0.528,
    0.5297, 0.53, 0.5375, 0.54, 0.5504, 0.56, 0.5711, 0.58, 0.5910, 0.595, 0.6023, 0.602, 0.6029, 0.600, 0.5977, 0.59, 0.5880, 0.585, 0.5830, 0.59, 0.5978, 0.61, 0.6345,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blackvio101_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0076, 0.008, 0.0094, 0.01, 0.0116, 0.012, 0.0139, 0.015, 0.0158, 0.0158, 0.0158, 0.0156, 0.0152, 0.015, 0.0142, 0.014, 0.0130, 0.012, 0.0111, 0.01, 0.0094, 0.009, 0.0090, 0.009,
    0.0085, 0.008, 0.0069, 0.006, 0.0058, 0.0057, 0.0057, 0.0058, 0.0058, 0.005, 0.0048, 0.0045, 0.0044, 0.005, 0.0060, 0.008, 0.0094, 0.011, 0.0126, 0.013, 0.0141, 0.014, 0.0141, 0.0145, 0.0148, 0.015, 0.0157, 0.016, 0.0172, 0.018, 0.0197, 0.021, 0.0227, 0.023, 0.0245, 0.0241,
    0.0242, 0.0235, 0.0233, 0.022, 0.0215, 0.021, 0.0205, 0.022, 0.0240, 0.026, 0.0321,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Whitebl92_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1652, 0.19, 0.2036, 0.24, 0.2848, 0.35, 0.4708, 0.55, 0.6230, 0.65, 0.6758, 0.6759, 0.6761, 0.665, 0.6619, 0.655, 0.6502, 0.64, 0.6343, 0.62, 0.6195, 0.6194, 0.6194, 0.618,
    0.6169, 0.605, 0.5939, 0.58, 0.5677, 0.5677, 0.5678, 0.568, 0.5695, 0.555, 0.5373, 0.52, 0.5105, 0.52, 0.5356, 0.55, 0.5778, 0.60, 0.6020, 0.605, 0.6073, 0.608, 0.6090, 0.61,
    0.6124, 0.615, 0.6187, 0.62, 0.6295, 0.64, 0.6456, 0.65, 0.6603, 0.665, 0.6693, 0.67, 0.6716, 0.671, 0.6704, 0.669, 0.6660, 0.665, 0.6640, 0.67, 0.6749, 0.69, 0.7003,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Greyredbl94_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0506, 0.053, 0.0569, 0.06, 0.0678, 0.08, 0.0906, 0.1, 0.1088, 0.11, 0.1160, 0.117, 0.1184, 0.119, 0.1195, 0.12, 0.1203, 0.1205, 0.1215, 0.125, 0.1293, 0.133, 0.1474, 0.15,
    0.1609, 0.155, 0.1538, 0.145, 0.1415, 0.143, 0.1466, 0.15, 0.1535, 0.14, 0.1364, 0.13, 0.1248, 0.14, 0.1494, 0.17, 0.1871, 0.200, 0.2098, 0.21, 0.2163, 0.217, 0.2179, 0.22,
    0.2217, 0.222, 0.2282, 0.23, 0.2372, 0.24, 0.2500, 0.256, 0.2631, 0.27, 0.2713, 0.272, 0.2739, 0.273, 0.2725, 0.27, 0.2678, 0.267, 0.2660, 0.272, 0.2770, 0.28, 0.3025,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blue32_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0933, 0.11,   0.1333, 0.17, 0.1967, 0.265, 0.3132, 0.35, 0.4096, 0.43, 0.4556, 0.465, 0.4763, 0.48, 0.4845, 0.486, .4870, 0.486, 0.4852, 0.486, 0.4868, 0.49, 0.4959, 0.496,
    0.4971, 0.48, 0.4755,  0.46, 0.4407, 0.42, 0.4085, 0.38, 0.3643, 0.32, 0.2922, 0.25, 0.2161, 0.19, 0.1658, 0.15, 0.1370, 0.12, 0.1190, 0.11,  0.1040, 0.10, 0.0947, 0.0948,
    0.0949, 0.097, 0.1001, 0.105, 0.1106, 0.12, 0.1306, 0.14, 0.1536, 0.16, 0.1652,  0.162, 0.1609, 0.154, 0.1480,  0.135, 0.1291, 0.12, 0.1188, 0.125, 0.1370, 0.16, 0.1933,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blue236_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0165, 0.021, 0.0259, 0.035, 0.0432, 0.05, 0.0684, 0.075, 0.0858, 0.087, 0.0895, 0.085, 0.0828, 0.075, 0.0683, 0.058, 0.0527, 0.045, 0.0365, 0.031, 0.0235, 0.021, 0.0177, 0.016,
    0.0143, 0.012, 0.0096, 0.0085, 0.0067, 0.0065, 0.0063, 0.0062, 0.0060, 0.006,  0.0052, 0.0052,  0.0053, 0.006,  0.0061, 0.007, 0.0074, 0.008, 0.0083, 0.0084, 0.0085, 0.0083,
    0.0080, 0.008,  0.0078, 0.0077, 0.0076, 0.008,  0.0081, 0.009, 0.0095, 0.01, 0.0112, 0.012, 0.0121, 0.012, 0.0114, 0.011, 0.0100, 0.009, 0.0086, 0.0085, 0.0083, 0.009, 0.0102, 0.0015, 0.0180,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Gre300_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0079, 0.008, 0.0085, 0.0086, 0.0087, 0.009,  0.0092, 0.0093, 0.0094, 0.0094,  0.0093, 0.0093, 0.0093, 0.0094, 0.0095, 0.0096, 0.0097, 0.01, 0.0110, 0.012, 0.0158, 0.045,
    0.0544, 0.134, 0.1907, 0.25, 0.3753, 0.435, 0.4883, 0.505, 0.5181, 0.512,  0.5044, 0.485, 0.4631, 0.433, 0.4070, 0.387,  0.3594, 0.334, 0.3262, 0.31,  0.3054,  0.295, 0.2873, 0.292,
    0.2772, 0.278, 0.2790, 0.281,  0.2875, 0.291,  0.3029, 0.3121,  0.3292, 0.333,  0.3571, 0.365,  0.3719,  0.365, 0.3693,  0.362, 0.3573, 0.342,  0.3371, 0.325,  0.3253, 0.336,  0.3471, 0.367,  0.4087,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Blue340_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0175, 0.02, 0.0320, 0.04, 0.0587, 0.08, 0.0989, 0.11, 0.1267, 0.13,  0.1322, 0.125, 0.1210, 0.111, 0.0978, 0.08, 0.0732, 0.06, 0.0487, 0.04, 0.0297, 0.025, 0.0212, 0.02,
    0.0167, 0.012, 0.0106, 0.008,  0.0069, 0.0065, 0.0064, 0.0063,  0.0062, 0.0055, 0.0051, 0.0051,  0.0052, 0.006, 0.0061, 0.007, 0.0073, 0.008, 0.0081, 0.008, 0.0080, 0.0078,
    0.0076, 0.0076, 0.0075, 0.0074, 0.0072, 0.0075, 0.0077, 0.008, 0.0088, 0.0092, 0.0105, 0.011, 0.0112, 0.0108, 0.0104, 0.01, 0.0092, 0.008,  0.0079, 0.0075, 0.0075, 0.008,
    0.0092, 0.01,  0.0167,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Gree110_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0954, 0.11, 0.1234, 0.15, 0.1702, 0.22, 0.2631, 0.312, 0.3363, 0.352, 0.3664, 0.372, 0.3799, 0.385, 0.3905, 0.395, 0.4002, 0.41, 0.4160, 0.43, 0.4582, 0.511,  0.5262, 0.544,
    0.5798, 0.585, 0.5915, 0.585, 0.5742, 0.563, 0.5465, 0.523, 0.5035, 0.465, 0.4364, 0.389, 0.3581, 0.3334, 0.2977, 0.275, 0.2589, 0.245,  0.2349, 0.223, 0.2152, 0.211,  0.2030, 0.204,
    0.2032, 0.206, 0.2106, 0.221, 0.2249, 0.243, 0.2511, 0.271,  0.2799, 0.286, 0.2951, 0.294, 0.2915, 0.2876, 0.2776, 0.2657, 0.2552, 0.2456, 0.2419, 0.253,  0.2638, 0.275,  0.3287,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Gree457_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0127, 0.0126, 0.0126, 0.012, 0.0112, 0.011, 0.0110, 0.011,  0.0109, 0.0107, 0.0106, 0.0105,  0.0105, 0.0105,  0.0104, 0.0107,  0.0110, 0.0124,  0.0136, 0.0234,  0.0318, 0.09,
    0.1246, 0.22,  0.3262, 0.43, 0.5051, 0.52, 0.5566, 0.54, 0.5181, 0.476, 0.4406, 0.398, 0.3429, 0.296, 0.2411, 0.203, 0.1647, 0.142, 0.1202, 0.1,  0.0968, 0.09,  0.0804, 0.08,
    0.0709, 0.0708, 0.0703, 0.0723,  0.0747, 0.08, 0.0840, 0.09,  0.1023, 0.11, 0.1243, 0.13, 0.1355, 0.132,  0.1311, 0.12, 0.1179, 0.10,  0.0993, 0.09,  0.0884, 0.09, 0.1043, 0.12,  0.1590,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Yel241_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0134, 0.012, 0.0106, 0.011, 0.0110, 0.011, 0.0108, 0.011, 0.0111, 0.0112, 0.0114, 0.0114, 0.0114, 0.0114, 0.0115, 0.0114, 0.0114, 0.012, 0.0122, 0.017, 0.0192, 0.05, 0.0731, 0.12,
    0.2455, 0.355, 0.4689, 0.556, 0.6183, 0.645, 0.6852, 0.698, 0.7107, 0.711, 0.7112, 0.708, 0.7059, 0.712, 0.7177, 0.724, 0.7335, 0.742, 0.7445, 0.746, 0.7487, 0.751,  0.7523, 0.753,
    0.7555, 0.758, 0.760, 0.7626, 0.7683, 0.771,  0.7779, 0.782, 0.7855, 0.791, 0.7915, 0.794, 0.7964, 0.799, 0.8011, 0.804, 0.8056, 0.807,  0.8097, 0.811,  0.8144, 0.820,  0.8239,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Ora321_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0247, 0.022, 0.0203, 0.019, 0.0182, 0.0182, 0.0183, 0.0182,  0.0182, 0.018, 0.0179, 0.018, 0.0182, 0.0185, 0.0188, 0.019, 0.0199, 0.022, 0.0249, 0.035, 0.0529, 0.112,
    0.1519, 0.231, 0.3116, 0.365, 0.4138, 0.421, 0.4410, 0.451, 0.4679, 0.485, 0.4906, 0.473, 0.4655, 0.455, 0.4517, 0.487, 0.5203, 0.578, 0.6238, 0.667, 0.6952, 0.711, 0.7270, 0.735,
    0.7406, 0.743, 0.7469, .751, 0.7527, 0.756, 0.7607, 0.765, 0.7708, 0.774, 0.7786, 0.782, 0.7849, 0.786, 0.7897, 0.79, 0.7940, 0.794, 0.7984, 0.799, 0.8025, 0.805, 0.8069, 0.811, 0.8160,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Yellow353_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0260, 0.027, 0.0294, 0.031, 0.0331, 0.037, 0.0420, 0.045, 0.0490, 0.051, 0.0517, 0.053, 0.0541, 0.056, 0.0572, 0.059, 0.0608, 0.063, 0.0664, 0.072, 0.0813, 0.096, 0.1070, 0.112,
    0.1291, 0.132, 0.1344, 0.133, 0.1317, 0.134, 0.1356, 0.137, 0.1390, 0.134, 0.1289, 0.123, 0.1195, 0.122, 0.1278, 0.134, 0.1420, 0.145, 0.1493, 0.15, 0.1500, 0.15, 0.1500, 0.153,
    0.1527, 0.154, 0.1576, 0.162, 0.1651, 0.172, 0.1759, 0.182, 0.1870, 0.191,  0.1942, 0.195, 0.1963, 0.196, 0.1952, 0.193, 0.1910, 0.19, 0.1898, 0.195, 0.1995, 0.21, 0.2209,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Mag465_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0388, 0.05, 0.0654, 0.09, 0.1020, 0.12, 0.1557, 0.17, 0.1880, 0.182, 0.1783, 0.162, 0.1434, 0.12, 0.1013, 0.09, 0.0684, 0.05, 0.0410, 0.03, 0.0219, 0.02, 0.0149, 0.012,
    0.0117, 0.009, 0.0080, 0.007, 0.0062, 0.0062, 0.0062, 0.0062, 0.0062, 0.006, 0.0056, 0.006,  0.0063, 0.008, 0.0098, 0.018, 0.0230, 0.03, 0.0440, 0.05, 0.0577, 0.06, 0.0617, 0.062,
    0.0645, 0.065, 0.0690, 0.07, 0.0766, 0.08,  0.0903, 0.1, 0.1064, 0.111,  0.1144, 0.112, 0.1113, 0.104, 0.1022, 0.09, 0.0898, 0.088, 0.0833, 0.09, 0.0962, 0.111,  0.1377,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Mag333_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1030, 0.12, 0.1508, 0.18, 0.2133, 0.26, 0.3191, 0.364, 0.3910, 0.39, 0.3895, 0.375, 0.3424, 0.312, 0.2770, 0.251, 0.2184, 0.183, 0.1593, 0.122, 0.1074, 0.1, 0.0825, 0.07,
    0.0669, 0.05, 0.0430, 0.03, 0.0265, 0.027, 0.0278, 0.03, 0.0315, 0.025, 0.0212, 0.02, 0.0163, 0.03, 0.0355, 0.05, 0.0861, 0.112, 0.1365, 0.143, 0.1565, 0.157, 0.1589, 0.16,
    0.1629, 0.165, 0.1713, 0.175, 0.1852, 0.19, 0.2099, 0.221, 0.2378, 0.245, 0.2517, 0.25, 0.2469, 0.238, 0.2322, 0.224, 0.2102, 0.206, 0.1973, 0.209, 0.2191, 0.231, 0.2855,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_Mag203_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0833, 0.11, 0.1329, 0.16, 0.2005, 0.25, 0.3099, 0.342, 0.3855, 0.39, 0.3916, 0.374, 0.3530, 0.321, 0.2926, 0.267, 0.2346, 0.21, 0.1741, 0.153, 0.1201, 0.1, 0.0934, 0.08,
    0.0759, 0.06, 0.0495, 0.04, 0.0306, 0.0307, 0.0308, 0.032, 0.0330, 0.025, 0.0214, 0.018, 0.0150, 0.02, 0.0256, 0.04, 0.0510, 0.06, 0.0723, 0.074, 0.0769, 0.075,  0.0748, 0.076,
    0.0761, 0.08, 0.0813, 0.09, 0.0911, 0.1, 0.1087, 0.115, 0.1295, 0.134, 0.1399, 0.136, 0.1353, 0.131, 0.1232, 0.114, 0.1064, 0.1, 0.0971, 0.105, 0.1137, 0.123, 0.1677,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueB6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1611, 0.18, 0.1947, 0.23, 0.2639, 0.31, 0.3488, 0.37, 0.4022, 0.43, 0.4517, 0.49, 0.501, 0.52, 0.5317, 0.534, 0.5367, 0.53, 0.5246, 0.51, 0.4905, 0.47, 0.4510, 0.43, 0.4059,
    0.37, 0.3351, 0.31, 0.2612, 0.24, 0.2177, 0.205, 0.1883, 0.16, 0.1444, 0.123, 0.1065, 0.1, 0.0889, 0.085, 0.0811, 0.08, 0.0757, 0.071, 0.0695, 0.067, 0.0648, 0.064, 0.0634,
    0.063, 0.0637,  0.065, 0.0662, 0.068, 0.0714, 0.075, 0.0787, 0.08,  0.0828, 0.082,  0.0822, 0.08, 0.0781, 0.075, 0.0726,  0.071, 0.0698, 0.075, 0.0770, 0.08,   0.0973,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueB15_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2207, 0.2, 0.1897, 0.21,  0.2274, 0.25, 0.2805, 0.30, 0.3216, 0.34, 0.3596, 0.38, 0.4003, 0.42, 0.4438, 0.47, 0.4909, 0.52, 0.5469, 0.57, 0.6009, 0.62, 0.6414, 0.65, 0.6557,
    0.64, 0.6275, 0.59, 0.5675, 0.53, 0.5022, 0.48, 0.4348, 0.39, 0.3473, 0.31, 0.2613, 0.24, 0.2039, 0.18, 0.1696, 0.15, 0.1465, 0.13, 0.1243, 0.11,  0.1055, 0.10, 0.0952, 0.094,
    0.0911, 0.091, 0.0906, 0.092, 0.0940, 0.1, 0.1006, 0.104, 0.1075, 0.11, 0.1125, 0.113,  0.1146, 0.113, 0.1129, 0.112, 0.1120, 0.115, 0.1198, 0.13,  0.1431,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueC2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2534, 0.25, 0.2454, 0.28, 0.3101, 0.33, 0.3567, 0.37, 0.3981, 0.41, 0.4258, 0.43, 0.4472, 0.45, 0.4574, 0.455, 0.4571, 0.45,  0.4468, 0.43, 0.4205, 0.41, 0.3947, 0.38,
    0.3647, 0.33, 0.3074, 0.27, 0.2471, 0.23, 0.2221, 0.22, 0.2128, 0.20,  0.1810, 0.17, 0.1510, 0.15,  0.1486, 0.15, 0.1546, 0.154, 0.1532, 0.15, 0.1426, 0.135, 0.1310, 0.125,
    0.1257, 0.125,  0.1246, 0.126,  0.1276, 0.13, 0.1360, 0.14, 0.1483, 0.15, 0.1566, 0.157, 0.1575, 0.155, 0.1535, 0.15, 0.1453, 0.143,  0.1403, 0.146, 0.1510, 0.16,  0.1855,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueC14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5697, 0.51, 0.4660, 0.49, 0.5000, 0.53, 0.5560, 0.58,  0.6072, 0.62, 0.6402, 0.65, 0.6632, 0.67, 0.6850, 0.69, 0.7069, 0.71, 0.7292, 0.74, 0.7488, 0.75, 0.7678, 0.77, 0.7786,
    0.775, 0.7721, 0.76, 0.7544, 0.75,  0.7394, 0.73, 0.7232, 0.70, 0.6889, 0.66, 0.6446, 0.63, 0.6171, 0.61, 0.5966, 0.58, 0.5743, 0.56, 0.5425, 0.53, 0.5093, 0.50,  0.4884, 0.48,
    0.4784, 0.475, 0.4774, 0.48, 0.4822, 0.49, 0.4944, 0.50, 0.5076, 0.51,  0.5186, 0.52, 0.5268, 0.53, 0.5303, 0.532, 0.5332, 0.54, 0.5454, 0.56,  0.5760,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueC16_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1909, 0.10, 0.0635, 0.06, 0.0554, 0.056,  0.0571, 0.057,  0.0580, 0.06, 0.0608, 0.063, 0.0662, 0.07,  0.0712, 0.072,  0.0742, 0.08,   0.0811, 0.09, 0.0985, 0.11, 0.1363, 0.15,
    0.1743, 0.173, 0.1720, 0.15, 0.1372, 0.12,  0.1005, 0.09, 0.0731, 0.06, 0.0509, 0.04,  0.0374, 0.036,   0.0322, 0.032,  0.0308, 0.031,  0.0309, 0.031,  0.0319, 0.033,
    0.0333, 0.034,  0.0349, 0.035,  0.0364, 0.037,  0.0377, 0.038,  0.0386, 0.0386, 0.0386, 0.0384, 0.0383, 0.038,  0.0377, 0.0377, 0.0378, 0.038,  0.0386, 0.04,   0.0417, 0.043,
    0.0461, 0.049,  0.0514,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueF1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0060, 0.04,  0.0975, 0.11, 0.1297, 0.13, 0.1426, 0.15,   0.1538, 0.16,   0.1624, 0.165,  0.1684, 0.17,   0.1727, 0.175,  0.1761, 0.178,  0.1798, 0.18,   0.1863, 0.19,   0.1993, 0.20,
    0.2091, 0.205, 0.2041, 0.20,    0.1902, 0.19,   0.1836, 0.182,  0.1809, 0.17,   0.1675, 0.16,   0.1527, 0.151,  0.1511, 0.153,  0.1541, 0.154,  0.1545, 0.153,  0.1516, 0.15,
    0.1484, 0.149,  0.1489, 0.15,   0.1516, 0.153,  0.1563, 0.16,   0.1654, 0.17,   0.1766, 0.18,   0.1827, 0.182,  0.1817, 0.18,   0.1766, 0.17,   0.1693, 0.166,  0.1643, 0.17,   0.1726, 0.18,   0.1978,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueF2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1613, 0.20, 0.2384, 0.30, 0.3288, 0.35,   0.4026, 0.43, 0.4630, 0.48, 0.4992, 0.51, 0.5294, 0.53, 0.5377, 0.53, 0.5261, 0.51, 0.4995, 0.48, 0.4513, 0.43, 0.4064, 0.38,
    0.3620, 0.31, 0.2935, 0.25, 0.2263, 0.21, 0.1977, 0.19, 0.1851, 0.17, 0.1519, 0.14, 0.1213, 0.12, 0.1149, 0.113, 0.1162, 0.114, 0.1133, 0.11, 0.1053, 0.10, 0.0976, 0.095,
    0.0947, 0.095,  0.0953, 0.097,  0.0985, 0.1, 0.1070, 0.11,  0.1188, 0.12,   0.1256, 0.124,  0.1246, 0.12,   0.1187, 0.11,   0.1099, 0.105,  0.1048, 0.11,   0.1145, 0.12,   0.145,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueF10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3003, 0.20,   0.1512, 0.20,   0.2149, 0.25,   0.2709, 0.30, 0.3127, 0.32, 0.3534, 0.37, 0.4001, 0.42, 0.4456, 0.46, 0.4882, 0.51, 0.5343, 0.55,   0.5728, 0.58,   0.5968, 0.595,
    0.5932, 0.56, 0.5490, 0.51, 0.4756, 0.43, 0.3989, 0.37, 0.3256, 0.28, 0.2424, 0.21, 0.1691, 0.15,   0.1239, 0.10,   0.0998, 0.09,   0.0866, 0.08, 0.0752, 0.07, 0.0664, 0.065,
    0.0623, 0.062,  0.0616, 0.062,  0.0626, 0.064,  0.0662, 0.07,   0.0720, 0.075, 0.0767, 0.077, 0.0784, 0.078,  0.0770, 0.075,  0.0734, 0.073,  0.0716, 0.075,  0.0779, 0.08,   0.0968,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueF13_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3075, 0.22,   0.1514, 0.13,   0.1216, 0.125,  0.1264, 0.13,   0.1312, 0.14,   0.1448, 0.15,   0.1665, 0.17,   0.1850, 0.19,   0.1922, 0.195,  0.1970, 0.20,   0.2036, 0.21,
    0.2136, 0.213,  0.2122, 0.19,   0.1832, 0.16,   0.1412, 0.12,   0.1066, 0.09,   0.0810, 0.07,   0.0579, 0.05,   0.0422, 0.04,   0.0358, 0.035,  0.0336, 0.033,  0.0335, 0.0335,
    0.0339, 0.034,  0.0348, 0.035,  0.0358, 0.036,  0.0371, 0.038,  0.0384, 0.039,  0.0394, 0.04,   0.0405, 0.0404, 0.0405, 0.04,   0.0399, 0.039,  0.0387, 0.039,
    0.0396, 0.04,   0.0422, 0.045,  0.0467, 0.05,   0.0527,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueG9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0937, 0.09,   0.0891, 0.07,   0.0621, 0.07,   0.0735, 0.08,   0.0842, 0.09,   0.0918, 0.1,    0.1009, 0.105,  0.1091, 0.11,   0.1158, 0.12,   0.1254, 0.13,   0.1452, 0.16,
    0.1805, 0.20,   0.2126, 0.213,  0.2128, 0.20,   0.1863, 0.17,   0.1550, 0.14,   0.1270, 0.1,    0.0973, 0.08,   0.0725, 0.06,   0.0585, 0.055,  0.0521, 0.05,   0.0486, 0.047,
    0.0458, 0.045,  0.0441, 0.044,  0.0439, 0.044,  0.0443, 0.045,  0.0455, 0.046,  0.0478, 0.049,  0.0500, 0.051,  0.0512, 0.051,  0.0511, 0.05,   0.0498, 0.049,
    0.0487, 0.049,  0.0495, 0.05,   0.0531, 0.055,  0.0620,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueG19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1198, 0.12,   0.1376, 0.15,   0.1735, 0.18,   0.1903, 0.20,   0.2116, 0.22,   0.2322, 0.24,   0.2555, 0.27,   0.2802, 0.29,   0.3077, 0.32,   0.3420, 0.36,   0.3822, 0.41,
    0.4261, 0.43,   0.4565, 0.45, 0.4481, 0.43, 0.4109, 0.39,   0.3736, 0.35,   0.3368, 0.30,   0.2811, 0.25,   0.2228, 0.19,   0.1839, 0.17,   0.1592, 0.15,   0.1394, 0.12,
    0.1183, 0.1,    0.0998, 0.09,   0.0897, 0.085,  0.0851, 0.085,  0.0844, 0.085,  0.0863, 0.09,   0.0912, 0.095,  0.0977, 0.1,    0.1027, 0.105,  0.1064, 0.106,  0.1064, 0.106,
    0.1067, 0.11,   0.1135, 0.12,   0.1319,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueI5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1096, 012, 0.1546, 0.16,  0.1855, 0.2,    0.2350, 0.25,   0.2720, 0.29,   0.3065, 0.32,   0.3404, 0.36,   0.3782, 0.4,    0.4229, 0.45,   0.4801, 0.51,   0.5416, 0.57,
    0.5962, 0.61,   0.6281, 0.62,   0.6144, 0.58,   0.5680, 0.55,   0.5211, 0.5,    0.4726, 0.43,   0.3991, 0.37,   0.3209, 0.29,   0.2674, 0.25,   0.2311, 0.22,   0.2000, 0.18,
    0.1656, 0.15,   0.1349, 0.12,   0.1161, 0.11,   0.1078, 0.105,  0.1046, 0.105,  0.1049, 0.107,  0.1097, 0.11,   0.1179, 0.12,   0.1264, 0.13,   0.1337, 0.135,  0.1378, 0.14,   0.1402, 0.145,  0.1474, 0.15,   0.1662,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueH15_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1047, 0.105,  0.1078, 0.12,   0.1550, 0.20,   0.2110, 0.23,   0.2487, 0.27,   0.2893, 0.31,   0.3386, 0.35,   0.3723, 0.375,  0.3773, 0.37,   0.3665, 0.35,   0.3373, 0.32,
    0.3037, 0.28,   0.2663, 0.24,   0.2106, 0.17,   0.1540, 0.13,   0.1200, 0.11,   0.0980, 0.08,   0.0721, 0.06,   0.0527, 0.05,   0.0448, 0.043,  0.0418, 0.041,  0.0403, 0.04,
    0.0390, 0.039,  0.0386, 0.039,  0.0392, 0.04,   0.0401, 0.041,  0.0413, 0.042,  0.0435, 0.044,  0.0456, 0.046,  0.0465, 0.046,  0.0457, 0.045,  0.0447, 0.044,  0.0435, 0.044,
    0.0442, 0.046,  0.0485, 0.05,   0.0576,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueI3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1759, 0.21,   0.2647, 0.31,   0.3209, 0.36,   0.4249, 0.45,   0.4957, 0.52,   0.5536, 0.57,   0.6073, 0.62,   0.6431, 0.65,   0.6538, 0.653,  0.6506, 0.64,   0.6282, 0.61,
    0.5973, 0.57,   0.5559, 0.52,   0.4856, 0.43,   0.4033, 0.37,   0.3417, 0.32,   0.2901, 0.26,   0.2211, 0.19,   0.1585, 0.14,   0.1236, 0.11,   0.1062, 0.1,    0.0959, 0.09,
    0.0865, 0.085,  0.0801, 0.08,   0.0780, 0.079,  0.0788, 0.08,   0.0819, 0.09,   0.0903, 0.1,    0.1009, 0.105,  0.1073, 0.106,  0.1055, 0.1,    0.0998, 0.095,  0.0914, 0.09,   0.0863, 0.09,   0.0961, 0.11,   0.1250,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueI19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3031, 0.28,   0.2588, 0.25,   0.2472, 0.26,   0.2738, 0.29,   0.3035, 0.31,   0.3271, 0.33,   0.3496, 0.36,   0.3724, 0.38,   0.3968, 0.41,   0.4252, 0.44,   0.4562, 0.47,
    0.4899, 0.5,    0.5147, 0.512,  0.5100, 0.49,   0.4841, 0.47,   0.4608, 0.45,   0.4373, 0.42,   0.3934, 0.37,   0.3420, 0.32,   0.3086, 0.30,   0.2850, 0.27,   0.2622, 0.25,
    0.2334, 0.22,   0.2054, 0.19,   0.1887, 0.185,  0.1806, 0.18,   0.1791, 0.18,   0.1825, 0.185,  0.1909, 0.2,    0.2018, 0.21,   0.2108, 0.213,  0.2164, 0.217,  0.2180, 0.219,
    0.2198, 0.22,   0.2295, 0.24,   0.2562,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueJ4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0290, 0.02,  0.0598, 0.1,    0.1278, 0.15,   0.1764, 0.19,   0.2052, 0.21,   0.2331, 0.25,   0.2642, 0.28,   0.2906, 0.3,    0.3093, 0.31,   0.3263, 0.33,   0.3413, 0.35,
    0.3550, 0.354,  0.3532, 0.33,   0.3206, 0.3,    0.2685, 0.24,   0.2212, 0.2,    0.1802, 0.16,   0.1346, 0.1,    0.0965, 0.08,   0.0751, 0.07,   0.0646, 0.06,   0.0586, 0.055,
    0.0535, 0.052,  0.0504, 0.05,   0.0491, 0.0495, 0.0493, 0.05,   0.0504, 0.051,  0.0529, 0.053,  0.0565, 0.057,  0.0591, 0.059,  0.0593, 0.058,  0.0576, 0.056,  0.0556, 0.055,
    0.0553, 0.06,   0.0602, 0.065,  0.0727,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueJ6_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3039, 0.3,    0.2571, 0.27,   0.2960, 0.31,   0.3446, 0.37,   0.3937, 0.41,   0.4405, 0.47,   0.4884, 0.51,   0.5337, 0.56,   0.5771, 0.59,   0.6221, 0.64,   0.6589, 0.67,
    0.6808, 0.68,   0.6770, 0.65,   0.6375, 0.59,   0.5693, 0.53,   0.4933, 0.45,   0.4160, 0.39,   0.3223, 0.26,   0.2338, 0.21,   0.1761, 0.16,   0.1447, 0.13,   0.1258, 0.12,
    0.1090, 0.1,    0.0961, 0.09,   0.0899, 0.089,  0.0877, 0.088,  0.0892, 0.09,   0.0949, 0.1,    0.1045, 0.11,   0.1115, 0.112,  0.1131, 0.112,  0.1115, 0.11,   0.1065, 0.105,
    0.1033, 0.11,   0.1122, 0.12,   0.1400,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_BlueJ11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1274, 0.1,    0.091, 0.08,    0.0656, 0.064,  0.0604, 0.06,   0.0570, 0.06,   0.0604, 0.061,  0.0644, 0.065,  0.0668, 0.07,   0.0700, 0.072,  0.0754, 0.08,   0.0874, 0.1,
    0.1111, 0.12,   0.1327, 0.132,  0.1313, 0.12,   0.1127, 0.1,    0.0931, 0.08,   0.0758, 0.06,   0.0580, 0.05,   0.0449, 0.04,   0.0385, 0.036,  0.0360, 0.035,  0.0351, 0.035,
    0.0351, 0.035,  0.0355, 0.036,  0.0371, 0.0375, 0.0379, 0.038,  0.0388, 0.04,   0.0406, 0.041,  0.0414, 0.0415, 0.0416, 0.041,  0.0409, 0.04,
    0.0398, 0.04,   0.0397, 0.04,   0.0424, 0.043,  0.0458, 0.05,   0.0522,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueJ13_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2802, 0.28,   0.2820, 0.3,    0.3461, 0.4,    0.4357, 0.45,   0.5027, 0.53,   0.5528, 0.58,   0.6001, 0.62,   0.6402, 0.66,   0.6730, 0.69,   0.7019, 0.71,   0.7216, 0.725,
    0.7288, 0.72,   0.7172, 0.7,    0.6779, 0.65,   0.6160, 0.58,   0.5478, 0.52,   0.4751, 0.43,   0.3816, 0.35,   0.2882, 0.25,   0.2260, 0.2,    0.1905, 0.18,   0.1699, 0.16,
    0.1514, 0.14,   0.1361, 0.13,   0.1295, 0.129,  0.1281, 0.13,   0.1309, 0.135,  0.1408, 0.145,  0.1546, 0.16,   0.1644, 0.165, 0.1656, 0.162,
    0.1606, 0.155,  0.1516, 0.15,   0.1461, 0.15,   0.1585, 0.16,   0.1975,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueK5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1727, 0.18,   0.1814, 0.19,   0.1931, 0.21,   0.2379, 0.25,   0.2728, 0.29,   0.3054, 0.31,   0.3396, 0.36,   0.3780, 0.4,    0.4243, 0.45,   0.4829, 0.51,   0.5464, 0.57,
    0.6024, 0.62,   0.6348, 0.63,   0.6209, 0.59,   0.5741, 0.55,   0.5258, 0.49,   0.4755, 0.43,   0.4003, 0.37,   0.3204, 0.3,    0.2656, 0.25,   0.2282, 0.21,   0.1966, 0.17,
    0.1624, 0.15,   0.1322, 0.12,   0.1137, 0.11,   0.1052, 0.105,  0.1025, 0.102,  0.1028, 0.105,  0.1075, 0.11,   0.1155, 0.12,   0.1237, 0.13,   0.1310, 0.133,
    0.1352, 0.136,  0.1370, 0.14,   0.1444, 0.15,   0.1632,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueN1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1936, 0.194,  0.1942, 0.22,  0.2692, 0.29,  0.3102, 0.33,  0.3535, 0.37,  0.3885, 0.4,  0.4230, 0.44,  0.4594, 0.48,  0.5011, 0.52,  0.5510, 0.58,  0.6022, 0.62,  0.6479, 0.66,
    0.6748, 0.67,  0.6640, 0.65,  0.6268, 0.61,  0.5919, 0.58,  0.5549, 0.53,  0.4921, 0.46,  0.4209, 0.39,  0.3723, 0.35,  0.3364, 0.32,  0.3005, 0.28,  0.2569, 0.23,  0.2154, 0.19,
    0.1897, 0.18,  0.1768, 0.175,  0.1723, 0.173,  0.1722, 0.175,  0.1785, 0.18,  0.1892, 0.19,  0.2017, 0.21,  0.2134, 0.22,  0.2202, 0.222,  0.2247, 0.23,  0.2333, 0.24,  0.2566,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueN4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3138, 0.25,   0.2038, 0.19,   0.1787, 0.18,   0.1840, 0.2,    0.2109, 0.23,   0.2400, 0.26,   0.2715, 0.29,   0.3075, 0.33,   0.3534, 0.39,   0.4149, 0.45,   0.4863, 0.52,
    0.5520, 0.57,   0.5904, 0.58,   0.5750, 0.55,   0.5198, 0.48,   0.4602, 0.43,   0.3997, 0.36,   0.3199, 0.27,   0.2411, 0.2,    0.1883, 0.17,   0.1553, 0.14,   0.1307, 0.12,
    0.1062, 0.1,    0.0859, 0.08,   0.0746, 0.07,   0.0697, 0.069,  0.0678, 0.068,  0.0683, 0.07, 0.0715, 0.075,  0.0765, 0.08,   0.0818, 0.085,  0.0862, 0.087,  0.0885, 0.089,
    0.0899, 0.09,   0.0951, 0.1,    0.1090,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueO19_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1231, 0.13,  0.1348, 0.12,  0.1152, 0.12,  0.1368, 0.15,  0.1627, 0.17,  0.1860, 0.2,  0.2130, 0.23,  0.2462, 0.26,  0.2859, 0.31,  0.3425, 0.38,  0.4152, 0.45,  0.4902, 0.51,
    0.5386, 0.53,  0.5259, 0.49,  0.4652, 0.43,  0.3950, 0.35,  0.3266, 0.28,  0.2469, 0.21,  0.1750, 0.15,  0.1293, 0.12,  0.1033, 0.09,  0.0859, 0.08,  0.0703, 0.06,  0.0578, 0.055,
    0.0521, 0.05,  0.0494, 0.049,  0.0492, 0.049,  0.0498, 0.05,  0.0522, 0.053,  0.0558, 0.058,  0.0590, 0.06,  0.0615, 0.062,  0.0625, 0.063,  0.0634, 0.065,  0.0670, 0.07,  0.0776,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_BlueU8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0776, 0.1,  0.1703, 0.23,  0.2942, 0.35,  0.4022, 0.45,  0.4795, 0.51,  0.5389, 0.56,  0.5968, 0.6,  0.6362, 0.64,  0.6512, 0.652,  0.6523, 0.64,  0.6340, 0.62,  0.6052, 0.58,
    0.5645, 0.53,  0.4952, 0.45,  0.4124, 0.38,  0.3457, 0.32,  0.2883, 0.25,  0.2164, 0.19,  0.1529, 0.14,  0.1168, 0.1,  0.0983, 0.09,  0.0883, 0.08,  0.0798, 0.075,  0.0735, 0.073,
    0.0720, 0.072,  0.0726, 0.074,  0.0757, 0.08,  0.0830, 0.09,  0.0930, 0.095,  0.0989, 0.097,  0.0971, 0.095,  0.0911, 0.085,  0.0832, 0.08,  0.0795, 0.082,  0.0877, 0.09,  0.1140,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuN8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0632, -0.04, -0.0244, 0.01,  0.0125, 0.02,   0.0294, 0.03,   0.0326, 0.033,  0.0352, 0.036,  0.0361, 0.037,  0.0374, 0.0374, 0.0373, 0.038,  0.0378, 0.039,  0.0397, 0.04,
    0.0421, 0.043,  0.0431, 0.042,  0.0417, 0.04,   0.0391, 0.038,  0.0378, 0.037,  0.0368, 0.035,  0.0347, 0.034,  0.0335, 0.034,  0.0341, 0.035,  0.0350, 0.035,  0.0355, 0.036,
    0.0357, 0.0357, 0.0358, 0.036,  0.0369, 0.037,  0.0372, 0.0376, 0.0378, 0.038,  0.0388, 0.039,  0.0397, 0.04,   0.0400, 0.0395, 0.0394, 0.039,  0.0386, 0.0385, 0.0384, 0.039,
    0.0395, 0.04,   0.0415, 0.043,  0.0448,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuN8_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0632, -0.04, -0.0244, 0.01,  0.0135, 0.02,   0.0294, 0.03,   0.0326, 0.033,  0.0352, 0.036,  0.0361, 0.037,  0.0374, 0.0374, 0.0373, 0.038,  0.0378, 0.039,  0.0397, 0.04,
    0.0421, 0.043,  0.0431, 0.042,  0.0417, 0.04,   0.0391, 0.038,  0.0378, 0.037,  0.0368, 0.035,  0.0347, 0.034,  0.0335, 0.034,  0.0341, 0.035,  0.0350, 0.035,  0.0355, 0.036,
    0.0357, 0.0357, 0.0358, 0.036,  0.0369, 0.037,  0.0372, 0.0376, 0.0378, 0.038,  0.0388, 0.039,  0.0397, 0.04,   0.0400, 0.0395, 0.0394, 0.039,  0.0386, 0.0385, 0.0384, 0.039,
    0.0395, 0.04,   0.0415, 0.043,  0.0448,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuN9_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0573, 0.054,  0.0516, 0.053,  0.0548, 0.05,   0.0401, 0.041,  0.0424, 0.043,  0.0449, 0.046,  0.0467, 0.047,  0.0473, 0.0473, 0.0474, 0.048,  0.0483, 0.05,   0.0508, 0.053,
    0.0558, 0.057,  0.0584, 0.058,  0.0550, 0.05,   0.0495, 0.048,  0.0468, 0.0465, 0.0460, 0.044,  0.0430, 0.042,  0.0411, 0.042,  0.0425, 0.044,  0.0457, 0.046,  0.0473, 0.0474,
    0.0475, 0.0475, 0.0474, 0.0475, 0.0476, 0.048,  0.0487, 0.049,  0.0499, 0.05,   0.0515, 0.052,  0.0533, 0.054,  0.0544, 0.054,  0.0539, 0.053,  0.0526, 0.052,  0.0512, 0.0515,
    0.0515, 0.053,  0.0538, 0.056,  0.0597,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuN9_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0573, 0.054,  0.0516, 0.053,  0.0548, 0.05,   0.0401, 0.041,  0.0424, 0.043,  0.0459, 0.046,  0.0467, 0.047,  0.0473, 0.0473, 0.0474, 0.048,  0.0483, 0.05,   0.0508, 0.053,
    0.0558, 0.057,  0.0584, 0.058,  0.0550, 0.05,   0.0495, 0.048,  0.0468, 0.0465, 0.0460, 0.044,  0.0430, 0.042,  0.0411, 0.042,  0.0425, 0.044,  0.0457, 0.046,  0.0473, 0.0474,
    0.0475, 0.0475, 0.0474, 0.0475, 0.0476, 0.048,  0.0487, 0.049,  0.0499, 0.05,   0.0515, 0.052,  0.0533, 0.054,  0.0544, 0.054,  0.0539, 0.053,  0.0526, 0.052,  0.0512, 0.0515,
    0.0515, 0.053,  0.0538, 0.056,  0.0597,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuO8_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0014, 0.02,  0.0806, 0.07,   0.0673, 0.07,   0.0854, 0.09,   0.0901, 0.095,  0.0960, 0.098,  0.0992, 0.1,    0.1017, 0.102,  0.1030, 0.104,  0.1052, 0.107,  0.1098, 0.11,
    0.1176, 0.12,   0.1230, 0.12,   0.1176, 0.11,   0.1071, 0.105,  0.1032, 0.103,  0.1032, 0.1,    0.0963, 0.09,   0.0899, 0.09,   0.0939, 0.095,  0.1007, 0.102,  0.1037, 0.104,
    0.1029, 0.102,  0.1014, 0.102,  0.1020, 0.103,  0.1039, 0.105,  0.1072, 0.11,   0.1134, 0.12,   0.1207, 0.122,  0.1245, 0.123,  0.1236, 0.121,  0.1205, 0.12,   0.1158, 0.115,
    0.1132, 0.115,  0.1185, 0.12,   0.1345,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuO8_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0014, 0.02,  0.0806, 0.07,   0.0673, 0.07,   0.0854, 0.09,   0.0901, 0.095,  0.0960, 0.098,  0.0992, 0.1,    0.1017, 0.102,  0.1030, 0.104,  0.1052, 0.107,  0.1098, 0.11,
    0.1176, 0.12,   0.1230, 0.12,   0.1276, 0.11,   0.1071, 0.105,  0.1032, 0.103,  0.1032, 0.1,    0.0963, 0.09,   0.0899, 0.09,   0.0939, 0.095,  0.1007, 0.102,  0.1037, 0.104,
    0.1029, 0.102,  0.1014, 0.102,  0.1220, 0.103,  0.1039, 0.105,  0.1072, 0.11,   0.1134, 0.12,   0.1207, 0.122,  0.1245, 0.123,  0.1236, 0.121,  0.1205, 0.12,   0.1158, 0.115,
    0.1132, 0.115,  0.1185, 0.12,   0.1345,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuO11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2926, 0.2,    0.1863, 0.16,   0.1428, 0.14,   0.1322, 0.134,  0.1396, 0.14,   0.1450, 0.146,  0.1498, 0.15,   0.1527, 0.155,  0.1554, 0.157,  0.1583, 0.16,   0.1631, 0.17,
    0.1754, 0.18,   0.1841, 0.18,   0.1761, 0.17,   0.1600, 0.155,  0.1549, 0.155,  0.1555, 0.15,   0.1449, 0.14,   0.1352, 0.14,   0.1414, 0.15,   0.1519, 0.153,  0.1568, 0.156,
    0.1556, 0.154,  0.1534, 0.154,  0.1547, 0.156,  0.1573, 0.16,   0.1622, 0.17,   0.1713, 0.18,   0.1823, 0.185,  0.1886, 0.188,  0.1873, 0.183,  0.1829, 0.18,   0.1753, 0.174,
    0.1716, 0.175,  0.1800, 0.2,    0.2039,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuO11_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2926, 0.2,    0.1863, 0.16,   0.1428, 0.14,   0.1322, 0.134,  0.1396, 0.14,   0.1450, 0.146,  0.1498, 0.15,   0.1527, 0.155,  0.1554, 0.157,  0.1583, 0.16,   0.1631, 0.17,
    0.1754, 0.18,   0.1841, 0.18,   0.1861, 0.17,   0.1600, 0.155,  0.1549, 0.155,  0.1555, 0.15,   0.1449, 0.14,   0.1352, 0.14,   0.1414, 0.15,   0.1519, 0.153,  0.1568, 0.156,
    0.1556, 0.154,  0.1534, 0.154,  0.1647, 0.156,  0.1573, 0.16,   0.1622, 0.17,   0.1713, 0.18,   0.1823, 0.185,  0.1886, 0.188,  0.1873, 0.183,  0.1829, 0.18,   0.1753, 0.174,
    0.1716, 0.175,  0.1800, 0.2,    0.2039,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuD5_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0840, 0.1,    0.1627, 0.18,   0.1934, 0.2,    0.2234, 0.23,   0.2430, 0.25,   0.2547, 0.26,   0.2618, 0.264,  0.2651, 0.265,  0.2655, 0.2655, 0.2659, 0.266,  0.2674, 0.27,
    0.2776, 0.28,   0.2841, 0.27,   0.2654, 0.25,   0.2351, 0.23,   0.2246, 0.225,  0.2247, 0.22,   0.2074, 0.20,   0.1913, 0.20,   0.2029, 0.21,   0.2231, 0.23,   0.2337, 0.233,
    0.2327, 0.23,   0.2291, 0.23,   0.2305, 0.232,  0.2344, 0.24,   0.2417, 0.25,   0.2553, 0.26,   0.2724, 0.28,   0.2816, 0.28,   0.2797, 0.276,  0.2720, 0.27,
    0.2603, 0.26,   0.2536, 0.26,   0.2660, 0.28,   0.3027,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuD5_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0840, 0.1,    0.1627, 0.18,   0.1934, 0.2,    0.2234, 0.23,   0.2430, 0.25,   0.2547, 0.26,   0.2618, 0.264,  0.2651, 0.265,  0.2655, 0.2655, 0.2659, 0.266,  0.2674, 0.27,
    0.2776, 0.28,   0.2841, 0.27,   0.2854, 0.25,   0.2351, 0.23,   0.2246, 0.225,  0.2247, 0.22,   0.2074, 0.20,   0.1913, 0.20,   0.2029, 0.21,   0.2231, 0.23,   0.2337, 0.233,
    0.2327, 0.23,   0.2291, 0.23,   0.2405, 0.232,  0.2344, 0.24,   0.2417, 0.25,   0.2553, 0.26,   0.2724, 0.28,   0.2816, 0.28,   0.2797, 0.276,  0.2720, 0.27,
    0.2603, 0.26,   0.2536, 0.26,   0.2660, 0.28,   0.3027,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuE11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1699, 0.18,  0.1971, 0.21,  0.2276, 0.23,  0.2483, 0.25,  0.2690, 0.28,  0.2820, 0.29,  0.2916, 0.295,  0.2992, 0.3,  0.3064, 0.31,  0.3151, 0.32,  0.3301, 0.34,  0.3593, 0.37,
    0.3873, 0.39,  0.3913, 0.38,  0.3793, 0.375,  0.3723, 0.37,  0.3678, 0.35,  0.3482, 0.33,  0.3249, 0.32,  0.3188, 0.319,  0.3188, 0.318,  0.3179, 0.315,  0.3128, 0.31,
    0.3086, 0.31,  0.3105, 0.312,  0.3148, 0.313,  0.3222, 0.33,  0.3364, 0.34,  0.3535, 0.36,  0.3629, 0.362,  0.3621, 0.36,  0.3549, 0.35,  0.3444, 0.34,  0.3394, 0.35,  0.3511, 0.36,  0.3862,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuE11_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1699, 0.18,  0.1971, 0.21,  0.2276, 0.23,  0.2483, 0.25,  0.2690, 0.28,  0.2820, 0.29,  0.2916, 0.295,  0.2992, 0.3,  0.3064, 0.31,  0.3151, 0.32,  0.3301, 0.34,  0.3593, 0.37,
    0.3873, 0.39,  0.3913, 0.38,  0.3993, 0.375,  0.3723, 0.37,  0.3678, 0.35,  0.3482, 0.33,  0.3249, 0.32,  0.3188, 0.319,  0.3188, 0.318,  0.3179, 0.315,  0.3128, 0.31,
    0.3086, 0.31,  0.3105, 0.312,  0.3248, 0.313,  0.3222, 0.33,  0.3364, 0.34,  0.3535, 0.36,  0.3629, 0.362,  0.3621, 0.36,  0.3549, 0.35,  0.3444, 0.34,  0.3394, 0.35,  0.3511, 0.36,  0.3862,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuK16_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5837, 0.45,  0.4117, 0.43,  0.4427, 0.47,  0.5098, 0.52,  0.5451, 0.55,  0.5698, 0.57,  0.5828, 0.59,  0.5939, 0.6,  0.6045, 0.607,  0.6140, 0.62,  0.6219, 0.63,  0.6330, 0.64,
    0.6419, 0.643,  0.6440, 0.642,  0.6417, 0.64,  0.6379, 0.631,  0.6309, 0.62,  0.6154, 0.6,  0.5911, 0.58,  0.5736, 0.57,  0.5612, 0.56,  0.5539, 0.55,  0.5462, 0.543,
    0.5406, 0.542,  0.5418, 0.543,  0.5452, 0.55,  0.5529, 0.56,  0.5654, 0.57,  0.5806, 0.584,  0.5888, 0.589,  0.5898, 0.586,  0.5858, 0.58,  0.5796, 0.577,  0.5770, 0.58,  0.5883, 0.59,  0.6190,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuK16_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5837, 0.45,  0.4117, 0.43,  0.4427, 0.47,  0.5098, 0.52,  0.5451, 0.55,  0.5698, 0.57,  0.5828, 0.59,  0.5939, 0.6,  0.6045, 0.607,  0.6140, 0.62,  0.6219, 0.63,  0.6330, 0.64,
    0.6419, 0.643,  0.6440, 0.642,  0.6617, 0.64,  0.6379, 0.631,  0.6309, 0.62,  0.6154, 0.6,  0.5911, 0.58,  0.5736, 0.57,  0.5612, 0.56,  0.5539, 0.55,  0.5462, 0.543,
    0.5406, 0.542,  0.5418, 0.543,  0.5652, 0.55,  0.5529, 0.56,  0.5654, 0.57,  0.5806, 0.584,  0.5888, 0.589,  0.5898, 0.586,  0.5858, 0.58,  0.5796, 0.577,  0.5770, 0.58,  0.5883, 0.59,  0.6190,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuM3_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2659, 0.255,  0.2526, 0.26,  0.2616, 0.27,  0.2854, 0.29,  0.3088, 0.31,  0.3231, 0.33,  0.3336, 0.34,  0.3421, 0.345,  0.347, 0.35,  0.3542, 0.36,  0.3647, 0.37,
    0.3854, 0.4,  0.4041, 0.402,  0.4012, 0.39,  0.3856, 0.38,  0.3769, 0.375,  0.3725, 0.36,  0.3525, 0.34,  0.3286, 0.325,  0.3247, 0.326,  0.3279, 0.328,  0.3285, 0.325,
    0.3240, 0.322,  0.3202, 0.321,  0.3220, 0.323,  0.3267, 0.33,  0.3342, 0.34,  0.3487, 0.35,  0.3667, 0.37,  0.3761, 0.375,  0.3746, 0.37,  0.3670, 0.36,  0.3559, 0.35,  0.3498, 0.35,  0.3630, 0.37,  0.3998,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuM3_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2659, 0.255,  0.2526, 0.26,  0.2616, 0.27,  0.2854, 0.29,  0.3088, 0.31,  0.3231, 0.33,  0.3336, 0.34,  0.3421, 0.345,  0.347, 0.35,  0.3542, 0.36,  0.3647, 0.37,
    0.3854, 0.4,  0.4041, 0.452,  0.4012, 0.39,  0.3856, 0.38,  0.3769, 0.375,  0.3725, 0.36,  0.3525, 0.34,  0.3286, 0.325,  0.3247, 0.326,  0.3279, 0.328,  0.3285, 0.325,
    0.3240, 0.322,  0.3202, 0.341,  0.3220, 0.323,  0.3267, 0.33,  0.3342, 0.34,  0.3487, 0.35,  0.3667, 0.37,  0.3761, 0.375,  0.3746, 0.37,  0.3670, 0.36,  0.3559, 0.35,  0.3498, 0.35,  0.3630, 0.37,  0.3998,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuN18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1284, 0.11,  0.1090, 0.13,  0.1573, 0.17,  0.1837, 0.19,  0.1971, 0.2,  0.2059, 0.21,  0.2143, 0.22,  0.2213, 0.225,  0.2271, 0.23,  0.2341, 0.24,  0.2487, 0.26,  0.2764, 0.29,
    0.3025, 0.303,  0.3052, 0.3,  0.2919, 0.29,  0.2843, 0.283,  0.2800, 0.27,  0.2612, 0.24,  0.2394, 0.235,  0.2339, 0.234,  0.2340, 0.233,  0.2326, 0.23,  0.2277, 0.225,
    0.2235, 0.224,  0.2246, 0.226,  0.2282, 0.23,  0.2349, 0.24,  0.2477, 0.25,  0.2632, 0.27,  0.2714, 0.271,  0.2702, 0.27,  0.2637, 0.26,  0.2538, 0.25,  0.2479, 0.25,  0.2589, 0.26,  0.2918,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuN18_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1284, 0.11,  0.1090, 0.13,  0.1573, 0.17,  0.1837, 0.19,  0.1971, 0.2,  0.2059, 0.21,  0.2143, 0.22,  0.2213, 0.225,  0.2271, 0.23,  0.2341, 0.24,  0.2487, 0.26,  0.2764, 0.29,
    0.3025, 0.303,  0.3052, 0.3,  0.2819, 0.29,  0.2843, 0.283,  0.2800, 0.27,  0.2612, 0.24,  0.2394, 0.235,  0.2339, 0.234,  0.2340, 0.233,  0.2326, 0.23,  0.2277, 0.225,
    0.2235, 0.224,  0.2246, 0.226,  0.2382, 0.23,  0.2349, 0.24,  0.2477, 0.25,  0.2632, 0.27,  0.2714, 0.271,  0.2702, 0.27,  0.2637, 0.26,  0.2538, 0.25,  0.2479, 0.25,  0.2589, 0.26,  0.2918,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuQ1_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0610, 0.06,  0.0592, 0.04,  0.0339, 0.04,  0.0338, 0.034,  0.0350, 0.036,  0.0363, 0.037,  0.0380, 0.038,  0.0383, 0.0383,  0.0385, 0.04,  0.0408, 0.042,  0.0451, 0.05,
    0.0524, 0.055,  0.0589, 0.058,  0.0585, 0.055,  0.0529, 0.05,  0.0456, 0.04,  0.0390, 0.035,  0.0330, 0.03,  0.0286, 0.028,  0.0275, 0.0276,  0.0275, 0.0278,  0.0279, 0.028,
    0.0289, 0.03,  0.0304, 0.031,  0.0320, 0.032,  0.0328, 0.033,  0.0341, 0.0345,  0.0346, 0.0346,  0.0347, 0.034,  0.0341, 0.034,  0.0336, 0.034,  0.0340, 0.035,
    0.0351, 0.036,  0.0373, 0.038,  0.0411, 0.042,  0.0446,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuQ1_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0610, 0.06,  0.0592, 0.04,  0.0359, 0.04,  0.0338, 0.034,  0.0350, 0.036,  0.0363, 0.037,  0.0380, 0.038,  0.0383, 0.0383,  0.0385, 0.04,  0.0408, 0.042,  0.0451, 0.05,
    0.0524, 0.055,  0.0589, 0.058,  0.0595, 0.055,  0.0529, 0.05,  0.0456, 0.04,  0.0390, 0.035,  0.0330, 0.03,  0.0286, 0.028,  0.0275, 0.0276,  0.0275, 0.0278,  0.0279, 0.028,
    0.0289, 0.03,  0.0304, 0.031,  0.0340, 0.032,  0.0328, 0.033,  0.0341, 0.0345,  0.0346, 0.0346,  0.0347, 0.034,  0.0341, 0.034,  0.0336, 0.034,  0.0340, 0.035,
    0.0351, 0.036,  0.0373, 0.038,  0.0411, 0.042,  0.0446,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuS7_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1447, 0.06,  0.0448, 0.042,  0.0411, 0.03,  0.0282, 0.028,  0.0270, 0.029,  0.0298, 0.03,  0.0319, 0.032,  0.0331, 0.0333,  0.0335, 0.036,  0.0361, 0.038,  0.0403, 0.045,
    0.0493, 0.05,  0.0599, 0.06,  0.0636, 0.062,  0.0606, 0.06,  0.0547, 0.05,  0.0488, 0.045,  0.0421, 0.04,  0.0366, 0.035,  0.0335, 0.033,  0.0323, 0.032,  0.0320, 0.032,
    0.0321, 0.0322,  0.0324, 0.033,  0.0333, 0.034,  0.0345, 0.035,  0.0356, 0.036,  0.0364, 0.037,  0.0372, 0.037,  0.0367, 0.0365,  0.0363, 0.0364,
    0.0361, 0.0362,  0.0363, 0.037,  0.0376, 0.04,  0.0412, 0.042,  0.0450,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuS7_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1447, 0.06,  0.0448, 0.042,  0.0511, 0.03,  0.0282, 0.028,  0.0270, 0.029,  0.0298, 0.03,  0.0319, 0.032,  0.0331, 0.0333,  0.0335, 0.036,  0.0361, 0.038,  0.0403, 0.045,
    0.0493, 0.05,  0.0599, 0.06,  0.0666, 0.062,  0.0606, 0.06,  0.0547, 0.05,  0.0488, 0.045,  0.0421, 0.04,  0.0366, 0.035,  0.0335, 0.033,  0.0323, 0.032,  0.0320, 0.032,
    0.0321, 0.0322,  0.0324, 0.033,  0.0333, 0.034,  0.0345, 0.035,  0.0356, 0.036,  0.0364, 0.037,  0.0372, 0.037,  0.0367, 0.0365,  0.0363, 0.0364,
    0.0361, 0.0362,  0.0363, 0.037,  0.0376, 0.04,  0.0412, 0.042,  0.0450,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::J570_NeuV10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1439, 0.07,  0.0583, 0.04,  0.0372, 0.037,  0.0362, 0.035,  0.0344, 0.034,  0.0340, 0.035,  0.0351, 0.036,  0.0361, 0.0361,  0.0361, 0.037,  0.0377, 0.039,  0.0404, 0.042,
    0.0450, 0.047,  0.0483, 0.048,  0.0475, 0.045,  0.0436, 0.04,  0.0387, 0.036,  0.0343, 0.03,  0.0299, 0.028,  0.0271, 0.027,  0.0262, 0.0262,  0.0262, 0.0267,  0.0269, 0.027,
    0.0283, 0.029,  0.0299, 0.03,  0.0308, 0.031,  0.0319, 0.032,  0.0331, 0.0333,  0.0337, 0.0337,  0.0337, 0.0333,  0.0332, 0.0332,  0.0331, 0.0331,  0.0331, 0.034,
    0.0341, 0.035,  0.0371, 0.038,  0.0399, 0.041,  0.0432,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuV10_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1439, 0.07,  0.0583, 0.04,  0.0392, 0.037,  0.0362, 0.035,  0.0344, 0.034,  0.0340, 0.035,  0.0351, 0.036,  0.0361, 0.0361,  0.0361, 0.037,  0.0377, 0.039,  0.0404, 0.042,
    0.0450, 0.047,  0.0483, 0.048,  0.0495, 0.045,  0.0436, 0.04,  0.0387, 0.036,  0.0343, 0.03,  0.0299, 0.028,  0.0271, 0.027,  0.0262, 0.0262,  0.0262, 0.0267,  0.0269, 0.027,
    0.0283, 0.029,  0.0299, 0.03,  0.0328, 0.031,  0.0319, 0.032,  0.0331, 0.0333,  0.0337, 0.0337,  0.0337, 0.0333,  0.0332, 0.0332,  0.0331, 0.0331,  0.0331, 0.034,
    0.0341, 0.035,  0.0371, 0.038,  0.0399, 0.041,  0.0432,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::J570_NeuW18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4685, 0.45,  0.4262, 0.5,  0.5061, 0.55,  0.5898, 0.6,  0.6487, 0.66,  0.6781, 0.68,  0.6947, 0.7,  0.7070, 0.71,  0.7185, 0.72,  0.7294, 0.73,  0.7383, 0.74,  0.7499, 0.75,
    0.7582, 0.758,  0.7582, 0.755,  0.7531, 0.75,  0.7484, 0.745,  0.7422, 0.73,  0.7263, 0.72,  0.7033, 0.7,  0.6913, 0.69,  0.6820, 0.68,  0.6738, 0.67,  0.6628, 0.66,
    0.6512, 0.65,  0.6462, 0.645,  0.6448, 0.645,  0.6485, 0.65,  0.6569, 0.66,  0.6698, 0.67,  0.6781, 0.68,  0.6822, 0.682,  0.6820, 0.682,  0.6815, 0.682,  0.6820, 0.69,  0.6907, 0.7,  0.7152,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuW18_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4685, 0.45,  0.4262, 0.5,  0.5061, 0.55,  0.5998, 0.6,  0.6487, 0.66,  0.6781, 0.68,  0.6947, 0.7,  0.7070, 0.71,  0.7185, 0.72,  0.7294, 0.73,  0.7383, 0.74,  0.7499, 0.75,
    0.7582, 0.758,  0.7582, 0.755,  0.7531, 0.75,  0.7584, 0.745,  0.7422, 0.73,  0.7263, 0.72,  0.7033, 0.7,  0.6913, 0.69,  0.6820, 0.68,  0.6738, 0.67,  0.6628, 0.66,
    0.6512, 0.65,  0.6462, 0.645,  0.6448, 0.645,  0.6485, 0.65,  0.6569, 0.66,  0.6698, 0.67,  0.6781, 0.68,  0.6822, 0.682,  0.6820, 0.682,  0.6815, 0.682,  0.6820, 0.69,  0.6907, 0.7,  0.7152,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuZ14_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2765, 0.2,  0.1352, 0.13,  0.1222, 0.121,  0.1206, 0.13,  0.1300, 0.134,  0.1357, 0.14,  0.1407, 0.142,  0.1455, 0.147,  0.1485, 0.15,  0.1539, 0.16,  0.1648, 0.17,
    0.1844, 0.2,  0.2015, 0.202,  0.2024, 0.2,  0.1922, 0.19,  0.1868, 0.185,  0.1841, 0.18,  0.1715, 0.16,  0.1566, 0.155,  0.1536, 0.154,  0.1545, 0.154,  0.1536, 0.151,
    0.1500, 0.148,  0.1471, 0.1472,  0.1478, 0.15,  0.1505, 0.153,  0.1552, 0.16,  0.1641, 0.17,  0.1751, 0.18,  0.1813, 0.181,  0.1801, 0.18,
    0.1757, 0.17,  0.1683, 0.165,  0.1642, 0.17,  0.1728, 0.18,  0.1970,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuZ14_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2765, 0.2,  0.1352, 0.13,  0.1222, 0.121,  0.1206, 0.13,  0.1300, 0.134,  0.1357, 0.14,  0.1407, 0.142,  0.1455, 0.147,  0.1485, 0.15,  0.1539, 0.16,  0.1648, 0.17,
    0.1844, 0.2,  0.2015, 0.202,  0.2024, 0.2,  0.2022, 0.19,  0.1868, 0.185,  0.1841, 0.18,  0.1715, 0.16,  0.1566, 0.155,  0.1536, 0.154,  0.1545, 0.154,  0.1536, 0.151,
    0.1500, 0.148,  0.1471, 0.1472,  0.1478, 0.15,  0.1605, 0.153,  0.1552, 0.16,  0.1641, 0.17,  0.1751, 0.18,  0.1813, 0.181,  0.1801, 0.18,
    0.1757, 0.17,  0.1683, 0.165,  0.1642, 0.17,  0.1728, 0.18,  0.1970,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuC18_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0555, 0.055,  0.0545, 0.055,  0.0585, 0.058,  0.0577, 0.056,  0.0554, 0.056,  0.0564, 0.058,  0.0590, 0.06,   0.0611, 0.062,  0.0638, 0.065,  0.0685, 0.07,   0.0797, 0.09,
    0.1009, 0.11,   0.1222, 0.124,  0.1298, 0.127,  0.1257, 0.123,  0.1208, 0.12,   0.1164, 0.11,   0.1067, 0.1,    0.0954, 0.09,   0.0895, 0.088,  0.0862, 0.085,  0.0834, 0.082,
    0.0806, 0.08,   0.0782, 0.078,  0.0780, 0.078,  0.0789, 0.08,   0.0813, 0.084,  0.0858, 0.09,   0.0911, 0.092,  0.0944, 0.093,  0.0938, 0.092,  0.0914, 0.09,   0.0878, 0.086,
    0.0858, 0.09,   0.0903, 0.1,    0.1037,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuC18_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0555, 0.055,  0.0545, 0.055,  0.0585, 0.058,  0.0577, 0.056,  0.0554, 0.056,  0.0564, 0.058,  0.0590, 0.06,   0.0611, 0.062,  0.0638, 0.065,  0.0685, 0.07,   0.0797, 0.09,
    0.1009, 0.11,   0.1222, 0.124,  0.1398, 0.127,  0.1257, 0.123,  0.1208, 0.12,   0.1164, 0.11,   0.1067, 0.1,    0.0954, 0.09,   0.0895, 0.088,  0.0862, 0.085,  0.0834, 0.082,
    0.0806, 0.08,   0.0782, 0.078,  0.0880, 0.078,  0.0789, 0.08,   0.0813, 0.084,  0.0858, 0.09,   0.0911, 0.092,  0.0944, 0.093,  0.0938, 0.092,  0.0914, 0.09,   0.0878, 0.086,
    0.0858, 0.09,   0.0903, 0.1,    0.1037,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuD17_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1980, 0.1,    0.0793, 0.06,   0.0578, 0.05,   0.0476, 0.046,  0.0454, 0.046,  0.0471, 0.048,  0.0499, 0.05,   0.0518, 0.052,  0.0533, 0.055,  0.0574, 0.06,   0.0676, 0.07,
    0.0897, 0.1,    0.1129, 0.113,  0.1140, 0.1,    0.0958, 0.08,   0.0743, 0.06,   0.0566, 0.05,   0.0422, 0.04, 0.0332, 0.03,   0.0297, 0.0295, 0.0292, 0.0293, 0.0294, 0.03,
    0.0306, 0.031,  0.0319, 0.032,  0.0339, 0.034,  0.0353, 0.036,  0.0363, 0.037,  0.0370, 0.037,  0.0372, 0.037,  0.0368, 0.0365, 0.0363, 0.036,
    0.0360, 0.037,  0.0376, 0.039,  0.0406, 0.042,  0.0448, 0.046,  0.0499,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuD17_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1980, 0.1,    0.0793, 0.06,   0.0578, 0.05,   0.0476, 0.046,  0.0454, 0.046,  0.0471, 0.048,  0.0499, 0.05,   0.0518, 0.052,  0.0533, 0.055,  0.0574, 0.06,   0.0676, 0.07,
    0.0897, 0.1,    0.1129, 0.113,  0.1240, 0.1,    0.0958, 0.08,   0.0743, 0.06,   0.0566, 0.05,   0.0422, 0.04, 0.0332, 0.03,   0.0297, 0.0295, 0.0292, 0.0293, 0.0294, 0.03,
    0.0306, 0.031,  0.0319, 0.032,  0.0439, 0.034,  0.0353, 0.036,  0.0363, 0.037,  0.0370, 0.037,  0.0372, 0.037,  0.0368, 0.0365, 0.0363, 0.036,
    0.0360, 0.037,  0.0376, 0.039,  0.0406, 0.042,  0.0448, 0.046,  0.0499,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::J570_NeuJ11_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1274, 0.1,    0.0916, 0.08,   0.0656, 0.061,  0.0604, 0.06,   0.0570, 0.06,   0.0604, 0.062,  0.0644, 0.065,  0.0668, 0.069,  0.0700, 0.072,  0.0754, 0.08,   0.0874, 0.1,
    0.1111, 0.12,   0.1327, 0.132,  0.1313, 0.12,   0.1127, 0.1,    0.0931, 0.08,   0.0758, 0.06,   0.0580, 0.05,   0.0449, 0.04,   0.0385, 0.037,  0.0360, 0.036,  0.0351, 0.035,
    0.0351, 0.0354, 0.0355, 0.036,  0.0371, 0.0375, 0.0379, 0.038,  0.0388, 0.04,   0.0406, 0.041,  0.0414, 0.0415, 0.0416, 0.041,  0.0409, 0.04,   0.0398, 0.0397, 0.0397, 0.04,
    0.0424, 0.043,  0.0458, 0.048,  0.0522,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::J570_NeuJ11_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1274, 0.1,    0.0916, 0.08,   0.0756, 0.061,  0.0604, 0.06,   0.0570, 0.06,   0.0604, 0.062,  0.0644, 0.065,  0.0668, 0.069,  0.0700, 0.072,  0.0754, 0.08,   0.0874, 0.1,
    0.1111, 0.12,   0.1327, 0.132,  0.1413, 0.12,   0.1127, 0.1,    0.0931, 0.08,   0.0758, 0.06,   0.0580, 0.05,   0.0449, 0.04,   0.0385, 0.037,  0.0360, 0.036,  0.0351, 0.035,
    0.0351, 0.0354, 0.0355, 0.036,  0.0371, 0.0375, 0.0379, 0.038,  0.0388, 0.04,   0.0406, 0.041,  0.0414, 0.0415, 0.0416, 0.041,  0.0409, 0.04,   0.0398, 0.0397, 0.0397, 0.04,
    0.0424, 0.043,  0.0458, 0.048,  0.0522,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::J570_NeuL4_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0348, 0.05,   0.0700, 0.09,   0.1043, 0.11,   0.1320, 0.14,   0.1505, 0.16,   0.1622, 0.17,   0.1721, 0.18,   0.1805, 0.185,  0.1877, 0.19,   0.1955, 0.2,    0.2068, 0.21,
    0.2226, 0.23,   0.2350, 0.235,  0.2352, 0.23,   0.2251, 0.22,   0.2128, 0.2,    0.1990, 0.18,   0.1761, 0.16,   0.1494, 0.13,   0.1296, 0.12,   0.1171, 0.11,   0.1089, 0.105,
    0.1010, 0.1,    0.0949, 0.093,  0.0926, 0.093,  0.0937, 0.095,  0.0961, 0.1,    0.1020, 0.11,   0.1104, 0.112,  0.1150, 0.115,  0.1155, 0.113,  0.1123, 0.11,
    0.1070, 0.105,  0.1040, 0.11,   0.1110, 0.12,   0.1323,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::J570_NeuL4_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0348, 0.05,   0.0700, 0.09,   0.1043, 0.11,   0.1320, 0.14,   0.1505, 0.16,   0.1622, 0.17,   0.1721, 0.18,   0.1805, 0.185,  0.1877, 0.19,   0.1955, 0.2,    0.2068, 0.21,
    0.2226, 0.23,   0.2350, 0.235,  0.2452, 0.23,   0.2251, 0.22,   0.2128, 0.2,    0.1990, 0.18,   0.1761, 0.16,   0.1494, 0.13,   0.1296, 0.12,   0.1171, 0.11,   0.1089, 0.105,
    0.1010, 0.1,    0.0949, 0.093,  0.0826, 0.093,  0.0937, 0.095,  0.0961, 0.1,    0.1020, 0.11,   0.1104, 0.112,  0.1150, 0.115,  0.1155, 0.113,  0.1123, 0.11,
    0.1070, 0.105,  0.1040, 0.11,   0.1110, 0.12,   0.1323,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_n72_n2_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0116, 0.01,  0.0171, 0.05,   0.0625, 0.08,   0.1486, 0.16,   0.1963, 0.2,    0.2409, 0.26,   0.2974, 0.31,   0.3468, 0.36,   0.3790, 0.39,   0.4075, 0.41,   0.4216, 0.43,
    0.4399,  0.47,  0.4878, 0.50,   0.5589, 0.57,   0.5882, 0.57,   0.5566, 0.52,   0.5030, 0.46,   0.4451, 0.42,   0.3928, 0.37,   0.3625, 0.35,   0.3396, 0.29,   0.2670, 0.15,
    0.1028, 0.05,   -0.0397, -0.08, -0.1151, -0.12, -0.1464, -0.15, -0.1582, -0.16, -0.1609, -0.16, -0.1581, -0.155,    -0.1556, -0.156,
    -0.1582, -0.16, -0.1621, -0.165,    -0.1683, -0.17, -0.1719, -0.17, -0.1696, -0.165,    -0.1623,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_10_n70_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0964, 0.1,    0.1534, 0.2,    0.2437, 0.33,   0.4663, 0.5,    0.6005, 0.65,   0.6958, 0.75,   0.8010, 0.83,   0.8598, 0.858,  0.8579, 0.85,   0.8432, 0.83,   0.8102, 0.79,
    0.7607, 0.7,    0.6760, 0.6,    0.5530, 0.5,    0.4212, 0.3,    0.2974, 0.2,    0.1839, 0.1,    0.0743, 0.03,   -0.0208, -0.05, -0.0747, -0.08, -0.0913, -0.05, -0.0458, 0.03,
    0.0806, 0.1,    0.1936, 0.2,    0.2556, 0.27,   0.2816, 0.29,   0.2925, 0.3,    0.3033, 0.31,   0.3175, 0.32,   0.3257, 0.325,  0.3246, 0.32,
    0.3187, 0.31,   0.3082, 0.305,  0.3014, 0.304,  0.3059, 0.31,   0.3253,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_10_n70_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0964, 0.1,    0.1534, 0.2,    0.2437, 0.33,   0.4663, 0.5,    0.6005, 0.65,   0.6958, 0.75,   0.8010, 0.83,   0.8598, 0.858,  0.8579, 0.85,   0.8432, 0.83,   0.8102, 0.79,
    0.7607, 0.7,    0.6760, 0.65,    0.5530, 0.55,    0.4212, 0.3,    0.2974, 0.25,    0.1839, 0.12,    0.0743, 0.03,   -0.0208, -0.05, -0.0747, -0.08, -0.0913, -0.05, -0.0458, 0.03,
    0.0806, 0.1,    0.1936, 0.2,    0.2556, 0.27,   0.2816, 0.29,   0.2925, 0.3,    0.3033, 0.31,   0.3175, 0.32,   0.3257, 0.325,  0.3246, 0.32,
    0.3187, 0.31,   0.3082, 0.305,  0.3014, 0.304,  0.3059, 0.34,   0.3353,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_10_n70_spect3[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0964, 0.1,    0.1534, 0.2,    0.2437, 0.33,   0.4663, 0.5,    0.6005, 0.65,   0.6958, 0.75,   0.8010, 0.83,   0.8598, 0.858,  0.8579, 0.85,   0.8432, 0.83,   0.8102, 0.79,
    0.7607, 0.7,    0.6760, 0.65,    0.5530, 0.55,    0.4212, 0.3,    0.2974, 0.25,    0.1839, 0.12,    0.0743, 0.03,   -0.0208, -0.05, -0.0747, -0.08, -0.0913, -0.05, -0.0458, 0.03,
    0.0806, 0.1,    0.1936, 0.22,    0.2556, 0.27,   0.2816, 0.29,   0.2925, 0.33,    0.3033, 0.33,   0.3175, 0.32,   0.3257, 0.325,  0.3246, 0.32,
    0.3187, 0.32,   0.3082, 0.305,  0.3014, 0.304,  0.3059, 0.36,   0.3353,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_10_n70_spect4[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0964, 0.1,    0.1534, 0.2,    0.2437, 0.34,   0.4663, 0.55,    0.6005, 0.65,   0.6958, 0.75,   0.8010, 0.83,   0.8598, 0.858,  0.8579, 0.86,   0.8432, 0.82,   0.8102, 0.79,
    0.7607, 0.7,    0.6760, 0.6,    0.5530, 0.5,    0.4212, 0.3,    0.2974, 0.20,    0.1839, 0.12,    0.0743, 0.03,   -0.0208, -0.05, -0.0747, -0.08, -0.0913, -0.05, -0.0458, 0.03,
    0.0806, 0.1,    0.1936, 0.22,    0.2556, 0.27,   0.2816, 0.29,   0.2925, 0.33,    0.3033, 0.33,   0.3175, 0.32,   0.3257, 0.325,  0.3246, 0.32,
    0.3187, 0.32,   0.3082, 0.305,  0.3014, 0.304,  0.3059, 0.36,   0.3353,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_n33_n70_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0615, 0.1,    0.1219, 0.2,    0.2179, 0.3,    0.4397, 0.5,    0.5722, 0.6,    0.6714, 0.7,    0.7834, 0.8,    0.8535, 0.86,   0.8647, 0.864,  0.8642, 0.864,  0.8429, 0.82,
    0.8035, 0.75,   0.7316, 0.7,    0.6238, 0.55,   0.4996, 0.45,   0.3717, 0.3,    0.2487, 0.2,    0.1299, 0.06,   0.0272, 0.01,   -0.0315, -0.04, -0.0557, -0.051,
    -0.0519, -0.03, -0.0234, 0.001, 0.0041, 0.01,   0.0201, 0.021,  0.0269, 0.028,  0.0298, 0.03,   0.0371, 0.04,   0.0497, 0.05,   0.0578, 0.056,
    0.0557, 0.05,   0.0490, 0.04,   0.0374, 0.03,   0.0299, 0.03,   0.0348, 0.04,   0.0537,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_n8_n74_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0603, 0.08,   0.1069, 0.15,   0.1810, 0.25,   0.3573, 0.4,    0.4634, 0.5,    0.5406, 0.6,    0.6266, 0.65,   0.6772, 0.679,  0.6800, 0.68,   0.6732, 0.66,   0.6511, 0.63,
    0.6136, 0.6,    0.5444, 0.5,    0.4415, 0.4,    0.3317, 0.3,    0.2286, 0.2,    0.1336, 0.1,    0.0425, 0.01,   -0.0360, -0.05, -0.0807, -0.09, -0.0967, -0.08,
    -0.0761, -0.05, -0.0119, 0.02,  0.0462, 0.05,   0.0784, 0.08,   0.0919, 0.095,  0.0974, 0.1,    0.1046, 0.11,   0.1152, 0.12,   0.1217, 0.121,
    0.1205, 0.12,   0.1155, 0.11,   0.1067, 0.104,  0.1010, 0.103,  0.1047, 0.11,   0.1199,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_19_n69_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1281, 0.16,   0.1941, 0.24,   0.2982, 0.4,    0.5607, 0.65,   0.7192, 0.8,    0.8299, 0.9,    0.9517, 1.0,    1.0174, 1.014,  1.0115, 1.0,    0.9899, 0.96,   0.9475, 0.91,
    0.8877, 0.82,   0.7902, 0.7,    0.6508, 0.57,   0.5008, 0.45,   0.3595, 0.3,    0.2302, 0.15,   0.1049, 0.02,   -0.0045, -0.03, -0.0662, -0.07, -0.0832, -0.05, -0.0161, 0.05,
    0.1648, 0.2,    0.3257, 0.37,   0.4137, 0.43,   0.4506, 0.46,   0.4661, 0.47,   0.4801, 0.49,   0.4974, 0.5,    0.5072, 0.505,  0.5062, 0.5,    0.4995, 0.49,   0.4876, 0.48,
    0.4799, 0.48,   0.4849, 0.49,   0.5081,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_n80_10_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0247, -0.01, -0.0057, 0.01,  0.0242, 0.05,   0.0736, 0.08,   0.0996, 0.1,    0.1281, 0.14,   0.1660, 0.19,   0.2037, 0.21,   0.2337, 0.25,   0.2618, 0.27,   0.2793, 0.29,
    0.3024, 0.32,   0.3564, 0.4,    0.4360, 0.44,   0.4788, 0.47,   0.4655, 0.45,   0.4308, 0.42,   0.3920, 0.37,   0.3564, 0.34,   0.3357, 0.32,   0.3167, 0.28,   0.2434, 0.1,
    0.0745, 0.01,   -0.0725, -0.09, -0.1506, -0.18, -0.1831, -0.19, -0.1955, -0.195,    -0.1995, -0.199,    -0.1989, -0.198,
    -0.1976, -0.198,    -0.1999, -0.2,  -0.2027, -0.208,    -0.2070, -0.208,    -0.2094, -0.208,    -0.2080, -0.207,    -0.2038,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_n80_26_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0325, -0.03, -0.0203, -0.01, -0.0012, 0.01,  0.0228, 0.03,   0.0332, 0.04,   0.0506, 0.06,   0.0762, 0.09,   0.1069, 0.12,   0.1370, 0.15,   0.1666, 0.17,   0.1877, 0.2,
    0.2174, 0.25,   0.2849, 0.33,   0.3849, 0.4,    0.4477, 0.45,   0.4509, 0.44,   0.4305, 0.42,   0.4049, 0.39,   0.3806, 0.37,   0.3664, 0.35,   0.3495, 0.29,   0.2724, 0.1,
    0.0921, 0.01,   -0.0651, -0.09, -0.1489, -0.15, -0.1837, -0.19, -0.1970, -0.2,  -0.2020, -0.202,    -0.2029, -0.203,    -0.2026, -0.204,
    -0.2047, -0.205,    -0.2069, -0.208,    -0.2099, -0.21, -0.2115, -0.21, -0.2106, -0.209,    -0.2086,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

/*
const double ColorTemp::JDC468_greyc14_66_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1767, 0.19, 0.2207, 0.25, 0.3142, 0.40, 0.5269, 0.63, 0.7018, 0.73, 0.7605, 0.76, 0.7580, 0.74, 0.7366, 0.725, 0.7182, 0.705, 0.6929, 0.68, 0.6661, 0.66, 0.6542, 0.65,
    0.642, 0.62, 0.6085, 0.585, 0.5752, 0.573, 0.5728, 0.573, 0.5723, 0.555, 0.5318, 0.51, 0.4982, 0.51, 0.5226, 0.54, 0.5670, 0.58, 0.5929, 0.594, 0.5977, 0.597, 0.5975, 0.6,
    0.6002, 0.602, 0.6065, 0.61, 0.6177, 0.62, 0.6352, 0.64, 0.6526, 0.66, 0.6623, 0.662, 0.6633, 0.66, 0.6593, 0.653, 0.6517, 0.65, 0.6479, 0.65, 0.6607, 0.69, 0.6908,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_greym13_325_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1325, 0.15, 0.1637, 0.2, 0.2222, 0.28, 0.3492, 0.40, 0.4523, 0.47, 0.4897, 0.49, 0.4918, 0.49, 0.4840, 0.48, 0.4761, 0.47, 0.4638, 0.46, 0.4538, 0.455, 0.4582, 0.458,
    0.4588, 0.45, 0.4360, 0.42, 0.4091, 0.41, 0.4101, 0.411, 0.4128, 0.405, 0.3785, 0.36, 0.3494, 0.36, 0.3720, 0.41, 0.4122, 0.425, 0.4339, 0.435, 0.4362, 0.597, 0.4355, 0.437,
    0.4395, 0.44, 0.4475, 0.45, 0.4606, 0.47, 0.4807, 0.49, 0.5006, 0.51, 0.5125, 0.513, 0.5145, 0.512, 0.5112, 0.51, 0.5029, 0.5, 0.4992, 0.51, 0.5150, 0.53, 0.5526,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::JDC468_greyf26_156_spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0823, 0.1, 0.1036, 0.11, 0.1337, 0.16, 0.1966, 0.22, 0.2468, 0.255, 0.2679, 0.27, 0.2728, 0.273, 0.2726, 0.273, 0.2724, 0.271, 0.2698, 0.27, 0.2705, 0.275, 0.2810, 0.285,
    0.2879, 0.28, 0.2756, 0.26, 0.2586, 0.26, 0.2601, 0.261, 0.2617, 0.25, 0.2357, 0.22, 0.2124, 0.22, 0.2241, 0.23, 0.2471, 0.25, 0.2581, 0.278, 0.2569, 0.255, 0.2548, 0.255,
    0.2579, 0.26, 0.2653, 0.27, 0.2765, 0.28, 0.2941, 0.30, 0.3126, 0.32, 0.3230, 0.323, 0.3238, 0.32, 0.3189, 0.31, 0.3091, 0.302, 0.3043, 0.31, 0.3200, 0.34, 0.3579,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
*/
const double ColorTemp::Colorlab_n80_5_9_5_9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0912, 0.1, 0.1228, 0.15, 0.1712, 0.2, 0.2978, 0.32, 0.3713, 0.41, 0.4241, 0.44, 0.4861, 0.51, 0.5255, 0.53, 0.5355, 0.534, 0.5363, 0.53, 0.5237, 0.524, 0.5251, 0.56,
    0.5722, 0.6, 0.6554, 0.67, 0.6936, 0.67, 0.6675, 0.65, 0.6203, 0.6, 0.5651, 0.54, 0.5116, 0.5, 0.4825, 0.48, 0.4714, 0.48, 0.4866, 0.5, 0.5320, 0.55, 0.5729, 0.58,
    0.5968, 0.6, 0.6069, 0.61, 0.6131, 0.615, 0.6198, 0.62, 0.6285, 0.63, 0.6325, 0.632, 0.6316, 0.63, 0.6282, 0.625, 0.6227, 0.62, 0.6196, 0.62, 0.6215, 0.63, 0.6337,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
/*
const double ColorTemp::Colorlab_n57_5_6_9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0385, 0.04, 0.0514, 0.06, 0.0711, 0.1, 0.1229, 0.14, 0.1528, 0.16, 0.1744, 0.18, 0.1999, 0.2, 0.2163, 0.22, 0.2209, 0.221, 0.2216, 0.22, 0.2167, 0.216, 0.2185, 0.23,
    0.2414, 0.26, 0.2813, 0.3, 0.3012, 0.3, 0.2922, 0.28, 0.2734, 0.26, 0.2511, 0.24, 0.2292, 0.22, 0.2173, 0.215, 0.2127, 0.215, 0.2183, 0.22, 0.2354, 0.24, 0.2508, 0.255,
    0.2599, 0.26, 0.2637, 0.263, 0.2662, 0.267, 0.2689, 0.27, 0.2725, 0.273, 0.2742, 0.274, 0.2738, 0.273, 0.2724, 0.271, 0.2701, 0.27, 0.2689, 0.269, 0.2697, 0.27, 0.2747,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
*/
const double ColorTemp::Colorlab_L61_110_110Rec2020spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2118, 0.2100, 0.1920, 0.1800, 0.1574, 0.1575, 0.1578, 0.1600, 0.1631, 0.1500, 0.1392, 0.1200, 0.1018, 0.650, 0.0430, 0.1000, -0.0212, 0.0543, -0.0929, -0.1111, -0.1519, -0.1600, -0.1876, -0.1923, -0.1950, -0.1850, -0.1775, -0.1600, -0.1448, -0.1000, -0.0914, -0.0100, -0.0271, 0.0120, 0.0300, 0.0550, 0.0732, 0.0900, 0.1017, 0.1234, 0.1425,//41
    0.1468, 0.3619, 0.5434, 0.8773, 1.1111, 1.3269, 1.4355, 1.5694, 1.6100, 1.6707, 1.7100, 1.7137, 1.7235, 1.7332, 1.7410, 1.7420, 1.7420, 1.7422, 1.7450, 1.7475, 1.7510, 1.7519, 1.7550, 1.7584, 1.7610, 1.7627, 1.7600, 1.7601, 1.7620, 1.7630,//30
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L63_120_m56Rec2020spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3223, 0.3500, 0.3687, 0.4000, 0.4394, 0.5100, 0.7018, 0.7500, 0.8667, 0.9000, 0.9539, 1.000, 1.0370, 1.040, 1.0410, 0.9800, 0.9653, 0.9200, 0.8675, 0.8100, 0.7631, 0.7100,
    0.6533, 0.5500, 0.5024, 0.4000, 0.3065, 0.2100, 0.1347, 0.1000, 0.0257, 0.0100, -0.0474, -0.0600, -0.1188, -0.1500, -0.1851, -0.2000, -0.2188, -0.2000, -0.1941, -0.0100,
    0.0977, 0.5000, 0.8071, 1.100, 1.4292, 1.5500, 1.7653, 1.8500, 1.9058, 1.9300, 1.9644, 1.9800, 1.9977, 2.010, 2.0234, 2.0300, 2.0337, 2.0250, 2.0381, 2.0360, 2.0358, 2.3010, 2.0302, 2.0280, 2.0265, 2.0227, 2.0292, 2.0290, 2.0547,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L63_m50_m60Rec2020spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0605, 0.1000, 0.1336, 0.2000, 0.2496, 0.4500, 0.5113, 0.6000, 0.6662, 0.7500, 0.7853, 0.9000, 0.9221, 0.9650, 1.0126, 1.0200, 1.0357, 1.0400, 1.0454, 1.0300, 1.0283, 1.0000,
    0.9940, 0.9500, 0.9371, 0.8800, 0.8515, 0.7900, 0.7337, 0.6500, 0.5897, 0.5300, 0.4428, 0.3200, 0.2986, 0.2100, 0.1731, 0.1300, 0.1012, 0.0800, 0.0678, 0.0500, 0.0461, 0.0300, 0.0176, 0.0100,
    -0.0042, -0.0100, -0.0143, -0.0150, -0.0183, -0.0190, -0.0194, -0.0150, -0.0127, -0.0100, 0.0014, 0.0050, 0.0108, 0.0090, 0.0076, 0.0001, -0.0007, -0.0050, -0.0150, -0.0200, -0.0240, -0.0200, -0.0182, 0.0010, 0.0042,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L63_m120_80Rec2020spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0708, -0.0700, -0.0651, -0.0600, -0.0562, -0.0600, 0.0685, -0.0700, -0.0813, -0.0750, -0.0762, -0.0700, -0.0617, -0.0400, -0.0305, 0.0100, 0.0120, 0.0300, 0.0568, 0.0800, 0.0933, 0.1000, 0.1433, 0.2000, 0.2482, 0.31000, 0.4019, 0.4500, 0.5081, 0.5200, 0.5339, 0.5300, 0.5266, 0.5200, 0.5121, 0.5000, 0.4971, 0.4900,
    0.4877, 0.4700, 0.4672, 0.4000, 0.3479, 0.2000, 0.0648, -0.0100, -0.1825, -0.0250, -0.3147, -0.3500, -0.3697, -0.3800, -0.3911, -0.4000, -0.4007, -0.4020, -0.4050,
    -0.4050, -0.4063, -0.4080, -0.4090, -0.4100, -0.4109, -0.4120, -0.4130, -0.4135, -0.4140, -0.4138, -0.4135, -0.4140, -0.4149,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L42_110_m100Prospect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2174, 0.2500, 0.2847, 0.3300, 0.3905, 0.5500, 0.6912, 0.7500, 0.8768, 0.9100, 0.9938, 1.0500, 1.1155, 1.1300, 1.1613, 1.1300, 1.1194, 1.0800, 1.0576, 1.0100, 0.9793, 0.9200,
    0.8769, 0.7700, 0.7035, 0.6000, 0.4573, 0.3400, 0.2303, 0.1100, 0.0648, 0.0100, -0.0656, -0.1100, -0.1884, -0.2300, -0.2951, -0.3200, -0.3541, -0.3542, -0.3546, -0.2300,
    -0.1748, 0.1000, 0.2751, 0.4100, 0.6713, 0.7500, 0.8857, 0.9200, 0.9753, 1.000, 1.0120, 1.0200, 1.0363, 1.0400, 1.0595, 1.0600, 1.0712, 1.0720, 1.0726, 1.0700, 1.0669, 1.0600, 1.0558, 1.0500, 1.0484, 1.0501, 1.0534, 1.0700, 1.0805,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L42_m70_m100Prospect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0736, 0.1000, 0.1529, 0.2200, 0.2791, 0.4200, 0.5702, 0.6600, 0.7447, 0.8200, 0.8749, 0.9500, 1.0210, 1.0600, 1.1109, 1.1200, 1.1222, 1.1200, 1.1187, 1.0900, 1.0887, 1.0500, 1.0294, 0.9500, 0.9114, 0.8600, 0.7320, 0.6600, 0.5414,
    0.4500, 0.3630, 0.2500, 0.1980, 0.1100, 0.0409, 0.0100, -0.0935, -0.01200, -0.1707, -0.1900, -0.2017, -0.2000, -0.1920, -0.1700, -0.1434, -0.1200, -0.0975, -0.0800, -0.0714,
    -0.7000, -0.0605, -0.0600, -0.0562, -0.0500, -0.0466, -0.0350, -0.0302, -0.0200, -0.0194, -0.0200, -0.0220, -0.0250, -0.0308, -0.0350, -0.0460, -0.0500, -0.0559, -0.0500, -0.0495, -0.0300, -0.0248,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L56_m120_90Prospect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0592, -0.0560, -0.0555, -0.0550, -0.0497, -0.0600, -0.0630, -0.0700, -0.0753, -0.0740, -0.0727, -0.0700, -0.0627, -0.0500, -0.0387, -0.0040, -0.0049, 0.0100, 0.0311, 0.0500,
    0.0607, 0.0800, 0.1009, 0.1500, 0.1840, 0.2500, 0.3055, 0.3400, 0.3899, 0.4000, 0.4115, 0.40800, 0.4069, 0.4000, 0.3968, 0.3900, 0.3862, 0.3800, 0.3795, 0.3700, 0.3633, 0.3200, 0.2672, 0.1300, 0.0388,
    0.0100, -0.1609, -0.2000, -0.2676, -0.3000, -0.3121, -0.3200, -0.3294, -0.3300, -0.3373, -0.3400, -0.3410, -0.3420, -0.3421, -0.3430, -0.3443, -0.3450, -0.3456, -0.3460, -0.3472, -0.3475, -0.3479, -0.3476, -0.3476, -0.3480, -0.3490,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L25_60_m120Prospect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1084, 0.1500, 0.1725, 0.2200, 0.2744, 0.4000, 0.5281, 0.6000, 0.6825, 0.7500, 0.7901, 0.8500, 0.9069, 0.9300, 0.9679, 0.9600, 0.9574, 0.9400, 0.9326, 0.9100, 0.8890, 0.8500, 0.8185, 0.7500,
    0.6837, 0.5400, 0.4837, 0.3500, 0.2905, 0.2000, 0.1352, 0.1000, 0.0028, -0.050, -0.1213, -0.2000, -0.2273, -0.2500, -0.2875, -0.2950, -0.3033, -0.2600, -0.2338, -0.1000, -0.0476, 0.0500,
    0.1181, 0.1500, 0.2081, 0.2200, 0.2458, 0.2550, 0.2608, 0.2650, 0.2741, 0.2800, 0.2906, 0.2950, 0.3003, 0.3000, 0.2994, 0.2950, 0.2930, 0.2850, 0.2813, 0.2800, 0.2735, 0.2750, 0.2786, 0.2900, 0.3006,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L75_50_120Prospect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1451, 0.1350, 0.1304, 0.1200, 0.1040, 0.1000, 0.0943, 0.0900, 0.0894, 0.8500, 0.0702, 0.0500, 0.0450, 0.0200, 0.0120, 0.0000, -0.0180, -0.0200, -0.0531, -0.0600, -0.0820, -0.0800, -0.0803, -0.0400, -0.0149, 0.0200, 0.1067, 0.1500,
    0.2101, 0.2500, 0.2775, 0.3000, 0.3323, 0.3500, 0.3756, 0.3900, 0.4057, 0.4100, 0.4259, 0.4300, 0.4494, 0.5000, 0.5688, 0.7500, 0.8473, 0.9500, 1.0899, 1.1500, 1.2215, 1.2500,
    1.2766, 1.2900, 1.3014, 1.305, 1.3127, 1.3150, 1.3176, 1.3170, 1.3169, 1.3180, 1.3199, 1.3200, 1.3227, 1.3250, 1.3272, 1.3300, 1.3306, 1.3300, 1.3285, 1.3295, 1.3301,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L75_m120_0Prospect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0548, -0.0200, -0.0126, 0.0100, 0.0544, 0.1100, 0.1678, 0.1900, 0.2289, 0.2500, 0.2929, 0.3300, 0.3761, 0.4100, 0.4556, 0.4800, 0.5149, 0.5400, 0.5703, 0.5900, 0.6035, 0.6300, 0.6419, 0.6800, 0.7270, 0.8000, 0.8497, 0.8700,
    0.9053, 0.8800, 0.8613, 0.8200, 0.7798, 0.7300, 0.6924, 0.6600, 0.6138, 0.5900, 0.5679, 0.5500, 0.5282, 0.4500, 0.3813, 0.2000, 0.0440, 0.0100, -0.2491, -0.3000, -0.4050, -0.4300, -0.4699, -0.4800, -0.4951, -0.5000,
    -0.5029, -0.5020, -0.5010, -0.5000, -0.4978, -0.5000, -0.5025, -0.5050, -0.5087, -0.5100, -0.5181, -0.5200, -0.5237, -0.5220, -0.5203, -0.5200, -0.5111,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L22_2_1_3Prospect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0074, 0.008, 0.0095, 0.01, 0.0127, 0.02, 0.0215, 0.025, 0.0267, 0.03, 0.0302, 0.033, 0.0343, 0.035, 0.0366, 0.036, 0.0367, 0.037, 0.0362, 0.035, 0.0349, 0.035, 0.0344, 0.036, 0.0367, 0.040, 0.0411, 0.042, 0.0428, 0.041,
    0.0409, 0.038, 0.0379, 0.035, 0.0343, 0.032, 0.0309, 0.030, 0.0290, 0.029, 0.0286, 0.03, 0.0312, 0.035, 0.0381, 0.040, 0.0443, 0.045, 0.0477, 0.048, 0.0491, 0.0490, 0.0498, 0.050,
    0.0504, 0.051, 0.0511, 0.0512, 0.0514, 0.0514, 0.0514, 0.0512, 0.0511, 0.0510, 0.0508, 0.0507, 0.0506, 0.0506, 0.0507, 0.0510, 0.0516,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp:: Colorlab_L44_2_8_3_9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0277, 0.030, 0.0351, 0.040, 0.0464, 0.060, 0.0782, 0.085, 0.0969, 0.010, 0.1096, 0.11, 0.1240, 0.130, 0.1321, 0.132, 0.1325, 0.131, 0.1305, 0.130, 0.1255, 0.125,
    0.1240, 0.130, 0.1328, 0.140, 0.1497, 0.150, 0.1569, 0.153, 0.1505, 0.140, 0.1399, 0.130, 0.1274, 0.120, 0.1152, 0.110, 0.1086, 0.108, 0.1070, 0.117, 0.1172, 0.130,
    0.1438, 0.150, 0.1672, 0.170, 0.1803, 0.181, 0.1858, 0.186, 0.1885, 0.190, 0.1907, 0.192, 0.1932, 0.194, 0.1942, 0.1942, 0.1941, 0.193, 0.1934, 0.193, 0.1922, 0.192, 0.1915, 0.191, 0.1919, 0.194, 0.1950,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L44_2_8_3_9spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0277, 0.030, 0.0351, 0.040, 0.0564, 0.060, 0.0782, 0.085, 0.0969, 0.010, 0.1096, 0.11, 0.1240, 0.130, 0.1321, 0.132, 0.1325, 0.131, 0.1305, 0.130, 0.1255, 0.125,
    0.1240, 0.130, 0.1328, 0.140, 0.1597, 0.150, 0.1569, 0.153, 0.1505, 0.140, 0.1399, 0.130, 0.1274, 0.120, 0.1152, 0.110, 0.1086, 0.108, 0.1070, 0.117, 0.1172, 0.130,
    0.1438, 0.150, 0.1672, 0.170, 0.1903, 0.181, 0.1858, 0.186, 0.1885, 0.190, 0.1907, 0.192, 0.1932, 0.194, 0.1942, 0.1942, 0.1941, 0.193, 0.1934, 0.193, 0.1922, 0.192, 0.1915, 0.191, 0.1919, 0.194, 0.1950,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L95_2_3_15_6spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1546, 0.18, 0.1934, 0.22, 0.2519, 0.35, 0.4184, 0.45, 0.5150, 0.55, 0.5811, 0.625, 0.6582, 0.689, 0.7035, 0.71, 0.7106, 0.705, 0.7042, 0.70, 0.6812, 0.682, 0.6845, 0.715,
    0.7647, 0.82, 0.9074, 0.95, 0.9847, 0.97, 0.9662, 0.94, 0.9160, 0.888, 0.8522, 0.81, 0.7878, 0.77, 0.7534, 0.75, 0.7438, 0.76, 0.900, 0.905, 0.9109, 0.95, 1.0180, 1.03,
    1.0784, 1.1, 1.1039, 1.11, 1.1176, 1.12, 1.1290, 1.13, 1.1417, 1.145, 1.1469, 1.146, 1.1463, 1.143, 1.1425, 1.30, 1.1360, 1.20, 1.1327, 1.133, 1.1348, 1.14, 1.1514,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L95_2_3_15_6spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1546, 0.18, 0.1934, 0.22, 0.2519, 0.35, 0.4284, 0.45, 0.5150, 0.55, 0.5811, 0.625, 0.6582, 0.689, 0.7035, 0.71, 0.7106, 0.705, 0.7042, 0.70, 0.6812, 0.682, 0.6845, 0.715,
    0.7647, 0.82, 0.9074, 0.95, 0.9847, 0.97, 0.9362, 0.91, 0.9160, 0.888, 0.8522, 0.81, 0.7878, 0.77, 0.7534, 0.75, 0.7438, 0.76, 0.900, 0.905, 0.9109, 0.95, 1.0180, 1.03,
    1.0784, 1.1, 1.1039, 1.11, 1.1176, 1.12, 1.1090, 1.13, 1.1417, 1.145, 1.1469, 1.146, 1.1463, 1.143, 1.1425, 1.30, 1.1360, 1.20, 1.1327, 1.133, 1.1348, 1.14, 1.1514,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_3_5_10_7spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0207, 0.022, 0.0249, 0.03, 0.0312, 0.04, 0.0503, 0.055, 0.0614, 0.065, 0.0687, 0.077, 0.0772, 0.08, 0.0819, 0.082, 0.0823, 0.085, 0.0810, 0.08, 0.0778, 0.078, 0.0784, 0.08,
    0.0893, 0.1, 0.1088, 0.11, 0.1202, 0.12, 0.1195, 0.115, 0.1147, 0.11, 0.1082, 0.105, 0.1012, 0.1, 0.0976, 0.97, 0.0969, 0.1, 0.1043, 0.11, 0.1231, 0.13, 0.1397, 0.145,
    0.1490, 0.15, 0.1529, 0.153, 0.1549, 0.155, 0.1564, 0.157, 0.1580, 0.1585, 0.1586, 0.1585, 0.1585, 0.1584, 0.1582, 0.158, 0.1575, 0.1573, 0.1572, 0.1573, 0.1574, 0.158, 0.1593,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_3_5_10_7spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0207, 0.022, 0.0249, 0.03, 0.0312, 0.04, 0.0603, 0.055, 0.0614, 0.065, 0.0687, 0.077, 0.0772, 0.08, 0.0819, 0.082, 0.0823, 0.085, 0.0810, 0.08, 0.0778, 0.078, 0.0784, 0.08,
    0.0893, 0.1, 0.1088, 0.11, 0.1202, 0.12, 0.1295, 0.115, 0.1147, 0.11, 0.1082, 0.105, 0.1012, 0.1, 0.0976, 0.97, 0.0969, 0.1, 0.1043, 0.11, 0.1231, 0.13, 0.1397, 0.145,
    0.1490, 0.15, 0.1529, 0.153, 0.1549, 0.155, 0.1664, 0.157, 0.1580, 0.1585, 0.1586, 0.1585, 0.1585, 0.1584, 0.1582, 0.158, 0.1575, 0.1573, 0.1572, 0.1573, 0.1574, 0.158, 0.1593,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_3_5_10_7spect3[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0207, 0.022, 0.0249, 0.03, 0.0212, 0.04, 0.0503, 0.055, 0.0614, 0.065, 0.0687, 0.077, 0.0772, 0.08, 0.0819, 0.082, 0.0823, 0.085, 0.0810, 0.08, 0.0778, 0.078, 0.0784, 0.08,
    0.0893, 0.1, 0.1088, 0.11, 0.1102, 0.12, 0.1195, 0.115, 0.1147, 0.11, 0.1082, 0.105, 0.1012, 0.1, 0.0976, 0.97, 0.0969, 0.1, 0.1043, 0.11, 0.1231, 0.13, 0.1397, 0.145,
    0.1490, 0.15, 0.1529, 0.153, 0.1449, 0.155, 0.1564, 0.157, 0.1580, 0.1585, 0.1486, 0.1585, 0.1585, 0.1584, 0.1582, 0.158, 0.1575, 0.1573, 0.1572, 0.1573, 0.1574, 0.158, 0.1593,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L34_1_8_1_9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0154, 0.018, 0.0199, 0.021, 0.0267, 0.03, 0.0454, 0.05, 0.0564, 0.06, 0.0639, 0.07, 0.0726, 0.075, 0.0776, 0.078, 0.0781, 0.078, 0.0771, 0.075, 0.0744, 0.074, 0.0737, 0.075,
    0.0788, 0.08, 0.0886, 0.09, 0.0926, 0.09, 0.0886, 0.085, 0.0821, 0.08, 0.0745, 0.07, 0.0672, 0.065, 0.0632, 0.063, 0.0621, 0.066, 0.0672, 0.08, 0.0807, 0.09, 0.0927, 0.095,
    0.0995, 0.1, 0.1023, 0.102, 0.1037, 0.104, 0.1049, 0.105, 0.1063, 0.1065, 0.1069, 0.1069, 0.1069, 0.1065, 0.1064, 0.106, 0.1057, 0.1055, 0.1053, 0.1054, 0.1055, 0.107, 0.1073,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L34_1_8_1_9spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0154, 0.018, 0.0199, 0.021, 0.0297, 0.03, 0.0454, 0.05, 0.0564, 0.06, 0.0639, 0.07, 0.0726, 0.075, 0.0776, 0.078, 0.0781, 0.078, 0.0771, 0.075, 0.0744, 0.074, 0.0737, 0.075,
    0.0788, 0.08, 0.0886, 0.09, 0.0826, 0.09, 0.0886, 0.085, 0.0821, 0.08, 0.0745, 0.07, 0.0672, 0.065, 0.0632, 0.063, 0.0621, 0.066, 0.0672, 0.08, 0.0807, 0.09, 0.0927, 0.095,
    0.0995, 0.1, 0.1023, 0.102, 0.1137, 0.104, 0.1049, 0.105, 0.1063, 0.1065, 0.1069, 0.1069, 0.1069, 0.1065, 0.1064, 0.106, 0.1057, 0.1055, 0.1053, 0.1054, 0.1055, 0.107, 0.1073,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L64_1_8_m1_9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0652, 0.08, 0.0864, 0.1, 0.1189, 0.15, 0.2063, 0.22, 0.2577, 0.28, 0.2935, 0.31, 0.3344, 0.34, 0.3582, 0.36, 0.3608, 0.36, 0.3569, 0.35, 0.3449, 0.34, 0.3395, 0.34,
    0.3558, 0.37, 0.3884, 0.39, 0.3971, 0.38, 0.3738, 0.35, 0.3408, 0.32, 0.3038, 0.28, 0.2686, 0.25, 0.2493, 0.245, 0.2435, 0.26, 0.2645, 0.3, 0.3207, 0.33, 0.3706, 0.38,
    0.398, 0.4, 0.4103, 0.415, 0.4163, 0.42, 0.4216, 0.425, 0.4278, 0.43, 0.4307, 0.4305, 0.4304, 0.43, 0.4282, 0.425, 0.4246, 0.424, 0.4225, 0.425, 0.4238, 0.43, 0.4321,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L84_0_8_m1spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1232, 0.15, 0.1635, 0.2, 0.2255, 0.3, 0.3913, 0.4, 0.4886, 0.5, 0.5567, 0.6, 0.6349, 0.65, 0.6810, 0.682, 0.6871, 0.685, 0.6811, 0.67, 0.6591, 0.655, 0.6507, 0.67, 0.6864, 0.70, 0.7553, 0.76, 0.7770, 0.75,
    0.7343, 0.7, 0.6719, 0.65, 0.6014, 0.55, 0.5340, 0.5, 0.4973, 0.49, 0.4858, 0.5, 0.5227, 0.55, 0.6222, 0.65, 0.7108, 0.75, 0.7606, 0.77, 0.7817, 0.79, 0.7925, 0.8,
    0.8023, 0.81, 0.8141, 0.815, 0.8196, 0.818, 0.8188, 0.817, 0.8147, 0.81, 0.8077, 0.805, 0.8037, 0.804, 0.8063, 0.81, 0.8221,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L63_1_3_m2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0646, 0.07, 0.0858, 0.1, 0.1184, 0.15, 0.2058, 0.22, 0.2571, 0.27, 0.2929, 0.31, 0.3340, 0.34, 0.3580, 0.36, 0.3608, 0.355, 0.3572, 0.35, 0.3454, 0.345, 0.3402, 0.35,
    0.3568, 0.37, 0.3896, 0.39, 0.3985, 0.38, 0.3752, 0.36, 0.3420, 0.32, 0.3049, 0.29, 0.2695, 0.255, 0.2501, 0.245, 0.2442, 0.25, 0.2644, 0.3, 0.3188, 0.35, 0.3671, 0.38,
    0.3942, 0.4, 0.4056, 0.41, 0.4115, 0.415, 0.4167, 0.42, 0.4229, 0.425, 0.4258, 0.4255, 0.4254, 0.424, 0.4232, 0.42, 0.4196, 0.417, 0.4174, 0.418, 0.4188, 0.42, 0.4271,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L44_2_3_m3spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0287, 0.03, 0.0380, 0.04, 0.0523, 0.08, 0.0910, 0.1, 0.1137, 0.12, 0.1295, 0.13, 0.1475, 0.15, 0.1578, 0.158, 0.1586, 0.157, 0.1566, 0.155, 0.1510, 0.15, 0.1481, 0.15,
    0.1538, 0.16, 0.1657, 0.167, 0.1678, 0.16, 0.1569, 0.15, 0.1421, 0.13, 0.1258, 0.12, 0.1103, 0.11, 0.1018, 0.1, 0.0994, 0.1, 0.1093, 0.12, 0.1356, 0.14, 0.1590, 0.16,
    0.1720, 0.173, 0.1775, 0.18, 0.1802, 0.181, 0.1826, 0.184, 0.1854, 0.186, 0.1867, 0.1865, 0.1865, 0.186, 0.1856, 0.185, 0.1840, 0.184, 0.1830, 0.1835, 0.1836, 0.186, 0.1873,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L65_96_45spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2103, 0.2, 0.1996, 0.19, 0.1795, 0.19, 0.2091, 0.22, 0.2312, 0.23, 0.2217, 0.22, 0.2014, 0.18, 0.1560, 0.1, 0.0981, 0.05, 0.0316, 0.0, -0.0261, -0.05, -0.0622, -0.07,
    -0.0700, -0.06, -0.0540, -0.04, -0.0294, -0.01, 0.0079, 0.03, 0.0536, 0.08, 0.0922, 0.1, 0.1193, 0.12, 0.1385, 0.16, 0.1735, 0.25, 0.3791, 0.5, 0.8646, 1.0, 1.2884, 1.35,
    1.5172, 1.6, 1.6128, 1.63, 1.6537, 1.66, 1.6730, 1.68, 1.682, 1.684, 1.6842, 1.685, 1.6888, 1.69, 1.6921, 1.695, 1.6967, 1.698, 1.6998, 1.699, 1.6979, 1.70, 1.7033,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L52_47_57spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0746, 0.7, 0.0693, 0.06, 0.0597, 0.06, 0.0637, 0.065, 0.0670, 0.063, 0.0611, 0.055, 0.0521, 0.04, 0.0367, 0.03, 0.0198, 0.0, -0.0001, -0.01, -0.0171, -0.02, -0.0218, -0.022,
    -0.0029, 0.0, 0.0366, 0.05, 0.0710, 0.08, 0.0956, 0.1, 0.1172, 0.12, 0.1343, 0.14, 0.1459, 0.15, 0.1539, 0.16, 0.1656, 0.20, 0.2305, 0.3, 0.3828, 0.45, 0.5156, 0.55,
    0.5875, 0.6, 0.6176, 0.62, 0.6309, 0.632, 0.6371, 0.64, 0.6401, 0.6401, 0.6402, 0.6415, 0.6417, 0.642, 0.6430, 0.644, 0.6448, 0.645, 0.6462, 0.646, 0.6454, 0.646, 0.6469,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L31_62_27spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0456, 0.045, 0.0430, 0.04, 0.0383, 0.04, 0.0441, 0.045, 0.0486, 0.047, 0.0462, 0.045, 0.0413, 0.04, 0.0310, 0.02, 0.0180, 0.01, 0.0031, 0.0, -0.0096, -0.01, -0.0179, -0.02,
    -0.0211, -0.02, -0.0198, -0.019, -0.0159, -0.01, -0.0081, 0.0, 0.0021, 0.02, 0.0109, 0.0105, 0.0172, 0.02, 0.0216, 0.025, 0.0294, 0.05, 0.0749, 0.1, 0.1824, 0.2, 0.2761, 0.3,
    0.3267, 0.33, 0.3479, 0.35, 0.3569, 0.36, 0.3611, 0.362, 0.3632, 0.363, 0.3635, 0.364, 0.3645, 0.365, 0.3653, 0.366, 0.3663, 0.367, 0.3670, 0.367, 0.3666, 0.367, 0.3677,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L79_m9_m28spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1188, 0.15, 0.1814, 0.2, 0.2793, 0.4, 0.5204, 0.6, 0.6631, 0.7, 0.7669, 0.8, 0.8855, 0.9, 0.9584, 0.96, 0.9703, 0.97, 0.9673, 0.96, 0.9410, 0.94, 0.9144, 0.91, 0.9020, 0.9,
    0.8941, 0.86, 0.8425, 0.8, 0.7410, 0.7, 0.6276, 0.6, 0.5106, 0.45, 0.4049, 0.35, 0.3456, 0.33, 0.3233, 0.33, 0.3475, 0.4, 0.4243, 0.45, 0.4941, 0.5, 0.5339, 0.54, 0.5507, 0.555,
    0.5593, 0.56, 0.5700, 0.575, 0.5852, 0.59, 0.5936, 0.592, 0.5918, 0.59, 0.5850, 0.58, 0.5735, 0.57, 0.5665, 0.57, 0.5710, 0.58, 0.5928,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L58_50_31spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1011, 0.11, 0.1006, 0.1, 0.0980, 0.11, 0.1257, 0.13, 0.1434, 0.145, 0.1459, 0.145, 0.1455, 0.14, 0.1325, 0.12, 0.1113, 0.1, 0.0852, 0.07, 0.0607, 0.05, 0.0494, 0.06,
    0.0625, 0.08, 0.0961, 0.1, 0.1233, 0.13, 0.1401, 0.15, 0.1549, 0.16, 0.1647, 0.165, 0.1692, 0.17, 0.1736, 0.18, 0.1864, 0.22, 0.2711, 0.35, 0.4721, 0.55, 0.6477, 0.7, 0.7428, 0.76, 0.7826, 0.79,
    0.8000, 0.804, 0.8088, 0.81, 0.8140, 0.815, 0.8150, 0.816, 0.8167, 0.817, 0.8176, 0.818, 0.8188, 0.819, 0.8198, 0.8195, 0.8192, 0.82, 0.8231,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L31_m52_27spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0093, -0.008, -0.0073, -0.005, -0.0042, -0.003, -0.0021, -0.002, -0.0018, 0.0, 0.0008, 0.004, 0.0053, 0.01, 0.0117, 0.015, 0.0191, 0.02, 0.0265, 0.03, 0.0321, 0.035, 0.0402, 0.05, 0.0582, 0.07, 0.0849, 0.1,
    0.1027, 0.103, 0.1056, 0.104, 0.1027, 0.1, 0.0984, 0.095, 0.0941, 0.093, 0.0915, 0.09, 0.0877, 0.08, 0.0684, 0.05, 0.0228, 0.0, -0.0170, -0.02, -0.0383, -0.04, -0.0471, -0.05,
    -0.0505, -0.051, -0.0519, -0.052, -0.0523, -0.0523, -0.0524, -0.0528, -0.0529, -0.053, -0.0533, -0.0538, -0.0538, -0.54, -0.0541, -0.054, -0.0540, -0.054, -0.0538,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L44_2_2_m7_35spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0303, 0.04, 0.0412, 0.05, 0.0579, 0.08, 0.1021, 0.11, 0.1283, 0.135, 0.1465, 0.16, 0.1672, 0.17, 0.1790, 0.179, 0.1798, 0.178, 0.1775, 0.173, 0.1712, 0.17, 0.1668, 0.168, 0.1695, 0.18, 0.1769, 0.175, 0.1746, 0.17, 0.1600, 0.15,
    0.1422, 0.13, 0.1229, 0.11, 0.1049, 0.1, 0.0950, 0.093, 0.0921, 0.1, 0.1027, 0.12, 0.1314, 0.14, 0.1569, 0.16, 0.1711, 0.175, 0.1771, 0.18, 0.1800, 0.181, 0.1826, 0.184, 0.1857, 0.187, 0.1872, 0.1871,
    0.1871, 0.186, 0.1860, 0.185, 0.1841, 0.184, 0.1830, 0.183, 0.1837, 0.186, 0.1878,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L47_m10_8_0_41spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0213, 0.03, 0.0316, 0.04, 0.0475, 0.06, 0.0864, 0.09, 0.1088, 0.11, 0.1258, 0.13, 0.1458, 0.15, 0.1595, 0.16, 0.1641, 0.165, 0.1662, 0.164, 0.1639, 0.164, 0.1647, 0.17, 0.1775, 0.18, 0.1995, 0.2, 0.2080, 0.2, 0.1975, 0.19, 0.1809, 0.17,
    0.1621, 0.15, 0.1445, 0.14, 0.1347, 0.131, 0.1302, 0.13, 0.1289, 0.129, 0.1288, 0.129, 0.1291, 0.1295, 0.1298, 0.13, 0.1301, 0.1305, 0.1308, 0.131, 0.1323, 0.133, 0.1347, 0.135,
    0.1360, 0.136, 0.1355, 0.135, 0.1344, 0.133, 0.1325, 0.132, 0.1314, 0.132, 0.1321, 0.134, 0.1357,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L32_4_8_m3_2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0163, 0.02, 0.0211, 0.025, 0.0285, 0.03, 0.0490, 0.05, 0.0611, 0.065, 0.0694, 0.07, 0.0786, 0.08, 0.0837, 0.083, 0.0835, 0.082, 0.0819, 0.08, 0.0785, 0.077, 0.0764, 0.077, 0.0784, 0.08, 0.0833, 0.083, 0.0835, 0.08, 0.0776, 0.075,
    0.0701, 0.065, 0.0617, 0.06, 0.0538, 0.05, 0.0494, 0.045, 0.0484, 0.05, 0.0552, 0.06, 0.0729, 0.08, 0.0885, 0.09, 0.0972, 0.1, 0.1008, 0.102, 0.1026, 0.0103, 0.1039, 0.0104,
    0.1055, 0.0106, 0.1062, 0.0161, 0.1061, 0.0106, 0.1056, 0.0105, 0.1048, 0.0104, 0.1044, 0.01045, 0.1047, 0.0105, 0.1066,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L57_m6_9_2_9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0379, 0.04, 0.0525, 0.06, 0.0750, 0.1, 0.1325, 0.14, 0.1658, 0.18, 0.1901, 0.2, 0.2187, 0.22, 0.2373, 0.24, 0.2425, 0.243, 0.2437, 0.24, 0.2387, 0.236, 0.2394, 0.24, 0.2594, 0.28, 0.2945, 0.3, 0.3096, 0.3,
    0.2963, 0.28, 0.2737, 0.26, 0.2478, 0.23, 0.2230, 0.21, 0.2093, 0.205, 0.2038, 0.206, 0.2078, 0.21, 0.2218, 0.225, 0.2346, 0.24, 0.2423, 0.244, 0.2456, 0.246, 0.2478, 0.25,
    0.2505, 0.252, 0.2543, 0.255, 0.2562, 0.257, 0.2557, 0.255, 0.2541, 0.253, 0.2515, 0.252, 0.2500, 0.25, 0.2509, 0.252, 0.2564,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L33_2_4_m4_5spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0167, 0.02, 0.0223, 0.03, 0.0310, 0.04, 0.054, 0.06, 0.0679, 0.07, 0.0774, 0.08, 0.0881, 0.09, 0.0943, 0.095, 0.0946, 0.094, 0.0933, 0.09, 0.0899, 0.088, 0.0877, 0.088,
    0.0898, 0.09, 0.0949, 0.095, 0.0947, 0.09, 0.0875, 0.08, 0.0785, 0.07, 0.0685, 0.06, 0.0592, 0.055, 0.0541, 0.053, 0.0527, 0.055, 0.0587, 0.06, 0.0748, 0.08, 0.0891, 0.09,
    0.0971, 0.1, 0.1005, 0.102, 0.1021, 0.103, 0.1035, 0.104, 0.1052, 0.1055, 0.1060, 0.106, 0.1059, 0.106, 0.1053, 0.105, 0.1043, 0.104, 0.1038, 0.104, 0.1041, 0.105, 0.1063,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L35_11_65_m1_1spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0226, 0.025, 0.0276, 0.03, 0.0351, 0.04, 0.0579, 0.06, 0.0715, 0.075, 0.0802, 0.085, 0.0897, 0.09, 0.0940, 0.093, 0.0925, 0.09, 0.0891, 0.085, 0.0841, 0.082, 0.0810, 0.082,
    0.0830, 0.085, 0.0888, 0.089, 0.0897, 0.085, 0.0842, 0.08, 0.0772, 0.07, 0.0691, 0.065, 0.0612, 0.055, 0.0570, 0.057, 0.0568, 0.06, 0.0691, 0.08, 0.0998, 0.1, 0.1269, 0.13,
    0.1417, 0.145, 0.1479, 0.15, 0.1508, 0.151, 0.1527, 0.153, 0.1546, 0.155, 0.1554, 0.1554, 0.1555, 0.155, 0.1550, 0.155, 0.1543, 0.153, 0.1538, 0.154, 0.1541, 0.155, 0.1564,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L52_m2_7_8_9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0316, 0.04, 0.0411, 0.05, 0.0556, 0.08, 0.0946, 0.1, 0.1172, 0.12, 0.1332, 0.14, 0.1521, 0.16, 0.1640, 0.165, 0.1670, 0.167, 0.1670, 0.165, 0.1628, 0.164, 0.1643, 0.17, 0.1829, 0.2, 0.2154, 0.22, 0.2324, 0.23,
    0.2267, 0.22, 0.2135, 0.2, 0.1973, 0.19, 0.1812, 0.18, 0.1725, 0.17, 0.1694, 0.17, 0.1756, 0.18, 0.1930, 0.2, 0.2085, 0.21, 0.2175, 0.22, 0.2213, 0.222, 0.2236, 0.224, 0.2259, 0.227,
    0.2287, 0.229, 0.2299, 0.2298, 0.2296, 0.229, 0.2286, 0.227, 0.2270, 0.227, 0.2261, 0.2265, 0.2267, 0.23, 0.2305,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L32_7_m2_5spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0170, 0.02, 0.0216, 0.025, 0.0285, 0.04, 0.0483, 0.05, 0.0600, 0.065, 0.0678, 0.07, 0.0765, 0.08, 0.0809, 0.0802, 0.0804, 0.079, 0.0783, 0.075, 0.0747, 0.074, 0.0724, 0.073, 0.0742, 0.077, 0.0789, 0.079, 0.0792, 0.075,
    0.0738, 0.07, 0.0670, 0.06, 0.0593, 0.055, 0.0519, 0.05, 0.0479, 0.0476, 0.0472, 0.05, 0.0552, 0.06, 0.0756, 0.09, 0.0937, 0.1, 0.1036, 0.104, 0.1078, 0.108, 0.1097, 0.11,
    0.1112, 0.112, 0.1127, 0.113, 0.1134, 0.1134, 0.1134, 0.113, 0.1130, 0.113, 0.1122, 0.112, 0.1118, 0.112, 0.1121, 0.113, 0.1140,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L32_3_4_m3_8spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0156, 0.02, 0.0205, 0.025, 0.0281, 0.03, 0.0488, 0.06, 0.0611, 0.065, 0.0694, 0.07, 0.0789, 0.08, 0.0842, 0.0843, 0.0843, 0.084, 0.0830, 0.08, 0.0798, 0.078, 0.0777, 0.078, 0.0797, 0.08,
    0.0844, 0.0844, 0.0843, 0.08, 0.0781, 0.075, 0.0702, 0.065, 0.0615, 0.06, 0.0534, 0.05, 0.0489, 0.048, 0.0477, 0.05, 0.0538, 0.06, 0.0697, 0.07, 0.0838, 0.09, 0.0916, 0.093,
    0.0949, 0.095, 0.0965, 0.097, 0.0978, 0.098, 0.0993, 0.1, 0.1000, 0.1, 0.0999, 0.0996, 0.0994, 0.099, 0.0986, 0.0985, 0.0981, 0.0985, 0.0984, 0.1, 0.1004,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L50_m5_3_6_5spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0275, 0.03, 0.0370, 0.04, 0.0515, 0.06, 0.0893, 0.1, 0.1111, 0.12, 0.1269, 0.13, 0.1456, 0.15, 0.1578, 0.16, 0.1614, 0.162, 0.1622, 0.16, 0.1589, 0.16, 0.1604, 0.17, 0.1776, 0.19, 0.2074, 0.21,
    0.2223, 0.22, 0.2158, 0.21, 0.2020, 0.19, 0.1855, 0.18, 0.1695, 0.165, 0.1607, 0.16, 0.1572, 0.16, 0.1605, 0.17, 0.1711, 0.18, 0.1807, 0.185, 0.1864, 0.188, 0.1889, 0.19,
    0.1906, 0.191, 0.1925, 0.194, 0.1951, 0.196, 0.1963, 0.196, 0.1960, 0.195, 0.1949, 0.194, 0.1933, 0.193, 0.1923, 0.1925, 0.1929, 0.195, 0.1966,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L44_3_96_m8_8spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0320, 0.04, 0.0431, 0.05, 0.0604, 0.09, 0.1063, 0.12, 0.1336, 0.14, 0.1524, 0.16, 0.1737, 0.18, 0.1856, 0.186, 0.1860, 0.185, 0.1831, 0.18, 0.1761, 0.175, 0.1709, 0.172, 0.1721, 0.175, 0.1773, 0.175,
    0.1732, 0.16, 0.1576, 0.15, 0.1391, 0.13, 0.1192, 0.11, 0.1007, 0.1, 0.0905, 0.09, 0.0877, 0.1, 0.1001, 0.12, 0.1329, 0.15, 0.1620, 0.17, 0.1782, 0.1850, 0.188, 0.1882, 0.19,
    0.1910, 0.193, 0.1942, 0.195, 0.1958, 0.1956, 0.195, 0.195, 0.1945, 0.193, 0.1926, 0.192, 0.1915, 0.192, 0.1922, 0.195, 0.1965,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L34_3_6_m5_4spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0184, 0.02, 0.0245, 0.03, 0.0339, 0.04, 0.0591, 0.06, 0.0741, 0.08, 0.0844, 0.09, 0.0960, 0.1, 0.1025, 0.105, 0.1026, 0.101, 0.1009, 0.1, 0.0970, 0.095, 0.0944, 0.095,
    0.0959, 0.1, 0.1004, 0.1, 0.0993, 0.095, 0.0913, 0.09, 0.0815, 0.08, 0.0707, 0.065, 0.0607, 0.06, 0.0552, 0.055, 0.0537, 0.06, 0.0609, 0.07, 0.0799, 0.08, 0.0968, 0.1, 0.1061, 0.11, 0.1101, 0.111, 0.1119, 0.112, 0.1135, 0.114,
    0.1153, 0.116, 0.1162, 0.1161, 0.1161, 0.116, 0.1155, 0.115, 0.1145, 0.114, 0.1139, 0.114, 0.1143, 0.115, 0.1166,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L31_5_9_m4spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0166, 0.02, 0.0214, 0.025, 0.0288, 0.04, 0.0495, 0.05, 0.0618, 0.065, 0.0700, 0.07, 0.0793, 0.08, 0.0842, 0.084, 0.0839, 0.083, 0.0820, 0.08, 0.0784, 0.077, 0.0761, 0.077, 0.0774, 0.08, 0.0813, 0.081,
    0.0808, 0.08, 0.0747, 0.07, 0.0671, 0.06, 0.0587, 0.055, 0.0508, 0.05, 0.0465, 0.046, 0.0456, 0.05, 0.0530, 0.06, 0.0720, 0.08, 0.0888, 0.09, 0.0981, 0.1, 0.1020, 0.102,
    0.1039, 0.104, 0.1053, 0.106, 0.1068, 0.107, 0.1076, 0.1075, 0.1075, 0.1074, 0.1071, 0.107, 0.1062, 0.106, 0.1058, 0.106, 0.1061, 0.107, 0.1080,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L35_3_4_m11spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0211, 0.025, 0.0291, 0.03, 0.0414, 0.05, 0.0739, 0.08, 0.0932, 0.1, 0.1066, 0.12, 0.1217, 0.13, 0.1302, 0.1302, 0.1303, 0.129, 0.1283, 0.125, 0.1235, 0.12, 0.1193, 0.118, 0.1180, 0.118, 0.1183, 0.115, 0.1129, 0.105, 0.1007, 0.9,
    0.0871, 0.08, 0.0727, 0.065, 0.0595, 0.053, 0.0521, 0.503, 0.0501, 0.055, 0.0583, 0.07, 0.0804, 0.09, 0.1000, 0.11, 0.1109, 0.115, 0.1154, 0.116, 0.1176, 0.118, 0.1195, 0.12,
    0.1217, 0.122, 0.1229, 0.122, 0.1227, 0.122, 0.1219, 0.121, 0.1205, 0.12, 0.1197, 0.12, 0.1202, 0.121, 0.1233,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L31_4_5_m4_7spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0163, 0.02, 0.0214, 0.025, 0.0292, 0.04, 0.0507, 0.06, 0.0634, 0.07, 0.0720, 0.08, 0.0818, 0.082, 0.0871, 0.087, 0.0870, 0.086, 0.0854, 0.084, 0.0820, 0.08, 0.0796, 0.08,
    0.0810, 0.082, 0.0849, 0.084, 0.0842, 0.08, 0.0776, 0.07, 0.0694, 0.065, 0.0605, 0.055, 0.0520, 0.05, 0.047, 0.046, 0.0463, 0.05, 0.0531, 0.06, 0.0706, 0.08,
    0.0862, 0.09, 0.0948, 0.0948, 0.0984, 0.1, 0.1001, 0.101, 0.1015, 0.102, 0.1031, 0.1032, 0.1038, 0.1038, 0.1038, 0.1035, 0.1033, 0.103, 0.1024, 0.102, 0.1019, 0.102, 0.1022, 0.103, 0.1043,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L35_4_8_m6_4spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0202, 0.025, 0.0268, 0.03, 0.0369, 0.05, 0.0643, 0.07, 0.0805, 0.09, 0.0917, 0.095, 0.1042, 0.11, 0.1110, 0.1105, 0.1109, 0.11, 0.1089, 0.105, 0.1045, 0.102,
    0.1013, 0.102, 0.1023, 0.105, 0.1060, 0.105, 0.1040, 0.1, 0.0951, 0.09, 0.0845, 0.08, 0.0729, 0.07, 0.0621, 0.06, 0.0562, 0.055, 0.0547, 0.06, 0.0631, 0.07,
    0.0851, 0.09, 0.1045, 0.11, 0.1153, 0.116, 0.1198, 0.12, 0.1220, 0.123, 0.1237, 0.124, 0.1257, 0.126, 0.1267, 0.1266, 0.1266, 0.126, 0.1259, 0.125, 0.1248, 0.1245, 0.1242, 0.1242, 0.1246, 0.126, 0.1272,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L95_10_7_m14_3spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2133, 0.25, 0.2824, 0.32, 0.3890, 0.5, 0.6783, 0.8, 0.8501, 0.9, 0.9674, 1.05, 1.0994, 1.11, 1.1716, 1.1705, 1.1704, 1.15, 1.1487, 1.12, 1.1019, 1.09, 1.0679, 1.07, 1.0770, 1.1, 1.1145, 1.1, 1.0929, 1.0, 0.9987, 0.9,
    0.8859, 0.8, 0.7633, 0.7, 0.6498, 0.6, 0.5873, 0.57, 0.5718, 0.6, 0.6609, 0.7, 0.8939, 1.0, 1.1003, 1.1, 1.2145, 1.25, 1.2625, 1.27, 1.2851, 1.3, 1.3037, 1.31, 1.3246, 1.33,
    1.3346, 1.334, 1.3339, 1.33, 1.3270, 1.32, 1.3152, 1.31, 1.3082, 1.31, 1.3127, 1.33, 1.340,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L36_34_m7_5spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0401, 0.045, 0.0459, 0.05, 0.0545, 0.07, 0.0861, 0.1, 0.1055, 0.11, 0.1162, 0.12, 0.1270, 0.128, 0.1290, 0.125, 0.1221, 0.12, 0.1125, 0.11, 0.1015, 0.1, 0.0924, 0.09, 0.0859, 0.085, 0.0806, 0.08, 0.0731, 0.07, 0.0646, 0.06,
    0.0570, 0.05, 0.0484, 0.04, 0.0399, 0.036, 0.0356, 0.037, 0.0378, 0.05, 0.0685, 0.1, 0.1433, 0.15, 0.2088, 0.22, 0.2444, 0.25, 0.2593, 0.26, 0.2658, 0.266, 0.2695, 0.27,
    0.2726, 0.273, 0.2738, 0.274, 0.2742, 0.274, 0.2738, 0.2733, 0.2731, 0.273, 0.2727, 0.273, 0.2730, 0.275, 0.2762,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L37_59_2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0604, 0.061, 0.0622, 0.063, 0.0644, 0.07, 0.0898, 0.1, 0.1061, 0.11, 0.1114, 0.113, 0.1147, 0.11, 0.1079, 0.1, 0.0930, 0.08, 0.0749, 0.06, 0.0576, 0.05, 0.0442, 0.04, 0.0342, 0.03, 0.0265, 0.021, 0.0208, 0.02, 0.0197, 0.02,
    0.0219, 0.022, 0.0228, 0.022, 0.0220, 0.0225, 0.0225, 0.025, 0.0300, 0.05, 0.0857, 0.2, 0.2187, 0.3, 0.3350, 0.35, 0.3978, 0.4, 0.4241, 0.43, 0.4353, 0.44, 0.4410, 0.445,
    0.4447, 0.445, 0.4457, 0.446, 0.4467, 0.447, 0.4471, 0.4472, 0.4474, 0.4475, 0.4476, 0.4475, 0.4475, 0.45, 0.4504,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::Colorlab_L69_14_m9spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1024, 0.12, 0.1316, 0.15, 0.1764, 0.2, 0.3024, 0.35, 0.3775, 0.4, 0.4274, 0.45, 0.4832, 0.5, 0.5118, 0.51, 0.5082, 0.5, 0.4953, 0.48, 0.4721, 0.46, 0.4554, 0.455, 0.4581, 0.46, 0.4737, 0.47, 0.4648, 0.44, 0.4258, 0.4,
    0.3795, 0.35, 0.3290, 0.3, 0.2813, 0.27, 0.2554, 0.253, 0.2506, 0.28, 0.2997, 0.33, 0.4254, 0.4, 0.5364, 0.55, 0.5976, 0.6, 0.6232, 0.63, 0.6351, 0.64, 0.6441, 0.65,
    0.6536, 0.655, 0.6580, 0.658, 0.6579, 0.656, 0.6551, 0.652, 0.6503, 0.65, 0.6474, 0.648, 0.6493, 0.67, 0.6613,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::Colorlab_L92_13_m16spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2033, 0.23, 0.2685, 0.32, 0.3690, 0.5, 0.6430, 0.7, 0.8059, 0.85, 0.9166, 1.0, 1.0407, 1.1, 1.1075, 1.105, 1.1043, 1.09, 1.0816, 1.05, 1.0357, 1.02, 1.0006, 1.0, 1.0019, 1.02, 1.0261, 1.0, 0.9978, 0.95, 0.9063, 0.8, 0.7994, 0.75, 0.6845, 0.6,
    0.5772, 0.55, 0.5185, 0.52, 0.5047, 0.57, 0.5942, 0.7, 0.8269, 0.9, 1.0329, 1.1, 1.1466, 1.15, 1.1944, 1.2, 1.2167, 1.22, 1.2346, 1.24, 1.2546, 1.26, 1.2641, 1.264,
    1.2635, 1.26, 1.2571, 1.25, 1.2460, 1.24, 1.2394, 1.24, 1.2437, 1.25, 1.2695,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L49_21_m12spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0571, 0.06, 0.0711, 0.08, 0.0928, 0.11, 0.1566, 0.17, 0.1950, 0.2, 0.2195, 0.23, 0.2461, 0.25, 0.2579, 0.255, 0.2527, 0.25, 0.2427, 0.23, 0.2282, 0.22, 0.2160, 0.21, 0.2094, 0.206, 0.2057, 0.2, 0.1934, 0.18, 0.1722, 0.16, 0.1498, 0.13, 0.1258, 0.11,
    0.1033, 0.1, 0.0911, 0.0905, 0.0904, 0.1, 0.1245, 0.18, 0.2101, 0.25, 0.2855, 0.3, 0.3266, 0.33, 0.3439, 0.35, 0.3516, 0.355, 0.3568, 0.36, 0.3619, 0.363, 0.3642, 0.3644,
    0.3644, 0.364, 0.3631, 0.361, 0.3609, 0.36, 0.3596, 0.36, 0.3605, 0.365, 0.3665,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L56_20_m15spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0778, 0.08, 0.0988, 0.11, 0.1312, 0.19, 0.2241, 0.25, 0.2800, 0.3, 0.3163, 0.34, 0.3559, 0.36, 0.3744, 0.37, 0.3683, 0.36, 0.3553, 0.34, 0.3356, 0.32, 0.3184, 0.31, 0.3083, 0.305, 0.3014, 0.3, 0.2817, 0.26, 0.2492, 0.23,
    0.2150, 0.2, 0.1787, 0.15, 0.1448, 0.13, 0.1264, 0.125, 0.1244, 0.15, 0.1690, 0.22, 0.2816, 0.32, 0.3810, 0.4, 0.4353, 0.44, 0.4580, 0.46, 0.4682, 0.47, 0.4754, 0.48,
    0.4826, 0.483, 0.4859, 0.486, 0.4861, 0.485, 0.4842, 0.483, 0.4807, 0.48, 0.4787, 0.48, 0.4800, 0.485, 0.4888,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L68_21_m19spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1192, 0.13, 0.1536, 0.18, 0.2066, 0.3, 0.3562, 0.4, 0.4460, 0.45, 0.5051, 0.55, 0.5700, 0.6, 0.6015, 0.6, 0.5935, 0.58, 0.5748, 0.56, 0.5447, 0.53, 0.5182, 0.51, 0.5024, 0.5, 0.4912, 0.48, 0.4591, 0.42, 0.4053, 0.37,
    0.3484, 0.3, 0.2883, 0.25, 0.2325, 0.22, 0.2021, 0.2, 0.1976, 0.23, 0.2631, 0.33, 0.4293, 0.5, 0.5760, 0.6564, 0.6, 0.6901, 0.7, 0.7051, 0.71, 0.7161, 0.72, 0.7275, 0.73,
    0.7328, 0.733, 0.7329, 0.73, 0.7297, 0.725, 0.7241, 0.721, 0.7206, 0.722, 0.7229, 0.73, 0.7370,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L98_m2_m32spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2248, 0.3, 0.3270, 0.4, 0.4865, 0.6, 0.8902, 1.0, 1.1296, 1.13, 1.3004, 1.4, 1.4939, 1.5, 1.6083, 1.61, 1.6196, 1.61, 1.6050, 1.6, 1.5533, 1.54, 1.5033, 1.5, 1.4786, 1.47, 1.4628, 1.4, 1.3770, 1.3, 1.2127, 1.1, 1.0306, 0.9, 0.8418, 0.7,
    0.6705, 0.6, 0.574, 0.55, 0.5427, 0.6, 0.6117, 0.7, 0.8072, 0.9, 0.9823, 1.0, 1.0802, 1.1, 1.1215, 1.13, 1.1414, 1.15, 1.1617, 1.17, 1.1884, 1.2, 1.2025, 1.2, 1.2001, 1.19,
    1.1894, 1.18, 1.1710, 1.16, 1.1597, 1.16, 1.1670, 1.2, 1.2037,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L98_m2_m32spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2248, 0.3, 0.3370, 0.4, 0.4865, 0.6, 0.8902, 1.0, 1.1296, 1.13, 1.3004, 1.4, 1.4939, 1.5, 1.6083, 1.61, 1.6196, 1.61, 1.6050, 1.6, 1.5533, 1.54, 1.5033, 1.5, 1.4786, 1.47, 1.4628, 1.4, 1.3770, 1.3, 1.2127, 1.1, 1.0306, 0.9, 0.8418, 0.7,
    0.6705, 0.6, 0.584, 0.55, 0.5427, 0.6, 0.6117, 0.7, 0.8072, 0.9, 0.9823, 1.0, 1.0802, 1.1, 1.1215, 1.13, 1.1414, 1.15, 1.1617, 1.17, 1.1884, 1.2, 1.2025, 1.2, 1.2001, 1.19,
    1.1894, 1.18, 1.1810, 1.16, 1.1597, 1.16, 1.1670, 1.2, 1.2037,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L41_m27_m16spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0125, 0.02, 0.0262, 0.03, 0.0479, 0.06, 0.0964, 0.1, 0.1246, 0.13, 0.1469, 0.16, 0.1731, 0.18, 0.1917, 0.195, 0.1986, 0.2, 0.2028, 0.201, 0.2015, 0.2, 0.1997, 0.2, 0.2021, 0.203, 0.2066, 0.2, 0.1994, 0.18, 0.1778, 0.16, 0.1518, 0.13,
    0.1251, 0.11, 0.1012, 0.1, 0.0876, 0.085, 0.0805, 0.08, 0.0713, 0.06, 0.0537, 0.04, 0.0389, 0.035, 0.0313, 0.03, 0.0282, 0.025, 0.0273, 0.028, 0.0283, 0.03, 0.0308, 0.031,
    0.0325, 0.032, 0.0318, 0.031, 0.0301, 0.03, 0.0274, 0.028, 0.0258, 0.026, 0.0268, 0.03, 0.0311,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L41_m27_m16spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0125, 0.02, 0.0262, 0.03, 0.0489, 0.06, 0.0964, 0.1, 0.1246, 0.13, 0.1469, 0.16, 0.1731, 0.18, 0.1917, 0.195, 0.1986, 0.2, 0.2028, 0.201, 0.2015, 0.2, 0.1997, 0.2, 0.2021, 0.203, 0.2066, 0.2, 0.1994, 0.18, 0.1778, 0.16, 0.1518, 0.13,
    0.1251, 0.11, 0.1012, 0.1, 0.0886, 0.085, 0.0805, 0.08, 0.0713, 0.06, 0.0537, 0.04, 0.0389, 0.035, 0.0313, 0.03, 0.0282, 0.025, 0.0273, 0.028, 0.0283, 0.03, 0.0308, 0.031,
    0.0325, 0.032, 0.0318, 0.031, 0.0311, 0.03, 0.0274, 0.028, 0.0258, 0.026, 0.0268, 0.03, 0.0311,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L15_m9_4spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0016, 0.002, 0.0027, 0.003, 0.0043, 0.006, 0.0081, 0.009, 0.0102, 0.011, 0.0119, 0.012, 0.0140, 0.014, 0.0156, 0.016, 0.0164, 0.017, 0.0170, 0.017, 0.0171, 0.0175, 0.0177, 0.02, 0.0201, 0.022, 0.0241, 0.025, 0.0261, 0.026, 0.0254, 0.025, 0.0238, 0.23,
    0.0218, 0.02, 0.0200, 0.019, 0.0189, 0.0185, 0.0183, 0.018, 0.0172, 0.016, 0.0148, 0.013, 0.0127, 0.012, 0.0116, 0.0115, 0.0112, 0.0112, 0.0111, 0.0111, 0.0111, 0.0112, 0.0113, 0.0114, 0.0115, 0.0115,
    0.0114, 0.0114, 0.0113, 0.0112, 0.0111, 0.011, 0.0109, 0.0110, 0.011, 0.0114,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L15_m9_4spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0016, 0.002, 0.0027, 0.003, 0.0043, 0.006, 0.0081, 0.009, 0.0102, 0.011, 0.0119, 0.012, 0.0140, 0.014, 0.0156, 0.016, 0.0164, 0.017, 0.0170, 0.017, 0.0171, 0.0175, 0.0177, 0.02, 0.0201, 0.022, 0.0241, 0.025, 0.0261, 0.026, 0.0254, 0.025, 0.0238, 0.23,
    0.0218, 0.02, 0.0220, 0.019, 0.0199, 0.0188, 0.0183, 0.018, 0.0172, 0.016, 0.0148, 0.013, 0.0127, 0.012, 0.0116, 0.0115, 0.0112, 0.0112, 0.0111, 0.0111, 0.0111, 0.0112, 0.0113, 0.0114, 0.0115, 0.0115,
    0.0114, 0.0114, 0.0113, 0.0112, 0.0111, 0.011, 0.0109, 0.0110, 0.011, 0.0114,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L11_m11_2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0008, 0.001, 0.0015, 0.002, 0.0027, 0.004, 0.0053, 0.006, 0.0067, 0.007, 0.0080, 0.009, 0.0095, 0.01, 0.0107, 0.0105, 0.0113, 0.011, 0.0118, 0.012, 0.0120, 0.012, 0.0124, 0.013, 0.0140, 0.015, 0.0165, 0.017, 0.0178, 0.0175, 0.0172, 0.017,
    0.0160, 0.015, 0.0145, 0.014, 0.0132, 0.013, 0.0124, 0.012, 0.0119, 0.011, 0.0108, 0.09, 0.0085, 0.007, 0.0064, 0.006, 0.0053, 0.005, 0.0048,
    0.0049, 0.0048, 0.0047, 0.0048, 0.0048, 0.0049, 0.005, 0.0050, 0.005, 0.0049, 0.0049, 0.0048, 0.0047, 0.0047, 0.0047, 0.0046, 0.0046, 0.0046, 0.0048, 0.0049,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L14_m4_3spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0024, 0.003, 0.0034, 0.004, 0.0049, 0.007, 0.0086, 0.01, 0.0108, 0.011, 0.0124, 0.013, 0.0144, 0.015, 0.0157, 0.016, 0.0162, 0.0163, 0.0164, 0.0163, 0.0162, 0.0164, 0.0165, 0.017, 0.0183, 0.02, 0.0213, 0.022, 0.0228, 0.0225, 0.0221, 0.021,
    0.0206, 0.02, 0.0188, 0.018, 0.0172, 0.017, 0.0162, 0.016, 0.0158, 0.0158, 0.0157, 0.0158, 0.0158, 0.0159, 0.0159, 0.016, 0.0160, 0.016, 0.0160, 0.0161, 0.0161, 0.0162,
    0.0163, 0.0164, 0.0165, 0.0165, 0.0166, 0.0166, 0.0166, 0.0166, 0.0165, 0.0164, 0.0163, 0.0163, 0.0162, 0.0162, 0.0162, 0.0165, 0.0166,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L41_38_24spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0455, 0.0455, 0.0453, 0.045, 0.0443, 0.05, 0.0570, 0.06, 0.0651, 0.065, 0.0664, 0.066, 0.0664, 0.065, 0.0607, 0.06, 0.0513, 0.04, 0.0397, 0.03, 0.0288, 0.025, 0.0239, 0.03, 0.0302, 0.04, 0.0460, 0.05, 0.0586, 0.06, 0.0662, 0.07,
    0.0727, 0.075, 0.0769, 0.077, 0.0788, 0.08, 0.0807, 0.085, 0.0863, 0.11, 0.1241, 0.18, 0.2137, 0.24, 0.2920, 0.3, 0.3344, 0.34, 0.3522, 0.354, 0.3599, 0.36, 0.3639, 0.365,
    0.3662, 0.366, 0.3667, 0.367, 0.3674, 0.3675, 0.3678, 0.368, 0.3684, 0.3688, 0.3688, 0.3685, 0.3686, 0.37, 0.3703,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L41_38_24spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0455, 0.0455, 0.0453, 0.045, 0.0443, 0.05, 0.0580, 0.06, 0.0651, 0.065, 0.0664, 0.066, 0.0664, 0.065, 0.0607, 0.06, 0.0513, 0.04, 0.0397, 0.03, 0.0288, 0.025, 0.0239, 0.03, 0.0302, 0.04, 0.0460, 0.05, 0.0586, 0.06, 0.0662, 0.07,
    0.0727, 0.075, 0.0769, 0.077, 0.0798, 0.08, 0.0807, 0.085, 0.0863, 0.11, 0.1241, 0.18, 0.2137, 0.24, 0.2920, 0.3, 0.3344, 0.34, 0.3522, 0.354, 0.3599, 0.36, 0.3639, 0.365,
    0.3662, 0.366, 0.3667, 0.367, 0.3684, 0.3675, 0.3678, 0.368, 0.3684, 0.3688, 0.3688, 0.3685, 0.3686, 0.37, 0.3703,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L53_48_58spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0768, 0.075, 0.0714, 0.07, 0.0615, 0.064, 0.0656, 0.067, 0.0689, 0.065, 0.0628, 0.06, 0.0534, 0.04, 0.0375, 0.03, 0.0201, 0.0, -0.0004, -0.01, -0.0180, -0.02, -0.0228, -0.03, -0.0035, 0.0, 0.0370, 0.05, 0.0724, 0.08, 0.0977, 0.1, 0.1200, 0.13,
    0.1376, 0.14, 0.1497, 0.15, 0.1580, 0.16, 0.1701, 0.2, 0.2370, 0.3, 0.3941, 0.45, 0.5311, 0.55, 0.6053, 0.62, 0.6364, 0.64, 0.6500, 0.65, 0.6564, 0.656, 0.6596, 0.6597,
    0.6597, 0.66, 0.6612, 0.662, 0.6625, 0.663, 0.6644, 0.665, 0.6658, 0.6652, 0.6650, 0.666, 0.6665,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L53_48_58spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0768, 0.075, 0.0714, 0.07, 0.0615, 0.064, 0.0656, 0.067, 0.0689, 0.065, 0.0628, 0.06, 0.0534, 0.04, 0.0375, 0.03, 0.0201, 0.0, -0.0004, -0.01, -0.0180, -0.02, -0.0228, -0.03, -0.0035, 0.0, 0.0370, 0.05, 0.0724, 0.08, 0.0977, 0.1, 0.1200, 0.13,
    0.1376, 0.14, 0.1497, 0.15, 0.1580, 0.165, 0.1801, 0.2, 0.2370, 0.3, 0.3941, 0.45, 0.5311, 0.55, 0.6053, 0.62, 0.6364, 0.64, 0.6500, 0.65, 0.6564, 0.656, 0.6596, 0.6597,
    0.6597, 0.66, 0.6612, 0.662, 0.6625, 0.683, 0.6644, 0.675, 0.6658, 0.6652, 0.6650, 0.666, 0.6665,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L70_44_86spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1186, 0.11, 0.1086, 0.1, 0.0903, 0.09, 0.0888, 0.088, 0.0885, 0.08, 0.0760, 0.06, 0.0594, 0.04, 0.0356, 0.02, 0.0128, 0.0, -0.0144, -0.02, -0.0376, -0.035, -0.0356, 0.0, 0.0195, 0.15, 0.1214, 0.2, 0.2062, 0.24, 0.2585, 0.26, 0.2993, 0.3, 0.3307, 0.34, 0.3516, 0.36,
    0.3660, 0.37, 0.3838, 0.41, 0.4778, 0.55, 0.6974, 0.8, 0.8888, 0.95, 0.9927, 1.0, 1.0362, 1.04, 1.0559, 1.06, 1.0651, 1.066, 1.0694, 1.069, 1.0690, 1.07, 1.0713, 1.072,
    1.0733, 1.074, 1.0766, 1.076, 1.0791, 1.078, 1.0775, 1.079, 1.0794,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L70_44_86spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1186, 0.11, 0.1086, 0.1, 0.103, 0.09, 0.0888, 0.088, 0.0885, 0.08, 0.0760, 0.06, 0.0594, 0.04, 0.0356, 0.02, 0.0128, 0.0, -0.0144, -0.02, -0.0376, -0.035, -0.0356, 0.0, 0.0195, 0.15, 0.1214, 0.2, 0.2062, 0.24, 0.2585, 0.26, 0.2993, 0.3, 0.3307, 0.34, 0.3516, 0.36,
    0.3660, 0.37, 0.3838, 0.41, 0.4878, 0.55, 0.6974, 0.8, 0.8888, 0.95, 0.9927, 1.0, 1.0362, 1.04, 1.0559, 1.06, 1.0651, 1.066, 1.0694, 1.069, 1.0690, 1.07, 1.0713, 1.072,
    1.0733, 1.074, 1.0866, 1.076, 1.0891, 1.078, 1.0775, 1.079, 1.0794,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L38_42_19spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0438, 0.044, 0.0438, 0.0435, 0.0431, 0.05, 0.0562, 0.06, 0.0646, 0.065, 0.0661, 0.0662, 0.0662, 0.061, 0.0604, 0.055, 0.0506, 0.04, 0.0387, 0.03, 0.0275, 0.025, 0.0213, 0.022, 0.0233, 0.03, 0.0322, 0.035, 0.0397, 0.04, 0.0451, 0.05,
    0.0506, 0.052, 0.0543, 0.055, 0.0560, 0.0565, 0.0577, 0.06, 0.0633, 0.09, 0.1014, 0.15, 0.1919, 0.22, 0.2710, 0.3, 0.3138, 0.32, 0.3317, 0.335, 0.3394, 0.34, 0.3433, 0.345,
    0.3457, 0.346, 0.3461, 0.3465, 0.3469, 0.347, 0.3473, 0.3475, 0.3478, 0.348, 0.3482, 0.348, 0.3480, 0.349, 0.3497,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L38_42_19spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0438, 0.044, 0.0438, 0.0435, 0.0441, 0.05, 0.0562, 0.06, 0.0646, 0.065, 0.0661, 0.0662, 0.0662, 0.061, 0.0604, 0.055, 0.0506, 0.04, 0.0387, 0.03, 0.0275, 0.025, 0.0213, 0.022, 0.0233, 0.03, 0.0322, 0.035, 0.0397, 0.04, 0.0451, 0.05,
    0.0506, 0.052, 0.0543, 0.055, 0.0570, 0.0565, 0.0577, 0.06, 0.0633, 0.09, 0.1014, 0.15, 0.1919, 0.22, 0.2710, 0.3, 0.3138, 0.32, 0.3317, 0.335, 0.3394, 0.34, 0.3433, 0.345,
    0.3457, 0.346, 0.3461, 0.3465, 0.3569, 0.348, 0.3473, 0.3475, 0.3478, 0.348, 0.3482, 0.348, 0.3480, 0.349, 0.3497,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L60_63_85spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1186, 0.11, 0.1079, 0.1, 0.0890, 0.089, 0.0882, 0.089, 0.0895, 0.08, 0.0764, 0.06, 0.0573, 0.04, 0.0286, 0.0, -0.0013, -0.02, -0.0355, -0.05, -0.0640, -0.07,
    -0.0738, -0.06, -0.0504, 0.0, 0.0021, 0.03, 0.0514, 0.08, 0.0918, 0.1, 0.1301, 0.14, 0.1619, 0.17, 0.1847, 0.19, 0.2000, 0.21, 0.2206, 0.28, 0.3301, 0.4, 0.5869, 0.7,
    0.8108, 0.9, 0.9318, 0.94, 0.9824, 1.0, 1.0045, 1.01, 1.0146, 1.015, 1.0192, 1.093, 1.0192, 1.02, 1.0218, 1.022, 1.0241, 1.024, 1.0275, 1.03, 1.0300, 1.029, 1.0285, 1.03, 1.0302,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L60_63_85spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1186, 0.11, 0.1079, 0.1, 0.0900, 0.089, 0.0882, 0.089, 0.0895, 0.08, 0.0764, 0.06, 0.0573, 0.04, 0.0286, 0.0, -0.0013, -0.02, -0.0355, -0.05, -0.0640, -0.07,
    -0.0738, -0.06, -0.0504, 0.0, 0.0021, 0.03, 0.0514, 0.08, 0.0918, 0.1, 0.1301, 0.14, 0.1619, 0.17, 0.1847, 0.19, 0.2000, 0.21, 0.2206, 0.28, 0.3301, 0.4, 0.5869, 0.7,
    0.8108, 0.9, 0.9418, 0.94, 0.9924, 1.0, 1.0045, 1.01, 1.0146, 1.015, 1.0192, 1.093, 1.0192, 1.05, 1.0218, 1.022, 1.0241, 1.024, 1.0275, 1.03, 1.0300, 1.029, 1.0285, 1.03, 1.0302,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L80_75_30spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2541, 0.255, 0.2557, 0.255, 0.2543, 0.3, 0.3359, 0.35, 0.3885, 0.39, 0.3999, 0.4, 0.4038, 0.4, 0.3722, 0.35, 0.3159, 0.3, 0.2468, 0.2, 0.1816, 0.15, 0.1439, 0.15, 0.1522, 0.18, 0.1981, 0.21, 0.2359, 0.25, 0.2630, 0.28, 0.2910, 0.3,
    0.3087, 0.31, 0.3154, 0.32, 0.3236, 0.34, 0.3555, 0.4, 0.5755, 0.8, 1.0987, 1.4, 1.5560, 1.7, 1.8035, 1.9, 1.9070, 1.92, 1.9519, 1.96, 1.9747, 1.98, 1.9884, 1.99,
    1.9914, 1.994, 1.9958, 1.996, 1.9979, 2.0, 2.0004, 2.01, 2.0024, 2.02, 2.0013, 2.012, 2.0117,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L80_75_30spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2541, 0.255, 0.2557, 0.255, 0.2543, 0.3, 0.3359, 0.35, 0.3885, 0.39, 0.3999, 0.4, 0.4038, 0.4, 0.3722, 0.35, 0.3159, 0.3, 0.2468, 0.2, 0.1816, 0.15, 0.1439, 0.15, 0.1522, 0.18, 0.1981, 0.21, 0.2359, 0.25, 0.2630, 0.28, 0.2910, 0.3,
    0.3087, 0.31, 0.3154, 0.32, 0.3236, 0.36, 0.3555, 0.4, 0.5755, 0.8, 1.0987, 1.4, 1.5560, 1.7, 1.8035, 1.9, 1.9070, 1.92, 1.9519, 1.96, 1.9747, 1.98, 1.9884, 1.99,
    1.9984, 1.994, 1.9958, 1.986, 1.9979, 2.06, 2.0004, 2.01, 2.0024, 2.02, 2.0013, 2.012, 2.0117,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L28_m21_24spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0003, 0.0, 0.0008, 0.002, 0.0025, 0.003, 0.0049, 0.005, 0.0059, 0.006, 0.0075, 0.01, 0.0101, 0.012, 0.0133, 0.015, 0.0167, 0.018, 0.0199, 0.02, 0.0221, 0.023, 0.0266, 0.03, 0.0388, 0.05, 0.0577, 0.07, 0.0706, 0.073, 0.0736, 0.073,
    0.0728, 0.071, 0.0708, 0.07, 0.0686, 0.069, 0.0673, 0.068, 0.0657, 0.06, 0.0582, 0.05, 0.0407, 0.03, 0.0255, 0.02, 0.0174, 0.015, 0.0141, 0.013, 0.0129, 0.0126,
    0.0125, 0.0125, 0.0125, 0.0125, 0.0125, 0.0124, 0.0123, 0.0122, 0.0121, 0.012, 0.0118, 0.0117, 0.0117, 0.0118, 0.0118, 0.012, 0.0121,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L28_m21_24spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0003, 0.0, 0.0008, 0.002, 0.0025, 0.003, 0.0049, 0.005, 0.0059, 0.006, 0.0075, 0.01, 0.0101, 0.012, 0.0133, 0.015, 0.0167, 0.018, 0.0199, 0.02, 0.0221, 0.023, 0.0266, 0.03, 0.0388, 0.05, 0.0577, 0.07, 0.0706, 0.073, 0.0736, 0.073,
    0.0728, 0.071, 0.0728, 0.07, 0.0686, 0.069, 0.0653, 0.068, 0.0657, 0.06, 0.0582, 0.05, 0.0407, 0.03, 0.0255, 0.02, 0.0174, 0.015, 0.0141, 0.013, 0.0129, 0.0126,
    0.0125, 0.0125, 0.0135, 0.0125, 0.0125, 0.0124, 0.0123, 0.0122, 0.0121, 0.012, 0.0118, 0.0117, 0.0117, 0.0118, 0.0118, 0.012, 0.0121,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L45_m33_47spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0048, -0.004,  -0.0032, -0.002, -0.0011, -0.001, -0.0007, -0.001, -0.0021, -0.001, 0.0003, 0.003, 0.0040, 0.009, 0.0115, 0.02, 0.0213, 0.03, 0.0311, 0.032, 0.0385, 0.04, 0.0527, 0.08, 0.0890, 0.1, 0.1449, 0.15, 0.1843, 0.19, 0.1961, 0.1967,
    0.1967, 0.195, 0.1942, 0.193, 0.1905, 0.19, 0.1884, 0.185, 0.1842, 0.17, 0.1610, 0.13, 0.1059, 0.09, 0.0578, 0.04, 0.0323, 0.03, 0.0217, 0.02,
    0.0179, 0.018, 0.0164, 0.016, 0.0159, 0.0158, 0.0157, 0.0154, 0.0151, 0.015, 0.0147, 0.0143, 0.0142, 0.0142, 0.0141, 0.0142, 0.0142, 0.0143, 0.0143,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L45_m33_47spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0048, -0.004,  -0.0032, -0.002, -0.0011, -0.001, -0.0007, -0.001, -0.0021, -0.001, 0.0003, 0.003, 0.0040, 0.009, 0.0115, 0.02, 0.0213, 0.03, 0.0311, 0.032, 0.0385, 0.04, 0.0527, 0.08, 0.0890, 0.1, 0.1449, 0.15, 0.1843, 0.19, 0.1961, 0.1967,
    0.1967, 0.195, 0.1942, 0.193, 0.2005, 0.19, 0.1884, 0.185, 0.1842, 0.19, 0.1610, 0.13, 0.1059, 0.09, 0.0578, 0.04, 0.0323, 0.03, 0.0217, 0.02,
    0.0179, 0.018, 0.0164, 0.016, 0.0159, 0.0158, 0.0157, 0.0154, 0.0151, 0.015, 0.0157, 0.0143, 0.0142, 0.0142, 0.0141, 0.0142, 0.0142, 0.0143, 0.0143,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L26_m7_404spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0064, 0.008, 0.0094, 0.01, 0.0142, 0.02, 0.0258, 0.03, 0.0324, 0.033, 0.0375, 0.04, 0.0435, 0.044, 0.0475, 0.048, 0.0489, 0.049, 0.0495, 0.049, 0.0489, 0.049, 0.0491, 0.05, 0.0529, 0.053, 0.0595, 0.06, 0.0621, 0.055, 0.0590, 0.058, 0.0541, 0.05,
    0.0485, 0.045, 0.0432, 0.042, 0.0403, 0.04, 0.0390, 0.038, 0.0386, 0.0386, 0.0386, 0.0387, 0.0387, 0.0388, 0.0389, 0.039, 0.0390, 0.0391, 0.0392, 0.0395, 0.0396, 0.04,
    0.0403, 0.0402, 0.0407, 0.0406, 0.0406, 0.0404, 0.0402, 0.04, 0.0397, 0.0395, 0.0394, 0.0395, 0.0396, 0.04, 0.0406,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L34_m61_2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0084, -0.004, -0.0019, 0.0, 0.0084, 0.01, 0.0259, 0.03, 0.0352, 0.04, 0.0451, 0.05, 0.0580, 0.06, 0.0705, 0.075, 0.0800, 0.085, 0.0888, 0.09, 0.0942, 0.1, 0.1007, 0.11,
    0.1154, 0.12, 0.1367, 0.14, 0.1471, 0.142, 0.1409, 0.13, 0.1285, 0.12, 0.1150, 0.11, 0.1028, 0.1, 0.0957, 0.09, 0.0894, 0.07, 0.0660, 0.04, 0.0123, 0.0, -0.0345, -0.04,
    -0.0593, -0.06, -0.0697, -0.07, -0.0737, -0.074, -0.0749, -0.0745, -0.0746, -0.0744, -0.0742, -0.0748, -0.0749, -0.075, -0.0759, -0.077, -0.0773, -0.078, -0.0782, -0.078, -0.0777, -0.077, -0.0762,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L32_m16_17spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0034, 0.004, 0.0056, 0.007, 0.0089, 0.01, 0.0160, 0.014, 0.0197, 0.02, 0.0232, 0.025, 0.0279, 0.03, 0.0322, 0.033, 0.0355, 0.037, 0.0383, 0.039, 0.0397, 0.04, 0.0436, 0.05, 0.0560, 0.07, 0.0757, 0.08, 0.0884, 0.09,
    0.0900, 0.089, 0.0875, 0.085, 0.0837, 0.08, 0.0796, 0.078, 0.0774, 0.076, 0.0756, 0.07, 0.0698, 0.06, 0.0566, 0.05, 0.0452, 0.04, 0.0392, 0.036, 0.0367,
    0.0360, 0.036, 0.0359, 0.036, 0.0363, 0.0364, 0.0364, 0.0363, 0.0362, 0.036, 0.0359, 0.0356, 0.0355, 0.0353, 0.0352, 0.0353, 0.0354, 0.036, 0.0361,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L30_m19_15spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0020, 0.003, 0.0043, 0.006, 0.0078, 0.01, 0.0149, 0.016, 0.0187, 0.02, 0.0223, 0.025, 0.0270, 0.03, 0.0315, 0.032, 0.0349, 0.036, 0.0379, 0.038, 0.0394, 0.04, 0.0431, 0.05,
    0.0542, 0.06, 0.0715, 0.08, 0.0824, 0.083, 0.0832, 0.081, 0.0801, 0.078, 0.0759, 0.074, 0.0717, 0.07, 0.0693, 0.068, 0.0673, 0.064, 0.0608, 0.05, 0.0458, 0.04,
    0.0328, 0.03, 0.0260, 0.025, 0.0232, 0.023, 0.0223, 0.0221, 0.0221, 0.0223, 0.0224, 0.0225, 0.0226, 0.0224, 0.0224, 0.0221, 0.0220, 0.022, 0.0216, 0.0215, 0.0213, 0.0214, 0.0215, 0.02, 0.0222,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L30_m17_16spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0027, 0.003, 0.0048, 0.006, 0.0080, 0.01, 0.0148, 0.015, 0.0183, 0.02, 0.0216, 0.025, 0.0261, 0.03, 0.0303, 0.032, 0.0334, 0.035, 0.0361, 0.037, 0.0375, 0.04, 0.0411, 0.05, 0.0524, 0.06, 0.0700, 0.08, 0.0813, 0.082, 0.0825, 0.08, 0.0799, 0.077,
    0.0761, 0.075, 0.0722, 0.07, 0.0700, 0.069, 0.0682, 0.067, 0.0624, 0.055, 0.0493, 0.04, 0.0378, 0.035, 0.0318, 0.03, 0.0294, 0.029, 0.0286, 0.0285, 0.0285, 0.0287,
    0.0288, 0.029, 0.0290, 0.029, 0.0287, 0.0285, 0.0284, 0.0282, 0.0280, 0.028, 0.0278, 0.0279, 0.0279, 0.028, 0.0286,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L35_m8_4spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0109, 0.014, 0.0158, 0.02, 0.0233, 0.03, 0.0419, 0.05, 0.0525, 0.06, 0.0606, 0.07, 0.0702, 0.075, 0.0768, 0.078, 0.0793, 0.08, 0.0805, 0.08, 0.0795, 0.08, 0.0806, 0.085, 0.0889, 0.1, 0.1028, 0.105, 0.1094, 0.106, 0.1054, 0.1, 0.0979, 0.09,
    0.0891, 0.085, 0.0807, 0.08, 0.0761, 0.075, 0.0739, 0.073, 0.0730, 0.073, 0.0722, 0.072, 0.0718, 0.0716, 0.0718, 0.0718, 0.0719, 0.072, 0.0722, 0.0725, 0.0729, 0.073,
    0.0741, 0.0745, 0.0746, 0.0744, 0.0744, 0.074, 0.0739, 0.073, 0.0730, 0.073, 0.0725, 0.0727, 0.0728, 0.073, 0.0745,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L37_m7_5spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0126, 0.015, 0.0178, 0.02, 0.0258, 0.03, 0.0457, 0.05, 0.0572, 0.06, 0.0658, 0.07, 0.0760, 0.08, 0.0830, 0.084, 0.0856, 0.086, 0.0867, 0.086, 0.0856, 0.086, 0.0868, 0.09, 0.0961, 0.1, 0.1120, 0.113, 0.1199, 0.118, 0.1160, 0.11, 0.1081, 0.1,
    0.0989, 0.09, 0.0900, 0.088, 0.0851, 0.084, 0.0828, 0.0825, 0.0825, 0.083, 0.0831, 0.0835, 0.0839, 0.084, 0.0846, 0.0845, 0.0849, 0.085, 0.0854, 0.086, 0.0862, 0.087,
    0.0875, 0.088, 0.0881, 0.088, 0.0879, 0.0875, 0.0873, 0.087, 0.0864, 0.086, 0.0858, 0.086, 0.0862, 0.087, 0.0881,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L45_m7_2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0213, 0.03, 0.0302, 0.04, 0.0438, 0.06, 0.0780, 0.09, 0.0979, 0.1, 0.1125, 0.12, 0.1297, 0.13, 0.1411, 0.143, 0.1445, 0.1454, 0.1456, 0.143, 0.1428, 0.143, 0.1433, 0.15, 0.1548, 0.16, 0.1750, 0.18, 0.1833, 0.18, 0.1749, 0.17, 0.1610, 0.15,
    0.1452, 0.14, 0.1301, 0.13, 0.1219, 0.12, 0.1183, 0.119, 0.1197, 0.12, 0.1254, 0.13, 0.1308, 0.133, 0.1342, 0.135, 0.1356, 0.136, 0.1367, 0.137, 0.1383, 0.14, 0.1405, 0.141,
    0.1416, 0.1415, 0.1413, 0.141, 0.1403, 0.14, 0.1387, 0.138, 0.1378, 0.138, 0.1384, 0.14, 0.1416,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_m6_5spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0164, 0.02, 0.0226, 0.03, 0.0320, 0.04, 0.0562, 0.06, 0.0701, 0.08, 0.0804, 0.09, 0.0925, 0.1, 0.1006, 0.103, 0.1031, 0.1035, 0.1039, 0.103, 0.1021, 0.103, 0.1030, 0.11, 0.1135, 0.12, 0.1316, 0.13, 0.1403, 0.14, 0.1356, 0.13, 0.1264, 0.12, 0.1155, 0.11, 0.1050, 0.1,
    0.0992, 0.097, 0.0968, 0.097, 0.0979, 0.1, 0.1024, 0.103, 0.1065, 0.108, 0.1091, 0.11, 0.1102, 0.111, 0.1110, 0.111, 0.1122, 0.113, 0.1138, 0.114, 0.1145, 0.1145, 0.1143, 0.114,
    0.1136, 0.113, 0.1125, 0.112, 0.1119, 0.112, 0.1123, 0.113, 0.1146,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L46_m6_2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0233, 0.03, 0.0322, 0.04, 0.0460, 0.06, 0.0811, 0.1, 0.1015, 0.11, 0.1163, 0.12, 0.1338, 0.14, 0.1451, 0.148, 0.1483, 0.145, 0.1489, 0.146, 0.1458, 0.146, 0.1461, 0.15, 0.1582, 0.17, 0.1795, 0.18, 0.1886, 0.181, 0.1804, 0.17, 0.1666, 0.16,
    0.1508, 0.14, 0.1356, 0.13, 0.1273, 0.125, 0.1239, 0.124, 0.1266, 0.13, 0.1356, 0.14, 0.1438, 0.145, 0.1488, 0.15, 0.1509, 0.152, 0.1523, 0.153, 0.1540, 0.155, 0.1563, 0.157,
    0.1574, 0.156, 0.1571, 0.157, 0.1562, 0.56, 0.1546, 0.154, 0.1537, 0.154, 0.1542, 0.156, 0.1576,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L48_m69_16spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0181, -0.01, -0.0091, 0.0, 0.0052, 0.01, 0.0259, 0.03, 0.0360, 0.04, 0.0492, 0.06, 0.0677, 0.08, 0.0879, 0.1, 0.1059, 0.11, 0.1233, 0.13, 0.1351, 0.14, 0.1511, 0.17, 0.1875, 0.2, 0.2413, 0.26, 0.2732, 0.271, 0.2706, 0.26, 0.2546, 0.24,
    0.2358, 0.22, 0.2184, 0.21, 0.2082, 0.2, 0.1975, 0.17, 0.1520, 0.06, 0.0462, 0.0, -0.0459, -0.06, -0.0950, -0.1, -0.1154, -0.12, -0.1232, -0.124, -0.1260, -0.126,
    -0.1261, -0.126, -0.1257, -0.126, -0.1270, -0.128, -0.1285, -0.13, -0.1306, -0.132, -0.1318, -0.1312, -0.1311, -0.13, -0.1294,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L51_89_53spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1314, 0.13, 0.1218, 0.11, 0.1046, 0.11, 0.1137, 0.12, 0.1221, 0.12, 0.1115, 0.1, 0.0933, 0.08, 0.0607, 0.04, 0.0225, 0.0, -0.0208, -0.04, -0.0572, -0.06, -0.0798, -0.08, -0.0852, -0.08, -0.0759, -0.06, -0.0588, -0.04, -0.0307, 0.0, 0.0037, 0.0036, 0.0336, 0.04,
    0.0557, 0.06, 0.0706, 0.08, 0.0944, 0.15, 0.2271, 0.4, 0.5396, 0.7, 0.8124, 0.9, 0.9595, 1.0, 1.0210, 1.03, 1.0472, 1.05, 1.0593, 1.06, 1.0651, 1.0656, 1.0656, 1.067,
    1.0686, 1.07, 1.0711, 1.073, 1.0745, 1.075, 1.0768, 1.076, 1.0754, 1.077, 1.0780,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L49_84_33spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1189, 0.115, 0.1130, 0.11, 0.1020, 0.11, 0.1198, 0.125, 0.1331, 0.13, 0.1281, 0.12, 0.1169, 0.1, 0.0909, 0.07, 0.0573, 0.04, 0.0189, 0.0, -0.0145, -0.02, -0.0366, -0.04,
    -0.0456, -0.044, -0.0438, -0.04, -0.0353, -0.02, -0.0165, 0.0, 0.0084, 0.01, 0.0297, 0.03, 0.0447, 0.05, 0.0554, 0.06, 0.0754, 0.1, 0.1935, 0.3, 0.4724, 0.6, 0.7159, 0.75,
    0.8473, 0.9, 0.9022, 0.91, 0.9255, 0.92, 0.9366, 0.94, 0.9423, 0.943, 0.9431, 0.944, 0.9457, 0.946, 0.9476, 0.95, 0.9501, 0.951, 0.9518, 0.951, 0.9508, 0.952, 0.9539,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L59_m51_31spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0123, -0.002, -0.0032, 0.0, 0.0108, 0.02, 0.0316, 0.04, 0.0410, 0.05, 0.0543, 0.06, 0.0737, 0.08, 0.0961, 0.1, 0.1175, 0.12, 0.1380, 0.14, 0.1519, 0.16, 0.1752, 0.2, 0.2340, 0.3, 0.3232, 0.35, 0.3810, 0.385,
    0.3880, 0.38, 0.3752, 0.36, 0.3573, 0.34, 0.3395, 0.33, 0.3292, 0.32, 0.3176, 0.3, 0.2669, 0.2, 0.1484, 0.1, 0.0451, 0.0, -0.0097, -0.02, -0.0324, -0.04, -0.0408, -0.042,
    -0.0437, -0.0436, -0.0438, -0.0436, -0.0436, -0.045, -0.0450, -0.046, -0.0465, -0.047, -0.0486, -0.049, -0.0497, -0.0493, -0.0491, -0.048, -0.0472,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L48_m69_16spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0181, -0.02, -0.0090, 0.004, 0.0052, 0.01, 0.0259, 0.03, 0.0361, 0.04, 0.0493, 0.05, 0.0678, 0.07, 0.0880, 0.1, 0.1060, 0.11, 0.1235, 0.13, 0.1353, 0.14, 0.1513, 0.17, 0.1879, 0.21, 0.2419, 0.25, 0.2739, 0.273, 0.2713, 0.26, 0.2553, 0.24, 0.2366, 0.22,
    0.2191, 0.21, 0.2089, 0.2, 0.1982, 0.17, 0.1527, 0.08, 0.0468, 0.0, -0.0455, -0.05, -0.0946, -0.1, -0.1150, -0.12, -0.1228, -0.123, -0.1256, -0.125, -0.1258, -0.1255,
    -0.1254, -0.125, -0.1267, -0.127, -0.1281, -0.13, -0.1303, -0.131, -0.1315, -0.131, -0.1308, -0.13, -0.1291,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L53_m71_6spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0168, -0.015, -0.0015, 0.01, 0.0226, 0.05, 0.0640, 0.07, 0.0860, 0.1, 0.1093, 0.12, 0.1397, 0.15, 0.1692, 0.18, 0.1916, 0.2, 0.2125, 0.22, 0.2250, 0.23, 0.2414, 0.27, 0.2802, 0.3, 0.3375, 0.35, 0.3670, 0.35, 0.3546, 0.33,
    0.3264, 0.3, 0.2951, 0.27, 0.2665, 0.25, 0.2499, 0.24, 0.2353, 0.2, 0.1811, 0.1, 0.0565, 0.0, -0.0519, -0.08, -0.1095, -0.1, -0.1334, -0.14, -0.1425, -0.143,
    -0.1453, -0.145, -0.1446, -0.144, -0.1435, -0.144, -0.1452, -0.146, -0.1475, -0.15, -0.1509, -0.151, -0.1528, -0.153, -0.1516, -0.15, -0.1482,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L51_m89_53spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1331, 0.13, 0.1233, 0.11, 0.1059, 0.11, 0.1151, 0.12, 0.1236, 0.12, 0.1129, 0.1, 0.0946, 0.07, 0.0615, 0.05, 0.0228, 0.0, -0.0210, -0.04, -0.0579, -0.07, -0.0808, -0.08, -0.0863, -0.08, -0.0769, -0.06, -0.0595, -0.04, -0.0311, 0.001, 0.0037, 0.035,
    0.0341, 0.04, 0.0564, 0.06, 0.0716, 0.08, 0.0956, 0.15, 0.2300, 0.4, 0.5466, 0.7, 0.8229, 0.9, 0.9720, 1.0, 1.0342, 1.05, 1.0608, 1.07, 1.0730, 1.075, 1.0789, 1.079,
    1.0794, 1.08, 1.0825, 1.083, 1.0850, 1.088, 1.0884, 1.09, 1.0908, 1.09, 1.0894, 1.09, 1.0920,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L49_84_33spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1189, 0.125, 0.1130, 0.11, 0.1020, 0.11, 0.1198, 0.125, 0.1331, 0.13, 0.1281, 0.12, 0.1369, 0.1, 0.0909, 0.07, 0.0573, 0.04, 0.0189, 0.0, -0.0145, -0.02, -0.0366, -0.04,
    -0.0456, -0.044, -0.0438, -0.04, -0.0353, -0.02, -0.0165, 0.0, 0.0084, 0.01, 0.0297, 0.03, 0.0447, 0.05, 0.0554, 0.06, 0.0754, 0.1, 0.1935, 0.3, 0.4724, 0.6, 0.7159, 0.75,
    0.8473, 0.9, 0.9122, 0.91, 0.9255, 0.92, 0.9366, 0.94, 0.9423, 0.943, 0.9431, 0.944, 0.9457, 0.946, 0.9476, 0.95, 0.9501, 0.951, 0.9518, 0.951, 0.9508, 0.952, 0.9539,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L36_m27_28spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0012, 0.0, 0.0007, 0.004, 0.0036, 0.007, 0.0079, 0.009, 0.0095, 0.01, 0.0123, 0.012, 0.0168, 0.02, 0.0223, 0.025, 0.0280, 0.03, 0.0335, 0.035, 0.0373, 0.04, 0.0447, 0.06, 0.0645, 0.08, 0.0949, 0.1, 0.1156, 0.12, 0.1202, 0.12,
    0.1185, 0.115, 0.1149, 0.113, 0.1110, 0.11, 0.1088, 0.107, 0.1060, 0.1, 0.0931, 0.08, 0.0628, 0.05, 0.0363, 0.03, 0.0224, 0.02, 0.0166, 0.015, 0.0145, 0.014, 0.0138, 0.0138,
    0.0138, 0.0138, 0.0138, 0.0135, 0.0134, 0.0132, 0.0131, 0.013, 0.0126, 0.0125, 0.0124, 0.0125, 0.0125, 0.013, 0.0130,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L36_m27_28spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0012, 0.0, 0.0007, 0.004, 0.0036, 0.007, 0.0079, 0.009, 0.0095, 0.01, 0.0153, 0.012, 0.0168, 0.02, 0.0223, 0.025, 0.0280, 0.03, 0.0335, 0.035, 0.0373, 0.04, 0.0447, 0.06, 0.0645, 0.08, 0.0949, 0.1, 0.1156, 0.12, 0.1202, 0.12,
    0.1185, 0.115, 0.1249, 0.113, 0.1110, 0.11, 0.1088, 0.107, 0.1060, 0.1, 0.0931, 0.08, 0.0628, 0.05, 0.0363, 0.03, 0.0224, 0.02, 0.0166, 0.015, 0.0145, 0.014, 0.0138, 0.0138,
    0.0138, 0.0138, 0.0138, 0.0135, 0.0134, 0.0132, 0.0131, 0.013, 0.0136, 0.0125, 0.0124, 0.0125, 0.0125, 0.013, 0.0130,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L36_m27_28spect3[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0012, 0.0, 0.0007, 0.004, 0.0036, 0.007, 0.0079, 0.009, 0.0095, 0.01, 0.0153, 0.012, 0.0168, 0.02, 0.0223, 0.025, 0.0280, 0.03, 0.0335, 0.035, 0.0373, 0.04, 0.0447, 0.06, 0.0645, 0.08, 0.0949, 0.1, 0.1156, 0.12, 0.1202, 0.12,
    0.1185, 0.115, 0.1049, 0.113, 0.1110, 0.11, 0.1088, 0.107, 0.1060, 0.1, 0.0931, 0.08, 0.0628, 0.05, 0.0363, 0.03, 0.0224, 0.02, 0.0166, 0.015, 0.0145, 0.014, 0.0138, 0.0138,
    0.0138, 0.0138, 0.0138, 0.0135, 0.0134, 0.0132, 0.0131, 0.012, 0.0136, 0.0115, 0.0124, 0.0125, 0.0125, 0.013, 0.0130,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L63_16_71spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0572, 0.055, 0.0533, 0.05, 0.0456, 0.0455, 0.0455, 0.045, 0.0446, 0.04, 0.0397, 0.035, 0.0344, 0.03, 0.0278, 0.025, 0.0233, 0.02, 0.0164, 0.01, 0.0099, 0.02, 0.0210, 0.05, 0.0751, 0.1, 0.1672, 0.2, 0.2386, 0.25, 0.2735, 0.28,
    0.2946, 0.3, 0.308, 0.31, 0.3164, 0.32, 0.3222, 0.325, 0.3282, 0.35, 0.3595, 0.4, 0.4319, 0.45, 0.4950, 0.5, 0.5297, 0.53, 0.5443, 0.55, 0.5516, 0.553, 0.5552, 0.556,
    0.5571, 0.557, 0.5568, 0.557, 0.5575, 0.558, 0.5582, 0.559, 0.5596, 0.56, 0.5608, 0.56, 0.5600, 0.56, 0.5611,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L84_4_46spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0967, 0.1, 0.1060, 0.11, 0.1183, 0.14, 0.1696, 0.18, 0.1976, 0.2, 0.2146, 0.22, 0.2361, 0.24, 0.2486, 0.25, 0.2527, 0.251, 0.2504, 0.25, 0.2420, 0.25, 0.2618, 0.3, 0.3581, 0.4, 0.5200, 0.6, 0.6329, 0.65, 0.6662, 0.67,
    0.6704, 0.67, 0.6620, 0.65, 0.6470, 0.645, 0.6405, 0.6402, 0.6404, 0.65, 0.6689, 0.7, 0.7383, 0.75, 0.7991, 0.8, 0.8336, 0.84, 0.8483, 0.85, 0.8569, 0.86, 0.8627, 0.865, 0.8680, 0.868, 0.8691, 0.869, 0.8693, 0.869,
    0.8685, 0.868, 0.8675, 0.8675, 0.8675, 0.8675, 0.8674, 0.87, 0.8737,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L84_4_46spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0967, 0.1, 0.1060, 0.12, 0.1183, 0.14, 0.1696, 0.18, 0.1976, 0.2, 0.2146, 0.22, 0.2361, 0.24, 0.2486, 0.25, 0.2527, 0.251, 0.2504, 0.25, 0.2420, 0.25, 0.2618, 0.3, 0.3581, 0.4, 0.5200, 0.6, 0.6329, 0.65, 0.6662, 0.67,
    0.6704, 0.67, 0.6620, 0.65, 0.6570, 0.645, 0.6405, 0.6402, 0.6404, 0.65, 0.6689, 0.7, 0.7383, 0.75, 0.7991, 0.8, 0.8336, 0.84, 0.8483, 0.85, 0.8569, 0.86, 0.8627, 0.865, 0.8680, 0.868, 0.8691, 0.869, 0.8693, 0.869,
    0.8685, 0.868, 0.8675, 0.8685, 0.8675, 0.8675, 0.8674, 0.87, 0.8737,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L75_m66_19spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0156, 0.0, 0.0098, 0.02, 0.0496, 0.1, 0.1201, 0.13, 0.1571, 0.18, 0.1958, 0.22, 0.2475, 0.28, 0.2978, 0.31, 0.3372, 0.35, 0.3735, 0.38, 0.3950, 0.4, 0.4291, 0.5, 0.5173, 0.6, 0.6511, 0.7, 0.7285, 0.72, 0.7189, 0.7, 0.6762, 0.67, 0.6258, 0.6, 0.5783, 0.56, 0.5509, 0.53,
    0.5265, 0.5, 0.4363, 0.3, 0.2287, 0.1, 0.0482, 0.0, -0.0474, -0.06, -0.0871, -0.1, -0.1017, -0.105, -0.1059, -0.105, -0.1043, -0.103, -0.1026, -0.104, -0.1055, -0.108,
    -0.1092, -0.11, -0.1148, -0.115, -0.1179, -0.116, -0.1160, -0.11, -0.1098,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L75_m66_19spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0156, 0.0, 0.0098, 0.03, 0.0496, 0.1, 0.1201, 0.13, 0.1571, 0.18, 0.1958, 0.22, 0.2475, 0.28, 0.2978, 0.31, 0.3372, 0.35, 0.3735, 0.38, 0.3950, 0.4, 0.4291, 0.5, 0.5173, 0.6, 0.6511, 0.7, 0.7285, 0.72, 0.7189, 0.7, 0.6762, 0.67, 0.6258, 0.6, 0.5783, 0.56, 0.5509, 0.53,
    0.5265, 0.5, 0.4363, 0.3, 0.2587, 0.1, 0.0482, 0.0, -0.0494, -0.06, -0.0871, -0.1, -0.1017, -0.105, -0.1059, -0.105, -0.1043, -0.103, -0.1026, -0.104, -0.1055, -0.108,
    -0.1092, -0.11, -0.1148, -0.115, -0.1179, -0.116, -0.1160, -0.11, -0.1098,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L64_m82_m6spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0171, 0.01, 0.0138, 0.05, 0.0628, 0.1, 0.1546, 0.2, 0.2056, 0.22, 0.2535, 0.3, 0.3138, 0.35, 0.3665, 0.4, 0.4005, 0.42, 0.4306, 0.44, 0.4457, 0.45, 0.4630, 0.5, 0.5062, 0.53, 0.5693, 0.58, 0.5909, 0.57, 0.5532, 0.5, 0.4940, 0.45, 0.4313, 0.4,
    0.3751, 0.35, 0.3425, 0.33, 0.3178, 0.3, 0.2396, 0.1, 0.0625, 0.0, -0.0910, -0.1, -0.1723, -0.2, -0.2062, -0.21, -0.2191, -0.22, -0.2221, -0.22, -0.2193, -0.219, -0.2165, -0.218,
    -0.2193, -0.22, -0.2235, -0.23, -0.2302, -0.232, -0.2341, -0.232, -0.2317, -0.23, -0.2240,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L64_m82_m6spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    -0.0171, 0.01, 0.0138, 0.05, 0.0648, 0.1, 0.1546, 0.2, 0.2056, 0.22, 0.2535, 0.3, 0.3138, 0.35, 0.3665, 0.4, 0.4105, 0.42, 0.4306, 0.44, 0.4457, 0.45, 0.4630, 0.5, 0.5062, 0.53, 0.5693, 0.58, 0.5909, 0.57, 0.5532, 0.5, 0.4940, 0.45, 0.4313, 0.4,
    0.3751, 0.35, 0.3425, 0.33, 0.3378, 0.3, 0.2396, 0.1, 0.0625, 0.0, -0.0910, -0.1, -0.1723, -0.2, -0.2062, -0.21, -0.2191, -0.22, -0.2221, -0.22, -0.2193, -0.219, -0.2165, -0.218,
    -0.2193, -0.22, -0.2235, -0.23, -0.2302, -0.232, -0.2341, -0.232, -0.2317, -0.23, -0.2240,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L66_m71_m17spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0007, 0.03, 0.0403, 0.08, 0.1029, 0.2, 0.2297, 0.25, 0.3019, 0.33, 0.3645, 0.4, 0.4407, 0.5, 0.5018, 0.52, 0.5344, 0.55, 0.5612, 0.57, 0.5702, 0.574, 0.5790, 0.6, 0.6080, 0.63, 0.6506, 0.65, 0.6510, 0.6, 0.5942, 0.55, 0.5179, 0.5, 0.4388, 0.4,
    0.3682, 0.34, 0.3275, 0.31, 0.3009, 0.25, 0.2329, 0.1, 0.0832, 0.0, -0.0461, -0.1, -0.1142, -0.12, -0.1425, -0.15, -0.1531, -0.154, -0.1542, -0.15, -0.1490, -0.145,
    -0.1449, -0.148, -0.1478, -0.15, -0.1529, -0.16, -0.1611, -0.165, -0.1661, -0.164, -0.1630, -0.16, -0.1522,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L66_m71_m17spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0007, 0.03, 0.0403, 0.08, 0.1429, 0.2, 0.2297, 0.25, 0.3019, 0.33, 0.3645, 0.4, 0.4407, 0.54, 0.5018, 0.52, 0.5344, 0.55, 0.5612, 0.57, 0.5702, 0.574, 0.5790, 0.6, 0.6080, 0.63, 0.6506, 0.65, 0.6510, 0.6, 0.5942, 0.55, 0.5179, 0.5, 0.4388, 0.4,
    0.3682, 0.34, 0.3275, 0.31, 0.3009, 0.28, 0.2329, 0.1, 0.0832, 0.0, -0.0461, -0.1, -0.1142, -0.12, -0.1425, -0.15, -0.1531, -0.154, -0.1542, -0.15, -0.1490, -0.145,
    -0.1449, -0.148, -0.1478, -0.15, -0.1529, -0.16, -0.1611, -0.165, -0.1661, -0.164, -0.1630, -0.16, -0.1522,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L22_m8_m60spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0228, 0.03, 0.0416, 0.06, 0.0715, 0.1, 0.1422, 0.15, 0.1848, 0.2, 0.2159, 0.24, 0.2505, 0.26, 0.2709, 0.272, 0.2720, 0.27, 0.2693, 0.265, 0.2605, 0.25, 0.2450, 0.22, 0.2153, 0.2, 0.1709, 0.15, 0.1245, 0.1, 0.0821, 0.06, 0.0434, 0.02, 0.0065, 0.0,
    -0.0251, -0.03, -0.0432, -0.045, -0.0497, -0.043, -0.0419, -0.02, -0.0173, 0.0, 0.0051, 0.01, 0.0174, 0.03, 0.0226, 0.0225, 0.0247, 0.025, 0.0275, 0.03, 0.0317, 0.032,
    0.0343, 0.034, 0.0338, 0.032, 0.0318, 0.03, 0.0282, 0.027, 0.0259, 0.027, 0.0274, 0.03, 0.0335,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L22_m8_m60spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0228, 0.03, 0.0416, 0.07, 0.0715, 0.1, 0.1422, 0.15, 0.1848, 0.2, 0.2159, 0.26, 0.2505, 0.26, 0.2709, 0.272, 0.2720, 0.27, 0.2693, 0.265, 0.2605, 0.25, 0.2450, 0.22, 0.2153, 0.2, 0.1709, 0.15, 0.1245, 0.1, 0.0821, 0.06, 0.0434, 0.02, 0.0065, 0.0,
    -0.0251, -0.03, -0.0432, -0.045, -0.0497, -0.043, -0.0419, -0.025, -0.0173, 0.0, 0.0051, 0.01, 0.0174, 0.03, 0.0226, 0.0225, 0.0247, 0.025, 0.0275, 0.03, 0.0317, 0.032,
    0.0343, 0.034, 0.0338, 0.032, 0.0318, 0.03, 0.0292, 0.027, 0.0259, 0.027, 0.0274, 0.03, 0.0335,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L15_m4_m42spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0109, 0.015, 0.0194, 0.02, 0.0328, 0.05, 0.0647, 0.07, 0.0839, 0.09, 0.0978, 0.1, 0.1134, 0.12, 0.1225, 0.123, 0.123, 0.122, 0.1217, 0.12, 0.1177, 0.115, 0.1108, 0.1, 0.0982, 0.08, 0.0795, 0.06, 0.0595, 0.05, 0.0408, 0.03, 0.0236, 0.01,
    0.0072, 0.007, -0.0070, -0.01, -0.0151, -0.016, -0.0180, -0.015, -0.0141, -0.001, -0.0023, 0.0, 0.0084, 0.01, 0.0143, 0.015, 0.0168, 0.017, 0.0178, 0.018, 0.0191, 0.02,
    0.0210, 0.022, 0.0222, 0.022, 0.0220, 0.022, 0.0211, 0.02, 0.0195, 0.019, 0.0185, 0.019, 0.0191, 0.02, 0.0219,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L15_m4_m42spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0109, 0.015, 0.0184, 0.02, 0.0328, 0.05, 0.0647, 0.07, 0.0839, 0.09, 0.0978, 0.1, 0.1134, 0.12, 0.1325, 0.123, 0.123, 0.122, 0.1217, 0.12, 0.1177, 0.115, 0.1108, 0.1, 0.0982, 0.08, 0.0795, 0.06, 0.0595, 0.05, 0.0408, 0.03, 0.0236, 0.01,
    0.0072, 0.007, -0.0070, -0.01, -0.0151, -0.016, -0.0180, -0.025, -0.0141, -0.001, -0.0023, 0.0, 0.0084, 0.01, 0.0143, 0.015, 0.0168, 0.017, 0.0178, 0.018, 0.0191, 0.02,
    0.0210, 0.022, 0.0222, 0.022, 0.0220, 0.022, 0.0211, 0.02, 0.0195, 0.019, 0.0185, 0.019, 0.0191, 0.02, 0.0219,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L13_3_m23spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0064, 0.007, 0.0098, 0.01, 0.0153, 0.02, 0.0290, 0.03, 0.0372, 0.04, 0.0430, 0.045, 0.0495, 0.05, 0.0531, 0.0531, 0.0531, 0.053, 0.0523, 0.051, 0.0503, 0.05, 0.0475, 0.045,
    0.0433, 0.04, 0.0373, 0.031, 0.0303, 0.025, 0.0232, 0.02, 0.0164, 0.01, 0.0098, 0.005, 0.0039, 0.001, 0.0007, 0.0, -0.0004, 0.0, 0.0024, 0.01, 0.0101, 0.012, 0.0170, 0.02, 0.0208, 0.023, 0.0223, 0.023, 0.0230, 0.0231, 0.0237, 0.024,
    0.0246, 0.025, 0.0251, 0.025, 0.0250, 0.0248, 0.0247, 0.0245, 0.0240, 0.024, 0.0236, 0.0236, 0.0239, 0.024, 0.0251,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L27_4_m90spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0543, 0.08, 0.0957, 0.12, 0.1617, 0.25, 0.3197, 0.35, 0.4152, 0.45, 0.4840, 0.5, 0.5600, 0.6, 0.6032, 0.603, 0.6029, 0.6, 0.5940, 0.595, 0.5722, 0.55, 0.5340, 0.5, 0.4601, 0.4, 0.3492, 0.3, 0.2374, 0.15, 0.1408, 0.1,
    0.0549, 0.0, -0.0262, -0.05, -0.0957, -0.1, -0.1353, -0.14, -0.1485, -0.13, -0.1237, -0.1, -0.0512, 0.0, 0.0139, 0.04, 0.0497, 0.05, 0.0646, 0.07, 0.0706, 0.071, 0.0773, 0.08,
    0.0870, 0.09, 0.0929, 0.092, 0.0920, 0.09, 0.0876, 0.08, 0.0798, 0.077, 0.0747, 0.076, 0.0781, 0.08, 0.0916,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L19_1_m29spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0109, 0.015, 0.0175, 0.02, 0.0278, 0.04, 0.0531, 0.06, 0.0682, 0.07, 0.0791, 0.08, 0.0912, 0.095, 0.0982, 0.098, 0.0983, 0.098, 0.0971, 0.095, 0.0936, 0.09, 0.0886, 0.085, 0.0806, 0.07, 0.0692, 0.06, 0.0561, 0.05, 0.0427, 0.03, 0.0299, 0.02, 0.0174, 0.01,
    0.0064, 0.003, 0.0002, 0.0, -0.0018, 0.0, 0.0025, 0.01, 0.0148, 0.02, 0.0259, 0.03, 0.0320, 0.033, 0.0346, 0.035, 0.0357, 0.036, 0.0368, 0.037, 0.0384, 0.039,
    0.0394, 0.0393, 0.0392, 0.039, 0.0385, 0.038, 0.0373, 0.037, 0.0366, 0.037, 0.0371, 0.038, 0.0393,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L27_4_m90spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0543, 0.09, 0.0957, 0.12, 0.1617, 0.28, 0.3197, 0.35, 0.4152, 0.45, 0.4840, 0.53, 0.5600, 0.6, 0.6032, 0.603, 0.6029, 0.6, 0.5940, 0.595, 0.5722, 0.55, 0.5340, 0.5, 0.4601, 0.4, 0.3492, 0.3, 0.2374, 0.15, 0.1408, 0.1,
    0.0549, 0.0, -0.0262, -0.05, -0.0957, -0.1, -0.1353, -0.15, -0.1485, -0.13, -0.1237, -0.1, -0.0512, 0.0, 0.0139, 0.04, 0.0497, 0.05, 0.0646, 0.07, 0.0706, 0.071, 0.0773, 0.08,
    0.0870, 0.09, 0.0929, 0.092, 0.0920, 0.09, 0.0876, 0.08, 0.0798, 0.077, 0.0747, 0.076, 0.0781, 0.08, 0.0916,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L16_0_m44spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0128, 0.02, 0.0219, 0.03, 0.0364, 0.06, 0.0714, 0.08, 0.0924, 0.1, 0.1076, 0.11, 0.1244, 0.13, 0.1341, 0.1341, 0.1343, 0.133, 0.1325, 0.13, 0.1278, 0.125, 0.1200, 0.11, 0.1057, 0.1, 0.0845, 0.07, 0.0623, 0.05, 0.0419, 0.03, 0.0232, 0.01,
    0.0055, 0.0, -0.0099, -0.01, -0.0186, -0.02, -0.0216, -0.02, -0.0163, -0.01, -0.0009, 0.0, 0.0130, 0.02, 0.0207, 0.022, 0.0239, 0.024, 0.0252, 0.026, 0.0267, 0.027,
    0.0288, 0.03, 0.0301, 0.03, 0.0299, 0.029, 0.0290, 0.028, 0.0273, 0.027, 0.0261, 0.0265, 0.0269, 0.028, 0.0299,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L16_0_m44spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0128, 0.02, 0.0219, 0.03, 0.0384, 0.06, 0.0714, 0.08, 0.0924, 0.1, 0.1076, 0.11, 0.1244, 0.13, 0.1371, 0.1341, 0.1343, 0.133, 0.1325, 0.13, 0.1278, 0.125, 0.1200, 0.11, 0.1057, 0.1, 0.0845, 0.07, 0.0623, 0.05, 0.0419, 0.03, 0.0232, 0.01,
    0.0055, 0.0, -0.0099, -0.01, -0.0186, -0.02, -0.0216, -0.02, -0.0163, -0.01, -0.0009, 0.0, 0.0130, 0.02, 0.0207, 0.022, 0.0239, 0.024, 0.0252, 0.026, 0.0267, 0.027,
    0.0288, 0.03, 0.0301, 0.03, 0.0299, 0.029, 0.0290, 0.028, 0.0283, 0.027, 0.0261, 0.0265, 0.0269, 0.028, 0.0299,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L13_m3_m36spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0078, 0.01, 0.0136, 0.02, 0.0228, 0.03, 0.0448, 0.05, 0.0581, 0.06, 0.0677, 0.07, 0.0784, 0.08, 0.0847, 0.085, 0.0850, 0.085, 0.0841, 0.083, 0.0813, 0.08, 0.0767, 0.07, 0.0682, 0.06, 0.0556, 0.05, 0.0420, 0.03, 0.0293, 0.02, 0.0175, 0.01,
    0.0062, 0.0, -0.0036, -0.005, -0.0092, -0.01, -0.0112, -0.003, -0.0084, -0.0001, -0.0001, 0.0, 0.0075, 0.01, 0.0117, 0.012, 0.0135, 0.014, 0.0142, 0.015, 0.0151, 0.016,
    0.0164, 0.017, 0.0172, 0.0171, 0.0171, 0.017, 0.0165, 0.016, 0.0154, 0.015, 0.0147, 0.015, 0.0151, 0.016, 0.0170,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L13_m3_m36spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0078, 0.01, 0.0136, 0.03, 0.0228, 0.03, 0.0448, 0.05, 0.0581, 0.06, 0.0677, 0.07, 0.0794, 0.08, 0.0847, 0.089, 0.0850, 0.085, 0.0841, 0.083, 0.0813, 0.08, 0.0767, 0.07, 0.0682, 0.06, 0.0556, 0.05, 0.0420, 0.03, 0.0293, 0.02, 0.0175, 0.01,
    0.0062, 0.0, -0.0036, -0.005, -0.0092, -0.01, -0.0112, -0.003, -0.0084, -0.0001, -0.0001, 0.0, 0.0085, 0.01, 0.0117, 0.012, 0.0135, 0.014, 0.0142, 0.015, 0.0151, 0.016,
    0.0164, 0.017, 0.0172, 0.0171, 0.0171, 0.017, 0.0165, 0.016, 0.0154, 0.015, 0.0147, 0.015, 0.0151, 0.016, 0.0170,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L31_m23_m60spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0283, 0.04, 0.0551, 0.07, 0.0978, 0.12, 0.1970, 0.23, 0.2564, 0.28, 0.3006, 0.34, 0.3502, 0.37, 0.3806, 0.381, 0.3845, 0.383, 0.3832, 0.38, 0.3728, 0.36, 0.3536, 0.33, 0.3176, 0.3, 0.2634, 0.25, 0.2037, 0.17, 0.1453, 0.1, 0.0903, 0.05,
    0.0375, 0.0, -0.0080, -0.05, -0.0340, -0.04, -0.0443, -0.04, -0.0399, -0.03, -0.0206, -0.02, -0.0026, 0.0, 0.0076, 0.01, 0.0119, 0.012, 0.0137, 0.015, 0.0171, 0.02, 0.0228, 0.025,
    0.0265, 0.026, 0.0256, 0.023, 0.0226, 0.02, 0.0175, 0.016, 0.0142, 0.016, 0.0164, 0.02, 0.0248,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L31_m23_m60spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0283, 0.04, 0.0551, 0.08, 0.0978, 0.12, 0.1970, 0.23, 0.2564, 0.28, 0.3006, 0.34, 0.3802, 0.37, 0.3806, 0.381, 0.3845, 0.385, 0.3832, 0.38, 0.3728, 0.36, 0.3536, 0.33, 0.3176, 0.3, 0.2634, 0.25, 0.2037, 0.17, 0.1453, 0.1, 0.0903, 0.05,
    0.0375, 0.0, -0.0080, -0.05, -0.0340, -0.04, -0.0443, -0.04, -0.0399, -0.03, -0.0206, -0.02, -0.0026, 0.0, 0.0076, 0.01, 0.0119, 0.012, 0.0137, 0.015, 0.0171, 0.02, 0.0228, 0.025,
    0.0265, 0.026, 0.0256, 0.023, 0.0246, 0.02, 0.0175, 0.016, 0.0142, 0.016, 0.0164, 0.02, 0.0248,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L17_3_m40spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0127, 0.02, 0.0210, 0.03, 0.0341, 0.05, 0.0660, 0.07, 0.0852, 0.09, 0.0990, 0.11, 0.1142, 0.12, 0.1229, 0.1228, 0.1228, 0.121, 0.1209, 0.12, 0.1164, 0.11, 0.1092, 0.1, 0.0964, 0.08, 0.0777, 0.06, 0.0578, 0.04, 0.0396, 0.03, 0.0229, 0.02,
    0.0069, 0.0, -0.0069, -0.01, -0.0148, -0.015, -0.0173, -0.015, -0.0116, 0.0, 0.0046, 0.01, 0.0191, 0.02, 0.0270, 0.03, 0.0304, 0.031, 0.0317, 0.032, 0.0332, 0.034,
    0.0352, 0.036, 0.0364, 0.0363, 0.0362, 0.036, 0.0353, 0.034, 0.0338, 0.033, 0.0328, 0.033, 0.0335, 0.035, 0.0362,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L17_3_m40spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0127, 0.02, 0.0210, 0.03, 0.0361, 0.05, 0.0660, 0.07, 0.0852, 0.09, 0.0990, 0.11, 0.1142, 0.12, 0.1229, 0.1248, 0.1228, 0.121, 0.1209, 0.12, 0.1164, 0.11, 0.1092, 0.1, 0.0964, 0.08, 0.0777, 0.06, 0.0578, 0.04, 0.0396, 0.03, 0.0229, 0.02,
    0.0069, 0.0, -0.0069, -0.01, -0.0148, -0.015, -0.0173, -0.015, -0.0116, 0.0, 0.0046, 0.015, 0.0191, 0.02, 0.0270, 0.03, 0.0304, 0.031, 0.0317, 0.032, 0.0332, 0.034,
    0.0352, 0.036, 0.0364, 0.0363, 0.0362, 0.036, 0.0353, 0.034, 0.0338, 0.033, 0.0328, 0.033, 0.0335, 0.035, 0.0362,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L17_3_m40spect3[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0127, 0.02, 0.0210, 0.03, 0.0341, 0.05, 0.0660, 0.07, 0.0852, 0.09, 0.0990, 0.11, 0.1142, 0.12, 0.1229, 0.1228, 0.1228, 0.121, 0.1209, 0.12, 0.1164, 0.11, 0.1092, 0.1, 0.0964, 0.08, 0.0777, 0.06, 0.0578, 0.04, 0.0396, 0.03, 0.0229, 0.02,
    0.0069, 0.0, -0.0069, -0.01, -0.0148, -0.015, -0.0173, -0.015, -0.0116, 0.0, 0.0046, 0.01, 0.0191, 0.02, 0.0270, 0.03, 0.0304, 0.031, 0.0317, 0.032, 0.0332, 0.034,
    0.0352, 0.036, 0.0364, 0.0363, 0.0362, 0.036, 0.0353, 0.034, 0.0338, 0.033, 0.0328, 0.033, 0.0335, 0.035, 0.0362,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L17_3_m40spect4[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0127, 0.02, 0.0210, 0.03, 0.0361, 0.05, 0.0660, 0.07, 0.0852, 0.09, 0.0990, 0.11, 0.1142, 0.12, 0.1229, 0.1248, 0.1228, 0.121, 0.1209, 0.12, 0.1164, 0.11, 0.1092, 0.1, 0.0964, 0.08, 0.0777, 0.06, 0.0578, 0.04, 0.0396, 0.03, 0.0229, 0.02,
    0.0069, 0.0, -0.0069, -0.01, -0.0148, -0.015, -0.0173, -0.015, -0.0116, 0.0, 0.0046, 0.015, 0.0191, 0.02, 0.0270, 0.03, 0.0304, 0.031, 0.0317, 0.032, 0.0332, 0.034,
    0.0352, 0.036, 0.0364, 0.0363, 0.0362, 0.036, 0.0353, 0.034, 0.0338, 0.033, 0.0328, 0.033, 0.0335, 0.035, 0.0362,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L17_3_m40spect5[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0127, 0.02, 0.0210, 0.03, 0.0361, 0.05, 0.0660, 0.07, 0.0852, 0.09, 0.0990, 0.11, 0.1142, 0.12, 0.1229, 0.1248, 0.1228, 0.121, 0.1209, 0.12, 0.1164, 0.11, 0.1092, 0.1, 0.0964, 0.08, 0.0777, 0.06, 0.0578, 0.04, 0.0396, 0.03, 0.0229, 0.02,
    0.0069, 0.0, -0.0069, -0.01, -0.0148, -0.015, -0.0173, -0.015, -0.0116, 0.0, 0.0046, 0.015, 0.0191, 0.02, 0.0270, 0.03, 0.0304, 0.031, 0.0317, 0.032, 0.0332, 0.034,
    0.0352, 0.036, 0.0364, 0.0363, 0.0362, 0.036, 0.0353, 0.034, 0.0338, 0.033, 0.0328, 0.033, 0.0335, 0.035, 0.0362,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L17_3_m40spect6[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0127, 0.02, 0.0210, 0.03, 0.0361, 0.05, 0.0660, 0.07, 0.0852, 0.09, 0.0990, 0.11, 0.1142, 0.12, 0.1229, 0.1248, 0.1228, 0.121, 0.1209, 0.12, 0.1164, 0.11, 0.1092, 0.1, 0.0964, 0.08, 0.0777, 0.06, 0.0578, 0.04, 0.0396, 0.03, 0.0229, 0.02,
    0.0069, 0.0, -0.0069, -0.01, -0.0148, -0.015, -0.0173, -0.015, -0.0116, 0.0, 0.0046, 0.015, 0.0191, 0.02, 0.0270, 0.03, 0.0304, 0.031, 0.0317, 0.032, 0.0332, 0.034,
    0.0352, 0.036, 0.0364, 0.0363, 0.0362, 0.036, 0.0353, 0.034, 0.0338, 0.033, 0.0328, 0.033, 0.0335, 0.035, 0.0362,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L21_9_m7spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0096, 0.01, 0.0122, 0.014, 0.0162, 0.02, 0.0277, 0.03, 0.0346, 0.035, 0.0391, 0.04, 0.0440, 0.045, 0.0464, 0.046, 0.0458, 0.045, 0.0443, 0.043, 0.0419, 0.041, 0.0400, 0.04, 0.0391, 0.039, 0.0389, 0.037, 0.0369, 0.035, 0.0330, 0.03, 0.0288, 0.025, 0.0243, 0.022,
    0.0201, 0.02, 0.0178, 0.0176, 0.0175, 0.02, 0.0227, 0.03, 0.0360, 0.04, 0.0477, 0.05, 0.0541, 0.055, 0.0568, 0.057, 0.0580, 0.058, 0.0589, 0.059, 0.0598, 0.06, 0.0602, 0.0602, 0.0602, 0.06, 0.0599, 0.0599,
    0.0595, 0.0595, 0.0592, 0.0594, 0.0594, 0.06, 0.0605,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L78_4_m74spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.2087, 0.3, 0.3297, 0.4, 0.5205, 0.8, 0.9908, 1.1, 1.2731, 1.3, 1.4747, 1.6, 1.6992, 1.75, 1.8277, 1.82, 1.8300, 1.82, 1.8050, 1.75, 1.7397, 1.7, 1.6470, 1.6, 1.5025, 1.4, 1.2968, 1.1, 1.0577, 1.0, 0.8101, 0.8, 0.5740, 0.4, 0.3430, 0.25,
    0.1407, 0.05, 0.0264, 0.0, -0.0107, 0.0, 0.0733, 0.2, 0.3116, 0.4, 0.5251, 0.6, 0.6429, 0.65, 0.6923, 0.7, 0.7137, 0.72, 0.7360, 0.75, 0.7658, 0.77, 0.7831, 0.782,
    0.7804, 0.77, 0.7677, 0.75, 0.7455, 0.74, 0.7313, 0.74, 0.7406, 0.77, 0.7819,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L31_m58_m66spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0229, 0.04, 0.0549, 0.08, 0.1059, 0.15, 0.2207, 0.24, 0.2892, 0.3, 0.3414, 0.36, 0.4006, 0.42, 0.4386, 0.44, 0.4461, 0.445, 0.4480, 0.44, 0.4387, 0.442, 0.4182, 0.4, 0.3771, 0.35, 0.3139, 0.3, 0.2436, 0.2, 0.1736, 0.15, 0.1070, 0.07,
    0.0432, 0.0, -0.0114, -0.02, -0.0429, -0.05, -0.0567, -0.06, -0.0618, -0.063, -0.0637, -0.0637, -0.0639, -0.0635, -0.0635, -0.0634, -0.0632, -0.0632, -0.0632, 0.06, -0.0601, -0.06, -0.0539, -0.05,
    -0.0497, 0.05, -0.0510, -0.054, -0.0546, -0.06, -0.0608, -0.062, -0.0649, -0.0622, -0.0622, -0.055, -0.0526,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L61_m11_m12spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0509, 0.06, 0.0770, 0.1, 0.1178, 0.2, 0.2177, 0.25, 0.2762, 0.3, 0.3194, 0.32, 0.3695, 0.37, 0.4016, 0.402, 0.4093, 0.41, 0.4106, 0.41, 0.4017, 0.4, 0.3957, 0.4, 0.4047, 0.41, 0.4237, 0.42, 0.4184, 0.4, 0.3820, 0.35, 0.3365, 0.3, 0.2881, 0.25,
    0.2436, 0.22, 0.2188, 0.21, 0.2086, 0.21, 0.2133, 0.22, 0.2323, 0.24, 0.2499, 0.25, 0.2605, 0.265, 0.2650, 0.266, 0.2679, 0.27, 0.2720, 0.275, 0.2783, 0.28, 0.2816, 0.281,
    0.2808, 0.28, 0.2779, 0.275, 0.2731, 0.271, 0.2702, 0.272, 0.2720, 0.28, 0.2811,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L61_m11_m12spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0509, 0.06, 0.0770, 0.1, 0.1578, 0.2, 0.2177, 0.25, 0.2762, 0.3, 0.3194, 0.32, 0.3695, 0.37, 0.4016, 0.402, 0.4093, 0.41, 0.4106, 0.41, 0.4017, 0.4, 0.3957, 0.4, 0.4047, 0.41, 0.4237, 0.42, 0.4184, 0.4, 0.3820, 0.35, 0.3365, 0.3, 0.2881, 0.25,
    0.2436, 0.22, 0.2188, 0.21, 0.2186, 0.21, 0.2133, 0.22, 0.2323, 0.24, 0.2499, 0.25, 0.2605, 0.265, 0.2650, 0.266, 0.2679, 0.27, 0.2720, 0.275, 0.2783, 0.28, 0.2816, 0.281,
    0.2808, 0.28, 0.2779, 0.275, 0.2831, 0.271, 0.2702, 0.272, 0.2720, 0.28, 0.2811,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L29_1_m13spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0147, 0.02, 0.0212, 0.03, 0.0313, 0.04, 0.0571, 0.06, 0.0724, 0.08, 0.0832, 0.09, 0.0955, 0.1, 0.1026, 0.102, 0.1031, 0.102, 0.1019, 0.1, 0.0985, 0.096, 0.0950, 0.094, 0.0930, 0.092, 0.0914, 0.09, 0.0855, 0.08, 0.0749, 0.07, 0.0634, 0.06, 0.0515, 0.05,
    0.0407, 0.04, 0.0346, 0.04, 0.0327, 0.033, 0.0377, 0.04, 0.0517, 0.06, 0.0642, 0.07, 0.0711, 0.072, 0.0740, 0.075, 0.0754, 0.076, 0.0768, 0.077, 0.0785, 0.079, 0.0794, 0.0793,
    0.0792, 0.079, 0.0786, 0.078, 0.0774, 0.077, 0.0767, 0.077, 0.0772, 0.078, 0.0795,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L29_1_m13spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0147, 0.02, 0.0212, 0.03, 0.0343, 0.04, 0.0571, 0.06, 0.0724, 0.08, 0.0832, 0.09, 0.0995, 0.1, 0.1026, 0.102, 0.1031, 0.102, 0.1019, 0.1, 0.0985, 0.096, 0.0950, 0.094, 0.0930, 0.092, 0.0914, 0.09, 0.0855, 0.08, 0.0749, 0.07, 0.0634, 0.06, 0.0515, 0.05,
    0.0407, 0.04, 0.0346, 0.04, 0.0327, 0.033, 0.0377, 0.04, 0.0517, 0.06, 0.0642, 0.07, 0.0711, 0.072, 0.0740, 0.075, 0.0754, 0.076, 0.0768, 0.077, 0.0785, 0.079, 0.0794, 0.0793,
    0.0792, 0.079, 0.0786, 0.078, 0.0794, 0.077, 0.0767, 0.077, 0.0772, 0.078, 0.0795,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L2_14_m1spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0020, 0.002, 0.0021, 0.0021, 0.0022, 0.003, 0.0030, 0.0033, 0.0036, 0.0035, 0.0038, 0.0038, 0.0039, 0.0038, 0.0036, 0.0035, 0.0030, 0.003, 0.0024, 0.002, 0.0017, 0.0015, 0.0011, 0.001, 0.0004, 0.0, -0.0004, -0.001, -0.0010, -0.0011, -0.0012, -0.0011, -0.0011, -0.0011, -0.0011, -0.001,
    -0.0011, -0.0011, -0.0011, -0.001, -0.0008, 0.0, 0.0012, 0.004, 0.0061, 0.01, 0.0104, 0.011, 0.0127, 0.013, 0.0136, 0.014, 0.0140, 0.0142, 0.0142, 0.0143, 0.0144, 0.0144,
    0.0144, 0.0144, 0.0144, 0.0145, 0.0145, 0.0145, 0.0145, 0.0145, 0.0145, 0.0145, 0.0145, 0.0145, 0.0146,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L5_39_m7spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0070, 0.007, 0.0072, 0.0075, 0.0076, 0.01, 0.0108, 0.011, 0.0129, 0.013, 0.0136, 0.014, 0.0140, 0.0135, 0.0131, 0.012, 0.0112, 0.01, 0.0088, 0.008, 0.0066, 0.005, 0.0046, 0.003, 0.0021, 0.0, -0.0009, -0.002, -0.0032, -0.003,
    -0.0039, -0.0039, -0.0039, -0.004, -0.0040, -0.004, -0.0041, -0.0041, -0.0041, -0.0035, -0.0032, 0.0, 0.0038, 0.03, 0.0205, 0.03, 0.0351, 0.04, 0.0429, 0.045, 0.0462, 0.047,
    0.0476, 0.048, 0.0483, 0.0485, 0.0487, 0.0488, 0.0489, 0.049, 0.0490, 0.049, 0.0490, 0.049, 0.0491, 0.0491, 0.0491, 0.0491, 0.0491, 0.0492, 0.0494,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L15_5_m13spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0064, 0.007, 0.0089, 0.01, 0.0129, 0.02, 0.0233, 0.025, 0.0295, 0.03, 0.0338, 0.035, 0.0386, 0.04, 0.0411, 0.041, 0.0409, 0.04, 0.0400, 0.039, 0.0383, 0.037, 0.0364, 0.035, 0.0345, 0.033, 0.0321, 0.03, 0.0285, 0.025, 0.0240, 0.02, 0.0194, 0.015, 0.0148, 0.012,
    0.0106, 0.01, 0.0083, 0.008, 0.0077, 0.01, 0.0107, 0.015, 0.0188, 0.02, 0.0260, 0.027, 0.0299, 0.03, 0.0316, 0.032, 0.0323, 0.033, 0.0330, 0.033, 0.0337, 0.034, 0.0341, 0.034,
    0.0340, 0.034, 0.0338, 0.033, 0.0333, 0.033, 0.0330, 0.033, 0.0332, 0.034, 0.0342,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::Colorlab_L12_5_m6spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0039, 0.004, 0.0052, 0.006, 0.0072, 0.01, 0.0127, 0.013, 0.0159, 0.016, 0.0181, 0.02, 0.0205, 0.021, 0.0218, 0.0217, 0.0217, 0.0215, 0.0212, 0.021, 0.0202, 0.02, 0.0194, 0.019, 0.0190, 0.019, 0.0187, 0.018, 0.0177, 0.016, 0.0157, 0.014,
    0.0135, 0.012, 0.0112, 0.01, 0.0091, 0.008, 0.0079, 0.0077, 0.0077, 0.008, 0.0096, 0.01, 0.0145, 0.015, 0.0189, 0.02, 0.0213, 0.021, 0.0223, 0.0222,  0.0227, 0.023,
    0.0231, 0.0231, 0.0235, 0.0236, 0.0237, 0.0237, 0.0237, 0.0236, 0.0235, 0.0233, 0.0233, 0.0232, 0.0233, 0.0233, 0.0235, 0.0238,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L12_5_m6spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0039, 0.004, 0.0062, 0.006, 0.0072, 0.01, 0.0147, 0.013, 0.0159, 0.016, 0.0181, 0.02, 0.0205, 0.021, 0.0218, 0.0257, 0.0217, 0.0215, 0.0212, 0.021, 0.0202, 0.02, 0.0194, 0.019, 0.0190, 0.019, 0.0187, 0.018, 0.0177, 0.016, 0.0157, 0.014,
    0.0135, 0.012, 0.0112, 0.01, 0.0099, 0.008, 0.0079, 0.0087, 0.0077, 0.008, 0.0096, 0.01, 0.0145, 0.015, 0.0189, 0.02, 0.0213, 0.021, 0.0223, 0.0222,  0.0227, 0.023,
    0.0231, 0.0231, 0.0235, 0.0236, 0.0237, 0.0237, 0.0237, 0.0236, 0.0235, 0.0233, 0.0233, 0.0232, 0.0233, 0.0233, 0.0235, 0.0238,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L37_m59_m24spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0007, 0.01, 0.0162, 0.03, 0.0407, 0.06, 0.0911, 0.1, 0.1203, 0.13, 0.1448, 0.16, 0.1741, 0.18, 0.1965, 0.2, 0.2068, 0.21, 0.2149, 0.215, 0.2165, 0.216, 0.2159, 0.216, 0.2165, 0.2165, 0.2165, 0.21, 0.2046, 0.19, 0.1781, 0.16, 0.1472, 0.13, 0.1161, 0.1,
    0.0889, 0.08, 0.0732, 0.07, 0.0636, 0.05, 0.0426, 0.0, -0.0029, -0.02, -0.0419, -0.05, -0.0625, -0.07, -0.0711, -0.073, -0.0743, -0.074, -0.0744, -0.073, -0.0722, -0.071,
    -0.0705, -0.071, -0.0715, -0.072, -0.0735, -0.075, -0.0766, -0.077, -0.0786, -0.078, -0.0773, -0.074, -0.0731,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};
const double ColorTemp::Colorlab_L37_m59_m24spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0007, 0.01, 0.0162, 0.03, 0.0427, 0.06, 0.0911, 0.1, 0.1203, 0.13, 0.1548, 0.16, 0.1741, 0.18, 0.1965, 0.2, 0.2068, 0.24, 0.2149, 0.215, 0.2165, 0.216, 0.2159, 0.216, 0.2165, 0.2165, 0.2165, 0.21, 0.2046, 0.19, 0.1781, 0.16, 0.1472, 0.13, 0.1161, 0.1,
    0.0889, 0.08, 0.0732, 0.07, 0.0636, 0.05, 0.0426, 0.0, -0.0029, -0.02, -0.0429, -0.05, -0.0625, -0.07, -0.0711, -0.073, -0.0743, -0.074, -0.0744, -0.073, -0.0722, -0.071,
    -0.0705, -0.071, -0.0715, -0.072, -0.0735, -0.075, -0.0766, -0.077, -0.0786, -0.078, -0.0773, -0.074, -0.0731,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L15_55_23spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0184, 0.017, 0.0169, 0.015, 0.0143, 0.015, 0.0153, 0.016, 0.0164, 0.015, 0.0147, 0.013, 0.0119, 0.01, 0.0070, 0.005, 0.0012, 0.0, -0.0052, -0.01, -0.0106, -0.012, -0.0143, -0.015, -0.0166, -0.016, -0.0177, -0.017, -0.0169, -0.015,
    -0.0134, -0.01, -0.0085, -0.005, -0.0041, -0.0008, -0.0008, 0.0, 0.0014, 0.002, 0.0049, 0.02, 0.0244, 0.05, 0.0701, 0.1, 0.1100, 0.12, 0.1315, 0.14, 0.1405, 0.142, 0.1443, 0.145,
    0.1461, 0.1463, 0.1469, 0.1469, 0.1469, 0.148, 0.1474, 0.1476, 0.1478, 0.148, 0.1483, 0.1485, 0.1486, 0.1486, 0.1484, 0.1485, 0.1487,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L11_m55_m11spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0164, 0.017, 0.0170, 0.017, 0.0179, 0.02, 0.0254, 0.03, 0.0304, 0.0302, 0.0321, 0.031, 0.0331, 0.032, 0.0310, 0.03, 0.0264, 0.025, 0.0209, 0.02, 0.0156, 0.0120, 0.0108, 0.01, 0.0048, 0.0, -0.0025, -0.005, -0.0079, -0.008, -0.0098, -0.0098, -0.0098, -0.0098, -0.0099, -0.01,
    -0.0103, -0.0102, -0.0102, -0.01, -0.0080, 0.0, 0.0084, 0.02, 0.0478, 0.06, 0.0823, 0.09, 0.1008, 0.105, 0.1086, 0.11, 0.1118, 0.112, 0.1135, 0.114, 0.1145, 0.1146, 0.1148, 0.115,
    0.1152, 0.1152, 0.1153, 0.1153, 0.1153, 0.1153, 0.1154, 0.1154, 0.1154, 0.116, 0.1162,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L8_m10_m2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0006, 0.001, 0.0013, 0.002, 0.0025, 0.004, 0.0050, 0.0055, 0.0064, 0.007, 0.0076, 0.008, 0.0090, 0.01, 0.0101, 0.0102, 0.0106, 0.011, 0.0111, 0.0111, 0.0111, 0.0112, 0.0113, 0.012, 0.0122, 0.013, 0.0135, 0.014, 0.0140, 0.014,
    0.0131, 0.012, 0.0118, 0.011, 0.0104, 0.01, 0.0091, 0.009, 0.0083, 0.008, 0.0079, 0.0075, 0.0070, 0.006, 0.0050, 0.004, 0.0033, 0.003, 0.0024, 0.0022, 0.0021, 0.002, 0.0019, 0.002, 0.0020, 0.002, 0.0021, 0.0022, 0.0022, 0.0021,
    0.0021, 0.002, 0.0020, 0.002, 0.0019, 0.0018, 0.0018, 0.0018, 0.0018, 0.0019, 0.0021,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L14_m10_m7spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0023, 0.003, 0.0042, 0.006, 0.0071, 0.01, 0.0139, 0.015, 0.0179, 0.02, 0.0209, 0.022, 0.0245, 0.026, 0.0270, 0.0275, 0.0278, 0.028, 0.0283, 0.028, 0.0280, 0.028, 0.0277, 0.028, 0.0282, 0.029, 0.0293, 0.029, 0.0286, 0.027, 0.0258, 0.024, 0.0224, 0.02,
    0.0188, 0.017, 0.0156, 0.014, 0.0137, 0.013, 0.0128, 0.0125, 0.0120, 0.011, 0.0107, 0.01, 0.0096, 0.0094, 0.0091, 0.009, 0.0089, 0.0089, 0.0089, 0.009, 0.0091, 0.0092, 0.0095, 0.0095,
    0.0097, 0.0096, 0.0096, 0.0095, 0.0094, 0.0093, 0.0090, 0.009, 0.0088, 0.0089, 0.0089, 0.009, 0.0095,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L20_m16_m13spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0039, 0.005, 0.0080, 0.01, 0.0144, 0.02, 0.0290, 0.03, 0.0375, 0.04, 0.0442, 0.05, 0.0519, 0.054, 0.0573, 0.058, 0.0591, 0.06, 0.0600, 0.06, 0.0594, 0.059, 0.0584, 0.058, 0.0579, 0.0578, 0.0575, 0.056, 0.0540, 0.05, 0.0471, 0.04,
    0.0393, 0.035, 0.0313, 0.03, 0.0242, 0.022, 0.0202, 0.02, 0.0182, 0.017, 0.0161, 0.014, 0.0125, 0.01, 0.0095, 0.009, 0.0081, 0.008, 0.0075, 0.0074, 0.0073, 0.0075,
    0.0076, 0.008, 0.0084, 0.0087, 0.0089, 0.0088, 0.0087, 0.0085, 0.0082, 0.008, 0.0074, 0.007, 0.0069, 0.007, 0.0073, 0.008, 0.0085,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L8_m10_m2spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0006, 0.001, 0.0013, 0.003, 0.0025, 0.004, 0.0050, 0.0055, 0.0064, 0.007, 0.0076, 0.009, 0.0090, 0.01, 0.0101, 0.0102, 0.0106, 0.012, 0.0111, 0.0111, 0.0121, 0.0122, 0.0113, 0.012, 0.0122, 0.013, 0.0135, 0.014, 0.0140, 0.014,
    0.0131, 0.012, 0.0128, 0.011, 0.0114, 0.01, 0.0091, 0.009, 0.0083, 0.008, 0.0079, 0.0075, 0.0070, 0.006, 0.0050, 0.004, 0.0033, 0.003, 0.0024, 0.0022, 0.0021, 0.002, 0.0019, 0.002, 0.0020, 0.002, 0.0021, 0.0022, 0.0022, 0.0021,
    0.0021, 0.002, 0.0020, 0.002, 0.0019, 0.0018, 0.0018, 0.0018, 0.0018, 0.0019, 0.0021,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L14_m10_m7spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0023, 0.003, 0.0042, 0.006, 0.0091, 0.01, 0.0149, 0.015, 0.0179, 0.02, 0.0209, 0.022, 0.0265, 0.026, 0.0270, 0.0275, 0.0278, 0.028, 0.0293, 0.028, 0.0280, 0.029, 0.0277, 0.028, 0.0282, 0.029, 0.0293, 0.029, 0.0286, 0.027, 0.0258, 0.024, 0.0224, 0.02,
    0.0198, 0.017, 0.0156, 0.014, 0.0137, 0.013, 0.0128, 0.0125, 0.0120, 0.011, 0.0107, 0.01, 0.0096, 0.0094, 0.0091, 0.009, 0.0089, 0.0089, 0.0089, 0.009, 0.0091, 0.0092, 0.0095, 0.0095,
    0.0097, 0.0096, 0.0096, 0.0095, 0.0094, 0.0093, 0.0090, 0.009, 0.0088, 0.0089, 0.0089, 0.009, 0.0095,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L20_m16_m13spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0039, 0.005, 0.0080, 0.01, 0.0144, 0.02, 0.0290, 0.03, 0.0375, 0.04, 0.0442, 0.05, 0.0519, 0.054, 0.0573, 0.06, 0.0591, 0.06, 0.0600, 0.06, 0.0654, 0.065, 0.0654, 0.065, 0.0659, 0.0578, 0.0575, 0.056, 0.0540, 0.05, 0.0471, 0.04,
    0.0393, 0.035, 0.0313, 0.03, 0.0242, 0.022, 0.0202, 0.02, 0.0182, 0.017, 0.0161, 0.014, 0.0125, 0.01, 0.0095, 0.009, 0.0081, 0.008, 0.0075, 0.0074, 0.0073, 0.0075,
    0.0076, 0.008, 0.0084, 0.0087, 0.0089, 0.0088, 0.0087, 0.0085, 0.0082, 0.008, 0.0074, 0.007, 0.0069, 0.007, 0.0073, 0.008, 0.0085,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L6_m9_1spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0002, 0.0005, 0.0007, 0.001, 0.0014, 0.002, 0.0030, 0.0035, 0.0039, 0.004, 0.0046, 0.005, 0.0056, 0.006, 0.0064, 0.0064, 0.0069, 0.007, 0.0073, 0.0073, 0.0074, 0.0076, 0.0077, 0.008, 0.0087, 0.009, 0.0101, 0.0102, 0.0108, 0.011,
    0.0103, 0.01, 0.0095, 0.009, 0.0085, 0.008, 0.0076, 0.0075, 0.0071, 0.007, 0.0068, 0.006, 0.0058, 0.004, 0.0037, 0.002, 0.0018, 0.001, 0.0008, 0.0006, 0.0004, 0.0004,
    0.0003, 0.0003, 0.0003, 0.0003, 0.0003, 0.0004, 0.0004, 0.0004, 0.0004, 0.0003, 0.0003, 0.0003, 0.0002, 0.0002, 0.0001, 0.0001, 0.0002, 0.0002, 0.0003,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L20_m9_m10spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0050, 0.006, 0.0084, 0.01, 0.0139, 0.02, 0.0266, 0.03, 0.0342, 0.035, 0.0398, 0.04, 0.0464, 0.05, 0.0507, 0.052, 0.0518, 0.052, 0.0522, 0.052, 0.0513, 0.051, 0.0503, 0.0502, 0.0502, 0.0503, 0.0504, 0.05, 0.0480, 0.048, 0.0425, 0.04, 0.0361, 0.03,
    0.0296, 0.025, 0.0236, 0.022, 0.0203, 0.02, 0.0188, 0.0185, 0.0185, 0.0188, 0.0188, 0.019, 0.0191, 0.0195, 0.0195, 0.0195, 0.0196, 0.0197, 0.0198, 0.02, 0.0202, 0.0205,
    0.0209, 0.021, 0.0214, 0.0214, 0.0213, 0.021, 0.0209, 0.0205, 0.0202, 0.02, 0.0198, 0.02, 0.0201, 0.021, 0.0212,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L85_10_45spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1141, 0.12, 0.1230, 0.13, 0.1346, 0.16, 0.1906, 0.2, 0.2220, 0.23, 0.2391, 0.25, 0.2598, 0.26, 0.2687, 0.267, 0.2673, 0.26, 0.2587, 0.25, 0.2448, 0.25, 0.2597, 0.3, 0.3501, 0.4, 0.5052, 0.55, 0.6139, 0.63, 0.6472, 0.65,
    0.6535, 0.65, 0.6470, 0.64, 0.6334, 0.63, 0.6279, 0.63, 0.6307, 0.66, 0.6778, 0.7, 0.7914, 0.82, 0.8909, 0.9, 0.9462, 0.95, 0.9695, 0.97, 0.9817, 0.982,
    0.9893, 0.99, 0.9956, 0.996, 0.9969, 0.997, 0.9975, 0.997, 0.9969, 0.9965, 0.9962, 0.9963, 0.9963, 0.9962, 0.9962, 1.0, 1.0030,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L90_m7_82spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0733, 0.073, 0.0730, 0.07, 0.0698, 0.07, 0.0777, 0.0777, 0.0779, 0.078, 0.0780, 0.08, 0.0835, 0.09, 0.0925, 0.1, 0.1077, 0.11, 0.1187, 0.12, 0.1248, 0.15, 0.1700, 0.25, 0.3233, 0.4, 0.5715, 0.65, 0.7545, 0.8, 0.8254, 0.84, 0.8529, 0.86, 0.8634, 0.864, 0.8632, 0.864,
    0.8651, 0.864, 0.8637, 0.86, 0.8584, 0.85, 0.8439, 0.84, 0.8309, 0.83, 0.8259, 0.825, 0.8241, 0.824, 0.8265, 0.827, 0.8286, 0.83, 0.8306, 0.83, 0.8299, 0.829,
    0.8296, 0.8296, 0.8296, 0.83, 0.8301, 0.831, 0.8314, 0.831, 0.8305, 0.832, 0.8331,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L95_2_18spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1508, 0.16, 0.1879, 0.2, 0.2437, 0.3, 0.4029, 0.45, 0.4951, 0.52, 0.5582, 0.6, 0.6321, 0.65, 0.6760, 0.68, 0.6838, 0.68, 0.6784, 0.66, 0.6570, 0.66, 0.6625, 0.7, 0.7471, 0.8, 0.8962, 0.9, 0.9795, 0.97, 0.9654, 0.95, 0.9187, 0.9,
    0.8583, 0.8, 0.7967, 0.78, 0.7639, 0.76, 0.7546, 0.77, 0.7979, 0.85, 0.9113, 1.0, 1.0117, 1.04, 1.0684, 1.08, 1.0924, 1.1, 1.1055, 1.11, 1.1164, 1.12, 1.1286, 1.13,
    1.1336, 1.133, 1.1330, 1.13, 1.1293, 1.125, 1.1232, 1.122, 1.1200, 1.121, 1.1220, 1.13, 1.1380,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L39_7_4spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0233, 0.025, 0.0285, 0.03, 0.0364, 0.04, 0.0599, 0.065, 0.0737, 0.08, 0.0827, 0.09, 0.0929, 0.095, 0.0982, 0.098, 0.0978, 0.096, 0.0954, 0.094, 0.0910, 0.09, 0.0896, 0.093, 0.0965, 0.1, 0.1100, 0.112, 0.1163, 0.113, 0.1124, 0.11, 0.1055, 0.1, 0.0970, 0.09,
    0.0885, 0.086, 0.0840, 0.084, 0.0834, 0.09, 0.0936, 0.1, 0.1194, 0.13, 0.1421, 0.15, 0.1547, 0.157, 0.1600, 0.162, 0.1625, 0.163, 0.1644, 0.165, 0.1663, 0.167, 0.1671, 0.1671,
    0.1671, 0.167, 0.1666, 0.166, 0.1658, 0.1656, 0.1653, 0.1655, 0.1656, 0.167, 0.1679,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L39_4_1spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0226, 0.025, 0.0290, 0.03, 0.0386, 0.05, 0.0656, 0.07, 0.0815, 0.09, 0.0923, 0.1, 0.1046, 0.11, 0.1114, 0.1113, 0.1116, 0.11, 0.1098, 0.108, 0.1056, 0.104, 0.1038, 0.108, 0.1097, 0.11, 0.1214, 0.123, 0.1256, 0.12, 0.1193, 0.115,
    0.1100, 0.1, 0.0992, 0.09, 0.0888, 0.085, 0.0831, 0.082, 0.0817, 0.085, 0.0903, 0.1, 0.1126, 0.12, 0.1324, 0.135, 0.1434, 0.145, 0.1480, 0.15, 0.1503, 0.152, 0.1521, 0.153,
    0.1541, 0.155, 0.1550, 0.155, 0.1549, 0.1545, 0.1543, 0.154, 0.1533, 0.153, 0.1527, 0.153, 0.1531, 0.154, 0.1557,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L39_3_m1spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0221, 0.025, 0.0288, 0.03, 0.0392, 0.05, 0.0673, 0.07, 0.0839, 0.09, 0.0953, 0.1, 0.1083, 0.11, 0.1158, 0.116, 0.1163, 0.115, 0.1147, 0.112, 0.1105, 0.11, 0.1087, 0.11, 0.1140, 0.12, 0.1248, 0.125, 0.1279, 0.125, 0.1207, 0.115, 0.1104, 0.1, 0.0988, 0.09, 0.0876, 0.085, 0.0816, 0.08, 0.0799, 0.08,
    0.0877, 0.09, 0.1083, 0.11, 0.1265, 0.13, 0.1367, 0.14, 0.1410, 0.142, 0.1431, 0.144, 0.1449, 0.146, 0.1470, 0.1474, 0.1479, 0.1478, 0.1478, 0.1475, 0.1472, 0.147, 0.1460, 0.146, 0.1454, 0.1456, 0.1458, 0.147, 0.1485,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_3_m2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0242, 0.03, 0.0316, 0.035, 0.0430, 0.06, 0.0740, 0.08, 0.0923, 0.1, 0.1049, 0.11, 0.1192, 0.12, 0.1272, 0.1275, 0.1275, 0.126, 0.1256, 0.124, 0.1208, 0.12, 0.1184, 0.12, 0.1232, 0.13, 0.1333, 0.134, 0.1355, 0.13, 0.1272, 0.12,
    0.1157, 0.11, 0.1029, 0.1, 0.0906, 0.09, 0.0840, 0.083, 0.0822, 0.09, 0.0913, 0.1, 0.1150, 0.12, 0.1360, 0.14, 0.1477, 0.15, 0.1526, 0.154, 0.1550, 0.156, 0.1570, 0.158,
    0.1593, 0.16, 0.1603, 0.1602, 0.1602, 0.16, 0.1595, 0.1594, 0.1582, 0.158, 0.1575, 0.158, 0.1580, 0.159, 0.1609,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L36_2_2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0173, 0.02, 0.0222, 0.025, 0.0296, 0.04, 0.0501, 0.055, 0.0621, 0.07, 0.0704, 0.075, 0.0798, 0.08, 0.0852, 0.085, 0.0856, 0.085, 0.0845, 0.083, 0.0815, 0.081, 0.0806, 0.084, 0.0864, 0.09, 0.0975, 0.1, 0.1021, 0.1, 0.0979, 0.095,
    0.0909, 0.09, 0.0827, 0.08, 0.0747, 0.073, 0.0703, 0.07, 0.0692, 0.07, 0.0752, 0.08, 0.0908, 0.1, 0.1046, 0.11, 0.1124, 0.113, 0.1156, 0.116, 0.1173, 0.118, 0.1187, 0.123,
    0.1202, 0.1205, 0.1209, 0.1208, 0.1208, 0.1205, 0.1203, 0.12, 0.1195, 0.1194, 0.1191, 0.1193, 0.1194, 0.12, 0.1213,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L39_7_4spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0233, 0.028, 0.0285, 0.03, 0.0364, 0.04, 0.0599, 0.065, 0.0737, 0.08, 0.0927, 0.09, 0.0929, 0.095, 0.0982, 0.098, 0.0978, 0.096, 0.0954, 0.094, 0.0910, 0.09, 0.0896, 0.093, 0.0965, 0.1, 0.1100, 0.112, 0.1163, 0.113, 0.1124, 0.11, 0.1055, 0.1, 0.0970, 0.09,
    0.0885, 0.086, 0.0840, 0.084, 0.0834, 0.09, 0.0936, 0.1, 0.1294, 0.15, 0.1521, 0.17, 0.1647, 0.167, 0.1600, 0.162, 0.1625, 0.163, 0.1644, 0.165, 0.1663, 0.167, 0.1671, 0.1671,
    0.1671, 0.167, 0.1666, 0.166, 0.1658, 0.1656, 0.1653, 0.1655, 0.1656, 0.167, 0.1679,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L39_4_1spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0226, 0.025, 0.0290, 0.035, 0.0386, 0.05, 0.0656, 0.07, 0.0855, 0.09, 0.0923, 0.105, 0.1046, 0.11, 0.1114, 0.1113, 0.1216, 0.13, 0.1298, 0.118, 0.1056, 0.104, 0.1038, 0.108, 0.1097, 0.11, 0.1214, 0.123, 0.1256, 0.12, 0.1193, 0.115,
    0.1100, 0.1, 0.0992, 0.09, 0.0888, 0.085, 0.0831, 0.082, 0.0817, 0.085, 0.0903, 0.1, 0.1126, 0.12, 0.1324, 0.135, 0.1434, 0.145, 0.1480, 0.15, 0.1503, 0.152, 0.1521, 0.153,
    0.1541, 0.155, 0.1550, 0.155, 0.1549, 0.1545, 0.1543, 0.154, 0.1533, 0.153, 0.1527, 0.153, 0.1531, 0.154, 0.1557,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L39_3_m1spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0221, 0.025, 0.0288, 0.03, 0.0392, 0.05, 0.0693, 0.07, 0.0839, 0.09, 0.0953, 0.1, 0.1083, 0.11, 0.1258, 0.126, 0.1263, 0.125, 0.1247, 0.122, 0.1205, 0.11, 0.1087, 0.11, 0.1140, 0.12, 0.1248, 0.125, 0.1279, 0.125, 0.1207, 0.115, 0.1104, 0.1, 0.0988, 0.09, 0.0876, 0.085, 0.0816, 0.08, 0.0799, 0.08,
    0.0877, 0.09, 0.1083, 0.11, 0.1265, 0.13, 0.1367, 0.14, 0.1410, 0.142, 0.1431, 0.144, 0.1449, 0.146, 0.1470, 0.1474, 0.1479, 0.1478, 0.1478, 0.1475, 0.1472, 0.147, 0.1460, 0.146, 0.1454, 0.1456, 0.1458, 0.147, 0.1485,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_3_m2spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0242, 0.03, 0.0316, 0.035, 0.0430, 0.06, 0.080, 0.088, 0.0923, 0.1, 0.1049, 0.11, 0.1192, 0.12, 0.1372, 0.1375, 0.1375, 0.136, 0.1356, 0.134, 0.1308, 0.13, 0.1284, 0.12, 0.1232, 0.13, 0.1333, 0.134, 0.1355, 0.13, 0.1272, 0.12,
    0.1157, 0.11, 0.1029, 0.1, 0.0906, 0.09, 0.0840, 0.083, 0.0822, 0.09, 0.0913, 0.1, 0.1150, 0.12, 0.1360, 0.14, 0.1477, 0.15, 0.1526, 0.154, 0.1550, 0.156, 0.1570, 0.158,
    0.1593, 0.16, 0.1603, 0.1602, 0.1602, 0.16, 0.1595, 0.1594, 0.1582, 0.158, 0.1575, 0.158, 0.1580, 0.159, 0.1609,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L36_2_2spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0173, 0.02, 0.0222, 0.025, 0.0296, 0.05, 0.0501, 0.055, 0.0621, 0.07, 0.0704, 0.075, 0.0798, 0.08, 0.0852, 0.085, 0.0856, 0.095, 0.0945, 0.083, 0.0815, 0.081, 0.0806, 0.084, 0.0864, 0.09, 0.0975, 0.1, 0.1021, 0.1, 0.0979, 0.095,
    0.0909, 0.09, 0.0827, 0.08, 0.0747, 0.073, 0.0703, 0.07, 0.0692, 0.07, 0.0752, 0.08, 0.0908, 0.11, 0.1146, 0.12, 0.1224, 0.123, 0.1256, 0.126, 0.1273, 0.128, 0.1287, 0.123,
    0.1202, 0.1205, 0.1209, 0.1208, 0.1208, 0.1205, 0.1203, 0.12, 0.1195, 0.1194, 0.1191, 0.1193, 0.1194, 0.12, 0.1213,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_4_m2spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0248, 0.03, 0.0323, 0.04, 0.0437, 0.06, 0.0752, 0.08, 0.0937, 0.1, 0.1064, 0.11, 0.1208, 0.124, 0.1289, 0.129, 0.1292, 0.128, 0.1271, 0.125, 0.1222, 0.12, 0.1197, 0.12, 0.1246, 0.13, 0.1350, 0.136, 0.1373, 0.13, 0.1289, 0.12, 0.1174, 0.11,
    0.1045, 0.1, 0.0921, 0.09, 0.0854, 0.084, 0.0837, 0.09, 0.0931, 0.1, 0.1177, 0.12, 0.1396, 0.14, 0.1517, 0.153, 0.1568, 0.158, 0.1593, 0.16, 0.1613, 0.162, 0.1636, 0.164,
    0.1647, 0.1646, 0.1646, 0.164, 0.1639, 0.163, 0.1626, 0.162, 0.1619, 0.162, 0.1624, 0.164, 0.1654,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L41_1_m6spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0259, 0.03, 0.0352, 0.04, 0.0494, 0.07, 0.0872, 0.1, 0.1094, 0.11, 0.1250, 0.13, 0.1427, 0.15, 0.1530, 0.153, 0.1539, 0.153, 0.1522, 0.15, 0.1470, 0.145, 0.1436, 0.145, 0.1469, 0.15, 0.1549, 0.155, 0.1541, 0.15, 0.1421, 0.13, 0.1269, 0.12,
    0.1105, 0.1, 0.0951, 0.09, 0.0866, 0.085, 0.0840, 0.09, 0.0926, 0.1, 0.1159, 0.12, 0.1366, 0.14, 0.1482, 0.15, 0.1531, 0.154, 0.1555, 0.156, 0.1577, 0.16, 0.1603, 0.161,
    0.1616, 0.1615, 0.1614, 0.161, 0.1605, 0.16, 0.1588, 0.158, 0.1579, 0.158, 0.1585, 0.16, 0.1620,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_4_m2spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0248, 0.03, 0.0323, 0.04, 0.0437, 0.06, 0.0752, 0.08, 0.0937, 0.1, 0.1064, 0.11, 0.1208, 0.134, 0.1389, 0.139, 0.1292, 0.128, 0.1271, 0.125, 0.1222, 0.12, 0.1197, 0.12, 0.1246, 0.13, 0.1350, 0.136, 0.1373, 0.13, 0.1289, 0.12, 0.1174, 0.11,
    0.1145, 0.12, 0.0921, 0.09, 0.0854, 0.084, 0.0837, 0.09, 0.0931, 0.1, 0.1177, 0.12, 0.1396, 0.14, 0.1517, 0.153, 0.1568, 0.158, 0.1593, 0.16, 0.1613, 0.162, 0.1636, 0.164,
    0.1647, 0.1646, 0.1646, 0.164, 0.1639, 0.163, 0.1626, 0.162, 0.1619, 0.162, 0.1624, 0.164, 0.1654,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L41_1_m6spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0259, 0.03, 0.0352, 0.04, 0.0494, 0.07, 0.0872, 0.1, 0.1194, 0.12, 0.1350, 0.14, 0.1527, 0.15, 0.1530, 0.153, 0.1539, 0.153, 0.1522, 0.15, 0.1470, 0.145, 0.1436, 0.145, 0.1469, 0.15, 0.1549, 0.155, 0.1541, 0.15, 0.1421, 0.13, 0.1269, 0.12,
    0.1105, 0.1, 0.0951, 0.09, 0.0866, 0.085, 0.0840, 0.09, 0.0926, 0.1, 0.1259, 0.13, 0.1366, 0.14, 0.1482, 0.15, 0.1531, 0.154, 0.1555, 0.156, 0.1577, 0.16, 0.1603, 0.161,
    0.1616, 0.1615, 0.1614, 0.161, 0.1605, 0.16, 0.1588, 0.158, 0.1579, 0.158, 0.1585, 0.16, 0.1620,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L41_12_14spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0272, 0.03, 0.0306, 0.033, 0.0354, 0.04, 0.0536, 0.06, 0.0644, 0.07, 0.0705, 0.074, 0.0774, 0.078, 0.0799, 0.079, 0.0780, 0.075, 0.0744, 0.07, 0.0694, 0.069, 0.0688, 0.07, 0.0795, 0.09, 0.0995, 0.1, 0.1121, 0.113, 0.1134, 0.112,
    0.1112, 0.11, 0.1069, 0.105, 0.1018, 0.1, 0.0993, 0.1, 0.1001, 0.11, 0.1147, 0.13, 0.1503, 0.16, 0.1815, 0.19, 0.1986, 0.2, 0.2058, 0.206, 0.2093, 0.21, 0.2114, 0.212,
    0.2132, 0.2134, 0.2137, 0.2139, 0.2139, 0.2138, 0.2137, 0.2134, 0.2133, 0.2132, 0.2131, 0.2132, 0.2132, 0.214, 0.2152,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L41_12_14spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0272, 0.03, 0.0306, 0.035, 0.0354, 0.04, 0.0536, 0.06, 0.0644, 0.07, 0.0705, 0.074, 0.0774, 0.078, 0.0799, 0.079, 0.0780, 0.075, 0.0744, 0.07, 0.0694, 0.069, 0.0688, 0.07, 0.0795, 0.09, 0.0995, 0.1, 0.1121, 0.113, 0.1134, 0.112,
    0.1112, 0.11, 0.1069, 0.105, 0.1018, 0.1, 0.0993, 0.11, 0.1101, 0.12, 0.1147, 0.13, 0.1503, 0.16, 0.1815, 0.19, 0.1986, 0.2, 0.2058, 0.206, 0.2093, 0.21, 0.2114, 0.212,
    0.2132, 0.2134, 0.2137, 0.2239, 0.2139, 0.2158, 0.2237, 0.2234, 0.2133, 0.2132, 0.2131, 0.2132, 0.2132, 0.214, 0.2152,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L10_0_m22spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0048, 0.006, 0.0077, 0.009, 0.0124, 0.015, 0.0238, 0.025, 0.0306, 0.033, 0.0356, 0.04, 0.0410, 0.042, 0.0442, 0.0443, 0.0443, 0.044, 0.0438, 0.043, 0.0422, 0.041, 0.0400, 0.038, 0.0363, 0.035, 0.0310, 0.03, 0.0250, 0.02, 0.0188, 0.015, 0.0130, 0.01,
    0.0073, 0.005, 0.0024, 0.0, -0.0004, -0.0005, -0.0014, 0.0, 0.0004, 0.001, 0.0057, 0.01, 0.0104, 0.012, 0.0130, 0.013, 0.0141, 0.0144, 0.0146, 0.015, 0.0151, 0.0155,
    0.0158, 0.016, 0.0162, 0.0162, 0.0161, 0.016, 0.0158, 0.0155, 0.0153, 0.015, 0.0149, 0.015, 0.0152, 0.016, 0.0162,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L38_60_8spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0621, 0.0625, 0.0627, 0.0627, 0.0627, 0.07, 0.0842, 0.09, 0.0982, 0.1, 0.1014, 0.102, 0.1024, 0.1, 0.0937, 0.08, 0.0780, 0.07, 0.0591, 0.05, 0.0416, 0.03, 0.0288, 0.025, 0.0210, 0.02, 0.0170, 0.016, 0.0150, 0.016, 0.0173, 0.02, 0.0226, 0.025,
    0.0263, 0.027, 0.0279, 0.029, 0.0298, 0.03, 0.0382, 0.07, 0.0962, 0.12, 0.2343, 0.3, 0.3551, 0.4, 0.4203, 0.43, 0.4476, 0.45, 0.4592, 0.46, 0.4651, 0.466, 0.4686, 0.469,
    0.4694, 0.48, 0.4706, 0.471, 0.4711, 0.4712, 0.4717, 0.472, 0.4721, 0.472, 0.4719, 0.473, 0.4745,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L49_85_39spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1217, 0.12, 0.1146, 0.11, 0.1016, 0.11, 0.1161, 0.12, 0.1275, 0.123, 0.1207, 0.11, 0.1072, 0.09, 0.0793, 0.06, 0.0447, 0.05, 0.0051, 0.0, -0.0289, -0.03, -0.0507, -0.055, -0.0582, -0.055, -0.0533, -0.05, -0.0417, -0.02, -0.0198, 0.0, 0.0081, 0.02, 0.0321, 0.04,
    0.0494, 0.05, 0.0614, 0.07, 0.0824, 0.1, 0.2039, 0.35, 0.4906, 0.6, 0.7408, 0.84, 0.8758, 0.9, 0.9322, 0.94, 0.9562, 0.96, 0.9675, 0.97, 0.9731, 0.974, 0.9738, 0.975,
    0.9766, 0.977, 0.9786, 0.98, 0.9814, 0.983, 0.9833, 0.983, 0.9822, 0.984, 0.9851,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L42_1_m18spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0312, 0.04, 0.0454, 0.06, 0.0676, 0.1, 0.1240, 0.14, 0.1575, 0.17, 0.1813, 0.19, 0.2081, 0.21, 0.2238, 0.223, 0.2249, 0.223, 0.2224, 0.22, 0.2149, 0.21, 0.2071, 0.205, 0.2014, 0.2, 0.1956, 0.19, 0.1811, 0.17, 0.1572, 0.14, 0.1316, 0.12, 0.1053, 0.1,
    0.0815, 0.07, 0.0682, 0.065, 0.0639, 0.07, 0.0744, 0.09, 0.1040, 0.11, 0.1304, 0.135, 0.1451, 0.15, 0.1513, 0.153, 0.1542, 0.155, 0.1571, 0.16, 0.1608, 0.161, 0.1628, 0.1625,
    0.1625, 0.162, 0.1610, 0.16, 0.1585, 0.157, 0.1569, 0.157, 0.1579, 0.16, 0.1630,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::Colorlab_L48_19_m25spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0617, 0.07, 0.0822, 0.1, 0.1139, 0.15, 0.2009, 0.22, 0.2533, 0.27, 0.2882, 0.3, 0.3264, 0.33, 0.3452, 0.342, 0.3406, 0.335, 0.3301, 0.32, 0.3131, 0.3, 0.2953, 0.28, 0.2766, 0.26, 0.2542, 0.24, 0.2234, 0.2, 0.1866, 0.17, 0.1506, 0.13, 0.1139, 0.1, 0.0806, 0.07,
    0.0623, 0.06, 0.0591, 0.08, 0.0942, 0.14, 0.1841, 0.21, 0.2635, 0.28, 0.3069, 0.31, 0.3251, 0.33, 0.3331, 0.335, 0.3391, 0.34, 0.3455, 0.346, 0.3487, 0.3467, 0.3486, 0.347,
    0.3467, 0.345, 0.3432, 0.342, 0.3411, 0.342, 0.3425, 0.35, 0.3505,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L30_21_m25spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0291, 0.03, 0.0386, 0.04, 0.0533, 0.08, 0.0940, 0.1, 0.1187, 0.12, 0.1349, 0.14, 0.1524, 0.16, 0.1605, 0.16, 0.1574, 0.155, 0.1515, 0.15, 0.1428, 0.14, 0.1330, 0.13, 0.1204, 0.11, 0.1039, 0.09, 0.0852, 0.07, 0.0667, 0.05,
    0.0496, 0.04, 0.0326, 0.02, 0.0173, 0.01, 0.0089, 0.008, 0.0078, 0.005, 0.0265, 0.005, 0.0738, 0.1, 0.1155, 0.12, 0.1383, 0.14, 0.1478, 0.15, 0.1519, 0.154, 0.1548, 0.156,
    0.1579, 0.158, 0.1594, 0.1594, 0.1594, 0.159, 0.1585, 0.1575, 0.1570, 0.156, 0.1559, 0.156, 0.1566, 0.159, 0.1603,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L15_10_m15spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0079, 0.01, 0.0107, 0.012, 0.0149, 0.02, 0.0264, 0.03, 0.0334, 0.035, 0.0381, 0.04, 0.0431, 0.044, 0.0456, 0.045, 0.0449, 0.044, 0.0435, 0.042, 0.0412, 0.04, 0.0386, 0.036, 0.0355, 0.033, 0.0316, 0.03, 0.0268, 0.025, 0.0216, 0.02, 0.0167, 0.015,
    0.0118, 0.01, 0.0073, 0.006, 0.0049, 0.004, 0.0045, 0.008, 0.0092, 0.001, 0.0212, 0.003, 0.0318, 0.0033, 0.0376, 0.04, 0.0401, 0.042, 0.0411, 0.0411, 0.0419, 0.042,
    0.0428, 0.043, 0.0432, 0.0432, 0.0432, 0.043, 0.0429, 0.0426, 0.0425, 0.0423, 0.0422, 0.0423, 0.0424, 0.043, 0.0434,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L48_19_m25spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0617, 0.07, 0.0822, 0.1, 0.1339, 0.25, 0.2009, 0.22, 0.2533, 0.29, 0.2882, 0.333, 0.3264, 0.38, 0.3452, 0.362, 0.3406, 0.335, 0.3301, 0.32, 0.3131, 0.3, 0.2953, 0.28, 0.2766, 0.26, 0.2542, 0.24, 0.2234, 0.2, 0.1866, 0.17, 0.1506, 0.13, 0.1139, 0.1, 0.0806, 0.07,
    0.0623, 0.06, 0.0591, 0.08, 0.0942, 0.14, 0.1841, 0.21, 0.2635, 0.28, 0.3069, 0.31, 0.3251, 0.33, 0.3331, 0.335, 0.3391, 0.34, 0.3455, 0.346, 0.3487, 0.3467, 0.3486, 0.347,
    0.3467, 0.345, 0.3432, 0.342, 0.3411, 0.342, 0.3425, 0.35, 0.3505,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L30_21_m25spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0291, 0.03, 0.0386, 0.04, 0.0533, 0.08, 0.0940, 0.1, 0.1187, 0.12, 0.1349, 0.15, 0.1624, 0.18, 0.1805, 0.18, 0.1774, 0.165, 0.1715, 0.15, 0.1428, 0.14, 0.1330, 0.13, 0.1204, 0.11, 0.1039, 0.09, 0.0852, 0.07, 0.0667, 0.05,
    0.0496, 0.04, 0.0326, 0.02, 0.0173, 0.01, 0.0089, 0.008, 0.0078, 0.005, 0.0265, 0.005, 0.0738, 0.1, 0.1155, 0.12, 0.1383, 0.14, 0.1478, 0.15, 0.1519, 0.154, 0.1548, 0.156,
    0.1579, 0.158, 0.1594, 0.1594, 0.1594, 0.159, 0.1585, 0.1575, 0.1570, 0.156, 0.1559, 0.156, 0.1566, 0.159, 0.1603,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L15_10_m15spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0079, 0.01, 0.0107, 0.012, 0.0149, 0.02, 0.0364, 0.04, 0.0434, 0.045, 0.0481, 0.04, 0.0431, 0.044, 0.0456, 0.045, 0.0449, 0.044, 0.0435, 0.042, 0.0412, 0.04, 0.0386, 0.036, 0.0355, 0.033, 0.0316, 0.03, 0.0268, 0.025, 0.0216, 0.02, 0.0167, 0.015,
    0.0118, 0.01, 0.0073, 0.006, 0.0049, 0.004, 0.0045, 0.008, 0.0092, 0.001, 0.0212, 0.003, 0.0318, 0.0033, 0.0376, 0.04, 0.0401, 0.042, 0.0411, 0.0411, 0.0419, 0.042,
    0.0528, 0.053, 0.0532, 0.0532, 0.0532, 0.043, 0.0429, 0.0426, 0.0425, 0.0423, 0.0422, 0.0423, 0.0424, 0.043, 0.0434,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L60_26_m25spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1040, 0.12, 0.1343, 0.15, 0.1812, 0.25, 0.3138, 0.35, 0.3938, 0.42, 0.4460, 0.48, 0.5027, 0.51, 0.5290, 0.52, 0.5196, 0.51, 0.5009, 0.49, 0.4726, 0.46, 0.4448, 0.43, 0.4186, 0.4, 0.3893, 0.36, 0.3469, 0.3, 0.2943, 0.28, 0.2426, 0.2, 0.1892, 0.16, 0.1403, 0.13,
    0.1136, 0.112, 0.1104, 0.15, 0.1729, 0.2, 0.3309, 0.4, 0.4702, 0.5, 0.5463, 0.56, 0.5782, 0.58, 0.5922, 0.6, 0.6022, 0.61, 0.6123, 0.615, 0.6171, 0.616, 0.6173, 0.616,
    0.6145, 0.61, 0.6095, 0.62, 0.6064, 0.607, 0.6085, 0.61, 0.6208,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_26_m45spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0615, 0.07, 0.0862, 0.1, 0.1249, 0.2, 0.2272, 0.25, 0.2893, 0.3, 0.3312, 0.35, 0.3766, 0.38, 0.3991, 0.395, 0.3933, 0.39, 0.3810, 0.37, 0.3613, 0.35, 0.3361, 0.3, 0.2979, 0.26, 0.2449, 0.2, 0.1884, 0.15, 0.1364, 0.1, 0.0895, 0.06, 0.0437, 0.04,
    0.0034, 0.0, -0.0191, -0.002, -0.0235, 0.0, 0.0145, 0.1, 0.1129, 0.15, 0.2000, 0.22, 0.2475, 0.25, 0.2673, 0.27, 0.2757, 0.28, 0.2822, 0.285, 0.2894, 0.29,
    0.2933, 0.293, 0.2932, 0.292, 0.2908, 0.29, 0.2865, 0.286, 0.2837, 0.285, 0.2855, 0.29, 0.2947,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_26_m45spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0615, 0.07, 0.0862, 0.1, 0.1249, 0.2, 0.2272, 0.25, 0.2893, 0.3, 0.3312, 0.35, 0.3766, 0.38, 0.3991, 0.395, 0.3933, 0.39, 0.3810, 0.37, 0.3613, 0.35, 0.3361, 0.3, 0.2979, 0.26, 0.2449, 0.2, 0.1884, 0.15, 0.1364, 0.1, 0.0895, 0.06, 0.0437, 0.04,
    0.0034, 0.0, -0.0191, -0.002, -0.0235, 0.0, 0.0145, 0.1, 0.1129, 0.15, 0.2000, 0.22, 0.2475, 0.25, 0.2673, 0.27, 0.2757, 0.28, 0.2822, 0.285, 0.2894, 0.29,
    0.2933, 0.293, 0.2932, 0.292, 0.2908, 0.29, 0.2865, 0.286, 0.2837, 0.285, 0.2855, 0.29, 0.2947,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L20_10_m45spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0196, 0.02, 0.0309, 0.04, 0.0487, 0.08, 0.0931, 0.1, 0.1199, 0.12, 0.1388, 0.14, 0.1595, 0.16, 0.1708, 0.17, 0.1699, 0.167, 0.1665, 0.16, 0.1595, 0.15, 0.1489, 0.14, 0.1303, 0.12, 0.1032, 0.09, 0.0751, 0.06, 0.0499, 0.03, 0.0273, 0.04,
    0.0056, 0.0, -0.0131, -0.02, -0.0237, -0.025, -0.0268, -0.02, -0.0165, 0.01, 0.0117, 0.02, 0.0369, 0.04, 0.0507, 0.051, 0.0564, 0.057, 0.0588, 0.06, 0.0610, 0.062,
    0.0639, 0.064, 0.0656, 0.0655, 0.0654, 0.065, 0.0642, 0.063, 0.0622, 0.061, 0.0608, 0.061, 0.0617, 0.064, 0.0656,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L20_10_m45spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0196, 0.02, 0.0309, 0.04, 0.0487, 0.08, 0.0931, 0.1, 0.1299, 0.14, 0.1588, 0.16, 0.1795, 0.18, 0.1908, 0.18, 0.1799, 0.187, 0.1665, 0.16, 0.1595, 0.15, 0.1489, 0.14, 0.1303, 0.12, 0.1032, 0.09, 0.0751, 0.06, 0.0499, 0.03, 0.0273, 0.04,
    0.0056, 0.0, 0., 0., 0., 0., 0., 0., 0., 0.01, 0.0117, 0.02, 0.0369, 0.04, 0.0507, 0.051, 0.0564, 0.057, 0.0588, 0.06, 0.0610, 0.062,
    0.0639, 0.064, 0.0656, 0.0655, 0.0654, 0.065, 0.0642, 0.063, 0.0622, 0.061, 0.0608, 0.061, 0.0617, 0.064, 0.0656,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L20_10_m45spect3[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0196, 0.02, 0.0309, 0.04, 0.0387, 0.08, 0.0831, 0.1, 0.1099, 0.11, 0.1188, 0.11, 0.1395, 0.14, 0.1508, 0.16, 0.1599, 0.157, 0.1465, 0.16, 0.1595, 0.15, 0.1489, 0.14, 0.1303, 0.12, 0.1032, 0.09, 0.0751, 0.06, 0.0499, 0.03, 0.0273, 0.04,
    0.0056, 0.0, -0.0131, -0.02, -0.0237, -0.025, -0.0268, -0.02, -0.0165, 0.01, 0.0117, 0.02, 0.0369, 0.04, 0.0507, 0.051, 0.0564, 0.057, 0.0588, 0.06, 0.0610, 0.062,
    0.0639, 0.064, 0.0656, 0.0655, 0.0654, 0.065, 0.0642, 0.063, 0.0622, 0.061, 0.0608, 0.061, 0.0617, 0.064, 0.0656,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyK3_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4939, 0.435, 0.3859, 0.403, 0.4298, 0.456, 0.4880, 0.525, 0.5528, 0.572, 0.5672, 0.578, 0.5880, 0.595, 0.5994, 0.602, 0.6029, 0.600, 0.5981, 0.588, 0.5808, 0.571, 0.5618, 0.551,
    0.5369, 0.503, 0.4819, 0.452, 0.4190, 0.404, 0.3921, 0.386, 0.3815, 0.364, 0.3400, 0.321, 0.2991, 0.298, 0.2977, 0.304, 0.3090, 0.309, 0.3088, 0.302, 0.2930, 0.284, 0.2753, 0.271,
    0.2660, 0.265, 0.2636, 0.266, 0.2678, 0.275, 0.2811, 0.290, 0.2995, 0.306, 0.3125, 0.314, 0.3153, 0.313, 0.3111, 0.307, 0.3006, 0.298, .2952, 0.306, 0.3116, 0.325, 0.3584,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyK9_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.4058, 0.441, 0.4734, 0.562, 0.5572, 0.65, 0.6051, 0.643, 0.7098, 0.725, 0.7392, 0.735, 0.7118, 0.712, 0.7135, 0.711, 0.7071, 0.702, 0.6938, 0.681, 0.6702, 0.663, 0.6511, 0.642,
    0.6282, 0.604, 0.5732, 0.542, 0.5103, 0.499, 0.4913, 0.492, 0.4926, 0.475, 0.4604, 0.452, 0.4341, 0.453, 0.4648, 0.496, 0.5111, 0.525, 0.5335, 0.531, 0.5283, 0.522, 0.5154, 0.512,
    0.5098, 0.509, 0.5093, 0.513, 0.5151, 0.523, 0.5309, 0.544, 0.5520, 0.562, 0.5642, 0.565, 0.5657, 0.562, 0.5598, 0.554, 0.5489, 0.546, 0.5430, 0.553, 0.5601, 0.576, 0.6067,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyC4_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3280, 0.2950, 0.2611, 0.334, 0.3981, 0.453, 0.4946, 0.548, 0.5692, 0.585, 0.5932, 0.611, 0.6512, 0.6621, 0.6507, 0.631, 0.6310, 0.625, 0.6181, 0.607, 0.5847, 0.563, 0.5488, 0.524,
    0.5066, 0.465, 0.4358, 0.398, 0.3585, 0.336, 0.3151, 0.302, 0.2855, 0.254, 0.2309, 0.203, 0.1786, 0.166, 0.1546, 0.149, 0.1443, 0.143, 0.1359, 0.131, 0.1245, 0.123, 0.115, 0.114,
    0.1120, 0.112, 0.1127, 0.114, 0.1169, 0.122, 0.1275, 0.133, 0.1421, 0.147, 0.1504, 0.149, 0.1488, 0.145, 0.1416, 0.136, 0.1303, 0.127, 0.1241, 0.132, 0.1355, 0.155, 0.1739,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyC14_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5697, 0.511, 0.4660, 0.481, 0.5500, 0.588, 0.5560, 0.633, 0.6572, 0.682, 0.6902, 0.693, 0.6932, 0.684, 0.6950, 0.699, 0.7069, 0.717, 0.7292, 0.735, 0.7488, 0.757, 0.7678, 0.773,
    0.7786, 0.776, 0.7721, 0.765, 0.7544, 0.746, 0.7394, 0.731, 0.7232, 0.704, 0.6889, 0.674, 0.6446, 0.631, 0.6171, 0.606, 0.5966, 0.585, 0.5743, 0.5570, 0.5425, 0.529, 0.5093, 0.498,
    0.4884, 0.482, 0.4784, 0.478, 0.4774, 0.481, 0.4822, 0.487, 0.4944, 0.503, 0.5076, 0.512, 0.5186, 0.522, 0.5268, 0.529, 0.5303, 0.532, 0.5332, 0.539, 0.5454, 0.565, 0.5760,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyE4_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1483, 0.161, 0.1756, 0.213, 0.2536, 0.283, 0.3584, 0.391, 0.3965, 0.437, 0.4589, 0.495, 0.4946, 0.526, 0.5427, 0.569, 0.5239, 0.522, 0.5193, 0.508, 0.4917, 0.476, 0.4569, 0.431,
    0.4123, 0.375, 0.3422, 0.309, 0.2672, 0.242, 0.2179, 0.208, 0.1820, 0.162, 0.1356, 0.113, 0.0972, 0.091, 0.0784, 0.073, 0.0698, 0.066, 0.0646, 0.062, 0.0592, 0.057, 0.0556, 0.055,
    0.0546, 0.055, 0.0551, 0.056, 0.0571, 0.059, 0.0611, 0.064, 0.0670, 0.069, 0.0701, 0.070, 0.0692, 0.067, 0.0661, 0.065, 0.0620, 0.061, 0.0606, 0.063, 0.0663, 0.072, 0.0834,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyM1_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3100, 0.303, 0.2922, 0.322, 0.3514, 0.376, 0.4342, 0.484, 0.4843, 0.497, 0.4969, 0.497, 0.5502, 0.557, 0.5633, 0.556, 0.5187, 0.518, 0.5179, 0.511, 0.5057, 0.497, 0.4928, 0.483,
    0.4729, 0.454, 0.4235, 0.398, 0.3643, 0.346, 0.3371, 0.329, 0.3234, 0.301, 0.2827, 0.263, 0.2418, 0.235, 0.2338, 0.235, 0.2370, 0.236, 0.2329, 0.226, 0.2184, 0.213, 0.2028, 0.198,
    0.1958, 0.194, 0.1937, 0.196, 0.1973, 0.203, 0.2084, 0.212, 0.2244, 0.233, 0.2351, 0.236, 0.2372, 0.234, 0.2331, 0.229, 0.2239, 0.222, 0.2178, 0.224, 0.2319, 0.251, 0.2731,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSky2B1_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5277, 0.485, 0.4431, 0.476, 0.5472, 0.599, 0.5920, 0.667, 0.6887, 0.693, 0.6950, 0.721, 0.7401, 0.737, 0.7640, 0.718, 0.7202, 0.720, 0.7193, 0.713, 0.7053, 0.695, 0.6891, 0.674,
    0.6657, 0.632, 0.6181, 0.587, 0.5614, 0.543, 0.5312, 0.521, 0.5101, 0.483, 0.4589, 0.431, 0.4045, 0.398, 0.3857, 0.385, 0.3826, 0.376, 0.3751, 0.364, 0.3574, 0.346, 0.3393, 0.335,
    0.3314, 0.331, 0.3304, 0.333, 0.3368, 0.346, 0.3523, 0.363, 0.3742, 0.382, 0.3874, 0.385, 0.3883, 0.384, 0.3818, 0.375, 0.3693, 0.364, 0.3616, 0.374, 0.3800, 0.396, 0.4324,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyT7_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.1943, 0.256, 0.3199, 0.376, 0.4836, 0.554, 0.5843, 0.592, 0.6643, 0.661, 0.6899, 0.694, 0.6939, 0.708, 0.7525, 0.756, 0.7329, 0.741, 0.7482, 0.751, 0.7527, 0.752, 0.7514, 0.745,
    0.7383, 0.721, 0.7028, 0.675, 0.6526, 0.631, 0.6034, 0.589, 0.5500, 0.512, 0.4708, 0.432, 0.3848, 0.342, 0.3268, 0.311, 0.2929, 0.282, 0.2712, 0.261, 0.2493, 0.236, 0.2316, 0.227,
    0.2243, 0.223, 0.2234, 0.229, 0.2288, 0.235, 0.2436, 0.255, 0.2640, 0.268, 0.2762, 0.277, 0.2767, 0.272, 0.2693, 0.263, 0.2566, 0.254, 0.2489, 0.255, 0.2665, 0.275, 0.3165,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyU19_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.5829, 0.534, 0.4865, 0.489, 0.5227, 0.552, 0.5890, 0.633, 0.6621, 0.661, 0.6832, 0.682, 0.6928, 0.694, 0.6932, 0.687, 0.6989, 0.688, 0.6884, 0.683, 0.6771, 0.671, 0.6648, 0.665,
    0.6465, 0.622, 0.6038, 0.583, 0.5524, 0.542, 0.5297, 0.523, 0.5194, 0.492, 0.4797, 0.451, 0.4387, 0.436, 0.4356, 0.442, 0.4455, 0.445, 0.4444, 0.432, 0.4282, 0.413, 0.4094, 0.404,
    0.4009, 0.400, 0.3992, 0.402, 0.4046, 0.411, 0.4185, 0.426, 0.4385, 0.446, 0.4515, 0.452, 0.4545, 0.452, 0.4505, 0.446, 0.4411, 0.438, 0.4368, 0.443, 0.4539, 0.467, 0.5013,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::ColorBlueSkyU2_spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.3594, 0.345, 0.3625, 0.363, 0.3614, 0.376, 0.3854, 0.397, 0.4497, 0.484, 0.4960, 0.496, 0.4990, 0.497, 0.4993, 0.494, 0.5302, 0.537, 0.5434, 0.538, 0.5476, 0.564, 0.5745, 0.583,
    0.5940, 0.593, 0.5901, 0.580, 0.5703, 0.563, 0.5545, 0.546, 0.5384, 0.521, 0.5029, 0.478, 0.4592, 0.444, 0.4334, 0.421, 0.4149, 0.408, 0.3947, 0.378, 0.3657, 0.352, 0.3363, 0.324,
    0.3177, 0.313, 0.3087, 0.308, 0.3077, 0.310, 0.3123, 0.317, 0.3231, 0.329, 0.3351, 0.339, 0.3454, 0.348, 0.3520, 0.353, 0.3545, 0.355, 0.3562, 0.359, 0.3674, 0.375, 0.3976,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L40_1_m40spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0411, 0.05, 0.0646, 0.09, 0.1016, 0.15, 0.1929, 0.22, 0.2476, 0.26, 0.2868, 0.3, 0.3304, 0.34, 0.3556, 0.356, 0.3564, 0.352, 0.3519, 0.34, 0.3395, 0.33, 0.3223, 0.3, 0.2967, 0.28, 0.2606, 0.23, 0.2170, 0.19, 0.1700, 0.15, 0.1245, 0.1, 0.0798, 0.06,
    0.0405, 0.03, 0.0183, 0.015, 0.0110, 0.02, 0.0267, 0.05, 0.0715, 0.1, 0.1117, 0.12, 0.1339, 0.14, 0.1432, 0.145, 0.1473, 0.15, 0.1516, 0.155, 0.1574, 0.16, 0.1608, 0.1605,
    0.1602, 0.16, 0.1578, 0.155, 0.1535, 0.153, 0.1507, 0.152, 0.1525, 0.158, 0.1605,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L30_4_m30spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0229, 0.03, 0.0346, 0.04, 0.0531, 0.08, 0.0995, 0.11, 0.1274, 0.13, 0.1471, 0.15, 0.1690, 0.17, 0.1813, 0.182, 0.1812, 0.18, 0.1784, 0.175, 0.1716, 0.17, 0.1627, 0.16, 0.1501, 0.14, 0.1326, 0.12, 0.1112, 0.1, 0.0880, 0.07, 0.0655, 0.05,
    0.0433, 0.03, 0.0237, 0.02, 0.0126, 0.009, 0.0092, 0.01, 0.0190, 0.03, 0.0460, 0.06, 0.0701, 0.075, 0.0834, 0.085, 0.0889, 0.09, 0.0914, 0.092, 0.0937, 0.095, 0.0967, 0.097,
    0.0984, 0.0982, 0.0982, 0.098, 0.0970, 0.096, 0.0949, 0.094, 0.0935, 0.094, 0.0944, 0.097, 0.0985,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L8_11_m25spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0057, 0.007, 0.0084, 0.01, 0.0126, 0.02, 0.0234, 0.025, 0.0300, 0.032, 0.0345, 0.036, 0.0395, 0.04, 0.0420, 0.042, 0.0415, 0.041, 0.0404, 0.04, 0.0385, 0.037, 0.0358, 0.034, 0.0314, 0.03, 0.0250, 0.02, 0.0184, 0.015, 0.0126, 0.01,
    0.0073, 0.005, 0.0022, 0.0, -0.0022, -0.003, -0.0047, -0.005, -0.0053, -0.003, -0.0019, 0.0, 0.0070, 0.01, 0.0149, 0.016, 0.0192, 0.02, 0.0210, 0.022, 0.0217, 0.022, 0.0224, 0.0224,
    0.0231, 0.0234, 0.0235, 0.0235, 0.0235, 0.0233, 0.0232, 0.023, 0.0227, 0.0225, 0.0224, 0.0225, 0.0226, 0.023, 0.0236,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::Colorlab_L40_1_m40spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0411, 0.05, 0.0686, 0.09, 0.1016, 0.15, 0.1929, 0.22, 0.2776, 0.28, 0.2968, 0.34, 0.3804, 0.39, 0.3756, 0.356, 0.3564, 0.352, 0.3519, 0.34, 0.3395, 0.33, 0.3223, 0.3, 0.2967, 0.28, 0.2606, 0.23, 0.2170, 0.19, 0.1700, 0.15, 0.1245, 0.1, 0.0798, 0.06,
    0.0405, 0.03, 0.0183, 0.015, 0.0110, 0.02, 0.0267, 0.05, 0.0715, 0.1, 0.1117, 0.12, 0.1339, 0.14, 0.1432, 0.145, 0.1473, 0.15, 0.1516, 0.155, 0.1574, 0.16, 0.1608, 0.1605,
    0.1602, 0.16, 0.1578, 0.155, 0.1535, 0.153, 0.1507, 0.152, 0.1525, 0.158, 0.1605,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L30_4_m30spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0229, 0.03, 0.0346, 0.04, 0.0531, 0.08, 0.0995, 0.11, 0.1274, 0.13, 0.1871, 0.19, 0.1990, 0.21, 0.2013, 0.212, 0.2012, 0.18, 0.1784, 0.175, 0.1716, 0.17, 0.1627, 0.16, 0.1501, 0.14, 0.1326, 0.12, 0.1112, 0.1, 0.0880, 0.07, 0.0655, 0.05,
    0.0433, 0.03, 0.0237, 0.02, 0.0126, 0.009, 0.0092, 0.01, 0.0190, 0.03, 0.0460, 0.06, 0.0701, 0.075, 0.0834, 0.085, 0.0889, 0.09, 0.0914, 0.092, 0.0937, 0.095, 0.0967, 0.097,
    0.0984, 0.0982, 0.0982, 0.098, 0.0970, 0.096, 0.0949, 0.094, 0.0935, 0.094, 0.0944, 0.097, 0.0985,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L8_11_m25spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0057, 0.007, 0.0094, 0.01, 0.0126, 0.02, 0.0234, 0.045, 0.0300, 0.032, 0.0445, 0.036, 0.0395, 0.04, 0.0440, 0.042, 0.0415, 0.041, 0.0404, 0.04, 0.0455, 0.037, 0.0358, 0.034, 0.0314, 0.03, 0.0250, 0.02, 0.0184, 0.015, 0.0126, 0.01,
    0.0073, 0.005, 0.0022, 0.0, -0.0022, -0.003, -0.0047, -0.005, -0.0053, -0.003, -0.0019, 0.0, 0.0070, 0.01, 0.0149, 0.016, 0.0192, 0.02, 0.0210, 0.022, 0.0217, 0.022, 0.0224, 0.0224,
    0.0231, 0.0234, 0.0235, 0.0235, 0.0235, 0.0233, 0.0232, 0.023, 0.0227, 0.0225, 0.0224, 0.0225, 0.0226, 0.023, 0.0236,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};


const double ColorTemp::Colorlab_L26_m8_m25spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0133, 0.02, 0.0223, 0.03, 0.0365, 0.05, 0.0706, 0.08, 0.0908, 0.1, 0.1057, 0.11, 0.1225, 0.13, 0.1328, 0.133, 0.1343, 0.134, 0.1337, 0.1232, 0.1300, 0.126, 0.1247, 0.12, 0.1174, 0.11, 0.1072, 0.1, 0.0930, 0.08, 0.0757, 0.06, 0.0583, 0.05,
    0.0410, 0.03, 0.0258, 0.02, 0.0172, 0.015, 0.0139, 0.015, 0.0166, 0.02, 0.0259, 0.03, 0.0344, 0.036, 0.0392, 0.04, 0.0412, 0.042, 0.0421, 0.043, 0.0435, 0.044,
    0.0456, 0.046, 0.0468, 0.0465, 0.0465, 0.046, 0.0455, 0.044, 0.0439, 0.043, 0.0428, 0.043, 0.0435, 0.045, 0.0465,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L26_m8_m25spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0133, 0.02, 0.0223, 0.03, 0.0385, 0.07, 0.0806, 0.09, 0.1008, 0.12, 0.1357, 0.13, 0.1425, 0.14, 0.1428, 0.143, 0.1443, 0.134, 0.1337, 0.1232, 0.1300, 0.126, 0.1247, 0.12, 0.1174, 0.11, 0.1072, 0.1, 0.0930, 0.08, 0.0757, 0.06, 0.0583, 0.05,
    0.0410, 0.03, 0.0258, 0.02, 0.0172, 0.015, 0.0139, 0.015, 0.0166, 0.02, 0.0259, 0.03, 0.0344, 0.036, 0.0392, 0.04, 0.0412, 0.042, 0.0421, 0.043, 0.0435, 0.044,
    0.0456, 0.046, 0.0468, 0.0465, 0.0465, 0.046, 0.0455, 0.044, 0.0439, 0.043, 0.0428, 0.043, 0.0435, 0.045, 0.0465,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L26_m8_m25spect3[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0133, 0.02, 0.0223, 0.03, 0.0265, 0.05, 0.0606, 0.08, 0.0808, 0.09, 0.1057, 0.10, 0.1125, 0.12, 0.1228, 0.123, 0.1243, 0.124, 0.1337, 0.1232, 0.1300, 0.126, 0.1247, 0.12, 0.1174, 0.11, 0.1072, 0.1, 0.0930, 0.08, 0.0757, 0.06, 0.0583, 0.05,
    0.0410, 0.03, 0.0258, 0.02, 0.0172, 0.015, 0.0139, 0.015, 0.0166, 0.02, 0.0259, 0.03, 0.0344, 0.036, 0.0392, 0.04, 0.0412, 0.042, 0.0421, 0.043, 0.0435, 0.044,
    0.0456, 0.046, 0.0468, 0.0465, 0.0465, 0.046, 0.0455, 0.044, 0.0439, 0.043, 0.0428, 0.043, 0.0435, 0.045, 0.0465,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

const double ColorTemp::Colorlab_L22_1_m42spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0170, 0.02, 0.0284, 0.03, 0.0463, 0.06, 0.0898, 0.1, 0.1160, 0.12, 0.1348, 0.14, 0.1557, 0.16, 0.1678, 0.168, 0.1680, 0.167, 0.1658, 0.164, 0.1600, 0.155, 0.1507, 0.14, 0.1346, 0.12, 0.1109, 0.1, 0.0853, 0.07, 0.0608, 0.05, 0.0380, 0.02, 0.0161, 0.0,
    -0.0028, -0.01, -0.0136, -0.015, -0.0172, -0.012, -0.0104, 0.0, 0.0095, 0.02, 0.0274, 0.03, 0.0372, 0.04, 0.0413, 0.042, 0.0431, 0.044, 0.0450, 0.046, 0.0477, 0.048, 0.0493, 0.049,
    0.0490, 0.048, 0.0478, 0.046, 0.0457, 0.045, 0.0444, 0.045, 0.0453, 0.048, 0.0490,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0    
};

const double ColorTemp::Colorlab_L22_1_m42spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0170, 0.02, 0.0284, 0.03, 0.0463, 0.06, 0.0898, 0.1, 0.1160, 0.12, 0.1548, 0.16, 0.1657, 0.17, 0.1778, 0.168, 0.1680, 0.167, 0.1658, 0.164, 0.1600, 0.155, 0.1507, 0.14, 0.1346, 0.12, 0.1109, 0.1, 0.0853, 0.07, 0.0608, 0.05, 0.0380, 0.02, 0.0161, 0.0,
    -0.0028, -0.01, -0.0136, -0.015, -0.0172, -0.012, -0.0104, 0.0, 0.0095, 0.02, 0.0274, 0.03, 0.0372, 0.04, 0.0413, 0.042, 0.0431, 0.044, 0.0450, 0.046, 0.0477, 0.048, 0.0493, 0.049,
    0.0490, 0.048, 0.0478, 0.046, 0.0457, 0.045, 0.0444, 0.045, 0.0453, 0.048, 0.0490,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0    
};

const double ColorTemp::Colorlab_L22_1_m42spect3[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0170, 0.02, 0.0284, 0.03, 0.0463, 0.06, 0.0898, 0.1, 0.1160, 0.12, 0.1548, 0.16, 0.1657, 0.17, 0.1778, 0.168, 0.1680, 0.167, 0.1658, 0.164, 0.1600, 0.155, 0.1507, 0.14, 0.1346, 0.12, 0.1109, 0.1, 0.0853, 0.07, 0.0608, 0.05, 0.0380, 0.02, 0.0161, 0.0,
    0.0, 0.0, 0.01, 0.01, 0.01, 0.01, 0.0, 0.0, 0.0095, 0.02, 0.0274, 0.03, 0.0372, 0.04, 0.0413, 0.042, 0.0431, 0.044, 0.0450, 0.046, 0.0477, 0.048, 0.0493, 0.049,
    0.0490, 0.048, 0.0478, 0.046, 0.0457, 0.045, 0.0444, 0.045, 0.0453, 0.048, 0.0490,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0    
};

const double ColorTemp::Colorlab_L22_1_m42spect4[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0170, 0.02, 0.0284, 0.03, 0.0463, 0.06, 0.0898, 0.1, 0.1160, 0.12, 0.1548, 0.16, 0.1657, 0.17, 0.1778, 0.168, 0.1680, 0.167, 0.1658, 0.164, 0.1600, 0.155, 0.1507, 0.14, 0.1346, 0.12, 0.1109, 0.1, 0.0853, 0.07, 0.0608, 0.05, 0.0380, 0.02, 0.0161, 0.0,
    0.0, 0.0, 0.01, 0.01, 0.01, 0.01, 0.0, 0.0, 0.0095, 0.02, 0.0274, 0.03, 0.0372, 0.04, 0.0413, 0.042, 0.0431, 0.044, 0.0450, 0.046, 0.0477, 0.048, 0.0493, 0.049,
    0.0490, 0.048, 0.0478, 0.046, 0.0457, 0.045, 0.0444, 0.045, 0.0453, 0.048, 0.0490,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0    
};

const double ColorTemp::Colorlab_L27_m1_m47spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0237, 0.03, 0.0400, 0.05, 0.0657, 0.09, 0.1278, 0.14, 0.1651, 0.18, 0.1921, 0.21, 0.2221, 0.23, 0.2395, 0.24, 0.2400, 0.24, 0.2371, 0.23, 0.2289, 0.22, 0.2159, 0.2, 0.1930, 0.18, 0.1595, 0.14, 0.1230, 0.1,
    0.0879, 0.07, 0.0553, 0.04, 0.0239, 0.0, -0.0034, -0.01, -0.0188, -0.02, -0.0241, -0.02, -0.0151, 0.0, 0.0117, 0.02, 0.0359, 0.04, 0.0492, 0.05, 0.0548, 0.055, 0.0571, 0.058,
    0.0598, 0.06, 0.0637, 0.065, 0.0660, 0.066, 0.0656, 0.064, 0.0638, 0.062, 0.0608, 0.06, 0.0589, 0.06, 0.0601, 0.065, 0.0655,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0       
};

const double ColorTemp::Colorlab_L27_m1_m47spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0237, 0.03, 0.0450, 0.05, 0.0657, 0.09, 0.1478, 0.16, 0.1751, 0.21, 0.2221, 0.23, 0.2421, 0.24, 0.2495, 0.25, 0.2500, 0.25, 0.2471, 0.24, 0.2389, 0.22, 0.2159, 0.2, 0.1930, 0.18, 0.1595, 0.14, 0.1230, 0.1,
    0.0879, 0.07, 0.0553, 0.04, 0.0239, 0.0, -0.0034, -0.01, -0.0188, -0.02, -0.0241, -0.02, -0.0151, 0.0, 0.0117, 0.02, 0.0359, 0.04, 0.0492, 0.05, 0.0548, 0.055, 0.0571, 0.058,
    0.0598, 0.06, 0.0637, 0.065, 0.0660, 0.066, 0.0656, 0.064, 0.0638, 0.062, 0.0608, 0.06, 0.0589, 0.06, 0.0601, 0.065, 0.0655,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0       
};

const double ColorTemp::Colorlab_L40_30_m30spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0552, 0.06, 0.0713, 0.08, 0.0964, 0.12, 0.1678, 0.19, 0.2112, 0.22, 0.2391, 0.25, 0.2689, 0.275, 0.2818, 0.28, 0.2749, 0.27, 0.2630, 0.25, 0.2464, 0.23, 0.2282, 0.21, 0.2049, 0.19, 0.1748, 0.16, 0.1415, 0.12, 0.1097, 0.1,
    0.0810, 0.07, 0.0523, 0.04, 0.0265, 0.02, 0.0123, 0.012, 0.0113, 0.03, 0.0487, 0.1, 0.1428, 0.18, 0.2257, 0.25, 0.2708, 0.28, 0.2897, 0.29, 0.2977, 0.3, 0.3033, 0.304,
    0.3088, 0.31, 0.3115, 0.3116, 0.3117, 0.3108, 0.3102, 0.31, 0.3076, 0.306, 0.3059, 0.306, 0.3070, 0.31, 0.3136,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0       
};

const double ColorTemp::Colorlab_L40_30_m30spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0552, 0.06, 0.0713, 0.08, 0.0864, 0.1, 0.1478, 0.19, 0.212, 0.22, 0.2391, 0.24, 0.2589, 0.255, 0.2818, 0.28, 0.2749, 0.27, 0.2630, 0.25, 0.2464, 0.23, 0.2282, 0.21, 0.2049, 0.19, 0.1748, 0.16, 0.1415, 0.12, 0.1097, 0.1,
    0.0810, 0.07, 0.0523, 0.04, 0.0265, 0.02, 0.0123, 0.012, 0.0113, 0.03, 0.0487, 0.1, 0.1428, 0.18, 0.2257, 0.25, 0.2708, 0.28, 0.2897, 0.29, 0.2977, 0.3, 0.3033, 0.304,
    0.3088, 0.3, 0.3015, 0.3016, 0.3017, 0.3108, 0.3002, 0.29, 0.2876, 0.286, 0.2859, 0.286, 0.2870, 0.28, 0.2836,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0       
};

const double ColorTemp::Colorlab_L40_20_m35spect[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0502, .06, 0.0694, 0.08, 0.0994, 0.13, 0.1794, 0.2, 0.2277, 0.25, 0.2604, 0.28, 0.2959, 0.3, 0.3137, 0.31, 0.3096, 0.305, 0.3004, 0.29, 0.2852, 0.27, 0.2669, 0.25, 0.2418, 0.22, 0.2080, 0.19,
    0.1696, 0.15, 0.1312, 0.1, 0.0955, 0.08, 0.0602, 0.04, 0.0288, 0.02, 0.0113, 0.01, 0.0078, 0.03, 0.0373, 0.1, 0.1136, 0.15, 0.1811, 0.2, 0.2180, 0.22, 0.2334, 0.24,
    0.2400, 0.241, 0.2452, 0.25, 0.2508, 0.252, 0.2538, 0.254, 0.2537, 0.252, 0.2519, 0.25, 0.2485, 0.246, 0.2464, 0.247, 0.2478, 0.26, 0.2550,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0       
};

const double ColorTemp::Colorlab_L40_20_m35spect2[97] = {
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    0.0502, .06, 0.0694, 0.08, 0.0994, 0.13, 0.1794, 0.2, 0.2277, 0.25, 0.2604, 0.28, 0.2959, 0.3, 0.3137, 0.31, 0.2896, 0.285, 0.2804, 0.27, 0.2652, 0.25, 0.2469, 0.25, 0.2418, 0.22, 0.2080, 0.19,
    0.1696, 0.15, 0.1312, 0.1, 0.0955, 0.08, 0.0602, 0.04, 0.0288, 0.02, 0.0113, 0.01, 0.0078, 0.03, 0.0373, 0.1, 0.1136, 0.15, 0.1811, 0.2, 0.2180, 0.22, 0.2334, 0.24,
    0.2400, 0.241, 0.2452, 0.23, 0.2308, 0.232, 0.2338, 0.234, 0.2337, 0.232, 0.2319, 0.23, 0.2385, 0.236, 0.2364, 0.237, 0.2378, 0.24, 0.2450,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0       
};



//3	A3	0.0552	0.0713	0.0964	0.1678	0.2112	0.2391	0.2689	0.2818	0.2749	0.2630	0.2464	0.2282	0.2049	0.1748	0.1415	0.1097	0.0810	0.0523	0.0265	0.0123	0.0113	0.0487	0.1428	0.2257	0.2708	0.2897	0.2977	0.3033	0.3088	0.3115	0.3117	0.3102	0.3076	0.3059	0.3070	0.3136
//4	A4	0.0502	0.0694	0.0994	0.1794	0.2277	0.2604	0.2959	0.3137	0.3096	0.3004	0.2852	0.2669	0.2418	0.2080	0.1696	0.1312	0.0955	0.0602	0.0288	0.0113	0.0078	0.0373	0.1136	0.1811	0.2180	0.2334	0.2400	0.2452	0.2508	0.2538	0.2537	0.2519	0.2485	0.2464	0.2478	0.2550


/*
 *      Name:   XYZtoCorColorTemp.c
 *
 *      Author: Bruce Justin Lindbloom
 *
 *      Copyright (c) 2003 Bruce Justin Lindbloom. All rights reserved.
 *
 *
 *      Description:
 *              This is an implementation of Robertson's method of computing the correlated color
 *              temperature of an XYZ color. It can compute correlated color temperatures in the
 *              range [1666.7K, infinity].
 *
 *      Reference:
 *              "Color Science: Concepts and Methods, Quantitative Data and Formulae", Second Edition,
 *              Gunter Wyszecki and W. S. Stiles, John Wiley & Sons, 1982, pp. 227, 228.
 */
//adaptation to RT by J.Desmis

int ColorTemp::XYZtoCorColorTemp(double x0, double y0, double z0, double &temp) const
{

    typedef struct UVT {
        double  u;
        double  v;
        double  t;
    } UVT;

    double rt[31] = {       /* reciprocal temperature (K) */
        DBL_MIN,  10.0e-6,  20.0e-6,  30.0e-6,  40.0e-6,  50.0e-6,
        60.0e-6,  70.0e-6,  80.0e-6,  90.0e-6, 100.0e-6, 125.0e-6,
        150.0e-6, 175.0e-6, 200.0e-6, 225.0e-6, 250.0e-6, 275.0e-6,
        300.0e-6, 325.0e-6, 350.0e-6, 375.0e-6, 400.0e-6, 425.0e-6,
        450.0e-6, 475.0e-6, 500.0e-6, 525.0e-6, 550.0e-6, 575.0e-6,
        600.0e-6
    };

    UVT uvt[31] = {
        {0.18006, 0.26352, -0.24341},
        {0.18066, 0.26589, -0.25479},
        {0.18133, 0.26846, -0.26876},
        {0.18208, 0.27119, -0.28539},
        {0.18293, 0.27407, -0.30470},
        {0.18388, 0.27709, -0.32675},
        {0.18494, 0.28021, -0.35156},
        {0.18611, 0.28342, -0.37915},
        {0.18740, 0.28668, -0.40955},
        {0.18880, 0.28997, -0.44278},
        {0.19032, 0.29326, -0.47888},
        {0.19462, 0.30141, -0.58204},
        {0.19962, 0.30921, -0.70471},
        {0.20525, 0.31647, -0.84901},
        {0.21142, 0.32312, -1.0182},
        {0.21807, 0.32909, -1.2168},
        {0.22511, 0.33439, -1.4512},
        {0.23247, 0.33904, -1.7298},
        {0.24010, 0.34308, -2.0637},
        {0.24792, 0.34655, -2.4681},
        {0.25591, 0.34951, -2.9641},
        {0.26400, 0.35200, -3.5814},
        {0.27218, 0.35407, -4.3633},
        {0.28039, 0.35577, -5.3762},
        {0.28863, 0.35714, -6.7262},
        {0.29685, 0.35823, -8.5955},
        {0.30505, 0.35907, -11.324},
        {0.31320, 0.35968, -15.628},
        {0.32129, 0.36011, -23.325},
        {0.32931, 0.36038, -40.770},
        {0.33724, 0.36051, -116.45}
    };

    double us, vs, p, di, dm;
    int i;

    if ((x0 < 1.0e-20) && (y0 < 1.0e-20) && (z0 < 1.0e-20)) {
        return -1;    /* protect against possible divide-by-zero failure */
    }

    us = (4.0 * x0) / (x0 + 15.0 * y0 + 3.0 * z0);
    vs = (6.0 * y0) / (x0 + 15.0 * y0 + 3.0 * z0);
    dm = 0.0;

    for (i = 0; i < 31; i++) {
        di = (vs - uvt[i].v) - uvt[i].t * (us - uvt[i].u);

        if ((i > 0) && (((di < 0.0) && (dm >= 0.0)) || ((di >= 0.0) && (dm < 0.0)))) {
            break;    /* found lines bounding (us, vs) : i-1 and i */
        }

        dm = di;
    }

    if (i == 31) {
        return -1;    /* bad XYZ input, color temp would be less than minimum of 1666.7 degrees, or too far towards blue */
    }

    di = di / sqrt(1.0 + uvt[i    ].t * uvt[i    ].t);
    dm = dm / sqrt(1.0 + uvt[i - 1].t * uvt[i - 1].t);
    p = dm / (dm - di);     /* p = interpolation parameter, 0.0 : i-1, 1.0 : i */
    p = 1.0 / rtengine::intp(p, rt[i], rt[i - 1]);
    temp = p;
    return 0;      /* success */
}

void ColorTemp::cieCAT02(double Xw, double Yw, double Zw, double &CAM02BB00, double &CAM02BB01, double &CAM02BB02, double &CAM02BB10, double &CAM02BB11, double &CAM02BB12, double &CAM02BB20, double &CAM02BB21, double &CAM02BB22, double adap)
{

// CIECAT02  - J.Desmis January 2012 review September 2012
    const double whiteD50p[3] = {0.9646019585, 1.0, 0.8244507152}; //calculate with these tools

    double cam_dest[3] = {0., 0., 0.};
    double cam_orig[3] = {0., 0., 0.};
    const double CAT02[3][3] = {{0.7328,   0.4296,  -0.1624},//CAT02 2002
        { -0.7036,  1.6975,   0.0061},
        {0.0030,   0.0136,   0.9834}
    };
    const double INVCAT02[3][3] = {{1.09612382083551,      -0.278869000218287,    0.182745179382773},  //Inverse CAT02
        {0.454369041975359,      0.4735331543070412,   0.0720978037172291},
        { -0.009627608738442936, -0.00569803121611342,  1.01532563995454}
    };

    double inv_white_orig[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    double intermed[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };

    double intermed_2[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    double CAM02[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    double D = adap;

    //white destination Wd : RT use always D50
    cam_dest[0] = CAT02[0][0] * whiteD50p[0] + CAT02[0][1] * whiteD50p[1] + CAT02[0][2] * whiteD50p[2]; //Cone response RoD
    cam_dest[1] = CAT02[1][0] * whiteD50p[0] + CAT02[1][1] * whiteD50p[1] + CAT02[1][2] * whiteD50p[2]; //GaD
    cam_dest[2] = CAT02[2][0] * whiteD50p[0] + CAT02[2][1] * whiteD50p[1] + CAT02[2][2] * whiteD50p[2]; //BeD

    //origin white Ws : A, D65, custom, etc.
    cam_orig[0] = CAT02[0][0] * Xw + CAT02[0][1] * Yw + CAT02[0][2] * Zw; //Cone response RoS
    cam_orig[1] = CAT02[1][0] * Xw + CAT02[1][1] * Yw + CAT02[1][2] * Zw;
    cam_orig[2] = CAT02[2][0] * Xw + CAT02[2][1] * Yw + CAT02[2][2] * Zw;

    //inverse white
    inv_white_orig[0][0] = 1. / cam_orig[0]; // 1/RoS
    inv_white_orig[1][1] = 1. / cam_orig[1]; // 1/GaS
    inv_white_orig[2][2] = 1. / cam_orig[2]; // 1/BeS

    //intermediates computation
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            intermed[i][j] = inv_white_orig[i][i] * CAT02[i][j];
        }

    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            intermed_2[i][j] = cam_dest[i] * intermed[i][j];
        }

    //and CAM02
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3; j++)
            for (int k = 0; k < 3; k++) {
                CAM02[i][j] += INVCAT02[i][k] * intermed_2[k][j];
            }

    //adaptation jdc : slightly different from CIECAM02 : Rc=(Yw(D/Rw)+(1-D))*R , but it's work !   true at 0 and 1
    CAM02[1][1] = (CAM02[1][1] - 1.0) * D + 1.0;
    CAM02[0][0] = (CAM02[0][0] - 1.0) * D + 1.0;
    CAM02[2][2] = (CAM02[2][2] - 1.0) * D + 1.0;
    CAM02[0][1] *= D;
    CAM02[0][2] *= D;
    CAM02[1][0] *= D;
    CAM02[1][2] *= D;
    CAM02[2][0] *= D;
    CAM02[2][1] *= D;
    //CAT02  matrix with D adaptation
    CAM02BB00 = CAM02[0][0];
    CAM02BB01 = CAM02[0][1];
    CAM02BB02 = CAM02[0][2];
    CAM02BB10 = CAM02[1][0];
    CAM02BB11 = CAM02[1][1];
    CAM02BB12 = CAM02[1][2];
    CAM02BB20 = CAM02[2][0];
    CAM02BB21 = CAM02[2][1];
    CAM02BB22 = CAM02[2][2];

}

void ColorTemp::cieCAT02float(float Xw, float Yw, float Zw, float &CAM02BB00, float &CAM02BB01, float &CAM02BB02, float &CAM02BB10, float &CAM02BB11, float &CAM02BB12, float &CAM02BB20, float &CAM02BB21, float &CAM02BB22, float adap)
{

// CIECAT02  - J.Desmis January 2012 review September 2012
    const float whiteD50p[3] = {0.9646019585, 1.0, 0.8244507152}; //calculate with these tools

    float cam_dest[3] = {0., 0., 0.};
    float cam_orig[3] = {0., 0., 0.};
    const float CAT02[3][3] = {{0.7328,   0.4296,  -0.1624},//CAT02 2002
        { -0.7036,  1.6975,   0.0061},
        {0.0030,   0.0136,   0.9834}
    };
    const float INVCAT02[3][3] = {{1.09612382083551,      -0.278869000218287,    0.182745179382773},  //Inverse CAT02
        {0.454369041975359,      0.4735331543070412,   0.0720978037172291},
        { -0.009627608738442936, -0.00569803121611342,  1.01532563995454}
    };

    float inv_white_orig[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float intermed[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };

    float intermed_2[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float CAM02[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float D = adap;

    //white destination Wd : RT use always D50
    cam_dest[0] = CAT02[0][0] * whiteD50p[0] + CAT02[0][1] * whiteD50p[1] + CAT02[0][2] * whiteD50p[2]; //Cone response RoD
    cam_dest[1] = CAT02[1][0] * whiteD50p[0] + CAT02[1][1] * whiteD50p[1] + CAT02[1][2] * whiteD50p[2]; //GaD
    cam_dest[2] = CAT02[2][0] * whiteD50p[0] + CAT02[2][1] * whiteD50p[1] + CAT02[2][2] * whiteD50p[2]; //BeD

    //origin white Ws : A, D65, custom, etc.
    cam_orig[0] = CAT02[0][0] * Xw + CAT02[0][1] * Yw + CAT02[0][2] * Zw; //Cone response RoS
    cam_orig[1] = CAT02[1][0] * Xw + CAT02[1][1] * Yw + CAT02[1][2] * Zw;
    cam_orig[2] = CAT02[2][0] * Xw + CAT02[2][1] * Yw + CAT02[2][2] * Zw;

    //inverse white
    inv_white_orig[0][0] = 1. / cam_orig[0]; // 1/RoS
    inv_white_orig[1][1] = 1. / cam_orig[1]; // 1/GaS
    inv_white_orig[2][2] = 1. / cam_orig[2]; // 1/BeS

    //intermediates computation
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            intermed[i][j] = inv_white_orig[i][i] * CAT02[i][j];
        }

    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            intermed_2[i][j] = cam_dest[i] * intermed[i][j];
        }

    //and CAM02
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3; j++)
            for (int k = 0; k < 3; k++) {
                CAM02[i][j] += INVCAT02[i][k] * intermed_2[k][j];
            }

    //adaptation jdc : slightly different from CIECAM02 : Rc=(Yw(D/Rw)+(1-D))*R , but it's work !   true at 0 and 1
    CAM02[1][1] = (CAM02[1][1] - 1.0) * D + 1.0;
    CAM02[0][0] = (CAM02[0][0] - 1.0) * D + 1.0;
    CAM02[2][2] = (CAM02[2][2] - 1.0) * D + 1.0;
//   CAM02[1][1] *= D;
//   CAM02[0][0] *= D;
//   CAM02[2][2] *= D;
    CAM02[0][1] *= D;
    CAM02[0][2] *= D;
    CAM02[1][0] *= D;
    CAM02[1][2] *= D;
    CAM02[2][0] *= D;
    CAM02[2][1] *= D;
    //CAT02  matrix with D adaptation
    CAM02BB00 = CAM02[0][0];
    CAM02BB01 = CAM02[0][1];
    CAM02BB02 = CAM02[0][2];
    CAM02BB10 = CAM02[1][0];
    CAM02BB11 = CAM02[1][1];
    CAM02BB12 = CAM02[1][2];
    CAM02BB20 = CAM02[2][0];
    CAM02BB21 = CAM02[2][1];
    CAM02BB22 = CAM02[2][2];

}

void ColorTemp::icieCAT02float(float Xw, float Yw, float Zw, float &iCAM02BB00, float &iCAM02BB01, float &iCAM02BB02, float &iCAM02BB10, float &iCAM02BB11, float &iCAM02BB12, float &iCAM02BB20, float &iCAM02BB21, float &iCAM02BB22, float adap)
{

// CIECAT02  - J.Desmis January 2012 review September 2017
    const float whiteD50p[3] = {0.9646019585, 1.0, 0.8244507152}; //calculate with these tools

    float cam_dest[3] = {0., 0., 0.};
    float cam_orig[3] = {0., 0., 0.};
    const float CAT02[3][3] = {{0.7328,   0.4296,  -0.1624},//CAT02 2002
        { -0.7036,  1.6975,   0.0061},
        {0.0030,   0.0136,   0.9834}
    };
    const float INVCAT02[3][3] = {{1.09612382083551,      -0.278869000218287,    0.182745179382773},  //Inverse CAT02
        {0.454369041975359,      0.4735331543070412,   0.0720978037172291},
        { -0.009627608738442936, -0.00569803121611342,  1.01532563995454}
    };

    float inv_white_orig[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float intermed[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };

    float intermed_2[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float INVCAM02[3][3] = {{0.,  0.,  0.},
        {0.,  0.,  0.},
        {0.,  0.,  0.}
    };
    float D = adap / 2.;

    //white destination Wd : RT use always D50
    cam_dest[0] = INVCAT02[0][0] * whiteD50p[0] + INVCAT02[0][1] * whiteD50p[1] + INVCAT02[0][2] * whiteD50p[2]; //Cone response RoD
    cam_dest[1] = INVCAT02[1][0] * whiteD50p[0] + INVCAT02[1][1] * whiteD50p[1] + INVCAT02[1][2] * whiteD50p[2]; //GaD
    cam_dest[2] = INVCAT02[2][0] * whiteD50p[0] + INVCAT02[2][1] * whiteD50p[1] + INVCAT02[2][2] * whiteD50p[2]; //BeD

    //origin white Ws : A, D65, custom, etc.
    cam_orig[0] = INVCAT02[0][0] * Xw + INVCAT02[0][1] * Yw + INVCAT02[0][2] * Zw; //Cone response RoS
    cam_orig[1] = INVCAT02[1][0] * Xw + INVCAT02[1][1] * Yw + INVCAT02[1][2] * Zw;
    cam_orig[2] = INVCAT02[2][0] * Xw + INVCAT02[2][1] * Yw + INVCAT02[2][2] * Zw;
//   cam_orig[0] = CAT02[0][0] * Xw + CAT02[0][1] * Yw + CAT02[0][2] * Zw; //Cone response RoS
//   cam_orig[1] = CAT02[1][0] * Xw + CAT02[1][1] * Yw + CAT02[1][2] * Zw;
    //  cam_orig[2] = CAT02[2][0] * Xw + CAT02[2][1] * Yw + CAT02[2][2] * Zw;

    //inverse white
    inv_white_orig[0][0] = 1. / cam_orig[0]; // 1/RoS
    inv_white_orig[1][1] = 1. / cam_orig[1]; // 1/GaS
    inv_white_orig[2][2] = 1. / cam_orig[2]; // 1/BeS

    //intermediates computation
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            // intermed[i][j] = inv_white_orig[i][i] * INVCAT02[i][j];
            intermed[i][j] = inv_white_orig[i][i] * CAT02[i][j];
        }

    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3 ; j++) {
            intermed_2[i][j] = cam_dest[i] * intermed[i][j];
        }

    //and CAM02
    for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3; j++)
            for (int k = 0; k < 3; k++) {
                INVCAM02[i][j] += INVCAT02[i][k] * intermed_2[k][j];
            }

    //adaptation jdc : slightly different from CIECAM02 : Rc=(Yw(D/Rw)+(1-D))*R , but it's work !   true at 0 and 1
    INVCAM02[0][0] = (INVCAM02[0][0] - 1.0) * D + 1.0;
    INVCAM02[2][2] = (INVCAM02[2][2] - 1.0) * D + 1.0;
    INVCAM02[0][1] *= D;
    INVCAM02[0][2] *= D;
    INVCAM02[1][0] *= D;
    INVCAM02[1][2] *= D;
    INVCAM02[2][0] *= D;
    INVCAM02[2][1] *= D;
    //CAT02  matrix with D adaptation
    iCAM02BB00 = INVCAM02[0][0];
    iCAM02BB01 = INVCAM02[0][1];
    iCAM02BB02 = INVCAM02[0][2];
    iCAM02BB10 = INVCAM02[1][0];
    iCAM02BB11 = INVCAM02[1][1];
    iCAM02BB12 = INVCAM02[1][2];
    iCAM02BB20 = INVCAM02[2][0];
    iCAM02BB21 = INVCAM02[2][1];
    iCAM02BB22 = INVCAM02[2][2];

}


void ColorTemp::temp2mulxyz(double temp, const std::string &method, StandardObserver observer, double &Xxyz, double &Zxyz)
{
    double x, y, z;

    // We first test for specially handled methods
    const auto iterator = spectMap.find(method);
    const auto &color_match = (observer == StandardObserver::TEN_DEGREES) ? cie_colour_match_jd : cie_colour_match_jd2;
/*
        if(observer == StandardObserver::TEN_DEGREES){
            printf("General Observer 10°\n");
        } else {
            printf("General Observer 2°\n");
        }
*/    
    if (iterator != spectMap.end()) {
        spectrum_to_xyz_preset(iterator->second, x, y, z, color_match);
    } else {
        // otherwise we use the Temp+Green generic solution
        if (temp <= INITIALBLACKBODY) {
            // if temperature is between 2000K and 4000K we use blackbody, because there will be no Daylight reference below 4000K...
            // of course, the previous version of RT used the "magical" but wrong formula of U.Fuchs (Ufraw).
            spectrum_to_xyz_blackbody(temp, x, y, z, color_match);
        } else {
            // from 4000K up to 25000K: using the D illuminant (daylight) which is standard
            double x_D, y_D;

            if (temp <= 7000) {
                x_D = -4.6070e9 / (temp * temp * temp) + 2.9678e6 / (temp * temp) + 0.09911e3 / temp + 0.244063;
            } else if (temp <= 25000) {
                x_D = -2.0064e9 / (temp * temp * temp) + 1.9018e6 / (temp * temp) + 0.24748e3 / temp + 0.237040;
            } else { /*if (temp > 25000)*/
                x_D = -2.0064e9 / (temp * temp * temp) + 1.9018e6 / (temp * temp) + 0.24748e3 / temp + 0.237040 - ((temp - 25000) / 25000) * 0.025;    //Jacques empirical adjustment for very high temp (underwater !)
            }

            y_D = -3.0 * x_D * x_D + 2.87 * x_D - 0.275; //modify blue / red action
            //calculate D -daylight in function of s0, s1, s2 and temp ==> x_D y_D
            //S(lamda)=So(lambda)+m1*s1(lambda)+m2*s2(lambda)
            double interm = 0.0241 + 0.2562 * x_D - 0.734 * y_D;
            double m1 = (-1.3515 - 1.7703 * x_D + 5.9114 * y_D) / interm;
            double m2 = (0.03 - 31.4424 * x_D + 30.0717 * y_D) / interm;
            spectrum_to_xyz_daylight(m1, m2, x, y, z, color_match);
        }
    }

    Xxyz = x / y;
    Zxyz = (1.0 - x - y) / y;
}

void ColorTemp::temp2mul(double temp, double green, double equal, StandardObserver observer, double& rmul, double& gmul, double& bmul) const
{
    clip(temp, green, equal);
    double Xwb, Zwb;
    temp2mulxyz(temp, method, observer, Xwb, Zwb);

    double adj = 1.0;

    if (equal < 0.9999 || equal > 1.0001) {
        adj = (100.0 + (1000.0 - (1000.0 * equal)) / 20.0) / 100.0;
    }


    /*if (isRaw) {
        rmul = sRGB_xyz[0][0]*X + sRGB_xyz[0][1]*Y + sRGB_xyz[0][2]*Z;
        gmul = sRGB_xyz[1][0]*X + sRGB_xyz[1][1]*Y + sRGB_xyz[1][2]*Z;
        bmul = sRGB_xyz[2][0]*X + sRGB_xyz[2][1]*Y + sRGB_xyz[2][2]*Z;
    } else {*/
    //recalculate channels multipliers with new values of XYZ tue to whitebalance
    rmul = sRGBd65_xyz[0][0] * Xwb * adj + sRGBd65_xyz[0][1] + sRGBd65_xyz[0][2] * Zwb / adj; // Jacques' empirical modification 5/2013
    gmul = sRGBd65_xyz[1][0] * Xwb       + sRGBd65_xyz[1][1] + sRGBd65_xyz[1][2] * Zwb;
    bmul = sRGBd65_xyz[2][0] * Xwb * adj + sRGBd65_xyz[2][1] + sRGBd65_xyz[2][2] * Zwb / adj;
    //};
    gmul /= green;
    //printf("rmul=%f gmul=%f bmul=%f\n",rmul, gmul, bmul);
    double maxRGB = rtengine::max(rmul, gmul, bmul);

    rmul /= maxRGB;
    gmul /= maxRGB;
    bmul /= maxRGB;


    if (settings->CRI_color != 0) { //activate if CRi_color !=0
        // begin CRI_RT : color rendering index RT - adaptation of CRI by J.Desmis
        // CRI = 100 for Blackbody and Daylight
        // calculate from spectral data  values X, Y, Z , for  color of colorchecker24 , SG, DC, JDC_468
        // only for lamp different of tungstene
        // first calcul with illuminant (choice)
        // and calcul with : blackbody at equivalent temp of lamp
        // CRI_color-1 = display Lab values of color CRI_color -1
        const double whiteD50[3] = {0.9646019585, 1.0, 0.8244507152}; //calculate with this tool : spect 5nm
        double Xchk[50], Ychk[50], Zchk[50]; //50 : I think it's a good limit for number of color : for CRI and Palette
        double Xcam02[50], Ycam02[50], Zcam02[50];

        double XchkLamp[50], YchkLamp[50], ZchkLamp[50];
        double Xcam02Lamp[50], Ycam02Lamp[50], Zcam02Lamp[50];
        const double epsilon = 0.008856; //Lab

        double xr[50], yr[50], zr[50];

        int illum;
        int numero_color = settings->CRI_color - 1;

        //spectral data illuminant (actually 21): only those necessary (lamp, fluorescent, LED) others CRI=100 (not Flash...)
        const double* spec_color[] = {
            ColorchechredC3_spect, ColorchechOraA2_spect, ColorchechYelD3_spect, ColorchechGreE2_spect, ColorchechGreB3_spect,
            ColorchechCyaF3_spect, ColorchechPurD2_spect, ColorchechMagE3_spect, ColorchechSkiA138_13_14_spect, ColorchechGraC4_67_spect,
            ColorchechSkiB166_18_18_spect, ColorchechBluC150_m5_m22_spect, ColorchechDCBluN881_m7_m14_spect, ColorchechSGSkiF763_14_26_spect,
            ColorchechSGSkiK285_11_17_spect, ColorchechWhiA496_spect, ColorchechGreD1_spect, ColorchechSGBlaN3_6_spect,
            JDC468_GraK14_44_spect, JDC468_BluH10_spect
        };

        int N_c = sizeof(spec_color) / sizeof(spec_color[0]); //number of color

        bool CRI_type = false;
        double tempw;

        if (method == "Fluo F1")              {
            CRI_type = true;
            tempw = 6430;
            illum = 1;
        } else if (method == "Fluo F2")              {
            CRI_type = true;
            tempw = 4230;
            illum = 2;
        } else if (method == "Fluo F3")              {
            CRI_type = true;
            tempw = 3450;
            illum = 3;
        } else if (method == "Fluo F4")              {
            CRI_type = true;
            tempw = 2940;
            illum = 4;
        } else if (method == "Fluo F5")              {
            CRI_type = true;
            tempw = 6350;
            illum = 5;
        } else if (method == "Fluo F6")              {
            CRI_type = true;
            tempw = 4150;
            illum = 6;
        } else if (method == "Fluo F7")              {
            CRI_type = true;
            tempw = 6500;
            illum = 7;
        } else if (method == "Fluo F8")              {
            CRI_type = true;
            tempw = 5020;
            illum = 8;
        } else if (method == "Fluo F9")              {
            CRI_type = true;
            tempw = 4330;
            illum = 9;
        } else if (method == "Fluo F10")             {
            CRI_type = true;
            tempw = 5300;
            illum = 10;
        } else if (method == "Fluo F11")             {
            CRI_type = true;
            tempw = 4000;
            illum = 11;
        } else if (method == "Fluo F12")             {
            CRI_type = true;
            tempw = 3000;
            illum = 12;
        } else if (method == "HMI Lamp")             {
            CRI_type = true;
            tempw = 4760;
            illum = 13;
        } else if (method == "GTI Lamp")             {
            CRI_type = true;
            tempw = 5000;
            illum = 14;
        } else if (method == "JudgeIII Lamp")        {
            CRI_type = true;
            tempw = 5100;
            illum = 15;
        } else if (method == "Solux Lamp 3500K")     {
            CRI_type = true;
            tempw = 3480;
            illum = 16;
        } else if (method == "Solux Lamp 4100K")     {
            CRI_type = true;
            tempw = 3930;
            illum = 17;
        } else if (method == "Solux Lamp 4700K")  {
            CRI_type = true;
            tempw = 4700;
            illum = 18;
        } else if (method == "NG Solux Lamp 4700K")  {
            CRI_type = true;
            tempw = 4480;
            illum = 19;
        } else if (method == "LED LSI Lumelex 2040") {
            CRI_type = true;
            tempw = 2970;
            illum = 20;
        } else if (method == "LED CRS SP12 WWMR16")  {
            CRI_type = true;
            tempw = 3050;
            illum = 21;
        } else                                       {
            CRI_type = false;
        }

        if (CRI_type) {
            double x, y, z;
            double Ywb = 1.0;

            const double* spect_illum[] = {
                Daylight5300_spect, Cloudy6200_spect, Shade7600_spect, A2856_spect, FluoF1_spect, FluoF2_spect, FluoF3_spect,
                FluoF4_spect, FluoF5_spect, FluoF6_spect, FluoF7_spect, FluoF8_spect, FluoF9_spect, FluoF10_spect, FluoF11_spect,
                FluoF12_spect, HMI_spect, GTI_spect, JudgeIII_spect, Solux3500_spect, Solux4100_spect, Solux4700_spect,
                NG_Solux4700_spect, NG_CRSSP12WWMR16_spect, NG_CRSSP12WWMR16_spect
            };

            float DeltaE[50], DeltaEs[8];
            float quadCRI = 0.0f, quadCRIs = 0.0f;
            float CRI_RT = 0.0, CRI[50];
            float CRI_RTs = 0.0, CRIs[8];

            const auto &color_match = (observer == StandardObserver::TEN_DEGREES) ? cie_colour_match_jd : cie_colour_match_jd2;

            //exceptional must be used by advice people
            if (observer == StandardObserver::TEN_DEGREES) {
                printf("CRI Observer 10°\n");
            } else {
                printf("CRI Observer 2°\n");
            }

            for (int i = 0; i < N_c; i++) {
                spectrum_to_color_xyz_preset(spec_color[i], spect_illum[illum + 3], XchkLamp[i], YchkLamp[i], ZchkLamp[i], color_match);
            }

            //calculate XYZ for each color : for Blackbody and Daylight at tempw
            if (tempw <= INITIALBLACKBODY) {
                for (int i = 0; i < N_c; i++) {
                    spectrum_to_color_xyz_blackbody(spec_color[i], tempw, Xchk[i], Ychk[i], Zchk[i], color_match);
                }

                spectrum_to_xyz_blackbody(tempw, x, y, z, color_match);//for white point
            } else { // after 6600K (arbitrary) I use daylight...because ...but there is no lamp...
                double m11, m22, x_DD, y_DD, interm2;

                if (tempw <= 7000) {
                    x_DD = -4.6070e9 / (tempw * tempw * tempw) + 2.9678e6 / (tempw * tempw) + 0.09911e3 / tempw + 0.244063;
                } else {
                    x_DD = -2.0064e9 / (tempw * tempw * tempw) + 1.9018e6 / (tempw * tempw) + 0.24748e3 / tempw + 0.237040;
                }

                y_DD = -3.0 * x_DD * x_DD + 2.87 * x_DD - 0.275;
                //calculate D -daylight in function of s0, s1, s2 and temp ==> x_D y_D
                //S(lamda)=So(lambda)+m1*s1(lambda)+m2*s2(lambda)
                interm2 = (0.0241 + 0.2562 * x_DD - 0.734 * y_DD);
                m11 = (-1.3515 - 1.7703 * x_DD + 5.9114 * y_DD) / interm2;
                m22 = (0.03 - 31.4424 * x_DD + 30.0717 * y_DD) / interm2;

                for (int i = 0; i < N_c; i++) {
                    spectrum_to_color_xyz_daylight(spec_color[i], m11, m22, Xchk[i], Ychk[i], Zchk[i], color_match);
                }

                spectrum_to_xyz_daylight(m11, m22, x, y, z, color_match);
            }

            if (settings->verbose) {
                double correl_temp = 0.0;
                XYZtoCorColorTemp(Xwb, Ywb, Zwb, correl_temp);
                printf("Correlated temperature (lamp)=%i\n", (int) correl_temp); //use only for lamp...otherwise It give an information!!
            }

            double Xwb_bb = x / y; //white balance for blackbody
            double Ywb_bb = 1.0;
            double Zwb_bb = (1.0 - x - y) / y;

            //calculate Matrix CAM02 : better than Von Kries and Bradford==> for Lamp
            double  adap = 1.0;
            double CAM02BB00, CAM02BB01, CAM02BB02, CAM02BB10, CAM02BB11, CAM02BB12, CAM02BB20, CAM02BB21, CAM02BB22; //for CIECAT02
            cieCAT02(Xwb, Ywb, Zwb, CAM02BB00, CAM02BB01, CAM02BB02, CAM02BB10, CAM02BB11, CAM02BB12, CAM02BB20, CAM02BB21, CAM02BB22, adap);

            //here new value of X,Y,Z for lamp with chromatic CAM02 adaptation
            for (int i = 0; i < N_c; i++) {
                Xcam02Lamp[i] = CAM02BB00 * XchkLamp[i] + CAM02BB01 * YchkLamp[i] + CAM02BB02 * ZchkLamp[i] ;
                Ycam02Lamp[i] = CAM02BB10 * XchkLamp[i] + CAM02BB11 * YchkLamp[i] + CAM02BB12 * ZchkLamp[i] ;
                Zcam02Lamp[i] = CAM02BB20 * XchkLamp[i] + CAM02BB21 * YchkLamp[i] + CAM02BB22 * ZchkLamp[i] ;
            }

            //now calculate CAM02 for Blackbody (or Daylight) at tempx

            cieCAT02(Xwb_bb, Ywb_bb, Zwb_bb, CAM02BB00, CAM02BB01, CAM02BB02, CAM02BB10, CAM02BB11, CAM02BB12, CAM02BB20, CAM02BB21, CAM02BB22, adap);

            //here new value of X,Y,Z for blackbody with chromatic CAM02 adaptation

            for (int i = 0; i < N_c; i++) {
                Xcam02[i] = CAM02BB00 * Xchk[i] + CAM02BB01 * Ychk[i] + CAM02BB02 * Zchk[i] ;
                Ycam02[i] = CAM02BB10 * Xchk[i] + CAM02BB11 * Ychk[i] + CAM02BB12 * Zchk[i] ;
                Zcam02[i] = CAM02BB20 * Xchk[i] + CAM02BB21 * Ychk[i] + CAM02BB22 * Zchk[i] ;
                //  printf("CoulXYZ %i X %f Y %f Z %f\n", i, Xchk[i],Ychk[i],Zchk[i]);
                //  printf("CoulCAM %i X %f Y %f Z %f\n", i, Xcam02[i],Ycam02[i],Zcam02[i]);
            }

            //now conversion to Lab
            // Lamp
            double fx[50], fy[50], fz[50];

            for (int i = 0; i < N_c; i++) {
                xr[i] = Xcam02Lamp[i] / whiteD50[0];
                yr[i] = Ycam02Lamp[i] / whiteD50[1];
                zr[i] = Zcam02Lamp[i] / whiteD50[2];

                // xr, yr , zr > epsilon
                if (xr[i] > epsilon) {
                    fx[i] = std::cbrt(xr[i]);
                } else {
                    fx[i] = (903.3 * xr[i] + 16.0) / 116.0;
                }

                if (yr[i] > epsilon) {
                    fy[i] = std::cbrt(yr[i]);
                } else {
                    fy[i] = (903.3 * yr[i] + 16.0) / 116.0;
                }

                if (zr[i] > epsilon) {
                    fz[i] = std::cbrt(zr[i]);
                } else {
                    fz[i] = (903.3 * zr[i] + 16.0) / 116.0;
                }
            }

            double Llamp[50], alamp[50], blamp[50];

            for (int i = 0; i < N_c; i++) {
                Llamp[i] = 116.0 * fy[i] - 16.0;
                alamp[i] = 500.0 * (fx[i] - fy[i]);
                blamp[i] = 200.0 * (fy[i] - fz[i]);
            }

            //blackbody at tempx
            for (int i = 0; i < N_c; i++) {
                xr[i] = Xcam02[i] / whiteD50[0];
                yr[i] = Ycam02[i] / whiteD50[1];
                zr[i] = Zcam02[i] / whiteD50[2];

                //
                if (xr[i] > epsilon) {
                    fx[i] = std::cbrt(xr[i]);
                } else {
                    fx[i] = (903.3 * xr[i] + 16.0) / 116.0;
                }

                if (yr[i] > epsilon) {
                    fy[i] = std::cbrt(yr[i]);
                } else {
                    fy[i] = (903.3 * yr[i] + 16.0) / 116.0;
                }

                if (zr[i] > epsilon) {
                    fz[i] = std::cbrt(zr[i]);
                } else {
                    fz[i] = (903.3 * zr[i] + 16.0) / 116.0;
                }
            }

            double Lbb[50], abb[50], bbb[50];

            for (int i = 0; i < N_c; i++) {
                Lbb[i] = 116.*fy[i] - 16.;
                abb[i] = 500.*(fx[i] - fy[i]);
                bbb[i] = 200.*(fy[i] - fz[i]);
            }

            //display value to verify calculus
            if (settings->CRI_color != 0) {
                printf("Color Number %i\n", numero_color);
                printf("L_refer=%2.2f a=%2.2f b=%2.2f\n", Lbb[numero_color], abb[numero_color], bbb[numero_color]);
                printf("L_lamp=%2.2f al=%2.2f bl=%2.2f\n", Llamp[numero_color], alamp[numero_color], blamp[numero_color]);
            }

            //then calculate DeltaE CIE 1976
            for (int i = 0; i < 8; i++) {
                DeltaEs[i] = sqrt((Lbb[i] - Llamp[i]) * (Lbb[i] - Llamp[i]) + (abb[i] - alamp[i]) * (abb[i] - alamp[i]) + (bbb[i] - blamp[i]) * (bbb[i] - blamp[i]));
            }

            for (int i = 0; i < 8; i++) {
                CRIs[i] = 100 - 3.f * DeltaEs[i];    //3.0 coef to adapt ==> same results than CRI "official"
            }

            for (int i = 0; i < 8; i++) {
                CRI_RTs += CRIs[i];
            }

            CRI_RTs /= 8;

            for (int i = 0; i < 8; i++) {
                quadCRIs += (CRIs[i] - CRI_RTs) * (CRIs[i] - CRI_RTs);
            }

            quadCRIs /= 8;

            for (int i = 0; i < N_c; i++) {
                DeltaE[i] = sqrt((Lbb[i] - Llamp[i]) * (Lbb[i] - Llamp[i]) + (abb[i] - alamp[i]) * (abb[i] - alamp[i]) + (bbb[i] - blamp[i]) * (bbb[i] - blamp[i]));
            }

            for (int i = 0; i < N_c; i++) {
                CRI[i] = 100 - 3.f * DeltaE[i];    //3.0 coef to adapt ==> same results than CRI "official"
            }

            for (int i = 0; i < N_c; i++) {
                CRI_RT += CRI[i];
            }

            CRI_RT /= N_c;

            for (int i = 0; i < N_c; i++) {
                quadCRI += (CRI[i] - CRI_RT) * (CRI[i] - CRI_RT);
            }

            quadCRI /= N_c;

            if (settings->CRI_color != 0) {
                printf("CRI_standard=%i CRI:1->8=%i %i %i %i %i %i %i %i  Sigma=%2.1f\n", (int) CRI_RTs, (int) CRIs[0], (int) CRIs[1], (int) CRIs[2], (int) CRIs[3], (int) CRIs[4], (int) CRIs[5], (int) CRIs[6], (int) CRIs[7], sqrt(static_cast<double>(quadCRIs)));
                printf("CRI_RT_exten=%i CRI:9->20=%i %i %i %i %i %i %i %i %i %i %i %i Sigma=%2.1f\n", (int) CRI_RT, (int) CRI[8], (int) CRI[9], (int) CRI[10], (int) CRI[11], (int) CRI[12], (int) CRI[13], (int) CRI[14], (int) CRI[15], (int) CRI[16], (int) CRI[17], (int) CRI[18], (int) CRI[19], static_cast<double>(sqrt(quadCRI)));
            }
        }
    }
}

/*
    Calculate Planck's radiation
*/
//calculate spectral data for blackbody at temp!
double ColorTemp::blackbody_spect(double wavelength, double temperature)
{
    const double wlm = wavelength * 1e-9;   /* Wavelength in meters */
    return (3.7417715247e-16 / rtengine::pow5(wlm)) /              //3.7417..= c1 = 2*Pi*h*c2  where h=Planck constant, c=velocity of light
           (xexp(1.438786e-2 / (wlm * temperature)) - 1.0); //1.4387..= c2 = h*c/k  where k=Boltzmann constant

}

/*
The next 3 methods are inspired from:

   a) Colour Rendering of Spectra  by John Walker
      http://www.fourmilab.ch/
      This program is in the public domain.

   b) Bruce Lindbloom
      Adapted to RawTherapee by J.Desmis

this values are often called xBar yBar zBar and are characteristics of a color / illuminant

values cie_colour_match2[][3] = 2° Standard Observer x2, y2, z2
E.g. for 380nm: x2=0.001368  y2=0.000039  z2=0.006451  round in J.Walker to 0.0014  0.0000 0.0065 above
I have increase precision used by J.Walker  and pass to 350nm to 830nm
And also add 10°  standard observer
*/

void ColorTemp::spectrum_to_xyz_daylight(double _m1, double _m2, double &x, double &y, double &z, const color_match_type &color_match)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, XYZ;

    for (i = 0, lambda = 350.; lambda < 830.1; i++, lambda += 5.) {
        double Me = daylight_spect(lambda, _m1, _m2);
        //  printf("Day Me=%f \n", Me);
        X += Me * color_match[i][0];
        Y += Me * color_match[i][1];
        Z += Me * color_match[i][2];
    }

    XYZ = (X + Y + Z);
    x = X / XYZ;
    y = Y / XYZ;
    z = Z / XYZ;
}

void ColorTemp::spectrum_to_xyz_blackbody(double _temp, double &x, double &y, double &z, const color_match_type &color_match)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, XYZ;

    for (i = 0, lambda = 350.; lambda < 830.1; i++, lambda += 5.) {
        double Me = blackbody_spect(lambda, _temp);
        X += Me * color_match[i][0];
        Y += Me * color_match[i][1];
        Z += Me * color_match[i][2];
    }

    XYZ = (X + Y + Z);
    x = X / XYZ;
    y = Y / XYZ;
    z = Z / XYZ;
}

void ColorTemp::spectrum_to_xyz_preset(const double* spec_intens, double &x, double &y, double &z, const color_match_type &color_match)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, XYZ;

    /*
    Inspired from:

       a) Colour Rendering of Spectra  by John Walker
          http://www.fourmilab.ch/
          This program is in the public domain.

       b) Bruce Lindbloom
          Adapted to RawTherapee by J.Desmis

    this values are often called xBar yBar zBar and are characteristics of a color / illuminant

    values cie_colour_match_jd2[][3] = 2° Standard Observer x2, y2, z2
    E.g. for 380nm: x2=0.001368  y2=0.000039  z2=0.006451  round in J.Walker to 0.0014  0.0000 0.0065 above
    I have increased the precision used by J.Walker and pass from 350nm to 830nm
    And also add standard observer 10°
    */
    for (i = 0, lambda = 350.; lambda < 830.1; i++, lambda += 5.) {
        double Me = get_spectral_color(lambda, spec_intens);
        X += Me * color_match[i][0];
        Y += Me * color_match[i][1];
        Z += Me * color_match[i][2];
    }

    XYZ = (X + Y + Z);
    x = X / XYZ;
    y = Y / XYZ;
    z = Z / XYZ;
}

//calculate XYZ from spectrum data (color) and illuminant : J.Desmis December 2011
void ColorTemp::spectrum_to_color_xyz_preset(const double* spec_color, const double* spec_intens, double &xx, double &yy, double &zz, const color_match_type &color_match)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, Yo = 0;

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {

        double Me;
        double Mc;

        Me = get_spectral_color(lambda, spec_color);
        Mc = get_spectral_color(lambda, spec_intens);
        X += Mc * color_match[i][0] * Me;
        Y += Mc * color_match[i][1] * Me;
        Z += Mc * color_match[i][2] * Me;
    }

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {

        double Ms;

        Ms = get_spectral_color(lambda, spec_intens);
        Yo += color_match[i][1] * Ms;
    }

    xx = X / Yo;
    yy = Y / Yo;
    zz = Z / Yo;
}

//calculate XYZ from spectrum data (color) and illuminant : J.Desmis december 2011
void ColorTemp::spectrum_to_color_xyz_daylight(const double* spec_color, double _m1, double _m2, double &xx, double &yy, double &zz, const color_match_type &color_match)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, Yo = 0;

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {
        const double Me = spec_color[i];
        const double Mc = daylight_spect(lambda, _m1, _m2);
        X += Mc * color_match[i][0] * Me;
        Y += Mc * color_match[i][1] * Me;
        Z += Mc * color_match[i][2] * Me;
    }

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {

        const double Mc1 = daylight_spect(lambda, _m1, _m2);
        Yo += color_match[i][1] * Mc1;
    }

    xx = X / Yo;
    yy = Y / Yo;
    zz = Z / Yo;
}

void ColorTemp::whitepoint(double tempw, double &xx, double &yy, double &zz, const color_match_type &color_match)
{
    if (tempw <= INITIALBLACKBODY) {
        spectrum_to_whitepoint_xyz_blackbody(tempw, xx, yy, zz, color_match);
    } else {
        double m11, m22, x_DD, y_DD, interm2;

        if (tempw <= 7000) {
            x_DD = -4.6070e9 / (tempw * tempw * tempw) + 2.9678e6 / (tempw * tempw) + 0.09911e3 / tempw + 0.244063;
        } else {
            x_DD = -2.0064e9 / (tempw * tempw * tempw) + 1.9018e6 / (tempw * tempw) + 0.24748e3 / tempw + 0.237040;
        }

        y_DD = -3.0 * x_DD * x_DD + 2.87 * x_DD - 0.275;
        //calculate D -daylight in function of s0, s1, s2 and temp ==> x_D y_D
        //S(lamda)=So(lambda)+m1*s1(lambda)+m2*s2(lambda)
        interm2 = (0.0241 + 0.2562 * x_DD - 0.734 * y_DD);
        m11 = (-1.3515 - 1.7703 * x_DD + 5.9114 * y_DD) / interm2;
        m22 = (0.03 - 31.4424 * x_DD + 30.0717 * y_DD) / interm2;

        spectrum_to_whitepoint_xyz_daylight(m11, m22, xx, yy, zz, color_match);
    }
}

void ColorTemp::spectrum_to_whitepoint_xyz_daylight(double _m1, double _m2, double &xx, double &yy, double &zz, const color_match_type &color_match)
{
    int i;
    double lambda, X = 0, Z = 0, Yo = 0;

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {
        const double Mc = daylight_spect(lambda, _m1, _m2);
        X += Mc * color_match[i][0];
        Z += Mc * color_match[i][2];
    }

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {
        const double Mc1 = daylight_spect(lambda, _m1, _m2);
        Yo += color_match[i][1] * Mc1;
    }

    xx = X / Yo;
    yy = 1.;
    zz = Z / Yo;
}

//calculate XYZ from spectrum data (color) and illuminant : J.Desmis december 2011 bug found 4/2023
void ColorTemp::spectrum_to_color_xyz_blackbody(const double* spec_color, double _temp, double &xx, double &yy, double &zz, const color_match_type &color_match)
{
    int i;
    double lambda, X = 0, Y = 0, Z = 0, Yo = 0;

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {
        const double Me = spec_color[i];
        const double Mc = blackbody_spect(lambda, _temp);
        X += Mc * color_match[i][0] * Me;
        Y += Mc * color_match[i][1] * Me;
        Z += Mc * color_match[i][2] * Me;
    }

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {
        const double Mc1 = blackbody_spect(lambda, _temp);
        Yo += color_match[i][1] * Mc1;
    }

    xx = X / Yo;
    yy = Y / Yo;
    zz = Z / Yo;
}

void ColorTemp::spectrum_to_whitepoint_xyz_blackbody(double _temp, double &xx, double &yy, double &zz, const color_match_type &color_match)
{
    int i;
    double lambda, X = 0, Z = 0, Yo = 0;

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {
        const double Mc = blackbody_spect(lambda, _temp);
        X += Mc * color_match[i][0];
        Z += Mc * color_match[i][2];
    }

    for (i = 0, lambda = 350; lambda < 830.1; i++, lambda += 5) {
        const double Mc1 = blackbody_spect(lambda, _temp);
        Yo += color_match[i][1] * Mc1;
    }

    xx = X / Yo;
    yy = 1.;
    zz = Z / Yo;
}



double ColorTemp::daylight_spect(double wavelength, double m1, double m2)
{
    //Values for Daylight illuminant: s0 s1 s2
    //s0
    static const double s0[97] = {61.80, 61.65, 61.50, 65.15, 68.80, 66.10, 63.40, 64.60, 65.80, 80.30, 94.80, 99.80, 104.80, 105.35, 105.90, 101.35, 96.80, 105.35, 113.90, 119.75, 125.60, 125.55, 125.50, 123.40, 121.30, 121.30, 121.30, 117.40, 113.50, 113.30,
                                  113.10, 111.95, 110.80, 108.65, 106.50, 107.65, 108.80, 107.05, 105.30, 104.85, 104.40, 102.20, 100.00, 98.00, 96.00, 95.55, 95.10, 92.10, 89.10, 89.80, 90.50, 90.40, 90.30, 89.35, 88.40, 86.20, 84.00, 84.55, 85.10,
                                  83.50, 81.90, 82.25, 82.60, 83.75, 84.90, 83.10, 81.30, 76.60, 71.90, 73.10, 74.30, 75.35, 76.40, 69.85, 63.30, 67.50, 71.70, 74.35, 77.00, 71.10, 65.20, 56.45, 47.70, 58.15, 68.60, 66.80, 65.00, 65.50, 66.00, 63.50, 61.00, 57.15,
                                  53.30, 56.10, 58.90, 60.40, 61.90
                                 };
    //s1
    static const double s1[97] = {41.60, 39.80, 38.00, 40.20, 42.40, 40.45, 38.50, 36.75, 35.00, 39.20, 43.40, 44.85, 46.30, 45.10, 43.90, 40.50, 37.10, 36.90, 36.70, 36.30, 35.90, 34.25, 32.60, 30.25, 27.90, 26.10, 24.30, 22.20, 20.10, 18.15, 16.20, 14.70,
                                  13.20, 10.90, 8.60, 7.35, 6.10, 5.15, 4.20, 3.05, 1.90, 0.95, 0.00, -0.80, -1.60, -2.55, -3.50, -3.50, -3.50, -4.65, -5.80, -6.50, -7.20, -7.90, -8.60, -9.05, -9.50, -10.20, -10.90, -10.80, -10.70, -11.35, -12.00, -13.00, -14.00,
                                  -13.80, -13.60, -12.80, -12.00, -12.65, -13.30, -13.10, -12.90, -11.75, -10.60, -11.10, -11.60, -11.90, -12.20, -11.20, -10.20, -9.00, -7.80, -9.50, -11.20, -10.80, -10.40, -10.50, -10.60, -10.15, -9.70, -9.00, -8.30,
                                  -8.80, -9.30, -9.55, -9.80
                                 };
    //s2
    static const double s2[97] = {6.70, 6.00, 5.30, 5.70, 6.10, 4.55, 3.00, 2.10, 1.20, 0.05, -1.10, -0.80, -0.50, -0.60, -0.70, -0.95, -1.20, -1.90, -2.60, -2.75, -2.90, -2.85, -2.80, -2.70, -2.60, -2.60, -2.60, -2.20, -1.80, -1.65, -1.50, -1.40, -1.30,
                                  -1.25, -1.20, -1.10, -1.00, -0.75, -0.50, -0.40, -0.30, -0.15, 0.00, 0.10, 0.20, 0.35, 0.50, 1.30, 2.10, 2.65, 3.20, 3.65, 4.10, 4.40, 4.70, 4.90, 5.10, 5.90, 6.70, 7.00, 7.30, 7.95, 8.60, 9.20, 9.80, 10.00, 10.20, 9.25, 8.30, 8.95,
                                  9.60, 9.05, 8.50, 7.75, 7.00, 7.30, 7.60, 7.80, 8.00, 7.35, 6.70, 5.95, 5.20, 6.30, 7.40, 7.10, 6.80, 6.90, 7.00, 6.70, 6.40, 5.95, 5.50, 5.80, 6.10, 6.30, 6.50
                                 };

    int wlm = (int)((wavelength - 350.) / 5.);
    return (s0[wlm] + m1 * s1[wlm] + m2 * s2[wlm]);
}



//tempxy : return x and y of xyY for 406 or more refreence color, and for T temperature from 2000K to 12000K
// we can change step for temperature and increase number  for T > 7500K if necessary
//these values Temp, x, y are references for all calculations and very precise.
//copyright J.Desmis august  2017 and june 2018 - 05/2023
void ColorTemp::tempxy(bool separated, int repref, float **Tx, float **Ty, float **Tz, float **Ta, float **Tb, float **TL, double *TX, double *TY, double *TZ, const procparams::WBParams & wbpar, int ttbeg, int ttend, double &wpx, double &wpz, double *WPX, double *WPZ)
{
    const double* spec_colorforxcyc[] = {//color references
        JDC468_BluH10_spect, JDC468_BluD6_spect, ColorchechCyaF3_spect, JDC468_BluM5_spect, // 0 3
        ColorGreenM25_spect,   JDC468_GreK7_spect, ColabSky42_0_m24_spect, ColabSky60_0_m31_spect,  ColorchechBluC150_m5_m22_spect,//8
        JDC468_GreQ7_spect, ColorchechDCBluN881_m7_m14_spect, ColorchechGreB3_spect, ColorchechPurD2_spect,  //12
        ColorchechSGBlaN3_6_spect, ColorchechGraC4_67_spect, JDC468_K15_87greyspect,//15
        JDC468_GraK14_44_spect, ColorGreenalsi_spect, Fictif_61greyspect, ColorchechGreD1_spect,//19
        ColorchechWhiA496_spect, JDC468_GreA10_spect, JDC468_GreI8_spect,//22
        ColabSkin91_4_14_spect, JDC468_PurE24_spect, //24
        ColorchechSGSkiK285_11_17_spect,  ColorchechGreE2_spect, ColorchechMagE3_spect, //27
        ColorchechSkiB166_18_18_spect, ColabSkin70_7_32_spect, ColorchechSGSkiF763_14_26_spect,//30
        ColorchechSkiA138_13_14_spect, ColabSkin57_22_18_spect, JDC468_YelN10_spect,//33
        ColabSkin35_15_17_spect, ColabSkin40_17_17_spect, ColorRedkurttu_spect, ColorYellowkeltano_spect,  ColorchechYelD3_spect, JDC468_OraO18_spect,//39
        JDC468_GreN7_spect, JDC468_RedG21va_spect, JDC468_OraD17_spect,//42
        ColorchechredC3_spect, JDC468_RedI9_spect, ColorRedpetunia_spect, ColorchechOraA2_spect,//46
        ColabSkin87_8_8_spect, ColabSkin89_8_21_spect, ColabSkin75_8_4_spect, ColabSkin75_10_33_spect,//50
        ColabSkin65_33_11_spect, ColabSkin65_7_24_spect, ColabSkin57_19_6_spect, ColabSkin57_4_19_spect, ColabSkin57_10_28_spect, ColabSkin40_17_6_spect,//56
        ColabSkin26_18_18_spect, ColabSkin90_m1_20_spect, ColorRedlupiini_spect, ColorRedhevosminttu_spect, //60
        ColorRedneilikka_spect, ColorRedpelagornia_spect, ColorRedtalvio_spect, ColorBrownpoimulehti_spect, ColorOrangetuntematon_spect,//65
        ColorOrangetlehmus_spect, ColorOrangvaahtera_spect, ColorBrownlehmus_spect, ColorBrownuotiosammal_spect,//69
        ColorBlacksoil_spect, ColorGraynahjajaekaelae_spect, //71
        ColorGreennuotisammal_spect, ColorGreenleskenlehti_spect, ColorGreenlinnunkaali_spect, //74
        ColorGreenpelto_spect, ColorGreenrodvoikukka, ColorGreenlehmus, ColorGreenlinden, ColorYellowlehmus, ColorYellowsuikeroalpi, //80
        ColorYellowpensashanhikki1, ColorYellowpensashanhikki2,  ColorBluehiidenvirna,  ColorBluekurkkuyrtti, //84
        ColorPinksiankaersaemoe, ColorVioletharakankello, ColorVioletalsikeapila, ColorVioletakilleija, ColorOrangekehaekukka,//89
        ColorRedpihlaja, ColorVioletpetunia, ColorVioletorvokki, ColorBluesinisievikki, ColorBlueiisoppi,  ColorBluelobelia, //95
        ColorWhiteojaka, ColorWhitepetunia, ColorWhitepelargonia, ColorWhitepaeivaen, JDC468_B14_75Redspect,//100
        ColorGreenkoriste, ColorGreenpoimulehti, ColorGreenhopeapaju, //103
        ColorReduuden, ColorRedpajuan, ColorRedjaloan, ColorBlueukon, ColorBlueorvokki, ColorBluemalvikki, //109
        ColorBlackmaito, ColorOrangpihlaja, ColorBlackpihlaja, //112
        ColorViolA1_spect, ColorViolA4_spect, ColorViolA6_spect, ColorBlueSkyK3_spect, ColorBlueSkyK9_spect, //117
        ColorBlueSkyC4_spect, ColorBlueSkyC14_spect, ColorBlueSkyE4_spect, //120
        ColorBlueSkyM1_spect, ColorBlueSky2B1_spect, ColorBlueSkyT7_spect, //123
        ColorBlueSkyU19_spect, ColorBlueSkyU2_spect, ColorBlueSkyT17_spect, //126
        ColorBlackM8_spect, ColorBlackM12_spect, ColorBlackM13_spect, ColorWhite2B12_spect, ColorWhite2B14_spect, //131
        JDC468_Blackred97_spect, JDC468_Blackredbl443_spect, JDC468_Blackbl27_spect, JDC468_Blackbl28_spect, //135
        JDC468_Blackgr214_spect, JDC468_Blackbl436_spect, JDC468_Whitebl455_spect, JDC468_Blackvio101_spect, JDC468_Whitebl92_spect, JDC468_Greyredbl94_spect, //141
        JDC468_Blue32_spect, JDC468_Blue236_spect, JDC468_Gre300_spect, //144
        JDC468_Blue340_spect, JDC468_Gree110_spect, JDC468_Gree457_spect, JDC468_Yel241_spect, JDC468_Ora321_spect, JDC468_Yellow353_spect, JDC468_Mag465_spect, //151
        JDC468_Mag333_spect, JDC468_Mag203_spect, J570_BlueB6_spect, J570_BlueB15_spect, J570_BlueC2_spect, J570_BlueC14_spect, J570_BlueC16_spect,//158
        J570_BlueF1_spect, J570_BlueF2_spect, J570_BlueF10_spect, J570_BlueF13_spect, J570_BlueG9_spect, J570_BlueG19_spect, J570_BlueI5_spect,//165
        J570_BlueI3_spect, J570_BlueI19_spect, J570_BlueJ4_spect, J570_BlueJ6_spect, J570_BlueJ11_spect, J570_BlueK5_spect, //171
        J570_BlueN1_spect, J570_BlueN4_spect, J570_BlueO19_spect, J570_BlueU8_spect, J570_NeuN8_spect,//176
        J570_NeuN9_spect, J570_NeuO8_spect, J570_NeuO11_spect, J570_NeuD5_spect,//180
        J570_NeuE11_spect, J570_NeuK16_spect, J570_NeuM3_spect, J570_NeuN18_spect, J570_NeuQ1_spect, J570_NeuS7_spect, J570_NeuV10_spect, J570_NeuW18_spect, J570_NeuZ14_spect, //189
        J570_NeuC18_spect, J570_NeuD17_spect, J570_NeuJ11_spect, J570_NeuL4_spect, Colorlab_n72_n2_spect,//194
        Colorlab_10_n70_spect, Colorlab_n33_n70_spect, Colorlab_n8_n74_spect, Colorlab_19_n69_spect, Colorlab_n80_10_spect, Colorlab_n80_26_spect,//200
        Colorlab_n80_5_9_5_9spect, /* JDC468_greyc14_66_spect, JDC468_greym13_325_spect, JDC468_greyf26_156_spect, Colorlab_n57_5_6_9spect,*/ Colorlab_L61_110_110Rec2020spect,//202
        Colorlab_L63_120_m56Rec2020spect, Colorlab_L63_m50_m60Rec2020spect, Colorlab_L63_m120_80Rec2020spect, Colorlab_L42_110_m100Prospect, Colorlab_L42_m70_m100Prospect,//207
        Colorlab_L56_m120_90Prospect, Colorlab_L25_60_m120Prospect, Colorlab_L75_50_120Prospect, Colorlab_L75_m120_0Prospect, J570_NeuN8_spect2, J570_NeuN9_spect2, //213
        J570_NeuO8_spect2, J570_NeuO11_spect2, J570_NeuD5_spect2, J570_NeuE11_spect2, J570_NeuK16_spect2, J570_NeuM3_spect2, Colorlab_L22_2_1_3Prospect,//220
        Colorlab_L44_2_8_3_9spect, Colorlab_L44_2_8_3_9spect2, Colorlab_L95_2_3_15_6spect, Colorlab_L95_2_3_15_6spect2, Colorlab_L40_3_5_10_7spect,//225
        Colorlab_L40_3_5_10_7spect2, /*Colorlab_L40_3_5_10_7Prospect3,*/ Colorlab_L34_1_8_1_9spect, Colorlab_L34_1_8_1_9spect2, Colorlab_L64_1_8_m1_9spect,//229
        Colorlab_L84_0_8_m1spect, Colorlab_L63_1_3_m2spect, Colorlab_L44_2_3_m3spect, Colorlab_L65_96_45spect, Colorlab_L52_47_57spect, Colorlab_L31_62_27spect,  //235
        Colorlab_L79_m9_m28spect, Colorlab_L58_50_31spect, Colorlab_L31_m52_27spect, Colorlab_L44_2_2_m7_35spect, Colorlab_L47_m10_8_0_41spect, Colorlab_L32_4_8_m3_2spect,//241
        Colorlab_L57_m6_9_2_9spect, Colorlab_L33_2_4_m4_5spect, Colorlab_L35_11_65_m1_1spect, Colorlab_L52_m2_7_8_9spect, Colorlab_L32_7_m2_5spect, Colorlab_L32_3_4_m3_8spect,//247
        Colorlab_L50_m5_3_6_5spect, Colorlab_L44_3_96_m8_8spect, Colorlab_L34_3_6_m5_4spect, Colorlab_L31_5_9_m4spect, Colorlab_L35_3_4_m11spect, Colorlab_L31_4_5_m4_7spect, //253
        Colorlab_L35_4_8_m6_4spect, Colorlab_L95_10_7_m14_3spect, Colorlab_L36_34_m7_5spect, Colorlab_L37_59_2spect, Colorlab_L69_14_m9spect, Colorlab_L92_13_m16spect,//259
        Colorlab_L49_21_m12spect, Colorlab_L56_20_m15spect, Colorlab_L68_21_m19spect, //262
        Colorlab_L98_m2_m32spect, Colorlab_L98_m2_m32spect2, Colorlab_L41_m27_m16spect, Colorlab_L41_m27_m16spect2, Colorlab_L15_m9_4spect, Colorlab_L15_m9_4spect2,//268
        Colorlab_L11_m11_2spect, Colorlab_L14_m4_3spect, Colorlab_L41_38_24spect, Colorlab_L41_38_24spect2, Colorlab_L53_48_58spect, Colorlab_L53_48_58spect2,//274
        Colorlab_L70_44_86spect, Colorlab_L70_44_86spect2, Colorlab_L38_42_19spect, Colorlab_L38_42_19spect2, //278
        Colorlab_L60_63_85spect, Colorlab_L60_63_85spect2, Colorlab_L80_75_30spect, Colorlab_L80_75_30spect2, Colorlab_L28_m21_24spect,//284
        Colorlab_L45_m33_47spect, Colorlab_L45_m33_47spect2,  Colorlab_L26_m7_404spect, Colorlab_L34_m61_2spect, Colorlab_L32_m16_17spect, Colorlab_L30_m19_15spect,
        Colorlab_L30_m17_16spect, Colorlab_L35_m8_4spect, Colorlab_L37_m7_5spect, Colorlab_L45_m7_2spect, Colorlab_L40_m6_5spect, Colorlab_L46_m6_2spect, Colorlab_L48_m69_16spect,
        Colorlab_L51_89_53spect, Colorlab_L49_84_33spect, Colorlab_L59_m51_31spect, Colorlab_L48_m69_16spect2, Colorlab_L53_m71_6spect, Colorlab_L51_m89_53spect2,
        Colorlab_L49_84_33spect2, Colorlab_L36_m27_28spect, Colorlab_L36_m27_28spect2, Colorlab_L36_m27_28spect3, Colorlab_L63_16_71spect,
        Colorlab_L84_4_46spect, Colorlab_L84_4_46spect2, Colorlab_L75_m66_19spect, Colorlab_L75_m66_19spect2, Colorlab_L64_m82_m6spect,  Colorlab_L64_m82_m6spect2,
        Colorlab_L66_m71_m17spect, Colorlab_L66_m71_m17spect2, Colorlab_L22_m8_m60spect, //317
        Colorlab_L22_m8_m60spect2, Colorlab_L15_m4_m42spect, Colorlab_L15_m4_m42spect2, Colorlab_L13_3_m23spect, Colorlab_L27_4_m90spect, Colorlab_L19_1_m29spect,
        Colorlab_L27_4_m90spect2, Colorlab_L16_0_m44spect, Colorlab_L16_0_m44spect2, Colorlab_L13_m3_m36spect, Colorlab_L13_m3_m36spect2,
        Colorlab_L31_m23_m60spect, Colorlab_L31_m23_m60spect2, Colorlab_L17_3_m40spect, Colorlab_L17_3_m40spect2, Colorlab_L21_9_m7spect, Colorlab_L78_4_m74spect,
        Colorlab_L31_m58_m66spect, Colorlab_L61_m11_m12spect, Colorlab_L61_m11_m12spect2, Colorlab_L29_1_m13spect, Colorlab_L29_1_m13spect2, Colorlab_L2_14_m1spect, //339
        Colorlab_L5_39_m7spect, Colorlab_L15_5_m13spect, Colorlab_L12_5_m6spect, Colorlab_L12_5_m6spect2, Colorlab_L37_m59_m24spect, Colorlab_L37_m59_m24spect2,//345
        Colorlab_L15_55_23spect, Colorlab_L11_m55_m11spect, Colorlab_L8_m10_m2spect, Colorlab_L14_m10_m7spect, Colorlab_L20_m16_m13spect, Colorlab_L8_m10_m2spect2, Colorlab_L14_m10_m7spect2, Colorlab_L20_m16_m13spect2, //353
        Colorlab_L6_m9_1spect, Colorlab_L20_m9_m10spect, Colorlab_L85_10_45spect, Colorlab_L90_m7_82spect, Colorlab_L95_2_18spect,
        Colorlab_L39_7_4spect, Colorlab_L39_4_1spect, Colorlab_L39_3_m1spect, Colorlab_L40_3_m2spect, Colorlab_L36_2_2spect,
        Colorlab_L39_7_4spect2, Colorlab_L39_4_1spect2, Colorlab_L39_3_m1spect2, Colorlab_L40_3_m2spect2, Colorlab_L36_2_2spect2, //369
        Colorlab_L40_4_m2spect, Colorlab_L41_1_m6spect, Colorlab_L40_4_m2spect2, Colorlab_L41_1_m6spect2, Colorlab_L41_12_14spect, Colorlab_L41_12_14spect2,
        Colorlab_L10_0_m22spect, Colorlab_L38_60_8spect, Colorlab_L49_85_39spect, Colorlab_L42_1_m18spect, //379
        Colorlab_L48_19_m25spect, Colorlab_L30_21_m25spect, Colorlab_L15_10_m15spect, Colorlab_L48_19_m25spect2, Colorlab_L30_21_m25spect2, Colorlab_L15_10_m15spect2,
        Colorlab_L60_26_m25spect, Colorlab_L40_26_m45spect, Colorlab_L20_10_m45spect, //388
        ColorBlueSkyK3_spect2, ColorBlueSkyK9_spect2, ColorBlueSkyC4_spect2, ColorBlueSkyC14_spect2, ColorBlueSkyE4_spect2,
        ColorBlueSkyM1_spect2, ColorBlueSky2B1_spect2, ColorBlueSkyT7_spect2, ColorBlueSkyU19_spect2, ColorBlueSkyU2_spect2,
        Colorlab_L40_1_m40spect, Colorlab_L30_4_m30spect, Colorlab_L8_11_m25spect, Colorlab_L40_1_m40spect2, Colorlab_L30_4_m30spect2, Colorlab_L8_11_m25spect2,
        Colorlab_L26_m8_m25spect, Colorlab_L26_m8_m25spect2, //406
        Colorlab_L17_3_m40spect3, Colorlab_L17_3_m40spect4, Colorlab_L17_3_m40spect5, Colorlab_L40_26_m45spect2,  Colorlab_L17_3_m40spect6, ColorchechCyaF3_spect2, ColorchechCyaF3_spect3,
        Colorlab_L22_1_m42spect, Colorlab_L27_m1_m47spect, Colorlab_L26_m8_m25spect3, Colorlab_L22_1_m42spect2, Colorlab_L27_m1_m47spect2,
        Colorlab_L40_30_m30spect, Colorlab_L40_20_m35spect, Colorlab_L20_10_m45spect2, Colorlab_L20_10_m45spect3,
        Colorlab_L40_30_m30spect2, Colorlab_L40_20_m35spect2, Colorlab_10_n70_spect2,  Colorlab_L22_1_m42spect3, Colorlab_10_n70_spect3, Colorlab_L22_1_m42spect4, Colorlab_10_n70_spect4
        //J570_NeuN18_spect2, J570_NeuQ1_spect2 //J570_NeuS7_spect2, J570_NeuV10_spect2, J570_NeuW18_spect2,
        //J570_NeuZ14_spect2,
        //J570_NeuC18_spect2,
        //J570_NeuD17_spect2,
        //J570_NeuJ11_spect2,
        //J570_NeuL4_spect2
    };//

    const double* spec_colorforxcyc_old[] = {//color references for 5.9
        JDC468_BluH10_spect, JDC468_BluD6_spect, ColorchechCyaF3_spect, JDC468_BluM5_spect, // 0 3
        ColorGreenM25_spect,   JDC468_GreK7_spect, ColabSky42_0_m24_spect, ColabSky60_0_m31_spect,  ColorchechBluC150_m5_m22_spect,//8
        JDC468_GreQ7_spect, ColorchechDCBluN881_m7_m14_spect, ColorchechGreB3_spect, ColorchechPurD2_spect,  //12
        ColorchechSGBlaN3_6_spect, ColorchechGraC4_67_spect, JDC468_K15_87greyspect,//15
        JDC468_GraK14_44_spect, ColorGreenalsi_spect, Fictif_61greyspect, ColorchechGreD1_spect,//19
        ColorchechWhiA496_spect, JDC468_GreA10_spect, JDC468_GreI8_spect,//22
        ColabSkin91_4_14_spect, JDC468_PurE24_spect, //24
        ColorchechSGSkiK285_11_17_spect,  ColorchechGreE2_spect, ColorchechMagE3_spect, //27
        ColorchechSkiB166_18_18_spect, ColabSkin70_7_32_spect, ColorchechSGSkiF763_14_26_spect,//30
        ColorchechSkiA138_13_14_spect, ColabSkin57_22_18_spect, JDC468_YelN10_spect,//33
        ColabSkin35_15_17_spect, ColabSkin40_17_17_spect, ColorRedkurttu_spect, ColorYellowkeltano_spect,  ColorchechYelD3_spect, JDC468_OraO18_spect,//39
        JDC468_GreN7_spect, JDC468_RedG21va_spect, JDC468_OraD17_spect,//42
        ColorchechredC3_spect, JDC468_RedI9_spect, ColorRedpetunia_spect, ColorchechOraA2_spect,//46
        ColabSkin87_8_8_spect, ColabSkin89_8_21_spect, ColabSkin75_8_4_spect, ColabSkin75_10_33_spect,//50
        ColabSkin65_33_11_spect, ColabSkin65_7_24_spect, ColabSkin57_19_6_spect, ColabSkin57_4_19_spect, ColabSkin57_10_28_spect, ColabSkin40_17_6_spect,//56
        ColabSkin26_18_18_spect, ColabSkin90_m1_20_spect, ColorRedlupiini_spect, ColorRedhevosminttu_spect, //60
        ColorRedneilikka_spect, ColorRedpelagornia_spect, ColorRedtalvio_spect, ColorBrownpoimulehti_spect, ColorOrangetuntematon_spect,//65
        ColorOrangetlehmus_spect, ColorOrangvaahtera_spect, ColorBrownlehmus_spect, ColorBrownuotiosammal_spect,//69
        ColorBlacksoil_spect, ColorGraynahjajaekaelae_spect, //71
        ColorGreennuotisammal_spect, ColorGreenleskenlehti_spect, ColorGreenlinnunkaali_spect, //74
        ColorGreenpelto_spect, ColorGreenrodvoikukka, ColorGreenlehmus, ColorGreenlinden, ColorYellowlehmus, ColorYellowsuikeroalpi, //80
        ColorYellowpensashanhikki1, ColorYellowpensashanhikki2,  ColorBluehiidenvirna,  ColorBluekurkkuyrtti, //84
        ColorPinksiankaersaemoe, ColorVioletharakankello, ColorVioletalsikeapila, ColorVioletakilleija, ColorOrangekehaekukka,//89
        ColorRedpihlaja, ColorVioletpetunia, ColorVioletorvokki, ColorBluesinisievikki, ColorBlueiisoppi,  ColorBluelobelia, //95
        ColorWhiteojaka, ColorWhitepetunia, ColorWhitepelargonia, ColorWhitepaeivaen, JDC468_B14_75Redspect,//100
        ColorGreenkoriste, ColorGreenpoimulehti, ColorGreenhopeapaju, //103
        ColorReduuden, ColorRedpajuan, ColorRedjaloan, ColorBlueukon, ColorBlueorvokki, ColorBluemalvikki, //109
        ColorBlackmaito, ColorOrangpihlaja, ColorBlackpihlaja, //112
        ColorViolA1_spect, ColorViolA4_spect, ColorViolA6_spect, ColorBlueSkyK3_spect, ColorBlueSkyK9_spect, //117
        ColorBlueSkyC4_spect, ColorBlueSkyC14_spect, ColorBlueSkyE4_spect, //120
        ColorBlueSkyM1_spect, ColorBlueSky2B1_spect, ColorBlueSkyT7_spect, //123
        ColorBlueSkyU19_spect, ColorBlueSkyU2_spect, ColorBlueSkyT17_spect, //126
        ColorBlackM8_spect, ColorBlackM12_spect, ColorBlackM13_spect, ColorWhite2B12_spect, ColorWhite2B14_spect, //131
        JDC468_Blackred97_spect, JDC468_Blackredbl443_spect, JDC468_Blackbl27_spect, JDC468_Blackbl28_spect, //135
        JDC468_Blackgr214_spect, JDC468_Blackbl436_spect, JDC468_Whitebl455_spect, JDC468_Blackvio101_spect, JDC468_Whitebl92_spect, JDC468_Greyredbl94_spect, //141
        JDC468_Blue32_spect, JDC468_Blue236_spect, JDC468_Gre300_spect, //144
        JDC468_Blue340_spect, JDC468_Gree110_spect, JDC468_Gree457_spect, JDC468_Yel241_spect, JDC468_Ora321_spect, JDC468_Yellow353_spect, JDC468_Mag465_spect, //151
        JDC468_Mag333_spect, JDC468_Mag203_spect, J570_BlueB6_spect, J570_BlueB15_spect, J570_BlueC2_spect, J570_BlueC14_spect, J570_BlueC16_spect,//158
        J570_BlueF1_spect, J570_BlueF2_spect, J570_BlueF10_spect, J570_BlueF13_spect, J570_BlueG9_spect, J570_BlueG19_spect, J570_BlueI5_spect,//165
        J570_BlueI3_spect, J570_BlueI19_spect, J570_BlueJ4_spect, J570_BlueJ6_spect, J570_BlueJ11_spect, J570_BlueK5_spect, //171
        J570_BlueN1_spect, J570_BlueN4_spect, J570_BlueO19_spect, J570_BlueU8_spect, J570_NeuN8_spect,//176
        J570_NeuN9_spect, J570_NeuO8_spect, J570_NeuO11_spect, J570_NeuD5_spect,//180
        J570_NeuE11_spect, J570_NeuK16_spect, J570_NeuM3_spect, J570_NeuN18_spect,
        J570_NeuQ1_spect, J570_NeuS7_spect,
        J570_NeuV10_spect, J570_NeuW18_spect, J570_NeuZ14_spect, //189
        J570_NeuC18_spect, J570_NeuD17_spect, J570_NeuJ11_spect, J570_NeuL4_spect, Colorlab_n72_n2_spect,
        Colorlab_10_n70_spect, Colorlab_n33_n70_spect, Colorlab_n8_n74_spect, Colorlab_19_n69_spect, Colorlab_n80_10_spect, Colorlab_n80_26_spect,
        Colorlab_n80_5_9_5_9spect //, Colorlab_n57_5_6_9spect

    };



    typedef struct WbTxyz {
        double Tem;
        double XX;
        double ZZ;
    } WbTxyz;
    //probably can be "passed" with rawimagesource.cc but I don't know how to do this.
    constexpr WbTxyz Txyz[191] = {//temperature Xwb Zwb 191 values - same table as in Rawimagesource.cc  x wb and y wb are calculated after
        {2001., 1.273842, 0.145295},
        {2051., 1.258802, 0.156066},
        {2101., 1.244008, 0.167533},
        {2151., 1.230570, 0.178778},
        {2201., 1.217338, 0.190697},
        {2251., 1.205305, 0.202338},
        {2301., 1.193444, 0.214632},
        {2351., 1.182648, 0.226598},
        {2401., 1.171996, 0.239195},
        {2451., 1.162290, 0.251421},
        {2501., 1.152883, 0.264539},
        {2551., 1.143965, 0.276682},
        {2605., 1.134667, 0.290722},
        {2655., 1.126659, 0.303556},
        {2705., 1.119049, 0.316446},
        {2755., 1.111814, 0.329381},
        {2790., 1.106961, 0.338455},
        {2803., 1.105381, 0.342193},
        {2825., 1.102275, 0.347542},
        {2856., 1.098258, 0.355599},
        {2880., 1.095233, 0.361840},
        {2910., 1.091550, 0.369645},
        {2930., 1.089155, 0.374849},
        {2960., 1.085649, 0.382655},
        {2980., 1.083369, 0.387858},
        {3003., 1.080982, 0.394258},
        {3025., 1.078397, 0.399561},
        {3050., 1.075727, 0.406057},
        {3075., 1.073122, 0.412550},
        {3103., 1.070277, 0.419815},
        {3128., 1.067801, 0.426296},
        {3153., 1.065384, 0.432769},
        {3175., 1.063305, 0.438459},
        {3203., 1.060906, 0.446161},
        {3225., 1.058738, 0.451367},
        {3250., 1.056535, 0.457806},
        {3280., 1.053960, 0.465519},
        {3303., 1.052034, 0.471422},
        {3353., 1.047990, 0.484218},
        {3400., 1.044547, 0.496719},
        {3450., 1.040667, 0.508891},
        {3500., 1.037145, 0.521523},
        {3550., 1.033783, 0.534090},
        {3600., 1.030574, 0.546590},
        {3650., 1.027510, 0.559020},
        {3699., 1.024834, 0.571722},
        {3801., 1.019072, 0.596102},
        {3851., 1.016527, 0.608221},
        {3902., 1.014244, 0.621136},
        {3952., 1.011729, 0.632447},
        {4002., 0.996153, 0.609518},
        {4052., 0.993720, 0.620805},
        {4102., 0.993908, 0.631520},
        {4152., 0.989179, 0.643262},
        {4202., 0.989283, 0.653999},
        {4252., 0.985039, 0.665536},
        {4302., 0.985067, 0.676288},
        {4352., 0.981271, 0.687599},
        {4402., 0.981228, 0.698349},
        {4452., 0.977843, 0.709425},
        {4502., 0.977736, 0.720159},
        {4552., 0.974728, 0.730993},
        {4602., 0.974562, 0.741698},
        {4652., 0.971899, 0.752284},
        {4702., 0.971681, 0.762949},
        {4752., 0.969335, 0.773285},
        {4802., 0.969069, 0.783899},
        {4827., 0.967570, 0.788836},
        {4852., 0.967011, 0.793982},
        {4877., 0.966465, 0.799108},
        {4902., 0.965933, 0.804214},
        {4914., 0.965682, 0.806658},
        {4927., 0.965414, 0.809229},
        {4940., 0.965149, 0.811937},
        {4952., 0.964908, 0.814366},
        {4965., 0.964650, 0.816993},
        {4977., 0.964415, 0.819412},
        {4990., 0.964163, 0.822028},
        {5002., 0.963934, 0.824438},//80
        {5015., 0.963689, 0.827044},
        {5027., 0.963465, 0.829444},
        {5040., 0.963226, 0.832039},
        {5051., 0.963008, 0.834429},
        {5065., 0.963226, 0.832039},
        {5077., 0.962563, 0.839395},
        {5090., 0.962336, 0.841968},
        {5102., 0.962129, 0.844339},
        {5115., 0.961907, 0.846902},
        {5127., 0.961706, 0.849263},
        {5140., 0.961490, 0.851815},
        {5151., 0.961294, 0.854166},
        {5177., 0.960893, 0.859049},
        {5202., 0.960501, 0.863911},
        {5253., 0.959749, 0.873572},
        {5302., 0.959313, 0.883815},
        {5351., 0.958361, 0.892644},
        {5402., 0.957903, 0.902793},
        {5452., 0.957116, 0.911379},
        {5502., 0.956639, 0.921431},
        {5553., 0.956002, 0.929779},
        {5602., 0.955509, 0.939728},
        {5652., 0.955008, 0.947842},
        {5702., 0.954502, 0.957685},
        {5752., 0.954124, 0.965569},
        {5802., 0.953608, 0.975303},
        {5852., 0.953342, 0.982963},
        {5902., 0.952818, 0.992584},
        {5952., 0.952652, 1.000025},
        {6002., 0.952122, 1.009532},
        {6052., 0.952047, 1.016759},
        {6102., 0.951514, 1.026149},
        {6152., 0.951520, 1.033168},
        {6202., 0.950985, 1.042439},
        {6252., 0.951064, 1.049256},
        {6302., 0.950530, 1.058406},
        {6352., 0.950674, 1.065027},
        {6380., 0.950576, 1.069386},
        {6402., 0.950143, 1.074055},
        {6425., 0.950428, 1.076341},
        {6452., 0.950345, 1.080484},
        {6475., 0.950277, 1.083996},
        {6502., 0.950201, 1.088097},
        {6525., 0.950139, 1.091573},
        {6552., 0.950070, 1.095633},
        {6575., 0.950014, 1.099075},
        {6602., 0.949952, 1.103094},
        {6625., 0.949902, 1.106501},
        {6652., 0.949846, 1.110479},
        {6675., 0.949801, 1.113852},
        {6702., 0.949752, 1.119138},
        {6725., 0.949712, 1.121128},
        {6752., 0.949668, 1.125027},
        {6802., 0.949596, 1.132190},
        {6852., 0.949533, 1.139281},
        {6902., 0.949033, 1.147691},
        {6952., 0.949437, 1.153246},
        {7002., 0.949402, 1.160129},
        {7052., 0.949376, 1.166966},
        {7102., 0.949358, 1.173732},
        {7152., 0.949348, 1.180429},
        {7202., 0.949346, 1.187058},
        {7252., 0.949350, 1.193619},
        {7301., 0.948896, 1.201432},
        {7352., 0.949380, 1.206541},
        {7402., 0.949405, 1.212904},
        {7451., 0.949434, 1.219076},
        {7501., 0.949471, 1.225312},
        {7551., 0.949512, 1.231485},
        {7601., 0.949099, 1.239061},
        {7675., 0.949638, 1.246525},
        {7751., 0.949729, 1.255559},
        {7825., 0.949828, 1.264225},
        {7901., 0.949498, 1.274460},
        {7952., 0.950018, 1.278800},
        {8025., 0.950137, 1.287013},
        {8095., 0.950259, 1.294777},
        {8151., 0.950361, 1.300912},
        {8225., 0.950501, 1.308915},
        {8301., 0.950253, 1.318464},
        {8375., 0.950804, 1.324786},
        {8451., 0.950966, 1.332651},
        {8525., 0.951129, 1.340199},
        {8601., 0.950941, 1.349261},
        {8701., 0.951533, 1.357724},
        {8801., 0.951772, 1.367421},
        {8901., 0.952018, 1.376935},
        {9001., 0.951969, 1.387639},
        {9201., 0.952784, 1.404422},
        {9401., 0.953081, 1.423213},//since 5 2023 I increased the number of temp references above 9401K
        {9651., 0.953993, 1.442883},
        {9901., 0.954537, 1.464134},
        {10201., 0.955520, 1.485825},
        {10501., 0.956321, 1.508623},
        {10751., 0.957057, 1.524806},
        {11001., 0.957747, 1.541281},
        {11251., 0.958436, 1.557207},
        {11501., 0.959112, 1.572366},
        {11751., 0.959784, 1.587037},
        {12001., 0.960440, 1.601019},//since 5 2023 I increased the number of temp references above 12000K
        {12251., 0.961090, 1.614566},
        {12501., 0.963963, 1.627492},
        {12751., 0.962350, 1.640031},
        {13001., 0.962962, 1.652055},
        {13251., 0.963561, 1.663638},
        {13501., 0.964147, 1.674804},
        {13751., 0.964720, 1.685571},
        {14001., 0.965279, 1.695919},
        {14251., 0.965827, 1.705950},
        {14501., 0.966363, 1.715637},
        {14751., 0.966886, 1.724998},
        {15001., 0.967397, 1.734047}
    };

    //compatibility 5.9
    constexpr WbTxyz Txyzs[118] = {//temperature Xwb Zwb 118 values - same table as in Rawimagesource.cc  x wb and y wb are calculated after for 5.9
        {2001., 1.273842, 0.145295},
        {2101., 1.244008, 0.167533},
        {2201., 1.217338, 0.190697},
        {2301., 1.193444, 0.214632},
        {2401., 1.171996, 0.239195},
        {2501., 1.152883, 0.264539},
        {2605., 1.134667, 0.290722},
        {2655., 1.126659, 0.303556},
        {2705., 1.119049, 0.316446},
        {2755., 1.111814, 0.329381},
        {2803., 1.105381, 0.342193},
        {2856., 1.098258, 0.355599},
        {2910., 1.091550, 0.369645},
        {2960., 1.085649, 0.382655},
        {3003., 1.080982, 0.394258},
        {3050., 1.075727, 0.406057},
        {3103., 1.070277, 0.419815},
        {3153., 1.065384, 0.432769},
        {3203., 1.060906, 0.446161},
        {3250., 1.056535, 0.457806},
        {3303., 1.052034, 0.471422},
        {3353., 1.047990, 0.484218},
        {3400., 1.044547, 0.496719},
        {3450., 1.040667, 0.508891},
        {3500., 1.037145, 0.521523},
        {3550., 1.033783, 0.534090},
        {3600., 1.030574, 0.546590},
        {3650., 1.027510, 0.559020},
        {3699., 1.024834, 0.571722},
        {3801., 1.019072, 0.596102},
        {3851., 1.016527, 0.608221},
        {3902., 1.014244, 0.621136},
        {3952., 1.011729, 0.632447},
        {4002., 0.996153, 0.609518},
        {4052., 0.993720, 0.620805},
        {4102., 0.993908, 0.631520},
        {4152., 0.989179, 0.643262},
        {4202., 0.989283, 0.653999},
        {4252., 0.985039, 0.665536},
        {4302., 0.985067, 0.676288},
        {4352., 0.981271, 0.687599},
        {4402., 0.981228, 0.698349},
        {4452., 0.977843, 0.709425},
        {4502., 0.977736, 0.720159},
        {4552., 0.974728, 0.730993},
        {4602., 0.974562, 0.741698},
        {4652., 0.971899, 0.752284},
        {4702., 0.971681, 0.762949},
        {4752., 0.969335, 0.773285},
        {4802., 0.969069, 0.783899},
        {4827., 0.967570, 0.788836},
        {4852., 0.967011, 0.793982},
        {4877., 0.966465, 0.799108},
        {4902., 0.965933, 0.804214},
        {4927., 0.965414, 0.809229},
        {4952., 0.964908, 0.814366},
        {4977., 0.964415, 0.819412},
        {5002., 0.963934, 0.824438},
        {5027., 0.963465, 0.829444},
        {5052., 0.963008, 0.834429},
        {5077., 0.962563, 0.839395},
        {5102., 0.962129, 0.844339},
        {5127., 0.961706, 0.849263},
        {5152., 0.961294, 0.854166},
        {5177., 0.960893, 0.859049},
        {5202., 0.960501, 0.863911},
        {5252., 0.959749, 0.873572},
        {5302., 0.959313, 0.883815},
        {5352., 0.958361, 0.892644},
        {5402., 0.957903, 0.902793},
        {5452., 0.957116, 0.911379},
        {5502., 0.956639, 0.921431},
        {5552., 0.956002, 0.929779},
        {5602., 0.955509, 0.939728},
        {5652., 0.955008, 0.947842},
        {5702., 0.954502, 0.957685},
        {5752., 0.954124, 0.965569},
        {5802., 0.953608, 0.975303},
        {5852., 0.953342, 0.982963},
        {5902., 0.952818, 0.992584},
        {5952., 0.952652, 1.000025},
        {6002., 0.952122, 1.009532},
        {6052., 0.952047, 1.016759},
        {6102., 0.951514, 1.026149},
        {6152., 0.951520, 1.033168},
        {6202., 0.950985, 1.042439},
        {6252., 0.951064, 1.049256},
        {6302., 0.950530, 1.058406},
        {6352., 0.950674, 1.065027},
        {6402., 0.950143, 1.074055},
        {6452., 0.950345, 1.080484},
        {6502., 0.950201, 1.088097},
        {6552., 0.950070, 1.095633},
        {6602., 0.949952, 1.103094},
        {6652., 0.949846, 1.110479},
        {6702., 0.949752, 1.119138},
        {6752., 0.949668, 1.125027},
        {6802., 0.949596, 1.132190},
        {6902., 0.949033, 1.147691},
        {7002., 0.949402, 1.160129},
        {7152., 0.949348, 1.180429},
        {7301., 0.948896, 1.201432},
        {7451., 0.949434, 1.219076},
        {7601., 0.949099, 1.239061},
        {7751., 0.949729, 1.255559},
        {7901., 0.949498, 1.274460},
        {8151., 0.950361, 1.300912},
        {8301., 0.950253, 1.318464},
        {8451., 0.950966, 1.332651},
        {8601., 0.950941, 1.349261},
        {8801., 0.951772, 1.367421},
        {9001., 0.951969, 1.387639},
        {9201., 0.952784, 1.404422},
        {9401., 0.953081, 1.423213},
        {9901., 0.954537, 1.464134},
        {10501., 0.956321, 1.508623},
        {11001., 0.957747, 1.541281},
        {12001., 0.960440, 1.601019}
    };

    int N_c = sizeof(spec_colorforxcyc) / sizeof(spec_colorforxcyc[0]);   //number of color

    if (wbpar.itcwb_sampling) {
        N_c = sizeof(spec_colorforxcyc_old) / sizeof(spec_colorforxcyc_old[0]);   //number of color 5.9
    }


    int N_t = sizeof(Txyz) / sizeof(Txyz[0]);   //number of temperature White point

    if (wbpar.itcwb_sampling) {
        N_t = sizeof(Txyzs) / sizeof(Txyzs[0]);   //number of temperature White point 5.9
    }

    if (settings->verbose) {
        printf("Number max spectral colors=%i Number sampling temp=%i\n", N_c, N_t);
    }

    typedef struct XYZref {
        double Xref;
        double Yref;
        double Zref;
    } XYZref;
    
    XYZref Refxyz[N_c + 1];

    for (int i = 0; i < N_c; i++) {
        Refxyz[i].Xref = 0.f;
        Refxyz[i].Yref = 0.f;
        Refxyz[i].Zref = 0.f;
    }

    bool observerchoice = false;

    observerchoice = (wbpar.observer == StandardObserver::TEN_DEGREES);

    if (wbpar.itcwb_sampling) {
        observerchoice = false;
    }

    const color_match_type &color_match = observerchoice ? cie_colour_match_jd : cie_colour_match_jd2;

    if (separated) {
        double tempw = Txyz[repref].Tem;

        if (wbpar.itcwb_sampling) {
            tempw = Txyzs[repref].Tem;
        }

        double yy = 0.;
        whitepoint(tempw, wpx, yy, wpz, color_match);

        if (tempw <= INITIALBLACKBODY) {
           for (int i = 0; i < N_c; i++) {
                if (! wbpar.itcwb_sampling) {
                    spectrum_to_color_xyz_blackbody(spec_colorforxcyc[i], tempw, TX[i], TY[i], TZ[i], color_match);
                } else {
                    spectrum_to_color_xyz_blackbody(spec_colorforxcyc_old[i], tempw, TX[i], TY[i], TZ[i], color_match);
                }
            }
        } else {
            double m11, m22, x_DD, y_DD, interm2;

            if (tempw <= 7000) {
                x_DD = -4.6070e9 / (tempw * tempw * tempw) + 2.9678e6 / (tempw * tempw) + 0.09911e3 / tempw + 0.244063;
            } else {
                x_DD = -2.0064e9 / (tempw * tempw * tempw) + 1.9018e6 / (tempw * tempw) + 0.24748e3 / tempw + 0.237040;
            }

            y_DD = -3.0 * x_DD * x_DD + 2.87 * x_DD - 0.275;
            //calculate D -daylight in function of s0, s1, s2 and temp ==> x_D y_D
            //S(lamda)=So(lambda)+m1*s1(lambda)+m2*s2(lambda)
            interm2 = (0.0241 + 0.2562 * x_DD - 0.734 * y_DD);
            m11 = (-1.3515 - 1.7703 * x_DD + 5.9114 * y_DD) / interm2;
            m22 = (0.03 - 31.4424 * x_DD + 30.0717 * y_DD) / interm2;

            for (int i = 0; i < N_c; i++) {
                if (! wbpar.itcwb_sampling) {
                    spectrum_to_color_xyz_daylight(spec_colorforxcyc[i], m11, m22, TX[i], TY[i], TZ[i], color_match);
                } else {
                    spectrum_to_color_xyz_daylight(spec_colorforxcyc_old[i], m11, m22, TX[i], TY[i], TZ[i], color_match);
                }
            }
        }
    } else {
        for (int tt = ttbeg; tt < ttend; tt++) {
            double tempw = Txyz[tt].Tem;

            if (wbpar.itcwb_sampling) {
                tempw = Txyzs[repref].Tem;
            }

            double yy = 0.;
            whitepoint(tempw, wpx, yy, wpz, color_match);
            WPX[tt] = wpx;
            WPZ[tt] = wpz;

            if (tempw <= INITIALBLACKBODY) {
                for (int i = 0; i < N_c; i++) {
                    if (! wbpar.itcwb_sampling) {
                        spectrum_to_color_xyz_blackbody(spec_colorforxcyc[i], tempw, Refxyz[i].Xref, Refxyz[i].Yref, Refxyz[i].Zref, color_match);
                    } else {
                        spectrum_to_color_xyz_blackbody(spec_colorforxcyc_old[i], tempw, Refxyz[i].Xref, Refxyz[i].Yref, Refxyz[i].Zref, color_match);
                    }
                }
            } else {
                double x_DD;

                if (tempw <= 7000) {
                    x_DD = -4.6070e9 / (tempw * tempw * tempw) + 2.9678e6 / (tempw * tempw) + 0.09911e3 / tempw + 0.244063;
                } else {
                    x_DD = -2.0064e9 / (tempw * tempw * tempw) + 1.9018e6 / (tempw * tempw) + 0.24748e3 / tempw + 0.237040;
                }

                const double y_DD = -3.0 * x_DD * x_DD + 2.87 * x_DD - 0.275;
                //calculate D -daylight in function of s0, s1, s2 and temp ==> x_D y_D
                //S(lamda)=So(lambda)+m1*s1(lambda)+m2*s2(lambda)
                const double interm2 = (0.0241 + 0.2562 * x_DD - 0.734 * y_DD);
                const double m11 = (-1.3515 - 1.7703 * x_DD + 5.9114 * y_DD) / interm2;
                const double m22 = (0.03 - 31.4424 * x_DD + 30.0717 * y_DD) / interm2;

                for (int i = 0; i < N_c; i++) {
                    if (! wbpar.itcwb_sampling) {
                        spectrum_to_color_xyz_daylight(spec_colorforxcyc[i], m11, m22, Refxyz[i].Xref, Refxyz[i].Yref, Refxyz[i].Zref, color_match);
                    } else {
                        spectrum_to_color_xyz_daylight(spec_colorforxcyc_old[i], m11, m22, Refxyz[i].Xref, Refxyz[i].Yref, Refxyz[i].Zref, color_match);
                    }
                }
            }

            for (int i = 0; i < N_c; i++) {
                Tx[i][tt] = Refxyz[i].Xref;
                Ty[i][tt] = Refxyz[i].Yref;
                Tz[i][tt] = Refxyz[i].Zref;
            }
        }
    }
}


}

