# -*- coding: utf-8 -*-


# PyMeeus: Python module implementing astronomical algorithms.
# Copyright (C) 2018  Dagoberto Salazar
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.


from math import sin, cos, tan, acos, atan2, sqrt, radians, log10

from pymeeus.Angle import Angle
from pymeeus.Epoch import Epoch, JDE2000
from pymeeus.Interpolation import Interpolation
from pymeeus.Coordinates import (
    geometric_vsop_pos, apparent_vsop_pos, orbital_elements,
    nutation_longitude, true_obliquity, ecliptical2equatorial,
    passage_nodes_elliptic
)
from pymeeus.Earth import Earth
from pymeeus.Sun import Sun


"""
.. module:: Uranus
   :synopsis: Class to model Uranus planet
   :license: GNU Lesser General Public License v3 (LGPLv3)

.. moduleauthor:: Dagoberto Salazar
"""


VSOP87_L = [
    # L0
    [
        [548129294.299, 0.00000000000, 0.00000000000],
        [9260408.252, 0.89106421530, 74.78159856730],
        [1504247.826, 3.62719262195, 1.48447270830],
        [365981.718, 1.89962189068, 73.29712585900],
        [272328.132, 3.35823710524, 149.56319713460],
        [70328.499, 5.39254431993, 63.73589830340],
        [68892.609, 6.09292489045, 76.26607127560],
        [61998.592, 2.26952040469, 2.96894541660],
        [61950.714, 2.85098907565, 11.04570026390],
        [26468.869, 3.14152087888, 71.81265315070],
        [25710.505, 6.11379842935, 454.90936652730],
        [21078.897, 4.36059465144, 148.07872442630],
        [17818.665, 1.74436982544, 36.64856292950],
        [14613.471, 4.73732047977, 3.93215326310],
        [11162.535, 5.82681993692, 224.34479570190],
        [10997.934, 0.48865493179, 138.51749687070],
        [9527.487, 2.95516893093, 35.16409022120],
        [7545.543, 5.23626440666, 109.94568878850],
        [4220.170, 3.23328535514, 70.84944530420],
        [4051.850, 2.27754158724, 151.04766984290],
        [3354.607, 1.06549008887, 4.45341812490],
        [2926.671, 4.62903695486, 9.56122755560],
        [3490.352, 5.48305567292, 146.59425171800],
        [3144.093, 4.75199307603, 77.75054398390],
        [2922.410, 5.35236743380, 85.82729883120],
        [2272.790, 4.36600802756, 70.32818044240],
        [2051.209, 1.51773563459, 0.11187458460],
        [2148.599, 0.60745800902, 38.13303563780],
        [1991.726, 4.92437290826, 277.03499374140],
        [1376.208, 2.04281409054, 65.22037101170],
        [1666.910, 3.62744580852, 380.12776796000],
        [1284.183, 3.11346336879, 202.25339517410],
        [1150.416, 0.93344454002, 3.18139373770],
        [1533.223, 2.58593414266, 52.69019803950],
        [1281.641, 0.54269869505, 222.86032299360],
        [1372.100, 4.19641615561, 111.43016149680],
        [1220.998, 0.19901396193, 108.46121608020],
        [946.195, 1.19249463066, 127.47179660680],
        [1150.993, 4.17898207045, 33.67961751290],
        [1244.342, 0.91612680579, 2.44768055480],
        [1072.008, 0.23564502877, 62.25142559510],
        [1090.461, 1.77501638912, 12.53017297220],
        [707.875, 5.18285226584, 213.29909543800],
        [653.401, 0.96586909116, 78.71375183040],
        [627.562, 0.18210181975, 984.60033162190],
        [524.495, 2.01276706996, 299.12639426920],
        [559.370, 3.35776737704, 0.52126486180],
        [606.827, 5.43209728952, 529.69096509460],
        [404.891, 5.98689011389, 8.07675484730],
        [467.211, 0.41484068933, 145.10977900970],
        [471.288, 1.40664336447, 184.72728735580],
        [483.219, 2.10553990154, 0.96320784650],
        [395.614, 5.87039580949, 351.81659230870],
        [433.532, 5.52142978255, 183.24281464750],
        [309.885, 5.83301304674, 145.63104387150],
        [378.609, 2.34975805006, 56.62235130260],
        [398.996, 0.33810765436, 415.55249061210],
        [300.379, 5.64353974146, 22.09140052780],
        [249.229, 4.74617120584, 225.82926841020],
        [239.334, 2.35045874708, 137.03302416240],
        [294.172, 5.83916826225, 39.61750834610],
        [216.480, 4.77847481363, 340.77089204480],
        [251.792, 1.63696775578, 221.37585028530],
        [219.621, 1.92212987979, 67.66805156650],
        [201.963, 1.29693040865, 0.04818410980],
        [224.097, 0.51574863468, 84.34282612290],
        [216.549, 6.14211862702, 5.93789083320],
        [222.588, 2.84309380331, 0.26063243090],
        [207.828, 5.58020570040, 68.84370773410],
        [187.474, 1.31924326253, 0.16005869440],
        [158.028, 0.73811997211, 54.17467074780],
        [199.146, 0.95634155010, 152.53214255120],
        [168.648, 5.87874000882, 18.15924726470],
        [170.300, 3.67717520688, 5.41662597140],
        [193.652, 1.88800122606, 456.39383923560],
        [192.998, 0.91616058506, 453.42489381900],
        [181.934, 3.53624029238, 79.23501669220],
        [173.145, 1.53860728054, 160.60889739850],
        [164.588, 1.42379714838, 106.97674337190],
        [171.968, 5.67952685533, 219.89137757700],
        [162.792, 3.05029377666, 112.91463420510],
        [146.653, 1.26300172265, 59.80374504030],
        [139.453, 5.38597723400, 32.19514480460],
        [138.585, 4.25994786673, 909.81873305460],
        [143.058, 1.29995487555, 35.42472265210],
        [123.840, 1.37359990336, 7.11354700080],
        [104.414, 5.02820888813, 0.75075952540],
        [103.277, 0.68095301267, 14.97785352700],
        [94.741, 0.90674090409, 74.66972398270],
        [82.978, 2.92828718445, 265.98929347750],
        [110.163, 2.02685778976, 554.06998748280],
        [94.226, 3.94266328260, 74.89347315190],
        [79.858, 1.01446829180, 6.59228213900],
        [109.376, 5.70581833286, 77.96299230500],
        [85.876, 1.70649435603, 82.85835341460],
        [103.562, 1.45770270246, 24.37902238820],
        [74.667, 4.63177552576, 69.36497259590],
        [79.919, 3.00974084247, 297.64192156090],
        [84.502, 0.36887189574, 186.21176006410],
        [88.810, 0.52481330563, 181.75834193920],
        [70.303, 1.18986880009, 66.70484372000],
        [69.965, 0.87476081875, 305.34616939270],
        [69.927, 3.76102749315, 131.40394986990],
        [84.604, 5.88725183325, 256.53994050650],
        [74.341, 6.24271323846, 447.79581952650],
        [62.310, 0.16901376623, 479.28838891550],
        [72.726, 2.84892775693, 462.02291352810],
        [69.060, 4.43934854374, 39.35687591520],
        [76.568, 4.58721110340, 6.21977512350],
        [73.387, 4.27603448634, 87.31177153950],
        [55.307, 1.49636544147, 71.60020482960],
        [57.291, 1.63015165542, 143.62530630140],
        [61.661, 3.18604743524, 77.22927912210],
        [57.634, 3.67180685401, 51.20572533120],
        [50.289, 1.12279384633, 20.60692781950],
        [53.744, 5.51890986247, 128.95626931510],
        [57.894, 2.66877593418, 381.61224066830],
        [58.112, 1.58629352171, 60.76695288680],
        [45.382, 0.48053933052, 14.01464568050],
        [37.581, 6.06822931932, 211.81462272970],
        [38.640, 3.43597050177, 153.49535039770],
        [46.087, 4.36201639577, 75.74480641380],
        [40.088, 4.57333927519, 46.20979048510],
        [34.229, 2.93967782207, 140.00196957900],
        [38.669, 5.58941074168, 99.16062095550],
        [34.827, 1.02792863024, 203.73786788240],
        [40.024, 0.69889667397, 218.40690486870],
        [32.538, 4.21625657443, 200.76892246580],
        [31.865, 5.50961503408, 72.33391801250],
        [41.695, 3.82438031124, 81.00137369080],
        [34.795, 0.39363490236, 1.37259812370],
        [39.775, 6.05600836903, 293.18850343600],
        [27.577, 2.18261286374, 125.98732389850],
        [36.279, 1.66586085405, 258.02441321480],
        [35.442, 1.96652806541, 835.03713448730],
        [35.361, 3.72258690030, 692.58748435350],
        [27.323, 2.10164372072, 209.36694217490],
        [26.530, 4.48265986115, 373.90799283650],
        [34.472, 1.07907945481, 191.20769491020],
        [29.915, 3.87358632506, 259.50888592310],
        [26.233, 3.63172504384, 490.33408917940],
        [25.848, 0.54461409359, 41.64449777560],
        [26.989, 6.27711247734, 28.57180808220],
        [26.391, 5.81110061049, 75.30286342910],
        [34.227, 6.05617272657, 275.55052103310],
        [24.279, 3.18776564878, 81.37388070630],
        [29.937, 1.88789751816, 269.92144674060],
        [26.235, 6.20105251336, 134.58534360760],
        [22.754, 0.92919725789, 288.08069400530],
        [25.180, 5.42547381962, 116.42609634290],
        [22.715, 0.53098783687, 1514.29129671650],
        [26.485, 4.77176167929, 284.14854074220],
        [27.008, 4.75281624832, 41.10198105440],
        [21.972, 4.58613057386, 404.50679034820],
        [22.012, 1.84389287183, 617.80588578620],
        [24.694, 4.70875195490, 378.64329525170],
        [28.949, 0.17127584792, 528.20649238630],
        [20.492, 0.10285646641, 195.13984817330],
        [20.696, 5.62143477633, 55.65914345610],
        [25.843, 0.74627159338, 278.51946644970],
        [22.990, 3.58378694661, 1.59634729290],
        [21.843, 0.05733533568, 173.94221952280],
        [19.050, 2.30351091243, 5.10780943070],
        [20.675, 2.64113858585, 105.49227066360],
        [21.856, 5.87352402691, 45.57665103870],
        [21.120, 1.98081790016, 114.39910691340],
        [19.279, 2.84304025179, 159.12442469020],
        [19.061, 0.50598371738, 67.35923502580],
        [20.434, 3.77601951414, 135.54855145410],
        [17.326, 4.47793157645, 120.35824960600],
        [20.547, 0.88695598555, 255.05546779820],
        [19.320, 1.48569290504, 0.89377187730],
        [21.331, 2.74470023060, 28.31117565130],
        [17.582, 4.09139636700, 296.15744885260],
        [15.918, 3.94525074972, 17.52610781830],
        [15.562, 0.92748407689, 300.61086697750],
        [16.439, 0.30868798605, 30.71067209630],
        [15.237, 4.93048601827, 7.42236354150],
        [19.284, 6.21950083268, 329.83706636550],
        [13.860, 0.56255266406, 144.14657116320],
        [16.206, 2.30292598693, 344.70304530790],
        [16.041, 0.19723295436, 103.09277421860],
        [14.414, 2.57606243208, 230.56457082540],
        [16.789, 4.93540052916, 565.11568774670],
        [17.052, 1.81844925116, 294.67297614430],
        [16.766, 0.27542186330, 73.81839072080],
        [15.428, 1.91577056305, 96.87299909510],
        [15.718, 3.87095025861, 98.89998852460],
        [11.923, 6.17545505441, 44.72531777680],
        [12.407, 6.22419970167, 80.19822453870],
        [13.040, 1.99652993223, 27.08733537390],
        [13.229, 3.43782440072, 227.31374111850],
        [11.669, 4.31526860843, 426.59819087600],
        [14.378, 5.78353646474, 1059.38193018920],
        [15.879, 0.98454960055, 6208.29425142410],
        [11.158, 1.74417430690, 220.41264243880],
        [11.989, 5.84388657950, 13.33332212430],
        [11.386, 2.55925734515, 19.12245511120],
        [13.281, 5.39472153462, 391.17346822390],
        [12.295, 4.57340278496, 23.57587323610],
        [12.827, 1.77410269070, 180.27386923090],
        [11.651, 4.29138607818, 142.44965013380],
        [12.248, 2.44241346243, 100.38446123290],
        [12.421, 2.32591770919, 80.71948940050],
        [9.774, 0.39898140151, 7.86430652620],
        [13.172, 2.74099358938, 177.87437278590],
        [12.262, 5.42795591646, 831.10498122420],
        [10.272, 5.90194483926, 74.52096613640],
        [9.317, 3.75869700774, 74.82978267710],
        [10.701, 4.00709797731, 235.39049596580],
        [9.243, 5.38492199672, 20.44686912510],
        [9.461, 2.60126707172, 92.30770638560],
        [12.066, 5.52163100220, 74.26033370550],
        [10.836, 1.88393779293, 241.61027108930],
        [9.317, 1.16483658310, 74.73341445750],
        [10.718, 5.50310449842, 187.69623277240],
        [12.057, 6.02120050390, 154.01661525950],
        [9.124, 1.15458738606, 0.63313944640],
        [11.526, 6.26425302826, 155.78297225810],
        [12.200, 5.79400179483, 1364.72809958190],
        [10.979, 5.76614513865, 628.85158605010],
        [8.532, 5.18016456150, 1.64453140270],
        [8.660, 3.78133822411, 74.94165726170],
        [11.227, 1.30788626675, 604.47256366190],
        [10.531, 1.05867421534, 291.70403072770],
        [8.446, 3.52020067595, 756.32338265690],
        [10.291, 5.30493908317, 75.04223099820],
        [8.015, 6.24347048958, 543.02428721890],
        [7.796, 5.23497582886, 58.10682401090],
        [9.310, 1.65210713729, 24.11838995730],
        [8.642, 1.14285691458, 74.62153987290],
        [7.797, 4.00208030502, 31.49256938900],
        [8.915, 3.63129389881, 408.43894361130],
        [7.191, 4.24536221306, 110.20632121940],
        [9.764, 0.10205649809, 366.48562929500],
        [8.710, 2.25910759480, 451.94042111070],
        [9.430, 2.00492467431, 331.32153907380],
        [9.008, 1.63146330622, 443.86366626340],
        [7.247, 4.35313018726, 88.11492069160],
        [7.659, 0.61918159043, 50.40257617910],
        [6.836, 5.12190844483, 0.80314915210],
        [9.367, 1.42664537007, 414.06801790380],
        [8.136, 0.45279998999, 25.60286266560],
        [9.199, 2.41000352664, 10138.50394764370],
        [6.896, 5.85017813531, 339.28641933650],
        [6.487, 6.03397141885, 1.22384027740],
        [7.186, 4.00480285222, 157.63995198190],
        [8.130, 0.21907525983, 422.66603761290],
        [8.022, 2.09953974305, 92.94084583200],
        [6.371, 4.47820781123, 79.88940799800],
        [7.957, 5.86499639179, 760.25553592000],
        [8.383, 2.33782809093, 417.03696332040],
        [7.931, 3.41952210669, 7.70424783180],
        [6.613, 1.39197711439, 16.67477455640],
        [7.142, 5.57864813931, 4.73530241520],
        [7.441, 0.48121969777, 68.18931642830],
        [7.187, 0.50441238918, 457.87831194390],
        [6.613, 2.84628770892, 142.14083359310],
        [6.399, 3.88681409308, 74.03083904190],
        [6.140, 1.65776909220, 350.33211960040],
        [7.053, 0.13890020306, 306.83064210100],
        [6.010, 5.46434004640, 48.75804477640],
        [6.810, 6.15448079403, 67.88049988760],
        [5.982, 2.36098472874, 2.00573757010],
        [5.709, 1.49928444044, 206.18554843720],
        [6.843, 1.08172913275, 465.95506679120],
        [6.408, 5.07331258075, 4.66586644600],
        [7.805, 3.98866710061, 3.62333672240],
        [5.527, 5.57881556653, 2.92076130680],
        [5.521, 3.38225987040, 149.45132255000],
        [6.107, 1.95504762856, 216.92243216040],
        [5.692, 2.83076925167, 260.99335863140],
        [6.586, 2.71085048651, 329.72519178090],
        [5.826, 3.98800970226, 347.88443904560],
        [5.745, 0.49160564101, 0.37250701550],
        [5.473, 5.69158856279, 1.69692102940],
        [4.982, 2.40870746521, 342.25536475310],
        [6.040, 4.78944090986, 558.00214074590],
        [5.317, 2.78403764459, 13.49338081870],
        [5.090, 5.47747622578, 372.42352012820],
        [4.894, 1.77082918618, 333.65734504400],
        [5.389, 2.94076732149, 9.40116886120],
        [4.757, 5.37129102802, 61.28821774860],
        [5.000, 3.43988321744, 518.64526483070],
        [5.664, 3.30309284254, 0.65439130580],
        [5.530, 0.45092393824, 162.09337010680],
        [5.746, 3.45964923866, 55.13787859430],
        [5.666, 1.23578675332, 328.35259365720],
        [5.071, 5.42033080481, 977.48678462110],
        [5.719, 0.66718965817, 92.04707395470],
        [4.994, 1.29267872727, 983.11585891360],
        [5.570, 2.36255927193, 6.90109867970],
        [5.189, 2.40682220291, 58.31927233200],
        [5.502, 0.13301359232, 149.67507171920],
        [4.706, 1.85473330365, 119.50691634410],
        [4.305, 4.18171934306, 90.82323367730],
        [5.953, 1.73036741041, 152.74459087230],
        [5.607, 5.53187692339, 1087.69310584050],
        [5.591, 5.75072223569, 358.93013930950],
        [4.441, 0.97726075887, 4.19278569400],
        [4.608, 3.31800103668, 89.75945209430],
        [4.677, 5.01422713233, 43.12897048390],
        [4.034, 1.08242564328, 75.53235809270],
        [5.626, 1.10270225604, 66.91729204110],
        [4.058, 1.94012136050, 17.26547538740],
        [4.770, 2.24207019076, 986.08480433020],
        [5.207, 1.36600428910, 767.36908292080],
        [4.940, 0.13733547633, 0.85133326190],
        [4.339, 5.33814728291, 152.01087768940],
        [3.917, 3.85320550575, 2.28762186040],
        [3.903, 5.87573410158, 16.46232623530],
        [4.655, 0.93665017107, 267.47376618580],
        [4.638, 5.42566923517, 16.04163511000],
        [5.177, 3.40845690805, 1289.94650101460],
        [3.825, 0.59888105730, 210.33015002140],
        [5.048, 2.16732539242, 367.97010200330],
        [3.819, 1.70901925915, 5.62907429250],
        [5.004, 0.26759264038, 403.13419222450],
        [4.202, 5.12029089394, 19.01058052660],
        [4.472, 2.88978371811, 59.28248017850],
        [4.700, 4.17709035394, 130.44074202340],
        [4.487, 0.85521839581, 969.62247809490],
        [3.802, 4.59721371468, 25.86349509650],
        [4.024, 4.98868930941, 30.05628079050],
        [4.722, 6.16359211847, 173.68158709190],
        [4.732, 3.76697693308, 373.01422095920],
        [3.542, 0.76768843819, 114.13847448250],
        [3.370, 2.00021522907, 286.59622129700],
        [3.626, 3.20240733896, 991.71387862270],
        [4.028, 0.46802022168, 387.24131496080],
        [3.442, 5.20959733350, 894.84087952760],
        [3.273, 5.46374958434, 192.69216761850],
        [3.291, 3.97847646998, 264.50482076920],
        [3.291, 1.62538722379, 681.54178408960],
        [3.198, 3.96580804800, 146.38180339690],
        [4.016, 1.40657464840, 383.09671337660],
        [3.844, 3.60271287642, 0.59070083100],
        [3.123, 4.50215520755, 1439.50969814920],
        [3.434, 5.18704419009, 97.41551581630],
        [3.123, 4.14198630214, 76.47851959670],
        [3.259, 1.65410614252, 214.78356814630],
        [3.557, 4.15769848885, 68.56182344380],
        [2.934, 2.95575531139, 120.99138905240],
        [2.914, 3.69930995976, 874.39401040250],
        [3.301, 2.14570582133, 253.57099508990],
        [3.362, 4.82277888708, 19.64371997300],
        [3.218, 2.56428709831, 60.55450456570],
        [3.059, 3.57539890234, 117.91056905120],
        [3.073, 3.54739757836, 95.38852638680],
        [2.789, 0.65190913388, 42.53826965290],
        [3.235, 5.31608088666, 546.95644048200],
        [3.657, 5.89905956226, 16.15350969460],
        [3.549, 2.76314903735, 82.48584639910],
        [3.627, 4.68663059919, 593.42686339800],
        [3.306, 1.57486085317, 312.45971639350],
        [3.602, 2.51921910142, 22.89454967990],
        [3.431, 4.95532928836, 49.72125262290],
        [2.675, 2.41314606353, 29.20494752860],
        [3.101, 1.42849885249, 17.63798240290],
        [3.399, 3.02815712113, 88.79624424780],
        [3.379, 2.65894323745, 771.30123618390],
        [2.547, 6.10642153361, 455.87257437380],
        [2.967, 0.31461418738, 150.52640498110],
        [2.681, 1.16839594153, 477.80391620720],
        [3.343, 3.09880618811, 552.58551477450],
        [2.678, 1.42841144096, 1.11196569280],
        [2.540, 1.94053883528, 6.48040755440],
        [2.491, 2.97226347939, 453.94615868080],
        [2.744, 1.93313970916, 73.18525127440],
        [2.935, 4.59394106280, 167.08930495290],
        [3.007, 1.38745560615, 365.00115658670],
        [3.053, 1.91792962252, 561.18353448360],
        [2.496, 5.45540866674, 66.18357885820],
        [2.622, 1.46324659292, 33.13710079170],
        [3.203, 4.01683757076, 555.55446019110],
        [2.317, 1.13727677715, 43.24084506850],
        [2.341, 5.87635018071, 228.27694896500],
        [2.841, 3.60234459541, 42.58645376270],
        [2.858, 1.53714262537, 353.30106501700],
        [2.484, 3.85791009894, 104.00779795530],
        [2.903, 5.21967656512, 73.40900044360],
        [2.760, 1.21343315367, 32.24332891440],
        [2.269, 3.31411391807, 4.14460158420],
        [2.241, 2.65636547591, 70.11573212130],
        [2.246, 4.08175081363, 123.53964334370],
        [2.583, 2.38305971478, 100.64509366380],
        [2.761, 5.71758409791, 43.28902917830],
        [2.827, 6.16734582851, 101.86893394120],
        [2.838, 0.54888495490, 20.49505323490],
        [2.145, 4.04195315408, 47.06112374700],
        [2.810, 1.90169260186, 273.10284047830],
        [2.922, 2.79808700183, 418.26080359780],
        [2.070, 5.50402718290, 47.69426319340],
        [2.071, 2.01973573060, 316.39186965660],
        [2.520, 3.12740527423, 905.88657979150],
        [2.170, 4.46196560050, 2.70831298570],
        [2.399, 3.78849518316, 75.58474771940],
        [2.746, 4.82558024832, 6.85291456990],
        [2.717, 2.33108458294, 404.61866493280],
        [2.416, 4.11932546205, 332.80601178210],
        [1.974, 5.79881978458, 11.15757484850],
        [1.967, 5.43918682709, 199.28444975750],
        [2.282, 2.79530897096, 22.63391724900],
        [1.910, 0.27727117649, 69.15252427480],
        [2.471, 3.51033894778, 8.59801970910],
        [1.904, 3.45282024423, 472.17484191470],
        [2.606, 0.43601023323, 439.78275515400],
        [2.663, 5.43112910549, 3265.83082813250],
        [2.058, 1.69362390174, 65.87476231750],
        [2.258, 5.32927779367, 908.33426034630],
        [1.873, 5.55399805910, 175.16605980020],
        [2.222, 0.96973865202, 39.09624348430],
        [2.121, 2.00302088316, 106.01353552540],
        [2.010, 1.49945418027, 29.22619938800],
        [2.222, 4.36573603431, 468.24268865160],
        [2.055, 0.05798973044, 205.22234059070],
        [2.086, 0.44287700052, 10.29494073850],
        [1.770, 4.32048805830, 0.45757438700],
        [2.112, 5.78682409103, 486.40193591630],
        [1.909, 0.82888506421, 254.94359321360],
        [1.973, 6.05826379648, 78.40493528970],
        [1.908, 5.55892482384, 15.49911838880],
        [1.888, 6.20874408008, 198.32124191100],
        [2.092, 2.55561831566, 49.50880430180],
        [1.775, 6.17741514589, 258.87574647670],
        [2.146, 1.40508118810, 526.72201967800],
        [1.800, 0.04238718337, 334.29048449040],
        [1.738, 1.99152421966, 77.06922042770],
        [1.641, 3.36410541913, 118.02244363580],
        [2.084, 5.21275540105, 134.06407874580],
        [1.861, 2.97480744198, 178.78939652260],
        [1.750, 2.01731567093, 142.66209845490],
        [1.600, 1.60963172329, 40.16002506730],
        [1.668, 1.53361997245, 0.83008140250],
        [1.769, 4.72689497119, 32.71640966640],
        [1.782, 2.60469159465, 166.82867252200],
        [1.765, 5.57583636983, 522.57741809380],
        [1.877, 1.09044005603, 274.06604832480],
        [1.544, 1.91798419140, 303.86169668440],
        [1.532, 3.44973397383, 124.50285119020],
        [1.658, 5.23946791059, 233.90602325750],
        [1.743, 2.32369273283, 290.21955801940],
        [1.528, 0.62020771152, 1033.35837639830],
        [1.522, 6.11272567668, 165.60483224460],
        [1.496, 2.02190170195, 150.08446199640],
        [1.490, 3.30997217921, 820.05928096030],
        [1.398, 3.41567259878, 4.99593484610],
        [1.886, 6.25585539882, 162.89651925890],
        [1.388, 0.62745508416, 448.68959140380],
        [1.918, 0.91483173263, 1819.63746610920],
        [1.682, 2.11545135265, 189.72322220190],
        [1.711, 2.55731536599, 1108.13997496560],
        [1.598, 0.89607036108, 115.88357962170],
        [1.477, 0.22838214106, 370.93904741990],
        [1.727, 1.54005322759, 401.64971951620],
        [1.432, 3.73381952953, 8.90683624980],
        [1.338, 0.19338311739, 81.89514556810],
        [1.618, 6.02595306259, 31.23193695810],
        [1.452, 0.11827434627, 72.77586099720],
        [1.626, 4.26332651029, 369.45457471160],
        [1.284, 3.18389639039, 362.86229257260],
        [1.755, 5.57436830525, 344.96367773880],
        [1.451, 4.42615381750, 189.18070548070],
        [1.294, 2.77775613125, 63.62402371880],
        [1.712, 2.16785817753, 536.80451209540],
        [1.266, 2.78408709208, 55.77101804070],
        [1.700, 2.49908604932, 441.26722786230],
        [1.645, 2.41257585783, 10.08249241740],
        [1.666, 5.00664377609, 79.51690098250],
        [1.674, 0.00394784675, 491.55792945680],
        [1.334, 5.62972572008, 129.91947716160],
        [1.328, 0.25606135840, 114.94162363460],
        [1.176, 1.63208422172, 84.18276742850],
        [1.177, 4.10449248614, 103.35340664950],
        [1.384, 5.73004529870, 89.33876096900],
        [1.287, 0.99537181009, 14.66903698630],
        [1.171, 5.98078310685, 57.14361616440],
        [1.126, 3.39586759408, 375.39246554480],
        [1.406, 5.18477940039, 113.87784205160],
        [1.397, 1.49751443233, 14.22709400160],
        [1.206, 3.60301196272, 480.77286162380],
        [1.113, 4.80418427391, 9.44935297100],
        [1.434, 1.57158177893, 419.74527630610],
        [1.473, 1.03881736383, 1215.16490244730],
        [1.311, 2.99704179684, 458.84151979040],
        [1.453, 6.10676427884, 64.69910614990],
        [1.070, 3.16542344402, 54.33472944220],
        [1.389, 2.78512263875, 26.02355379090],
        [1.203, 0.20627563214, 0.56944897160],
        [1.343, 5.58004577468, 95.22846769240],
        [1.062, 2.40616687148, 154.97982310600],
        [1.399, 1.66776602336, 240.38643081190],
        [1.036, 5.53891715915, 403.02231763990],
        [1.269, 2.37684527290, 37.87240320690],
        [1.197, 4.87553746725, 1044.40407666220],
        [1.009, 2.74619462960, 80.41067285980],
        [1.330, 0.99603813295, 483.22054217860],
        [1.348, 0.58829539202, 476.43131808350],
        [0.989, 3.31666847329, 18.91000679010],
        [1.054, 2.85972567059, 616.32141307790],
        [1.276, 4.72938141859, 691.10301164520],
        [1.219, 3.62909689220, 106.27416795630],
        [1.269, 1.53301050628, 280.96714700450],
        [0.968, 2.73688433893, 218.92816973050],
        [1.330, 5.69234088687, 694.07195706180],
        [1.121, 2.80542439790, 148.59998928810],
        [0.980, 6.04026702553, 5.46901559810],
        [1.235, 0.61136787453, 237.67811782620],
        [1.161, 5.32209024820, 369.08206769610],
        [0.944, 1.14261861393, 384.05992122310],
        [1.017, 2.49896409345, 147.11551657980],
        [1.082, 2.46236323548, 326.86812094890],
        [1.037, 5.92126063748, 4.82592514040],
        [1.232, 5.83725190224, 63.84777288800],
        [0.914, 0.36627060914, 10.93382567930],
        [1.165, 5.32140830393, 308.31511480930],
        [0.952, 4.78982033367, 93.90405367850],
        [0.993, 3.39918521663, 10.78506783300],
        [1.206, 3.64530500530, 699.70103135430],
        [0.893, 2.97140509591, 248.46318565920],
        [0.883, 2.85605198817, 15.19030184810],
        [0.875, 3.43925709280, 3.08082000120],
        [1.103, 4.80576195766, 6133.51265285680],
        [1.178, 6.01576659565, 377.15882254340],
        [0.974, 2.51206761828, 121.84272231430],
        [0.865, 5.68958102479, 141.69889060840],
        [0.974, 3.15729174941, 215.43795945210],
        [0.847, 0.84854843713, 2043.98226181110],
        [0.961, 0.22181374419, 0.91502373670],
        [1.167, 2.37544946421, 33.94024994380],
        [1.013, 3.43778868786, 36.90919536040],
        [0.838, 1.63355479706, 2.33580597020],
        [1.113, 2.50989694970, 405.99126305650],
        [0.987, 1.14030030863, 82.20396210880],
        [1.060, 1.70915765427, 438.29828244570],
        [0.829, 2.97491446672, 62.77269045690],
        [0.991, 4.44869793177, 406.10313764110],
        [0.952, 6.13897716036, 184.98791978670],
        [1.033, 4.19932839584, 141.48644228730],
        [1.029, 5.13205530996, 157.26744496640],
        [0.805, 3.11000318272, 93.79217909390],
        [0.865, 4.44578048207, 295.19424100610],
        [0.921, 4.88190545687, 12.00890811040],
        [0.949, 5.99910796869, 606.76018552230],
        [0.991, 1.68012021428, 40.58071619260],
        [0.854, 0.37823682862, 217.23124870110],
        [0.883, 2.36224385140, 3.77209456870],
        [1.016, 3.90745585959, 194.28851491140],
        [0.807, 5.93051451738, 302.09533968580],
        [0.879, 0.52695940866, 1057.89745748090],
        [1.009, 1.19621149495, 490.07345674850],
        [0.801, 4.96724351781, 661.09491496450],
        [0.843, 0.97496000705, 73.88782669000],
        [1.028, 2.65189503651, 477.91579079180],
        [0.846, 3.36343733960, 40.84134862350],
        [0.772, 4.93551925711, 425.11371816770],
        [0.781, 0.59382881638, 97.67614824720],
        [0.770, 4.25621058806, 488.84961647110],
        [0.935, 1.24971148781, 624.91943278700],
        [0.984, 4.44298060183, 171.65459766240],
        [0.806, 0.09410536829, 440.68227252570],
        [0.769, 4.09296452529, 140.65636088480],
        [0.802, 0.78515729603, 11.84884941600],
        [1.007, 4.06909438635, 76.15419669100],
        [0.935, 0.03808890956, 156.15547927360],
        [0.712, 5.35549696452, 610.69233878540],
        [0.911, 4.70177653335, 81.68269724700],
        [0.926, 3.09765550633, 833.55266177900],
        [0.812, 3.54377377085, 149.40313844020],
        [0.801, 5.48034970408, 21.97952594320],
        [0.900, 4.21739347020, 778.41478318470],
        [0.689, 2.56862945159, 109.31254934210],
        [0.685, 3.19965828980, 31.65262808340],
        [0.952, 4.82579978820, 1744.85586754190],
        [0.724, 2.16878848875, 1171.87587326900],
        [0.897, 3.94746491183, 75.67537044460],
        [0.739, 5.45802693622, 252.65597135320],
        [0.821, 4.26046087515, 1246.65747183630],
        [0.663, 5.83767831921, 86.63044798330],
        [0.664, 1.72432848951, 216.48048917570],
        [0.721, 3.98089320988, 902.70518605380],
        [0.663, 4.98388191647, 958.57677783100],
        [0.750, 2.66119349235, 363.51668387840],
        [0.828, 3.62849315181, 14.81779483260],
        [0.663, 3.23521229496, 207.88246946660],
        [0.681, 3.29667467046, 25.06034594440],
        [0.887, 4.81199147907, 155.50108796780],
        [0.646, 2.65129054432, 685.47393735270],
        [0.808, 5.68764955594, 280.00393915800],
        [0.619, 1.32296886424, 193.65537546500],
        [0.621, 2.57837200005, 703.63318461740],
        [0.854, 3.63173686962, 411.62033734900],
        [0.728, 1.05845783695, 916.93228005540],
        [0.733, 0.75817076935, 44.09217833040],
        [0.629, 1.52747752455, 397.39324334740],
        [0.599, 5.30671888409, 180.16199464630],
        [0.615, 5.68908944293, 25.27279426550],
        [0.709, 0.64922689354, 14.55716240170],
        [0.729, 4.79389069212, 479.40026350010],
        [0.667, 1.98320895029, 37.61177077600],
        [0.586, 1.51157853976, 668.20846196530],
        [0.639, 1.69085181319, 262.47783133970],
        [0.616, 4.62035066985, 12.26954054130],
        [0.687, 2.24368916079, 228.79821382680],
        [0.599, 2.08317681783, 149.30256470370],
        [0.597, 3.14660293947, 137.55428902420],
        [0.581, 2.69049614736, 823.99143422340],
        [0.709, 4.38514100216, 184.09414790940],
        [0.671, 3.46925958949, 105.38039607900],
        [0.619, 2.91325544152, 236.87496867410],
        [0.558, 2.96177194880, 34.20088237470],
        [0.648, 5.56457931302, 140.96517742550],
        [0.581, 0.55427680962, 331.20966448920],
        [0.585, 0.15548306049, 232.42155054920],
        [0.548, 3.57525860446, 497.44763618020],
        [0.574, 5.60908001848, 118.87377689770],
        [0.702, 1.74156189506, 149.04193227280],
        [0.543, 0.49890445043, 133.10087089930],
        [0.716, 3.04149734724, 131.92521473170],
        [0.544, 6.22369103738, 149.61138124440],
        [0.539, 0.74276113752, 911.30320576290],
        [0.614, 5.83710659138, 181.05576652360],
        [0.601, 0.30768922616, 407.58761034940],
        [0.635, 4.03476459045, 136.06981631590],
        [0.526, 5.42874995984, 450.97721326420],
        [0.547, 3.07676037032, 204.70107572890],
        [0.622, 3.08666523105, 268.43697403230],
        [0.537, 4.25467814241, 217.44369702220],
        [0.566, 4.84686444604, 842.15068148810],
        [0.636, 2.57425168783, 621.73803904930],
        [0.623, 5.84341335807, 52.80207262410],
        [0.544, 3.62983006500, 149.51501302480],
        [0.578, 0.34796271917, 139.48070471720],
        [0.537, 5.99181083831, 246.97871295090],
        [0.674, 2.57972741298, 602.98809095360],
        [0.539, 6.19662961610, 696.51963761660],
        [0.516, 2.19916575703, 458.09076026500],
        [0.632, 5.26658553640, 67.07735073550],
        [0.581, 0.05320320337, 95.93104310800],
        [0.504, 6.24600928623, 149.72325582900],
        [0.638, 6.23121553223, 10063.72234907640],
        [0.528, 0.20662780149, 310.17209453310],
        [0.537, 2.96207822442, 73.13706716460],
        [0.508, 5.29969144068, 335.77495719870],
        [0.487, 2.83772541949, 143.93412284210],
        [0.537, 3.36808372143, 252.08652238160],
        [0.632, 5.88494938125, 920.86443331850],
        [0.523, 6.13183488285, 1589.07289528380],
        [0.579, 0.04597861846, 563.63121503840],
        [0.613, 0.34938781762, 343.47920503050],
        [0.495, 2.89212499810, 61.44827644300],
        [0.623, 0.71740350315, 513.07988101300],
        [0.498, 2.53375871592, 41.75637236020],
        [0.564, 2.01612524784, 449.28029223480],
        [0.480, 0.10535023009, 69.67378913660],
        [0.494, 3.25187012728, 428.08266358430],
        [0.536, 5.81149025999, 282.66406803390],
        [0.468, 0.89483830828, 541.53981451060],
        [0.533, 2.44239677121, 393.46109008430],
        [0.589, 6.24067076234, 29.79564835960],
        [0.465, 0.25006743710, 57.25549074900],
        [0.622, 3.89339038121, 416.77633088950],
        [0.519, 2.72375973888, 469.13646052890],
        [0.498, 1.76422801185, 380.38840039090],
        [0.596, 0.83642843095, 98.35747180340],
        [0.459, 0.39052216206, 197.79997704920],
        [0.574, 2.86366933069, 170.76082578510],
        [0.518, 3.38058345605, 535.91074021810],
        [0.564, 1.20395155832, 832.58945393250],
        [0.556, 4.39974034374, 196.62432088160],
        [0.537, 3.93637064940, 460.53844081980],
        [0.482, 2.41562148830, 827.92358748650],
        [0.578, 2.39644032176, 1670.07426897460],
        [0.475, 4.19223519775, 271.40591944890],
        [0.452, 3.99146251480, 135.33610313300],
        [0.514, 6.11377193423, 1894.41906467650],
        [0.531, 3.45607724228, 450.45594840240],
        [0.492, 5.87591888758, 170.17012495410],
        [0.588, 2.66953705406, 310.97524368520],
        [0.564, 1.04491117370, 446.31134681820],
        [0.440, 5.85084571537, 224.23292111730],
        [0.441, 2.19799439287, 119.39504175950],
        [0.549, 6.05651523611, 76.37794586020],
        [0.573, 2.29898526750, 122.47586176070],
        [0.473, 4.61187869812, 291.26208774300],
        [0.553, 4.90464199013, 463.50738623640],
        [0.599, 1.48666209087, 149.82382956550],
        [0.499, 3.72896978991, 853.19638175200],
        [0.440, 5.05024690419, 79.44746501330],
        [0.558, 0.42332722744, 283.62727588040],
        [0.458, 4.49655973916, 754.83890994860],
        [0.449, 1.32861330901, 308.68762182480],
        [0.565, 0.45628024105, 241.87090352020],
        [0.510, 3.70202346104, 452.46168597250],
        [0.404, 0.12335821240, 1097.09427470170],
        [0.409, 4.02092464698, 735.87651353180],
        [0.464, 3.82915608692, 1094.54602041040],
        [0.439, 2.72266354653, 376.19561469690],
        [0.478, 0.23380952322, 1182.92157353290],
        [0.412, 1.21971515436, 419.48464387520],
        [0.546, 0.22296640745, 829.62050851590],
        [0.510, 2.69499052512, 412.58354519550],
        [0.409, 2.51935747849, 409.07208305770],
        [0.451, 0.56137272347, 758.77106321170],
        [0.399, 0.86037315330, 337.80194662820],
        [0.393, 0.40202463200, 107.49800823370],
        [0.494, 5.02745190154, 619.29035849450],
        [0.404, 0.08539758465, 18.96239641680],
        [0.504, 4.18251931021, 449.49274055590],
        [0.406, 3.80082989682, 34.53095077480],
        [0.545, 2.80919248176, 514.56435372130],
        [0.390, 1.65941826256, 447.20511869550],
        [0.448, 2.81540452771, 400.16524680790],
        [0.405, 4.86073222353, 1404.08497549710],
        [0.476, 1.61050626902, 54.28654533240],
        [0.406, 1.29798079034, 226.79247625670],
        [0.526, 5.35780726572, 838.21852822500],
        [0.403, 2.75405589772, 285.11174858870],
        [0.381, 4.97702366598, 309.27832265580],
        [0.448, 1.37926537411, 745.27768239300],
        [0.419, 0.90546724862, 451.72797278960],
        [0.450, 1.92391706975, 474.94684537520],
        [0.474, 5.54351717465, 494.26624244250],
        [0.460, 5.09575399931, 289.56516671360],
        [0.455, 3.14755087330, 168.57377766120],
        [0.372, 4.87645271422, 116.53797092750],
        [0.479, 1.08512503555, 154.29849954980],
        [0.374, 0.16389070181, 1190.78588005910],
        [0.422, 3.51871257208, 706.81457835510],
        [0.354, 4.06405413226, 124.29040286910],
        [0.494, 4.57924296149, 167.72244439930],
        [0.357, 5.78050145791, 1265.56747862640],
        [0.402, 2.67652703260, 464.47059408290],
        [0.370, 2.21677703856, 232.04904353370],
        [0.462, 4.10424305270, 27.72047482030],
        [0.364, 3.07518732480, 442.63982598600],
        [0.403, 1.69214233165, 90.28071695610],
        [0.348, 1.16647947937, 357.44566660120],
        [0.443, 1.32306861852, 298.23262239190],
        [0.386, 4.34980428548, 227.52618943960],
        [0.337, 5.92030047826, 445.34813897170],
        [0.356, 2.38824200660, 511.59540830470],
        [0.357, 3.19265737844, 21.19762865050],
        [0.406, 3.72223708907, 3116.26763099790],
        [0.341, 3.05116722794, 15.78100267910],
        [0.379, 4.72472516443, 30.59879751170],
        [0.400, 0.71868453904, 836.52160719560],
        [0.370, 0.17259001853, 6531.66165626500],
        [0.456, 0.83408547295, 674.80074410430],
        [0.376, 5.91068811321, 1617.38407093510],
        [0.392, 1.90856045571, 25863.55834587229],
        [0.333, 4.77074940789, 76.78733613740],
        [0.332, 0.85699402720, 749.20983565610],
        [0.321, 3.88221470645, 38.60611638980],
        [0.321, 2.59404134515, 1300.99220127850],
        [0.438, 2.60178805278, 224.45667028650],
        [0.371, 3.73501205989, 328.24071907260],
        [0.357, 0.03656571669, 148.81243760920],
        [0.389, 3.06990362181, 1012.91150727320],
        [0.392, 3.14428599675, 125.18417474640],
        [0.326, 2.23565995627, 89.59939339990],
        [0.314, 4.56810921721, 1681.11996923850],
        [0.333, 4.37613329736, 147.96684984170],
        [0.326, 4.15448016748, 21.14944454070],
        [0.371, 4.09066371754, 239.16259053450],
        [0.372, 1.12230345314, 321.76031151820],
        [0.305, 6.12924444546, 19.97378837310],
        [0.407, 3.65906570714, 679.25416222920],
        [0.406, 3.58469900333, 26013.12154300690],
        [0.310, 3.92339533494, 229.08009811710],
        [0.321, 1.35118535306, 172.45774681450],
        [0.332, 3.54513021513, 288.73508531110],
        [0.312, 2.87878773413, 806.72595883600],
        [0.372, 1.99953045718, 192.80404220310],
        [0.302, 0.01867543539, 501.37978944330],
        [0.299, 3.96468960950, 20277.00789528740],
        [0.308, 2.66235795286, 248.72381809010],
        [0.298, 3.52867456736, 21.45826108140],
        [0.370, 3.51061046963, 91.45637312370],
        [0.297, 0.79872983355, 742.99006053260],
        [0.414, 4.81163687199, 589.49471013490],
        [0.302, 5.26079338050, 27.56041612590],
        [0.314, 3.14643487607, 361.37781986430],
        [0.356, 4.63707521448, 442.75170057060],
        [0.357, 3.44793069844, 44.61344319220],
        [0.370, 4.25032151516, 304.23420369990],
        [0.366, 5.43115395433, 625.99451521810],
        [0.365, 0.30454498410, 6283.07584999140],
        [0.322, 2.32892248876, 229.45260513260],
        [0.369, 3.28573365074, 104.52906281710],
        [0.327, 2.98588869318, 348.84764689210],
        [0.315, 3.66842071994, 230.93707784090],
        [0.356, 0.90433599977, 29.74746424980],
        [0.384, 0.91820739126, 549.72844394250],
        [0.317, 3.86462587284, 639.89728631400],
        [0.333, 4.95319798125, 881.50755740330],
        [0.285, 4.09883967296, 904.18965876210],
        [0.338, 5.65177488491, 195.77298761970],
        [0.365, 0.57418860616, 285.63301345050],
        [0.363, 0.98999016221, 839.70300093330],
        [0.330, 2.26308969695, 49.17873590170],
        [0.335, 2.25619157817, 272.58157561650],
        [0.376, 5.87496858487, 268.95823889410],
        [0.362, 4.94491380965, 572.22923474750],
        [0.320, 5.58342880588, 459.36278465220],
        [0.299, 2.86286938521, 883.79517926370],
        [0.286, 2.49409963193, 156.04360468900],
        [0.272, 3.62976505444, 754.03576079650],
        [0.349, 2.34615857088, 6069.77675455340],
        [0.279, 4.04872155075, 180.79513409270],
        [0.328, 1.21350330743, 148.19059901090],
        [0.315, 0.74270298817, 320.27583880990],
        [0.324, 5.54296698387, 1507.17774971570],
        [0.266, 4.36134021576, 1310.39337013970],
        [0.262, 5.84934968714, 450.17406411210],
        [0.280, 5.05848320657, 102.52332524700],
        [0.347, 4.70068639620, 282.14280317210],
        [0.342, 5.47365149093, 163.57784281510],
        [0.364, 3.29301824378, 378.90392768260],
        [0.330, 4.63426494882, 341.99473232220],
        [0.259, 2.43682741156, 170.01006625970],
        [0.336, 3.79029047358, 9999.98645077300],
        [0.284, 2.52583672467, 266.10116806210],
        [0.281, 3.93593342516, 194.17664032680],
        [0.297, 0.18595848541, 491.81856188770],
        [0.290, 4.49575150721, 151.85081899500],
        [0.284, 1.59495665161, 336.83873878170],
        [0.255, 5.85817353877, 229.34073054800],
        [0.260, 5.92834225312, 455.06942522170],
        [0.274, 0.53977064975, 380.23964254460],
        [0.259, 3.15728797958, 454.74930783290],
        [0.303, 0.17851964142, 384.58118608490],
        [0.285, 1.29732672572, 25.12978191360],
        [0.273, 4.18776699292, 177.30492381430],
        [0.326, 1.68159391466, 161.72086309130],
        [0.321, 2.36931686576, 2274.54683263650],
        [0.241, 3.57660492473, 150.31395666000],
        [0.273, 3.78780099400, 380.01589337540],
        [0.265, 3.02540120552, 454.79749194270],
        [0.265, 6.06087280189, 455.02124111190],
        [0.279, 3.82392760479, 31.54075349880],
        [0.273, 2.64820862667, 838.00607990390],
        [0.238, 5.17487170793, 263.02034806090],
        [0.299, 3.94906046599, 531.97858695500],
        [0.317, 2.28289083599, 44.07092647100],
        [0.250, 1.35950829789, 304.12232911530],
        [0.312, 2.73327875294, 442.37919355510],
        [0.314, 3.86400459047, 734.45573129830],
        [0.251, 0.15148137746, 221.16340196420],
        [0.272, 5.71864670101, 164.54105066160],
        [0.305, 4.96642198943, 1140.38330388000],
        [0.282, 5.46073788901, 550.13783421970],
        [0.273, 5.68721459468, 92.41958097020],
        [0.256, 0.88787970870, 418.52143602870],
        [0.229, 5.02021405557, 144.89733068860],
        [0.231, 5.70236752870, 132.88842257820],
        [0.218, 2.05623736180, 303.05854753230],
        [0.300, 1.76109447754, 371.52974825090],
        [0.216, 2.97122807313, 176.65053250850],
        [0.224, 3.12798198868, 188.92007304980],
        [0.212, 1.30757526083, 74.14845912090],
        [0.238, 4.65119406609, 385.75684225250],
        [0.216, 0.07086910120, 893.35640681930],
        [0.262, 5.78959872639, 635.96513305090],
        [0.250, 4.47327859711, 551.10104206620],
        [0.222, 1.64692618955, 76.42612997000],
        [0.254, 2.61838408005, 525.23754696970],
        [0.284, 4.13290731223, 544.50875992720],
        [0.262, 2.82476092056, 971.10695080320],
        [0.266, 4.12467258610, 375.67434983510],
        [0.212, 3.61675003296, 75.41473801370],
        [0.253, 3.14224867483, 270.18207917150],
        [0.250, 4.32883971376, 346.44815044710],
        [0.273, 1.95676918609, 402.21916848780],
        [0.259, 1.96371242284, 968.13800538660],
        [0.238, 1.29663338057, 421.18156490460],
        [0.243, 5.94961177434, 117.36805233000],
        [0.241, 4.70849619029, 406.95447090300],
        [0.216, 3.31232425021, 190.66517818900],
        [0.201, 1.23733749784, 799.61241183520],
        [0.223, 0.98087204684, 627.36711334180],
        [0.226, 1.66139333004, 1366.21257229020],
        [0.205, 0.33839683950, 143.34342201110],
        [0.240, 0.71559872262, 525.75881183150],
        [0.223, 2.57722930616, 981.63138620530],
        [0.193, 4.48974066435, 172.19711438360],
        [0.221, 2.88670838151, 238.90195810360],
        [0.177, 3.87122035013, 389.68899551560],
        [0.175, 5.85737374379, 170.71264167530],
        [0.233, 0.63169996424, 980.66817835880],
        [0.225, 4.81504648561, 88.27497938600],
        [0.210, 2.49819501106, 128.43500445330],
        [0.172, 0.91255921858, 210.85141488320],
        [0.176, 3.85133296117, 605.95703637020],
        [0.211, 1.72999855867, 10213.28554621100],
        [0.170, 1.61405340100, 1512.80682400820],
        [0.216, 1.54874566441, 1060.86640289750],
        [0.194, 6.07381783144, 520.12973753900],
        [0.170, 2.58526747515, 1515.77576942480],
        [0.172, 1.45073604378, 995.64603188580],
        [0.160, 4.20015913513, 433.71173787680],
        [0.219, 1.63985385568, 630.33605875840],
        [0.219, 1.96273394194, 313.68355667090],
        [0.159, 3.62343846627, 73.97844941520],
        [0.206, 0.83764718449, 104.83787935780],
        [0.203, 0.68701289007, 1363.24362687360],
        [0.158, 0.81666479221, 987.56927703850],
        [0.172, 2.97938676702, 327.43756992050],
        [0.147, 2.15517092627, 73.24894174920],
        [0.156, 1.29643469200, 216.26804085460],
        [0.153, 3.77492020946, 768.85355562910],
        [0.147, 4.74903152642, 73.34530996880],
        [0.198, 2.80749830888, 225.30800354840],
        [0.179, 0.35410767087, 421.22974901440],
        [0.204, 3.47423038287, 564.85505531580],
        [0.172, 2.88420642820, 233.53351624200],
        [0.195, 3.09114364733, 294.30046912880],
        [0.181, 4.75684139861, 71.15826184490],
        [0.160, 3.16134209902, 70.04629615210],
        [0.155, 2.07144269719, 91.24392480260],
        [0.154, 3.37462128317, 138.40562228610],
        [0.192, 6.19629054455, 312.19908396260],
        [0.162, 0.60896273267, 73.03649342810],
        [0.172, 2.01712700688, 973.55463135800],
        [0.145, 1.72582731815, 302.37722397610],
        [0.170, 2.97437423989, 3191.04922956520],
        [0.153, 0.11158566352, 138.62937145530],
        [0.164, 5.37129563471, 457.35704708210],
        [0.132, 3.15466226029, 523.47118997110],
        [0.137, 2.88313323946, 765.88461021250],
        [0.128, 1.75773421717, 77.70235987410],
        [0.173, 5.03552846066, 415.29185818120],
        [0.137, 4.77137510538, 73.45718455340],
        [0.163, 0.99316178485, 94.42531854030],
        [0.138, 4.68330115148, 517.16079212240],
        [0.133, 2.95376828791, 75.15410558280],
        [0.134, 1.35779558361, 249.94765836750],
        [0.168, 4.73164542970, 108.72184851110],
        [0.119, 1.48229220689, 237.41748539530],
        [0.134, 1.52277177920, 154.67100656530],
        [0.117, 5.48642273610, 437.64389113990],
        [0.131, 5.76525548944, 75.43598987310],
        [0.127, 2.38309227230, 208.84567731310],
        [0.115, 4.38951350436, 343.21857259960],
        [0.114, 4.48142283161, 224.86606056370],
        [0.117, 5.37802827323, 293.70976829780],
        [0.136, 2.80772094137, 374.49869366750],
    ],
    # L1
    [
        [7502543121.646, 0.00000000000, 0.00000000000],
        [154458.244, 5.24201658072, 74.78159856730],
        [24456.413, 1.71255705309, 1.48447270830],
        [9257.828, 0.42844639064, 11.04570026390],
        [8265.977, 1.50220035110, 63.73589830340],
        [7841.715, 1.31983607251, 149.56319713460],
        [3899.105, 0.46483574024, 3.93215326310],
        [2283.777, 4.17367533997, 76.26607127560],
        [1926.600, 0.53013080152, 2.96894541660],
        [1232.727, 1.58634458237, 70.84944530420],
        [791.206, 5.43641224143, 3.18139373770],
        [766.954, 1.99555409575, 73.29712585900],
        [481.671, 2.98401996914, 85.82729883120],
        [449.798, 4.13826237508, 138.51749687070],
        [445.600, 3.72300400331, 224.34479570190],
        [426.554, 4.73126059388, 71.81265315070],
        [347.735, 2.45372261286, 9.56122755560],
        [353.752, 2.58324496886, 148.07872442630],
        [317.084, 5.57855232072, 52.69019803950],
        [179.920, 5.68367730922, 12.53017297220],
        [171.084, 3.00060075287, 78.71375183040],
        [205.585, 2.36263144251, 2.44768055480],
        [158.029, 2.90931969498, 0.96320784650],
        [189.068, 4.20242881378, 56.62235130260],
        [154.670, 5.59083925605, 4.45341812490],
        [183.762, 0.28371004654, 151.04766984290],
        [143.464, 2.59049246726, 62.25142559510],
        [151.984, 2.94217326890, 77.75054398390],
        [153.515, 4.65186885939, 35.16409022120],
        [121.452, 4.14839204920, 127.47179660680],
        [115.546, 3.73224603791, 65.22037101170],
        [102.022, 4.18754517993, 145.63104387150],
        [101.718, 6.03385875009, 0.11187458460],
        [88.202, 3.99035787994, 18.15924726470],
        [87.549, 6.15520787584, 202.25339517410],
        [80.530, 2.64124743934, 22.09140052780],
        [72.047, 6.04545933578, 70.32818044240],
        [68.570, 4.05071895264, 77.96299230500],
        [59.173, 3.70413919082, 67.66805156650],
        [47.267, 3.54312460519, 351.81659230870],
        [42.534, 5.72357370899, 5.41662597140],
        [44.339, 5.90865821911, 7.11354700080],
        [35.605, 3.29197259183, 8.07675484730],
        [35.524, 3.32784616138, 71.60020482960],
        [36.116, 5.89964278801, 33.67961751290],
        [30.608, 5.46414592601, 160.60889739850],
        [31.454, 5.62015632303, 984.60033162190],
        [38.544, 4.91519003848, 222.86032299360],
        [34.996, 5.08034112149, 38.13303563780],
        [30.811, 5.49591403863, 59.80374504030],
        [28.947, 4.51867390414, 84.34282612290],
        [26.627, 5.54127301037, 131.40394986990],
        [29.866, 1.65980844667, 447.79581952650],
        [29.206, 1.14722640419, 462.02291352810],
        [25.753, 4.99362028417, 137.03302416240],
        [25.373, 5.73584678604, 380.12776796000],
        [21.672, 2.80556379586, 69.36497259590],
        [26.605, 6.14640604128, 299.12639426920],
        [22.995, 2.24925345862, 111.43016149680],
        [19.246, 3.55645739672, 54.17467074780],
        [21.780, 0.93285892393, 213.29909543800],
        [19.338, 1.86249384092, 108.46121608020],
        [16.153, 3.10208165842, 14.97785352700],
        [13.126, 1.95385539499, 87.31177153950],
        [13.907, 1.54149045800, 340.77089204480],
        [13.549, 4.38455126720, 5.93789083320],
        [13.102, 5.88301410143, 6.21977512350],
        [11.810, 0.32615567587, 35.42472265210],
        [10.980, 1.69230280951, 45.57665103870],
        [12.351, 0.32823896833, 51.20572533120],
        [10.906, 5.97068444790, 265.98929347750],
        [11.446, 3.37831545858, 72.33391801250],
        [12.013, 3.60395709253, 269.92144674060],
        [11.662, 1.74504271366, 79.23501669220],
        [13.777, 2.69028726334, 225.82926841020],
        [12.006, 5.34430562395, 152.53214255120],
        [9.866, 5.50316093605, 153.49535039770],
        [10.436, 4.16875643286, 24.37902238820],
        [10.632, 3.06875158069, 284.14854074220],
        [9.613, 0.49590148788, 209.36694217490],
        [9.283, 3.54479191952, 41.64449777560],
        [9.536, 5.60054956443, 82.85835341460],
        [9.740, 1.01087744586, 68.84370773410],
        [9.187, 4.49472579228, 20.60692781950],
        [10.159, 3.51765739489, 529.69096509460],
        [8.612, 3.88869873588, 60.76695288680],
        [10.030, 4.64790204580, 77.22927912210],
        [8.689, 1.96813580258, 195.13984817330],
        [8.370, 4.40914764204, 134.58534360760],
        [9.273, 3.93291227900, 39.61750834610],
        [7.784, 5.35626068469, 75.74480641380],
        [7.724, 5.77176047568, 73.81839072080],
        [7.683, 4.44252070929, 14.01464568050],
        [8.355, 2.44425910430, 146.59425171800],
        [7.954, 5.73093878181, 184.72728735580],
        [7.465, 2.18972405572, 145.10977900970],
        [6.430, 0.84582374839, 32.19514480460],
        [6.257, 2.17085130003, 74.89347315190],
        [7.911, 0.17275924476, 120.35824960600],
        [7.036, 4.12047266896, 191.20769491020],
        [6.860, 2.13462553365, 116.42609634290],
        [5.191, 3.11155355454, 106.97674337190],
        [4.798, 2.25093144226, 46.20979048510],
        [4.566, 3.45427648666, 0.75075952540],
        [4.401, 3.94058045671, 6.59228213900],
        [4.214, 5.17805765625, 144.14657116320],
        [4.409, 0.24427052932, 92.94084583200],
        [4.866, 1.15344187054, 112.91463420510],
        [4.744, 5.18229292013, 81.00137369080],
        [4.332, 2.52429167546, 99.16062095550],
        [3.876, 2.78430217652, 565.11568774670],
        [3.801, 0.75133837939, 58.10682401090],
        [4.146, 5.84943984597, 221.37585028530],
        [3.885, 4.95626104286, 125.98732389850],
        [3.815, 3.23004401930, 479.28838891550],
        [3.679, 5.28098232097, 66.91729204110],
        [3.479, 2.95514470947, 74.66972398270],
        [3.514, 4.90090391308, 28.31117565130],
        [4.515, 4.15474629145, 344.70304530790],
        [4.036, 2.28903172191, 109.94568878850],
        [4.266, 2.68534591099, 7.86430652620],
        [3.428, 0.02846652682, 140.00196957900],
        [3.644, 5.32002093810, 408.43894361130],
        [3.252, 1.44975192429, 128.95626931510],
        [4.143, 1.89070487241, 277.03499374140],
        [3.177, 0.04197149544, 220.41264243880],
        [3.901, 6.25926496244, 0.89377187730],
        [3.787, 0.02516903921, 152.74459087230],
        [3.200, 0.52009458683, 2.28762186040],
        [2.995, 1.94615440691, 80.19822453870],
        [4.029, 5.24603808726, 96.87299909510],
        [3.302, 4.81033551060, 422.66603761290],
        [3.189, 6.26156603400, 456.39383923560],
        [2.804, 1.35626949052, 404.50679034820],
        [2.970, 0.54327361056, 159.12442469020],
        [3.465, 5.88337990735, 16.67477455640],
        [3.518, 4.99649404130, 36.64856292950],
        [3.081, 2.82772472086, 453.42489381900],
        [3.320, 1.56223495893, 23.57587323610],
        [2.573, 6.19617997586, 135.54855145410],
        [2.547, 5.19937103778, 173.94221952280],
        [2.534, 1.85452635674, 490.33408917940],
        [3.106, 6.07067928601, 142.44965013380],
        [3.302, 1.02846689671, 297.64192156090],
        [2.429, 1.33640100979, 211.81462272970],
        [2.792, 3.89897022917, 358.93013930950],
        [2.947, 5.31528985588, 55.13787859430],
        [2.449, 3.44007536754, 206.18554843720],
        [2.407, 4.38551271701, 60.55450456570],
        [2.425, 2.22643225523, 66.70484372000],
        [2.295, 2.31690029267, 31.49256938900],
        [2.225, 0.41365126245, 81.37388070630],
        [2.196, 0.76281798713, 17.52610781830],
        [2.301, 3.60815987923, 288.08069400530],
        [2.557, 0.73679737974, 200.76892246580],
        [2.158, 2.61924330277, 13.33332212430],
        [2.048, 6.27204714771, 98.89998852460],
        [2.054, 3.61072687338, 333.65734504400],
        [2.190, 2.49696729700, 76.47851959670],
        [2.092, 1.66496421654, 235.39049596580],
        [2.206, 2.35938756479, 347.88443904560],
        [2.469, 4.70656858928, 186.21176006410],
        [2.226, 5.97327738150, 1514.29129671650],
        [1.851, 2.19455296942, 203.73786788240],
        [1.865, 4.98207204280, 5.10780943070],
        [2.171, 5.49034081907, 373.01422095920],
        [1.999, 5.80509154216, 146.38180339690],
        [1.903, 4.32950489567, 49.50880430180],
        [1.732, 3.94794012202, 24.11838995730],
        [1.747, 2.46883637489, 55.65914345610],
        [1.833, 3.35110048460, 143.62530630140],
        [1.686, 1.28621563322, 103.09277421860],
        [1.720, 2.35857527806, 1.64453140270],
        [1.641, 2.99507314472, 391.17346822390],
        [1.610, 0.97420709262, 977.48678462110],
        [1.696, 4.98332661473, 387.24131496080],
        [1.527, 3.15107379811, 7.42236354150],
        [1.570, 1.61119571428, 991.71387862270],
        [1.497, 2.89637638984, 19.64371997300],
        [1.507, 3.32822127349, 909.81873305460],
        [1.375, 5.75263837916, 19.12245511120],
        [1.407, 2.20244941425, 67.35923502580],
        [1.364, 4.40006421418, 27.08733537390],
        [1.357, 4.33780029649, 70.11573212130],
        [1.311, 4.62202930578, 81.89514556810],
        [1.307, 2.79964247834, 25.60286266560],
        [1.312, 3.73623252660, 628.85158605010],
        [1.286, 3.96557527092, 61.28821774860],
        [1.723, 4.56068809303, 305.34616939270],
        [1.313, 4.90611014973, 617.80588578620],
        [1.508, 6.25017976193, 604.47256366190],
        [1.235, 5.93779486368, 415.55249061210],
        [1.278, 3.21119872139, 92.04707395470],
        [1.357, 0.72647086107, 546.95644048200],
        [1.552, 5.05296247763, 10.29494073850],
        [1.230, 1.52077038294, 157.63995198190],
        [1.210, 2.63049415027, 426.59819087600],
        [1.206, 4.83219370572, 100.38446123290],
        [1.234, 4.46203104116, 162.09337010680],
        [1.174, 5.32356191090, 17.26547538740],
        [1.431, 6.18138614295, 14.22709400160],
        [1.244, 0.16929250603, 29.20494752860],
        [1.180, 4.09719023908, 443.86366626340],
        [1.180, 3.31438239649, 44.72531777680],
        [1.259, 1.88793196065, 0.65439130580],
        [1.263, 3.49967730885, 230.56457082540],
        [1.168, 2.04071854201, 30.71067209630],
        [1.523, 2.28101186489, 373.90799283650],
        [1.429, 2.05075136274, 181.75834193920],
        [1.065, 2.95960854361, 241.61027108930],
        [1.253, 0.23639539817, 561.18353448360],
        [1.255, 1.25819925760, 155.78297225810],
        [1.044, 2.89293032709, 543.02428721890],
        [1.062, 3.26314901318, 28.57180808220],
        [1.124, 1.06535506684, 88.11492069160],
        [1.186, 5.73445278027, 329.72519178090],
        [1.190, 2.82438170535, 41.10198105440],
        [1.067, 0.27101806190, 58.31927233200],
        [1.017, 4.30527610005, 67.88049988760],
        [0.959, 5.20504598622, 42.53826965290],
        [0.984, 4.90934403664, 465.95506679120],
        [0.944, 0.66925769374, 88.79624424780],
        [1.017, 4.37095088461, 13.49338081870],
        [1.222, 5.13450955699, 300.61086697750],
        [0.989, 0.53937909300, 80.71948940050],
        [0.890, 3.09802121989, 110.20632121940],
        [1.095, 1.70637576740, 43.12897048390],
        [0.992, 4.17968869928, 154.01661525950],
        [0.950, 0.09841899432, 273.10284047830],
        [0.870, 4.77500238443, 33.13710079170],
        [0.867, 4.22078052532, 20.44686912510],
        [0.830, 5.23785245773, 472.17484191470],
        [0.884, 4.34377463442, 105.49227066360],
        [0.812, 3.53258780148, 39.35687591520],
        [1.055, 1.52219418153, 227.31374111850],
        [0.963, 1.87806076896, 259.50888592310],
        [0.859, 0.57844232244, 152.01087768940],
        [0.861, 4.69213709412, 1059.38193018920],
        [1.084, 2.79612346618, 48.75804477640],
        [0.994, 2.87052008214, 454.90936652730],
        [0.831, 1.62068330602, 554.06998748280],
        [0.891, 2.85026036860, 32.24332891440],
        [0.876, 0.83921717739, 4.73530241520],
        [0.707, 6.16918394997, 3.62333672240],
        [0.787, 1.95585343912, 16.46232623530],
        [0.702, 5.49557046240, 558.00214074590],
        [0.817, 0.38724470336, 378.64329525170],
        [0.804, 2.25693582099, 16.04163511000],
        [0.866, 1.80814575866, 258.87574647670],
        [0.651, 3.72120167607, 286.59622129700],
        [0.672, 1.00052727778, 522.57741809380],
        [0.631, 4.14839739363, 141.69889060840],
        [0.748, 4.19441869839, 486.40193591630],
        [0.668, 0.77754011576, 120.99138905240],
        [0.619, 4.31040053492, 455.87257437380],
        [0.619, 4.77556598202, 453.94615868080],
        [0.647, 5.74952736928, 119.50691634410],
        [0.609, 0.24149609998, 117.91056905120],
        [0.630, 1.79018649942, 440.68227252570],
        [0.601, 1.41196883461, 218.92816973050],
        [0.719, 4.27398947015, 50.40257617910],
        [0.594, 3.92150462249, 25.27279426550],
        [0.710, 0.45768559438, 536.80451209540],
        [0.706, 6.15599144951, 258.02441321480],
        [0.617, 2.80636989892, 68.56182344380],
        [0.587, 5.47247350993, 767.36908292080],
        [0.690, 3.48978614301, 835.03713448730],
        [0.537, 4.06668446648, 450.97721326420],
        [0.511, 0.60155300709, 264.50482076920],
        [0.694, 1.18127476921, 129.91947716160],
        [0.584, 1.94104733057, 106.27416795630],
        [0.522, 5.95180617510, 518.64526483070],
        [0.507, 4.39658523394, 121.84272231430],
        [0.627, 2.24582628581, 218.40690486870],
        [0.485, 0.02058107411, 106.01353552540],
        [0.592, 2.06072766194, 296.15744885260],
        [0.587, 0.18557470860, 458.09076026500],
        [0.483, 1.50333774574, 150.52640498110],
        [0.474, 4.99848521665, 458.84151979040],
        [0.566, 1.94435189030, 699.70103135430],
        [0.472, 1.86519720200, 180.16199464630],
        [0.472, 0.07145793467, 216.48048917570],
        [0.571, 6.01195273302, 47.06112374700],
        [0.460, 3.76890954025, 342.25536475310],
        [0.489, 2.96084966272, 385.75684225250],
        [0.458, 1.99730631732, 275.55052103310],
        [0.460, 5.75982407113, 89.75945209430],
        [0.549, 1.43219978325, 171.65459766240],
        [0.544, 0.04821904056, 114.39910691340],
        [0.450, 1.94933296558, 148.59998928810],
        [0.444, 2.94093732205, 692.58748435350],
        [0.442, 2.15938034999, 173.68158709190],
        [0.543, 2.61197342701, 451.72797278960],
        [0.465, 0.31777753866, 756.32338265690],
        [0.441, 2.82271922049, 32.71640966640],
        [0.538, 2.39420182072, 339.28641933650],
        [0.569, 0.84686482736, 260.99335863140],
        [0.572, 5.40379754526, 278.51946644970],
        [0.422, 4.61520857062, 40.16002506730],
        [0.451, 4.50911201020, 142.14083359310],
        [0.501, 0.18290112601, 331.32153907380],
        [0.468, 0.97688759019, 760.25553592000],
        [0.443, 4.58896013561, 149.67507171920],
        [0.428, 1.02564850231, 469.13646052890],
        [0.500, 4.34235307579, 166.82867252200],
        [0.412, 5.69502940499, 92.30770638560],
        [0.404, 5.18855270166, 22.63391724900],
        [0.396, 3.98515136901, 31.23193695810],
        [0.421, 5.47567810199, 104.00779795530],
        [0.425, 3.50702044406, 180.27386923090],
        [0.415, 1.52291071520, 497.44763618020],
        [0.430, 2.39159932023, 39.09624348430],
        [0.401, 0.55271143649, 95.38852638680],
        [0.384, 2.48712922138, 210.33015002140],
        [0.422, 1.02056886848, 468.24268865160],
        [0.465, 5.72323435231, 183.24281464750],
        [0.383, 2.63486938783, 685.47393735270],
        [0.367, 5.39331524988, 874.39401040250],
        [0.461, 3.57961254790, 187.69623277240],
        [0.409, 4.21780704807, 181.05576652360],
        [0.440, 0.36380766054, 367.97010200330],
        [0.392, 5.44355925956, 26.02355379090],
        [0.431, 3.83885208954, 254.94359321360],
        [0.366, 2.92275490656, 291.26208774300],
        [0.416, 2.54190330826, 255.05546779820],
        [0.348, 0.35176743482, 46.47042291600],
        [0.413, 2.41518097006, 483.22054217860],
        [0.386, 4.76483292968, 268.43697403230],
        [0.344, 0.20350283971, 184.09414790940],
        [0.350, 1.24205287122, 97.41551581630],
        [0.361, 5.68393391400, 353.30106501700],
        [0.359, 2.62171903648, 162.89651925890],
        [0.381, 3.38777292581, 114.94162363460],
        [0.352, 2.76374792259, 295.19424100610],
        [0.340, 1.34666360560, 34.20088237470],
        [0.433, 1.90504858871, 123.53964334370],
        [0.389, 2.41268196916, 381.61224066830],
        [0.383, 3.20416581825, 79.44746501330],
        [0.369, 2.15185889720, 555.55446019110],
        [0.370, 2.19402183275, 562.66800719190],
        [0.327, 3.40081544565, 309.27832265580],
        [0.378, 5.75737470182, 916.93228005540],
        [0.318, 4.53066393124, 350.33211960040],
        [0.376, 1.74845257914, 545.47196777370],
        [0.346, 4.15815107375, 282.66406803390],
        [0.319, 4.38123849114, 154.97982310600],
        [0.320, 0.81846631878, 610.69233878540],
        [0.327, 5.07873875355, 189.72322220190],
        [0.306, 1.71903179875, 394.35486196160],
        [0.329, 1.82999432252, 706.81457835510],
        [0.335, 4.78622577105, 109.31254934210],
        [0.310, 5.08120849869, 376.19561469690],
        [0.327, 1.87637598331, 207.88246946660],
        [0.323, 1.88845451800, 192.69216761850],
        [0.284, 2.88222063053, 384.05992122310],
        [0.283, 4.63187254084, 332.17287233570],
        [0.294, 2.84554743359, 267.47376618580],
        [0.285, 0.97965330777, 113.87784205160],
        [0.319, 5.09582764612, 285.63301345050],
        [0.280, 0.67871105907, 312.45971639350],
        [0.300, 5.93285242876, 124.29040286910],
        [0.320, 4.86151247369, 448.68959140380],
        [0.310, 3.75000484412, 253.57099508990],
        [0.311, 5.59686590720, 271.40591944890],
        [0.316, 1.89686876211, 228.27694896500],
        [0.269, 0.14585942744, 142.66209845490],
        [0.270, 2.12904548682, 778.41478318470],
        [0.267, 0.96560769114, 90.82323367730],
        [0.308, 1.38454900684, 375.39246554480],
        [0.298, 3.99595366039, 451.94042111070],
        [0.278, 3.38339026214, 346.39996633730],
        [0.287, 1.01918432834, 905.88657979150],
        [0.263, 0.16921968622, 124.50285119020],
        [0.283, 5.95865378023, 362.86229257260],
        [0.266, 3.07331582044, 193.65537546500],
        [0.264, 5.47114459575, 133.10087089930],
        [0.288, 2.78232740152, 1812.52391910840],
        [0.361, 4.30140629884, 198.32124191100],
        [0.257, 1.60206491208, 369.08206769610],
        [0.274, 2.88347680082, 233.90602325750],
        [0.267, 4.90554019072, 681.54178408960],
        [0.305, 1.55983861329, 49.72125262290],
        [0.253, 0.50457429429, 316.39186965660],
        [0.258, 5.81453094409, 630.33605875840],
        [0.280, 1.15452517706, 986.08480433020],
        [0.265, 4.93584097286, 831.10498122420],
        [0.246, 1.25186233620, 134.06407874580],
        [0.298, 5.75927878031, 902.70518605380],
        [0.240, 2.84888261768, 44.09217833040],
        [0.261, 2.20643594285, 73.08467753790],
        [0.321, 3.46864827820, 372.42352012820],
        [0.234, 6.06783988023, 147.11551657980],
        [0.262, 2.69623862046, 167.72244439930],
        [0.267, 4.05985113852, 75.30286342910],
        [0.240, 0.48471871511, 172.19711438360],
        [0.261, 4.64354183979, 535.32003938710],
        [0.244, 5.85987959874, 507.59956456680],
        [0.237, 4.79666486485, 377.15882254340],
        [0.224, 1.94589447357, 593.42686339800],
        [0.226, 3.71637531808, 449.28029223480],
        [0.233, 5.98739382153, 219.89137757700],
        [0.240, 2.71609791875, 227.52618943960],
        [0.221, 2.23218400256, 460.53844081980],
        [0.226, 2.74516124394, 446.31134681820],
        [0.221, 3.20339807670, 463.50738623640],
        [0.203, 5.04975483055, 457.87831194390],
        [0.191, 4.24841510229, 4.66586644600],
        [0.169, 0.59358171769, 983.11585891360],
        [0.179, 4.12060524413, 310.17209453310],
        [0.187, 6.22165475247, 294.67297614430],
        [0.162, 1.30776665222, 248.72381809010],
        [0.219, 4.17413407057, 303.86169668440],
        [0.193, 1.64715944768, 91.45637312370],
        [0.170, 2.18067759964, 66.18357885820],
        [0.156, 4.92094728667, 68.18931642830],
        [0.145, 5.51404722738, 280.96714700450],
        [0.144, 5.81835834612, 75.53235809270],
        [0.153, 0.48549989656, 144.89733068860],
        [0.168, 5.81402201452, 149.45132255000],
        [0.150, 4.66632209585, 306.83064210100],
        [0.131, 1.01359934164, 175.16605980020],
        [0.174, 3.03279013213, 298.23262239190],
        [0.163, 1.97665571311, 221.16340196420],
        [0.144, 2.59058085010, 217.23124870110],
        [0.167, 2.74604167580, 69.15252427480],
        [0.129, 2.87574897902, 156.15547927360],
    ],
    # L2
    [
        [53033.277, 0.00000000000, 0.00000000000],
        [2357.636, 2.26014661705, 74.78159856730],
        [769.129, 4.52561041823, 11.04570026390],
        [551.533, 3.25814281023, 63.73589830340],
        [541.532, 2.27573907424, 3.93215326310],
        [529.473, 4.92348433826, 1.48447270830],
        [257.521, 3.69059216858, 3.18139373770],
        [238.835, 5.85806638405, 149.56319713460],
        [181.904, 6.21763603405, 70.84944530420],
        [49.401, 6.03101301723, 56.62235130260],
        [53.504, 1.44225240953, 76.26607127560],
        [38.222, 1.78467827781, 52.69019803950],
        [44.753, 3.90904910523, 2.44768055480],
        [44.530, 0.81152639478, 85.82729883120],
        [37.403, 4.46228598032, 2.96894541660],
        [33.029, 0.86388149962, 9.56122755560],
        [24.292, 2.10702559049, 18.15924726470],
        [29.423, 5.09818697708, 73.29712585900],
        [22.135, 4.81730808582, 78.71375183040],
        [22.491, 5.99320728691, 138.51749687070],
        [17.226, 2.53537183199, 145.63104387150],
        [21.392, 2.39880709309, 77.96299230500],
        [20.578, 2.16918786539, 224.34479570190],
        [16.777, 3.46631344086, 12.53017297220],
        [12.012, 0.01941361902, 22.09140052780],
        [10.466, 4.45556032593, 62.25142559510],
        [11.010, 0.08496274370, 127.47179660680],
        [8.668, 4.25550086984, 7.11354700080],
        [10.476, 5.16453084068, 71.60020482960],
        [7.160, 1.24903906391, 5.41662597140],
        [8.387, 5.50115930045, 67.66805156650],
        [6.087, 5.44611674384, 65.22037101170],
        [6.013, 4.51836836347, 151.04766984290],
        [5.718, 1.82933915340, 202.25339517410],
        [6.109, 3.36320161279, 447.79581952650],
        [6.003, 5.72500086735, 462.02291352810],
        [5.111, 3.52374555791, 59.80374504030],
        [5.155, 1.05810305746, 131.40394986990],
        [5.969, 5.61147374852, 148.07872442630],
        [5.065, 3.36477113418, 4.45341812490],
        [4.845, 1.20298837109, 71.81265315070],
        [3.979, 0.67629577193, 77.75054398390],
        [3.673, 1.76315074166, 351.81659230870],
        [3.149, 3.83590892865, 45.57665103870],
        [3.036, 3.32062892682, 160.60889739850],
        [3.033, 6.14532331482, 77.22927912210],
        [3.596, 4.57256025582, 454.90936652730],
        [2.664, 5.36121614612, 269.92144674060],
        [2.498, 1.04819496324, 69.36497259590],
        [2.307, 2.69282373897, 84.34282612290],
        [2.249, 5.07693376112, 14.97785352700],
        [2.228, 1.38937510191, 284.14854074220],
        [2.064, 4.34647674542, 984.60033162190],
        [2.105, 2.32047802326, 120.35824960600],
        [1.864, 5.70354779393, 54.17467074780],
        [2.005, 3.87177765185, 195.13984817330],
        [1.622, 5.07964536529, 209.36694217490],
        [1.597, 0.48807990368, 137.03302416240],
        [1.583, 2.90536212187, 51.20572533120],
        [1.725, 6.25703202673, 41.64449777560],
        [2.073, 1.24032244487, 35.16409022120],
        [1.543, 2.15414338268, 70.32818044240],
        [1.671, 6.28283232471, 277.03499374140],
        [1.494, 6.04572758571, 87.31177153950],
        [1.418, 1.15843502159, 213.29909543800],
        [1.239, 4.63223076077, 92.94084583200],
        [1.238, 2.65969680342, 134.58534360760],
        [1.273, 5.87964059822, 60.55450456570],
        [1.160, 1.03320781667, 153.49535039770],
        [1.430, 4.68022239016, 299.12639426920],
        [1.117, 2.62506108047, 72.33391801250],
        [1.142, 4.64615099782, 152.74459087230],
        [0.974, 2.85233132493, 222.86032299360],
        [1.046, 4.81299135934, 116.42609634290],
        [0.872, 3.49659835508, 340.77089204480],
        [0.952, 2.10837480840, 20.60692781950],
        [0.964, 2.46471453524, 380.12776796000],
        [0.843, 6.12869288891, 49.50880430180],
        [0.821, 0.27134156683, 191.20769491020],
        [0.813, 4.08930465981, 14.22709400160],
        [0.796, 6.17066846300, 344.70304530790],
        [0.924, 2.11096444289, 14.01464568050],
        [0.791, 2.38927423348, 58.10682401090],
        [0.781, 0.74223115950, 408.43894361130],
        [0.759, 3.77564054479, 80.19822453870],
        [0.884, 1.99930014838, 265.98929347750],
        [0.722, 3.10001033669, 422.66603761290],
        [0.750, 2.33167721991, 358.93013930950],
        [0.687, 2.02866342040, 33.67961751290],
        [0.603, 1.10391172652, 55.13787859430],
        [0.655, 3.85415269764, 16.67477455640],
        [0.606, 0.15052747979, 28.31117565130],
        [0.639, 5.16714934188, 23.57587323610],
        [0.658, 0.75636229109, 76.47851959670],
        [0.590, 1.73778850095, 8.07675484730],
        [0.565, 4.92645232089, 35.42472265210],
        [0.656, 2.34273264083, 38.13303563780],
        [0.542, 5.97968975563, 146.59425171800],
        [0.518, 3.19086220901, 152.53214255120],
        [0.536, 4.52808465499, 220.41264243880],
        [0.489, 4.80633294199, 159.12442469020],
        [0.491, 0.85765309118, 565.11568774670],
        [0.483, 3.52583593251, 144.14657116320],
        [0.521, 5.21561656321, 206.18554843720],
        [0.477, 4.25420753202, 365.90067395840],
        [0.466, 5.13219663072, 297.64192156090],
        [0.557, 0.98387565952, 225.82926841020],
        [0.531, 4.22534657450, 29.20494752860],
        [0.500, 3.49663062387, 128.95626931510],
        [0.445, 2.60797570173, 96.87299909510],
        [0.466, 6.05585106742, 70.11573212130],
        [0.425, 1.04692398351, 19.64371997300],
        [0.491, 2.26123398680, 152.01087768940],
        [0.455, 5.45520675000, 333.65734504400],
        [0.458, 0.91654899383, 373.01422095920],
        [0.520, 5.72828536642, 111.43016149680],
        [0.432, 1.04604024916, 125.98732389850],
        [0.387, 2.82547341355, 200.76892246580],
        [0.383, 1.91679738697, 5.62907429250],
        [0.504, 1.95816731769, 415.55249061210],
        [0.370, 3.21958844151, 387.24131496080],
        [0.379, 2.75940848661, 81.89514556810],
        [0.345, 2.98021540638, 429.77958461370],
        [0.368, 6.20331898497, 456.39383923560],
        [0.335, 5.29062985955, 13.33332212430],
        [0.320, 0.74685222907, 347.88443904560],
        [0.307, 1.65925943351, 99.16062095550],
        [0.284, 2.09437476480, 129.91947716160],
        [0.275, 0.62680026669, 31.49256938900],
        [0.339, 1.65968150805, 142.44965013380],
        [0.270, 2.79378345550, 977.48678462110],
        [0.284, 2.42261530322, 546.95644048200],
        [0.271, 0.59225635449, 1894.41906467650],
        [0.263, 3.49309481771, 440.68227252570],
        [0.295, 0.43376026627, 373.90799283650],
        [0.271, 4.82853730065, 561.18353448360],
        [0.248, 5.72379940676, 79.23501669220],
        [0.252, 5.76728095309, 235.39049596580],
        [0.261, 0.09830739366, 991.71387862270],
        [0.238, 1.14634663242, 288.08069400530],
        [0.220, 5.12386263728, 479.28838891550],
        [0.263, 1.52366362053, 146.38180339690],
        [0.230, 0.35983101967, 109.94568878850],
        [0.214, 2.94388765580, 184.72728735580],
        [0.198, 5.72107161521, 453.42489381900],
        [0.185, 4.71600176780, 108.46121608020],
        [0.139, 2.99832472431, 211.81462272970],
        [0.167, 1.19643522280, 39.61750834610],
        [0.180, 0.80198096578, 183.24281464750],
        [0.131, 2.73236351123, 522.57741809380],
        [0.142, 5.03489222377, 536.80451209540],
    ],
    # L3
    [
        [120.936, 0.02418789918, 74.78159856730],
        [68.064, 4.12084267733, 3.93215326310],
        [52.828, 2.38964061260, 11.04570026390],
        [43.754, 2.95965039734, 1.48447270830],
        [45.300, 2.04423798410, 3.18139373770],
        [45.806, 0.00000000000, 0.00000000000],
        [24.969, 4.88741307918, 63.73589830340],
        [21.061, 4.54511486862, 70.84944530420],
        [19.897, 2.31320314136, 149.56319713460],
        [8.901, 1.57548871761, 56.62235130260],
        [4.271, 0.22777319552, 18.15924726470],
        [3.613, 5.39244611308, 76.26607127560],
        [3.488, 4.97622811775, 85.82729883120],
        [3.479, 4.12969359977, 52.69019803950],
        [3.572, 0.95052448578, 77.96299230500],
        [2.328, 0.85770961794, 145.63104387150],
        [2.696, 0.37287796344, 78.71375183040],
        [1.946, 2.67997393431, 7.11354700080],
        [2.156, 5.65647821519, 9.56122755560],
        [1.363, 4.86983744746, 224.34479570190],
        [1.333, 1.25032115614, 12.53017297220],
        [1.613, 0.48764377311, 71.60020482960],
        [1.475, 5.19957293069, 73.29712585900],
        [1.225, 3.93406822032, 22.09140052780],
        [0.911, 2.18921999026, 127.47179660680],
        [0.811, 3.98323855938, 462.02291352810],
        [0.808, 5.06374463008, 447.79581952650],
        [0.718, 0.34600103024, 5.62907429250],
        [0.722, 1.05856935832, 138.51749687070],
        [0.687, 2.93752748595, 131.40394986990],
        [0.463, 1.58927254512, 151.04766984290],
        [0.414, 4.32904287620, 120.35824960600],
        [0.372, 0.73596518002, 269.92144674060],
        [0.345, 3.05968942771, 561.18353448360],
        [0.338, 5.94221536204, 284.14854074220],
        [0.343, 4.01891371998, 546.95644048200],
        [0.382, 5.93515231196, 45.57665103870],
        [0.276, 3.44212110991, 202.25339517410],
        [0.300, 1.13119175675, 160.60889739850],
        [0.268, 3.24615387968, 536.80451209540],
        [0.358, 1.10916640253, 333.65734504400],
        [0.236, 4.65292396535, 387.24131496080],
        [0.204, 5.81663798296, 373.01422095920],
        [0.145, 2.75632381347, 92.94084583200],
        [0.149, 0.13764106563, 71.81265315070],
        [0.156, 2.90936922804, 153.49535039770],
    ],
    # L4
    [
        [113.855, 3.14159265359, 0.00000000000],
        [5.599, 4.57882424417, 74.78159856730],
        [3.203, 0.34623003207, 11.04570026390],
        [1.217, 3.42199121826, 56.62235130260],
        [0.634, 4.65759668097, 18.15924726470],
        [0.171, 3.80393539303, 149.56319713460],
        [0.133, 4.35519131657, 63.73589830340],
    ],
    # L5
    [
        [0.873, 3.14159265359, 0.00000000000]
    ],
]
"""This table contains Uranus' periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric longitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in pages
445-448."""


VSOP87_B = [
    # B0
    [
        [1346277.639, 2.61877810545, 74.78159856730],
        [62341.405, 5.08111175856, 149.56319713460],
        [61601.203, 3.14159265359, 0.00000000000],
        [9963.744, 1.61603876357, 76.26607127560],
        [9926.151, 0.57630387917, 73.29712585900],
        [3259.455, 1.26119385960, 224.34479570190],
        [2972.318, 2.24367035538, 1.48447270830],
        [2010.257, 6.05550401088, 148.07872442630],
        [1522.172, 0.27960386377, 63.73589830340],
        [924.055, 4.03822927853, 151.04766984290],
        [760.624, 6.14000431923, 71.81265315070],
        [420.265, 5.21279984788, 11.04570026390],
        [430.668, 3.55445034854, 213.29909543800],
        [436.843, 3.38082524317, 529.69096509460],
        [522.309, 3.32085194770, 138.51749687070],
        [434.625, 0.34065281858, 77.75054398390],
        [462.630, 0.74256727574, 85.82729883120],
        [232.649, 2.25716421383, 222.86032299360],
        [215.838, 1.59121704940, 38.13303563780],
        [244.698, 0.78795150326, 2.96894541660],
        [179.935, 3.72487952673, 299.12639426920],
        [174.895, 1.23550262213, 146.59425171800],
        [173.667, 1.93654269131, 380.12776796000],
        [160.368, 5.33635436463, 111.43016149680],
        [144.064, 5.96239326415, 35.16409022120],
        [102.049, 2.61876256513, 78.71375183040],
        [116.363, 5.73877190007, 70.84944530420],
        [106.441, 0.94103112994, 70.32818044240],
        [86.163, 0.70262506622, 39.61750834610],
        [72.617, 0.20564696113, 225.82926841020],
        [71.172, 0.83343269975, 109.94568878850],
        [57.502, 2.67039425415, 108.46121608020],
        [54.255, 3.35166579613, 184.72728735580],
        [44.470, 2.74408231138, 152.53214255120],
        [38.591, 5.17394663303, 202.25339517410],
        [39.157, 2.17108251341, 351.81659230870],
        [41.346, 3.22134319551, 160.60889739850],
        [35.140, 4.00111634363, 112.91463420510],
        [33.073, 3.61378095742, 221.37585028530],
        [31.315, 2.71969470781, 145.10977900970],
        [37.336, 4.02053241202, 52.69019803950],
        [32.028, 1.29160071142, 145.63104387150],
        [27.574, 3.70064266960, 36.64856292950],
        [24.277, 2.84989187496, 127.47179660680],
        [24.635, 1.11645461259, 3.93215326310],
        [24.315, 5.48987913644, 79.23501669220],
        [21.418, 0.63722900407, 277.03499374140],
        [19.826, 2.59334182230, 84.34282612290],
        [22.373, 5.73687615457, 4.45341812490],
        [19.137, 1.30214105578, 62.25142559510],
        [19.789, 4.72260849557, 297.64192156090],
        [20.299, 1.06070151806, 454.90936652730],
        [19.768, 5.77906142568, 305.34616939270],
        [21.348, 1.01350946382, 33.67961751290],
        [15.142, 2.91786832554, 426.59819087600],
        [16.000, 1.95535748902, 186.21176006410],
        [13.819, 2.67163927171, 74.66972398270],
        [11.463, 5.73391138419, 41.10198105440],
        [10.741, 3.73401569675, 1059.38193018920],
        [11.450, 3.98177764866, 106.97674337190],
        [10.360, 4.75567608732, 183.24281464750],
        [10.232, 6.18772866993, 373.90799283650],
        [13.803, 5.70712120608, 74.89347315190],
        [10.553, 3.78602881738, 490.33408917940],
        [11.838, 5.96756415681, 87.31177153950],
        [10.030, 1.74828757238, 22.09140052780],
        [10.107, 0.92911975959, 65.22037101170],
        [9.127, 5.11093790809, 153.49535039770],
        [12.093, 2.53736362742, 9.56122755560],
        [8.646, 4.18351923569, 12.53017297220],
        [9.978, 5.83600622359, 604.47256366190],
        [11.352, 2.12645777694, 68.84370773410],
        [8.472, 3.36885457285, 72.33391801250],
        [7.797, 5.11771906359, 77.96299230500],
        [8.302, 5.19247905162, 77.22927912210],
        [7.696, 3.25189037096, 71.60020482960],
        [7.513, 2.98265970100, 114.39910691340],
        [6.947, 3.31871016057, 56.62235130260],
        [6.490, 0.88434578474, 288.08069400530],
        [6.394, 3.51142812432, 220.41264243880],
        [6.211, 0.58222518453, 340.77089204480],
        [6.772, 4.09374798222, 137.03302416240],
        [5.595, 5.68643434536, 259.50888592310],
        [5.309, 2.65421183211, 300.61086697750],
        [4.950, 4.99672086239, 219.89137757700],
        [6.419, 0.44895727879, 140.00196957900],
        [4.975, 4.06722486039, 143.62530630140],
        [5.692, 3.76563463180, 67.66805156650],
        [4.853, 2.03383592524, 415.55249061210],
        [3.796, 1.26231186682, 75.30286342910],
        [3.807, 2.24787582155, 909.81873305460],
        [3.812, 3.10475682509, 181.75834193920],
        [3.764, 5.20052090560, 227.31374111850],
        [3.445, 5.91769433069, 296.15744885260],
        [3.517, 5.44397685665, 628.85158605010],
        [3.943, 4.95136058926, 265.98929347750],
        [3.472, 0.34737998380, 131.40394986990],
        [3.390, 0.53497504164, 206.18554843720],
        [3.038, 4.68314286209, 159.12442469020],
        [3.190, 5.68929316349, 235.39049596580],
        [3.303, 2.09359507373, 73.81839072080],
        [3.069, 4.49065085092, 66.70484372000],
        [3.285, 0.00780313833, 75.74480641380],
        [2.917, 4.21615078632, 258.02441321480],
        [3.747, 0.82999983666, 74.26033370550],
        [2.814, 3.96708337625, 82.85835341460],
        [2.474, 3.81319259323, 7.11354700080],
        [2.394, 2.21483198491, 54.17467074780],
        [2.555, 2.97023907145, 378.64329525170],
        [2.631, 1.55153254691, 154.01661525950],
        [2.633, 2.28385552693, 32.19514480460],
        [2.643, 3.96832729680, 381.61224066830],
        [2.206, 3.06995275892, 59.80374504030],
        [2.635, 0.53987945692, 211.81462272970],
        [2.071, 1.97429082033, 18.15924726470],
        [2.485, 3.55433846990, 96.87299909510],
        [2.061, 4.50102695788, 5.93789083320],
        [1.916, 1.60538526374, 80.19822453870],
        [2.480, 0.63321072542, 187.69623277240],
        [2.039, 2.97351088965, 191.20769491020],
        [1.833, 1.95824865568, 81.89514556810],
        [1.719, 2.22526635038, 479.28838891550],
        [1.745, 3.22821992592, 218.40690486870],
        [1.857, 1.66304484985, 984.60033162190],
        [1.766, 5.24239122261, 105.49227066360],
        [1.524, 6.05374020264, 99.16062095550],
        [1.519, 0.94716867229, 372.42352012820],
        [1.614, 3.39986066169, 230.56457082540],
        [1.711, 3.44237080993, 522.57741809380],
        [1.504, 1.34653259405, 74.52096613640],
        [1.577, 4.38020936720, 80.71948940050],
        [1.360, 5.48691240270, 74.82978267710],
        [1.364, 4.56045715617, 42.58645376270],
        [1.398, 0.33827838973, 142.44965013380],
        [1.709, 3.63188407264, 554.06998748280],
        [1.360, 2.89305157919, 74.73341445750],
        [1.260, 5.50922979275, 74.94165726170],
        [1.374, 4.39897993200, 260.99335863140],
        [1.366, 2.15288773765, 162.09337010680],
        [1.244, 4.97789913094, 149.45132255000],
        [1.269, 0.84167691738, 767.36908292080],
        [1.278, 4.53585702916, 294.67297614430],
        [1.342, 5.11117141196, 51.20572533120],
        [1.180, 0.79882196802, 116.42609634290],
        [1.495, 0.74986873597, 75.04223099820],
        [1.207, 3.67288675913, 20.60692781950],
        [1.181, 4.39598416757, 180.27386923090],
        [1.248, 3.13312504066, 67.35923502580],
        [1.263, 2.87116663203, 74.62153987290],
        [1.380, 3.76141611602, 92.94084583200],
        [1.113, 3.87133607367, 39.35687591520],
        [1.018, 6.20393099094, 835.03713448730],
        [0.962, 3.30343472839, 255.05546779820],
        [1.238, 1.73023505315, 149.67507171920],
        [0.970, 1.71236273285, 115.88357962170],
        [1.001, 5.49914631698, 256.53994050650],
        [0.921, 3.07729879788, 8.07675484730],
        [0.914, 0.00764291274, 536.80451209540],
        [0.911, 6.23753038018, 200.76892246580],
        [0.956, 5.76811833839, 128.95626931510],
        [0.999, 0.33400530567, 404.50679034820],
        [0.952, 3.00456073496, 14.97785352700],
        [0.765, 3.45454533660, 214.78356814630],
        [0.800, 3.20912932090, 28.31117565130],
        [0.799, 4.11425365829, 125.98732389850],
        [1.021, 1.79905869707, 3.18139373770],
        [0.706, 5.80210566917, 157.63995198190],
        [0.715, 5.56313177065, 146.38180339690],
        [0.689, 1.84748347121, 41.64449777560],
        [0.682, 5.16479782395, 74.03083904190],
        [0.673, 2.65544175682, 75.53235809270],
        [0.723, 4.75905991606, 331.32153907380],
        [0.730, 2.25510749124, 453.42489381900],
        [0.691, 3.58561635364, 362.86229257260],
        [0.641, 5.77408891198, 110.20632121940],
        [0.671, 5.96862039131, 135.54855145410],
        [0.631, 1.97807297205, 639.89728631400],
        [0.774, 0.42035450706, 565.11568774670],
        [0.705, 2.94649553712, 60.76695288680],
        [0.663, 5.32574112049, 142.14083359310],
        [0.612, 2.01741578932, 195.13984817330],
        [0.749, 5.56828487823, 2.44768055480],
        [0.798, 0.90969731665, 152.01087768940],
        [0.747, 5.07639466593, 89.75945209430],
        [0.650, 4.56215718085, 216.92243216040],
        [0.651, 2.80626026285, 50.40257617910],
        [0.593, 1.89556258897, 203.73786788240],
        [0.550, 2.90625551534, 68.18931642830],
        [0.548, 3.75628845322, 617.80588578620],
        [0.554, 2.78135114877, 14.01464568050],
        [0.530, 3.51385025328, 291.70403072770],
        [0.506, 4.94619342366, 81.37388070630],
        [0.649, 5.74895589744, 141.48644228730],
        [0.593, 5.40734033998, 692.58748435350],
        [0.544, 2.99910512780, 152.74459087230],
        [0.485, 2.36317665443, 448.68959140380],
        [0.481, 5.81647231299, 134.58534360760],
        [0.517, 4.97759795528, 387.24131496080],
        [0.573, 2.46311368783, 81.00137369080],
        [0.470, 1.30184316812, 228.27694896500],
        [0.475, 0.53480492526, 303.86169668440],
        [0.485, 6.21247575899, 5.41662597140],
        [0.468, 0.55881267334, 23.57587323610],
        [0.585, 4.67924542643, 88.79624424780],
        [0.512, 5.86200059955, 293.18850343600],
        [0.445, 2.50076311432, 905.88657979150],
        [0.501, 4.79997295899, 306.83064210100],
        [0.418, 5.21379084769, 35.42472265210],
        [0.408, 5.69107313998, 284.14854074220],
        [0.474, 3.03149617428, 286.59622129700],
        [0.432, 4.20907682097, 278.51946644970],
        [0.506, 2.05348204197, 373.01422095920],
        [0.410, 5.30637634877, 95.38852638680],
        [0.478, 2.41106642594, 358.93013930950],
        [0.536, 1.82614772260, 114.13847448250],
        [0.383, 5.54541241459, 419.48464387520],
        [0.413, 0.03813081773, 103.09277421860],
        [0.368, 4.08526832792, 1589.07289528380],
        [0.369, 1.82533858431, 334.29048449040],
        [0.491, 5.58913582973, 68.56182344380],
        [0.387, 0.56619310480, 602.98809095360],
        [0.427, 5.08349119654, 367.97010200330],
        [0.475, 0.17507881032, 120.35824960600],
        [0.342, 5.27187859255, 28.57180808220],
        [0.407, 2.00853504718, 679.25416222920],
        [0.381, 4.61366060949, 329.72519178090],
        [0.388, 0.88856038803, 483.22054217860],
        [0.376, 1.28856348513, 155.78297225810],
        [0.326, 6.09140263554, 456.39383923560],
        [0.318, 0.09996195379, 69.36497259590],
        [0.292, 5.11578046796, 375.39246554480],
        [0.299, 6.04121646505, 332.80601178210],
        [0.328, 3.47252263966, 73.40900044360],
        [0.283, 1.81773059325, 647.01083331480],
        [0.308, 3.50154864071, 30.71067209630],
        [0.283, 1.88910019831, 24.37902238820],
        [0.278, 3.85507901929, 760.25553592000],
        [0.273, 4.22941219477, 391.17346822390],
        [0.293, 5.44464406156, 477.91579079180],
        [0.289, 3.85492516765, 209.36694217490],
        [0.309, 1.97045147502, 543.02428721890],
        [0.324, 5.57191515554, 501.37978944330],
        [0.248, 2.17741598681, 611.58611066270],
        [0.245, 1.04008534095, 1332.48477066750],
        [0.248, 6.19516038159, 1134.16352875650],
        [0.242, 1.37720813333, 121.25202148330],
        [0.233, 0.50933224249, 462.02291352810],
        [0.319, 4.24219881040, 328.35259365720],
        [0.244, 6.00644853640, 295.19424100610],
        [0.228, 0.72608678727, 233.90602325750],
        [0.223, 5.35785607671, 983.11585891360],
        [0.304, 5.68101077712, 189.18070548070],
        [0.223, 2.04168197187, 370.93904741990],
        [0.245, 4.69742022955, 316.39186965660],
        [0.217, 4.33845164683, 269.92144674060],
        [0.267, 0.15744446718, 10213.28554621100],
        [0.210, 5.84975171904, 147.11551657980],
        [0.224, 3.04829549918, 1439.50969814920],
        [0.226, 0.72248476212, 45.57665103870],
        [0.202, 1.37341689387, 302.09533968580],
        [0.207, 6.13936312021, 344.70304530790],
        [0.228, 2.33602531859, 150.52640498110],
        [0.203, 2.38070591912, 275.55052103310],
        [0.194, 5.11344829813, 1215.16490244730],
        [0.259, 2.78974616768, 144.14657116320],
        [0.199, 3.21010368905, 7.42236354150],
        [0.246, 6.15106498377, 6.21977512350],
        [0.180, 4.70377036870, 518.64526483070],
        [0.186, 3.85070006482, 46.20979048510],
        [0.175, 3.72163733058, 150.08446199640],
        [0.165, 1.87245210311, 310.17209453310],
        [0.166, 3.24028819042, 173.94221952280],
        [0.181, 0.39521867351, 0.96320784650],
        [0.144, 1.78180338482, 531.17543780290],
        [0.137, 6.20635855175, 369.45457471160],
        [0.136, 4.00164375048, 526.72201967800],
        [0.141, 4.02238453909, 350.33211960040],
        [0.125, 5.28865037145, 329.83706636550],
        [0.134, 3.97421067761, 228.79821382680],
        [0.132, 4.52023845365, 148.59998928810],
        [0.125, 4.24724390191, 248.72381809010],
        [0.120, 4.08565783859, 154.97982310600],
        [0.133, 3.16576403244, 262.47783133970],
    ],
    # B1
    [
        [206366.162, 4.12394311407, 74.78159856730],
        [8563.230, 0.33819986165, 149.56319713460],
        [1725.703, 2.12193159895, 73.29712585900],
        [1368.860, 3.06861722047, 76.26607127560],
        [1374.449, 0.00000000000, 0.00000000000],
        [399.847, 2.84767037795, 224.34479570190],
        [450.639, 3.77656180977, 1.48447270830],
        [307.214, 1.25456766737, 148.07872442630],
        [154.336, 3.78575467747, 63.73589830340],
        [110.888, 5.32888676461, 138.51749687070],
        [112.432, 5.57299891505, 151.04766984290],
        [83.493, 3.59152795558, 71.81265315070],
        [55.573, 3.40135416354, 85.82729883120],
        [41.377, 4.45476669141, 78.71375183040],
        [53.690, 1.70455769943, 77.75054398390],
        [41.912, 1.21476607434, 11.04570026390],
        [31.959, 3.77446207748, 222.86032299360],
        [30.297, 2.56371683644, 2.96894541660],
        [26.977, 5.33695500294, 213.29909543800],
        [26.222, 0.41620628369, 380.12776796000],
        [20.094, 5.93085633510, 529.69096509460],
        [22.992, 2.48887389394, 146.59425171800],
        [19.590, 5.37213500014, 299.12639426920],
        [20.408, 3.70179681605, 70.84944530420],
        [19.102, 1.09213276596, 111.43016149680],
        [19.411, 3.83015453768, 38.13303563780],
        [10.847, 2.66326308043, 3.93215326310],
        [10.249, 2.31278807720, 109.94568878850],
        [9.405, 2.76950513184, 39.61750834610],
        [7.660, 1.81108462850, 225.82926841020],
        [8.082, 4.69064168719, 184.72728735580],
        [6.584, 5.50417589189, 35.16409022120],
        [7.410, 1.17879753422, 65.22037101170],
        [6.451, 4.98294064391, 71.60020482960],
        [6.089, 1.31830108565, 52.69019803950],
        [4.768, 5.90574941745, 145.63104387150],
        [4.840, 4.86390682412, 221.37585028530],
        [4.192, 3.29643787103, 77.96299230500],
        [4.711, 4.24289069791, 152.53214255120],
        [4.894, 6.01164167429, 160.60889739850],
        [3.738, 4.75287390209, 70.32818044240],
        [3.481, 0.64108927026, 153.49535039770],
        [3.758, 3.94715595219, 351.81659230870],
        [3.114, 0.10537144899, 112.91463420510],
        [2.788, 4.24118032837, 74.66972398270],
        [2.505, 0.04576283378, 297.64192156090],
        [2.563, 4.15665405963, 305.34616939270],
        [2.544, 5.25903565788, 56.62235130260],
        [2.247, 4.24726481845, 36.64856292950],
        [2.541, 0.40106060407, 77.22927912210],
        [2.212, 2.88960413468, 277.03499374140],
        [2.299, 3.57748029365, 186.21176006410],
        [2.661, 0.53230319176, 79.23501669220],
        [2.157, 2.10150995852, 127.47179660680],
        [2.265, 1.41055702214, 4.45341812490],
        [2.103, 4.27438518414, 22.09140052780],
        [1.861, 3.75619999278, 145.10977900970],
        [1.759, 2.10240976488, 131.40394986990],
        [1.661, 4.84483054269, 62.25142559510],
        [1.496, 1.72084298116, 220.41264243880],
        [1.659, 5.86539712478, 454.90936652730],
        [1.428, 0.31508367934, 137.03302416240],
        [1.522, 1.00801468633, 75.74480641380],
        [1.459, 6.17427145114, 426.59819087600],
        [1.463, 5.14953143442, 84.34282612290],
        [1.453, 2.22988903923, 206.18554843720],
        [1.358, 5.85111427068, 183.24281464750],
        [1.405, 2.43582184515, 87.31177153950],
        [1.495, 5.55621838458, 67.66805156650],
        [1.317, 1.91178535183, 140.00196957900],
        [1.068, 1.51430678116, 373.90799283650],
        [1.439, 0.99170994448, 74.89347315190],
        [1.065, 4.15616015505, 288.08069400530],
        [1.096, 1.63909426062, 41.10198105440],
        [1.189, 0.90595784409, 33.67961751290],
        [0.961, 5.48175535705, 490.33408917940],
        [0.851, 0.95029849401, 909.81873305460],
        [0.820, 0.78610123063, 259.50888592310],
        [0.881, 4.31294603221, 9.56122755560],
        [0.708, 0.00007309836, 81.89514556810],
        [0.709, 3.18853632737, 80.19822453870],
        [0.786, 5.18884635415, 114.39910691340],
        [0.822, 0.01949759759, 18.15924726470],
        [0.656, 6.16899483115, 96.87299909510],
        [0.879, 1.82006610038, 73.81839072080],
        [0.872, 0.30134022304, 12.53017297220],
        [0.860, 0.21225398802, 3.18139373770],
        [0.637, 2.64378420008, 75.30286342910],
        [0.727, 0.02846968582, 66.70484372000],
        [0.600, 4.42462853209, 415.55249061210],
        [0.590, 4.17885957237, 300.61086697750],
        [0.610, 6.07202921132, 219.89137757700],
        [0.611, 0.97629869063, 296.15744885260],
        [0.635, 2.21125075603, 74.26033370550],
        [0.529, 2.35940463062, 7.11354700080],
        [0.622, 2.21801944850, 211.81462272970],
        [0.519, 2.01872911223, 142.44965013380],
        [0.489, 1.83419944488, 92.94084583200],
        [0.445, 2.51784247184, 604.47256366190],
        [0.413, 5.36482818305, 82.85835341460],
        [0.445, 0.33164113115, 227.31374111850],
        [0.456, 2.44190834824, 381.61224066830],
        [0.378, 1.91873737843, 202.25339517410],
        [0.509, 5.83556856314, 191.20769491020],
        [0.455, 5.15414537021, 522.57741809380],
        [0.419, 2.80644155875, 72.33391801250],
        [0.333, 0.32014837950, 2.44768055480],
        [0.360, 1.47248643716, 378.64329525170],
        [0.306, 0.15517399606, 159.12442469020],
        [0.301, 4.46417652272, 536.80451209540],
        [0.353, 0.48749845867, 128.95626931510],
        [0.351, 6.24769322491, 5.93789083320],
        [0.298, 1.71815652029, 235.39049596580],
        [0.315, 2.44922921309, 187.69623277240],
        [0.318, 0.70176359510, 181.75834193920],
        [0.314, 4.68400251693, 14.97785352700],
        [0.282, 3.70093718573, 108.46121608020],
        [0.272, 3.91340553608, 617.80588578620],
        [0.273, 3.29483889428, 387.24131496080],
        [0.323, 4.90410549341, 258.02441321480],
        [0.288, 4.42249612833, 195.13984817330],
        [0.250, 1.23231297183, 703.63318461740],
        [0.338, 2.84645768890, 154.01661525950],
        [0.297, 1.16538119842, 146.38180339690],
        [0.248, 4.90614051989, 41.64449777560],
        [0.275, 5.35665949805, 80.71948940050],
        [0.257, 1.82441994046, 230.56457082540],
        [0.234, 0.27679874465, 33.13710079170],
        [0.280, 1.73679618032, 265.98929347750],
        [0.229, 0.49529839431, 74.82978267710],
        [0.229, 4.18462288684, 74.73341445750],
        [0.253, 2.63817804331, 74.52096613640],
        [0.252, 2.04143912495, 75.04223099820],
        [0.213, 4.16218259902, 74.62153987290],
        [0.212, 0.51761494342, 74.94165726170],
        [0.201, 4.54140547837, 20.60692781950],
        [0.194, 4.17282454759, 116.42609634290],
        [0.213, 2.29528235429, 32.19514480460],
        [0.174, 3.15418942153, 228.27694896500],
        [0.194, 0.06960211137, 42.58645376270],
        [0.173, 5.59700344643, 68.84370773410],
        [0.159, 4.91721631097, 143.62530630140],
        [0.150, 0.92771324396, 404.50679034820],
        [0.136, 2.51083022906, 372.42352012820],
        [0.134, 3.22507836958, 479.28838891550],
        [0.124, 0.42063711585, 149.45132255000],
        [0.122, 1.20639876458, 5.41662597140],
        [0.119, 0.45375065997, 74.03083904190],
        [0.146, 5.08207330360, 294.67297614430],
        [0.118, 4.22640788890, 75.53235809270],
        [0.118, 1.50613822829, 344.70304530790],
        [0.121, 2.13544759505, 209.36694217490],
        [0.121, 6.08239076370, 260.99335863140],
        [0.154, 4.18369977366, 39.35687591520],
    ],
    # B2
    [
        [9211.656, 5.80044305785, 74.78159856730],
        [556.926, 0.00000000000, 0.00000000000],
        [286.265, 2.17729776353, 149.56319713460],
        [94.969, 3.84237569809, 73.29712585900],
        [45.419, 4.87822046064, 76.26607127560],
        [20.107, 5.46264485369, 1.48447270830],
        [14.793, 0.87983715652, 138.51749687070],
        [13.963, 5.07234043994, 63.73589830340],
        [14.261, 2.84517742687, 148.07872442630],
        [10.122, 5.00290894862, 224.34479570190],
        [8.299, 6.26655615197, 78.71375183040],
        [4.729, 5.16274174929, 71.81265315070],
        [3.816, 6.28224514574, 85.82729883120],
        [3.488, 3.53472172445, 11.04570026390],
        [2.555, 1.44444215715, 151.04766984290],
        [2.353, 4.23069776466, 3.93215326310],
        [2.585, 0.41383633246, 71.60020482960],
        [1.394, 4.13126838571, 146.59425171800],
        [1.183, 3.68471361409, 77.75054398390],
        [1.103, 5.54212014132, 222.86032299360],
        [1.205, 5.05109252937, 380.12776796000],
        [1.146, 1.95280464754, 529.69096509460],
        [0.977, 1.52652616357, 77.96299230500],
        [1.025, 4.33698643491, 2.96894541660],
        [0.858, 2.78728745263, 111.43016149680],
        [0.868, 5.55175791193, 38.13303563780],
        [0.633, 0.41074353315, 213.29909543800],
        [0.596, 5.39265533517, 127.47179660680],
        [0.586, 4.00404667232, 109.94568878850],
        [0.543, 2.40369406419, 153.49535039770],
        [0.486, 2.05237757516, 299.12639426920],
        [0.557, 3.13408880388, 65.22037101170],
        [0.457, 3.96543219832, 454.90936652730],
        [0.481, 2.81511187371, 160.60889739850],
        [0.421, 2.16819778071, 56.62235130260],
        [0.326, 4.52920012430, 39.61750834610],
        [0.308, 6.26508780547, 70.32818044240],
        [0.338, 5.14594268587, 3.18139373770],
        [0.288, 0.47061435406, 22.09140052780],
        [0.336, 4.23512034174, 35.16409022120],
        [0.316, 3.93430525759, 52.69019803950],
        [0.306, 1.10359318443, 70.84944530420],
        [0.250, 3.58780257084, 202.25339517410],
        [0.239, 4.77679306080, 87.31177153950],
        [0.227, 2.74138067839, 12.53017297220],
        [0.263, 1.59203582407, 84.34282612290],
        [0.215, 3.88195737361, 131.40394986990],
        [0.216, 5.80700510713, 74.66972398270],
        [0.264, 1.82574036051, 77.22927912210],
        [0.222, 0.04111883550, 184.72728735580],
        [0.197, 1.56602555362, 9.56122755560],
        [0.193, 2.26416938160, 75.74480641380],
        [0.179, 2.69065316892, 145.63104387150],
        [0.170, 2.75844544119, 73.81839072080],
        [0.155, 3.55393249110, 18.15924726470],
        [0.174, 4.76111441901, 277.03499374140],
        [0.140, 0.28714762870, 221.37585028530],
        [0.134, 1.29065526326, 206.18554843720],
        [0.127, 6.17908901556, 62.25142559510],
        [0.116, 6.26646620658, 220.41264243880],
    ],
    # B3
    [
        [267.832, 1.25097888291, 74.78159856730],
        [11.048, 3.14159265359, 0.00000000000],
        [6.154, 4.00663614486, 149.56319713460],
        [3.361, 5.77804694935, 73.29712585900],
        [1.602, 1.05657834344, 63.73589830340],
        [1.265, 1.66795295537, 78.71375183040],
        [1.183, 2.58856450374, 138.51749687070],
        [1.087, 0.28687213135, 76.26607127560],
        [0.640, 1.87238784591, 71.60020482960],
        [0.590, 0.80206040001, 1.48447270830],
        [0.467, 4.42872012006, 148.07872442630],
        [0.272, 4.00684090176, 85.82729883120],
        [0.203, 0.60406901282, 71.81265315070],
        [0.180, 5.55657564049, 3.93215326310],
        [0.168, 4.67745630044, 70.84944530420],
        [0.170, 2.93672195979, 11.04570026390],
    ],
    # B4
    [
        [5.719, 2.85499529315, 74.78159856730],
        [0.300, 3.14159265359, 0.00000000000]
    ],
]
"""This table contains Uranus' periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric latitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in pages
448-449."""


VSOP87_R = [
    # R0
    [
        [1921264847.881, 0.00000000000, 0.00000000000],
        [88784984.055, 5.60377526994, 74.78159856730],
        [3440835.545, 0.32836098991, 73.29712585900],
        [2055653.495, 1.78295170028, 149.56319713460],
        [649321.851, 4.52247298119, 76.26607127560],
        [602248.144, 3.86003820462, 63.73589830340],
        [496404.171, 1.40139934716, 454.90936652730],
        [338525.522, 1.58002682946, 138.51749687070],
        [243508.222, 1.57086595074, 71.81265315070],
        [190521.915, 1.99809364502, 1.48447270830],
        [161858.251, 2.79137863469, 148.07872442630],
        [143705.902, 1.38368574483, 11.04570026390],
        [93192.359, 0.17437193645, 36.64856292950],
        [71424.265, 4.24509327405, 224.34479570190],
        [89805.842, 3.66105366329, 109.94568878850],
        [39009.624, 1.66971128869, 70.84944530420],
        [46677.322, 1.39976563936, 35.16409022120],
        [39025.681, 3.36234710692, 277.03499374140],
        [36755.160, 3.88648934736, 146.59425171800],
        [30348.875, 0.70100446346, 151.04766984290],
        [29156.264, 3.18056174556, 77.75054398390],
        [20471.584, 1.55588961500, 202.25339517410],
        [25620.360, 5.25656292802, 380.12776796000],
        [25785.805, 3.78537741503, 85.82729883120],
        [22637.152, 0.72519137745, 529.69096509460],
        [20473.163, 2.79639811626, 70.32818044240],
        [17900.561, 0.55455488605, 2.96894541660],
        [12328.151, 5.96039150918, 127.47179660680],
        [14701.566, 4.90434406648, 108.46121608020],
        [11494.701, 0.43774027872, 65.22037101170],
        [15502.809, 5.35405037603, 38.13303563780],
        [10792.699, 1.42104858472, 213.29909543800],
        [11696.085, 3.29825599114, 3.93215326310],
        [11959.355, 1.75044072173, 984.60033162190],
        [12896.507, 2.62154018241, 111.43016149680],
        [11852.996, 0.99342814582, 52.69019803950],
        [9111.446, 4.99638600045, 62.25142559510],
        [8420.550, 5.25350716616, 222.86032299360],
        [7449.125, 0.79491905956, 351.81659230870],
        [8402.147, 5.03877516489, 415.55249061210],
        [6046.370, 5.67960948357, 78.71375183040],
        [5524.133, 3.11499484161, 9.56122755560],
        [7329.454, 3.97277527840, 183.24281464750],
        [5444.878, 5.10575635361, 145.10977900970],
        [5238.103, 2.62960141797, 33.67961751290],
        [4079.167, 3.22064788674, 340.77089204480],
        [3801.606, 6.10985558505, 184.72728735580],
        [3919.476, 4.25015288873, 39.61750834610],
        [2940.492, 2.14637460319, 137.03302416240],
        [3781.219, 3.45840272873, 456.39383923560],
        [2942.239, 0.42393808854, 299.12639426920],
        [3686.787, 2.48718116535, 453.42489381900],
        [3101.743, 4.14031063896, 219.89137757700],
        [2962.641, 0.82977991995, 56.62235130260],
        [2937.799, 3.67657450930, 140.00196957900],
        [2865.128, 0.30996903761, 12.53017297220],
        [2538.032, 4.85457831993, 131.40394986990],
        [1962.510, 5.24342224065, 84.34282612290],
        [2363.550, 0.44253328372, 554.06998748280],
        [1979.394, 6.12836181686, 106.97674337190],
        [2182.572, 2.94040431638, 305.34616939270],
        [1962.974, 0.04114739120, 221.37585028530],
        [1829.560, 4.01105771632, 68.84370773410],
        [1642.920, 0.35564102554, 67.66805156650],
        [1584.850, 3.16267171762, 225.82926841020],
        [1848.655, 2.91111759376, 909.81873305460],
        [1632.430, 4.23061792837, 22.09140052780],
        [1401.390, 1.39084023521, 265.98929347750],
        [1403.717, 5.63563637532, 4.45341812490],
        [1655.866, 1.96431297431, 79.23501669220],
        [1248.978, 5.44027380866, 54.17467074780],
        [1563.447, 1.47917835549, 112.91463420510],
        [1248.054, 4.88984353601, 479.28838891550],
        [1197.439, 2.52185744943, 145.63104387150],
        [1506.952, 5.24186185583, 181.75834193920],
        [1481.746, 5.66203046912, 152.53214255120],
        [1438.838, 1.53046287618, 447.79581952650],
        [1408.514, 4.41921749601, 462.02291352810],
        [1477.112, 4.32214690647, 256.53994050650],
        [1228.314, 5.97703331040, 59.80374504030],
        [1249.958, 6.24484546141, 160.60889739850],
        [906.468, 5.62025869483, 74.66972398270],
        [1090.681, 4.15393813845, 77.96299230500],
        [844.931, 0.12943398585, 82.85835341460],
        [900.363, 2.37315925843, 74.89347315190],
        [1071.957, 1.74286714339, 528.20649238630],
        [689.708, 3.08097059985, 69.36497259590],
        [593.798, 4.50074517056, 8.07675484730],
        [718.559, 4.00047509264, 128.95626931510],
        [699.574, 0.03987168068, 143.62530630140],
        [575.656, 5.89552672641, 66.70484372000],
        [759.004, 2.13700057433, 692.58748435350],
        [710.449, 5.41605755095, 218.40690486870],
        [548.672, 5.62811496970, 3.18139373770],
        [651.632, 4.42340061551, 18.15924726470],
        [539.825, 6.20788667166, 71.60020482960],
        [544.539, 5.69375108253, 203.73786788240],
        [710.276, 4.21967260022, 381.61224066830],
        [593.819, 3.83805798523, 32.19514480460],
        [710.134, 4.48972171999, 293.18850343600],
        [705.482, 0.45521177725, 835.03713448730],
        [588.000, 5.08252923316, 186.21176006410],
        [598.231, 0.35815291076, 269.92144674060],
        [641.914, 2.71127457036, 87.31177153950],
        [495.621, 2.65094755989, 200.76892246580],
        [630.252, 4.46146214548, 275.55052103310],
        [575.195, 5.57862480486, 2.44768055480],
        [569.870, 1.63930932740, 77.22927912210],
        [556.672, 1.07231961344, 1059.38193018920],
        [449.439, 0.27981733949, 617.80588578620],
        [463.608, 1.43448297993, 297.64192156090],
        [436.547, 0.52802035072, 209.36694217490],
        [463.938, 2.35443114417, 211.81462272970],
        [435.943, 2.10077211065, 1514.29129671650],
        [515.534, 3.23274579379, 284.14854074220],
        [454.879, 4.08364210459, 99.16062095550],
        [477.430, 2.89397217998, 39.35687591520],
        [542.331, 5.39481705077, 278.51946644970],
        [410.087, 3.04968860441, 404.50679034820],
        [367.848, 0.71159607058, 125.98732389850],
        [503.096, 5.83931251717, 191.20769491020],
        [487.532, 0.06402454583, 60.76695288680],
        [455.043, 2.59321186669, 490.33408917940],
        [436.291, 2.08183813746, 51.20572533120],
        [435.803, 2.79445203085, 75.74480641380],
        [323.546, 4.82899980859, 195.13984817330],
        [359.363, 0.00868012078, 35.42472265210],
        [429.314, 3.08031550488, 41.10198105440],
        [320.021, 5.48625497747, 14.97785352700],
        [414.331, 0.09012800478, 258.02441321480],
        [379.715, 0.05832815311, 378.64329525170],
        [420.062, 2.25393983318, 81.00137369080],
        [357.721, 4.71414305625, 173.94221952280],
        [358.922, 0.35213227553, 426.59819087600],
        [405.410, 6.12263257999, 24.37902238820],
        [365.158, 5.59483211224, 255.05546779820],
        [308.102, 3.92355394354, 116.42609634290],
        [325.660, 4.71996698332, 134.58534360760],
        [292.781, 3.99521194830, 72.33391801250],
        [386.543, 0.68619006966, 230.56457082540],
        [305.686, 3.76108783519, 344.70304530790],
        [286.972, 1.84990335310, 153.49535039770],
        [353.640, 4.65717995107, 329.83706636550],
        [302.051, 0.13190003806, 565.11568774670],
        [241.128, 1.60454142389, 81.37388070630],
        [249.829, 4.24205256241, 75.30286342910],
        [245.063, 5.94905404273, 20.60692781950],
        [248.277, 1.06282887181, 105.49227066360],
        [305.353, 2.55534744586, 6208.29425142410],
        [296.328, 4.21100245276, 1364.72809958190],
        [219.938, 2.96119055727, 120.35824960600],
        [233.564, 2.97074409938, 46.20979048510],
        [262.422, 3.83652250971, 831.10498122420],
        [233.546, 4.48117006140, 628.85158605010],
        [187.432, 3.03529190348, 135.54855145410],
        [216.776, 3.42907414802, 241.61027108930],
        [255.760, 1.16707893460, 177.87437278590],
        [220.458, 0.19633492290, 180.27386923090],
        [224.519, 0.40677777819, 114.39910691340],
        [205.398, 2.30380942634, 259.50888592310],
        [211.106, 4.93079982424, 103.09277421860],
        [175.758, 5.50822822216, 7.11354700080],
        [188.512, 2.23588941288, 5.41662597140],
        [171.718, 5.21730232334, 41.64449777560],
        [176.136, 1.95958319897, 756.32338265690],
        [170.447, 4.94978757413, 206.18554843720],
        [169.454, 4.04319823722, 55.65914345610],
        [219.015, 0.24790282027, 294.67297614430],
        [187.768, 2.04538775456, 408.43894361130],
        [182.258, 0.70728384467, 391.17346822390],
        [192.095, 5.76718231319, 291.70403072770],
        [153.684, 4.70659406659, 543.02428721890],
        [170.043, 4.50995820508, 288.08069400530],
        [164.097, 5.22527540372, 67.35923502580],
        [194.341, 6.11690364710, 414.06801790380],
        [168.027, 5.25810639105, 518.64526483070],
        [156.641, 0.66304836778, 220.41264243880],
        [182.330, 0.78383856974, 417.03696332040],
        [167.462, 4.92241597775, 422.66603761290],
        [170.770, 2.30927162659, 98.89998852460],
        [161.678, 3.27259601116, 443.86366626340],
        [132.763, 2.88875442023, 373.90799283650],
        [161.140, 3.82341391177, 451.94042111070],
        [179.292, 4.82405681293, 366.48562929500],
        [178.153, 3.98026039043, 10138.50394764370],
        [141.929, 1.26972581554, 159.12442469020],
        [153.750, 4.27847681414, 45.57665103870],
        [161.513, 4.99545008738, 73.81839072080],
        [146.315, 2.65664902119, 465.95506679120],
        [124.875, 4.30470898895, 339.28641933650],
        [154.620, 4.32046228120, 760.25553592000],
        [142.894, 2.07773752143, 457.87831194390],
        [152.408, 4.64742446768, 155.78297225810],
        [116.389, 4.43513730944, 5.93789083320],
        [113.444, 4.65351596266, 80.19822453870],
        [107.611, 3.77290419929, 142.44965013380],
        [133.740, 5.30894739047, 14.01464568050],
        [116.104, 2.51182725670, 296.15744885260],
        [129.106, 0.36277717661, 96.87299909510],
        [122.766, 2.38341351026, 141.48644228730],
        [101.368, 1.05739625315, 92.30770638560],
        [114.669, 6.24863527978, 767.36908292080],
        [113.283, 0.83051319425, 100.38446123290],
        [107.199, 2.39365512354, 347.88443904560],
        [95.443, 0.80094579583, 342.25536475310],
        [110.789, 0.38651051525, 216.92243216040],
        [126.978, 0.42359358250, 331.32153907380],
        [112.635, 0.08107814739, 558.00214074590],
        [103.166, 0.69792283389, 358.93013930950],
        [111.474, 0.75023459027, 80.71948940050],
        [90.902, 5.16530481614, 144.14657116320],
        [90.677, 0.22036476597, 333.65734504400],
        [98.568, 4.33164222339, 74.52096613640],
        [89.306, 2.18851161761, 74.82978267710],
        [117.216, 3.94965784596, 74.26033370550],
        [89.088, 5.87783179087, 74.73341445750],
        [97.316, 0.69429695020, 977.48678462110],
        [116.587, 1.83677031994, 1289.94650101460],
        [85.449, 5.80255966149, 6.59228213900],
        [86.823, 5.61973473261, 300.61086697750],
        [105.226, 5.94513614941, 328.35259365720],
        [112.117, 1.21168089807, 329.72519178090],
        [82.982, 2.20797412496, 74.94165726170],
        [94.345, 4.53937998713, 28.57180808220],
        [106.847, 1.82071328579, 306.83064210100],
        [103.572, 2.99368274596, 6.21977512350],
        [106.357, 0.81583874750, 1087.69310584050],
        [77.728, 2.73390123734, 110.20632121940],
        [98.405, 3.73478182667, 75.04223099820],
        [86.231, 2.83316881064, 983.11585891360],
        [89.023, 4.73754458960, 604.47256366190],
        [83.013, 1.88273535999, 387.24131496080],
        [90.227, 3.80367274711, 986.08480433020],
        [84.598, 1.25774132938, 142.14083359310],
        [74.690, 1.35097482767, 350.33211960040],
        [95.770, 5.54845504768, 969.62247809490],
        [90.277, 0.36773710508, 0.96320784650],
        [82.748, 5.85590525764, 74.62153987290],
        [75.828, 2.78019216029, 88.11492069160],
        [83.850, 1.84386358668, 227.31374111850],
        [70.705, 4.65567024014, 44.72531777680],
        [71.322, 3.64963906751, 894.84087952760],
        [94.141, 4.98819201726, 403.13419222450],
        [88.966, 4.43895583278, 154.01661525950],
        [79.436, 5.66662613679, 267.47376618580],
        [75.615, 5.40971072536, 50.40257617910],
        [68.583, 4.76679841388, 991.71387862270],
        [65.256, 0.69286370395, 152.74459087230],
        [63.031, 2.89946567712, 79.88940799800],
        [63.878, 0.09820555288, 681.54178408960],
        [80.101, 2.97520561915, 526.72201967800],
        [69.693, 3.95281159807, 187.69623277240],
        [59.492, 3.59642351692, 58.10682401090],
        [59.273, 0.50930692071, 28.31117565130],
        [68.590, 2.41880311530, 235.39049596580],
        [66.007, 5.04558399435, 30.71067209630],
        [70.223, 3.73647415486, 546.95644048200],
        [66.836, 0.85506033017, 522.57741809380],
        [63.027, 0.29269109052, 119.50691634410],
        [62.023, 2.31557510311, 74.03083904190],
        [71.379, 3.16967571102, 23.57587323610],
        [74.827, 5.36812537961, 373.01422095920],
        [64.204, 2.36817149460, 157.63995198190],
        [70.712, 0.55830476304, 92.94084583200],
        [55.762, 5.27011035858, 874.39401040250],
        [75.638, 4.66344127677, 101.86893394120],
        [73.727, 6.20581665991, 312.45971639350],
        [72.940, 0.58406607757, 367.97010200330],
        [53.230, 2.24728742995, 17.52610781830],
        [63.139, 4.59563922296, 67.88049988760],
        [60.550, 0.57591315857, 253.57099508990],
        [52.946, 2.45947017614, 264.50482076920],
        [70.236, 1.51860943454, 552.58551477450],
        [68.624, 2.44507780453, 555.55446019110],
        [62.796, 0.33786296181, 561.18353448360],
        [49.009, 1.09233728279, 19.12245511120],
        [64.636, 5.27469970900, 68.18931642830],
        [62.957, 5.35891188483, 92.04707395470],
        [47.664, 3.90924952181, 192.69216761850],
        [65.279, 4.23629510074, 771.30123618390],
        [65.190, 3.73942854797, 536.80451209540],
        [59.452, 6.10554259948, 365.00115658670],
        [52.153, 1.71734604937, 905.88657979150],
        [46.035, 3.87093684776, 210.33015002140],
        [46.429, 5.97423131576, 477.80391620720],
        [62.115, 2.67544358037, 130.44074202340],
        [46.038, 3.89378239085, 48.75804477640],
        [42.663, 3.81519760715, 61.28821774860],
        [53.909, 2.86457147106, 353.30106501700],
        [46.936, 1.00011046774, 166.82867252200],
        [42.217, 2.61748790314, 90.82323367730],
        [43.324, 4.15777895713, 173.68158709190],
        [41.296, 1.79930408254, 149.45132255000],
        [44.960, 1.76623306927, 0.52126486180],
        [51.904, 2.97773319756, 383.09671337660],
        [42.931, 1.57416456203, 120.99138905240],
        [49.611, 4.03427920470, 303.86169668440],
        [45.263, 3.58382163089, 97.41551581630],
        [38.695, 2.39404211169, 31.49256938900],
        [38.072, 5.79473670350, 75.53235809270],
        [50.126, 4.76412907201, 911.30320576290],
        [50.884, 5.15513957132, 439.78275515400],
        [43.148, 0.84999004804, 58.31927233200],
        [42.732, 5.17318058934, 162.09337010680],
        [50.298, 5.81603435915, 66.91729204110],
        [35.639, 1.87447823723, 472.17484191470],
        [49.963, 1.88943490790, 42.58645376270],
        [39.974, 1.74262050679, 89.75945209430],
        [45.252, 1.92511912328, 55.13787859430],
        [44.896, 1.48355901890, 450.97721326420],
        [34.297, 5.20257496546, 316.39186965660],
        [46.355, 0.33942039181, 273.10284047830],
        [37.152, 2.03757941865, 117.91056905120],
        [46.106, 5.62315633955, 1819.63746610920],
        [39.368, 4.19402806344, 486.40193591630],
        [41.039, 4.82994471947, 149.67507171920],
        [44.959, 0.72694662195, 3265.83082813250],
        [43.617, 0.75332422672, 404.61866493280],
        [31.823, 3.84768075667, 20.44686912510],
        [44.196, 4.36769721266, 418.26080359780],
        [37.900, 3.02928044053, 167.08930495290],
        [43.684, 1.57328182739, 491.55792945680],
        [34.004, 1.26257052908, 260.99335863140],
        [31.276, 4.16123711648, 13.33332212430],
        [39.984, 2.86626125620, 468.24268865160],
        [36.490, 2.58804294589, 68.56182344380],
        [32.364, 3.11577354875, 103.35340664950],
        [33.857, 0.15592410716, 24.11838995730],
        [35.933, 1.36784550071, 59.28248017850],
        [33.633, 0.75501177400, 290.21955801940],
        [29.751, 5.33178627038, 1033.35837639830],
        [32.036, 4.67549858000, 205.22234059070],
        [30.991, 4.62823866461, 258.87574647670],
        [35.268, 1.00718464327, 1108.13997496560],
        [33.366, 3.40738625377, 43.12897048390],
        [32.638, 5.25485850258, 114.13847448250],
        [29.825, 5.64157476876, 254.94359321360],
        [31.613, 3.78231393110, 152.01087768940],
        [30.980, 2.26660677937, 104.00779795530],
        [34.591, 5.17326577255, 25.60286266560],
        [28.398, 1.76872790446, 820.05928096030],
        [27.991, 3.92486885309, 199.28444975750],
        [28.986, 2.58171811759, 76.47851959670],
        [33.772, 5.79359878723, 274.06604832480],
        [29.401, 5.93638676504, 280.96714700450],
        [31.094, 1.39352495971, 178.78939652260],
        [30.118, 0.44367887423, 27.08733537390],
        [33.820, 6.26168443513, 401.64971951620],
        [27.513, 2.15194454461, 480.77286162380],
        [26.880, 2.51300272780, 123.53964334370],
        [26.139, 0.21985367371, 286.59622129700],
        [26.455, 3.88229792258, 372.42352012820],
        [33.974, 1.44637843871, 88.79624424780],
        [30.107, 0.82723915882, 100.64509366380],
        [27.715, 4.64827434185, 198.32124191100],
        [33.687, 1.14348201049, 82.48584639910],
        [26.493, 1.97889544238, 95.38852638680],
        [24.355, 2.37839176150, 146.38180339690],
        [26.590, 0.39881920389, 106.01353552540],
        [27.006, 2.10206230691, 1057.89745748090],
        [23.976, 6.21233637686, 16.67477455640],
        [30.970, 5.34005431547, 476.43131808350],
        [24.073, 3.42953641968, 1044.40407666220],
        [27.023, 0.71284764471, 248.72381809010],
        [29.098, 3.99184722502, 908.33426034630],
        [22.862, 2.26978781393, 175.16605980020],
        [24.026, 0.36584131268, 73.18525127440],
        [28.024, 3.46485782266, 1439.50969814920],
        [22.034, 0.05163807300, 33.13710079170],
        [22.185, 5.32252126255, 483.22054217860],
        [21.027, 0.37224660652, 214.78356814630],
        [20.548, 1.80004483299, 118.02244363580],
        [27.835, 4.12412553530, 694.07195706180],
        [25.500, 5.49632191634, 115.88357962170],
        [21.377, 3.89179204956, 66.18357885820],
        [27.201, 5.76148797900, 1215.16490244730],
        [24.984, 0.65339418015, 132.88842257820],
        [23.976, 4.56161326826, 458.84151979040],
        [21.116, 1.13610706250, 60.55450456570],
        [26.263, 2.77532723118, 490.07345674850],
        [26.369, 3.37120039300, 49.72125262290],
        [22.870, 4.53135637620, 78.40493528970],
        [26.872, 3.26037129303, 691.10301164520],
        [25.004, 3.65018677651, 73.40900044360],
        [20.874, 3.92589972978, 134.06407874580],
        [20.915, 5.53955400138, 129.91947716160],
        [23.067, 2.56806856688, 332.80601178210],
        [22.630, 5.02721554401, 150.52640498110],
        [19.123, 1.92386327535, 124.50285119020],
        [20.678, 0.98302410602, 29.20494752860],
        [18.755, 1.07911898422, 70.11573212130],
        [19.458, 1.33847349577, 616.32141307790],
        [23.071, 3.93152899657, 43.28902917830],
        [23.313, 0.61185525008, 189.72322220190],
        [19.660, 1.40884902649, 1589.07289528380],
        [24.990, 0.91842956919, 441.26722786230],
        [23.555, 0.02127675886, 593.42686339800],
        [18.288, 4.55111843462, 165.60483224460],
        [20.980, 0.88504201898, 326.86812094890],
        [24.940, 4.63470443286, 162.89651925890],
        [18.941, 5.10763304564, 81.89514556810],
        [18.911, 1.23351635328, 13.49338081870],
        [17.358, 4.05768226252, 403.02231763990],
        [17.362, 5.28607227640, 7.86430652620],
        [22.513, 3.15059891398, 419.74527630610],
        [21.237, 2.14856256664, 75.58474771940],
        [17.845, 2.54349200329, 47.06112374700],
        [16.995, 2.48647736969, 2043.98226181110],
        [23.676, 5.80355919955, 232.04904353370],
        [22.639, 2.07623129509, 699.70103135430],
        [19.261, 1.56494156016, 425.11371816770],
        [21.067, 5.30844438236, 237.67811782620],
        [22.733, 0.28303126440, 0.11187458460],
        [16.372, 3.45984005656, 0.75075952540],
        [21.213, 0.95828006612, 405.99126305650],
        [18.033, 1.60723214246, 215.43795945210],
        [16.267, 4.89002016360, 69.15252427480],
        [21.738, 3.24738839789, 1744.85586754190],
        [16.149, 0.35803995032, 77.06922042770],
        [21.710, 0.88800040769, 344.96367773880],
        [17.204, 6.04366142241, 32.24332891440],
        [17.883, 4.01076173641, 280.00393915800],
        [15.918, 2.96623390816, 25.86349509650],
        [14.769, 3.73887340623, 610.69233878540],
        [15.033, 4.24825484707, 228.27694896500],
        [15.586, 5.07987082740, 114.94162363460],
        [15.392, 0.22971106129, 17.26547538740],
        [15.354, 0.25482391126, 661.09491496450],
        [14.617, 1.13349626273, 823.99143422340],
        [16.232, 3.43499743810, 147.11551657980],
        [14.654, 1.68288566884, 207.88246946660],
        [17.682, 5.94376629143, 624.91943278700],
        [18.837, 1.38335408070, 377.15882254340],
        [15.425, 1.66489033237, 440.68227252570],
        [14.764, 4.41710614445, 16.46232623530],
        [14.402, 0.41359448817, 142.66209845490],
        [16.992, 0.16042368544, 438.29828244570],
        [13.268, 3.04634728126, 668.20846196530],
        [16.460, 0.92068542861, 369.08206769610],
        [17.239, 4.51659246818, 606.76018552230],
        [13.238, 0.13650358961, 216.48048917570],
        [15.832, 4.94315562971, 124.29040286910],
        [14.374, 2.93700606008, 419.48464387520],
        [12.927, 1.65950183061, 54.33472944220],
        [14.224, 4.42286781619, 47.69426319340],
        [12.753, 0.03020931725, 217.23124870110],
        [14.792, 1.08447500622, 49.50880430180],
        [14.031, 3.68785757687, 16.04163511000],
        [13.709, 4.78890618802, 72.77586099720],
        [13.073, 1.54064778942, 218.92816973050],
        [17.474, 5.05621281434, 564.85505531580],
        [12.686, 3.44640888880, 958.57677783100],
        [13.035, 0.56445754615, 1171.87587326900],
        [12.458, 3.29187197133, 902.70518605380],
        [11.893, 1.41294011193, 55.77101804070],
        [15.018, 3.43209569509, 19.01058052660],
        [16.470, 2.04067754807, 411.62033734900],
        [15.619, 1.53464600544, 833.55266177900],
        [15.678, 5.92839374034, 778.41478318470],
        [12.039, 5.17748353434, 135.33610313300],
        [15.523, 3.54656631824, 113.87784205160],
        [14.364, 4.19825110964, 89.33876096900],
        [15.424, 2.12697366269, 106.27416795630],
        [11.957, 1.43314130608, 455.87257437380],
        [15.938, 5.49575810978, 513.07988101300],
        [13.532, 4.11463529983, 95.22846769240],
        [15.105, 1.86350524526, 7.70424783180],
        [15.832, 3.42498484109, 79.51690098250],
        [11.492, 4.65187455620, 149.61138124440],
        [11.406, 1.31085455047, 63.62402371880],
        [14.469, 3.35284802718, 19.64371997300],
        [11.953, 0.20979051344, 65.87476231750],
        [12.039, 0.01423238410, 397.39324334740],
        [14.157, 1.87440535404, 6283.07584999140],
        [11.357, 0.19079103112, 5.62907429250],
        [14.109, 0.09348109701, 6133.51265285680],
        [15.322, 3.54468546172, 252.65597135320],
        [11.681, 0.85100356112, 5.10780943070],
        [14.134, 5.66340426198, 639.89728631400],
        [11.052, 0.47607339302, 150.08446199640],
        [11.507, 5.19480309409, 1182.92157353290],
        [11.492, 2.05801478181, 149.51501302480],
        [11.571, 4.78210724970, 334.29048449040],
        [10.671, 4.67373109923, 149.72325582900],
        [11.651, 3.13272450186, 93.90405367850],
        [14.316, 0.08421279341, 240.38643081190],
        [10.855, 4.52379396618, 453.94615868080],
        [11.900, 1.41784572428, 26.02355379090],
        [10.851, 4.40625021974, 57.14361616440],
        [13.385, 0.76174742916, 37.87240320690],
        [10.664, 5.81644528276, 193.65537546500],
        [10.700, 5.34595506070, 331.20966448920],
        [10.465, 3.82648204886, 180.16199464630],
        [13.350, 0.86920479636, 22.89454967990],
        [10.324, 2.99969783109, 525.75881183150],
        [14.293, 1.06904465002, 477.91579079180],
        [12.341, 4.62813430535, 1894.41906467650],
        [12.539, 3.70404881494, 67.07735073550],
        [11.771, 1.07971321862, 363.51668387840],
        [11.466, 0.93528386040, 121.84272231430],
        [12.839, 0.31988787839, 474.94684537520],
        [10.194, 6.23976471898, 84.18276742850],
        [12.300, 2.85238700423, 184.09414790940],
        [13.861, 3.48367688770, 157.26744496640],
        [11.395, 4.25533440680, 181.05576652360],
        [10.146, 6.01371693363, 43.24084506850],
        [9.889, 0.01753733887, 40.16002506730],
        [10.798, 2.46364243940, 140.65636088480],
        [9.893, 2.72219544398, 384.05992122310],
        [11.012, 3.77284307154, 494.26624244250],
        [10.226, 1.49169427598, 80.41067285980],
        [11.981, 2.69741212203, 369.45457471160],
        [10.658, 1.78702880970, 252.08652238160],
        [12.506, 4.66852994807, 64.69910614990],
        [11.640, 5.70405852396, 39.09624348430],
        [12.305, 1.73322306013, 229.08009811710],
        [9.954, 3.72074935511, 233.90602325750],
        [11.004, 3.58723041577, 449.28029223480],
        [10.581, 2.79550711884, 1246.65747183630],
        [10.411, 2.86727145530, 189.18070548070],
        [9.259, 5.19834823120, 749.20983565610],
        [12.468, 0.76477162698, 122.47586176070],
        [10.099, 6.06894682979, 156.15547927360],
        [12.671, 6.19797171716, 149.82382956550],
        [9.053, 0.99447742265, 109.31254934210],
        [9.595, 1.00827882958, 393.46109008430],
        [10.455, 1.23531019351, 148.59998928810],
        [10.645, 5.50399216121, 460.53844081980],
        [12.433, 5.29843661600, 20.49505323490],
        [9.002, 5.51216892840, 133.10087089930],
        [9.882, 5.68987366321, 42.53826965290],
        [12.110, 3.14577799081, 30.05628079050],
        [10.380, 3.54528360301, 619.29035849450],
        [10.139, 1.90528799801, 25.06034594440],
        [11.206, 5.87823238990, 832.58945393250],
        [9.283, 3.00514787072, 754.83890994860],
        [10.994, 0.05721003392, 54.28654533240],
        [8.981, 5.82462023723, 248.46318565920],
        [8.634, 5.49123270314, 448.68959140380],
        [9.390, 1.32674472283, 9.40116886120],
        [9.621, 5.69090299390, 73.88782669000],
        [10.843, 1.42812648220, 268.43697403230],
        [10.874, 2.61361741990, 446.31134681820],
        [9.681, 4.27051176079, 282.66406803390],
        [10.770, 0.19304986906, 463.50738623640],
        [8.223, 3.30114329151, 172.19711438360],
        [10.216, 5.04428111805, 241.87090352020],
        [8.890, 2.53320903906, 271.40591944890],
        [10.621, 4.39013117792, 63.84777288800],
        [9.085, 0.99085954236, 6.90109867970],
        [10.414, 4.96367476854, 97.67614824720],
        [8.382, 5.03764591595, 370.93904741990],
        [8.791, 3.05426995163, 291.26208774300],
        [8.629, 0.14674938050, 262.47783133970],
        [9.673, 4.25597330570, 602.98809095360],
        [7.939, 5.71230451368, 541.53981451060],
        [7.981, 2.35900017752, 196.62432088160],
        [7.941, 0.88239951788, 154.97982310600],
        [8.212, 0.00845991197, 76.42612997000],
        [10.135, 1.90258069764, 91.45637312370],
        [8.948, 4.31891786278, 469.13646052890],
        [9.906, 0.61122653279, 308.31511480930],
        [8.257, 4.61012292958, 69.67378913660],
        [10.291, 3.58217488981, 842.15068148810],
        [10.672, 2.28920805112, 194.28851491140],
        [9.024, 0.70282370018, 685.47393735270],
        [7.552, 4.86800510978, 1097.09427470170],
        [9.496, 1.06662350720, 93.79217909390],
        [8.413, 3.15290837718, 32.71640966640],
        [8.914, 5.03579282562, 450.45594840240],
        [8.008, 4.33420849497, 302.09533968580],
        [10.397, 4.90564475822, 829.62050851590],
        [7.401, 5.67595616187, 337.80194662820],
        [7.419, 3.04672439120, 7.42236354150],
        [7.936, 4.37116642726, 464.47059408290],
        [9.731, 5.70393303400, 98.35747180340],
        [9.287, 4.16913084905, 15.49911838880],
        [10.072, 1.18963356664, 621.73803904930],
        [9.867, 1.34469368253, 142.97091499560],
        [8.913, 1.33679256423, 0.26063243090],
        [8.734, 2.03651443558, 149.40313844020],
        [7.745, 0.14834225031, 1404.08497549710],
        [9.037, 3.53203542312, 636.66770846650],
        [7.707, 1.87083579542, 31.65262808340],
        [8.566, 2.48280357910, 497.44763618020],
        [10.074, 2.59066128203, 711.44930703380],
        [8.726, 5.88905195509, 82.20396210880],
        [9.218, 1.10861658464, 412.58354519550],
        [9.483, 6.02304095716, 916.93228005540],
        [7.012, 6.27635752748, 376.19561469690],
        [9.867, 1.93140204720, 62.77269045690],
        [7.121, 1.54192963162, 679.25416222920],
        [8.440, 5.06400284478, 1.37259812370],
        [8.915, 2.39052372377, 76.15419669100],
        [9.637, 3.78128664809, 838.21852822500],
        [7.059, 5.91499106809, 74.14845912090],
        [9.570, 1.97721363991, 703.63318461740],
        [7.516, 4.87738017916, 310.17209453310],
        [7.059, 1.94098053303, 75.41473801370],
        [8.552, 6.14581140636, 17.63798240290],
        [6.859, 5.11679089849, 107.49800823370],
        [7.148, 1.73466140387, 1190.78588005910],
        [9.112, 0.88368663290, 362.86229257260],
        [9.300, 1.44257902224, 763.43692965770],
        [9.073, 4.31998777457, 16.15350969460],
        [9.126, 3.74417347717, 4.66586644600],
        [7.869, 4.65596954763, 232.42155054920],
        [9.168, 3.25096522859, 155.50108796780],
        [7.624, 0.88232424215, 459.36278465220],
        [8.200, 1.51866334747, 10063.72234907640],
        [8.579, 2.37726234500, 75.67537044460],
        [7.595, 2.63499505823, 657.16276170140],
        [8.131, 6.15861249482, 745.27768239300],
        [6.398, 0.61376490225, 73.24894174920],
        [7.710, 4.02552779925, 4.73530241520],
        [6.380, 3.20688120531, 73.34530996880],
        [7.505, 0.67397826037, 228.79821382680],
        [7.129, 1.33525552417, 236.87496867410],
        [7.444, 3.05581518163, 171.65459766240],
        [6.361, 6.05108999867, 95.97922721780],
        [7.086, 4.88497877319, 6531.66165626500],
        [6.527, 4.01380149030, 118.87377689770],
        [7.984, 1.70695215254, 104.52906281710],
        [6.239, 1.08160874504, 143.93412284210],
        [7.200, 1.19830150903, 1617.38407093510],
        [6.390, 4.83649966441, 1072.71525231350],
        [7.857, 3.06062400692, 341.99473232220],
        [6.158, 0.04340975323, 627.36711334180],
        [8.664, 5.60425824325, 2810.92146160520],
        [6.147, 4.16482048084, 1300.99220127850],
        [8.148, 3.11700641910, 10213.28554621100],
        [8.603, 3.80404544682, 25558.21217647960],
        [7.586, 2.86885781812, 406.10313764110],
        [7.925, 3.11650504000, 81.68269724700],
        [7.063, 5.35078594952, 73.03649342810],
        [6.265, 3.27996815317, 116.53797092750],
        [7.345, 0.45699353973, 192.80404220310],
        [6.878, 1.29690429239, 22.63391724900],
        [6.102, 2.06174377751, 73.97844941520],
        [7.867, 1.45455437627, 131.92521473170],
        [6.837, 0.07237438837, 90.28071695610],
        [7.740, 2.87078307084, 79.44746501330],
        [6.744, 0.01897075429, 572.22923474750],
        [6.034, 0.96635268225, 476.31944349890],
        [8.276, 2.40596529645, 674.80074410430],
        [6.002, 3.24986200464, 76.78733613740],
        [6.863, 1.24850658686, 400.16524680790],
        [6.375, 1.37050567525, 75.15410558280],
        [7.445, 5.47946546419, 50.66320861000],
        [5.889, 5.83364715391, 164.12035953630],
        [7.190, 3.21396813566, 71.15826184490],
        [6.305, 1.65443603478, 70.04629615210],
        [6.628, 1.99043744200, 1.59634729290],
        [6.081, 1.26599890530, 61.44827644300],
        [7.795, 0.70956881527, 44.07092647100],
        [7.784, 5.47416712189, 416.77633088950],
        [5.727, 2.39389303582, 20277.00789528740],
        [6.363, 1.96472326808, 288.73508531110],
        [5.724, 4.30532783970, 86.63044798330],
        [7.348, 5.28008775997, 285.63301345050],
        [5.693, 1.13590287643, 445.34813897170],
        [5.799, 2.43420435064, 180.79513409270],
        [6.927, 4.19297070447, 525.23754696970],
        [6.143, 2.23902465258, 1310.39337013970],
        [6.680, 5.36285833774, 452.46168597250],
        [5.709, 3.65303501856, 442.37919355510],
        [6.238, 4.60538870228, 137.55428902420],
        [5.932, 3.22993254909, 73.45718455340],
        [7.734, 5.72805659987, 154.29849954980],
        [5.782, 2.17577477697, 2.28762186040],
        [6.016, 3.46585546590, 346.39996633730],
        [7.517, 5.63641104589, 549.72844394250],
        [6.685, 4.84971092735, 148.81243760920],
        [5.797, 0.51568102762, 149.30256470370],
        [7.739, 3.23104533276, 589.49471013490],
        [7.590, 2.49504939023, 321.76031151820],
        [6.706, 5.23367324742, 769.81676347560],
        [7.210, 2.56243122515, 375.67434983510],
        [5.491, 1.84139760824, 375.39246554480],
        [6.592, 3.39703193659, 389.68899551560],
        [6.414, 3.56513278405, 488.84961647110],
        [6.376, 0.24081237769, 881.50755740330],
        [5.785, 3.49508162978, 102.52332524700],
        [6.079, 0.84517404881, 89.59939339990],
        [5.799, 2.19388408812, 8.90683624980],
        [5.305, 1.97785611200, 150.31395666000],
        [5.422, 1.21990761182, 332.17287233570],
        [7.235, 2.78896876234, 748.09786996330],
        [6.360, 0.36304095923, 74.40909155180],
        [5.285, 4.41108441808, 12.00890811040],
        [6.602, 3.01699401726, 442.75170057060],
        [6.509, 0.49819168917, 1147.49685088080],
        [5.585, 6.04761249158, 172.45774681450],
        [6.673, 0.77536194908, 6069.77675455340],
        [7.125, 0.34356180793, 511.59540830470],
        [5.288, 4.27462942653, 11.15757484850],
        [6.767, 2.00969662613, 105.38039607900],
        [5.611, 2.40776057824, 1286.01434775150],
        [6.456, 4.48699081452, 31.23193695810],
        [5.898, 5.26174074234, 757.80785536520],
        [5.153, 2.41386832919, 742.99006053260],
        [5.087, 6.03592089039, 980.66817835880],
        [6.198, 0.83056505252, 1507.17774971570],
        [5.425, 1.93107713343, 40.84134862350],
        [6.091, 5.18564204379, 487.10451133190],
        [5.715, 1.96802719384, 394.35486196160],
        [5.509, 1.31275092080, 883.79517926370],
        [5.320, 4.22718652038, 65.38042970610],
        [5.357, 1.80483136985, 139.48070471720],
        [5.204, 3.39869589191, 1400.15282223400],
        [6.537, 2.31923989568, 328.24071907260],
        [5.041, 2.94673346440, 361.37781986430],
        [4.969, 6.27367198215, 0.16005869440],
        [5.334, 2.78985718428, 217.44369702220],
        [5.654, 0.51056760715, 285.11174858870],
        [6.432, 2.21948959433, 9999.98645077300],
        [5.768, 5.10735836078, 216.26804085460],
        [4.977, 2.62435916254, 194.17664032680],
        [6.932, 1.71722863424, 378.90392768260],
        [5.187, 3.04429850681, 1083.76095257740],
        [5.791, 3.94061476250, 550.13783421970],
        [5.816, 2.24843661305, 230.93707784090],
        [5.319, 0.06998825350, 336.83873878170],
        [5.427, 0.30577275388, 40.58071619260],
        [6.469, 3.02579309025, 298.23262239190],
        [4.974, 1.21594265105, 455.06942522170],
        [4.960, 4.72806210230, 454.74930783290],
        [5.619, 2.27500303004, 227.52618943960],
        [6.328, 0.97544932086, 249.94765836750],
        [5.319, 4.59867974067, 454.79749194270],
        [4.791, 4.40360629153, 853.19638175200],
        [6.519, 3.03043401282, 167.72244439930],
        [5.450, 6.00971547441, 25.12978191360],
        [5.094, 3.96693309189, 1066.49547719000],
        [5.779, 0.65954416303, 272.58157561650],
        [6.491, 4.68529651540, 312.19908396260],
        [5.234, 4.34712255335, 233.53351624200],
        [5.396, 5.62885554221, 418.52143602870],
        [5.048, 2.46802064424, 987.56927703850],
        [6.152, 0.79853332272, 2274.54683263650],
        [6.506, 1.72915575120, 125.18417474640],
        [4.993, 3.75975860404, 57.25549074900],
        [6.295, 0.84778953014, 10.08249241740],
        [6.251, 4.78782138567, 270.18207917150],
        [5.785, 4.31237764709, 374.49869366750],
        [5.406, 5.49902863401, 632.78373931320],
        [6.224, 0.12733845417, 149.04193227280],
        [4.921, 1.43037646364, 73.13706716460],
        [5.076, 1.34845106372, 455.02124111190],
        [5.730, 3.30386575867, 88.27497938600],
        [4.618, 0.64720625124, 119.39504175950],
        [6.213, 2.58827934841, 544.50875992720],
        [4.825, 6.08615765986, 304.12232911530],
        [4.825, 6.00483903794, 226.79247625670],
        [5.528, 5.69752791882, 548.44091319030],
        [5.108, 2.74489127167, 423.62924545940],
        [5.426, 2.44835106987, 531.97858695500],
        [4.573, 1.61098293427, 357.44566660120],
        [4.487, 6.09067660554, 204.70107572890],
        [5.866, 6.20513223441, 772.78570889220],
        [5.334, 2.49860553733, 1131.19458333990],
        [5.656, 4.75744184558, 491.81856188770],
        [4.441, 0.23590452375, 35.68535508300],
        [4.370, 3.81136490830, 1329.30337692980],
        [4.406, 3.42865095493, 144.89733068860],
        [5.251, 4.72114047741, 535.32003938710],
        [5.174, 1.37807596858, 520.12973753900],
        [4.331, 2.64717426456, 1517.26024213310],
        [4.802, 2.60569463520, 177.30492381430],
        [4.368, 3.36272561974, 1503.24559645260],
        [4.335, 2.73379207096, 289.56516671360],
        [5.198, 0.97116582962, 128.43500445330],
        [4.437, 0.56678131875, 253.45912050530],
        [5.386, 5.84886051674, 268.69760646320],
        [5.376, 1.30096148962, 436.81380973740],
        [5.797, 4.33049740199, 208.84567731310],
        [4.353, 1.66111524192, 1261.63532536330],
        [4.812, 4.95769337401, 545.47196777370],
        [5.897, 2.04201205180, 8.59801970910],
        [5.032, 2.80550759770, 360.41461201780],
        [4.921, 2.55658380096, 260.36021918500],
        [4.413, 3.23825819993, 973.55463135800],
        [4.506, 0.17509624151, 380.38840039090],
        [4.947, 5.50324549675, 365.90067395840],
        [4.174, 2.99974290843, 136.06981631590],
        [4.553, 2.77416673233, 147.96684984170],
        [5.344, 1.81213470593, 521.09294538550],
        [5.155, 0.78324341489, 1670.07426897460],
        [5.133, 1.37435234967, 271.61836777000],
        [5.708, 2.52872222038, 501.37978944330],
        [4.933, 1.36454104948, 238.90195810360],
        [4.973, 1.92960964594, 535.91074021810],
        [4.935, 5.04375067678, 697.80716836880],
        [4.129, 1.50064332826, 71.86083726050],
        [5.207, 4.01877367340, 92.41958097020],
        [4.587, 4.78553156868, 95.93104310800],
        [5.050, 5.41251268131, 758.77106321170],
        [5.012, 4.50266403888, 635.96513305090],
        [4.248, 0.65406962267, 920.86443331850],
        [5.150, 1.13490701556, 310.97524368520],
        [4.146, 5.54040372231, 1048.33622992530],
        [4.256, 4.20942901957, 25.27279426550],
        [4.569, 5.19758291396, 10.29494073850],
        [4.259, 5.53202386861, 184.98791978670],
        [4.315, 2.80569687202, 213.95348674380],
        [4.604, 2.51643176466, 962.50893109410],
        [5.105, 1.26007002216, 971.10695080320],
        [4.500, 6.15796742231, 1052.26838318840],
        [4.095, 0.63467124507, 1321.43907040360],
        [3.968, 0.07377679014, 77.70235987410],
        [4.617, 2.77367751889, 406.95447090300],
        [4.899, 4.65767840428, 305.60680182360],
        [3.900, 1.66467970991, 945.24345570670],
        [4.133, 3.76396043787, 263.02034806090],
        [3.899, 4.28677450975, 224.23292111730],
        [5.030, 6.24112139981, 1162.47470440780],
        [5.024, 0.39738855487, 968.13800538660],
        [4.894, 2.50422546622, 355.74874557180],
        [4.283, 3.15267059582, 846.08283475120],
        [3.941, 0.04342429962, 1235.61177157240],
        [4.077, 5.68854469970, 695.55642977010],
        [3.842, 0.37429373422, 774.48262992160],
        [4.674, 0.08112657673, 1366.21257229020],
        [4.671, 4.38923533828, 117.36805233000],
        [5.313, 4.38472090135, 689.61853893690],
        [3.787, 1.41443617212, 48.91810347080],
        [4.236, 1.60316940746, 367.59759498780],
        [4.569, 2.88138923862, 551.10104206620],
        [4.636, 5.93442268083, 148.19059901090],
        [5.128, 0.17600225009, 433.71173787680],
        [4.264, 2.08657038625, 325.38364824060],
        [3.885, 5.85359840623, 450.71658083330],
        [4.753, 2.58442943928, 358.40887444770],
        [4.226, 6.24596640453, 448.97147569410],
        [3.776, 1.78756451192, 71.70077856610],
        [4.912, 4.45665056284, 51.88704888740],
        [3.854, 2.72138633161, 151.85081899500],
        [4.561, 0.07201979569, 2349.32843120380],
        [4.291, 5.39929339966, 523.75307426140],
        [4.143, 0.17158866270, 735.87651353180],
        [3.806, 1.44358694049, 138.62937145530],
        [3.654, 2.41520715554, 348.84764689210],
        [3.728, 1.69745141654, 984.71220620650],
        [4.176, 4.01139155515, 195.77298761970],
        [4.986, 1.03562905920, 224.45667028650],
        [4.031, 0.92145122185, 76.00543884470],
        [4.098, 3.51214223942, 72.49397670690],
        [3.812, 4.41246815759, 1511.32235129990],
        [4.098, 2.39702785276, 239.16259053450],
        [4.894, 5.26621064696, 601.50361824530],
        [4.459, 5.76440378473, 836.52160719560],
        [4.373, 4.08948598951, 75.43598987310],
        [4.363, 6.01127167247, 421.18156490460],
        [4.414, 0.35830661600, 168.31314523030],
        [3.700, 2.04103813925, 63.21463344160],
        [4.648, 2.07482117651, 1106.65550225730],
        [4.304, 3.03122452327, 1109.62444767390],
        [3.618, 4.84400847177, 893.35640681930],
        [4.023, 0.59685650686, 91.24392480260],
        [4.937, 5.49417275871, 976.73602509570],
        [4.373, 3.76648561161, 74.12720726150],
        [4.310, 3.72983787822, 673.31627139600],
        [4.680, 3.89631254150, 163.57784281510],
        [4.172, 0.75349427039, 1500.06420271490],
        [4.492, 1.87283145714, 141.17762574660],
        [4.174, 5.82910805335, 346.44815044710],
        [3.775, 0.83052387256, 827.92358748650],
        [3.481, 2.70828672792, 818.57480825200],
        [4.853, 0.95917381603, 58.17051448570],
        [4.021, 3.11274455034, 377.41945497430],
        [4.409, 0.16607520728, 630.33605875840],
        [3.597, 1.02560564654, 515.46387109300],
        [3.444, 1.38488805947, 117.31986822020],
        [4.021, 5.68447974866, 3.49021027840],
        [4.244, 3.75845344717, 733.42883297700],
        [3.933, 4.55157642432, 240.12579838100],
        [4.421, 1.51263319894, 1610.27052393430],
        [3.390, 4.00215380112, 74.99404688840],
        [4.556, 3.41531360529, 1140.38330388000],
        [4.171, 0.76417016678, 623.43496007870],
        [3.441, 2.56450835637, 14.81779483260],
        [3.587, 4.10186965494, 343.21857259960],
        [3.997, 5.74857613262, 6212.22640468720],
        [4.215, 0.84469743228, 176.65053250850],
        [4.098, 1.41920746453, 559.69906177530],
        [4.553, 1.09016692751, 561.88610989920],
        [3.493, 0.11837510368, 1031.87390369000],
        [3.596, 5.33968666729, 394.94556279260],
        [3.501, 5.53309359866, 594.91133610630],
        [3.564, 1.57868308864, 354.99798604640],
        [4.279, 2.35436288262, 562.66800719190],
        [3.398, 1.62086348540, 941.31130244360],
        [3.803, 1.78948693511, 251.17149864490],
        [3.299, 5.13631478768, 477.00076705510],
        [3.422, 3.55674695079, 256.42806592190],
        [3.802, 4.09287840097, 268.95823889410],
        [3.818, 4.64162443046, 71.92452773530],
        [4.623, 2.72836206211, 6244.94281435360],
        [3.466, 1.91387688001, 58.73996345730],
        [3.435, 2.96178782926, 995.64603188580],
        [3.626, 5.35614681493, 57.51612317990],
        [4.512, 1.18350543284, 170.76082578510],
        [4.378, 0.27346037700, 469.72716135990],
        [3.323, 4.51516827363, 454.86118241750],
        [3.320, 1.42938752044, 454.95755063710],
        [4.578, 4.74980514730, 731.94436026870],
        [3.249, 0.67719975914, 74.04788538440],
        [4.145, 5.58064267022, 57.79800747020],
        [3.512, 6.09122971288, 70.58881287330],
        [3.814, 2.49565462974, 6204.36209816100],
        [3.828, 4.39751907192, 586.31331639720],
        [3.639, 4.85097208169, 138.40562228610],
        [3.518, 0.52105043625, 262.80789973980],
        [3.671, 1.99667387765, 511.53171782990],
        [3.215, 0.64628330219, 887.72733252680],
        [3.718, 3.27473813045, 454.64873409640],
        [3.638, 2.63250736806, 455.16999895820],
        [3.772, 0.88810300052, 10142.43610090680],
        [3.190, 4.87960158471, 455.66012605270],
        [3.669, 4.11456655271, 409.92341631960],
        [3.166, 1.24948126394, 82.64590509350],
        [3.530, 4.02075420346, 388.20452280730],
        [3.163, 1.65294183878, 765.88461021250],
        [3.568, 5.95965909592, 460.84725736050],
        [3.450, 0.08821515281, 49.17873590170],
        [3.270, 1.81146731641, 34.20088237470],
        [3.188, 1.74587038709, 18.91000679010],
        [3.305, 5.77382040863, 10.52443540210],
        [3.345, 4.15802505352, 1515.77576942480],
        [3.160, 1.06549289762, 454.15860700190],
        [3.525, 2.56091667232, 78114.14622758799],
        [4.124, 1.06751791085, 388.46515523820],
        [4.016, 2.48751669586, 531.17543780290],
        [3.147, 5.05814757549, 1521.40484371730],
        [4.180, 1.16833674781, 514.56435372130],
        [3.212, 3.18682610058, 1512.80682400820],
        [3.486, 0.22630227172, 36.53668834490],
        [4.211, 3.21876950029, 761.74000862830],
        [3.485, 3.26177495276, 36.76043751410],
        [3.506, 6.26633354904, 545.27502581760],
        [3.733, 2.55776517455, 279.48267429620],
        [4.227, 1.41381723926, 41.75637236020],
        [3.187, 1.62296832026, 138.46931276090],
        [3.934, 2.86731965547, 832.06818907070],
        [3.684, 3.94174060300, 179.31066138440],
        [3.115, 5.67364420834, 73.55775828990],
        [3.663, 1.80556740809, 31.54075349880],
        [3.171, 4.85529878165, 138.56568098050],
        [3.777, 5.94890597104, 873.17017012510],
        [4.357, 4.15105623366, 10175.15251057320],
        [3.951, 1.60185888278, 576.16138801060],
        [3.525, 4.80316435970, 429.77958461370],
        [3.330, 5.62171319933, 1116.00428149180],
        [3.943, 4.62641543020, 898.77303279070],
        [3.382, 1.45717307307, 5983.94945572220],
        [3.215, 3.73878297941, 335.77495719870],
        [3.423, 5.07987216951, 143.34342201110],
        [4.161, 5.39091883238, 1363.24362687360],
        [3.457, 3.94796907904, 444.82687410990],
        [3.593, 3.91831549069, 10134.57179438060],
        [3.666, 4.22620338990, 36.17548217750],
        [3.580, 3.43130119859, 912.78767847120],
        [3.700, 0.67445695843, 73.93026530540],
        [3.679, 1.10949079061, 686.95841006100],
        [3.547, 5.63096398237, 440.89472084680],
        [3.622, 1.10742531477, 2250.16781024830],
        [3.562, 3.80604468765, 1525.33699698040],
        [3.330, 1.76480149289, 78.97438426130],
        [3.738, 1.76754910180, 384.58118608490],
        [3.986, 1.06842874470, 743.79320968470],
        [3.032, 5.77852412826, 612.17681149370],
        [3.008, 0.64086342534, 210.85141488320],
        [3.334, 4.81681647959, 597.35901666110],
        [3.141, 3.11768616608, 6607.92772754060],
        [3.022, 2.08709314702, 34.53095077480],
        [3.226, 3.19780030434, 377.68008740520],
        [4.065, 3.53637930424, 402.21916848780],
        [4.138, 0.28927701421, 517.16079212240],
        [3.697, 0.89932694516, 75.63293182920],
        [3.918, 5.73859894835, 94.42531854030],
        [3.374, 0.40974405580, 677.76968952090],
        [3.194, 4.56998602897, 1385.17496870700],
        [3.180, 1.03370427552, 885.43971066640],
        [3.235, 5.09681747179, 464.99185894470],
        [3.798, 5.76464888795, 586.37700687200],
        [3.173, 5.68964342749, 4.19278569400],
        [3.181, 2.87968862974, 9914.15915194180],
        [3.355, 5.45857968674, 73.08467753790],
        [2.899, 5.12928266291, 448.31708438830],
        [3.706, 2.47342147635, 64.25716316520],
        [3.796, 4.29502458131, 164.54105066160],
        [3.534, 4.42464754991, 46.47042291600],
        [3.488, 5.62713714766, 3189.56475685690],
        [2.940, 4.33606107945, 78263.70942472259],
        [3.309, 3.10770680369, 519.60847267720],
        [4.030, 4.41794838679, 772.58876693610],
        [2.867, 5.17129632099, 346.18751801620],
        [3.842, 1.87994191354, 299.71709510020],
        [3.846, 5.38315286213, 980.14691349700],
        [3.724, 4.94511644698, 984.48845703730],
        [2.814, 2.17260061398, 191.31956949480],
        [3.392, 3.08552116090, 245.49424024260],
        [2.923, 0.37358823115, 6.48040755440],
        [3.025, 1.19297242418, 104.83787935780],
        [3.470, 1.89084704021, 44.61344319220],
        [2.931, 0.78809830626, 540.05534180230],
        [3.707, 4.14868763219, 6136.48159827340],
        [3.931, 5.52289695589, 6171.64568849460],
        [3.056, 2.97000936733, 250.60204967330],
        [3.117, 0.76332399369, 229.45260513260],
        [3.091, 4.91941279780, 221.16340196420],
        [3.378, 3.82658652472, 25936.85547173129],
        [3.041, 2.14503983522, 6604.95878212400],
        [2.865, 0.53734608663, 273.85360000370],
        [2.845, 0.34922064899, 85.93917341580],
        [3.657, 6.02755271763, 340.88276662940],
        [2.818, 4.44508352472, 369.34270012700],
        [2.861, 2.19284349075, 295.19424100610],
        [2.865, 3.21935127992, 3.82027867850],
        [2.797, 5.97725967979, 2014.02655475710],
        [3.280, 0.74832416123, 422.71422172270],
        [3.509, 1.92501559437, 343.47920503050],
        [2.962, 2.29867992492, 661.15860543930],
        [3.485, 4.53205302380, 676.28521681260],
        [2.836, 1.20779683660, 1119.18567522950],
        [3.603, 2.72183511139, 508.62646288810],
        [3.620, 5.57691156197, 10066.69129449300],
        [2.731, 0.96663411770, 582.38116313410],
        [3.306, 6.20620840278, 11.56696512570],
        [3.279, 6.13563821647, 276.07178589490],
        [2.747, 4.57476263020, 226.63241756230],
        [2.825, 1.24120423378, 989.05374974680],
        [2.707, 0.46257342768, 1458.47209456600],
        [2.755, 5.43548338507, 246.97871295090],
        [3.338, 3.98641322371, 488.58898404020],
        [2.960, 5.31788128818, 1467.07011427510],
        [3.062, 1.93542241882, 987.78172535960],
        [3.504, 4.10521239427, 6280.10690457480],
        [3.584, 3.12196206517, 108.72184851110],
        [2.975, 0.13746189123, 9987.45627780080],
        [2.634, 5.61201014857, 412.37109687440],
        [3.702, 0.66231252049, 10101.85538471420],
        [3.261, 4.39228048501, 75.37229939830],
        [2.939, 3.26319979850, 130.55261660800],
        [2.742, 6.24317126103, 447.20511869550],
        [3.008, 4.24451493185, 170.17012495410],
        [3.001, 2.45489658954, 230.82520325630],
        [2.722, 4.98348297926, 754.03576079650],
        [2.928, 5.73784691627, 14.66903698630],
        [2.699, 2.98043546816, 27.72047482030],
        [3.678, 3.46436124301, 26468.03090953420],
        [3.500, 4.41810854452, 322.61164478010],
        [2.943, 3.58544468129, 12489.88562870720],
        [2.894, 2.26999120840, 1615.89959822680],
        [2.679, 1.32002304425, 236.19364511790],
        [2.711, 1.25250599577, 52250.58788171570],
        [2.573, 4.41371056719, 262.05714021440],
        [3.483, 1.97113718781, 655.93892142400],
        [2.600, 3.80107226970, 70.63699698310],
        [3.280, 1.75059058234, 683.98946464440],
        [3.260, 3.46047977615, 74.19089773630],
        [3.097, 5.47452034532, 302.37722397610],
        [2.565, 3.47443238116, 2042.49778910280],
        [3.158, 4.58160924364, 12492.85457412380],
        [3.373, 5.52806001629, 10210.31660079440],
        [2.619, 4.43681016753, 949.17560896980],
        [3.290, 1.35055242247, 515.67631941410],
        [3.180, 4.49938964982, 694.83822295220],
        [2.558, 5.07921716363, 197.79997704920],
        [2.902, 2.61388996483, 115.36231475990],
        [2.586, 4.09771865336, 1448.91086701040],
        [3.160, 4.82168102018, 714.67888488130],
        [3.016, 3.16691101108, 385.75684225250],
        [3.001, 3.24181300229, 1618.86854364340],
        [3.319, 5.98200177347, 533.83556667880],
        [3.307, 3.31452419197, 732.97125859000],
        [2.530, 0.13809025963, 591.94239068970],
        [2.975, 5.72770980032, 1011.42703456490],
        [2.857, 4.11031053901, 2267.43328563570],
        [3.091, 4.87979664891, 582.64179556500],
        [3.422, 6.18592254593, 281.48841186630],
        [2.501, 0.50645055144, 29.22619938800],
        [3.502, 0.17203520151, 371.52974825090],
        [2.607, 3.51828908958, 112.39336934330],
        [2.547, 4.45612695304, 901.22071334550],
        [2.778, 4.97805873371, 132.57960603750],
        [2.476, 1.55163657371, 1234.12729886410],
        [2.929, 2.34725672182, 273.15102458810],
        [2.724, 5.73177362443, 688.65533109040],
        [2.667, 6.13733078138, 161.41204655060],
        [2.877, 5.47506403197, 1436.54075273260],
        [2.779, 1.54196175338, 680.05731138130],
        [2.427, 4.64913431310, 392.65794093220],
        [3.223, 2.57976952494, 267.58564077040],
        [2.750, 1.29662662582, 108.98248094200],
        [3.220, 1.06290187940, 388.72578766910],
        [3.236, 3.47802643973, 283.62727588040],
        [3.027, 5.38233284458, 44.09217833040],
        [2.830, 5.70274947128, 327.43756992050],
        [2.768, 5.42256168790, 482.25733433210],
        [3.230, 5.65661970187, 134.37289528650],
        [2.685, 5.03687735302, 763.22448133660],
        [2.441, 5.19830386978, 380.23964254460],
        [2.608, 3.11386505876, 578.44900987100],
        [2.465, 1.49242672830, 141.69889060840],
        [2.332, 3.19371804203, 683.02625679790],
        [2.307, 4.07685834192, 78.92620015150],
        [2.280, 2.52476606956, 156.67674413540],
        [2.260, 6.23958645351, 400.57463708510],
        [2.301, 2.47762267951, 107.91869935900],
        [3.089, 5.69639540598, 537.39521292640],
        [3.074, 1.16740737548, 58.62808887270],
        [2.846, 0.02430566936, 563.37058260750],
        [2.423, 4.24559706691, 27.74172667970],
        [2.670, 3.48404619442, 123.01837848190],
        [2.228, 3.33159258912, 1257.70317210020],
        [2.445, 2.29979337782, 280.21638747910],
        [2.260, 3.85026258012, 753.14198891920],
        [2.195, 4.94038065567, 1222.27844944810],
        [2.579, 1.76841912074, 710.74673161820],
        [2.402, 4.47942458428, 569.04784100980],
        [2.181, 5.39048760967, 318.67949151700],
        [2.317, 6.27946729049, 493.04240216510],
        [2.420, 3.66436222896, 3.62333672240],
    ],
    # R1
    [
        [1479896.370, 3.67205705317, 74.78159856730],
        [71212.085, 6.22601006675, 63.73589830340],
        [68626.972, 6.13411265052, 149.56319713460],
        [20857.262, 5.24625494219, 11.04570026390],
        [21468.152, 2.60176704270, 76.26607127560],
        [24059.649, 3.14159265359, 0.00000000000],
        [11405.346, 0.01848461561, 70.84944530420],
        [7496.775, 0.42360033283, 73.29712585900],
        [4243.800, 1.41692350371, 85.82729883120],
        [3505.936, 2.58354048851, 138.51749687070],
        [3228.835, 5.25499602896, 3.93215326310],
        [3926.694, 3.15513991323, 71.81265315070],
        [3060.010, 0.15321893225, 1.48447270830],
        [3578.446, 2.31160668309, 224.34479570190],
        [2564.251, 0.98076846352, 148.07872442630],
        [2429.445, 3.99440122468, 52.69019803950],
        [1644.719, 2.65349313124, 127.47179660680],
        [1583.766, 1.43045619196, 78.71375183040],
        [1413.112, 4.57461892062, 202.25339517410],
        [1489.525, 2.67559167316, 56.62235130260],
        [1403.237, 1.36985349744, 77.75054398390],
        [1228.220, 1.04703640149, 62.25142559510],
        [1508.028, 5.05996325425, 151.04766984290],
        [992.085, 2.17168865909, 65.22037101170],
        [1032.731, 0.26459059027, 131.40394986990],
        [861.867, 5.05530802218, 351.81659230870],
        [744.445, 3.07640148939, 35.16409022120],
        [604.362, 0.90717667985, 984.60033162190],
        [646.851, 4.47290422910, 70.32818044240],
        [574.710, 3.23070708457, 447.79581952650],
        [687.470, 2.49912565674, 77.96299230500],
        [623.602, 0.86253073820, 9.56122755560],
        [527.794, 5.15136007084, 2.96894541660],
        [561.839, 2.71778158980, 462.02291352810],
        [530.364, 5.91655309045, 213.29909543800],
        [460.080, 4.22302465979, 12.53017297220],
        [494.280, 0.46291078127, 145.63104387150],
        [487.336, 0.70614146398, 380.12776796000],
        [380.908, 3.85089591694, 3.18139373770],
        [444.352, 2.15558291251, 67.66805156650],
        [338.800, 2.53820897704, 18.15924726470],
        [372.947, 5.05141251694, 529.69096509460],
        [348.345, 1.74874852104, 71.60020482960],
        [405.881, 1.22961727600, 22.09140052780],
        [268.913, 6.24069521597, 340.77089204480],
        [255.585, 2.95695013627, 84.34282612290],
        [259.465, 3.92053708924, 59.80374504030],
        [224.731, 3.90961468562, 160.60889739850],
        [221.710, 3.64727173951, 137.03302416240],
        [254.591, 3.50411592815, 38.13303563780],
        [238.290, 2.04879982674, 269.92144674060],
        [272.355, 3.38363105223, 222.86032299360],
        [200.648, 1.24861003313, 69.36497259590],
        [234.153, 0.27825220612, 108.46121608020],
        [188.515, 4.41307507326, 265.98929347750],
        [211.691, 0.68027381802, 111.43016149680],
        [205.946, 1.53379817229, 284.14854074220],
        [196.179, 4.77152996605, 299.12639426920],
        [153.102, 5.21761881347, 209.36694217490],
        [162.563, 4.34054353610, 33.67961751290],
        [150.563, 1.98966326297, 54.17467074780],
        [137.012, 0.40323866041, 195.13984817330],
        [117.171, 0.39649791652, 87.31177153950],
        [127.913, 2.40333045173, 39.61750834610],
        [104.218, 2.92152185788, 134.58534360760],
        [103.862, 1.81622936156, 72.33391801250],
        [105.741, 0.17067407327, 79.23501669220],
        [106.419, 0.69799543514, 2.44768055480],
        [95.326, 4.02880266738, 82.85835341460],
        [104.772, 4.43616414428, 305.34616939270],
        [93.825, 5.01823592717, 51.20572533120],
        [103.739, 2.57553519741, 191.20769491020],
        [106.679, 1.22996874093, 225.82926841020],
        [93.452, 3.09274255916, 77.22927912210],
        [97.398, 3.81380841075, 152.53214255120],
        [84.583, 5.72473747348, 68.84370773410],
        [77.395, 0.08281157747, 45.57665103870],
        [76.207, 4.20384370842, 73.81839072080],
        [86.249, 0.53131085736, 145.10977900970],
        [75.795, 3.78559826812, 75.74480641380],
        [77.592, 1.63628139623, 479.28838891550],
        [84.612, 0.61662456010, 116.42609634290],
        [100.209, 4.94084867643, 120.35824960600],
        [72.142, 4.30505812564, 565.11568774670],
        [70.733, 2.38450718488, 60.76695288680],
        [71.585, 3.93906647867, 153.49535039770],
        [84.566, 5.56037336584, 344.70304530790],
        [63.556, 1.93742986679, 41.64449777560],
        [71.619, 3.71213491656, 408.43894361130],
        [61.594, 3.90006698249, 4.45341812490],
        [64.973, 1.55845503407, 106.97674337190],
        [59.913, 0.60110866128, 74.89347315190],
        [62.000, 4.39369268007, 453.42489381900],
        [63.361, 4.19159979468, 184.72728735580],
        [62.301, 3.23773103318, 422.66603761290],
        [54.427, 3.72545550857, 7.11354700080],
        [52.474, 6.08562717749, 404.50679034820],
        [59.073, 1.55568469603, 456.39383923560],
        [52.597, 3.50492233970, 125.98732389850],
        [52.835, 5.20100035142, 358.93013930950],
        [58.123, 5.33480562448, 220.41264243880],
        [52.909, 4.44819701196, 426.59819087600],
        [50.934, 0.52638534200, 490.33408917940],
        [54.968, 1.60146090981, 14.97785352700],
        [49.491, 4.25534603275, 5.41662597140],
        [51.303, 0.36772379136, 206.18554843720],
        [51.821, 1.75832999538, 8.07675484730],
        [56.964, 0.84114552694, 146.59425171800],
        [49.109, 0.94061875871, 99.16062095550],
        [46.361, 5.35115472594, 152.74459087230],
        [48.023, 1.97249712347, 288.08069400530],
        [43.772, 3.03713403879, 20.60692781950],
        [49.493, 5.84619560979, 112.91463420510],
        [41.987, 0.04620500196, 128.95626931510],
        [48.628, 3.62817742782, 81.00137369080],
        [41.472, 2.33730376429, 277.03499374140],
        [39.983, 5.09525356576, 35.42472265210],
        [41.948, 2.51050760642, 24.37902238820],
        [38.325, 3.61946898382, 173.94221952280],
        [38.385, 2.06003220130, 333.65734504400],
        [42.597, 1.26088737300, 1514.29129671650],
        [38.855, 0.74239364306, 347.88443904560],
        [38.535, 4.95064283065, 92.94084583200],
        [33.234, 1.38358507432, 74.66972398270],
        [33.788, 3.68407945156, 66.91729204110],
        [38.953, 5.49236040328, 200.76892246580],
        [31.850, 0.53990592534, 203.73786788240],
        [33.320, 6.26012644668, 1059.38193018920],
        [30.806, 2.53797566903, 977.48678462110],
        [29.198, 5.43116906000, 58.10682401090],
        [30.059, 0.19481555617, 387.24131496080],
        [28.997, 3.10546504714, 991.71387862270],
        [35.640, 3.72863820177, 96.87299909510],
        [27.607, 0.37142052647, 80.19822453870],
        [32.492, 4.38403518987, 221.37585028530],
        [27.029, 1.35552416596, 0.96320784650],
        [31.276, 0.79566430555, 373.01422095920],
        [31.122, 2.05381353845, 230.56457082540],
        [25.883, 3.46808071409, 144.14657116320],
        [30.201, 0.71392007232, 109.94568878850],
        [24.688, 3.04162764358, 14.01464568050],
        [27.882, 4.76559523368, 415.55249061210],
        [25.110, 5.12405829717, 81.37388070630],
        [25.582, 2.56904073164, 522.57741809380],
        [24.351, 2.20289059750, 628.85158605010],
        [25.479, 1.79521877300, 143.62530630140],
        [24.182, 5.67160913092, 443.86366626340],
        [25.679, 5.43185950751, 546.95644048200],
        [24.177, 5.59982039849, 32.19514480460],
        [24.428, 3.30271734903, 617.80588578620],
        [23.535, 0.65842590604, 46.20979048510],
        [22.371, 4.82094751058, 135.54855145410],
        [27.179, 2.02720001624, 536.80451209540],
        [22.213, 4.61664624220, 391.17346822390],
        [21.973, 4.59216260632, 241.61027108930],
        [20.813, 0.24392941148, 465.95506679120],
        [27.264, 2.15210992383, 140.00196957900],
        [21.356, 5.27168432406, 159.12442469020],
        [23.632, 4.94972840898, 561.18353448360],
        [24.921, 0.54550733267, 181.75834193920],
        [23.027, 3.80632203913, 55.13787859430],
        [19.799, 1.30259938601, 518.64526483070],
        [19.252, 1.31448491434, 543.02428721890],
        [19.704, 4.90869636976, 909.81873305460],
        [20.801, 0.91178207093, 76.47851959670],
        [19.876, 0.66494008343, 66.70484372000],
        [18.957, 4.67998817036, 98.89998852460],
        [25.913, 4.52903186569, 454.90936652730],
        [21.888, 1.23372931740, 41.10198105440],
        [18.703, 6.09640927844, 103.09277421860],
        [18.207, 0.97283864525, 55.65914345610],
        [21.247, 4.19373732137, 329.72519178090],
        [19.408, 4.31468230800, 6.21977512350],
        [18.497, 5.78624335074, 142.44965013380],
        [22.588, 5.84591645052, 297.64192156090],
        [16.770, 6.09084656811, 211.81462272970],
        [16.432, 2.50008469020, 61.28821774860],
        [20.361, 3.16137245375, 186.21176006410],
        [15.955, 2.98317221345, 81.89514556810],
        [18.953, 6.01226591746, 155.78297225810],
        [17.686, 4.82613965176, 273.10284047830],
        [15.141, 3.65588411561, 472.17484191470],
        [18.440, 3.47582817224, 36.64856292950],
        [16.303, 0.13086415177, 554.06998748280],
        [18.633, 0.23932740251, 23.57587323610],
        [14.352, 2.69389896537, 70.11573212130],
        [15.190, 2.43789398875, 486.40193591630],
        [14.002, 5.12389205028, 29.20494752860],
        [15.758, 4.24947053051, 146.38180339690],
        [14.125, 1.55719788547, 110.20632121940],
        [17.477, 1.94549668506, 835.03713448730],
        [13.691, 1.63831110442, 92.04707395470],
        [13.801, 0.13721153975, 235.39049596580],
        [13.573, 2.85427895075, 49.50880430180],
        [12.563, 3.20921738646, 100.38446123290],
        [12.390, 2.88595800082, 60.55450456570],
        [14.986, 0.32593957273, 259.50888592310],
        [12.922, 2.77565630582, 105.49227066360],
        [12.323, 3.36427641421, 440.68227252570],
        [15.233, 0.25589845180, 258.87574647670],
        [12.106, 0.10857558014, 157.63995198190],
        [12.883, 0.30655541587, 124.29040286910],
        [10.900, 3.42905554547, 33.13710079170],
        [11.206, 4.98840478043, 604.47256366190],
        [10.812, 3.86253020441, 767.36908292080],
        [11.561, 2.60450144944, 166.82867252200],
        [10.200, 5.27810824796, 264.50482076920],
        [10.926, 0.64149188846, 558.00214074590],
        [12.315, 4.33998516461, 16.67477455640],
        [9.946, 0.67298666287, 31.49256938900],
        [12.641, 4.83194943583, 114.39910691340],
        [10.479, 0.20404797652, 275.55052103310],
        [11.291, 0.96120625051, 373.90799283650],
        [12.144, 1.91712815063, 378.64329525170],
        [12.229, 0.70465454670, 218.40690486870],
        [10.753, 5.74480767273, 88.11492069160],
        [9.481, 0.65566927406, 353.30106501700],
        [11.006, 2.62953946665, 154.01661525950],
        [9.113, 2.99457723478, 681.54178408960],
        [10.429, 2.33056994007, 132.88842257820],
        [9.169, 4.79284571455, 216.48048917570],
        [9.341, 0.75923548315, 129.91947716160],
        [8.917, 0.78008399009, 67.35923502580],
        [8.757, 6.12717748848, 150.52640498110],
        [9.637, 2.88664912193, 67.88049988760],
        [10.465, 0.36943456465, 699.70103135430],
        [9.301, 1.49620591593, 19.64371997300],
        [9.367, 5.26481516822, 80.71948940050],
        [10.076, 3.56540311122, 278.51946644970],
        [9.455, 3.06088968751, 149.67507171920],
        [9.168, 3.02528121597, 162.09337010680],
        [8.395, 2.18455001650, 342.25536475310],
        [9.233, 5.32613442062, 152.01087768940],
        [9.786, 2.43713607191, 75.30286342910],
        [10.029, 0.81917102953, 339.28641933650],
        [9.429, 1.93671715384, 147.11551657980],
        [7.861, 4.71717822837, 106.01353552540],
        [8.813, 0.01616162513, 42.58645376270],
        [7.808, 0.61104170424, 135.33610313300],
        [8.193, 2.59644466423, 469.13646052890],
        [10.084, 2.58619215129, 50.40257617910],
        [8.574, 5.69115937472, 760.25553592000],
        [7.525, 2.64764195045, 5.93789083320],
        [8.699, 0.54050826161, 66.18357885820],
        [8.027, 1.94079002321, 180.27386923090],
        [7.547, 5.94593031762, 97.41551581630],
        [7.597, 5.80197738402, 450.97721326420],
        [8.666, 3.69933873164, 300.61086697750],
        [7.685, 1.47377256329, 32.24332891440],
        [8.195, 2.30769657654, 254.94359321360],
        [8.473, 1.27680705911, 39.35687591520],
        [7.026, 0.68091865104, 874.39401040250],
        [8.898, 0.16273040357, 43.12897048390],
        [7.205, 4.98177531040, 117.91056905120],
        [7.389, 4.09295183164, 92.30770638560],
        [7.314, 5.04313738379, 756.32338265690],
        [8.454, 1.22026161161, 79.44746501330],
        [6.925, 6.04100189247, 350.33211960040],
        [8.793, 1.33398658801, 48.75804477640],
        [7.270, 3.32609286227, 68.18931642830],
        [6.825, 4.77832275072, 142.66209845490],
        [6.816, 3.90452052962, 480.77286162380],
        [7.062, 1.27536949417, 68.56182344380],
        [7.947, 4.29940380231, 624.91943278700],
        [6.741, 5.43264472273, 610.69233878540],
        [6.529, 5.43599941795, 88.79624424780],
        [7.635, 4.81180007736, 312.45971639350],
        [7.235, 3.18370421558, 268.43697403230],
        [8.133, 1.98936178361, 692.58748435350],
        [6.477, 1.05238958778, 685.47393735270],
        [6.630, 1.37656948077, 291.26208774300],
        [6.878, 2.59188446778, 282.66406803390],
        [7.123, 5.79744758808, 468.24268865160],
        [6.320, 2.58497126634, 458.09076026500],
        [6.222, 5.68982546821, 113.87784205160],
        [7.635, 0.49482302003, 296.15744885260],
        [8.521, 0.00576688485, 227.31374111850],
        [6.520, 3.99093726386, 42.53826965290],
        [6.435, 1.03721543102, 365.90067395840],
        [6.107, 0.35071886662, 148.59998928810],
        [8.199, 1.13448902886, 69.15252427480],
        [6.102, 0.94101111641, 13.33332212430],
        [5.989, 4.98445156102, 184.09414790940],
        [6.355, 0.16346166674, 228.27694896500],
        [7.955, 4.03567630186, 183.24281464750],
        [5.884, 4.40842406038, 19.12245511120],
        [5.938, 5.40863870407, 17.52610781830],
        [5.869, 5.39494525133, 95.38852638680],
        [5.775, 2.81250784939, 121.84272231430],
        [6.070, 4.23605170027, 119.50691634410],
        [6.349, 3.52304701692, 285.63301345050],
        [5.780, 0.17831551537, 458.84151979040],
        [5.674, 4.16711163603, 89.75945209430],
        [5.534, 4.24741728108, 75.53235809270],
        [5.648, 2.81224199321, 154.97982310600],
        [6.939, 3.31979953743, 306.83064210100],
        [5.682, 4.79764449768, 248.72381809010],
        [6.087, 4.04640130992, 271.40591944890],
        [6.869, 1.34392408836, 7.86430652620],
        [5.611, 5.32955957046, 920.86443331850],
        [6.495, 0.45735814276, 106.27416795630],
        [5.353, 2.49825965802, 24.11838995730],
        [6.612, 5.24626646696, 58.31927233200],
        [5.552, 0.24515487696, 173.68158709190],
        [5.209, 6.07866998675, 134.06407874580],
        [5.176, 3.69984512887, 778.41478318470],
        [5.949, 3.63204266272, 189.72322220190],
        [6.360, 0.35370738262, 411.62033734900],
        [5.147, 1.55440402971, 193.65537546500],
        [6.436, 5.18759014405, 120.99138905240],
        [6.994, 4.85978914075, 419.48464387520],
        [5.323, 0.50787742639, 16.46232623530],
        [5.085, 1.28917723765, 267.47376618580],
        [5.993, 4.70505267412, 298.23262239190],
        [5.507, 2.72405080404, 986.08480433020],
        [6.163, 1.87793216012, 397.39324334740],
        [4.846, 5.66714115710, 90.82323367730],
        [4.875, 1.24385851949, 25.60286266560],
        [5.374, 0.31175745933, 192.69216761850],
        [5.262, 1.85699096844, 114.94162363460],
        [5.373, 6.22242588334, 91.45637312370],
        [5.050, 3.39322756907, 831.10498122420],
        [4.637, 0.84958882655, 403.02231763990],
        [6.382, 2.77560901069, 198.32124191100],
        [4.685, 4.94029403928, 902.70518605380],
        [5.005, 1.40309022449, 6.15033915430],
        [5.014, 5.57665259095, 451.94042111070],
        [4.580, 2.47734499363, 31.23193695810],
        [5.129, 3.23528704150, 109.31254934210],
        [4.459, 6.22635092697, 207.88246946660],
        [5.734, 0.96616252776, 483.22054217860],
        [4.425, 2.74721673213, 823.99143422340],
        [4.575, 1.87994871749, 44.72531777680],
        [4.748, 0.34902594832, 457.87831194390],
        [4.268, 4.89983575247, 124.50285119020],
        [4.709, 5.28612293112, 449.28029223480],
        [5.761, 2.09247769051, 187.69623277240],
        [4.284, 0.66132439268, 210.33015002140],
        [4.318, 1.68857333749, 309.27832265580],
        [4.332, 1.41872733238, 25.12978191360],
        [4.305, 1.05990546337, 606.76018552230],
        [4.519, 5.84384426255, 905.88657979150],
        [3.934, 0.41768897300, 180.16199464630],
        [3.973, 3.22666150606, 639.89728631400],
        [4.871, 4.61331971606, 258.02441321480],
        [4.604, 4.77631056831, 463.50738623640],
        [3.943, 3.31312639875, 107.49800823370],
        [4.217, 0.73383451512, 497.44763618020],
        [4.057, 1.67333716577, 7.42236354150],
        [3.854, 6.13547145503, 34.20088237470],
        [4.617, 5.89829880253, 303.86169668440],
        [5.086, 2.85235518740, 28.31117565130],
        [5.337, 2.36556705745, 477.80391620720],
        [4.456, 1.74674336635, 95.97922721780],
        [4.138, 3.80344455465, 460.53844081980],
        [3.812, 2.48508006441, 25.27279426550],
        [4.732, 0.87519409311, 255.05546779820],
        [3.843, 4.02615028031, 104.00779795530],
        [3.776, 2.89171052095, 27.08733537390],
        [4.932, 0.36238909407, 123.53964334370],
        [4.371, 3.74322467592, 376.19561469690],
        [3.747, 3.04126115463, 142.14083359310],
        [4.232, 4.31629167726, 446.31134681820],
        [3.685, 3.26448469664, 170.76082578510],
        [3.575, 4.31199276037, 572.22923474750],
        [3.567, 4.08542270507, 433.71173787680],
        [4.496, 2.10358455875, 838.21852822500],
        [3.505, 3.53902384390, 520.12973753900],
        [3.524, 3.75716903766, 473.06861379200],
        [3.962, 5.33706246667, 43.28902917830],
        [3.597, 3.65066955203, 976.00231191280],
        [3.487, 2.12239114397, 316.39186965660],
        [3.475, 4.44351326599, 384.05992122310],
        [3.628, 2.11511417759, 73.18525127440],
        [3.702, 3.86923731076, 981.63138620530],
        [3.687, 5.18698183343, 993.19835133100],
        [3.599, 2.07986409347, 47.69426319340],
        [3.807, 4.21821126511, 196.62432088160],
        [4.707, 4.56309173897, 47.06112374700],
        [4.312, 0.38740046308, 988.53248488500],
        [3.867, 2.08559458308, 457.35704708210],
        [4.723, 4.16947683948, 219.89137757700],
        [3.527, 0.20371576470, 394.35486196160],
        [3.644, 5.82023483708, 586.31331639720],
        [3.328, 2.93840719007, 535.91074021810],
        [3.321, 4.19289134366, 114.13847448250],
        [4.128, 3.06165703109, 377.15882254340],
        [3.545, 4.41886084391, 1293.87865427770],
        [3.295, 2.97049569593, 15.19030184810],
        [3.337, 6.23473900765, 9947.05568153210],
        [3.253, 5.22412177835, 425.11371816770],
        [3.677, 5.31389484415, 141.69889060840],
        [3.242, 4.68868636498, 978.97125732940],
        [3.266, 3.57072306171, 17.26547538740],
        [3.435, 0.52794358986, 141.48644228730],
        [3.242, 2.62760698007, 6.59228213900],
        [3.613, 1.94737668557, 661.09491496450],
        [3.182, 0.36603315110, 449.49274055590],
        [3.311, 1.25616383318, 233.90602325750],
        [3.403, 6.03792583170, 199.28444975750],
        [4.196, 4.26442082589, 381.61224066830],
        [3.961, 4.53281422377, 916.93228005540],
        [3.846, 3.76849990033, 8.59801970910],
        [3.350, 5.63661413371, 444.82687410990],
        [3.780, 5.35722293289, 328.24071907260],
        [3.166, 2.16351748263, 983.11585891360],
        [3.538, 1.89746744103, 280.96714700450],
        [3.930, 2.09444900058, 653.98136796370],
        [3.282, 1.91872815218, 2349.32843120380],
        [3.269, 0.52855777633, 450.45594840240],
        [3.582, 1.60170266832, 1587.58842257550],
        [3.522, 2.51782036180, 237.67811782620],
        [3.024, 3.54567524563, 94.42531854030],
        [3.528, 4.79818282081, 406.95447090300],
        [2.996, 2.59155293620, 6133.51265285680],
        [3.146, 2.18094827839, 216.92243216040],
        [3.610, 6.15486273902, 171.65459766240],
        [2.977, 0.69478628170, 294.30046912880],
        [3.377, 1.21382647091, 162.89651925890],
        [3.347, 4.14981703949, 214.78356814630],
        [2.953, 2.18721777019, 597.35901666110],
        [4.049, 3.15153850922, 833.55266177900],
        [3.725, 5.84743216544, 6058.73105428950],
        [3.390, 1.18412112871, 167.72244439930],
        [3.142, 2.26934209337, 517.16079212240],
        [4.077, 0.07273073033, 1190.03512053370],
        [3.020, 2.64998251178, 20.44686912510],
        [3.926, 1.41612569694, 346.18751801620],
        [3.110, 1.11431255827, 1044.40407666220],
        [2.836, 0.62522723719, 749.20983565610],
        [2.831, 4.78996738581, 820.05928096030],
        [2.824, 0.87232289414, 30.71067209630],
        [3.114, 1.79734939525, 414.06801790380],
        [2.801, 3.99301180541, 10063.72234907640],
        [3.489, 1.86982946081, 371.52974825090],
        [3.725, 1.68366366742, 683.98946464440],
        [3.763, 3.28247771799, 432.81796599950],
        [3.493, 0.98765698465, 9988.94075050910],
        [3.523, 5.12512607932, 105.38039607900],
        [2.839, 4.22662576295, 990.22940591440],
        [3.432, 2.80483162230, 764.18768918310],
        [2.733, 0.42373696972, 354.99798604640],
        [3.146, 5.19208910201, 417.03696332040],
        [3.041, 5.75641149588, 409.92341631960],
        [3.379, 5.47448876584, 1396.22066897090],
        [3.102, 0.41684444780, 521.09294538550],
        [2.863, 0.41519700992, 894.84087952760],
        [2.707, 3.60084311477, 621.73803904930],
        [3.128, 5.23384180625, 424.15051032120],
        [3.107, 2.44919355737, 4.66586644600],
        [2.683, 3.88682711832, 133.10087089930],
        [2.660, 4.78670985324, 362.86229257260],
        [3.200, 1.88004939357, 331.20966448920],
        [2.730, 4.12217979791, 600.54041039880],
        [3.414, 4.93712749827, 1140.38330388000],
        [2.653, 5.10283251074, 118.02244363580],
        [3.222, 4.76521772319, 294.67297614430],
        [3.289, 4.26401031509, 544.50875992720],
        [3.100, 5.47928527930, 701.18550406260],
        [2.785, 5.19343849039, 144.89733068860],
        [2.607, 4.72531286187, 122.47586176070],
        [2.581, 6.27329466695, 908.33426034630],
        [3.285, 1.95972622670, 372.42352012820],
        [2.897, 0.37378090180, 582.38116313410],
        [2.615, 2.25516923974, 74.99404688840],
        [3.582, 1.27992264402, 987.56927703850],
        [3.115, 5.10929689813, 459.05396811150],
        [2.857, 5.90256930211, 525.23754696970],
        [2.589, 1.83177157032, 657.16276170140],
        [2.539, 4.14968938109, 74.73341445750],
        [2.797, 2.82242772664, 2036.86871481030],
        [2.688, 2.16500211397, 262.80789973980],
        [2.744, 1.54445470732, 28.57180808220],
        [2.539, 0.46036497385, 74.82978267710],
        [3.322, 3.50108539407, 82.64590509350],
        [2.810, 6.06709915335, 374.49869366750],
        [2.504, 3.52394801700, 1183.67233305830],
        [2.565, 1.64023845161, 73.40900044360],
        [2.531, 3.50486296784, 293.18850343600],
        [2.663, 4.23321349902, 421.18156490460],
        [2.793, 2.00644423849, 75.04223099820],
        [2.430, 1.56119387576, 136.06981631590],
        [2.553, 1.25909246207, 670.49608382570],
        [2.604, 3.87350462519, 74.03083904190],
        [2.510, 3.35948960782, 464.99185894470],
        [3.005, 0.81031349171, 73.88782669000],
        [3.110, 6.14956891318, 118.87377689770],
        [3.234, 2.45751141361, 98.35747180340],
        [2.774, 6.26134027482, 1022.31267613440],
        [2.402, 4.38353347008, 1303.27982313890],
        [3.296, 3.84350963765, 511.53171782990],
        [2.800, 2.60339313269, 74.52096613640],
        [3.005, 0.76247280223, 75.67537044460],
        [2.434, 4.94784679430, 969.62247809490],
        [2.632, 0.63557110200, 227.52618943960],
        [2.669, 0.73340228210, 73.08467753790],
        [2.465, 1.30648773380, 77.06922042770],
        [3.237, 3.19110274211, 1887.30551767570],
        [2.395, 2.76580569447, 768.85355562910],
        [3.230, 0.01981320255, 881.50755740330],
        [2.747, 5.59085990261, 388.72578766910],
        [3.008, 5.65955463660, 1969.20066324380],
        [3.008, 0.91409756228, 2118.76386037840],
        [2.465, 0.26629856014, 72.49397670690],
        [2.629, 4.00618677646, 26.02355379090],
        [2.272, 2.77069357315, 515.46387109300],
        [2.360, 4.12736987374, 74.62153987290],
        [2.255, 3.36574443950, 286.59622129700],
        [2.211, 5.18239546182, 59.28248017850],
        [2.994, 2.83179016989, 184.98791978670],
        [2.492, 1.19872353228, 383.09671337660],
        [2.355, 0.48259604722, 74.94165726170],
        [2.185, 6.07997119980, 63.62402371880],
        [2.228, 1.42452148910, 6219.33995168800],
    ],
    # R2
    [
        [22439.904, 0.69953118760, 74.78159856730],
        [4727.037, 1.69901641488, 63.73589830340],
        [1681.903, 4.64833551727, 70.84944530420],
        [1433.755, 3.52119917947, 149.56319713460],
        [1649.559, 3.09660078980, 11.04570026390],
        [770.188, 0.00000000000, 0.00000000000],
        [461.009, 0.76676632849, 3.93215326310],
        [500.429, 6.17229032223, 76.26607127560],
        [390.371, 4.49605283502, 56.62235130260],
        [389.945, 5.52673426377, 85.82729883120],
        [292.097, 0.20389012095, 52.69019803950],
        [272.898, 3.84707823651, 138.51749687070],
        [286.579, 3.53357683270, 73.29712585900],
        [205.449, 3.24758017121, 78.71375183040],
        [219.674, 1.96418942891, 131.40394986990],
        [215.788, 0.84812474187, 77.96299230500],
        [128.834, 2.08146849515, 3.18139373770],
        [148.554, 4.89840863841, 127.47179660680],
        [117.452, 4.93414907433, 447.79581952650],
        [112.690, 1.01361852218, 462.02291352810],
        [98.875, 6.15817742611, 224.34479570190],
        [91.379, 0.67973399531, 18.15924726470],
        [89.217, 0.23425778826, 202.25339517410],
        [88.206, 2.93094837724, 62.25142559510],
        [114.066, 4.78741873960, 145.63104387150],
        [103.858, 3.58561789629, 71.60020482960],
        [61.819, 3.29964272893, 351.81659230870],
        [57.782, 4.90737420887, 22.09140052780],
        [64.369, 3.39006689398, 1.48447270830],
        [71.110, 6.10490061068, 454.90936652730],
        [50.990, 3.86691997779, 65.22037101170],
        [63.537, 3.96202309168, 67.66805156650],
        [58.957, 5.55530463687, 9.56122755560],
        [48.700, 3.74709235789, 269.92144674060],
        [43.584, 1.92568752002, 59.80374504030],
        [42.170, 2.61650997054, 151.04766984290],
        [42.420, 6.13634453301, 284.14854074220],
        [44.340, 5.89997845114, 71.81265315070],
        [37.328, 5.91300114911, 984.60033162190],
        [36.201, 5.40315761474, 77.75054398390],
        [41.989, 2.09071623849, 12.53017297220],
        [31.411, 4.59200004835, 148.07872442630],
        [31.289, 2.26696307388, 195.13984817330],
        [27.150, 3.53242984046, 209.36694217490],
        [28.152, 4.57845964163, 77.22927912210],
        [26.097, 0.65978256272, 120.35824960600],
        [24.372, 5.86680440531, 69.36497259590],
        [23.037, 1.03776963677, 84.34282612290],
        [22.679, 1.71434243970, 160.60889739850],
        [27.650, 4.91488946525, 277.03499374140],
        [20.816, 2.19643268155, 45.57665103870],
        [19.961, 2.32077356180, 2.44768055480],
        [16.584, 4.77529536873, 213.29909543800],
        [16.578, 1.85615182154, 340.77089204480],
        [17.196, 4.36852462522, 54.17467074780],
        [16.053, 3.64619586667, 152.74459087230],
        [14.806, 5.43824503068, 408.43894361130],
        [13.872, 3.38531100784, 358.93013930950],
        [13.328, 5.25179190495, 137.03302416240],
        [13.286, 1.26285812368, 134.58534360760],
        [12.890, 3.03270380745, 92.94084583200],
        [12.467, 1.33213558369, 51.20572533120],
        [13.450, 1.53176996919, 422.66603761290],
        [16.442, 0.40190549188, 265.98929347750],
        [11.996, 5.10426418352, 191.20769491020],
        [12.898, 4.43242192513, 87.31177153950],
        [11.449, 2.02645622099, 7.11354700080],
        [11.826, 4.65645290272, 41.64449777560],
        [12.045, 3.23910807852, 116.42609634290],
        [11.680, 3.73278249629, 220.41264243880],
        [11.573, 4.16500659139, 60.55450456570],
        [10.175, 0.32936886913, 70.32818044240],
        [11.332, 1.07613885149, 72.33391801250],
        [9.655, 3.05950236129, 2.96894541660],
        [9.279, 2.43997351068, 565.11568774670],
        [8.986, 5.18839740735, 225.82926841020],
        [10.284, 1.18602582060, 344.70304530790],
        [8.844, 6.00894470528, 5.41662597140],
        [8.508, 5.24741470216, 347.88443904560],
        [8.319, 3.71723808749, 14.97785352700],
        [8.276, 2.27408171672, 299.12639426920],
        [8.064, 5.71681525179, 55.13787859430],
        [7.830, 0.90313686798, 222.86032299360],
        [8.335, 4.48600419464, 70.11573212130],
        [8.763, 5.81519440120, 153.49535039770],
        [8.472, 3.91387041805, 333.65734504400],
        [9.874, 5.96526143660, 35.16409022120],
        [9.647, 0.38872626737, 415.55249061210],
        [7.106, 1.50598488470, 991.71387862270],
        [6.596, 1.18068235818, 96.87299909510],
        [8.065, 2.25930653257, 206.18554843720],
        [6.479, 2.99461362786, 380.12776796000],
        [9.012, 6.05343622530, 146.38180339690],
        [6.131, 0.05596259493, 99.16062095550],
        [5.799, 0.82465326137, 142.44965013380],
        [5.816, 4.63029217647, 49.50880430180],
        [5.608, 0.66268449799, 58.10682401090],
        [5.966, 2.48916255408, 373.01422095920],
        [5.710, 2.23566160404, 80.19822453870],
        [5.272, 5.06746739956, 440.68227252570],
        [5.162, 4.36457872885, 977.48678462110],
        [5.428, 0.85181859845, 546.95644048200],
        [5.766, 0.34229025692, 536.80451209540],
        [5.924, 5.48443563529, 76.47851959670],
        [5.340, 3.73073116400, 23.57587323610],
        [5.174, 4.13873402677, 132.88842257820],
        [5.310, 6.14059082194, 39.61750834610],
        [5.790, 3.39593613152, 458.09076026500],
        [5.007, 4.25821412289, 522.57741809380],
        [4.967, 4.79184817938, 387.24131496080],
        [5.183, 3.25775152471, 561.18353448360],
        [4.602, 1.69262282455, 152.53214255120],
        [5.302, 1.83522660093, 124.29040286910],
        [5.005, 0.36630565950, 60.76695288680],
        [4.454, 2.30288945184, 312.45971639350],
        [4.457, 0.45775730382, 33.13710079170],
        [5.722, 0.89523844278, 81.89514556810],
        [5.842, 0.92039543147, 20.60692781950],
        [5.743, 0.66226484448, 38.13303563780],
        [4.255, 3.55373860346, 479.28838891550],
        [4.190, 4.37674804409, 79.23501669220],
        [4.194, 1.64986267170, 128.95626931510],
        [5.125, 1.40553011416, 144.14657116320],
        [4.045, 6.07362424798, 19.64371997300],
        [3.984, 5.77178406410, 288.08069400530],
        [5.017, 2.99521887648, 29.20494752860],
        [3.842, 2.60024827897, 426.59819087600],
        [3.861, 3.19886211335, 159.12442469020],
        [3.870, 4.43713601497, 141.69889060840],
        [5.316, 4.07970979457, 111.43016149680],
        [4.553, 0.01384318412, 298.23262239190],
        [3.737, 5.28319518103, 353.30106501700],
        [3.939, 5.27301148162, 521.09294538550],
        [3.710, 5.15385470848, 490.33408917940],
        [4.039, 0.60924359087, 152.01087768940],
        [3.861, 1.34394383700, 535.32003938710],
        [4.385, 0.62057680100, 827.17282796110],
        [3.567, 4.71986443303, 6.90109867970],
        [3.576, 3.24526237368, 230.56457082540],
        [3.469, 0.79054323335, 983.11585891360],
        [4.524, 2.86819565712, 129.91947716160],
        [3.648, 5.59395544992, 774.48262992160],
        [3.513, 4.49630054276, 376.19561469690],
        [3.432, 2.55614913808, 258.87574647670],
        [4.352, 2.09804374929, 404.50679034820],
        [3.336, 0.89628904042, 469.13646052890],
        [3.274, 3.86236880159, 42.53826965290],
        [3.201, 2.76459652868, 248.72381809010],
        [3.184, 0.07709843451, 1514.29129671650],
        [3.783, 5.29835962126, 369.08206769610],
        [3.266, 2.24754480216, 73.81839072080],
        [3.055, 2.60120354415, 433.71173787680],
        [3.051, 4.54953369151, 980.66817835880],
        [3.062, 1.27089879603, 200.76892246580],
        [3.055, 1.70878161343, 639.89728631400],
        [3.110, 3.63187411723, 16.67477455640],
        [3.472, 4.93521260607, 411.62033734900],
        [3.531, 4.49372794858, 881.50755740330],
        [3.284, 5.59170577331, 472.17484191470],
        [3.015, 6.02967446446, 291.26208774300],
        [3.467, 2.17484439267, 554.06998748280],
        [3.138, 0.52367930477, 1094.80665284130],
        [3.257, 2.49339546514, 451.72797278960],
        [2.881, 0.50481204892, 305.34616939270],
        [3.082, 4.20145474081, 146.59425171800],
        [2.883, 2.44983947531, 135.33610313300],
        [2.965, 0.39294995530, 25.27279426550],
        [2.831, 2.52728803131, 867.28046340170],
        [2.728, 5.29491477549, 125.98732389850],
        [2.857, 4.71106805785, 218.92816973050],
        [2.763, 4.27510031656, 350.33211960040],
        [2.730, 1.98552777251, 82.85835341460],
        [2.857, 3.08706426922, 216.48048917570],
        [3.365, 3.67691210011, 661.09491496450],
        [2.925, 1.43646759644, 381.61224066830],
        [2.753, 0.39468041761, 33.67961751290],
        [2.756, 4.62672498840, 1357.61455258110],
        [3.450, 2.12911756067, 685.47393735270],
        [2.571, 5.92862393284, 89.75945209430],
        [2.677, 0.76342313946, 486.40193591630],
        [2.689, 4.16436463826, 235.39049596580],
        [2.646, 3.81808560938, 550.88859374510],
        [3.369, 3.17071565345, 108.46121608020],
        [2.613, 5.68333838067, 24.37902238820],
        [2.736, 1.87107584495, 529.69096509460],
        [2.606, 4.36605237304, 1080.57955883970],
        [2.407, 3.07343136742, 391.17346822390],
        [2.446, 5.73846381540, 535.91074021810],
        [2.334, 5.18878243102, 1059.38193018920],
        [2.568, 1.09886876369, 913.00012679230],
        [2.236, 6.10115874045, 140.00196957900],
        [3.053, 5.35047433775, 681.54178408960],
    ],
    # R3
    [
        [1164.382, 4.73453291602, 74.78159856730],
        [212.367, 3.34255734999, 63.73589830340],
        [196.408, 2.98004616318, 70.84944530420],
        [104.527, 0.95807937648, 11.04570026390],
        [71.681, 0.02528455665, 56.62235130260],
        [72.540, 0.99701907912, 149.56319713460],
        [54.875, 2.59436811267, 3.93215326310],
        [34.029, 3.81553325635, 76.26607127560],
        [32.081, 3.59825177840, 131.40394986990],
        [29.641, 3.44111535957, 85.82729883120],
        [36.377, 5.65035573017, 77.96299230500],
        [27.663, 0.42836001470, 3.18139373770],
        [27.464, 2.55126467481, 52.69019803950],
        [24.569, 5.14034173566, 78.71375183040],
        [19.390, 5.13477648625, 18.15924726470],
        [15.767, 0.37116951743, 447.79581952650],
        [15.441, 5.57271837433, 462.02291352810],
        [15.232, 3.85998573509, 73.29712585900],
        [15.475, 2.97496547327, 145.63104387150],
        [17.951, 0.00000000000, 0.00000000000],
        [15.958, 5.19915553904, 71.60020482960],
        [11.056, 6.03152659562, 138.51749687070],
        [10.529, 3.58261852497, 224.34479570190],
        [7.606, 1.44542030704, 1.48447270830],
        [8.121, 2.61579604319, 22.09140052780],
        [7.107, 5.43946774526, 269.92144674060],
        [6.459, 4.37142319461, 284.14854074220],
        [6.818, 0.01396812984, 151.04766984290],
        [8.101, 0.29563819537, 127.47179660680],
        [5.768, 4.22672716593, 373.01422095920],
        [5.022, 1.84154937974, 202.25339517410],
        [4.692, 2.78404575440, 120.35824960600],
        [5.087, 0.77745294804, 62.25142559510],
        [4.160, 1.83820502779, 72.33391801250],
        [3.922, 1.88900691473, 209.36694217490],
        [5.201, 4.15791319343, 195.13984817330],
        [3.636, 1.99709010456, 65.22037101170],
        [3.582, 3.92592140377, 124.29040286910],
        [3.808, 1.04818660873, 92.94084583200],
        [4.241, 3.95755998904, 9.56122755560],
        [3.497, 1.54139696251, 148.07872442630],
        [3.195, 2.98608971329, 387.24131496080],
        [3.950, 1.85721204932, 152.74459087230],
        [3.277, 1.40881404192, 351.81659230870],
        [3.605, 1.17366167402, 153.49535039770],
        [2.940, 6.03594958459, 12.53017297220],
        [2.744, 5.64674283515, 134.58534360760],
        [2.800, 0.79480255927, 572.22923474750],
        [3.054, 5.84310939105, 160.60889739850],
        [2.662, 1.98593312104, 450.97721326420],
        [2.700, 2.77036653988, 213.29909543800],
        [2.323, 1.67918985468, 358.93013930950],
        [2.254, 5.77129530133, 84.34282612290],
        [2.291, 4.81424601600, 536.80451209540],
        [2.213, 2.20360299816, 465.95506679120],
    ],
    # R4
    [
        [52.996, 3.00838033088, 74.78159856730],
        [9.887, 1.91399083603, 56.62235130260],
        [7.008, 5.08677527404, 11.04570026390],
        [6.728, 5.42924958121, 149.56319713460],
        [3.703, 5.22728581851, 131.40394986990],
        [3.361, 1.29695290266, 85.82729883120],
        [2.664, 0.44064577837, 63.73589830340],
        [2.309, 0.92380720934, 145.63104387150],
        [2.383, 6.21390585593, 358.93013930950],
        [2.288, 2.23425399117, 440.68227252570],
        [3.093, 3.14159265359, 0.00000000000],
    ],
]
"""This table contains Uranus' periodic terms (all of them) from the planetary
theory VSOP87 for the radius vector at the equinox of date (taken from the 'D'
solution). In Meeus' book a shortened version can be found in pages 449-451."""


ORBITAL_ELEM = [
    [314.055005, 429.8640561, 0.0003039, 0.000000026],          # L
    [19.218446062, -0.0000000372, 0.00000000098, 0.0],          # a
    [0.04638122, -0.000027293, 0.0000000789, 0.00000000024],    # e
    [0.773197, 0.0007744, 0.00003749, -0.000000092],            # i
    [74.005957, 0.5211278, 0.00133947, 0.000018484],            # Omega
    [173.005291, 1.486379, 0.00021406, 0.000000434]             # pie
]
"""This table contains the parameters to compute Uranus's orbital elements for
the mean equinox of date. Based in Table 31.A, page 213"""


ORBITAL_ELEM_J2000 = [
    [314.055005, 428.4669983, -0.00000486, 0.000000006],    # L
    [0.773197, -0.0016869, 0.00000349, 0.000000016],        # i
    [74.005957, 0.0741431, 0.00040539, 0.000000119],        # Omega
    [173.005291, 0.0893212, -0.0000947, 0.000000414]        # pie
]
"""This table contains the parameters to compute Uranus's orbital elements for
the standard equinox J2000.0. Based on Table 31.B, page 215"""


class Uranus(object):
    """
    Class Uranus models that planet.
    """

    @staticmethod
    def geometric_heliocentric_position(epoch, tofk5=True):
        """This method computes the geometric heliocentric position of planet
        Uranus for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Uranus position, as an Epoch object
        :type epoch: :py:class:`Epoch`
        :param tofk5: Whether or not the small correction to convert to the FK5
            system will be applied or not
        :type tofk5: bool

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2018, 10, 27.0)
        >>> l, b, r = Uranus.geometric_heliocentric_position(epoch)
        >>> print(round(l.to_positive(), 4))
        30.5888
        >>> print(round(b, 4))
        -0.5315
        >>> print(round(r, 5))
        19.86964
        """

        return geometric_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R, tofk5)

    @staticmethod
    def apparent_heliocentric_position(epoch):
        """This method computes the apparent heliocentric position of planet
        Uranus for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Uranus position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.
        """

        return apparent_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R)

    @staticmethod
    def orbital_elements_mean_equinox(epoch):
        """This method computes the orbital elements of Uranus for the mean
        equinox of the date for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Uranus.orbital_elements_mean_equinox(epoch)
        >>> print(round(l, 6))
        235.517526
        >>> print(round(a, 8))
        19.21844604
        >>> print(round(e, 7))
        0.0463634
        >>> print(round(i, 6))
        0.77372
        >>> print(round(ome, 5))
        74.34776
        >>> print(round(arg, 6))
        99.630865
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM)

    @staticmethod
    def orbital_elements_j2000(epoch):
        """This method computes the orbital elements of Uranus for the
        standard equinox J2000.0 for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Uranus.orbital_elements_j2000(epoch)
        >>> print(round(l, 6))
        234.602641
        >>> print(round(a, 8))
        19.21844604
        >>> print(round(e, 7))
        0.0463634
        >>> print(round(i, 6))
        0.772094
        >>> print(round(ome, 5))
        74.05468
        >>> print(round(arg, 6))
        99.009058
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM_J2000)

    @staticmethod
    def geocentric_position(epoch):
        """This method computes the geocentric position of Uranus (right
        ascension and declination) for the given epoch, as well as the
        elongation angle.

        :param epoch: Epoch to compute geocentric position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the right ascension, the declination and
            the elongation angle as Angle objects
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.

        >>> epoch = Epoch(1992, 12, 20.0)
        >>> ra, dec, elon = Uranus.geocentric_position(epoch)
        >>> print(ra.ra_str(n_dec=1))
        19h 13' 48.7''
        >>> print(dec.dms_str(n_dec=1))
        -22d 46' 13.0''
        >>> print(elon.dms_str(n_dec=1))
        18d 44' 18.7''
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Compute the heliocentric position of Uranus
        l, b, r = Uranus.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute the heliocentric position of the Earth
        l0, b0, r0 = Earth.geometric_heliocentric_position(epoch, tofk5=False)
        # Convert to radians
        lr = l.rad()
        br = b.rad()
        l0r = l0.rad()
        b0r = b0.rad()
        # Compute first iteration
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        delta = sqrt(x * x + y * y + z * z)
        tau = 0.0057755183 * delta
        # Adjust the epoch for light-time
        epoch -= tau
        # Compute again Uranus coordinates with this correction
        l, b, r = Uranus.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute second iteration
        lr = l.rad()
        br = b.rad()
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        # Compute longitude and latitude
        lamb = atan2(y, x)
        beta = atan2(z, sqrt(x * x + y * y))
        # Now, let's compute the aberration effect
        t = (epoch - JDE2000) / 36525
        e = 0.016708634 + t * (-0.000042037 - t * 0.0000001267)
        pie = 102.93735 + t * (1.71946 + t * 0.00046)
        pie = radians(pie)
        lon = l0 + 180.0
        lon = lon.rad()
        k = 20.49552    # The constant of aberration
        deltal1 = k * (-cos(lon - lamb) + e * cos(pie - lamb)) / cos(beta)
        deltab1 = -k * sin(beta) * (sin(lon - lamb) - e * sin(pie - lamb))
        deltal1 = Angle(0, 0, deltal1)
        deltab1 = Angle(0, 0, deltab1)
        # Correction to FK5 system
        lamb = Angle(lamb, radians=True)
        lamb = lamb.to_positive()
        beta = Angle(beta, radians=True)
        l_prime = lamb - t * (1.397 + t * 0.00031)
        deltal2 = Angle(0, 0, -0.09033)
        a = 0.03916 * (cos(l_prime.rad()) + sin(l_prime.rad()))
        a = a * tan(b.rad())
        deltal2 += Angle(0, 0, a)
        deltab2 = 0.03916 * (cos(l_prime.rad()) - sin(l_prime.rad()))
        deltab2 = Angle(0, 0, deltab2)
        # Apply the corrections
        lamb = lamb + deltal1 + deltal2
        beta = beta + deltab1 + deltab2
        # Correction for nutation
        dpsi = nutation_longitude(epoch)
        lamb += dpsi
        e = true_obliquity(epoch)
        ra, dec = ecliptical2equatorial(lamb, beta, e)
        # Let's compute the elongation angle
        lons, lats, rs = Sun.apparent_geocentric_position(epoch)
        lambr = lamb.rad()
        lsr = lons.rad()
        betar = beta.rad()
        elon = acos(cos(betar) * cos(lambr - lsr))
        elon = Angle(elon, radians=True)
        return ra, dec, elon

    @staticmethod
    def conjunction(epoch):
        """This method computes the time of the conjunction closest to the
        given epoch.

        :param epoch: Epoch close to the desired conjunction
        :type epoch: :py:class:`Epoch`

        :returns: The time when the conjunction happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1993, 10, 1.0)
        >>> conj = Uranus.conjunction(epoch)
        >>> y, m, d = conj.get_date()
        >>> print(y)
        1994
        >>> print(m)
        1
        >>> print(round(d, 4))
        12.7365
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Uranus' conjunction
        a = 2451579.489
        b = 369.656035
        m0 = 31.5219
        m1 = 4.333093
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute a couple auxiliary angles
        ee = 207.83 + 8.51 * t
        ff = 108.84 + 419.96 * t
        # Convert to radians
        ee = Angle(ee).rad()
        ff = Angle(ff).rad()
        corr = (-0.0859 + t * 0.0003 +
                sin(m) * (-3.8179 + t * (-0.0148 + t * 0.00003)) +
                cos(m) * (5.1228 + t * (-0.0105 - t * 0.00002)) +
                sin(2.0 * m) * (-0.0803 + t * 0.0011) +
                cos(2.0 * m) * (-0.1905 - t * 0.0006) +
                sin(3.0 * m) * (0.0088 + t * 0.0001) +
                cos(ee) * (0.885) +
                cos(ff) * (0.2153))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def opposition(epoch):
        """This method computes the time of the opposition closest to the given
        epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: The time when the opposition happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1780, 12, 1.0)
        >>> oppo = Uranus.opposition(epoch)
        >>> y, m, d = oppo.get_date()
        >>> print(y)
        1780
        >>> print(m)
        12
        >>> print(round(d, 4))
        17.5998
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Uranus' opposition
        a = 2451764.317
        b = 369.656035
        m0 = 213.6884
        m1 = 4.333093
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute a couple auxiliary angles
        ee = 207.83 + 8.51 * t
        ff = 108.84 + 419.96 * t
        # Convert to radians
        ee = Angle(ee).rad()
        ff = Angle(ff).rad()
        corr = (0.0844 - t * 0.0006 +
                sin(m) * (-0.1048 + t * 0.0246) +
                cos(m) * (-5.1221 + t * (0.0104 + t * 0.00003)) +
                sin(2.0 * m) * (-0.1428 + t * 0.0005) +
                cos(2.0 * m) * (-0.0148 - t * 0.0013) +
                cos(3.0 * m) * (0.0055) +
                cos(ee) * (0.885) +
                cos(ff) * (0.2153))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def perihelion_aphelion(epoch, perihelion=True):
        """This method computes the time of Perihelion (or Aphelion) closer to
        a given epoch.

        :param epoch: Epoch close to the desired Perihelion (or Aphelion)
        :type epoch: :py:class:`Epoch`
        :param peihelion: If True, the epoch of the closest Perihelion is
            computed, if False, the epoch of the closest Aphelion is found.
        :type bool:

        :returns: The epoch of the desired Perihelion (or Aphelion)
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input values are of wrong type.

        .. note:: The solution provided by this method may have several days of
            error.

        >>> epoch = Epoch(1880, 1, 1.0)
        >>> e = Uranus.perihelion_aphelion(epoch)
        >>> y, m, d = e.get_date()
        >>> print(y)
        1882
        >>> print(m)
        3
        >>> print(int(d))
        18
        >>> epoch = Epoch(2090, 1, 1.0)
        >>> e = Uranus.perihelion_aphelion(epoch, perihelion=False)
        >>> y, m, d = e.get_date()
        >>> print(y)
        2092
        >>> print(m)
        11
        >>> print(int(d))
        22
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input value")
        # First approximation
        k = 0.0119 * (epoch.year() - 2051.1)
        if perihelion:
            k = round(k)
        else:
            k = round(k + 0.5) - 0.5
        jde = 2470213.5 + k * (30694.8767 - k * 0.00541)
        # Compute the epochs 1 year before and after
        jde_before = jde - 360.0
        jde_after = jde + 360.0
        # Compute the Sun-Uranus distance for each epoch
        l, b, r_b = Uranus.geometric_heliocentric_position(Epoch(jde_before))
        l, b, r = Uranus.geometric_heliocentric_position(Epoch(jde))
        l, b, r_a = Uranus.geometric_heliocentric_position(Epoch(jde_after))
        # Call an interpolation object
        m = Interpolation([jde_before, jde, jde_after], [r_b, r, r_a])
        sol = m.minmax()
        return Epoch(sol)

    @staticmethod
    def passage_nodes(epoch, ascending=True):
        """This function computes the time of passage by the nodes (ascending
        or descending) of Uranus, nearest to the given epoch.

        :param epoch: Epoch closest to the node passage
        :type epoch: :py:class:`Epoch`
        :param ascending: Whether the time of passage by the ascending (True)
            or descending (False) node will be computed
        :type ascending: bool

        :returns: Tuple containing:
            - Time of passage through the node (:py:class:`Epoch`)
            - Radius vector when passing through the node (in AU, float)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2019, 1, 1)
        >>> time, r = Uranus.passage_nodes(epoch)
        >>> year, month, day = time.get_date()
        >>> print(year)
        2028
        >>> print(month)
        8
        >>> print(round(day, 1))
        23.2
        >>> print(round(r, 4))
        19.3201
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input types")
        # Get the orbital parameters
        l, a, e, i, ome, arg = Uranus.orbital_elements_mean_equinox(epoch)
        # Compute the time of passage through perihelion
        t = Uranus.perihelion_aphelion(epoch)
        # Get the time of passage through the node
        time, r = passage_nodes_elliptic(arg, e, a, t, ascending)
        return time, r

    @staticmethod
    def magnitude(sun_dist, earth_dist):
        """This function computes the approximate magnitude of Uranus.

        :param sun_dist: Distance from Uranus to Sun, in Astronomical Units
        :type sun_dist: float
        :param earth_dist: Distance from Uranus to Earth, in Astronomical Units
        :type earth_dist: float

        :returns: Uranus's magnitude
        :rtype: float
        :raises: TypeError if input values are of wrong type.
        """

        if not (isinstance(sun_dist, float) and isinstance(earth_dist, float)):
            raise TypeError("Invalid input types")
        m = -6.85 + 5.0 * log10(sun_dist * earth_dist)
        return round(m, 1)


def main():

    # Let's define a small helper function
    def print_me(msg, val):
        print("{}: {}".format(msg, val))

    # Let's show some uses of Uranus class
    print("\n" + 35 * "*")
    print("*** Use of Uranus class")
    print(35 * "*" + "\n")

    # Let's now compute the heliocentric position for a given epoch
    epoch = Epoch(2018, 10, 27.0)
    lon, lat, r = Uranus.geometric_heliocentric_position(epoch)
    print_me("Geometric Heliocentric Longitude", lon.to_positive())
    print_me("Geometric Heliocentric Latitude", lat)
    print_me("Radius vector", r)

    print("")

    # Compute the geocentric position for 1992/12/20:
    epoch = Epoch(1992, 12, 20.0)
    ra, dec, elon = Uranus.geocentric_position(epoch)
    print_me("Right ascension", ra.ra_str(n_dec=1))
    print_me("Declination", dec.dms_str(n_dec=1))
    print_me("Elongation", elon.dms_str(n_dec=1))

    print("")

    # Print mean orbital elements for Uranus at 2065.6.24
    epoch = Epoch(2065, 6, 24.0)
    l, a, e, i, ome, arg = Uranus.orbital_elements_mean_equinox(epoch)
    print_me("Mean longitude of the planet", round(l, 6))  # 235.517526
    print_me("Semimajor axis of the orbit (UA)", round(a, 8))  # 19.21844604
    print_me("Eccentricity of the orbit", round(e, 7))  # 0.0463634
    print_me("Inclination on plane of the ecliptic", round(i, 6))  # 0.77372
    print_me("Longitude of the ascending node", round(ome, 5))  # 74.34776
    print_me("Argument of the perihelion", round(arg, 6))  # 99.630865

    print("")

    # Compute the time of the conjunction close to 1993/10/1
    epoch = Epoch(1993, 10, 1.0)
    conj = Uranus.conjunction(epoch)
    y, m, d = conj.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Conjunction date", date)

    # Compute the time of the opposition close to 1780/12/1
    epoch = Epoch(1780, 12, 1.0)
    oppo = Uranus.opposition(epoch)
    y, m, d = oppo.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Opposition date", date)

    print("")

    # Find the epoch of the Perihelion closer to 1780/1/1
    epoch = Epoch(1780, 1, 1.0)
    e = Uranus.perihelion_aphelion(epoch)
    y, m, d = e.get_date()
    peri = str(y) + '/' + str(m) + '/' + str(int(d))
    print_me("The Perihelion closest to 1780/1/1 happened on", peri)

    print("")

    # Compute the time of passage through an ascending node
    epoch = Epoch(2019, 1, 1)
    time, r = Uranus.passage_nodes(epoch)
    y, m, d = time.get_date()
    d = round(d, 1)
    print("Time of passage through ascending node: {}/{}/{}".format(y, m, d))
    # 2028/8/23.2
    print("Radius vector at ascending node: {}".format(round(r, 4)))  # 19.3201


if __name__ == "__main__":

    main()
