/*
 * Copyright (c) 1983, 1988, 1993
 *  The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *  This product includes software developed by the University of
 *  California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if defined(LIBC_SCCS) && !defined(lint)
static char sccsid[] = "@(#)syslog.c  8.4 (Berkeley) 3/18/94";
#endif /* LIBC_SCCS and not lint */

/*
 * SYSLOG -- print message on log file
 *
 * This routine looks a lot like printf, except that it outputs to the
 * log file instead of the standard output.  Also:
 *  adds a timestamp,
 *  prints the module name in front of the message,
 *  has some other formatting types (or will sometime),
 *  adds a newline on the end of the message.
 *
 * The output of this routine is intended to be read by syslogd(8).
 *
 * Author: Eric Allman
 * Modified to use UNIX domain IPC by Ralph Campbell
 * Modified by Russell Coker to use the syslog() library function.  I have
 * changed this so much that any blame or credit for this code is better
 * directed at me than at the above people.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/uio.h>

#include <sys/syslog.h>

#include <netdb.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <paths.h>
#include <stdlib.h>
#include <netinet/in.h>

#include "server.h"

/* On libc5 writev() is not defined in sys/uio.h */

#ifndef _SYS_UIO_H
extern int writev(int fd, const struct iovec *vector, size_t count);
#endif

/* value -1 means closed, value -2 means local logging */
static int  LogFile = -1;    /* fd for log */
static bool connected;    /* have done connect */
static int  LogStat = 0;    /* status bits, set by openlog() */
static char *LogTag;  /* string to tag the entry with */
static int  LogFacility = LOG_LOCAL2;  /* default facility code */
static int  LogMask = 0xff;    /* mask of priorities to be logged */

static void 
closelog_intern(int to_default)
{
  if(LogFile == -2)
    closelog();
  else
    close(LogFile);
  LogFile = -1;
  connected = false;
  if(to_default)
  {
    LogStat = 0;
    free(LogTag);
    LogTag = NULL;
    LogFacility = LOG_LOCAL2;
    LogMask = 0xff;
  }
}

static void nvsyslog(int pri, const char *fmt, va_list ap);

/*
 * syslog, vsyslog --
 *     print message on log file; output is intended for syslogd(8).
 */
void
nsyslog(int pri, const char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  nvsyslog(pri, fmt, ap);
  va_end(ap);
}

void nopenlog(const char *ident, int logstat, int logfac);

static void
nvsyslog(int pri, const char *fmt, va_list ap)
{
  int cnt;
  char *tmp_ptr;
  time_t now;
  int saved_errno;
  char tbuf[2048], fmt_cpy[1024], *stdp = 0;
  char *real_message = tbuf;

  saved_errno = errno;

  /* See if we should just throw out this message. */
  if(pri && LogMask < LOG_PRI(pri))
    return;

  if(LogFile < 0 || !connected)
    nopenlog(NULL, LogStat | LOG_NDELAY, 0);

  /* Set default facility if none specified. */
  if((pri & LOG_FACMASK) == 0)
  {
    pri |= LogFacility;
  }

  tmp_ptr = tbuf;
  if(LogFile != -2) /* If not doing local logging */
  {
    /* Build the message. */
    time(&now);
    snprintf(tbuf, sizeof(tbuf), "<%d>%.15s ", pri, ctime(&now) + 4);
    for(tmp_ptr = tbuf; *tmp_ptr; ++tmp_ptr);
    if(LogStat & LOG_PERROR)
      stdp = tmp_ptr;
    /* NB LogTag is no longer than 50 bytes so we won't run out of buffer */
    if(LogTag)
      strcpy(tmp_ptr, LogTag);
    else
      strcpy(tmp_ptr, "portslave");
    for (; *tmp_ptr; ++tmp_ptr);
    if(LogStat & LOG_PID)
    {
      snprintf(tmp_ptr, sizeof(tbuf)-(tmp_ptr-tbuf), "[%d]", getpid());
      for (; *tmp_ptr; ++tmp_ptr);
    }
    *tmp_ptr++ = ':';
  } /* end if(LogFile != -2) */

  /* Substitute error message for %m. */
  {
    /* We have to make sure we don't overrun fmt_cpy. */
    int fmt_ind = 0, fmt_cpy_ind = 0;
    for (; fmt[fmt_ind] != '\0' && fmt_cpy_ind < ((int)sizeof(fmt_cpy) - 1); )
    {
      if(fmt[fmt_ind] == '%' && fmt[fmt_ind + 1] == 'm')
      {
        fmt_ind++;
        strncpy(&fmt_cpy[fmt_cpy_ind], strerror(saved_errno), sizeof(fmt_cpy) - fmt_cpy_ind - 1);
        fmt_cpy[sizeof(fmt_cpy) - 1] = '\0';
        fmt_cpy_ind = strlen(fmt_cpy);
      }
      else
      {
        fmt_cpy[fmt_cpy_ind] = fmt[fmt_ind];
        fmt_cpy_ind++;
      }
      fmt_ind++;
    }
    fmt_cpy[fmt_cpy_ind] = '\0';
  }

  real_message = tmp_ptr;
  tmp_ptr += vsnprintf(tmp_ptr, sizeof(tbuf) - (tmp_ptr - tbuf), fmt_cpy, ap);
  cnt = tmp_ptr - tbuf;

  /* Output to stderr if requested. */
  if(LogStat & LOG_PERROR)
  {
    struct iovec iov[2];
    struct iovec *v = iov;
    char c = '\n'; /* final new-line on string */

    v->iov_base = stdp;
    v->iov_len = cnt - (stdp - tbuf);
    ++v;
    v->iov_base = &c;
    v->iov_len = 1;
    writev(STDERR_FILENO, iov, 2);
  }

  /* Output the message to the local logger. */
  /* Use NULL as a message delimiter. */
  if(LogFile == -2)
  {
    syslog(pri, "%s", real_message);
  }
  else if(write(LogFile, tbuf, cnt + 1) >= 1)
  {
    return;
  }
  else if(!lineconf.syslog)
  {
    /* If the write fails, we try to reconnect it next
     * time. */
    closelog_intern (0);
  }
}

static int nsetlogmask(int pmask);

/*
 * OPENLOG -- open system log
 */
void
nopenlog(const char *ident, int logstat, int logfac)
{
  struct sockaddr_in s_in;

  closelog();

  if(LogTag)
    free(LogTag);
  if(ident)
  {
    LogTag = xstrdup(ident);
    if(strlen(LogTag) > 50)
      LogTag[50] = '\0';
  }
  else
  {
    if(GetPortNo() >= 0)
    {
      char buf[64];
      snprintf(buf, sizeof (buf), "port[S%d]", GetPortNo());
      LogTag = xstrdup(buf);
    }
    else
    {
      LogTag = xstrdup("portslave");
    }
  }
  switch(lineconf.debug)
  {
  case 1:
    nsetlogmask(LOG_INFO);
  break;
  case 2:
    nsetlogmask(LOG_DEBUG);
  break;
  default:
    nsetlogmask(LOG_NOTICE);
  }
  LogStat = logstat;
  if(logfac != 0 && (logfac &~ LOG_FACMASK) == 0)
    LogFacility = LOG_MAKEPRI(LOG_FAC(logfac), 0);
  else
  {
    if(lineconf.facility)
      LogFacility = LOG_MAKEPRI(LOG_FAC(LOG_LOCAL0) + lineconf.facility, 0);
  }
  if(LogFile >= 0)
    close(LogFile);
  if(lineconf.syslog)
  {
    s_in.sin_family = AF_INET;
    s_in.sin_port   = htons(514);
    s_in.sin_addr.s_addr = lineconf.syslog;
    if((LogFile = socket(AF_INET, SOCK_DGRAM, 0)) != -1)
    {
      if(connect(LogFile, (const struct sockaddr *)&s_in, sizeof(s_in)) != -1)
        connected = true;
    }
  }
  else
  {
    openlog(LogTag, logstat, LogFacility);
    LogFile = -2;
    connected = true;
  }
}

/*
 * CLOSELOG -- close the system log
 */
void
ncloselog()
{
  closelog_intern(1);
}

/*
 * SETLOGMASK -- set the log mask level
 */
static int
nsetlogmask(int pmask)
{
  int omask;

  omask = LogMask;
  if(pmask != 0)
    LogMask = LOG_PRI(pmask);
  return (omask);
}

