//
// SecureStreamSocket.h
//
// $Id: //poco/1.3/NetSSL_OpenSSL/include/Poco/Net/SecureStreamSocket.h#6 $
//
// Library: NetSSL_OpenSSL
// Package: SSLSockets
// Module:  SecureStreamSocket
//
// Definition of the SecureStreamSocket class.
//
// Copyright (c) 2006-2009, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef NetSSL_SecureStreamSocket_INCLUDED
#define NetSSL_SecureStreamSocket_INCLUDED


#include "Poco/Net/NetSSL.h"
#include "Poco/Net/StreamSocket.h"
#include "Poco/Net/Context.h"
#include "Poco/Net/X509Certificate.h"


namespace Poco {
namespace Net {


class NetSSL_API SecureStreamSocket: public StreamSocket
	/// A subclass of StreamSocket for secure SSL sockets.
	///
	/// A few notes about nonblocking IO:
	/// sendBytes() and receiveBytes() can return a
	/// negative value when using a nonblocking socket, which means 
	/// a SSL handshake is currently in progress and more data
	/// needs to be read or written for the handshake to continue.
	/// If sendBytes() or receiveBytes() return ERR_SSL_WANT_WRITE,
	/// sendBytes() must be called as soon as possible (usually, after
	/// select() indicates that data can be written). Likewise, if
	/// ERR_SSL_WANT_READ is returned, receiveBytes() must be called
	/// as soon as data is available for reading (indicated by select()).
	///
	/// The SSL handshake is delayed until the first sendBytes() or 
	/// receiveBytes() operation is performed on the socket. No automatic
	/// post connection check (checking the peer certificate for a valid
	/// hostname) is performed when using nonblocking I/O. To manually
	/// perform peer certificate validation, call verifyPeerCertificate()
	/// after the SSL handshake has been completed.
{
public:
	enum
	{
		ERR_SSL_WANT_READ  = -1,
		ERR_SSL_WANT_WRITE = -2
	};
	
	SecureStreamSocket();
		/// Creates an unconnected secure stream socket
		/// using the default client SSL context.
		///
		/// Before sending or receiving data, the socket
		/// must be connected with a call to connect().

	explicit SecureStreamSocket(Context::Ptr pContext);
		/// Creates an unconnected secure stream socket
		/// using the given SSL context.
		///
		/// Before sending or receiving data, the socket
		/// must be connected with a call to connect().

	explicit SecureStreamSocket(const SocketAddress& address);
		/// Creates a secure stream socket using the default 
		/// client SSL context and connects it to
		/// the socket specified by address.

	SecureStreamSocket(const SocketAddress& address, Context::Ptr pContext);
		/// Creates a secure stream socket using the given 
		/// client SSL context and connects it to
		/// the socket specified by address.

	SecureStreamSocket(const SocketAddress& address, const std::string& hostName);
		/// Creates a secure stream socket using the default 
		/// client SSL context and connects it to
		/// the socket specified by address.
		///
		/// The given host name is used for certificate verification.

	SecureStreamSocket(const SocketAddress& address, const std::string& hostName, Context::Ptr pContext);
		/// Creates a secure stream socket using the given 
		/// client SSL context and connects it to
		/// the socket specified by address.
		///
		/// The given host name is used for certificate verification.

	SecureStreamSocket(const Socket& socket);
		/// Creates the SecureStreamSocket with the SocketImpl
		/// from another socket. The SocketImpl must be
		/// a SecureStreamSocketImpl, otherwise an InvalidArgumentException
		/// will be thrown.

	virtual ~SecureStreamSocket();
		/// Destroys the StreamSocket.

	SecureStreamSocket& operator = (const Socket& socket);
		/// Assignment operator.
		///
		/// Releases the socket's SocketImpl and
		/// attaches the SocketImpl from the other socket and
		/// increments the reference count of the SocketImpl.	
	
	X509Certificate peerCertificate() const;
		/// Returns the peer's X509 certificate.
		
	void setPeerHostName(const std::string& hostName);
		/// Sets the peer's host name used for certificate validation.
		
	const std::string& getPeerHostName() const;
		/// Returns the peer's host name used for certificate validation.

	static SecureStreamSocket attach(const StreamSocket& streamSocket);
		/// Creates a SecureStreamSocket over an existing socket
		/// connection. The given StreamSocket must be connected.
		/// A SSL handshake will be performed.

	static SecureStreamSocket attach(const StreamSocket& streamSocket, Context::Ptr pContext);
		/// Creates a SecureStreamSocket over an existing socket
		/// connection. The given StreamSocket must be connected.
		/// A SSL handshake will be performed.

	static SecureStreamSocket attach(const StreamSocket& streamSocket, const std::string& peerHostName);
		/// Creates a SecureStreamSocket over an existing socket
		/// connection. The given StreamSocket must be connected.
		/// A SSL handshake will be performed.

	static SecureStreamSocket attach(const StreamSocket& streamSocket, const std::string& peerHostName, Context::Ptr pContext);
		/// Creates a SecureStreamSocket over an existing socket
		/// connection. The given StreamSocket must be connected.
		/// A SSL handshake will be performed.

	Context::Ptr context() const;
		/// Returns the SSL context used by this socket.
		
	void setLazyHandshake(bool flag = true);
		/// Enable lazy SSL handshake. If enabled, the SSL handshake
		/// will be performed the first time date is sent or
		/// received over the connection.
		
	bool getLazyHandshake() const;
		/// Returns true if setLazyHandshake(true) has been called.
		
	void verifyPeerCertificate();
		/// Performs post-connect (or post-accept) peer certificate validation,
		/// using the peer host name set with setPeerHostName(), or the peer's
		/// IP address string if no peer host name has been set.
		///
		/// Should only be used for non-blocking connections, after the
		/// initial SSL handshake has been performed (see completeHandshake()).

	void verifyPeerCertificate(const std::string& hostName);
		/// Performs post-connect (or post-accept) peer certificate validation
		/// using the given host name.
		///
		/// Should only be used for non-blocking connections, after the
		/// initial SSL handshake has been performed (see completeHandshake()).
		
	int completeHandshake();
		/// Completes the SSL handshake.
		///
		/// If the SSL connection was the result of an accept(),
		/// the server-side handshake is completed, otherwise
		/// a client-side handshake is performed.
		///
		/// Returns 1 if the handshake was successful, ERR_SSL_WANT_READ or
		/// ERR_SSL_WANT_WRITE if more data is required to complete the
		/// handshake. In this case, completeHandshake() should be called
		/// again, after the necessary condition has been met.
		
protected:
	SecureStreamSocket(SocketImpl* pImpl);

	friend class SecureServerSocket;
};


} } // namespace Poco::Net


#endif // NetSSL_SecureStreamSocket_INCLUDED
