//*******************************************************************
//
// License:  See top level LICENSE.txt file.
// 
// DESCRIPTION:
//   Contains main module for ecg2ocg program. This command line app converts
//   MET enhanced coarse grid geometry/grid files into the OSSIM coarse grid
//   format. Usage:
//
//   ecg2ocg <ecg_geom_file> <ecg_grid_file>
//
//   The output are 2 files: the geometry keywordlist file
//   (<ecg_geom_file>_ocg.kwl) and the grid data file
//   (<ecg_geom_file>_ocg.dat).
//
// SOFTWARE HISTORY:
//>
//   12JUN2001  Oscar Kramer (okramer@imagelinks.com)
//              Initial coding.
//<
//*****************************************************************************
//  $Id: ecg2ocg.cpp 9966 2006-11-29 02:01:07Z gpotts $

#include <ossim/projection/ossimCoarseGridModel.h>
#include <fstream>
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimKeywordNames.h>
#include <ossim/init/ossimInit.h>

int main (int argc, char** argv)
{
   ossimInit::instance()->initialize(argc, argv);

   if (argc < 3)
   {
      cout << "\nUsage: " << argv[0] << " <ecg_geom_file> <ecg_grid_file> <optional output geom name>"
           << endl;
      return -1;
   }

   ossimFilename metGeomName (argv[1]);
   ossimFilename metGridName (argv[2]);

   ossimCoarseGridModel model;
   model.loadMetEcgGeom(metGeomName, metGridName);
   
   if (model.getErrorStatus())
   {
      cout << "Error status set during MET ECG load. Check taht the MET ECG "
           << "files specified exist, are readable, and are properly formatted"
           << ". " << endl;
      return -1;
   }
   ossimKeywordlist kwl;
   model.saveState(kwl);

   ossimFilename geomFile;
   ossimFilename gridFile;
   if(argc == 4)
   {
      geomFile = argv[3];
   }
   else
   {
      geomFile = metGeomName.fileNoExtension();
      geomFile += "_ocg";
      geomFile.setExtension("geom");
      
   }

   gridFile = geomFile;
   gridFile.setExtension("dat");

   //---
   // NOTE:
   // The ossimCoarseGridModel::saveState has a hard coded grid file name!
   // So:
   // Override the "grid_file_name" keyword with the correct name.
   //---
   kwl.add("grid_file_name",         // keyword
           gridFile.file().c_str(),  // file name with no absolute path
           true);                    // replace existing
           
   kwl.write(geomFile);
   model.saveCoarseGrid(gridFile);
   
   cout << "\nECG geometry and grid converted successfully."
        << "\nWrote geometry file:  " << geomFile.c_str()
        << "\nWrote grid file:      " << gridFile.c_str()      
        << endl;
   return 0;
}   
