/*
 * Copyright 2014 Higher Frequency Trading
 *
 * http://www.higherfrequencytrading.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.chronicle.queue.stateless;

import net.openhft.chronicle.bytes.Bytes;
import net.openhft.chronicle.queue.ChronicleQueue;
import net.openhft.chronicle.queue.Excerpt;
import net.openhft.chronicle.queue.ExcerptComparator;
import net.openhft.chronicle.queue.stateless.bytes.StatelessRawBytesTailer;
import net.openhft.chronicle.wire.Wire;
import net.openhft.chronicle.wire.WireIn;
import org.jetbrains.annotations.NotNull;

import java.util.function.Consumer;
import java.util.function.Function;

/**
 * todo : currently work in process
 *
 * Created by Rob Austin
 */
public class StatelessExcerpt implements Excerpt {

    private final Function<Bytes, Wire> wireFunction;
    private final StatelessRawBytesTailer statelessRawBytesTailer;
    private final ChronicleQueue chronicleQueue;
    private long index;

    public StatelessExcerpt(ChronicleQueue chronicleQueue, Function<Bytes, Wire> wireFunction,
                            StatelessRawBytesTailer statelessRawBytesTailer) {
        this.wireFunction = wireFunction;
        this.statelessRawBytesTailer = statelessRawBytesTailer;
        this.chronicleQueue = chronicleQueue;
    }

    /**
     * The wire associated with the current index, use {@link StatelessTailer#index} to move onto
     * the next index
     *
     * @return the wire generated by the {@code wireFunction} and populated with the {@code bytes}
     */
    @Override
    public Wire wire() {
        final Bytes bytes = statelessRawBytesTailer.readExcept(index);
        return wireFunction.apply(bytes);
    }

    @Override
    public boolean readDocument(@NotNull ReadMarshallable reader) {
        reader.readMarshallable(wire());
        return true;
    }

    @Override
    public long findMatch(@NotNull ExcerptComparator comparator) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void findRange(@NotNull long[] startEnd, @NotNull ExcerptComparator comparator) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean index(long l) {
        index = l;
        return statelessRawBytesTailer.index(l);
    }

    @NotNull
    @Override
    public Excerpt toStart() {
        index = 0;
        return this;
    }

    @NotNull
    @Override
    public Excerpt toEnd() {
        index = statelessRawBytesTailer.lastWrittenIndex();
        return this;
    }

    @Override
    public ChronicleQueue chronicle() {
        return chronicleQueue;
    }
}
