------------------------------------------------------------------------------
--                                  G P S                                   --
--                                                                          --
--                     Copyright (C) 2007-2013, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

--  <summary>Main package for the docgen2 module</summary>
--  <description>
--  Docgen2 is the module handling automatic documentation generation in GPS.
--
--  The documentation analysis is performed in 3 different steps:
--
--  * A first step analyse the involved file/project and collects various
--  informations on all entitities
--
--  * A second step dumps those information to document the APIs
--
--  * The last step preforms a pretty print pass on all source files, and
--  creates hyperlinks when an entity references a documented API.
--
--  Finaly, index files are created, and the appropriate viewer is launched.
--
--  Docgen2 is intended to support several backends for documentation
--  generation but only an HTML backend is implemented for now.
--  </description>

with Ada.Containers.Vectors;
with Ada.Strings.Unbounded;
with GNAT.Expect;
with GNATCOLL.Projects;       use GNATCOLL.Projects;
with Docgen2_Backend;
with GPS.Kernel;
with GNATCOLL.Traces;
with GNATCOLL.VFS;

package Docgen2 is
   DOCGEN_V3 : constant GNATCOLL.Traces.Trace_Handle :=
     GNATCOLL.Traces.Create ("Docgen.V3", GNATCOLL.Traces.Off);
   --  Whether to enable the prototype of Docgen version 3

   package User_Tags_List is new Ada.Containers.Vectors
     (Natural,
      Ada.Strings.Unbounded.Unbounded_String,
      Ada.Strings.Unbounded."=");

   type Docgen_Options is record
      Process_Body_Files      : Boolean := False;
      --  Create also the body documentation
      Comments_Filter         : GNAT.Expect.Pattern_Matcher_Access;
      --  Filter comments
      Show_Private            : Boolean := False;
      --  Show also private entities
      References              : Boolean := False;
      --  True if the program should search for the references
      --  Adding information like "subprogram called by..."
      Process_Up_To_Date_Only : Boolean := True;
      --  True if docgen should process only files having up-to-date cross refs
      Spawn_Browser           : Boolean := True;
      --  True if docgen should spawn a browser after having generated the doc
      Generate_Comment_Xref   : Boolean := True;
      --  True if docgen should try to find entity names in comments and
      --  generate cross references in this case
   end record;

   procedure Generate
     (Kernel  : not null access GPS.Kernel.Kernel_Handle_Record'Class;
      Backend : Docgen2_Backend.Backend_Handle;
      File    : GNATCOLL.VFS.Virtual_File;
      Options : Docgen_Options);
   --  <summary>
   --  Generate documentation for a single file using Backend.
   --  </summary>
   --  <parameter name="Kernel">
   --    The GPS kernel object
   --  </parameter>
   --  <parameter name="Backend">
   --    The backend used for generating the doc
   --  </parameter>
   --  <parameter name="File">
   --    The file that is to be documented
   --  </parameter>
   --  <parameter name="Options">
   --    Docgen user options
   --  </parameter>

   procedure Generate
     (Kernel    : not null access GPS.Kernel.Kernel_Handle_Record'Class;
      Backend   : Docgen2_Backend.Backend_Handle;
      Project   : Project_Type;
      Options   : Docgen_Options;
      Recursive : Boolean := False);
   --  <summary>
   --  Generate documentation for a project using Backend.
   --  </summary>
   --  <parameter name="Kernel">
   --    The GPS kernel object
   --  </parameter>
   --  <parameter name="Backend">
   --    The backend used for generating the doc
   --  </parameter>
   --  <parameter name="Project">
   --    The project that is to be documented
   --  </parameter>
   --  <parameter name="Options">
   --    Docgen user options
   --  </parameter>
   --  <parameter name="Recursive">
   --    If false, then only the project's source files are documented.
   --    Else, imported project's source files are also documented.
   --  </parameter>

   type Docgen_Object is private;

   procedure Generate_Custom_Docgen_File
     (Command  : Docgen_Object;
      Name     : String;
      Filename : String;
      Content  : String);
   --  Used by docgen2.hooks for customized user-generated files.

   function Get_Kernel (D : Docgen_Object) return GPS.Kernel.Kernel_Handle;
   --  Get kernel from docgen object

   function Get_Backend
     (D : Docgen_Object) return Docgen2_Backend.Backend_Handle;
   --  Get selected backend from docgen object

   function Get_Options
     (D : Docgen_Object) return Docgen_Options;
   --  Get current docgen options from docgen object

   function Get_Doc_Directory
     (Object : Docgen_Object) return GNATCOLL.VFS.Virtual_File;
   --  Return the directory in which the documentation will be generated

   function Get_Current_File
     (Object : Docgen_Object) return GNATCOLL.VFS.Virtual_File;
   --  Return the file currently analysed

private

   type Docgen_Command;
   type Docgen_Object is access all Docgen_Command'Class;

end Docgen2;
