#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif

#include <stdlib.h>
#include <string.h>

#include <ghmm/ghmm.h>
#include <ghmm/sequence.h>
#include <ghmm/viterbi.h>
#include <ghmm/foba.h>

void analyse(ghmm_dmodel *model, ghmm_dseq *seqs, int max_seqlen) {
    int i, s, seqlen, pathlen;
    int *O, *path;
    double vlogp, flogp, *scale, **alpha;
    
    scale = malloc(max_seqlen * sizeof(*scale));
    alpha = malloc(max_seqlen * sizeof(*alpha));
    alpha[0] = malloc(max_seqlen * model->N * sizeof(**alpha));
    for (i=1; i<max_seqlen; ++i)
        alpha[i] = alpha[0] + i*model->N;

    for (i=0; i<seqs->seq_number; ++i) {
        seqlen = seqs->seq_len[i];
        O = seqs->seq[i];
        path = ghmm_dmodel_viterbi(model, O, seqlen, &pathlen, &vlogp);

        if (pathlen > 40) {
            for (s=0; s<20; s++)
                printf("%d, ", path[s]);
            printf("... , ");
            for (s=pathlen-20; s<pathlen-1; s++)
                printf("%d, ", path[s]);
            printf("%d\n", path[s]);
        }
        else if (pathlen > 0) {
            for (s=0; s<pathlen-1; s++)
                printf("%d, ", path[s]);
            printf("%d\n", path[s]);
        }

        printf("returned pathlen: %d\n", pathlen);

        free(path);

        ghmm_dmodel_forward(model, O, seqlen, alpha, scale, &flogp);

        printf("sequence %d: forward logp %g, viterbi logp %g\n", i, flogp, vlogp);        
    }
    free(alpha[0]);
    free(alpha);
    free(scale);
}

/*===========================================================================*/
int main(int argc, char **argv) {

    char *modelfile, *seqfile;
    int argnr=1, mo_number, i, max_seqlen=0;
    ghmm_dseq *seqs;
    ghmm_dmodel **readmodels, *model;

    while (argnr < argc) {
        if (!strncmp(argv[argnr], "-m", 2) ||
            !strncmp(argv[argnr], "--model", 7)) {
            argnr++;
            if (argnr < argc)
                modelfile = argv[argnr++];
            
        }
        else
            seqfile = argv[argnr++];
    }

    printf("Analysing Sequences from %s with model %s\n", seqfile, modelfile);

    readmodels = ghmm_dmodel_xml_read(modelfile, &mo_number);
    if (!readmodels || mo_number != 1)
        exit(1);
    
    model = readmodels[0];

    seqs = ghmm_dseq_open_fasta(seqfile, model->alphabet);

    for (i=0; i<seqs->seq_number; ++i) {
        if (max_seqlen < seqs->seq_len[i])
            max_seqlen = seqs->seq_len[i];
    }

    analyse(model, seqs, max_seqlen);
    return 0;
}
