//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Tests/Functional/LibFit/MinimizerTests.cpp
//! @brief     Implements classes from MinimizerTest family.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Fit/Kernel/Minimizer.h"
#include "Tests/Functional/LibFit/PlanCases.h"
#include "Tests/GTestWrapper/google_test.h"

bool runMinimizerTest(const std::string& minimizer_name, const std::string& algorithm_name,
                      MinimizerTestPlan& plan, const std::string& options = "")
{
    mumufit::Minimizer minimizer;
    minimizer.setMinimizer(minimizer_name, algorithm_name, options);
    return plan.checkMinimizer(minimizer);
}


TEST(Minimize, Minuit_Rosenbrock)
{
    RosenbrockPlan plan;
    EXPECT_TRUE(runMinimizerTest("Minuit2", "Migrad", plan));
}

TEST(Minimize, Minuit_WoodFour)
{
    WoodFourPlan plan;
    EXPECT_TRUE(runMinimizerTest("Minuit2", "Migrad", plan));
}

TEST(Minimize, Minuit_DecayingSin)
{
    DecayingSinPlan plan;
    EXPECT_TRUE(runMinimizerTest("Minuit2", "Migrad", plan));
}

/* known to fail
TEST(Minimize, SteepestDescent_Rosenbrock)
{
    RosenbrockPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "SteepestDescent", plan));
}
*/

TEST(Minimize, SteepestDescent_WoodFour)
{
    WoodFourPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "SteepestDescent", plan));
}

TEST(Minimize, ConjugateFR_Rosenbrock)
{
    RosenbrockPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "ConjugateFR", plan));
}

TEST(Minimize, ConjugateFR_WoodFour)
{
    WoodFourPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "ConjugateFR", plan));
}

TEST(Minimize, ConjugatePR_Rosenbrock)
{
    RosenbrockPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "ConjugatePR", plan));
}

TEST(Minimize, ConjugatePR_WoodFour)
{
    WoodFourPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "ConjugatePR", plan));
}

TEST(Minimize, Bfgs_Rosenbrock)
{
    RosenbrockPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "BFGS", plan));
}

TEST(Minimize, Bfgs_WoodFour)
{
    WoodFourPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "BFGS", plan));
}

TEST(Minimize, Bfgs2_Rosenbrock)
{
    RosenbrockPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "BFGS2", plan));
}

TEST(Minimize, Bfgs2_WoodFour)
{
    WoodFourPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLMultiMin", "BFGS2", plan));
}

TEST(Minimize, GSLSimAn_EasyRosenbrock)
{
    EasyRosenbrockPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLSimAn", "Simulated annealing", plan));
}

TEST(Minimize, GSLSimAn_EasyWoodFour)
{
    EasyWoodFourPlan plan;
    EXPECT_TRUE(runMinimizerTest("GSLSimAn", "Simulated annealing", plan));
}

TEST(Minimize, Genetic_EasyRosenbrock)
{
    EasyRosenbrockPlan plan;
    EXPECT_TRUE(runMinimizerTest("Genetic", "Genetic", plan, "RandomSeed=1"));
}

TEST(Minimize, Genetic_EasyWoodFour)
{
    EasyWoodFourPlan plan;
    EXPECT_TRUE(runMinimizerTest("Genetic", "Genetic", plan, "RandomSeed=1"));
}

TEST(Minimize, Fumili_DecayingSin)
{
    DecayingSinPlan plan;
    EXPECT_TRUE(runMinimizerTest("Minuit2", "Fumili", plan, "MaxFunctionCalls=10000"));
}

TEST(Minimize, LevenbergMarquardtV3)
{
    DecayingSinPlanV2 plan;
    EXPECT_TRUE(runMinimizerTest("GSLLMA", "Levenberg-Marquardt", plan));
}
