//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/ItemWithParticlesCatalog.cpp
//! @brief     Implements class ItemWithParticlesCatalog
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/ItemWithParticlesCatalog.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Sample/CompoundItem.h"
#include "GUI/Model/Sample/CoreAndShellItem.h"
#include "GUI/Model/Sample/MesocrystalItem.h"
#include "GUI/Model/Sample/ParticleItem.h"

ItemWithParticles* ItemWithParticlesCatalog::create(Type type, const MaterialModel* materials)
{
    switch (type) {
    case Type::Particle:
        return new ParticleItem(materials);
    case Type::Composition:
        return new CompoundItem(materials);
    case Type::CoreShell:
        return new CoreAndShellItem(materials);
    case Type::Mesocrystal:
        return new MesocrystalItem(materials);
    default:
        ASSERT(false);
    }
}

QVector<ItemWithParticlesCatalog::Type> ItemWithParticlesCatalog::types()
{
    return {Type::Particle, Type::Composition, Type::CoreShell, Type::Mesocrystal};
}

QVector<ItemWithParticlesCatalog::Type> ItemWithParticlesCatalog::assemblyTypes()
{
    return {Type::Composition, Type::CoreShell, Type::Mesocrystal};
}

UiInfo ItemWithParticlesCatalog::uiInfo(Type type)
{
    switch (type) {
    case Type::Particle:
        return {"Particle", "", ""}; // particle is not on UI, only its form factor
    case Type::Composition:
        return {"Particle Composition", "Composition of particles with fixed positions",
                ":/SampleDesignerToolbox/images/Compound_64x64.png"};
    case Type::CoreShell:
        return {"Core shell particle", "A particle with a core/shell geometry",
                ":/SampleDesignerToolbox/images/CoreAndShell_64x64.png"};
    case Type::Mesocrystal:
        return {"Meso Crystal", "A 3D crystal structure of nanoparticles",
                ":/SampleDesignerToolbox/images/Mesocrystal_64x64.png"};
    default:
        ASSERT(false);
    }
}

ItemWithParticlesCatalog::Type ItemWithParticlesCatalog::type(const ItemWithParticles* item)
{
    ASSERT(item);

    if (dynamic_cast<const ParticleItem*>(item))
        return Type::Particle;
    if (dynamic_cast<const CompoundItem*>(item))
        return Type::Composition;
    if (dynamic_cast<const MesocrystalItem*>(item))
        return Type::Mesocrystal;
    if (dynamic_cast<const CoreAndShellItem*>(item))
        return Type::CoreShell;

    ASSERT(false);
}
