/*
 * Copyright (c) 2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <cstdint>

namespace arm_gemm {

void a64_hybrid_s8qs_mmla_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int col_base
)
{
    struct KernelArgs {
        const int32_t *multiplier_ptr = {};
        const int32_t *shift_ptr = {};
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    if (qp->per_channel_requant) {
        flags |= 0x10;
        ka.multiplier_ptr=qp->per_channel_muls + col_base;
        ka.shift_ptr=qp->per_channel_right_shifts + col_base;
    }
    if (qp->c_offset > qp->minval) {
        flags |= 0x20;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 146f\n"
      "cmp %x[M], #0x4\n"
      "bgt 117f\n"
      "beq 88f\n"
      "cmp %x[M], #0x2\n"
      "bgt 59f\n"
      "beq 30f\n"
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "2:"  // Height 1: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "3:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "blt 9f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q7, [x9, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q6, [x9, #0x10]\n"
      "blt 8f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      "trn1 v18.2d, v1.2d, v21.2d\n"
      ".inst 0x4e87a648  // smmla v8.4s, v18.16b, v7.16b\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4e86a64c  // smmla v12.4s, v18.16b, v6.16b\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4e91a649  // smmla v9.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x40]\n"
      ".inst 0x4e90a64d  // smmla v13.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4e91a64a  // smmla v10.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x60]\n"
      ".inst 0x4e90a64e  // smmla v14.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x70]\n"
      "trn2 v1.2d, v1.2d, v21.2d\n"
      ".inst 0x4e91a64b  // smmla v11.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x80]\n"
      ".inst 0x4e90a64f  // smmla v15.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x90]\n"
      ".inst 0x4e91a428  // smmla v8.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xa0]\n"
      ".inst 0x4e90a42c  // smmla v12.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xb0]\n"
      ".inst 0x4e91a429  // smmla v9.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xc0]\n"
      ".inst 0x4e90a42d  // smmla v13.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xd0]\n"
      ".inst 0x4e91a42a  // smmla v10.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xe0]\n"
      ".inst 0x4e90a42e  // smmla v14.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xf0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e91a42b  // smmla v11.4s, v1.16b, v17.16b\n"
      ".inst 0x4e90a42f  // smmla v15.4s, v1.16b, v16.16b\n"
      "ldr q1, [x26, #0x0]\n"
      "add x9, x9, #0x100\n"
      "ldr q7, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "bge 7b\n"
      "8:"  // Height 1: Multiply loop: Single iteration only
      "trn1 v18.2d, v1.2d, v19.2d\n"
      ".inst 0x4e87a648  // smmla v8.4s, v18.16b, v7.16b\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4e86a64c  // smmla v12.4s, v18.16b, v6.16b\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4e91a649  // smmla v9.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x40]\n"
      ".inst 0x4e90a64d  // smmla v13.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4e91a64a  // smmla v10.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x60]\n"
      ".inst 0x4e90a64e  // smmla v14.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x70]\n"
      "trn2 v1.2d, v1.2d, v19.2d\n"
      ".inst 0x4e91a64b  // smmla v11.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x80]\n"
      ".inst 0x4e90a64f  // smmla v15.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x90]\n"
      ".inst 0x4e91a428  // smmla v8.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xa0]\n"
      ".inst 0x4e90a42c  // smmla v12.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xb0]\n"
      ".inst 0x4e91a429  // smmla v9.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xc0]\n"
      ".inst 0x4e90a42d  // smmla v13.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xd0]\n"
      ".inst 0x4e91a42a  // smmla v10.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xe0]\n"
      ".inst 0x4e90a42e  // smmla v14.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xf0]\n"
      "add x26, x26, #0x10\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e91a42b  // smmla v11.4s, v1.16b, v17.16b\n"
      ".inst 0x4e90a42f  // smmla v15.4s, v1.16b, v16.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "add x9, x9, #0x100\n"
      "9:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 16f\n"
      "cmp x27, #0x8\n"
      "blt 11f\n"
      "10:"  // Height 1: Multiply loop: Odd block loop
      "ldr d18, [x26], #0x8\n"
      "ldr q17, [x9, #0x0]\n"
      "trn1 v18.2d, v18.2d, v16.2d\n"
      "ldr q31, [x9, #0x10]\n"
      ".inst 0x4e91a648  // smmla v8.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4e9fa64c  // smmla v12.4s, v18.16b, v31.16b\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4e91a649  // smmla v9.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x40]\n"
      ".inst 0x4e90a64d  // smmla v13.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4e91a64a  // smmla v10.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x60]\n"
      ".inst 0x4e90a64e  // smmla v14.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x70]\n"
      "sub x27, x27, #0x8\n"
      "cmp x27, #0x8\n"
      ".inst 0x4e91a64b  // smmla v11.4s, v18.16b, v17.16b\n"
      ".inst 0x4e90a64f  // smmla v15.4s, v18.16b, v16.16b\n"
      "add x9, x9, #0x80\n"
      "bge 10b\n"
      "11:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x27, 16f\n"
      "tbz x27, #2, 13f\n"
      "ldr s1, [x26], #0x4\n"
      "tbz x27, #1, 12f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "tbz x27, #0, 15f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "b 15f\n"
      "12:"  // Height 1: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 15f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "b 15f\n"
      "13:"  // Height 1: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 14f\n"
      "ldr h1, [x26], #0x2\n"
      "tbz x27, #0, 15f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "b 15f\n"
      "14:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "15:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q17, [x9, #0x0]\n"
      "ldr q19, [x9, #0x10]\n"
      "trn1 v18.2d, v1.2d, v16.2d\n"
      ".inst 0x4e91a648  // smmla v8.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4e93a64c  // smmla v12.4s, v18.16b, v19.16b\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4e91a649  // smmla v9.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x40]\n"
      ".inst 0x4e90a64d  // smmla v13.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4e91a64a  // smmla v10.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x60]\n"
      ".inst 0x4e90a64e  // smmla v14.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x70]\n"
      ".inst 0x4e91a64b  // smmla v11.4s, v18.16b, v17.16b\n"
      ".inst 0x4e90a64f  // smmla v15.4s, v18.16b, v16.16b\n"
      "add x9, x9, #0x80\n"
      "16:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 4b\n"
      "ldr q19, [x14, #0x0]\n"
      "ldr q18, [x14, #0x10]\n"
      "uzp1 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v9.2d, v9.2d, v13.2d\n"
      "ldr q17, [x14, #0x20]\n"
      "ldr q16, [x14, #0x30]\n"
      "uzp1 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v11.2d, v11.2d, v15.2d\n"
      "mov v15.16b, v8.16b\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add v15.4s, v15.4s, v19.4s\n"
      "add x14, x14, #0x40\n"
      "add v9.4s, v9.4s, v18.4s\n"
      "add v10.4s, v10.4s, v17.4s\n"
      "add v11.4s, v11.4s, v16.4s\n"
      "tbz %x[flags], #4, 17f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 18f\n"
      "17:"  // Height 1: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "18:"  // Height 1: parameters loaded
      "sqrdmulh v15.4s, v15.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "tbz %x[flags], #5, 19f\n"
      "and v17.16b, v15.16b, v0.16b\n"
      "and v16.16b, v9.16b, v1.16b\n"
      "and v25.16b, v10.16b, v2.16b\n"
      "and v18.16b, v11.16b, v3.16b\n"
      "sshr v17.4s, v17.4s, #0x1f\n"
      "sshr v16.4s, v16.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sshr v18.4s, v18.4s, #0x1f\n"
      "sqadd v15.4s, v15.4s, v17.4s\n"
      "sqadd v9.4s, v9.4s, v16.4s\n"
      "sqadd v10.4s, v10.4s, v25.4s\n"
      "sqadd v11.4s, v11.4s, v18.4s\n"
      "19:"  // Height 1: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v18.4s }, [x20]\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add v15.4s, v15.4s, v18.4s\n"
      "add v9.4s, v9.4s, v18.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v16.4s }, [x20]\n"
      "add v10.4s, v10.4s, v18.4s\n"
      "add v11.4s, v11.4s, v18.4s\n"
      "cmp x10, #0x10\n"
      "smin v15.4s, v15.4s, v17.4s\n"
      "smin v9.4s, v9.4s, v17.4s\n"
      "smin v10.4s, v10.4s, v17.4s\n"
      "smin v11.4s, v11.4s, v17.4s\n"
      "smax v15.4s, v15.4s, v16.4s\n"
      "smax v9.4s, v9.4s, v16.4s\n"
      "smax v10.4s, v10.4s, v16.4s\n"
      "smax v11.4s, v11.4s, v16.4s\n"
      "uzp1 v15.8h, v15.8h, v9.8h\n"
      "uzp1 v16.8h, v10.8h, v11.8h\n"
      "uzp1 v15.16b, v15.16b, v16.16b\n"
      "bge 28f\n"
      "tbz x10, #3, 23f\n"
      "str d15, [x11], #0x8\n"
      "tbz x10, #2, 21f\n"
      "st1 { v15.s }[2], [x11], #0x4\n"
      "tbz x10, #1, 20f\n"
      "st1 { v15.h }[6], [x11], #0x2\n"
      "tbz x10, #0, 27f\n"
      "st1 { v15.b }[14], [x11]\n"
      "b 27f\n"
      "20:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x10, #0, 27f\n"
      "st1 { v15.b }[12], [x11]\n"
      "b 27f\n"
      "21:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x10, #1, 22f\n"
      "st1 { v15.h }[4], [x11], #0x2\n"
      "tbz x10, #0, 27f\n"
      "st1 { v15.b }[10], [x11]\n"
      "b 27f\n"
      "22:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x10, #0, 27f\n"
      "st1 { v15.b }[8], [x11]\n"
      "b 27f\n"
      "23:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x10, #2, 25f\n"
      "str s15, [x11], #0x4\n"
      "tbz x10, #1, 24f\n"
      "st1 { v15.h }[2], [x11], #0x2\n"
      "tbz x10, #0, 27f\n"
      "st1 { v15.b }[6], [x11]\n"
      "b 27f\n"
      "24:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x10, #0, 27f\n"
      "st1 { v15.b }[4], [x11]\n"
      "b 27f\n"
      "25:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x10, #1, 26f\n"
      "str h15, [x11], #0x2\n"
      "tbz x10, #0, 27f\n"
      "st1 { v15.b }[2], [x11]\n"
      "b 27f\n"
      "26:"  // Height 1: Partial direct writeback: partial_1_0
      "str b15, [x11, #0x0]\n"
      "27:"  // Height 1: Partial direct writeback: Done
      "b 29f\n"
      "28:"  // Height 1: Full writeback
      "str q15, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "29:"  // Height 1: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 2b\n"
      "b 176f\n"
      "30:"  // Height 2
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "31:"  // Height 2: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "32:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "33:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 34f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 35f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 35f\n"
      "34:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "35:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "blt 38f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q7, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "blt 37f\n"
      "36:"  // Height 2: Multiply loop: Main loop head
      "trn1 v18.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a648  // smmla v8.4s, v18.16b, v7.16b\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4e86a64c  // smmla v12.4s, v18.16b, v6.16b\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4e91a649  // smmla v9.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x40]\n"
      ".inst 0x4e90a64d  // smmla v13.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4e91a64a  // smmla v10.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x60]\n"
      ".inst 0x4e90a64e  // smmla v14.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x70]\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e91a64b  // smmla v11.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x80]\n"
      ".inst 0x4e90a64f  // smmla v15.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x90]\n"
      ".inst 0x4e91a428  // smmla v8.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xa0]\n"
      ".inst 0x4e90a42c  // smmla v12.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xb0]\n"
      ".inst 0x4e91a429  // smmla v9.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xc0]\n"
      ".inst 0x4e90a42d  // smmla v13.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xd0]\n"
      ".inst 0x4e91a42a  // smmla v10.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xe0]\n"
      ".inst 0x4e90a42e  // smmla v14.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xf0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e91a42b  // smmla v11.4s, v1.16b, v17.16b\n"
      "add x9, x9, #0x100\n"
      "ldr q7, [x9, #0x0]\n"
      ".inst 0x4e90a42f  // smmla v15.4s, v1.16b, v16.16b\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "bge 36b\n"
      "37:"  // Height 2: Multiply loop: Single iteration only
      "trn1 v18.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a648  // smmla v8.4s, v18.16b, v7.16b\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4e86a64c  // smmla v12.4s, v18.16b, v6.16b\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4e91a649  // smmla v9.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x40]\n"
      ".inst 0x4e90a64d  // smmla v13.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4e91a64a  // smmla v10.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x60]\n"
      ".inst 0x4e90a64e  // smmla v14.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x70]\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e91a64b  // smmla v11.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x80]\n"
      ".inst 0x4e90a64f  // smmla v15.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x90]\n"
      ".inst 0x4e91a428  // smmla v8.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xa0]\n"
      ".inst 0x4e90a42c  // smmla v12.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xb0]\n"
      ".inst 0x4e91a429  // smmla v9.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xc0]\n"
      ".inst 0x4e90a42d  // smmla v13.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xd0]\n"
      ".inst 0x4e91a42a  // smmla v10.4s, v1.16b, v17.16b\n"
      "ldr q17, [x9, #0xe0]\n"
      ".inst 0x4e90a42e  // smmla v14.4s, v1.16b, v16.16b\n"
      "ldr q16, [x9, #0xf0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e91a42b  // smmla v11.4s, v1.16b, v17.16b\n"
      ".inst 0x4e90a42f  // smmla v15.4s, v1.16b, v16.16b\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "add x9, x9, #0x100\n"
      "38:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 45f\n"
      "cmp x27, #0x8\n"
      "blt 40f\n"
      "39:"  // Height 2: Multiply loop: Odd block loop
      "ldr d17, [x26], #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "trn1 v18.2d, v17.2d, v16.2d\n"
      "sub x27, x27, #0x8\n"
      "ldr q17, [x9, #0x0]\n"
      "ldr q16, [x9, #0x10]\n"
      ".inst 0x4e91a648  // smmla v8.4s, v18.16b, v17.16b\n"
      ".inst 0x4e90a64c  // smmla v12.4s, v18.16b, v16.16b\n"
      "ldr q17, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4e91a649  // smmla v9.4s, v18.16b, v17.16b\n"
      ".inst 0x4e90a64d  // smmla v13.4s, v18.16b, v16.16b\n"
      "ldr q17, [x9, #0x40]\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4e91a64a  // smmla v10.4s, v18.16b, v17.16b\n"
      ".inst 0x4e90a64e  // smmla v14.4s, v18.16b, v16.16b\n"
      "ldr q17, [x9, #0x60]\n"
      "ldr q16, [x9, #0x70]\n"
      "cmp x27, #0x8\n"
      ".inst 0x4e91a64b  // smmla v11.4s, v18.16b, v17.16b\n"
      ".inst 0x4e90a64f  // smmla v15.4s, v18.16b, v16.16b\n"
      "add x9, x9, #0x80\n"
      "bge 39b\n"
      "40:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x27, 45f\n"
      "tbz x27, #2, 42f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "tbz x27, #1, 41f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "tbz x27, #0, 44f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "b 44f\n"
      "41:"  // Height 2: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 44f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "b 44f\n"
      "42:"  // Height 2: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 43f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "tbz x27, #0, 44f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "b 44f\n"
      "43:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "44:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q17, [x9, #0x0]\n"
      "ldr q16, [x9, #0x10]\n"
      "trn1 v18.2d, v1.2d, v2.2d\n"
      ".inst 0x4e91a648  // smmla v8.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x20]\n"
      ".inst 0x4e90a64c  // smmla v12.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x30]\n"
      ".inst 0x4e91a649  // smmla v9.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x40]\n"
      ".inst 0x4e90a64d  // smmla v13.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x50]\n"
      ".inst 0x4e91a64a  // smmla v10.4s, v18.16b, v17.16b\n"
      "ldr q17, [x9, #0x60]\n"
      ".inst 0x4e90a64e  // smmla v14.4s, v18.16b, v16.16b\n"
      "ldr q16, [x9, #0x70]\n"
      ".inst 0x4e91a64b  // smmla v11.4s, v18.16b, v17.16b\n"
      ".inst 0x4e90a64f  // smmla v15.4s, v18.16b, v16.16b\n"
      "add x9, x9, #0x80\n"
      "45:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 33b\n"
      "ldr q19, [x14, #0x0]\n"
      "ldr q18, [x14, #0x10]\n"
      "uzp1 v17.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q5, [x14, #0x20]\n"
      "ldr q16, [x14, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x25, x11, x20\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "mov v15.16b, v17.16b\n"
      "add v15.4s, v15.4s, v19.4s\n"
      "add x14, x14, #0x40\n"
      "add v12.4s, v12.4s, v18.4s\n"
      "add v13.4s, v13.4s, v5.4s\n"
      "add v14.4s, v14.4s, v16.4s\n"
      "add v8.4s, v8.4s, v19.4s\n"
      "add v9.4s, v9.4s, v18.4s\n"
      "add v10.4s, v10.4s, v5.4s\n"
      "add v11.4s, v11.4s, v16.4s\n"
      "tbz %x[flags], #4, 46f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 47f\n"
      "46:"  // Height 2: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "47:"  // Height 2: parameters loaded
      "sqrdmulh v15.4s, v15.4s, v4.4s\n"
      "sqrdmulh v12.4s, v12.4s, v5.4s\n"
      "sqrdmulh v13.4s, v13.4s, v6.4s\n"
      "sqrdmulh v14.4s, v14.4s, v7.4s\n"
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "tbz %x[flags], #5, 48f\n"
      "and v19.16b, v15.16b, v0.16b\n"
      "and v18.16b, v12.16b, v1.16b\n"
      "and v17.16b, v13.16b, v2.16b\n"
      "and v16.16b, v14.16b, v3.16b\n"
      "sshr v19.4s, v19.4s, #0x1f\n"
      "sshr v18.4s, v18.4s, #0x1f\n"
      "sshr v17.4s, v17.4s, #0x1f\n"
      "sshr v16.4s, v16.4s, #0x1f\n"
      "sqadd v15.4s, v15.4s, v19.4s\n"
      "sqadd v12.4s, v12.4s, v18.4s\n"
      "sqadd v13.4s, v13.4s, v17.4s\n"
      "sqadd v14.4s, v14.4s, v16.4s\n"
      "and v19.16b, v8.16b, v0.16b\n"
      "and v18.16b, v9.16b, v1.16b\n"
      "and v17.16b, v10.16b, v2.16b\n"
      "and v16.16b, v11.16b, v3.16b\n"
      "sshr v19.4s, v19.4s, #0x1f\n"
      "sshr v18.4s, v18.4s, #0x1f\n"
      "sshr v17.4s, v17.4s, #0x1f\n"
      "sshr v16.4s, v16.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v19.4s\n"
      "sqadd v9.4s, v9.4s, v18.4s\n"
      "sqadd v10.4s, v10.4s, v17.4s\n"
      "sqadd v11.4s, v11.4s, v16.4s\n"
      "48:"  // Height 2: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v18.4s }, [x20]\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v17.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v16.4s }, [x20]\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "add v15.4s, v15.4s, v18.4s\n"
      "add v12.4s, v12.4s, v18.4s\n"
      "add v13.4s, v13.4s, v18.4s\n"
      "add v14.4s, v14.4s, v18.4s\n"
      "add v8.4s, v8.4s, v18.4s\n"
      "add v9.4s, v9.4s, v18.4s\n"
      "add v10.4s, v10.4s, v18.4s\n"
      "add v11.4s, v11.4s, v18.4s\n"
      "smin v15.4s, v15.4s, v17.4s\n"
      "smin v12.4s, v12.4s, v17.4s\n"
      "smin v13.4s, v13.4s, v17.4s\n"
      "smin v14.4s, v14.4s, v17.4s\n"
      "smin v8.4s, v8.4s, v17.4s\n"
      "smin v9.4s, v9.4s, v17.4s\n"
      "smin v10.4s, v10.4s, v17.4s\n"
      "smin v11.4s, v11.4s, v17.4s\n"
      "smax v15.4s, v15.4s, v16.4s\n"
      "smax v12.4s, v12.4s, v16.4s\n"
      "smax v13.4s, v13.4s, v16.4s\n"
      "smax v14.4s, v14.4s, v16.4s\n"
      "smax v8.4s, v8.4s, v16.4s\n"
      "smax v9.4s, v9.4s, v16.4s\n"
      "smax v10.4s, v10.4s, v16.4s\n"
      "smax v11.4s, v11.4s, v16.4s\n"
      "uzp1 v15.8h, v15.8h, v12.8h\n"
      "uzp1 v17.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v16.8h, v10.8h, v11.8h\n"
      "uzp1 v15.16b, v15.16b, v17.16b\n"
      "uzp1 v8.16b, v8.16b, v16.16b\n"
      "bge 57f\n"
      "tbz x10, #3, 52f\n"
      "str d15, [x11], #0x8\n"
      "str d8, [x25], #0x8\n"
      "tbz x10, #2, 50f\n"
      "st1 { v15.s }[2], [x11], #0x4\n"
      "st1 { v8.s }[2], [x25], #0x4\n"
      "tbz x10, #1, 49f\n"
      "st1 { v15.h }[6], [x11], #0x2\n"
      "st1 { v8.h }[6], [x25], #0x2\n"
      "tbz x10, #0, 56f\n"
      "st1 { v15.b }[14], [x11]\n"
      "st1 { v8.b }[14], [x25]\n"
      "b 56f\n"
      "49:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x10, #0, 56f\n"
      "st1 { v15.b }[12], [x11]\n"
      "st1 { v8.b }[12], [x25]\n"
      "b 56f\n"
      "50:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x10, #1, 51f\n"
      "st1 { v15.h }[4], [x11], #0x2\n"
      "st1 { v8.h }[4], [x25], #0x2\n"
      "tbz x10, #0, 56f\n"
      "st1 { v15.b }[10], [x11]\n"
      "st1 { v8.b }[10], [x25]\n"
      "b 56f\n"
      "51:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x10, #0, 56f\n"
      "st1 { v15.b }[8], [x11]\n"
      "st1 { v8.b }[8], [x25]\n"
      "b 56f\n"
      "52:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x10, #2, 54f\n"
      "str s15, [x11], #0x4\n"
      "str s8, [x25], #0x4\n"
      "tbz x10, #1, 53f\n"
      "st1 { v15.h }[2], [x11], #0x2\n"
      "st1 { v8.h }[2], [x25], #0x2\n"
      "tbz x10, #0, 56f\n"
      "st1 { v15.b }[6], [x11]\n"
      "st1 { v8.b }[6], [x25]\n"
      "b 56f\n"
      "53:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x10, #0, 56f\n"
      "st1 { v15.b }[4], [x11]\n"
      "st1 { v8.b }[4], [x25]\n"
      "b 56f\n"
      "54:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x10, #1, 55f\n"
      "str h15, [x11], #0x2\n"
      "str h8, [x25], #0x2\n"
      "tbz x10, #0, 56f\n"
      "st1 { v15.b }[2], [x11]\n"
      "st1 { v8.b }[2], [x25]\n"
      "b 56f\n"
      "55:"  // Height 2: Partial direct writeback: partial_1_0
      "str b15, [x11, #0x0]\n"
      "str b8, [x25, #0x0]\n"
      "56:"  // Height 2: Partial direct writeback: Done
      "b 58f\n"
      "57:"  // Height 2: Full writeback
      "str q15, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q8, [x25, #0x0]\n"
      "58:"  // Height 2: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 31b\n"
      "b 176f\n"
      "59:"  // Height 3
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "60:"  // Height 3: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "61:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "62:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 63f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 64f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 64f\n"
      "63:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "64:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "blt 67f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q7, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "blt 66f\n"
      "65:"  // Height 3: Multiply loop: Main loop head
      "trn1 v27.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a768  // smmla v8.4s, v27.16b, v7.16b\n"
      "trn1 v26.2d, v3.2d, v28.2d\n"
      ".inst 0x4e87a750  // smmla v16.4s, v26.16b, v7.16b\n"
      "ldr q25, [x9, #0x20]\n"
      ".inst 0x4e86a76c  // smmla v12.4s, v27.16b, v6.16b\n"
      ".inst 0x4e86a754  // smmla v20.4s, v26.16b, v6.16b\n"
      "ldr q24, [x9, #0x30]\n"
      ".inst 0x4e99a769  // smmla v9.4s, v27.16b, v25.16b\n"
      "trn2 v3.2d, v3.2d, v28.2d\n"
      ".inst 0x4e99a751  // smmla v17.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x40]\n"
      ".inst 0x4e98a76d  // smmla v13.4s, v27.16b, v24.16b\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e98a755  // smmla v21.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4e99a76a  // smmla v10.4s, v27.16b, v25.16b\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4e99a752  // smmla v18.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4e98a76e  // smmla v14.4s, v27.16b, v24.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e98a756  // smmla v22.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x70]\n"
      ".inst 0x4e99a76b  // smmla v11.4s, v27.16b, v25.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e99a753  // smmla v19.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x80]\n"
      ".inst 0x4e98a76f  // smmla v15.4s, v27.16b, v24.16b\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e98a757  // smmla v23.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x90]\n"
      "ldr q2, [x25, #0x0]\n"
      ".inst 0x4e99a428  // smmla v8.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a470  // smmla v16.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xa0]\n"
      ".inst 0x4e98a42c  // smmla v12.4s, v1.16b, v24.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e98a474  // smmla v20.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xb0]\n"
      ".inst 0x4e99a429  // smmla v9.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e99a471  // smmla v17.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xc0]\n"
      ".inst 0x4e98a42d  // smmla v13.4s, v1.16b, v24.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e98a475  // smmla v21.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xd0]\n"
      ".inst 0x4e99a42a  // smmla v10.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a472  // smmla v18.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xe0]\n"
      ".inst 0x4e98a42e  // smmla v14.4s, v1.16b, v24.16b\n"
      ".inst 0x4e98a476  // smmla v22.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e99a42b  // smmla v11.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a473  // smmla v19.4s, v3.16b, v25.16b\n"
      "ldr q7, [x9, #0x0]\n"
      ".inst 0x4e98a42f  // smmla v15.4s, v1.16b, v24.16b\n"
      "ldr q1, [x26, #0x0]\n"
      ".inst 0x4e98a477  // smmla v23.4s, v3.16b, v24.16b\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "bge 65b\n"
      "66:"  // Height 3: Multiply loop: Single iteration only
      "trn1 v27.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a768  // smmla v8.4s, v27.16b, v7.16b\n"
      "trn1 v26.2d, v3.2d, v25.2d\n"
      ".inst 0x4e87a750  // smmla v16.4s, v26.16b, v7.16b\n"
      "ldr q24, [x9, #0x20]\n"
      ".inst 0x4e86a76c  // smmla v12.4s, v27.16b, v6.16b\n"
      ".inst 0x4e86a754  // smmla v20.4s, v26.16b, v6.16b\n"
      "ldr q0, [x9, #0x30]\n"
      ".inst 0x4e98a769  // smmla v9.4s, v27.16b, v24.16b\n"
      "trn2 v3.2d, v3.2d, v25.2d\n"
      ".inst 0x4e98a751  // smmla v17.4s, v26.16b, v24.16b\n"
      "ldr q25, [x9, #0x40]\n"
      ".inst 0x4e80a76d  // smmla v13.4s, v27.16b, v0.16b\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4e80a755  // smmla v21.4s, v26.16b, v0.16b\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4e99a76a  // smmla v10.4s, v27.16b, v25.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e99a752  // smmla v18.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4e98a76e  // smmla v14.4s, v27.16b, v24.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e98a756  // smmla v22.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x70]\n"
      ".inst 0x4e99a76b  // smmla v11.4s, v27.16b, v25.16b\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e99a753  // smmla v19.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x80]\n"
      ".inst 0x4e98a76f  // smmla v15.4s, v27.16b, v24.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e98a757  // smmla v23.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x90]\n"
      ".inst 0x4e99a428  // smmla v8.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e99a470  // smmla v16.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xa0]\n"
      ".inst 0x4e98a42c  // smmla v12.4s, v1.16b, v24.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e98a474  // smmla v20.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xb0]\n"
      ".inst 0x4e99a429  // smmla v9.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a471  // smmla v17.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xc0]\n"
      ".inst 0x4e98a42d  // smmla v13.4s, v1.16b, v24.16b\n"
      ".inst 0x4e98a475  // smmla v21.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xd0]\n"
      ".inst 0x4e99a42a  // smmla v10.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a472  // smmla v18.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xe0]\n"
      ".inst 0x4e98a42e  // smmla v14.4s, v1.16b, v24.16b\n"
      ".inst 0x4e98a476  // smmla v22.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e99a42b  // smmla v11.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a473  // smmla v19.4s, v3.16b, v25.16b\n"
      ".inst 0x4e98a42f  // smmla v15.4s, v1.16b, v24.16b\n"
      ".inst 0x4e98a477  // smmla v23.4s, v3.16b, v24.16b\n"
      "67:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 74f\n"
      "cmp x27, #0x8\n"
      "blt 69f\n"
      "68:"  // Height 3: Multiply loop: Odd block loop
      "ldr d25, [x26], #0x8\n"
      "ldr d24, [x25], #0x8\n"
      "trn1 v27.2d, v25.2d, v24.2d\n"
      "ldr d24, [x24], #0x8\n"
      "ldr q25, [x9, #0x0]\n"
      "trn1 v26.2d, v24.2d, v26.2d\n"
      ".inst 0x4e99a768  // smmla v8.4s, v27.16b, v25.16b\n"
      "ldr q24, [x9, #0x10]\n"
      ".inst 0x4e99a750  // smmla v16.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x20]\n"
      ".inst 0x4e98a76c  // smmla v12.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a754  // smmla v20.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x30]\n"
      ".inst 0x4e99a769  // smmla v9.4s, v27.16b, v25.16b\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x4e99a751  // smmla v17.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x40]\n"
      ".inst 0x4e98a76d  // smmla v13.4s, v27.16b, v24.16b\n"
      "cmp x27, #0x8\n"
      ".inst 0x4e98a755  // smmla v21.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4e99a76a  // smmla v10.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a752  // smmla v18.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4e98a76e  // smmla v14.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a756  // smmla v22.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x70]\n"
      ".inst 0x4e99a76b  // smmla v11.4s, v27.16b, v25.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e99a753  // smmla v19.4s, v26.16b, v25.16b\n"
      ".inst 0x4e98a76f  // smmla v15.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a757  // smmla v23.4s, v26.16b, v24.16b\n"
      "bge 68b\n"
      "69:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x27, 74f\n"
      "tbz x27, #2, 71f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "tbz x27, #1, 70f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "tbz x27, #0, 73f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "b 73f\n"
      "70:"  // Height 3: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 73f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "b 73f\n"
      "71:"  // Height 3: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 72f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "tbz x27, #0, 73f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "b 73f\n"
      "72:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "ldr b3, [x24, #0x0]\n"
      "73:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q25, [x9, #0x0]\n"
      "ldr q28, [x9, #0x10]\n"
      "trn1 v27.2d, v1.2d, v2.2d\n"
      "trn1 v26.2d, v3.2d, v24.2d\n"
      ".inst 0x4e99a768  // smmla v8.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a750  // smmla v16.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x20]\n"
      ".inst 0x4e9ca76c  // smmla v12.4s, v27.16b, v28.16b\n"
      ".inst 0x4e9ca754  // smmla v20.4s, v26.16b, v28.16b\n"
      "ldr q24, [x9, #0x30]\n"
      ".inst 0x4e99a769  // smmla v9.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a751  // smmla v17.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x40]\n"
      ".inst 0x4e98a76d  // smmla v13.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a755  // smmla v21.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4e99a76a  // smmla v10.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a752  // smmla v18.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4e98a76e  // smmla v14.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a756  // smmla v22.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e99a76b  // smmla v11.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a753  // smmla v19.4s, v26.16b, v25.16b\n"
      ".inst 0x4e98a76f  // smmla v15.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a757  // smmla v23.4s, v26.16b, v24.16b\n"
      "74:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 62b\n"
      "ldr q28, [x14, #0x0]\n"
      "ldr q27, [x14, #0x10]\n"
      "uzp1 v26.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q25, [x14, #0x20]\n"
      "ldr q24, [x14, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x25, x11, x20\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "add x24, x25, x20\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "uzp1 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v17.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp1 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v19.2d, v19.2d, v23.2d\n"
      "add x14, x14, #0x40\n"
      "mov v23.16b, v26.16b\n"
      "add v23.4s, v23.4s, v28.4s\n"
      "add v12.4s, v12.4s, v27.4s\n"
      "add v13.4s, v13.4s, v25.4s\n"
      "add v14.4s, v14.4s, v24.4s\n"
      "add v8.4s, v8.4s, v28.4s\n"
      "add v9.4s, v9.4s, v27.4s\n"
      "add v10.4s, v10.4s, v25.4s\n"
      "add v11.4s, v11.4s, v24.4s\n"
      "add v16.4s, v16.4s, v28.4s\n"
      "add v17.4s, v17.4s, v27.4s\n"
      "add v18.4s, v18.4s, v25.4s\n"
      "add v19.4s, v19.4s, v24.4s\n"
      "tbz %x[flags], #4, 75f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 76f\n"
      "75:"  // Height 3: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "76:"  // Height 3: parameters loaded
      "sqrdmulh v23.4s, v23.4s, v4.4s\n"
      "sqrdmulh v12.4s, v12.4s, v5.4s\n"
      "sqrdmulh v13.4s, v13.4s, v6.4s\n"
      "sqrdmulh v14.4s, v14.4s, v7.4s\n"
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "sqrdmulh v16.4s, v16.4s, v4.4s\n"
      "sqrdmulh v17.4s, v17.4s, v5.4s\n"
      "sqrdmulh v18.4s, v18.4s, v6.4s\n"
      "sqrdmulh v19.4s, v19.4s, v7.4s\n"
      "tbz %x[flags], #5, 77f\n"
      "and v24.16b, v23.16b, v0.16b\n"
      "and v22.16b, v12.16b, v1.16b\n"
      "and v21.16b, v13.16b, v2.16b\n"
      "and v20.16b, v14.16b, v3.16b\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sshr v22.4s, v22.4s, #0x1f\n"
      "sshr v21.4s, v21.4s, #0x1f\n"
      "sshr v20.4s, v20.4s, #0x1f\n"
      "sqadd v23.4s, v23.4s, v24.4s\n"
      "sqadd v12.4s, v12.4s, v22.4s\n"
      "sqadd v13.4s, v13.4s, v21.4s\n"
      "sqadd v14.4s, v14.4s, v20.4s\n"
      "and v24.16b, v8.16b, v0.16b\n"
      "and v22.16b, v9.16b, v1.16b\n"
      "and v21.16b, v10.16b, v2.16b\n"
      "and v20.16b, v11.16b, v3.16b\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sshr v22.4s, v22.4s, #0x1f\n"
      "sshr v21.4s, v21.4s, #0x1f\n"
      "sshr v20.4s, v20.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v24.4s\n"
      "sqadd v9.4s, v9.4s, v22.4s\n"
      "sqadd v10.4s, v10.4s, v21.4s\n"
      "sqadd v11.4s, v11.4s, v20.4s\n"
      "and v24.16b, v16.16b, v0.16b\n"
      "and v22.16b, v17.16b, v1.16b\n"
      "and v21.16b, v18.16b, v2.16b\n"
      "and v20.16b, v19.16b, v3.16b\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sshr v22.4s, v22.4s, #0x1f\n"
      "sshr v21.4s, v21.4s, #0x1f\n"
      "sshr v20.4s, v20.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v24.4s\n"
      "sqadd v17.4s, v17.4s, v22.4s\n"
      "sqadd v18.4s, v18.4s, v21.4s\n"
      "sqadd v19.4s, v19.4s, v20.4s\n"
      "77:"  // Height 3: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v22.4s }, [x20]\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v21.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v20.4s }, [x20]\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "add v23.4s, v23.4s, v22.4s\n"
      "add v12.4s, v12.4s, v22.4s\n"
      "add v13.4s, v13.4s, v22.4s\n"
      "add v14.4s, v14.4s, v22.4s\n"
      "add v8.4s, v8.4s, v22.4s\n"
      "add v9.4s, v9.4s, v22.4s\n"
      "add v10.4s, v10.4s, v22.4s\n"
      "add v11.4s, v11.4s, v22.4s\n"
      "add v16.4s, v16.4s, v22.4s\n"
      "add v17.4s, v17.4s, v22.4s\n"
      "add v18.4s, v18.4s, v22.4s\n"
      "add v19.4s, v19.4s, v22.4s\n"
      "smin v23.4s, v23.4s, v21.4s\n"
      "smin v12.4s, v12.4s, v21.4s\n"
      "smin v13.4s, v13.4s, v21.4s\n"
      "smin v14.4s, v14.4s, v21.4s\n"
      "smin v8.4s, v8.4s, v21.4s\n"
      "smin v9.4s, v9.4s, v21.4s\n"
      "smin v10.4s, v10.4s, v21.4s\n"
      "smin v11.4s, v11.4s, v21.4s\n"
      "smin v16.4s, v16.4s, v21.4s\n"
      "smin v17.4s, v17.4s, v21.4s\n"
      "smin v18.4s, v18.4s, v21.4s\n"
      "smin v19.4s, v19.4s, v21.4s\n"
      "smax v23.4s, v23.4s, v20.4s\n"
      "smax v12.4s, v12.4s, v20.4s\n"
      "smax v13.4s, v13.4s, v20.4s\n"
      "smax v14.4s, v14.4s, v20.4s\n"
      "smax v8.4s, v8.4s, v20.4s\n"
      "smax v9.4s, v9.4s, v20.4s\n"
      "smax v10.4s, v10.4s, v20.4s\n"
      "smax v11.4s, v11.4s, v20.4s\n"
      "smax v16.4s, v16.4s, v20.4s\n"
      "smax v17.4s, v17.4s, v20.4s\n"
      "smax v18.4s, v18.4s, v20.4s\n"
      "smax v19.4s, v19.4s, v20.4s\n"
      "uzp1 v23.8h, v23.8h, v12.8h\n"
      "uzp1 v21.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v20.8h, v10.8h, v11.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v23.16b, v23.16b, v21.16b\n"
      "uzp1 v8.16b, v8.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 86f\n"
      "tbz x10, #3, 81f\n"
      "str d23, [x11], #0x8\n"
      "str d8, [x25], #0x8\n"
      "str d16, [x24], #0x8\n"
      "tbz x10, #2, 79f\n"
      "st1 { v23.s }[2], [x11], #0x4\n"
      "st1 { v8.s }[2], [x25], #0x4\n"
      "st1 { v16.s }[2], [x24], #0x4\n"
      "tbz x10, #1, 78f\n"
      "st1 { v23.h }[6], [x11], #0x2\n"
      "st1 { v8.h }[6], [x25], #0x2\n"
      "st1 { v16.h }[6], [x24], #0x2\n"
      "tbz x10, #0, 85f\n"
      "st1 { v23.b }[14], [x11]\n"
      "st1 { v8.b }[14], [x25]\n"
      "st1 { v16.b }[14], [x24]\n"
      "b 85f\n"
      "78:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x10, #0, 85f\n"
      "st1 { v23.b }[12], [x11]\n"
      "st1 { v8.b }[12], [x25]\n"
      "st1 { v16.b }[12], [x24]\n"
      "b 85f\n"
      "79:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x10, #1, 80f\n"
      "st1 { v23.h }[4], [x11], #0x2\n"
      "st1 { v8.h }[4], [x25], #0x2\n"
      "st1 { v16.h }[4], [x24], #0x2\n"
      "tbz x10, #0, 85f\n"
      "st1 { v23.b }[10], [x11]\n"
      "st1 { v8.b }[10], [x25]\n"
      "st1 { v16.b }[10], [x24]\n"
      "b 85f\n"
      "80:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x10, #0, 85f\n"
      "st1 { v23.b }[8], [x11]\n"
      "st1 { v8.b }[8], [x25]\n"
      "st1 { v16.b }[8], [x24]\n"
      "b 85f\n"
      "81:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x10, #2, 83f\n"
      "str s23, [x11], #0x4\n"
      "str s8, [x25], #0x4\n"
      "str s16, [x24], #0x4\n"
      "tbz x10, #1, 82f\n"
      "st1 { v23.h }[2], [x11], #0x2\n"
      "st1 { v8.h }[2], [x25], #0x2\n"
      "st1 { v16.h }[2], [x24], #0x2\n"
      "tbz x10, #0, 85f\n"
      "st1 { v23.b }[6], [x11]\n"
      "st1 { v8.b }[6], [x25]\n"
      "st1 { v16.b }[6], [x24]\n"
      "b 85f\n"
      "82:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x10, #0, 85f\n"
      "st1 { v23.b }[4], [x11]\n"
      "st1 { v8.b }[4], [x25]\n"
      "st1 { v16.b }[4], [x24]\n"
      "b 85f\n"
      "83:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x10, #1, 84f\n"
      "str h23, [x11], #0x2\n"
      "str h8, [x25], #0x2\n"
      "str h16, [x24], #0x2\n"
      "tbz x10, #0, 85f\n"
      "st1 { v23.b }[2], [x11]\n"
      "st1 { v8.b }[2], [x25]\n"
      "st1 { v16.b }[2], [x24]\n"
      "b 85f\n"
      "84:"  // Height 3: Partial direct writeback: partial_1_0
      "str b23, [x11, #0x0]\n"
      "str b8, [x25, #0x0]\n"
      "str b16, [x24, #0x0]\n"
      "85:"  // Height 3: Partial direct writeback: Done
      "b 87f\n"
      "86:"  // Height 3: Full writeback
      "str q23, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q8, [x25, #0x0]\n"
      "str q16, [x24, #0x0]\n"
      "87:"  // Height 3: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 60b\n"
      "b 176f\n"
      "88:"  // Height 4
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "89:"  // Height 4: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "90:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "91:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 92f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 93f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 93f\n"
      "92:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "93:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "blt 96f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q4, [x23, #0x0]\n"
      "ldr q7, [x9, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "blt 95f\n"
      "94:"  // Height 4: Multiply loop: Main loop head
      "trn1 v27.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a768  // smmla v8.4s, v27.16b, v7.16b\n"
      "sub x27, x27, #0x10\n"
      "trn1 v26.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a750  // smmla v16.4s, v26.16b, v7.16b\n"
      "ldr q25, [x9, #0x20]\n"
      ".inst 0x4e86a76c  // smmla v12.4s, v27.16b, v6.16b\n"
      ".inst 0x4e86a754  // smmla v20.4s, v26.16b, v6.16b\n"
      "ldr q24, [x9, #0x30]\n"
      ".inst 0x4e99a769  // smmla v9.4s, v27.16b, v25.16b\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e99a751  // smmla v17.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x40]\n"
      ".inst 0x4e98a76d  // smmla v13.4s, v27.16b, v24.16b\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4e98a755  // smmla v21.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4e99a76a  // smmla v10.4s, v27.16b, v25.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e99a752  // smmla v18.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4e98a76e  // smmla v14.4s, v27.16b, v24.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e98a756  // smmla v22.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x70]\n"
      ".inst 0x4e99a76b  // smmla v11.4s, v27.16b, v25.16b\n"
      "add x23, x23, #0x10\n"
      "ldr q4, [x23, #0x0]\n"
      ".inst 0x4e99a753  // smmla v19.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x80]\n"
      ".inst 0x4e98a76f  // smmla v15.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a757  // smmla v23.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x90]\n"
      "ldr q2, [x25, #0x0]\n"
      ".inst 0x4e99a428  // smmla v8.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a470  // smmla v16.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xa0]\n"
      ".inst 0x4e98a42c  // smmla v12.4s, v1.16b, v24.16b\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e98a474  // smmla v20.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xb0]\n"
      ".inst 0x4e99a429  // smmla v9.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e99a471  // smmla v17.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xc0]\n"
      ".inst 0x4e98a42d  // smmla v13.4s, v1.16b, v24.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e98a475  // smmla v21.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xd0]\n"
      ".inst 0x4e99a42a  // smmla v10.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e99a472  // smmla v18.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xe0]\n"
      ".inst 0x4e98a42e  // smmla v14.4s, v1.16b, v24.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e98a476  // smmla v22.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e99a42b  // smmla v11.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a473  // smmla v19.4s, v3.16b, v25.16b\n"
      "ldr q7, [x9, #0x0]\n"
      ".inst 0x4e98a42f  // smmla v15.4s, v1.16b, v24.16b\n"
      "ldr q1, [x26, #0x0]\n"
      ".inst 0x4e98a477  // smmla v23.4s, v3.16b, v24.16b\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q6, [x9, #0x10]\n"
      "bge 94b\n"
      "95:"  // Height 4: Multiply loop: Single iteration only
      "trn1 v27.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a768  // smmla v8.4s, v27.16b, v7.16b\n"
      "add x26, x26, #0x10\n"
      "trn1 v26.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a750  // smmla v16.4s, v26.16b, v7.16b\n"
      "ldr q25, [x9, #0x20]\n"
      ".inst 0x4e86a76c  // smmla v12.4s, v27.16b, v6.16b\n"
      ".inst 0x4e86a754  // smmla v20.4s, v26.16b, v6.16b\n"
      "ldr q24, [x9, #0x30]\n"
      ".inst 0x4e99a769  // smmla v9.4s, v27.16b, v25.16b\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e99a751  // smmla v17.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x40]\n"
      ".inst 0x4e98a76d  // smmla v13.4s, v27.16b, v24.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e98a755  // smmla v21.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4e99a76a  // smmla v10.4s, v27.16b, v25.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e99a752  // smmla v18.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4e98a76e  // smmla v14.4s, v27.16b, v24.16b\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e98a756  // smmla v22.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x70]\n"
      ".inst 0x4e99a76b  // smmla v11.4s, v27.16b, v25.16b\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e99a753  // smmla v19.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x80]\n"
      ".inst 0x4e98a76f  // smmla v15.4s, v27.16b, v24.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e98a757  // smmla v23.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x90]\n"
      ".inst 0x4e99a428  // smmla v8.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e99a470  // smmla v16.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xa0]\n"
      ".inst 0x4e98a42c  // smmla v12.4s, v1.16b, v24.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e98a474  // smmla v20.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xb0]\n"
      ".inst 0x4e99a429  // smmla v9.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e99a471  // smmla v17.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xc0]\n"
      ".inst 0x4e98a42d  // smmla v13.4s, v1.16b, v24.16b\n"
      ".inst 0x4e98a475  // smmla v21.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xd0]\n"
      ".inst 0x4e99a42a  // smmla v10.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a472  // smmla v18.4s, v3.16b, v25.16b\n"
      "ldr q25, [x9, #0xe0]\n"
      ".inst 0x4e98a42e  // smmla v14.4s, v1.16b, v24.16b\n"
      ".inst 0x4e98a476  // smmla v22.4s, v3.16b, v24.16b\n"
      "ldr q24, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e99a42b  // smmla v11.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a473  // smmla v19.4s, v3.16b, v25.16b\n"
      ".inst 0x4e98a42f  // smmla v15.4s, v1.16b, v24.16b\n"
      ".inst 0x4e98a477  // smmla v23.4s, v3.16b, v24.16b\n"
      "96:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 103f\n"
      "cmp x27, #0x8\n"
      "blt 98f\n"
      "97:"  // Height 4: Multiply loop: Odd block loop
      "ldr d25, [x26], #0x8\n"
      "ldr d24, [x25], #0x8\n"
      "trn1 v27.2d, v25.2d, v24.2d\n"
      "sub x27, x27, #0x8\n"
      "ldr d25, [x24], #0x8\n"
      "ldr d24, [x23], #0x8\n"
      "trn1 v26.2d, v25.2d, v24.2d\n"
      "cmp x27, #0x8\n"
      "ldr q25, [x9, #0x0]\n"
      "ldr q24, [x9, #0x10]\n"
      ".inst 0x4e99a768  // smmla v8.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a750  // smmla v16.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x20]\n"
      ".inst 0x4e98a76c  // smmla v12.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a754  // smmla v20.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x30]\n"
      ".inst 0x4e99a769  // smmla v9.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a751  // smmla v17.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x40]\n"
      ".inst 0x4e98a76d  // smmla v13.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a755  // smmla v21.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4e99a76a  // smmla v10.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a752  // smmla v18.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4e98a76e  // smmla v14.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a756  // smmla v22.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e99a76b  // smmla v11.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a753  // smmla v19.4s, v26.16b, v25.16b\n"
      ".inst 0x4e98a76f  // smmla v15.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a757  // smmla v23.4s, v26.16b, v24.16b\n"
      "bge 97b\n"
      "98:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x27, 103f\n"
      "tbz x27, #2, 100f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "tbz x27, #1, 99f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "tbz x27, #0, 102f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "b 102f\n"
      "99:"  // Height 4: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 102f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "b 102f\n"
      "100:"  // Height 4: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 101f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "tbz x27, #0, 102f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "b 102f\n"
      "101:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "ldr b3, [x24, #0x0]\n"
      "ldr b4, [x23, #0x0]\n"
      "102:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q25, [x9, #0x0]\n"
      "ldr q24, [x9, #0x10]\n"
      "trn1 v27.2d, v1.2d, v2.2d\n"
      "trn1 v26.2d, v3.2d, v4.2d\n"
      ".inst 0x4e99a768  // smmla v8.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a750  // smmla v16.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x20]\n"
      ".inst 0x4e98a76c  // smmla v12.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a754  // smmla v20.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x30]\n"
      ".inst 0x4e99a769  // smmla v9.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a751  // smmla v17.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x40]\n"
      ".inst 0x4e98a76d  // smmla v13.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a755  // smmla v21.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x50]\n"
      ".inst 0x4e99a76a  // smmla v10.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a752  // smmla v18.4s, v26.16b, v25.16b\n"
      "ldr q25, [x9, #0x60]\n"
      ".inst 0x4e98a76e  // smmla v14.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a756  // smmla v22.4s, v26.16b, v24.16b\n"
      "ldr q24, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e99a76b  // smmla v11.4s, v27.16b, v25.16b\n"
      ".inst 0x4e99a753  // smmla v19.4s, v26.16b, v25.16b\n"
      ".inst 0x4e98a76f  // smmla v15.4s, v27.16b, v24.16b\n"
      ".inst 0x4e98a757  // smmla v23.4s, v26.16b, v24.16b\n"
      "103:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 91b\n"
      "ldr q28, [x14, #0x0]\n"
      "ldr q27, [x14, #0x10]\n"
      "uzp1 v26.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q25, [x14, #0x20]\n"
      "ldr q24, [x14, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x25, x11, x20\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "add x24, x25, x20\n"
      "add x23, x24, x20\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x14, x14, #0x40\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "mov v23.16b, v26.16b\n"
      "add v23.4s, v23.4s, v28.4s\n"
      "add v12.4s, v12.4s, v27.4s\n"
      "add v13.4s, v13.4s, v25.4s\n"
      "add v14.4s, v14.4s, v24.4s\n"
      "add v8.4s, v8.4s, v28.4s\n"
      "add v9.4s, v9.4s, v27.4s\n"
      "add v10.4s, v10.4s, v25.4s\n"
      "add v11.4s, v11.4s, v24.4s\n"
      "add v15.4s, v15.4s, v28.4s\n"
      "add v20.4s, v20.4s, v27.4s\n"
      "add v21.4s, v21.4s, v25.4s\n"
      "add v22.4s, v22.4s, v24.4s\n"
      "add v16.4s, v16.4s, v28.4s\n"
      "add v17.4s, v17.4s, v27.4s\n"
      "add v18.4s, v18.4s, v25.4s\n"
      "add v19.4s, v19.4s, v24.4s\n"
      "tbz %x[flags], #4, 104f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 105f\n"
      "104:"  // Height 4: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "105:"  // Height 4: parameters loaded
      "sqrdmulh v23.4s, v23.4s, v4.4s\n"
      "sqrdmulh v12.4s, v12.4s, v5.4s\n"
      "sqrdmulh v13.4s, v13.4s, v6.4s\n"
      "sqrdmulh v14.4s, v14.4s, v7.4s\n"
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "sqrdmulh v15.4s, v15.4s, v4.4s\n"
      "sqrdmulh v20.4s, v20.4s, v5.4s\n"
      "sqrdmulh v21.4s, v21.4s, v6.4s\n"
      "sqrdmulh v22.4s, v22.4s, v7.4s\n"
      "sqrdmulh v16.4s, v16.4s, v4.4s\n"
      "sqrdmulh v17.4s, v17.4s, v5.4s\n"
      "sqrdmulh v18.4s, v18.4s, v6.4s\n"
      "sqrdmulh v19.4s, v19.4s, v7.4s\n"
      "tbz %x[flags], #5, 106f\n"
      "and v27.16b, v23.16b, v0.16b\n"
      "and v26.16b, v12.16b, v1.16b\n"
      "and v25.16b, v13.16b, v2.16b\n"
      "and v24.16b, v14.16b, v3.16b\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v23.4s, v23.4s, v27.4s\n"
      "sqadd v12.4s, v12.4s, v26.4s\n"
      "sqadd v13.4s, v13.4s, v25.4s\n"
      "sqadd v14.4s, v14.4s, v24.4s\n"
      "and v27.16b, v8.16b, v0.16b\n"
      "and v26.16b, v9.16b, v1.16b\n"
      "and v25.16b, v10.16b, v2.16b\n"
      "and v24.16b, v11.16b, v3.16b\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v27.4s\n"
      "sqadd v9.4s, v9.4s, v26.4s\n"
      "sqadd v10.4s, v10.4s, v25.4s\n"
      "sqadd v11.4s, v11.4s, v24.4s\n"
      "and v27.16b, v15.16b, v0.16b\n"
      "and v26.16b, v20.16b, v1.16b\n"
      "and v25.16b, v21.16b, v2.16b\n"
      "and v24.16b, v22.16b, v3.16b\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v15.4s, v15.4s, v27.4s\n"
      "sqadd v20.4s, v20.4s, v26.4s\n"
      "sqadd v21.4s, v21.4s, v25.4s\n"
      "sqadd v22.4s, v22.4s, v24.4s\n"
      "and v27.16b, v16.16b, v0.16b\n"
      "and v26.16b, v17.16b, v1.16b\n"
      "and v25.16b, v18.16b, v2.16b\n"
      "and v24.16b, v19.16b, v3.16b\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v27.4s\n"
      "sqadd v17.4s, v17.4s, v26.4s\n"
      "sqadd v18.4s, v18.4s, v25.4s\n"
      "sqadd v19.4s, v19.4s, v24.4s\n"
      "106:"  // Height 4: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v26.4s }, [x20]\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v25.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v24.4s }, [x20]\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v1.4s\n"
      "srshl v21.4s, v21.4s, v2.4s\n"
      "srshl v22.4s, v22.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "add v23.4s, v23.4s, v26.4s\n"
      "add v12.4s, v12.4s, v26.4s\n"
      "add v13.4s, v13.4s, v26.4s\n"
      "add v14.4s, v14.4s, v26.4s\n"
      "add v8.4s, v8.4s, v26.4s\n"
      "add v9.4s, v9.4s, v26.4s\n"
      "add v10.4s, v10.4s, v26.4s\n"
      "add v11.4s, v11.4s, v26.4s\n"
      "add v15.4s, v15.4s, v26.4s\n"
      "add v20.4s, v20.4s, v26.4s\n"
      "add v21.4s, v21.4s, v26.4s\n"
      "add v22.4s, v22.4s, v26.4s\n"
      "add v16.4s, v16.4s, v26.4s\n"
      "add v17.4s, v17.4s, v26.4s\n"
      "add v18.4s, v18.4s, v26.4s\n"
      "add v19.4s, v19.4s, v26.4s\n"
      "smin v23.4s, v23.4s, v25.4s\n"
      "smin v12.4s, v12.4s, v25.4s\n"
      "smin v13.4s, v13.4s, v25.4s\n"
      "smin v14.4s, v14.4s, v25.4s\n"
      "smin v8.4s, v8.4s, v25.4s\n"
      "smin v9.4s, v9.4s, v25.4s\n"
      "smin v10.4s, v10.4s, v25.4s\n"
      "smin v11.4s, v11.4s, v25.4s\n"
      "smin v15.4s, v15.4s, v25.4s\n"
      "smin v20.4s, v20.4s, v25.4s\n"
      "smin v21.4s, v21.4s, v25.4s\n"
      "smin v22.4s, v22.4s, v25.4s\n"
      "smin v16.4s, v16.4s, v25.4s\n"
      "smin v17.4s, v17.4s, v25.4s\n"
      "smin v18.4s, v18.4s, v25.4s\n"
      "smin v19.4s, v19.4s, v25.4s\n"
      "smax v23.4s, v23.4s, v24.4s\n"
      "smax v12.4s, v12.4s, v24.4s\n"
      "smax v13.4s, v13.4s, v24.4s\n"
      "smax v14.4s, v14.4s, v24.4s\n"
      "smax v8.4s, v8.4s, v24.4s\n"
      "smax v9.4s, v9.4s, v24.4s\n"
      "smax v10.4s, v10.4s, v24.4s\n"
      "smax v11.4s, v11.4s, v24.4s\n"
      "smax v15.4s, v15.4s, v24.4s\n"
      "smax v20.4s, v20.4s, v24.4s\n"
      "smax v21.4s, v21.4s, v24.4s\n"
      "smax v22.4s, v22.4s, v24.4s\n"
      "smax v16.4s, v16.4s, v24.4s\n"
      "smax v17.4s, v17.4s, v24.4s\n"
      "smax v18.4s, v18.4s, v24.4s\n"
      "smax v19.4s, v19.4s, v24.4s\n"
      "uzp1 v23.8h, v23.8h, v12.8h\n"
      "uzp1 v25.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v24.8h, v10.8h, v11.8h\n"
      "uzp1 v15.8h, v15.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v23.16b, v23.16b, v25.16b\n"
      "uzp1 v8.16b, v8.16b, v24.16b\n"
      "uzp1 v15.16b, v15.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 115f\n"
      "tbz x10, #3, 110f\n"
      "str d23, [x11], #0x8\n"
      "str d8, [x25], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "tbz x10, #2, 108f\n"
      "st1 { v23.s }[2], [x11], #0x4\n"
      "st1 { v8.s }[2], [x25], #0x4\n"
      "st1 { v15.s }[2], [x24], #0x4\n"
      "st1 { v16.s }[2], [x23], #0x4\n"
      "tbz x10, #1, 107f\n"
      "st1 { v23.h }[6], [x11], #0x2\n"
      "st1 { v8.h }[6], [x25], #0x2\n"
      "st1 { v15.h }[6], [x24], #0x2\n"
      "st1 { v16.h }[6], [x23], #0x2\n"
      "tbz x10, #0, 114f\n"
      "st1 { v23.b }[14], [x11]\n"
      "st1 { v8.b }[14], [x25]\n"
      "st1 { v15.b }[14], [x24]\n"
      "st1 { v16.b }[14], [x23]\n"
      "b 114f\n"
      "107:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x10, #0, 114f\n"
      "st1 { v23.b }[12], [x11]\n"
      "st1 { v8.b }[12], [x25]\n"
      "st1 { v15.b }[12], [x24]\n"
      "st1 { v16.b }[12], [x23]\n"
      "b 114f\n"
      "108:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x10, #1, 109f\n"
      "st1 { v23.h }[4], [x11], #0x2\n"
      "st1 { v8.h }[4], [x25], #0x2\n"
      "st1 { v15.h }[4], [x24], #0x2\n"
      "st1 { v16.h }[4], [x23], #0x2\n"
      "tbz x10, #0, 114f\n"
      "st1 { v23.b }[10], [x11]\n"
      "st1 { v8.b }[10], [x25]\n"
      "st1 { v15.b }[10], [x24]\n"
      "st1 { v16.b }[10], [x23]\n"
      "b 114f\n"
      "109:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x10, #0, 114f\n"
      "st1 { v23.b }[8], [x11]\n"
      "st1 { v8.b }[8], [x25]\n"
      "st1 { v15.b }[8], [x24]\n"
      "st1 { v16.b }[8], [x23]\n"
      "b 114f\n"
      "110:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x10, #2, 112f\n"
      "str s23, [x11], #0x4\n"
      "str s8, [x25], #0x4\n"
      "str s15, [x24], #0x4\n"
      "str s16, [x23], #0x4\n"
      "tbz x10, #1, 111f\n"
      "st1 { v23.h }[2], [x11], #0x2\n"
      "st1 { v8.h }[2], [x25], #0x2\n"
      "st1 { v15.h }[2], [x24], #0x2\n"
      "st1 { v16.h }[2], [x23], #0x2\n"
      "tbz x10, #0, 114f\n"
      "st1 { v23.b }[6], [x11]\n"
      "st1 { v8.b }[6], [x25]\n"
      "st1 { v15.b }[6], [x24]\n"
      "st1 { v16.b }[6], [x23]\n"
      "b 114f\n"
      "111:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x10, #0, 114f\n"
      "st1 { v23.b }[4], [x11]\n"
      "st1 { v8.b }[4], [x25]\n"
      "st1 { v15.b }[4], [x24]\n"
      "st1 { v16.b }[4], [x23]\n"
      "b 114f\n"
      "112:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x10, #1, 113f\n"
      "str h23, [x11], #0x2\n"
      "str h8, [x25], #0x2\n"
      "str h15, [x24], #0x2\n"
      "str h16, [x23], #0x2\n"
      "tbz x10, #0, 114f\n"
      "st1 { v23.b }[2], [x11]\n"
      "st1 { v8.b }[2], [x25]\n"
      "st1 { v15.b }[2], [x24]\n"
      "st1 { v16.b }[2], [x23]\n"
      "b 114f\n"
      "113:"  // Height 4: Partial direct writeback: partial_1_0
      "str b23, [x11, #0x0]\n"
      "str b8, [x25, #0x0]\n"
      "str b15, [x24, #0x0]\n"
      "str b16, [x23, #0x0]\n"
      "114:"  // Height 4: Partial direct writeback: Done
      "b 116f\n"
      "115:"  // Height 4: Full writeback
      "str q23, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q8, [x25, #0x0]\n"
      "str q15, [x24, #0x0]\n"
      "str q16, [x23, #0x0]\n"
      "116:"  // Height 4: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 89b\n"
      "b 176f\n"
      "117:"  // Height 5
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "118:"  // Height 5: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "119:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "120:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 121f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 122f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 122f\n"
      "121:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "122:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "blt 125f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q4, [x23, #0x0]\n"
      "ldr q5, [x22, #0x0]\n"
      "ldr q7, [x9, #0x0]\n"
      "blt 124f\n"
      "123:"  // Height 5: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "sub x27, x27, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x9, #0x10]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x40]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x60]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "ldr q2, [x25, #0x0]\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q0, [x9, #0x90]\n"
      "ldr q4, [x23, #0x0]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q6, [x9, #0xa0]\n"
      ".inst 0x4e80a42c  // smmla v12.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bc  // smmla v28.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xb0]\n"
      ".inst 0x4e86a429  // smmla v9.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a471  // smmla v17.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4b9  // smmla v25.4s, v5.16b, v6.16b\n"
      "ldr q6, [x9, #0xc0]\n"
      ".inst 0x4e80a42d  // smmla v13.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bd  // smmla v29.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xd0]\n"
      ".inst 0x4e86a42a  // smmla v10.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a472  // smmla v18.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4ba  // smmla v26.4s, v5.16b, v6.16b\n"
      "ldr q6, [x9, #0xe0]\n"
      ".inst 0x4e80a42e  // smmla v14.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4be  // smmla v30.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e86a42b  // smmla v11.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a473  // smmla v19.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bb  // smmla v27.4s, v5.16b, v6.16b\n"
      "ldr q7, [x9, #0x0]\n"
      ".inst 0x4e80a42f  // smmla v15.4s, v1.16b, v0.16b\n"
      "ldr q1, [x26, #0x0]\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      "ldr q3, [x24, #0x0]\n"
      ".inst 0x4e80a4bf  // smmla v31.4s, v5.16b, v0.16b\n"
      "ldr q5, [x22, #0x0]\n"
      "bge 123b\n"
      "124:"  // Height 5: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "add x26, x26, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x9, #0x10]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x40]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "add x22, x22, #0x10\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x60]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q0, [x9, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q2, [x9, #0xa0]\n"
      ".inst 0x4e80a42c  // smmla v12.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bc  // smmla v28.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xb0]\n"
      ".inst 0x4e82a429  // smmla v9.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a471  // smmla v17.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4b9  // smmla v25.4s, v5.16b, v2.16b\n"
      "ldr q2, [x9, #0xc0]\n"
      ".inst 0x4e80a42d  // smmla v13.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bd  // smmla v29.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xd0]\n"
      ".inst 0x4e82a42a  // smmla v10.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a472  // smmla v18.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4ba  // smmla v26.4s, v5.16b, v2.16b\n"
      "ldr q2, [x9, #0xe0]\n"
      ".inst 0x4e80a42e  // smmla v14.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4be  // smmla v30.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e82a42b  // smmla v11.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a473  // smmla v19.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4bb  // smmla v27.4s, v5.16b, v2.16b\n"
      ".inst 0x4e80a42f  // smmla v15.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bf  // smmla v31.4s, v5.16b, v0.16b\n"
      "125:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 132f\n"
      "cmp x27, #0x8\n"
      "blt 127f\n"
      "126:"  // Height 5: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d0, [x25], #0x8\n"
      "trn1 v4.2d, v1.2d, v0.2d\n"
      "ldr d1, [x24], #0x8\n"
      "ldr d0, [x23], #0x8\n"
      "trn1 v3.2d, v1.2d, v0.2d\n"
      "sub x27, x27, #0x8\n"
      "ldr d0, [x22], #0x8\n"
      "ldr q1, [x9, #0x0]\n"
      "trn1 v2.2d, v0.2d, v2.2d\n"
      ".inst 0x4e81a488  // smmla v8.4s, v4.16b, v1.16b\n"
      "ldr q0, [x9, #0x10]\n"
      ".inst 0x4e81a470  // smmla v16.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a458  // smmla v24.4s, v2.16b, v1.16b\n"
      "ldr q1, [x9, #0x20]\n"
      ".inst 0x4e80a48c  // smmla v12.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      "cmp x27, #0x8\n"
      ".inst 0x4e80a45c  // smmla v28.4s, v2.16b, v0.16b\n"
      "ldr q0, [x9, #0x30]\n"
      ".inst 0x4e81a489  // smmla v9.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a471  // smmla v17.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a459  // smmla v25.4s, v2.16b, v1.16b\n"
      "ldr q1, [x9, #0x40]\n"
      ".inst 0x4e80a48d  // smmla v13.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45d  // smmla v29.4s, v2.16b, v0.16b\n"
      "ldr q0, [x9, #0x50]\n"
      ".inst 0x4e81a48a  // smmla v10.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a472  // smmla v18.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45a  // smmla v26.4s, v2.16b, v1.16b\n"
      "ldr q1, [x9, #0x60]\n"
      ".inst 0x4e80a48e  // smmla v14.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45e  // smmla v30.4s, v2.16b, v0.16b\n"
      "ldr q0, [x9, #0x70]\n"
      ".inst 0x4e81a48b  // smmla v11.4s, v4.16b, v1.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e81a473  // smmla v19.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45b  // smmla v27.4s, v2.16b, v1.16b\n"
      ".inst 0x4e80a48f  // smmla v15.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45f  // smmla v31.4s, v2.16b, v0.16b\n"
      "bge 126b\n"
      "127:"  // Height 5: Multiply loop: Skip odd blocks
      "cbz x27, 132f\n"
      "tbz x27, #2, 129f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "ldr s5, [x22], #0x4\n"
      "tbz x27, #1, 128f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "ld1 { v5.h }[2], [x22], #0x2\n"
      "tbz x27, #0, 131f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "ld1 { v5.b }[6], [x22]\n"
      "b 131f\n"
      "128:"  // Height 5: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 131f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "ld1 { v5.b }[4], [x22]\n"
      "b 131f\n"
      "129:"  // Height 5: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 130f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "ldr h5, [x22], #0x2\n"
      "tbz x27, #0, 131f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "ld1 { v5.b }[2], [x22]\n"
      "b 131f\n"
      "130:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "ldr b3, [x24, #0x0]\n"
      "ldr b4, [x23, #0x0]\n"
      "ldr b5, [x22, #0x0]\n"
      "131:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q7, [x9, #0x0]\n"
      "trn1 v6.2d, v1.2d, v2.2d\n"
      "trn1 v3.2d, v3.2d, v4.2d\n"
      "trn1 v2.2d, v5.2d, v0.2d\n"
      "ldr q0, [x9, #0x10]\n"
      ".inst 0x4e87a4c8  // smmla v8.4s, v6.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a458  // smmla v24.4s, v2.16b, v7.16b\n"
      "ldr q1, [x9, #0x20]\n"
      ".inst 0x4e80a4cc  // smmla v12.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45c  // smmla v28.4s, v2.16b, v0.16b\n"
      "ldr q0, [x9, #0x30]\n"
      ".inst 0x4e81a4c9  // smmla v9.4s, v6.16b, v1.16b\n"
      ".inst 0x4e81a471  // smmla v17.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a459  // smmla v25.4s, v2.16b, v1.16b\n"
      "ldr q1, [x9, #0x40]\n"
      ".inst 0x4e80a4cd  // smmla v13.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45d  // smmla v29.4s, v2.16b, v0.16b\n"
      "ldr q0, [x9, #0x50]\n"
      ".inst 0x4e81a4ca  // smmla v10.4s, v6.16b, v1.16b\n"
      ".inst 0x4e81a472  // smmla v18.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45a  // smmla v26.4s, v2.16b, v1.16b\n"
      "ldr q1, [x9, #0x60]\n"
      ".inst 0x4e80a4ce  // smmla v14.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45e  // smmla v30.4s, v2.16b, v0.16b\n"
      "ldr q0, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e81a4cb  // smmla v11.4s, v6.16b, v1.16b\n"
      ".inst 0x4e81a473  // smmla v19.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45b  // smmla v27.4s, v2.16b, v1.16b\n"
      ".inst 0x4e80a4cf  // smmla v15.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45f  // smmla v31.4s, v2.16b, v0.16b\n"
      "132:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 120b\n"
      "ldr q4, [x14, #0x0]\n"
      "ldr q3, [x14, #0x10]\n"
      "uzp1 v2.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q1, [x14, #0x20]\n"
      "ldr q0, [x14, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x11, x20\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "add x24, x25, x20\n"
      "add x23, x24, x20\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "add x22, x23, x20\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "add x14, x14, #0x40\n"
      "uzp1 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v27.2d, v27.2d, v31.2d\n"
      "mov v31.16b, v2.16b\n"
      "add v31.4s, v31.4s, v4.4s\n"
      "add v12.4s, v12.4s, v3.4s\n"
      "add v13.4s, v13.4s, v1.4s\n"
      "add v14.4s, v14.4s, v0.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v3.4s\n"
      "add v10.4s, v10.4s, v1.4s\n"
      "add v11.4s, v11.4s, v0.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v20.4s, v20.4s, v3.4s\n"
      "add v21.4s, v21.4s, v1.4s\n"
      "add v22.4s, v22.4s, v0.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v3.4s\n"
      "add v18.4s, v18.4s, v1.4s\n"
      "add v19.4s, v19.4s, v0.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v3.4s\n"
      "add v26.4s, v26.4s, v1.4s\n"
      "add v27.4s, v27.4s, v0.4s\n"
      "tbz %x[flags], #4, 133f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 134f\n"
      "133:"  // Height 5: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "134:"  // Height 5: parameters loaded
      "sqrdmulh v31.4s, v31.4s, v4.4s\n"
      "sqrdmulh v12.4s, v12.4s, v5.4s\n"
      "sqrdmulh v13.4s, v13.4s, v6.4s\n"
      "sqrdmulh v14.4s, v14.4s, v7.4s\n"
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "sqrdmulh v15.4s, v15.4s, v4.4s\n"
      "sqrdmulh v20.4s, v20.4s, v5.4s\n"
      "sqrdmulh v21.4s, v21.4s, v6.4s\n"
      "sqrdmulh v22.4s, v22.4s, v7.4s\n"
      "sqrdmulh v16.4s, v16.4s, v4.4s\n"
      "sqrdmulh v17.4s, v17.4s, v5.4s\n"
      "sqrdmulh v18.4s, v18.4s, v6.4s\n"
      "sqrdmulh v19.4s, v19.4s, v7.4s\n"
      "sqrdmulh v24.4s, v24.4s, v4.4s\n"
      "sqrdmulh v25.4s, v25.4s, v5.4s\n"
      "sqrdmulh v26.4s, v26.4s, v6.4s\n"
      "sqrdmulh v27.4s, v27.4s, v7.4s\n"
      "tbz %x[flags], #5, 135f\n"
      "and v30.16b, v31.16b, v0.16b\n"
      "and v29.16b, v12.16b, v1.16b\n"
      "and v28.16b, v13.16b, v2.16b\n"
      "and v23.16b, v14.16b, v3.16b\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sqadd v31.4s, v31.4s, v30.4s\n"
      "sqadd v12.4s, v12.4s, v29.4s\n"
      "sqadd v13.4s, v13.4s, v28.4s\n"
      "sqadd v14.4s, v14.4s, v23.4s\n"
      "and v30.16b, v8.16b, v0.16b\n"
      "and v29.16b, v9.16b, v1.16b\n"
      "and v28.16b, v10.16b, v2.16b\n"
      "and v23.16b, v11.16b, v3.16b\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v30.4s\n"
      "sqadd v9.4s, v9.4s, v29.4s\n"
      "sqadd v10.4s, v10.4s, v28.4s\n"
      "sqadd v11.4s, v11.4s, v23.4s\n"
      "and v30.16b, v15.16b, v0.16b\n"
      "and v29.16b, v20.16b, v1.16b\n"
      "and v28.16b, v21.16b, v2.16b\n"
      "and v23.16b, v22.16b, v3.16b\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sqadd v15.4s, v15.4s, v30.4s\n"
      "sqadd v20.4s, v20.4s, v29.4s\n"
      "sqadd v21.4s, v21.4s, v28.4s\n"
      "sqadd v22.4s, v22.4s, v23.4s\n"
      "and v30.16b, v16.16b, v0.16b\n"
      "and v29.16b, v17.16b, v1.16b\n"
      "and v28.16b, v18.16b, v2.16b\n"
      "and v23.16b, v19.16b, v3.16b\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v30.4s\n"
      "sqadd v17.4s, v17.4s, v29.4s\n"
      "sqadd v18.4s, v18.4s, v28.4s\n"
      "sqadd v19.4s, v19.4s, v23.4s\n"
      "and v30.16b, v24.16b, v0.16b\n"
      "and v29.16b, v25.16b, v1.16b\n"
      "and v28.16b, v26.16b, v2.16b\n"
      "and v23.16b, v27.16b, v3.16b\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sqadd v24.4s, v24.4s, v30.4s\n"
      "sqadd v25.4s, v25.4s, v29.4s\n"
      "sqadd v26.4s, v26.4s, v28.4s\n"
      "sqadd v27.4s, v27.4s, v23.4s\n"
      "135:"  // Height 5: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v29.4s }, [x20]\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v28.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v23.4s }, [x20]\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v1.4s\n"
      "srshl v21.4s, v21.4s, v2.4s\n"
      "srshl v22.4s, v22.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v1.4s\n"
      "srshl v26.4s, v26.4s, v2.4s\n"
      "srshl v27.4s, v27.4s, v3.4s\n"
      "add v31.4s, v31.4s, v29.4s\n"
      "add v12.4s, v12.4s, v29.4s\n"
      "add v13.4s, v13.4s, v29.4s\n"
      "add v14.4s, v14.4s, v29.4s\n"
      "add v8.4s, v8.4s, v29.4s\n"
      "add v9.4s, v9.4s, v29.4s\n"
      "add v10.4s, v10.4s, v29.4s\n"
      "add v11.4s, v11.4s, v29.4s\n"
      "add v15.4s, v15.4s, v29.4s\n"
      "add v20.4s, v20.4s, v29.4s\n"
      "add v21.4s, v21.4s, v29.4s\n"
      "add v22.4s, v22.4s, v29.4s\n"
      "add v16.4s, v16.4s, v29.4s\n"
      "add v17.4s, v17.4s, v29.4s\n"
      "add v18.4s, v18.4s, v29.4s\n"
      "add v19.4s, v19.4s, v29.4s\n"
      "add v24.4s, v24.4s, v29.4s\n"
      "add v25.4s, v25.4s, v29.4s\n"
      "add v26.4s, v26.4s, v29.4s\n"
      "add v27.4s, v27.4s, v29.4s\n"
      "smin v31.4s, v31.4s, v28.4s\n"
      "smin v12.4s, v12.4s, v28.4s\n"
      "smin v13.4s, v13.4s, v28.4s\n"
      "smin v14.4s, v14.4s, v28.4s\n"
      "smin v8.4s, v8.4s, v28.4s\n"
      "smin v9.4s, v9.4s, v28.4s\n"
      "smin v10.4s, v10.4s, v28.4s\n"
      "smin v11.4s, v11.4s, v28.4s\n"
      "smin v15.4s, v15.4s, v28.4s\n"
      "smin v20.4s, v20.4s, v28.4s\n"
      "smin v21.4s, v21.4s, v28.4s\n"
      "smin v22.4s, v22.4s, v28.4s\n"
      "smin v16.4s, v16.4s, v28.4s\n"
      "smin v17.4s, v17.4s, v28.4s\n"
      "smin v18.4s, v18.4s, v28.4s\n"
      "smin v19.4s, v19.4s, v28.4s\n"
      "smin v24.4s, v24.4s, v28.4s\n"
      "smin v25.4s, v25.4s, v28.4s\n"
      "smin v26.4s, v26.4s, v28.4s\n"
      "smin v27.4s, v27.4s, v28.4s\n"
      "smax v31.4s, v31.4s, v23.4s\n"
      "smax v12.4s, v12.4s, v23.4s\n"
      "smax v13.4s, v13.4s, v23.4s\n"
      "smax v14.4s, v14.4s, v23.4s\n"
      "smax v8.4s, v8.4s, v23.4s\n"
      "smax v9.4s, v9.4s, v23.4s\n"
      "smax v10.4s, v10.4s, v23.4s\n"
      "smax v11.4s, v11.4s, v23.4s\n"
      "smax v15.4s, v15.4s, v23.4s\n"
      "smax v20.4s, v20.4s, v23.4s\n"
      "smax v21.4s, v21.4s, v23.4s\n"
      "smax v22.4s, v22.4s, v23.4s\n"
      "smax v16.4s, v16.4s, v23.4s\n"
      "smax v17.4s, v17.4s, v23.4s\n"
      "smax v18.4s, v18.4s, v23.4s\n"
      "smax v19.4s, v19.4s, v23.4s\n"
      "smax v24.4s, v24.4s, v23.4s\n"
      "smax v25.4s, v25.4s, v23.4s\n"
      "smax v26.4s, v26.4s, v23.4s\n"
      "smax v27.4s, v27.4s, v23.4s\n"
      "uzp1 v31.8h, v31.8h, v12.8h\n"
      "uzp1 v28.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v23.8h, v10.8h, v11.8h\n"
      "uzp1 v15.8h, v15.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v18.8h, v18.8h, v19.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v17.8h, v26.8h, v27.8h\n"
      "uzp1 v31.16b, v31.16b, v28.16b\n"
      "uzp1 v8.16b, v8.16b, v23.16b\n"
      "uzp1 v15.16b, v15.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v18.16b\n"
      "uzp1 v24.16b, v24.16b, v17.16b\n"
      "bge 144f\n"
      "tbz x10, #3, 139f\n"
      "str d31, [x11], #0x8\n"
      "str d8, [x25], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d24, [x22], #0x8\n"
      "tbz x10, #2, 137f\n"
      "st1 { v31.s }[2], [x11], #0x4\n"
      "st1 { v8.s }[2], [x25], #0x4\n"
      "st1 { v15.s }[2], [x24], #0x4\n"
      "st1 { v16.s }[2], [x23], #0x4\n"
      "st1 { v24.s }[2], [x22], #0x4\n"
      "tbz x10, #1, 136f\n"
      "st1 { v31.h }[6], [x11], #0x2\n"
      "st1 { v8.h }[6], [x25], #0x2\n"
      "st1 { v15.h }[6], [x24], #0x2\n"
      "st1 { v16.h }[6], [x23], #0x2\n"
      "st1 { v24.h }[6], [x22], #0x2\n"
      "tbz x10, #0, 143f\n"
      "st1 { v31.b }[14], [x11]\n"
      "st1 { v8.b }[14], [x25]\n"
      "st1 { v15.b }[14], [x24]\n"
      "st1 { v16.b }[14], [x23]\n"
      "st1 { v24.b }[14], [x22]\n"
      "b 143f\n"
      "136:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x10, #0, 143f\n"
      "st1 { v31.b }[12], [x11]\n"
      "st1 { v8.b }[12], [x25]\n"
      "st1 { v15.b }[12], [x24]\n"
      "st1 { v16.b }[12], [x23]\n"
      "st1 { v24.b }[12], [x22]\n"
      "b 143f\n"
      "137:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x10, #1, 138f\n"
      "st1 { v31.h }[4], [x11], #0x2\n"
      "st1 { v8.h }[4], [x25], #0x2\n"
      "st1 { v15.h }[4], [x24], #0x2\n"
      "st1 { v16.h }[4], [x23], #0x2\n"
      "st1 { v24.h }[4], [x22], #0x2\n"
      "tbz x10, #0, 143f\n"
      "st1 { v31.b }[10], [x11]\n"
      "st1 { v8.b }[10], [x25]\n"
      "st1 { v15.b }[10], [x24]\n"
      "st1 { v16.b }[10], [x23]\n"
      "st1 { v24.b }[10], [x22]\n"
      "b 143f\n"
      "138:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x10, #0, 143f\n"
      "st1 { v31.b }[8], [x11]\n"
      "st1 { v8.b }[8], [x25]\n"
      "st1 { v15.b }[8], [x24]\n"
      "st1 { v16.b }[8], [x23]\n"
      "st1 { v24.b }[8], [x22]\n"
      "b 143f\n"
      "139:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x10, #2, 141f\n"
      "str s31, [x11], #0x4\n"
      "str s8, [x25], #0x4\n"
      "str s15, [x24], #0x4\n"
      "str s16, [x23], #0x4\n"
      "str s24, [x22], #0x4\n"
      "tbz x10, #1, 140f\n"
      "st1 { v31.h }[2], [x11], #0x2\n"
      "st1 { v8.h }[2], [x25], #0x2\n"
      "st1 { v15.h }[2], [x24], #0x2\n"
      "st1 { v16.h }[2], [x23], #0x2\n"
      "st1 { v24.h }[2], [x22], #0x2\n"
      "tbz x10, #0, 143f\n"
      "st1 { v31.b }[6], [x11]\n"
      "st1 { v8.b }[6], [x25]\n"
      "st1 { v15.b }[6], [x24]\n"
      "st1 { v16.b }[6], [x23]\n"
      "st1 { v24.b }[6], [x22]\n"
      "b 143f\n"
      "140:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x10, #0, 143f\n"
      "st1 { v31.b }[4], [x11]\n"
      "st1 { v8.b }[4], [x25]\n"
      "st1 { v15.b }[4], [x24]\n"
      "st1 { v16.b }[4], [x23]\n"
      "st1 { v24.b }[4], [x22]\n"
      "b 143f\n"
      "141:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x10, #1, 142f\n"
      "str h31, [x11], #0x2\n"
      "str h8, [x25], #0x2\n"
      "str h15, [x24], #0x2\n"
      "str h16, [x23], #0x2\n"
      "str h24, [x22], #0x2\n"
      "tbz x10, #0, 143f\n"
      "st1 { v31.b }[2], [x11]\n"
      "st1 { v8.b }[2], [x25]\n"
      "st1 { v15.b }[2], [x24]\n"
      "st1 { v16.b }[2], [x23]\n"
      "st1 { v24.b }[2], [x22]\n"
      "b 143f\n"
      "142:"  // Height 5: Partial direct writeback: partial_1_0
      "str b31, [x11, #0x0]\n"
      "str b8, [x25, #0x0]\n"
      "str b15, [x24, #0x0]\n"
      "str b16, [x23, #0x0]\n"
      "str b24, [x22, #0x0]\n"
      "143:"  // Height 5: Partial direct writeback: Done
      "b 145f\n"
      "144:"  // Height 5: Full writeback
      "str q31, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q8, [x25, #0x0]\n"
      "str q15, [x24, #0x0]\n"
      "str q16, [x23, #0x0]\n"
      "str q24, [x22, #0x0]\n"
      "145:"  // Height 5: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 118b\n"
      "b 176f\n"
      "146:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x6\n"
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x11, %x[output_ptr]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "147:"  // Height 6: Column loop
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "148:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "149:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 150f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 151f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 151f\n"
      "150:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "151:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "blt 154f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q4, [x23, #0x0]\n"
      "ldr q5, [x22, #0x0]\n"
      "ldr q6, [x21, #0x0]\n"
      "ldr q7, [x9, #0x0]\n"
      "blt 153f\n"
      "152:"  // Height 6: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "sub x27, x27, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "add x26, x26, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x9, #0x10]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x40]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x60]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "ldr q2, [x25, #0x0]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q0, [x9, #0x90]\n"
      "ldr q4, [x23, #0x0]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q6, [x9, #0xa0]\n"
      ".inst 0x4e80a42c  // smmla v12.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bc  // smmla v28.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xb0]\n"
      ".inst 0x4e86a429  // smmla v9.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a471  // smmla v17.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4b9  // smmla v25.4s, v5.16b, v6.16b\n"
      "ldr q6, [x9, #0xc0]\n"
      ".inst 0x4e80a42d  // smmla v13.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bd  // smmla v29.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xd0]\n"
      ".inst 0x4e86a42a  // smmla v10.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a472  // smmla v18.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4ba  // smmla v26.4s, v5.16b, v6.16b\n"
      "ldr q6, [x9, #0xe0]\n"
      ".inst 0x4e80a42e  // smmla v14.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4be  // smmla v30.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e86a42b  // smmla v11.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a473  // smmla v19.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bb  // smmla v27.4s, v5.16b, v6.16b\n"
      "ldr q7, [x9, #0x0]\n"
      ".inst 0x4e80a42f  // smmla v15.4s, v1.16b, v0.16b\n"
      "ldr q1, [x26, #0x0]\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      "ldr q3, [x24, #0x0]\n"
      ".inst 0x4e80a4bf  // smmla v31.4s, v5.16b, v0.16b\n"
      "ldr q5, [x22, #0x0]\n"
      "ldr q6, [x21, #0x0]\n"
      "bge 152b\n"
      "153:"  // Height 6: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "add x26, x26, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "add x25, x25, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x9, #0x10]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x40]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "add x21, x21, #0x10\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x60]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x9, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x9, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q0, [x9, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q2, [x9, #0xa0]\n"
      ".inst 0x4e80a42c  // smmla v12.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bc  // smmla v28.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xb0]\n"
      ".inst 0x4e82a429  // smmla v9.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a471  // smmla v17.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4b9  // smmla v25.4s, v5.16b, v2.16b\n"
      "ldr q2, [x9, #0xc0]\n"
      ".inst 0x4e80a42d  // smmla v13.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bd  // smmla v29.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xd0]\n"
      ".inst 0x4e82a42a  // smmla v10.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a472  // smmla v18.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4ba  // smmla v26.4s, v5.16b, v2.16b\n"
      "ldr q2, [x9, #0xe0]\n"
      ".inst 0x4e80a42e  // smmla v14.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4be  // smmla v30.4s, v5.16b, v0.16b\n"
      "ldr q0, [x9, #0xf0]\n"
      "add x9, x9, #0x100\n"
      ".inst 0x4e82a42b  // smmla v11.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a473  // smmla v19.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4bb  // smmla v27.4s, v5.16b, v2.16b\n"
      ".inst 0x4e80a42f  // smmla v15.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bf  // smmla v31.4s, v5.16b, v0.16b\n"
      "154:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 161f\n"
      "cmp x27, #0x8\n"
      "blt 156f\n"
      "155:"  // Height 6: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d0, [x25], #0x8\n"
      "trn1 v4.2d, v1.2d, v0.2d\n"
      "sub x27, x27, #0x8\n"
      "ldr d1, [x24], #0x8\n"
      "ldr d0, [x23], #0x8\n"
      "trn1 v3.2d, v1.2d, v0.2d\n"
      "cmp x27, #0x8\n"
      "ldr d1, [x22], #0x8\n"
      "ldr d0, [x21], #0x8\n"
      "trn1 v2.2d, v1.2d, v0.2d\n"
      "ldr q1, [x9, #0x0]\n"
      "ldr q0, [x9, #0x10]\n"
      ".inst 0x4e81a488  // smmla v8.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a470  // smmla v16.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a458  // smmla v24.4s, v2.16b, v1.16b\n"
      "ldr q1, [x9, #0x20]\n"
      ".inst 0x4e80a48c  // smmla v12.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45c  // smmla v28.4s, v2.16b, v0.16b\n"
      "ldr q0, [x9, #0x30]\n"
      ".inst 0x4e81a489  // smmla v9.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a471  // smmla v17.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a459  // smmla v25.4s, v2.16b, v1.16b\n"
      "ldr q1, [x9, #0x40]\n"
      ".inst 0x4e80a48d  // smmla v13.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45d  // smmla v29.4s, v2.16b, v0.16b\n"
      "ldr q0, [x9, #0x50]\n"
      ".inst 0x4e81a48a  // smmla v10.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a472  // smmla v18.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45a  // smmla v26.4s, v2.16b, v1.16b\n"
      "ldr q1, [x9, #0x60]\n"
      ".inst 0x4e80a48e  // smmla v14.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45e  // smmla v30.4s, v2.16b, v0.16b\n"
      "ldr q0, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e81a48b  // smmla v11.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a473  // smmla v19.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45b  // smmla v27.4s, v2.16b, v1.16b\n"
      ".inst 0x4e80a48f  // smmla v15.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45f  // smmla v31.4s, v2.16b, v0.16b\n"
      "bge 155b\n"
      "156:"  // Height 6: Multiply loop: Skip odd blocks
      "cbz x27, 161f\n"
      "tbz x27, #2, 158f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "ldr s5, [x22], #0x4\n"
      "ldr s6, [x21], #0x4\n"
      "tbz x27, #1, 157f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "ld1 { v5.h }[2], [x22], #0x2\n"
      "ld1 { v6.h }[2], [x21], #0x2\n"
      "tbz x27, #0, 160f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "ld1 { v5.b }[6], [x22]\n"
      "ld1 { v6.b }[6], [x21]\n"
      "b 160f\n"
      "157:"  // Height 6: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 160f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "ld1 { v5.b }[4], [x22]\n"
      "ld1 { v6.b }[4], [x21]\n"
      "b 160f\n"
      "158:"  // Height 6: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 159f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "ldr h5, [x22], #0x2\n"
      "ldr h6, [x21], #0x2\n"
      "tbz x27, #0, 160f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "ld1 { v5.b }[2], [x22]\n"
      "ld1 { v6.b }[2], [x21]\n"
      "b 160f\n"
      "159:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "ldr b3, [x24, #0x0]\n"
      "ldr b4, [x23, #0x0]\n"
      "ldr b5, [x22, #0x0]\n"
      "ldr b6, [x21, #0x0]\n"
      "160:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q7, [x9, #0x0]\n"
      "trn1 v2.2d, v1.2d, v2.2d\n"
      "trn1 v4.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a448  // smmla v8.4s, v2.16b, v7.16b\n"
      "trn1 v3.2d, v5.2d, v6.2d\n"
      "ldr q0, [x9, #0x10]\n"
      ".inst 0x4e87a490  // smmla v16.4s, v4.16b, v7.16b\n"
      ".inst 0x4e87a478  // smmla v24.4s, v3.16b, v7.16b\n"
      "ldr q1, [x9, #0x20]\n"
      ".inst 0x4e80a44c  // smmla v12.4s, v2.16b, v0.16b\n"
      ".inst 0x4e80a494  // smmla v20.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a47c  // smmla v28.4s, v3.16b, v0.16b\n"
      "ldr q0, [x9, #0x30]\n"
      ".inst 0x4e81a449  // smmla v9.4s, v2.16b, v1.16b\n"
      ".inst 0x4e81a491  // smmla v17.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a479  // smmla v25.4s, v3.16b, v1.16b\n"
      "ldr q1, [x9, #0x40]\n"
      ".inst 0x4e80a44d  // smmla v13.4s, v2.16b, v0.16b\n"
      ".inst 0x4e80a495  // smmla v21.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a47d  // smmla v29.4s, v3.16b, v0.16b\n"
      "ldr q0, [x9, #0x50]\n"
      ".inst 0x4e81a44a  // smmla v10.4s, v2.16b, v1.16b\n"
      ".inst 0x4e81a492  // smmla v18.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a47a  // smmla v26.4s, v3.16b, v1.16b\n"
      "ldr q1, [x9, #0x60]\n"
      ".inst 0x4e80a44e  // smmla v14.4s, v2.16b, v0.16b\n"
      ".inst 0x4e80a496  // smmla v22.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a47e  // smmla v30.4s, v3.16b, v0.16b\n"
      "ldr q0, [x9, #0x70]\n"
      ".inst 0x4e81a44b  // smmla v11.4s, v2.16b, v1.16b\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4e81a493  // smmla v19.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a47b  // smmla v27.4s, v3.16b, v1.16b\n"
      ".inst 0x4e80a44f  // smmla v15.4s, v2.16b, v0.16b\n"
      ".inst 0x4e80a497  // smmla v23.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a47f  // smmla v31.4s, v3.16b, v0.16b\n"
      "161:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 149b\n"
      "ldr q4, [x14, #0x0]\n"
      "ldr q3, [x14, #0x10]\n"
      "uzp1 v2.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q1, [x14, #0x20]\n"
      "ldr q0, [x14, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x11, x20\n"
      "add x24, x25, x20\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "add x23, x24, x20\n"
      "add x22, x23, x20\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "add x21, x22, x20\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "add x14, x14, #0x40\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "uzp1 v23.2d, v24.2d, v28.2d\n"
      "uzp2 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v28.2d, v25.2d, v29.2d\n"
      "uzp2 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v29.2d, v26.2d, v30.2d\n"
      "uzp2 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v30.2d, v27.2d, v31.2d\n"
      "uzp2 v27.2d, v27.2d, v31.2d\n"
      "mov v31.16b, v2.16b\n"
      "add v31.4s, v31.4s, v4.4s\n"
      "add v12.4s, v12.4s, v3.4s\n"
      "add v13.4s, v13.4s, v1.4s\n"
      "add v14.4s, v14.4s, v0.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v3.4s\n"
      "add v10.4s, v10.4s, v1.4s\n"
      "add v11.4s, v11.4s, v0.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v20.4s, v20.4s, v3.4s\n"
      "add v21.4s, v21.4s, v1.4s\n"
      "add v22.4s, v22.4s, v0.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v3.4s\n"
      "add v18.4s, v18.4s, v1.4s\n"
      "add v19.4s, v19.4s, v0.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v28.4s, v28.4s, v3.4s\n"
      "add v29.4s, v29.4s, v1.4s\n"
      "add v30.4s, v30.4s, v0.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v3.4s\n"
      "add v26.4s, v26.4s, v1.4s\n"
      "add v27.4s, v27.4s, v0.4s\n"
      "tbz %x[flags], #4, 162f\n"
      "ldr q0, [x12, #0x0]\n"
      "ldr q4, [x13, #0x0]\n"
      "ldr q1, [x12, #0x10]\n"
      "ldr q5, [x13, #0x10]\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q6, [x13, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "ldr q7, [x13, #0x30]\n"
      "add x12, x12, #0x40\n"
      "add x13, x13, #0x40\n"
      "b 163f\n"
      "162:"  // Height 6: per layer parameters
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "ld1r { v0.4s }, [x20]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "163:"  // Height 6: parameters loaded
      "sqrdmulh v31.4s, v31.4s, v4.4s\n"
      "sqrdmulh v12.4s, v12.4s, v5.4s\n"
      "sqrdmulh v13.4s, v13.4s, v6.4s\n"
      "sqrdmulh v14.4s, v14.4s, v7.4s\n"
      "sqrdmulh v8.4s, v8.4s, v4.4s\n"
      "sqrdmulh v9.4s, v9.4s, v5.4s\n"
      "sqrdmulh v10.4s, v10.4s, v6.4s\n"
      "sqrdmulh v11.4s, v11.4s, v7.4s\n"
      "sqrdmulh v15.4s, v15.4s, v4.4s\n"
      "sqrdmulh v20.4s, v20.4s, v5.4s\n"
      "sqrdmulh v21.4s, v21.4s, v6.4s\n"
      "sqrdmulh v22.4s, v22.4s, v7.4s\n"
      "sqrdmulh v16.4s, v16.4s, v4.4s\n"
      "sqrdmulh v17.4s, v17.4s, v5.4s\n"
      "sqrdmulh v18.4s, v18.4s, v6.4s\n"
      "sqrdmulh v19.4s, v19.4s, v7.4s\n"
      "sqrdmulh v23.4s, v23.4s, v4.4s\n"
      "sqrdmulh v28.4s, v28.4s, v5.4s\n"
      "sqrdmulh v29.4s, v29.4s, v6.4s\n"
      "sqrdmulh v30.4s, v30.4s, v7.4s\n"
      "sqrdmulh v24.4s, v24.4s, v4.4s\n"
      "sqrdmulh v25.4s, v25.4s, v5.4s\n"
      "sqrdmulh v26.4s, v26.4s, v6.4s\n"
      "sqrdmulh v27.4s, v27.4s, v7.4s\n"
      "tbz %x[flags], #5, 164f\n"
      "and v7.16b, v31.16b, v0.16b\n"
      "and v6.16b, v12.16b, v1.16b\n"
      "and v5.16b, v13.16b, v2.16b\n"
      "and v4.16b, v14.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v31.4s, v31.4s, v7.4s\n"
      "sqadd v12.4s, v12.4s, v6.4s\n"
      "sqadd v13.4s, v13.4s, v5.4s\n"
      "sqadd v14.4s, v14.4s, v4.4s\n"
      "and v7.16b, v8.16b, v0.16b\n"
      "and v6.16b, v9.16b, v1.16b\n"
      "and v5.16b, v10.16b, v2.16b\n"
      "and v4.16b, v11.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v8.4s, v8.4s, v7.4s\n"
      "sqadd v9.4s, v9.4s, v6.4s\n"
      "sqadd v10.4s, v10.4s, v5.4s\n"
      "sqadd v11.4s, v11.4s, v4.4s\n"
      "and v7.16b, v15.16b, v0.16b\n"
      "and v6.16b, v20.16b, v1.16b\n"
      "and v5.16b, v21.16b, v2.16b\n"
      "and v4.16b, v22.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v15.4s, v15.4s, v7.4s\n"
      "sqadd v20.4s, v20.4s, v6.4s\n"
      "sqadd v21.4s, v21.4s, v5.4s\n"
      "sqadd v22.4s, v22.4s, v4.4s\n"
      "and v7.16b, v16.16b, v0.16b\n"
      "and v6.16b, v17.16b, v1.16b\n"
      "and v5.16b, v18.16b, v2.16b\n"
      "and v4.16b, v19.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v7.4s\n"
      "sqadd v17.4s, v17.4s, v6.4s\n"
      "sqadd v18.4s, v18.4s, v5.4s\n"
      "sqadd v19.4s, v19.4s, v4.4s\n"
      "and v7.16b, v23.16b, v0.16b\n"
      "and v6.16b, v28.16b, v1.16b\n"
      "and v5.16b, v29.16b, v2.16b\n"
      "and v4.16b, v30.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v23.4s, v23.4s, v7.4s\n"
      "sqadd v28.4s, v28.4s, v6.4s\n"
      "sqadd v29.4s, v29.4s, v5.4s\n"
      "sqadd v30.4s, v30.4s, v4.4s\n"
      "and v7.16b, v24.16b, v0.16b\n"
      "and v6.16b, v25.16b, v1.16b\n"
      "and v5.16b, v26.16b, v2.16b\n"
      "and v4.16b, v27.16b, v3.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sqadd v24.4s, v24.4s, v7.4s\n"
      "sqadd v25.4s, v25.4s, v6.4s\n"
      "sqadd v26.4s, v26.4s, v5.4s\n"
      "sqadd v27.4s, v27.4s, v4.4s\n"
      "164:"  // Height 6: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      "ld1r { v6.4s }, [x20]\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v5.4s }, [x20]\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "ld1r { v4.4s }, [x20]\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "cmp x10, #0x10\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v1.4s\n"
      "srshl v21.4s, v21.4s, v2.4s\n"
      "srshl v22.4s, v22.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v28.4s, v28.4s, v1.4s\n"
      "srshl v29.4s, v29.4s, v2.4s\n"
      "srshl v30.4s, v30.4s, v3.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v1.4s\n"
      "srshl v26.4s, v26.4s, v2.4s\n"
      "srshl v27.4s, v27.4s, v3.4s\n"
      "add v31.4s, v31.4s, v6.4s\n"
      "add v12.4s, v12.4s, v6.4s\n"
      "add v13.4s, v13.4s, v6.4s\n"
      "add v14.4s, v14.4s, v6.4s\n"
      "add v8.4s, v8.4s, v6.4s\n"
      "add v9.4s, v9.4s, v6.4s\n"
      "add v10.4s, v10.4s, v6.4s\n"
      "add v11.4s, v11.4s, v6.4s\n"
      "add v15.4s, v15.4s, v6.4s\n"
      "add v20.4s, v20.4s, v6.4s\n"
      "add v21.4s, v21.4s, v6.4s\n"
      "add v22.4s, v22.4s, v6.4s\n"
      "add v16.4s, v16.4s, v6.4s\n"
      "add v17.4s, v17.4s, v6.4s\n"
      "add v18.4s, v18.4s, v6.4s\n"
      "add v19.4s, v19.4s, v6.4s\n"
      "add v23.4s, v23.4s, v6.4s\n"
      "add v28.4s, v28.4s, v6.4s\n"
      "add v29.4s, v29.4s, v6.4s\n"
      "add v30.4s, v30.4s, v6.4s\n"
      "add v24.4s, v24.4s, v6.4s\n"
      "add v25.4s, v25.4s, v6.4s\n"
      "add v26.4s, v26.4s, v6.4s\n"
      "add v27.4s, v27.4s, v6.4s\n"
      "smin v31.4s, v31.4s, v5.4s\n"
      "smin v12.4s, v12.4s, v5.4s\n"
      "smin v13.4s, v13.4s, v5.4s\n"
      "smin v14.4s, v14.4s, v5.4s\n"
      "smin v8.4s, v8.4s, v5.4s\n"
      "smin v9.4s, v9.4s, v5.4s\n"
      "smin v10.4s, v10.4s, v5.4s\n"
      "smin v11.4s, v11.4s, v5.4s\n"
      "smin v15.4s, v15.4s, v5.4s\n"
      "smin v20.4s, v20.4s, v5.4s\n"
      "smin v21.4s, v21.4s, v5.4s\n"
      "smin v22.4s, v22.4s, v5.4s\n"
      "smin v16.4s, v16.4s, v5.4s\n"
      "smin v17.4s, v17.4s, v5.4s\n"
      "smin v18.4s, v18.4s, v5.4s\n"
      "smin v19.4s, v19.4s, v5.4s\n"
      "smin v23.4s, v23.4s, v5.4s\n"
      "smin v28.4s, v28.4s, v5.4s\n"
      "smin v29.4s, v29.4s, v5.4s\n"
      "smin v30.4s, v30.4s, v5.4s\n"
      "smin v24.4s, v24.4s, v5.4s\n"
      "smin v25.4s, v25.4s, v5.4s\n"
      "smin v26.4s, v26.4s, v5.4s\n"
      "smin v27.4s, v27.4s, v5.4s\n"
      "smax v31.4s, v31.4s, v4.4s\n"
      "smax v12.4s, v12.4s, v4.4s\n"
      "smax v13.4s, v13.4s, v4.4s\n"
      "smax v14.4s, v14.4s, v4.4s\n"
      "smax v8.4s, v8.4s, v4.4s\n"
      "smax v9.4s, v9.4s, v4.4s\n"
      "smax v10.4s, v10.4s, v4.4s\n"
      "smax v11.4s, v11.4s, v4.4s\n"
      "smax v15.4s, v15.4s, v4.4s\n"
      "smax v20.4s, v20.4s, v4.4s\n"
      "smax v21.4s, v21.4s, v4.4s\n"
      "smax v22.4s, v22.4s, v4.4s\n"
      "smax v16.4s, v16.4s, v4.4s\n"
      "smax v17.4s, v17.4s, v4.4s\n"
      "smax v18.4s, v18.4s, v4.4s\n"
      "smax v19.4s, v19.4s, v4.4s\n"
      "smax v23.4s, v23.4s, v4.4s\n"
      "smax v28.4s, v28.4s, v4.4s\n"
      "smax v29.4s, v29.4s, v4.4s\n"
      "smax v30.4s, v30.4s, v4.4s\n"
      "smax v24.4s, v24.4s, v4.4s\n"
      "smax v25.4s, v25.4s, v4.4s\n"
      "smax v26.4s, v26.4s, v4.4s\n"
      "smax v27.4s, v27.4s, v4.4s\n"
      "uzp1 v31.8h, v31.8h, v12.8h\n"
      "uzp1 v1.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v0.8h, v10.8h, v11.8h\n"
      "uzp1 v15.8h, v15.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v19.8h, v18.8h, v19.8h\n"
      "uzp1 v23.8h, v23.8h, v28.8h\n"
      "uzp1 v18.8h, v29.8h, v30.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v17.8h, v26.8h, v27.8h\n"
      "uzp1 v31.16b, v31.16b, v1.16b\n"
      "uzp1 v8.16b, v8.16b, v0.16b\n"
      "uzp1 v15.16b, v15.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v19.16b\n"
      "uzp1 v23.16b, v23.16b, v18.16b\n"
      "uzp1 v24.16b, v24.16b, v17.16b\n"
      "bge 173f\n"
      "tbz x10, #3, 168f\n"
      "str d31, [x11], #0x8\n"
      "str d8, [x25], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "tbz x10, #2, 166f\n"
      "st1 { v31.s }[2], [x11], #0x4\n"
      "st1 { v8.s }[2], [x25], #0x4\n"
      "st1 { v15.s }[2], [x24], #0x4\n"
      "st1 { v16.s }[2], [x23], #0x4\n"
      "st1 { v23.s }[2], [x22], #0x4\n"
      "st1 { v24.s }[2], [x21], #0x4\n"
      "tbz x10, #1, 165f\n"
      "st1 { v31.h }[6], [x11], #0x2\n"
      "st1 { v8.h }[6], [x25], #0x2\n"
      "st1 { v15.h }[6], [x24], #0x2\n"
      "st1 { v16.h }[6], [x23], #0x2\n"
      "st1 { v23.h }[6], [x22], #0x2\n"
      "st1 { v24.h }[6], [x21], #0x2\n"
      "tbz x10, #0, 172f\n"
      "st1 { v31.b }[14], [x11]\n"
      "st1 { v8.b }[14], [x25]\n"
      "st1 { v15.b }[14], [x24]\n"
      "st1 { v16.b }[14], [x23]\n"
      "st1 { v23.b }[14], [x22]\n"
      "st1 { v24.b }[14], [x21]\n"
      "b 172f\n"
      "165:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x10, #0, 172f\n"
      "st1 { v31.b }[12], [x11]\n"
      "st1 { v8.b }[12], [x25]\n"
      "st1 { v15.b }[12], [x24]\n"
      "st1 { v16.b }[12], [x23]\n"
      "st1 { v23.b }[12], [x22]\n"
      "st1 { v24.b }[12], [x21]\n"
      "b 172f\n"
      "166:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x10, #1, 167f\n"
      "st1 { v31.h }[4], [x11], #0x2\n"
      "st1 { v8.h }[4], [x25], #0x2\n"
      "st1 { v15.h }[4], [x24], #0x2\n"
      "st1 { v16.h }[4], [x23], #0x2\n"
      "st1 { v23.h }[4], [x22], #0x2\n"
      "st1 { v24.h }[4], [x21], #0x2\n"
      "tbz x10, #0, 172f\n"
      "st1 { v31.b }[10], [x11]\n"
      "st1 { v8.b }[10], [x25]\n"
      "st1 { v15.b }[10], [x24]\n"
      "st1 { v16.b }[10], [x23]\n"
      "st1 { v23.b }[10], [x22]\n"
      "st1 { v24.b }[10], [x21]\n"
      "b 172f\n"
      "167:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x10, #0, 172f\n"
      "st1 { v31.b }[8], [x11]\n"
      "st1 { v8.b }[8], [x25]\n"
      "st1 { v15.b }[8], [x24]\n"
      "st1 { v16.b }[8], [x23]\n"
      "st1 { v23.b }[8], [x22]\n"
      "st1 { v24.b }[8], [x21]\n"
      "b 172f\n"
      "168:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x10, #2, 170f\n"
      "str s31, [x11], #0x4\n"
      "str s8, [x25], #0x4\n"
      "str s15, [x24], #0x4\n"
      "str s16, [x23], #0x4\n"
      "str s23, [x22], #0x4\n"
      "str s24, [x21], #0x4\n"
      "tbz x10, #1, 169f\n"
      "st1 { v31.h }[2], [x11], #0x2\n"
      "st1 { v8.h }[2], [x25], #0x2\n"
      "st1 { v15.h }[2], [x24], #0x2\n"
      "st1 { v16.h }[2], [x23], #0x2\n"
      "st1 { v23.h }[2], [x22], #0x2\n"
      "st1 { v24.h }[2], [x21], #0x2\n"
      "tbz x10, #0, 172f\n"
      "st1 { v31.b }[6], [x11]\n"
      "st1 { v8.b }[6], [x25]\n"
      "st1 { v15.b }[6], [x24]\n"
      "st1 { v16.b }[6], [x23]\n"
      "st1 { v23.b }[6], [x22]\n"
      "st1 { v24.b }[6], [x21]\n"
      "b 172f\n"
      "169:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x10, #0, 172f\n"
      "st1 { v31.b }[4], [x11]\n"
      "st1 { v8.b }[4], [x25]\n"
      "st1 { v15.b }[4], [x24]\n"
      "st1 { v16.b }[4], [x23]\n"
      "st1 { v23.b }[4], [x22]\n"
      "st1 { v24.b }[4], [x21]\n"
      "b 172f\n"
      "170:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x10, #1, 171f\n"
      "str h31, [x11], #0x2\n"
      "str h8, [x25], #0x2\n"
      "str h15, [x24], #0x2\n"
      "str h16, [x23], #0x2\n"
      "str h23, [x22], #0x2\n"
      "str h24, [x21], #0x2\n"
      "tbz x10, #0, 172f\n"
      "st1 { v31.b }[2], [x11]\n"
      "st1 { v8.b }[2], [x25]\n"
      "st1 { v15.b }[2], [x24]\n"
      "st1 { v16.b }[2], [x23]\n"
      "st1 { v23.b }[2], [x22]\n"
      "st1 { v24.b }[2], [x21]\n"
      "b 172f\n"
      "171:"  // Height 6: Partial direct writeback: partial_1_0
      "str b31, [x11, #0x0]\n"
      "str b8, [x25, #0x0]\n"
      "str b15, [x24, #0x0]\n"
      "str b16, [x23, #0x0]\n"
      "str b23, [x22, #0x0]\n"
      "str b24, [x21, #0x0]\n"
      "172:"  // Height 6: Partial direct writeback: Done
      "b 174f\n"
      "173:"  // Height 6: Full writeback
      "str q31, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q8, [x25, #0x0]\n"
      "str q15, [x24, #0x0]\n"
      "str q16, [x23, #0x0]\n"
      "str q23, [x22, #0x0]\n"
      "str q24, [x21, #0x0]\n"
      "174:"  // Height 6: Writeback done
      "subs x10, x10, #0x10\n"
      "bgt 147b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 176f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 175f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "175:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "176:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [flags] "r" (flags), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_multiplier_ptr] "I" (offsetof(KernelArgs, multiplier_ptr)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_shift_ptr] "I" (offsetof(KernelArgs, shift_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x13", "x14", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
