/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#ifndef __GCS_CELLRENDERERSWATCH_H
#define __GCS_CELLRENDERERSWATCH_H

#include "core/gcs-types.h"
#include <gtkmm/cellrenderer.h>

namespace gcs
{
    namespace Widgets
    {

        /** A custom CellRenderer that is used to display a color swatch based
         * on a color value contained in a model row of a TreeView
         *
         * To use, create an instance of the renderer and add it to a
         * TreeViewColumn.  Then Connect the 'color' property of the
         * CellRenderer to one of the columns of the treemodel that specifies a
         * color hex string (i.e. "#FFFFFF").  This is done like this:
         *
         * pColumn->add_attribute(pRenderer->property_color(), columns.hexString)
         *
         * where pRenderer is the instance of the renderer that you created and pColumn is the TreeViewColumn
         */
        class CellRendererSwatch : public Gtk::CellRenderer
        {
            public:
                CellRendererSwatch(void);
                virtual ~CellRendererSwatch(void);

                /** The size of the colorswatch -- defaults to 18
                 */
                Glib::PropertyProxy<gint> property_size(void);
                Glib::PropertyProxy_ReadOnly<gint> property_size(void) const;

                /** The color of the colorswatch -- defaults to '#FFFFFF'
                 */
                Glib::PropertyProxy<ColorPtr> property_color(void);
                Glib::PropertyProxy_ReadOnly<ColorPtr> property_color(void)
                    const;

                /** The width of the border around the swatch in pixels --
                 * defaults to 1
                 */
                Glib::PropertyProxy<gint> property_border_width(void);
                Glib::PropertyProxy_ReadOnly<gint> property_border_width(void)
                    const;

                /** The color of the border around the swatch -- Currently this
                 * is not implemented and it's just always drawn in black
                 */
                Glib::PropertyProxy<ColorPtr> property_border_color(void);
                Glib::PropertyProxy_ReadOnly<ColorPtr>
                    property_border_color(void) const;

            protected:
                /** This function calculates the size of the cell.  It
                 * uses the values for padding and from the
                 * property_size() of the CellRenderer.  It gets called
                 * indirectly from the render_vfunc function for each cell that
                 * needs to be rendered.  It also calculates the offsets for
                 * the cell.
                 */
                virtual void get_size_vfunc(Gtk::Widget& widget,
                        const Gdk::Rectangle* cell_area,
                        int* x_offset, int* y_offset,
                        int* width, int* height) const;

                /** This function actually renders the cell.  It gets called
                 * automatically for each cell in the TreeView.  It calculates
                 * how much size it needs and then draws the swatch in this
                 * area
                 */
                virtual void render_vfunc(
                        const Glib::RefPtr<Gdk::Drawable>& window,
                        Gtk::Widget& widget,
                        const Gdk::Rectangle& background_area,
                        const Gdk::Rectangle& cell_area,
                        const Gdk::Rectangle& expose_area,
                        Gtk::CellRendererState flags);

            private:
                Glib::Property<gint> m_property_size;
                Glib::Property<ColorPtr> m_property_color;
                Glib::Property<gint> m_property_border_width;
                Glib::Property<ColorPtr> m_property_border_color;
        };

    } // namespace Widgets
} // namespace gcs
#endif // __GCS_CELLRENDERERSWATCH_H
