!{\src2tex{textfont=tt}}
!!****f* ABINIT/opt_proj
!! NAME
!! opt_proj
!!
!! FUNCTION
!! Apply Real Space Optimization (RSO) on non-local projectors in order
!! to smooth them and cut large reciprocal vectors contribution.
!! Directly written from:
!!  RD King-smith, MC Payne and JS Lin, Phys. Rev. B, 44 (1991), 13063
!!
!! COPYRIGHT
!! Copyright (C) 1998-2005 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! INPUTS
!!  pawrso
!!    %ecut_rso=Real Space Optimization parameter: plane wave cutoff = 1/2 Gmax**2
!!    %gfact_rso=Real Space Optimization parameter: Gamma/Gmax ratio
!!    %userso=TRUE if REAL Space Optimization is required
!!    %werror_rso=Real Space Optimization parameter: max. error W_l allowed
!!  pshead
!!    %basis_size= Number of elements for the paw nl basis
!!    %log_step= Logaritmic step corresponding to radial mesh
!!    %orbitals(basis_size)= Quantum number l for each basis function
!!    %rad_step= Radial step corresponding to radial mesh
!!    %wav_meshsz=Mesh size for partial waves and projectors inherited from uspp
!!  un_log= Unit number for log file (comments)
!!
!! SIDE EFFECTS
!!  pshead
!!    %prj_meshsz= Dimension of radial mesh for tproj
!!  pawps
!!    %tproj(prj_msz_max,basis_size)= projectors on partial waves
!!
!! PARENTS
!!      uspp2abinit
!!
!! CHILDREN
!!      aamat,bess,bessp,gauleg
!!
!! SOURCE

 subroutine opt_proj(pawps,pawrso,pshead,un_log)

  use defs_pawps
  use defs_basis

  implicit none

!Arguments ---------------------------------------------
 integer :: un_log
!These types are defined in defs_pawps
 type(pawps_type)    :: pawps
 type(pawrso_type)   :: pawrso
 type(pshead_type)   :: pshead

!Local variables ---------------------------------------
 integer,parameter :: ngaussmax=50
 integer, parameter :: nqmax=500

 integer :: ib,info,iq,iqp,ir,ll,nqgauss1,nqgauss2,r0_meshsz,rp_meshsz
 real(dp) :: aamat,bess,bessp,dq,gamma,gmax,r0,rc_prj,stepint,wwmax,wwl,xx

 integer, allocatable :: iwork(:)
 real(dp),allocatable :: am(:,:),bm(:),chi1g(:,:),chi2g(:),chireg(:,:),ff(:),&
&                        gg(:),qgauss1(:),qgauss2(:),rr(:),rr_(:),&
&                        wgauss1(:),wgauss2(:)
!-------------------------------------------------------

 if (.not.pawrso%userso) return

 write(un_log,'(3(/,a),/,a,f7.2,/,a,f6.2,/,a,g11.3)') &
&     '> USpp->Abinit translator INFO:',&
&     '   Optimizing non-local projectors',&
&     '   using Real Space Optimization...',&
&     '   Parameters: Ecut (Hartree)=',pawrso%ecut,&
&     '               Gamma/Gmax    =',pawrso%gfact,&
&     '               Wl max (error)=',pawrso%werror

!Initialize data for optimization
!=======================================================
 rc_prj=pshead%rad_step*(exp(dble(pshead%prj_meshsz-1)*pshead%log_step)-1.d0)
 r0=rc_prj/1.035d0
 gmax=sqrt(2.d0*pawrso%ecut)
 gamma=pawrso%gfact*gmax
 wwmax=1.d10

!Define q mesh for reciprocal space
!=======================================================
 nqgauss1=ngaussmax
 allocate(qgauss1(nqgauss1),wgauss1(nqgauss1))
 call gauleg(0.d0,gmax,qgauss1,wgauss1,nqgauss1)
 nqgauss2=ngaussmax
 allocate(qgauss2(nqgauss2),wgauss2(nqgauss2))
 call gauleg(gmax,gamma,qgauss2,wgauss2,nqgauss2)
 allocate(chi1g(nqgauss1,pshead%basis_size),chireg(nqmax,pshead%basis_size))
 allocate(am(nqgauss2,nqgauss2),bm(nqgauss2),chi2g(nqgauss2))

!Define r mesh for real space
!=======================================================
 rp_meshsz=pshead%prj_meshsz;allocate(ff(rp_meshsz))
 stepint=pshead%log_step
 allocate(rr(pshead%prj_msz_max),rr_(pshead%prj_msz_max))
 do ir=1,pshead%prj_msz_max
  rr(ir)=pshead%rad_step*(exp(dble(ir-1)*pshead%log_step)-1.d0)
  rr_(ir)=rr(ir)+pshead%rad_step
 enddo

!Transfer tproj(r) into chi(q) for 0<q<gmax
!=======================================================
!On a Gaussian mesh
 do ib=1,pshead%basis_size
  ll=pshead%orbitals(ib)
  do iq=1,nqgauss1
   do ir=1,rp_meshsz
    ff(ir)=bess(qgauss1(iq)*rr(ir),ll)*rr(ir)*rr_(ir)*pawps%tproj(ir,ib)
   enddo
   call ctrap(rp_meshsz,ff,stepint,chi1g(iq,ib))
  enddo
!On a regular mesh
  dq=gmax/dble(nqmax-1)
  do iq=1,nqmax
   do ir=1,rp_meshsz
    ff(ir)=bess(dble(iq-1)*dq*rr(ir),ll)*rr(ir)*rr_(ir)*pawps%tproj(ir,ib)
   enddo
   call ctrap(rp_meshsz,ff,stepint,chireg(iq,ib))
  enddo
 enddo

!Loop on error Wl
!=======================================================
 do while (wwmax>pawrso%werror)
  wwmax=-1.d0
  r0_meshsz=max(int(log(1.d0+(r0*1.035d0)/pshead%rad_step)/pshead%log_step)+1,pshead%prj_meshsz+1)
  if (r0_meshsz>pshead%prj_msz_max) then
   write(un_log,'(/,a,/,a)') '> USpp->Abinit translator ERROR:',&
&                            '    opt_proj: ro_meshsz too big !'
   stop
  endif
  r0=rr(r0_meshsz)
  allocate(gg(r0_meshsz))

!Loop on (l,n) basis
!=======================================================
 do ib=1,pshead%basis_size
  ll=pshead%orbitals(ib)

!Compute chi(q) for gmax<q<gamma on Gauss mesh
!=======================================================
! Loop on q
  do iq=1,nqgauss2

!  Compute bm(q)
   bm(iq)=zero
   do iqp=1,nqgauss1
    bm(iq)=bm(iq)+aamat(qgauss2(iq),qgauss1(iqp),r0,ll)*chi1g(iqp,ib)*wgauss1(iqp)
   enddo

!  Compute am(q,qp)
   do iqp=1,iq
    xx=-aamat(qgauss2(iq),qgauss2(iqp),r0,ll)
    am(iq,iqp)=xx*wgauss2(iqp)
    am(iqp,iq)=xx*wgauss2(iq)
   enddo
   am(iq,iq)=am(iq,iq)+(pi/2.d0)*qgauss2(iq)**2

! End loop on q
  enddo


! Solve Am(q,qp).X(q)=Bm(q)
  allocate(iwork(nqgauss2))
  call DGETRF(nqgauss2,nqgauss2,am,nqgauss2,iwork,info)
  call DGETRS('N',nqgauss2,1,am,nqgauss2,iwork,bm,nqgauss2,info)
  deallocate(iwork)

  chi2g=bm

!Transfer back chi(q) into tproj(r)
!=======================================================
  do ir=1,r0_meshsz
   xx=zero
   do iq=1,nqgauss1
    xx=xx+wgauss1(iq)*bess(qgauss1(iq)*rr(ir),ll)*chi1g(iq,ib)*qgauss1(iq)**2
   enddo
   do iq=1,nqgauss2
    xx=xx+wgauss2(iq)*bess(qgauss2(iq)*rr(ir),ll)*chi2g(iq)*qgauss2(iq)**2
   enddo
   pawps%tproj(ir,ib)=2.d0*rr(ir)*xx/pi
  enddo

!Estimate the error W_l(q)
!=======================================================
! Compute Int(0,R0) [r**2.chi(r).jl(qr)] (and Wl)
! for each q of the regular mesh
  wwl=-1.d0
  do iq=1,nqmax
   do ir=1,r0_meshsz
    gg(ir)=bess(dble(iq-1)*dq*rr(ir),ll)*rr(ir)*rr_(ir)*pawps%tproj(ir,ib)
   enddo
   call ctrap(r0_meshsz,gg,stepint,xx)
   wwl=max(abs(chireg(iq,ib)-xx),wwl)
  enddo
  wwl=wwl/maxval(abs(chireg(:,ib)))
  wwmax=max(wwl,wwmax)

!End loop on ib
!=======================================================
 enddo

!End loop on error
!=======================================================
 deallocate(gg)
 enddo

 deallocate(am,bm,chi1g,chi2g,chireg,ff,wgauss1,qgauss1,rr,rr_,wgauss2,qgauss2)

 pshead%prj_meshsz=r0_meshsz
 write(un_log,'(2(a,f7.4),a)') '   New radius R0 for nl projectors (Bohr)=',&
&                         r0,' (=',r0/rc_prj,'*Rc(proj))'
 if (r0>1.55d0*rc_prj) &
&  write(un_log,'(4(/,a))') &
&                'Warning:',&
&                '  Radius for nl projectors (R0) seems to be high !',&
&                '  You should change parameters of Real Space Optimization',&
&                '   (increase Ecut, Gamma/Gmax or Wl).'

 end subroutine opt_proj



!==============================================
 double precision function aamat(qq,qqp,r0,ll)
!==============================================

 use defs_basis

 implicit none

 integer :: ll
 real(dp) :: bess,bessp,qq,qqp,r0

 if (dabs(qq-qqp)<tol10) then
  aamat=qq**3*r0**2/2.d0*(bess(qqp*r0,ll)*bessp(qqp*r0,ll) &
&                         +qq*r0*(bessp(qqp*r0,ll))**2 &
&                         +qq*r0*(bess (qqp*r0,ll))**2 &
&                         -dble(ll*(ll+1))/(qq*r0)*(bess(qqp*r0,ll))**2)
 else
  aamat=qq**2*qqp**2/(qqp**2-qq**2)*r0**2 &
&       *(bess(qqp*r0,ll)*bessp(qq*r0,ll)*qq-bess(qq*r0,ll)*bessp(qqp*r0,ll)*qqp)
 endif

 end function aamat
