/*
 * Decompiled with CFR 0.152.
 */
package picard.analysis;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileReader;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.reference.ReferenceSequenceFileFactory;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.PeekableIterator;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.QualityUtil;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.text.NumberFormat;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import picard.PicardException;
import picard.analysis.GcBiasDetailMetrics;
import picard.analysis.GcBiasSummaryMetrics;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.Metrics;
import picard.util.RExecutor;

@CommandLineProgramProperties(usage="Tool to collect information about GC bias in the reads in a given BAM file. Computes the number of windows (of size specified by WINDOW_SIZE) in the genome at each GC% and counts the number of read starts in each GC bin.  What is output and plotted is the \"normalized coverage\" in each bin - i.e. the number of reads per window normalized to the average number of reads per window across the whole genome..\n", usageShort="Collects information about GC bias in the reads in the provided SAM or BAM", programGroup=Metrics.class)
public class CollectGcBiasMetrics
extends CommandLineProgram {
    private static final String R_SCRIPT = "picard/analysis/gcBias.R";
    @Option(shortName="R", doc="The reference sequence fasta file.")
    public File REFERENCE_SEQUENCE;
    @Option(shortName="I", doc="The BAM or SAM file containing aligned reads.  Must be coordinate-sorted.")
    public File INPUT;
    @Option(shortName="O", doc="The text file to write the metrics table to.")
    public File OUTPUT;
    @Option(shortName="CHART", doc="The PDF file to render the chart to.")
    public File CHART_OUTPUT;
    @Option(shortName="S", doc="The text file to write summary metrics to.", optional=true)
    public File SUMMARY_OUTPUT;
    @Option(doc="The size of windows on the genome that are used to bin reads.")
    public int WINDOW_SIZE = 100;
    @Option(doc="For summary metrics, exclude GC windows that include less than this fraction of the genome.")
    public double MINIMUM_GENOME_FRACTION = 1.0E-5;
    @Option(doc="If true, assume that the input file is coordinate sorted, even if the header says otherwise.", shortName="AS")
    public boolean ASSUME_SORTED = false;
    @Option(shortName="BS", doc="Whether the SAM or BAM file consists of bisulfite sequenced reads.  ")
    public boolean IS_BISULFITE_SEQUENCED = false;
    private static final Log log = Log.getInstance(CollectGcBiasMetrics.class);
    private int totalClusters = 0;
    private int totalAlignedReads = 0;

    public static void main(String[] args) {
        System.exit(new CollectGcBiasMetrics().instanceMain(args));
    }

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable((File)this.REFERENCE_SEQUENCE);
        IOUtil.assertFileIsReadable((File)this.INPUT);
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        IOUtil.assertFileIsWritable((File)this.CHART_OUTPUT);
        if (this.SUMMARY_OUTPUT != null) {
            IOUtil.assertFileIsWritable((File)this.SUMMARY_OUTPUT);
        }
        int[] windowsByGc = new int[101];
        int[] readsByGc = new int[101];
        long[] basesByGc = new long[101];
        long[] errorsByGc = new long[101];
        SAMFileReader sam = new SAMFileReader(this.INPUT);
        if (!this.ASSUME_SORTED && sam.getFileHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new PicardException("Header of input file " + this.INPUT.getAbsolutePath() + " indicates that it is not coordinate sorted.  " + "If you believe the records are in coordinate order, pass option ASSUME_SORTED=true.  If not, sort the file with SortSam.");
        }
        PeekableIterator iterator = new PeekableIterator((Iterator)sam.iterator());
        ReferenceSequenceFile referenceFile = ReferenceSequenceFileFactory.getReferenceSequenceFile((File)this.REFERENCE_SEQUENCE);
        SAMSequenceDictionary referenceDictionary = referenceFile.getSequenceDictionary();
        SAMSequenceDictionary samFileDictionary = sam.getFileHeader().getSequenceDictionary();
        if (referenceDictionary != null && samFileDictionary != null) {
            SequenceUtil.assertSequenceDictionariesEqual((SAMSequenceDictionary)referenceDictionary, (SAMSequenceDictionary)samFileDictionary);
        }
        ReferenceSequence ref = null;
        ProgressLogger progress = new ProgressLogger(log);
        while ((ref = referenceFile.nextSequence()) != null) {
            byte[] refBases = ref.getBases();
            StringUtil.toUpperCase((byte[])refBases);
            int refLength = refBases.length;
            int lastWindowStart = refLength - this.WINDOW_SIZE;
            byte[] gc = this.calculateAllGcs(refBases, windowsByGc, lastWindowStart);
            while (iterator.hasNext() && ((SAMRecord)iterator.peek()).getReferenceIndex().intValue() == ref.getContigIndex()) {
                SAMRecord rec = (SAMRecord)iterator.next();
                if (!rec.getReadPairedFlag() || rec.getFirstOfPairFlag()) {
                    ++this.totalClusters;
                }
                if (!rec.getReadUnmappedFlag()) {
                    byte windowGc;
                    int pos = rec.getReadNegativeStrandFlag() ? rec.getAlignmentEnd() - this.WINDOW_SIZE : rec.getAlignmentStart();
                    ++this.totalAlignedReads;
                    if (pos > 0 && (windowGc = gc[pos]) >= 0) {
                        byte by = windowGc;
                        readsByGc[by] = readsByGc[by] + 1;
                        byte by2 = windowGc;
                        basesByGc[by2] = basesByGc[by2] + (long)rec.getReadLength();
                        byte by3 = windowGc;
                        errorsByGc[by3] = errorsByGc[by3] + (long)(SequenceUtil.countMismatches((SAMRecord)rec, (byte[])refBases, (boolean)this.IS_BISULFITE_SEQUENCED) + SequenceUtil.countInsertedBases((SAMRecord)rec) + SequenceUtil.countDeletedBases((SAMRecord)rec));
                    }
                }
                progress.record(rec);
            }
            log.info(new Object[]{"Processed reference sequence: " + ref.getName()});
        }
        while (iterator.hasNext()) {
            SAMRecord rec = (SAMRecord)iterator.next();
            if (rec.getReadPairedFlag() && !rec.getFirstOfPairFlag()) continue;
            ++this.totalClusters;
        }
        MetricsFile metricsFile = this.getMetricsFile();
        double totalWindows = this.sum(windowsByGc);
        double totalReads = this.sum(readsByGc);
        double meanReadsPerWindow = totalReads / totalWindows;
        double minimumWindowsToCountInSummary = totalWindows * this.MINIMUM_GENOME_FRACTION;
        for (int i = 0; i < windowsByGc.length; ++i) {
            if (windowsByGc[i] == 0) continue;
            GcBiasDetailMetrics m = new GcBiasDetailMetrics();
            m.GC = i;
            m.WINDOWS = windowsByGc[i];
            m.READ_STARTS = readsByGc[i];
            if (errorsByGc[i] > 0L) {
                m.MEAN_BASE_QUALITY = QualityUtil.getPhredScoreFromObsAndErrors((double)basesByGc[i], (double)errorsByGc[i]);
            }
            m.NORMALIZED_COVERAGE = (double)m.READ_STARTS / (double)m.WINDOWS / meanReadsPerWindow;
            m.ERROR_BAR_WIDTH = Math.sqrt(m.READ_STARTS) / (double)m.WINDOWS / meanReadsPerWindow;
            metricsFile.addMetric((MetricBase)m);
        }
        metricsFile.write(this.OUTPUT);
        if (this.SUMMARY_OUTPUT != null) {
            MetricsFile summaryMetricsFile = this.getMetricsFile();
            GcBiasSummaryMetrics summary = new GcBiasSummaryMetrics();
            summary.WINDOW_SIZE = this.WINDOW_SIZE;
            summary.TOTAL_CLUSTERS = this.totalClusters;
            summary.ALIGNED_READS = this.totalAlignedReads;
            this.calculateDropoutMetrics(metricsFile.getMetrics(), summary);
            summaryMetricsFile.addMetric((MetricBase)summary);
            summaryMetricsFile.write(this.SUMMARY_OUTPUT);
        }
        NumberFormat fmt = NumberFormat.getIntegerInstance();
        fmt.setGroupingUsed(true);
        String subtitle = "Total clusters: " + fmt.format(this.totalClusters) + ", Aligned reads: " + fmt.format(this.totalAlignedReads);
        String title = this.INPUT.getName().replace(".duplicates_marked", "").replace(".aligned.bam", "");
        List readGroups = sam.getFileHeader().getReadGroups();
        if (readGroups.size() == 1) {
            title = title + "." + ((SAMReadGroupRecord)readGroups.get(0)).getLibrary();
        }
        RExecutor.executeFromClasspath(R_SCRIPT, this.OUTPUT.getAbsolutePath(), this.CHART_OUTPUT.getAbsolutePath(), title, subtitle, String.valueOf(this.WINDOW_SIZE));
        return 0;
    }

    private double sum(int[] values) {
        int length = values.length;
        double total = 0.0;
        for (int i = 0; i < length; ++i) {
            total += (double)values[i];
        }
        return total;
    }

    private void calculateDropoutMetrics(Collection<GcBiasDetailMetrics> details, GcBiasSummaryMetrics summary) {
        double totalReads = 0.0;
        double totalWindows = 0.0;
        for (GcBiasDetailMetrics detail : details) {
            totalReads += (double)detail.READ_STARTS;
            totalWindows += (double)detail.WINDOWS;
        }
        double atDropout = 0.0;
        double gcDropout = 0.0;
        for (GcBiasDetailMetrics detail : details) {
            double relativeWindows = (double)detail.WINDOWS / totalWindows;
            double relativeReads = (double)detail.READ_STARTS / totalReads;
            double dropout = (relativeWindows - relativeReads) * 100.0;
            if (!(dropout > 0.0)) continue;
            if (detail.GC <= 50) {
                atDropout += dropout;
            }
            if (detail.GC < 50) continue;
            gcDropout += dropout;
        }
        summary.AT_DROPOUT = atDropout;
        summary.GC_DROPOUT = gcDropout;
    }

    private byte[] calculateAllGcs(byte[] refBases, int[] windowsByGc, int lastWindowStart) {
        int refLength = refBases.length;
        byte[] gc = new byte[refLength + 1];
        CalculateGcState state = new CalculateGcState();
        for (int i = 1; i < lastWindowStart; ++i) {
            int windowEnd = i + this.WINDOW_SIZE;
            int windowGc = this.calculateGc(refBases, i, windowEnd, state);
            gc[i] = (byte)windowGc;
            if (windowGc == -1) continue;
            int n = windowGc;
            windowsByGc[n] = windowsByGc[n] + 1;
        }
        return gc;
    }

    private int calculateGc(byte[] bases, int startIndex, int endIndex, CalculateGcState state) {
        if (state.init) {
            state.init = false;
            state.gcCount = 0;
            state.nCount = 0;
            for (int i = startIndex; i < endIndex; ++i) {
                byte base = bases[i];
                if (base == 71 || base == 67) {
                    ++state.gcCount;
                    continue;
                }
                if (base != 78) continue;
                ++state.nCount;
            }
        } else {
            byte newBase = bases[endIndex - 1];
            if (newBase == 71 || newBase == 67) {
                ++state.gcCount;
            } else if (newBase == 78) {
                ++state.nCount;
            }
            if (state.priorBase == 71 || state.priorBase == 67) {
                --state.gcCount;
            } else if (state.priorBase == 78) {
                --state.nCount;
            }
        }
        state.priorBase = bases[startIndex];
        if (state.nCount > 4) {
            return -1;
        }
        return state.gcCount * 100 / (endIndex - startIndex);
    }

    class CalculateGcState {
        boolean init = true;
        int nCount;
        int gcCount;
        byte priorBase;

        CalculateGcState() {
        }
    }
}

