/*  Copyright (C) 2010 University of Oxford  

    Stamatios Sotiropoulos - FMRIB Image Analysis Group  */

/*  Part of FSL - FMRIB's Software Library
    http://www.fmrib.ox.ac.uk/fsl
    fsl@fmrib.ox.ac.uk
    
    Developed at FMRIB (Oxford Centre for Functional Magnetic Resonance
    Imaging of the Brain), Department of Clinical Neurology, Oxford
    University, Oxford, UK
    
    
    LICENCE
    
    FMRIB Software Library, Release 5.0 (c) 2012, The University of
    Oxford (the "Software")
    
    The Software remains the property of the University of Oxford ("the
    University").
    
    The Software is distributed "AS IS" under this Licence solely for
    non-commercial use in the hope that it will be useful, but in order
    that the University as a charitable foundation protects its assets for
    the benefit of its educational and research purposes, the University
    makes clear that no condition is made or to be implied, nor is any
    warranty given or to be implied, as to the accuracy of the Software,
    or that it will be suitable for any particular purpose or for use
    under any specific conditions. Furthermore, the University disclaims
    all responsibility for the use which is made of the Software. It
    further disclaims any liability for the outcomes arising from using
    the Software.
    
    The Licensee agrees to indemnify the University and hold the
    University harmless from and against any and all claims, damages and
    liabilities asserted by third parties (including claims for
    negligence) which arise directly or indirectly from the use of the
    Software or the sale of any products based on the Software.
    
    No part of the Software may be reproduced, modified, transmitted or
    transferred in any form or by any means, electronic or mechanical,
    without the express permission of the University. The permission of
    the University is not required if the said reproduction, modification,
    transmission or transference is done without financial return, the
    conditions of this Licence are imposed upon the receiver of the
    product, and all original and amended source code is included in any
    transmitted product. You may be held legally responsible for any
    copyright infringement that is caused or encouraged by your failure to
    abide by these terms and conditions.
    
    You are not permitted under this Licence to use this Software
    commercially. Use for which any financial return is received shall be
    defined as commercial use, and includes (1) integration of all or part
    of the source code or the Software into a product for sale or license
    by or on behalf of Licensee to third parties or (2) use of the
    Software or any derivative of it for research with the final aim of
    developing software products for sale or license to a third party or
    (3) use of the Software or any derivative of it for research with the
    final aim of developing non-software products for sale or license to a
    third party, or (4) use of the Software to provide any service to an
    external organisation for which payment is received. If you are
    interested in using the Software commercially, please contact Isis
    Innovation Limited ("Isis"), the technology transfer company of the
    University, to negotiate a licence. Contact details are:
    innovation@isis.ox.ac.uk quoting reference DE/9564. */

#if !defined(sphere_tessellation_h)
#define sphere_tessellation_h

using namespace NEWMAT;

namespace ODFs{
  
  //Class that holds points uniformly distributed on the surface of a sphere
  //Points are obtained using an icosahedral tessellation.
  //The Matrix index contains indices of the closest num_neighbours of each point.
  class Points{
    Matrix Tess_Points;           //Matrix that keeps points from a sphere tesselation (3 x Npoints)
    int num_points;               //Number of points obtained after tesselation 
    Matrix index;
    double mean_Eucl_dist;        //Mean Euclidean distance between neighbouring points
    double mean_ang_dist;         //Mean angular distance between neighbouring points
 
    const int& degree;            //Tessellation degree (0-5)
    const int& num_neighbours;    //Number of neighbours that will be considered 

  public:
  //constructor
  Points(const int& tessel_degree, const int& neighbours): degree(tessel_degree), num_neighbours(neighbours){
      sphere_tessellation_points();  
      num_points=Tess_Points.Ncols();
      index.ReSize(num_points, num_neighbours);
      find_closest();
      distances();
  } 
    
  //destructor
  ~Points(){}
    
  Matrix& get_TessPoints_Ref() { return Tess_Points; }
  Matrix get_TessPoints() { return Tess_Points; }
  ColumnVector get_TessPoints_coord(int coord_index) { return Tess_Points.t().Column(coord_index); } //Return x, y or z coordinates of Points (coord_index=1,2 or 3)
  Matrix& get_index_Ref() { return index; }
  Matrix get_index() { return index; }
  double get_Eucl_dist() const {return mean_Eucl_dist;}
  double get_ang_dist() const {return mean_ang_dist;}


  //Function that returns a matrix with pre-calculated points evenly distributed on the sphere.
  //Points have been calculated using a tessellation of an icosahedron. Degree corresponds to degree of
  //tesselation (from 0 to 5).
  void sphere_tessellation_points(){
    switch (degree){
      case 0: Tess_Points=tessellation_points0();
        break;
      case 1: Tess_Points=tessellation_points1();
        break;
      case 2: Tess_Points=tessellation_points2();
        break;
      case 3: Tess_Points=tessellation_points3();
        break;
      case 4: Tess_Points=tessellation_points4();
        break;
      case 5: Tess_Points=tessellation_points5();
        break;
    }
  }
  
  
  //For each of the Tess_Points returns the indices of the "num_neighbours" closer points. So index(j,k) will have an index to Tess_Points
  //for the kth closest point of the grid to point j (k<=num_neighbours). Stored indices start from 1! 
  //The algortihm computes the dot product of each point with all the others and keeps the points that correspond to the num_neighbours maximum dot products.
  void find_closest(){
    float min_entry, dot_prod;
    vector <float> max_dots(num_neighbours);
    vector <int> temp_index(num_neighbours);
    int min_index,i,k,j;

    for (i=1; i<=num_points; i++){	        //for each point
      for (k=0; k<num_neighbours; k++)          //max_dots=Array that keeps the max dot products, temp_index=array that keeps the index
	{max_dots[k]=0; temp_index[k]=0;}       //of the point corresponding to each dot product
      min_entry=0; min_index=0;		        //min_entry is the smallest value of the max_dots array occuring at min_index position
      for (j=1; j<=num_points; j++){	        //Compute the dot product with each other point.
	dot_prod=Tess_Points(1,i)*Tess_Points(1,j)+Tess_Points(2,i)*Tess_Points(2,j)+Tess_Points(3,i)*Tess_Points(3,j);
	if (i!=j && dot_prod>min_entry){        //if the dot product is larger than the smaller value of tha max_dots array
	  max_dots[min_index]=dot_prod;  	//replace it
	  temp_index[min_index]=j;		//include this point in the list of closest neighbours
	  min_entry=1;				//find the min and its location of the updated max_dots array
	  for (k=0; k<num_neighbours; k++)
	    if (max_dots[k]<min_entry)
	      { min_entry=max_dots[k]; min_index=k;}
	}
      }
      for (k=0; k<num_neighbours; k++)		//Once done with all points store the list of closest neighbours
	index(i,k+1)=temp_index[k];
    }
    
  } 
  
  //Compute the mean Euclidean and the mean angular distance between neighbouring points in the lattice 
  void distances(){
    double Eucl_dis, ang_dis;
    float dot_prod;
    ColumnVector p1(3),p2(3);

    mean_Eucl_dist=0;
    mean_ang_dist=0;

    for (int i=1; i<=num_points; i++){	        //for each point
      Eucl_dis=0; ang_dis=0;
      p1<<Tess_Points(1,i)<<Tess_Points(2,i)<<Tess_Points(3,i);
      for (int k=1; k<=num_neighbours; k++){       //Get the Euclidean and angular distance with each neighbour
	p2<<Tess_Points(1,(int)index(i,k))<<Tess_Points(2,(int)index(i,k))<<Tess_Points(3,(int)index(i,k));
	dot_prod=p1(1)*p2(1)+p1(2)*p2(2)+p1(3)*p2(3);  //dot product with neighbour
	if (dot_prod<0)
	  p2=-p2;
	Eucl_dis+=sqrt((p1(1)-p2(1))*(p1(1)-p2(1))+(p1(2)-p2(2))*(p1(2)-p2(2))+(p1(3)-p2(3))*(p1(3)-p2(3))); //Get the euclidean distance
	ang_dis+=fabs(dot_prod);	
      }
      Eucl_dis/=num_neighbours; ang_dis/=num_neighbours;   //Average those across neighbours
      mean_Eucl_dist+=Eucl_dis;                //And add to the mean across points
      mean_ang_dist+=ang_dis;
      }
    mean_Eucl_dist/=num_points;
    mean_ang_dist/=num_points;                 //So far this is the mean dot product between neighbours
    mean_ang_dist=acos(mean_ang_dist)*180/M_PI;//Get the angle now
  }



  //0th order tessellation
  ReturnMatrix tessellation_points0() const{
    Matrix Points(12,3);
    Real v[]={0.850651, 0.525731, 0.000000,
			    -0.850651, 0.525731, 0.000000,
			    -0.850651, -0.525731, 0.000000,
			    0.850651, -0.525731, 0.000000,
			    0.525731, 0.000000, 0.850651,
			    0.525731, 0.000000, -0.850651,
			    -0.525731, 0.000000, -0.850651,
			    -0.525731, 0.000000, 0.850651,
			    0.000000, 0.850651, 0.525731,
			    0.000000, -0.850651, 0.525731,
			    0.000000, -0.850651, -0.525731,
			    0.000000, 0.850651, -0.525731};
  
    Points << v;
    Points << Points.t();
    Points.Release();
    return Points;
  }



  //1st order tessellation
  ReturnMatrix tessellation_points1() const{
    Matrix Points(42,3);
    Real v[]={0.850651, 0.525731, 0.000000,
			    -0.850651, 0.525731, 0.000000,
			    -0.850651, -0.525731, 0.000000,
			    0.850651, -0.525731, 0.000000,
			    0.525731, 0.000000, 0.850651,
			    0.525731, 0.000000, -0.850651,
			    -0.525731, 0.000000, -0.850651,
			    -0.525731, 0.000000, 0.850651,
			    0.000000, 0.850651, 0.525731,
			    0.000000, -0.850651, 0.525731,
			    0.000000, -0.850651, -0.525731,
			    0.000000, 0.850651, -0.525731,
			    0.309017, 0.500000, 0.809017,
			    -0.309017, 0.500000, 0.809017,
			    0.000000, 0.000000, 1.000000,
			    -0.309017, -0.500000, 0.809017,
			    0.309017, -0.500000, 0.809017,
			    0.000000, 0.000000, -1.000000,
			    -0.309017, 0.500000, -0.809017,
			    0.309017, 0.500000, -0.809017,
			    0.309017, -0.500000, -0.809017,
			    -0.309017, -0.500000, -0.809017,
			    0.809017, 0.309017, 0.500000,
			    0.809017, -0.309017, 0.500000,
			    1.000000, 0.000000, 0.000000,
			    0.809017, -0.309017, -0.500000,
			    0.809017, 0.309017, -0.500000,
			    -0.809017, -0.309017, 0.500000,
			    -0.809017, 0.309017, 0.500000,
			    -1.000000, 0.000000, 0.000000,
			    -0.809017, 0.309017, -0.500000,
			    -0.809017, -0.309017, -0.500000,
			    0.500000, 0.809017, 0.309017,
			    0.500000, 0.809017, -0.309017,
			    0.000000, 1.000000, 0.000000,
			    -0.500000, 0.809017, -0.309017,
			    -0.500000, 0.809017, 0.309017,
			    0.000000, -1.000000, 0.000000,
			    0.500000, -0.809017, -0.309017,
			    0.500000, -0.809017, 0.309017,
			    -0.500000, -0.809017, 0.309017,
			    -0.500000, -0.809017, -0.309017};

    Points << v;
    Points << Points.t();
    Points.Release();
    return Points;
  }


  //2nd order tessellation
  ReturnMatrix tessellation_points2() const{
    Matrix Points(162,3);
    Real v[]={0.850651, 0.525731, 0.000000,
			     -0.850651, 0.525731, 0.000000,
			     -0.850651, -0.525731, 0.000000,
			     0.850651, -0.525731, 0.000000,
			     0.525731, 0.000000, 0.850651,
			     0.525731, 0.000000, -0.850651,
			     -0.525731, 0.000000, -0.850651,
			     -0.525731, 0.000000, 0.850651,
			     0.000000, 0.850651, 0.525731,
			     0.000000, -0.850651, 0.525731,
			     0.000000, -0.850651, -0.525731,
			     0.000000, 0.850651, -0.525731,
			     0.309017, 0.500000, 0.809017,
			     -0.309017, 0.500000, 0.809017,
			     0.000000, 0.000000, 1.000000,
			     -0.309017, -0.500000, 0.809017,
			     0.309017, -0.500000, 0.809017,
			     0.000000, 0.000000, -1.000000,
			     -0.309017, 0.500000, -0.809017,
			     0.309017, 0.500000, -0.809017,
			     0.309017, -0.500000, -0.809017,
			     -0.309017, -0.500000, -0.809017,
			     0.809017, 0.309017, 0.500000,
			     0.809017, -0.309017, 0.500000,
			     1.000000, 0.000000, 0.000000,
			     0.809017, -0.309017, -0.500000,
			     0.809017, 0.309017, -0.500000,
			     -0.809017, -0.309017, 0.500000,
			     -0.809017, 0.309017, 0.500000,
			     -1.000000, 0.000000, 0.000000,
			     -0.809017, 0.309017, -0.500000,
			     -0.809017, -0.309017, -0.500000,
			     0.500000, 0.809017, 0.309017,
			     0.500000, 0.809017, -0.309017,
			     0.000000, 1.000000, 0.000000,
			     -0.500000, 0.809017, -0.309017,
			     -0.500000, 0.809017, 0.309017,
			     0.000000, -1.000000, 0.000000,
			     0.500000, -0.809017, -0.309017,
			     0.500000, -0.809017, 0.309017,
			     -0.500000, -0.809017, 0.309017,
			     -0.500000, -0.809017, -0.309017,
			     0.433889, 0.259892, 0.862669,
			     0.162460, 0.262866, 0.951057,
			     0.273267, 0.000000, 0.961938,
			     0.000000, 0.525731, 0.850651,
			     -0.162460, 0.262866, 0.951057,
			     -0.433889, 0.259892, 0.862669,
			     -0.273267, 0.000000, 0.961938,
			     0.160622, 0.702046, 0.693780,
			     -0.160622, 0.702046, 0.693780,
			     0.162460, -0.262866, 0.951057,
			     0.433889, -0.259892, 0.862669,
			     -0.162460, -0.262866, 0.951057,
			     0.000000, -0.525731, 0.850651,
			     -0.160622, -0.702046, 0.693780,
			     0.160622, -0.702046, 0.693780,
			     -0.433889, -0.259892, 0.862669,
			     0.273267, 0.000000, -0.961938,
			     0.162460, 0.262866, -0.951057,
			     0.433889, 0.259892, -0.862669,
			     -0.162460, 0.262866, -0.951057,
			     0.000000, 0.525731, -0.850651,
			     -0.160622, 0.702046, -0.693780,
			     0.160622, 0.702046, -0.693780,
			     -0.273267, 0.000000, -0.961938,
			     -0.433889, 0.259892, -0.862669,
			     0.433889, -0.259892, -0.862669,
			     0.162460, -0.262866, -0.951057,
			     0.000000, -0.525731, -0.850651,
			     -0.162460, -0.262866, -0.951057,
			     -0.433889, -0.259892, -0.862669,
			     0.160622, -0.702046, -0.693780,
			     -0.160622, -0.702046, -0.693780,
			     0.862669, 0.433889, 0.259892,
			     0.951057, 0.162460, 0.262866,
			     0.961938, 0.273267, 0.000000,
			     0.850651, 0.000000, 0.525731,
			     0.951057, -0.162460, 0.262866,
			     0.862669, -0.433889, 0.259892,
			     0.961938, -0.273267, 0.000000,
			     0.693780, 0.160622, 0.702046,
			     0.693780, -0.160622, 0.702046,
			     0.951057, 0.162460, -0.262866,
			     0.862669, 0.433889, -0.259892,
			     0.951057, -0.162460, -0.262866,
			     0.850651, 0.000000, -0.525731,
			     0.693780, -0.160622, -0.702046,
			     0.693780, 0.160622, -0.702046,
			     0.862669, -0.433889, -0.259892,
			     -0.862669, -0.433889, 0.259892,
			     -0.951057, -0.162460, 0.262866,
			     -0.961938, -0.273267, 0.000000,
			     -0.850651, 0.000000, 0.525731,
			     -0.951057, 0.162460, 0.262866,
			     -0.862669, 0.433889, 0.259892,
			     -0.961938, 0.273267, 0.000000,
			     -0.693780, -0.160622, 0.702046,
			     -0.693780, 0.160622, 0.702046,
			     -0.951057, -0.162460, -0.262866,
			     -0.862669, -0.433889, -0.259892,
			     -0.951057, 0.162460, -0.262866,
			     -0.850651, 0.000000, -0.525731,
			     -0.693780, 0.160622, -0.702046,
			     -0.693780, -0.160622, -0.702046,
			     -0.862669, 0.433889, -0.259892,
			     0.259892, 0.862669, 0.433889,
			     0.262866, 0.951057, 0.162460,
			     0.000000, 0.961938, 0.273267,
			     0.525731, 0.850651, 0.000000,
			     0.262866, 0.951057, -0.162460,
			     0.259892, 0.862669, -0.433889,
			     0.000000, 0.961938, -0.273267,
			     0.702046, 0.693780, 0.160622,
			     0.702046, 0.693780, -0.160622,
			     -0.262866, 0.951057, 0.162460,
			     -0.259892, 0.862669, 0.433889,
			     -0.262866, 0.951057, -0.162460,
			     -0.525731, 0.850651, 0.000000,
			     -0.702046, 0.693780, -0.160622,
			     -0.702046, 0.693780, 0.160622,
			     -0.259892, 0.862669, -0.433889,
			     0.000000, -0.961938, 0.273267,
			     0.262866, -0.951057, 0.162460,
			     0.259892, -0.862669, 0.433889,
			     0.262866, -0.951057, -0.162460,
			     0.525731, -0.850651, 0.000000,
			     0.702046, -0.693780, -0.160622,
			     0.702046, -0.693780, 0.160622,
			     0.000000, -0.961938, -0.273267,
			     0.259892, -0.862669, -0.433889,
			     -0.259892, -0.862669, 0.433889,
			     -0.262866, -0.951057, 0.162460,
			     -0.525731, -0.850651, 0.000000,
			     -0.262866, -0.951057, -0.162460,
			     -0.259892, -0.862669, -0.433889,
			     -0.702046, -0.693780, 0.160622,
			     -0.702046, -0.693780, -0.160622,
			     0.425325, 0.688191, 0.587785,
			     0.587785, 0.425325, 0.688191,
			     0.688191, 0.587785, 0.425325,
			     0.425325, 0.688191, -0.587785,
			     0.688191, 0.587785, -0.425325,
			     0.587785, 0.425325, -0.688191,
			     0.587785, -0.425325, 0.688191,
			     0.425325, -0.688191, 0.587785,
			     0.688191, -0.587785, 0.425325,
			     0.587785, -0.425325, -0.688191,
			     0.688191, -0.587785, -0.425325,
			     0.425325, -0.688191, -0.587785,
			     -0.587785, 0.425325, 0.688191,
			     -0.425325, 0.688191, 0.587785,
			     -0.688191, 0.587785, 0.425325,
			     -0.587785, 0.425325, -0.688191,
			     -0.688191, 0.587785, -0.425325,
			     -0.425325, 0.688191, -0.587785,
			     -0.587785, -0.425325, 0.688191,
			     -0.688191, -0.587785, 0.425325,
			     -0.425325, -0.688191, 0.587785,
			     -0.587785, -0.425325, -0.688191,
			     -0.425325, -0.688191, -0.587785,
			     -0.688191, -0.587785, -0.425325};

    Points << v;
    Points << Points.t();
    Points.Release();
    return Points;
  }

  //3rd order tessellation
  ReturnMatrix tessellation_points3() const {
    Matrix Points(642,3);
    Real v[]={0.850651, 0.525731, 0.000000,
			    -0.850651, 0.525731, 0.000000,
			    -0.850651, -0.525731, 0.000000,
			    0.850651, -0.525731, 0.000000,
			    0.525731, 0.000000, 0.850651,
			    0.525731, 0.000000, -0.850651,
			    -0.525731, 0.000000, -0.850651,
			    -0.525731, 0.000000, 0.850651,
			    0.000000, 0.850651, 0.525731,
			    0.000000, -0.850651, 0.525731,
			    0.000000, -0.850651, -0.525731,
			    0.000000, 0.850651, -0.525731,
			    0.309017, 0.500000, 0.809017,
			    -0.309017, 0.500000, 0.809017,
			    0.000000, 0.000000, 1.000000,
			    -0.309017, -0.500000, 0.809017,
			    0.309017, -0.500000, 0.809017,
			    0.000000, 0.000000, -1.000000,
			    -0.309017, 0.500000, -0.809017,
			    0.309017, 0.500000, -0.809017,
			    0.309017, -0.500000, -0.809017,
			    -0.309017, -0.500000, -0.809017,
			    0.809017, 0.309017, 0.500000,
			    0.809017, -0.309017, 0.500000,
			    1.000000, 0.000000, 0.000000,
			    0.809017, -0.309017, -0.500000,
			    0.809017, 0.309017, -0.500000,
			    -0.809017, -0.309017, 0.500000,
			    -0.809017, 0.309017, 0.500000,
			    -1.000000, 0.000000, 0.000000,
			    -0.809017, 0.309017, -0.500000,
			    -0.809017, -0.309017, -0.500000,
			    0.500000, 0.809017, 0.309017,
			    0.500000, 0.809017, -0.309017,
			    0.000000, 1.000000, 0.000000,
			    -0.500000, 0.809017, -0.309017,
			    -0.500000, 0.809017, 0.309017,
			    0.000000, -1.000000, 0.000000,
			    0.500000, -0.809017, -0.309017,
			    0.500000, -0.809017, 0.309017,
			    -0.500000, -0.809017, 0.309017,
			    -0.500000, -0.809017, -0.309017,
			    0.433889, 0.259892, 0.862669,
			    0.162460, 0.262866, 0.951057,
			    0.273267, 0.000000, 0.961938,
			    0.000000, 0.525731, 0.850651,
			    -0.162460, 0.262866, 0.951057,
			    -0.433889, 0.259892, 0.862669,
			    -0.273267, 0.000000, 0.961938,
			    0.160622, 0.702046, 0.693780,
			    -0.160622, 0.702046, 0.693780,
			    0.162460, -0.262866, 0.951057,
			    0.433889, -0.259892, 0.862669,
			    -0.162460, -0.262866, 0.951057,
			    0.000000, -0.525731, 0.850651,
			    -0.160622, -0.702046, 0.693780,
			    0.160622, -0.702046, 0.693780,
			    -0.433889, -0.259892, 0.862669,
			    0.273267, 0.000000, -0.961938,
			    0.162460, 0.262866, -0.951057,
			    0.433889, 0.259892, -0.862669,
			    -0.162460, 0.262866, -0.951057,
			    0.000000, 0.525731, -0.850651,
			    -0.160622, 0.702046, -0.693780,
			    0.160622, 0.702046, -0.693780,
			    -0.273267, 0.000000, -0.961938,
			    -0.433889, 0.259892, -0.862669,
			    0.433889, -0.259892, -0.862669,
			    0.162460, -0.262866, -0.951057,
			    0.000000, -0.525731, -0.850651,
			    -0.162460, -0.262866, -0.951057,
			    -0.433889, -0.259892, -0.862669,
			    0.160622, -0.702046, -0.693780,
			    -0.160622, -0.702046, -0.693780,
			    0.862669, 0.433889, 0.259892,
			    0.951057, 0.162460, 0.262866,
			    0.961938, 0.273267, 0.000000,
			    0.850651, 0.000000, 0.525731,
			    0.951057, -0.162460, 0.262866,
			    0.862669, -0.433889, 0.259892,
			    0.961938, -0.273267, 0.000000,
			    0.693780, 0.160622, 0.702046,
			    0.693780, -0.160622, 0.702046,
			    0.951057, 0.162460, -0.262866,
			    0.862669, 0.433889, -0.259892,
			    0.951057, -0.162460, -0.262866,
			    0.850651, 0.000000, -0.525731,
			    0.693780, -0.160622, -0.702046,
			    0.693780, 0.160622, -0.702046,
			    0.862669, -0.433889, -0.259892,
			    -0.862669, -0.433889, 0.259892,
			    -0.951057, -0.162460, 0.262866,
			    -0.961938, -0.273267, 0.000000,
			    -0.850651, 0.000000, 0.525731,
			    -0.951057, 0.162460, 0.262866,
			    -0.862669, 0.433889, 0.259892,
			    -0.961938, 0.273267, 0.000000,
			    -0.693780, -0.160622, 0.702046,
			    -0.693780, 0.160622, 0.702046,
			    -0.951057, -0.162460, -0.262866,
			    -0.862669, -0.433889, -0.259892,
			    -0.951057, 0.162460, -0.262866,
			    -0.850651, 0.000000, -0.525731,
			    -0.693780, 0.160622, -0.702046,
			    -0.693780, -0.160622, -0.702046,
			    -0.862669, 0.433889, -0.259892,
			    0.259892, 0.862669, 0.433889,
			    0.262866, 0.951057, 0.162460,
			    0.000000, 0.961938, 0.273267,
			    0.525731, 0.850651, 0.000000,
			    0.262866, 0.951057, -0.162460,
			    0.259892, 0.862669, -0.433889,
			    0.000000, 0.961938, -0.273267,
			    0.702046, 0.693780, 0.160622,
			    0.702046, 0.693780, -0.160622,
			    -0.262866, 0.951057, 0.162460,
			    -0.259892, 0.862669, 0.433889,
			    -0.262866, 0.951057, -0.162460,
			    -0.525731, 0.850651, 0.000000,
			    -0.702046, 0.693780, -0.160622,
			    -0.702046, 0.693780, 0.160622,
			    -0.259892, 0.862669, -0.433889,
			    0.000000, -0.961938, 0.273267,
			    0.262866, -0.951057, 0.162460,
			    0.259892, -0.862669, 0.433889,
			    0.262866, -0.951057, -0.162460,
			    0.525731, -0.850651, 0.000000,
			    0.702046, -0.693780, -0.160622,
			    0.702046, -0.693780, 0.160622,
			    0.000000, -0.961938, -0.273267,
			    0.259892, -0.862669, -0.433889,
			    -0.259892, -0.862669, 0.433889,
			    -0.262866, -0.951057, 0.162460,
			    -0.525731, -0.850651, 0.000000,
			    -0.262866, -0.951057, -0.162460,
			    -0.259892, -0.862669, -0.433889,
			    -0.702046, -0.693780, 0.160622,
			    -0.702046, -0.693780, -0.160622,
			    0.425325, 0.688191, 0.587785,
			    0.587785, 0.425325, 0.688191,
			    0.688191, 0.587785, 0.425325,
			    0.425325, 0.688191, -0.587785,
			    0.688191, 0.587785, -0.425325,
			    0.587785, 0.425325, -0.688191,
			    0.587785, -0.425325, 0.688191,
			    0.425325, -0.688191, 0.587785,
			    0.688191, -0.587785, 0.425325,
			    0.587785, -0.425325, -0.688191,
			    0.688191, -0.587785, -0.425325,
			    0.425325, -0.688191, -0.587785,
			    -0.587785, 0.425325, 0.688191,
			    -0.425325, 0.688191, 0.587785,
			    -0.688191, 0.587785, 0.425325,
			    -0.587785, 0.425325, -0.688191,
			    -0.688191, 0.587785, -0.425325,
			    -0.425325, 0.688191, -0.587785,
			    -0.587785, -0.425325, 0.688191,
			    -0.688191, -0.587785, 0.425325,
			    -0.425325, -0.688191, 0.587785,
			    -0.587785, -0.425325, -0.688191,
			    -0.425325, -0.688191, -0.587785,
			    -0.688191, -0.587785, -0.425325,
			    0.484442, 0.131200, 0.864929,
			    0.358229, 0.131655, 0.924305,
			    0.403355, 0.000000, 0.915043,
			    0.301259, 0.264083, 0.916244,
			    0.220117, 0.132792, 0.966393,
			    0.082242, 0.133071, 0.987688,
			    0.137952, 0.000000, 0.990439,
			    0.375039, 0.383614, 0.843911,
			    0.238677, 0.386187, 0.891006,
			    0.000000, 0.266405, 0.963861,
			    -0.082242, 0.133071, 0.987688,
			    0.082324, 0.399607, 0.912982,
			    -0.082324, 0.399607, 0.912982,
			    -0.156434, 0.519258, 0.840178,
			    -0.238677, 0.386187, 0.891006,
			    0.156434, 0.519258, 0.840178,
			    -0.220117, 0.132792, 0.966393,
			    -0.137952, 0.000000, 0.990439,
			    -0.301259, 0.264083, 0.916244,
			    -0.358229, 0.131655, 0.924305,
			    -0.484442, 0.131200, 0.864929,
			    -0.403355, 0.000000, 0.915043,
			    -0.375039, 0.383614, 0.843911,
			    0.237086, 0.606825, 0.758652,
			    0.081142, 0.620240, 0.780204,
			    0.000000, 0.711282, 0.702907,
			    -0.081142, 0.620240, 0.780204,
			    -0.237086, 0.606825, 0.758652,
			    0.081086, 0.783843, 0.615642,
			    -0.081086, 0.783843, 0.615642,
			    0.358229, -0.131655, 0.924305,
			    0.484442, -0.131200, 0.864929,
			    0.220117, -0.132792, 0.966393,
			    0.301259, -0.264083, 0.916244,
			    0.238677, -0.386187, 0.891006,
			    0.375039, -0.383614, 0.843911,
			    0.082242, -0.133071, 0.987688,
			    0.082324, -0.399607, 0.912982,
			    0.156434, -0.519258, 0.840178,
			    0.000000, -0.266405, 0.963861,
			    -0.082324, -0.399607, 0.912982,
			    -0.238677, -0.386187, 0.891006,
			    -0.156434, -0.519258, 0.840178,
			    -0.082242, -0.133071, 0.987688,
			    0.081142, -0.620240, 0.780204,
			    0.237086, -0.606825, 0.758652,
			    -0.081142, -0.620240, 0.780204,
			    0.000000, -0.711282, 0.702907,
			    -0.081086, -0.783843, 0.615642,
			    0.081086, -0.783843, 0.615642,
			    -0.237086, -0.606825, 0.758652,
			    -0.220117, -0.132792, 0.966393,
			    -0.358229, -0.131655, 0.924305,
			    -0.301259, -0.264083, 0.916244,
			    -0.375039, -0.383614, 0.843911,
			    -0.484442, -0.131200, 0.864929,
			    0.403355, 0.000000, -0.915043,
			    0.358229, 0.131655, -0.924305,
			    0.484442, 0.131200, -0.864929,
			    0.220117, 0.132792, -0.966393,
			    0.301259, 0.264083, -0.916244,
			    0.238677, 0.386187, -0.891006,
			    0.375039, 0.383614, -0.843911,
			    0.137952, 0.000000, -0.990439,
			    0.082242, 0.133071, -0.987688,
			    0.082324, 0.399607, -0.912982,
			    0.156434, 0.519258, -0.840178,
			    0.000000, 0.266405, -0.963861,
			    -0.082324, 0.399607, -0.912982,
			    -0.238677, 0.386187, -0.891006,
			    -0.156434, 0.519258, -0.840178,
			    -0.082242, 0.133071, -0.987688,
			    0.081142, 0.620240, -0.780204,
			    0.237086, 0.606825, -0.758652,
			    -0.081142, 0.620240, -0.780204,
			    0.000000, 0.711282, -0.702907,
			    -0.081086, 0.783843, -0.615642,
			    0.081086, 0.783843, -0.615642,
			    -0.237086, 0.606825, -0.758652,
			    -0.137952, 0.000000, -0.990439,
			    -0.220117, 0.132792, -0.966393,
			    -0.358229, 0.131655, -0.924305,
			    -0.301259, 0.264083, -0.916244,
			    -0.375039, 0.383614, -0.843911,
			    -0.403355, 0.000000, -0.915043,
			    -0.484442, 0.131200, -0.864929,
			    0.484442, -0.131200, -0.864929,
			    0.358229, -0.131655, -0.924305,
			    0.301259, -0.264083, -0.916244,
			    0.220117, -0.132792, -0.966393,
			    0.082242, -0.133071, -0.987688,
			    0.375039, -0.383614, -0.843911,
			    0.238677, -0.386187, -0.891006,
			    0.000000, -0.266405, -0.963861,
			    -0.082242, -0.133071, -0.987688,
			    0.082324, -0.399607, -0.912982,
			    -0.082324, -0.399607, -0.912982,
			    -0.156434, -0.519258, -0.840178,
			    -0.238677, -0.386187, -0.891006,
			    0.156434, -0.519258, -0.840178,
			    -0.220117, -0.132792, -0.966393,
			    -0.301259, -0.264083, -0.916244,
			    -0.358229, -0.131655, -0.924305,
			    -0.484442, -0.131200, -0.864929,
			    -0.375039, -0.383614, -0.843911,
			    0.237086, -0.606825, -0.758652,
			    0.081142, -0.620240, -0.780204,
			    0.000000, -0.711282, -0.702907,
			    -0.081142, -0.620240, -0.780204,
			    -0.237086, -0.606825, -0.758652,
			    0.081086, -0.783843, -0.615642,
			    -0.081086, -0.783843, -0.615642,
			    0.864929, 0.484442, 0.131200,
			    0.924305, 0.358229, 0.131655,
			    0.915043, 0.403355, 0.000000,
			    0.916244, 0.301259, 0.264083,
			    0.966393, 0.220117, 0.132792,
			    0.987688, 0.082242, 0.133071,
			    0.990439, 0.137952, 0.000000,
			    0.843911, 0.375039, 0.383614,
			    0.891006, 0.238677, 0.386187,
			    0.963861, 0.000000, 0.266405,
			    0.987688, -0.082242, 0.133071,
			    0.912982, 0.082324, 0.399607,
			    0.912982, -0.082324, 0.399607,
			    0.840178, -0.156434, 0.519258,
			    0.891006, -0.238677, 0.386187,
			    0.840178, 0.156434, 0.519258,
			    0.966393, -0.220117, 0.132792,
			    0.990439, -0.137952, 0.000000,
			    0.916244, -0.301259, 0.264083,
			    0.924305, -0.358229, 0.131655,
			    0.864929, -0.484442, 0.131200,
			    0.915043, -0.403355, 0.000000,
			    0.843911, -0.375039, 0.383614,
			    0.758652, 0.237086, 0.606825,
			    0.780204, 0.081142, 0.620240,
			    0.702907, 0.000000, 0.711282,
			    0.780204, -0.081142, 0.620240,
			    0.758652, -0.237086, 0.606825,
			    0.615642, 0.081086, 0.783843,
			    0.615642, -0.081086, 0.783843,
			    0.924305, 0.358229, -0.131655,
			    0.864929, 0.484442, -0.131200,
			    0.966393, 0.220117, -0.132792,
			    0.916244, 0.301259, -0.264083,
			    0.891006, 0.238677, -0.386187,
			    0.843911, 0.375039, -0.383614,
			    0.987688, 0.082242, -0.133071,
			    0.912982, 0.082324, -0.399607,
			    0.840178, 0.156434, -0.519258,
			    0.963861, 0.000000, -0.266405,
			    0.912982, -0.082324, -0.399607,
			    0.891006, -0.238677, -0.386187,
			    0.840178, -0.156434, -0.519258,
			    0.987688, -0.082242, -0.133071,
			    0.780204, 0.081142, -0.620240,
			    0.758652, 0.237086, -0.606825,
			    0.780204, -0.081142, -0.620240,
			    0.702907, 0.000000, -0.711282,
			    0.615642, -0.081086, -0.783843,
			    0.615642, 0.081086, -0.783843,
			    0.758652, -0.237086, -0.606825,
			    0.966393, -0.220117, -0.132792,
			    0.924305, -0.358229, -0.131655,
			    0.916244, -0.301259, -0.264083,
			    0.843911, -0.375039, -0.383614,
			    0.864929, -0.484442, -0.131200,
			    -0.864929, -0.484442, 0.131200,
			    -0.924305, -0.358229, 0.131655,
			    -0.915043, -0.403355, 0.000000,
			    -0.916244, -0.301259, 0.264083,
			    -0.966393, -0.220117, 0.132792,
			    -0.987688, -0.082242, 0.133071,
			    -0.990439, -0.137952, 0.000000,
			    -0.843911, -0.375039, 0.383614,
			    -0.891006, -0.238677, 0.386187,
			    -0.963861, 0.000000, 0.266405,
			    -0.987688, 0.082242, 0.133071,
			    -0.912982, -0.082324, 0.399607,
			    -0.912982, 0.082324, 0.399607,
			    -0.840178, 0.156434, 0.519258,
			    -0.891006, 0.238677, 0.386187,
			    -0.840178, -0.156434, 0.519258,
			    -0.966393, 0.220117, 0.132792,
			    -0.990439, 0.137952, 0.000000,
			    -0.916244, 0.301259, 0.264083,
			    -0.924305, 0.358229, 0.131655,
			    -0.864929, 0.484442, 0.131200,
			    -0.915043, 0.403355, 0.000000,
			    -0.843911, 0.375039, 0.383614,
			    -0.758652, -0.237086, 0.606825,
			    -0.780204, -0.081142, 0.620240,
			    -0.702907, 0.000000, 0.711282,
			    -0.780204, 0.081142, 0.620240,
			    -0.758652, 0.237086, 0.606825,
			    -0.615642, -0.081086, 0.783843,
			    -0.615642, 0.081086, 0.783843,
			    -0.924305, -0.358229, -0.131655,
			    -0.864929, -0.484442, -0.131200,
			    -0.966393, -0.220117, -0.132792,
			    -0.916244, -0.301259, -0.264083,
			    -0.891006, -0.238677, -0.386187,
			    -0.843911, -0.375039, -0.383614,
			    -0.987688, -0.082242, -0.133071,
			    -0.912982, -0.082324, -0.399607,
			    -0.840178, -0.156434, -0.519258,
			    -0.963861, 0.000000, -0.266405,
			    -0.912982, 0.082324, -0.399607,
			    -0.891006, 0.238677, -0.386187,
			    -0.840178, 0.156434, -0.519258,
			    -0.987688, 0.082242, -0.133071,
			    -0.780204, -0.081142, -0.620240,
			    -0.758652, -0.237086, -0.606825,
			    -0.780204, 0.081142, -0.620240,
			    -0.702907, 0.000000, -0.711282,
			    -0.615642, 0.081086, -0.783843,
			    -0.615642, -0.081086, -0.783843,
			    -0.758652, 0.237086, -0.606825,
			    -0.966393, 0.220117, -0.132792,
			    -0.924305, 0.358229, -0.131655,
			    -0.916244, 0.301259, -0.264083,
			    -0.843911, 0.375039, -0.383614,
			    -0.864929, 0.484442, -0.131200,
			    0.131200, 0.864929, 0.484442,
			    0.131655, 0.924305, 0.358229,
			    0.000000, 0.915043, 0.403355,
			    0.264083, 0.916244, 0.301259,
			    0.132792, 0.966393, 0.220117,
			    0.133071, 0.987688, 0.082242,
			    0.000000, 0.990439, 0.137952,
			    0.383614, 0.843911, 0.375039,
			    0.386187, 0.891006, 0.238677,
			    0.266405, 0.963861, 0.000000,
			    0.133071, 0.987688, -0.082242,
			    0.399607, 0.912982, 0.082324,
			    0.399607, 0.912982, -0.082324,
			    0.519258, 0.840178, -0.156434,
			    0.386187, 0.891006, -0.238677,
			    0.519258, 0.840178, 0.156434,
			    0.132792, 0.966393, -0.220117,
			    0.000000, 0.990439, -0.137952,
			    0.264083, 0.916244, -0.301259,
			    0.131655, 0.924305, -0.358229,
			    0.131200, 0.864929, -0.484442,
			    0.000000, 0.915043, -0.403355,
			    0.383614, 0.843911, -0.375039,
			    0.606825, 0.758652, 0.237086,
			    0.620240, 0.780204, 0.081142,
			    0.711282, 0.702907, 0.000000,
			    0.620240, 0.780204, -0.081142,
			    0.606825, 0.758652, -0.237086,
			    0.783843, 0.615642, 0.081086,
			    0.783843, 0.615642, -0.081086,
			    -0.131655, 0.924305, 0.358229,
			    -0.131200, 0.864929, 0.484442,
			    -0.132792, 0.966393, 0.220117,
			    -0.264083, 0.916244, 0.301259,
			    -0.386187, 0.891006, 0.238677,
			    -0.383614, 0.843911, 0.375039,
			    -0.133071, 0.987688, 0.082242,
			    -0.399607, 0.912982, 0.082324,
			    -0.519258, 0.840178, 0.156434,
			    -0.266405, 0.963861, 0.000000,
			    -0.399607, 0.912982, -0.082324,
			    -0.386187, 0.891006, -0.238677,
			    -0.519258, 0.840178, -0.156434,
			    -0.133071, 0.987688, -0.082242,
			    -0.620240, 0.780204, 0.081142,
			    -0.606825, 0.758652, 0.237086,
			    -0.620240, 0.780204, -0.081142,
			    -0.711282, 0.702907, 0.000000,
			    -0.783843, 0.615642, -0.081086,
			    -0.783843, 0.615642, 0.081086,
			    -0.606825, 0.758652, -0.237086,
			    -0.132792, 0.966393, -0.220117,
			    -0.131655, 0.924305, -0.358229,
			    -0.264083, 0.916244, -0.301259,
			    -0.383614, 0.843911, -0.375039,
			    -0.131200, 0.864929, -0.484442,
			    0.000000, -0.915043, 0.403355,
			    0.131655, -0.924305, 0.358229,
			    0.131200, -0.864929, 0.484442,
			    0.132792, -0.966393, 0.220117,
			    0.264083, -0.916244, 0.301259,
			    0.386187, -0.891006, 0.238677,
			    0.383614, -0.843911, 0.375039,
			    0.000000, -0.990439, 0.137952,
			    0.133071, -0.987688, 0.082242,
			    0.399607, -0.912982, 0.082324,
			    0.519258, -0.840178, 0.156434,
			    0.266405, -0.963861, 0.000000,
			    0.399607, -0.912982, -0.082324,
			    0.386187, -0.891006, -0.238677,
			    0.519258, -0.840178, -0.156434,
			    0.133071, -0.987688, -0.082242,
			    0.620240, -0.780204, 0.081142,
			    0.606825, -0.758652, 0.237086,
			    0.620240, -0.780204, -0.081142,
			    0.711282, -0.702907, 0.000000,
			    0.783843, -0.615642, -0.081086,
			    0.783843, -0.615642, 0.081086,
			    0.606825, -0.758652, -0.237086,
			    0.000000, -0.990439, -0.137952,
			    0.132792, -0.966393, -0.220117,
			    0.131655, -0.924305, -0.358229,
			    0.264083, -0.916244, -0.301259,
			    0.383614, -0.843911, -0.375039,
			    0.000000, -0.915043, -0.403355,
			    0.131200, -0.864929, -0.484442,
			    -0.131200, -0.864929, 0.484442,
			    -0.131655, -0.924305, 0.358229,
			    -0.264083, -0.916244, 0.301259,
			    -0.132792, -0.966393, 0.220117,
			    -0.133071, -0.987688, 0.082242,
			    -0.383614, -0.843911, 0.375039,
			    -0.386187, -0.891006, 0.238677,
			    -0.266405, -0.963861, 0.000000,
			    -0.133071, -0.987688, -0.082242,
			    -0.399607, -0.912982, 0.082324,
			    -0.399607, -0.912982, -0.082324,
			    -0.519258, -0.840178, -0.156434,
			    -0.386187, -0.891006, -0.238677,
			    -0.519258, -0.840178, 0.156434,
			    -0.132792, -0.966393, -0.220117,
			    -0.264083, -0.916244, -0.301259,
			    -0.131655, -0.924305, -0.358229,
			    -0.131200, -0.864929, -0.484442,
			    -0.383614, -0.843911, -0.375039,
			    -0.606825, -0.758652, 0.237086,
			    -0.620240, -0.780204, 0.081142,
			    -0.711282, -0.702907, 0.000000,
			    -0.620240, -0.780204, -0.081142,
			    -0.606825, -0.758652, -0.237086,
			    -0.783843, -0.615642, 0.081086,
			    -0.783843, -0.615642, -0.081086,
			    0.213023, 0.792649, 0.571252,
			    0.296005, 0.702310, 0.647412,
			    0.346153, 0.783452, 0.516122,
			    0.468430, 0.757935, 0.453990,
			    0.371748, 0.601501, 0.707107,
			    0.564254, 0.646578, 0.513375,
			    0.601501, 0.707107, 0.371748,
			    0.513375, 0.564254, 0.646578,
			    0.646578, 0.513375, 0.564254,
			    0.707107, 0.371748, 0.601501,
			    0.757935, 0.453990, 0.468430,
			    0.453990, 0.468430, 0.757935,
			    0.702310, 0.647412, 0.296005,
			    0.783452, 0.516122, 0.346153,
			    0.792649, 0.571252, 0.213023,
			    0.516122, 0.346153, 0.783452,
			    0.571252, 0.213023, 0.792649,
			    0.647412, 0.296005, 0.702310,
			    0.213023, 0.792649, -0.571252,
			    0.346153, 0.783452, -0.516122,
			    0.296005, 0.702310, -0.647412,
			    0.371748, 0.601501, -0.707107,
			    0.468430, 0.757935, -0.453990,
			    0.513375, 0.564254, -0.646578,
			    0.453990, 0.468430, -0.757935,
			    0.564254, 0.646578, -0.513375,
			    0.646578, 0.513375, -0.564254,
			    0.757935, 0.453990, -0.468430,
			    0.707107, 0.371748, -0.601501,
			    0.601501, 0.707107, -0.371748,
			    0.516122, 0.346153, -0.783452,
			    0.647412, 0.296005, -0.702310,
			    0.571252, 0.213023, -0.792649,
			    0.702310, 0.647412, -0.296005,
			    0.792649, 0.571252, -0.213023,
			    0.783452, 0.516122, -0.346153,
			    0.571252, -0.213023, 0.792649,
			    0.516122, -0.346153, 0.783452,
			    0.647412, -0.296005, 0.702310,
			    0.707107, -0.371748, 0.601501,
			    0.453990, -0.468430, 0.757935,
			    0.646578, -0.513375, 0.564254,
			    0.757935, -0.453990, 0.468430,
			    0.513375, -0.564254, 0.646578,
			    0.564254, -0.646578, 0.513375,
			    0.468430, -0.757935, 0.453990,
			    0.601501, -0.707107, 0.371748,
			    0.371748, -0.601501, 0.707107,
			    0.783452, -0.516122, 0.346153,
			    0.702310, -0.647412, 0.296005,
			    0.792649, -0.571252, 0.213023,
			    0.296005, -0.702310, 0.647412,
			    0.213023, -0.792649, 0.571252,
			    0.346153, -0.783452, 0.516122,
			    0.571252, -0.213023, -0.792649,
			    0.647412, -0.296005, -0.702310,
			    0.516122, -0.346153, -0.783452,
			    0.453990, -0.468430, -0.757935,
			    0.707107, -0.371748, -0.601501,
			    0.513375, -0.564254, -0.646578,
			    0.371748, -0.601501, -0.707107,
			    0.646578, -0.513375, -0.564254,
			    0.564254, -0.646578, -0.513375,
			    0.601501, -0.707107, -0.371748,
			    0.468430, -0.757935, -0.453990,
			    0.757935, -0.453990, -0.468430,
			    0.296005, -0.702310, -0.647412,
			    0.346153, -0.783452, -0.516122,
			    0.213023, -0.792649, -0.571252,
			    0.783452, -0.516122, -0.346153,
			    0.792649, -0.571252, -0.213023,
			    0.702310, -0.647412, -0.296005,
			    -0.571252, 0.213023, 0.792649,
			    -0.516122, 0.346153, 0.783452,
			    -0.647412, 0.296005, 0.702310,
			    -0.707107, 0.371748, 0.601501,
			    -0.453990, 0.468430, 0.757935,
			    -0.646578, 0.513375, 0.564254,
			    -0.757935, 0.453990, 0.468430,
			    -0.513375, 0.564254, 0.646578,
			    -0.564254, 0.646578, 0.513375,
			    -0.468430, 0.757935, 0.453990,
			    -0.601501, 0.707107, 0.371748,
			    -0.371748, 0.601501, 0.707107,
			    -0.783452, 0.516122, 0.346153,
			    -0.702310, 0.647412, 0.296005,
			    -0.792649, 0.571252, 0.213023,
			    -0.296005, 0.702310, 0.647412,
			    -0.213023, 0.792649, 0.571252,
			    -0.346153, 0.783452, 0.516122,
			    -0.571252, 0.213023, -0.792649,
			    -0.647412, 0.296005, -0.702310,
			    -0.516122, 0.346153, -0.783452,
			    -0.453990, 0.468430, -0.757935,
			    -0.707107, 0.371748, -0.601501,
			    -0.513375, 0.564254, -0.646578,
			    -0.371748, 0.601501, -0.707107,
			    -0.646578, 0.513375, -0.564254,
			    -0.564254, 0.646578, -0.513375,
			    -0.601501, 0.707107, -0.371748,
			    -0.468430, 0.757935, -0.453990,
			    -0.757935, 0.453990, -0.468430,
			    -0.296005, 0.702310, -0.647412,
			    -0.346153, 0.783452, -0.516122,
			    -0.213023, 0.792649, -0.571252,
			    -0.783452, 0.516122, -0.346153,
			    -0.792649, 0.571252, -0.213023,
			    -0.702310, 0.647412, -0.296005,
			    -0.571252, -0.213023, 0.792649,
			    -0.647412, -0.296005, 0.702310,
			    -0.516122, -0.346153, 0.783452,
			    -0.453990, -0.468430, 0.757935,
			    -0.707107, -0.371748, 0.601501,
			    -0.513375, -0.564254, 0.646578,
			    -0.371748, -0.601501, 0.707107,
			    -0.646578, -0.513375, 0.564254,
			    -0.564254, -0.646578, 0.513375,
			    -0.601501, -0.707107, 0.371748,
			    -0.468430, -0.757935, 0.453990,
			    -0.757935, -0.453990, 0.468430,
			    -0.296005, -0.702310, 0.647412,
			    -0.346153, -0.783452, 0.516122,
			    -0.213023, -0.792649, 0.571252,
			    -0.783452, -0.516122, 0.346153,
			    -0.792649, -0.571252, 0.213023,
			    -0.702310, -0.647412, 0.296005,
			    -0.571252, -0.213023, -0.792649,
			    -0.516122, -0.346153, -0.783452,
			    -0.647412, -0.296005, -0.702310,
			    -0.707107, -0.371748, -0.601501,
			    -0.453990, -0.468430, -0.757935,
			    -0.646578, -0.513375, -0.564254,
			    -0.757935, -0.453990, -0.468430,
			    -0.513375, -0.564254, -0.646578,
			    -0.564254, -0.646578, -0.513375,
			    -0.468430, -0.757935, -0.453990,
			    -0.601501, -0.707107, -0.371748,
			    -0.371748, -0.601501, -0.707107,
			    -0.783452, -0.516122, -0.346153,
			    -0.702310, -0.647412, -0.296005,
			    -0.792649, -0.571252, -0.213023,
			    -0.296005, -0.702310, -0.647412,
			    -0.213023, -0.792649, -0.571252,
			    -0.346153, -0.783452, -0.516122};

    Points << v;
    Points << Points.t();
    Points.Release();
    return Points;
  }


  //4th order tessellation
  ReturnMatrix tessellation_points4() const {
    Matrix Points(2562,3);
    Real v[]={0.850651, 0.525731, 0.000000,
-0.850651, 0.525731, 0.000000,
-0.850651, -0.525731, 0.000000,
0.850651, -0.525731, 0.000000,
0.525731, 0.000000, 0.850651,
0.525731, 0.000000, -0.850651,
-0.525731, 0.000000, -0.850651,
-0.525731, 0.000000, 0.850651,
0.000000, 0.850651, 0.525731,
0.000000, -0.850651, 0.525731,
0.000000, -0.850651, -0.525731,
0.000000, 0.850651, -0.525731,
0.309017, 0.500000, 0.809017,
-0.309017, 0.500000, 0.809017,
0.000000, 0.000000, 1.000000,
-0.309017, -0.500000, 0.809017,
0.309017, -0.500000, 0.809017,
0.000000, 0.000000, -1.000000,
-0.309017, 0.500000, -0.809017,
0.309017, 0.500000, -0.809017,
0.309017, -0.500000, -0.809017,
-0.309017, -0.500000, -0.809017,
0.809017, 0.309017, 0.500000,
0.809017, -0.309017, 0.500000,
1.000000, 0.000000, 0.000000,
0.809017, -0.309017, -0.500000,
0.809017, 0.309017, -0.500000,
-0.809017, -0.309017, 0.500000,
-0.809017, 0.309017, 0.500000,
-1.000000, 0.000000, 0.000000,
-0.809017, 0.309017, -0.500000,
-0.809017, -0.309017, -0.500000,
0.500000, 0.809017, 0.309017,
0.500000, 0.809017, -0.309017,
0.000000, 1.000000, 0.000000,
-0.500000, 0.809017, -0.309017,
-0.500000, 0.809017, 0.309017,
0.000000, -1.000000, 0.000000,
0.500000, -0.809017, -0.309017,
0.500000, -0.809017, 0.309017,
-0.500000, -0.809017, 0.309017,
-0.500000, -0.809017, -0.309017,
0.433889, 0.259892, 0.862669,
0.162460, 0.262866, 0.951057,
0.273267, 0.000000, 0.961938,
0.000000, 0.525731, 0.850651,
-0.162460, 0.262866, 0.951057,
-0.433889, 0.259892, 0.862669,
-0.273267, 0.000000, 0.961938,
0.160622, 0.702046, 0.693780,
-0.160622, 0.702046, 0.693780,
0.162460, -0.262866, 0.951057,
0.433889, -0.259892, 0.862669,
-0.162460, -0.262866, 0.951057,
0.000000, -0.525731, 0.850651,
-0.160622, -0.702046, 0.693780,
0.160622, -0.702046, 0.693780,
-0.433889, -0.259892, 0.862669,
0.273267, 0.000000, -0.961938,
0.162460, 0.262866, -0.951057,
0.433889, 0.259892, -0.862669,
-0.162460, 0.262866, -0.951057,
0.000000, 0.525731, -0.850651,
-0.160622, 0.702046, -0.693780,
0.160622, 0.702046, -0.693780,
-0.273267, 0.000000, -0.961938,
-0.433889, 0.259892, -0.862669,
0.433889, -0.259892, -0.862669,
0.162460, -0.262866, -0.951057,
0.000000, -0.525731, -0.850651,
-0.162460, -0.262866, -0.951057,
-0.433889, -0.259892, -0.862669,
0.160622, -0.702046, -0.693780,
-0.160622, -0.702046, -0.693780,
0.862669, 0.433889, 0.259892,
0.951057, 0.162460, 0.262866,
0.961938, 0.273267, 0.000000,
0.850651, 0.000000, 0.525731,
0.951057, -0.162460, 0.262866,
0.862669, -0.433889, 0.259892,
0.961938, -0.273267, 0.000000,
0.693780, 0.160622, 0.702046,
0.693780, -0.160622, 0.702046,
0.951057, 0.162460, -0.262866,
0.862669, 0.433889, -0.259892,
0.951057, -0.162460, -0.262866,
0.850651, 0.000000, -0.525731,
0.693780, -0.160622, -0.702046,
0.693780, 0.160622, -0.702046,
0.862669, -0.433889, -0.259892,
-0.862669, -0.433889, 0.259892,
-0.951057, -0.162460, 0.262866,
-0.961938, -0.273267, 0.000000,
-0.850651, 0.000000, 0.525731,
-0.951057, 0.162460, 0.262866,
-0.862669, 0.433889, 0.259892,
-0.961938, 0.273267, 0.000000,
-0.693780, -0.160622, 0.702046,
-0.693780, 0.160622, 0.702046,
-0.951057, -0.162460, -0.262866,
-0.862669, -0.433889, -0.259892,
-0.951057, 0.162460, -0.262866,
-0.850651, 0.000000, -0.525731,
-0.693780, 0.160622, -0.702046,
-0.693780, -0.160622, -0.702046,
-0.862669, 0.433889, -0.259892,
0.259892, 0.862669, 0.433889,
0.262866, 0.951057, 0.162460,
0.000000, 0.961938, 0.273267,
0.525731, 0.850651, 0.000000,
0.262866, 0.951057, -0.162460,
0.259892, 0.862669, -0.433889,
0.000000, 0.961938, -0.273267,
0.702046, 0.693780, 0.160622,
0.702046, 0.693780, -0.160622,
-0.262866, 0.951057, 0.162460,
-0.259892, 0.862669, 0.433889,
-0.262866, 0.951057, -0.162460,
-0.525731, 0.850651, 0.000000,
-0.702046, 0.693780, -0.160622,
-0.702046, 0.693780, 0.160622,
-0.259892, 0.862669, -0.433889,
0.000000, -0.961938, 0.273267,
0.262866, -0.951057, 0.162460,
0.259892, -0.862669, 0.433889,
0.262866, -0.951057, -0.162460,
0.525731, -0.850651, 0.000000,
0.702046, -0.693780, -0.160622,
0.702046, -0.693780, 0.160622,
0.000000, -0.961938, -0.273267,
0.259892, -0.862669, -0.433889,
-0.259892, -0.862669, 0.433889,
-0.262866, -0.951057, 0.162460,
-0.525731, -0.850651, 0.000000,
-0.262866, -0.951057, -0.162460,
-0.259892, -0.862669, -0.433889,
-0.702046, -0.693780, 0.160622,
-0.702046, -0.693780, -0.160622,
0.425325, 0.688191, 0.587785,
0.587785, 0.425325, 0.688191,
0.688191, 0.587785, 0.425325,
0.425325, 0.688191, -0.587785,
0.688191, 0.587785, -0.425325,
0.587785, 0.425325, -0.688191,
0.587785, -0.425325, 0.688191,
0.425325, -0.688191, 0.587785,
0.688191, -0.587785, 0.425325,
0.587785, -0.425325, -0.688191,
0.688191, -0.587785, -0.425325,
0.425325, -0.688191, -0.587785,
-0.587785, 0.425325, 0.688191,
-0.425325, 0.688191, 0.587785,
-0.688191, 0.587785, 0.425325,
-0.587785, 0.425325, -0.688191,
-0.688191, 0.587785, -0.425325,
-0.425325, 0.688191, -0.587785,
-0.587785, -0.425325, 0.688191,
-0.688191, -0.587785, 0.425325,
-0.425325, -0.688191, 0.587785,
-0.587785, -0.425325, -0.688191,
-0.425325, -0.688191, -0.587785,
-0.688191, -0.587785, -0.425325,
0.484442, 0.131200, 0.864929,
0.358229, 0.131655, 0.924305,
0.403355, 0.000000, 0.915043,
0.301259, 0.264083, 0.916244,
0.220117, 0.132792, 0.966393,
0.082242, 0.133071, 0.987688,
0.137952, 0.000000, 0.990439,
0.375039, 0.383614, 0.843911,
0.238677, 0.386187, 0.891006,
0.000000, 0.266405, 0.963861,
-0.082242, 0.133071, 0.987688,
0.082324, 0.399607, 0.912982,
-0.082324, 0.399607, 0.912982,
-0.156434, 0.519258, 0.840178,
-0.238677, 0.386187, 0.891006,
0.156434, 0.519258, 0.840178,
-0.220117, 0.132792, 0.966393,
-0.137952, 0.000000, 0.990439,
-0.301259, 0.264083, 0.916244,
-0.358229, 0.131655, 0.924305,
-0.484442, 0.131200, 0.864929,
-0.403355, 0.000000, 0.915043,
-0.375039, 0.383614, 0.843911,
0.237086, 0.606825, 0.758652,
0.081142, 0.620240, 0.780204,
0.000000, 0.711282, 0.702907,
-0.081142, 0.620240, 0.780204,
-0.237086, 0.606825, 0.758652,
0.081086, 0.783843, 0.615642,
-0.081086, 0.783843, 0.615642,
0.358229, -0.131655, 0.924305,
0.484442, -0.131200, 0.864929,
0.220117, -0.132792, 0.966393,
0.301259, -0.264083, 0.916244,
0.238677, -0.386187, 0.891006,
0.375039, -0.383614, 0.843911,
0.082242, -0.133071, 0.987688,
0.082324, -0.399607, 0.912982,
0.156434, -0.519258, 0.840178,
0.000000, -0.266405, 0.963861,
-0.082324, -0.399607, 0.912982,
-0.238677, -0.386187, 0.891006,
-0.156434, -0.519258, 0.840178,
-0.082242, -0.133071, 0.987688,
0.081142, -0.620240, 0.780204,
0.237086, -0.606825, 0.758652,
-0.081142, -0.620240, 0.780204,
0.000000, -0.711282, 0.702907,
-0.081086, -0.783843, 0.615642,
0.081086, -0.783843, 0.615642,
-0.237086, -0.606825, 0.758652,
-0.220117, -0.132792, 0.966393,
-0.358229, -0.131655, 0.924305,
-0.301259, -0.264083, 0.916244,
-0.375039, -0.383614, 0.843911,
-0.484442, -0.131200, 0.864929,
0.403355, 0.000000, -0.915043,
0.358229, 0.131655, -0.924305,
0.484442, 0.131200, -0.864929,
0.220117, 0.132792, -0.966393,
0.301259, 0.264083, -0.916244,
0.238677, 0.386187, -0.891006,
0.375039, 0.383614, -0.843911,
0.137952, 0.000000, -0.990439,
0.082242, 0.133071, -0.987688,
0.082324, 0.399607, -0.912982,
0.156434, 0.519258, -0.840178,
0.000000, 0.266405, -0.963861,
-0.082324, 0.399607, -0.912982,
-0.238677, 0.386187, -0.891006,
-0.156434, 0.519258, -0.840178,
-0.082242, 0.133071, -0.987688,
0.081142, 0.620240, -0.780204,
0.237086, 0.606825, -0.758652,
-0.081142, 0.620240, -0.780204,
0.000000, 0.711282, -0.702907,
-0.081086, 0.783843, -0.615642,
0.081086, 0.783843, -0.615642,
-0.237086, 0.606825, -0.758652,
-0.137952, 0.000000, -0.990439,
-0.220117, 0.132792, -0.966393,
-0.358229, 0.131655, -0.924305,
-0.301259, 0.264083, -0.916244,
-0.375039, 0.383614, -0.843911,
-0.403355, 0.000000, -0.915043,
-0.484442, 0.131200, -0.864929,
0.484442, -0.131200, -0.864929,
0.358229, -0.131655, -0.924305,
0.301259, -0.264083, -0.916244,
0.220117, -0.132792, -0.966393,
0.082242, -0.133071, -0.987688,
0.375039, -0.383614, -0.843911,
0.238677, -0.386187, -0.891006,
0.000000, -0.266405, -0.963861,
-0.082242, -0.133071, -0.987688,
0.082324, -0.399607, -0.912982,
-0.082324, -0.399607, -0.912982,
-0.156434, -0.519258, -0.840178,
-0.238677, -0.386187, -0.891006,
0.156434, -0.519258, -0.840178,
-0.220117, -0.132792, -0.966393,
-0.301259, -0.264083, -0.916244,
-0.358229, -0.131655, -0.924305,
-0.484442, -0.131200, -0.864929,
-0.375039, -0.383614, -0.843911,
0.237086, -0.606825, -0.758652,
0.081142, -0.620240, -0.780204,
0.000000, -0.711282, -0.702907,
-0.081142, -0.620240, -0.780204,
-0.237086, -0.606825, -0.758652,
0.081086, -0.783843, -0.615642,
-0.081086, -0.783843, -0.615642,
0.864929, 0.484442, 0.131200,
0.924305, 0.358229, 0.131655,
0.915043, 0.403355, 0.000000,
0.916244, 0.301259, 0.264083,
0.966393, 0.220117, 0.132792,
0.987688, 0.082242, 0.133071,
0.990439, 0.137952, 0.000000,
0.843911, 0.375039, 0.383614,
0.891006, 0.238677, 0.386187,
0.963861, 0.000000, 0.266405,
0.987688, -0.082242, 0.133071,
0.912982, 0.082324, 0.399607,
0.912982, -0.082324, 0.399607,
0.840178, -0.156434, 0.519258,
0.891006, -0.238677, 0.386187,
0.840178, 0.156434, 0.519258,
0.966393, -0.220117, 0.132792,
0.990439, -0.137952, 0.000000,
0.916244, -0.301259, 0.264083,
0.924305, -0.358229, 0.131655,
0.864929, -0.484442, 0.131200,
0.915043, -0.403355, 0.000000,
0.843911, -0.375039, 0.383614,
0.758652, 0.237086, 0.606825,
0.780204, 0.081142, 0.620240,
0.702907, 0.000000, 0.711282,
0.780204, -0.081142, 0.620240,
0.758652, -0.237086, 0.606825,
0.615642, 0.081086, 0.783843,
0.615642, -0.081086, 0.783843,
0.924305, 0.358229, -0.131655,
0.864929, 0.484442, -0.131200,
0.966393, 0.220117, -0.132792,
0.916244, 0.301259, -0.264083,
0.891006, 0.238677, -0.386187,
0.843911, 0.375039, -0.383614,
0.987688, 0.082242, -0.133071,
0.912982, 0.082324, -0.399607,
0.840178, 0.156434, -0.519258,
0.963861, 0.000000, -0.266405,
0.912982, -0.082324, -0.399607,
0.891006, -0.238677, -0.386187,
0.840178, -0.156434, -0.519258,
0.987688, -0.082242, -0.133071,
0.780204, 0.081142, -0.620240,
0.758652, 0.237086, -0.606825,
0.780204, -0.081142, -0.620240,
0.702907, 0.000000, -0.711282,
0.615642, -0.081086, -0.783843,
0.615642, 0.081086, -0.783843,
0.758652, -0.237086, -0.606825,
0.966393, -0.220117, -0.132792,
0.924305, -0.358229, -0.131655,
0.916244, -0.301259, -0.264083,
0.843911, -0.375039, -0.383614,
0.864929, -0.484442, -0.131200,
-0.864929, -0.484442, 0.131200,
-0.924305, -0.358229, 0.131655,
-0.915043, -0.403355, 0.000000,
-0.916244, -0.301259, 0.264083,
-0.966393, -0.220117, 0.132792,
-0.987688, -0.082242, 0.133071,
-0.990439, -0.137952, 0.000000,
-0.843911, -0.375039, 0.383614,
-0.891006, -0.238677, 0.386187,
-0.963861, 0.000000, 0.266405,
-0.987688, 0.082242, 0.133071,
-0.912982, -0.082324, 0.399607,
-0.912982, 0.082324, 0.399607,
-0.840178, 0.156434, 0.519258,
-0.891006, 0.238677, 0.386187,
-0.840178, -0.156434, 0.519258,
-0.966393, 0.220117, 0.132792,
-0.990439, 0.137952, 0.000000,
-0.916244, 0.301259, 0.264083,
-0.924305, 0.358229, 0.131655,
-0.864929, 0.484442, 0.131200,
-0.915043, 0.403355, 0.000000,
-0.843911, 0.375039, 0.383614,
-0.758652, -0.237086, 0.606825,
-0.780204, -0.081142, 0.620240,
-0.702907, 0.000000, 0.711282,
-0.780204, 0.081142, 0.620240,
-0.758652, 0.237086, 0.606825,
-0.615642, -0.081086, 0.783843,
-0.615642, 0.081086, 0.783843,
-0.924305, -0.358229, -0.131655,
-0.864929, -0.484442, -0.131200,
-0.966393, -0.220117, -0.132792,
-0.916244, -0.301259, -0.264083,
-0.891006, -0.238677, -0.386187,
-0.843911, -0.375039, -0.383614,
-0.987688, -0.082242, -0.133071,
-0.912982, -0.082324, -0.399607,
-0.840178, -0.156434, -0.519258,
-0.963861, 0.000000, -0.266405,
-0.912982, 0.082324, -0.399607,
-0.891006, 0.238677, -0.386187,
-0.840178, 0.156434, -0.519258,
-0.987688, 0.082242, -0.133071,
-0.780204, -0.081142, -0.620240,
-0.758652, -0.237086, -0.606825,
-0.780204, 0.081142, -0.620240,
-0.702907, 0.000000, -0.711282,
-0.615642, 0.081086, -0.783843,
-0.615642, -0.081086, -0.783843,
-0.758652, 0.237086, -0.606825,
-0.966393, 0.220117, -0.132792,
-0.924305, 0.358229, -0.131655,
-0.916244, 0.301259, -0.264083,
-0.843911, 0.375039, -0.383614,
-0.864929, 0.484442, -0.131200,
0.131200, 0.864929, 0.484442,
0.131655, 0.924305, 0.358229,
0.000000, 0.915043, 0.403355,
0.264083, 0.916244, 0.301259,
0.132792, 0.966393, 0.220117,
0.133071, 0.987688, 0.082242,
0.000000, 0.990439, 0.137952,
0.383614, 0.843911, 0.375039,
0.386187, 0.891006, 0.238677,
0.266405, 0.963861, 0.000000,
0.133071, 0.987688, -0.082242,
0.399607, 0.912982, 0.082324,
0.399607, 0.912982, -0.082324,
0.519258, 0.840178, -0.156434,
0.386187, 0.891006, -0.238677,
0.519258, 0.840178, 0.156434,
0.132792, 0.966393, -0.220117,
0.000000, 0.990439, -0.137952,
0.264083, 0.916244, -0.301259,
0.131655, 0.924305, -0.358229,
0.131200, 0.864929, -0.484442,
0.000000, 0.915043, -0.403355,
0.383614, 0.843911, -0.375039,
0.606825, 0.758652, 0.237086,
0.620240, 0.780204, 0.081142,
0.711282, 0.702907, 0.000000,
0.620240, 0.780204, -0.081142,
0.606825, 0.758652, -0.237086,
0.783843, 0.615642, 0.081086,
0.783843, 0.615642, -0.081086,
-0.131655, 0.924305, 0.358229,
-0.131200, 0.864929, 0.484442,
-0.132792, 0.966393, 0.220117,
-0.264083, 0.916244, 0.301259,
-0.386187, 0.891006, 0.238677,
-0.383614, 0.843911, 0.375039,
-0.133071, 0.987688, 0.082242,
-0.399607, 0.912982, 0.082324,
-0.519258, 0.840178, 0.156434,
-0.266405, 0.963861, 0.000000,
-0.399607, 0.912982, -0.082324,
-0.386187, 0.891006, -0.238677,
-0.519258, 0.840178, -0.156434,
-0.133071, 0.987688, -0.082242,
-0.620240, 0.780204, 0.081142,
-0.606825, 0.758652, 0.237086,
-0.620240, 0.780204, -0.081142,
-0.711282, 0.702907, 0.000000,
-0.783843, 0.615642, -0.081086,
-0.783843, 0.615642, 0.081086,
-0.606825, 0.758652, -0.237086,
-0.132792, 0.966393, -0.220117,
-0.131655, 0.924305, -0.358229,
-0.264083, 0.916244, -0.301259,
-0.383614, 0.843911, -0.375039,
-0.131200, 0.864929, -0.484442,
0.000000, -0.915043, 0.403355,
0.131655, -0.924305, 0.358229,
0.131200, -0.864929, 0.484442,
0.132792, -0.966393, 0.220117,
0.264083, -0.916244, 0.301259,
0.386187, -0.891006, 0.238677,
0.383614, -0.843911, 0.375039,
0.000000, -0.990439, 0.137952,
0.133071, -0.987688, 0.082242,
0.399607, -0.912982, 0.082324,
0.519258, -0.840178, 0.156434,
0.266405, -0.963861, 0.000000,
0.399607, -0.912982, -0.082324,
0.386187, -0.891006, -0.238677,
0.519258, -0.840178, -0.156434,
0.133071, -0.987688, -0.082242,
0.620240, -0.780204, 0.081142,
0.606825, -0.758652, 0.237086,
0.620240, -0.780204, -0.081142,
0.711282, -0.702907, 0.000000,
0.783843, -0.615642, -0.081086,
0.783843, -0.615642, 0.081086,
0.606825, -0.758652, -0.237086,
0.000000, -0.990439, -0.137952,
0.132792, -0.966393, -0.220117,
0.131655, -0.924305, -0.358229,
0.264083, -0.916244, -0.301259,
0.383614, -0.843911, -0.375039,
0.000000, -0.915043, -0.403355,
0.131200, -0.864929, -0.484442,
-0.131200, -0.864929, 0.484442,
-0.131655, -0.924305, 0.358229,
-0.264083, -0.916244, 0.301259,
-0.132792, -0.966393, 0.220117,
-0.133071, -0.987688, 0.082242,
-0.383614, -0.843911, 0.375039,
-0.386187, -0.891006, 0.238677,
-0.266405, -0.963861, 0.000000,
-0.133071, -0.987688, -0.082242,
-0.399607, -0.912982, 0.082324,
-0.399607, -0.912982, -0.082324,
-0.519258, -0.840178, -0.156434,
-0.386187, -0.891006, -0.238677,
-0.519258, -0.840178, 0.156434,
-0.132792, -0.966393, -0.220117,
-0.264083, -0.916244, -0.301259,
-0.131655, -0.924305, -0.358229,
-0.131200, -0.864929, -0.484442,
-0.383614, -0.843911, -0.375039,
-0.606825, -0.758652, 0.237086,
-0.620240, -0.780204, 0.081142,
-0.711282, -0.702907, 0.000000,
-0.620240, -0.780204, -0.081142,
-0.606825, -0.758652, -0.237086,
-0.783843, -0.615642, 0.081086,
-0.783843, -0.615642, -0.081086,
0.213023, 0.792649, 0.571252,
0.296005, 0.702310, 0.647412,
0.346153, 0.783452, 0.516122,
0.468430, 0.757935, 0.453990,
0.371748, 0.601501, 0.707107,
0.564254, 0.646578, 0.513375,
0.601501, 0.707107, 0.371748,
0.513375, 0.564254, 0.646578,
0.646578, 0.513375, 0.564254,
0.707107, 0.371748, 0.601501,
0.757935, 0.453990, 0.468430,
0.453990, 0.468430, 0.757935,
0.702310, 0.647412, 0.296005,
0.783452, 0.516122, 0.346153,
0.792649, 0.571252, 0.213023,
0.516122, 0.346153, 0.783452,
0.571252, 0.213023, 0.792649,
0.647412, 0.296005, 0.702310,
0.213023, 0.792649, -0.571252,
0.346153, 0.783452, -0.516122,
0.296005, 0.702310, -0.647412,
0.371748, 0.601501, -0.707107,
0.468430, 0.757935, -0.453990,
0.513375, 0.564254, -0.646578,
0.453990, 0.468430, -0.757935,
0.564254, 0.646578, -0.513375,
0.646578, 0.513375, -0.564254,
0.757935, 0.453990, -0.468430,
0.707107, 0.371748, -0.601501,
0.601501, 0.707107, -0.371748,
0.516122, 0.346153, -0.783452,
0.647412, 0.296005, -0.702310,
0.571252, 0.213023, -0.792649,
0.702310, 0.647412, -0.296005,
0.792649, 0.571252, -0.213023,
0.783452, 0.516122, -0.346153,
0.571252, -0.213023, 0.792649,
0.516122, -0.346153, 0.783452,
0.647412, -0.296005, 0.702310,
0.707107, -0.371748, 0.601501,
0.453990, -0.468430, 0.757935,
0.646578, -0.513375, 0.564254,
0.757935, -0.453990, 0.468430,
0.513375, -0.564254, 0.646578,
0.564254, -0.646578, 0.513375,
0.468430, -0.757935, 0.453990,
0.601501, -0.707107, 0.371748,
0.371748, -0.601501, 0.707107,
0.783452, -0.516122, 0.346153,
0.702310, -0.647412, 0.296005,
0.792649, -0.571252, 0.213023,
0.296005, -0.702310, 0.647412,
0.213023, -0.792649, 0.571252,
0.346153, -0.783452, 0.516122,
0.571252, -0.213023, -0.792649,
0.647412, -0.296005, -0.702310,
0.516122, -0.346153, -0.783452,
0.453990, -0.468430, -0.757935,
0.707107, -0.371748, -0.601501,
0.513375, -0.564254, -0.646578,
0.371748, -0.601501, -0.707107,
0.646578, -0.513375, -0.564254,
0.564254, -0.646578, -0.513375,
0.601501, -0.707107, -0.371748,
0.468430, -0.757935, -0.453990,
0.757935, -0.453990, -0.468430,
0.296005, -0.702310, -0.647412,
0.346153, -0.783452, -0.516122,
0.213023, -0.792649, -0.571252,
0.783452, -0.516122, -0.346153,
0.792649, -0.571252, -0.213023,
0.702310, -0.647412, -0.296005,
-0.571252, 0.213023, 0.792649,
-0.516122, 0.346153, 0.783452,
-0.647412, 0.296005, 0.702310,
-0.707107, 0.371748, 0.601501,
-0.453990, 0.468430, 0.757935,
-0.646578, 0.513375, 0.564254,
-0.757935, 0.453990, 0.468430,
-0.513375, 0.564254, 0.646578,
-0.564254, 0.646578, 0.513375,
-0.468430, 0.757935, 0.453990,
-0.601501, 0.707107, 0.371748,
-0.371748, 0.601501, 0.707107,
-0.783452, 0.516122, 0.346153,
-0.702310, 0.647412, 0.296005,
-0.792649, 0.571252, 0.213023,
-0.296005, 0.702310, 0.647412,
-0.213023, 0.792649, 0.571252,
-0.346153, 0.783452, 0.516122,
-0.571252, 0.213023, -0.792649,
-0.647412, 0.296005, -0.702310,
-0.516122, 0.346153, -0.783452,
-0.453990, 0.468430, -0.757935,
-0.707107, 0.371748, -0.601501,
-0.513375, 0.564254, -0.646578,
-0.371748, 0.601501, -0.707107,
-0.646578, 0.513375, -0.564254,
-0.564254, 0.646578, -0.513375,
-0.601501, 0.707107, -0.371748,
-0.468430, 0.757935, -0.453990,
-0.757935, 0.453990, -0.468430,
-0.296005, 0.702310, -0.647412,
-0.346153, 0.783452, -0.516122,
-0.213023, 0.792649, -0.571252,
-0.783452, 0.516122, -0.346153,
-0.792649, 0.571252, -0.213023,
-0.702310, 0.647412, -0.296005,
-0.571252, -0.213023, 0.792649,
-0.647412, -0.296005, 0.702310,
-0.516122, -0.346153, 0.783452,
-0.453990, -0.468430, 0.757935,
-0.707107, -0.371748, 0.601501,
-0.513375, -0.564254, 0.646578,
-0.371748, -0.601501, 0.707107,
-0.646578, -0.513375, 0.564254,
-0.564254, -0.646578, 0.513375,
-0.601501, -0.707107, 0.371748,
-0.468430, -0.757935, 0.453990,
-0.757935, -0.453990, 0.468430,
-0.296005, -0.702310, 0.647412,
-0.346153, -0.783452, 0.516122,
-0.213023, -0.792649, 0.571252,
-0.783452, -0.516122, 0.346153,
-0.792649, -0.571252, 0.213023,
-0.702310, -0.647412, 0.296005,
-0.571252, -0.213023, -0.792649,
-0.516122, -0.346153, -0.783452,
-0.647412, -0.296005, -0.702310,
-0.707107, -0.371748, -0.601501,
-0.453990, -0.468430, -0.757935,
-0.646578, -0.513375, -0.564254,
-0.757935, -0.453990, -0.468430,
-0.513375, -0.564254, -0.646578,
-0.564254, -0.646578, -0.513375,
-0.468430, -0.757935, -0.453990,
-0.601501, -0.707107, -0.371748,
-0.371748, -0.601501, -0.707107,
-0.783452, -0.516122, -0.346153,
-0.702310, -0.647412, -0.296005,
-0.792649, -0.571252, -0.213023,
-0.296005, -0.702310, -0.647412,
-0.213023, -0.792649, -0.571252,
-0.346153, -0.783452, -0.516122,
0.506298, 0.065758, 0.859848,
0.445365, 0.065817, 0.892927,
0.465658, 0.000000, 0.884965,
0.422364, 0.131749, 0.896801,
0.381722, 0.065988, 0.921919,
0.316778, 0.066042, 0.946198,
0.339123, 0.000000, 0.940742,
0.460267, 0.196015, 0.865871,
0.397351, 0.196412, 0.896401,
0.289929, 0.132570, 0.947822,
0.247326, 0.066567, 0.966643,
0.330606, 0.198387, 0.922682,
0.261550, 0.199094, 0.944433,
0.232455, 0.264151, 0.936050,
0.191780, 0.198337, 0.961188,
0.368518, 0.262661, 0.891742,
0.179596, 0.066604, 0.981483,
0.206103, 0.000000, 0.978530,
0.151549, 0.133256, 0.979426,
0.110385, 0.066709, 0.991648,
0.041248, 0.066741, 0.996917,
0.069142, 0.000000, 0.997607,
0.122729, 0.198580, 0.972370,
0.405434, 0.322525, 0.855337,
0.339209, 0.324864, 0.882837,
0.307659, 0.385906, 0.869725,
0.270627, 0.325929, 0.905832,
0.201189, 0.325530, 0.923879,
0.342848, 0.442867, 0.828447,
0.274694, 0.444464, 0.852640,
0.000000, 0.133523, 0.991046,
-0.041248, 0.066741, 0.996917,
0.041251, 0.200368, 0.978852,
-0.041251, 0.200368, 0.978852,
-0.081501, 0.265518, 0.960655,
-0.122729, 0.198580, 0.972370,
0.081501, 0.265518, 0.960655,
-0.041302, 0.334140, 0.941618,
-0.122800, 0.332342, 0.935130,
0.041302, 0.334140, 0.941618,
0.000000, 0.400968, 0.916092,
0.041299, 0.464213, 0.884760,
-0.041299, 0.464213, 0.884760,
0.122800, 0.332342, 0.935130,
-0.161006, 0.394136, 0.904839,
-0.201189, 0.325530, 0.923879,
-0.119755, 0.460882, 0.879344,
-0.198227, 0.454262, 0.868535,
-0.233445, 0.511205, 0.827147,
-0.274694, 0.444464, 0.852640,
-0.078459, 0.524110, 0.848029,
0.161006, 0.394136, 0.904839,
0.198227, 0.454262, 0.868535,
0.119755, 0.460882, 0.879344,
0.078459, 0.524110, 0.848029,
0.233445, 0.511205, 0.827147,
-0.110385, 0.066709, 0.991648,
-0.069142, 0.000000, 0.997607,
-0.151549, 0.133256, 0.979426,
-0.179596, 0.066604, 0.981483,
-0.247326, 0.066567, 0.966643,
-0.206103, 0.000000, 0.978530,
-0.191780, 0.198337, 0.961188,
-0.289929, 0.132570, 0.947822,
-0.316778, 0.066042, 0.946198,
-0.261550, 0.199094, 0.944433,
-0.330606, 0.198387, 0.922682,
-0.368518, 0.262661, 0.891742,
-0.397351, 0.196412, 0.896401,
-0.232455, 0.264151, 0.936050,
-0.381722, 0.065988, 0.921919,
-0.339123, 0.000000, 0.940742,
-0.422364, 0.131749, 0.896801,
-0.445365, 0.065817, 0.892927,
-0.506298, 0.065758, 0.859848,
-0.465658, 0.000000, 0.884965,
-0.460267, 0.196015, 0.865871,
-0.270627, 0.325929, 0.905832,
-0.307659, 0.385906, 0.869725,
-0.339209, 0.324864, 0.882837,
-0.405434, 0.322525, 0.855337,
-0.342848, 0.442867, 0.828447,
0.273707, 0.554740, 0.785715,
0.197274, 0.564513, 0.801504,
0.159613, 0.615456, 0.771841,
0.119078, 0.571140, 0.812169,
0.040675, 0.574458, 0.817523,
0.199331, 0.656006, 0.727958,
0.121193, 0.662842, 0.738886,
0.000000, 0.622292, 0.782786,
-0.040675, 0.574458, 0.817523,
0.040677, 0.667502, 0.743496,
-0.040677, 0.667502, 0.743496,
-0.080573, 0.708969, 0.700622,
-0.121193, 0.662842, 0.738886,
0.080573, 0.708969, 0.700622,
-0.119078, 0.571140, 0.812169,
-0.159613, 0.615456, 0.771841,
-0.197274, 0.564513, 0.801504,
-0.273707, 0.554740, 0.785715,
-0.199331, 0.656006, 0.727958,
0.121144, 0.744727, 0.656282,
0.040642, 0.749387, 0.660884,
0.000000, 0.786433, 0.617676,
-0.040642, 0.749387, 0.660884,
-0.121144, 0.744727, 0.656282,
0.040640, 0.819207, 0.572056,
-0.040640, 0.819207, 0.572056,
0.445365, -0.065817, 0.892927,
0.506298, -0.065758, 0.859848,
0.381722, -0.065988, 0.921919,
0.422364, -0.131749, 0.896801,
0.397351, -0.196412, 0.896401,
0.460267, -0.196015, 0.865871,
0.316778, -0.066042, 0.946198,
0.330606, -0.198387, 0.922682,
0.368518, -0.262661, 0.891742,
0.289929, -0.132570, 0.947822,
0.261550, -0.199094, 0.944433,
0.191780, -0.198337, 0.961188,
0.232455, -0.264151, 0.936050,
0.247326, -0.066567, 0.966643,
0.339209, -0.324864, 0.882837,
0.405434, -0.322525, 0.855337,
0.270627, -0.325929, 0.905832,
0.307659, -0.385906, 0.869725,
0.274694, -0.444464, 0.852640,
0.342848, -0.442867, 0.828447,
0.201189, -0.325530, 0.923879,
0.179596, -0.066604, 0.981483,
0.110385, -0.066709, 0.991648,
0.151549, -0.133256, 0.979426,
0.122729, -0.198580, 0.972370,
0.041248, -0.066741, 0.996917,
0.198227, -0.454262, 0.868535,
0.233445, -0.511205, 0.827147,
0.161006, -0.394136, 0.904839,
0.119755, -0.460882, 0.879344,
0.041299, -0.464213, 0.884760,
0.078459, -0.524110, 0.848029,
0.122800, -0.332342, 0.935130,
0.000000, -0.400968, 0.916092,
-0.041299, -0.464213, 0.884760,
0.041302, -0.334140, 0.941618,
-0.041302, -0.334140, 0.941618,
-0.081501, -0.265518, 0.960655,
-0.122800, -0.332342, 0.935130,
0.081501, -0.265518, 0.960655,
-0.119755, -0.460882, 0.879344,
-0.078459, -0.524110, 0.848029,
-0.161006, -0.394136, 0.904839,
-0.198227, -0.454262, 0.868535,
-0.274694, -0.444464, 0.852640,
-0.233445, -0.511205, 0.827147,
-0.201189, -0.325530, 0.923879,
0.041251, -0.200368, 0.978852,
0.000000, -0.133523, 0.991046,
-0.041251, -0.200368, 0.978852,
-0.122729, -0.198580, 0.972370,
-0.041248, -0.066741, 0.996917,
0.197274, -0.564513, 0.801504,
0.273707, -0.554740, 0.785715,
0.119078, -0.571140, 0.812169,
0.159613, -0.615456, 0.771841,
0.121193, -0.662842, 0.738886,
0.199331, -0.656006, 0.727958,
0.040675, -0.574458, 0.817523,
0.040677, -0.667502, 0.743496,
0.080573, -0.708969, 0.700622,
0.000000, -0.622292, 0.782786,
-0.040677, -0.667502, 0.743496,
-0.121193, -0.662842, 0.738886,
-0.080573, -0.708969, 0.700622,
-0.040675, -0.574458, 0.817523,
0.040642, -0.749387, 0.660884,
0.121144, -0.744727, 0.656282,
-0.040642, -0.749387, 0.660884,
0.000000, -0.786433, 0.617676,
-0.040640, -0.819207, 0.572056,
0.040640, -0.819207, 0.572056,
-0.121144, -0.744727, 0.656282,
-0.119078, -0.571140, 0.812169,
-0.197274, -0.564513, 0.801504,
-0.159613, -0.615456, 0.771841,
-0.199331, -0.656006, 0.727958,
-0.273707, -0.554740, 0.785715,
-0.110385, -0.066709, 0.991648,
-0.179596, -0.066604, 0.981483,
-0.151549, -0.133256, 0.979426,
-0.191780, -0.198337, 0.961188,
-0.247326, -0.066567, 0.966643,
-0.261550, -0.199094, 0.944433,
-0.232455, -0.264151, 0.936050,
-0.289929, -0.132570, 0.947822,
-0.330606, -0.198387, 0.922682,
-0.397351, -0.196412, 0.896401,
-0.368518, -0.262661, 0.891742,
-0.316778, -0.066042, 0.946198,
-0.270627, -0.325929, 0.905832,
-0.339209, -0.324864, 0.882837,
-0.307659, -0.385906, 0.869725,
-0.342848, -0.442867, 0.828447,
-0.405434, -0.322525, 0.855337,
-0.381722, -0.065988, 0.921919,
-0.445365, -0.065817, 0.892927,
-0.422364, -0.131749, 0.896801,
-0.460267, -0.196015, 0.865871,
-0.506298, -0.065758, 0.859848,
0.465658, 0.000000, -0.884965,
0.445365, 0.065817, -0.892927,
0.506298, 0.065758, -0.859848,
0.381722, 0.065988, -0.921919,
0.422364, 0.131749, -0.896801,
0.397351, 0.196412, -0.896401,
0.460267, 0.196015, -0.865871,
0.339123, 0.000000, -0.940742,
0.316778, 0.066042, -0.946198,
0.330606, 0.198387, -0.922682,
0.368518, 0.262661, -0.891742,
0.289929, 0.132570, -0.947822,
0.261550, 0.199094, -0.944433,
0.191780, 0.198337, -0.961188,
0.232455, 0.264151, -0.936050,
0.247326, 0.066567, -0.966643,
0.339209, 0.324864, -0.882837,
0.405434, 0.322525, -0.855337,
0.270627, 0.325929, -0.905832,
0.307659, 0.385906, -0.869725,
0.274694, 0.444464, -0.852640,
0.342848, 0.442867, -0.828447,
0.201189, 0.325530, -0.923879,
0.206103, 0.000000, -0.978530,
0.179596, 0.066604, -0.981483,
0.110385, 0.066709, -0.991648,
0.151549, 0.133256, -0.979426,
0.122729, 0.198580, -0.972370,
0.069142, 0.000000, -0.997607,
0.041248, 0.066741, -0.996917,
0.198227, 0.454262, -0.868535,
0.233445, 0.511205, -0.827147,
0.161006, 0.394136, -0.904839,
0.119755, 0.460882, -0.879344,
0.041299, 0.464213, -0.884760,
0.078459, 0.524110, -0.848029,
0.122800, 0.332342, -0.935130,
0.000000, 0.400968, -0.916092,
-0.041299, 0.464213, -0.884760,
0.041302, 0.334140, -0.941618,
-0.041302, 0.334140, -0.941618,
-0.081501, 0.265518, -0.960655,
-0.122800, 0.332342, -0.935130,
0.081501, 0.265518, -0.960655,
-0.119755, 0.460882, -0.879344,
-0.078459, 0.524110, -0.848029,
-0.161006, 0.394136, -0.904839,
-0.198227, 0.454262, -0.868535,
-0.274694, 0.444464, -0.852640,
-0.233445, 0.511205, -0.827147,
-0.201189, 0.325530, -0.923879,
0.041251, 0.200368, -0.978852,
0.000000, 0.133523, -0.991046,
-0.041251, 0.200368, -0.978852,
-0.122729, 0.198580, -0.972370,
-0.041248, 0.066741, -0.996917,
0.197274, 0.564513, -0.801504,
0.273707, 0.554740, -0.785715,
0.119078, 0.571140, -0.812169,
0.159613, 0.615456, -0.771841,
0.121193, 0.662842, -0.738886,
0.199331, 0.656006, -0.727958,
0.040675, 0.574458, -0.817523,
0.040677, 0.667502, -0.743496,
0.080573, 0.708969, -0.700622,
0.000000, 0.622292, -0.782786,
-0.040677, 0.667502, -0.743496,
-0.121193, 0.662842, -0.738886,
-0.080573, 0.708969, -0.700622,
-0.040675, 0.574458, -0.817523,
0.040642, 0.749387, -0.660884,
0.121144, 0.744727, -0.656282,
-0.040642, 0.749387, -0.660884,
0.000000, 0.786433, -0.617676,
-0.040640, 0.819207, -0.572056,
0.040640, 0.819207, -0.572056,
-0.121144, 0.744727, -0.656282,
-0.119078, 0.571140, -0.812169,
-0.197274, 0.564513, -0.801504,
-0.159613, 0.615456, -0.771841,
-0.199331, 0.656006, -0.727958,
-0.273707, 0.554740, -0.785715,
-0.069142, 0.000000, -0.997607,
-0.110385, 0.066709, -0.991648,
-0.179596, 0.066604, -0.981483,
-0.151549, 0.133256, -0.979426,
-0.191780, 0.198337, -0.961188,
-0.206103, 0.000000, -0.978530,
-0.247326, 0.066567, -0.966643,
-0.261550, 0.199094, -0.944433,
-0.232455, 0.264151, -0.936050,
-0.289929, 0.132570, -0.947822,
-0.330606, 0.198387, -0.922682,
-0.397351, 0.196412, -0.896401,
-0.368518, 0.262661, -0.891742,
-0.316778, 0.066042, -0.946198,
-0.270627, 0.325929, -0.905832,
-0.339209, 0.324864, -0.882837,
-0.307659, 0.385906, -0.869725,
-0.342848, 0.442867, -0.828447,
-0.405434, 0.322525, -0.855337,
-0.339123, 0.000000, -0.940742,
-0.381722, 0.065988, -0.921919,
-0.445365, 0.065817, -0.892927,
-0.422364, 0.131749, -0.896801,
-0.460267, 0.196015, -0.865871,
-0.465658, 0.000000, -0.884965,
-0.506298, 0.065758, -0.859848,
0.506298, -0.065758, -0.859848,
0.445365, -0.065817, -0.892927,
0.422364, -0.131749, -0.896801,
0.381722, -0.065988, -0.921919,
0.316778, -0.066042, -0.946198,
0.460267, -0.196015, -0.865871,
0.397351, -0.196412, -0.896401,
0.289929, -0.132570, -0.947822,
0.247326, -0.066567, -0.966643,
0.330606, -0.198387, -0.922682,
0.261550, -0.199094, -0.944433,
0.232455, -0.264151, -0.936050,
0.191780, -0.198337, -0.961188,
0.368518, -0.262661, -0.891742,
0.179596, -0.066604, -0.981483,
0.151549, -0.133256, -0.979426,
0.110385, -0.066709, -0.991648,
0.041248, -0.066741, -0.996917,
0.122729, -0.198580, -0.972370,
0.405434, -0.322525, -0.855337,
0.339209, -0.324864, -0.882837,
0.307659, -0.385906, -0.869725,
0.270627, -0.325929, -0.905832,
0.201189, -0.325530, -0.923879,
0.342848, -0.442867, -0.828447,
0.274694, -0.444464, -0.852640,
0.000000, -0.133523, -0.991046,
-0.041248, -0.066741, -0.996917,
0.041251, -0.200368, -0.978852,
-0.041251, -0.200368, -0.978852,
-0.081501, -0.265518, -0.960655,
-0.122729, -0.198580, -0.972370,
0.081501, -0.265518, -0.960655,
-0.041302, -0.334140, -0.941618,
-0.122800, -0.332342, -0.935130,
0.041302, -0.334140, -0.941618,
0.000000, -0.400968, -0.916092,
0.041299, -0.464213, -0.884760,
-0.041299, -0.464213, -0.884760,
0.122800, -0.332342, -0.935130,
-0.161006, -0.394136, -0.904839,
-0.201189, -0.325530, -0.923879,
-0.119755, -0.460882, -0.879344,
-0.198227, -0.454262, -0.868535,
-0.233445, -0.511205, -0.827147,
-0.274694, -0.444464, -0.852640,
-0.078459, -0.524110, -0.848029,
0.161006, -0.394136, -0.904839,
0.198227, -0.454262, -0.868535,
0.119755, -0.460882, -0.879344,
0.078459, -0.524110, -0.848029,
0.233445, -0.511205, -0.827147,
-0.110385, -0.066709, -0.991648,
-0.151549, -0.133256, -0.979426,
-0.179596, -0.066604, -0.981483,
-0.247326, -0.066567, -0.966643,
-0.191780, -0.198337, -0.961188,
-0.289929, -0.132570, -0.947822,
-0.316778, -0.066042, -0.946198,
-0.261550, -0.199094, -0.944433,
-0.330606, -0.198387, -0.922682,
-0.368518, -0.262661, -0.891742,
-0.397351, -0.196412, -0.896401,
-0.232455, -0.264151, -0.936050,
-0.381722, -0.065988, -0.921919,
-0.422364, -0.131749, -0.896801,
-0.445365, -0.065817, -0.892927,
-0.506298, -0.065758, -0.859848,
-0.460267, -0.196015, -0.865871,
-0.270627, -0.325929, -0.905832,
-0.307659, -0.385906, -0.869725,
-0.339209, -0.324864, -0.882837,
-0.405434, -0.322525, -0.855337,
-0.342848, -0.442867, -0.828447,
0.273707, -0.554740, -0.785715,
0.197274, -0.564513, -0.801504,
0.159613, -0.615456, -0.771841,
0.119078, -0.571140, -0.812169,
0.040675, -0.574458, -0.817523,
0.199331, -0.656006, -0.727958,
0.121193, -0.662842, -0.738886,
0.000000, -0.622292, -0.782786,
-0.040675, -0.574458, -0.817523,
0.040677, -0.667502, -0.743496,
-0.040677, -0.667502, -0.743496,
-0.080573, -0.708969, -0.700622,
-0.121193, -0.662842, -0.738886,
0.080573, -0.708969, -0.700622,
-0.119078, -0.571140, -0.812169,
-0.159613, -0.615456, -0.771841,
-0.197274, -0.564513, -0.801504,
-0.273707, -0.554740, -0.785715,
-0.199331, -0.656006, -0.727958,
0.121144, -0.744727, -0.656282,
0.040642, -0.749387, -0.660884,
0.000000, -0.786433, -0.617676,
-0.040642, -0.749387, -0.660884,
-0.121144, -0.744727, -0.656282,
0.040640, -0.819207, -0.572056,
-0.040640, -0.819207, -0.572056,
0.859848, 0.506298, 0.065758,
0.892927, 0.445365, 0.065817,
0.884965, 0.465658, 0.000000,
0.896801, 0.422364, 0.131749,
0.921919, 0.381722, 0.065988,
0.946198, 0.316778, 0.066042,
0.940742, 0.339123, 0.000000,
0.865871, 0.460267, 0.196015,
0.896401, 0.397351, 0.196412,
0.947822, 0.289929, 0.132570,
0.966643, 0.247326, 0.066567,
0.922682, 0.330606, 0.198387,
0.944433, 0.261550, 0.199094,
0.936050, 0.232455, 0.264151,
0.961188, 0.191780, 0.198337,
0.891742, 0.368518, 0.262661,
0.981483, 0.179596, 0.066604,
0.978530, 0.206103, 0.000000,
0.979426, 0.151549, 0.133256,
0.991648, 0.110385, 0.066709,
0.996917, 0.041248, 0.066741,
0.997607, 0.069142, 0.000000,
0.972370, 0.122729, 0.198580,
0.855337, 0.405434, 0.322525,
0.882837, 0.339209, 0.324864,
0.869725, 0.307659, 0.385906,
0.905832, 0.270627, 0.325929,
0.923879, 0.201189, 0.325530,
0.828447, 0.342848, 0.442867,
0.852640, 0.274694, 0.444464,
0.991046, 0.000000, 0.133523,
0.996917, -0.041248, 0.066741,
0.978852, 0.041251, 0.200368,
0.978852, -0.041251, 0.200368,
0.960655, -0.081501, 0.265518,
0.972370, -0.122729, 0.198580,
0.960655, 0.081501, 0.265518,
0.941618, -0.041302, 0.334140,
0.935130, -0.122800, 0.332342,
0.941618, 0.041302, 0.334140,
0.916092, 0.000000, 0.400968,
0.884760, 0.041299, 0.464213,
0.884760, -0.041299, 0.464213,
0.935130, 0.122800, 0.332342,
0.904839, -0.161006, 0.394136,
0.923879, -0.201189, 0.325530,
0.879344, -0.119755, 0.460882,
0.868535, -0.198227, 0.454262,
0.827147, -0.233445, 0.511205,
0.852640, -0.274694, 0.444464,
0.848029, -0.078459, 0.524110,
0.904839, 0.161006, 0.394136,
0.868535, 0.198227, 0.454262,
0.879344, 0.119755, 0.460882,
0.848029, 0.078459, 0.524110,
0.827147, 0.233445, 0.511205,
0.991648, -0.110385, 0.066709,
0.997607, -0.069142, 0.000000,
0.979426, -0.151549, 0.133256,
0.981483, -0.179596, 0.066604,
0.966643, -0.247326, 0.066567,
0.978530, -0.206103, 0.000000,
0.961188, -0.191780, 0.198337,
0.947822, -0.289929, 0.132570,
0.946198, -0.316778, 0.066042,
0.944433, -0.261550, 0.199094,
0.922682, -0.330606, 0.198387,
0.891742, -0.368518, 0.262661,
0.896401, -0.397351, 0.196412,
0.936050, -0.232455, 0.264151,
0.921919, -0.381722, 0.065988,
0.940742, -0.339123, 0.000000,
0.896801, -0.422364, 0.131749,
0.892927, -0.445365, 0.065817,
0.859848, -0.506298, 0.065758,
0.884965, -0.465658, 0.000000,
0.865871, -0.460267, 0.196015,
0.905832, -0.270627, 0.325929,
0.869725, -0.307659, 0.385906,
0.882837, -0.339209, 0.324864,
0.855337, -0.405434, 0.322525,
0.828447, -0.342848, 0.442867,
0.785715, 0.273707, 0.554740,
0.801504, 0.197274, 0.564513,
0.771841, 0.159613, 0.615456,
0.812169, 0.119078, 0.571140,
0.817523, 0.040675, 0.574458,
0.727958, 0.199331, 0.656006,
0.738886, 0.121193, 0.662842,
0.782786, 0.000000, 0.622292,
0.817523, -0.040675, 0.574458,
0.743496, 0.040677, 0.667502,
0.743496, -0.040677, 0.667502,
0.700622, -0.080573, 0.708969,
0.738886, -0.121193, 0.662842,
0.700622, 0.080573, 0.708969,
0.812169, -0.119078, 0.571140,
0.771841, -0.159613, 0.615456,
0.801504, -0.197274, 0.564513,
0.785715, -0.273707, 0.554740,
0.727958, -0.199331, 0.656006,
0.656282, 0.121144, 0.744727,
0.660884, 0.040642, 0.749387,
0.617676, 0.000000, 0.786433,
0.660884, -0.040642, 0.749387,
0.656282, -0.121144, 0.744727,
0.572056, 0.040640, 0.819207,
0.572056, -0.040640, 0.819207,
0.892927, 0.445365, -0.065817,
0.859848, 0.506298, -0.065758,
0.921919, 0.381722, -0.065988,
0.896801, 0.422364, -0.131749,
0.896401, 0.397351, -0.196412,
0.865871, 0.460267, -0.196015,
0.946198, 0.316778, -0.066042,
0.922682, 0.330606, -0.198387,
0.891742, 0.368518, -0.262661,
0.947822, 0.289929, -0.132570,
0.944433, 0.261550, -0.199094,
0.961188, 0.191780, -0.198337,
0.936050, 0.232455, -0.264151,
0.966643, 0.247326, -0.066567,
0.882837, 0.339209, -0.324864,
0.855337, 0.405434, -0.322525,
0.905832, 0.270627, -0.325929,
0.869725, 0.307659, -0.385906,
0.852640, 0.274694, -0.444464,
0.828447, 0.342848, -0.442867,
0.923879, 0.201189, -0.325530,
0.981483, 0.179596, -0.066604,
0.991648, 0.110385, -0.066709,
0.979426, 0.151549, -0.133256,
0.972370, 0.122729, -0.198580,
0.996917, 0.041248, -0.066741,
0.868535, 0.198227, -0.454262,
0.827147, 0.233445, -0.511205,
0.904839, 0.161006, -0.394136,
0.879344, 0.119755, -0.460882,
0.884760, 0.041299, -0.464213,
0.848029, 0.078459, -0.524110,
0.935130, 0.122800, -0.332342,
0.916092, 0.000000, -0.400968,
0.884760, -0.041299, -0.464213,
0.941618, 0.041302, -0.334140,
0.941618, -0.041302, -0.334140,
0.960655, -0.081501, -0.265518,
0.935130, -0.122800, -0.332342,
0.960655, 0.081501, -0.265518,
0.879344, -0.119755, -0.460882,
0.848029, -0.078459, -0.524110,
0.904839, -0.161006, -0.394136,
0.868535, -0.198227, -0.454262,
0.852640, -0.274694, -0.444464,
0.827147, -0.233445, -0.511205,
0.923879, -0.201189, -0.325530,
0.978852, 0.041251, -0.200368,
0.991046, 0.000000, -0.133523,
0.978852, -0.041251, -0.200368,
0.972370, -0.122729, -0.198580,
0.996917, -0.041248, -0.066741,
0.801504, 0.197274, -0.564513,
0.785715, 0.273707, -0.554740,
0.812169, 0.119078, -0.571140,
0.771841, 0.159613, -0.615456,
0.738886, 0.121193, -0.662842,
0.727958, 0.199331, -0.656006,
0.817523, 0.040675, -0.574458,
0.743496, 0.040677, -0.667502,
0.700622, 0.080573, -0.708969,
0.782786, 0.000000, -0.622292,
0.743496, -0.040677, -0.667502,
0.738886, -0.121193, -0.662842,
0.700622, -0.080573, -0.708969,
0.817523, -0.040675, -0.574458,
0.660884, 0.040642, -0.749387,
0.656282, 0.121144, -0.744727,
0.660884, -0.040642, -0.749387,
0.617676, 0.000000, -0.786433,
0.572056, -0.040640, -0.819207,
0.572056, 0.040640, -0.819207,
0.656282, -0.121144, -0.744727,
0.812169, -0.119078, -0.571140,
0.801504, -0.197274, -0.564513,
0.771841, -0.159613, -0.615456,
0.727958, -0.199331, -0.656006,
0.785715, -0.273707, -0.554740,
0.991648, -0.110385, -0.066709,
0.981483, -0.179596, -0.066604,
0.979426, -0.151549, -0.133256,
0.961188, -0.191780, -0.198337,
0.966643, -0.247326, -0.066567,
0.944433, -0.261550, -0.199094,
0.936050, -0.232455, -0.264151,
0.947822, -0.289929, -0.132570,
0.922682, -0.330606, -0.198387,
0.896401, -0.397351, -0.196412,
0.891742, -0.368518, -0.262661,
0.946198, -0.316778, -0.066042,
0.905832, -0.270627, -0.325929,
0.882837, -0.339209, -0.324864,
0.869725, -0.307659, -0.385906,
0.828447, -0.342848, -0.442867,
0.855337, -0.405434, -0.322525,
0.921919, -0.381722, -0.065988,
0.892927, -0.445365, -0.065817,
0.896801, -0.422364, -0.131749,
0.865871, -0.460267, -0.196015,
0.859848, -0.506298, -0.065758,
-0.859848, -0.506298, 0.065758,
-0.892927, -0.445365, 0.065817,
-0.884965, -0.465658, 0.000000,
-0.896801, -0.422364, 0.131749,
-0.921919, -0.381722, 0.065988,
-0.946198, -0.316778, 0.066042,
-0.940742, -0.339123, 0.000000,
-0.865871, -0.460267, 0.196015,
-0.896401, -0.397351, 0.196412,
-0.947822, -0.289929, 0.132570,
-0.966643, -0.247326, 0.066567,
-0.922682, -0.330606, 0.198387,
-0.944433, -0.261550, 0.199094,
-0.936050, -0.232455, 0.264151,
-0.961188, -0.191780, 0.198337,
-0.891742, -0.368518, 0.262661,
-0.981483, -0.179596, 0.066604,
-0.978530, -0.206103, 0.000000,
-0.979426, -0.151549, 0.133256,
-0.991648, -0.110385, 0.066709,
-0.996917, -0.041248, 0.066741,
-0.997607, -0.069142, 0.000000,
-0.972370, -0.122729, 0.198580,
-0.855337, -0.405434, 0.322525,
-0.882837, -0.339209, 0.324864,
-0.869725, -0.307659, 0.385906,
-0.905832, -0.270627, 0.325929,
-0.923879, -0.201189, 0.325530,
-0.828447, -0.342848, 0.442867,
-0.852640, -0.274694, 0.444464,
-0.991046, 0.000000, 0.133523,
-0.996917, 0.041248, 0.066741,
-0.978852, -0.041251, 0.200368,
-0.978852, 0.041251, 0.200368,
-0.960655, 0.081501, 0.265518,
-0.972370, 0.122729, 0.198580,
-0.960655, -0.081501, 0.265518,
-0.941618, 0.041302, 0.334140,
-0.935130, 0.122800, 0.332342,
-0.941618, -0.041302, 0.334140,
-0.916092, 0.000000, 0.400968,
-0.884760, -0.041299, 0.464213,
-0.884760, 0.041299, 0.464213,
-0.935130, -0.122800, 0.332342,
-0.904839, 0.161006, 0.394136,
-0.923879, 0.201189, 0.325530,
-0.879344, 0.119755, 0.460882,
-0.868535, 0.198227, 0.454262,
-0.827147, 0.233445, 0.511205,
-0.852640, 0.274694, 0.444464,
-0.848029, 0.078459, 0.524110,
-0.904839, -0.161006, 0.394136,
-0.868535, -0.198227, 0.454262,
-0.879344, -0.119755, 0.460882,
-0.848029, -0.078459, 0.524110,
-0.827147, -0.233445, 0.511205,
-0.991648, 0.110385, 0.066709,
-0.997607, 0.069142, 0.000000,
-0.979426, 0.151549, 0.133256,
-0.981483, 0.179596, 0.066604,
-0.966643, 0.247326, 0.066567,
-0.978530, 0.206103, 0.000000,
-0.961188, 0.191780, 0.198337,
-0.947822, 0.289929, 0.132570,
-0.946198, 0.316778, 0.066042,
-0.944433, 0.261550, 0.199094,
-0.922682, 0.330606, 0.198387,
-0.891742, 0.368518, 0.262661,
-0.896401, 0.397351, 0.196412,
-0.936050, 0.232455, 0.264151,
-0.921919, 0.381722, 0.065988,
-0.940742, 0.339123, 0.000000,
-0.896801, 0.422364, 0.131749,
-0.892927, 0.445365, 0.065817,
-0.859848, 0.506298, 0.065758,
-0.884965, 0.465658, 0.000000,
-0.865871, 0.460267, 0.196015,
-0.905832, 0.270627, 0.325929,
-0.869725, 0.307659, 0.385906,
-0.882837, 0.339209, 0.324864,
-0.855337, 0.405434, 0.322525,
-0.828447, 0.342848, 0.442867,
-0.785715, -0.273707, 0.554740,
-0.801504, -0.197274, 0.564513,
-0.771841, -0.159613, 0.615456,
-0.812169, -0.119078, 0.571140,
-0.817523, -0.040675, 0.574458,
-0.727958, -0.199331, 0.656006,
-0.738886, -0.121193, 0.662842,
-0.782786, 0.000000, 0.622292,
-0.817523, 0.040675, 0.574458,
-0.743496, -0.040677, 0.667502,
-0.743496, 0.040677, 0.667502,
-0.700622, 0.080573, 0.708969,
-0.738886, 0.121193, 0.662842,
-0.700622, -0.080573, 0.708969,
-0.812169, 0.119078, 0.571140,
-0.771841, 0.159613, 0.615456,
-0.801504, 0.197274, 0.564513,
-0.785715, 0.273707, 0.554740,
-0.727958, 0.199331, 0.656006,
-0.656282, -0.121144, 0.744727,
-0.660884, -0.040642, 0.749387,
-0.617676, 0.000000, 0.786433,
-0.660884, 0.040642, 0.749387,
-0.656282, 0.121144, 0.744727,
-0.572056, -0.040640, 0.819207,
-0.572056, 0.040640, 0.819207,
-0.892927, -0.445365, -0.065817,
-0.859848, -0.506298, -0.065758,
-0.921919, -0.381722, -0.065988,
-0.896801, -0.422364, -0.131749,
-0.896401, -0.397351, -0.196412,
-0.865871, -0.460267, -0.196015,
-0.946198, -0.316778, -0.066042,
-0.922682, -0.330606, -0.198387,
-0.891742, -0.368518, -0.262661,
-0.947822, -0.289929, -0.132570,
-0.944433, -0.261550, -0.199094,
-0.961188, -0.191780, -0.198337,
-0.936050, -0.232455, -0.264151,
-0.966643, -0.247326, -0.066567,
-0.882837, -0.339209, -0.324864,
-0.855337, -0.405434, -0.322525,
-0.905832, -0.270627, -0.325929,
-0.869725, -0.307659, -0.385906,
-0.852640, -0.274694, -0.444464,
-0.828447, -0.342848, -0.442867,
-0.923879, -0.201189, -0.325530,
-0.981483, -0.179596, -0.066604,
-0.991648, -0.110385, -0.066709,
-0.979426, -0.151549, -0.133256,
-0.972370, -0.122729, -0.198580,
-0.996917, -0.041248, -0.066741,
-0.868535, -0.198227, -0.454262,
-0.827147, -0.233445, -0.511205,
-0.904839, -0.161006, -0.394136,
-0.879344, -0.119755, -0.460882,
-0.884760, -0.041299, -0.464213,
-0.848029, -0.078459, -0.524110,
-0.935130, -0.122800, -0.332342,
-0.916092, 0.000000, -0.400968,
-0.884760, 0.041299, -0.464213,
-0.941618, -0.041302, -0.334140,
-0.941618, 0.041302, -0.334140,
-0.960655, 0.081501, -0.265518,
-0.935130, 0.122800, -0.332342,
-0.960655, -0.081501, -0.265518,
-0.879344, 0.119755, -0.460882,
-0.848029, 0.078459, -0.524110,
-0.904839, 0.161006, -0.394136,
-0.868535, 0.198227, -0.454262,
-0.852640, 0.274694, -0.444464,
-0.827147, 0.233445, -0.511205,
-0.923879, 0.201189, -0.325530,
-0.978852, -0.041251, -0.200368,
-0.991046, 0.000000, -0.133523,
-0.978852, 0.041251, -0.200368,
-0.972370, 0.122729, -0.198580,
-0.996917, 0.041248, -0.066741,
-0.801504, -0.197274, -0.564513,
-0.785715, -0.273707, -0.554740,
-0.812169, -0.119078, -0.571140,
-0.771841, -0.159613, -0.615456,
-0.738886, -0.121193, -0.662842,
-0.727958, -0.199331, -0.656006,
-0.817523, -0.040675, -0.574458,
-0.743496, -0.040677, -0.667502,
-0.700622, -0.080573, -0.708969,
-0.782786, 0.000000, -0.622292,
-0.743496, 0.040677, -0.667502,
-0.738886, 0.121193, -0.662842,
-0.700622, 0.080573, -0.708969,
-0.817523, 0.040675, -0.574458,
-0.660884, -0.040642, -0.749387,
-0.656282, -0.121144, -0.744727,
-0.660884, 0.040642, -0.749387,
-0.617676, 0.000000, -0.786433,
-0.572056, 0.040640, -0.819207,
-0.572056, -0.040640, -0.819207,
-0.656282, 0.121144, -0.744727,
-0.812169, 0.119078, -0.571140,
-0.801504, 0.197274, -0.564513,
-0.771841, 0.159613, -0.615456,
-0.727958, 0.199331, -0.656006,
-0.785715, 0.273707, -0.554740,
-0.991648, 0.110385, -0.066709,
-0.981483, 0.179596, -0.066604,
-0.979426, 0.151549, -0.133256,
-0.961188, 0.191780, -0.198337,
-0.966643, 0.247326, -0.066567,
-0.944433, 0.261550, -0.199094,
-0.936050, 0.232455, -0.264151,
-0.947822, 0.289929, -0.132570,
-0.922682, 0.330606, -0.198387,
-0.896401, 0.397351, -0.196412,
-0.891742, 0.368518, -0.262661,
-0.946198, 0.316778, -0.066042,
-0.905832, 0.270627, -0.325929,
-0.882837, 0.339209, -0.324864,
-0.869725, 0.307659, -0.385906,
-0.828447, 0.342848, -0.442867,
-0.855337, 0.405434, -0.322525,
-0.921919, 0.381722, -0.065988,
-0.892927, 0.445365, -0.065817,
-0.896801, 0.422364, -0.131749,
-0.865871, 0.460267, -0.196015,
-0.859848, 0.506298, -0.065758,
0.065758, 0.859848, 0.506298,
0.065817, 0.892927, 0.445365,
0.000000, 0.884965, 0.465658,
0.131749, 0.896801, 0.422364,
0.065988, 0.921919, 0.381722,
0.066042, 0.946198, 0.316778,
0.000000, 0.940742, 0.339123,
0.196015, 0.865871, 0.460267,
0.196412, 0.896401, 0.397351,
0.132570, 0.947822, 0.289929,
0.066567, 0.966643, 0.247326,
0.198387, 0.922682, 0.330606,
0.199094, 0.944433, 0.261550,
0.264151, 0.936050, 0.232455,
0.198337, 0.961188, 0.191780,
0.262661, 0.891742, 0.368518,
0.066604, 0.981483, 0.179596,
0.000000, 0.978530, 0.206103,
0.133256, 0.979426, 0.151549,
0.066709, 0.991648, 0.110385,
0.066741, 0.996917, 0.041248,
0.000000, 0.997607, 0.069142,
0.198580, 0.972370, 0.122729,
0.322525, 0.855337, 0.405434,
0.324864, 0.882837, 0.339209,
0.385906, 0.869725, 0.307659,
0.325929, 0.905832, 0.270627,
0.325530, 0.923879, 0.201189,
0.442867, 0.828447, 0.342848,
0.444464, 0.852640, 0.274694,
0.133523, 0.991046, 0.000000,
0.066741, 0.996917, -0.041248,
0.200368, 0.978852, 0.041251,
0.200368, 0.978852, -0.041251,
0.265518, 0.960655, -0.081501,
0.198580, 0.972370, -0.122729,
0.265518, 0.960655, 0.081501,
0.334140, 0.941618, -0.041302,
0.332342, 0.935130, -0.122800,
0.334140, 0.941618, 0.041302,
0.400968, 0.916092, 0.000000,
0.464213, 0.884760, 0.041299,
0.464213, 0.884760, -0.041299,
0.332342, 0.935130, 0.122800,
0.394136, 0.904839, -0.161006,
0.325530, 0.923879, -0.201189,
0.460882, 0.879344, -0.119755,
0.454262, 0.868535, -0.198227,
0.511205, 0.827147, -0.233445,
0.444464, 0.852640, -0.274694,
0.524110, 0.848029, -0.078459,
0.394136, 0.904839, 0.161006,
0.454262, 0.868535, 0.198227,
0.460882, 0.879344, 0.119755,
0.524110, 0.848029, 0.078459,
0.511205, 0.827147, 0.233445,
0.066709, 0.991648, -0.110385,
0.000000, 0.997607, -0.069142,
0.133256, 0.979426, -0.151549,
0.066604, 0.981483, -0.179596,
0.066567, 0.966643, -0.247326,
0.000000, 0.978530, -0.206103,
0.198337, 0.961188, -0.191780,
0.132570, 0.947822, -0.289929,
0.066042, 0.946198, -0.316778,
0.199094, 0.944433, -0.261550,
0.198387, 0.922682, -0.330606,
0.262661, 0.891742, -0.368518,
0.196412, 0.896401, -0.397351,
0.264151, 0.936050, -0.232455,
0.065988, 0.921919, -0.381722,
0.000000, 0.940742, -0.339123,
0.131749, 0.896801, -0.422364,
0.065817, 0.892927, -0.445365,
0.065758, 0.859848, -0.506298,
0.000000, 0.884965, -0.465658,
0.196015, 0.865871, -0.460267,
0.325929, 0.905832, -0.270627,
0.385906, 0.869725, -0.307659,
0.324864, 0.882837, -0.339209,
0.322525, 0.855337, -0.405434,
0.442867, 0.828447, -0.342848,
0.554740, 0.785715, 0.273707,
0.564513, 0.801504, 0.197274,
0.615456, 0.771841, 0.159613,
0.571140, 0.812169, 0.119078,
0.574458, 0.817523, 0.040675,
0.656006, 0.727958, 0.199331,
0.662842, 0.738886, 0.121193,
0.622292, 0.782786, 0.000000,
0.574458, 0.817523, -0.040675,
0.667502, 0.743496, 0.040677,
0.667502, 0.743496, -0.040677,
0.708969, 0.700622, -0.080573,
0.662842, 0.738886, -0.121193,
0.708969, 0.700622, 0.080573,
0.571140, 0.812169, -0.119078,
0.615456, 0.771841, -0.159613,
0.564513, 0.801504, -0.197274,
0.554740, 0.785715, -0.273707,
0.656006, 0.727958, -0.199331,
0.744727, 0.656282, 0.121144,
0.749387, 0.660884, 0.040642,
0.786433, 0.617676, 0.000000,
0.749387, 0.660884, -0.040642,
0.744727, 0.656282, -0.121144,
0.819207, 0.572056, 0.040640,
0.819207, 0.572056, -0.040640,
-0.065817, 0.892927, 0.445365,
-0.065758, 0.859848, 0.506298,
-0.065988, 0.921919, 0.381722,
-0.131749, 0.896801, 0.422364,
-0.196412, 0.896401, 0.397351,
-0.196015, 0.865871, 0.460267,
-0.066042, 0.946198, 0.316778,
-0.198387, 0.922682, 0.330606,
-0.262661, 0.891742, 0.368518,
-0.132570, 0.947822, 0.289929,
-0.199094, 0.944433, 0.261550,
-0.198337, 0.961188, 0.191780,
-0.264151, 0.936050, 0.232455,
-0.066567, 0.966643, 0.247326,
-0.324864, 0.882837, 0.339209,
-0.322525, 0.855337, 0.405434,
-0.325929, 0.905832, 0.270627,
-0.385906, 0.869725, 0.307659,
-0.444464, 0.852640, 0.274694,
-0.442867, 0.828447, 0.342848,
-0.325530, 0.923879, 0.201189,
-0.066604, 0.981483, 0.179596,
-0.066709, 0.991648, 0.110385,
-0.133256, 0.979426, 0.151549,
-0.198580, 0.972370, 0.122729,
-0.066741, 0.996917, 0.041248,
-0.454262, 0.868535, 0.198227,
-0.511205, 0.827147, 0.233445,
-0.394136, 0.904839, 0.161006,
-0.460882, 0.879344, 0.119755,
-0.464213, 0.884760, 0.041299,
-0.524110, 0.848029, 0.078459,
-0.332342, 0.935130, 0.122800,
-0.400968, 0.916092, 0.000000,
-0.464213, 0.884760, -0.041299,
-0.334140, 0.941618, 0.041302,
-0.334140, 0.941618, -0.041302,
-0.265518, 0.960655, -0.081501,
-0.332342, 0.935130, -0.122800,
-0.265518, 0.960655, 0.081501,
-0.460882, 0.879344, -0.119755,
-0.524110, 0.848029, -0.078459,
-0.394136, 0.904839, -0.161006,
-0.454262, 0.868535, -0.198227,
-0.444464, 0.852640, -0.274694,
-0.511205, 0.827147, -0.233445,
-0.325530, 0.923879, -0.201189,
-0.200368, 0.978852, 0.041251,
-0.133523, 0.991046, 0.000000,
-0.200368, 0.978852, -0.041251,
-0.198580, 0.972370, -0.122729,
-0.066741, 0.996917, -0.041248,
-0.564513, 0.801504, 0.197274,
-0.554740, 0.785715, 0.273707,
-0.571140, 0.812169, 0.119078,
-0.615456, 0.771841, 0.159613,
-0.662842, 0.738886, 0.121193,
-0.656006, 0.727958, 0.199331,
-0.574458, 0.817523, 0.040675,
-0.667502, 0.743496, 0.040677,
-0.708969, 0.700622, 0.080573,
-0.622292, 0.782786, 0.000000,
-0.667502, 0.743496, -0.040677,
-0.662842, 0.738886, -0.121193,
-0.708969, 0.700622, -0.080573,
-0.574458, 0.817523, -0.040675,
-0.749387, 0.660884, 0.040642,
-0.744727, 0.656282, 0.121144,
-0.749387, 0.660884, -0.040642,
-0.786433, 0.617676, 0.000000,
-0.819207, 0.572056, -0.040640,
-0.819207, 0.572056, 0.040640,
-0.744727, 0.656282, -0.121144,
-0.571140, 0.812169, -0.119078,
-0.564513, 0.801504, -0.197274,
-0.615456, 0.771841, -0.159613,
-0.656006, 0.727958, -0.199331,
-0.554740, 0.785715, -0.273707,
-0.066709, 0.991648, -0.110385,
-0.066604, 0.981483, -0.179596,
-0.133256, 0.979426, -0.151549,
-0.198337, 0.961188, -0.191780,
-0.066567, 0.966643, -0.247326,
-0.199094, 0.944433, -0.261550,
-0.264151, 0.936050, -0.232455,
-0.132570, 0.947822, -0.289929,
-0.198387, 0.922682, -0.330606,
-0.196412, 0.896401, -0.397351,
-0.262661, 0.891742, -0.368518,
-0.066042, 0.946198, -0.316778,
-0.325929, 0.905832, -0.270627,
-0.324864, 0.882837, -0.339209,
-0.385906, 0.869725, -0.307659,
-0.442867, 0.828447, -0.342848,
-0.322525, 0.855337, -0.405434,
-0.065988, 0.921919, -0.381722,
-0.065817, 0.892927, -0.445365,
-0.131749, 0.896801, -0.422364,
-0.196015, 0.865871, -0.460267,
-0.065758, 0.859848, -0.506298,
0.000000, -0.884965, 0.465658,
0.065817, -0.892927, 0.445365,
0.065758, -0.859848, 0.506298,
0.065988, -0.921919, 0.381722,
0.131749, -0.896801, 0.422364,
0.196412, -0.896401, 0.397351,
0.196015, -0.865871, 0.460267,
0.000000, -0.940742, 0.339123,
0.066042, -0.946198, 0.316778,
0.198387, -0.922682, 0.330606,
0.262661, -0.891742, 0.368518,
0.132570, -0.947822, 0.289929,
0.199094, -0.944433, 0.261550,
0.198337, -0.961188, 0.191780,
0.264151, -0.936050, 0.232455,
0.066567, -0.966643, 0.247326,
0.324864, -0.882837, 0.339209,
0.322525, -0.855337, 0.405434,
0.325929, -0.905832, 0.270627,
0.385906, -0.869725, 0.307659,
0.444464, -0.852640, 0.274694,
0.442867, -0.828447, 0.342848,
0.325530, -0.923879, 0.201189,
0.000000, -0.978530, 0.206103,
0.066604, -0.981483, 0.179596,
0.066709, -0.991648, 0.110385,
0.133256, -0.979426, 0.151549,
0.198580, -0.972370, 0.122729,
0.000000, -0.997607, 0.069142,
0.066741, -0.996917, 0.041248,
0.454262, -0.868535, 0.198227,
0.511205, -0.827147, 0.233445,
0.394136, -0.904839, 0.161006,
0.460882, -0.879344, 0.119755,
0.464213, -0.884760, 0.041299,
0.524110, -0.848029, 0.078459,
0.332342, -0.935130, 0.122800,
0.400968, -0.916092, 0.000000,
0.464213, -0.884760, -0.041299,
0.334140, -0.941618, 0.041302,
0.334140, -0.941618, -0.041302,
0.265518, -0.960655, -0.081501,
0.332342, -0.935130, -0.122800,
0.265518, -0.960655, 0.081501,
0.460882, -0.879344, -0.119755,
0.524110, -0.848029, -0.078459,
0.394136, -0.904839, -0.161006,
0.454262, -0.868535, -0.198227,
0.444464, -0.852640, -0.274694,
0.511205, -0.827147, -0.233445,
0.325530, -0.923879, -0.201189,
0.200368, -0.978852, 0.041251,
0.133523, -0.991046, 0.000000,
0.200368, -0.978852, -0.041251,
0.198580, -0.972370, -0.122729,
0.066741, -0.996917, -0.041248,
0.564513, -0.801504, 0.197274,
0.554740, -0.785715, 0.273707,
0.571140, -0.812169, 0.119078,
0.615456, -0.771841, 0.159613,
0.662842, -0.738886, 0.121193,
0.656006, -0.727958, 0.199331,
0.574458, -0.817523, 0.040675,
0.667502, -0.743496, 0.040677,
0.708969, -0.700622, 0.080573,
0.622292, -0.782786, 0.000000,
0.667502, -0.743496, -0.040677,
0.662842, -0.738886, -0.121193,
0.708969, -0.700622, -0.080573,
0.574458, -0.817523, -0.040675,
0.749387, -0.660884, 0.040642,
0.744727, -0.656282, 0.121144,
0.749387, -0.660884, -0.040642,
0.786433, -0.617676, 0.000000,
0.819207, -0.572056, -0.040640,
0.819207, -0.572056, 0.040640,
0.744727, -0.656282, -0.121144,
0.571140, -0.812169, -0.119078,
0.564513, -0.801504, -0.197274,
0.615456, -0.771841, -0.159613,
0.656006, -0.727958, -0.199331,
0.554740, -0.785715, -0.273707,
0.000000, -0.997607, -0.069142,
0.066709, -0.991648, -0.110385,
0.066604, -0.981483, -0.179596,
0.133256, -0.979426, -0.151549,
0.198337, -0.961188, -0.191780,
0.000000, -0.978530, -0.206103,
0.066567, -0.966643, -0.247326,
0.199094, -0.944433, -0.261550,
0.264151, -0.936050, -0.232455,
0.132570, -0.947822, -0.289929,
0.198387, -0.922682, -0.330606,
0.196412, -0.896401, -0.397351,
0.262661, -0.891742, -0.368518,
0.066042, -0.946198, -0.316778,
0.325929, -0.905832, -0.270627,
0.324864, -0.882837, -0.339209,
0.385906, -0.869725, -0.307659,
0.442867, -0.828447, -0.342848,
0.322525, -0.855337, -0.405434,
0.000000, -0.940742, -0.339123,
0.065988, -0.921919, -0.381722,
0.065817, -0.892927, -0.445365,
0.131749, -0.896801, -0.422364,
0.196015, -0.865871, -0.460267,
0.000000, -0.884965, -0.465658,
0.065758, -0.859848, -0.506298,
-0.065758, -0.859848, 0.506298,
-0.065817, -0.892927, 0.445365,
-0.131749, -0.896801, 0.422364,
-0.065988, -0.921919, 0.381722,
-0.066042, -0.946198, 0.316778,
-0.196015, -0.865871, 0.460267,
-0.196412, -0.896401, 0.397351,
-0.132570, -0.947822, 0.289929,
-0.066567, -0.966643, 0.247326,
-0.198387, -0.922682, 0.330606,
-0.199094, -0.944433, 0.261550,
-0.264151, -0.936050, 0.232455,
-0.198337, -0.961188, 0.191780,
-0.262661, -0.891742, 0.368518,
-0.066604, -0.981483, 0.179596,
-0.133256, -0.979426, 0.151549,
-0.066709, -0.991648, 0.110385,
-0.066741, -0.996917, 0.041248,
-0.198580, -0.972370, 0.122729,
-0.322525, -0.855337, 0.405434,
-0.324864, -0.882837, 0.339209,
-0.385906, -0.869725, 0.307659,
-0.325929, -0.905832, 0.270627,
-0.325530, -0.923879, 0.201189,
-0.442867, -0.828447, 0.342848,
-0.444464, -0.852640, 0.274694,
-0.133523, -0.991046, 0.000000,
-0.066741, -0.996917, -0.041248,
-0.200368, -0.978852, 0.041251,
-0.200368, -0.978852, -0.041251,
-0.265518, -0.960655, -0.081501,
-0.198580, -0.972370, -0.122729,
-0.265518, -0.960655, 0.081501,
-0.334140, -0.941618, -0.041302,
-0.332342, -0.935130, -0.122800,
-0.334140, -0.941618, 0.041302,
-0.400968, -0.916092, 0.000000,
-0.464213, -0.884760, 0.041299,
-0.464213, -0.884760, -0.041299,
-0.332342, -0.935130, 0.122800,
-0.394136, -0.904839, -0.161006,
-0.325530, -0.923879, -0.201189,
-0.460882, -0.879344, -0.119755,
-0.454262, -0.868535, -0.198227,
-0.511205, -0.827147, -0.233445,
-0.444464, -0.852640, -0.274694,
-0.524110, -0.848029, -0.078459,
-0.394136, -0.904839, 0.161006,
-0.454262, -0.868535, 0.198227,
-0.460882, -0.879344, 0.119755,
-0.524110, -0.848029, 0.078459,
-0.511205, -0.827147, 0.233445,
-0.066709, -0.991648, -0.110385,
-0.133256, -0.979426, -0.151549,
-0.066604, -0.981483, -0.179596,
-0.066567, -0.966643, -0.247326,
-0.198337, -0.961188, -0.191780,
-0.132570, -0.947822, -0.289929,
-0.066042, -0.946198, -0.316778,
-0.199094, -0.944433, -0.261550,
-0.198387, -0.922682, -0.330606,
-0.262661, -0.891742, -0.368518,
-0.196412, -0.896401, -0.397351,
-0.264151, -0.936050, -0.232455,
-0.065988, -0.921919, -0.381722,
-0.131749, -0.896801, -0.422364,
-0.065817, -0.892927, -0.445365,
-0.065758, -0.859848, -0.506298,
-0.196015, -0.865871, -0.460267,
-0.325929, -0.905832, -0.270627,
-0.385906, -0.869725, -0.307659,
-0.324864, -0.882837, -0.339209,
-0.322525, -0.855337, -0.405434,
-0.442867, -0.828447, -0.342848,
-0.554740, -0.785715, 0.273707,
-0.564513, -0.801504, 0.197274,
-0.615456, -0.771841, 0.159613,
-0.571140, -0.812169, 0.119078,
-0.574458, -0.817523, 0.040675,
-0.656006, -0.727958, 0.199331,
-0.662842, -0.738886, 0.121193,
-0.622292, -0.782786, 0.000000,
-0.574458, -0.817523, -0.040675,
-0.667502, -0.743496, 0.040677,
-0.667502, -0.743496, -0.040677,
-0.708969, -0.700622, -0.080573,
-0.662842, -0.738886, -0.121193,
-0.708969, -0.700622, 0.080573,
-0.571140, -0.812169, -0.119078,
-0.615456, -0.771841, -0.159613,
-0.564513, -0.801504, -0.197274,
-0.554740, -0.785715, -0.273707,
-0.656006, -0.727958, -0.199331,
-0.744727, -0.656282, 0.121144,
-0.749387, -0.660884, 0.040642,
-0.786433, -0.617676, 0.000000,
-0.749387, -0.660884, -0.040642,
-0.744727, -0.656282, -0.121144,
-0.819207, -0.572056, 0.040640,
-0.819207, -0.572056, -0.040640,
0.106494, 0.827110, 0.551859,
0.147414, 0.790170, 0.594895,
0.172532, 0.830812, 0.529135,
0.237229, 0.830359, 0.504209,
0.187432, 0.749786, 0.634579,
0.280319, 0.790112, 0.545109,
0.303801, 0.825175, 0.476226,
0.255180, 0.749435, 0.610926,
0.322141, 0.745339, 0.583691,
0.361592, 0.697037, 0.619186,
0.386730, 0.737712, 0.553372,
0.228900, 0.703983, 0.672319,
0.366027, 0.816233, 0.446977,
0.408286, 0.772575, 0.486240,
0.427135, 0.803016, 0.415597,
0.485712, 0.785899, 0.382683,
0.448259, 0.725299, 0.522499,
0.267381, 0.656620, 0.705236,
0.305212, 0.605742, 0.734794,
0.334692, 0.653497, 0.678913,
0.399769, 0.646840, 0.649448,
0.341435, 0.552453, 0.760406,
0.536784, 0.735011, 0.414273,
0.552453, 0.760406, 0.341435,
0.517970, 0.704471, 0.485208,
0.584716, 0.678977, 0.443957,
0.628313, 0.619242, 0.470917,
0.646840, 0.649448, 0.399769,
0.496441, 0.669612, 0.552418,
0.607478, 0.581952, 0.540650,
0.669612, 0.552418, 0.496441,
0.540650, 0.607478, 0.581952,
0.581952, 0.540650, 0.607478,
0.552418, 0.496441, 0.669612,
0.619242, 0.470917, 0.628313,
0.470917, 0.628313, 0.619242,
0.704471, 0.485208, 0.517970,
0.725299, 0.522499, 0.448259,
0.678977, 0.443957, 0.584716,
0.735011, 0.414273, 0.536784,
0.760406, 0.341435, 0.552453,
0.785899, 0.382683, 0.485712,
0.649448, 0.399769, 0.646840,
0.443957, 0.584716, 0.678977,
0.414273, 0.536784, 0.735011,
0.485208, 0.517970, 0.704471,
0.522499, 0.448259, 0.725299,
0.382683, 0.485712, 0.785899,
0.605742, 0.734794, 0.305212,
0.653497, 0.678913, 0.334692,
0.656620, 0.705236, 0.267381,
0.703983, 0.672319, 0.228900,
0.697037, 0.619186, 0.361592,
0.749435, 0.610926, 0.255180,
0.749786, 0.634579, 0.187432,
0.745339, 0.583691, 0.322141,
0.790112, 0.545109, 0.280319,
0.825175, 0.476226, 0.303801,
0.830359, 0.504209, 0.237229,
0.737712, 0.553372, 0.386730,
0.790170, 0.594895, 0.147414,
0.830812, 0.529135, 0.172532,
0.827110, 0.551859, 0.106494,
0.772575, 0.486240, 0.408286,
0.803016, 0.415597, 0.427135,
0.816233, 0.446977, 0.366027,
0.415597, 0.427135, 0.803016,
0.446977, 0.366027, 0.816233,
0.486240, 0.408286, 0.772575,
0.553372, 0.386730, 0.737712,
0.476226, 0.303801, 0.825175,
0.583691, 0.322141, 0.745339,
0.619186, 0.361592, 0.697037,
0.545109, 0.280319, 0.790112,
0.610926, 0.255180, 0.749435,
0.634579, 0.187432, 0.749786,
0.672319, 0.228900, 0.703983,
0.504209, 0.237229, 0.830359,
0.678913, 0.334692, 0.653497,
0.705236, 0.267381, 0.656620,
0.734794, 0.305212, 0.605742,
0.529135, 0.172532, 0.830812,
0.551859, 0.106494, 0.827110,
0.594895, 0.147414, 0.790170,
0.106494, 0.827110, -0.551859,
0.172532, 0.830812, -0.529135,
0.147414, 0.790170, -0.594895,
0.187432, 0.749786, -0.634579,
0.237229, 0.830359, -0.504209,
0.255180, 0.749435, -0.610926,
0.228900, 0.703983, -0.672319,
0.280319, 0.790112, -0.545109,
0.322141, 0.745339, -0.583691,
0.386730, 0.737712, -0.553372,
0.361592, 0.697037, -0.619186,
0.303801, 0.825175, -0.476226,
0.267381, 0.656620, -0.705236,
0.334692, 0.653497, -0.678913,
0.305212, 0.605742, -0.734794,
0.341435, 0.552453, -0.760406,
0.399769, 0.646840, -0.649448,
0.366027, 0.816233, -0.446977,
0.427135, 0.803016, -0.415597,
0.408286, 0.772575, -0.486240,
0.448259, 0.725299, -0.522499,
0.485712, 0.785899, -0.382683,
0.414273, 0.536784, -0.735011,
0.382683, 0.485712, -0.785899,
0.443957, 0.584716, -0.678977,
0.485208, 0.517970, -0.704471,
0.552418, 0.496441, -0.669612,
0.522499, 0.448259, -0.725299,
0.470917, 0.628313, -0.619242,
0.581952, 0.540650, -0.607478,
0.619242, 0.470917, -0.628313,
0.540650, 0.607478, -0.581952,
0.607478, 0.581952, -0.540650,
0.628313, 0.619242, -0.470917,
0.669612, 0.552418, -0.496441,
0.496441, 0.669612, -0.552418,
0.678977, 0.443957, -0.584716,
0.649448, 0.399769, -0.646840,
0.704471, 0.485208, -0.517970,
0.735011, 0.414273, -0.536784,
0.785899, 0.382683, -0.485712,
0.760406, 0.341435, -0.552453,
0.725299, 0.522499, -0.448259,
0.517970, 0.704471, -0.485208,
0.536784, 0.735011, -0.414273,
0.584716, 0.678977, -0.443957,
0.646840, 0.649448, -0.399769,
0.552453, 0.760406, -0.341435,
0.415597, 0.427135, -0.803016,
0.486240, 0.408286, -0.772575,
0.446977, 0.366027, -0.816233,
0.476226, 0.303801, -0.825175,
0.553372, 0.386730, -0.737712,
0.545109, 0.280319, -0.790112,
0.504209, 0.237229, -0.830359,
0.583691, 0.322141, -0.745339,
0.610926, 0.255180, -0.749435,
0.672319, 0.228900, -0.703983,
0.634579, 0.187432, -0.749786,
0.619186, 0.361592, -0.697037,
0.529135, 0.172532, -0.830812,
0.594895, 0.147414, -0.790170,
0.551859, 0.106494, -0.827110,
0.678913, 0.334692, -0.653497,
0.734794, 0.305212, -0.605742,
0.705236, 0.267381, -0.656620,
0.605742, 0.734794, -0.305212,
0.656620, 0.705236, -0.267381,
0.653497, 0.678913, -0.334692,
0.697037, 0.619186, -0.361592,
0.703983, 0.672319, -0.228900,
0.745339, 0.583691, -0.322141,
0.737712, 0.553372, -0.386730,
0.749435, 0.610926, -0.255180,
0.790112, 0.545109, -0.280319,
0.830359, 0.504209, -0.237229,
0.825175, 0.476226, -0.303801,
0.749786, 0.634579, -0.187432,
0.772575, 0.486240, -0.408286,
0.816233, 0.446977, -0.366027,
0.803016, 0.415597, -0.427135,
0.790170, 0.594895, -0.147414,
0.827110, 0.551859, -0.106494,
0.830812, 0.529135, -0.172532,
0.551859, -0.106494, 0.827110,
0.529135, -0.172532, 0.830812,
0.594895, -0.147414, 0.790170,
0.634579, -0.187432, 0.749786,
0.504209, -0.237229, 0.830359,
0.610926, -0.255180, 0.749435,
0.672319, -0.228900, 0.703983,
0.545109, -0.280319, 0.790112,
0.583691, -0.322141, 0.745339,
0.553372, -0.386730, 0.737712,
0.619186, -0.361592, 0.697037,
0.476226, -0.303801, 0.825175,
0.705236, -0.267381, 0.656620,
0.678913, -0.334692, 0.653497,
0.734794, -0.305212, 0.605742,
0.760406, -0.341435, 0.552453,
0.649448, -0.399769, 0.646840,
0.446977, -0.366027, 0.816233,
0.415597, -0.427135, 0.803016,
0.486240, -0.408286, 0.772575,
0.522499, -0.448259, 0.725299,
0.382683, -0.485712, 0.785899,
0.735011, -0.414273, 0.536784,
0.785899, -0.382683, 0.485712,
0.678977, -0.443957, 0.584716,
0.704471, -0.485208, 0.517970,
0.669612, -0.552418, 0.496441,
0.725299, -0.522499, 0.448259,
0.619242, -0.470917, 0.628313,
0.607478, -0.581952, 0.540650,
0.628313, -0.619242, 0.470917,
0.581952, -0.540650, 0.607478,
0.540650, -0.607478, 0.581952,
0.470917, -0.628313, 0.619242,
0.496441, -0.669612, 0.552418,
0.552418, -0.496441, 0.669612,
0.584716, -0.678977, 0.443957,
0.646840, -0.649448, 0.399769,
0.517970, -0.704471, 0.485208,
0.536784, -0.735011, 0.414273,
0.485712, -0.785899, 0.382683,
0.552453, -0.760406, 0.341435,
0.448259, -0.725299, 0.522499,
0.485208, -0.517970, 0.704471,
0.414273, -0.536784, 0.735011,
0.443957, -0.584716, 0.678977,
0.399769, -0.646840, 0.649448,
0.341435, -0.552453, 0.760406,
0.803016, -0.415597, 0.427135,
0.772575, -0.486240, 0.408286,
0.816233, -0.446977, 0.366027,
0.825175, -0.476226, 0.303801,
0.737712, -0.553372, 0.386730,
0.790112, -0.545109, 0.280319,
0.830359, -0.504209, 0.237229,
0.745339, -0.583691, 0.322141,
0.749435, -0.610926, 0.255180,
0.703983, -0.672319, 0.228900,
0.749786, -0.634579, 0.187432,
0.697037, -0.619186, 0.361592,
0.830812, -0.529135, 0.172532,
0.790170, -0.594895, 0.147414,
0.827110, -0.551859, 0.106494,
0.653497, -0.678913, 0.334692,
0.605742, -0.734794, 0.305212,
0.656620, -0.705236, 0.267381,
0.305212, -0.605742, 0.734794,
0.267381, -0.656620, 0.705236,
0.334692, -0.653497, 0.678913,
0.361592, -0.697037, 0.619186,
0.228900, -0.703983, 0.672319,
0.322141, -0.745339, 0.583691,
0.386730, -0.737712, 0.553372,
0.255180, -0.749435, 0.610926,
0.280319, -0.790112, 0.545109,
0.237229, -0.830359, 0.504209,
0.303801, -0.825175, 0.476226,
0.187432, -0.749786, 0.634579,
0.408286, -0.772575, 0.486240,
0.366027, -0.816233, 0.446977,
0.427135, -0.803016, 0.415597,
0.147414, -0.790170, 0.594895,
0.106494, -0.827110, 0.551859,
0.172532, -0.830812, 0.529135,
0.551859, -0.106494, -0.827110,
0.594895, -0.147414, -0.790170,
0.529135, -0.172532, -0.830812,
0.504209, -0.237229, -0.830359,
0.634579, -0.187432, -0.749786,
0.545109, -0.280319, -0.790112,
0.476226, -0.303801, -0.825175,
0.610926, -0.255180, -0.749435,
0.583691, -0.322141, -0.745339,
0.619186, -0.361592, -0.697037,
0.553372, -0.386730, -0.737712,
0.672319, -0.228900, -0.703983,
0.446977, -0.366027, -0.816233,
0.486240, -0.408286, -0.772575,
0.415597, -0.427135, -0.803016,
0.382683, -0.485712, -0.785899,
0.522499, -0.448259, -0.725299,
0.705236, -0.267381, -0.656620,
0.734794, -0.305212, -0.605742,
0.678913, -0.334692, -0.653497,
0.649448, -0.399769, -0.646840,
0.760406, -0.341435, -0.552453,
0.414273, -0.536784, -0.735011,
0.341435, -0.552453, -0.760406,
0.485208, -0.517970, -0.704471,
0.443957, -0.584716, -0.678977,
0.470917, -0.628313, -0.619242,
0.399769, -0.646840, -0.649448,
0.552418, -0.496441, -0.669612,
0.540650, -0.607478, -0.581952,
0.496441, -0.669612, -0.552418,
0.581952, -0.540650, -0.607478,
0.607478, -0.581952, -0.540650,
0.669612, -0.552418, -0.496441,
0.628313, -0.619242, -0.470917,
0.619242, -0.470917, -0.628313,
0.517970, -0.704471, -0.485208,
0.448259, -0.725299, -0.522499,
0.584716, -0.678977, -0.443957,
0.536784, -0.735011, -0.414273,
0.552453, -0.760406, -0.341435,
0.485712, -0.785899, -0.382683,
0.646840, -0.649448, -0.399769,
0.678977, -0.443957, -0.584716,
0.735011, -0.414273, -0.536784,
0.704471, -0.485208, -0.517970,
0.725299, -0.522499, -0.448259,
0.785899, -0.382683, -0.485712,
0.305212, -0.605742, -0.734794,
0.334692, -0.653497, -0.678913,
0.267381, -0.656620, -0.705236,
0.228900, -0.703983, -0.672319,
0.361592, -0.697037, -0.619186,
0.255180, -0.749435, -0.610926,
0.187432, -0.749786, -0.634579,
0.322141, -0.745339, -0.583691,
0.280319, -0.790112, -0.545109,
0.303801, -0.825175, -0.476226,
0.237229, -0.830359, -0.504209,
0.386730, -0.737712, -0.553372,
0.147414, -0.790170, -0.594895,
0.172532, -0.830812, -0.529135,
0.106494, -0.827110, -0.551859,
0.408286, -0.772575, -0.486240,
0.427135, -0.803016, -0.415597,
0.366027, -0.816233, -0.446977,
0.803016, -0.415597, -0.427135,
0.816233, -0.446977, -0.366027,
0.772575, -0.486240, -0.408286,
0.737712, -0.553372, -0.386730,
0.825175, -0.476226, -0.303801,
0.745339, -0.583691, -0.322141,
0.697037, -0.619186, -0.361592,
0.790112, -0.545109, -0.280319,
0.749435, -0.610926, -0.255180,
0.749786, -0.634579, -0.187432,
0.703983, -0.672319, -0.228900,
0.830359, -0.504209, -0.237229,
0.653497, -0.678913, -0.334692,
0.656620, -0.705236, -0.267381,
0.605742, -0.734794, -0.305212,
0.830812, -0.529135, -0.172532,
0.827110, -0.551859, -0.106494,
0.790170, -0.594895, -0.147414,
-0.551859, 0.106494, 0.827110,
-0.529135, 0.172532, 0.830812,
-0.594895, 0.147414, 0.790170,
-0.634579, 0.187432, 0.749786,
-0.504209, 0.237229, 0.830359,
-0.610926, 0.255180, 0.749435,
-0.672319, 0.228900, 0.703983,
-0.545109, 0.280319, 0.790112,
-0.583691, 0.322141, 0.745339,
-0.553372, 0.386730, 0.737712,
-0.619186, 0.361592, 0.697037,
-0.476226, 0.303801, 0.825175,
-0.705236, 0.267381, 0.656620,
-0.678913, 0.334692, 0.653497,
-0.734794, 0.305212, 0.605742,
-0.760406, 0.341435, 0.552453,
-0.649448, 0.399769, 0.646840,
-0.446977, 0.366027, 0.816233,
-0.415597, 0.427135, 0.803016,
-0.486240, 0.408286, 0.772575,
-0.522499, 0.448259, 0.725299,
-0.382683, 0.485712, 0.785899,
-0.735011, 0.414273, 0.536784,
-0.785899, 0.382683, 0.485712,
-0.678977, 0.443957, 0.584716,
-0.704471, 0.485208, 0.517970,
-0.669612, 0.552418, 0.496441,
-0.725299, 0.522499, 0.448259,
-0.619242, 0.470917, 0.628313,
-0.607478, 0.581952, 0.540650,
-0.628313, 0.619242, 0.470917,
-0.581952, 0.540650, 0.607478,
-0.540650, 0.607478, 0.581952,
-0.470917, 0.628313, 0.619242,
-0.496441, 0.669612, 0.552418,
-0.552418, 0.496441, 0.669612,
-0.584716, 0.678977, 0.443957,
-0.646840, 0.649448, 0.399769,
-0.517970, 0.704471, 0.485208,
-0.536784, 0.735011, 0.414273,
-0.485712, 0.785899, 0.382683,
-0.552453, 0.760406, 0.341435,
-0.448259, 0.725299, 0.522499,
-0.485208, 0.517970, 0.704471,
-0.414273, 0.536784, 0.735011,
-0.443957, 0.584716, 0.678977,
-0.399769, 0.646840, 0.649448,
-0.341435, 0.552453, 0.760406,
-0.803016, 0.415597, 0.427135,
-0.772575, 0.486240, 0.408286,
-0.816233, 0.446977, 0.366027,
-0.825175, 0.476226, 0.303801,
-0.737712, 0.553372, 0.386730,
-0.790112, 0.545109, 0.280319,
-0.830359, 0.504209, 0.237229,
-0.745339, 0.583691, 0.322141,
-0.749435, 0.610926, 0.255180,
-0.703983, 0.672319, 0.228900,
-0.749786, 0.634579, 0.187432,
-0.697037, 0.619186, 0.361592,
-0.830812, 0.529135, 0.172532,
-0.790170, 0.594895, 0.147414,
-0.827110, 0.551859, 0.106494,
-0.653497, 0.678913, 0.334692,
-0.605742, 0.734794, 0.305212,
-0.656620, 0.705236, 0.267381,
-0.305212, 0.605742, 0.734794,
-0.267381, 0.656620, 0.705236,
-0.334692, 0.653497, 0.678913,
-0.361592, 0.697037, 0.619186,
-0.228900, 0.703983, 0.672319,
-0.322141, 0.745339, 0.583691,
-0.386730, 0.737712, 0.553372,
-0.255180, 0.749435, 0.610926,
-0.280319, 0.790112, 0.545109,
-0.237229, 0.830359, 0.504209,
-0.303801, 0.825175, 0.476226,
-0.187432, 0.749786, 0.634579,
-0.408286, 0.772575, 0.486240,
-0.366027, 0.816233, 0.446977,
-0.427135, 0.803016, 0.415597,
-0.147414, 0.790170, 0.594895,
-0.106494, 0.827110, 0.551859,
-0.172532, 0.830812, 0.529135,
-0.551859, 0.106494, -0.827110,
-0.594895, 0.147414, -0.790170,
-0.529135, 0.172532, -0.830812,
-0.504209, 0.237229, -0.830359,
-0.634579, 0.187432, -0.749786,
-0.545109, 0.280319, -0.790112,
-0.476226, 0.303801, -0.825175,
-0.610926, 0.255180, -0.749435,
-0.583691, 0.322141, -0.745339,
-0.619186, 0.361592, -0.697037,
-0.553372, 0.386730, -0.737712,
-0.672319, 0.228900, -0.703983,
-0.446977, 0.366027, -0.816233,
-0.486240, 0.408286, -0.772575,
-0.415597, 0.427135, -0.803016,
-0.382683, 0.485712, -0.785899,
-0.522499, 0.448259, -0.725299,
-0.705236, 0.267381, -0.656620,
-0.734794, 0.305212, -0.605742,
-0.678913, 0.334692, -0.653497,
-0.649448, 0.399769, -0.646840,
-0.760406, 0.341435, -0.552453,
-0.414273, 0.536784, -0.735011,
-0.341435, 0.552453, -0.760406,
-0.485208, 0.517970, -0.704471,
-0.443957, 0.584716, -0.678977,
-0.470917, 0.628313, -0.619242,
-0.399769, 0.646840, -0.649448,
-0.552418, 0.496441, -0.669612,
-0.540650, 0.607478, -0.581952,
-0.496441, 0.669612, -0.552418,
-0.581952, 0.540650, -0.607478,
-0.607478, 0.581952, -0.540650,
-0.669612, 0.552418, -0.496441,
-0.628313, 0.619242, -0.470917,
-0.619242, 0.470917, -0.628313,
-0.517970, 0.704471, -0.485208,
-0.448259, 0.725299, -0.522499,
-0.584716, 0.678977, -0.443957,
-0.536784, 0.735011, -0.414273,
-0.552453, 0.760406, -0.341435,
-0.485712, 0.785899, -0.382683,
-0.646840, 0.649448, -0.399769,
-0.678977, 0.443957, -0.584716,
-0.735011, 0.414273, -0.536784,
-0.704471, 0.485208, -0.517970,
-0.725299, 0.522499, -0.448259,
-0.785899, 0.382683, -0.485712,
-0.305212, 0.605742, -0.734794,
-0.334692, 0.653497, -0.678913,
-0.267381, 0.656620, -0.705236,
-0.228900, 0.703983, -0.672319,
-0.361592, 0.697037, -0.619186,
-0.255180, 0.749435, -0.610926,
-0.187432, 0.749786, -0.634579,
-0.322141, 0.745339, -0.583691,
-0.280319, 0.790112, -0.545109,
-0.303801, 0.825175, -0.476226,
-0.237229, 0.830359, -0.504209,
-0.386730, 0.737712, -0.553372,
-0.147414, 0.790170, -0.594895,
-0.172532, 0.830812, -0.529135,
-0.106494, 0.827110, -0.551859,
-0.408286, 0.772575, -0.486240,
-0.427135, 0.803016, -0.415597,
-0.366027, 0.816233, -0.446977,
-0.803016, 0.415597, -0.427135,
-0.816233, 0.446977, -0.366027,
-0.772575, 0.486240, -0.408286,
-0.737712, 0.553372, -0.386730,
-0.825175, 0.476226, -0.303801,
-0.745339, 0.583691, -0.322141,
-0.697037, 0.619186, -0.361592,
-0.790112, 0.545109, -0.280319,
-0.749435, 0.610926, -0.255180,
-0.749786, 0.634579, -0.187432,
-0.703983, 0.672319, -0.228900,
-0.830359, 0.504209, -0.237229,
-0.653497, 0.678913, -0.334692,
-0.656620, 0.705236, -0.267381,
-0.605742, 0.734794, -0.305212,
-0.830812, 0.529135, -0.172532,
-0.827110, 0.551859, -0.106494,
-0.790170, 0.594895, -0.147414,
-0.551859, -0.106494, 0.827110,
-0.594895, -0.147414, 0.790170,
-0.529135, -0.172532, 0.830812,
-0.504209, -0.237229, 0.830359,
-0.634579, -0.187432, 0.749786,
-0.545109, -0.280319, 0.790112,
-0.476226, -0.303801, 0.825175,
-0.610926, -0.255180, 0.749435,
-0.583691, -0.322141, 0.745339,
-0.619186, -0.361592, 0.697037,
-0.553372, -0.386730, 0.737712,
-0.672319, -0.228900, 0.703983,
-0.446977, -0.366027, 0.816233,
-0.486240, -0.408286, 0.772575,
-0.415597, -0.427135, 0.803016,
-0.382683, -0.485712, 0.785899,
-0.522499, -0.448259, 0.725299,
-0.705236, -0.267381, 0.656620,
-0.734794, -0.305212, 0.605742,
-0.678913, -0.334692, 0.653497,
-0.649448, -0.399769, 0.646840,
-0.760406, -0.341435, 0.552453,
-0.414273, -0.536784, 0.735011,
-0.341435, -0.552453, 0.760406,
-0.485208, -0.517970, 0.704471,
-0.443957, -0.584716, 0.678977,
-0.470917, -0.628313, 0.619242,
-0.399769, -0.646840, 0.649448,
-0.552418, -0.496441, 0.669612,
-0.540650, -0.607478, 0.581952,
-0.496441, -0.669612, 0.552418,
-0.581952, -0.540650, 0.607478,
-0.607478, -0.581952, 0.540650,
-0.669612, -0.552418, 0.496441,
-0.628313, -0.619242, 0.470917,
-0.619242, -0.470917, 0.628313,
-0.517970, -0.704471, 0.485208,
-0.448259, -0.725299, 0.522499,
-0.584716, -0.678977, 0.443957,
-0.536784, -0.735011, 0.414273,
-0.552453, -0.760406, 0.341435,
-0.485712, -0.785899, 0.382683,
-0.646840, -0.649448, 0.399769,
-0.678977, -0.443957, 0.584716,
-0.735011, -0.414273, 0.536784,
-0.704471, -0.485208, 0.517970,
-0.725299, -0.522499, 0.448259,
-0.785899, -0.382683, 0.485712,
-0.305212, -0.605742, 0.734794,
-0.334692, -0.653497, 0.678913,
-0.267381, -0.656620, 0.705236,
-0.228900, -0.703983, 0.672319,
-0.361592, -0.697037, 0.619186,
-0.255180, -0.749435, 0.610926,
-0.187432, -0.749786, 0.634579,
-0.322141, -0.745339, 0.583691,
-0.280319, -0.790112, 0.545109,
-0.303801, -0.825175, 0.476226,
-0.237229, -0.830359, 0.504209,
-0.386730, -0.737712, 0.553372,
-0.147414, -0.790170, 0.594895,
-0.172532, -0.830812, 0.529135,
-0.106494, -0.827110, 0.551859,
-0.408286, -0.772575, 0.486240,
-0.427135, -0.803016, 0.415597,
-0.366027, -0.816233, 0.446977,
-0.803016, -0.415597, 0.427135,
-0.816233, -0.446977, 0.366027,
-0.772575, -0.486240, 0.408286,
-0.737712, -0.553372, 0.386730,
-0.825175, -0.476226, 0.303801,
-0.745339, -0.583691, 0.322141,
-0.697037, -0.619186, 0.361592,
-0.790112, -0.545109, 0.280319,
-0.749435, -0.610926, 0.255180,
-0.749786, -0.634579, 0.187432,
-0.703983, -0.672319, 0.228900,
-0.830359, -0.504209, 0.237229,
-0.653497, -0.678913, 0.334692,
-0.656620, -0.705236, 0.267381,
-0.605742, -0.734794, 0.305212,
-0.830812, -0.529135, 0.172532,
-0.827110, -0.551859, 0.106494,
-0.790170, -0.594895, 0.147414,
-0.551859, -0.106494, -0.827110,
-0.529135, -0.172532, -0.830812,
-0.594895, -0.147414, -0.790170,
-0.634579, -0.187432, -0.749786,
-0.504209, -0.237229, -0.830359,
-0.610926, -0.255180, -0.749435,
-0.672319, -0.228900, -0.703983,
-0.545109, -0.280319, -0.790112,
-0.583691, -0.322141, -0.745339,
-0.553372, -0.386730, -0.737712,
-0.619186, -0.361592, -0.697037,
-0.476226, -0.303801, -0.825175,
-0.705236, -0.267381, -0.656620,
-0.678913, -0.334692, -0.653497,
-0.734794, -0.305212, -0.605742,
-0.760406, -0.341435, -0.552453,
-0.649448, -0.399769, -0.646840,
-0.446977, -0.366027, -0.816233,
-0.415597, -0.427135, -0.803016,
-0.486240, -0.408286, -0.772575,
-0.522499, -0.448259, -0.725299,
-0.382683, -0.485712, -0.785899,
-0.735011, -0.414273, -0.536784,
-0.785899, -0.382683, -0.485712,
-0.678977, -0.443957, -0.584716,
-0.704471, -0.485208, -0.517970,
-0.669612, -0.552418, -0.496441,
-0.725299, -0.522499, -0.448259,
-0.619242, -0.470917, -0.628313,
-0.607478, -0.581952, -0.540650,
-0.628313, -0.619242, -0.470917,
-0.581952, -0.540650, -0.607478,
-0.540650, -0.607478, -0.581952,
-0.470917, -0.628313, -0.619242,
-0.496441, -0.669612, -0.552418,
-0.552418, -0.496441, -0.669612,
-0.584716, -0.678977, -0.443957,
-0.646840, -0.649448, -0.399769,
-0.517970, -0.704471, -0.485208,
-0.536784, -0.735011, -0.414273,
-0.485712, -0.785899, -0.382683,
-0.552453, -0.760406, -0.341435,
-0.448259, -0.725299, -0.522499,
-0.485208, -0.517970, -0.704471,
-0.414273, -0.536784, -0.735011,
-0.443957, -0.584716, -0.678977,
-0.399769, -0.646840, -0.649448,
-0.341435, -0.552453, -0.760406,
-0.803016, -0.415597, -0.427135,
-0.772575, -0.486240, -0.408286,
-0.816233, -0.446977, -0.366027,
-0.825175, -0.476226, -0.303801,
-0.737712, -0.553372, -0.386730,
-0.790112, -0.545109, -0.280319,
-0.830359, -0.504209, -0.237229,
-0.745339, -0.583691, -0.322141,
-0.749435, -0.610926, -0.255180,
-0.703983, -0.672319, -0.228900,
-0.749786, -0.634579, -0.187432,
-0.697037, -0.619186, -0.361592,
-0.830812, -0.529135, -0.172532,
-0.790170, -0.594895, -0.147414,
-0.827110, -0.551859, -0.106494,
-0.653497, -0.678913, -0.334692,
-0.605742, -0.734794, -0.305212,
-0.656620, -0.705236, -0.267381,
-0.305212, -0.605742, -0.734794,
-0.267381, -0.656620, -0.705236,
-0.334692, -0.653497, -0.678913,
-0.361592, -0.697037, -0.619186,
-0.228900, -0.703983, -0.672319,
-0.322141, -0.745339, -0.583691,
-0.386730, -0.737712, -0.553372,
-0.255180, -0.749435, -0.610926,
-0.280319, -0.790112, -0.545109,
-0.237229, -0.830359, -0.504209,
-0.303801, -0.825175, -0.476226,
-0.187432, -0.749786, -0.634579,
-0.408286, -0.772575, -0.486240,
-0.366027, -0.816233, -0.446977,
-0.427135, -0.803016, -0.415597,
-0.147414, -0.790170, -0.594895,
-0.106494, -0.827110, -0.551859,
-0.172532, -0.830812, -0.529135};

    Points << v;
    Points << Points.t();
    Points.Release();
    return Points;
  }


  //5th order tessellation
  ReturnMatrix tessellation_points5() const{
    Matrix Points(10242,3);
    Real v[]={0.850651, 0.525731, 0.000000,
-0.850651, 0.525731, 0.000000,
-0.850651, -0.525731, 0.000000,
0.850651, -0.525731, 0.000000,
0.525731, 0.000000, 0.850651,
0.525731, 0.000000, -0.850651,
-0.525731, 0.000000, -0.850651,
-0.525731, 0.000000, 0.850651,
0.000000, 0.850651, 0.525731,
0.000000, -0.850651, 0.525731,
0.000000, -0.850651, -0.525731,
0.000000, 0.850651, -0.525731,
0.309017, 0.500000, 0.809017,
-0.309017, 0.500000, 0.809017,
0.000000, 0.000000, 1.000000,
-0.309017, -0.500000, 0.809017,
0.309017, -0.500000, 0.809017,
0.000000, 0.000000, -1.000000,
-0.309017, 0.500000, -0.809017,
0.309017, 0.500000, -0.809017,
0.309017, -0.500000, -0.809017,
-0.309017, -0.500000, -0.809017,
0.809017, 0.309017, 0.500000,
0.809017, -0.309017, 0.500000,
1.000000, 0.000000, 0.000000,
0.809017, -0.309017, -0.500000,
0.809017, 0.309017, -0.500000,
-0.809017, -0.309017, 0.500000,
-0.809017, 0.309017, 0.500000,
-1.000000, 0.000000, 0.000000,
-0.809017, 0.309017, -0.500000,
-0.809017, -0.309017, -0.500000,
0.500000, 0.809017, 0.309017,
0.500000, 0.809017, -0.309017,
0.000000, 1.000000, 0.000000,
-0.500000, 0.809017, -0.309017,
-0.500000, 0.809017, 0.309017,
0.000000, -1.000000, 0.000000,
0.500000, -0.809017, -0.309017,
0.500000, -0.809017, 0.309017,
-0.500000, -0.809017, 0.309017,
-0.500000, -0.809017, -0.309017,
0.433889, 0.259892, 0.862669,
0.162460, 0.262866, 0.951057,
0.273267, 0.000000, 0.961938,
0.000000, 0.525731, 0.850651,
-0.162460, 0.262866, 0.951057,
-0.433889, 0.259892, 0.862669,
-0.273267, 0.000000, 0.961938,
0.160622, 0.702046, 0.693780,
-0.160622, 0.702046, 0.693780,
0.162460, -0.262866, 0.951057,
0.433889, -0.259892, 0.862669,
-0.162460, -0.262866, 0.951057,
0.000000, -0.525731, 0.850651,
-0.160622, -0.702046, 0.693780,
0.160622, -0.702046, 0.693780,
-0.433889, -0.259892, 0.862669,
0.273267, 0.000000, -0.961938,
0.162460, 0.262866, -0.951057,
0.433889, 0.259892, -0.862669,
-0.162460, 0.262866, -0.951057,
0.000000, 0.525731, -0.850651,
-0.160622, 0.702046, -0.693780,
0.160622, 0.702046, -0.693780,
-0.273267, 0.000000, -0.961938,
-0.433889, 0.259892, -0.862669,
0.433889, -0.259892, -0.862669,
0.162460, -0.262866, -0.951057,
0.000000, -0.525731, -0.850651,
-0.162460, -0.262866, -0.951057,
-0.433889, -0.259892, -0.862669,
0.160622, -0.702046, -0.693780,
-0.160622, -0.702046, -0.693780,
0.862669, 0.433889, 0.259892,
0.951057, 0.162460, 0.262866,
0.961938, 0.273267, 0.000000,
0.850651, 0.000000, 0.525731,
0.951057, -0.162460, 0.262866,
0.862669, -0.433889, 0.259892,
0.961938, -0.273267, 0.000000,
0.693780, 0.160622, 0.702046,
0.693780, -0.160622, 0.702046,
0.951057, 0.162460, -0.262866,
0.862669, 0.433889, -0.259892,
0.951057, -0.162460, -0.262866,
0.850651, 0.000000, -0.525731,
0.693780, -0.160622, -0.702046,
0.693780, 0.160622, -0.702046,
0.862669, -0.433889, -0.259892,
-0.862669, -0.433889, 0.259892,
-0.951057, -0.162460, 0.262866,
-0.961938, -0.273267, 0.000000,
-0.850651, 0.000000, 0.525731,
-0.951057, 0.162460, 0.262866,
-0.862669, 0.433889, 0.259892,
-0.961938, 0.273267, 0.000000,
-0.693780, -0.160622, 0.702046,
-0.693780, 0.160622, 0.702046,
-0.951057, -0.162460, -0.262866,
-0.862669, -0.433889, -0.259892,
-0.951057, 0.162460, -0.262866,
-0.850651, 0.000000, -0.525731,
-0.693780, 0.160622, -0.702046,
-0.693780, -0.160622, -0.702046,
-0.862669, 0.433889, -0.259892,
0.259892, 0.862669, 0.433889,
0.262866, 0.951057, 0.162460,
0.000000, 0.961938, 0.273267,
0.525731, 0.850651, 0.000000,
0.262866, 0.951057, -0.162460,
0.259892, 0.862669, -0.433889,
0.000000, 0.961938, -0.273267,
0.702046, 0.693780, 0.160622,
0.702046, 0.693780, -0.160622,
-0.262866, 0.951057, 0.162460,
-0.259892, 0.862669, 0.433889,
-0.262866, 0.951057, -0.162460,
-0.525731, 0.850651, 0.000000,
-0.702046, 0.693780, -0.160622,
-0.702046, 0.693780, 0.160622,
-0.259892, 0.862669, -0.433889,
0.000000, -0.961938, 0.273267,
0.262866, -0.951057, 0.162460,
0.259892, -0.862669, 0.433889,
0.262866, -0.951057, -0.162460,
0.525731, -0.850651, 0.000000,
0.702046, -0.693780, -0.160622,
0.702046, -0.693780, 0.160622,
0.000000, -0.961938, -0.273267,
0.259892, -0.862669, -0.433889,
-0.259892, -0.862669, 0.433889,
-0.262866, -0.951057, 0.162460,
-0.525731, -0.850651, 0.000000,
-0.262866, -0.951057, -0.162460,
-0.259892, -0.862669, -0.433889,
-0.702046, -0.693780, 0.160622,
-0.702046, -0.693780, -0.160622,
0.425325, 0.688191, 0.587785,
0.587785, 0.425325, 0.688191,
0.688191, 0.587785, 0.425325,
0.425325, 0.688191, -0.587785,
0.688191, 0.587785, -0.425325,
0.587785, 0.425325, -0.688191,
0.587785, -0.425325, 0.688191,
0.425325, -0.688191, 0.587785,
0.688191, -0.587785, 0.425325,
0.587785, -0.425325, -0.688191,
0.688191, -0.587785, -0.425325,
0.425325, -0.688191, -0.587785,
-0.587785, 0.425325, 0.688191,
-0.425325, 0.688191, 0.587785,
-0.688191, 0.587785, 0.425325,
-0.587785, 0.425325, -0.688191,
-0.688191, 0.587785, -0.425325,
-0.425325, 0.688191, -0.587785,
-0.587785, -0.425325, 0.688191,
-0.688191, -0.587785, 0.425325,
-0.425325, -0.688191, 0.587785,
-0.587785, -0.425325, -0.688191,
-0.425325, -0.688191, -0.587785,
-0.688191, -0.587785, -0.425325,
0.484442, 0.131200, 0.864929,
0.358229, 0.131655, 0.924305,
0.403355, 0.000000, 0.915043,
0.301259, 0.264083, 0.916244,
0.220117, 0.132792, 0.966393,
0.082242, 0.133071, 0.987688,
0.137952, 0.000000, 0.990439,
0.375039, 0.383614, 0.843911,
0.238677, 0.386187, 0.891006,
0.000000, 0.266405, 0.963861,
-0.082242, 0.133071, 0.987688,
0.082324, 0.399607, 0.912982,
-0.082324, 0.399607, 0.912982,
-0.156434, 0.519258, 0.840178,
-0.238677, 0.386187, 0.891006,
0.156434, 0.519258, 0.840178,
-0.220117, 0.132792, 0.966393,
-0.137952, 0.000000, 0.990439,
-0.301259, 0.264083, 0.916244,
-0.358229, 0.131655, 0.924305,
-0.484442, 0.131200, 0.864929,
-0.403355, 0.000000, 0.915043,
-0.375039, 0.383614, 0.843911,
0.237086, 0.606825, 0.758652,
0.081142, 0.620240, 0.780204,
0.000000, 0.711282, 0.702907,
-0.081142, 0.620240, 0.780204,
-0.237086, 0.606825, 0.758652,
0.081086, 0.783843, 0.615642,
-0.081086, 0.783843, 0.615642,
0.358229, -0.131655, 0.924305,
0.484442, -0.131200, 0.864929,
0.220117, -0.132792, 0.966393,
0.301259, -0.264083, 0.916244,
0.238677, -0.386187, 0.891006,
0.375039, -0.383614, 0.843911,
0.082242, -0.133071, 0.987688,
0.082324, -0.399607, 0.912982,
0.156434, -0.519258, 0.840178,
0.000000, -0.266405, 0.963861,
-0.082324, -0.399607, 0.912982,
-0.238677, -0.386187, 0.891006,
-0.156434, -0.519258, 0.840178,
-0.082242, -0.133071, 0.987688,
0.081142, -0.620240, 0.780204,
0.237086, -0.606825, 0.758652,
-0.081142, -0.620240, 0.780204,
0.000000, -0.711282, 0.702907,
-0.081086, -0.783843, 0.615642,
0.081086, -0.783843, 0.615642,
-0.237086, -0.606825, 0.758652,
-0.220117, -0.132792, 0.966393,
-0.358229, -0.131655, 0.924305,
-0.301259, -0.264083, 0.916244,
-0.375039, -0.383614, 0.843911,
-0.484442, -0.131200, 0.864929,
0.403355, 0.000000, -0.915043,
0.358229, 0.131655, -0.924305,
0.484442, 0.131200, -0.864929,
0.220117, 0.132792, -0.966393,
0.301259, 0.264083, -0.916244,
0.238677, 0.386187, -0.891006,
0.375039, 0.383614, -0.843911,
0.137952, 0.000000, -0.990439,
0.082242, 0.133071, -0.987688,
0.082324, 0.399607, -0.912982,
0.156434, 0.519258, -0.840178,
0.000000, 0.266405, -0.963861,
-0.082324, 0.399607, -0.912982,
-0.238677, 0.386187, -0.891006,
-0.156434, 0.519258, -0.840178,
-0.082242, 0.133071, -0.987688,
0.081142, 0.620240, -0.780204,
0.237086, 0.606825, -0.758652,
-0.081142, 0.620240, -0.780204,
0.000000, 0.711282, -0.702907,
-0.081086, 0.783843, -0.615642,
0.081086, 0.783843, -0.615642,
-0.237086, 0.606825, -0.758652,
-0.137952, 0.000000, -0.990439,
-0.220117, 0.132792, -0.966393,
-0.358229, 0.131655, -0.924305,
-0.301259, 0.264083, -0.916244,
-0.375039, 0.383614, -0.843911,
-0.403355, 0.000000, -0.915043,
-0.484442, 0.131200, -0.864929,
0.484442, -0.131200, -0.864929,
0.358229, -0.131655, -0.924305,
0.301259, -0.264083, -0.916244,
0.220117, -0.132792, -0.966393,
0.082242, -0.133071, -0.987688,
0.375039, -0.383614, -0.843911,
0.238677, -0.386187, -0.891006,
0.000000, -0.266405, -0.963861,
-0.082242, -0.133071, -0.987688,
0.082324, -0.399607, -0.912982,
-0.082324, -0.399607, -0.912982,
-0.156434, -0.519258, -0.840178,
-0.238677, -0.386187, -0.891006,
0.156434, -0.519258, -0.840178,
-0.220117, -0.132792, -0.966393,
-0.301259, -0.264083, -0.916244,
-0.358229, -0.131655, -0.924305,
-0.484442, -0.131200, -0.864929,
-0.375039, -0.383614, -0.843911,
0.237086, -0.606825, -0.758652,
0.081142, -0.620240, -0.780204,
0.000000, -0.711282, -0.702907,
-0.081142, -0.620240, -0.780204,
-0.237086, -0.606825, -0.758652,
0.081086, -0.783843, -0.615642,
-0.081086, -0.783843, -0.615642,
0.864929, 0.484442, 0.131200,
0.924305, 0.358229, 0.131655,
0.915043, 0.403355, 0.000000,
0.916244, 0.301259, 0.264083,
0.966393, 0.220117, 0.132792,
0.987688, 0.082242, 0.133071,
0.990439, 0.137952, 0.000000,
0.843911, 0.375039, 0.383614,
0.891006, 0.238677, 0.386187,
0.963861, 0.000000, 0.266405,
0.987688, -0.082242, 0.133071,
0.912982, 0.082324, 0.399607,
0.912982, -0.082324, 0.399607,
0.840178, -0.156434, 0.519258,
0.891006, -0.238677, 0.386187,
0.840178, 0.156434, 0.519258,
0.966393, -0.220117, 0.132792,
0.990439, -0.137952, 0.000000,
0.916244, -0.301259, 0.264083,
0.924305, -0.358229, 0.131655,
0.864929, -0.484442, 0.131200,
0.915043, -0.403355, 0.000000,
0.843911, -0.375039, 0.383614,
0.758652, 0.237086, 0.606825,
0.780204, 0.081142, 0.620240,
0.702907, 0.000000, 0.711282,
0.780204, -0.081142, 0.620240,
0.758652, -0.237086, 0.606825,
0.615642, 0.081086, 0.783843,
0.615642, -0.081086, 0.783843,
0.924305, 0.358229, -0.131655,
0.864929, 0.484442, -0.131200,
0.966393, 0.220117, -0.132792,
0.916244, 0.301259, -0.264083,
0.891006, 0.238677, -0.386187,
0.843911, 0.375039, -0.383614,
0.987688, 0.082242, -0.133071,
0.912982, 0.082324, -0.399607,
0.840178, 0.156434, -0.519258,
0.963861, 0.000000, -0.266405,
0.912982, -0.082324, -0.399607,
0.891006, -0.238677, -0.386187,
0.840178, -0.156434, -0.519258,
0.987688, -0.082242, -0.133071,
0.780204, 0.081142, -0.620240,
0.758652, 0.237086, -0.606825,
0.780204, -0.081142, -0.620240,
0.702907, 0.000000, -0.711282,
0.615642, -0.081086, -0.783843,
0.615642, 0.081086, -0.783843,
0.758652, -0.237086, -0.606825,
0.966393, -0.220117, -0.132792,
0.924305, -0.358229, -0.131655,
0.916244, -0.301259, -0.264083,
0.843911, -0.375039, -0.383614,
0.864929, -0.484442, -0.131200,
-0.864929, -0.484442, 0.131200,
-0.924305, -0.358229, 0.131655,
-0.915043, -0.403355, 0.000000,
-0.916244, -0.301259, 0.264083,
-0.966393, -0.220117, 0.132792,
-0.987688, -0.082242, 0.133071,
-0.990439, -0.137952, 0.000000,
-0.843911, -0.375039, 0.383614,
-0.891006, -0.238677, 0.386187,
-0.963861, 0.000000, 0.266405,
-0.987688, 0.082242, 0.133071,
-0.912982, -0.082324, 0.399607,
-0.912982, 0.082324, 0.399607,
-0.840178, 0.156434, 0.519258,
-0.891006, 0.238677, 0.386187,
-0.840178, -0.156434, 0.519258,
-0.966393, 0.220117, 0.132792,
-0.990439, 0.137952, 0.000000,
-0.916244, 0.301259, 0.264083,
-0.924305, 0.358229, 0.131655,
-0.864929, 0.484442, 0.131200,
-0.915043, 0.403355, 0.000000,
-0.843911, 0.375039, 0.383614,
-0.758652, -0.237086, 0.606825,
-0.780204, -0.081142, 0.620240,
-0.702907, 0.000000, 0.711282,
-0.780204, 0.081142, 0.620240,
-0.758652, 0.237086, 0.606825,
-0.615642, -0.081086, 0.783843,
-0.615642, 0.081086, 0.783843,
-0.924305, -0.358229, -0.131655,
-0.864929, -0.484442, -0.131200,
-0.966393, -0.220117, -0.132792,
-0.916244, -0.301259, -0.264083,
-0.891006, -0.238677, -0.386187,
-0.843911, -0.375039, -0.383614,
-0.987688, -0.082242, -0.133071,
-0.912982, -0.082324, -0.399607,
-0.840178, -0.156434, -0.519258,
-0.963861, 0.000000, -0.266405,
-0.912982, 0.082324, -0.399607,
-0.891006, 0.238677, -0.386187,
-0.840178, 0.156434, -0.519258,
-0.987688, 0.082242, -0.133071,
-0.780204, -0.081142, -0.620240,
-0.758652, -0.237086, -0.606825,
-0.780204, 0.081142, -0.620240,
-0.702907, 0.000000, -0.711282,
-0.615642, 0.081086, -0.783843,
-0.615642, -0.081086, -0.783843,
-0.758652, 0.237086, -0.606825,
-0.966393, 0.220117, -0.132792,
-0.924305, 0.358229, -0.131655,
-0.916244, 0.301259, -0.264083,
-0.843911, 0.375039, -0.383614,
-0.864929, 0.484442, -0.131200,
0.131200, 0.864929, 0.484442,
0.131655, 0.924305, 0.358229,
0.000000, 0.915043, 0.403355,
0.264083, 0.916244, 0.301259,
0.132792, 0.966393, 0.220117,
0.133071, 0.987688, 0.082242,
0.000000, 0.990439, 0.137952,
0.383614, 0.843911, 0.375039,
0.386187, 0.891006, 0.238677,
0.266405, 0.963861, 0.000000,
0.133071, 0.987688, -0.082242,
0.399607, 0.912982, 0.082324,
0.399607, 0.912982, -0.082324,
0.519258, 0.840178, -0.156434,
0.386187, 0.891006, -0.238677,
0.519258, 0.840178, 0.156434,
0.132792, 0.966393, -0.220117,
0.000000, 0.990439, -0.137952,
0.264083, 0.916244, -0.301259,
0.131655, 0.924305, -0.358229,
0.131200, 0.864929, -0.484442,
0.000000, 0.915043, -0.403355,
0.383614, 0.843911, -0.375039,
0.606825, 0.758652, 0.237086,
0.620240, 0.780204, 0.081142,
0.711282, 0.702907, 0.000000,
0.620240, 0.780204, -0.081142,
0.606825, 0.758652, -0.237086,
0.783843, 0.615642, 0.081086,
0.783843, 0.615642, -0.081086,
-0.131655, 0.924305, 0.358229,
-0.131200, 0.864929, 0.484442,
-0.132792, 0.966393, 0.220117,
-0.264083, 0.916244, 0.301259,
-0.386187, 0.891006, 0.238677,
-0.383614, 0.843911, 0.375039,
-0.133071, 0.987688, 0.082242,
-0.399607, 0.912982, 0.082324,
-0.519258, 0.840178, 0.156434,
-0.266405, 0.963861, 0.000000,
-0.399607, 0.912982, -0.082324,
-0.386187, 0.891006, -0.238677,
-0.519258, 0.840178, -0.156434,
-0.133071, 0.987688, -0.082242,
-0.620240, 0.780204, 0.081142,
-0.606825, 0.758652, 0.237086,
-0.620240, 0.780204, -0.081142,
-0.711282, 0.702907, 0.000000,
-0.783843, 0.615642, -0.081086,
-0.783843, 0.615642, 0.081086,
-0.606825, 0.758652, -0.237086,
-0.132792, 0.966393, -0.220117,
-0.131655, 0.924305, -0.358229,
-0.264083, 0.916244, -0.301259,
-0.383614, 0.843911, -0.375039,
-0.131200, 0.864929, -0.484442,
0.000000, -0.915043, 0.403355,
0.131655, -0.924305, 0.358229,
0.131200, -0.864929, 0.484442,
0.132792, -0.966393, 0.220117,
0.264083, -0.916244, 0.301259,
0.386187, -0.891006, 0.238677,
0.383614, -0.843911, 0.375039,
0.000000, -0.990439, 0.137952,
0.133071, -0.987688, 0.082242,
0.399607, -0.912982, 0.082324,
0.519258, -0.840178, 0.156434,
0.266405, -0.963861, 0.000000,
0.399607, -0.912982, -0.082324,
0.386187, -0.891006, -0.238677,
0.519258, -0.840178, -0.156434,
0.133071, -0.987688, -0.082242,
0.620240, -0.780204, 0.081142,
0.606825, -0.758652, 0.237086,
0.620240, -0.780204, -0.081142,
0.711282, -0.702907, 0.000000,
0.783843, -0.615642, -0.081086,
0.783843, -0.615642, 0.081086,
0.606825, -0.758652, -0.237086,
0.000000, -0.990439, -0.137952,
0.132792, -0.966393, -0.220117,
0.131655, -0.924305, -0.358229,
0.264083, -0.916244, -0.301259,
0.383614, -0.843911, -0.375039,
0.000000, -0.915043, -0.403355,
0.131200, -0.864929, -0.484442,
-0.131200, -0.864929, 0.484442,
-0.131655, -0.924305, 0.358229,
-0.264083, -0.916244, 0.301259,
-0.132792, -0.966393, 0.220117,
-0.133071, -0.987688, 0.082242,
-0.383614, -0.843911, 0.375039,
-0.386187, -0.891006, 0.238677,
-0.266405, -0.963861, 0.000000,
-0.133071, -0.987688, -0.082242,
-0.399607, -0.912982, 0.082324,
-0.399607, -0.912982, -0.082324,
-0.519258, -0.840178, -0.156434,
-0.386187, -0.891006, -0.238677,
-0.519258, -0.840178, 0.156434,
-0.132792, -0.966393, -0.220117,
-0.264083, -0.916244, -0.301259,
-0.131655, -0.924305, -0.358229,
-0.131200, -0.864929, -0.484442,
-0.383614, -0.843911, -0.375039,
-0.606825, -0.758652, 0.237086,
-0.620240, -0.780204, 0.081142,
-0.711282, -0.702907, 0.000000,
-0.620240, -0.780204, -0.081142,
-0.606825, -0.758652, -0.237086,
-0.783843, -0.615642, 0.081086,
-0.783843, -0.615642, -0.081086,
0.213023, 0.792649, 0.571252,
0.296005, 0.702310, 0.647412,
0.346153, 0.783452, 0.516122,
0.468430, 0.757935, 0.453990,
0.371748, 0.601501, 0.707107,
0.564254, 0.646578, 0.513375,
0.601501, 0.707107, 0.371748,
0.513375, 0.564254, 0.646578,
0.646578, 0.513375, 0.564254,
0.707107, 0.371748, 0.601501,
0.757935, 0.453990, 0.468430,
0.453990, 0.468430, 0.757935,
0.702310, 0.647412, 0.296005,
0.783452, 0.516122, 0.346153,
0.792649, 0.571252, 0.213023,
0.516122, 0.346153, 0.783452,
0.571252, 0.213023, 0.792649,
0.647412, 0.296005, 0.702310,
0.213023, 0.792649, -0.571252,
0.346153, 0.783452, -0.516122,
0.296005, 0.702310, -0.647412,
0.371748, 0.601501, -0.707107,
0.468430, 0.757935, -0.453990,
0.513375, 0.564254, -0.646578,
0.453990, 0.468430, -0.757935,
0.564254, 0.646578, -0.513375,
0.646578, 0.513375, -0.564254,
0.757935, 0.453990, -0.468430,
0.707107, 0.371748, -0.601501,
0.601501, 0.707107, -0.371748,
0.516122, 0.346153, -0.783452,
0.647412, 0.296005, -0.702310,
0.571252, 0.213023, -0.792649,
0.702310, 0.647412, -0.296005,
0.792649, 0.571252, -0.213023,
0.783452, 0.516122, -0.346153,
0.571252, -0.213023, 0.792649,
0.516122, -0.346153, 0.783452,
0.647412, -0.296005, 0.702310,
0.707107, -0.371748, 0.601501,
0.453990, -0.468430, 0.757935,
0.646578, -0.513375, 0.564254,
0.757935, -0.453990, 0.468430,
0.513375, -0.564254, 0.646578,
0.564254, -0.646578, 0.513375,
0.468430, -0.757935, 0.453990,
0.601501, -0.707107, 0.371748,
0.371748, -0.601501, 0.707107,
0.783452, -0.516122, 0.346153,
0.702310, -0.647412, 0.296005,
0.792649, -0.571252, 0.213023,
0.296005, -0.702310, 0.647412,
0.213023, -0.792649, 0.571252,
0.346153, -0.783452, 0.516122,
0.571252, -0.213023, -0.792649,
0.647412, -0.296005, -0.702310,
0.516122, -0.346153, -0.783452,
0.453990, -0.468430, -0.757935,
0.707107, -0.371748, -0.601501,
0.513375, -0.564254, -0.646578,
0.371748, -0.601501, -0.707107,
0.646578, -0.513375, -0.564254,
0.564254, -0.646578, -0.513375,
0.601501, -0.707107, -0.371748,
0.468430, -0.757935, -0.453990,
0.757935, -0.453990, -0.468430,
0.296005, -0.702310, -0.647412,
0.346153, -0.783452, -0.516122,
0.213023, -0.792649, -0.571252,
0.783452, -0.516122, -0.346153,
0.792649, -0.571252, -0.213023,
0.702310, -0.647412, -0.296005,
-0.571252, 0.213023, 0.792649,
-0.516122, 0.346153, 0.783452,
-0.647412, 0.296005, 0.702310,
-0.707107, 0.371748, 0.601501,
-0.453990, 0.468430, 0.757935,
-0.646578, 0.513375, 0.564254,
-0.757935, 0.453990, 0.468430,
-0.513375, 0.564254, 0.646578,
-0.564254, 0.646578, 0.513375,
-0.468430, 0.757935, 0.453990,
-0.601501, 0.707107, 0.371748,
-0.371748, 0.601501, 0.707107,
-0.783452, 0.516122, 0.346153,
-0.702310, 0.647412, 0.296005,
-0.792649, 0.571252, 0.213023,
-0.296005, 0.702310, 0.647412,
-0.213023, 0.792649, 0.571252,
-0.346153, 0.783452, 0.516122,
-0.571252, 0.213023, -0.792649,
-0.647412, 0.296005, -0.702310,
-0.516122, 0.346153, -0.783452,
-0.453990, 0.468430, -0.757935,
-0.707107, 0.371748, -0.601501,
-0.513375, 0.564254, -0.646578,
-0.371748, 0.601501, -0.707107,
-0.646578, 0.513375, -0.564254,
-0.564254, 0.646578, -0.513375,
-0.601501, 0.707107, -0.371748,
-0.468430, 0.757935, -0.453990,
-0.757935, 0.453990, -0.468430,
-0.296005, 0.702310, -0.647412,
-0.346153, 0.783452, -0.516122,
-0.213023, 0.792649, -0.571252,
-0.783452, 0.516122, -0.346153,
-0.792649, 0.571252, -0.213023,
-0.702310, 0.647412, -0.296005,
-0.571252, -0.213023, 0.792649,
-0.647412, -0.296005, 0.702310,
-0.516122, -0.346153, 0.783452,
-0.453990, -0.468430, 0.757935,
-0.707107, -0.371748, 0.601501,
-0.513375, -0.564254, 0.646578,
-0.371748, -0.601501, 0.707107,
-0.646578, -0.513375, 0.564254,
-0.564254, -0.646578, 0.513375,
-0.601501, -0.707107, 0.371748,
-0.468430, -0.757935, 0.453990,
-0.757935, -0.453990, 0.468430,
-0.296005, -0.702310, 0.647412,
-0.346153, -0.783452, 0.516122,
-0.213023, -0.792649, 0.571252,
-0.783452, -0.516122, 0.346153,
-0.792649, -0.571252, 0.213023,
-0.702310, -0.647412, 0.296005,
-0.571252, -0.213023, -0.792649,
-0.516122, -0.346153, -0.783452,
-0.647412, -0.296005, -0.702310,
-0.707107, -0.371748, -0.601501,
-0.453990, -0.468430, -0.757935,
-0.646578, -0.513375, -0.564254,
-0.757935, -0.453990, -0.468430,
-0.513375, -0.564254, -0.646578,
-0.564254, -0.646578, -0.513375,
-0.468430, -0.757935, -0.453990,
-0.601501, -0.707107, -0.371748,
-0.371748, -0.601501, -0.707107,
-0.783452, -0.516122, -0.346153,
-0.702310, -0.647412, -0.296005,
-0.792649, -0.571252, -0.213023,
-0.296005, -0.702310, -0.647412,
-0.213023, -0.792649, -0.571252,
-0.346153, -0.783452, -0.516122,
0.506298, 0.065758, 0.859848,
0.445365, 0.065817, 0.892927,
0.465658, 0.000000, 0.884965,
0.422364, 0.131749, 0.896801,
0.381722, 0.065988, 0.921919,
0.316778, 0.066042, 0.946198,
0.339123, 0.000000, 0.940742,
0.460267, 0.196015, 0.865871,
0.397351, 0.196412, 0.896401,
0.289929, 0.132570, 0.947822,
0.247326, 0.066567, 0.966643,
0.330606, 0.198387, 0.922682,
0.261550, 0.199094, 0.944433,
0.232455, 0.264151, 0.936050,
0.191780, 0.198337, 0.961188,
0.368518, 0.262661, 0.891742,
0.179596, 0.066604, 0.981483,
0.206103, 0.000000, 0.978530,
0.151549, 0.133256, 0.979426,
0.110385, 0.066709, 0.991648,
0.041248, 0.066741, 0.996917,
0.069142, 0.000000, 0.997607,
0.122729, 0.198580, 0.972370,
0.405434, 0.322525, 0.855337,
0.339209, 0.324864, 0.882837,
0.307659, 0.385906, 0.869725,
0.270627, 0.325929, 0.905832,
0.201189, 0.325530, 0.923879,
0.342848, 0.442867, 0.828447,
0.274694, 0.444464, 0.852640,
0.000000, 0.133523, 0.991046,
-0.041248, 0.066741, 0.996917,
0.041251, 0.200368, 0.978852,
-0.041251, 0.200368, 0.978852,
-0.081501, 0.265518, 0.960655,
-0.122729, 0.198580, 0.972370,
0.081501, 0.265518, 0.960655,
-0.041302, 0.334140, 0.941618,
-0.122800, 0.332342, 0.935130,
0.041302, 0.334140, 0.941618,
0.000000, 0.400968, 0.916092,
0.041299, 0.464213, 0.884760,
-0.041299, 0.464213, 0.884760,
0.122800, 0.332342, 0.935130,
-0.161006, 0.394136, 0.904839,
-0.201189, 0.325530, 0.923879,
-0.119755, 0.460882, 0.879344,
-0.198227, 0.454262, 0.868535,
-0.233445, 0.511205, 0.827147,
-0.274694, 0.444464, 0.852640,
-0.078459, 0.524110, 0.848029,
0.161006, 0.394136, 0.904839,
0.198227, 0.454262, 0.868535,
0.119755, 0.460882, 0.879344,
0.078459, 0.524110, 0.848029,
0.233445, 0.511205, 0.827147,
-0.110385, 0.066709, 0.991648,
-0.069142, 0.000000, 0.997607,
-0.151549, 0.133256, 0.979426,
-0.179596, 0.066604, 0.981483,
-0.247326, 0.066567, 0.966643,
-0.206103, 0.000000, 0.978530,
-0.191780, 0.198337, 0.961188,
-0.289929, 0.132570, 0.947822,
-0.316778, 0.066042, 0.946198,
-0.261550, 0.199094, 0.944433,
-0.330606, 0.198387, 0.922682,
-0.368518, 0.262661, 0.891742,
-0.397351, 0.196412, 0.896401,
-0.232455, 0.264151, 0.936050,
-0.381722, 0.065988, 0.921919,
-0.339123, 0.000000, 0.940742,
-0.422364, 0.131749, 0.896801,
-0.445365, 0.065817, 0.892927,
-0.506298, 0.065758, 0.859848,
-0.465658, 0.000000, 0.884965,
-0.460267, 0.196015, 0.865871,
-0.270627, 0.325929, 0.905832,
-0.307659, 0.385906, 0.869725,
-0.339209, 0.324864, 0.882837,
-0.405434, 0.322525, 0.855337,
-0.342848, 0.442867, 0.828447,
0.273707, 0.554740, 0.785715,
0.197274, 0.564513, 0.801504,
0.159613, 0.615456, 0.771841,
0.119078, 0.571140, 0.812169,
0.040675, 0.574458, 0.817523,
0.199331, 0.656006, 0.727958,
0.121193, 0.662842, 0.738886,
0.000000, 0.622292, 0.782786,
-0.040675, 0.574458, 0.817523,
0.040677, 0.667502, 0.743496,
-0.040677, 0.667502, 0.743496,
-0.080573, 0.708969, 0.700622,
-0.121193, 0.662842, 0.738886,
0.080573, 0.708969, 0.700622,
-0.119078, 0.571140, 0.812169,
-0.159613, 0.615456, 0.771841,
-0.197274, 0.564513, 0.801504,
-0.273707, 0.554740, 0.785715,
-0.199331, 0.656006, 0.727958,
0.121144, 0.744727, 0.656282,
0.040642, 0.749387, 0.660884,
0.000000, 0.786433, 0.617676,
-0.040642, 0.749387, 0.660884,
-0.121144, 0.744727, 0.656282,
0.040640, 0.819207, 0.572056,
-0.040640, 0.819207, 0.572056,
0.445365, -0.065817, 0.892927,
0.506298, -0.065758, 0.859848,
0.381722, -0.065988, 0.921919,
0.422364, -0.131749, 0.896801,
0.397351, -0.196412, 0.896401,
0.460267, -0.196015, 0.865871,
0.316778, -0.066042, 0.946198,
0.330606, -0.198387, 0.922682,
0.368518, -0.262661, 0.891742,
0.289929, -0.132570, 0.947822,
0.261550, -0.199094, 0.944433,
0.191780, -0.198337, 0.961188,
0.232455, -0.264151, 0.936050,
0.247326, -0.066567, 0.966643,
0.339209, -0.324864, 0.882837,
0.405434, -0.322525, 0.855337,
0.270627, -0.325929, 0.905832,
0.307659, -0.385906, 0.869725,
0.274694, -0.444464, 0.852640,
0.342848, -0.442867, 0.828447,
0.201189, -0.325530, 0.923879,
0.179596, -0.066604, 0.981483,
0.110385, -0.066709, 0.991648,
0.151549, -0.133256, 0.979426,
0.122729, -0.198580, 0.972370,
0.041248, -0.066741, 0.996917,
0.198227, -0.454262, 0.868535,
0.233445, -0.511205, 0.827147,
0.161006, -0.394136, 0.904839,
0.119755, -0.460882, 0.879344,
0.041299, -0.464213, 0.884760,
0.078459, -0.524110, 0.848029,
0.122800, -0.332342, 0.935130,
0.000000, -0.400968, 0.916092,
-0.041299, -0.464213, 0.884760,
0.041302, -0.334140, 0.941618,
-0.041302, -0.334140, 0.941618,
-0.081501, -0.265518, 0.960655,
-0.122800, -0.332342, 0.935130,
0.081501, -0.265518, 0.960655,
-0.119755, -0.460882, 0.879344,
-0.078459, -0.524110, 0.848029,
-0.161006, -0.394136, 0.904839,
-0.198227, -0.454262, 0.868535,
-0.274694, -0.444464, 0.852640,
-0.233445, -0.511205, 0.827147,
-0.201189, -0.325530, 0.923879,
0.041251, -0.200368, 0.978852,
0.000000, -0.133523, 0.991046,
-0.041251, -0.200368, 0.978852,
-0.122729, -0.198580, 0.972370,
-0.041248, -0.066741, 0.996917,
0.197274, -0.564513, 0.801504,
0.273707, -0.554740, 0.785715,
0.119078, -0.571140, 0.812169,
0.159613, -0.615456, 0.771841,
0.121193, -0.662842, 0.738886,
0.199331, -0.656006, 0.727958,
0.040675, -0.574458, 0.817523,
0.040677, -0.667502, 0.743496,
0.080573, -0.708969, 0.700622,
0.000000, -0.622292, 0.782786,
-0.040677, -0.667502, 0.743496,
-0.121193, -0.662842, 0.738886,
-0.080573, -0.708969, 0.700622,
-0.040675, -0.574458, 0.817523,
0.040642, -0.749387, 0.660884,
0.121144, -0.744727, 0.656282,
-0.040642, -0.749387, 0.660884,
0.000000, -0.786433, 0.617676,
-0.040640, -0.819207, 0.572056,
0.040640, -0.819207, 0.572056,
-0.121144, -0.744727, 0.656282,
-0.119078, -0.571140, 0.812169,
-0.197274, -0.564513, 0.801504,
-0.159613, -0.615456, 0.771841,
-0.199331, -0.656006, 0.727958,
-0.273707, -0.554740, 0.785715,
-0.110385, -0.066709, 0.991648,
-0.179596, -0.066604, 0.981483,
-0.151549, -0.133256, 0.979426,
-0.191780, -0.198337, 0.961188,
-0.247326, -0.066567, 0.966643,
-0.261550, -0.199094, 0.944433,
-0.232455, -0.264151, 0.936050,
-0.289929, -0.132570, 0.947822,
-0.330606, -0.198387, 0.922682,
-0.397351, -0.196412, 0.896401,
-0.368518, -0.262661, 0.891742,
-0.316778, -0.066042, 0.946198,
-0.270627, -0.325929, 0.905832,
-0.339209, -0.324864, 0.882837,
-0.307659, -0.385906, 0.869725,
-0.342848, -0.442867, 0.828447,
-0.405434, -0.322525, 0.855337,
-0.381722, -0.065988, 0.921919,
-0.445365, -0.065817, 0.892927,
-0.422364, -0.131749, 0.896801,
-0.460267, -0.196015, 0.865871,
-0.506298, -0.065758, 0.859848,
0.465658, 0.000000, -0.884965,
0.445365, 0.065817, -0.892927,
0.506298, 0.065758, -0.859848,
0.381722, 0.065988, -0.921919,
0.422364, 0.131749, -0.896801,
0.397351, 0.196412, -0.896401,
0.460267, 0.196015, -0.865871,
0.339123, 0.000000, -0.940742,
0.316778, 0.066042, -0.946198,
0.330606, 0.198387, -0.922682,
0.368518, 0.262661, -0.891742,
0.289929, 0.132570, -0.947822,
0.261550, 0.199094, -0.944433,
0.191780, 0.198337, -0.961188,
0.232455, 0.264151, -0.936050,
0.247326, 0.066567, -0.966643,
0.339209, 0.324864, -0.882837,
0.405434, 0.322525, -0.855337,
0.270627, 0.325929, -0.905832,
0.307659, 0.385906, -0.869725,
0.274694, 0.444464, -0.852640,
0.342848, 0.442867, -0.828447,
0.201189, 0.325530, -0.923879,
0.206103, 0.000000, -0.978530,
0.179596, 0.066604, -0.981483,
0.110385, 0.066709, -0.991648,
0.151549, 0.133256, -0.979426,
0.122729, 0.198580, -0.972370,
0.069142, 0.000000, -0.997607,
0.041248, 0.066741, -0.996917,
0.198227, 0.454262, -0.868535,
0.233445, 0.511205, -0.827147,
0.161006, 0.394136, -0.904839,
0.119755, 0.460882, -0.879344,
0.041299, 0.464213, -0.884760,
0.078459, 0.524110, -0.848029,
0.122800, 0.332342, -0.935130,
0.000000, 0.400968, -0.916092,
-0.041299, 0.464213, -0.884760,
0.041302, 0.334140, -0.941618,
-0.041302, 0.334140, -0.941618,
-0.081501, 0.265518, -0.960655,
-0.122800, 0.332342, -0.935130,
0.081501, 0.265518, -0.960655,
-0.119755, 0.460882, -0.879344,
-0.078459, 0.524110, -0.848029,
-0.161006, 0.394136, -0.904839,
-0.198227, 0.454262, -0.868535,
-0.274694, 0.444464, -0.852640,
-0.233445, 0.511205, -0.827147,
-0.201189, 0.325530, -0.923879,
0.041251, 0.200368, -0.978852,
0.000000, 0.133523, -0.991046,
-0.041251, 0.200368, -0.978852,
-0.122729, 0.198580, -0.972370,
-0.041248, 0.066741, -0.996917,
0.197274, 0.564513, -0.801504,
0.273707, 0.554740, -0.785715,
0.119078, 0.571140, -0.812169,
0.159613, 0.615456, -0.771841,
0.121193, 0.662842, -0.738886,
0.199331, 0.656006, -0.727958,
0.040675, 0.574458, -0.817523,
0.040677, 0.667502, -0.743496,
0.080573, 0.708969, -0.700622,
0.000000, 0.622292, -0.782786,
-0.040677, 0.667502, -0.743496,
-0.121193, 0.662842, -0.738886,
-0.080573, 0.708969, -0.700622,
-0.040675, 0.574458, -0.817523,
0.040642, 0.749387, -0.660884,
0.121144, 0.744727, -0.656282,
-0.040642, 0.749387, -0.660884,
0.000000, 0.786433, -0.617676,
-0.040640, 0.819207, -0.572056,
0.040640, 0.819207, -0.572056,
-0.121144, 0.744727, -0.656282,
-0.119078, 0.571140, -0.812169,
-0.197274, 0.564513, -0.801504,
-0.159613, 0.615456, -0.771841,
-0.199331, 0.656006, -0.727958,
-0.273707, 0.554740, -0.785715,
-0.069142, 0.000000, -0.997607,
-0.110385, 0.066709, -0.991648,
-0.179596, 0.066604, -0.981483,
-0.151549, 0.133256, -0.979426,
-0.191780, 0.198337, -0.961188,
-0.206103, 0.000000, -0.978530,
-0.247326, 0.066567, -0.966643,
-0.261550, 0.199094, -0.944433,
-0.232455, 0.264151, -0.936050,
-0.289929, 0.132570, -0.947822,
-0.330606, 0.198387, -0.922682,
-0.397351, 0.196412, -0.896401,
-0.368518, 0.262661, -0.891742,
-0.316778, 0.066042, -0.946198,
-0.270627, 0.325929, -0.905832,
-0.339209, 0.324864, -0.882837,
-0.307659, 0.385906, -0.869725,
-0.342848, 0.442867, -0.828447,
-0.405434, 0.322525, -0.855337,
-0.339123, 0.000000, -0.940742,
-0.381722, 0.065988, -0.921919,
-0.445365, 0.065817, -0.892927,
-0.422364, 0.131749, -0.896801,
-0.460267, 0.196015, -0.865871,
-0.465658, 0.000000, -0.884965,
-0.506298, 0.065758, -0.859848,
0.506298, -0.065758, -0.859848,
0.445365, -0.065817, -0.892927,
0.422364, -0.131749, -0.896801,
0.381722, -0.065988, -0.921919,
0.316778, -0.066042, -0.946198,
0.460267, -0.196015, -0.865871,
0.397351, -0.196412, -0.896401,
0.289929, -0.132570, -0.947822,
0.247326, -0.066567, -0.966643,
0.330606, -0.198387, -0.922682,
0.261550, -0.199094, -0.944433,
0.232455, -0.264151, -0.936050,
0.191780, -0.198337, -0.961188,
0.368518, -0.262661, -0.891742,
0.179596, -0.066604, -0.981483,
0.151549, -0.133256, -0.979426,
0.110385, -0.066709, -0.991648,
0.041248, -0.066741, -0.996917,
0.122729, -0.198580, -0.972370,
0.405434, -0.322525, -0.855337,
0.339209, -0.324864, -0.882837,
0.307659, -0.385906, -0.869725,
0.270627, -0.325929, -0.905832,
0.201189, -0.325530, -0.923879,
0.342848, -0.442867, -0.828447,
0.274694, -0.444464, -0.852640,
0.000000, -0.133523, -0.991046,
-0.041248, -0.066741, -0.996917,
0.041251, -0.200368, -0.978852,
-0.041251, -0.200368, -0.978852,
-0.081501, -0.265518, -0.960655,
-0.122729, -0.198580, -0.972370,
0.081501, -0.265518, -0.960655,
-0.041302, -0.334140, -0.941618,
-0.122800, -0.332342, -0.935130,
0.041302, -0.334140, -0.941618,
0.000000, -0.400968, -0.916092,
0.041299, -0.464213, -0.884760,
-0.041299, -0.464213, -0.884760,
0.122800, -0.332342, -0.935130,
-0.161006, -0.394136, -0.904839,
-0.201189, -0.325530, -0.923879,
-0.119755, -0.460882, -0.879344,
-0.198227, -0.454262, -0.868535,
-0.233445, -0.511205, -0.827147,
-0.274694, -0.444464, -0.852640,
-0.078459, -0.524110, -0.848029,
0.161006, -0.394136, -0.904839,
0.198227, -0.454262, -0.868535,
0.119755, -0.460882, -0.879344,
0.078459, -0.524110, -0.848029,
0.233445, -0.511205, -0.827147,
-0.110385, -0.066709, -0.991648,
-0.151549, -0.133256, -0.979426,
-0.179596, -0.066604, -0.981483,
-0.247326, -0.066567, -0.966643,
-0.191780, -0.198337, -0.961188,
-0.289929, -0.132570, -0.947822,
-0.316778, -0.066042, -0.946198,
-0.261550, -0.199094, -0.944433,
-0.330606, -0.198387, -0.922682,
-0.368518, -0.262661, -0.891742,
-0.397351, -0.196412, -0.896401,
-0.232455, -0.264151, -0.936050,
-0.381722, -0.065988, -0.921919,
-0.422364, -0.131749, -0.896801,
-0.445365, -0.065817, -0.892927,
-0.506298, -0.065758, -0.859848,
-0.460267, -0.196015, -0.865871,
-0.270627, -0.325929, -0.905832,
-0.307659, -0.385906, -0.869725,
-0.339209, -0.324864, -0.882837,
-0.405434, -0.322525, -0.855337,
-0.342848, -0.442867, -0.828447,
0.273707, -0.554740, -0.785715,
0.197274, -0.564513, -0.801504,
0.159613, -0.615456, -0.771841,
0.119078, -0.571140, -0.812169,
0.040675, -0.574458, -0.817523,
0.199331, -0.656006, -0.727958,
0.121193, -0.662842, -0.738886,
0.000000, -0.622292, -0.782786,
-0.040675, -0.574458, -0.817523,
0.040677, -0.667502, -0.743496,
-0.040677, -0.667502, -0.743496,
-0.080573, -0.708969, -0.700622,
-0.121193, -0.662842, -0.738886,
0.080573, -0.708969, -0.700622,
-0.119078, -0.571140, -0.812169,
-0.159613, -0.615456, -0.771841,
-0.197274, -0.564513, -0.801504,
-0.273707, -0.554740, -0.785715,
-0.199331, -0.656006, -0.727958,
0.121144, -0.744727, -0.656282,
0.040642, -0.749387, -0.660884,
0.000000, -0.786433, -0.617676,
-0.040642, -0.749387, -0.660884,
-0.121144, -0.744727, -0.656282,
0.040640, -0.819207, -0.572056,
-0.040640, -0.819207, -0.572056,
0.859848, 0.506298, 0.065758,
0.892927, 0.445365, 0.065817,
0.884965, 0.465658, 0.000000,
0.896801, 0.422364, 0.131749,
0.921919, 0.381722, 0.065988,
0.946198, 0.316778, 0.066042,
0.940742, 0.339123, 0.000000,
0.865871, 0.460267, 0.196015,
0.896401, 0.397351, 0.196412,
0.947822, 0.289929, 0.132570,
0.966643, 0.247326, 0.066567,
0.922682, 0.330606, 0.198387,
0.944433, 0.261550, 0.199094,
0.936050, 0.232455, 0.264151,
0.961188, 0.191780, 0.198337,
0.891742, 0.368518, 0.262661,
0.981483, 0.179596, 0.066604,
0.978530, 0.206103, 0.000000,
0.979426, 0.151549, 0.133256,
0.991648, 0.110385, 0.066709,
0.996917, 0.041248, 0.066741,
0.997607, 0.069142, 0.000000,
0.972370, 0.122729, 0.198580,
0.855337, 0.405434, 0.322525,
0.882837, 0.339209, 0.324864,
0.869725, 0.307659, 0.385906,
0.905832, 0.270627, 0.325929,
0.923879, 0.201189, 0.325530,
0.828447, 0.342848, 0.442867,
0.852640, 0.274694, 0.444464,
0.991046, 0.000000, 0.133523,
0.996917, -0.041248, 0.066741,
0.978852, 0.041251, 0.200368,
0.978852, -0.041251, 0.200368,
0.960655, -0.081501, 0.265518,
0.972370, -0.122729, 0.198580,
0.960655, 0.081501, 0.265518,
0.941618, -0.041302, 0.334140,
0.935130, -0.122800, 0.332342,
0.941618, 0.041302, 0.334140,
0.916092, 0.000000, 0.400968,
0.884760, 0.041299, 0.464213,
0.884760, -0.041299, 0.464213,
0.935130, 0.122800, 0.332342,
0.904839, -0.161006, 0.394136,
0.923879, -0.201189, 0.325530,
0.879344, -0.119755, 0.460882,
0.868535, -0.198227, 0.454262,
0.827147, -0.233445, 0.511205,
0.852640, -0.274694, 0.444464,
0.848029, -0.078459, 0.524110,
0.904839, 0.161006, 0.394136,
0.868535, 0.198227, 0.454262,
0.879344, 0.119755, 0.460882,
0.848029, 0.078459, 0.524110,
0.827147, 0.233445, 0.511205,
0.991648, -0.110385, 0.066709,
0.997607, -0.069142, 0.000000,
0.979426, -0.151549, 0.133256,
0.981483, -0.179596, 0.066604,
0.966643, -0.247326, 0.066567,
0.978530, -0.206103, 0.000000,
0.961188, -0.191780, 0.198337,
0.947822, -0.289929, 0.132570,
0.946198, -0.316778, 0.066042,
0.944433, -0.261550, 0.199094,
0.922682, -0.330606, 0.198387,
0.891742, -0.368518, 0.262661,
0.896401, -0.397351, 0.196412,
0.936050, -0.232455, 0.264151,
0.921919, -0.381722, 0.065988,
0.940742, -0.339123, 0.000000,
0.896801, -0.422364, 0.131749,
0.892927, -0.445365, 0.065817,
0.859848, -0.506298, 0.065758,
0.884965, -0.465658, 0.000000,
0.865871, -0.460267, 0.196015,
0.905832, -0.270627, 0.325929,
0.869725, -0.307659, 0.385906,
0.882837, -0.339209, 0.324864,
0.855337, -0.405434, 0.322525,
0.828447, -0.342848, 0.442867,
0.785715, 0.273707, 0.554740,
0.801504, 0.197274, 0.564513,
0.771841, 0.159613, 0.615456,
0.812169, 0.119078, 0.571140,
0.817523, 0.040675, 0.574458,
0.727958, 0.199331, 0.656006,
0.738886, 0.121193, 0.662842,
0.782786, 0.000000, 0.622292,
0.817523, -0.040675, 0.574458,
0.743496, 0.040677, 0.667502,
0.743496, -0.040677, 0.667502,
0.700622, -0.080573, 0.708969,
0.738886, -0.121193, 0.662842,
0.700622, 0.080573, 0.708969,
0.812169, -0.119078, 0.571140,
0.771841, -0.159613, 0.615456,
0.801504, -0.197274, 0.564513,
0.785715, -0.273707, 0.554740,
0.727958, -0.199331, 0.656006,
0.656282, 0.121144, 0.744727,
0.660884, 0.040642, 0.749387,
0.617676, 0.000000, 0.786433,
0.660884, -0.040642, 0.749387,
0.656282, -0.121144, 0.744727,
0.572056, 0.040640, 0.819207,
0.572056, -0.040640, 0.819207,
0.892927, 0.445365, -0.065817,
0.859848, 0.506298, -0.065758,
0.921919, 0.381722, -0.065988,
0.896801, 0.422364, -0.131749,
0.896401, 0.397351, -0.196412,
0.865871, 0.460267, -0.196015,
0.946198, 0.316778, -0.066042,
0.922682, 0.330606, -0.198387,
0.891742, 0.368518, -0.262661,
0.947822, 0.289929, -0.132570,
0.944433, 0.261550, -0.199094,
0.961188, 0.191780, -0.198337,
0.936050, 0.232455, -0.264151,
0.966643, 0.247326, -0.066567,
0.882837, 0.339209, -0.324864,
0.855337, 0.405434, -0.322525,
0.905832, 0.270627, -0.325929,
0.869725, 0.307659, -0.385906,
0.852640, 0.274694, -0.444464,
0.828447, 0.342848, -0.442867,
0.923879, 0.201189, -0.325530,
0.981483, 0.179596, -0.066604,
0.991648, 0.110385, -0.066709,
0.979426, 0.151549, -0.133256,
0.972370, 0.122729, -0.198580,
0.996917, 0.041248, -0.066741,
0.868535, 0.198227, -0.454262,
0.827147, 0.233445, -0.511205,
0.904839, 0.161006, -0.394136,
0.879344, 0.119755, -0.460882,
0.884760, 0.041299, -0.464213,
0.848029, 0.078459, -0.524110,
0.935130, 0.122800, -0.332342,
0.916092, 0.000000, -0.400968,
0.884760, -0.041299, -0.464213,
0.941618, 0.041302, -0.334140,
0.941618, -0.041302, -0.334140,
0.960655, -0.081501, -0.265518,
0.935130, -0.122800, -0.332342,
0.960655, 0.081501, -0.265518,
0.879344, -0.119755, -0.460882,
0.848029, -0.078459, -0.524110,
0.904839, -0.161006, -0.394136,
0.868535, -0.198227, -0.454262,
0.852640, -0.274694, -0.444464,
0.827147, -0.233445, -0.511205,
0.923879, -0.201189, -0.325530,
0.978852, 0.041251, -0.200368,
0.991046, 0.000000, -0.133523,
0.978852, -0.041251, -0.200368,
0.972370, -0.122729, -0.198580,
0.996917, -0.041248, -0.066741,
0.801504, 0.197274, -0.564513,
0.785715, 0.273707, -0.554740,
0.812169, 0.119078, -0.571140,
0.771841, 0.159613, -0.615456,
0.738886, 0.121193, -0.662842,
0.727958, 0.199331, -0.656006,
0.817523, 0.040675, -0.574458,
0.743496, 0.040677, -0.667502,
0.700622, 0.080573, -0.708969,
0.782786, 0.000000, -0.622292,
0.743496, -0.040677, -0.667502,
0.738886, -0.121193, -0.662842,
0.700622, -0.080573, -0.708969,
0.817523, -0.040675, -0.574458,
0.660884, 0.040642, -0.749387,
0.656282, 0.121144, -0.744727,
0.660884, -0.040642, -0.749387,
0.617676, 0.000000, -0.786433,
0.572056, -0.040640, -0.819207,
0.572056, 0.040640, -0.819207,
0.656282, -0.121144, -0.744727,
0.812169, -0.119078, -0.571140,
0.801504, -0.197274, -0.564513,
0.771841, -0.159613, -0.615456,
0.727958, -0.199331, -0.656006,
0.785715, -0.273707, -0.554740,
0.991648, -0.110385, -0.066709,
0.981483, -0.179596, -0.066604,
0.979426, -0.151549, -0.133256,
0.961188, -0.191780, -0.198337,
0.966643, -0.247326, -0.066567,
0.944433, -0.261550, -0.199094,
0.936050, -0.232455, -0.264151,
0.947822, -0.289929, -0.132570,
0.922682, -0.330606, -0.198387,
0.896401, -0.397351, -0.196412,
0.891742, -0.368518, -0.262661,
0.946198, -0.316778, -0.066042,
0.905832, -0.270627, -0.325929,
0.882837, -0.339209, -0.324864,
0.869725, -0.307659, -0.385906,
0.828447, -0.342848, -0.442867,
0.855337, -0.405434, -0.322525,
0.921919, -0.381722, -0.065988,
0.892927, -0.445365, -0.065817,
0.896801, -0.422364, -0.131749,
0.865871, -0.460267, -0.196015,
0.859848, -0.506298, -0.065758,
-0.859848, -0.506298, 0.065758,
-0.892927, -0.445365, 0.065817,
-0.884965, -0.465658, 0.000000,
-0.896801, -0.422364, 0.131749,
-0.921919, -0.381722, 0.065988,
-0.946198, -0.316778, 0.066042,
-0.940742, -0.339123, 0.000000,
-0.865871, -0.460267, 0.196015,
-0.896401, -0.397351, 0.196412,
-0.947822, -0.289929, 0.132570,
-0.966643, -0.247326, 0.066567,
-0.922682, -0.330606, 0.198387,
-0.944433, -0.261550, 0.199094,
-0.936050, -0.232455, 0.264151,
-0.961188, -0.191780, 0.198337,
-0.891742, -0.368518, 0.262661,
-0.981483, -0.179596, 0.066604,
-0.978530, -0.206103, 0.000000,
-0.979426, -0.151549, 0.133256,
-0.991648, -0.110385, 0.066709,
-0.996917, -0.041248, 0.066741,
-0.997607, -0.069142, 0.000000,
-0.972370, -0.122729, 0.198580,
-0.855337, -0.405434, 0.322525,
-0.882837, -0.339209, 0.324864,
-0.869725, -0.307659, 0.385906,
-0.905832, -0.270627, 0.325929,
-0.923879, -0.201189, 0.325530,
-0.828447, -0.342848, 0.442867,
-0.852640, -0.274694, 0.444464,
-0.991046, 0.000000, 0.133523,
-0.996917, 0.041248, 0.066741,
-0.978852, -0.041251, 0.200368,
-0.978852, 0.041251, 0.200368,
-0.960655, 0.081501, 0.265518,
-0.972370, 0.122729, 0.198580,
-0.960655, -0.081501, 0.265518,
-0.941618, 0.041302, 0.334140,
-0.935130, 0.122800, 0.332342,
-0.941618, -0.041302, 0.334140,
-0.916092, 0.000000, 0.400968,
-0.884760, -0.041299, 0.464213,
-0.884760, 0.041299, 0.464213,
-0.935130, -0.122800, 0.332342,
-0.904839, 0.161006, 0.394136,
-0.923879, 0.201189, 0.325530,
-0.879344, 0.119755, 0.460882,
-0.868535, 0.198227, 0.454262,
-0.827147, 0.233445, 0.511205,
-0.852640, 0.274694, 0.444464,
-0.848029, 0.078459, 0.524110,
-0.904839, -0.161006, 0.394136,
-0.868535, -0.198227, 0.454262,
-0.879344, -0.119755, 0.460882,
-0.848029, -0.078459, 0.524110,
-0.827147, -0.233445, 0.511205,
-0.991648, 0.110385, 0.066709,
-0.997607, 0.069142, 0.000000,
-0.979426, 0.151549, 0.133256,
-0.981483, 0.179596, 0.066604,
-0.966643, 0.247326, 0.066567,
-0.978530, 0.206103, 0.000000,
-0.961188, 0.191780, 0.198337,
-0.947822, 0.289929, 0.132570,
-0.946198, 0.316778, 0.066042,
-0.944433, 0.261550, 0.199094,
-0.922682, 0.330606, 0.198387,
-0.891742, 0.368518, 0.262661,
-0.896401, 0.397351, 0.196412,
-0.936050, 0.232455, 0.264151,
-0.921919, 0.381722, 0.065988,
-0.940742, 0.339123, 0.000000,
-0.896801, 0.422364, 0.131749,
-0.892927, 0.445365, 0.065817,
-0.859848, 0.506298, 0.065758,
-0.884965, 0.465658, 0.000000,
-0.865871, 0.460267, 0.196015,
-0.905832, 0.270627, 0.325929,
-0.869725, 0.307659, 0.385906,
-0.882837, 0.339209, 0.324864,
-0.855337, 0.405434, 0.322525,
-0.828447, 0.342848, 0.442867,
-0.785715, -0.273707, 0.554740,
-0.801504, -0.197274, 0.564513,
-0.771841, -0.159613, 0.615456,
-0.812169, -0.119078, 0.571140,
-0.817523, -0.040675, 0.574458,
-0.727958, -0.199331, 0.656006,
-0.738886, -0.121193, 0.662842,
-0.782786, 0.000000, 0.622292,
-0.817523, 0.040675, 0.574458,
-0.743496, -0.040677, 0.667502,
-0.743496, 0.040677, 0.667502,
-0.700622, 0.080573, 0.708969,
-0.738886, 0.121193, 0.662842,
-0.700622, -0.080573, 0.708969,
-0.812169, 0.119078, 0.571140,
-0.771841, 0.159613, 0.615456,
-0.801504, 0.197274, 0.564513,
-0.785715, 0.273707, 0.554740,
-0.727958, 0.199331, 0.656006,
-0.656282, -0.121144, 0.744727,
-0.660884, -0.040642, 0.749387,
-0.617676, 0.000000, 0.786433,
-0.660884, 0.040642, 0.749387,
-0.656282, 0.121144, 0.744727,
-0.572056, -0.040640, 0.819207,
-0.572056, 0.040640, 0.819207,
-0.892927, -0.445365, -0.065817,
-0.859848, -0.506298, -0.065758,
-0.921919, -0.381722, -0.065988,
-0.896801, -0.422364, -0.131749,
-0.896401, -0.397351, -0.196412,
-0.865871, -0.460267, -0.196015,
-0.946198, -0.316778, -0.066042,
-0.922682, -0.330606, -0.198387,
-0.891742, -0.368518, -0.262661,
-0.947822, -0.289929, -0.132570,
-0.944433, -0.261550, -0.199094,
-0.961188, -0.191780, -0.198337,
-0.936050, -0.232455, -0.264151,
-0.966643, -0.247326, -0.066567,
-0.882837, -0.339209, -0.324864,
-0.855337, -0.405434, -0.322525,
-0.905832, -0.270627, -0.325929,
-0.869725, -0.307659, -0.385906,
-0.852640, -0.274694, -0.444464,
-0.828447, -0.342848, -0.442867,
-0.923879, -0.201189, -0.325530,
-0.981483, -0.179596, -0.066604,
-0.991648, -0.110385, -0.066709,
-0.979426, -0.151549, -0.133256,
-0.972370, -0.122729, -0.198580,
-0.996917, -0.041248, -0.066741,
-0.868535, -0.198227, -0.454262,
-0.827147, -0.233445, -0.511205,
-0.904839, -0.161006, -0.394136,
-0.879344, -0.119755, -0.460882,
-0.884760, -0.041299, -0.464213,
-0.848029, -0.078459, -0.524110,
-0.935130, -0.122800, -0.332342,
-0.916092, 0.000000, -0.400968,
-0.884760, 0.041299, -0.464213,
-0.941618, -0.041302, -0.334140,
-0.941618, 0.041302, -0.334140,
-0.960655, 0.081501, -0.265518,
-0.935130, 0.122800, -0.332342,
-0.960655, -0.081501, -0.265518,
-0.879344, 0.119755, -0.460882,
-0.848029, 0.078459, -0.524110,
-0.904839, 0.161006, -0.394136,
-0.868535, 0.198227, -0.454262,
-0.852640, 0.274694, -0.444464,
-0.827147, 0.233445, -0.511205,
-0.923879, 0.201189, -0.325530,
-0.978852, -0.041251, -0.200368,
-0.991046, 0.000000, -0.133523,
-0.978852, 0.041251, -0.200368,
-0.972370, 0.122729, -0.198580,
-0.996917, 0.041248, -0.066741,
-0.801504, -0.197274, -0.564513,
-0.785715, -0.273707, -0.554740,
-0.812169, -0.119078, -0.571140,
-0.771841, -0.159613, -0.615456,
-0.738886, -0.121193, -0.662842,
-0.727958, -0.199331, -0.656006,
-0.817523, -0.040675, -0.574458,
-0.743496, -0.040677, -0.667502,
-0.700622, -0.080573, -0.708969,
-0.782786, 0.000000, -0.622292,
-0.743496, 0.040677, -0.667502,
-0.738886, 0.121193, -0.662842,
-0.700622, 0.080573, -0.708969,
-0.817523, 0.040675, -0.574458,
-0.660884, -0.040642, -0.749387,
-0.656282, -0.121144, -0.744727,
-0.660884, 0.040642, -0.749387,
-0.617676, 0.000000, -0.786433,
-0.572056, 0.040640, -0.819207,
-0.572056, -0.040640, -0.819207,
-0.656282, 0.121144, -0.744727,
-0.812169, 0.119078, -0.571140,
-0.801504, 0.197274, -0.564513,
-0.771841, 0.159613, -0.615456,
-0.727958, 0.199331, -0.656006,
-0.785715, 0.273707, -0.554740,
-0.991648, 0.110385, -0.066709,
-0.981483, 0.179596, -0.066604,
-0.979426, 0.151549, -0.133256,
-0.961188, 0.191780, -0.198337,
-0.966643, 0.247326, -0.066567,
-0.944433, 0.261550, -0.199094,
-0.936050, 0.232455, -0.264151,
-0.947822, 0.289929, -0.132570,
-0.922682, 0.330606, -0.198387,
-0.896401, 0.397351, -0.196412,
-0.891742, 0.368518, -0.262661,
-0.946198, 0.316778, -0.066042,
-0.905832, 0.270627, -0.325929,
-0.882837, 0.339209, -0.324864,
-0.869725, 0.307659, -0.385906,
-0.828447, 0.342848, -0.442867,
-0.855337, 0.405434, -0.322525,
-0.921919, 0.381722, -0.065988,
-0.892927, 0.445365, -0.065817,
-0.896801, 0.422364, -0.131749,
-0.865871, 0.460267, -0.196015,
-0.859848, 0.506298, -0.065758,
0.065758, 0.859848, 0.506298,
0.065817, 0.892927, 0.445365,
0.000000, 0.884965, 0.465658,
0.131749, 0.896801, 0.422364,
0.065988, 0.921919, 0.381722,
0.066042, 0.946198, 0.316778,
0.000000, 0.940742, 0.339123,
0.196015, 0.865871, 0.460267,
0.196412, 0.896401, 0.397351,
0.132570, 0.947822, 0.289929,
0.066567, 0.966643, 0.247326,
0.198387, 0.922682, 0.330606,
0.199094, 0.944433, 0.261550,
0.264151, 0.936050, 0.232455,
0.198337, 0.961188, 0.191780,
0.262661, 0.891742, 0.368518,
0.066604, 0.981483, 0.179596,
0.000000, 0.978530, 0.206103,
0.133256, 0.979426, 0.151549,
0.066709, 0.991648, 0.110385,
0.066741, 0.996917, 0.041248,
0.000000, 0.997607, 0.069142,
0.198580, 0.972370, 0.122729,
0.322525, 0.855337, 0.405434,
0.324864, 0.882837, 0.339209,
0.385906, 0.869725, 0.307659,
0.325929, 0.905832, 0.270627,
0.325530, 0.923879, 0.201189,
0.442867, 0.828447, 0.342848,
0.444464, 0.852640, 0.274694,
0.133523, 0.991046, 0.000000,
0.066741, 0.996917, -0.041248,
0.200368, 0.978852, 0.041251,
0.200368, 0.978852, -0.041251,
0.265518, 0.960655, -0.081501,
0.198580, 0.972370, -0.122729,
0.265518, 0.960655, 0.081501,
0.334140, 0.941618, -0.041302,
0.332342, 0.935130, -0.122800,
0.334140, 0.941618, 0.041302,
0.400968, 0.916092, 0.000000,
0.464213, 0.884760, 0.041299,
0.464213, 0.884760, -0.041299,
0.332342, 0.935130, 0.122800,
0.394136, 0.904839, -0.161006,
0.325530, 0.923879, -0.201189,
0.460882, 0.879344, -0.119755,
0.454262, 0.868535, -0.198227,
0.511205, 0.827147, -0.233445,
0.444464, 0.852640, -0.274694,
0.524110, 0.848029, -0.078459,
0.394136, 0.904839, 0.161006,
0.454262, 0.868535, 0.198227,
0.460882, 0.879344, 0.119755,
0.524110, 0.848029, 0.078459,
0.511205, 0.827147, 0.233445,
0.066709, 0.991648, -0.110385,
0.000000, 0.997607, -0.069142,
0.133256, 0.979426, -0.151549,
0.066604, 0.981483, -0.179596,
0.066567, 0.966643, -0.247326,
0.000000, 0.978530, -0.206103,
0.198337, 0.961188, -0.191780,
0.132570, 0.947822, -0.289929,
0.066042, 0.946198, -0.316778,
0.199094, 0.944433, -0.261550,
0.198387, 0.922682, -0.330606,
0.262661, 0.891742, -0.368518,
0.196412, 0.896401, -0.397351,
0.264151, 0.936050, -0.232455,
0.065988, 0.921919, -0.381722,
0.000000, 0.940742, -0.339123,
0.131749, 0.896801, -0.422364,
0.065817, 0.892927, -0.445365,
0.065758, 0.859848, -0.506298,
0.000000, 0.884965, -0.465658,
0.196015, 0.865871, -0.460267,
0.325929, 0.905832, -0.270627,
0.385906, 0.869725, -0.307659,
0.324864, 0.882837, -0.339209,
0.322525, 0.855337, -0.405434,
0.442867, 0.828447, -0.342848,
0.554740, 0.785715, 0.273707,
0.564513, 0.801504, 0.197274,
0.615456, 0.771841, 0.159613,
0.571140, 0.812169, 0.119078,
0.574458, 0.817523, 0.040675,
0.656006, 0.727958, 0.199331,
0.662842, 0.738886, 0.121193,
0.622292, 0.782786, 0.000000,
0.574458, 0.817523, -0.040675,
0.667502, 0.743496, 0.040677,
0.667502, 0.743496, -0.040677,
0.708969, 0.700622, -0.080573,
0.662842, 0.738886, -0.121193,
0.708969, 0.700622, 0.080573,
0.571140, 0.812169, -0.119078,
0.615456, 0.771841, -0.159613,
0.564513, 0.801504, -0.197274,
0.554740, 0.785715, -0.273707,
0.656006, 0.727958, -0.199331,
0.744727, 0.656282, 0.121144,
0.749387, 0.660884, 0.040642,
0.786433, 0.617676, 0.000000,
0.749387, 0.660884, -0.040642,
0.744727, 0.656282, -0.121144,
0.819207, 0.572056, 0.040640,
0.819207, 0.572056, -0.040640,
-0.065817, 0.892927, 0.445365,
-0.065758, 0.859848, 0.506298,
-0.065988, 0.921919, 0.381722,
-0.131749, 0.896801, 0.422364,
-0.196412, 0.896401, 0.397351,
-0.196015, 0.865871, 0.460267,
-0.066042, 0.946198, 0.316778,
-0.198387, 0.922682, 0.330606,
-0.262661, 0.891742, 0.368518,
-0.132570, 0.947822, 0.289929,
-0.199094, 0.944433, 0.261550,
-0.198337, 0.961188, 0.191780,
-0.264151, 0.936050, 0.232455,
-0.066567, 0.966643, 0.247326,
-0.324864, 0.882837, 0.339209,
-0.322525, 0.855337, 0.405434,
-0.325929, 0.905832, 0.270627,
-0.385906, 0.869725, 0.307659,
-0.444464, 0.852640, 0.274694,
-0.442867, 0.828447, 0.342848,
-0.325530, 0.923879, 0.201189,
-0.066604, 0.981483, 0.179596,
-0.066709, 0.991648, 0.110385,
-0.133256, 0.979426, 0.151549,
-0.198580, 0.972370, 0.122729,
-0.066741, 0.996917, 0.041248,
-0.454262, 0.868535, 0.198227,
-0.511205, 0.827147, 0.233445,
-0.394136, 0.904839, 0.161006,
-0.460882, 0.879344, 0.119755,
-0.464213, 0.884760, 0.041299,
-0.524110, 0.848029, 0.078459,
-0.332342, 0.935130, 0.122800,
-0.400968, 0.916092, 0.000000,
-0.464213, 0.884760, -0.041299,
-0.334140, 0.941618, 0.041302,
-0.334140, 0.941618, -0.041302,
-0.265518, 0.960655, -0.081501,
-0.332342, 0.935130, -0.122800,
-0.265518, 0.960655, 0.081501,
-0.460882, 0.879344, -0.119755,
-0.524110, 0.848029, -0.078459,
-0.394136, 0.904839, -0.161006,
-0.454262, 0.868535, -0.198227,
-0.444464, 0.852640, -0.274694,
-0.511205, 0.827147, -0.233445,
-0.325530, 0.923879, -0.201189,
-0.200368, 0.978852, 0.041251,
-0.133523, 0.991046, 0.000000,
-0.200368, 0.978852, -0.041251,
-0.198580, 0.972370, -0.122729,
-0.066741, 0.996917, -0.041248,
-0.564513, 0.801504, 0.197274,
-0.554740, 0.785715, 0.273707,
-0.571140, 0.812169, 0.119078,
-0.615456, 0.771841, 0.159613,
-0.662842, 0.738886, 0.121193,
-0.656006, 0.727958, 0.199331,
-0.574458, 0.817523, 0.040675,
-0.667502, 0.743496, 0.040677,
-0.708969, 0.700622, 0.080573,
-0.622292, 0.782786, 0.000000,
-0.667502, 0.743496, -0.040677,
-0.662842, 0.738886, -0.121193,
-0.708969, 0.700622, -0.080573,
-0.574458, 0.817523, -0.040675,
-0.749387, 0.660884, 0.040642,
-0.744727, 0.656282, 0.121144,
-0.749387, 0.660884, -0.040642,
-0.786433, 0.617676, 0.000000,
-0.819207, 0.572056, -0.040640,
-0.819207, 0.572056, 0.040640,
-0.744727, 0.656282, -0.121144,
-0.571140, 0.812169, -0.119078,
-0.564513, 0.801504, -0.197274,
-0.615456, 0.771841, -0.159613,
-0.656006, 0.727958, -0.199331,
-0.554740, 0.785715, -0.273707,
-0.066709, 0.991648, -0.110385,
-0.066604, 0.981483, -0.179596,
-0.133256, 0.979426, -0.151549,
-0.198337, 0.961188, -0.191780,
-0.066567, 0.966643, -0.247326,
-0.199094, 0.944433, -0.261550,
-0.264151, 0.936050, -0.232455,
-0.132570, 0.947822, -0.289929,
-0.198387, 0.922682, -0.330606,
-0.196412, 0.896401, -0.397351,
-0.262661, 0.891742, -0.368518,
-0.066042, 0.946198, -0.316778,
-0.325929, 0.905832, -0.270627,
-0.324864, 0.882837, -0.339209,
-0.385906, 0.869725, -0.307659,
-0.442867, 0.828447, -0.342848,
-0.322525, 0.855337, -0.405434,
-0.065988, 0.921919, -0.381722,
-0.065817, 0.892927, -0.445365,
-0.131749, 0.896801, -0.422364,
-0.196015, 0.865871, -0.460267,
-0.065758, 0.859848, -0.506298,
0.000000, -0.884965, 0.465658,
0.065817, -0.892927, 0.445365,
0.065758, -0.859848, 0.506298,
0.065988, -0.921919, 0.381722,
0.131749, -0.896801, 0.422364,
0.196412, -0.896401, 0.397351,
0.196015, -0.865871, 0.460267,
0.000000, -0.940742, 0.339123,
0.066042, -0.946198, 0.316778,
0.198387, -0.922682, 0.330606,
0.262661, -0.891742, 0.368518,
0.132570, -0.947822, 0.289929,
0.199094, -0.944433, 0.261550,
0.198337, -0.961188, 0.191780,
0.264151, -0.936050, 0.232455,
0.066567, -0.966643, 0.247326,
0.324864, -0.882837, 0.339209,
0.322525, -0.855337, 0.405434,
0.325929, -0.905832, 0.270627,
0.385906, -0.869725, 0.307659,
0.444464, -0.852640, 0.274694,
0.442867, -0.828447, 0.342848,
0.325530, -0.923879, 0.201189,
0.000000, -0.978530, 0.206103,
0.066604, -0.981483, 0.179596,
0.066709, -0.991648, 0.110385,
0.133256, -0.979426, 0.151549,
0.198580, -0.972370, 0.122729,
0.000000, -0.997607, 0.069142,
0.066741, -0.996917, 0.041248,
0.454262, -0.868535, 0.198227,
0.511205, -0.827147, 0.233445,
0.394136, -0.904839, 0.161006,
0.460882, -0.879344, 0.119755,
0.464213, -0.884760, 0.041299,
0.524110, -0.848029, 0.078459,
0.332342, -0.935130, 0.122800,
0.400968, -0.916092, 0.000000,
0.464213, -0.884760, -0.041299,
0.334140, -0.941618, 0.041302,
0.334140, -0.941618, -0.041302,
0.265518, -0.960655, -0.081501,
0.332342, -0.935130, -0.122800,
0.265518, -0.960655, 0.081501,
0.460882, -0.879344, -0.119755,
0.524110, -0.848029, -0.078459,
0.394136, -0.904839, -0.161006,
0.454262, -0.868535, -0.198227,
0.444464, -0.852640, -0.274694,
0.511205, -0.827147, -0.233445,
0.325530, -0.923879, -0.201189,
0.200368, -0.978852, 0.041251,
0.133523, -0.991046, 0.000000,
0.200368, -0.978852, -0.041251,
0.198580, -0.972370, -0.122729,
0.066741, -0.996917, -0.041248,
0.564513, -0.801504, 0.197274,
0.554740, -0.785715, 0.273707,
0.571140, -0.812169, 0.119078,
0.615456, -0.771841, 0.159613,
0.662842, -0.738886, 0.121193,
0.656006, -0.727958, 0.199331,
0.574458, -0.817523, 0.040675,
0.667502, -0.743496, 0.040677,
0.708969, -0.700622, 0.080573,
0.622292, -0.782786, 0.000000,
0.667502, -0.743496, -0.040677,
0.662842, -0.738886, -0.121193,
0.708969, -0.700622, -0.080573,
0.574458, -0.817523, -0.040675,
0.749387, -0.660884, 0.040642,
0.744727, -0.656282, 0.121144,
0.749387, -0.660884, -0.040642,
0.786433, -0.617676, 0.000000,
0.819207, -0.572056, -0.040640,
0.819207, -0.572056, 0.040640,
0.744727, -0.656282, -0.121144,
0.571140, -0.812169, -0.119078,
0.564513, -0.801504, -0.197274,
0.615456, -0.771841, -0.159613,
0.656006, -0.727958, -0.199331,
0.554740, -0.785715, -0.273707,
0.000000, -0.997607, -0.069142,
0.066709, -0.991648, -0.110385,
0.066604, -0.981483, -0.179596,
0.133256, -0.979426, -0.151549,
0.198337, -0.961188, -0.191780,
0.000000, -0.978530, -0.206103,
0.066567, -0.966643, -0.247326,
0.199094, -0.944433, -0.261550,
0.264151, -0.936050, -0.232455,
0.132570, -0.947822, -0.289929,
0.198387, -0.922682, -0.330606,
0.196412, -0.896401, -0.397351,
0.262661, -0.891742, -0.368518,
0.066042, -0.946198, -0.316778,
0.325929, -0.905832, -0.270627,
0.324864, -0.882837, -0.339209,
0.385906, -0.869725, -0.307659,
0.442867, -0.828447, -0.342848,
0.322525, -0.855337, -0.405434,
0.000000, -0.940742, -0.339123,
0.065988, -0.921919, -0.381722,
0.065817, -0.892927, -0.445365,
0.131749, -0.896801, -0.422364,
0.196015, -0.865871, -0.460267,
0.000000, -0.884965, -0.465658,
0.065758, -0.859848, -0.506298,
-0.065758, -0.859848, 0.506298,
-0.065817, -0.892927, 0.445365,
-0.131749, -0.896801, 0.422364,
-0.065988, -0.921919, 0.381722,
-0.066042, -0.946198, 0.316778,
-0.196015, -0.865871, 0.460267,
-0.196412, -0.896401, 0.397351,
-0.132570, -0.947822, 0.289929,
-0.066567, -0.966643, 0.247326,
-0.198387, -0.922682, 0.330606,
-0.199094, -0.944433, 0.261550,
-0.264151, -0.936050, 0.232455,
-0.198337, -0.961188, 0.191780,
-0.262661, -0.891742, 0.368518,
-0.066604, -0.981483, 0.179596,
-0.133256, -0.979426, 0.151549,
-0.066709, -0.991648, 0.110385,
-0.066741, -0.996917, 0.041248,
-0.198580, -0.972370, 0.122729,
-0.322525, -0.855337, 0.405434,
-0.324864, -0.882837, 0.339209,
-0.385906, -0.869725, 0.307659,
-0.325929, -0.905832, 0.270627,
-0.325530, -0.923879, 0.201189,
-0.442867, -0.828447, 0.342848,
-0.444464, -0.852640, 0.274694,
-0.133523, -0.991046, 0.000000,
-0.066741, -0.996917, -0.041248,
-0.200368, -0.978852, 0.041251,
-0.200368, -0.978852, -0.041251,
-0.265518, -0.960655, -0.081501,
-0.198580, -0.972370, -0.122729,
-0.265518, -0.960655, 0.081501,
-0.334140, -0.941618, -0.041302,
-0.332342, -0.935130, -0.122800,
-0.334140, -0.941618, 0.041302,
-0.400968, -0.916092, 0.000000,
-0.464213, -0.884760, 0.041299,
-0.464213, -0.884760, -0.041299,
-0.332342, -0.935130, 0.122800,
-0.394136, -0.904839, -0.161006,
-0.325530, -0.923879, -0.201189,
-0.460882, -0.879344, -0.119755,
-0.454262, -0.868535, -0.198227,
-0.511205, -0.827147, -0.233445,
-0.444464, -0.852640, -0.274694,
-0.524110, -0.848029, -0.078459,
-0.394136, -0.904839, 0.161006,
-0.454262, -0.868535, 0.198227,
-0.460882, -0.879344, 0.119755,
-0.524110, -0.848029, 0.078459,
-0.511205, -0.827147, 0.233445,
-0.066709, -0.991648, -0.110385,
-0.133256, -0.979426, -0.151549,
-0.066604, -0.981483, -0.179596,
-0.066567, -0.966643, -0.247326,
-0.198337, -0.961188, -0.191780,
-0.132570, -0.947822, -0.289929,
-0.066042, -0.946198, -0.316778,
-0.199094, -0.944433, -0.261550,
-0.198387, -0.922682, -0.330606,
-0.262661, -0.891742, -0.368518,
-0.196412, -0.896401, -0.397351,
-0.264151, -0.936050, -0.232455,
-0.065988, -0.921919, -0.381722,
-0.131749, -0.896801, -0.422364,
-0.065817, -0.892927, -0.445365,
-0.065758, -0.859848, -0.506298,
-0.196015, -0.865871, -0.460267,
-0.325929, -0.905832, -0.270627,
-0.385906, -0.869725, -0.307659,
-0.324864, -0.882837, -0.339209,
-0.322525, -0.855337, -0.405434,
-0.442867, -0.828447, -0.342848,
-0.554740, -0.785715, 0.273707,
-0.564513, -0.801504, 0.197274,
-0.615456, -0.771841, 0.159613,
-0.571140, -0.812169, 0.119078,
-0.574458, -0.817523, 0.040675,
-0.656006, -0.727958, 0.199331,
-0.662842, -0.738886, 0.121193,
-0.622292, -0.782786, 0.000000,
-0.574458, -0.817523, -0.040675,
-0.667502, -0.743496, 0.040677,
-0.667502, -0.743496, -0.040677,
-0.708969, -0.700622, -0.080573,
-0.662842, -0.738886, -0.121193,
-0.708969, -0.700622, 0.080573,
-0.571140, -0.812169, -0.119078,
-0.615456, -0.771841, -0.159613,
-0.564513, -0.801504, -0.197274,
-0.554740, -0.785715, -0.273707,
-0.656006, -0.727958, -0.199331,
-0.744727, -0.656282, 0.121144,
-0.749387, -0.660884, 0.040642,
-0.786433, -0.617676, 0.000000,
-0.749387, -0.660884, -0.040642,
-0.744727, -0.656282, -0.121144,
-0.819207, -0.572056, 0.040640,
-0.819207, -0.572056, -0.040640,
0.106494, 0.827110, 0.551859,
0.147414, 0.790170, 0.594895,
0.172532, 0.830812, 0.529135,
0.237229, 0.830359, 0.504209,
0.187432, 0.749786, 0.634579,
0.280319, 0.790112, 0.545109,
0.303801, 0.825175, 0.476226,
0.255180, 0.749435, 0.610926,
0.322141, 0.745339, 0.583691,
0.361592, 0.697037, 0.619186,
0.386730, 0.737712, 0.553372,
0.228900, 0.703983, 0.672319,
0.366027, 0.816233, 0.446977,
0.408286, 0.772575, 0.486240,
0.427135, 0.803016, 0.415597,
0.485712, 0.785899, 0.382683,
0.448259, 0.725299, 0.522499,
0.267381, 0.656620, 0.705236,
0.305212, 0.605742, 0.734794,
0.334692, 0.653497, 0.678913,
0.399769, 0.646840, 0.649448,
0.341435, 0.552453, 0.760406,
0.536784, 0.735011, 0.414273,
0.552453, 0.760406, 0.341435,
0.517970, 0.704471, 0.485208,
0.584716, 0.678977, 0.443957,
0.628313, 0.619242, 0.470917,
0.646840, 0.649448, 0.399769,
0.496441, 0.669612, 0.552418,
0.607478, 0.581952, 0.540650,
0.669612, 0.552418, 0.496441,
0.540650, 0.607478, 0.581952,
0.581952, 0.540650, 0.607478,
0.552418, 0.496441, 0.669612,
0.619242, 0.470917, 0.628313,
0.470917, 0.628313, 0.619242,
0.704471, 0.485208, 0.517970,
0.725299, 0.522499, 0.448259,
0.678977, 0.443957, 0.584716,
0.735011, 0.414273, 0.536784,
0.760406, 0.341435, 0.552453,
0.785899, 0.382683, 0.485712,
0.649448, 0.399769, 0.646840,
0.443957, 0.584716, 0.678977,
0.414273, 0.536784, 0.735011,
0.485208, 0.517970, 0.704471,
0.522499, 0.448259, 0.725299,
0.382683, 0.485712, 0.785899,
0.605742, 0.734794, 0.305212,
0.653497, 0.678913, 0.334692,
0.656620, 0.705236, 0.267381,
0.703983, 0.672319, 0.228900,
0.697037, 0.619186, 0.361592,
0.749435, 0.610926, 0.255180,
0.749786, 0.634579, 0.187432,
0.745339, 0.583691, 0.322141,
0.790112, 0.545109, 0.280319,
0.825175, 0.476226, 0.303801,
0.830359, 0.504209, 0.237229,
0.737712, 0.553372, 0.386730,
0.790170, 0.594895, 0.147414,
0.830812, 0.529135, 0.172532,
0.827110, 0.551859, 0.106494,
0.772575, 0.486240, 0.408286,
0.803016, 0.415597, 0.427135,
0.816233, 0.446977, 0.366027,
0.415597, 0.427135, 0.803016,
0.446977, 0.366027, 0.816233,
0.486240, 0.408286, 0.772575,
0.553372, 0.386730, 0.737712,
0.476226, 0.303801, 0.825175,
0.583691, 0.322141, 0.745339,
0.619186, 0.361592, 0.697037,
0.545109, 0.280319, 0.790112,
0.610926, 0.255180, 0.749435,
0.634579, 0.187432, 0.749786,
0.672319, 0.228900, 0.703983,
0.504209, 0.237229, 0.830359,
0.678913, 0.334692, 0.653497,
0.705236, 0.267381, 0.656620,
0.734794, 0.305212, 0.605742,
0.529135, 0.172532, 0.830812,
0.551859, 0.106494, 0.827110,
0.594895, 0.147414, 0.790170,
0.106494, 0.827110, -0.551859,
0.172532, 0.830812, -0.529135,
0.147414, 0.790170, -0.594895,
0.187432, 0.749786, -0.634579,
0.237229, 0.830359, -0.504209,
0.255180, 0.749435, -0.610926,
0.228900, 0.703983, -0.672319,
0.280319, 0.790112, -0.545109,
0.322141, 0.745339, -0.583691,
0.386730, 0.737712, -0.553372,
0.361592, 0.697037, -0.619186,
0.303801, 0.825175, -0.476226,
0.267381, 0.656620, -0.705236,
0.334692, 0.653497, -0.678913,
0.305212, 0.605742, -0.734794,
0.341435, 0.552453, -0.760406,
0.399769, 0.646840, -0.649448,
0.366027, 0.816233, -0.446977,
0.427135, 0.803016, -0.415597,
0.408286, 0.772575, -0.486240,
0.448259, 0.725299, -0.522499,
0.485712, 0.785899, -0.382683,
0.414273, 0.536784, -0.735011,
0.382683, 0.485712, -0.785899,
0.443957, 0.584716, -0.678977,
0.485208, 0.517970, -0.704471,
0.552418, 0.496441, -0.669612,
0.522499, 0.448259, -0.725299,
0.470917, 0.628313, -0.619242,
0.581952, 0.540650, -0.607478,
0.619242, 0.470917, -0.628313,
0.540650, 0.607478, -0.581952,
0.607478, 0.581952, -0.540650,
0.628313, 0.619242, -0.470917,
0.669612, 0.552418, -0.496441,
0.496441, 0.669612, -0.552418,
0.678977, 0.443957, -0.584716,
0.649448, 0.399769, -0.646840,
0.704471, 0.485208, -0.517970,
0.735011, 0.414273, -0.536784,
0.785899, 0.382683, -0.485712,
0.760406, 0.341435, -0.552453,
0.725299, 0.522499, -0.448259,
0.517970, 0.704471, -0.485208,
0.536784, 0.735011, -0.414273,
0.584716, 0.678977, -0.443957,
0.646840, 0.649448, -0.399769,
0.552453, 0.760406, -0.341435,
0.415597, 0.427135, -0.803016,
0.486240, 0.408286, -0.772575,
0.446977, 0.366027, -0.816233,
0.476226, 0.303801, -0.825175,
0.553372, 0.386730, -0.737712,
0.545109, 0.280319, -0.790112,
0.504209, 0.237229, -0.830359,
0.583691, 0.322141, -0.745339,
0.610926, 0.255180, -0.749435,
0.672319, 0.228900, -0.703983,
0.634579, 0.187432, -0.749786,
0.619186, 0.361592, -0.697037,
0.529135, 0.172532, -0.830812,
0.594895, 0.147414, -0.790170,
0.551859, 0.106494, -0.827110,
0.678913, 0.334692, -0.653497,
0.734794, 0.305212, -0.605742,
0.705236, 0.267381, -0.656620,
0.605742, 0.734794, -0.305212,
0.656620, 0.705236, -0.267381,
0.653497, 0.678913, -0.334692,
0.697037, 0.619186, -0.361592,
0.703983, 0.672319, -0.228900,
0.745339, 0.583691, -0.322141,
0.737712, 0.553372, -0.386730,
0.749435, 0.610926, -0.255180,
0.790112, 0.545109, -0.280319,
0.830359, 0.504209, -0.237229,
0.825175, 0.476226, -0.303801,
0.749786, 0.634579, -0.187432,
0.772575, 0.486240, -0.408286,
0.816233, 0.446977, -0.366027,
0.803016, 0.415597, -0.427135,
0.790170, 0.594895, -0.147414,
0.827110, 0.551859, -0.106494,
0.830812, 0.529135, -0.172532,
0.551859, -0.106494, 0.827110,
0.529135, -0.172532, 0.830812,
0.594895, -0.147414, 0.790170,
0.634579, -0.187432, 0.749786,
0.504209, -0.237229, 0.830359,
0.610926, -0.255180, 0.749435,
0.672319, -0.228900, 0.703983,
0.545109, -0.280319, 0.790112,
0.583691, -0.322141, 0.745339,
0.553372, -0.386730, 0.737712,
0.619186, -0.361592, 0.697037,
0.476226, -0.303801, 0.825175,
0.705236, -0.267381, 0.656620,
0.678913, -0.334692, 0.653497,
0.734794, -0.305212, 0.605742,
0.760406, -0.341435, 0.552453,
0.649448, -0.399769, 0.646840,
0.446977, -0.366027, 0.816233,
0.415597, -0.427135, 0.803016,
0.486240, -0.408286, 0.772575,
0.522499, -0.448259, 0.725299,
0.382683, -0.485712, 0.785899,
0.735011, -0.414273, 0.536784,
0.785899, -0.382683, 0.485712,
0.678977, -0.443957, 0.584716,
0.704471, -0.485208, 0.517970,
0.669612, -0.552418, 0.496441,
0.725299, -0.522499, 0.448259,
0.619242, -0.470917, 0.628313,
0.607478, -0.581952, 0.540650,
0.628313, -0.619242, 0.470917,
0.581952, -0.540650, 0.607478,
0.540650, -0.607478, 0.581952,
0.470917, -0.628313, 0.619242,
0.496441, -0.669612, 0.552418,
0.552418, -0.496441, 0.669612,
0.584716, -0.678977, 0.443957,
0.646840, -0.649448, 0.399769,
0.517970, -0.704471, 0.485208,
0.536784, -0.735011, 0.414273,
0.485712, -0.785899, 0.382683,
0.552453, -0.760406, 0.341435,
0.448259, -0.725299, 0.522499,
0.485208, -0.517970, 0.704471,
0.414273, -0.536784, 0.735011,
0.443957, -0.584716, 0.678977,
0.399769, -0.646840, 0.649448,
0.341435, -0.552453, 0.760406,
0.803016, -0.415597, 0.427135,
0.772575, -0.486240, 0.408286,
0.816233, -0.446977, 0.366027,
0.825175, -0.476226, 0.303801,
0.737712, -0.553372, 0.386730,
0.790112, -0.545109, 0.280319,
0.830359, -0.504209, 0.237229,
0.745339, -0.583691, 0.322141,
0.749435, -0.610926, 0.255180,
0.703983, -0.672319, 0.228900,
0.749786, -0.634579, 0.187432,
0.697037, -0.619186, 0.361592,
0.830812, -0.529135, 0.172532,
0.790170, -0.594895, 0.147414,
0.827110, -0.551859, 0.106494,
0.653497, -0.678913, 0.334692,
0.605742, -0.734794, 0.305212,
0.656620, -0.705236, 0.267381,
0.305212, -0.605742, 0.734794,
0.267381, -0.656620, 0.705236,
0.334692, -0.653497, 0.678913,
0.361592, -0.697037, 0.619186,
0.228900, -0.703983, 0.672319,
0.322141, -0.745339, 0.583691,
0.386730, -0.737712, 0.553372,
0.255180, -0.749435, 0.610926,
0.280319, -0.790112, 0.545109,
0.237229, -0.830359, 0.504209,
0.303801, -0.825175, 0.476226,
0.187432, -0.749786, 0.634579,
0.408286, -0.772575, 0.486240,
0.366027, -0.816233, 0.446977,
0.427135, -0.803016, 0.415597,
0.147414, -0.790170, 0.594895,
0.106494, -0.827110, 0.551859,
0.172532, -0.830812, 0.529135,
0.551859, -0.106494, -0.827110,
0.594895, -0.147414, -0.790170,
0.529135, -0.172532, -0.830812,
0.504209, -0.237229, -0.830359,
0.634579, -0.187432, -0.749786,
0.545109, -0.280319, -0.790112,
0.476226, -0.303801, -0.825175,
0.610926, -0.255180, -0.749435,
0.583691, -0.322141, -0.745339,
0.619186, -0.361592, -0.697037,
0.553372, -0.386730, -0.737712,
0.672319, -0.228900, -0.703983,
0.446977, -0.366027, -0.816233,
0.486240, -0.408286, -0.772575,
0.415597, -0.427135, -0.803016,
0.382683, -0.485712, -0.785899,
0.522499, -0.448259, -0.725299,
0.705236, -0.267381, -0.656620,
0.734794, -0.305212, -0.605742,
0.678913, -0.334692, -0.653497,
0.649448, -0.399769, -0.646840,
0.760406, -0.341435, -0.552453,
0.414273, -0.536784, -0.735011,
0.341435, -0.552453, -0.760406,
0.485208, -0.517970, -0.704471,
0.443957, -0.584716, -0.678977,
0.470917, -0.628313, -0.619242,
0.399769, -0.646840, -0.649448,
0.552418, -0.496441, -0.669612,
0.540650, -0.607478, -0.581952,
0.496441, -0.669612, -0.552418,
0.581952, -0.540650, -0.607478,
0.607478, -0.581952, -0.540650,
0.669612, -0.552418, -0.496441,
0.628313, -0.619242, -0.470917,
0.619242, -0.470917, -0.628313,
0.517970, -0.704471, -0.485208,
0.448259, -0.725299, -0.522499,
0.584716, -0.678977, -0.443957,
0.536784, -0.735011, -0.414273,
0.552453, -0.760406, -0.341435,
0.485712, -0.785899, -0.382683,
0.646840, -0.649448, -0.399769,
0.678977, -0.443957, -0.584716,
0.735011, -0.414273, -0.536784,
0.704471, -0.485208, -0.517970,
0.725299, -0.522499, -0.448259,
0.785899, -0.382683, -0.485712,
0.305212, -0.605742, -0.734794,
0.334692, -0.653497, -0.678913,
0.267381, -0.656620, -0.705236,
0.228900, -0.703983, -0.672319,
0.361592, -0.697037, -0.619186,
0.255180, -0.749435, -0.610926,
0.187432, -0.749786, -0.634579,
0.322141, -0.745339, -0.583691,
0.280319, -0.790112, -0.545109,
0.303801, -0.825175, -0.476226,
0.237229, -0.830359, -0.504209,
0.386730, -0.737712, -0.553372,
0.147414, -0.790170, -0.594895,
0.172532, -0.830812, -0.529135,
0.106494, -0.827110, -0.551859,
0.408286, -0.772575, -0.486240,
0.427135, -0.803016, -0.415597,
0.366027, -0.816233, -0.446977,
0.803016, -0.415597, -0.427135,
0.816233, -0.446977, -0.366027,
0.772575, -0.486240, -0.408286,
0.737712, -0.553372, -0.386730,
0.825175, -0.476226, -0.303801,
0.745339, -0.583691, -0.322141,
0.697037, -0.619186, -0.361592,
0.790112, -0.545109, -0.280319,
0.749435, -0.610926, -0.255180,
0.749786, -0.634579, -0.187432,
0.703983, -0.672319, -0.228900,
0.830359, -0.504209, -0.237229,
0.653497, -0.678913, -0.334692,
0.656620, -0.705236, -0.267381,
0.605742, -0.734794, -0.305212,
0.830812, -0.529135, -0.172532,
0.827110, -0.551859, -0.106494,
0.790170, -0.594895, -0.147414,
-0.551859, 0.106494, 0.827110,
-0.529135, 0.172532, 0.830812,
-0.594895, 0.147414, 0.790170,
-0.634579, 0.187432, 0.749786,
-0.504209, 0.237229, 0.830359,
-0.610926, 0.255180, 0.749435,
-0.672319, 0.228900, 0.703983,
-0.545109, 0.280319, 0.790112,
-0.583691, 0.322141, 0.745339,
-0.553372, 0.386730, 0.737712,
-0.619186, 0.361592, 0.697037,
-0.476226, 0.303801, 0.825175,
-0.705236, 0.267381, 0.656620,
-0.678913, 0.334692, 0.653497,
-0.734794, 0.305212, 0.605742,
-0.760406, 0.341435, 0.552453,
-0.649448, 0.399769, 0.646840,
-0.446977, 0.366027, 0.816233,
-0.415597, 0.427135, 0.803016,
-0.486240, 0.408286, 0.772575,
-0.522499, 0.448259, 0.725299,
-0.382683, 0.485712, 0.785899,
-0.735011, 0.414273, 0.536784,
-0.785899, 0.382683, 0.485712,
-0.678977, 0.443957, 0.584716,
-0.704471, 0.485208, 0.517970,
-0.669612, 0.552418, 0.496441,
-0.725299, 0.522499, 0.448259,
-0.619242, 0.470917, 0.628313,
-0.607478, 0.581952, 0.540650,
-0.628313, 0.619242, 0.470917,
-0.581952, 0.540650, 0.607478,
-0.540650, 0.607478, 0.581952,
-0.470917, 0.628313, 0.619242,
-0.496441, 0.669612, 0.552418,
-0.552418, 0.496441, 0.669612,
-0.584716, 0.678977, 0.443957,
-0.646840, 0.649448, 0.399769,
-0.517970, 0.704471, 0.485208,
-0.536784, 0.735011, 0.414273,
-0.485712, 0.785899, 0.382683,
-0.552453, 0.760406, 0.341435,
-0.448259, 0.725299, 0.522499,
-0.485208, 0.517970, 0.704471,
-0.414273, 0.536784, 0.735011,
-0.443957, 0.584716, 0.678977,
-0.399769, 0.646840, 0.649448,
-0.341435, 0.552453, 0.760406,
-0.803016, 0.415597, 0.427135,
-0.772575, 0.486240, 0.408286,
-0.816233, 0.446977, 0.366027,
-0.825175, 0.476226, 0.303801,
-0.737712, 0.553372, 0.386730,
-0.790112, 0.545109, 0.280319,
-0.830359, 0.504209, 0.237229,
-0.745339, 0.583691, 0.322141,
-0.749435, 0.610926, 0.255180,
-0.703983, 0.672319, 0.228900,
-0.749786, 0.634579, 0.187432,
-0.697037, 0.619186, 0.361592,
-0.830812, 0.529135, 0.172532,
-0.790170, 0.594895, 0.147414,
-0.827110, 0.551859, 0.106494,
-0.653497, 0.678913, 0.334692,
-0.605742, 0.734794, 0.305212,
-0.656620, 0.705236, 0.267381,
-0.305212, 0.605742, 0.734794,
-0.267381, 0.656620, 0.705236,
-0.334692, 0.653497, 0.678913,
-0.361592, 0.697037, 0.619186,
-0.228900, 0.703983, 0.672319,
-0.322141, 0.745339, 0.583691,
-0.386730, 0.737712, 0.553372,
-0.255180, 0.749435, 0.610926,
-0.280319, 0.790112, 0.545109,
-0.237229, 0.830359, 0.504209,
-0.303801, 0.825175, 0.476226,
-0.187432, 0.749786, 0.634579,
-0.408286, 0.772575, 0.486240,
-0.366027, 0.816233, 0.446977,
-0.427135, 0.803016, 0.415597,
-0.147414, 0.790170, 0.594895,
-0.106494, 0.827110, 0.551859,
-0.172532, 0.830812, 0.529135,
-0.551859, 0.106494, -0.827110,
-0.594895, 0.147414, -0.790170,
-0.529135, 0.172532, -0.830812,
-0.504209, 0.237229, -0.830359,
-0.634579, 0.187432, -0.749786,
-0.545109, 0.280319, -0.790112,
-0.476226, 0.303801, -0.825175,
-0.610926, 0.255180, -0.749435,
-0.583691, 0.322141, -0.745339,
-0.619186, 0.361592, -0.697037,
-0.553372, 0.386730, -0.737712,
-0.672319, 0.228900, -0.703983,
-0.446977, 0.366027, -0.816233,
-0.486240, 0.408286, -0.772575,
-0.415597, 0.427135, -0.803016,
-0.382683, 0.485712, -0.785899,
-0.522499, 0.448259, -0.725299,
-0.705236, 0.267381, -0.656620,
-0.734794, 0.305212, -0.605742,
-0.678913, 0.334692, -0.653497,
-0.649448, 0.399769, -0.646840,
-0.760406, 0.341435, -0.552453,
-0.414273, 0.536784, -0.735011,
-0.341435, 0.552453, -0.760406,
-0.485208, 0.517970, -0.704471,
-0.443957, 0.584716, -0.678977,
-0.470917, 0.628313, -0.619242,
-0.399769, 0.646840, -0.649448,
-0.552418, 0.496441, -0.669612,
-0.540650, 0.607478, -0.581952,
-0.496441, 0.669612, -0.552418,
-0.581952, 0.540650, -0.607478,
-0.607478, 0.581952, -0.540650,
-0.669612, 0.552418, -0.496441,
-0.628313, 0.619242, -0.470917,
-0.619242, 0.470917, -0.628313,
-0.517970, 0.704471, -0.485208,
-0.448259, 0.725299, -0.522499,
-0.584716, 0.678977, -0.443957,
-0.536784, 0.735011, -0.414273,
-0.552453, 0.760406, -0.341435,
-0.485712, 0.785899, -0.382683,
-0.646840, 0.649448, -0.399769,
-0.678977, 0.443957, -0.584716,
-0.735011, 0.414273, -0.536784,
-0.704471, 0.485208, -0.517970,
-0.725299, 0.522499, -0.448259,
-0.785899, 0.382683, -0.485712,
-0.305212, 0.605742, -0.734794,
-0.334692, 0.653497, -0.678913,
-0.267381, 0.656620, -0.705236,
-0.228900, 0.703983, -0.672319,
-0.361592, 0.697037, -0.619186,
-0.255180, 0.749435, -0.610926,
-0.187432, 0.749786, -0.634579,
-0.322141, 0.745339, -0.583691,
-0.280319, 0.790112, -0.545109,
-0.303801, 0.825175, -0.476226,
-0.237229, 0.830359, -0.504209,
-0.386730, 0.737712, -0.553372,
-0.147414, 0.790170, -0.594895,
-0.172532, 0.830812, -0.529135,
-0.106494, 0.827110, -0.551859,
-0.408286, 0.772575, -0.486240,
-0.427135, 0.803016, -0.415597,
-0.366027, 0.816233, -0.446977,
-0.803016, 0.415597, -0.427135,
-0.816233, 0.446977, -0.366027,
-0.772575, 0.486240, -0.408286,
-0.737712, 0.553372, -0.386730,
-0.825175, 0.476226, -0.303801,
-0.745339, 0.583691, -0.322141,
-0.697037, 0.619186, -0.361592,
-0.790112, 0.545109, -0.280319,
-0.749435, 0.610926, -0.255180,
-0.749786, 0.634579, -0.187432,
-0.703983, 0.672319, -0.228900,
-0.830359, 0.504209, -0.237229,
-0.653497, 0.678913, -0.334692,
-0.656620, 0.705236, -0.267381,
-0.605742, 0.734794, -0.305212,
-0.830812, 0.529135, -0.172532,
-0.827110, 0.551859, -0.106494,
-0.790170, 0.594895, -0.147414,
-0.551859, -0.106494, 0.827110,
-0.594895, -0.147414, 0.790170,
-0.529135, -0.172532, 0.830812,
-0.504209, -0.237229, 0.830359,
-0.634579, -0.187432, 0.749786,
-0.545109, -0.280319, 0.790112,
-0.476226, -0.303801, 0.825175,
-0.610926, -0.255180, 0.749435,
-0.583691, -0.322141, 0.745339,
-0.619186, -0.361592, 0.697037,
-0.553372, -0.386730, 0.737712,
-0.672319, -0.228900, 0.703983,
-0.446977, -0.366027, 0.816233,
-0.486240, -0.408286, 0.772575,
-0.415597, -0.427135, 0.803016,
-0.382683, -0.485712, 0.785899,
-0.522499, -0.448259, 0.725299,
-0.705236, -0.267381, 0.656620,
-0.734794, -0.305212, 0.605742,
-0.678913, -0.334692, 0.653497,
-0.649448, -0.399769, 0.646840,
-0.760406, -0.341435, 0.552453,
-0.414273, -0.536784, 0.735011,
-0.341435, -0.552453, 0.760406,
-0.485208, -0.517970, 0.704471,
-0.443957, -0.584716, 0.678977,
-0.470917, -0.628313, 0.619242,
-0.399769, -0.646840, 0.649448,
-0.552418, -0.496441, 0.669612,
-0.540650, -0.607478, 0.581952,
-0.496441, -0.669612, 0.552418,
-0.581952, -0.540650, 0.607478,
-0.607478, -0.581952, 0.540650,
-0.669612, -0.552418, 0.496441,
-0.628313, -0.619242, 0.470917,
-0.619242, -0.470917, 0.628313,
-0.517970, -0.704471, 0.485208,
-0.448259, -0.725299, 0.522499,
-0.584716, -0.678977, 0.443957,
-0.536784, -0.735011, 0.414273,
-0.552453, -0.760406, 0.341435,
-0.485712, -0.785899, 0.382683,
-0.646840, -0.649448, 0.399769,
-0.678977, -0.443957, 0.584716,
-0.735011, -0.414273, 0.536784,
-0.704471, -0.485208, 0.517970,
-0.725299, -0.522499, 0.448259,
-0.785899, -0.382683, 0.485712,
-0.305212, -0.605742, 0.734794,
-0.334692, -0.653497, 0.678913,
-0.267381, -0.656620, 0.705236,
-0.228900, -0.703983, 0.672319,
-0.361592, -0.697037, 0.619186,
-0.255180, -0.749435, 0.610926,
-0.187432, -0.749786, 0.634579,
-0.322141, -0.745339, 0.583691,
-0.280319, -0.790112, 0.545109,
-0.303801, -0.825175, 0.476226,
-0.237229, -0.830359, 0.504209,
-0.386730, -0.737712, 0.553372,
-0.147414, -0.790170, 0.594895,
-0.172532, -0.830812, 0.529135,
-0.106494, -0.827110, 0.551859,
-0.408286, -0.772575, 0.486240,
-0.427135, -0.803016, 0.415597,
-0.366027, -0.816233, 0.446977,
-0.803016, -0.415597, 0.427135,
-0.816233, -0.446977, 0.366027,
-0.772575, -0.486240, 0.408286,
-0.737712, -0.553372, 0.386730,
-0.825175, -0.476226, 0.303801,
-0.745339, -0.583691, 0.322141,
-0.697037, -0.619186, 0.361592,
-0.790112, -0.545109, 0.280319,
-0.749435, -0.610926, 0.255180,
-0.749786, -0.634579, 0.187432,
-0.703983, -0.672319, 0.228900,
-0.830359, -0.504209, 0.237229,
-0.653497, -0.678913, 0.334692,
-0.656620, -0.705236, 0.267381,
-0.605742, -0.734794, 0.305212,
-0.830812, -0.529135, 0.172532,
-0.827110, -0.551859, 0.106494,
-0.790170, -0.594895, 0.147414,
-0.551859, -0.106494, -0.827110,
-0.529135, -0.172532, -0.830812,
-0.594895, -0.147414, -0.790170,
-0.634579, -0.187432, -0.749786,
-0.504209, -0.237229, -0.830359,
-0.610926, -0.255180, -0.749435,
-0.672319, -0.228900, -0.703983,
-0.545109, -0.280319, -0.790112,
-0.583691, -0.322141, -0.745339,
-0.553372, -0.386730, -0.737712,
-0.619186, -0.361592, -0.697037,
-0.476226, -0.303801, -0.825175,
-0.705236, -0.267381, -0.656620,
-0.678913, -0.334692, -0.653497,
-0.734794, -0.305212, -0.605742,
-0.760406, -0.341435, -0.552453,
-0.649448, -0.399769, -0.646840,
-0.446977, -0.366027, -0.816233,
-0.415597, -0.427135, -0.803016,
-0.486240, -0.408286, -0.772575,
-0.522499, -0.448259, -0.725299,
-0.382683, -0.485712, -0.785899,
-0.735011, -0.414273, -0.536784,
-0.785899, -0.382683, -0.485712,
-0.678977, -0.443957, -0.584716,
-0.704471, -0.485208, -0.517970,
-0.669612, -0.552418, -0.496441,
-0.725299, -0.522499, -0.448259,
-0.619242, -0.470917, -0.628313,
-0.607478, -0.581952, -0.540650,
-0.628313, -0.619242, -0.470917,
-0.581952, -0.540650, -0.607478,
-0.540650, -0.607478, -0.581952,
-0.470917, -0.628313, -0.619242,
-0.496441, -0.669612, -0.552418,
-0.552418, -0.496441, -0.669612,
-0.584716, -0.678977, -0.443957,
-0.646840, -0.649448, -0.399769,
-0.517970, -0.704471, -0.485208,
-0.536784, -0.735011, -0.414273,
-0.485712, -0.785899, -0.382683,
-0.552453, -0.760406, -0.341435,
-0.448259, -0.725299, -0.522499,
-0.485208, -0.517970, -0.704471,
-0.414273, -0.536784, -0.735011,
-0.443957, -0.584716, -0.678977,
-0.399769, -0.646840, -0.649448,
-0.341435, -0.552453, -0.760406,
-0.803016, -0.415597, -0.427135,
-0.772575, -0.486240, -0.408286,
-0.816233, -0.446977, -0.366027,
-0.825175, -0.476226, -0.303801,
-0.737712, -0.553372, -0.386730,
-0.790112, -0.545109, -0.280319,
-0.830359, -0.504209, -0.237229,
-0.745339, -0.583691, -0.322141,
-0.749435, -0.610926, -0.255180,
-0.703983, -0.672319, -0.228900,
-0.749786, -0.634579, -0.187432,
-0.697037, -0.619186, -0.361592,
-0.830812, -0.529135, -0.172532,
-0.790170, -0.594895, -0.147414,
-0.827110, -0.551859, -0.106494,
-0.653497, -0.678913, -0.334692,
-0.605742, -0.734794, -0.305212,
-0.656620, -0.705236, -0.267381,
-0.305212, -0.605742, -0.734794,
-0.267381, -0.656620, -0.705236,
-0.334692, -0.653497, -0.678913,
-0.361592, -0.697037, -0.619186,
-0.228900, -0.703983, -0.672319,
-0.322141, -0.745339, -0.583691,
-0.386730, -0.737712, -0.553372,
-0.255180, -0.749435, -0.610926,
-0.280319, -0.790112, -0.545109,
-0.237229, -0.830359, -0.504209,
-0.303801, -0.825175, -0.476226,
-0.187432, -0.749786, -0.634579,
-0.408286, -0.772575, -0.486240,
-0.366027, -0.816233, -0.446977,
-0.427135, -0.803016, -0.415597,
-0.147414, -0.790170, -0.594895,
-0.106494, -0.827110, -0.551859,
-0.172532, -0.830812, -0.529135,
0.516324, 0.032898, 0.855762,
0.486380, 0.032906, 0.873128,
0.495991, 0.000000, 0.868328,
0.476118, 0.065827, 0.876914,
0.455785, 0.032928, 0.889480,
0.424710, 0.032936, 0.904730,
0.434767, 0.000000, 0.900543,
0.495666, 0.098538, 0.862905,
0.465287, 0.098590, 0.879653,
0.413796, 0.065943, 0.907978,
0.392778, 0.033014, 0.919041,
0.434130, 0.098843, 0.895411,
0.402375, 0.098950, 0.910112,
0.390534, 0.131782, 0.911107,
0.370201, 0.098882, 0.923674,
0.453679, 0.131555, 0.881402,
0.360716, 0.033021, 0.932091,
0.371461, 0.000000, 0.928448,
0.349460, 0.066055, 0.934620,
0.328151, 0.033041, 0.944047,
0.295262, 0.033048, 0.954845,
0.306378, 0.000000, 0.951910,
0.337778, 0.098929, 0.936012,
0.472637, 0.163706, 0.865919,
0.441675, 0.164016, 0.882055,
0.429071, 0.196334, 0.881675,
0.410104, 0.164179, 0.897140,
0.378097, 0.164167, 0.911094,
0.447345, 0.228090, 0.864787,
0.415958, 0.228338, 0.880251,
0.282237, 0.066348, 0.957048,
0.260463, 0.033305, 0.964909,
0.303549, 0.099370, 0.947620,
0.268847, 0.099650, 0.958014,
0.255190, 0.132768, 0.957732,
0.233871, 0.099744, 0.967138,
0.324291, 0.132199, 0.936674,
0.275921, 0.165941, 0.946748,
0.241032, 0.166080, 0.956201,
0.310525, 0.165615, 0.936027,
0.296273, 0.198871, 0.934170,
0.316139, 0.231386, 0.920064,
0.281637, 0.231780, 0.931106,
0.344643, 0.165129, 0.924096,
0.226811, 0.198844, 0.953424,
0.206081, 0.165671, 0.964409,
0.247162, 0.231772, 0.940847,
0.212293, 0.231436, 0.949405,
0.197584, 0.263677, 0.944158,
0.177234, 0.230749, 0.956736,
0.267028, 0.264286, 0.926741,
0.364213, 0.197527, 0.910128,
0.383186, 0.229687, 0.894658,
0.349848, 0.230712, 0.907953,
0.335103, 0.263541, 0.904573,
0.401461, 0.261444, 0.877768,
0.226892, 0.033310, 0.973350,
0.239828, 0.000000, 0.970815,
0.213589, 0.066626, 0.974649,
0.192973, 0.033324, 0.980638,
0.158898, 0.033328, 0.986732,
0.172130, 0.000000, 0.985074,
0.200013, 0.099776, 0.974700,
0.145079, 0.066698, 0.987169,
0.124250, 0.033376, 0.991690,
0.165681, 0.099996, 0.981097,
0.131070, 0.100061, 0.986311,
0.116967, 0.133245, 0.984157,
0.096377, 0.099955, 0.990313,
0.185946, 0.133106, 0.973502,
0.089833, 0.033381, 0.995397,
0.103609, 0.000000, 0.994618,
0.075862, 0.066765, 0.994881,
0.055231, 0.033392, 0.997915,
0.020640, 0.033396, 0.999229,
0.034591, 0.000000, 0.999402,
0.061793, 0.099983, 0.993068,
0.171797, 0.165925, 0.971058,
0.157351, 0.198580, 0.967371,
0.137228, 0.166025, 0.976527,
0.102565, 0.165954, 0.980785,
0.142705, 0.230901, 0.962455,
0.419912, 0.291383, 0.859517,
0.387280, 0.292822, 0.874225,
0.372561, 0.323903, 0.869647,
0.354076, 0.293939, 0.887823,
0.320484, 0.294703, 0.900244,
0.390470, 0.353281, 0.850133,
0.357403, 0.354516, 0.864050,
0.305118, 0.325609, 0.894920,
0.286117, 0.295186, 0.911593,
0.323632, 0.355602, 0.876818,
0.289370, 0.356197, 0.888475,
0.273346, 0.386299, 0.880940,
0.254807, 0.356275, 0.898967,
0.341572, 0.385011, 0.857377,
0.251736, 0.295269, 0.921654,
0.236060, 0.325939, 0.915445,
0.216955, 0.295021, 0.930534,
0.181964, 0.294425, 0.938191,
0.220102, 0.356133, 0.908143,
0.359158, 0.413488, 0.836680,
0.325506, 0.414707, 0.849744,
0.308973, 0.443956, 0.841094,
0.291352, 0.415435, 0.861701,
0.256883, 0.415646, 0.872496,
0.326128, 0.471716, 0.819222,
0.292081, 0.472596, 0.831470,
0.000000, 0.066798, 0.997767,
-0.020640, 0.033396, 0.999229,
0.020640, 0.100210, 0.994752,
-0.020640, 0.100210, 0.994752,
-0.041156, 0.133410, 0.990206,
-0.061793, 0.099983, 0.993068,
0.041156, 0.133410, 0.990206,
-0.020642, 0.167078, 0.985728,
-0.061795, 0.166851, 0.984044,
0.020642, 0.167078, 0.985728,
0.000000, 0.200538, 0.979686,
0.020642, 0.233570, 0.972121,
-0.020642, 0.233570, 0.972121,
0.061795, 0.166851, 0.984044,
-0.082059, 0.199641, 0.976427,
-0.102565, 0.165954, 0.980785,
-0.061424, 0.233124, 0.970505,
-0.102196, 0.232233, 0.967277,
-0.122082, 0.264412, 0.956652,
-0.142705, 0.230901, 0.962455,
-0.040784, 0.266183, 0.963059,
0.082059, 0.199641, 0.976427,
0.102196, 0.232233, 0.967277,
0.061424, 0.233124, 0.970505,
0.040784, 0.266183, 0.963059,
0.122082, 0.264412, 0.956652,
-0.102238, 0.299185, 0.948702,
-0.142749, 0.297852, 0.943879,
-0.061453, 0.300080, 0.951932,
-0.082120, 0.333520, 0.939160,
-0.061865, 0.367185, 0.928088,
-0.102647, 0.366279, 0.924826,
-0.020669, 0.300527, 0.953549,
-0.020669, 0.367868, 0.929648,
-0.041197, 0.400628, 0.915314,
0.000000, 0.334425, 0.942422,
0.020669, 0.367868, 0.929648,
0.061865, 0.367185, 0.928088,
0.041197, 0.400628, 0.915314,
0.020669, 0.300527, 0.953549,
-0.020667, 0.432953, 0.901180,
-0.061863, 0.432270, 0.899620,
0.020667, 0.432953, 0.901180,
0.000000, 0.464610, 0.885516,
0.020667, 0.495384, 0.868428,
-0.020667, 0.495384, 0.868428,
0.061863, 0.432270, 0.899620,
0.061453, 0.300080, 0.951932,
0.102238, 0.299185, 0.948702,
0.082120, 0.333520, 0.939160,
0.102647, 0.366279, 0.924826,
0.142749, 0.297852, 0.943879,
-0.162122, 0.329196, 0.930240,
-0.181964, 0.294425, 0.938191,
-0.142013, 0.363521, 0.920698,
-0.181249, 0.360134, 0.915124,
-0.199999, 0.390469, 0.898629,
-0.220102, 0.356133, 0.908143,
-0.121761, 0.397184, 0.909626,
-0.179759, 0.424534, 0.887388,
-0.218637, 0.420581, 0.880517,
-0.140501, 0.427873, 0.892852,
-0.159117, 0.457934, 0.874631,
-0.138204, 0.490456, 0.860438,
-0.177481, 0.487173, 0.855081,
-0.101119, 0.430583, 0.896869,
-0.236644, 0.449711, 0.861255,
-0.256883, 0.415646, 0.872496,
-0.216004, 0.483108, 0.848498,
-0.254286, 0.478242, 0.840609,
-0.271440, 0.505993, 0.818713,
-0.292081, 0.472596, 0.831470,
-0.195090, 0.515629, 0.834306,
-0.080590, 0.462906, 0.882736,
-0.059926, 0.494553, 0.867079,
-0.099190, 0.492908, 0.864409,
-0.117537, 0.522087, 0.844754,
-0.039260, 0.525326, 0.849995,
0.162122, 0.329196, 0.930240,
0.181249, 0.360134, 0.915124,
0.142013, 0.363521, 0.920698,
0.121761, 0.397184, 0.909626,
0.199999, 0.390469, 0.898629,
0.140501, 0.427873, 0.892852,
0.101119, 0.430583, 0.896869,
0.179759, 0.424534, 0.887388,
0.159117, 0.457934, 0.874631,
0.177481, 0.487173, 0.855081,
0.138204, 0.490456, 0.860438,
0.218637, 0.420581, 0.880517,
0.080590, 0.462906, 0.882736,
0.099190, 0.492908, 0.864409,
0.059926, 0.494553, 0.867079,
0.039260, 0.525326, 0.849995,
0.117537, 0.522087, 0.844754,
0.236644, 0.449711, 0.861255,
0.254286, 0.478242, 0.840609,
0.216004, 0.483108, 0.848498,
0.195090, 0.515629, 0.834306,
0.271440, 0.505993, 0.818713,
-0.055231, 0.033392, 0.997915,
-0.034591, 0.000000, 0.999402,
-0.075862, 0.066765, 0.994881,
-0.089833, 0.033381, 0.995397,
-0.124250, 0.033376, 0.991690,
-0.103609, 0.000000, 0.994618,
-0.096377, 0.099955, 0.990313,
-0.145079, 0.066698, 0.987169,
-0.158898, 0.033328, 0.986732,
-0.131070, 0.100061, 0.986311,
-0.165681, 0.099996, 0.981097,
-0.185946, 0.133106, 0.973502,
-0.200013, 0.099776, 0.974700,
-0.116967, 0.133245, 0.984157,
-0.192973, 0.033324, 0.980638,
-0.172130, 0.000000, 0.985074,
-0.213589, 0.066626, 0.974649,
-0.226892, 0.033310, 0.973350,
-0.260463, 0.033305, 0.964909,
-0.239828, 0.000000, 0.970815,
-0.233871, 0.099744, 0.967138,
-0.137228, 0.166025, 0.976527,
-0.157351, 0.198580, 0.967371,
-0.171797, 0.165925, 0.971058,
-0.206081, 0.165671, 0.964409,
-0.177234, 0.230749, 0.956736,
-0.282237, 0.066348, 0.957048,
-0.295262, 0.033048, 0.954845,
-0.268847, 0.099650, 0.958014,
-0.303549, 0.099370, 0.947620,
-0.324291, 0.132199, 0.936674,
-0.337778, 0.098929, 0.936012,
-0.255190, 0.132768, 0.957732,
-0.310525, 0.165615, 0.936027,
-0.344643, 0.165129, 0.924096,
-0.275921, 0.165941, 0.946748,
-0.296273, 0.198871, 0.934170,
-0.281637, 0.231780, 0.931106,
-0.316139, 0.231386, 0.920064,
-0.241032, 0.166080, 0.956201,
-0.364213, 0.197527, 0.910128,
-0.378097, 0.164167, 0.911094,
-0.349848, 0.230712, 0.907953,
-0.383186, 0.229687, 0.894658,
-0.401461, 0.261444, 0.877768,
-0.415958, 0.228338, 0.880251,
-0.335103, 0.263541, 0.904573,
-0.226811, 0.198844, 0.953424,
-0.212293, 0.231436, 0.949405,
-0.247162, 0.231772, 0.940847,
-0.267028, 0.264286, 0.926741,
-0.197584, 0.263677, 0.944158,
-0.328151, 0.033041, 0.944047,
-0.306378, 0.000000, 0.951910,
-0.349460, 0.066055, 0.934620,
-0.360716, 0.033021, 0.932091,
-0.392778, 0.033014, 0.919041,
-0.371461, 0.000000, 0.928448,
-0.370201, 0.098882, 0.923674,
-0.413796, 0.065943, 0.907978,
-0.424710, 0.032936, 0.904730,
-0.402375, 0.098950, 0.910112,
-0.434130, 0.098843, 0.895411,
-0.453679, 0.131555, 0.881402,
-0.465287, 0.098590, 0.879653,
-0.390534, 0.131782, 0.911107,
-0.455785, 0.032928, 0.889480,
-0.434767, 0.000000, 0.900543,
-0.476118, 0.065827, 0.876914,
-0.486380, 0.032906, 0.873128,
-0.516324, 0.032898, 0.855762,
-0.495991, 0.000000, 0.868328,
-0.495666, 0.098538, 0.862905,
-0.410104, 0.164179, 0.897140,
-0.429071, 0.196334, 0.881675,
-0.441675, 0.164016, 0.882055,
-0.472637, 0.163706, 0.865919,
-0.447345, 0.228090, 0.864787,
-0.216955, 0.295021, 0.930534,
-0.236060, 0.325939, 0.915445,
-0.251736, 0.295269, 0.921654,
-0.286117, 0.295186, 0.911593,
-0.254807, 0.356275, 0.898967,
-0.305118, 0.325609, 0.894920,
-0.320484, 0.294703, 0.900244,
-0.289370, 0.356197, 0.888475,
-0.323632, 0.355602, 0.876818,
-0.341572, 0.385011, 0.857377,
-0.357403, 0.354516, 0.864050,
-0.273346, 0.386299, 0.880940,
-0.354076, 0.293939, 0.887823,
-0.372561, 0.323903, 0.869647,
-0.387280, 0.292822, 0.874225,
-0.419912, 0.291383, 0.859517,
-0.390470, 0.353281, 0.850133,
-0.291352, 0.415435, 0.861701,
-0.308973, 0.443956, 0.841094,
-0.325506, 0.414707, 0.849744,
-0.359158, 0.413488, 0.836680,
-0.326128, 0.471716, 0.819222,
0.291536, 0.527686, 0.797844,
0.253742, 0.533322, 0.806959,
0.235673, 0.560060, 0.794224,
0.215489, 0.538182, 0.814815,
0.176970, 0.542239, 0.821376,
0.255549, 0.581130, 0.772646,
0.217322, 0.586051, 0.780587,
0.158301, 0.568272, 0.807470,
0.137840, 0.545532, 0.826677,
0.178553, 0.590346, 0.787154,
0.139437, 0.593687, 0.792524,
0.120472, 0.618331, 0.776630,
0.100171, 0.596053, 0.796672,
0.198505, 0.611619, 0.765845,
0.098832, 0.547978, 0.830634,
0.079939, 0.573242, 0.815477,
0.059604, 0.549620, 0.833286,
0.020351, 0.550447, 0.834622,
0.060948, 0.597732, 0.799376,
0.218339, 0.631794, 0.743751,
0.179588, 0.636140, 0.750382,
0.160388, 0.659941, 0.733998,
0.140487, 0.639533, 0.755818,
0.101232, 0.641952, 0.760032,
0.180084, 0.679433, 0.711295,
0.140998, 0.682882, 0.716793,
0.000000, 0.574934, 0.818200,
-0.020351, 0.550447, 0.834622,
0.020351, 0.598760, 0.800670,
-0.020351, 0.598760, 0.800670,
-0.040604, 0.621778, 0.782140,
-0.060948, 0.597732, 0.799376,
0.040604, 0.621778, 0.782140,
-0.020352, 0.645320, 0.763641,
-0.060949, 0.644291, 0.762348,
0.020352, 0.645320, 0.763641,
0.000000, 0.668055, 0.744112,
0.020352, 0.689842, 0.723674,
-0.020352, 0.689842, 0.723674,
0.060949, 0.644291, 0.762348,
-0.081001, 0.665716, 0.741796,
-0.101232, 0.641952, 0.760032,
-0.060664, 0.688679, 0.722524,
-0.100949, 0.686355, 0.720226,
-0.120696, 0.706082, 0.697769,
-0.140998, 0.682882, 0.716793,
-0.040319, 0.710703, 0.702335,
0.081001, 0.665716, 0.741796,
0.100949, 0.686355, 0.720226,
0.060664, 0.688679, 0.722524,
0.040319, 0.710703, 0.702335,
0.120696, 0.706082, 0.697769,
-0.059604, 0.549620, 0.833286,
-0.079939, 0.573242, 0.815477,
-0.098832, 0.547978, 0.830634,
-0.137840, 0.545532, 0.826677,
-0.100171, 0.596053, 0.796672,
-0.158301, 0.568272, 0.807470,
-0.176970, 0.542239, 0.821376,
-0.139437, 0.593687, 0.792524,
-0.178553, 0.590346, 0.787154,
-0.198505, 0.611619, 0.765845,
-0.217322, 0.586051, 0.780587,
-0.120472, 0.618331, 0.776630,
-0.215489, 0.538182, 0.814815,
-0.235673, 0.560060, 0.794224,
-0.253742, 0.533322, 0.806959,
-0.291536, 0.527686, 0.797844,
-0.255549, 0.581130, 0.772646,
-0.140487, 0.639533, 0.755818,
-0.160388, 0.659941, 0.733998,
-0.179588, 0.636140, 0.750382,
-0.218339, 0.631794, 0.743751,
-0.180084, 0.679433, 0.711295,
0.140967, 0.723820, 0.675435,
0.100920, 0.727293, 0.678867,
0.080959, 0.747667, 0.659120,
0.060644, 0.729617, 0.681162,
0.020333, 0.730779, 0.682311,
0.101176, 0.764743, 0.636343,
0.060901, 0.767082, 0.638652,
0.000000, 0.750007, 0.661430,
-0.020333, 0.730779, 0.682311,
0.020333, 0.768380, 0.639671,
-0.020333, 0.768380, 0.639671,
-0.040577, 0.785785, 0.617167,
-0.060901, 0.767082, 0.638652,
0.040577, 0.785785, 0.617167,
-0.060644, 0.729617, 0.681162,
-0.080959, 0.747667, 0.659120,
-0.100920, 0.727293, 0.678867,
-0.140967, 0.723820, 0.675435,
-0.101176, 0.764743, 0.636343,
0.060900, 0.802005, 0.594204,
0.020332, 0.803303, 0.595224,
0.000000, 0.819885, 0.572529,
-0.020332, 0.803303, 0.595224,
-0.060900, 0.802005, 0.594204,
0.020332, 0.835429, 0.549222,
-0.020332, 0.835429, 0.549222,
0.486380, -0.032906, 0.873128,
0.516324, -0.032898, 0.855762,
0.455785, -0.032928, 0.889480,
0.476118, -0.065827, 0.876914,
0.465287, -0.098590, 0.879653,
0.495666, -0.098538, 0.862905,
0.424710, -0.032936, 0.904730,
0.434130, -0.098843, 0.895411,
0.453679, -0.131555, 0.881402,
0.413796, -0.065943, 0.907978,
0.402375, -0.098950, 0.910112,
0.370201, -0.098882, 0.923674,
0.390534, -0.131782, 0.911107,
0.392778, -0.033014, 0.919041,
0.441675, -0.164016, 0.882055,
0.472637, -0.163706, 0.865919,
0.410104, -0.164179, 0.897140,
0.429071, -0.196334, 0.881675,
0.415958, -0.228338, 0.880251,
0.447345, -0.228090, 0.864787,
0.378097, -0.164167, 0.911094,
0.360716, -0.033021, 0.932091,
0.328151, -0.033041, 0.944047,
0.349460, -0.066055, 0.934620,
0.337778, -0.098929, 0.936012,
0.295262, -0.033048, 0.954845,
0.383186, -0.229687, 0.894658,
0.401461, -0.261444, 0.877768,
0.364213, -0.197527, 0.910128,
0.349848, -0.230712, 0.907953,
0.316139, -0.231386, 0.920064,
0.335103, -0.263541, 0.904573,
0.344643, -0.165129, 0.924096,
0.296273, -0.198871, 0.934170,
0.281637, -0.231780, 0.931106,
0.310525, -0.165615, 0.936027,
0.275921, -0.165941, 0.946748,
0.255190, -0.132768, 0.957732,
0.241032, -0.166080, 0.956201,
0.324291, -0.132199, 0.936674,
0.247162, -0.231772, 0.940847,
0.267028, -0.264286, 0.926741,
0.226811, -0.198844, 0.953424,
0.212293, -0.231436, 0.949405,
0.177234, -0.230749, 0.956736,
0.197584, -0.263677, 0.944158,
0.206081, -0.165671, 0.964409,
0.303549, -0.099370, 0.947620,
0.282237, -0.066348, 0.957048,
0.268847, -0.099650, 0.958014,
0.233871, -0.099744, 0.967138,
0.260463, -0.033305, 0.964909,
0.387280, -0.292822, 0.874225,
0.419912, -0.291383, 0.859517,
0.354076, -0.293939, 0.887823,
0.372561, -0.323903, 0.869647,
0.357403, -0.354516, 0.864050,
0.390470, -0.353281, 0.850133,
0.320484, -0.294703, 0.900244,
0.323632, -0.355602, 0.876818,
0.341572, -0.385011, 0.857377,
0.305118, -0.325609, 0.894920,
0.289370, -0.356197, 0.888475,
0.254807, -0.356275, 0.898967,
0.273346, -0.386299, 0.880940,
0.286117, -0.295186, 0.911593,
0.325506, -0.414707, 0.849744,
0.359158, -0.413488, 0.836680,
0.291352, -0.415435, 0.861701,
0.308973, -0.443956, 0.841094,
0.292081, -0.472596, 0.831470,
0.326128, -0.471716, 0.819222,
0.256883, -0.415646, 0.872496,
0.251736, -0.295269, 0.921654,
0.216955, -0.295021, 0.930534,
0.236060, -0.325939, 0.915445,
0.220102, -0.356133, 0.908143,
0.181964, -0.294425, 0.938191,
0.226892, -0.033310, 0.973350,
0.192973, -0.033324, 0.980638,
0.213589, -0.066626, 0.974649,
0.200013, -0.099776, 0.974700,
0.158898, -0.033328, 0.986732,
0.165681, -0.099996, 0.981097,
0.185946, -0.133106, 0.973502,
0.145079, -0.066698, 0.987169,
0.131070, -0.100061, 0.986311,
0.096377, -0.099955, 0.990313,
0.116967, -0.133245, 0.984157,
0.124250, -0.033376, 0.991690,
0.171797, -0.165925, 0.971058,
0.137228, -0.166025, 0.976527,
0.157351, -0.198580, 0.967371,
0.142705, -0.230901, 0.962455,
0.102565, -0.165954, 0.980785,
0.089833, -0.033381, 0.995397,
0.055231, -0.033392, 0.997915,
0.075862, -0.066765, 0.994881,
0.061793, -0.099983, 0.993068,
0.020640, -0.033396, 0.999229,
0.254286, -0.478242, 0.840609,
0.271440, -0.505993, 0.818713,
0.236644, -0.449711, 0.861255,
0.216004, -0.483108, 0.848498,
0.177481, -0.487173, 0.855081,
0.195090, -0.515629, 0.834306,
0.218637, -0.420581, 0.880517,
0.159117, -0.457934, 0.874631,
0.138204, -0.490456, 0.860438,
0.179759, -0.424534, 0.887388,
0.140501, -0.427873, 0.892852,
0.121761, -0.397184, 0.909626,
0.101119, -0.430583, 0.896869,
0.199999, -0.390469, 0.898629,
0.099190, -0.492908, 0.864409,
0.117537, -0.522087, 0.844754,
0.080590, -0.462906, 0.882736,
0.059926, -0.494553, 0.867079,
0.020667, -0.495384, 0.868428,
0.039260, -0.525326, 0.849995,
0.061863, -0.432270, 0.899620,
0.181249, -0.360134, 0.915124,
0.162122, -0.329196, 0.930240,
0.142013, -0.363521, 0.920698,
0.102647, -0.366279, 0.924826,
0.142749, -0.297852, 0.943879,
0.000000, -0.464610, 0.885516,
-0.020667, -0.495384, 0.868428,
0.020667, -0.432953, 0.901180,
-0.020667, -0.432953, 0.901180,
-0.041197, -0.400628, 0.915314,
-0.061863, -0.432270, 0.899620,
0.041197, -0.400628, 0.915314,
-0.020669, -0.367868, 0.929648,
-0.061865, -0.367185, 0.928088,
0.020669, -0.367868, 0.929648,
0.000000, -0.334425, 0.942422,
0.020669, -0.300527, 0.953549,
-0.020669, -0.300527, 0.953549,
0.061865, -0.367185, 0.928088,
-0.082120, -0.333520, 0.939160,
-0.102647, -0.366279, 0.924826,
-0.061453, -0.300080, 0.951932,
-0.102238, -0.299185, 0.948702,
-0.122082, -0.264412, 0.956652,
-0.142749, -0.297852, 0.943879,
-0.040784, -0.266183, 0.963059,
0.082120, -0.333520, 0.939160,
0.102238, -0.299185, 0.948702,
0.061453, -0.300080, 0.951932,
0.040784, -0.266183, 0.963059,
0.122082, -0.264412, 0.956652,
-0.059926, -0.494553, 0.867079,
-0.039260, -0.525326, 0.849995,
-0.080590, -0.462906, 0.882736,
-0.099190, -0.492908, 0.864409,
-0.138204, -0.490456, 0.860438,
-0.117537, -0.522087, 0.844754,
-0.101119, -0.430583, 0.896869,
-0.159117, -0.457934, 0.874631,
-0.177481, -0.487173, 0.855081,
-0.140501, -0.427873, 0.892852,
-0.179759, -0.424534, 0.887388,
-0.199999, -0.390469, 0.898629,
-0.218637, -0.420581, 0.880517,
-0.121761, -0.397184, 0.909626,
-0.216004, -0.483108, 0.848498,
-0.195090, -0.515629, 0.834306,
-0.236644, -0.449711, 0.861255,
-0.254286, -0.478242, 0.840609,
-0.292081, -0.472596, 0.831470,
-0.271440, -0.505993, 0.818713,
-0.256883, -0.415646, 0.872496,
-0.142013, -0.363521, 0.920698,
-0.162122, -0.329196, 0.930240,
-0.181249, -0.360134, 0.915124,
-0.220102, -0.356133, 0.908143,
-0.181964, -0.294425, 0.938191,
0.102196, -0.232233, 0.967277,
0.082059, -0.199641, 0.976427,
0.061424, -0.233124, 0.970505,
0.020642, -0.233570, 0.972121,
0.061795, -0.166851, 0.984044,
0.000000, -0.200538, 0.979686,
-0.020642, -0.233570, 0.972121,
0.020642, -0.167078, 0.985728,
-0.020642, -0.167078, 0.985728,
-0.041156, -0.133410, 0.990206,
-0.061795, -0.166851, 0.984044,
0.041156, -0.133410, 0.990206,
-0.061424, -0.233124, 0.970505,
-0.082059, -0.199641, 0.976427,
-0.102196, -0.232233, 0.967277,
-0.142705, -0.230901, 0.962455,
-0.102565, -0.165954, 0.980785,
0.020640, -0.100210, 0.994752,
0.000000, -0.066798, 0.997767,
-0.020640, -0.100210, 0.994752,
-0.061793, -0.099983, 0.993068,
-0.020640, -0.033396, 0.999229,
0.253742, -0.533322, 0.806959,
0.291536, -0.527686, 0.797844,
0.215489, -0.538182, 0.814815,
0.235673, -0.560060, 0.794224,
0.217322, -0.586051, 0.780587,
0.255549, -0.581130, 0.772646,
0.176970, -0.542239, 0.821376,
0.178553, -0.590346, 0.787154,
0.198505, -0.611619, 0.765845,
0.158301, -0.568272, 0.807470,
0.139437, -0.593687, 0.792524,
0.100171, -0.596053, 0.796672,
0.120472, -0.618331, 0.776630,
0.137840, -0.545532, 0.826677,
0.179588, -0.636140, 0.750382,
0.218339, -0.631794, 0.743751,
0.140487, -0.639533, 0.755818,
0.160388, -0.659941, 0.733998,
0.140998, -0.682882, 0.716793,
0.180084, -0.679433, 0.711295,
0.101232, -0.641952, 0.760032,
0.098832, -0.547978, 0.830634,
0.059604, -0.549620, 0.833286,
0.079939, -0.573242, 0.815477,
0.060948, -0.597732, 0.799376,
0.020351, -0.550447, 0.834622,
0.100949, -0.686355, 0.720226,
0.120696, -0.706082, 0.697769,
0.081001, -0.665716, 0.741796,
0.060664, -0.688679, 0.722524,
0.020352, -0.689842, 0.723674,
0.040319, -0.710703, 0.702335,
0.060949, -0.644291, 0.762348,
0.000000, -0.668055, 0.744112,
-0.020352, -0.689842, 0.723674,
0.020352, -0.645320, 0.763641,
-0.020352, -0.645320, 0.763641,
-0.040604, -0.621778, 0.782140,
-0.060949, -0.644291, 0.762348,
0.040604, -0.621778, 0.782140,
-0.060664, -0.688679, 0.722524,
-0.040319, -0.710703, 0.702335,
-0.081001, -0.665716, 0.741796,
-0.100949, -0.686355, 0.720226,
-0.140998, -0.682882, 0.716793,
-0.120696, -0.706082, 0.697769,
-0.101232, -0.641952, 0.760032,
0.020351, -0.598760, 0.800670,
0.000000, -0.574934, 0.818200,
-0.020351, -0.598760, 0.800670,
-0.060948, -0.597732, 0.799376,
-0.020351, -0.550447, 0.834622,
0.100920, -0.727293, 0.678867,
0.140967, -0.723820, 0.675435,
0.060644, -0.729617, 0.681162,
0.080959, -0.747667, 0.659120,
0.060901, -0.767082, 0.638652,
0.101176, -0.764743, 0.636343,
0.020333, -0.730779, 0.682311,
0.020333, -0.768380, 0.639671,
0.040577, -0.785785, 0.617167,
0.000000, -0.750007, 0.661430,
-0.020333, -0.768380, 0.639671,
-0.060901, -0.767082, 0.638652,
-0.040577, -0.785785, 0.617167,
-0.020333, -0.730779, 0.682311,
0.020332, -0.803303, 0.595224,
0.060900, -0.802005, 0.594204,
-0.020332, -0.803303, 0.595224,
0.000000, -0.819885, 0.572529,
-0.020332, -0.835429, 0.549222,
0.020332, -0.835429, 0.549222,
-0.060900, -0.802005, 0.594204,
-0.060644, -0.729617, 0.681162,
-0.100920, -0.727293, 0.678867,
-0.080959, -0.747667, 0.659120,
-0.101176, -0.764743, 0.636343,
-0.140967, -0.723820, 0.675435,
-0.059604, -0.549620, 0.833286,
-0.098832, -0.547978, 0.830634,
-0.079939, -0.573242, 0.815477,
-0.100171, -0.596053, 0.796672,
-0.137840, -0.545532, 0.826677,
-0.139437, -0.593687, 0.792524,
-0.120472, -0.618331, 0.776630,
-0.158301, -0.568272, 0.807470,
-0.178553, -0.590346, 0.787154,
-0.217322, -0.586051, 0.780587,
-0.198505, -0.611619, 0.765845,
-0.176970, -0.542239, 0.821376,
-0.140487, -0.639533, 0.755818,
-0.179588, -0.636140, 0.750382,
-0.160388, -0.659941, 0.733998,
-0.180084, -0.679433, 0.711295,
-0.218339, -0.631794, 0.743751,
-0.215489, -0.538182, 0.814815,
-0.253742, -0.533322, 0.806959,
-0.235673, -0.560060, 0.794224,
-0.255549, -0.581130, 0.772646,
-0.291536, -0.527686, 0.797844,
-0.055231, -0.033392, 0.997915,
-0.089833, -0.033381, 0.995397,
-0.075862, -0.066765, 0.994881,
-0.096377, -0.099955, 0.990313,
-0.124250, -0.033376, 0.991690,
-0.131070, -0.100061, 0.986311,
-0.116967, -0.133245, 0.984157,
-0.145079, -0.066698, 0.987169,
-0.165681, -0.099996, 0.981097,
-0.200013, -0.099776, 0.974700,
-0.185946, -0.133106, 0.973502,
-0.158898, -0.033328, 0.986732,
-0.137228, -0.166025, 0.976527,
-0.171797, -0.165925, 0.971058,
-0.157351, -0.198580, 0.967371,
-0.177234, -0.230749, 0.956736,
-0.206081, -0.165671, 0.964409,
-0.192973, -0.033324, 0.980638,
-0.226892, -0.033310, 0.973350,
-0.213589, -0.066626, 0.974649,
-0.233871, -0.099744, 0.967138,
-0.260463, -0.033305, 0.964909,
-0.212293, -0.231436, 0.949405,
-0.197584, -0.263677, 0.944158,
-0.226811, -0.198844, 0.953424,
-0.247162, -0.231772, 0.940847,
-0.281637, -0.231780, 0.931106,
-0.267028, -0.264286, 0.926741,
-0.241032, -0.166080, 0.956201,
-0.296273, -0.198871, 0.934170,
-0.316139, -0.231386, 0.920064,
-0.275921, -0.165941, 0.946748,
-0.310525, -0.165615, 0.936027,
-0.324291, -0.132199, 0.936674,
-0.344643, -0.165129, 0.924096,
-0.255190, -0.132768, 0.957732,
-0.349848, -0.230712, 0.907953,
-0.335103, -0.263541, 0.904573,
-0.364213, -0.197527, 0.910128,
-0.383186, -0.229687, 0.894658,
-0.415958, -0.228338, 0.880251,
-0.401461, -0.261444, 0.877768,
-0.378097, -0.164167, 0.911094,
-0.268847, -0.099650, 0.958014,
-0.282237, -0.066348, 0.957048,
-0.303549, -0.099370, 0.947620,
-0.337778, -0.098929, 0.936012,
-0.295262, -0.033048, 0.954845,
-0.216955, -0.295021, 0.930534,
-0.251736, -0.295269, 0.921654,
-0.236060, -0.325939, 0.915445,
-0.254807, -0.356275, 0.898967,
-0.286117, -0.295186, 0.911593,
-0.289370, -0.356197, 0.888475,
-0.273346, -0.386299, 0.880940,
-0.305118, -0.325609, 0.894920,
-0.323632, -0.355602, 0.876818,
-0.357403, -0.354516, 0.864050,
-0.341572, -0.385011, 0.857377,
-0.320484, -0.294703, 0.900244,
-0.291352, -0.415435, 0.861701,
-0.325506, -0.414707, 0.849744,
-0.308973, -0.443956, 0.841094,
-0.326128, -0.471716, 0.819222,
-0.359158, -0.413488, 0.836680,
-0.354076, -0.293939, 0.887823,
-0.387280, -0.292822, 0.874225,
-0.372561, -0.323903, 0.869647,
-0.390470, -0.353281, 0.850133,
-0.419912, -0.291383, 0.859517,
-0.328151, -0.033041, 0.944047,
-0.360716, -0.033021, 0.932091,
-0.349460, -0.066055, 0.934620,
-0.370201, -0.098882, 0.923674,
-0.392778, -0.033014, 0.919041,
-0.402375, -0.098950, 0.910112,
-0.390534, -0.131782, 0.911107,
-0.413796, -0.065943, 0.907978,
-0.434130, -0.098843, 0.895411,
-0.465287, -0.098590, 0.879653,
-0.453679, -0.131555, 0.881402,
-0.424710, -0.032936, 0.904730,
-0.410104, -0.164179, 0.897140,
-0.441675, -0.164016, 0.882055,
-0.429071, -0.196334, 0.881675,
-0.447345, -0.228090, 0.864787,
-0.472637, -0.163706, 0.865919,
-0.455785, -0.032928, 0.889480,
-0.486380, -0.032906, 0.873128,
-0.476118, -0.065827, 0.876914,
-0.495666, -0.098538, 0.862905,
-0.516324, -0.032898, 0.855762,
0.495991, 0.000000, -0.868328,
0.486380, 0.032906, -0.873128,
0.516324, 0.032898, -0.855762,
0.455785, 0.032928, -0.889480,
0.476118, 0.065827, -0.876914,
0.465287, 0.098590, -0.879653,
0.495666, 0.098538, -0.862905,
0.434767, 0.000000, -0.900543,
0.424710, 0.032936, -0.904730,
0.434130, 0.098843, -0.895411,
0.453679, 0.131555, -0.881402,
0.413796, 0.065943, -0.907978,
0.402375, 0.098950, -0.910112,
0.370201, 0.098882, -0.923674,
0.390534, 0.131782, -0.911107,
0.392778, 0.033014, -0.919041,
0.441675, 0.164016, -0.882055,
0.472637, 0.163706, -0.865919,
0.410104, 0.164179, -0.897140,
0.429071, 0.196334, -0.881675,
0.415958, 0.228338, -0.880251,
0.447345, 0.228090, -0.864787,
0.378097, 0.164167, -0.911094,
0.371461, 0.000000, -0.928448,
0.360716, 0.033021, -0.932091,
0.328151, 0.033041, -0.944047,
0.349460, 0.066055, -0.934620,
0.337778, 0.098929, -0.936012,
0.306378, 0.000000, -0.951910,
0.295262, 0.033048, -0.954845,
0.383186, 0.229687, -0.894658,
0.401461, 0.261444, -0.877768,
0.364213, 0.197527, -0.910128,
0.349848, 0.230712, -0.907953,
0.316139, 0.231386, -0.920064,
0.335103, 0.263541, -0.904573,
0.344643, 0.165129, -0.924096,
0.296273, 0.198871, -0.934170,
0.281637, 0.231780, -0.931106,
0.310525, 0.165615, -0.936027,
0.275921, 0.165941, -0.946748,
0.255190, 0.132768, -0.957732,
0.241032, 0.166080, -0.956201,
0.324291, 0.132199, -0.936674,
0.247162, 0.231772, -0.940847,
0.267028, 0.264286, -0.926741,
0.226811, 0.198844, -0.953424,
0.212293, 0.231436, -0.949405,
0.177234, 0.230749, -0.956736,
0.197584, 0.263677, -0.944158,
0.206081, 0.165671, -0.964409,
0.303549, 0.099370, -0.947620,
0.282237, 0.066348, -0.957048,
0.268847, 0.099650, -0.958014,
0.233871, 0.099744, -0.967138,
0.260463, 0.033305, -0.964909,
0.387280, 0.292822, -0.874225,
0.419912, 0.291383, -0.859517,
0.354076, 0.293939, -0.887823,
0.372561, 0.323903, -0.869647,
0.357403, 0.354516, -0.864050,
0.390470, 0.353281, -0.850133,
0.320484, 0.294703, -0.900244,
0.323632, 0.355602, -0.876818,
0.341572, 0.385011, -0.857377,
0.305118, 0.325609, -0.894920,
0.289370, 0.356197, -0.888475,
0.254807, 0.356275, -0.898967,
0.273346, 0.386299, -0.880940,
0.286117, 0.295186, -0.911593,
0.325506, 0.414707, -0.849744,
0.359158, 0.413488, -0.836680,
0.291352, 0.415435, -0.861701,
0.308973, 0.443956, -0.841094,
0.292081, 0.472596, -0.831470,
0.326128, 0.471716, -0.819222,
0.256883, 0.415646, -0.872496,
0.251736, 0.295269, -0.921654,
0.216955, 0.295021, -0.930534,
0.236060, 0.325939, -0.915445,
0.220102, 0.356133, -0.908143,
0.181964, 0.294425, -0.938191,
0.239828, 0.000000, -0.970815,
0.226892, 0.033310, -0.973350,
0.192973, 0.033324, -0.980638,
0.213589, 0.066626, -0.974649,
0.200013, 0.099776, -0.974700,
0.172130, 0.000000, -0.985074,
0.158898, 0.033328, -0.986732,
0.165681, 0.099996, -0.981097,
0.185946, 0.133106, -0.973502,
0.145079, 0.066698, -0.987169,
0.131070, 0.100061, -0.986311,
0.096377, 0.099955, -0.990313,
0.116967, 0.133245, -0.984157,
0.124250, 0.033376, -0.991690,
0.171797, 0.165925, -0.971058,
0.137228, 0.166025, -0.976527,
0.157351, 0.198580, -0.967371,
0.142705, 0.230901, -0.962455,
0.102565, 0.165954, -0.980785,
0.103609, 0.000000, -0.994618,
0.089833, 0.033381, -0.995397,
0.055231, 0.033392, -0.997915,
0.075862, 0.066765, -0.994881,
0.061793, 0.099983, -0.993068,
0.034591, 0.000000, -0.999402,
0.020640, 0.033396, -0.999229,
0.254286, 0.478242, -0.840609,
0.271440, 0.505993, -0.818713,
0.236644, 0.449711, -0.861255,
0.216004, 0.483108, -0.848498,
0.177481, 0.487173, -0.855081,
0.195090, 0.515629, -0.834306,
0.218637, 0.420581, -0.880517,
0.159117, 0.457934, -0.874631,
0.138204, 0.490456, -0.860438,
0.179759, 0.424534, -0.887388,
0.140501, 0.427873, -0.892852,
0.121761, 0.397184, -0.909626,
0.101119, 0.430583, -0.896869,
0.199999, 0.390469, -0.898629,
0.099190, 0.492908, -0.864409,
0.117537, 0.522087, -0.844754,
0.080590, 0.462906, -0.882736,
0.059926, 0.494553, -0.867079,
0.020667, 0.495384, -0.868428,
0.039260, 0.525326, -0.849995,
0.061863, 0.432270, -0.899620,
0.181249, 0.360134, -0.915124,
0.162122, 0.329196, -0.930240,
0.142013, 0.363521, -0.920698,
0.102647, 0.366279, -0.924826,
0.142749, 0.297852, -0.943879,
0.000000, 0.464610, -0.885516,
-0.020667, 0.495384, -0.868428,
0.020667, 0.432953, -0.901180,
-0.020667, 0.432953, -0.901180,
-0.041197, 0.400628, -0.915314,
-0.061863, 0.432270, -0.899620,
0.041197, 0.400628, -0.915314,
-0.020669, 0.367868, -0.929648,
-0.061865, 0.367185, -0.928088,
0.020669, 0.367868, -0.929648,
0.000000, 0.334425, -0.942422,
0.020669, 0.300527, -0.953549,
-0.020669, 0.300527, -0.953549,
0.061865, 0.367185, -0.928088,
-0.082120, 0.333520, -0.939160,
-0.102647, 0.366279, -0.924826,
-0.061453, 0.300080, -0.951932,
-0.102238, 0.299185, -0.948702,
-0.122082, 0.264412, -0.956652,
-0.142749, 0.297852, -0.943879,
-0.040784, 0.266183, -0.963059,
0.082120, 0.333520, -0.939160,
0.102238, 0.299185, -0.948702,
0.061453, 0.300080, -0.951932,
0.040784, 0.266183, -0.963059,
0.122082, 0.264412, -0.956652,
-0.059926, 0.494553, -0.867079,
-0.039260, 0.525326, -0.849995,
-0.080590, 0.462906, -0.882736,
-0.099190, 0.492908, -0.864409,
-0.138204, 0.490456, -0.860438,
-0.117537, 0.522087, -0.844754,
-0.101119, 0.430583, -0.896869,
-0.159117, 0.457934, -0.874631,
-0.177481, 0.487173, -0.855081,
-0.140501, 0.427873, -0.892852,
-0.179759, 0.424534, -0.887388,
-0.199999, 0.390469, -0.898629,
-0.218637, 0.420581, -0.880517,
-0.121761, 0.397184, -0.909626,
-0.216004, 0.483108, -0.848498,
-0.195090, 0.515629, -0.834306,
-0.236644, 0.449711, -0.861255,
-0.254286, 0.478242, -0.840609,
-0.292081, 0.472596, -0.831470,
-0.271440, 0.505993, -0.818713,
-0.256883, 0.415646, -0.872496,
-0.142013, 0.363521, -0.920698,
-0.162122, 0.329196, -0.930240,
-0.181249, 0.360134, -0.915124,
-0.220102, 0.356133, -0.908143,
-0.181964, 0.294425, -0.938191,
0.102196, 0.232233, -0.967277,
0.082059, 0.199641, -0.976427,
0.061424, 0.233124, -0.970505,
0.020642, 0.233570, -0.972121,
0.061795, 0.166851, -0.984044,
0.000000, 0.200538, -0.979686,
-0.020642, 0.233570, -0.972121,
0.020642, 0.167078, -0.985728,
-0.020642, 0.167078, -0.985728,
-0.041156, 0.133410, -0.990206,
-0.061795, 0.166851, -0.984044,
0.041156, 0.133410, -0.990206,
-0.061424, 0.233124, -0.970505,
-0.082059, 0.199641, -0.976427,
-0.102196, 0.232233, -0.967277,
-0.142705, 0.230901, -0.962455,
-0.102565, 0.165954, -0.980785,
0.020640, 0.100210, -0.994752,
0.000000, 0.066798, -0.997767,
-0.020640, 0.100210, -0.994752,
-0.061793, 0.099983, -0.993068,
-0.020640, 0.033396, -0.999229,
0.253742, 0.533322, -0.806959,
0.291536, 0.527686, -0.797844,
0.215489, 0.538182, -0.814815,
0.235673, 0.560060, -0.794224,
0.217322, 0.586051, -0.780587,
0.255549, 0.581130, -0.772646,
0.176970, 0.542239, -0.821376,
0.178553, 0.590346, -0.787154,
0.198505, 0.611619, -0.765845,
0.158301, 0.568272, -0.807470,
0.139437, 0.593687, -0.792524,
0.100171, 0.596053, -0.796672,
0.120472, 0.618331, -0.776630,
0.137840, 0.545532, -0.826677,
0.179588, 0.636140, -0.750382,
0.218339, 0.631794, -0.743751,
0.140487, 0.639533, -0.755818,
0.160388, 0.659941, -0.733998,
0.140998, 0.682882, -0.716793,
0.180084, 0.679433, -0.711295,
0.101232, 0.641952, -0.760032,
0.098832, 0.547978, -0.830634,
0.059604, 0.549620, -0.833286,
0.079939, 0.573242, -0.815477,
0.060948, 0.597732, -0.799376,
0.020351, 0.550447, -0.834622,
0.100949, 0.686355, -0.720226,
0.120696, 0.706082, -0.697769,
0.081001, 0.665716, -0.741796,
0.060664, 0.688679, -0.722524,
0.020352, 0.689842, -0.723674,
0.040319, 0.710703, -0.702335,
0.060949, 0.644291, -0.762348,
0.000000, 0.668055, -0.744112,
-0.020352, 0.689842, -0.723674,
0.020352, 0.645320, -0.763641,
-0.020352, 0.645320, -0.763641,
-0.040604, 0.621778, -0.782140,
-0.060949, 0.644291, -0.762348,
0.040604, 0.621778, -0.782140,
-0.060664, 0.688679, -0.722524,
-0.040319, 0.710703, -0.702335,
-0.081001, 0.665716, -0.741796,
-0.100949, 0.686355, -0.720226,
-0.140998, 0.682882, -0.716793,
-0.120696, 0.706082, -0.697769,
-0.101232, 0.641952, -0.760032,
0.020351, 0.598760, -0.800670,
0.000000, 0.574934, -0.818200,
-0.020351, 0.598760, -0.800670,
-0.060948, 0.597732, -0.799376,
-0.020351, 0.550447, -0.834622,
0.100920, 0.727293, -0.678867,
0.140967, 0.723820, -0.675435,
0.060644, 0.729617, -0.681162,
0.080959, 0.747667, -0.659120,
0.060901, 0.767082, -0.638652,
0.101176, 0.764743, -0.636343,
0.020333, 0.730779, -0.682311,
0.020333, 0.768380, -0.639671,
0.040577, 0.785785, -0.617167,
0.000000, 0.750007, -0.661430,
-0.020333, 0.768380, -0.639671,
-0.060901, 0.767082, -0.638652,
-0.040577, 0.785785, -0.617167,
-0.020333, 0.730779, -0.682311,
0.020332, 0.803303, -0.595224,
0.060900, 0.802005, -0.594204,
-0.020332, 0.803303, -0.595224,
0.000000, 0.819885, -0.572529,
-0.020332, 0.835429, -0.549222,
0.020332, 0.835429, -0.549222,
-0.060900, 0.802005, -0.594204,
-0.060644, 0.729617, -0.681162,
-0.100920, 0.727293, -0.678867,
-0.080959, 0.747667, -0.659120,
-0.101176, 0.764743, -0.636343,
-0.140967, 0.723820, -0.675435,
-0.059604, 0.549620, -0.833286,
-0.098832, 0.547978, -0.830634,
-0.079939, 0.573242, -0.815477,
-0.100171, 0.596053, -0.796672,
-0.137840, 0.545532, -0.826677,
-0.139437, 0.593687, -0.792524,
-0.120472, 0.618331, -0.776630,
-0.158301, 0.568272, -0.807470,
-0.178553, 0.590346, -0.787154,
-0.217322, 0.586051, -0.780587,
-0.198505, 0.611619, -0.765845,
-0.176970, 0.542239, -0.821376,
-0.140487, 0.639533, -0.755818,
-0.179588, 0.636140, -0.750382,
-0.160388, 0.659941, -0.733998,
-0.180084, 0.679433, -0.711295,
-0.218339, 0.631794, -0.743751,
-0.215489, 0.538182, -0.814815,
-0.253742, 0.533322, -0.806959,
-0.235673, 0.560060, -0.794224,
-0.255549, 0.581130, -0.772646,
-0.291536, 0.527686, -0.797844,
-0.034591, 0.000000, -0.999402,
-0.055231, 0.033392, -0.997915,
-0.089833, 0.033381, -0.995397,
-0.075862, 0.066765, -0.994881,
-0.096377, 0.099955, -0.990313,
-0.103609, 0.000000, -0.994618,
-0.124250, 0.033376, -0.991690,
-0.131070, 0.100061, -0.986311,
-0.116967, 0.133245, -0.984157,
-0.145079, 0.066698, -0.987169,
-0.165681, 0.099996, -0.981097,
-0.200013, 0.099776, -0.974700,
-0.185946, 0.133106, -0.973502,
-0.158898, 0.033328, -0.986732,
-0.137228, 0.166025, -0.976527,
-0.171797, 0.165925, -0.971058,
-0.157351, 0.198580, -0.967371,
-0.177234, 0.230749, -0.956736,
-0.206081, 0.165671, -0.964409,
-0.172130, 0.000000, -0.985074,
-0.192973, 0.033324, -0.980638,
-0.226892, 0.033310, -0.973350,
-0.213589, 0.066626, -0.974649,
-0.233871, 0.099744, -0.967138,
-0.239828, 0.000000, -0.970815,
-0.260463, 0.033305, -0.964909,
-0.212293, 0.231436, -0.949405,
-0.197584, 0.263677, -0.944158,
-0.226811, 0.198844, -0.953424,
-0.247162, 0.231772, -0.940847,
-0.281637, 0.231780, -0.931106,
-0.267028, 0.264286, -0.926741,
-0.241032, 0.166080, -0.956201,
-0.296273, 0.198871, -0.934170,
-0.316139, 0.231386, -0.920064,
-0.275921, 0.165941, -0.946748,
-0.310525, 0.165615, -0.936027,
-0.324291, 0.132199, -0.936674,
-0.344643, 0.165129, -0.924096,
-0.255190, 0.132768, -0.957732,
-0.349848, 0.230712, -0.907953,
-0.335103, 0.263541, -0.904573,
-0.364213, 0.197527, -0.910128,
-0.383186, 0.229687, -0.894658,
-0.415958, 0.228338, -0.880251,
-0.401461, 0.261444, -0.877768,
-0.378097, 0.164167, -0.911094,
-0.268847, 0.099650, -0.958014,
-0.282237, 0.066348, -0.957048,
-0.303549, 0.099370, -0.947620,
-0.337778, 0.098929, -0.936012,
-0.295262, 0.033048, -0.954845,
-0.216955, 0.295021, -0.930534,
-0.251736, 0.295269, -0.921654,
-0.236060, 0.325939, -0.915445,
-0.254807, 0.356275, -0.898967,
-0.286117, 0.295186, -0.911593,
-0.289370, 0.356197, -0.888475,
-0.273346, 0.386299, -0.880940,
-0.305118, 0.325609, -0.894920,
-0.323632, 0.355602, -0.876818,
-0.357403, 0.354516, -0.864050,
-0.341572, 0.385011, -0.857377,
-0.320484, 0.294703, -0.900244,
-0.291352, 0.415435, -0.861701,
-0.325506, 0.414707, -0.849744,
-0.308973, 0.443956, -0.841094,
-0.326128, 0.471716, -0.819222,
-0.359158, 0.413488, -0.836680,
-0.354076, 0.293939, -0.887823,
-0.387280, 0.292822, -0.874225,
-0.372561, 0.323903, -0.869647,
-0.390470, 0.353281, -0.850133,
-0.419912, 0.291383, -0.859517,
-0.306378, 0.000000, -0.951910,
-0.328151, 0.033041, -0.944047,
-0.360716, 0.033021, -0.932091,
-0.349460, 0.066055, -0.934620,
-0.370201, 0.098882, -0.923674,
-0.371461, 0.000000, -0.928448,
-0.392778, 0.033014, -0.919041,
-0.402375, 0.098950, -0.910112,
-0.390534, 0.131782, -0.911107,
-0.413796, 0.065943, -0.907978,
-0.434130, 0.098843, -0.895411,
-0.465287, 0.098590, -0.879653,
-0.453679, 0.131555, -0.881402,
-0.424710, 0.032936, -0.904730,
-0.410104, 0.164179, -0.897140,
-0.441675, 0.164016, -0.882055,
-0.429071, 0.196334, -0.881675,
-0.447345, 0.228090, -0.864787,
-0.472637, 0.163706, -0.865919,
-0.434767, 0.000000, -0.900543,
-0.455785, 0.032928, -0.889480,
-0.486380, 0.032906, -0.873128,
-0.476118, 0.065827, -0.876914,
-0.495666, 0.098538, -0.862905,
-0.495991, 0.000000, -0.868328,
-0.516324, 0.032898, -0.855762,
0.516324, -0.032898, -0.855762,
0.486380, -0.032906, -0.873128,
0.476118, -0.065827, -0.876914,
0.455785, -0.032928, -0.889480,
0.424710, -0.032936, -0.904730,
0.495666, -0.098538, -0.862905,
0.465287, -0.098590, -0.879653,
0.413796, -0.065943, -0.907978,
0.392778, -0.033014, -0.919041,
0.434130, -0.098843, -0.895411,
0.402375, -0.098950, -0.910112,
0.390534, -0.131782, -0.911107,
0.370201, -0.098882, -0.923674,
0.453679, -0.131555, -0.881402,
0.360716, -0.033021, -0.932091,
0.349460, -0.066055, -0.934620,
0.328151, -0.033041, -0.944047,
0.295262, -0.033048, -0.954845,
0.337778, -0.098929, -0.936012,
0.472637, -0.163706, -0.865919,
0.441675, -0.164016, -0.882055,
0.429071, -0.196334, -0.881675,
0.410104, -0.164179, -0.897140,
0.378097, -0.164167, -0.911094,
0.447345, -0.228090, -0.864787,
0.415958, -0.228338, -0.880251,
0.282237, -0.066348, -0.957048,
0.260463, -0.033305, -0.964909,
0.303549, -0.099370, -0.947620,
0.268847, -0.099650, -0.958014,
0.255190, -0.132768, -0.957732,
0.233871, -0.099744, -0.967138,
0.324291, -0.132199, -0.936674,
0.275921, -0.165941, -0.946748,
0.241032, -0.166080, -0.956201,
0.310525, -0.165615, -0.936027,
0.296273, -0.198871, -0.934170,
0.316139, -0.231386, -0.920064,
0.281637, -0.231780, -0.931106,
0.344643, -0.165129, -0.924096,
0.226811, -0.198844, -0.953424,
0.206081, -0.165671, -0.964409,
0.247162, -0.231772, -0.940847,
0.212293, -0.231436, -0.949405,
0.197584, -0.263677, -0.944158,
0.177234, -0.230749, -0.956736,
0.267028, -0.264286, -0.926741,
0.364213, -0.197527, -0.910128,
0.383186, -0.229687, -0.894658,
0.349848, -0.230712, -0.907953,
0.335103, -0.263541, -0.904573,
0.401461, -0.261444, -0.877768,
0.226892, -0.033310, -0.973350,
0.213589, -0.066626, -0.974649,
0.192973, -0.033324, -0.980638,
0.158898, -0.033328, -0.986732,
0.200013, -0.099776, -0.974700,
0.145079, -0.066698, -0.987169,
0.124250, -0.033376, -0.991690,
0.165681, -0.099996, -0.981097,
0.131070, -0.100061, -0.986311,
0.116967, -0.133245, -0.984157,
0.096377, -0.099955, -0.990313,
0.185946, -0.133106, -0.973502,
0.089833, -0.033381, -0.995397,
0.075862, -0.066765, -0.994881,
0.055231, -0.033392, -0.997915,
0.020640, -0.033396, -0.999229,
0.061793, -0.099983, -0.993068,
0.171797, -0.165925, -0.971058,
0.157351, -0.198580, -0.967371,
0.137228, -0.166025, -0.976527,
0.102565, -0.165954, -0.980785,
0.142705, -0.230901, -0.962455,
0.419912, -0.291383, -0.859517,
0.387280, -0.292822, -0.874225,
0.372561, -0.323903, -0.869647,
0.354076, -0.293939, -0.887823,
0.320484, -0.294703, -0.900244,
0.390470, -0.353281, -0.850133,
0.357403, -0.354516, -0.864050,
0.305118, -0.325609, -0.894920,
0.286117, -0.295186, -0.911593,
0.323632, -0.355602, -0.876818,
0.289370, -0.356197, -0.888475,
0.273346, -0.386299, -0.880940,
0.254807, -0.356275, -0.898967,
0.341572, -0.385011, -0.857377,
0.251736, -0.295269, -0.921654,
0.236060, -0.325939, -0.915445,
0.216955, -0.295021, -0.930534,
0.181964, -0.294425, -0.938191,
0.220102, -0.356133, -0.908143,
0.359158, -0.413488, -0.836680,
0.325506, -0.414707, -0.849744,
0.308973, -0.443956, -0.841094,
0.291352, -0.415435, -0.861701,
0.256883, -0.415646, -0.872496,
0.326128, -0.471716, -0.819222,
0.292081, -0.472596, -0.831470,
0.000000, -0.066798, -0.997767,
-0.020640, -0.033396, -0.999229,
0.020640, -0.100210, -0.994752,
-0.020640, -0.100210, -0.994752,
-0.041156, -0.133410, -0.990206,
-0.061793, -0.099983, -0.993068,
0.041156, -0.133410, -0.990206,
-0.020642, -0.167078, -0.985728,
-0.061795, -0.166851, -0.984044,
0.020642, -0.167078, -0.985728,
0.000000, -0.200538, -0.979686,
0.020642, -0.233570, -0.972121,
-0.020642, -0.233570, -0.972121,
0.061795, -0.166851, -0.984044,
-0.082059, -0.199641, -0.976427,
-0.102565, -0.165954, -0.980785,
-0.061424, -0.233124, -0.970505,
-0.102196, -0.232233, -0.967277,
-0.122082, -0.264412, -0.956652,
-0.142705, -0.230901, -0.962455,
-0.040784, -0.266183, -0.963059,
0.082059, -0.199641, -0.976427,
0.102196, -0.232233, -0.967277,
0.061424, -0.233124, -0.970505,
0.040784, -0.266183, -0.963059,
0.122082, -0.264412, -0.956652,
-0.102238, -0.299185, -0.948702,
-0.142749, -0.297852, -0.943879,
-0.061453, -0.300080, -0.951932,
-0.082120, -0.333520, -0.939160,
-0.061865, -0.367185, -0.928088,
-0.102647, -0.366279, -0.924826,
-0.020669, -0.300527, -0.953549,
-0.020669, -0.367868, -0.929648,
-0.041197, -0.400628, -0.915314,
0.000000, -0.334425, -0.942422,
0.020669, -0.367868, -0.929648,
0.061865, -0.367185, -0.928088,
0.041197, -0.400628, -0.915314,
0.020669, -0.300527, -0.953549,
-0.020667, -0.432953, -0.901180,
-0.061863, -0.432270, -0.899620,
0.020667, -0.432953, -0.901180,
0.000000, -0.464610, -0.885516,
0.020667, -0.495384, -0.868428,
-0.020667, -0.495384, -0.868428,
0.061863, -0.432270, -0.899620,
0.061453, -0.300080, -0.951932,
0.102238, -0.299185, -0.948702,
0.082120, -0.333520, -0.939160,
0.102647, -0.366279, -0.924826,
0.142749, -0.297852, -0.943879,
-0.162122, -0.329196, -0.930240,
-0.181964, -0.294425, -0.938191,
-0.142013, -0.363521, -0.920698,
-0.181249, -0.360134, -0.915124,
-0.199999, -0.390469, -0.898629,
-0.220102, -0.356133, -0.908143,
-0.121761, -0.397184, -0.909626,
-0.179759, -0.424534, -0.887388,
-0.218637, -0.420581, -0.880517,
-0.140501, -0.427873, -0.892852,
-0.159117, -0.457934, -0.874631,
-0.138204, -0.490456, -0.860438,
-0.177481, -0.487173, -0.855081,
-0.101119, -0.430583, -0.896869,
-0.236644, -0.449711, -0.861255,
-0.256883, -0.415646, -0.872496,
-0.216004, -0.483108, -0.848498,
-0.254286, -0.478242, -0.840609,
-0.271440, -0.505993, -0.818713,
-0.292081, -0.472596, -0.831470,
-0.195090, -0.515629, -0.834306,
-0.080590, -0.462906, -0.882736,
-0.059926, -0.494553, -0.867079,
-0.099190, -0.492908, -0.864409,
-0.117537, -0.522087, -0.844754,
-0.039260, -0.525326, -0.849995,
0.162122, -0.329196, -0.930240,
0.181249, -0.360134, -0.915124,
0.142013, -0.363521, -0.920698,
0.121761, -0.397184, -0.909626,
0.199999, -0.390469, -0.898629,
0.140501, -0.427873, -0.892852,
0.101119, -0.430583, -0.896869,
0.179759, -0.424534, -0.887388,
0.159117, -0.457934, -0.874631,
0.177481, -0.487173, -0.855081,
0.138204, -0.490456, -0.860438,
0.218637, -0.420581, -0.880517,
0.080590, -0.462906, -0.882736,
0.099190, -0.492908, -0.864409,
0.059926, -0.494553, -0.867079,
0.039260, -0.525326, -0.849995,
0.117537, -0.522087, -0.844754,
0.236644, -0.449711, -0.861255,
0.254286, -0.478242, -0.840609,
0.216004, -0.483108, -0.848498,
0.195090, -0.515629, -0.834306,
0.271440, -0.505993, -0.818713,
-0.055231, -0.033392, -0.997915,
-0.075862, -0.066765, -0.994881,
-0.089833, -0.033381, -0.995397,
-0.124250, -0.033376, -0.991690,
-0.096377, -0.099955, -0.990313,
-0.145079, -0.066698, -0.987169,
-0.158898, -0.033328, -0.986732,
-0.131070, -0.100061, -0.986311,
-0.165681, -0.099996, -0.981097,
-0.185946, -0.133106, -0.973502,
-0.200013, -0.099776, -0.974700,
-0.116967, -0.133245, -0.984157,
-0.192973, -0.033324, -0.980638,
-0.213589, -0.066626, -0.974649,
-0.226892, -0.033310, -0.973350,
-0.260463, -0.033305, -0.964909,
-0.233871, -0.099744, -0.967138,
-0.137228, -0.166025, -0.976527,
-0.157351, -0.198580, -0.967371,
-0.171797, -0.165925, -0.971058,
-0.206081, -0.165671, -0.964409,
-0.177234, -0.230749, -0.956736,
-0.282237, -0.066348, -0.957048,
-0.295262, -0.033048, -0.954845,
-0.268847, -0.099650, -0.958014,
-0.303549, -0.099370, -0.947620,
-0.324291, -0.132199, -0.936674,
-0.337778, -0.098929, -0.936012,
-0.255190, -0.132768, -0.957732,
-0.310525, -0.165615, -0.936027,
-0.344643, -0.165129, -0.924096,
-0.275921, -0.165941, -0.946748,
-0.296273, -0.198871, -0.934170,
-0.281637, -0.231780, -0.931106,
-0.316139, -0.231386, -0.920064,
-0.241032, -0.166080, -0.956201,
-0.364213, -0.197527, -0.910128,
-0.378097, -0.164167, -0.911094,
-0.349848, -0.230712, -0.907953,
-0.383186, -0.229687, -0.894658,
-0.401461, -0.261444, -0.877768,
-0.415958, -0.228338, -0.880251,
-0.335103, -0.263541, -0.904573,
-0.226811, -0.198844, -0.953424,
-0.212293, -0.231436, -0.949405,
-0.247162, -0.231772, -0.940847,
-0.267028, -0.264286, -0.926741,
-0.197584, -0.263677, -0.944158,
-0.328151, -0.033041, -0.944047,
-0.349460, -0.066055, -0.934620,
-0.360716, -0.033021, -0.932091,
-0.392778, -0.033014, -0.919041,
-0.370201, -0.098882, -0.923674,
-0.413796, -0.065943, -0.907978,
-0.424710, -0.032936, -0.904730,
-0.402375, -0.098950, -0.910112,
-0.434130, -0.098843, -0.895411,
-0.453679, -0.131555, -0.881402,
-0.465287, -0.098590, -0.879653,
-0.390534, -0.131782, -0.911107,
-0.455785, -0.032928, -0.889480,
-0.476118, -0.065827, -0.876914,
-0.486380, -0.032906, -0.873128,
-0.516324, -0.032898, -0.855762,
-0.495666, -0.098538, -0.862905,
-0.410104, -0.164179, -0.897140,
-0.429071, -0.196334, -0.881675,
-0.441675, -0.164016, -0.882055,
-0.472637, -0.163706, -0.865919,
-0.447345, -0.228090, -0.864787,
-0.216955, -0.295021, -0.930534,
-0.236060, -0.325939, -0.915445,
-0.251736, -0.295269, -0.921654,
-0.286117, -0.295186, -0.911593,
-0.254807, -0.356275, -0.898967,
-0.305118, -0.325609, -0.894920,
-0.320484, -0.294703, -0.900244,
-0.289370, -0.356197, -0.888475,
-0.323632, -0.355602, -0.876818,
-0.341572, -0.385011, -0.857377,
-0.357403, -0.354516, -0.864050,
-0.273346, -0.386299, -0.880940,
-0.354076, -0.293939, -0.887823,
-0.372561, -0.323903, -0.869647,
-0.387280, -0.292822, -0.874225,
-0.419912, -0.291383, -0.859517,
-0.390470, -0.353281, -0.850133,
-0.291352, -0.415435, -0.861701,
-0.308973, -0.443956, -0.841094,
-0.325506, -0.414707, -0.849744,
-0.359158, -0.413488, -0.836680,
-0.326128, -0.471716, -0.819222,
0.291536, -0.527686, -0.797844,
0.253742, -0.533322, -0.806959,
0.235673, -0.560060, -0.794224,
0.215489, -0.538182, -0.814815,
0.176970, -0.542239, -0.821376,
0.255549, -0.581130, -0.772646,
0.217322, -0.586051, -0.780587,
0.158301, -0.568272, -0.807470,
0.137840, -0.545532, -0.826677,
0.178553, -0.590346, -0.787154,
0.139437, -0.593687, -0.792524,
0.120472, -0.618331, -0.776630,
0.100171, -0.596053, -0.796672,
0.198505, -0.611619, -0.765845,
0.098832, -0.547978, -0.830634,
0.079939, -0.573242, -0.815477,
0.059604, -0.549620, -0.833286,
0.020351, -0.550447, -0.834622,
0.060948, -0.597732, -0.799376,
0.218339, -0.631794, -0.743751,
0.179588, -0.636140, -0.750382,
0.160388, -0.659941, -0.733998,
0.140487, -0.639533, -0.755818,
0.101232, -0.641952, -0.760032,
0.180084, -0.679433, -0.711295,
0.140998, -0.682882, -0.716793,
0.000000, -0.574934, -0.818200,
-0.020351, -0.550447, -0.834622,
0.020351, -0.598760, -0.800670,
-0.020351, -0.598760, -0.800670,
-0.040604, -0.621778, -0.782140,
-0.060948, -0.597732, -0.799376,
0.040604, -0.621778, -0.782140,
-0.020352, -0.645320, -0.763641,
-0.060949, -0.644291, -0.762348,
0.020352, -0.645320, -0.763641,
0.000000, -0.668055, -0.744112,
0.020352, -0.689842, -0.723674,
-0.020352, -0.689842, -0.723674,
0.060949, -0.644291, -0.762348,
-0.081001, -0.665716, -0.741796,
-0.101232, -0.641952, -0.760032,
-0.060664, -0.688679, -0.722524,
-0.100949, -0.686355, -0.720226,
-0.120696, -0.706082, -0.697769,
-0.140998, -0.682882, -0.716793,
-0.040319, -0.710703, -0.702335,
0.081001, -0.665716, -0.741796,
0.100949, -0.686355, -0.720226,
0.060664, -0.688679, -0.722524,
0.040319, -0.710703, -0.702335,
0.120696, -0.706082, -0.697769,
-0.059604, -0.549620, -0.833286,
-0.079939, -0.573242, -0.815477,
-0.098832, -0.547978, -0.830634,
-0.137840, -0.545532, -0.826677,
-0.100171, -0.596053, -0.796672,
-0.158301, -0.568272, -0.807470,
-0.176970, -0.542239, -0.821376,
-0.139437, -0.593687, -0.792524,
-0.178553, -0.590346, -0.787154,
-0.198505, -0.611619, -0.765845,
-0.217322, -0.586051, -0.780587,
-0.120472, -0.618331, -0.776630,
-0.215489, -0.538182, -0.814815,
-0.235673, -0.560060, -0.794224,
-0.253742, -0.533322, -0.806959,
-0.291536, -0.527686, -0.797844,
-0.255549, -0.581130, -0.772646,
-0.140487, -0.639533, -0.755818,
-0.160388, -0.659941, -0.733998,
-0.179588, -0.636140, -0.750382,
-0.218339, -0.631794, -0.743751,
-0.180084, -0.679433, -0.711295,
0.140967, -0.723820, -0.675435,
0.100920, -0.727293, -0.678867,
0.080959, -0.747667, -0.659120,
0.060644, -0.729617, -0.681162,
0.020333, -0.730779, -0.682311,
0.101176, -0.764743, -0.636343,
0.060901, -0.767082, -0.638652,
0.000000, -0.750007, -0.661430,
-0.020333, -0.730779, -0.682311,
0.020333, -0.768380, -0.639671,
-0.020333, -0.768380, -0.639671,
-0.040577, -0.785785, -0.617167,
-0.060901, -0.767082, -0.638652,
0.040577, -0.785785, -0.617167,
-0.060644, -0.729617, -0.681162,
-0.080959, -0.747667, -0.659120,
-0.100920, -0.727293, -0.678867,
-0.140967, -0.723820, -0.675435,
-0.101176, -0.764743, -0.636343,
0.060900, -0.802005, -0.594204,
0.020332, -0.803303, -0.595224,
0.000000, -0.819885, -0.572529,
-0.020332, -0.803303, -0.595224,
-0.060900, -0.802005, -0.594204,
0.020332, -0.835429, -0.549222,
-0.020332, -0.835429, -0.549222,
0.855762, 0.516324, 0.032898,
0.873128, 0.486380, 0.032906,
0.868328, 0.495991, 0.000000,
0.876914, 0.476118, 0.065827,
0.889480, 0.455785, 0.032928,
0.904730, 0.424710, 0.032936,
0.900543, 0.434767, 0.000000,
0.862905, 0.495666, 0.098538,
0.879653, 0.465287, 0.098590,
0.907978, 0.413796, 0.065943,
0.919041, 0.392778, 0.033014,
0.895411, 0.434130, 0.098843,
0.910112, 0.402375, 0.098950,
0.911107, 0.390534, 0.131782,
0.923674, 0.370201, 0.098882,
0.881402, 0.453679, 0.131555,
0.932091, 0.360716, 0.033021,
0.928448, 0.371461, 0.000000,
0.934620, 0.349460, 0.066055,
0.944047, 0.328151, 0.033041,
0.954845, 0.295262, 0.033048,
0.951910, 0.306378, 0.000000,
0.936012, 0.337778, 0.098929,
0.865919, 0.472637, 0.163706,
0.882055, 0.441675, 0.164016,
0.881675, 0.429071, 0.196334,
0.897140, 0.410104, 0.164179,
0.911094, 0.378097, 0.164167,
0.864787, 0.447345, 0.228090,
0.880251, 0.415958, 0.228338,
0.957048, 0.282237, 0.066348,
0.964909, 0.260463, 0.033305,
0.947620, 0.303549, 0.099370,
0.958014, 0.268847, 0.099650,
0.957732, 0.255190, 0.132768,
0.967138, 0.233871, 0.099744,
0.936674, 0.324291, 0.132199,
0.946748, 0.275921, 0.165941,
0.956201, 0.241032, 0.166080,
0.936027, 0.310525, 0.165615,
0.934170, 0.296273, 0.198871,
0.920064, 0.316139, 0.231386,
0.931106, 0.281637, 0.231780,
0.924096, 0.344643, 0.165129,
0.953424, 0.226811, 0.198844,
0.964409, 0.206081, 0.165671,
0.940847, 0.247162, 0.231772,
0.949405, 0.212293, 0.231436,
0.944158, 0.197584, 0.263677,
0.956736, 0.177234, 0.230749,
0.926741, 0.267028, 0.264286,
0.910128, 0.364213, 0.197527,
0.894658, 0.383186, 0.229687,
0.907953, 0.349848, 0.230712,
0.904573, 0.335103, 0.263541,
0.877768, 0.401461, 0.261444,
0.973350, 0.226892, 0.033310,
0.970815, 0.239828, 0.000000,
0.974649, 0.213589, 0.066626,
0.980638, 0.192973, 0.033324,
0.986732, 0.158898, 0.033328,
0.985074, 0.172130, 0.000000,
0.974700, 0.200013, 0.099776,
0.987169, 0.145079, 0.066698,
0.991690, 0.124250, 0.033376,
0.981097, 0.165681, 0.099996,
0.986311, 0.131070, 0.100061,
0.984157, 0.116967, 0.133245,
0.990313, 0.096377, 0.099955,
0.973502, 0.185946, 0.133106,
0.995397, 0.089833, 0.033381,
0.994618, 0.103609, 0.000000,
0.994881, 0.075862, 0.066765,
0.997915, 0.055231, 0.033392,
0.999229, 0.020640, 0.033396,
0.999402, 0.034591, 0.000000,
0.993068, 0.061793, 0.099983,
0.971058, 0.171797, 0.165925,
0.967371, 0.157351, 0.198580,
0.976527, 0.137228, 0.166025,
0.980785, 0.102565, 0.165954,
0.962455, 0.142705, 0.230901,
0.859517, 0.419912, 0.291383,
0.874225, 0.387280, 0.292822,
0.869647, 0.372561, 0.323903,
0.887823, 0.354076, 0.293939,
0.900244, 0.320484, 0.294703,
0.850133, 0.390470, 0.353281,
0.864050, 0.357403, 0.354516,
0.894920, 0.305118, 0.325609,
0.911593, 0.286117, 0.295186,
0.876818, 0.323632, 0.355602,
0.888475, 0.289370, 0.356197,
0.880940, 0.273346, 0.386299,
0.898967, 0.254807, 0.356275,
0.857377, 0.341572, 0.385011,
0.921654, 0.251736, 0.295269,
0.915445, 0.236060, 0.325939,
0.930534, 0.216955, 0.295021,
0.938191, 0.181964, 0.294425,
0.908143, 0.220102, 0.356133,
0.836680, 0.359158, 0.413488,
0.849744, 0.325506, 0.414707,
0.841094, 0.308973, 0.443956,
0.861701, 0.291352, 0.415435,
0.872496, 0.256883, 0.415646,
0.819222, 0.326128, 0.471716,
0.831470, 0.292081, 0.472596,
0.997767, 0.000000, 0.066798,
0.999229, -0.020640, 0.033396,
0.994752, 0.020640, 0.100210,
0.994752, -0.020640, 0.100210,
0.990206, -0.041156, 0.133410,
0.993068, -0.061793, 0.099983,
0.990206, 0.041156, 0.133410,
0.985728, -0.020642, 0.167078,
0.984044, -0.061795, 0.166851,
0.985728, 0.020642, 0.167078,
0.979686, 0.000000, 0.200538,
0.972121, 0.020642, 0.233570,
0.972121, -0.020642, 0.233570,
0.984044, 0.061795, 0.166851,
0.976427, -0.082059, 0.199641,
0.980785, -0.102565, 0.165954,
0.970505, -0.061424, 0.233124,
0.967277, -0.102196, 0.232233,
0.956652, -0.122082, 0.264412,
0.962455, -0.142705, 0.230901,
0.963059, -0.040784, 0.266183,
0.976427, 0.082059, 0.199641,
0.967277, 0.102196, 0.232233,
0.970505, 0.061424, 0.233124,
0.963059, 0.040784, 0.266183,
0.956652, 0.122082, 0.264412,
0.948702, -0.102238, 0.299185,
0.943879, -0.142749, 0.297852,
0.951932, -0.061453, 0.300080,
0.939160, -0.082120, 0.333520,
0.928088, -0.061865, 0.367185,
0.924826, -0.102647, 0.366279,
0.953549, -0.020669, 0.300527,
0.929648, -0.020669, 0.367868,
0.915314, -0.041197, 0.400628,
0.942422, 0.000000, 0.334425,
0.929648, 0.020669, 0.367868,
0.928088, 0.061865, 0.367185,
0.915314, 0.041197, 0.400628,
0.953549, 0.020669, 0.300527,
0.901180, -0.020667, 0.432953,
0.899620, -0.061863, 0.432270,
0.901180, 0.020667, 0.432953,
0.885516, 0.000000, 0.464610,
0.868428, 0.020667, 0.495384,
0.868428, -0.020667, 0.495384,
0.899620, 0.061863, 0.432270,
0.951932, 0.061453, 0.300080,
0.948702, 0.102238, 0.299185,
0.939160, 0.082120, 0.333520,
0.924826, 0.102647, 0.366279,
0.943879, 0.142749, 0.297852,
0.930240, -0.162122, 0.329196,
0.938191, -0.181964, 0.294425,
0.920698, -0.142013, 0.363521,
0.915124, -0.181249, 0.360134,
0.898629, -0.199999, 0.390469,
0.908143, -0.220102, 0.356133,
0.909626, -0.121761, 0.397184,
0.887388, -0.179759, 0.424534,
0.880517, -0.218637, 0.420581,
0.892852, -0.140501, 0.427873,
0.874631, -0.159117, 0.457934,
0.860438, -0.138204, 0.490456,
0.855081, -0.177481, 0.487173,
0.896869, -0.101119, 0.430583,
0.861255, -0.236644, 0.449711,
0.872496, -0.256883, 0.415646,
0.848498, -0.216004, 0.483108,
0.840609, -0.254286, 0.478242,
0.818713, -0.271440, 0.505993,
0.831470, -0.292081, 0.472596,
0.834306, -0.195090, 0.515629,
0.882736, -0.080590, 0.462906,
0.867079, -0.059926, 0.494553,
0.864409, -0.099190, 0.492908,
0.844754, -0.117537, 0.522087,
0.849995, -0.039260, 0.525326,
0.930240, 0.162122, 0.329196,
0.915124, 0.181249, 0.360134,
0.920698, 0.142013, 0.363521,
0.909626, 0.121761, 0.397184,
0.898629, 0.199999, 0.390469,
0.892852, 0.140501, 0.427873,
0.896869, 0.101119, 0.430583,
0.887388, 0.179759, 0.424534,
0.874631, 0.159117, 0.457934,
0.855081, 0.177481, 0.487173,
0.860438, 0.138204, 0.490456,
0.880517, 0.218637, 0.420581,
0.882736, 0.080590, 0.462906,
0.864409, 0.099190, 0.492908,
0.867079, 0.059926, 0.494553,
0.849995, 0.039260, 0.525326,
0.844754, 0.117537, 0.522087,
0.861255, 0.236644, 0.449711,
0.840609, 0.254286, 0.478242,
0.848498, 0.216004, 0.483108,
0.834306, 0.195090, 0.515629,
0.818713, 0.271440, 0.505993,
0.997915, -0.055231, 0.033392,
0.999402, -0.034591, 0.000000,
0.994881, -0.075862, 0.066765,
0.995397, -0.089833, 0.033381,
0.991690, -0.124250, 0.033376,
0.994618, -0.103609, 0.000000,
0.990313, -0.096377, 0.099955,
0.987169, -0.145079, 0.066698,
0.986732, -0.158898, 0.033328,
0.986311, -0.131070, 0.100061,
0.981097, -0.165681, 0.099996,
0.973502, -0.185946, 0.133106,
0.974700, -0.200013, 0.099776,
0.984157, -0.116967, 0.133245,
0.980638, -0.192973, 0.033324,
0.985074, -0.172130, 0.000000,
0.974649, -0.213589, 0.066626,
0.973350, -0.226892, 0.033310,
0.964909, -0.260463, 0.033305,
0.970815, -0.239828, 0.000000,
0.967138, -0.233871, 0.099744,
0.976527, -0.137228, 0.166025,
0.967371, -0.157351, 0.198580,
0.971058, -0.171797, 0.165925,
0.964409, -0.206081, 0.165671,
0.956736, -0.177234, 0.230749,
0.957048, -0.282237, 0.066348,
0.954845, -0.295262, 0.033048,
0.958014, -0.268847, 0.099650,
0.947620, -0.303549, 0.099370,
0.936674, -0.324291, 0.132199,
0.936012, -0.337778, 0.098929,
0.957732, -0.255190, 0.132768,
0.936027, -0.310525, 0.165615,
0.924096, -0.344643, 0.165129,
0.946748, -0.275921, 0.165941,
0.934170, -0.296273, 0.198871,
0.931106, -0.281637, 0.231780,
0.920064, -0.316139, 0.231386,
0.956201, -0.241032, 0.166080,
0.910128, -0.364213, 0.197527,
0.911094, -0.378097, 0.164167,
0.907953, -0.349848, 0.230712,
0.894658, -0.383186, 0.229687,
0.877768, -0.401461, 0.261444,
0.880251, -0.415958, 0.228338,
0.904573, -0.335103, 0.263541,
0.953424, -0.226811, 0.198844,
0.949405, -0.212293, 0.231436,
0.940847, -0.247162, 0.231772,
0.926741, -0.267028, 0.264286,
0.944158, -0.197584, 0.263677,
0.944047, -0.328151, 0.033041,
0.951910, -0.306378, 0.000000,
0.934620, -0.349460, 0.066055,
0.932091, -0.360716, 0.033021,
0.919041, -0.392778, 0.033014,
0.928448, -0.371461, 0.000000,
0.923674, -0.370201, 0.098882,
0.907978, -0.413796, 0.065943,
0.904730, -0.424710, 0.032936,
0.910112, -0.402375, 0.098950,
0.895411, -0.434130, 0.098843,
0.881402, -0.453679, 0.131555,
0.879653, -0.465287, 0.098590,
0.911107, -0.390534, 0.131782,
0.889480, -0.455785, 0.032928,
0.900543, -0.434767, 0.000000,
0.876914, -0.476118, 0.065827,
0.873128, -0.486380, 0.032906,
0.855762, -0.516324, 0.032898,
0.868328, -0.495991, 0.000000,
0.862905, -0.495666, 0.098538,
0.897140, -0.410104, 0.164179,
0.881675, -0.429071, 0.196334,
0.882055, -0.441675, 0.164016,
0.865919, -0.472637, 0.163706,
0.864787, -0.447345, 0.228090,
0.930534, -0.216955, 0.295021,
0.915445, -0.236060, 0.325939,
0.921654, -0.251736, 0.295269,
0.911593, -0.286117, 0.295186,
0.898967, -0.254807, 0.356275,
0.894920, -0.305118, 0.325609,
0.900244, -0.320484, 0.294703,
0.888475, -0.289370, 0.356197,
0.876818, -0.323632, 0.355602,
0.857377, -0.341572, 0.385011,
0.864050, -0.357403, 0.354516,
0.880940, -0.273346, 0.386299,
0.887823, -0.354076, 0.293939,
0.869647, -0.372561, 0.323903,
0.874225, -0.387280, 0.292822,
0.859517, -0.419912, 0.291383,
0.850133, -0.390470, 0.353281,
0.861701, -0.291352, 0.415435,
0.841094, -0.308973, 0.443956,
0.849744, -0.325506, 0.414707,
0.836680, -0.359158, 0.413488,
0.819222, -0.326128, 0.471716,
0.797844, 0.291536, 0.527686,
0.806959, 0.253742, 0.533322,
0.794224, 0.235673, 0.560060,
0.814815, 0.215489, 0.538182,
0.821376, 0.176970, 0.542239,
0.772646, 0.255549, 0.581130,
0.780587, 0.217322, 0.586051,
0.807470, 0.158301, 0.568272,
0.826677, 0.137840, 0.545532,
0.787154, 0.178553, 0.590346,
0.792524, 0.139437, 0.593687,
0.776630, 0.120472, 0.618331,
0.796672, 0.100171, 0.596053,
0.765845, 0.198505, 0.611619,
0.830634, 0.098832, 0.547978,
0.815477, 0.079939, 0.573242,
0.833286, 0.059604, 0.549620,
0.834622, 0.020351, 0.550447,
0.799376, 0.060948, 0.597732,
0.743751, 0.218339, 0.631794,
0.750382, 0.179588, 0.636140,
0.733998, 0.160388, 0.659941,
0.755818, 0.140487, 0.639533,
0.760032, 0.101232, 0.641952,
0.711295, 0.180084, 0.679433,
0.716793, 0.140998, 0.682882,
0.818200, 0.000000, 0.574934,
0.834622, -0.020351, 0.550447,
0.800670, 0.020351, 0.598760,
0.800670, -0.020351, 0.598760,
0.782140, -0.040604, 0.621778,
0.799376, -0.060948, 0.597732,
0.782140, 0.040604, 0.621778,
0.763641, -0.020352, 0.645320,
0.762348, -0.060949, 0.644291,
0.763641, 0.020352, 0.645320,
0.744112, 0.000000, 0.668055,
0.723674, 0.020352, 0.689842,
0.723674, -0.020352, 0.689842,
0.762348, 0.060949, 0.644291,
0.741796, -0.081001, 0.665716,
0.760032, -0.101232, 0.641952,
0.722524, -0.060664, 0.688679,
0.720226, -0.100949, 0.686355,
0.697769, -0.120696, 0.706082,
0.716793, -0.140998, 0.682882,
0.702335, -0.040319, 0.710703,
0.741796, 0.081001, 0.665716,
0.720226, 0.100949, 0.686355,
0.722524, 0.060664, 0.688679,
0.702335, 0.040319, 0.710703,
0.697769, 0.120696, 0.706082,
0.833286, -0.059604, 0.549620,
0.815477, -0.079939, 0.573242,
0.830634, -0.098832, 0.547978,
0.826677, -0.137840, 0.545532,
0.796672, -0.100171, 0.596053,
0.807470, -0.158301, 0.568272,
0.821376, -0.176970, 0.542239,
0.792524, -0.139437, 0.593687,
0.787154, -0.178553, 0.590346,
0.765845, -0.198505, 0.611619,
0.780587, -0.217322, 0.586051,
0.776630, -0.120472, 0.618331,
0.814815, -0.215489, 0.538182,
0.794224, -0.235673, 0.560060,
0.806959, -0.253742, 0.533322,
0.797844, -0.291536, 0.527686,
0.772646, -0.255549, 0.581130,
0.755818, -0.140487, 0.639533,
0.733998, -0.160388, 0.659941,
0.750382, -0.179588, 0.636140,
0.743751, -0.218339, 0.631794,
0.711295, -0.180084, 0.679433,
0.675435, 0.140967, 0.723820,
0.678867, 0.100920, 0.727293,
0.659120, 0.080959, 0.747667,
0.681162, 0.060644, 0.729617,
0.682311, 0.020333, 0.730779,
0.636343, 0.101176, 0.764743,
0.638652, 0.060901, 0.767082,
0.661430, 0.000000, 0.750007,
0.682311, -0.020333, 0.730779,
0.639671, 0.020333, 0.768380,
0.639671, -0.020333, 0.768380,
0.617167, -0.040577, 0.785785,
0.638652, -0.060901, 0.767082,
0.617167, 0.040577, 0.785785,
0.681162, -0.060644, 0.729617,
0.659120, -0.080959, 0.747667,
0.678867, -0.100920, 0.727293,
0.675435, -0.140967, 0.723820,
0.636343, -0.101176, 0.764743,
0.594204, 0.060900, 0.802005,
0.595224, 0.020332, 0.803303,
0.572529, 0.000000, 0.819885,
0.595224, -0.020332, 0.803303,
0.594204, -0.060900, 0.802005,
0.549222, 0.020332, 0.835429,
0.549222, -0.020332, 0.835429,
0.873128, 0.486380, -0.032906,
0.855762, 0.516324, -0.032898,
0.889480, 0.455785, -0.032928,
0.876914, 0.476118, -0.065827,
0.879653, 0.465287, -0.098590,
0.862905, 0.495666, -0.098538,
0.904730, 0.424710, -0.032936,
0.895411, 0.434130, -0.098843,
0.881402, 0.453679, -0.131555,
0.907978, 0.413796, -0.065943,
0.910112, 0.402375, -0.098950,
0.923674, 0.370201, -0.098882,
0.911107, 0.390534, -0.131782,
0.919041, 0.392778, -0.033014,
0.882055, 0.441675, -0.164016,
0.865919, 0.472637, -0.163706,
0.897140, 0.410104, -0.164179,
0.881675, 0.429071, -0.196334,
0.880251, 0.415958, -0.228338,
0.864787, 0.447345, -0.228090,
0.911094, 0.378097, -0.164167,
0.932091, 0.360716, -0.033021,
0.944047, 0.328151, -0.033041,
0.934620, 0.349460, -0.066055,
0.936012, 0.337778, -0.098929,
0.954845, 0.295262, -0.033048,
0.894658, 0.383186, -0.229687,
0.877768, 0.401461, -0.261444,
0.910128, 0.364213, -0.197527,
0.907953, 0.349848, -0.230712,
0.920064, 0.316139, -0.231386,
0.904573, 0.335103, -0.263541,
0.924096, 0.344643, -0.165129,
0.934170, 0.296273, -0.198871,
0.931106, 0.281637, -0.231780,
0.936027, 0.310525, -0.165615,
0.946748, 0.275921, -0.165941,
0.957732, 0.255190, -0.132768,
0.956201, 0.241032, -0.166080,
0.936674, 0.324291, -0.132199,
0.940847, 0.247162, -0.231772,
0.926741, 0.267028, -0.264286,
0.953424, 0.226811, -0.198844,
0.949405, 0.212293, -0.231436,
0.956736, 0.177234, -0.230749,
0.944158, 0.197584, -0.263677,
0.964409, 0.206081, -0.165671,
0.947620, 0.303549, -0.099370,
0.957048, 0.282237, -0.066348,
0.958014, 0.268847, -0.099650,
0.967138, 0.233871, -0.099744,
0.964909, 0.260463, -0.033305,
0.874225, 0.387280, -0.292822,
0.859517, 0.419912, -0.291383,
0.887823, 0.354076, -0.293939,
0.869647, 0.372561, -0.323903,
0.864050, 0.357403, -0.354516,
0.850133, 0.390470, -0.353281,
0.900244, 0.320484, -0.294703,
0.876818, 0.323632, -0.355602,
0.857377, 0.341572, -0.385011,
0.894920, 0.305118, -0.325609,
0.888475, 0.289370, -0.356197,
0.898967, 0.254807, -0.356275,
0.880940, 0.273346, -0.386299,
0.911593, 0.286117, -0.295186,
0.849744, 0.325506, -0.414707,
0.836680, 0.359158, -0.413488,
0.861701, 0.291352, -0.415435,
0.841094, 0.308973, -0.443956,
0.831470, 0.292081, -0.472596,
0.819222, 0.326128, -0.471716,
0.872496, 0.256883, -0.415646,
0.921654, 0.251736, -0.295269,
0.930534, 0.216955, -0.295021,
0.915445, 0.236060, -0.325939,
0.908143, 0.220102, -0.356133,
0.938191, 0.181964, -0.294425,
0.973350, 0.226892, -0.033310,
0.980638, 0.192973, -0.033324,
0.974649, 0.213589, -0.066626,
0.974700, 0.200013, -0.099776,
0.986732, 0.158898, -0.033328,
0.981097, 0.165681, -0.099996,
0.973502, 0.185946, -0.133106,
0.987169, 0.145079, -0.066698,
0.986311, 0.131070, -0.100061,
0.990313, 0.096377, -0.099955,
0.984157, 0.116967, -0.133245,
0.991690, 0.124250, -0.033376,
0.971058, 0.171797, -0.165925,
0.976527, 0.137228, -0.166025,
0.967371, 0.157351, -0.198580,
0.962455, 0.142705, -0.230901,
0.980785, 0.102565, -0.165954,
0.995397, 0.089833, -0.033381,
0.997915, 0.055231, -0.033392,
0.994881, 0.075862, -0.066765,
0.993068, 0.061793, -0.099983,
0.999229, 0.020640, -0.033396,
0.840609, 0.254286, -0.478242,
0.818713, 0.271440, -0.505993,
0.861255, 0.236644, -0.449711,
0.848498, 0.216004, -0.483108,
0.855081, 0.177481, -0.487173,
0.834306, 0.195090, -0.515629,
0.880517, 0.218637, -0.420581,
0.874631, 0.159117, -0.457934,
0.860438, 0.138204, -0.490456,
0.887388, 0.179759, -0.424534,
0.892852, 0.140501, -0.427873,
0.909626, 0.121761, -0.397184,
0.896869, 0.101119, -0.430583,
0.898629, 0.199999, -0.390469,
0.864409, 0.099190, -0.492908,
0.844754, 0.117537, -0.522087,
0.882736, 0.080590, -0.462906,
0.867079, 0.059926, -0.494553,
0.868428, 0.020667, -0.495384,
0.849995, 0.039260, -0.525326,
0.899620, 0.061863, -0.432270,
0.915124, 0.181249, -0.360134,
0.930240, 0.162122, -0.329196,
0.920698, 0.142013, -0.363521,
0.924826, 0.102647, -0.366279,
0.943879, 0.142749, -0.297852,
0.885516, 0.000000, -0.464610,
0.868428, -0.020667, -0.495384,
0.901180, 0.020667, -0.432953,
0.901180, -0.020667, -0.432953,
0.915314, -0.041197, -0.400628,
0.899620, -0.061863, -0.432270,
0.915314, 0.041197, -0.400628,
0.929648, -0.020669, -0.367868,
0.928088, -0.061865, -0.367185,
0.929648, 0.020669, -0.367868,
0.942422, 0.000000, -0.334425,
0.953549, 0.020669, -0.300527,
0.953549, -0.020669, -0.300527,
0.928088, 0.061865, -0.367185,
0.939160, -0.082120, -0.333520,
0.924826, -0.102647, -0.366279,
0.951932, -0.061453, -0.300080,
0.948702, -0.102238, -0.299185,
0.956652, -0.122082, -0.264412,
0.943879, -0.142749, -0.297852,
0.963059, -0.040784, -0.266183,
0.939160, 0.082120, -0.333520,
0.948702, 0.102238, -0.299185,
0.951932, 0.061453, -0.300080,
0.963059, 0.040784, -0.266183,
0.956652, 0.122082, -0.264412,
0.867079, -0.059926, -0.494553,
0.849995, -0.039260, -0.525326,
0.882736, -0.080590, -0.462906,
0.864409, -0.099190, -0.492908,
0.860438, -0.138204, -0.490456,
0.844754, -0.117537, -0.522087,
0.896869, -0.101119, -0.430583,
0.874631, -0.159117, -0.457934,
0.855081, -0.177481, -0.487173,
0.892852, -0.140501, -0.427873,
0.887388, -0.179759, -0.424534,
0.898629, -0.199999, -0.390469,
0.880517, -0.218637, -0.420581,
0.909626, -0.121761, -0.397184,
0.848498, -0.216004, -0.483108,
0.834306, -0.195090, -0.515629,
0.861255, -0.236644, -0.449711,
0.840609, -0.254286, -0.478242,
0.831470, -0.292081, -0.472596,
0.818713, -0.271440, -0.505993,
0.872496, -0.256883, -0.415646,
0.920698, -0.142013, -0.363521,
0.930240, -0.162122, -0.329196,
0.915124, -0.181249, -0.360134,
0.908143, -0.220102, -0.356133,
0.938191, -0.181964, -0.294425,
0.967277, 0.102196, -0.232233,
0.976427, 0.082059, -0.199641,
0.970505, 0.061424, -0.233124,
0.972121, 0.020642, -0.233570,
0.984044, 0.061795, -0.166851,
0.979686, 0.000000, -0.200538,
0.972121, -0.020642, -0.233570,
0.985728, 0.020642, -0.167078,
0.985728, -0.020642, -0.167078,
0.990206, -0.041156, -0.133410,
0.984044, -0.061795, -0.166851,
0.990206, 0.041156, -0.133410,
0.970505, -0.061424, -0.233124,
0.976427, -0.082059, -0.199641,
0.967277, -0.102196, -0.232233,
0.962455, -0.142705, -0.230901,
0.980785, -0.102565, -0.165954,
0.994752, 0.020640, -0.100210,
0.997767, 0.000000, -0.066798,
0.994752, -0.020640, -0.100210,
0.993068, -0.061793, -0.099983,
0.999229, -0.020640, -0.033396,
0.806959, 0.253742, -0.533322,
0.797844, 0.291536, -0.527686,
0.814815, 0.215489, -0.538182,
0.794224, 0.235673, -0.560060,
0.780587, 0.217322, -0.586051,
0.772646, 0.255549, -0.581130,
0.821376, 0.176970, -0.542239,
0.787154, 0.178553, -0.590346,
0.765845, 0.198505, -0.611619,
0.807470, 0.158301, -0.568272,
0.792524, 0.139437, -0.593687,
0.796672, 0.100171, -0.596053,
0.776630, 0.120472, -0.618331,
0.826677, 0.137840, -0.545532,
0.750382, 0.179588, -0.636140,
0.743751, 0.218339, -0.631794,
0.755818, 0.140487, -0.639533,
0.733998, 0.160388, -0.659941,
0.716793, 0.140998, -0.682882,
0.711295, 0.180084, -0.679433,
0.760032, 0.101232, -0.641952,
0.830634, 0.098832, -0.547978,
0.833286, 0.059604, -0.549620,
0.815477, 0.079939, -0.573242,
0.799376, 0.060948, -0.597732,
0.834622, 0.020351, -0.550447,
0.720226, 0.100949, -0.686355,
0.697769, 0.120696, -0.706082,
0.741796, 0.081001, -0.665716,
0.722524, 0.060664, -0.688679,
0.723674, 0.020352, -0.689842,
0.702335, 0.040319, -0.710703,
0.762348, 0.060949, -0.644291,
0.744112, 0.000000, -0.668055,
0.723674, -0.020352, -0.689842,
0.763641, 0.020352, -0.645320,
0.763641, -0.020352, -0.645320,
0.782140, -0.040604, -0.621778,
0.762348, -0.060949, -0.644291,
0.782140, 0.040604, -0.621778,
0.722524, -0.060664, -0.688679,
0.702335, -0.040319, -0.710703,
0.741796, -0.081001, -0.665716,
0.720226, -0.100949, -0.686355,
0.716793, -0.140998, -0.682882,
0.697769, -0.120696, -0.706082,
0.760032, -0.101232, -0.641952,
0.800670, 0.020351, -0.598760,
0.818200, 0.000000, -0.574934,
0.800670, -0.020351, -0.598760,
0.799376, -0.060948, -0.597732,
0.834622, -0.020351, -0.550447,
0.678867, 0.100920, -0.727293,
0.675435, 0.140967, -0.723820,
0.681162, 0.060644, -0.729617,
0.659120, 0.080959, -0.747667,
0.638652, 0.060901, -0.767082,
0.636343, 0.101176, -0.764743,
0.682311, 0.020333, -0.730779,
0.639671, 0.020333, -0.768380,
0.617167, 0.040577, -0.785785,
0.661430, 0.000000, -0.750007,
0.639671, -0.020333, -0.768380,
0.638652, -0.060901, -0.767082,
0.617167, -0.040577, -0.785785,
0.682311, -0.020333, -0.730779,
0.595224, 0.020332, -0.803303,
0.594204, 0.060900, -0.802005,
0.595224, -0.020332, -0.803303,
0.572529, 0.000000, -0.819885,
0.549222, -0.020332, -0.835429,
0.549222, 0.020332, -0.835429,
0.594204, -0.060900, -0.802005,
0.681162, -0.060644, -0.729617,
0.678867, -0.100920, -0.727293,
0.659120, -0.080959, -0.747667,
0.636343, -0.101176, -0.764743,
0.675435, -0.140967, -0.723820,
0.833286, -0.059604, -0.549620,
0.830634, -0.098832, -0.547978,
0.815477, -0.079939, -0.573242,
0.796672, -0.100171, -0.596053,
0.826677, -0.137840, -0.545532,
0.792524, -0.139437, -0.593687,
0.776630, -0.120472, -0.618331,
0.807470, -0.158301, -0.568272,
0.787154, -0.178553, -0.590346,
0.780587, -0.217322, -0.586051,
0.765845, -0.198505, -0.611619,
0.821376, -0.176970, -0.542239,
0.755818, -0.140487, -0.639533,
0.750382, -0.179588, -0.636140,
0.733998, -0.160388, -0.659941,
0.711295, -0.180084, -0.679433,
0.743751, -0.218339, -0.631794,
0.814815, -0.215489, -0.538182,
0.806959, -0.253742, -0.533322,
0.794224, -0.235673, -0.560060,
0.772646, -0.255549, -0.581130,
0.797844, -0.291536, -0.527686,
0.997915, -0.055231, -0.033392,
0.995397, -0.089833, -0.033381,
0.994881, -0.075862, -0.066765,
0.990313, -0.096377, -0.099955,
0.991690, -0.124250, -0.033376,
0.986311, -0.131070, -0.100061,
0.984157, -0.116967, -0.133245,
0.987169, -0.145079, -0.066698,
0.981097, -0.165681, -0.099996,
0.974700, -0.200013, -0.099776,
0.973502, -0.185946, -0.133106,
0.986732, -0.158898, -0.033328,
0.976527, -0.137228, -0.166025,
0.971058, -0.171797, -0.165925,
0.967371, -0.157351, -0.198580,
0.956736, -0.177234, -0.230749,
0.964409, -0.206081, -0.165671,
0.980638, -0.192973, -0.033324,
0.973350, -0.226892, -0.033310,
0.974649, -0.213589, -0.066626,
0.967138, -0.233871, -0.099744,
0.964909, -0.260463, -0.033305,
0.949405, -0.212293, -0.231436,
0.944158, -0.197584, -0.263677,
0.953424, -0.226811, -0.198844,
0.940847, -0.247162, -0.231772,
0.931106, -0.281637, -0.231780,
0.926741, -0.267028, -0.264286,
0.956201, -0.241032, -0.166080,
0.934170, -0.296273, -0.198871,
0.920064, -0.316139, -0.231386,
0.946748, -0.275921, -0.165941,
0.936027, -0.310525, -0.165615,
0.936674, -0.324291, -0.132199,
0.924096, -0.344643, -0.165129,
0.957732, -0.255190, -0.132768,
0.907953, -0.349848, -0.230712,
0.904573, -0.335103, -0.263541,
0.910128, -0.364213, -0.197527,
0.894658, -0.383186, -0.229687,
0.880251, -0.415958, -0.228338,
0.877768, -0.401461, -0.261444,
0.911094, -0.378097, -0.164167,
0.958014, -0.268847, -0.099650,
0.957048, -0.282237, -0.066348,
0.947620, -0.303549, -0.099370,
0.936012, -0.337778, -0.098929,
0.954845, -0.295262, -0.033048,
0.930534, -0.216955, -0.295021,
0.921654, -0.251736, -0.295269,
0.915445, -0.236060, -0.325939,
0.898967, -0.254807, -0.356275,
0.911593, -0.286117, -0.295186,
0.888475, -0.289370, -0.356197,
0.880940, -0.273346, -0.386299,
0.894920, -0.305118, -0.325609,
0.876818, -0.323632, -0.355602,
0.864050, -0.357403, -0.354516,
0.857377, -0.341572, -0.385011,
0.900244, -0.320484, -0.294703,
0.861701, -0.291352, -0.415435,
0.849744, -0.325506, -0.414707,
0.841094, -0.308973, -0.443956,
0.819222, -0.326128, -0.471716,
0.836680, -0.359158, -0.413488,
0.887823, -0.354076, -0.293939,
0.874225, -0.387280, -0.292822,
0.869647, -0.372561, -0.323903,
0.850133, -0.390470, -0.353281,
0.859517, -0.419912, -0.291383,
0.944047, -0.328151, -0.033041,
0.932091, -0.360716, -0.033021,
0.934620, -0.349460, -0.066055,
0.923674, -0.370201, -0.098882,
0.919041, -0.392778, -0.033014,
0.910112, -0.402375, -0.098950,
0.911107, -0.390534, -0.131782,
0.907978, -0.413796, -0.065943,
0.895411, -0.434130, -0.098843,
0.879653, -0.465287, -0.098590,
0.881402, -0.453679, -0.131555,
0.904730, -0.424710, -0.032936,
0.897140, -0.410104, -0.164179,
0.882055, -0.441675, -0.164016,
0.881675, -0.429071, -0.196334,
0.864787, -0.447345, -0.228090,
0.865919, -0.472637, -0.163706,
0.889480, -0.455785, -0.032928,
0.873128, -0.486380, -0.032906,
0.876914, -0.476118, -0.065827,
0.862905, -0.495666, -0.098538,
0.855762, -0.516324, -0.032898,
-0.855762, -0.516324, 0.032898,
-0.873128, -0.486380, 0.032906,
-0.868328, -0.495991, 0.000000,
-0.876914, -0.476118, 0.065827,
-0.889480, -0.455785, 0.032928,
-0.904730, -0.424710, 0.032936,
-0.900543, -0.434767, 0.000000,
-0.862905, -0.495666, 0.098538,
-0.879653, -0.465287, 0.098590,
-0.907978, -0.413796, 0.065943,
-0.919041, -0.392778, 0.033014,
-0.895411, -0.434130, 0.098843,
-0.910112, -0.402375, 0.098950,
-0.911107, -0.390534, 0.131782,
-0.923674, -0.370201, 0.098882,
-0.881402, -0.453679, 0.131555,
-0.932091, -0.360716, 0.033021,
-0.928448, -0.371461, 0.000000,
-0.934620, -0.349460, 0.066055,
-0.944047, -0.328151, 0.033041,
-0.954845, -0.295262, 0.033048,
-0.951910, -0.306378, 0.000000,
-0.936012, -0.337778, 0.098929,
-0.865919, -0.472637, 0.163706,
-0.882055, -0.441675, 0.164016,
-0.881675, -0.429071, 0.196334,
-0.897140, -0.410104, 0.164179,
-0.911094, -0.378097, 0.164167,
-0.864787, -0.447345, 0.228090,
-0.880251, -0.415958, 0.228338,
-0.957048, -0.282237, 0.066348,
-0.964909, -0.260463, 0.033305,
-0.947620, -0.303549, 0.099370,
-0.958014, -0.268847, 0.099650,
-0.957732, -0.255190, 0.132768,
-0.967138, -0.233871, 0.099744,
-0.936674, -0.324291, 0.132199,
-0.946748, -0.275921, 0.165941,
-0.956201, -0.241032, 0.166080,
-0.936027, -0.310525, 0.165615,
-0.934170, -0.296273, 0.198871,
-0.920064, -0.316139, 0.231386,
-0.931106, -0.281637, 0.231780,
-0.924096, -0.344643, 0.165129,
-0.953424, -0.226811, 0.198844,
-0.964409, -0.206081, 0.165671,
-0.940847, -0.247162, 0.231772,
-0.949405, -0.212293, 0.231436,
-0.944158, -0.197584, 0.263677,
-0.956736, -0.177234, 0.230749,
-0.926741, -0.267028, 0.264286,
-0.910128, -0.364213, 0.197527,
-0.894658, -0.383186, 0.229687,
-0.907953, -0.349848, 0.230712,
-0.904573, -0.335103, 0.263541,
-0.877768, -0.401461, 0.261444,
-0.973350, -0.226892, 0.033310,
-0.970815, -0.239828, 0.000000,
-0.974649, -0.213589, 0.066626,
-0.980638, -0.192973, 0.033324,
-0.986732, -0.158898, 0.033328,
-0.985074, -0.172130, 0.000000,
-0.974700, -0.200013, 0.099776,
-0.987169, -0.145079, 0.066698,
-0.991690, -0.124250, 0.033376,
-0.981097, -0.165681, 0.099996,
-0.986311, -0.131070, 0.100061,
-0.984157, -0.116967, 0.133245,
-0.990313, -0.096377, 0.099955,
-0.973502, -0.185946, 0.133106,
-0.995397, -0.089833, 0.033381,
-0.994618, -0.103609, 0.000000,
-0.994881, -0.075862, 0.066765,
-0.997915, -0.055231, 0.033392,
-0.999229, -0.020640, 0.033396,
-0.999402, -0.034591, 0.000000,
-0.993068, -0.061793, 0.099983,
-0.971058, -0.171797, 0.165925,
-0.967371, -0.157351, 0.198580,
-0.976527, -0.137228, 0.166025,
-0.980785, -0.102565, 0.165954,
-0.962455, -0.142705, 0.230901,
-0.859517, -0.419912, 0.291383,
-0.874225, -0.387280, 0.292822,
-0.869647, -0.372561, 0.323903,
-0.887823, -0.354076, 0.293939,
-0.900244, -0.320484, 0.294703,
-0.850133, -0.390470, 0.353281,
-0.864050, -0.357403, 0.354516,
-0.894920, -0.305118, 0.325609,
-0.911593, -0.286117, 0.295186,
-0.876818, -0.323632, 0.355602,
-0.888475, -0.289370, 0.356197,
-0.880940, -0.273346, 0.386299,
-0.898967, -0.254807, 0.356275,
-0.857377, -0.341572, 0.385011,
-0.921654, -0.251736, 0.295269,
-0.915445, -0.236060, 0.325939,
-0.930534, -0.216955, 0.295021,
-0.938191, -0.181964, 0.294425,
-0.908143, -0.220102, 0.356133,
-0.836680, -0.359158, 0.413488,
-0.849744, -0.325506, 0.414707,
-0.841094, -0.308973, 0.443956,
-0.861701, -0.291352, 0.415435,
-0.872496, -0.256883, 0.415646,
-0.819222, -0.326128, 0.471716,
-0.831470, -0.292081, 0.472596,
-0.997767, 0.000000, 0.066798,
-0.999229, 0.020640, 0.033396,
-0.994752, -0.020640, 0.100210,
-0.994752, 0.020640, 0.100210,
-0.990206, 0.041156, 0.133410,
-0.993068, 0.061793, 0.099983,
-0.990206, -0.041156, 0.133410,
-0.985728, 0.020642, 0.167078,
-0.984044, 0.061795, 0.166851,
-0.985728, -0.020642, 0.167078,
-0.979686, 0.000000, 0.200538,
-0.972121, -0.020642, 0.233570,
-0.972121, 0.020642, 0.233570,
-0.984044, -0.061795, 0.166851,
-0.976427, 0.082059, 0.199641,
-0.980785, 0.102565, 0.165954,
-0.970505, 0.061424, 0.233124,
-0.967277, 0.102196, 0.232233,
-0.956652, 0.122082, 0.264412,
-0.962455, 0.142705, 0.230901,
-0.963059, 0.040784, 0.266183,
-0.976427, -0.082059, 0.199641,
-0.967277, -0.102196, 0.232233,
-0.970505, -0.061424, 0.233124,
-0.963059, -0.040784, 0.266183,
-0.956652, -0.122082, 0.264412,
-0.948702, 0.102238, 0.299185,
-0.943879, 0.142749, 0.297852,
-0.951932, 0.061453, 0.300080,
-0.939160, 0.082120, 0.333520,
-0.928088, 0.061865, 0.367185,
-0.924826, 0.102647, 0.366279,
-0.953549, 0.020669, 0.300527,
-0.929648, 0.020669, 0.367868,
-0.915314, 0.041197, 0.400628,
-0.942422, 0.000000, 0.334425,
-0.929648, -0.020669, 0.367868,
-0.928088, -0.061865, 0.367185,
-0.915314, -0.041197, 0.400628,
-0.953549, -0.020669, 0.300527,
-0.901180, 0.020667, 0.432953,
-0.899620, 0.061863, 0.432270,
-0.901180, -0.020667, 0.432953,
-0.885516, 0.000000, 0.464610,
-0.868428, -0.020667, 0.495384,
-0.868428, 0.020667, 0.495384,
-0.899620, -0.061863, 0.432270,
-0.951932, -0.061453, 0.300080,
-0.948702, -0.102238, 0.299185,
-0.939160, -0.082120, 0.333520,
-0.924826, -0.102647, 0.366279,
-0.943879, -0.142749, 0.297852,
-0.930240, 0.162122, 0.329196,
-0.938191, 0.181964, 0.294425,
-0.920698, 0.142013, 0.363521,
-0.915124, 0.181249, 0.360134,
-0.898629, 0.199999, 0.390469,
-0.908143, 0.220102, 0.356133,
-0.909626, 0.121761, 0.397184,
-0.887388, 0.179759, 0.424534,
-0.880517, 0.218637, 0.420581,
-0.892852, 0.140501, 0.427873,
-0.874631, 0.159117, 0.457934,
-0.860438, 0.138204, 0.490456,
-0.855081, 0.177481, 0.487173,
-0.896869, 0.101119, 0.430583,
-0.861255, 0.236644, 0.449711,
-0.872496, 0.256883, 0.415646,
-0.848498, 0.216004, 0.483108,
-0.840609, 0.254286, 0.478242,
-0.818713, 0.271440, 0.505993,
-0.831470, 0.292081, 0.472596,
-0.834306, 0.195090, 0.515629,
-0.882736, 0.080590, 0.462906,
-0.867079, 0.059926, 0.494553,
-0.864409, 0.099190, 0.492908,
-0.844754, 0.117537, 0.522087,
-0.849995, 0.039260, 0.525326,
-0.930240, -0.162122, 0.329196,
-0.915124, -0.181249, 0.360134,
-0.920698, -0.142013, 0.363521,
-0.909626, -0.121761, 0.397184,
-0.898629, -0.199999, 0.390469,
-0.892852, -0.140501, 0.427873,
-0.896869, -0.101119, 0.430583,
-0.887388, -0.179759, 0.424534,
-0.874631, -0.159117, 0.457934,
-0.855081, -0.177481, 0.487173,
-0.860438, -0.138204, 0.490456,
-0.880517, -0.218637, 0.420581,
-0.882736, -0.080590, 0.462906,
-0.864409, -0.099190, 0.492908,
-0.867079, -0.059926, 0.494553,
-0.849995, -0.039260, 0.525326,
-0.844754, -0.117537, 0.522087,
-0.861255, -0.236644, 0.449711,
-0.840609, -0.254286, 0.478242,
-0.848498, -0.216004, 0.483108,
-0.834306, -0.195090, 0.515629,
-0.818713, -0.271440, 0.505993,
-0.997915, 0.055231, 0.033392,
-0.999402, 0.034591, 0.000000,
-0.994881, 0.075862, 0.066765,
-0.995397, 0.089833, 0.033381,
-0.991690, 0.124250, 0.033376,
-0.994618, 0.103609, 0.000000,
-0.990313, 0.096377, 0.099955,
-0.987169, 0.145079, 0.066698,
-0.986732, 0.158898, 0.033328,
-0.986311, 0.131070, 0.100061,
-0.981097, 0.165681, 0.099996,
-0.973502, 0.185946, 0.133106,
-0.974700, 0.200013, 0.099776,
-0.984157, 0.116967, 0.133245,
-0.980638, 0.192973, 0.033324,
-0.985074, 0.172130, 0.000000,
-0.974649, 0.213589, 0.066626,
-0.973350, 0.226892, 0.033310,
-0.964909, 0.260463, 0.033305,
-0.970815, 0.239828, 0.000000,
-0.967138, 0.233871, 0.099744,
-0.976527, 0.137228, 0.166025,
-0.967371, 0.157351, 0.198580,
-0.971058, 0.171797, 0.165925,
-0.964409, 0.206081, 0.165671,
-0.956736, 0.177234, 0.230749,
-0.957048, 0.282237, 0.066348,
-0.954845, 0.295262, 0.033048,
-0.958014, 0.268847, 0.099650,
-0.947620, 0.303549, 0.099370,
-0.936674, 0.324291, 0.132199,
-0.936012, 0.337778, 0.098929,
-0.957732, 0.255190, 0.132768,
-0.936027, 0.310525, 0.165615,
-0.924096, 0.344643, 0.165129,
-0.946748, 0.275921, 0.165941,
-0.934170, 0.296273, 0.198871,
-0.931106, 0.281637, 0.231780,
-0.920064, 0.316139, 0.231386,
-0.956201, 0.241032, 0.166080,
-0.910128, 0.364213, 0.197527,
-0.911094, 0.378097, 0.164167,
-0.907953, 0.349848, 0.230712,
-0.894658, 0.383186, 0.229687,
-0.877768, 0.401461, 0.261444,
-0.880251, 0.415958, 0.228338,
-0.904573, 0.335103, 0.263541,
-0.953424, 0.226811, 0.198844,
-0.949405, 0.212293, 0.231436,
-0.940847, 0.247162, 0.231772,
-0.926741, 0.267028, 0.264286,
-0.944158, 0.197584, 0.263677,
-0.944047, 0.328151, 0.033041,
-0.951910, 0.306378, 0.000000,
-0.934620, 0.349460, 0.066055,
-0.932091, 0.360716, 0.033021,
-0.919041, 0.392778, 0.033014,
-0.928448, 0.371461, 0.000000,
-0.923674, 0.370201, 0.098882,
-0.907978, 0.413796, 0.065943,
-0.904730, 0.424710, 0.032936,
-0.910112, 0.402375, 0.098950,
-0.895411, 0.434130, 0.098843,
-0.881402, 0.453679, 0.131555,
-0.879653, 0.465287, 0.098590,
-0.911107, 0.390534, 0.131782,
-0.889480, 0.455785, 0.032928,
-0.900543, 0.434767, 0.000000,
-0.876914, 0.476118, 0.065827,
-0.873128, 0.486380, 0.032906,
-0.855762, 0.516324, 0.032898,
-0.868328, 0.495991, 0.000000,
-0.862905, 0.495666, 0.098538,
-0.897140, 0.410104, 0.164179,
-0.881675, 0.429071, 0.196334,
-0.882055, 0.441675, 0.164016,
-0.865919, 0.472637, 0.163706,
-0.864787, 0.447345, 0.228090,
-0.930534, 0.216955, 0.295021,
-0.915445, 0.236060, 0.325939,
-0.921654, 0.251736, 0.295269,
-0.911593, 0.286117, 0.295186,
-0.898967, 0.254807, 0.356275,
-0.894920, 0.305118, 0.325609,
-0.900244, 0.320484, 0.294703,
-0.888475, 0.289370, 0.356197,
-0.876818, 0.323632, 0.355602,
-0.857377, 0.341572, 0.385011,
-0.864050, 0.357403, 0.354516,
-0.880940, 0.273346, 0.386299,
-0.887823, 0.354076, 0.293939,
-0.869647, 0.372561, 0.323903,
-0.874225, 0.387280, 0.292822,
-0.859517, 0.419912, 0.291383,
-0.850133, 0.390470, 0.353281,
-0.861701, 0.291352, 0.415435,
-0.841094, 0.308973, 0.443956,
-0.849744, 0.325506, 0.414707,
-0.836680, 0.359158, 0.413488,
-0.819222, 0.326128, 0.471716,
-0.797844, -0.291536, 0.527686,
-0.806959, -0.253742, 0.533322,
-0.794224, -0.235673, 0.560060,
-0.814815, -0.215489, 0.538182,
-0.821376, -0.176970, 0.542239,
-0.772646, -0.255549, 0.581130,
-0.780587, -0.217322, 0.586051,
-0.807470, -0.158301, 0.568272,
-0.826677, -0.137840, 0.545532,
-0.787154, -0.178553, 0.590346,
-0.792524, -0.139437, 0.593687,
-0.776630, -0.120472, 0.618331,
-0.796672, -0.100171, 0.596053,
-0.765845, -0.198505, 0.611619,
-0.830634, -0.098832, 0.547978,
-0.815477, -0.079939, 0.573242,
-0.833286, -0.059604, 0.549620,
-0.834622, -0.020351, 0.550447,
-0.799376, -0.060948, 0.597732,
-0.743751, -0.218339, 0.631794,
-0.750382, -0.179588, 0.636140,
-0.733998, -0.160388, 0.659941,
-0.755818, -0.140487, 0.639533,
-0.760032, -0.101232, 0.641952,
-0.711295, -0.180084, 0.679433,
-0.716793, -0.140998, 0.682882,
-0.818200, 0.000000, 0.574934,
-0.834622, 0.020351, 0.550447,
-0.800670, -0.020351, 0.598760,
-0.800670, 0.020351, 0.598760,
-0.782140, 0.040604, 0.621778,
-0.799376, 0.060948, 0.597732,
-0.782140, -0.040604, 0.621778,
-0.763641, 0.020352, 0.645320,
-0.762348, 0.060949, 0.644291,
-0.763641, -0.020352, 0.645320,
-0.744112, 0.000000, 0.668055,
-0.723674, -0.020352, 0.689842,
-0.723674, 0.020352, 0.689842,
-0.762348, -0.060949, 0.644291,
-0.741796, 0.081001, 0.665716,
-0.760032, 0.101232, 0.641952,
-0.722524, 0.060664, 0.688679,
-0.720226, 0.100949, 0.686355,
-0.697769, 0.120696, 0.706082,
-0.716793, 0.140998, 0.682882,
-0.702335, 0.040319, 0.710703,
-0.741796, -0.081001, 0.665716,
-0.720226, -0.100949, 0.686355,
-0.722524, -0.060664, 0.688679,
-0.702335, -0.040319, 0.710703,
-0.697769, -0.120696, 0.706082,
-0.833286, 0.059604, 0.549620,
-0.815477, 0.079939, 0.573242,
-0.830634, 0.098832, 0.547978,
-0.826677, 0.137840, 0.545532,
-0.796672, 0.100171, 0.596053,
-0.807470, 0.158301, 0.568272,
-0.821376, 0.176970, 0.542239,
-0.792524, 0.139437, 0.593687,
-0.787154, 0.178553, 0.590346,
-0.765845, 0.198505, 0.611619,
-0.780587, 0.217322, 0.586051,
-0.776630, 0.120472, 0.618331,
-0.814815, 0.215489, 0.538182,
-0.794224, 0.235673, 0.560060,
-0.806959, 0.253742, 0.533322,
-0.797844, 0.291536, 0.527686,
-0.772646, 0.255549, 0.581130,
-0.755818, 0.140487, 0.639533,
-0.733998, 0.160388, 0.659941,
-0.750382, 0.179588, 0.636140,
-0.743751, 0.218339, 0.631794,
-0.711295, 0.180084, 0.679433,
-0.675435, -0.140967, 0.723820,
-0.678867, -0.100920, 0.727293,
-0.659120, -0.080959, 0.747667,
-0.681162, -0.060644, 0.729617,
-0.682311, -0.020333, 0.730779,
-0.636343, -0.101176, 0.764743,
-0.638652, -0.060901, 0.767082,
-0.661430, 0.000000, 0.750007,
-0.682311, 0.020333, 0.730779,
-0.639671, -0.020333, 0.768380,
-0.639671, 0.020333, 0.768380,
-0.617167, 0.040577, 0.785785,
-0.638652, 0.060901, 0.767082,
-0.617167, -0.040577, 0.785785,
-0.681162, 0.060644, 0.729617,
-0.659120, 0.080959, 0.747667,
-0.678867, 0.100920, 0.727293,
-0.675435, 0.140967, 0.723820,
-0.636343, 0.101176, 0.764743,
-0.594204, -0.060900, 0.802005,
-0.595224, -0.020332, 0.803303,
-0.572529, 0.000000, 0.819885,
-0.595224, 0.020332, 0.803303,
-0.594204, 0.060900, 0.802005,
-0.549222, -0.020332, 0.835429,
-0.549222, 0.020332, 0.835429,
-0.873128, -0.486380, -0.032906,
-0.855762, -0.516324, -0.032898,
-0.889480, -0.455785, -0.032928,
-0.876914, -0.476118, -0.065827,
-0.879653, -0.465287, -0.098590,
-0.862905, -0.495666, -0.098538,
-0.904730, -0.424710, -0.032936,
-0.895411, -0.434130, -0.098843,
-0.881402, -0.453679, -0.131555,
-0.907978, -0.413796, -0.065943,
-0.910112, -0.402375, -0.098950,
-0.923674, -0.370201, -0.098882,
-0.911107, -0.390534, -0.131782,
-0.919041, -0.392778, -0.033014,
-0.882055, -0.441675, -0.164016,
-0.865919, -0.472637, -0.163706,
-0.897140, -0.410104, -0.164179,
-0.881675, -0.429071, -0.196334,
-0.880251, -0.415958, -0.228338,
-0.864787, -0.447345, -0.228090,
-0.911094, -0.378097, -0.164167,
-0.932091, -0.360716, -0.033021,
-0.944047, -0.328151, -0.033041,
-0.934620, -0.349460, -0.066055,
-0.936012, -0.337778, -0.098929,
-0.954845, -0.295262, -0.033048,
-0.894658, -0.383186, -0.229687,
-0.877768, -0.401461, -0.261444,
-0.910128, -0.364213, -0.197527,
-0.907953, -0.349848, -0.230712,
-0.920064, -0.316139, -0.231386,
-0.904573, -0.335103, -0.263541,
-0.924096, -0.344643, -0.165129,
-0.934170, -0.296273, -0.198871,
-0.931106, -0.281637, -0.231780,
-0.936027, -0.310525, -0.165615,
-0.946748, -0.275921, -0.165941,
-0.957732, -0.255190, -0.132768,
-0.956201, -0.241032, -0.166080,
-0.936674, -0.324291, -0.132199,
-0.940847, -0.247162, -0.231772,
-0.926741, -0.267028, -0.264286,
-0.953424, -0.226811, -0.198844,
-0.949405, -0.212293, -0.231436,
-0.956736, -0.177234, -0.230749,
-0.944158, -0.197584, -0.263677,
-0.964409, -0.206081, -0.165671,
-0.947620, -0.303549, -0.099370,
-0.957048, -0.282237, -0.066348,
-0.958014, -0.268847, -0.099650,
-0.967138, -0.233871, -0.099744,
-0.964909, -0.260463, -0.033305,
-0.874225, -0.387280, -0.292822,
-0.859517, -0.419912, -0.291383,
-0.887823, -0.354076, -0.293939,
-0.869647, -0.372561, -0.323903,
-0.864050, -0.357403, -0.354516,
-0.850133, -0.390470, -0.353281,
-0.900244, -0.320484, -0.294703,
-0.876818, -0.323632, -0.355602,
-0.857377, -0.341572, -0.385011,
-0.894920, -0.305118, -0.325609,
-0.888475, -0.289370, -0.356197,
-0.898967, -0.254807, -0.356275,
-0.880940, -0.273346, -0.386299,
-0.911593, -0.286117, -0.295186,
-0.849744, -0.325506, -0.414707,
-0.836680, -0.359158, -0.413488,
-0.861701, -0.291352, -0.415435,
-0.841094, -0.308973, -0.443956,
-0.831470, -0.292081, -0.472596,
-0.819222, -0.326128, -0.471716,
-0.872496, -0.256883, -0.415646,
-0.921654, -0.251736, -0.295269,
-0.930534, -0.216955, -0.295021,
-0.915445, -0.236060, -0.325939,
-0.908143, -0.220102, -0.356133,
-0.938191, -0.181964, -0.294425,
-0.973350, -0.226892, -0.033310,
-0.980638, -0.192973, -0.033324,
-0.974649, -0.213589, -0.066626,
-0.974700, -0.200013, -0.099776,
-0.986732, -0.158898, -0.033328,
-0.981097, -0.165681, -0.099996,
-0.973502, -0.185946, -0.133106,
-0.987169, -0.145079, -0.066698,
-0.986311, -0.131070, -0.100061,
-0.990313, -0.096377, -0.099955,
-0.984157, -0.116967, -0.133245,
-0.991690, -0.124250, -0.033376,
-0.971058, -0.171797, -0.165925,
-0.976527, -0.137228, -0.166025,
-0.967371, -0.157351, -0.198580,
-0.962455, -0.142705, -0.230901,
-0.980785, -0.102565, -0.165954,
-0.995397, -0.089833, -0.033381,
-0.997915, -0.055231, -0.033392,
-0.994881, -0.075862, -0.066765,
-0.993068, -0.061793, -0.099983,
-0.999229, -0.020640, -0.033396,
-0.840609, -0.254286, -0.478242,
-0.818713, -0.271440, -0.505993,
-0.861255, -0.236644, -0.449711,
-0.848498, -0.216004, -0.483108,
-0.855081, -0.177481, -0.487173,
-0.834306, -0.195090, -0.515629,
-0.880517, -0.218637, -0.420581,
-0.874631, -0.159117, -0.457934,
-0.860438, -0.138204, -0.490456,
-0.887388, -0.179759, -0.424534,
-0.892852, -0.140501, -0.427873,
-0.909626, -0.121761, -0.397184,
-0.896869, -0.101119, -0.430583,
-0.898629, -0.199999, -0.390469,
-0.864409, -0.099190, -0.492908,
-0.844754, -0.117537, -0.522087,
-0.882736, -0.080590, -0.462906,
-0.867079, -0.059926, -0.494553,
-0.868428, -0.020667, -0.495384,
-0.849995, -0.039260, -0.525326,
-0.899620, -0.061863, -0.432270,
-0.915124, -0.181249, -0.360134,
-0.930240, -0.162122, -0.329196,
-0.920698, -0.142013, -0.363521,
-0.924826, -0.102647, -0.366279,
-0.943879, -0.142749, -0.297852,
-0.885516, 0.000000, -0.464610,
-0.868428, 0.020667, -0.495384,
-0.901180, -0.020667, -0.432953,
-0.901180, 0.020667, -0.432953,
-0.915314, 0.041197, -0.400628,
-0.899620, 0.061863, -0.432270,
-0.915314, -0.041197, -0.400628,
-0.929648, 0.020669, -0.367868,
-0.928088, 0.061865, -0.367185,
-0.929648, -0.020669, -0.367868,
-0.942422, 0.000000, -0.334425,
-0.953549, -0.020669, -0.300527,
-0.953549, 0.020669, -0.300527,
-0.928088, -0.061865, -0.367185,
-0.939160, 0.082120, -0.333520,
-0.924826, 0.102647, -0.366279,
-0.951932, 0.061453, -0.300080,
-0.948702, 0.102238, -0.299185,
-0.956652, 0.122082, -0.264412,
-0.943879, 0.142749, -0.297852,
-0.963059, 0.040784, -0.266183,
-0.939160, -0.082120, -0.333520,
-0.948702, -0.102238, -0.299185,
-0.951932, -0.061453, -0.300080,
-0.963059, -0.040784, -0.266183,
-0.956652, -0.122082, -0.264412,
-0.867079, 0.059926, -0.494553,
-0.849995, 0.039260, -0.525326,
-0.882736, 0.080590, -0.462906,
-0.864409, 0.099190, -0.492908,
-0.860438, 0.138204, -0.490456,
-0.844754, 0.117537, -0.522087,
-0.896869, 0.101119, -0.430583,
-0.874631, 0.159117, -0.457934,
-0.855081, 0.177481, -0.487173,
-0.892852, 0.140501, -0.427873,
-0.887388, 0.179759, -0.424534,
-0.898629, 0.199999, -0.390469,
-0.880517, 0.218637, -0.420581,
-0.909626, 0.121761, -0.397184,
-0.848498, 0.216004, -0.483108,
-0.834306, 0.195090, -0.515629,
-0.861255, 0.236644, -0.449711,
-0.840609, 0.254286, -0.478242,
-0.831470, 0.292081, -0.472596,
-0.818713, 0.271440, -0.505993,
-0.872496, 0.256883, -0.415646,
-0.920698, 0.142013, -0.363521,
-0.930240, 0.162122, -0.329196,
-0.915124, 0.181249, -0.360134,
-0.908143, 0.220102, -0.356133,
-0.938191, 0.181964, -0.294425,
-0.967277, -0.102196, -0.232233,
-0.976427, -0.082059, -0.199641,
-0.970505, -0.061424, -0.233124,
-0.972121, -0.020642, -0.233570,
-0.984044, -0.061795, -0.166851,
-0.979686, 0.000000, -0.200538,
-0.972121, 0.020642, -0.233570,
-0.985728, -0.020642, -0.167078,
-0.985728, 0.020642, -0.167078,
-0.990206, 0.041156, -0.133410,
-0.984044, 0.061795, -0.166851,
-0.990206, -0.041156, -0.133410,
-0.970505, 0.061424, -0.233124,
-0.976427, 0.082059, -0.199641,
-0.967277, 0.102196, -0.232233,
-0.962455, 0.142705, -0.230901,
-0.980785, 0.102565, -0.165954,
-0.994752, -0.020640, -0.100210,
-0.997767, 0.000000, -0.066798,
-0.994752, 0.020640, -0.100210,
-0.993068, 0.061793, -0.099983,
-0.999229, 0.020640, -0.033396,
-0.806959, -0.253742, -0.533322,
-0.797844, -0.291536, -0.527686,
-0.814815, -0.215489, -0.538182,
-0.794224, -0.235673, -0.560060,
-0.780587, -0.217322, -0.586051,
-0.772646, -0.255549, -0.581130,
-0.821376, -0.176970, -0.542239,
-0.787154, -0.178553, -0.590346,
-0.765845, -0.198505, -0.611619,
-0.807470, -0.158301, -0.568272,
-0.792524, -0.139437, -0.593687,
-0.796672, -0.100171, -0.596053,
-0.776630, -0.120472, -0.618331,
-0.826677, -0.137840, -0.545532,
-0.750382, -0.179588, -0.636140,
-0.743751, -0.218339, -0.631794,
-0.755818, -0.140487, -0.639533,
-0.733998, -0.160388, -0.659941,
-0.716793, -0.140998, -0.682882,
-0.711295, -0.180084, -0.679433,
-0.760032, -0.101232, -0.641952,
-0.830634, -0.098832, -0.547978,
-0.833286, -0.059604, -0.549620,
-0.815477, -0.079939, -0.573242,
-0.799376, -0.060948, -0.597732,
-0.834622, -0.020351, -0.550447,
-0.720226, -0.100949, -0.686355,
-0.697769, -0.120696, -0.706082,
-0.741796, -0.081001, -0.665716,
-0.722524, -0.060664, -0.688679,
-0.723674, -0.020352, -0.689842,
-0.702335, -0.040319, -0.710703,
-0.762348, -0.060949, -0.644291,
-0.744112, 0.000000, -0.668055,
-0.723674, 0.020352, -0.689842,
-0.763641, -0.020352, -0.645320,
-0.763641, 0.020352, -0.645320,
-0.782140, 0.040604, -0.621778,
-0.762348, 0.060949, -0.644291,
-0.782140, -0.040604, -0.621778,
-0.722524, 0.060664, -0.688679,
-0.702335, 0.040319, -0.710703,
-0.741796, 0.081001, -0.665716,
-0.720226, 0.100949, -0.686355,
-0.716793, 0.140998, -0.682882,
-0.697769, 0.120696, -0.706082,
-0.760032, 0.101232, -0.641952,
-0.800670, -0.020351, -0.598760,
-0.818200, 0.000000, -0.574934,
-0.800670, 0.020351, -0.598760,
-0.799376, 0.060948, -0.597732,
-0.834622, 0.020351, -0.550447,
-0.678867, -0.100920, -0.727293,
-0.675435, -0.140967, -0.723820,
-0.681162, -0.060644, -0.729617,
-0.659120, -0.080959, -0.747667,
-0.638652, -0.060901, -0.767082,
-0.636343, -0.101176, -0.764743,
-0.682311, -0.020333, -0.730779,
-0.639671, -0.020333, -0.768380,
-0.617167, -0.040577, -0.785785,
-0.661430, 0.000000, -0.750007,
-0.639671, 0.020333, -0.768380,
-0.638652, 0.060901, -0.767082,
-0.617167, 0.040577, -0.785785,
-0.682311, 0.020333, -0.730779,
-0.595224, -0.020332, -0.803303,
-0.594204, -0.060900, -0.802005,
-0.595224, 0.020332, -0.803303,
-0.572529, 0.000000, -0.819885,
-0.549222, 0.020332, -0.835429,
-0.549222, -0.020332, -0.835429,
-0.594204, 0.060900, -0.802005,
-0.681162, 0.060644, -0.729617,
-0.678867, 0.100920, -0.727293,
-0.659120, 0.080959, -0.747667,
-0.636343, 0.101176, -0.764743,
-0.675435, 0.140967, -0.723820,
-0.833286, 0.059604, -0.549620,
-0.830634, 0.098832, -0.547978,
-0.815477, 0.079939, -0.573242,
-0.796672, 0.100171, -0.596053,
-0.826677, 0.137840, -0.545532,
-0.792524, 0.139437, -0.593687,
-0.776630, 0.120472, -0.618331,
-0.807470, 0.158301, -0.568272,
-0.787154, 0.178553, -0.590346,
-0.780587, 0.217322, -0.586051,
-0.765845, 0.198505, -0.611619,
-0.821376, 0.176970, -0.542239,
-0.755818, 0.140487, -0.639533,
-0.750382, 0.179588, -0.636140,
-0.733998, 0.160388, -0.659941,
-0.711295, 0.180084, -0.679433,
-0.743751, 0.218339, -0.631794,
-0.814815, 0.215489, -0.538182,
-0.806959, 0.253742, -0.533322,
-0.794224, 0.235673, -0.560060,
-0.772646, 0.255549, -0.581130,
-0.797844, 0.291536, -0.527686,
-0.997915, 0.055231, -0.033392,
-0.995397, 0.089833, -0.033381,
-0.994881, 0.075862, -0.066765,
-0.990313, 0.096377, -0.099955,
-0.991690, 0.124250, -0.033376,
-0.986311, 0.131070, -0.100061,
-0.984157, 0.116967, -0.133245,
-0.987169, 0.145079, -0.066698,
-0.981097, 0.165681, -0.099996,
-0.974700, 0.200013, -0.099776,
-0.973502, 0.185946, -0.133106,
-0.986732, 0.158898, -0.033328,
-0.976527, 0.137228, -0.166025,
-0.971058, 0.171797, -0.165925,
-0.967371, 0.157351, -0.198580,
-0.956736, 0.177234, -0.230749,
-0.964409, 0.206081, -0.165671,
-0.980638, 0.192973, -0.033324,
-0.973350, 0.226892, -0.033310,
-0.974649, 0.213589, -0.066626,
-0.967138, 0.233871, -0.099744,
-0.964909, 0.260463, -0.033305,
-0.949405, 0.212293, -0.231436,
-0.944158, 0.197584, -0.263677,
-0.953424, 0.226811, -0.198844,
-0.940847, 0.247162, -0.231772,
-0.931106, 0.281637, -0.231780,
-0.926741, 0.267028, -0.264286,
-0.956201, 0.241032, -0.166080,
-0.934170, 0.296273, -0.198871,
-0.920064, 0.316139, -0.231386,
-0.946748, 0.275921, -0.165941,
-0.936027, 0.310525, -0.165615,
-0.936674, 0.324291, -0.132199,
-0.924096, 0.344643, -0.165129,
-0.957732, 0.255190, -0.132768,
-0.907953, 0.349848, -0.230712,
-0.904573, 0.335103, -0.263541,
-0.910128, 0.364213, -0.197527,
-0.894658, 0.383186, -0.229687,
-0.880251, 0.415958, -0.228338,
-0.877768, 0.401461, -0.261444,
-0.911094, 0.378097, -0.164167,
-0.958014, 0.268847, -0.099650,
-0.957048, 0.282237, -0.066348,
-0.947620, 0.303549, -0.099370,
-0.936012, 0.337778, -0.098929,
-0.954845, 0.295262, -0.033048,
-0.930534, 0.216955, -0.295021,
-0.921654, 0.251736, -0.295269,
-0.915445, 0.236060, -0.325939,
-0.898967, 0.254807, -0.356275,
-0.911593, 0.286117, -0.295186,
-0.888475, 0.289370, -0.356197,
-0.880940, 0.273346, -0.386299,
-0.894920, 0.305118, -0.325609,
-0.876818, 0.323632, -0.355602,
-0.864050, 0.357403, -0.354516,
-0.857377, 0.341572, -0.385011,
-0.900244, 0.320484, -0.294703,
-0.861701, 0.291352, -0.415435,
-0.849744, 0.325506, -0.414707,
-0.841094, 0.308973, -0.443956,
-0.819222, 0.326128, -0.471716,
-0.836680, 0.359158, -0.413488,
-0.887823, 0.354076, -0.293939,
-0.874225, 0.387280, -0.292822,
-0.869647, 0.372561, -0.323903,
-0.850133, 0.390470, -0.353281,
-0.859517, 0.419912, -0.291383,
-0.944047, 0.328151, -0.033041,
-0.932091, 0.360716, -0.033021,
-0.934620, 0.349460, -0.066055,
-0.923674, 0.370201, -0.098882,
-0.919041, 0.392778, -0.033014,
-0.910112, 0.402375, -0.098950,
-0.911107, 0.390534, -0.131782,
-0.907978, 0.413796, -0.065943,
-0.895411, 0.434130, -0.098843,
-0.879653, 0.465287, -0.098590,
-0.881402, 0.453679, -0.131555,
-0.904730, 0.424710, -0.032936,
-0.897140, 0.410104, -0.164179,
-0.882055, 0.441675, -0.164016,
-0.881675, 0.429071, -0.196334,
-0.864787, 0.447345, -0.228090,
-0.865919, 0.472637, -0.163706,
-0.889480, 0.455785, -0.032928,
-0.873128, 0.486380, -0.032906,
-0.876914, 0.476118, -0.065827,
-0.862905, 0.495666, -0.098538,
-0.855762, 0.516324, -0.032898,
0.032898, 0.855762, 0.516324,
0.032906, 0.873128, 0.486380,
0.000000, 0.868328, 0.495991,
0.065827, 0.876914, 0.476118,
0.032928, 0.889480, 0.455785,
0.032936, 0.904730, 0.424710,
0.000000, 0.900543, 0.434767,
0.098538, 0.862905, 0.495666,
0.098590, 0.879653, 0.465287,
0.065943, 0.907978, 0.413796,
0.033014, 0.919041, 0.392778,
0.098843, 0.895411, 0.434130,
0.098950, 0.910112, 0.402375,
0.131782, 0.911107, 0.390534,
0.098882, 0.923674, 0.370201,
0.131555, 0.881402, 0.453679,
0.033021, 0.932091, 0.360716,
0.000000, 0.928448, 0.371461,
0.066055, 0.934620, 0.349460,
0.033041, 0.944047, 0.328151,
0.033048, 0.954845, 0.295262,
0.000000, 0.951910, 0.306378,
0.098929, 0.936012, 0.337778,
0.163706, 0.865919, 0.472637,
0.164016, 0.882055, 0.441675,
0.196334, 0.881675, 0.429071,
0.164179, 0.897140, 0.410104,
0.164167, 0.911094, 0.378097,
0.228090, 0.864787, 0.447345,
0.228338, 0.880251, 0.415958,
0.066348, 0.957048, 0.282237,
0.033305, 0.964909, 0.260463,
0.099370, 0.947620, 0.303549,
0.099650, 0.958014, 0.268847,
0.132768, 0.957732, 0.255190,
0.099744, 0.967138, 0.233871,
0.132199, 0.936674, 0.324291,
0.165941, 0.946748, 0.275921,
0.166080, 0.956201, 0.241032,
0.165615, 0.936027, 0.310525,
0.198871, 0.934170, 0.296273,
0.231386, 0.920064, 0.316139,
0.231780, 0.931106, 0.281637,
0.165129, 0.924096, 0.344643,
0.198844, 0.953424, 0.226811,
0.165671, 0.964409, 0.206081,
0.231772, 0.940847, 0.247162,
0.231436, 0.949405, 0.212293,
0.263677, 0.944158, 0.197584,
0.230749, 0.956736, 0.177234,
0.264286, 0.926741, 0.267028,
0.197527, 0.910128, 0.364213,
0.229687, 0.894658, 0.383186,
0.230712, 0.907953, 0.349848,
0.263541, 0.904573, 0.335103,
0.261444, 0.877768, 0.401461,
0.033310, 0.973350, 0.226892,
0.000000, 0.970815, 0.239828,
0.066626, 0.974649, 0.213589,
0.033324, 0.980638, 0.192973,
0.033328, 0.986732, 0.158898,
0.000000, 0.985074, 0.172130,
0.099776, 0.974700, 0.200013,
0.066698, 0.987169, 0.145079,
0.033376, 0.991690, 0.124250,
0.099996, 0.981097, 0.165681,
0.100061, 0.986311, 0.131070,
0.133245, 0.984157, 0.116967,
0.099955, 0.990313, 0.096377,
0.133106, 0.973502, 0.185946,
0.033381, 0.995397, 0.089833,
0.000000, 0.994618, 0.103609,
0.066765, 0.994881, 0.075862,
0.033392, 0.997915, 0.055231,
0.033396, 0.999229, 0.020640,
0.000000, 0.999402, 0.034591,
0.099983, 0.993068, 0.061793,
0.165925, 0.971058, 0.171797,
0.198580, 0.967371, 0.157351,
0.166025, 0.976527, 0.137228,
0.165954, 0.980785, 0.102565,
0.230901, 0.962455, 0.142705,
0.291383, 0.859517, 0.419912,
0.292822, 0.874225, 0.387280,
0.323903, 0.869647, 0.372561,
0.293939, 0.887823, 0.354076,
0.294703, 0.900244, 0.320484,
0.353281, 0.850133, 0.390470,
0.354516, 0.864050, 0.357403,
0.325609, 0.894920, 0.305118,
0.295186, 0.911593, 0.286117,
0.355602, 0.876818, 0.323632,
0.356197, 0.888475, 0.289370,
0.386299, 0.880940, 0.273346,
0.356275, 0.898967, 0.254807,
0.385011, 0.857377, 0.341572,
0.295269, 0.921654, 0.251736,
0.325939, 0.915445, 0.236060,
0.295021, 0.930534, 0.216955,
0.294425, 0.938191, 0.181964,
0.356133, 0.908143, 0.220102,
0.413488, 0.836680, 0.359158,
0.414707, 0.849744, 0.325506,
0.443956, 0.841094, 0.308973,
0.415435, 0.861701, 0.291352,
0.415646, 0.872496, 0.256883,
0.471716, 0.819222, 0.326128,
0.472596, 0.831470, 0.292081,
0.066798, 0.997767, 0.000000,
0.033396, 0.999229, -0.020640,
0.100210, 0.994752, 0.020640,
0.100210, 0.994752, -0.020640,
0.133410, 0.990206, -0.041156,
0.099983, 0.993068, -0.061793,
0.133410, 0.990206, 0.041156,
0.167078, 0.985728, -0.020642,
0.166851, 0.984044, -0.061795,
0.167078, 0.985728, 0.020642,
0.200538, 0.979686, 0.000000,
0.233570, 0.972121, 0.020642,
0.233570, 0.972121, -0.020642,
0.166851, 0.984044, 0.061795,
0.199641, 0.976427, -0.082059,
0.165954, 0.980785, -0.102565,
0.233124, 0.970505, -0.061424,
0.232233, 0.967277, -0.102196,
0.264412, 0.956652, -0.122082,
0.230901, 0.962455, -0.142705,
0.266183, 0.963059, -0.040784,
0.199641, 0.976427, 0.082059,
0.232233, 0.967277, 0.102196,
0.233124, 0.970505, 0.061424,
0.266183, 0.963059, 0.040784,
0.264412, 0.956652, 0.122082,
0.299185, 0.948702, -0.102238,
0.297852, 0.943879, -0.142749,
0.300080, 0.951932, -0.061453,
0.333520, 0.939160, -0.082120,
0.367185, 0.928088, -0.061865,
0.366279, 0.924826, -0.102647,
0.300527, 0.953549, -0.020669,
0.367868, 0.929648, -0.020669,
0.400628, 0.915314, -0.041197,
0.334425, 0.942422, 0.000000,
0.367868, 0.929648, 0.020669,
0.367185, 0.928088, 0.061865,
0.400628, 0.915314, 0.041197,
0.300527, 0.953549, 0.020669,
0.432953, 0.901180, -0.020667,
0.432270, 0.899620, -0.061863,
0.432953, 0.901180, 0.020667,
0.464610, 0.885516, 0.000000,
0.495384, 0.868428, 0.020667,
0.495384, 0.868428, -0.020667,
0.432270, 0.899620, 0.061863,
0.300080, 0.951932, 0.061453,
0.299185, 0.948702, 0.102238,
0.333520, 0.939160, 0.082120,
0.366279, 0.924826, 0.102647,
0.297852, 0.943879, 0.142749,
0.329196, 0.930240, -0.162122,
0.294425, 0.938191, -0.181964,
0.363521, 0.920698, -0.142013,
0.360134, 0.915124, -0.181249,
0.390469, 0.898629, -0.199999,
0.356133, 0.908143, -0.220102,
0.397184, 0.909626, -0.121761,
0.424534, 0.887388, -0.179759,
0.420581, 0.880517, -0.218637,
0.427873, 0.892852, -0.140501,
0.457934, 0.874631, -0.159117,
0.490456, 0.860438, -0.138204,
0.487173, 0.855081, -0.177481,
0.430583, 0.896869, -0.101119,
0.449711, 0.861255, -0.236644,
0.415646, 0.872496, -0.256883,
0.483108, 0.848498, -0.216004,
0.478242, 0.840609, -0.254286,
0.505993, 0.818713, -0.271440,
0.472596, 0.831470, -0.292081,
0.515629, 0.834306, -0.195090,
0.462906, 0.882736, -0.080590,
0.494553, 0.867079, -0.059926,
0.492908, 0.864409, -0.099190,
0.522087, 0.844754, -0.117537,
0.525326, 0.849995, -0.039260,
0.329196, 0.930240, 0.162122,
0.360134, 0.915124, 0.181249,
0.363521, 0.920698, 0.142013,
0.397184, 0.909626, 0.121761,
0.390469, 0.898629, 0.199999,
0.427873, 0.892852, 0.140501,
0.430583, 0.896869, 0.101119,
0.424534, 0.887388, 0.179759,
0.457934, 0.874631, 0.159117,
0.487173, 0.855081, 0.177481,
0.490456, 0.860438, 0.138204,
0.420581, 0.880517, 0.218637,
0.462906, 0.882736, 0.080590,
0.492908, 0.864409, 0.099190,
0.494553, 0.867079, 0.059926,
0.525326, 0.849995, 0.039260,
0.522087, 0.844754, 0.117537,
0.449711, 0.861255, 0.236644,
0.478242, 0.840609, 0.254286,
0.483108, 0.848498, 0.216004,
0.515629, 0.834306, 0.195090,
0.505993, 0.818713, 0.271440,
0.033392, 0.997915, -0.055231,
0.000000, 0.999402, -0.034591,
0.066765, 0.994881, -0.075862,
0.033381, 0.995397, -0.089833,
0.033376, 0.991690, -0.124250,
0.000000, 0.994618, -0.103609,
0.099955, 0.990313, -0.096377,
0.066698, 0.987169, -0.145079,
0.033328, 0.986732, -0.158898,
0.100061, 0.986311, -0.131070,
0.099996, 0.981097, -0.165681,
0.133106, 0.973502, -0.185946,
0.099776, 0.974700, -0.200013,
0.133245, 0.984157, -0.116967,
0.033324, 0.980638, -0.192973,
0.000000, 0.985074, -0.172130,
0.066626, 0.974649, -0.213589,
0.033310, 0.973350, -0.226892,
0.033305, 0.964909, -0.260463,
0.000000, 0.970815, -0.239828,
0.099744, 0.967138, -0.233871,
0.166025, 0.976527, -0.137228,
0.198580, 0.967371, -0.157351,
0.165925, 0.971058, -0.171797,
0.165671, 0.964409, -0.206081,
0.230749, 0.956736, -0.177234,
0.066348, 0.957048, -0.282237,
0.033048, 0.954845, -0.295262,
0.099650, 0.958014, -0.268847,
0.099370, 0.947620, -0.303549,
0.132199, 0.936674, -0.324291,
0.098929, 0.936012, -0.337778,
0.132768, 0.957732, -0.255190,
0.165615, 0.936027, -0.310525,
0.165129, 0.924096, -0.344643,
0.165941, 0.946748, -0.275921,
0.198871, 0.934170, -0.296273,
0.231780, 0.931106, -0.281637,
0.231386, 0.920064, -0.316139,
0.166080, 0.956201, -0.241032,
0.197527, 0.910128, -0.364213,
0.164167, 0.911094, -0.378097,
0.230712, 0.907953, -0.349848,
0.229687, 0.894658, -0.383186,
0.261444, 0.877768, -0.401461,
0.228338, 0.880251, -0.415958,
0.263541, 0.904573, -0.335103,
0.198844, 0.953424, -0.226811,
0.231436, 0.949405, -0.212293,
0.231772, 0.940847, -0.247162,
0.264286, 0.926741, -0.267028,
0.263677, 0.944158, -0.197584,
0.033041, 0.944047, -0.328151,
0.000000, 0.951910, -0.306378,
0.066055, 0.934620, -0.349460,
0.033021, 0.932091, -0.360716,
0.033014, 0.919041, -0.392778,
0.000000, 0.928448, -0.371461,
0.098882, 0.923674, -0.370201,
0.065943, 0.907978, -0.413796,
0.032936, 0.904730, -0.424710,
0.098950, 0.910112, -0.402375,
0.098843, 0.895411, -0.434130,
0.131555, 0.881402, -0.453679,
0.098590, 0.879653, -0.465287,
0.131782, 0.911107, -0.390534,
0.032928, 0.889480, -0.455785,
0.000000, 0.900543, -0.434767,
0.065827, 0.876914, -0.476118,
0.032906, 0.873128, -0.486380,
0.032898, 0.855762, -0.516324,
0.000000, 0.868328, -0.495991,
0.098538, 0.862905, -0.495666,
0.164179, 0.897140, -0.410104,
0.196334, 0.881675, -0.429071,
0.164016, 0.882055, -0.441675,
0.163706, 0.865919, -0.472637,
0.228090, 0.864787, -0.447345,
0.295021, 0.930534, -0.216955,
0.325939, 0.915445, -0.236060,
0.295269, 0.921654, -0.251736,
0.295186, 0.911593, -0.286117,
0.356275, 0.898967, -0.254807,
0.325609, 0.894920, -0.305118,
0.294703, 0.900244, -0.320484,
0.356197, 0.888475, -0.289370,
0.355602, 0.876818, -0.323632,
0.385011, 0.857377, -0.341572,
0.354516, 0.864050, -0.357403,
0.386299, 0.880940, -0.273346,
0.293939, 0.887823, -0.354076,
0.323903, 0.869647, -0.372561,
0.292822, 0.874225, -0.387280,
0.291383, 0.859517, -0.419912,
0.353281, 0.850133, -0.390470,
0.415435, 0.861701, -0.291352,
0.443956, 0.841094, -0.308973,
0.414707, 0.849744, -0.325506,
0.413488, 0.836680, -0.359158,
0.471716, 0.819222, -0.326128,
0.527686, 0.797844, 0.291536,
0.533322, 0.806959, 0.253742,
0.560060, 0.794224, 0.235673,
0.538182, 0.814815, 0.215489,
0.542239, 0.821376, 0.176970,
0.581130, 0.772646, 0.255549,
0.586051, 0.780587, 0.217322,
0.568272, 0.807470, 0.158301,
0.545532, 0.826677, 0.137840,
0.590346, 0.787154, 0.178553,
0.593687, 0.792524, 0.139437,
0.618331, 0.776630, 0.120472,
0.596053, 0.796672, 0.100171,
0.611619, 0.765845, 0.198505,
0.547978, 0.830634, 0.098832,
0.573242, 0.815477, 0.079939,
0.549620, 0.833286, 0.059604,
0.550447, 0.834622, 0.020351,
0.597732, 0.799376, 0.060948,
0.631794, 0.743751, 0.218339,
0.636140, 0.750382, 0.179588,
0.659941, 0.733998, 0.160388,
0.639533, 0.755818, 0.140487,
0.641952, 0.760032, 0.101232,
0.679433, 0.711295, 0.180084,
0.682882, 0.716793, 0.140998,
0.574934, 0.818200, 0.000000,
0.550447, 0.834622, -0.020351,
0.598760, 0.800670, 0.020351,
0.598760, 0.800670, -0.020351,
0.621778, 0.782140, -0.040604,
0.597732, 0.799376, -0.060948,
0.621778, 0.782140, 0.040604,
0.645320, 0.763641, -0.020352,
0.644291, 0.762348, -0.060949,
0.645320, 0.763641, 0.020352,
0.668055, 0.744112, 0.000000,
0.689842, 0.723674, 0.020352,
0.689842, 0.723674, -0.020352,
0.644291, 0.762348, 0.060949,
0.665716, 0.741796, -0.081001,
0.641952, 0.760032, -0.101232,
0.688679, 0.722524, -0.060664,
0.686355, 0.720226, -0.100949,
0.706082, 0.697769, -0.120696,
0.682882, 0.716793, -0.140998,
0.710703, 0.702335, -0.040319,
0.665716, 0.741796, 0.081001,
0.686355, 0.720226, 0.100949,
0.688679, 0.722524, 0.060664,
0.710703, 0.702335, 0.040319,
0.706082, 0.697769, 0.120696,
0.549620, 0.833286, -0.059604,
0.573242, 0.815477, -0.079939,
0.547978, 0.830634, -0.098832,
0.545532, 0.826677, -0.137840,
0.596053, 0.796672, -0.100171,
0.568272, 0.807470, -0.158301,
0.542239, 0.821376, -0.176970,
0.593687, 0.792524, -0.139437,
0.590346, 0.787154, -0.178553,
0.611619, 0.765845, -0.198505,
0.586051, 0.780587, -0.217322,
0.618331, 0.776630, -0.120472,
0.538182, 0.814815, -0.215489,
0.560060, 0.794224, -0.235673,
0.533322, 0.806959, -0.253742,
0.527686, 0.797844, -0.291536,
0.581130, 0.772646, -0.255549,
0.639533, 0.755818, -0.140487,
0.659941, 0.733998, -0.160388,
0.636140, 0.750382, -0.179588,
0.631794, 0.743751, -0.218339,
0.679433, 0.711295, -0.180084,
0.723820, 0.675435, 0.140967,
0.727293, 0.678867, 0.100920,
0.747667, 0.659120, 0.080959,
0.729617, 0.681162, 0.060644,
0.730779, 0.682311, 0.020333,
0.764743, 0.636343, 0.101176,
0.767082, 0.638652, 0.060901,
0.750007, 0.661430, 0.000000,
0.730779, 0.682311, -0.020333,
0.768380, 0.639671, 0.020333,
0.768380, 0.639671, -0.020333,
0.785785, 0.617167, -0.040577,
0.767082, 0.638652, -0.060901,
0.785785, 0.617167, 0.040577,
0.729617, 0.681162, -0.060644,
0.747667, 0.659120, -0.080959,
0.727293, 0.678867, -0.100920,
0.723820, 0.675435, -0.140967,
0.764743, 0.636343, -0.101176,
0.802005, 0.594204, 0.060900,
0.803303, 0.595224, 0.020332,
0.819885, 0.572529, 0.000000,
0.803303, 0.595224, -0.020332,
0.802005, 0.594204, -0.060900,
0.835429, 0.549222, 0.020332,
0.835429, 0.549222, -0.020332,
-0.032906, 0.873128, 0.486380,
-0.032898, 0.855762, 0.516324,
-0.032928, 0.889480, 0.455785,
-0.065827, 0.876914, 0.476118,
-0.098590, 0.879653, 0.465287,
-0.098538, 0.862905, 0.495666,
-0.032936, 0.904730, 0.424710,
-0.098843, 0.895411, 0.434130,
-0.131555, 0.881402, 0.453679,
-0.065943, 0.907978, 0.413796,
-0.098950, 0.910112, 0.402375,
-0.098882, 0.923674, 0.370201,
-0.131782, 0.911107, 0.390534,
-0.033014, 0.919041, 0.392778,
-0.164016, 0.882055, 0.441675,
-0.163706, 0.865919, 0.472637,
-0.164179, 0.897140, 0.410104,
-0.196334, 0.881675, 0.429071,
-0.228338, 0.880251, 0.415958,
-0.228090, 0.864787, 0.447345,
-0.164167, 0.911094, 0.378097,
-0.033021, 0.932091, 0.360716,
-0.033041, 0.944047, 0.328151,
-0.066055, 0.934620, 0.349460,
-0.098929, 0.936012, 0.337778,
-0.033048, 0.954845, 0.295262,
-0.229687, 0.894658, 0.383186,
-0.261444, 0.877768, 0.401461,
-0.197527, 0.910128, 0.364213,
-0.230712, 0.907953, 0.349848,
-0.231386, 0.920064, 0.316139,
-0.263541, 0.904573, 0.335103,
-0.165129, 0.924096, 0.344643,
-0.198871, 0.934170, 0.296273,
-0.231780, 0.931106, 0.281637,
-0.165615, 0.936027, 0.310525,
-0.165941, 0.946748, 0.275921,
-0.132768, 0.957732, 0.255190,
-0.166080, 0.956201, 0.241032,
-0.132199, 0.936674, 0.324291,
-0.231772, 0.940847, 0.247162,
-0.264286, 0.926741, 0.267028,
-0.198844, 0.953424, 0.226811,
-0.231436, 0.949405, 0.212293,
-0.230749, 0.956736, 0.177234,
-0.263677, 0.944158, 0.197584,
-0.165671, 0.964409, 0.206081,
-0.099370, 0.947620, 0.303549,
-0.066348, 0.957048, 0.282237,
-0.099650, 0.958014, 0.268847,
-0.099744, 0.967138, 0.233871,
-0.033305, 0.964909, 0.260463,
-0.292822, 0.874225, 0.387280,
-0.291383, 0.859517, 0.419912,
-0.293939, 0.887823, 0.354076,
-0.323903, 0.869647, 0.372561,
-0.354516, 0.864050, 0.357403,
-0.353281, 0.850133, 0.390470,
-0.294703, 0.900244, 0.320484,
-0.355602, 0.876818, 0.323632,
-0.385011, 0.857377, 0.341572,
-0.325609, 0.894920, 0.305118,
-0.356197, 0.888475, 0.289370,
-0.356275, 0.898967, 0.254807,
-0.386299, 0.880940, 0.273346,
-0.295186, 0.911593, 0.286117,
-0.414707, 0.849744, 0.325506,
-0.413488, 0.836680, 0.359158,
-0.415435, 0.861701, 0.291352,
-0.443956, 0.841094, 0.308973,
-0.472596, 0.831470, 0.292081,
-0.471716, 0.819222, 0.326128,
-0.415646, 0.872496, 0.256883,
-0.295269, 0.921654, 0.251736,
-0.295021, 0.930534, 0.216955,
-0.325939, 0.915445, 0.236060,
-0.356133, 0.908143, 0.220102,
-0.294425, 0.938191, 0.181964,
-0.033310, 0.973350, 0.226892,
-0.033324, 0.980638, 0.192973,
-0.066626, 0.974649, 0.213589,
-0.099776, 0.974700, 0.200013,
-0.033328, 0.986732, 0.158898,
-0.099996, 0.981097, 0.165681,
-0.133106, 0.973502, 0.185946,
-0.066698, 0.987169, 0.145079,
-0.100061, 0.986311, 0.131070,
-0.099955, 0.990313, 0.096377,
-0.133245, 0.984157, 0.116967,
-0.033376, 0.991690, 0.124250,
-0.165925, 0.971058, 0.171797,
-0.166025, 0.976527, 0.137228,
-0.198580, 0.967371, 0.157351,
-0.230901, 0.962455, 0.142705,
-0.165954, 0.980785, 0.102565,
-0.033381, 0.995397, 0.089833,
-0.033392, 0.997915, 0.055231,
-0.066765, 0.994881, 0.075862,
-0.099983, 0.993068, 0.061793,
-0.033396, 0.999229, 0.020640,
-0.478242, 0.840609, 0.254286,
-0.505993, 0.818713, 0.271440,
-0.449711, 0.861255, 0.236644,
-0.483108, 0.848498, 0.216004,
-0.487173, 0.855081, 0.177481,
-0.515629, 0.834306, 0.195090,
-0.420581, 0.880517, 0.218637,
-0.457934, 0.874631, 0.159117,
-0.490456, 0.860438, 0.138204,
-0.424534, 0.887388, 0.179759,
-0.427873, 0.892852, 0.140501,
-0.397184, 0.909626, 0.121761,
-0.430583, 0.896869, 0.101119,
-0.390469, 0.898629, 0.199999,
-0.492908, 0.864409, 0.099190,
-0.522087, 0.844754, 0.117537,
-0.462906, 0.882736, 0.080590,
-0.494553, 0.867079, 0.059926,
-0.495384, 0.868428, 0.020667,
-0.525326, 0.849995, 0.039260,
-0.432270, 0.899620, 0.061863,
-0.360134, 0.915124, 0.181249,
-0.329196, 0.930240, 0.162122,
-0.363521, 0.920698, 0.142013,
-0.366279, 0.924826, 0.102647,
-0.297852, 0.943879, 0.142749,
-0.464610, 0.885516, 0.000000,
-0.495384, 0.868428, -0.020667,
-0.432953, 0.901180, 0.020667,
-0.432953, 0.901180, -0.020667,
-0.400628, 0.915314, -0.041197,
-0.432270, 0.899620, -0.061863,
-0.400628, 0.915314, 0.041197,
-0.367868, 0.929648, -0.020669,
-0.367185, 0.928088, -0.061865,
-0.367868, 0.929648, 0.020669,
-0.334425, 0.942422, 0.000000,
-0.300527, 0.953549, 0.020669,
-0.300527, 0.953549, -0.020669,
-0.367185, 0.928088, 0.061865,
-0.333520, 0.939160, -0.082120,
-0.366279, 0.924826, -0.102647,
-0.300080, 0.951932, -0.061453,
-0.299185, 0.948702, -0.102238,
-0.264412, 0.956652, -0.122082,
-0.297852, 0.943879, -0.142749,
-0.266183, 0.963059, -0.040784,
-0.333520, 0.939160, 0.082120,
-0.299185, 0.948702, 0.102238,
-0.300080, 0.951932, 0.061453,
-0.266183, 0.963059, 0.040784,
-0.264412, 0.956652, 0.122082,
-0.494553, 0.867079, -0.059926,
-0.525326, 0.849995, -0.039260,
-0.462906, 0.882736, -0.080590,
-0.492908, 0.864409, -0.099190,
-0.490456, 0.860438, -0.138204,
-0.522087, 0.844754, -0.117537,
-0.430583, 0.896869, -0.101119,
-0.457934, 0.874631, -0.159117,
-0.487173, 0.855081, -0.177481,
-0.427873, 0.892852, -0.140501,
-0.424534, 0.887388, -0.179759,
-0.390469, 0.898629, -0.199999,
-0.420581, 0.880517, -0.218637,
-0.397184, 0.909626, -0.121761,
-0.483108, 0.848498, -0.216004,
-0.515629, 0.834306, -0.195090,
-0.449711, 0.861255, -0.236644,
-0.478242, 0.840609, -0.254286,
-0.472596, 0.831470, -0.292081,
-0.505993, 0.818713, -0.271440,
-0.415646, 0.872496, -0.256883,
-0.363521, 0.920698, -0.142013,
-0.329196, 0.930240, -0.162122,
-0.360134, 0.915124, -0.181249,
-0.356133, 0.908143, -0.220102,
-0.294425, 0.938191, -0.181964,
-0.232233, 0.967277, 0.102196,
-0.199641, 0.976427, 0.082059,
-0.233124, 0.970505, 0.061424,
-0.233570, 0.972121, 0.020642,
-0.166851, 0.984044, 0.061795,
-0.200538, 0.979686, 0.000000,
-0.233570, 0.972121, -0.020642,
-0.167078, 0.985728, 0.020642,
-0.167078, 0.985728, -0.020642,
-0.133410, 0.990206, -0.041156,
-0.166851, 0.984044, -0.061795,
-0.133410, 0.990206, 0.041156,
-0.233124, 0.970505, -0.061424,
-0.199641, 0.976427, -0.082059,
-0.232233, 0.967277, -0.102196,
-0.230901, 0.962455, -0.142705,
-0.165954, 0.980785, -0.102565,
-0.100210, 0.994752, 0.020640,
-0.066798, 0.997767, 0.000000,
-0.100210, 0.994752, -0.020640,
-0.099983, 0.993068, -0.061793,
-0.033396, 0.999229, -0.020640,
-0.533322, 0.806959, 0.253742,
-0.527686, 0.797844, 0.291536,
-0.538182, 0.814815, 0.215489,
-0.560060, 0.794224, 0.235673,
-0.586051, 0.780587, 0.217322,
-0.581130, 0.772646, 0.255549,
-0.542239, 0.821376, 0.176970,
-0.590346, 0.787154, 0.178553,
-0.611619, 0.765845, 0.198505,
-0.568272, 0.807470, 0.158301,
-0.593687, 0.792524, 0.139437,
-0.596053, 0.796672, 0.100171,
-0.618331, 0.776630, 0.120472,
-0.545532, 0.826677, 0.137840,
-0.636140, 0.750382, 0.179588,
-0.631794, 0.743751, 0.218339,
-0.639533, 0.755818, 0.140487,
-0.659941, 0.733998, 0.160388,
-0.682882, 0.716793, 0.140998,
-0.679433, 0.711295, 0.180084,
-0.641952, 0.760032, 0.101232,
-0.547978, 0.830634, 0.098832,
-0.549620, 0.833286, 0.059604,
-0.573242, 0.815477, 0.079939,
-0.597732, 0.799376, 0.060948,
-0.550447, 0.834622, 0.020351,
-0.686355, 0.720226, 0.100949,
-0.706082, 0.697769, 0.120696,
-0.665716, 0.741796, 0.081001,
-0.688679, 0.722524, 0.060664,
-0.689842, 0.723674, 0.020352,
-0.710703, 0.702335, 0.040319,
-0.644291, 0.762348, 0.060949,
-0.668055, 0.744112, 0.000000,
-0.689842, 0.723674, -0.020352,
-0.645320, 0.763641, 0.020352,
-0.645320, 0.763641, -0.020352,
-0.621778, 0.782140, -0.040604,
-0.644291, 0.762348, -0.060949,
-0.621778, 0.782140, 0.040604,
-0.688679, 0.722524, -0.060664,
-0.710703, 0.702335, -0.040319,
-0.665716, 0.741796, -0.081001,
-0.686355, 0.720226, -0.100949,
-0.682882, 0.716793, -0.140998,
-0.706082, 0.697769, -0.120696,
-0.641952, 0.760032, -0.101232,
-0.598760, 0.800670, 0.020351,
-0.574934, 0.818200, 0.000000,
-0.598760, 0.800670, -0.020351,
-0.597732, 0.799376, -0.060948,
-0.550447, 0.834622, -0.020351,
-0.727293, 0.678867, 0.100920,
-0.723820, 0.675435, 0.140967,
-0.729617, 0.681162, 0.060644,
-0.747667, 0.659120, 0.080959,
-0.767082, 0.638652, 0.060901,
-0.764743, 0.636343, 0.101176,
-0.730779, 0.682311, 0.020333,
-0.768380, 0.639671, 0.020333,
-0.785785, 0.617167, 0.040577,
-0.750007, 0.661430, 0.000000,
-0.768380, 0.639671, -0.020333,
-0.767082, 0.638652, -0.060901,
-0.785785, 0.617167, -0.040577,
-0.730779, 0.682311, -0.020333,
-0.803303, 0.595224, 0.020332,
-0.802005, 0.594204, 0.060900,
-0.803303, 0.595224, -0.020332,
-0.819885, 0.572529, 0.000000,
-0.835429, 0.549222, -0.020332,
-0.835429, 0.549222, 0.020332,
-0.802005, 0.594204, -0.060900,
-0.729617, 0.681162, -0.060644,
-0.727293, 0.678867, -0.100920,
-0.747667, 0.659120, -0.080959,
-0.764743, 0.636343, -0.101176,
-0.723820, 0.675435, -0.140967,
-0.549620, 0.833286, -0.059604,
-0.547978, 0.830634, -0.098832,
-0.573242, 0.815477, -0.079939,
-0.596053, 0.796672, -0.100171,
-0.545532, 0.826677, -0.137840,
-0.593687, 0.792524, -0.139437,
-0.618331, 0.776630, -0.120472,
-0.568272, 0.807470, -0.158301,
-0.590346, 0.787154, -0.178553,
-0.586051, 0.780587, -0.217322,
-0.611619, 0.765845, -0.198505,
-0.542239, 0.821376, -0.176970,
-0.639533, 0.755818, -0.140487,
-0.636140, 0.750382, -0.179588,
-0.659941, 0.733998, -0.160388,
-0.679433, 0.711295, -0.180084,
-0.631794, 0.743751, -0.218339,
-0.538182, 0.814815, -0.215489,
-0.533322, 0.806959, -0.253742,
-0.560060, 0.794224, -0.235673,
-0.581130, 0.772646, -0.255549,
-0.527686, 0.797844, -0.291536,
-0.033392, 0.997915, -0.055231,
-0.033381, 0.995397, -0.089833,
-0.066765, 0.994881, -0.075862,
-0.099955, 0.990313, -0.096377,
-0.033376, 0.991690, -0.124250,
-0.100061, 0.986311, -0.131070,
-0.133245, 0.984157, -0.116967,
-0.066698, 0.987169, -0.145079,
-0.099996, 0.981097, -0.165681,
-0.099776, 0.974700, -0.200013,
-0.133106, 0.973502, -0.185946,
-0.033328, 0.986732, -0.158898,
-0.166025, 0.976527, -0.137228,
-0.165925, 0.971058, -0.171797,
-0.198580, 0.967371, -0.157351,
-0.230749, 0.956736, -0.177234,
-0.165671, 0.964409, -0.206081,
-0.033324, 0.980638, -0.192973,
-0.033310, 0.973350, -0.226892,
-0.066626, 0.974649, -0.213589,
-0.099744, 0.967138, -0.233871,
-0.033305, 0.964909, -0.260463,
-0.231436, 0.949405, -0.212293,
-0.263677, 0.944158, -0.197584,
-0.198844, 0.953424, -0.226811,
-0.231772, 0.940847, -0.247162,
-0.231780, 0.931106, -0.281637,
-0.264286, 0.926741, -0.267028,
-0.166080, 0.956201, -0.241032,
-0.198871, 0.934170, -0.296273,
-0.231386, 0.920064, -0.316139,
-0.165941, 0.946748, -0.275921,
-0.165615, 0.936027, -0.310525,
-0.132199, 0.936674, -0.324291,
-0.165129, 0.924096, -0.344643,
-0.132768, 0.957732, -0.255190,
-0.230712, 0.907953, -0.349848,
-0.263541, 0.904573, -0.335103,
-0.197527, 0.910128, -0.364213,
-0.229687, 0.894658, -0.383186,
-0.228338, 0.880251, -0.415958,
-0.261444, 0.877768, -0.401461,
-0.164167, 0.911094, -0.378097,
-0.099650, 0.958014, -0.268847,
-0.066348, 0.957048, -0.282237,
-0.099370, 0.947620, -0.303549,
-0.098929, 0.936012, -0.337778,
-0.033048, 0.954845, -0.295262,
-0.295021, 0.930534, -0.216955,
-0.295269, 0.921654, -0.251736,
-0.325939, 0.915445, -0.236060,
-0.356275, 0.898967, -0.254807,
-0.295186, 0.911593, -0.286117,
-0.356197, 0.888475, -0.289370,
-0.386299, 0.880940, -0.273346,
-0.325609, 0.894920, -0.305118,
-0.355602, 0.876818, -0.323632,
-0.354516, 0.864050, -0.357403,
-0.385011, 0.857377, -0.341572,
-0.294703, 0.900244, -0.320484,
-0.415435, 0.861701, -0.291352,
-0.414707, 0.849744, -0.325506,
-0.443956, 0.841094, -0.308973,
-0.471716, 0.819222, -0.326128,
-0.413488, 0.836680, -0.359158,
-0.293939, 0.887823, -0.354076,
-0.292822, 0.874225, -0.387280,
-0.323903, 0.869647, -0.372561,
-0.353281, 0.850133, -0.390470,
-0.291383, 0.859517, -0.419912,
-0.033041, 0.944047, -0.328151,
-0.033021, 0.932091, -0.360716,
-0.066055, 0.934620, -0.349460,
-0.098882, 0.923674, -0.370201,
-0.033014, 0.919041, -0.392778,
-0.098950, 0.910112, -0.402375,
-0.131782, 0.911107, -0.390534,
-0.065943, 0.907978, -0.413796,
-0.098843, 0.895411, -0.434130,
-0.098590, 0.879653, -0.465287,
-0.131555, 0.881402, -0.453679,
-0.032936, 0.904730, -0.424710,
-0.164179, 0.897140, -0.410104,
-0.164016, 0.882055, -0.441675,
-0.196334, 0.881675, -0.429071,
-0.228090, 0.864787, -0.447345,
-0.163706, 0.865919, -0.472637,
-0.032928, 0.889480, -0.455785,
-0.032906, 0.873128, -0.486380,
-0.065827, 0.876914, -0.476118,
-0.098538, 0.862905, -0.495666,
-0.032898, 0.855762, -0.516324,
0.000000, -0.868328, 0.495991,
0.032906, -0.873128, 0.486380,
0.032898, -0.855762, 0.516324,
0.032928, -0.889480, 0.455785,
0.065827, -0.876914, 0.476118,
0.098590, -0.879653, 0.465287,
0.098538, -0.862905, 0.495666,
0.000000, -0.900543, 0.434767,
0.032936, -0.904730, 0.424710,
0.098843, -0.895411, 0.434130,
0.131555, -0.881402, 0.453679,
0.065943, -0.907978, 0.413796,
0.098950, -0.910112, 0.402375,
0.098882, -0.923674, 0.370201,
0.131782, -0.911107, 0.390534,
0.033014, -0.919041, 0.392778,
0.164016, -0.882055, 0.441675,
0.163706, -0.865919, 0.472637,
0.164179, -0.897140, 0.410104,
0.196334, -0.881675, 0.429071,
0.228338, -0.880251, 0.415958,
0.228090, -0.864787, 0.447345,
0.164167, -0.911094, 0.378097,
0.000000, -0.928448, 0.371461,
0.033021, -0.932091, 0.360716,
0.033041, -0.944047, 0.328151,
0.066055, -0.934620, 0.349460,
0.098929, -0.936012, 0.337778,
0.000000, -0.951910, 0.306378,
0.033048, -0.954845, 0.295262,
0.229687, -0.894658, 0.383186,
0.261444, -0.877768, 0.401461,
0.197527, -0.910128, 0.364213,
0.230712, -0.907953, 0.349848,
0.231386, -0.920064, 0.316139,
0.263541, -0.904573, 0.335103,
0.165129, -0.924096, 0.344643,
0.198871, -0.934170, 0.296273,
0.231780, -0.931106, 0.281637,
0.165615, -0.936027, 0.310525,
0.165941, -0.946748, 0.275921,
0.132768, -0.957732, 0.255190,
0.166080, -0.956201, 0.241032,
0.132199, -0.936674, 0.324291,
0.231772, -0.940847, 0.247162,
0.264286, -0.926741, 0.267028,
0.198844, -0.953424, 0.226811,
0.231436, -0.949405, 0.212293,
0.230749, -0.956736, 0.177234,
0.263677, -0.944158, 0.197584,
0.165671, -0.964409, 0.206081,
0.099370, -0.947620, 0.303549,
0.066348, -0.957048, 0.282237,
0.099650, -0.958014, 0.268847,
0.099744, -0.967138, 0.233871,
0.033305, -0.964909, 0.260463,
0.292822, -0.874225, 0.387280,
0.291383, -0.859517, 0.419912,
0.293939, -0.887823, 0.354076,
0.323903, -0.869647, 0.372561,
0.354516, -0.864050, 0.357403,
0.353281, -0.850133, 0.390470,
0.294703, -0.900244, 0.320484,
0.355602, -0.876818, 0.323632,
0.385011, -0.857377, 0.341572,
0.325609, -0.894920, 0.305118,
0.356197, -0.888475, 0.289370,
0.356275, -0.898967, 0.254807,
0.386299, -0.880940, 0.273346,
0.295186, -0.911593, 0.286117,
0.414707, -0.849744, 0.325506,
0.413488, -0.836680, 0.359158,
0.415435, -0.861701, 0.291352,
0.443956, -0.841094, 0.308973,
0.472596, -0.831470, 0.292081,
0.471716, -0.819222, 0.326128,
0.415646, -0.872496, 0.256883,
0.295269, -0.921654, 0.251736,
0.295021, -0.930534, 0.216955,
0.325939, -0.915445, 0.236060,
0.356133, -0.908143, 0.220102,
0.294425, -0.938191, 0.181964,
0.000000, -0.970815, 0.239828,
0.033310, -0.973350, 0.226892,
0.033324, -0.980638, 0.192973,
0.066626, -0.974649, 0.213589,
0.099776, -0.974700, 0.200013,
0.000000, -0.985074, 0.172130,
0.033328, -0.986732, 0.158898,
0.099996, -0.981097, 0.165681,
0.133106, -0.973502, 0.185946,
0.066698, -0.987169, 0.145079,
0.100061, -0.986311, 0.131070,
0.099955, -0.990313, 0.096377,
0.133245, -0.984157, 0.116967,
0.033376, -0.991690, 0.124250,
0.165925, -0.971058, 0.171797,
0.166025, -0.976527, 0.137228,
0.198580, -0.967371, 0.157351,
0.230901, -0.962455, 0.142705,
0.165954, -0.980785, 0.102565,
0.000000, -0.994618, 0.103609,
0.033381, -0.995397, 0.089833,
0.033392, -0.997915, 0.055231,
0.066765, -0.994881, 0.075862,
0.099983, -0.993068, 0.061793,
0.000000, -0.999402, 0.034591,
0.033396, -0.999229, 0.020640,
0.478242, -0.840609, 0.254286,
0.505993, -0.818713, 0.271440,
0.449711, -0.861255, 0.236644,
0.483108, -0.848498, 0.216004,
0.487173, -0.855081, 0.177481,
0.515629, -0.834306, 0.195090,
0.420581, -0.880517, 0.218637,
0.457934, -0.874631, 0.159117,
0.490456, -0.860438, 0.138204,
0.424534, -0.887388, 0.179759,
0.427873, -0.892852, 0.140501,
0.397184, -0.909626, 0.121761,
0.430583, -0.896869, 0.101119,
0.390469, -0.898629, 0.199999,
0.492908, -0.864409, 0.099190,
0.522087, -0.844754, 0.117537,
0.462906, -0.882736, 0.080590,
0.494553, -0.867079, 0.059926,
0.495384, -0.868428, 0.020667,
0.525326, -0.849995, 0.039260,
0.432270, -0.899620, 0.061863,
0.360134, -0.915124, 0.181249,
0.329196, -0.930240, 0.162122,
0.363521, -0.920698, 0.142013,
0.366279, -0.924826, 0.102647,
0.297852, -0.943879, 0.142749,
0.464610, -0.885516, 0.000000,
0.495384, -0.868428, -0.020667,
0.432953, -0.901180, 0.020667,
0.432953, -0.901180, -0.020667,
0.400628, -0.915314, -0.041197,
0.432270, -0.899620, -0.061863,
0.400628, -0.915314, 0.041197,
0.367868, -0.929648, -0.020669,
0.367185, -0.928088, -0.061865,
0.367868, -0.929648, 0.020669,
0.334425, -0.942422, 0.000000,
0.300527, -0.953549, 0.020669,
0.300527, -0.953549, -0.020669,
0.367185, -0.928088, 0.061865,
0.333520, -0.939160, -0.082120,
0.366279, -0.924826, -0.102647,
0.300080, -0.951932, -0.061453,
0.299185, -0.948702, -0.102238,
0.264412, -0.956652, -0.122082,
0.297852, -0.943879, -0.142749,
0.266183, -0.963059, -0.040784,
0.333520, -0.939160, 0.082120,
0.299185, -0.948702, 0.102238,
0.300080, -0.951932, 0.061453,
0.266183, -0.963059, 0.040784,
0.264412, -0.956652, 0.122082,
0.494553, -0.867079, -0.059926,
0.525326, -0.849995, -0.039260,
0.462906, -0.882736, -0.080590,
0.492908, -0.864409, -0.099190,
0.490456, -0.860438, -0.138204,
0.522087, -0.844754, -0.117537,
0.430583, -0.896869, -0.101119,
0.457934, -0.874631, -0.159117,
0.487173, -0.855081, -0.177481,
0.427873, -0.892852, -0.140501,
0.424534, -0.887388, -0.179759,
0.390469, -0.898629, -0.199999,
0.420581, -0.880517, -0.218637,
0.397184, -0.909626, -0.121761,
0.483108, -0.848498, -0.216004,
0.515629, -0.834306, -0.195090,
0.449711, -0.861255, -0.236644,
0.478242, -0.840609, -0.254286,
0.472596, -0.831470, -0.292081,
0.505993, -0.818713, -0.271440,
0.415646, -0.872496, -0.256883,
0.363521, -0.920698, -0.142013,
0.329196, -0.930240, -0.162122,
0.360134, -0.915124, -0.181249,
0.356133, -0.908143, -0.220102,
0.294425, -0.938191, -0.181964,
0.232233, -0.967277, 0.102196,
0.199641, -0.976427, 0.082059,
0.233124, -0.970505, 0.061424,
0.233570, -0.972121, 0.020642,
0.166851, -0.984044, 0.061795,
0.200538, -0.979686, 0.000000,
0.233570, -0.972121, -0.020642,
0.167078, -0.985728, 0.020642,
0.167078, -0.985728, -0.020642,
0.133410, -0.990206, -0.041156,
0.166851, -0.984044, -0.061795,
0.133410, -0.990206, 0.041156,
0.233124, -0.970505, -0.061424,
0.199641, -0.976427, -0.082059,
0.232233, -0.967277, -0.102196,
0.230901, -0.962455, -0.142705,
0.165954, -0.980785, -0.102565,
0.100210, -0.994752, 0.020640,
0.066798, -0.997767, 0.000000,
0.100210, -0.994752, -0.020640,
0.099983, -0.993068, -0.061793,
0.033396, -0.999229, -0.020640,
0.533322, -0.806959, 0.253742,
0.527686, -0.797844, 0.291536,
0.538182, -0.814815, 0.215489,
0.560060, -0.794224, 0.235673,
0.586051, -0.780587, 0.217322,
0.581130, -0.772646, 0.255549,
0.542239, -0.821376, 0.176970,
0.590346, -0.787154, 0.178553,
0.611619, -0.765845, 0.198505,
0.568272, -0.807470, 0.158301,
0.593687, -0.792524, 0.139437,
0.596053, -0.796672, 0.100171,
0.618331, -0.776630, 0.120472,
0.545532, -0.826677, 0.137840,
0.636140, -0.750382, 0.179588,
0.631794, -0.743751, 0.218339,
0.639533, -0.755818, 0.140487,
0.659941, -0.733998, 0.160388,
0.682882, -0.716793, 0.140998,
0.679433, -0.711295, 0.180084,
0.641952, -0.760032, 0.101232,
0.547978, -0.830634, 0.098832,
0.549620, -0.833286, 0.059604,
0.573242, -0.815477, 0.079939,
0.597732, -0.799376, 0.060948,
0.550447, -0.834622, 0.020351,
0.686355, -0.720226, 0.100949,
0.706082, -0.697769, 0.120696,
0.665716, -0.741796, 0.081001,
0.688679, -0.722524, 0.060664,
0.689842, -0.723674, 0.020352,
0.710703, -0.702335, 0.040319,
0.644291, -0.762348, 0.060949,
0.668055, -0.744112, 0.000000,
0.689842, -0.723674, -0.020352,
0.645320, -0.763641, 0.020352,
0.645320, -0.763641, -0.020352,
0.621778, -0.782140, -0.040604,
0.644291, -0.762348, -0.060949,
0.621778, -0.782140, 0.040604,
0.688679, -0.722524, -0.060664,
0.710703, -0.702335, -0.040319,
0.665716, -0.741796, -0.081001,
0.686355, -0.720226, -0.100949,
0.682882, -0.716793, -0.140998,
0.706082, -0.697769, -0.120696,
0.641952, -0.760032, -0.101232,
0.598760, -0.800670, 0.020351,
0.574934, -0.818200, 0.000000,
0.598760, -0.800670, -0.020351,
0.597732, -0.799376, -0.060948,
0.550447, -0.834622, -0.020351,
0.727293, -0.678867, 0.100920,
0.723820, -0.675435, 0.140967,
0.729617, -0.681162, 0.060644,
0.747667, -0.659120, 0.080959,
0.767082, -0.638652, 0.060901,
0.764743, -0.636343, 0.101176,
0.730779, -0.682311, 0.020333,
0.768380, -0.639671, 0.020333,
0.785785, -0.617167, 0.040577,
0.750007, -0.661430, 0.000000,
0.768380, -0.639671, -0.020333,
0.767082, -0.638652, -0.060901,
0.785785, -0.617167, -0.040577,
0.730779, -0.682311, -0.020333,
0.803303, -0.595224, 0.020332,
0.802005, -0.594204, 0.060900,
0.803303, -0.595224, -0.020332,
0.819885, -0.572529, 0.000000,
0.835429, -0.549222, -0.020332,
0.835429, -0.549222, 0.020332,
0.802005, -0.594204, -0.060900,
0.729617, -0.681162, -0.060644,
0.727293, -0.678867, -0.100920,
0.747667, -0.659120, -0.080959,
0.764743, -0.636343, -0.101176,
0.723820, -0.675435, -0.140967,
0.549620, -0.833286, -0.059604,
0.547978, -0.830634, -0.098832,
0.573242, -0.815477, -0.079939,
0.596053, -0.796672, -0.100171,
0.545532, -0.826677, -0.137840,
0.593687, -0.792524, -0.139437,
0.618331, -0.776630, -0.120472,
0.568272, -0.807470, -0.158301,
0.590346, -0.787154, -0.178553,
0.586051, -0.780587, -0.217322,
0.611619, -0.765845, -0.198505,
0.542239, -0.821376, -0.176970,
0.639533, -0.755818, -0.140487,
0.636140, -0.750382, -0.179588,
0.659941, -0.733998, -0.160388,
0.679433, -0.711295, -0.180084,
0.631794, -0.743751, -0.218339,
0.538182, -0.814815, -0.215489,
0.533322, -0.806959, -0.253742,
0.560060, -0.794224, -0.235673,
0.581130, -0.772646, -0.255549,
0.527686, -0.797844, -0.291536,
0.000000, -0.999402, -0.034591,
0.033392, -0.997915, -0.055231,
0.033381, -0.995397, -0.089833,
0.066765, -0.994881, -0.075862,
0.099955, -0.990313, -0.096377,
0.000000, -0.994618, -0.103609,
0.033376, -0.991690, -0.124250,
0.100061, -0.986311, -0.131070,
0.133245, -0.984157, -0.116967,
0.066698, -0.987169, -0.145079,
0.099996, -0.981097, -0.165681,
0.099776, -0.974700, -0.200013,
0.133106, -0.973502, -0.185946,
0.033328, -0.986732, -0.158898,
0.166025, -0.976527, -0.137228,
0.165925, -0.971058, -0.171797,
0.198580, -0.967371, -0.157351,
0.230749, -0.956736, -0.177234,
0.165671, -0.964409, -0.206081,
0.000000, -0.985074, -0.172130,
0.033324, -0.980638, -0.192973,
0.033310, -0.973350, -0.226892,
0.066626, -0.974649, -0.213589,
0.099744, -0.967138, -0.233871,
0.000000, -0.970815, -0.239828,
0.033305, -0.964909, -0.260463,
0.231436, -0.949405, -0.212293,
0.263677, -0.944158, -0.197584,
0.198844, -0.953424, -0.226811,
0.231772, -0.940847, -0.247162,
0.231780, -0.931106, -0.281637,
0.264286, -0.926741, -0.267028,
0.166080, -0.956201, -0.241032,
0.198871, -0.934170, -0.296273,
0.231386, -0.920064, -0.316139,
0.165941, -0.946748, -0.275921,
0.165615, -0.936027, -0.310525,
0.132199, -0.936674, -0.324291,
0.165129, -0.924096, -0.344643,
0.132768, -0.957732, -0.255190,
0.230712, -0.907953, -0.349848,
0.263541, -0.904573, -0.335103,
0.197527, -0.910128, -0.364213,
0.229687, -0.894658, -0.383186,
0.228338, -0.880251, -0.415958,
0.261444, -0.877768, -0.401461,
0.164167, -0.911094, -0.378097,
0.099650, -0.958014, -0.268847,
0.066348, -0.957048, -0.282237,
0.099370, -0.947620, -0.303549,
0.098929, -0.936012, -0.337778,
0.033048, -0.954845, -0.295262,
0.295021, -0.930534, -0.216955,
0.295269, -0.921654, -0.251736,
0.325939, -0.915445, -0.236060,
0.356275, -0.898967, -0.254807,
0.295186, -0.911593, -0.286117,
0.356197, -0.888475, -0.289370,
0.386299, -0.880940, -0.273346,
0.325609, -0.894920, -0.305118,
0.355602, -0.876818, -0.323632,
0.354516, -0.864050, -0.357403,
0.385011, -0.857377, -0.341572,
0.294703, -0.900244, -0.320484,
0.415435, -0.861701, -0.291352,
0.414707, -0.849744, -0.325506,
0.443956, -0.841094, -0.308973,
0.471716, -0.819222, -0.326128,
0.413488, -0.836680, -0.359158,
0.293939, -0.887823, -0.354076,
0.292822, -0.874225, -0.387280,
0.323903, -0.869647, -0.372561,
0.353281, -0.850133, -0.390470,
0.291383, -0.859517, -0.419912,
0.000000, -0.951910, -0.306378,
0.033041, -0.944047, -0.328151,
0.033021, -0.932091, -0.360716,
0.066055, -0.934620, -0.349460,
0.098882, -0.923674, -0.370201,
0.000000, -0.928448, -0.371461,
0.033014, -0.919041, -0.392778,
0.098950, -0.910112, -0.402375,
0.131782, -0.911107, -0.390534,
0.065943, -0.907978, -0.413796,
0.098843, -0.895411, -0.434130,
0.098590, -0.879653, -0.465287,
0.131555, -0.881402, -0.453679,
0.032936, -0.904730, -0.424710,
0.164179, -0.897140, -0.410104,
0.164016, -0.882055, -0.441675,
0.196334, -0.881675, -0.429071,
0.228090, -0.864787, -0.447345,
0.163706, -0.865919, -0.472637,
0.000000, -0.900543, -0.434767,
0.032928, -0.889480, -0.455785,
0.032906, -0.873128, -0.486380,
0.065827, -0.876914, -0.476118,
0.098538, -0.862905, -0.495666,
0.000000, -0.868328, -0.495991,
0.032898, -0.855762, -0.516324,
-0.032898, -0.855762, 0.516324,
-0.032906, -0.873128, 0.486380,
-0.065827, -0.876914, 0.476118,
-0.032928, -0.889480, 0.455785,
-0.032936, -0.904730, 0.424710,
-0.098538, -0.862905, 0.495666,
-0.098590, -0.879653, 0.465287,
-0.065943, -0.907978, 0.413796,
-0.033014, -0.919041, 0.392778,
-0.098843, -0.895411, 0.434130,
-0.098950, -0.910112, 0.402375,
-0.131782, -0.911107, 0.390534,
-0.098882, -0.923674, 0.370201,
-0.131555, -0.881402, 0.453679,
-0.033021, -0.932091, 0.360716,
-0.066055, -0.934620, 0.349460,
-0.033041, -0.944047, 0.328151,
-0.033048, -0.954845, 0.295262,
-0.098929, -0.936012, 0.337778,
-0.163706, -0.865919, 0.472637,
-0.164016, -0.882055, 0.441675,
-0.196334, -0.881675, 0.429071,
-0.164179, -0.897140, 0.410104,
-0.164167, -0.911094, 0.378097,
-0.228090, -0.864787, 0.447345,
-0.228338, -0.880251, 0.415958,
-0.066348, -0.957048, 0.282237,
-0.033305, -0.964909, 0.260463,
-0.099370, -0.947620, 0.303549,
-0.099650, -0.958014, 0.268847,
-0.132768, -0.957732, 0.255190,
-0.099744, -0.967138, 0.233871,
-0.132199, -0.936674, 0.324291,
-0.165941, -0.946748, 0.275921,
-0.166080, -0.956201, 0.241032,
-0.165615, -0.936027, 0.310525,
-0.198871, -0.934170, 0.296273,
-0.231386, -0.920064, 0.316139,
-0.231780, -0.931106, 0.281637,
-0.165129, -0.924096, 0.344643,
-0.198844, -0.953424, 0.226811,
-0.165671, -0.964409, 0.206081,
-0.231772, -0.940847, 0.247162,
-0.231436, -0.949405, 0.212293,
-0.263677, -0.944158, 0.197584,
-0.230749, -0.956736, 0.177234,
-0.264286, -0.926741, 0.267028,
-0.197527, -0.910128, 0.364213,
-0.229687, -0.894658, 0.383186,
-0.230712, -0.907953, 0.349848,
-0.263541, -0.904573, 0.335103,
-0.261444, -0.877768, 0.401461,
-0.033310, -0.973350, 0.226892,
-0.066626, -0.974649, 0.213589,
-0.033324, -0.980638, 0.192973,
-0.033328, -0.986732, 0.158898,
-0.099776, -0.974700, 0.200013,
-0.066698, -0.987169, 0.145079,
-0.033376, -0.991690, 0.124250,
-0.099996, -0.981097, 0.165681,
-0.100061, -0.986311, 0.131070,
-0.133245, -0.984157, 0.116967,
-0.099955, -0.990313, 0.096377,
-0.133106, -0.973502, 0.185946,
-0.033381, -0.995397, 0.089833,
-0.066765, -0.994881, 0.075862,
-0.033392, -0.997915, 0.055231,
-0.033396, -0.999229, 0.020640,
-0.099983, -0.993068, 0.061793,
-0.165925, -0.971058, 0.171797,
-0.198580, -0.967371, 0.157351,
-0.166025, -0.976527, 0.137228,
-0.165954, -0.980785, 0.102565,
-0.230901, -0.962455, 0.142705,
-0.291383, -0.859517, 0.419912,
-0.292822, -0.874225, 0.387280,
-0.323903, -0.869647, 0.372561,
-0.293939, -0.887823, 0.354076,
-0.294703, -0.900244, 0.320484,
-0.353281, -0.850133, 0.390470,
-0.354516, -0.864050, 0.357403,
-0.325609, -0.894920, 0.305118,
-0.295186, -0.911593, 0.286117,
-0.355602, -0.876818, 0.323632,
-0.356197, -0.888475, 0.289370,
-0.386299, -0.880940, 0.273346,
-0.356275, -0.898967, 0.254807,
-0.385011, -0.857377, 0.341572,
-0.295269, -0.921654, 0.251736,
-0.325939, -0.915445, 0.236060,
-0.295021, -0.930534, 0.216955,
-0.294425, -0.938191, 0.181964,
-0.356133, -0.908143, 0.220102,
-0.413488, -0.836680, 0.359158,
-0.414707, -0.849744, 0.325506,
-0.443956, -0.841094, 0.308973,
-0.415435, -0.861701, 0.291352,
-0.415646, -0.872496, 0.256883,
-0.471716, -0.819222, 0.326128,
-0.472596, -0.831470, 0.292081,
-0.066798, -0.997767, 0.000000,
-0.033396, -0.999229, -0.020640,
-0.100210, -0.994752, 0.020640,
-0.100210, -0.994752, -0.020640,
-0.133410, -0.990206, -0.041156,
-0.099983, -0.993068, -0.061793,
-0.133410, -0.990206, 0.041156,
-0.167078, -0.985728, -0.020642,
-0.166851, -0.984044, -0.061795,
-0.167078, -0.985728, 0.020642,
-0.200538, -0.979686, 0.000000,
-0.233570, -0.972121, 0.020642,
-0.233570, -0.972121, -0.020642,
-0.166851, -0.984044, 0.061795,
-0.199641, -0.976427, -0.082059,
-0.165954, -0.980785, -0.102565,
-0.233124, -0.970505, -0.061424,
-0.232233, -0.967277, -0.102196,
-0.264412, -0.956652, -0.122082,
-0.230901, -0.962455, -0.142705,
-0.266183, -0.963059, -0.040784,
-0.199641, -0.976427, 0.082059,
-0.232233, -0.967277, 0.102196,
-0.233124, -0.970505, 0.061424,
-0.266183, -0.963059, 0.040784,
-0.264412, -0.956652, 0.122082,
-0.299185, -0.948702, -0.102238,
-0.297852, -0.943879, -0.142749,
-0.300080, -0.951932, -0.061453,
-0.333520, -0.939160, -0.082120,
-0.367185, -0.928088, -0.061865,
-0.366279, -0.924826, -0.102647,
-0.300527, -0.953549, -0.020669,
-0.367868, -0.929648, -0.020669,
-0.400628, -0.915314, -0.041197,
-0.334425, -0.942422, 0.000000,
-0.367868, -0.929648, 0.020669,
-0.367185, -0.928088, 0.061865,
-0.400628, -0.915314, 0.041197,
-0.300527, -0.953549, 0.020669,
-0.432953, -0.901180, -0.020667,
-0.432270, -0.899620, -0.061863,
-0.432953, -0.901180, 0.020667,
-0.464610, -0.885516, 0.000000,
-0.495384, -0.868428, 0.020667,
-0.495384, -0.868428, -0.020667,
-0.432270, -0.899620, 0.061863,
-0.300080, -0.951932, 0.061453,
-0.299185, -0.948702, 0.102238,
-0.333520, -0.939160, 0.082120,
-0.366279, -0.924826, 0.102647,
-0.297852, -0.943879, 0.142749,
-0.329196, -0.930240, -0.162122,
-0.294425, -0.938191, -0.181964,
-0.363521, -0.920698, -0.142013,
-0.360134, -0.915124, -0.181249,
-0.390469, -0.898629, -0.199999,
-0.356133, -0.908143, -0.220102,
-0.397184, -0.909626, -0.121761,
-0.424534, -0.887388, -0.179759,
-0.420581, -0.880517, -0.218637,
-0.427873, -0.892852, -0.140501,
-0.457934, -0.874631, -0.159117,
-0.490456, -0.860438, -0.138204,
-0.487173, -0.855081, -0.177481,
-0.430583, -0.896869, -0.101119,
-0.449711, -0.861255, -0.236644,
-0.415646, -0.872496, -0.256883,
-0.483108, -0.848498, -0.216004,
-0.478242, -0.840609, -0.254286,
-0.505993, -0.818713, -0.271440,
-0.472596, -0.831470, -0.292081,
-0.515629, -0.834306, -0.195090,
-0.462906, -0.882736, -0.080590,
-0.494553, -0.867079, -0.059926,
-0.492908, -0.864409, -0.099190,
-0.522087, -0.844754, -0.117537,
-0.525326, -0.849995, -0.039260,
-0.329196, -0.930240, 0.162122,
-0.360134, -0.915124, 0.181249,
-0.363521, -0.920698, 0.142013,
-0.397184, -0.909626, 0.121761,
-0.390469, -0.898629, 0.199999,
-0.427873, -0.892852, 0.140501,
-0.430583, -0.896869, 0.101119,
-0.424534, -0.887388, 0.179759,
-0.457934, -0.874631, 0.159117,
-0.487173, -0.855081, 0.177481,
-0.490456, -0.860438, 0.138204,
-0.420581, -0.880517, 0.218637,
-0.462906, -0.882736, 0.080590,
-0.492908, -0.864409, 0.099190,
-0.494553, -0.867079, 0.059926,
-0.525326, -0.849995, 0.039260,
-0.522087, -0.844754, 0.117537,
-0.449711, -0.861255, 0.236644,
-0.478242, -0.840609, 0.254286,
-0.483108, -0.848498, 0.216004,
-0.515629, -0.834306, 0.195090,
-0.505993, -0.818713, 0.271440,
-0.033392, -0.997915, -0.055231,
-0.066765, -0.994881, -0.075862,
-0.033381, -0.995397, -0.089833,
-0.033376, -0.991690, -0.124250,
-0.099955, -0.990313, -0.096377,
-0.066698, -0.987169, -0.145079,
-0.033328, -0.986732, -0.158898,
-0.100061, -0.986311, -0.131070,
-0.099996, -0.981097, -0.165681,
-0.133106, -0.973502, -0.185946,
-0.099776, -0.974700, -0.200013,
-0.133245, -0.984157, -0.116967,
-0.033324, -0.980638, -0.192973,
-0.066626, -0.974649, -0.213589,
-0.033310, -0.973350, -0.226892,
-0.033305, -0.964909, -0.260463,
-0.099744, -0.967138, -0.233871,
-0.166025, -0.976527, -0.137228,
-0.198580, -0.967371, -0.157351,
-0.165925, -0.971058, -0.171797,
-0.165671, -0.964409, -0.206081,
-0.230749, -0.956736, -0.177234,
-0.066348, -0.957048, -0.282237,
-0.033048, -0.954845, -0.295262,
-0.099650, -0.958014, -0.268847,
-0.099370, -0.947620, -0.303549,
-0.132199, -0.936674, -0.324291,
-0.098929, -0.936012, -0.337778,
-0.132768, -0.957732, -0.255190,
-0.165615, -0.936027, -0.310525,
-0.165129, -0.924096, -0.344643,
-0.165941, -0.946748, -0.275921,
-0.198871, -0.934170, -0.296273,
-0.231780, -0.931106, -0.281637,
-0.231386, -0.920064, -0.316139,
-0.166080, -0.956201, -0.241032,
-0.197527, -0.910128, -0.364213,
-0.164167, -0.911094, -0.378097,
-0.230712, -0.907953, -0.349848,
-0.229687, -0.894658, -0.383186,
-0.261444, -0.877768, -0.401461,
-0.228338, -0.880251, -0.415958,
-0.263541, -0.904573, -0.335103,
-0.198844, -0.953424, -0.226811,
-0.231436, -0.949405, -0.212293,
-0.231772, -0.940847, -0.247162,
-0.264286, -0.926741, -0.267028,
-0.263677, -0.944158, -0.197584,
-0.033041, -0.944047, -0.328151,
-0.066055, -0.934620, -0.349460,
-0.033021, -0.932091, -0.360716,
-0.033014, -0.919041, -0.392778,
-0.098882, -0.923674, -0.370201,
-0.065943, -0.907978, -0.413796,
-0.032936, -0.904730, -0.424710,
-0.098950, -0.910112, -0.402375,
-0.098843, -0.895411, -0.434130,
-0.131555, -0.881402, -0.453679,
-0.098590, -0.879653, -0.465287,
-0.131782, -0.911107, -0.390534,
-0.032928, -0.889480, -0.455785,
-0.065827, -0.876914, -0.476118,
-0.032906, -0.873128, -0.486380,
-0.032898, -0.855762, -0.516324,
-0.098538, -0.862905, -0.495666,
-0.164179, -0.897140, -0.410104,
-0.196334, -0.881675, -0.429071,
-0.164016, -0.882055, -0.441675,
-0.163706, -0.865919, -0.472637,
-0.228090, -0.864787, -0.447345,
-0.295021, -0.930534, -0.216955,
-0.325939, -0.915445, -0.236060,
-0.295269, -0.921654, -0.251736,
-0.295186, -0.911593, -0.286117,
-0.356275, -0.898967, -0.254807,
-0.325609, -0.894920, -0.305118,
-0.294703, -0.900244, -0.320484,
-0.356197, -0.888475, -0.289370,
-0.355602, -0.876818, -0.323632,
-0.385011, -0.857377, -0.341572,
-0.354516, -0.864050, -0.357403,
-0.386299, -0.880940, -0.273346,
-0.293939, -0.887823, -0.354076,
-0.323903, -0.869647, -0.372561,
-0.292822, -0.874225, -0.387280,
-0.291383, -0.859517, -0.419912,
-0.353281, -0.850133, -0.390470,
-0.415435, -0.861701, -0.291352,
-0.443956, -0.841094, -0.308973,
-0.414707, -0.849744, -0.325506,
-0.413488, -0.836680, -0.359158,
-0.471716, -0.819222, -0.326128,
-0.527686, -0.797844, 0.291536,
-0.533322, -0.806959, 0.253742,
-0.560060, -0.794224, 0.235673,
-0.538182, -0.814815, 0.215489,
-0.542239, -0.821376, 0.176970,
-0.581130, -0.772646, 0.255549,
-0.586051, -0.780587, 0.217322,
-0.568272, -0.807470, 0.158301,
-0.545532, -0.826677, 0.137840,
-0.590346, -0.787154, 0.178553,
-0.593687, -0.792524, 0.139437,
-0.618331, -0.776630, 0.120472,
-0.596053, -0.796672, 0.100171,
-0.611619, -0.765845, 0.198505,
-0.547978, -0.830634, 0.098832,
-0.573242, -0.815477, 0.079939,
-0.549620, -0.833286, 0.059604,
-0.550447, -0.834622, 0.020351,
-0.597732, -0.799376, 0.060948,
-0.631794, -0.743751, 0.218339,
-0.636140, -0.750382, 0.179588,
-0.659941, -0.733998, 0.160388,
-0.639533, -0.755818, 0.140487,
-0.641952, -0.760032, 0.101232,
-0.679433, -0.711295, 0.180084,
-0.682882, -0.716793, 0.140998,
-0.574934, -0.818200, 0.000000,
-0.550447, -0.834622, -0.020351,
-0.598760, -0.800670, 0.020351,
-0.598760, -0.800670, -0.020351,
-0.621778, -0.782140, -0.040604,
-0.597732, -0.799376, -0.060948,
-0.621778, -0.782140, 0.040604,
-0.645320, -0.763641, -0.020352,
-0.644291, -0.762348, -0.060949,
-0.645320, -0.763641, 0.020352,
-0.668055, -0.744112, 0.000000,
-0.689842, -0.723674, 0.020352,
-0.689842, -0.723674, -0.020352,
-0.644291, -0.762348, 0.060949,
-0.665716, -0.741796, -0.081001,
-0.641952, -0.760032, -0.101232,
-0.688679, -0.722524, -0.060664,
-0.686355, -0.720226, -0.100949,
-0.706082, -0.697769, -0.120696,
-0.682882, -0.716793, -0.140998,
-0.710703, -0.702335, -0.040319,
-0.665716, -0.741796, 0.081001,
-0.686355, -0.720226, 0.100949,
-0.688679, -0.722524, 0.060664,
-0.710703, -0.702335, 0.040319,
-0.706082, -0.697769, 0.120696,
-0.549620, -0.833286, -0.059604,
-0.573242, -0.815477, -0.079939,
-0.547978, -0.830634, -0.098832,
-0.545532, -0.826677, -0.137840,
-0.596053, -0.796672, -0.100171,
-0.568272, -0.807470, -0.158301,
-0.542239, -0.821376, -0.176970,
-0.593687, -0.792524, -0.139437,
-0.590346, -0.787154, -0.178553,
-0.611619, -0.765845, -0.198505,
-0.586051, -0.780587, -0.217322,
-0.618331, -0.776630, -0.120472,
-0.538182, -0.814815, -0.215489,
-0.560060, -0.794224, -0.235673,
-0.533322, -0.806959, -0.253742,
-0.527686, -0.797844, -0.291536,
-0.581130, -0.772646, -0.255549,
-0.639533, -0.755818, -0.140487,
-0.659941, -0.733998, -0.160388,
-0.636140, -0.750382, -0.179588,
-0.631794, -0.743751, -0.218339,
-0.679433, -0.711295, -0.180084,
-0.723820, -0.675435, 0.140967,
-0.727293, -0.678867, 0.100920,
-0.747667, -0.659120, 0.080959,
-0.729617, -0.681162, 0.060644,
-0.730779, -0.682311, 0.020333,
-0.764743, -0.636343, 0.101176,
-0.767082, -0.638652, 0.060901,
-0.750007, -0.661430, 0.000000,
-0.730779, -0.682311, -0.020333,
-0.768380, -0.639671, 0.020333,
-0.768380, -0.639671, -0.020333,
-0.785785, -0.617167, -0.040577,
-0.767082, -0.638652, -0.060901,
-0.785785, -0.617167, 0.040577,
-0.729617, -0.681162, -0.060644,
-0.747667, -0.659120, -0.080959,
-0.727293, -0.678867, -0.100920,
-0.723820, -0.675435, -0.140967,
-0.764743, -0.636343, -0.101176,
-0.802005, -0.594204, 0.060900,
-0.803303, -0.595224, 0.020332,
-0.819885, -0.572529, 0.000000,
-0.803303, -0.595224, -0.020332,
-0.802005, -0.594204, -0.060900,
-0.835429, -0.549222, 0.020332,
-0.835429, -0.549222, -0.020332,
0.053243, 0.840222, 0.539623,
0.073611, 0.823654, 0.562295,
0.086178, 0.843986, 0.529397,
0.118945, 0.846717, 0.518577,
0.093867, 0.806140, 0.584232,
0.139598, 0.829468, 0.540828,
0.151959, 0.848387, 0.507097,
0.127031, 0.809135, 0.573728,
0.160105, 0.811161, 0.562480,
0.180328, 0.791892, 0.583429,
0.192895, 0.812225, 0.550529,
0.114320, 0.787486, 0.605637,
0.184424, 0.849034, 0.495105,
0.205003, 0.831085, 0.516983,
0.216754, 0.848634, 0.482533,
0.248763, 0.847202, 0.469431,
0.225309, 0.812164, 0.538168,
0.134388, 0.768075, 0.626099,
0.154382, 0.747714, 0.645826,
0.167523, 0.770441, 0.615107,
0.200390, 0.771845, 0.603406,
0.174169, 0.726507, 0.664720,
0.270692, 0.828310, 0.490539,
0.282027, 0.844463, 0.455349,
0.258941, 0.810757, 0.524997,
0.292299, 0.808303, 0.511084,
0.313441, 0.787296, 0.530962,
0.325185, 0.804829, 0.496492,
0.246832, 0.791897, 0.558545,
0.301428, 0.768229, 0.564770,
0.334423, 0.765026, 0.550360,
0.267971, 0.770411, 0.578496,
0.288850, 0.747877, 0.597700,
0.275772, 0.726347, 0.629580,
0.309328, 0.724422, 0.616060,
0.234254, 0.771546, 0.591475,
0.354664, 0.742003, 0.568898,
0.366677, 0.761070, 0.535090,
0.342087, 0.721652, 0.601826,
0.374471, 0.717969, 0.586764,
0.393711, 0.693058, 0.603872,
0.406288, 0.713409, 0.570944,
0.329009, 0.700122, 0.633705,
0.221448, 0.750093, 0.623154,
0.208302, 0.727361, 0.653878,
0.242237, 0.727302, 0.642147,
0.262621, 0.703597, 0.660289,
0.194886, 0.703465, 0.683488,
0.313409, 0.840916, 0.441176,
0.335116, 0.821198, 0.461879,
0.344498, 0.836323, 0.426480,
0.375114, 0.830721, 0.411329,
0.356369, 0.800468, 0.481926,
0.396824, 0.810120, 0.431551,
0.405639, 0.824001, 0.395576,
0.387410, 0.794924, 0.466914,
0.418038, 0.788414, 0.451273,
0.438625, 0.765722, 0.470402,
0.448074, 0.780985, 0.435077,
0.377449, 0.778488, 0.501486,
0.435338, 0.816363, 0.379516,
0.456698, 0.794935, 0.399381,
0.464593, 0.807701, 0.363003,
0.493236, 0.798073, 0.346117,
0.477439, 0.772513, 0.418660,
0.397816, 0.755728, 0.520209,
0.417751, 0.731954, 0.538265,
0.428549, 0.749420, 0.504694,
0.458698, 0.742189, 0.488621,
0.437129, 0.707290, 0.555570,
0.214284, 0.680528, 0.700689,
0.233506, 0.656735, 0.717059,
0.248290, 0.680710, 0.689192,
0.281913, 0.679996, 0.676853,
0.252431, 0.632217, 0.732517,
0.301233, 0.655487, 0.692528,
0.315540, 0.678317, 0.663567,
0.286472, 0.631567, 0.720456,
0.320203, 0.630114, 0.707408,
0.338701, 0.604015, 0.721421,
0.353435, 0.627882, 0.693432,
0.271326, 0.606679, 0.747210,
0.348411, 0.675790, 0.649552,
0.367467, 0.650587, 0.664608,
0.380913, 0.672350, 0.634705,
0.412866, 0.668030, 0.619094,
0.386056, 0.624652, 0.678801,
0.289684, 0.580690, 0.760843,
0.307772, 0.553959, 0.773567,
0.323518, 0.579446, 0.748050,
0.356867, 0.577422, 0.734322,
0.325477, 0.526633, 0.785317,
0.519525, 0.773811, 0.362368,
0.526633, 0.785317, 0.325477,
0.511644, 0.761045, 0.398787,
0.545041, 0.748288, 0.378147,
0.569625, 0.721670, 0.393344,
0.577422, 0.734322, 0.356867,
0.503033, 0.747106, 0.434500,
0.561193, 0.707553, 0.429454,
0.593575, 0.693587, 0.408174,
0.527794, 0.720310, 0.450096,
0.551813, 0.692313, 0.464979,
0.541538, 0.676056, 0.499685,
0.574937, 0.663299, 0.479043,
0.493588, 0.731779, 0.469969,
0.616293, 0.664768, 0.422216,
0.624652, 0.678801, 0.386056,
0.606985, 0.649612, 0.457792,
0.638080, 0.634846, 0.435687,
0.658800, 0.604016, 0.448494,
0.668030, 0.619094, 0.412866,
0.596780, 0.633437, 0.492556,
0.483519, 0.715483, 0.504275,
0.472724, 0.698007, 0.537883,
0.507599, 0.687574, 0.519216,
0.530789, 0.658643, 0.533341,
0.461267, 0.679467, 0.570576,
0.649517, 0.586330, 0.484092,
0.679467, 0.570576, 0.461267,
0.618412, 0.601099, 0.506207,
0.639079, 0.567660, 0.518980,
0.627561, 0.548129, 0.552922,
0.658643, 0.533341, 0.530789,
0.586364, 0.614787, 0.527461,
0.595223, 0.561780, 0.574554,
0.614787, 0.527461, 0.586364,
0.574554, 0.595223, 0.561780,
0.561780, 0.574554, 0.595223,
0.527461, 0.586364, 0.614787,
0.548129, 0.552922, 0.627561,
0.552922, 0.627561, 0.548129,
0.601099, 0.506207, 0.618412,
0.633437, 0.492556, 0.596780,
0.567660, 0.518980, 0.639079,
0.586330, 0.484092, 0.649517,
0.570576, 0.461267, 0.679467,
0.604016, 0.448494, 0.658800,
0.533341, 0.530789, 0.658643,
0.518980, 0.639079, 0.567660,
0.484092, 0.649517, 0.586330,
0.506207, 0.618412, 0.601099,
0.492556, 0.596780, 0.633437,
0.448494, 0.658800, 0.604016,
0.698007, 0.537883, 0.472724,
0.707290, 0.555570, 0.437129,
0.687574, 0.519216, 0.507599,
0.715483, 0.504275, 0.483519,
0.731779, 0.469969, 0.493588,
0.742189, 0.488621, 0.458698,
0.676056, 0.499685, 0.541538,
0.720310, 0.450096, 0.527794,
0.747106, 0.434500, 0.503033,
0.692313, 0.464979, 0.551813,
0.707553, 0.429454, 0.561193,
0.693587, 0.408174, 0.593575,
0.721670, 0.393344, 0.569625,
0.663299, 0.479043, 0.574937,
0.761045, 0.398787, 0.511644,
0.772513, 0.418660, 0.477439,
0.748288, 0.378147, 0.545041,
0.773811, 0.362368, 0.519525,
0.785317, 0.325477, 0.526633,
0.798073, 0.346117, 0.493236,
0.734322, 0.356867, 0.577422,
0.649612, 0.457792, 0.606985,
0.634846, 0.435687, 0.638080,
0.664768, 0.422216, 0.616293,
0.678801, 0.386056, 0.624652,
0.619094, 0.412866, 0.668030,
0.435687, 0.638080, 0.634846,
0.422216, 0.616293, 0.664768,
0.457792, 0.606985, 0.649612,
0.479043, 0.574937, 0.663299,
0.408174, 0.593575, 0.693587,
0.464979, 0.551813, 0.692313,
0.499685, 0.541538, 0.676056,
0.429454, 0.561193, 0.707553,
0.450096, 0.527794, 0.720310,
0.434500, 0.503033, 0.747106,
0.469969, 0.493588, 0.731779,
0.393344, 0.569625, 0.721670,
0.519216, 0.507599, 0.687574,
0.504275, 0.483519, 0.715483,
0.537883, 0.472724, 0.698007,
0.555570, 0.437129, 0.707290,
0.488621, 0.458698, 0.742189,
0.378147, 0.545041, 0.748288,
0.362368, 0.519525, 0.773811,
0.398787, 0.511644, 0.761045,
0.418660, 0.477439, 0.772513,
0.346117, 0.493236, 0.798073,
0.553959, 0.773567, 0.307772,
0.579446, 0.748050, 0.323518,
0.580690, 0.760843, 0.289684,
0.606679, 0.747210, 0.271326,
0.604015, 0.721421, 0.338701,
0.631567, 0.720456, 0.286472,
0.632217, 0.732517, 0.252431,
0.630114, 0.707408, 0.320203,
0.655487, 0.692528, 0.301233,
0.678317, 0.663567, 0.315540,
0.679996, 0.676853, 0.281913,
0.627882, 0.693432, 0.353435,
0.656735, 0.717059, 0.233506,
0.680710, 0.689192, 0.248290,
0.680528, 0.700689, 0.214284,
0.703465, 0.683488, 0.194886,
0.703597, 0.660289, 0.262621,
0.650587, 0.664608, 0.367467,
0.672350, 0.634705, 0.380913,
0.675790, 0.649552, 0.348411,
0.700122, 0.633705, 0.329009,
0.693058, 0.603872, 0.393711,
0.727361, 0.653878, 0.208302,
0.726507, 0.664720, 0.174169,
0.727302, 0.642147, 0.242237,
0.750093, 0.623154, 0.221448,
0.771546, 0.591475, 0.234254,
0.771845, 0.603406, 0.200390,
0.726347, 0.629580, 0.275772,
0.770411, 0.578496, 0.267971,
0.791897, 0.558545, 0.246832,
0.747877, 0.597700, 0.288850,
0.768229, 0.564770, 0.301428,
0.765026, 0.550360, 0.334423,
0.787296, 0.530962, 0.313441,
0.724422, 0.616060, 0.309328,
0.810757, 0.524997, 0.258941,
0.812164, 0.538168, 0.225309,
0.808303, 0.511084, 0.292299,
0.828310, 0.490539, 0.270692,
0.844463, 0.455349, 0.282027,
0.847202, 0.469431, 0.248763,
0.804829, 0.496492, 0.325185,
0.721652, 0.601826, 0.342087,
0.717969, 0.586764, 0.374471,
0.742003, 0.568898, 0.354664,
0.761070, 0.535090, 0.366677,
0.713409, 0.570944, 0.406288,
0.747714, 0.645826, 0.154382,
0.770441, 0.615107, 0.167523,
0.768075, 0.626099, 0.134388,
0.787486, 0.605637, 0.114320,
0.791892, 0.583429, 0.180328,
0.809135, 0.573728, 0.127031,
0.806140, 0.584232, 0.093867,
0.811161, 0.562480, 0.160105,
0.829468, 0.540828, 0.139598,
0.848387, 0.507097, 0.151959,
0.846717, 0.518577, 0.118945,
0.812225, 0.550529, 0.192895,
0.823654, 0.562295, 0.073611,
0.843986, 0.529397, 0.086178,
0.840222, 0.539623, 0.053243,
0.831085, 0.516983, 0.205003,
0.848634, 0.482533, 0.216754,
0.849034, 0.495105, 0.184424,
0.731954, 0.538265, 0.417751,
0.749420, 0.504694, 0.428549,
0.755728, 0.520209, 0.397816,
0.778488, 0.501486, 0.377449,
0.765722, 0.470402, 0.438625,
0.794924, 0.466914, 0.387410,
0.800468, 0.481926, 0.356369,
0.788414, 0.451273, 0.418038,
0.810120, 0.431551, 0.396824,
0.824001, 0.395576, 0.405639,
0.830721, 0.411329, 0.375114,
0.780985, 0.435077, 0.448074,
0.821198, 0.461879, 0.335116,
0.836323, 0.426480, 0.344498,
0.840916, 0.441176, 0.313409,
0.794935, 0.399381, 0.456698,
0.807701, 0.363003, 0.464593,
0.816363, 0.379516, 0.435338,
0.363003, 0.464593, 0.807701,
0.379516, 0.435338, 0.816363,
0.399381, 0.456698, 0.794935,
0.435077, 0.448074, 0.780985,
0.395576, 0.405639, 0.824001,
0.451273, 0.418038, 0.788414,
0.470402, 0.438625, 0.765722,
0.431551, 0.396824, 0.810120,
0.466914, 0.387410, 0.794924,
0.481926, 0.356369, 0.800468,
0.501486, 0.377449, 0.778488,
0.411329, 0.375114, 0.830721,
0.504694, 0.428549, 0.749420,
0.520209, 0.397816, 0.755728,
0.538265, 0.417751, 0.731954,
0.570944, 0.406288, 0.713409,
0.535090, 0.366677, 0.761070,
0.426480, 0.344498, 0.836323,
0.441176, 0.313409, 0.840916,
0.461879, 0.335116, 0.821198,
0.496492, 0.325185, 0.804829,
0.455349, 0.282027, 0.844463,
0.586764, 0.374471, 0.717969,
0.603872, 0.393711, 0.693058,
0.568898, 0.354664, 0.742003,
0.601826, 0.342087, 0.721652,
0.616060, 0.309328, 0.724422,
0.633705, 0.329009, 0.700122,
0.550360, 0.334423, 0.765026,
0.597700, 0.288850, 0.747877,
0.629580, 0.275772, 0.726347,
0.564770, 0.301428, 0.768229,
0.578496, 0.267971, 0.770411,
0.558545, 0.246832, 0.791897,
0.591475, 0.234254, 0.771546,
0.530962, 0.313441, 0.787296,
0.642147, 0.242237, 0.727302,
0.660289, 0.262621, 0.703597,
0.623154, 0.221448, 0.750093,
0.653878, 0.208302, 0.727361,
0.664720, 0.174169, 0.726507,
0.683488, 0.194886, 0.703465,
0.603406, 0.200390, 0.771845,
0.511084, 0.292299, 0.808303,
0.490539, 0.270692, 0.828310,
0.524997, 0.258941, 0.810757,
0.538168, 0.225309, 0.812164,
0.469431, 0.248763, 0.847202,
0.634705, 0.380913, 0.672350,
0.649552, 0.348411, 0.675790,
0.664608, 0.367467, 0.650587,
0.693432, 0.353435, 0.627882,
0.663567, 0.315540, 0.678317,
0.707408, 0.320203, 0.630114,
0.721421, 0.338701, 0.604015,
0.692528, 0.301233, 0.655487,
0.720456, 0.286472, 0.631567,
0.732517, 0.252431, 0.632217,
0.747210, 0.271326, 0.606679,
0.676853, 0.281913, 0.679996,
0.748050, 0.323518, 0.579446,
0.760843, 0.289684, 0.580690,
0.773567, 0.307772, 0.553959,
0.689192, 0.248290, 0.680710,
0.700689, 0.214284, 0.680528,
0.717059, 0.233506, 0.656735,
0.482533, 0.216754, 0.848634,
0.495105, 0.184424, 0.849034,
0.516983, 0.205003, 0.831085,
0.550529, 0.192895, 0.812225,
0.507097, 0.151959, 0.848387,
0.562480, 0.160105, 0.811161,
0.583429, 0.180328, 0.791892,
0.540828, 0.139598, 0.829468,
0.573728, 0.127031, 0.809135,
0.584232, 0.093867, 0.806140,
0.605637, 0.114320, 0.787486,
0.518577, 0.118945, 0.846717,
0.615107, 0.167523, 0.770441,
0.626099, 0.134388, 0.768075,
0.645826, 0.154382, 0.747714,
0.529397, 0.086178, 0.843986,
0.539623, 0.053243, 0.840222,
0.562295, 0.073611, 0.823654,
0.053243, 0.840222, -0.539623,
0.086178, 0.843986, -0.529397,
0.073611, 0.823654, -0.562295,
0.093867, 0.806140, -0.584232,
0.118945, 0.846717, -0.518577,
0.127031, 0.809135, -0.573728,
0.114320, 0.787486, -0.605637,
0.139598, 0.829468, -0.540828,
0.160105, 0.811161, -0.562480,
0.192895, 0.812225, -0.550529,
0.180328, 0.791892, -0.583429,
0.151959, 0.848387, -0.507097,
0.134388, 0.768075, -0.626099,
0.167523, 0.770441, -0.615107,
0.154382, 0.747714, -0.645826,
0.174169, 0.726507, -0.664720,
0.200390, 0.771845, -0.603406,
0.184424, 0.849034, -0.495105,
0.216754, 0.848634, -0.482533,
0.205003, 0.831085, -0.516983,
0.225309, 0.812164, -0.538168,
0.248763, 0.847202, -0.469431,
0.208302, 0.727361, -0.653878,
0.194886, 0.703465, -0.683488,
0.221448, 0.750093, -0.623154,
0.242237, 0.727302, -0.642147,
0.275772, 0.726347, -0.629580,
0.262621, 0.703597, -0.660289,
0.234254, 0.771546, -0.591475,
0.288850, 0.747877, -0.597700,
0.309328, 0.724422, -0.616060,
0.267971, 0.770411, -0.578496,
0.301428, 0.768229, -0.564770,
0.313441, 0.787296, -0.530962,
0.334423, 0.765026, -0.550360,
0.246832, 0.791897, -0.558545,
0.342087, 0.721652, -0.601826,
0.329009, 0.700122, -0.633705,
0.354664, 0.742003, -0.568898,
0.374471, 0.717969, -0.586764,
0.406288, 0.713409, -0.570944,
0.393711, 0.693058, -0.603872,
0.366677, 0.761070, -0.535090,
0.258941, 0.810757, -0.524997,
0.270692, 0.828310, -0.490539,
0.292299, 0.808303, -0.511084,
0.325185, 0.804829, -0.496492,
0.282027, 0.844463, -0.455349,
0.214284, 0.680528, -0.700689,
0.248290, 0.680710, -0.689192,
0.233506, 0.656735, -0.717059,
0.252431, 0.632217, -0.732517,
0.281913, 0.679996, -0.676853,
0.286472, 0.631567, -0.720456,
0.271326, 0.606679, -0.747210,
0.301233, 0.655487, -0.692528,
0.320203, 0.630114, -0.707408,
0.353435, 0.627882, -0.693432,
0.338701, 0.604015, -0.721421,
0.315540, 0.678317, -0.663567,
0.289684, 0.580690, -0.760843,
0.323518, 0.579446, -0.748050,
0.307772, 0.553959, -0.773567,
0.325477, 0.526633, -0.785317,
0.356867, 0.577422, -0.734322,
0.348411, 0.675790, -0.649552,
0.380913, 0.672350, -0.634705,
0.367467, 0.650587, -0.664608,
0.386056, 0.624652, -0.678801,
0.412866, 0.668030, -0.619094,
0.313409, 0.840916, -0.441176,
0.344498, 0.836323, -0.426480,
0.335116, 0.821198, -0.461879,
0.356369, 0.800468, -0.481926,
0.375114, 0.830721, -0.411329,
0.387410, 0.794924, -0.466914,
0.377449, 0.778488, -0.501486,
0.396824, 0.810120, -0.431551,
0.418038, 0.788414, -0.451273,
0.448074, 0.780985, -0.435077,
0.438625, 0.765722, -0.470402,
0.405639, 0.824001, -0.395576,
0.397816, 0.755728, -0.520209,
0.428549, 0.749420, -0.504694,
0.417751, 0.731954, -0.538265,
0.437129, 0.707290, -0.555570,
0.458698, 0.742189, -0.488621,
0.435338, 0.816363, -0.379516,
0.464593, 0.807701, -0.363003,
0.456698, 0.794935, -0.399381,
0.477439, 0.772513, -0.418660,
0.493236, 0.798073, -0.346117,
0.362368, 0.519525, -0.773811,
0.346117, 0.493236, -0.798073,
0.378147, 0.545041, -0.748288,
0.398787, 0.511644, -0.761045,
0.434500, 0.503033, -0.747106,
0.418660, 0.477439, -0.772513,
0.393344, 0.569625, -0.721670,
0.450096, 0.527794, -0.720310,
0.469969, 0.493588, -0.731779,
0.429454, 0.561193, -0.707553,
0.464979, 0.551813, -0.692313,
0.479043, 0.574937, -0.663299,
0.499685, 0.541538, -0.676056,
0.408174, 0.593575, -0.693587,
0.504275, 0.483519, -0.715483,
0.488621, 0.458698, -0.742189,
0.519216, 0.507599, -0.687574,
0.537883, 0.472724, -0.698007,
0.570576, 0.461267, -0.679467,
0.555570, 0.437129, -0.707290,
0.533341, 0.530789, -0.658643,
0.422216, 0.616293, -0.664768,
0.435687, 0.638080, -0.634846,
0.457792, 0.606985, -0.649612,
0.492556, 0.596780, -0.633437,
0.448494, 0.658800, -0.604016,
0.586330, 0.484092, -0.649517,
0.604016, 0.448494, -0.658800,
0.567660, 0.518980, -0.639079,
0.601099, 0.506207, -0.618412,
0.614787, 0.527461, -0.586364,
0.633437, 0.492556, -0.596780,
0.548129, 0.552922, -0.627561,
0.595223, 0.561780, -0.574554,
0.627561, 0.548129, -0.552922,
0.561780, 0.574554, -0.595223,
0.574554, 0.595223, -0.561780,
0.552922, 0.627561, -0.548129,
0.586364, 0.614787, -0.527461,
0.527461, 0.586364, -0.614787,
0.639079, 0.567660, -0.518980,
0.658643, 0.533341, -0.530789,
0.618412, 0.601099, -0.506207,
0.649517, 0.586330, -0.484092,
0.658800, 0.604016, -0.448494,
0.679467, 0.570576, -0.461267,
0.596780, 0.633437, -0.492556,
0.506207, 0.618412, -0.601099,
0.484092, 0.649517, -0.586330,
0.518980, 0.639079, -0.567660,
0.530789, 0.658643, -0.533341,
0.461267, 0.679467, -0.570576,
0.634846, 0.435687, -0.638080,
0.619094, 0.412866, -0.668030,
0.649612, 0.457792, -0.606985,
0.664768, 0.422216, -0.616293,
0.693587, 0.408174, -0.593575,
0.678801, 0.386056, -0.624652,
0.663299, 0.479043, -0.574937,
0.707553, 0.429454, -0.561193,
0.721670, 0.393344, -0.569625,
0.692313, 0.464979, -0.551813,
0.720310, 0.450096, -0.527794,
0.731779, 0.469969, -0.493588,
0.747106, 0.434500, -0.503033,
0.676056, 0.499685, -0.541538,
0.748288, 0.378147, -0.545041,
0.734322, 0.356867, -0.577422,
0.761045, 0.398787, -0.511644,
0.773811, 0.362368, -0.519525,
0.798073, 0.346117, -0.493236,
0.785317, 0.325477, -0.526633,
0.772513, 0.418660, -0.477439,
0.687574, 0.519216, -0.507599,
0.698007, 0.537883, -0.472724,
0.715483, 0.504275, -0.483519,
0.742189, 0.488621, -0.458698,
0.707290, 0.555570, -0.437129,
0.472724, 0.698007, -0.537883,
0.483519, 0.715483, -0.504275,
0.507599, 0.687574, -0.519216,
0.541538, 0.676056, -0.499685,
0.493588, 0.731779, -0.469969,
0.551813, 0.692313, -0.464979,
0.574937, 0.663299, -0.479043,
0.527794, 0.720310, -0.450096,
0.561193, 0.707553, -0.429454,
0.569625, 0.721670, -0.393344,
0.593575, 0.693587, -0.408174,
0.503033, 0.747106, -0.434500,
0.606985, 0.649612, -0.457792,
0.616293, 0.664768, -0.422216,
0.638080, 0.634846, -0.435687,
0.668030, 0.619094, -0.412866,
0.624652, 0.678801, -0.386056,
0.511644, 0.761045, -0.398787,
0.519525, 0.773811, -0.362368,
0.545041, 0.748288, -0.378147,
0.577422, 0.734322, -0.356867,
0.526633, 0.785317, -0.325477,
0.363003, 0.464593, -0.807701,
0.399381, 0.456698, -0.794935,
0.379516, 0.435338, -0.816363,
0.395576, 0.405639, -0.824001,
0.435077, 0.448074, -0.780985,
0.431551, 0.396824, -0.810120,
0.411329, 0.375114, -0.830721,
0.451273, 0.418038, -0.788414,
0.466914, 0.387410, -0.794924,
0.501486, 0.377449, -0.778488,
0.481926, 0.356369, -0.800468,
0.470402, 0.438625, -0.765722,
0.426480, 0.344498, -0.836323,
0.461879, 0.335116, -0.821198,
0.441176, 0.313409, -0.840916,
0.455349, 0.282027, -0.844463,
0.496492, 0.325185, -0.804829,
0.504694, 0.428549, -0.749420,
0.538265, 0.417751, -0.731954,
0.520209, 0.397816, -0.755728,
0.535090, 0.366677, -0.761070,
0.570944, 0.406288, -0.713409,
0.490539, 0.270692, -0.828310,
0.469431, 0.248763, -0.847202,
0.511084, 0.292299, -0.808303,
0.524997, 0.258941, -0.810757,
0.558545, 0.246832, -0.791897,
0.538168, 0.225309, -0.812164,
0.530962, 0.313441, -0.787296,
0.578496, 0.267971, -0.770411,
0.591475, 0.234254, -0.771546,
0.564770, 0.301428, -0.768229,
0.597700, 0.288850, -0.747877,
0.616060, 0.309328, -0.724422,
0.629580, 0.275772, -0.726347,
0.550360, 0.334423, -0.765026,
0.623154, 0.221448, -0.750093,
0.603406, 0.200390, -0.771845,
0.642147, 0.242237, -0.727302,
0.653878, 0.208302, -0.727361,
0.683488, 0.194886, -0.703465,
0.664720, 0.174169, -0.726507,
0.660289, 0.262621, -0.703597,
0.568898, 0.354664, -0.742003,
0.586764, 0.374471, -0.717969,
0.601826, 0.342087, -0.721652,
0.633705, 0.329009, -0.700122,
0.603872, 0.393711, -0.693058,
0.482533, 0.216754, -0.848634,
0.516983, 0.205003, -0.831085,
0.495105, 0.184424, -0.849034,
0.507097, 0.151959, -0.848387,
0.550529, 0.192895, -0.812225,
0.540828, 0.139598, -0.829468,
0.518577, 0.118945, -0.846717,
0.562480, 0.160105, -0.811161,
0.573728, 0.127031, -0.809135,
0.605637, 0.114320, -0.787486,
0.584232, 0.093867, -0.806140,
0.583429, 0.180328, -0.791892,
0.529397, 0.086178, -0.843986,
0.562295, 0.073611, -0.823654,
0.539623, 0.053243, -0.840222,
0.615107, 0.167523, -0.770441,
0.645826, 0.154382, -0.747714,
0.626099, 0.134388, -0.768075,
0.634705, 0.380913, -0.672350,
0.664608, 0.367467, -0.650587,
0.649552, 0.348411, -0.675790,
0.663567, 0.315540, -0.678317,
0.693432, 0.353435, -0.627882,
0.692528, 0.301233, -0.655487,
0.676853, 0.281913, -0.679996,
0.707408, 0.320203, -0.630114,
0.720456, 0.286472, -0.631567,
0.747210, 0.271326, -0.606679,
0.732517, 0.252431, -0.632217,
0.721421, 0.338701, -0.604015,
0.689192, 0.248290, -0.680710,
0.717059, 0.233506, -0.656735,
0.700689, 0.214284, -0.680528,
0.748050, 0.323518, -0.579446,
0.773567, 0.307772, -0.553959,
0.760843, 0.289684, -0.580690,
0.553959, 0.773567, -0.307772,
0.580690, 0.760843, -0.289684,
0.579446, 0.748050, -0.323518,
0.604015, 0.721421, -0.338701,
0.606679, 0.747210, -0.271326,
0.630114, 0.707408, -0.320203,
0.627882, 0.693432, -0.353435,
0.631567, 0.720456, -0.286472,
0.655487, 0.692528, -0.301233,
0.679996, 0.676853, -0.281913,
0.678317, 0.663567, -0.315540,
0.632217, 0.732517, -0.252431,
0.650587, 0.664608, -0.367467,
0.675790, 0.649552, -0.348411,
0.672350, 0.634705, -0.380913,
0.693058, 0.603872, -0.393711,
0.700122, 0.633705, -0.329009,
0.656735, 0.717059, -0.233506,
0.680528, 0.700689, -0.214284,
0.680710, 0.689192, -0.248290,
0.703597, 0.660289, -0.262621,
0.703465, 0.683488, -0.194886,
0.717969, 0.586764, -0.374471,
0.713409, 0.570944, -0.406288,
0.721652, 0.601826, -0.342087,
0.742003, 0.568898, -0.354664,
0.765026, 0.550360, -0.334423,
0.761070, 0.535090, -0.366677,
0.724422, 0.616060, -0.309328,
0.768229, 0.564770, -0.301428,
0.787296, 0.530962, -0.313441,
0.747877, 0.597700, -0.288850,
0.770411, 0.578496, -0.267971,
0.771546, 0.591475, -0.234254,
0.791897, 0.558545, -0.246832,
0.726347, 0.629580, -0.275772,
0.808303, 0.511084, -0.292299,
0.804829, 0.496492, -0.325185,
0.810757, 0.524997, -0.258941,
0.828310, 0.490539, -0.270692,
0.847202, 0.469431, -0.248763,
0.844463, 0.455349, -0.282027,
0.812164, 0.538168, -0.225309,
0.727302, 0.642147, -0.242237,
0.727361, 0.653878, -0.208302,
0.750093, 0.623154, -0.221448,
0.771845, 0.603406, -0.200390,
0.726507, 0.664720, -0.174169,
0.731954, 0.538265, -0.417751,
0.755728, 0.520209, -0.397816,
0.749420, 0.504694, -0.428549,
0.765722, 0.470402, -0.438625,
0.778488, 0.501486, -0.377449,
0.788414, 0.451273, -0.418038,
0.780985, 0.435077, -0.448074,
0.794924, 0.466914, -0.387410,
0.810120, 0.431551, -0.396824,
0.830721, 0.411329, -0.375114,
0.824001, 0.395576, -0.405639,
0.800468, 0.481926, -0.356369,
0.794935, 0.399381, -0.456698,
0.816363, 0.379516, -0.435338,
0.807701, 0.363003, -0.464593,
0.821198, 0.461879, -0.335116,
0.840916, 0.441176, -0.313409,
0.836323, 0.426480, -0.344498,
0.747714, 0.645826, -0.154382,
0.768075, 0.626099, -0.134388,
0.770441, 0.615107, -0.167523,
0.791892, 0.583429, -0.180328,
0.787486, 0.605637, -0.114320,
0.811161, 0.562480, -0.160105,
0.812225, 0.550529, -0.192895,
0.809135, 0.573728, -0.127031,
0.829468, 0.540828, -0.139598,
0.846717, 0.518577, -0.118945,
0.848387, 0.507097, -0.151959,
0.806140, 0.584232, -0.093867,
0.831085, 0.516983, -0.205003,
0.849034, 0.495105, -0.184424,
0.848634, 0.482533, -0.216754,
0.823654, 0.562295, -0.073611,
0.840222, 0.539623, -0.053243,
0.843986, 0.529397, -0.086178,
0.539623, -0.053243, 0.840222,
0.529397, -0.086178, 0.843986,
0.562295, -0.073611, 0.823654,
0.584232, -0.093867, 0.806140,
0.518577, -0.118945, 0.846717,
0.573728, -0.127031, 0.809135,
0.605637, -0.114320, 0.787486,
0.540828, -0.139598, 0.829468,
0.562480, -0.160105, 0.811161,
0.550529, -0.192895, 0.812225,
0.583429, -0.180328, 0.791892,
0.507097, -0.151959, 0.848387,
0.626099, -0.134388, 0.768075,
0.615107, -0.167523, 0.770441,
0.645826, -0.154382, 0.747714,
0.664720, -0.174169, 0.726507,
0.603406, -0.200390, 0.771845,
0.495105, -0.184424, 0.849034,
0.482533, -0.216754, 0.848634,
0.516983, -0.205003, 0.831085,
0.538168, -0.225309, 0.812164,
0.469431, -0.248763, 0.847202,
0.653878, -0.208302, 0.727361,
0.683488, -0.194886, 0.703465,
0.623154, -0.221448, 0.750093,
0.642147, -0.242237, 0.727302,
0.629580, -0.275772, 0.726347,
0.660289, -0.262621, 0.703597,
0.591475, -0.234254, 0.771546,
0.597700, -0.288850, 0.747877,
0.616060, -0.309328, 0.724422,
0.578496, -0.267971, 0.770411,
0.564770, -0.301428, 0.768229,
0.530962, -0.313441, 0.787296,
0.550360, -0.334423, 0.765026,
0.558545, -0.246832, 0.791897,
0.601826, -0.342087, 0.721652,
0.633705, -0.329009, 0.700122,
0.568898, -0.354664, 0.742003,
0.586764, -0.374471, 0.717969,
0.570944, -0.406288, 0.713409,
0.603872, -0.393711, 0.693058,
0.535090, -0.366677, 0.761070,
0.524997, -0.258941, 0.810757,
0.490539, -0.270692, 0.828310,
0.511084, -0.292299, 0.808303,
0.496492, -0.325185, 0.804829,
0.455349, -0.282027, 0.844463,
0.700689, -0.214284, 0.680528,
0.689192, -0.248290, 0.680710,
0.717059, -0.233506, 0.656735,
0.732517, -0.252431, 0.632217,
0.676853, -0.281913, 0.679996,
0.720456, -0.286472, 0.631567,
0.747210, -0.271326, 0.606679,
0.692528, -0.301233, 0.655487,
0.707408, -0.320203, 0.630114,
0.693432, -0.353435, 0.627882,
0.721421, -0.338701, 0.604015,
0.663567, -0.315540, 0.678317,
0.760843, -0.289684, 0.580690,
0.748050, -0.323518, 0.579446,
0.773567, -0.307772, 0.553959,
0.785317, -0.325477, 0.526633,
0.734322, -0.356867, 0.577422,
0.649552, -0.348411, 0.675790,
0.634705, -0.380913, 0.672350,
0.664608, -0.367467, 0.650587,
0.678801, -0.386056, 0.624652,
0.619094, -0.412866, 0.668030,
0.441176, -0.313409, 0.840916,
0.426480, -0.344498, 0.836323,
0.461879, -0.335116, 0.821198,
0.481926, -0.356369, 0.800468,
0.411329, -0.375114, 0.830721,
0.466914, -0.387410, 0.794924,
0.501486, -0.377449, 0.778488,
0.431551, -0.396824, 0.810120,
0.451273, -0.418038, 0.788414,
0.435077, -0.448074, 0.780985,
0.470402, -0.438625, 0.765722,
0.395576, -0.405639, 0.824001,
0.520209, -0.397816, 0.755728,
0.504694, -0.428549, 0.749420,
0.538265, -0.417751, 0.731954,
0.555570, -0.437129, 0.707290,
0.488621, -0.458698, 0.742189,
0.379516, -0.435338, 0.816363,
0.363003, -0.464593, 0.807701,
0.399381, -0.456698, 0.794935,
0.418660, -0.477439, 0.772513,
0.346117, -0.493236, 0.798073,
0.773811, -0.362368, 0.519525,
0.798073, -0.346117, 0.493236,
0.748288, -0.378147, 0.545041,
0.761045, -0.398787, 0.511644,
0.747106, -0.434500, 0.503033,
0.772513, -0.418660, 0.477439,
0.721670, -0.393344, 0.569625,
0.720310, -0.450096, 0.527794,
0.731779, -0.469969, 0.493588,
0.707553, -0.429454, 0.561193,
0.692313, -0.464979, 0.551813,
0.663299, -0.479043, 0.574937,
0.676056, -0.499685, 0.541538,
0.693587, -0.408174, 0.593575,
0.715483, -0.504275, 0.483519,
0.742189, -0.488621, 0.458698,
0.687574, -0.519216, 0.507599,
0.698007, -0.537883, 0.472724,
0.679467, -0.570576, 0.461267,
0.707290, -0.555570, 0.437129,
0.658643, -0.533341, 0.530789,
0.664768, -0.422216, 0.616293,
0.634846, -0.435687, 0.638080,
0.649612, -0.457792, 0.606985,
0.633437, -0.492556, 0.596780,
0.604016, -0.448494, 0.658800,
0.649517, -0.586330, 0.484092,
0.658800, -0.604016, 0.448494,
0.639079, -0.567660, 0.518980,
0.618412, -0.601099, 0.506207,
0.586364, -0.614787, 0.527461,
0.596780, -0.633437, 0.492556,
0.627561, -0.548129, 0.552922,
0.574554, -0.595223, 0.561780,
0.552922, -0.627561, 0.548129,
0.595223, -0.561780, 0.574554,
0.561780, -0.574554, 0.595223,
0.548129, -0.552922, 0.627561,
0.527461, -0.586364, 0.614787,
0.614787, -0.527461, 0.586364,
0.518980, -0.639079, 0.567660,
0.530789, -0.658643, 0.533341,
0.506207, -0.618412, 0.601099,
0.484092, -0.649517, 0.586330,
0.448494, -0.658800, 0.604016,
0.461267, -0.679467, 0.570576,
0.492556, -0.596780, 0.633437,
0.601099, -0.506207, 0.618412,
0.586330, -0.484092, 0.649517,
0.567660, -0.518980, 0.639079,
0.533341, -0.530789, 0.658643,
0.570576, -0.461267, 0.679467,
0.638080, -0.634846, 0.435687,
0.668030, -0.619094, 0.412866,
0.606985, -0.649612, 0.457792,
0.616293, -0.664768, 0.422216,
0.593575, -0.693587, 0.408174,
0.624652, -0.678801, 0.386056,
0.574937, -0.663299, 0.479043,
0.561193, -0.707553, 0.429454,
0.569625, -0.721670, 0.393344,
0.551813, -0.692313, 0.464979,
0.527794, -0.720310, 0.450096,
0.493588, -0.731779, 0.469969,
0.503033, -0.747106, 0.434500,
0.541538, -0.676056, 0.499685,
0.545041, -0.748288, 0.378147,
0.577422, -0.734322, 0.356867,
0.511644, -0.761045, 0.398787,
0.519525, -0.773811, 0.362368,
0.493236, -0.798073, 0.346117,
0.526633, -0.785317, 0.325477,
0.477439, -0.772513, 0.418660,
0.507599, -0.687574, 0.519216,
0.472724, -0.698007, 0.537883,
0.483519, -0.715483, 0.504275,
0.458698, -0.742189, 0.488621,
0.437129, -0.707290, 0.555570,
0.537883, -0.472724, 0.698007,
0.504275, -0.483519, 0.715483,
0.519216, -0.507599, 0.687574,
0.499685, -0.541538, 0.676056,
0.469969, -0.493588, 0.731779,
0.464979, -0.551813, 0.692313,
0.479043, -0.574937, 0.663299,
0.450096, -0.527794, 0.720310,
0.429454, -0.561193, 0.707553,
0.393344, -0.569625, 0.721670,
0.408174, -0.593575, 0.693587,
0.434500, -0.503033, 0.747106,
0.457792, -0.606985, 0.649612,
0.422216, -0.616293, 0.664768,
0.435687, -0.638080, 0.634846,
0.412866, -0.668030, 0.619094,
0.386056, -0.624652, 0.678801,
0.398787, -0.511644, 0.761045,
0.362368, -0.519525, 0.773811,
0.378147, -0.545041, 0.748288,
0.356867, -0.577422, 0.734322,
0.325477, -0.526633, 0.785317,
0.807701, -0.363003, 0.464593,
0.794935, -0.399381, 0.456698,
0.816363, -0.379516, 0.435338,
0.824001, -0.395576, 0.405639,
0.780985, -0.435077, 0.448074,
0.810120, -0.431551, 0.396824,
0.830721, -0.411329, 0.375114,
0.788414, -0.451273, 0.418038,
0.794924, -0.466914, 0.387410,
0.778488, -0.501486, 0.377449,
0.800468, -0.481926, 0.356369,
0.765722, -0.470402, 0.438625,
0.836323, -0.426480, 0.344498,
0.821198, -0.461879, 0.335116,
0.840916, -0.441176, 0.313409,
0.844463, -0.455349, 0.282027,
0.804829, -0.496492, 0.325185,
0.749420, -0.504694, 0.428549,
0.731954, -0.538265, 0.417751,
0.755728, -0.520209, 0.397816,
0.761070, -0.535090, 0.366677,
0.713409, -0.570944, 0.406288,
0.828310, -0.490539, 0.270692,
0.847202, -0.469431, 0.248763,
0.808303, -0.511084, 0.292299,
0.810757, -0.524997, 0.258941,
0.791897, -0.558545, 0.246832,
0.812164, -0.538168, 0.225309,
0.787296, -0.530962, 0.313441,
0.770411, -0.578496, 0.267971,
0.771546, -0.591475, 0.234254,
0.768229, -0.564770, 0.301428,
0.747877, -0.597700, 0.288850,
0.724422, -0.616060, 0.309328,
0.726347, -0.629580, 0.275772,
0.765026, -0.550360, 0.334423,
0.750093, -0.623154, 0.221448,
0.771845, -0.603406, 0.200390,
0.727302, -0.642147, 0.242237,
0.727361, -0.653878, 0.208302,
0.703465, -0.683488, 0.194886,
0.726507, -0.664720, 0.174169,
0.703597, -0.660289, 0.262621,
0.742003, -0.568898, 0.354664,
0.717969, -0.586764, 0.374471,
0.721652, -0.601826, 0.342087,
0.700122, -0.633705, 0.329009,
0.693058, -0.603872, 0.393711,
0.848634, -0.482533, 0.216754,
0.831085, -0.516983, 0.205003,
0.849034, -0.495105, 0.184424,
0.848387, -0.507097, 0.151959,
0.812225, -0.550529, 0.192895,
0.829468, -0.540828, 0.139598,
0.846717, -0.518577, 0.118945,
0.811161, -0.562480, 0.160105,
0.809135, -0.573728, 0.127031,
0.787486, -0.605637, 0.114320,
0.806140, -0.584232, 0.093867,
0.791892, -0.583429, 0.180328,
0.843986, -0.529397, 0.086178,
0.823654, -0.562295, 0.073611,
0.840222, -0.539623, 0.053243,
0.770441, -0.615107, 0.167523,
0.747714, -0.645826, 0.154382,
0.768075, -0.626099, 0.134388,
0.672350, -0.634705, 0.380913,
0.650587, -0.664608, 0.367467,
0.675790, -0.649552, 0.348411,
0.678317, -0.663567, 0.315540,
0.627882, -0.693432, 0.353435,
0.655487, -0.692528, 0.301233,
0.679996, -0.676853, 0.281913,
0.630114, -0.707408, 0.320203,
0.631567, -0.720456, 0.286472,
0.606679, -0.747210, 0.271326,
0.632217, -0.732517, 0.252431,
0.604015, -0.721421, 0.338701,
0.680710, -0.689192, 0.248290,
0.656735, -0.717059, 0.233506,
0.680528, -0.700689, 0.214284,
0.579446, -0.748050, 0.323518,
0.553959, -0.773567, 0.307772,
0.580690, -0.760843, 0.289684,
0.307772, -0.553959, 0.773567,
0.289684, -0.580690, 0.760843,
0.323518, -0.579446, 0.748050,
0.338701, -0.604015, 0.721421,
0.271326, -0.606679, 0.747210,
0.320203, -0.630114, 0.707408,
0.353435, -0.627882, 0.693432,
0.286472, -0.631567, 0.720456,
0.301233, -0.655487, 0.692528,
0.281913, -0.679996, 0.676853,
0.315540, -0.678317, 0.663567,
0.252431, -0.632217, 0.732517,
0.367467, -0.650587, 0.664608,
0.348411, -0.675790, 0.649552,
0.380913, -0.672350, 0.634705,
0.393711, -0.693058, 0.603872,
0.329009, -0.700122, 0.633705,
0.233506, -0.656735, 0.717059,
0.214284, -0.680528, 0.700689,
0.248290, -0.680710, 0.689192,
0.262621, -0.703597, 0.660289,
0.194886, -0.703465, 0.683488,
0.374471, -0.717969, 0.586764,
0.406288, -0.713409, 0.570944,
0.342087, -0.721652, 0.601826,
0.354664, -0.742003, 0.568898,
0.334423, -0.765026, 0.550360,
0.366677, -0.761070, 0.535090,
0.309328, -0.724422, 0.616060,
0.301428, -0.768229, 0.564770,
0.313441, -0.787296, 0.530962,
0.288850, -0.747877, 0.597700,
0.267971, -0.770411, 0.578496,
0.234254, -0.771546, 0.591475,
0.246832, -0.791897, 0.558545,
0.275772, -0.726347, 0.629580,
0.292299, -0.808303, 0.511084,
0.325185, -0.804829, 0.496492,
0.258941, -0.810757, 0.524997,
0.270692, -0.828310, 0.490539,
0.248763, -0.847202, 0.469431,
0.282027, -0.844463, 0.455349,
0.225309, -0.812164, 0.538168,
0.242237, -0.727302, 0.642147,
0.208302, -0.727361, 0.653878,
0.221448, -0.750093, 0.623154,
0.200390, -0.771845, 0.603406,
0.174169, -0.726507, 0.664720,
0.417751, -0.731954, 0.538265,
0.397816, -0.755728, 0.520209,
0.428549, -0.749420, 0.504694,
0.438625, -0.765722, 0.470402,
0.377449, -0.778488, 0.501486,
0.418038, -0.788414, 0.451273,
0.448074, -0.780985, 0.435077,
0.387410, -0.794924, 0.466914,
0.396824, -0.810120, 0.431551,
0.375114, -0.830721, 0.411329,
0.405639, -0.824001, 0.395576,
0.356369, -0.800468, 0.481926,
0.456698, -0.794935, 0.399381,
0.435338, -0.816363, 0.379516,
0.464593, -0.807701, 0.363003,
0.335116, -0.821198, 0.461879,
0.313409, -0.840916, 0.441176,
0.344498, -0.836323, 0.426480,
0.154382, -0.747714, 0.645826,
0.134388, -0.768075, 0.626099,
0.167523, -0.770441, 0.615107,
0.180328, -0.791892, 0.583429,
0.114320, -0.787486, 0.605637,
0.160105, -0.811161, 0.562480,
0.192895, -0.812225, 0.550529,
0.127031, -0.809135, 0.573728,
0.139598, -0.829468, 0.540828,
0.118945, -0.846717, 0.518577,
0.151959, -0.848387, 0.507097,
0.093867, -0.806140, 0.584232,
0.205003, -0.831085, 0.516983,
0.184424, -0.849034, 0.495105,
0.216754, -0.848634, 0.482533,
0.073611, -0.823654, 0.562295,
0.053243, -0.840222, 0.539623,
0.086178, -0.843986, 0.529397,
0.539623, -0.053243, -0.840222,
0.562295, -0.073611, -0.823654,
0.529397, -0.086178, -0.843986,
0.518577, -0.118945, -0.846717,
0.584232, -0.093867, -0.806140,
0.540828, -0.139598, -0.829468,
0.507097, -0.151959, -0.848387,
0.573728, -0.127031, -0.809135,
0.562480, -0.160105, -0.811161,
0.583429, -0.180328, -0.791892,
0.550529, -0.192895, -0.812225,
0.605637, -0.114320, -0.787486,
0.495105, -0.184424, -0.849034,
0.516983, -0.205003, -0.831085,
0.482533, -0.216754, -0.848634,
0.469431, -0.248763, -0.847202,
0.538168, -0.225309, -0.812164,
0.626099, -0.134388, -0.768075,
0.645826, -0.154382, -0.747714,
0.615107, -0.167523, -0.770441,
0.603406, -0.200390, -0.771845,
0.664720, -0.174169, -0.726507,
0.490539, -0.270692, -0.828310,
0.455349, -0.282027, -0.844463,
0.524997, -0.258941, -0.810757,
0.511084, -0.292299, -0.808303,
0.530962, -0.313441, -0.787296,
0.496492, -0.325185, -0.804829,
0.558545, -0.246832, -0.791897,
0.564770, -0.301428, -0.768229,
0.550360, -0.334423, -0.765026,
0.578496, -0.267971, -0.770411,
0.597700, -0.288850, -0.747877,
0.629580, -0.275772, -0.726347,
0.616060, -0.309328, -0.724422,
0.591475, -0.234254, -0.771546,
0.568898, -0.354664, -0.742003,
0.535090, -0.366677, -0.761070,
0.601826, -0.342087, -0.721652,
0.586764, -0.374471, -0.717969,
0.603872, -0.393711, -0.693058,
0.570944, -0.406288, -0.713409,
0.633705, -0.329009, -0.700122,
0.623154, -0.221448, -0.750093,
0.653878, -0.208302, -0.727361,
0.642147, -0.242237, -0.727302,
0.660289, -0.262621, -0.703597,
0.683488, -0.194886, -0.703465,
0.441176, -0.313409, -0.840916,
0.461879, -0.335116, -0.821198,
0.426480, -0.344498, -0.836323,
0.411329, -0.375114, -0.830721,
0.481926, -0.356369, -0.800468,
0.431551, -0.396824, -0.810120,
0.395576, -0.405639, -0.824001,
0.466914, -0.387410, -0.794924,
0.451273, -0.418038, -0.788414,
0.470402, -0.438625, -0.765722,
0.435077, -0.448074, -0.780985,
0.501486, -0.377449, -0.778488,
0.379516, -0.435338, -0.816363,
0.399381, -0.456698, -0.794935,
0.363003, -0.464593, -0.807701,
0.346117, -0.493236, -0.798073,
0.418660, -0.477439, -0.772513,
0.520209, -0.397816, -0.755728,
0.538265, -0.417751, -0.731954,
0.504694, -0.428549, -0.749420,
0.488621, -0.458698, -0.742189,
0.555570, -0.437129, -0.707290,
0.700689, -0.214284, -0.680528,
0.717059, -0.233506, -0.656735,
0.689192, -0.248290, -0.680710,
0.676853, -0.281913, -0.679996,
0.732517, -0.252431, -0.632217,
0.692528, -0.301233, -0.655487,
0.663567, -0.315540, -0.678317,
0.720456, -0.286472, -0.631567,
0.707408, -0.320203, -0.630114,
0.721421, -0.338701, -0.604015,
0.693432, -0.353435, -0.627882,
0.747210, -0.271326, -0.606679,
0.649552, -0.348411, -0.675790,
0.664608, -0.367467, -0.650587,
0.634705, -0.380913, -0.672350,
0.619094, -0.412866, -0.668030,
0.678801, -0.386056, -0.624652,
0.760843, -0.289684, -0.580690,
0.773567, -0.307772, -0.553959,
0.748050, -0.323518, -0.579446,
0.734322, -0.356867, -0.577422,
0.785317, -0.325477, -0.526633,
0.362368, -0.519525, -0.773811,
0.325477, -0.526633, -0.785317,
0.398787, -0.511644, -0.761045,
0.378147, -0.545041, -0.748288,
0.393344, -0.569625, -0.721670,
0.356867, -0.577422, -0.734322,
0.434500, -0.503033, -0.747106,
0.429454, -0.561193, -0.707553,
0.408174, -0.593575, -0.693587,
0.450096, -0.527794, -0.720310,
0.464979, -0.551813, -0.692313,
0.499685, -0.541538, -0.676056,
0.479043, -0.574937, -0.663299,
0.469969, -0.493588, -0.731779,
0.422216, -0.616293, -0.664768,
0.386056, -0.624652, -0.678801,
0.457792, -0.606985, -0.649612,
0.435687, -0.638080, -0.634846,
0.448494, -0.658800, -0.604016,
0.412866, -0.668030, -0.619094,
0.492556, -0.596780, -0.633437,
0.504275, -0.483519, -0.715483,
0.537883, -0.472724, -0.698007,
0.519216, -0.507599, -0.687574,
0.533341, -0.530789, -0.658643,
0.570576, -0.461267, -0.679467,
0.484092, -0.649517, -0.586330,
0.461267, -0.679467, -0.570576,
0.506207, -0.618412, -0.601099,
0.518980, -0.639079, -0.567660,
0.552922, -0.627561, -0.548129,
0.530789, -0.658643, -0.533341,
0.527461, -0.586364, -0.614787,
0.574554, -0.595223, -0.561780,
0.586364, -0.614787, -0.527461,
0.561780, -0.574554, -0.595223,
0.595223, -0.561780, -0.574554,
0.614787, -0.527461, -0.586364,
0.627561, -0.548129, -0.552922,
0.548129, -0.552922, -0.627561,
0.618412, -0.601099, -0.506207,
0.596780, -0.633437, -0.492556,
0.639079, -0.567660, -0.518980,
0.649517, -0.586330, -0.484092,
0.679467, -0.570576, -0.461267,
0.658800, -0.604016, -0.448494,
0.658643, -0.533341, -0.530789,
0.567660, -0.518980, -0.639079,
0.586330, -0.484092, -0.649517,
0.601099, -0.506207, -0.618412,
0.633437, -0.492556, -0.596780,
0.604016, -0.448494, -0.658800,
0.472724, -0.698007, -0.537883,
0.437129, -0.707290, -0.555570,
0.507599, -0.687574, -0.519216,
0.483519, -0.715483, -0.504275,
0.493588, -0.731779, -0.469969,
0.458698, -0.742189, -0.488621,
0.541538, -0.676056, -0.499685,
0.527794, -0.720310, -0.450096,
0.503033, -0.747106, -0.434500,
0.551813, -0.692313, -0.464979,
0.561193, -0.707553, -0.429454,
0.593575, -0.693587, -0.408174,
0.569625, -0.721670, -0.393344,
0.574937, -0.663299, -0.479043,
0.511644, -0.761045, -0.398787,
0.477439, -0.772513, -0.418660,
0.545041, -0.748288, -0.378147,
0.519525, -0.773811, -0.362368,
0.526633, -0.785317, -0.325477,
0.493236, -0.798073, -0.346117,
0.577422, -0.734322, -0.356867,
0.606985, -0.649612, -0.457792,
0.638080, -0.634846, -0.435687,
0.616293, -0.664768, -0.422216,
0.624652, -0.678801, -0.386056,
0.668030, -0.619094, -0.412866,
0.634846, -0.435687, -0.638080,
0.664768, -0.422216, -0.616293,
0.649612, -0.457792, -0.606985,
0.663299, -0.479043, -0.574937,
0.693587, -0.408174, -0.593575,
0.692313, -0.464979, -0.551813,
0.676056, -0.499685, -0.541538,
0.707553, -0.429454, -0.561193,
0.720310, -0.450096, -0.527794,
0.747106, -0.434500, -0.503033,
0.731779, -0.469969, -0.493588,
0.721670, -0.393344, -0.569625,
0.687574, -0.519216, -0.507599,
0.715483, -0.504275, -0.483519,
0.698007, -0.537883, -0.472724,
0.707290, -0.555570, -0.437129,
0.742189, -0.488621, -0.458698,
0.748288, -0.378147, -0.545041,
0.773811, -0.362368, -0.519525,
0.761045, -0.398787, -0.511644,
0.772513, -0.418660, -0.477439,
0.798073, -0.346117, -0.493236,
0.307772, -0.553959, -0.773567,
0.323518, -0.579446, -0.748050,
0.289684, -0.580690, -0.760843,
0.271326, -0.606679, -0.747210,
0.338701, -0.604015, -0.721421,
0.286472, -0.631567, -0.720456,
0.252431, -0.632217, -0.732517,
0.320203, -0.630114, -0.707408,
0.301233, -0.655487, -0.692528,
0.315540, -0.678317, -0.663567,
0.281913, -0.679996, -0.676853,
0.353435, -0.627882, -0.693432,
0.233506, -0.656735, -0.717059,
0.248290, -0.680710, -0.689192,
0.214284, -0.680528, -0.700689,
0.194886, -0.703465, -0.683488,
0.262621, -0.703597, -0.660289,
0.367467, -0.650587, -0.664608,
0.380913, -0.672350, -0.634705,
0.348411, -0.675790, -0.649552,
0.329009, -0.700122, -0.633705,
0.393711, -0.693058, -0.603872,
0.208302, -0.727361, -0.653878,
0.174169, -0.726507, -0.664720,
0.242237, -0.727302, -0.642147,
0.221448, -0.750093, -0.623154,
0.234254, -0.771546, -0.591475,
0.200390, -0.771845, -0.603406,
0.275772, -0.726347, -0.629580,
0.267971, -0.770411, -0.578496,
0.246832, -0.791897, -0.558545,
0.288850, -0.747877, -0.597700,
0.301428, -0.768229, -0.564770,
0.334423, -0.765026, -0.550360,
0.313441, -0.787296, -0.530962,
0.309328, -0.724422, -0.616060,
0.258941, -0.810757, -0.524997,
0.225309, -0.812164, -0.538168,
0.292299, -0.808303, -0.511084,
0.270692, -0.828310, -0.490539,
0.282027, -0.844463, -0.455349,
0.248763, -0.847202, -0.469431,
0.325185, -0.804829, -0.496492,
0.342087, -0.721652, -0.601826,
0.374471, -0.717969, -0.586764,
0.354664, -0.742003, -0.568898,
0.366677, -0.761070, -0.535090,
0.406288, -0.713409, -0.570944,
0.154382, -0.747714, -0.645826,
0.167523, -0.770441, -0.615107,
0.134388, -0.768075, -0.626099,
0.114320, -0.787486, -0.605637,
0.180328, -0.791892, -0.583429,
0.127031, -0.809135, -0.573728,
0.093867, -0.806140, -0.584232,
0.160105, -0.811161, -0.562480,
0.139598, -0.829468, -0.540828,
0.151959, -0.848387, -0.507097,
0.118945, -0.846717, -0.518577,
0.192895, -0.812225, -0.550529,
0.073611, -0.823654, -0.562295,
0.086178, -0.843986, -0.529397,
0.053243, -0.840222, -0.539623,
0.205003, -0.831085, -0.516983,
0.216754, -0.848634, -0.482533,
0.184424, -0.849034, -0.495105,
0.417751, -0.731954, -0.538265,
0.428549, -0.749420, -0.504694,
0.397816, -0.755728, -0.520209,
0.377449, -0.778488, -0.501486,
0.438625, -0.765722, -0.470402,
0.387410, -0.794924, -0.466914,
0.356369, -0.800468, -0.481926,
0.418038, -0.788414, -0.451273,
0.396824, -0.810120, -0.431551,
0.405639, -0.824001, -0.395576,
0.375114, -0.830721, -0.411329,
0.448074, -0.780985, -0.435077,
0.335116, -0.821198, -0.461879,
0.344498, -0.836323, -0.426480,
0.313409, -0.840916, -0.441176,
0.456698, -0.794935, -0.399381,
0.464593, -0.807701, -0.363003,
0.435338, -0.816363, -0.379516,
0.807701, -0.363003, -0.464593,
0.816363, -0.379516, -0.435338,
0.794935, -0.399381, -0.456698,
0.780985, -0.435077, -0.448074,
0.824001, -0.395576, -0.405639,
0.788414, -0.451273, -0.418038,
0.765722, -0.470402, -0.438625,
0.810120, -0.431551, -0.396824,
0.794924, -0.466914, -0.387410,
0.800468, -0.481926, -0.356369,
0.778488, -0.501486, -0.377449,
0.830721, -0.411329, -0.375114,
0.749420, -0.504694, -0.428549,
0.755728, -0.520209, -0.397816,
0.731954, -0.538265, -0.417751,
0.713409, -0.570944, -0.406288,
0.761070, -0.535090, -0.366677,
0.836323, -0.426480, -0.344498,
0.840916, -0.441176, -0.313409,
0.821198, -0.461879, -0.335116,
0.804829, -0.496492, -0.325185,
0.844463, -0.455349, -0.282027,
0.717969, -0.586764, -0.374471,
0.693058, -0.603872, -0.393711,
0.742003, -0.568898, -0.354664,
0.721652, -0.601826, -0.342087,
0.724422, -0.616060, -0.309328,
0.700122, -0.633705, -0.329009,
0.765026, -0.550360, -0.334423,
0.747877, -0.597700, -0.288850,
0.726347, -0.629580, -0.275772,
0.768229, -0.564770, -0.301428,
0.770411, -0.578496, -0.267971,
0.791897, -0.558545, -0.246832,
0.771546, -0.591475, -0.234254,
0.787296, -0.530962, -0.313441,
0.727302, -0.642147, -0.242237,
0.703597, -0.660289, -0.262621,
0.750093, -0.623154, -0.221448,
0.727361, -0.653878, -0.208302,
0.726507, -0.664720, -0.174169,
0.703465, -0.683488, -0.194886,
0.771845, -0.603406, -0.200390,
0.808303, -0.511084, -0.292299,
0.828310, -0.490539, -0.270692,
0.810757, -0.524997, -0.258941,
0.812164, -0.538168, -0.225309,
0.847202, -0.469431, -0.248763,
0.672350, -0.634705, -0.380913,
0.675790, -0.649552, -0.348411,
0.650587, -0.664608, -0.367467,
0.627882, -0.693432, -0.353435,
0.678317, -0.663567, -0.315540,
0.630114, -0.707408, -0.320203,
0.604015, -0.721421, -0.338701,
0.655487, -0.692528, -0.301233,
0.631567, -0.720456, -0.286472,
0.632217, -0.732517, -0.252431,
0.606679, -0.747210, -0.271326,
0.679996, -0.676853, -0.281913,
0.579446, -0.748050, -0.323518,
0.580690, -0.760843, -0.289684,
0.553959, -0.773567, -0.307772,
0.680710, -0.689192, -0.248290,
0.680528, -0.700689, -0.214284,
0.656735, -0.717059, -0.233506,
0.848634, -0.482533, -0.216754,
0.849034, -0.495105, -0.184424,
0.831085, -0.516983, -0.205003,
0.812225, -0.550529, -0.192895,
0.848387, -0.507097, -0.151959,
0.811161, -0.562480, -0.160105,
0.791892, -0.583429, -0.180328,
0.829468, -0.540828, -0.139598,
0.809135, -0.573728, -0.127031,
0.806140, -0.584232, -0.093867,
0.787486, -0.605637, -0.114320,
0.846717, -0.518577, -0.118945,
0.770441, -0.615107, -0.167523,
0.768075, -0.626099, -0.134388,
0.747714, -0.645826, -0.154382,
0.843986, -0.529397, -0.086178,
0.840222, -0.539623, -0.053243,
0.823654, -0.562295, -0.073611,
-0.539623, 0.053243, 0.840222,
-0.529397, 0.086178, 0.843986,
-0.562295, 0.073611, 0.823654,
-0.584232, 0.093867, 0.806140,
-0.518577, 0.118945, 0.846717,
-0.573728, 0.127031, 0.809135,
-0.605637, 0.114320, 0.787486,
-0.540828, 0.139598, 0.829468,
-0.562480, 0.160105, 0.811161,
-0.550529, 0.192895, 0.812225,
-0.583429, 0.180328, 0.791892,
-0.507097, 0.151959, 0.848387,
-0.626099, 0.134388, 0.768075,
-0.615107, 0.167523, 0.770441,
-0.645826, 0.154382, 0.747714,
-0.664720, 0.174169, 0.726507,
-0.603406, 0.200390, 0.771845,
-0.495105, 0.184424, 0.849034,
-0.482533, 0.216754, 0.848634,
-0.516983, 0.205003, 0.831085,
-0.538168, 0.225309, 0.812164,
-0.469431, 0.248763, 0.847202,
-0.653878, 0.208302, 0.727361,
-0.683488, 0.194886, 0.703465,
-0.623154, 0.221448, 0.750093,
-0.642147, 0.242237, 0.727302,
-0.629580, 0.275772, 0.726347,
-0.660289, 0.262621, 0.703597,
-0.591475, 0.234254, 0.771546,
-0.597700, 0.288850, 0.747877,
-0.616060, 0.309328, 0.724422,
-0.578496, 0.267971, 0.770411,
-0.564770, 0.301428, 0.768229,
-0.530962, 0.313441, 0.787296,
-0.550360, 0.334423, 0.765026,
-0.558545, 0.246832, 0.791897,
-0.601826, 0.342087, 0.721652,
-0.633705, 0.329009, 0.700122,
-0.568898, 0.354664, 0.742003,
-0.586764, 0.374471, 0.717969,
-0.570944, 0.406288, 0.713409,
-0.603872, 0.393711, 0.693058,
-0.535090, 0.366677, 0.761070,
-0.524997, 0.258941, 0.810757,
-0.490539, 0.270692, 0.828310,
-0.511084, 0.292299, 0.808303,
-0.496492, 0.325185, 0.804829,
-0.455349, 0.282027, 0.844463,
-0.700689, 0.214284, 0.680528,
-0.689192, 0.248290, 0.680710,
-0.717059, 0.233506, 0.656735,
-0.732517, 0.252431, 0.632217,
-0.676853, 0.281913, 0.679996,
-0.720456, 0.286472, 0.631567,
-0.747210, 0.271326, 0.606679,
-0.692528, 0.301233, 0.655487,
-0.707408, 0.320203, 0.630114,
-0.693432, 0.353435, 0.627882,
-0.721421, 0.338701, 0.604015,
-0.663567, 0.315540, 0.678317,
-0.760843, 0.289684, 0.580690,
-0.748050, 0.323518, 0.579446,
-0.773567, 0.307772, 0.553959,
-0.785317, 0.325477, 0.526633,
-0.734322, 0.356867, 0.577422,
-0.649552, 0.348411, 0.675790,
-0.634705, 0.380913, 0.672350,
-0.664608, 0.367467, 0.650587,
-0.678801, 0.386056, 0.624652,
-0.619094, 0.412866, 0.668030,
-0.441176, 0.313409, 0.840916,
-0.426480, 0.344498, 0.836323,
-0.461879, 0.335116, 0.821198,
-0.481926, 0.356369, 0.800468,
-0.411329, 0.375114, 0.830721,
-0.466914, 0.387410, 0.794924,
-0.501486, 0.377449, 0.778488,
-0.431551, 0.396824, 0.810120,
-0.451273, 0.418038, 0.788414,
-0.435077, 0.448074, 0.780985,
-0.470402, 0.438625, 0.765722,
-0.395576, 0.405639, 0.824001,
-0.520209, 0.397816, 0.755728,
-0.504694, 0.428549, 0.749420,
-0.538265, 0.417751, 0.731954,
-0.555570, 0.437129, 0.707290,
-0.488621, 0.458698, 0.742189,
-0.379516, 0.435338, 0.816363,
-0.363003, 0.464593, 0.807701,
-0.399381, 0.456698, 0.794935,
-0.418660, 0.477439, 0.772513,
-0.346117, 0.493236, 0.798073,
-0.773811, 0.362368, 0.519525,
-0.798073, 0.346117, 0.493236,
-0.748288, 0.378147, 0.545041,
-0.761045, 0.398787, 0.511644,
-0.747106, 0.434500, 0.503033,
-0.772513, 0.418660, 0.477439,
-0.721670, 0.393344, 0.569625,
-0.720310, 0.450096, 0.527794,
-0.731779, 0.469969, 0.493588,
-0.707553, 0.429454, 0.561193,
-0.692313, 0.464979, 0.551813,
-0.663299, 0.479043, 0.574937,
-0.676056, 0.499685, 0.541538,
-0.693587, 0.408174, 0.593575,
-0.715483, 0.504275, 0.483519,
-0.742189, 0.488621, 0.458698,
-0.687574, 0.519216, 0.507599,
-0.698007, 0.537883, 0.472724,
-0.679467, 0.570576, 0.461267,
-0.707290, 0.555570, 0.437129,
-0.658643, 0.533341, 0.530789,
-0.664768, 0.422216, 0.616293,
-0.634846, 0.435687, 0.638080,
-0.649612, 0.457792, 0.606985,
-0.633437, 0.492556, 0.596780,
-0.604016, 0.448494, 0.658800,
-0.649517, 0.586330, 0.484092,
-0.658800, 0.604016, 0.448494,
-0.639079, 0.567660, 0.518980,
-0.618412, 0.601099, 0.506207,
-0.586364, 0.614787, 0.527461,
-0.596780, 0.633437, 0.492556,
-0.627561, 0.548129, 0.552922,
-0.574554, 0.595223, 0.561780,
-0.552922, 0.627561, 0.548129,
-0.595223, 0.561780, 0.574554,
-0.561780, 0.574554, 0.595223,
-0.548129, 0.552922, 0.627561,
-0.527461, 0.586364, 0.614787,
-0.614787, 0.527461, 0.586364,
-0.518980, 0.639079, 0.567660,
-0.530789, 0.658643, 0.533341,
-0.506207, 0.618412, 0.601099,
-0.484092, 0.649517, 0.586330,
-0.448494, 0.658800, 0.604016,
-0.461267, 0.679467, 0.570576,
-0.492556, 0.596780, 0.633437,
-0.601099, 0.506207, 0.618412,
-0.586330, 0.484092, 0.649517,
-0.567660, 0.518980, 0.639079,
-0.533341, 0.530789, 0.658643,
-0.570576, 0.461267, 0.679467,
-0.638080, 0.634846, 0.435687,
-0.668030, 0.619094, 0.412866,
-0.606985, 0.649612, 0.457792,
-0.616293, 0.664768, 0.422216,
-0.593575, 0.693587, 0.408174,
-0.624652, 0.678801, 0.386056,
-0.574937, 0.663299, 0.479043,
-0.561193, 0.707553, 0.429454,
-0.569625, 0.721670, 0.393344,
-0.551813, 0.692313, 0.464979,
-0.527794, 0.720310, 0.450096,
-0.493588, 0.731779, 0.469969,
-0.503033, 0.747106, 0.434500,
-0.541538, 0.676056, 0.499685,
-0.545041, 0.748288, 0.378147,
-0.577422, 0.734322, 0.356867,
-0.511644, 0.761045, 0.398787,
-0.519525, 0.773811, 0.362368,
-0.493236, 0.798073, 0.346117,
-0.526633, 0.785317, 0.325477,
-0.477439, 0.772513, 0.418660,
-0.507599, 0.687574, 0.519216,
-0.472724, 0.698007, 0.537883,
-0.483519, 0.715483, 0.504275,
-0.458698, 0.742189, 0.488621,
-0.437129, 0.707290, 0.555570,
-0.537883, 0.472724, 0.698007,
-0.504275, 0.483519, 0.715483,
-0.519216, 0.507599, 0.687574,
-0.499685, 0.541538, 0.676056,
-0.469969, 0.493588, 0.731779,
-0.464979, 0.551813, 0.692313,
-0.479043, 0.574937, 0.663299,
-0.450096, 0.527794, 0.720310,
-0.429454, 0.561193, 0.707553,
-0.393344, 0.569625, 0.721670,
-0.408174, 0.593575, 0.693587,
-0.434500, 0.503033, 0.747106,
-0.457792, 0.606985, 0.649612,
-0.422216, 0.616293, 0.664768,
-0.435687, 0.638080, 0.634846,
-0.412866, 0.668030, 0.619094,
-0.386056, 0.624652, 0.678801,
-0.398787, 0.511644, 0.761045,
-0.362368, 0.519525, 0.773811,
-0.378147, 0.545041, 0.748288,
-0.356867, 0.577422, 0.734322,
-0.325477, 0.526633, 0.785317,
-0.807701, 0.363003, 0.464593,
-0.794935, 0.399381, 0.456698,
-0.816363, 0.379516, 0.435338,
-0.824001, 0.395576, 0.405639,
-0.780985, 0.435077, 0.448074,
-0.810120, 0.431551, 0.396824,
-0.830721, 0.411329, 0.375114,
-0.788414, 0.451273, 0.418038,
-0.794924, 0.466914, 0.387410,
-0.778488, 0.501486, 0.377449,
-0.800468, 0.481926, 0.356369,
-0.765722, 0.470402, 0.438625,
-0.836323, 0.426480, 0.344498,
-0.821198, 0.461879, 0.335116,
-0.840916, 0.441176, 0.313409,
-0.844463, 0.455349, 0.282027,
-0.804829, 0.496492, 0.325185,
-0.749420, 0.504694, 0.428549,
-0.731954, 0.538265, 0.417751,
-0.755728, 0.520209, 0.397816,
-0.761070, 0.535090, 0.366677,
-0.713409, 0.570944, 0.406288,
-0.828310, 0.490539, 0.270692,
-0.847202, 0.469431, 0.248763,
-0.808303, 0.511084, 0.292299,
-0.810757, 0.524997, 0.258941,
-0.791897, 0.558545, 0.246832,
-0.812164, 0.538168, 0.225309,
-0.787296, 0.530962, 0.313441,
-0.770411, 0.578496, 0.267971,
-0.771546, 0.591475, 0.234254,
-0.768229, 0.564770, 0.301428,
-0.747877, 0.597700, 0.288850,
-0.724422, 0.616060, 0.309328,
-0.726347, 0.629580, 0.275772,
-0.765026, 0.550360, 0.334423,
-0.750093, 0.623154, 0.221448,
-0.771845, 0.603406, 0.200390,
-0.727302, 0.642147, 0.242237,
-0.727361, 0.653878, 0.208302,
-0.703465, 0.683488, 0.194886,
-0.726507, 0.664720, 0.174169,
-0.703597, 0.660289, 0.262621,
-0.742003, 0.568898, 0.354664,
-0.717969, 0.586764, 0.374471,
-0.721652, 0.601826, 0.342087,
-0.700122, 0.633705, 0.329009,
-0.693058, 0.603872, 0.393711,
-0.848634, 0.482533, 0.216754,
-0.831085, 0.516983, 0.205003,
-0.849034, 0.495105, 0.184424,
-0.848387, 0.507097, 0.151959,
-0.812225, 0.550529, 0.192895,
-0.829468, 0.540828, 0.139598,
-0.846717, 0.518577, 0.118945,
-0.811161, 0.562480, 0.160105,
-0.809135, 0.573728, 0.127031,
-0.787486, 0.605637, 0.114320,
-0.806140, 0.584232, 0.093867,
-0.791892, 0.583429, 0.180328,
-0.843986, 0.529397, 0.086178,
-0.823654, 0.562295, 0.073611,
-0.840222, 0.539623, 0.053243,
-0.770441, 0.615107, 0.167523,
-0.747714, 0.645826, 0.154382,
-0.768075, 0.626099, 0.134388,
-0.672350, 0.634705, 0.380913,
-0.650587, 0.664608, 0.367467,
-0.675790, 0.649552, 0.348411,
-0.678317, 0.663567, 0.315540,
-0.627882, 0.693432, 0.353435,
-0.655487, 0.692528, 0.301233,
-0.679996, 0.676853, 0.281913,
-0.630114, 0.707408, 0.320203,
-0.631567, 0.720456, 0.286472,
-0.606679, 0.747210, 0.271326,
-0.632217, 0.732517, 0.252431,
-0.604015, 0.721421, 0.338701,
-0.680710, 0.689192, 0.248290,
-0.656735, 0.717059, 0.233506,
-0.680528, 0.700689, 0.214284,
-0.579446, 0.748050, 0.323518,
-0.553959, 0.773567, 0.307772,
-0.580690, 0.760843, 0.289684,
-0.307772, 0.553959, 0.773567,
-0.289684, 0.580690, 0.760843,
-0.323518, 0.579446, 0.748050,
-0.338701, 0.604015, 0.721421,
-0.271326, 0.606679, 0.747210,
-0.320203, 0.630114, 0.707408,
-0.353435, 0.627882, 0.693432,
-0.286472, 0.631567, 0.720456,
-0.301233, 0.655487, 0.692528,
-0.281913, 0.679996, 0.676853,
-0.315540, 0.678317, 0.663567,
-0.252431, 0.632217, 0.732517,
-0.367467, 0.650587, 0.664608,
-0.348411, 0.675790, 0.649552,
-0.380913, 0.672350, 0.634705,
-0.393711, 0.693058, 0.603872,
-0.329009, 0.700122, 0.633705,
-0.233506, 0.656735, 0.717059,
-0.214284, 0.680528, 0.700689,
-0.248290, 0.680710, 0.689192,
-0.262621, 0.703597, 0.660289,
-0.194886, 0.703465, 0.683488,
-0.374471, 0.717969, 0.586764,
-0.406288, 0.713409, 0.570944,
-0.342087, 0.721652, 0.601826,
-0.354664, 0.742003, 0.568898,
-0.334423, 0.765026, 0.550360,
-0.366677, 0.761070, 0.535090,
-0.309328, 0.724422, 0.616060,
-0.301428, 0.768229, 0.564770,
-0.313441, 0.787296, 0.530962,
-0.288850, 0.747877, 0.597700,
-0.267971, 0.770411, 0.578496,
-0.234254, 0.771546, 0.591475,
-0.246832, 0.791897, 0.558545,
-0.275772, 0.726347, 0.629580,
-0.292299, 0.808303, 0.511084,
-0.325185, 0.804829, 0.496492,
-0.258941, 0.810757, 0.524997,
-0.270692, 0.828310, 0.490539,
-0.248763, 0.847202, 0.469431,
-0.282027, 0.844463, 0.455349,
-0.225309, 0.812164, 0.538168,
-0.242237, 0.727302, 0.642147,
-0.208302, 0.727361, 0.653878,
-0.221448, 0.750093, 0.623154,
-0.200390, 0.771845, 0.603406,
-0.174169, 0.726507, 0.664720,
-0.417751, 0.731954, 0.538265,
-0.397816, 0.755728, 0.520209,
-0.428549, 0.749420, 0.504694,
-0.438625, 0.765722, 0.470402,
-0.377449, 0.778488, 0.501486,
-0.418038, 0.788414, 0.451273,
-0.448074, 0.780985, 0.435077,
-0.387410, 0.794924, 0.466914,
-0.396824, 0.810120, 0.431551,
-0.375114, 0.830721, 0.411329,
-0.405639, 0.824001, 0.395576,
-0.356369, 0.800468, 0.481926,
-0.456698, 0.794935, 0.399381,
-0.435338, 0.816363, 0.379516,
-0.464593, 0.807701, 0.363003,
-0.335116, 0.821198, 0.461879,
-0.313409, 0.840916, 0.441176,
-0.344498, 0.836323, 0.426480,
-0.154382, 0.747714, 0.645826,
-0.134388, 0.768075, 0.626099,
-0.167523, 0.770441, 0.615107,
-0.180328, 0.791892, 0.583429,
-0.114320, 0.787486, 0.605637,
-0.160105, 0.811161, 0.562480,
-0.192895, 0.812225, 0.550529,
-0.127031, 0.809135, 0.573728,
-0.139598, 0.829468, 0.540828,
-0.118945, 0.846717, 0.518577,
-0.151959, 0.848387, 0.507097,
-0.093867, 0.806140, 0.584232,
-0.205003, 0.831085, 0.516983,
-0.184424, 0.849034, 0.495105,
-0.216754, 0.848634, 0.482533,
-0.073611, 0.823654, 0.562295,
-0.053243, 0.840222, 0.539623,
-0.086178, 0.843986, 0.529397,
-0.539623, 0.053243, -0.840222,
-0.562295, 0.073611, -0.823654,
-0.529397, 0.086178, -0.843986,
-0.518577, 0.118945, -0.846717,
-0.584232, 0.093867, -0.806140,
-0.540828, 0.139598, -0.829468,
-0.507097, 0.151959, -0.848387,
-0.573728, 0.127031, -0.809135,
-0.562480, 0.160105, -0.811161,
-0.583429, 0.180328, -0.791892,
-0.550529, 0.192895, -0.812225,
-0.605637, 0.114320, -0.787486,
-0.495105, 0.184424, -0.849034,
-0.516983, 0.205003, -0.831085,
-0.482533, 0.216754, -0.848634,
-0.469431, 0.248763, -0.847202,
-0.538168, 0.225309, -0.812164,
-0.626099, 0.134388, -0.768075,
-0.645826, 0.154382, -0.747714,
-0.615107, 0.167523, -0.770441,
-0.603406, 0.200390, -0.771845,
-0.664720, 0.174169, -0.726507,
-0.490539, 0.270692, -0.828310,
-0.455349, 0.282027, -0.844463,
-0.524997, 0.258941, -0.810757,
-0.511084, 0.292299, -0.808303,
-0.530962, 0.313441, -0.787296,
-0.496492, 0.325185, -0.804829,
-0.558545, 0.246832, -0.791897,
-0.564770, 0.301428, -0.768229,
-0.550360, 0.334423, -0.765026,
-0.578496, 0.267971, -0.770411,
-0.597700, 0.288850, -0.747877,
-0.629580, 0.275772, -0.726347,
-0.616060, 0.309328, -0.724422,
-0.591475, 0.234254, -0.771546,
-0.568898, 0.354664, -0.742003,
-0.535090, 0.366677, -0.761070,
-0.601826, 0.342087, -0.721652,
-0.586764, 0.374471, -0.717969,
-0.603872, 0.393711, -0.693058,
-0.570944, 0.406288, -0.713409,
-0.633705, 0.329009, -0.700122,
-0.623154, 0.221448, -0.750093,
-0.653878, 0.208302, -0.727361,
-0.642147, 0.242237, -0.727302,
-0.660289, 0.262621, -0.703597,
-0.683488, 0.194886, -0.703465,
-0.441176, 0.313409, -0.840916,
-0.461879, 0.335116, -0.821198,
-0.426480, 0.344498, -0.836323,
-0.411329, 0.375114, -0.830721,
-0.481926, 0.356369, -0.800468,
-0.431551, 0.396824, -0.810120,
-0.395576, 0.405639, -0.824001,
-0.466914, 0.387410, -0.794924,
-0.451273, 0.418038, -0.788414,
-0.470402, 0.438625, -0.765722,
-0.435077, 0.448074, -0.780985,
-0.501486, 0.377449, -0.778488,
-0.379516, 0.435338, -0.816363,
-0.399381, 0.456698, -0.794935,
-0.363003, 0.464593, -0.807701,
-0.346117, 0.493236, -0.798073,
-0.418660, 0.477439, -0.772513,
-0.520209, 0.397816, -0.755728,
-0.538265, 0.417751, -0.731954,
-0.504694, 0.428549, -0.749420,
-0.488621, 0.458698, -0.742189,
-0.555570, 0.437129, -0.707290,
-0.700689, 0.214284, -0.680528,
-0.717059, 0.233506, -0.656735,
-0.689192, 0.248290, -0.680710,
-0.676853, 0.281913, -0.679996,
-0.732517, 0.252431, -0.632217,
-0.692528, 0.301233, -0.655487,
-0.663567, 0.315540, -0.678317,
-0.720456, 0.286472, -0.631567,
-0.707408, 0.320203, -0.630114,
-0.721421, 0.338701, -0.604015,
-0.693432, 0.353435, -0.627882,
-0.747210, 0.271326, -0.606679,
-0.649552, 0.348411, -0.675790,
-0.664608, 0.367467, -0.650587,
-0.634705, 0.380913, -0.672350,
-0.619094, 0.412866, -0.668030,
-0.678801, 0.386056, -0.624652,
-0.760843, 0.289684, -0.580690,
-0.773567, 0.307772, -0.553959,
-0.748050, 0.323518, -0.579446,
-0.734322, 0.356867, -0.577422,
-0.785317, 0.325477, -0.526633,
-0.362368, 0.519525, -0.773811,
-0.325477, 0.526633, -0.785317,
-0.398787, 0.511644, -0.761045,
-0.378147, 0.545041, -0.748288,
-0.393344, 0.569625, -0.721670,
-0.356867, 0.577422, -0.734322,
-0.434500, 0.503033, -0.747106,
-0.429454, 0.561193, -0.707553,
-0.408174, 0.593575, -0.693587,
-0.450096, 0.527794, -0.720310,
-0.464979, 0.551813, -0.692313,
-0.499685, 0.541538, -0.676056,
-0.479043, 0.574937, -0.663299,
-0.469969, 0.493588, -0.731779,
-0.422216, 0.616293, -0.664768,
-0.386056, 0.624652, -0.678801,
-0.457792, 0.606985, -0.649612,
-0.435687, 0.638080, -0.634846,
-0.448494, 0.658800, -0.604016,
-0.412866, 0.668030, -0.619094,
-0.492556, 0.596780, -0.633437,
-0.504275, 0.483519, -0.715483,
-0.537883, 0.472724, -0.698007,
-0.519216, 0.507599, -0.687574,
-0.533341, 0.530789, -0.658643,
-0.570576, 0.461267, -0.679467,
-0.484092, 0.649517, -0.586330,
-0.461267, 0.679467, -0.570576,
-0.506207, 0.618412, -0.601099,
-0.518980, 0.639079, -0.567660,
-0.552922, 0.627561, -0.548129,
-0.530789, 0.658643, -0.533341,
-0.527461, 0.586364, -0.614787,
-0.574554, 0.595223, -0.561780,
-0.586364, 0.614787, -0.527461,
-0.561780, 0.574554, -0.595223,
-0.595223, 0.561780, -0.574554,
-0.614787, 0.527461, -0.586364,
-0.627561, 0.548129, -0.552922,
-0.548129, 0.552922, -0.627561,
-0.618412, 0.601099, -0.506207,
-0.596780, 0.633437, -0.492556,
-0.639079, 0.567660, -0.518980,
-0.649517, 0.586330, -0.484092,
-0.679467, 0.570576, -0.461267,
-0.658800, 0.604016, -0.448494,
-0.658643, 0.533341, -0.530789,
-0.567660, 0.518980, -0.639079,
-0.586330, 0.484092, -0.649517,
-0.601099, 0.506207, -0.618412,
-0.633437, 0.492556, -0.596780,
-0.604016, 0.448494, -0.658800,
-0.472724, 0.698007, -0.537883,
-0.437129, 0.707290, -0.555570,
-0.507599, 0.687574, -0.519216,
-0.483519, 0.715483, -0.504275,
-0.493588, 0.731779, -0.469969,
-0.458698, 0.742189, -0.488621,
-0.541538, 0.676056, -0.499685,
-0.527794, 0.720310, -0.450096,
-0.503033, 0.747106, -0.434500,
-0.551813, 0.692313, -0.464979,
-0.561193, 0.707553, -0.429454,
-0.593575, 0.693587, -0.408174,
-0.569625, 0.721670, -0.393344,
-0.574937, 0.663299, -0.479043,
-0.511644, 0.761045, -0.398787,
-0.477439, 0.772513, -0.418660,
-0.545041, 0.748288, -0.378147,
-0.519525, 0.773811, -0.362368,
-0.526633, 0.785317, -0.325477,
-0.493236, 0.798073, -0.346117,
-0.577422, 0.734322, -0.356867,
-0.606985, 0.649612, -0.457792,
-0.638080, 0.634846, -0.435687,
-0.616293, 0.664768, -0.422216,
-0.624652, 0.678801, -0.386056,
-0.668030, 0.619094, -0.412866,
-0.634846, 0.435687, -0.638080,
-0.664768, 0.422216, -0.616293,
-0.649612, 0.457792, -0.606985,
-0.663299, 0.479043, -0.574937,
-0.693587, 0.408174, -0.593575,
-0.692313, 0.464979, -0.551813,
-0.676056, 0.499685, -0.541538,
-0.707553, 0.429454, -0.561193,
-0.720310, 0.450096, -0.527794,
-0.747106, 0.434500, -0.503033,
-0.731779, 0.469969, -0.493588,
-0.721670, 0.393344, -0.569625,
-0.687574, 0.519216, -0.507599,
-0.715483, 0.504275, -0.483519,
-0.698007, 0.537883, -0.472724,
-0.707290, 0.555570, -0.437129,
-0.742189, 0.488621, -0.458698,
-0.748288, 0.378147, -0.545041,
-0.773811, 0.362368, -0.519525,
-0.761045, 0.398787, -0.511644,
-0.772513, 0.418660, -0.477439,
-0.798073, 0.346117, -0.493236,
-0.307772, 0.553959, -0.773567,
-0.323518, 0.579446, -0.748050,
-0.289684, 0.580690, -0.760843,
-0.271326, 0.606679, -0.747210,
-0.338701, 0.604015, -0.721421,
-0.286472, 0.631567, -0.720456,
-0.252431, 0.632217, -0.732517,
-0.320203, 0.630114, -0.707408,
-0.301233, 0.655487, -0.692528,
-0.315540, 0.678317, -0.663567,
-0.281913, 0.679996, -0.676853,
-0.353435, 0.627882, -0.693432,
-0.233506, 0.656735, -0.717059,
-0.248290, 0.680710, -0.689192,
-0.214284, 0.680528, -0.700689,
-0.194886, 0.703465, -0.683488,
-0.262621, 0.703597, -0.660289,
-0.367467, 0.650587, -0.664608,
-0.380913, 0.672350, -0.634705,
-0.348411, 0.675790, -0.649552,
-0.329009, 0.700122, -0.633705,
-0.393711, 0.693058, -0.603872,
-0.208302, 0.727361, -0.653878,
-0.174169, 0.726507, -0.664720,
-0.242237, 0.727302, -0.642147,
-0.221448, 0.750093, -0.623154,
-0.234254, 0.771546, -0.591475,
-0.200390, 0.771845, -0.603406,
-0.275772, 0.726347, -0.629580,
-0.267971, 0.770411, -0.578496,
-0.246832, 0.791897, -0.558545,
-0.288850, 0.747877, -0.597700,
-0.301428, 0.768229, -0.564770,
-0.334423, 0.765026, -0.550360,
-0.313441, 0.787296, -0.530962,
-0.309328, 0.724422, -0.616060,
-0.258941, 0.810757, -0.524997,
-0.225309, 0.812164, -0.538168,
-0.292299, 0.808303, -0.511084,
-0.270692, 0.828310, -0.490539,
-0.282027, 0.844463, -0.455349,
-0.248763, 0.847202, -0.469431,
-0.325185, 0.804829, -0.496492,
-0.342087, 0.721652, -0.601826,
-0.374471, 0.717969, -0.586764,
-0.354664, 0.742003, -0.568898,
-0.366677, 0.761070, -0.535090,
-0.406288, 0.713409, -0.570944,
-0.154382, 0.747714, -0.645826,
-0.167523, 0.770441, -0.615107,
-0.134388, 0.768075, -0.626099,
-0.114320, 0.787486, -0.605637,
-0.180328, 0.791892, -0.583429,
-0.127031, 0.809135, -0.573728,
-0.093867, 0.806140, -0.584232,
-0.160105, 0.811161, -0.562480,
-0.139598, 0.829468, -0.540828,
-0.151959, 0.848387, -0.507097,
-0.118945, 0.846717, -0.518577,
-0.192895, 0.812225, -0.550529,
-0.073611, 0.823654, -0.562295,
-0.086178, 0.843986, -0.529397,
-0.053243, 0.840222, -0.539623,
-0.205003, 0.831085, -0.516983,
-0.216754, 0.848634, -0.482533,
-0.184424, 0.849034, -0.495105,
-0.417751, 0.731954, -0.538265,
-0.428549, 0.749420, -0.504694,
-0.397816, 0.755728, -0.520209,
-0.377449, 0.778488, -0.501486,
-0.438625, 0.765722, -0.470402,
-0.387410, 0.794924, -0.466914,
-0.356369, 0.800468, -0.481926,
-0.418038, 0.788414, -0.451273,
-0.396824, 0.810120, -0.431551,
-0.405639, 0.824001, -0.395576,
-0.375114, 0.830721, -0.411329,
-0.448074, 0.780985, -0.435077,
-0.335116, 0.821198, -0.461879,
-0.344498, 0.836323, -0.426480,
-0.313409, 0.840916, -0.441176,
-0.456698, 0.794935, -0.399381,
-0.464593, 0.807701, -0.363003,
-0.435338, 0.816363, -0.379516,
-0.807701, 0.363003, -0.464593,
-0.816363, 0.379516, -0.435338,
-0.794935, 0.399381, -0.456698,
-0.780985, 0.435077, -0.448074,
-0.824001, 0.395576, -0.405639,
-0.788414, 0.451273, -0.418038,
-0.765722, 0.470402, -0.438625,
-0.810120, 0.431551, -0.396824,
-0.794924, 0.466914, -0.387410,
-0.800468, 0.481926, -0.356369,
-0.778488, 0.501486, -0.377449,
-0.830721, 0.411329, -0.375114,
-0.749420, 0.504694, -0.428549,
-0.755728, 0.520209, -0.397816,
-0.731954, 0.538265, -0.417751,
-0.713409, 0.570944, -0.406288,
-0.761070, 0.535090, -0.366677,
-0.836323, 0.426480, -0.344498,
-0.840916, 0.441176, -0.313409,
-0.821198, 0.461879, -0.335116,
-0.804829, 0.496492, -0.325185,
-0.844463, 0.455349, -0.282027,
-0.717969, 0.586764, -0.374471,
-0.693058, 0.603872, -0.393711,
-0.742003, 0.568898, -0.354664,
-0.721652, 0.601826, -0.342087,
-0.724422, 0.616060, -0.309328,
-0.700122, 0.633705, -0.329009,
-0.765026, 0.550360, -0.334423,
-0.747877, 0.597700, -0.288850,
-0.726347, 0.629580, -0.275772,
-0.768229, 0.564770, -0.301428,
-0.770411, 0.578496, -0.267971,
-0.791897, 0.558545, -0.246832,
-0.771546, 0.591475, -0.234254,
-0.787296, 0.530962, -0.313441,
-0.727302, 0.642147, -0.242237,
-0.703597, 0.660289, -0.262621,
-0.750093, 0.623154, -0.221448,
-0.727361, 0.653878, -0.208302,
-0.726507, 0.664720, -0.174169,
-0.703465, 0.683488, -0.194886,
-0.771845, 0.603406, -0.200390,
-0.808303, 0.511084, -0.292299,
-0.828310, 0.490539, -0.270692,
-0.810757, 0.524997, -0.258941,
-0.812164, 0.538168, -0.225309,
-0.847202, 0.469431, -0.248763,
-0.672350, 0.634705, -0.380913,
-0.675790, 0.649552, -0.348411,
-0.650587, 0.664608, -0.367467,
-0.627882, 0.693432, -0.353435,
-0.678317, 0.663567, -0.315540,
-0.630114, 0.707408, -0.320203,
-0.604015, 0.721421, -0.338701,
-0.655487, 0.692528, -0.301233,
-0.631567, 0.720456, -0.286472,
-0.632217, 0.732517, -0.252431,
-0.606679, 0.747210, -0.271326,
-0.679996, 0.676853, -0.281913,
-0.579446, 0.748050, -0.323518,
-0.580690, 0.760843, -0.289684,
-0.553959, 0.773567, -0.307772,
-0.680710, 0.689192, -0.248290,
-0.680528, 0.700689, -0.214284,
-0.656735, 0.717059, -0.233506,
-0.848634, 0.482533, -0.216754,
-0.849034, 0.495105, -0.184424,
-0.831085, 0.516983, -0.205003,
-0.812225, 0.550529, -0.192895,
-0.848387, 0.507097, -0.151959,
-0.811161, 0.562480, -0.160105,
-0.791892, 0.583429, -0.180328,
-0.829468, 0.540828, -0.139598,
-0.809135, 0.573728, -0.127031,
-0.806140, 0.584232, -0.093867,
-0.787486, 0.605637, -0.114320,
-0.846717, 0.518577, -0.118945,
-0.770441, 0.615107, -0.167523,
-0.768075, 0.626099, -0.134388,
-0.747714, 0.645826, -0.154382,
-0.843986, 0.529397, -0.086178,
-0.840222, 0.539623, -0.053243,
-0.823654, 0.562295, -0.073611,
-0.539623, -0.053243, 0.840222,
-0.562295, -0.073611, 0.823654,
-0.529397, -0.086178, 0.843986,
-0.518577, -0.118945, 0.846717,
-0.584232, -0.093867, 0.806140,
-0.540828, -0.139598, 0.829468,
-0.507097, -0.151959, 0.848387,
-0.573728, -0.127031, 0.809135,
-0.562480, -0.160105, 0.811161,
-0.583429, -0.180328, 0.791892,
-0.550529, -0.192895, 0.812225,
-0.605637, -0.114320, 0.787486,
-0.495105, -0.184424, 0.849034,
-0.516983, -0.205003, 0.831085,
-0.482533, -0.216754, 0.848634,
-0.469431, -0.248763, 0.847202,
-0.538168, -0.225309, 0.812164,
-0.626099, -0.134388, 0.768075,
-0.645826, -0.154382, 0.747714,
-0.615107, -0.167523, 0.770441,
-0.603406, -0.200390, 0.771845,
-0.664720, -0.174169, 0.726507,
-0.490539, -0.270692, 0.828310,
-0.455349, -0.282027, 0.844463,
-0.524997, -0.258941, 0.810757,
-0.511084, -0.292299, 0.808303,
-0.530962, -0.313441, 0.787296,
-0.496492, -0.325185, 0.804829,
-0.558545, -0.246832, 0.791897,
-0.564770, -0.301428, 0.768229,
-0.550360, -0.334423, 0.765026,
-0.578496, -0.267971, 0.770411,
-0.597700, -0.288850, 0.747877,
-0.629580, -0.275772, 0.726347,
-0.616060, -0.309328, 0.724422,
-0.591475, -0.234254, 0.771546,
-0.568898, -0.354664, 0.742003,
-0.535090, -0.366677, 0.761070,
-0.601826, -0.342087, 0.721652,
-0.586764, -0.374471, 0.717969,
-0.603872, -0.393711, 0.693058,
-0.570944, -0.406288, 0.713409,
-0.633705, -0.329009, 0.700122,
-0.623154, -0.221448, 0.750093,
-0.653878, -0.208302, 0.727361,
-0.642147, -0.242237, 0.727302,
-0.660289, -0.262621, 0.703597,
-0.683488, -0.194886, 0.703465,
-0.441176, -0.313409, 0.840916,
-0.461879, -0.335116, 0.821198,
-0.426480, -0.344498, 0.836323,
-0.411329, -0.375114, 0.830721,
-0.481926, -0.356369, 0.800468,
-0.431551, -0.396824, 0.810120,
-0.395576, -0.405639, 0.824001,
-0.466914, -0.387410, 0.794924,
-0.451273, -0.418038, 0.788414,
-0.470402, -0.438625, 0.765722,
-0.435077, -0.448074, 0.780985,
-0.501486, -0.377449, 0.778488,
-0.379516, -0.435338, 0.816363,
-0.399381, -0.456698, 0.794935,
-0.363003, -0.464593, 0.807701,
-0.346117, -0.493236, 0.798073,
-0.418660, -0.477439, 0.772513,
-0.520209, -0.397816, 0.755728,
-0.538265, -0.417751, 0.731954,
-0.504694, -0.428549, 0.749420,
-0.488621, -0.458698, 0.742189,
-0.555570, -0.437129, 0.707290,
-0.700689, -0.214284, 0.680528,
-0.717059, -0.233506, 0.656735,
-0.689192, -0.248290, 0.680710,
-0.676853, -0.281913, 0.679996,
-0.732517, -0.252431, 0.632217,
-0.692528, -0.301233, 0.655487,
-0.663567, -0.315540, 0.678317,
-0.720456, -0.286472, 0.631567,
-0.707408, -0.320203, 0.630114,
-0.721421, -0.338701, 0.604015,
-0.693432, -0.353435, 0.627882,
-0.747210, -0.271326, 0.606679,
-0.649552, -0.348411, 0.675790,
-0.664608, -0.367467, 0.650587,
-0.634705, -0.380913, 0.672350,
-0.619094, -0.412866, 0.668030,
-0.678801, -0.386056, 0.624652,
-0.760843, -0.289684, 0.580690,
-0.773567, -0.307772, 0.553959,
-0.748050, -0.323518, 0.579446,
-0.734322, -0.356867, 0.577422,
-0.785317, -0.325477, 0.526633,
-0.362368, -0.519525, 0.773811,
-0.325477, -0.526633, 0.785317,
-0.398787, -0.511644, 0.761045,
-0.378147, -0.545041, 0.748288,
-0.393344, -0.569625, 0.721670,
-0.356867, -0.577422, 0.734322,
-0.434500, -0.503033, 0.747106,
-0.429454, -0.561193, 0.707553,
-0.408174, -0.593575, 0.693587,
-0.450096, -0.527794, 0.720310,
-0.464979, -0.551813, 0.692313,
-0.499685, -0.541538, 0.676056,
-0.479043, -0.574937, 0.663299,
-0.469969, -0.493588, 0.731779,
-0.422216, -0.616293, 0.664768,
-0.386056, -0.624652, 0.678801,
-0.457792, -0.606985, 0.649612,
-0.435687, -0.638080, 0.634846,
-0.448494, -0.658800, 0.604016,
-0.412866, -0.668030, 0.619094,
-0.492556, -0.596780, 0.633437,
-0.504275, -0.483519, 0.715483,
-0.537883, -0.472724, 0.698007,
-0.519216, -0.507599, 0.687574,
-0.533341, -0.530789, 0.658643,
-0.570576, -0.461267, 0.679467,
-0.484092, -0.649517, 0.586330,
-0.461267, -0.679467, 0.570576,
-0.506207, -0.618412, 0.601099,
-0.518980, -0.639079, 0.567660,
-0.552922, -0.627561, 0.548129,
-0.530789, -0.658643, 0.533341,
-0.527461, -0.586364, 0.614787,
-0.574554, -0.595223, 0.561780,
-0.586364, -0.614787, 0.527461,
-0.561780, -0.574554, 0.595223,
-0.595223, -0.561780, 0.574554,
-0.614787, -0.527461, 0.586364,
-0.627561, -0.548129, 0.552922,
-0.548129, -0.552922, 0.627561,
-0.618412, -0.601099, 0.506207,
-0.596780, -0.633437, 0.492556,
-0.639079, -0.567660, 0.518980,
-0.649517, -0.586330, 0.484092,
-0.679467, -0.570576, 0.461267,
-0.658800, -0.604016, 0.448494,
-0.658643, -0.533341, 0.530789,
-0.567660, -0.518980, 0.639079,
-0.586330, -0.484092, 0.649517,
-0.601099, -0.506207, 0.618412,
-0.633437, -0.492556, 0.596780,
-0.604016, -0.448494, 0.658800,
-0.472724, -0.698007, 0.537883,
-0.437129, -0.707290, 0.555570,
-0.507599, -0.687574, 0.519216,
-0.483519, -0.715483, 0.504275,
-0.493588, -0.731779, 0.469969,
-0.458698, -0.742189, 0.488621,
-0.541538, -0.676056, 0.499685,
-0.527794, -0.720310, 0.450096,
-0.503033, -0.747106, 0.434500,
-0.551813, -0.692313, 0.464979,
-0.561193, -0.707553, 0.429454,
-0.593575, -0.693587, 0.408174,
-0.569625, -0.721670, 0.393344,
-0.574937, -0.663299, 0.479043,
-0.511644, -0.761045, 0.398787,
-0.477439, -0.772513, 0.418660,
-0.545041, -0.748288, 0.378147,
-0.519525, -0.773811, 0.362368,
-0.526633, -0.785317, 0.325477,
-0.493236, -0.798073, 0.346117,
-0.577422, -0.734322, 0.356867,
-0.606985, -0.649612, 0.457792,
-0.638080, -0.634846, 0.435687,
-0.616293, -0.664768, 0.422216,
-0.624652, -0.678801, 0.386056,
-0.668030, -0.619094, 0.412866,
-0.634846, -0.435687, 0.638080,
-0.664768, -0.422216, 0.616293,
-0.649612, -0.457792, 0.606985,
-0.663299, -0.479043, 0.574937,
-0.693587, -0.408174, 0.593575,
-0.692313, -0.464979, 0.551813,
-0.676056, -0.499685, 0.541538,
-0.707553, -0.429454, 0.561193,
-0.720310, -0.450096, 0.527794,
-0.747106, -0.434500, 0.503033,
-0.731779, -0.469969, 0.493588,
-0.721670, -0.393344, 0.569625,
-0.687574, -0.519216, 0.507599,
-0.715483, -0.504275, 0.483519,
-0.698007, -0.537883, 0.472724,
-0.707290, -0.555570, 0.437129,
-0.742189, -0.488621, 0.458698,
-0.748288, -0.378147, 0.545041,
-0.773811, -0.362368, 0.519525,
-0.761045, -0.398787, 0.511644,
-0.772513, -0.418660, 0.477439,
-0.798073, -0.346117, 0.493236,
-0.307772, -0.553959, 0.773567,
-0.323518, -0.579446, 0.748050,
-0.289684, -0.580690, 0.760843,
-0.271326, -0.606679, 0.747210,
-0.338701, -0.604015, 0.721421,
-0.286472, -0.631567, 0.720456,
-0.252431, -0.632217, 0.732517,
-0.320203, -0.630114, 0.707408,
-0.301233, -0.655487, 0.692528,
-0.315540, -0.678317, 0.663567,
-0.281913, -0.679996, 0.676853,
-0.353435, -0.627882, 0.693432,
-0.233506, -0.656735, 0.717059,
-0.248290, -0.680710, 0.689192,
-0.214284, -0.680528, 0.700689,
-0.194886, -0.703465, 0.683488,
-0.262621, -0.703597, 0.660289,
-0.367467, -0.650587, 0.664608,
-0.380913, -0.672350, 0.634705,
-0.348411, -0.675790, 0.649552,
-0.329009, -0.700122, 0.633705,
-0.393711, -0.693058, 0.603872,
-0.208302, -0.727361, 0.653878,
-0.174169, -0.726507, 0.664720,
-0.242237, -0.727302, 0.642147,
-0.221448, -0.750093, 0.623154,
-0.234254, -0.771546, 0.591475,
-0.200390, -0.771845, 0.603406,
-0.275772, -0.726347, 0.629580,
-0.267971, -0.770411, 0.578496,
-0.246832, -0.791897, 0.558545,
-0.288850, -0.747877, 0.597700,
-0.301428, -0.768229, 0.564770,
-0.334423, -0.765026, 0.550360,
-0.313441, -0.787296, 0.530962,
-0.309328, -0.724422, 0.616060,
-0.258941, -0.810757, 0.524997,
-0.225309, -0.812164, 0.538168,
-0.292299, -0.808303, 0.511084,
-0.270692, -0.828310, 0.490539,
-0.282027, -0.844463, 0.455349,
-0.248763, -0.847202, 0.469431,
-0.325185, -0.804829, 0.496492,
-0.342087, -0.721652, 0.601826,
-0.374471, -0.717969, 0.586764,
-0.354664, -0.742003, 0.568898,
-0.366677, -0.761070, 0.535090,
-0.406288, -0.713409, 0.570944,
-0.154382, -0.747714, 0.645826,
-0.167523, -0.770441, 0.615107,
-0.134388, -0.768075, 0.626099,
-0.114320, -0.787486, 0.605637,
-0.180328, -0.791892, 0.583429,
-0.127031, -0.809135, 0.573728,
-0.093867, -0.806140, 0.584232,
-0.160105, -0.811161, 0.562480,
-0.139598, -0.829468, 0.540828,
-0.151959, -0.848387, 0.507097,
-0.118945, -0.846717, 0.518577,
-0.192895, -0.812225, 0.550529,
-0.073611, -0.823654, 0.562295,
-0.086178, -0.843986, 0.529397,
-0.053243, -0.840222, 0.539623,
-0.205003, -0.831085, 0.516983,
-0.216754, -0.848634, 0.482533,
-0.184424, -0.849034, 0.495105,
-0.417751, -0.731954, 0.538265,
-0.428549, -0.749420, 0.504694,
-0.397816, -0.755728, 0.520209,
-0.377449, -0.778488, 0.501486,
-0.438625, -0.765722, 0.470402,
-0.387410, -0.794924, 0.466914,
-0.356369, -0.800468, 0.481926,
-0.418038, -0.788414, 0.451273,
-0.396824, -0.810120, 0.431551,
-0.405639, -0.824001, 0.395576,
-0.375114, -0.830721, 0.411329,
-0.448074, -0.780985, 0.435077,
-0.335116, -0.821198, 0.461879,
-0.344498, -0.836323, 0.426480,
-0.313409, -0.840916, 0.441176,
-0.456698, -0.794935, 0.399381,
-0.464593, -0.807701, 0.363003,
-0.435338, -0.816363, 0.379516,
-0.807701, -0.363003, 0.464593,
-0.816363, -0.379516, 0.435338,
-0.794935, -0.399381, 0.456698,
-0.780985, -0.435077, 0.448074,
-0.824001, -0.395576, 0.405639,
-0.788414, -0.451273, 0.418038,
-0.765722, -0.470402, 0.438625,
-0.810120, -0.431551, 0.396824,
-0.794924, -0.466914, 0.387410,
-0.800468, -0.481926, 0.356369,
-0.778488, -0.501486, 0.377449,
-0.830721, -0.411329, 0.375114,
-0.749420, -0.504694, 0.428549,
-0.755728, -0.520209, 0.397816,
-0.731954, -0.538265, 0.417751,
-0.713409, -0.570944, 0.406288,
-0.761070, -0.535090, 0.366677,
-0.836323, -0.426480, 0.344498,
-0.840916, -0.441176, 0.313409,
-0.821198, -0.461879, 0.335116,
-0.804829, -0.496492, 0.325185,
-0.844463, -0.455349, 0.282027,
-0.717969, -0.586764, 0.374471,
-0.693058, -0.603872, 0.393711,
-0.742003, -0.568898, 0.354664,
-0.721652, -0.601826, 0.342087,
-0.724422, -0.616060, 0.309328,
-0.700122, -0.633705, 0.329009,
-0.765026, -0.550360, 0.334423,
-0.747877, -0.597700, 0.288850,
-0.726347, -0.629580, 0.275772,
-0.768229, -0.564770, 0.301428,
-0.770411, -0.578496, 0.267971,
-0.791897, -0.558545, 0.246832,
-0.771546, -0.591475, 0.234254,
-0.787296, -0.530962, 0.313441,
-0.727302, -0.642147, 0.242237,
-0.703597, -0.660289, 0.262621,
-0.750093, -0.623154, 0.221448,
-0.727361, -0.653878, 0.208302,
-0.726507, -0.664720, 0.174169,
-0.703465, -0.683488, 0.194886,
-0.771845, -0.603406, 0.200390,
-0.808303, -0.511084, 0.292299,
-0.828310, -0.490539, 0.270692,
-0.810757, -0.524997, 0.258941,
-0.812164, -0.538168, 0.225309,
-0.847202, -0.469431, 0.248763,
-0.672350, -0.634705, 0.380913,
-0.675790, -0.649552, 0.348411,
-0.650587, -0.664608, 0.367467,
-0.627882, -0.693432, 0.353435,
-0.678317, -0.663567, 0.315540,
-0.630114, -0.707408, 0.320203,
-0.604015, -0.721421, 0.338701,
-0.655487, -0.692528, 0.301233,
-0.631567, -0.720456, 0.286472,
-0.632217, -0.732517, 0.252431,
-0.606679, -0.747210, 0.271326,
-0.679996, -0.676853, 0.281913,
-0.579446, -0.748050, 0.323518,
-0.580690, -0.760843, 0.289684,
-0.553959, -0.773567, 0.307772,
-0.680710, -0.689192, 0.248290,
-0.680528, -0.700689, 0.214284,
-0.656735, -0.717059, 0.233506,
-0.848634, -0.482533, 0.216754,
-0.849034, -0.495105, 0.184424,
-0.831085, -0.516983, 0.205003,
-0.812225, -0.550529, 0.192895,
-0.848387, -0.507097, 0.151959,
-0.811161, -0.562480, 0.160105,
-0.791892, -0.583429, 0.180328,
-0.829468, -0.540828, 0.139598,
-0.809135, -0.573728, 0.127031,
-0.806140, -0.584232, 0.093867,
-0.787486, -0.605637, 0.114320,
-0.846717, -0.518577, 0.118945,
-0.770441, -0.615107, 0.167523,
-0.768075, -0.626099, 0.134388,
-0.747714, -0.645826, 0.154382,
-0.843986, -0.529397, 0.086178,
-0.840222, -0.539623, 0.053243,
-0.823654, -0.562295, 0.073611,
-0.539623, -0.053243, -0.840222,
-0.529397, -0.086178, -0.843986,
-0.562295, -0.073611, -0.823654,
-0.584232, -0.093867, -0.806140,
-0.518577, -0.118945, -0.846717,
-0.573728, -0.127031, -0.809135,
-0.605637, -0.114320, -0.787486,
-0.540828, -0.139598, -0.829468,
-0.562480, -0.160105, -0.811161,
-0.550529, -0.192895, -0.812225,
-0.583429, -0.180328, -0.791892,
-0.507097, -0.151959, -0.848387,
-0.626099, -0.134388, -0.768075,
-0.615107, -0.167523, -0.770441,
-0.645826, -0.154382, -0.747714,
-0.664720, -0.174169, -0.726507,
-0.603406, -0.200390, -0.771845,
-0.495105, -0.184424, -0.849034,
-0.482533, -0.216754, -0.848634,
-0.516983, -0.205003, -0.831085,
-0.538168, -0.225309, -0.812164,
-0.469431, -0.248763, -0.847202,
-0.653878, -0.208302, -0.727361,
-0.683488, -0.194886, -0.703465,
-0.623154, -0.221448, -0.750093,
-0.642147, -0.242237, -0.727302,
-0.629580, -0.275772, -0.726347,
-0.660289, -0.262621, -0.703597,
-0.591475, -0.234254, -0.771546,
-0.597700, -0.288850, -0.747877,
-0.616060, -0.309328, -0.724422,
-0.578496, -0.267971, -0.770411,
-0.564770, -0.301428, -0.768229,
-0.530962, -0.313441, -0.787296,
-0.550360, -0.334423, -0.765026,
-0.558545, -0.246832, -0.791897,
-0.601826, -0.342087, -0.721652,
-0.633705, -0.329009, -0.700122,
-0.568898, -0.354664, -0.742003,
-0.586764, -0.374471, -0.717969,
-0.570944, -0.406288, -0.713409,
-0.603872, -0.393711, -0.693058,
-0.535090, -0.366677, -0.761070,
-0.524997, -0.258941, -0.810757,
-0.490539, -0.270692, -0.828310,
-0.511084, -0.292299, -0.808303,
-0.496492, -0.325185, -0.804829,
-0.455349, -0.282027, -0.844463,
-0.700689, -0.214284, -0.680528,
-0.689192, -0.248290, -0.680710,
-0.717059, -0.233506, -0.656735,
-0.732517, -0.252431, -0.632217,
-0.676853, -0.281913, -0.679996,
-0.720456, -0.286472, -0.631567,
-0.747210, -0.271326, -0.606679,
-0.692528, -0.301233, -0.655487,
-0.707408, -0.320203, -0.630114,
-0.693432, -0.353435, -0.627882,
-0.721421, -0.338701, -0.604015,
-0.663567, -0.315540, -0.678317,
-0.760843, -0.289684, -0.580690,
-0.748050, -0.323518, -0.579446,
-0.773567, -0.307772, -0.553959,
-0.785317, -0.325477, -0.526633,
-0.734322, -0.356867, -0.577422,
-0.649552, -0.348411, -0.675790,
-0.634705, -0.380913, -0.672350,
-0.664608, -0.367467, -0.650587,
-0.678801, -0.386056, -0.624652,
-0.619094, -0.412866, -0.668030,
-0.441176, -0.313409, -0.840916,
-0.426480, -0.344498, -0.836323,
-0.461879, -0.335116, -0.821198,
-0.481926, -0.356369, -0.800468,
-0.411329, -0.375114, -0.830721,
-0.466914, -0.387410, -0.794924,
-0.501486, -0.377449, -0.778488,
-0.431551, -0.396824, -0.810120,
-0.451273, -0.418038, -0.788414,
-0.435077, -0.448074, -0.780985,
-0.470402, -0.438625, -0.765722,
-0.395576, -0.405639, -0.824001,
-0.520209, -0.397816, -0.755728,
-0.504694, -0.428549, -0.749420,
-0.538265, -0.417751, -0.731954,
-0.555570, -0.437129, -0.707290,
-0.488621, -0.458698, -0.742189,
-0.379516, -0.435338, -0.816363,
-0.363003, -0.464593, -0.807701,
-0.399381, -0.456698, -0.794935,
-0.418660, -0.477439, -0.772513,
-0.346117, -0.493236, -0.798073,
-0.773811, -0.362368, -0.519525,
-0.798073, -0.346117, -0.493236,
-0.748288, -0.378147, -0.545041,
-0.761045, -0.398787, -0.511644,
-0.747106, -0.434500, -0.503033,
-0.772513, -0.418660, -0.477439,
-0.721670, -0.393344, -0.569625,
-0.720310, -0.450096, -0.527794,
-0.731779, -0.469969, -0.493588,
-0.707553, -0.429454, -0.561193,
-0.692313, -0.464979, -0.551813,
-0.663299, -0.479043, -0.574937,
-0.676056, -0.499685, -0.541538,
-0.693587, -0.408174, -0.593575,
-0.715483, -0.504275, -0.483519,
-0.742189, -0.488621, -0.458698,
-0.687574, -0.519216, -0.507599,
-0.698007, -0.537883, -0.472724,
-0.679467, -0.570576, -0.461267,
-0.707290, -0.555570, -0.437129,
-0.658643, -0.533341, -0.530789,
-0.664768, -0.422216, -0.616293,
-0.634846, -0.435687, -0.638080,
-0.649612, -0.457792, -0.606985,
-0.633437, -0.492556, -0.596780,
-0.604016, -0.448494, -0.658800,
-0.649517, -0.586330, -0.484092,
-0.658800, -0.604016, -0.448494,
-0.639079, -0.567660, -0.518980,
-0.618412, -0.601099, -0.506207,
-0.586364, -0.614787, -0.527461,
-0.596780, -0.633437, -0.492556,
-0.627561, -0.548129, -0.552922,
-0.574554, -0.595223, -0.561780,
-0.552922, -0.627561, -0.548129,
-0.595223, -0.561780, -0.574554,
-0.561780, -0.574554, -0.595223,
-0.548129, -0.552922, -0.627561,
-0.527461, -0.586364, -0.614787,
-0.614787, -0.527461, -0.586364,
-0.518980, -0.639079, -0.567660,
-0.530789, -0.658643, -0.533341,
-0.506207, -0.618412, -0.601099,
-0.484092, -0.649517, -0.586330,
-0.448494, -0.658800, -0.604016,
-0.461267, -0.679467, -0.570576,
-0.492556, -0.596780, -0.633437,
-0.601099, -0.506207, -0.618412,
-0.586330, -0.484092, -0.649517,
-0.567660, -0.518980, -0.639079,
-0.533341, -0.530789, -0.658643,
-0.570576, -0.461267, -0.679467,
-0.638080, -0.634846, -0.435687,
-0.668030, -0.619094, -0.412866,
-0.606985, -0.649612, -0.457792,
-0.616293, -0.664768, -0.422216,
-0.593575, -0.693587, -0.408174,
-0.624652, -0.678801, -0.386056,
-0.574937, -0.663299, -0.479043,
-0.561193, -0.707553, -0.429454,
-0.569625, -0.721670, -0.393344,
-0.551813, -0.692313, -0.464979,
-0.527794, -0.720310, -0.450096,
-0.493588, -0.731779, -0.469969,
-0.503033, -0.747106, -0.434500,
-0.541538, -0.676056, -0.499685,
-0.545041, -0.748288, -0.378147,
-0.577422, -0.734322, -0.356867,
-0.511644, -0.761045, -0.398787,
-0.519525, -0.773811, -0.362368,
-0.493236, -0.798073, -0.346117,
-0.526633, -0.785317, -0.325477,
-0.477439, -0.772513, -0.418660,
-0.507599, -0.687574, -0.519216,
-0.472724, -0.698007, -0.537883,
-0.483519, -0.715483, -0.504275,
-0.458698, -0.742189, -0.488621,
-0.437129, -0.707290, -0.555570,
-0.537883, -0.472724, -0.698007,
-0.504275, -0.483519, -0.715483,
-0.519216, -0.507599, -0.687574,
-0.499685, -0.541538, -0.676056,
-0.469969, -0.493588, -0.731779,
-0.464979, -0.551813, -0.692313,
-0.479043, -0.574937, -0.663299,
-0.450096, -0.527794, -0.720310,
-0.429454, -0.561193, -0.707553,
-0.393344, -0.569625, -0.721670,
-0.408174, -0.593575, -0.693587,
-0.434500, -0.503033, -0.747106,
-0.457792, -0.606985, -0.649612,
-0.422216, -0.616293, -0.664768,
-0.435687, -0.638080, -0.634846,
-0.412866, -0.668030, -0.619094,
-0.386056, -0.624652, -0.678801,
-0.398787, -0.511644, -0.761045,
-0.362368, -0.519525, -0.773811,
-0.378147, -0.545041, -0.748288,
-0.356867, -0.577422, -0.734322,
-0.325477, -0.526633, -0.785317,
-0.807701, -0.363003, -0.464593,
-0.794935, -0.399381, -0.456698,
-0.816363, -0.379516, -0.435338,
-0.824001, -0.395576, -0.405639,
-0.780985, -0.435077, -0.448074,
-0.810120, -0.431551, -0.396824,
-0.830721, -0.411329, -0.375114,
-0.788414, -0.451273, -0.418038,
-0.794924, -0.466914, -0.387410,
-0.778488, -0.501486, -0.377449,
-0.800468, -0.481926, -0.356369,
-0.765722, -0.470402, -0.438625,
-0.836323, -0.426480, -0.344498,
-0.821198, -0.461879, -0.335116,
-0.840916, -0.441176, -0.313409,
-0.844463, -0.455349, -0.282027,
-0.804829, -0.496492, -0.325185,
-0.749420, -0.504694, -0.428549,
-0.731954, -0.538265, -0.417751,
-0.755728, -0.520209, -0.397816,
-0.761070, -0.535090, -0.366677,
-0.713409, -0.570944, -0.406288,
-0.828310, -0.490539, -0.270692,
-0.847202, -0.469431, -0.248763,
-0.808303, -0.511084, -0.292299,
-0.810757, -0.524997, -0.258941,
-0.791897, -0.558545, -0.246832,
-0.812164, -0.538168, -0.225309,
-0.787296, -0.530962, -0.313441,
-0.770411, -0.578496, -0.267971,
-0.771546, -0.591475, -0.234254,
-0.768229, -0.564770, -0.301428,
-0.747877, -0.597700, -0.288850,
-0.724422, -0.616060, -0.309328,
-0.726347, -0.629580, -0.275772,
-0.765026, -0.550360, -0.334423,
-0.750093, -0.623154, -0.221448,
-0.771845, -0.603406, -0.200390,
-0.727302, -0.642147, -0.242237,
-0.727361, -0.653878, -0.208302,
-0.703465, -0.683488, -0.194886,
-0.726507, -0.664720, -0.174169,
-0.703597, -0.660289, -0.262621,
-0.742003, -0.568898, -0.354664,
-0.717969, -0.586764, -0.374471,
-0.721652, -0.601826, -0.342087,
-0.700122, -0.633705, -0.329009,
-0.693058, -0.603872, -0.393711,
-0.848634, -0.482533, -0.216754,
-0.831085, -0.516983, -0.205003,
-0.849034, -0.495105, -0.184424,
-0.848387, -0.507097, -0.151959,
-0.812225, -0.550529, -0.192895,
-0.829468, -0.540828, -0.139598,
-0.846717, -0.518577, -0.118945,
-0.811161, -0.562480, -0.160105,
-0.809135, -0.573728, -0.127031,
-0.787486, -0.605637, -0.114320,
-0.806140, -0.584232, -0.093867,
-0.791892, -0.583429, -0.180328,
-0.843986, -0.529397, -0.086178,
-0.823654, -0.562295, -0.073611,
-0.840222, -0.539623, -0.053243,
-0.770441, -0.615107, -0.167523,
-0.747714, -0.645826, -0.154382,
-0.768075, -0.626099, -0.134388,
-0.672350, -0.634705, -0.380913,
-0.650587, -0.664608, -0.367467,
-0.675790, -0.649552, -0.348411,
-0.678317, -0.663567, -0.315540,
-0.627882, -0.693432, -0.353435,
-0.655487, -0.692528, -0.301233,
-0.679996, -0.676853, -0.281913,
-0.630114, -0.707408, -0.320203,
-0.631567, -0.720456, -0.286472,
-0.606679, -0.747210, -0.271326,
-0.632217, -0.732517, -0.252431,
-0.604015, -0.721421, -0.338701,
-0.680710, -0.689192, -0.248290,
-0.656735, -0.717059, -0.233506,
-0.680528, -0.700689, -0.214284,
-0.579446, -0.748050, -0.323518,
-0.553959, -0.773567, -0.307772,
-0.580690, -0.760843, -0.289684,
-0.307772, -0.553959, -0.773567,
-0.289684, -0.580690, -0.760843,
-0.323518, -0.579446, -0.748050,
-0.338701, -0.604015, -0.721421,
-0.271326, -0.606679, -0.747210,
-0.320203, -0.630114, -0.707408,
-0.353435, -0.627882, -0.693432,
-0.286472, -0.631567, -0.720456,
-0.301233, -0.655487, -0.692528,
-0.281913, -0.679996, -0.676853,
-0.315540, -0.678317, -0.663567,
-0.252431, -0.632217, -0.732517,
-0.367467, -0.650587, -0.664608,
-0.348411, -0.675790, -0.649552,
-0.380913, -0.672350, -0.634705,
-0.393711, -0.693058, -0.603872,
-0.329009, -0.700122, -0.633705,
-0.233506, -0.656735, -0.717059,
-0.214284, -0.680528, -0.700689,
-0.248290, -0.680710, -0.689192,
-0.262621, -0.703597, -0.660289,
-0.194886, -0.703465, -0.683488,
-0.374471, -0.717969, -0.586764,
-0.406288, -0.713409, -0.570944,
-0.342087, -0.721652, -0.601826,
-0.354664, -0.742003, -0.568898,
-0.334423, -0.765026, -0.550360,
-0.366677, -0.761070, -0.535090,
-0.309328, -0.724422, -0.616060,
-0.301428, -0.768229, -0.564770,
-0.313441, -0.787296, -0.530962,
-0.288850, -0.747877, -0.597700,
-0.267971, -0.770411, -0.578496,
-0.234254, -0.771546, -0.591475,
-0.246832, -0.791897, -0.558545,
-0.275772, -0.726347, -0.629580,
-0.292299, -0.808303, -0.511084,
-0.325185, -0.804829, -0.496492,
-0.258941, -0.810757, -0.524997,
-0.270692, -0.828310, -0.490539,
-0.248763, -0.847202, -0.469431,
-0.282027, -0.844463, -0.455349,
-0.225309, -0.812164, -0.538168,
-0.242237, -0.727302, -0.642147,
-0.208302, -0.727361, -0.653878,
-0.221448, -0.750093, -0.623154,
-0.200390, -0.771845, -0.603406,
-0.174169, -0.726507, -0.664720,
-0.417751, -0.731954, -0.538265,
-0.397816, -0.755728, -0.520209,
-0.428549, -0.749420, -0.504694,
-0.438625, -0.765722, -0.470402,
-0.377449, -0.778488, -0.501486,
-0.418038, -0.788414, -0.451273,
-0.448074, -0.780985, -0.435077,
-0.387410, -0.794924, -0.466914,
-0.396824, -0.810120, -0.431551,
-0.375114, -0.830721, -0.411329,
-0.405639, -0.824001, -0.395576,
-0.356369, -0.800468, -0.481926,
-0.456698, -0.794935, -0.399381,
-0.435338, -0.816363, -0.379516,
-0.464593, -0.807701, -0.363003,
-0.335116, -0.821198, -0.461879,
-0.313409, -0.840916, -0.441176,
-0.344498, -0.836323, -0.426480,
-0.154382, -0.747714, -0.645826,
-0.134388, -0.768075, -0.626099,
-0.167523, -0.770441, -0.615107,
-0.180328, -0.791892, -0.583429,
-0.114320, -0.787486, -0.605637,
-0.160105, -0.811161, -0.562480,
-0.192895, -0.812225, -0.550529,
-0.127031, -0.809135, -0.573728,
-0.139598, -0.829468, -0.540828,
-0.118945, -0.846717, -0.518577,
-0.151959, -0.848387, -0.507097,
-0.093867, -0.806140, -0.584232,
-0.205003, -0.831085, -0.516983,
-0.184424, -0.849034, -0.495105,
-0.216754, -0.848634, -0.482533,
-0.073611, -0.823654, -0.562295,
-0.053243, -0.840222, -0.539623,
-0.086178, -0.843986, -0.529397};

    Points << v;
    Points << Points.t();
    Points.Release();
    return Points;
    }  

  }; 
}

#endif
