/* Copyright (C) 2007-2021 Open Information Security Foundation
 *
 * You can copy, redistribute or modify this Program under the terms of
 * the GNU General Public License version 2 as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

/**
 * \ingroup httplayer
 *
 * @{
 */

/**
 * \file
 *
 * \author Victor Julien <victor@inliniac.net>
 *
 * Implements http_start
 */

#include "suricata-common.h"
#include "threads.h"
#include "decode.h"

#include "detect.h"
#include "detect-parse.h"
#include "detect-engine.h"
#include "detect-engine-buffer.h"
#include "detect-engine-mpm.h"
#include "detect-engine-state.h"
#include "detect-engine-prefilter.h"
#include "detect-engine-content-inspection.h"
#include "detect-content.h"
#include "detect-pcre.h"
#include "detect-http-header-common.h"
#include "detect-http-start.h"

#include "flow.h"
#include "flow-var.h"
#include "flow-util.h"

#include "util-debug.h"
#include "util-unittest.h"
#include "util-unittest-helper.h"
#include "util-spm.h"
#include "util-print.h"

#include "app-layer.h"
#include "app-layer-parser.h"

#include "app-layer-htp.h"
#include "detect-http-header.h"
#include "stream-tcp.h"

#define KEYWORD_NAME "http.start"
#define KEYWORD_NAME_LEGACY "http_start"
#define KEYWORD_DOC "http-keywords.html#http-start"
#define BUFFER_NAME "http_start"
#define BUFFER_DESC "http start: request/response line + headers"
static int g_buffer_id = 0;
static int g_keyword_thread_id = 0;

#define BUFFER_SIZE_STEP    2048
static HttpHeaderThreadDataConfig g_td_config = { BUFFER_SIZE_STEP };

static uint8_t *GetBufferForTX(
        htp_tx_t *tx, DetectEngineThreadCtx *det_ctx, Flow *f, uint8_t flags, uint32_t *buffer_len)
{
    *buffer_len = 0;

    HttpHeaderThreadData *hdr_td = NULL;
    HttpHeaderBuffer *buf =
            HttpHeaderGetBufferSpace(det_ctx, f, flags, g_keyword_thread_id, &hdr_td);
    if (unlikely(buf == NULL)) {
        return NULL;
    }

    const bstr *line = NULL;
    const htp_headers_t *headers;
    if (flags & STREAM_TOSERVER) {
        if (AppLayerParserGetStateProgress(IPPROTO_TCP, ALPROTO_HTTP1, tx, flags) <=
                HTP_REQUEST_PROGRESS_HEADERS)
            return NULL;
        line = htp_tx_request_line(tx);
        headers = htp_tx_request_headers(tx);
    } else {
        if (AppLayerParserGetStateProgress(IPPROTO_TCP, ALPROTO_HTTP1, tx, flags) <=
                HTP_RESPONSE_PROGRESS_HEADERS)
            return NULL;
        headers = htp_tx_response_headers(tx);
        line = htp_tx_response_line(tx);
    }
    if (line == NULL || headers == NULL)
        return NULL;

    size_t line_size = bstr_len(line) + 2;
    if (line_size + buf->len > buf->size) {
        if (HttpHeaderExpandBuffer(hdr_td, buf, line_size) != 0) {
            return NULL;
        }
    }
    memcpy(buf->buffer + buf->len, bstr_ptr(line), bstr_size(line));
    buf->len += bstr_size(line);
    buf->buffer[buf->len++] = '\r';
    buf->buffer[buf->len++] = '\n';

    size_t i = 0;
    size_t no_of_headers = htp_headers_size(headers);
    for (; i < no_of_headers; i++) {
        const htp_header_t *h = htp_headers_get_index(headers, i);
        size_t size1 = htp_header_name_len(h);
        size_t size2 = htp_header_value_len(h);
        size_t size = size1 + size2 + 4;
        if (i + 1 == no_of_headers)
            size += 2;
        if (size + buf->len > buf->size) {
            if (HttpHeaderExpandBuffer(hdr_td, buf, size) != 0) {
                return NULL;
            }
        }

        memcpy(buf->buffer + buf->len, htp_header_name_ptr(h), htp_header_name_len(h));
        buf->len += htp_header_name_len(h);
        buf->buffer[buf->len++] = ':';
        buf->buffer[buf->len++] = ' ';
        memcpy(buf->buffer + buf->len, htp_header_value_ptr(h), htp_header_value_len(h));
        buf->len += htp_header_value_len(h);
        buf->buffer[buf->len++] = '\r';
        buf->buffer[buf->len++] = '\n';
        if (i + 1 == no_of_headers) {
            buf->buffer[buf->len++] = '\r';
            buf->buffer[buf->len++] = '\n';
        }
    }

    *buffer_len = buf->len;
    return buf->buffer;
}

static InspectionBuffer *GetBuffer1ForTX(DetectEngineThreadCtx *det_ctx,
        const DetectEngineTransforms *transforms, Flow *f, const uint8_t flow_flags, void *txv,
        const int list_id)
{
    InspectionBuffer *buffer = InspectionBufferGet(det_ctx, list_id);
    if (buffer->inspect == NULL) {
        uint32_t rawdata_len = 0;
        uint8_t *rawdata = GetBufferForTX(txv, det_ctx, f, flow_flags, &rawdata_len);
        if (rawdata_len == 0)
            return NULL;

        InspectionBufferSetupAndApplyTransforms(
                det_ctx, list_id, buffer, rawdata, rawdata_len, transforms);
    }

    return buffer;
}

static int DetectHttpStartSetup(DetectEngineCtx *de_ctx, Signature *s, const char *arg)
{
    if (SCDetectBufferSetActiveList(de_ctx, s, g_buffer_id) < 0)
        return -1;

    if (SCDetectSignatureSetAppProto(s, ALPROTO_HTTP1) < 0)
        return -1;

    return 0;
}

/**
 * \brief Registers the keyword handlers for the "http_start" keyword.
 */
void DetectHttpStartRegister(void)
{
    sigmatch_table[DETECT_HTTP_START].name = KEYWORD_NAME;
    sigmatch_table[DETECT_HTTP_START].alias = KEYWORD_NAME_LEGACY;
    sigmatch_table[DETECT_HTTP_START].desc = BUFFER_NAME " sticky buffer";
    sigmatch_table[DETECT_HTTP_START].url = "/rules/" KEYWORD_DOC;
    sigmatch_table[DETECT_HTTP_START].Setup = DetectHttpStartSetup;
    sigmatch_table[DETECT_HTTP_START].flags |= SIGMATCH_NOOPT | SIGMATCH_INFO_STICKY_BUFFER;

    DetectAppLayerMpmRegister(BUFFER_NAME, SIG_FLAG_TOSERVER, 2, PrefilterGenericMpmRegister,
            GetBuffer1ForTX, ALPROTO_HTTP1, HTP_REQUEST_PROGRESS_HEADERS);
    DetectAppLayerMpmRegister(BUFFER_NAME, SIG_FLAG_TOCLIENT, 2, PrefilterGenericMpmRegister,
            GetBuffer1ForTX, ALPROTO_HTTP1, HTP_RESPONSE_PROGRESS_HEADERS);

    DetectAppLayerInspectEngineRegister(BUFFER_NAME, ALPROTO_HTTP1, SIG_FLAG_TOSERVER,
            HTP_REQUEST_PROGRESS_HEADERS, DetectEngineInspectBufferGeneric, GetBuffer1ForTX);
    DetectAppLayerInspectEngineRegister(BUFFER_NAME, ALPROTO_HTTP1, SIG_FLAG_TOCLIENT,
            HTP_RESPONSE_PROGRESS_HEADERS, DetectEngineInspectBufferGeneric, GetBuffer1ForTX);

    DetectBufferTypeSetDescriptionByName(BUFFER_NAME,
            BUFFER_DESC);

    g_buffer_id = DetectBufferTypeGetByName(BUFFER_NAME);

    g_keyword_thread_id = DetectRegisterThreadCtxGlobalFuncs(KEYWORD_NAME,
            HttpHeaderThreadDataInit, &g_td_config, HttpHeaderThreadDataFree);

    SCLogDebug("keyword %s registered. Thread id %d. "
            "Buffer %s registered. Buffer id %d",
            KEYWORD_NAME, g_keyword_thread_id,
            BUFFER_NAME, g_buffer_id);
}
