/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Compiler_p.h"

#include <llvm/Module.h>
#include <llvm/PassManager.h>

#include "GTLCore/AST/Expression.h"
#include "GTLCore/AST/FunctionDeclaration.h"
#include "GTLCore/AST/Tree.h"
#include "GTLCore/LLVMBackend/CodeGenerator_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/CompilationMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/Type.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/Function_p.h"
#include "GTLCore/Value.h"

#include "GTLCore/Debug.h"
#include "Lexer_p.h"
#include "Library.h"
#include "LibrariesManager.h"
#include "Parser_p.h"
#include <GTLCore/CompilationMessages.h>
#include "Library_p.h"

using namespace GTLFragment;

struct Compiler::Private {
  llvm::Module* llvmModule;
  GTLCore::ModuleData* moduleData;
  LLVMBackend::CodeGenerator* codeGenerator;
  Lexer* lexer;
  Parser* parser;
  int channelsNb;
  Library::Type type;
  WhichStdLibrary stdLibrary;
  std::map< GTLCore::String, GTLCore::Value > parameters;
};

Compiler::Compiler( Library::Type _type, int _channelsNb ) : d(new Private)
{
  d->moduleData = 0;
  d->codeGenerator = 0;
  d->channelsNb = _channelsNb;
  d->type = _type;
}

Compiler::~Compiler()
{
  delete d;
}

WhichStdLibrary Compiler::whichStdLibrary() const
{
  return d->stdLibrary;
}

Library::Type Compiler::libraryType() const
{
  return d->type;
}

bool Compiler::compile(WhichStdLibrary _stdLibrary, const GTLCore::String& _sourceCode, const GTLCore::String& _kernelName, GTLCore::ModuleData* _moduleData, llvm::Module* _module, GTLCore::String& _nameSpace, const std::vector< ParameterInfo >& parameters)
{
  d->stdLibrary = _stdLibrary;
  GTL_DEBUG("Compile: " << _kernelName << " : " << _sourceCode);
  // Initialise the module structure
  d->llvmModule = _module;
  d->moduleData = _moduleData;
  d->codeGenerator = new LLVMBackend::CodeGenerator( d->moduleData );
  setModuleData(d->moduleData, _module);
  
  foreach(ParameterInfo pi, parameters)
  {
    d->parameters[pi.name] = pi.value;
  }

  llvm::LLVMContext& context = d->moduleData->llvmModule()->getContext();
  
  // Create Standard Library functions
  // CtlStdLib functions (except print which get a special treatement)
  createStdLibFunction( context, "assert", "assert", GTLCore::Type::Void, 1, GTLCore::Type::Boolean, false);
  createStdLibFunction( context, "isnan_f", "isnan_f", GTLCore::Type::Boolean, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "isnan_h", "isnan_h", GTLCore::Type::Boolean, 1, GTLCore::Type::Float32, false);
  // C Math functions
  createStdLibFunction( context, "rand", "intRandomAt", GTLCore::Type::Integer32, 3, GTLCore::Type::Integer32, false, GTLCore::Type::Integer32, false, GTLCore::Type::Integer32, false);
  createStdLibFunction( context, "frand", "floatRandomAt", GTLCore::Type::Float32, 3, GTLCore::Type::Integer32, false, GTLCore::Type::Integer32, false, GTLCore::Type::Integer32, false);
  createStdLibFunction( context, "acos", "acosf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "asin", "asinf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "atan", "atanf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "atan2", "atan2f", GTLCore::Type::Float32, 2, GTLCore::Type::Float32, false, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "cos", "cosf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "sin", "sinf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "tan", "tanf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "cosh", "coshf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "sinh", "sinhf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "tanh", "tanhf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "exp", "expf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "log", "logf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "log10", "log10f", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "pow", "powf", GTLCore::Type::Float32, 2, GTLCore::Type::Float32, false, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "sqrt", "sqrtf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "floor", "floorf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "ceil", "ceilf", GTLCore::Type::Float32, 1, GTLCore::Type::Float32, false);
  createStdLibFunction( context, "mod", "fmodf", GTLCore::Type::Float32, 2, GTLCore::Type::Float32, false, GTLCore::Type::Float32, false);
  
  // Init the lexer
  std::istringstream iss(_sourceCode);
  d->lexer = new Lexer( &iss, d->type );
  d->parser = new Parser( this, d->lexer );
  
  // Import standard libaries
  if( d->stdLibrary != RijnWrappersLibrary)
  {
    if( d->stdLibrary != FragmentStdLibrary)
    { // of course you don't want to import the standard library when building the standard library
      importModule("stdlib");
    }
    if( d->stdLibrary != RijnStdLibrary and (d->type == Library::RIJN_LIBRARY or d->type == Library::RIJN_SKETCH) )
    {
      importModule("rijnstdlib");
    }
  }

  // Parse
  GTLCore::AST::Tree* tree = d->parser->parse();
  bool success = (tree and errorMessages().errors().size() == 0);
  if( success )
  {
    tree->generate( d->moduleData, _module, d->codeGenerator, &context );
    GTL_DEBUG( *_module );
    for( std::list<GTLCore::AST::FunctionDeclaration*>::const_iterator it = tree->functionsDeclarations().begin();
         it != tree->functionsDeclarations().end(); ++it)
    {
      _moduleData->appendFunction( (*it)->function()->name(), (*it)->function());
    }
  } else {
    // Failure
    GTL_DEBUG("failure " << errorMessages().toString());
//     delete d->module;
  }
  _nameSpace = d->parser->nameSpace();
  // Clean up
  GTL_DEBUG("Clean up");
  
  delete tree;

/*  foreach(GTLCore::Function* function, functionsToDelete())
  {
    delete function;
  }*/
  GTL_DEBUG("Namespace = " << d->parser->nameSpace() );
  delete d->lexer;
  d->lexer = 0;
  delete d->parser;
  d->parser = 0;
  delete d->codeGenerator;
  d->codeGenerator = 0;
  d->moduleData = 0;
  GTL_DEBUG("Compilation finished");
  return success;
}

bool Compiler::importModule(const GTLCore::String& name)
{
  Library* library = LibrariesManager::instance()->loadLibrary( name, d->channelsNb, d->type );
  if( library )
  {
    if(not library->isCompiled())
    {
      library->compile();
      if(not library->isCompiled())
      {
        GTL_DEBUG("Compilation error: " << std::endl << library->compilationMessages().toString() ); // TODO: report the error
        appendErrors( library->compilationMessages().errors() );
        return false;
      }
    }
    d->moduleData->linkWith( library->data() );
    // Append the function coming from the imported module
    std::list<GTLCore::Function*> functions = library->functions();
    foreach( GTLCore::Function* function, functions )
    {
      GTL_DEBUG(function->name());
      GTLCore::Function* newFunction = GTLCore::Function::Private::createExternalFunction( d->moduleData, d->llvmModule, d->moduleData->llvmModule()->getContext(), function );
      declareFunction( newFunction->name(), newFunction );
      functionsToDelete().push_back( newFunction );
    }
    
    // Declare constant
    GTL_ASSERT(d->parser);
    for(std::map<GTLCore::ScopedName, const GTLCore::Type*>::const_iterator it = library->data()->constants().begin();
        it != library->data()->constants().end(); ++it)
    {
      
      d->parser->appendGlobalConstantDeclaration( new GTLCore::AST::GlobalConstantDeclaration(it->first, it->second, GTLCore::AST::Expression::fromValue(library->data()->constantsValue().at(it->first)), false, true ));
    }
  }
  return library;
}

GTLCore::TypesManager* Compiler::typesManager()
{
  return d->moduleData->typesManager();
}

GTLCore::AST::Expression* Compiler::standardConstant( const GTLCore::String& _name )
{
  // TODO standardConstant
  
/*  // Check if it is a parameter
  std::map< GTLCore::String, GTLCore::Value >::iterator it = d->parameters.find(_name);
  if( it != d->parameters.end() )
  {
    return GTLCore::AST::Expression::fromValue( it->second );
  }*/
  
  return 0;
}

const std::map< GTLCore::String, GTLCore::Value >& Compiler::parameters() const
{
  return d->parameters;
}

int Compiler::channelsNb() const
{
  return d->channelsNb;
}
