/*<replacement>*/
var bufferShim = require('safe-buffer').Buffer;
/*</replacement>*/
require('../common');
var assert = require('assert/');
var stream = require('../../');
var Writable = stream.Writable;

// Test the buffering behaviour of Writable streams.
//
// The call to cork() triggers storing chunks which are flushed
// on calling uncork() in the same tick.
//
// node version target: 0.12

var expectedChunks = ['please', 'buffer', 'me', 'kindly'];
var inputChunks = expectedChunks.slice(0);
var seenChunks = [];
var seenEnd = false;

var w = new Writable();
// lets arrange to store the chunks
w._write = function (chunk, encoding, cb) {
  // default encoding given none was specified
  assert.strictEqual(encoding, 'buffer');

  seenChunks.push(chunk);
  cb();
};
// lets record the stream end event
w.on('finish', function () {
  seenEnd = true;
});

function writeChunks(remainingChunks, callback) {
  var writeChunk = remainingChunks.shift();
  var writeState = void 0;

  if (writeChunk) {
    setImmediate(function () {
      writeState = w.write(writeChunk);
      // we were not told to stop writing
      assert.ok(writeState);

      writeChunks(remainingChunks, callback);
    });
  } else {
    callback();
  }
}

// do an initial write
w.write('stuff');
// the write was immediate
assert.strictEqual(seenChunks.length, 1);
// reset the chunks seen so far
seenChunks = [];

// trigger stream buffering
w.cork();

// write the bufferedChunks
writeChunks(inputChunks, function () {
  // should not have seen anything yet
  assert.strictEqual(seenChunks.length, 0);

  // trigger writing out the buffer
  w.uncork();

  // buffered bytes shoud be seen in current tick
  assert.strictEqual(seenChunks.length, 4);

  // did the chunks match
  for (var i = 0, l = expectedChunks.length; i < l; i++) {
    var seen = seenChunks[i];
    // there was a chunk
    assert.ok(seen);

    var expected = bufferShim.from(expectedChunks[i]);
    // it was what we expected
    assert.deepEqual(seen, expected);
  }

  setImmediate(function () {
    // the stream should not have been ended
    assert.ok(!seenEnd);
  });
});