/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
 *  \file pdl_variable.c
 *
 *  \brief Implementation of the pdl variables.
 *
 *  Not all functions defined in this file are accessible to everyone. A
 *  subset is used by the pdl variable functions themselves. For the list
 *  API functions look in pdl_variables.h.
 *
 *
 *  \author  G.M. Venekamp  (venekamp@nikhef.nl)
 *  \version $Revision: 16234 $
 *  \date    $Date: 2012-03-22 10:33:06 +0100 (Thu, 22 Mar 2012) $
 *
 */


#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lcmaps_log.h"
#include "pdl_variable.h"
#include "pdl_rule.h"

static var_t *top_var=0, *last_var=0;

BOOL   _lcmaps_add_variable(record_t* name, record_t* value);
void   free_vars(void);
/* var_t* lcmaps_find_variable(char* name); */
var_t* lcmaps_detect_loop(char* name, char* value);
void   show_vars(void);


/*!
 *  Wrapper function for the _lcmaps_add_variable() function call. The hard
 *  work is done in the _lcmaps_add_variable() call. When that call succeeds
 *  only the resources allocated for holding the name and value
 *  parameters are freed, i.e. the structures name and value. In case
 *  the _lcmaps_add_variable() calls fails, the string that is contained
 *  within the name and value strutures is freed as well.
 *
 *  \param  name  Name of the variable.
 *  \param  value Value of the variable.
 *
 */
void lcmaps_add_variable(record_t* name, record_t* value)
{
  if (!_lcmaps_add_variable(name, value)) {
    free(name->string);  name->string = NULL;
    free(value->string); name->string = NULL;
  }

  free(name);  name  = NULL;
  free(value); value = NULL;
}


/*!
 *  Actual implementation of the lcmaps_add_variable call. When the variable
 *  has been added the call returns TRUE, otherwise its FALSE. There
 *  can be several reasons for failure:
 *   - Variable allready exists;
 *   - Variable refers to itself through a loop;
 *   - No more resources to allocate for variable.
 *
 *  \param name  Name of the variable to be added.
 *  \param value Value of the variable.
 *  \return TRUE in case the variable has been added, FALSE otherwise.
 *
 */
BOOL _lcmaps_add_variable(record_t* name, record_t* value)
{
  var_t *var;

  if ((var = lcmaps_find_variable(name->string))) {
    lcmaps_warning(PDL_ERROR, "variable '%s' already defined at line %d; ", var->name, var->lineno);
    lcmaps_warning(PDL_SAME,  "previous value: '%s'.", var->value);
    return FALSE;
  }

  if ((var = lcmaps_detect_loop(name->string, value->string))) {
    lcmaps_warning(PDL_ERROR, "loop detected on variable '%s'; %s = %s", name->string, name->string, value->string);
    while (var) {
      var_t* tmp = var;
      lcmaps_warning(PDL_SAME,  "see also line: %d  %s = %s", var->lineno, var->name, var->value);
      var = var->next;
      /*
       *  A list is created during the lcmaps_detect_loop call. This list must be
       *  freed. However, the elements of the structure are direct copies
       *  of existing elements. Therefore, only the list elements need be
       *  freed and not any of the structure elements!
       */
      free(tmp);
    }
    return FALSE;
  }

  if (!(var = (var_t *)malloc(sizeof(var_t)))) {
    lcmaps_warning(PDL_ERROR, "Out of memory; cannot add variable '%s'.\n", name->string);
    return FALSE;
  }

  var->name   = name->string;
  var->value  = value->string;
  var->okay   = FALSE;
  var->lineno = name->lineno;
  var->next   = 0;
  
  if (top_var)
    last_var->next = var;
  else
    top_var = var;
  
  last_var = var;

  return TRUE;
}


/*!
 *  Free the resources allocated for the variables.
 *
 */
void lcmaps_free_variables(void)
{
  var_t* var = top_var;

  while (var) {
    var_t* t = var->next;

    free((char *)var->name);   var->name  = NULL;
    free((char *)var->value);  var->value = NULL;
    free(var);

    var = t;
  }

  top_var = 0;
}


/*!
 *  Find a variable based on the variable name. This way the value
 *  of a variable can be retrieved.
 *
 *  \param  name Name of the variable to find.
 *  \return Pointer to the corresponding variable, or 0 when not found.
 *
 */
var_t* lcmaps_find_variable(char* name)
{
  var_t* var = top_var;

  if (!name)
    return 0;

  while (var && strcmp(name, var->name)!=0) {
    var = var->next;
  }

  return var;
}


/*!
 *  Try to detect a loop in the variable references. When e.g. a=b,
 *  b=c and c=a, then the call should detect a loop.
 *
 *  \param name  Name of the variable.
 *  \param value Value of the variable.
 *  \return 0 if no loop was detected. When a loop is detected, the
 *          first variable in the loop is returned.
 *
 */
var_t* lcmaps_detect_loop(char* name, char* value)
{
  var_t *loop=0, *top_loop=0, *var = lcmaps_find_variable(value);

  while (var) {
    var_t* tmp = (var_t *)malloc(sizeof(var_t));

    if (loop) 
      loop->next = tmp;
    else
      top_loop = tmp;

    loop = tmp;
    memcpy(loop, var, sizeof(var_t));
    loop->next = 0;

    tmp = top_loop;
    while (tmp && strcmp(name, tmp->value)!=0) {
      tmp = tmp->next;
    }
    
    if (tmp) return top_loop;

    var = lcmaps_find_variable(var->value);
  }

  /*  No loop was detected, free the allocated memory.  */
  while (top_loop) {
    var_t* tmp = top_loop;
    free(top_loop);
    top_loop = tmp->next;
  }

  return 0;
}


/*!
 *  Reduce the variable to its real value. When a variable has another
 *  variable as its value, the variable will be reduced to the value
 *  of the referring variable.
 *
 *  \param name  Name of the variable to be reduced.
 *  \param rule_type  Type of the rule.
 *  \return Real value of the redunced variable.
 *
 */
void lcmaps_reduce_to_var(char** name, rule_type_t rule_type)
{
  var_t *var=0, *tmp;
  char* n = *name;
  rule_t* t;

  while ((tmp = lcmaps_find_variable(n))) {
    var = tmp;
    n = var->value;
  }

  if (var) {
    if (var->okay || !(t = lcmaps_get_rule(n, rule_type==STATE ? right_side : left_side))) {
      var->okay = TRUE;
      free(*name);
      *name = strdup(n);
      if (*name==NULL)
	  lcmaps_warning(PDL_WARNING, "Out of memory when dupping %s\n",n);
    } else {
      lineno = var->lineno;
      lcmaps_warning(PDL_WARNING, "Variable %s points to state %s. This is considered dangerous.", var->name, n);
    }
  }
}


/*!
 *  Get a list of all variables in the configure file.
 *
 *  \return First variable of the list.
 *
 */
var_t* lcmaps_get_variables(void)
{
  return top_var;
}


/*!
 *  Print all variables and their value as described in the configure
 *  file to stdout.
 *
 */
void lcmaps_show_variables(void)
{
  var_t* var = top_var;

  while (var) {
    lcmaps_log_debug(1, "var: %s = %s\n", var->name, var->value);
    var = var->next;
  }
}
