///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo bamg inputs / outputs
//
// author:
//      Pierre.Saramito@imag.fr
//
// date: 2 may 2001
//

// ============================================================================
//  includes
// ============================================================================

#include "rheolef/georep.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
#include "rheolef/field.h"
#include "rheolef/form.h"
#include "rheolef/tiny_element.h"
using namespace std;
namespace rheolef { 

// --------------------------------------------------------------------
// grummp input
// --------------------------------------------------------------------
istream&
georep::get_grummp(istream& s)
{
  // grummp specific format
  // GRUMMP-0.1.7/src/IO_generator/user2D.template and user3D.template
  // 	newfile g
  // 	"grummp" 
  // 	"2" nverts ncells nbfaces nfaces
  // 	verts: coords
  // 	cells: verts region
  // 	bdryfaces: verts bc face
  // 	faces: cells verts
  //
  // => this is only a version 1 mesh file format type
  //    not consistent for P2 elements
  _version = 1;

  if (!s) error_macro("bad input stream for geo.");
  bool verbose = iorheo::getverbose(s);
  if (verbose) warning_macro ("grummp mesh needs upgrade to geo version 2");
  if (!scatch(s,"grummp")) {
	// grummp-0.1.7 has buggy output : `m' of mesh is missing in 2d...
        error_macro("input stream does not contains a grummp mesh file.");
  }
  //
  // get coordinates
  //
  size_type nverts, ncells, nbfaces, nface;
  s >> _dim >> nverts >> ncells >> nbfaces >> nface;
  _x.resize(nverts);
  get_nodes (s, begin_node(), end_node(), _xmin, _xmax, _dim);
  _count_geo [0] = _count_element [0] = nverts;
  if (!s.good()) error_macro ("a problem occurs while loading a grummp mesh");
  //
  // get elements
  //
  resize(ncells);
  vector<vector<tiny_element> > region_table;
  size_type K_idx = 0;
  iterator last_elt = end();
  for (iterator iter_elt = begin();
	s.good() && iter_elt != last_elt; iter_elt++, K_idx++) {

      geo_element& K = (*iter_elt);
      if (_dim == 2)
          K.set_name('t');
      else
          K.set_name('T');
      K.set_index(K_idx) ;
      for (size_type i = 0 ; i < K.size(); i++) {
	s >> K[i];
	if (K[i] >= nverts) {
	    error_macro("grummp input: element " << K_idx+1 << ": " << i+1 << "-th vertice index "
              << K[i] << " out of range 0:" << nverts-1);
	}
      }
      _count_geo    [K.dimension()]++;
      _count_element[K.variant()]++;

      // get region id
      size_type r;
      s >> r;
      if (r >= region_table.size()) region_table.resize(r+1);
      region_table[r].push_back (tiny_element(K));
  }
  // skip the last record : faces: cells verts
  //
  // build regions if more than one is defined
  //
  size_type nreg = 0;
  for (size_type k = 0; k < region_table.size(); k++) {
      if (region_table[k].size() != 0) nreg++;
  }
  if (nreg == 1) {
      // the whole domain is already defined
      nreg = 0;
  } else {
      _domlist.resize(nreg);
      domlist_type::iterator q = _domlist.begin();
      for (size_type k = 0; k < region_table.size(); k++) {

          if (region_table[k].size() == 0) continue;
          (*q).set(region_table[k].begin(), region_table[k].size(), _dim, "region"+itos(k));
          q++;
      }
  }
  //
  // get boundary faces and slip it by domain id
  //
  map<size_type,size_type> bdr_map;        // (bdr_id, ibdr) map
  vector<vector<tiny_element> > bdr_table; // indexed by ibdr
  for (size_type j = 0; s.good() && j < nbfaces; j++) {
    tiny_element e;
    if (_dim == 2) {
	e.set_variant(reference_element::e);
        s >> e[0] >> e[1];
    } else {
	e.set_variant(reference_element::t);
        s >> e[0] >> e[1] >> e[2];
    }
    // handle region indexes
    // since there are arbitrarily large, we use a map
    size_type bdr_id, iface;
    s >> bdr_id;
    map<size_type,size_type>::iterator p = bdr_map.find(bdr_id);
    size_type ibdr;
    if (p == bdr_map.end()) {
	ibdr = bdr_map.size();
	bdr_map.insert(make_pair(bdr_id,ibdr));
        bdr_table.resize(ibdr+1);
    } else {
	ibdr = (*p).second;
    }
    bdr_table[ibdr].push_back (e);
    // extra iface that is used by meshopt2d meshopt3d
    s >> iface;
  }
  if (!s.good()) error_macro ("a problem occurs while loading a grummp mesh");
  //
  // copy bdr_table into domlist
  //
  size_type nbdr = bdr_table.size();
  _domlist.resize(nreg+nbdr);
  domlist_type::iterator q = _domlist.begin() + nreg;
  for (map<size_type,size_type>::iterator p = bdr_map.begin();
	p != bdr_map.end(); p++, q++) {
      size_type bdr_id = (*p).first;
      size_type ibdr   = (*p).second;
      (*q).set(bdr_table[ibdr].begin(), bdr_table[ibdr].size(), _dim-1, "boundary"+itos(bdr_id));
  }
  //
  // extension to grummp file format: optional domain names
  //
  if ((_dim == 2 && scatch(s,"EdgeDomainNames")) ||
      (_dim == 3 && scatch(s,"FaceDomainNames"))   ) {

      // get boundary domain names
      size_type ndombdr; s >> ndombdr;
      if (ndombdr != nbdr)
          error_macro("grummp input(bdr domain name size): expect " 
	    << nbdr << ", read " << ndombdr);
      for (size_type k = 0; k < nbdr; k++) {
          domlist_type::iterator p = _domlist.begin() + nreg + k;
	  string name;
          s >> name;
          (*p).set_name(name);
      }
  }
  if ((_dim == 2 && scatch(s,"FaceDomainNames")) ||
      (_dim == 3 && scatch(s,"VolumeDomainNames"))   ) {

      // get region domain names
      size_type ndomreg; s >> ndomreg;
      if (ndomreg != nreg)
          error_macro("grummp input(region domain name size): expect " 
	    << nreg << ", read " << ndomreg);
      for (size_type k = 0; k < nreg; k++) {
          domlist_type::iterator p = _domlist.begin() + k;
	  string name;
          s >> name;
          (*p).set_name(name);
      }
  }
  return s;
}
}// namespace rheolef
