/* eenc.c Enigma-like encryption/decryption command-line tool */
/* markus@mhoenicka.de 2003-03-08 */

/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "getopt.h"
#include "enigma.h"
#include "refdb.h"

/* declaration of the svn version function */
const char* svn_version(void);

int main (int argc, char** argv) {
  int n_opt;
  char wheelpos[PREFS_BUF_LEN] = "";
  char passwd[PASSWD_LENGTH+1];
  char scrambled_passwd[PASSWD_LENGTH+1];
  char numberized_passwd[3*PASSWD_LENGTH+1];

  /* read command line settings */
  while ((n_opt = getopt(argc, argv, "hp:v")) != -1) {
    switch (n_opt) {
    case 'h':
      fprintf(stderr, "eenc encodes a string of up to 16 chars using an Enigma-like algorithm\nUsage: eenc -p wheelpos string\nwheelpos is a string ABC-DE-FG-HI where ABC denotes the wheel sequence and DE, FG, and HI denote the position of the wheels in the first, second, and third position, respectively\nRun eenc -v to display version info\n");
      exit(0);
      break;
    case 'p':
      strncpy(wheelpos, optarg, PREFS_BUF_LEN);
      wheelpos[PREFS_BUF_LEN-1] = '\0';
      break;
    case 'v':
      printf("eenc %s built from svn revision %s markus@mhoenicka.de\nYou may redistribute and modify this software under the terms of the GNU General Public License.\n", VERSION, svn_version());
      exit (0);
      break;
    case ':':
      fprintf(stderr, "Usage: eenc -p wheelpos string\nwheelpos is a string ABC-DE-FG-HI where ABC denotes the wheel sequence and DE, FG, and HI denote the position of the wheels in the first, second, and third position, respectively\n");
      exit(0);
      break;
    case '?':
      fprintf(stderr, "unknown option %c: use eenc -h to display usage\n", optopt);
      break;
    }
  }

  if (!*wheelpos) {
    fprintf(stderr, "no scramble string specified\n");
    exit(1);
  }

  /* if password was passed on the command line */
  if (argv[optind] && *(argv[optind])) {
    strncpy(passwd, argv[optind], PASSWD_LENGTH);
    passwd[PASSWD_LENGTH] = '\0';
  }
  else {
    if (!fgets(passwd, PASSWD_LENGTH+1, stdin)) {
      exit(0);
    }
    /* remove trailing newline */
    if (passwd[strlen(passwd)-1] == '\n') {
      passwd[strlen(passwd)-1] = '\0';
    }
  }

  if (enigma_encrypt(passwd, scrambled_passwd, wheelpos)) {
    fprintf(stderr, "password encryption failed\n");
    exit(1);
  }

  numberize_passwd(numberized_passwd, scrambled_passwd);

  printf("%s", numberized_passwd);
  exit(0);
}
