% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ranova.R
\name{ranova}
\alias{ranova}
\alias{rand}
\title{ANOVA-Like Table for Random-Effects}
\usage{
ranova(model, reduce.terms = TRUE, ...)

rand(model, reduce.terms = TRUE, ...)
}
\arguments{
\item{model}{a linear mixed effect model fitted with \code{lmer()}
(inheriting from class \code{lmerMod}).}

\item{reduce.terms}{if \code{TRUE} (default) random-effect terms are
reduced (if possible). If \code{FALSE} random-effect terms are simply
removed.}

\item{...}{currently ignored}
}
\value{
an ANOVA-like table with single term deletions of random-effects
inheriting from class \code{anova} and \code{data.frame} with the columns:
\item{npar}{number of model parameters.}
\item{logLik}{the log-likelihood for the model. Note that this is the
REML-logLik if the model is fitted with REML.}
\item{AIC}{the AIC for the model evaluated as \code{-2*(logLik - npar)}.
Smaller is better.}
\item{LRT}{the likelihood ratio test statistic; twice the difference in
log-likelihood, which is asymptotically chi-square distributed.}
\item{Df}{degrees of freedom for the likelihood ratio test: the difference in
number of model parameters.}
\item{Pr(>Chisq)}{the p-value.}
}
\description{
Compute an ANOVA-like table with tests of random-effect terms in the model.
Each random-effect term is reduced or removed and likelihood ratio tests of
model reductions are presented in a form similar to that of
\code{\link[=drop1.lmerModLmerTest]{drop1}}.
\code{rand} is an alias for \code{ranova}.
}
\details{
If the model is fitted with REML the tests are REML-likelihood ratio tests.

A random-effect term of the form \code{(f1 + f2 | gr)} is reduced to
terms of the form \code{(f2 | gr)} and \code{(f1 | gr)} and these reduced
models are compared to the original model.
If \code{reduce.terms} is \code{FALSE} \code{(f1 + f2 | gr)} is removed
instead.

A random-effect term of the form \code{(f1 | gr)} is reduced to \code{(1 | gr)}
(unless \code{reduce.terms} is \code{FALSE}).

A random-effect term of the form \code{(1 | gr)} is not reduced but
simply removed.

A random-effect term of the form \code{(0 + f1 | gr)} or \code{(-1 + f1 | gr)}
is reduced (if \code{reduce.terms = TRUE}) to \code{(1 | gr)}.

A random-effect term of the form \code{(1 | gr1/gr2)} is automatically
expanded to two terms: \code{(1 | gr2:gr1)} and \code{(1 | gr1)} using
\code{\link[lme4]{findbars}}.

In this exposition it is immaterial whether \code{f1} and \code{f2} are
factors or continuous variables.
}
\note{
Note that \code{anova} can be used to compare two models and will often
be able to produce the same tests as \code{ranova}. This is, however, not always the
case as illustrated in the examples.
}
\section{Warning}{

In certain cases tests of non-nested models may be generated. An example
is when \code{(0 + poly(x, 2) | gr)} is reduced (the default) to \code{(1 | gr)}.
To our best knowledge non-nested model comparisons are only generated in
cases which are statistical nonsense anyway (such as in this example where
the random intercept is suppressed).
}

\examples{

# Test reduction of (Days | Subject) to (1 | Subject):
fm1 <- lmer(Reaction ~ Days + (Days|Subject), sleepstudy)
ranova(fm1) # 2 df test

# This test can also be achieved with anova():
fm2 <- lmer(Reaction ~ Days + (1|Subject), sleepstudy)
anova(fm1, fm2, refit=FALSE)

# Illustrate reduce.test argument:
# Test removal of (Days | Subject):
ranova(fm1, reduce.terms = FALSE) # 3 df test

# The likelihood ratio test statistic is in this case:
fm3 <- lm(Reaction ~ Days, sleepstudy)
2*c(logLik(fm1, REML=TRUE) - logLik(fm3, REML=TRUE)) # LRT

# anova() is not always able to perform the same tests as ranova(),
# for example:
anova(fm1, fm3, refit=FALSE) # compares REML with ML and should not be used
anova(fm1, fm3, refit=TRUE) # is a test of ML fits and not what we seek

# Also note that the lmer-fit needs to come first - not an lm-fit:
# anova(fm3, fm1) # does not work and gives an error

# ranova() may not generate all relevant test:
# For the following model ranova() indicates that we should not reduce
# (TVset | Assessor):
fm <- lmer(Coloursaturation ~ TVset * Picture + (TVset | Assessor), data=TVbo)
ranova(fm)
# However, a more appropriate model is:
fm2 <- lmer(Coloursaturation ~ TVset * Picture + (1 | TVset:Assessor), data=TVbo)
anova(fm, fm2, refit=FALSE)
# fm and fm2 has essentially the same fit to data but fm uses 5 parameters
# more than fm.

}
\seealso{
\code{\link[=drop1.lmerModLmerTest]{drop1}} for tests of marginal
fixed-effect terms and
\code{\link{anova}} for usual anova tables for fixed-effect terms.
}
\author{
Rune Haubo B. Christensen and Alexandra Kuznetsova
}
