

#ifndef _COMMAND_DISPATCHER_H_
#define _COMMAND_DISPATCHER_H_

#include <boost/shared_ptr.hpp>
#include <boost/utility.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/condition.hpp>
#include <boost/any.hpp>

#include <string>
#include <map>
#include <list>
#include <set>

#include "inputbuffer.h"
#include "datainputstream.h"
#include "protectedbuffer.h"
#include "rpccommandhandler.h"

namespace rpc {

/**
 * Class that parses and forwards commands that are received in
 * the RpcDriver.
 */
class CommandDispatcher : private boost::noncopyable
{
public:
	
	typedef std::list<RpcCommandHandlerPtr> RpcCommandHandlerList;
	
	CommandDispatcher() : requireAuthentication(false) { }
	~CommandDispatcher() { }
	
	/**
	 * Register a command with us so that it receives notifications
	 * when a command that carries its name is received.
	 * @param pCmdHandler Command to register.
	 */
	void registerCommand( RpcCommandHandlerPtr pCmdHandler );
	
	/**
	 * Handle a command that has been received in the RpcDriver. This includes
	 * parsing the parameters and notifying all RpcCommandHandlers that listen
	 * for this type of command.
	 * @param sender Id of the client that sent this command.
	 * @param cmd Input buffer containing the data of the command. This should
	 *			  be in the form of <cmd name>, <param>, <param>, ...
	 */
	void handleCommand( int sender, CmdInputBufferPtr cmd );
	
	/**
	 * Tell the thread to stop executing.
	 */
	void abort();
	
	/**
	 * Wait for commands to start coming in. This should be the entry point
	 * for the thread that sits and handles commands.
	 */
	void waitForCommand();

	void setClientAuthenticated(int clientId,bool authenticated);
	bool isClientAuthenticated(int clientId);
	void enableAuthenticationChecking(bool yes) {requireAuthentication=yes;}

private:
	
	/**
	 * Parse the suplied params and put them in a variant.
	 * @param params The parameters to be parsed.
	 * @return The resulting variant constructed from the params.
	 */
	void parseCommandParams( DataInputStreamPtr paramStream, std::list<boost::any>& params );
	boost::any getNextParam(DataInputStreamPtr paramStream, 
											std::list<boost::any>& params) ;

	/**
	 * Retreive the name of the command from the input buffer representing
	 * the command. It will also advance the buffer past the command name 
	 * so that others can keep reading to get the rest of the params.
	 * @param cmd Contains all the data for the command.
	 */
	std::string retrieveCommandName( DataInputStreamPtr cmdStream );
	
	/**
	 * Send the command specified by cmdName to all the registerd 
	 * RpcCommandHandlers and pass a variant containing the params along.
	 * @param cmdName Name of the command.
	 * @param sender Id of the client that sent this command.
	 * @param params Variant containing parameters for the command.
	 */
	void dispatchCommand( const std::string& cmdName,
						  int sender,
						  const std::list<boost::any>& params );
	
	/// Make sure that two threads don't access the map of commands at
	/// the same time.
	boost::mutex handlerListMutex;
	
	/// Mutex used in a condition to make the receiver thread wait
	/// for data to arrive in the outgoing queue
	boost::mutex commandMonitor;
	
	/// Condition used to wait for data to be present in the command queue.
	/// Used together with m_queueMonitor
	boost::condition commandAvailable;
	
	/// Map containing all the commands that have registered wich maps
	/// a command name to a list of RpcCommandHandlers registerd for that name.
	std::map< std::string, RpcCommandHandlerList> registeredCommands;
	
	/// Buffer holding the commands that are waiting to be handled
	ProtectedBuffer< std::list<boost::any> > commandBuffer;
	
	/// Set this to enable authentication checking
	bool requireAuthentication;

	/// Holds authenticated clients
	std::set<int> authenticatedClients;
};

/// Smart pointer class for convinience
typedef boost::shared_ptr<CommandDispatcher> CommandDispatcherPtr;

}

#endif
