/*
 * CRRCsim - the Charles River Radio Control Club Flight Simulator Project
 *
 * Copyright (C) 2004 Kees Lemmens (original author)
 * Copyright (C) 2004 Lionel Cailler
 * Copyright (C) 2005, 2006, 2008, 2009 Jan Reucker
 * Copyright (C) 2005, 2008 Jens Wilhelm Wulf
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */
  

/** \file crrc_sound.h
 *
 *  CRRCsim-specific stuff related to the soundserver.
 *
 */
#ifndef CRRC_SOUND_H
#define CRRC_SOUND_H

#include "crrc_soundserver.h"

/// Definition of the sound type constants
#define SOUND_TYPE_GLOW       (0)     ///< glow engine, sound also plays when engine is idle
#define SOUND_TYPE_ELECTRIC   (1)     ///< electric engine, sound stops when engine stops
#define SOUND_TYPE_GLIDER     (2)     ///< sound is generated by the airflow

// Inter-thread exchange function
void soundUpdate3D(float flDist, float flPropFreq, float flH, float flRelV);


/** \brief The CRRCsim-specific mother of all airplane sound classes.
 *
 *  This class extends T_PitchVariableLoop with 
 *  some CRRCsim-specific attributes.
 */
class T_AirplaneSound : public T_PitchVariableLoop
{
  public:
    /// The constructor
    T_AirplaneSound(const char *filename, SDL_AudioSpec *fmt);
  
    /// The destructor
    ~T_AirplaneSound();
  
    // Get a pointer to a chunk of data. Description: see implementation.
    Uint8* getMixableData(Uint32 playpos, Uint32 *len);
  
    /// Get the pitch factor for this sound sample
    double  getPitchFactor() const {return dPitchFactor;};
    /// Set the pitch factor for this sound sample
    void    setPitchFactor(double d) {dPitchFactor = d;};
    
    /// Get the maximum volume for this sound sample
    double  getMaxVolume() const {return dMaxVolume;};
    /// Set the maximum volume for this sound sample
    void    setMaxVolume(double v) {dMaxVolume = v;};
    
    /// Get the type of this sound sample (SOUND_TYPE_XYZ)
    int     getType() const {return type;};
    /// Set the type of this sound sample (SOUND_TYPE_XYZ)
    void    setType(int t) {type = t;};

    /// Get the channel on which this sound sample is playing
    int     getChannel() const {return channel;};
    /// Set the channel on which this sound sample is playing
    void    setChannel(int c) {channel = c;};

    // calculate Doppler constant (see implementation for details)
    float calculate_Doppler(float dist);

  protected:
    double  dPitchFactor;   ///< The relation of sample pitch to e.g. motor speed
    double  dMaxVolume;     ///< The maximum sample volume
    int     type;           ///< Type of the sound sample (SOUND_TYPE_XYZ)
    int     channel;        ///< The soundserver channel on which this sample is playing

  private:
    /// Interface to the routine that calculates pitch and volume for the sample
    virtual void calculate() = 0;

    // for Doppler calculations:
    float  last_dist;       ///< Doppler effect: last distance to sound source
    float  v_rel_filter;    ///< Doppler effect: filtered relative velocity
};


/** \brief The CRRCsim-specific engine sound class.
 *
 *  This class extends T_PitchVariableLoop with a
 *  calculation routine which creates some basic
 *  "special FX" for the engine sound: Doppler,
 *  distance-dependant attenuation and speed-dependant
 *  engine pitch.
 */
class T_EngineSound : public T_AirplaneSound
{
  public:
    T_EngineSound(const char *filename, SDL_AudioSpec *fmt);
    ~T_EngineSound();

  private:
    void    calculate();
};


/** \brief The CRRCsim-specific glider sound class.
 *
 *  This class extends T_PitchVariableLoop with a
 *  calculation routine which creates some basic
 *  "special FX" for glider sounds: Doppler,
 *  distance-dependant attenuation and speed-dependant
 *  pitch.
 */
class T_GliderSound : public T_AirplaneSound
{
  public:
    /// The constructor
    T_GliderSound(const char *filename, SDL_AudioSpec *fmt);
  
    /// The destructor
    ~T_GliderSound();
  
    /// Set the minimum relative velocity
    void setMinRelVelocity(float f) {flMinRelV = f;};
  
    /// Set the maximum relative velocity
    void setMaxRelVelocity(float f) {flMaxRelV = f;};
    
    /// Set the maximum distance (in ft.)
    void setMaxDistanceFeet(float f) {flMaxDist = f;};

  private:
    void calculate();
    
    float flMinRelV;  ///< minimal relative velocity that triggers the sound
    float flMaxRelV;  ///< relative velocity where the sound reaches max volume
    float flMaxDist;  ///< distance where the sound drops to min volume
};


/** \brief A variometer sound class.
 *
 *  This class implements the variometer sound for CRRCsim.
 */
class T_VariometerSound : protected T_SoundSample
{
  public:
    T_VariometerSound(SDL_AudioSpec *fmt);
    virtual ~T_VariometerSound();
    
    void    convert(SDL_AudioSpec *fmt);
  
    Uint8*  getMixableData(Uint32 playpos, Uint32 *len);

  protected:
    void init(SDL_AudioSpec *fmt);
    void calculate();

    std::vector<Uint8>  dyn_buffer;
    std::vector<Sint16> sinewave;
    int   nBeepFInc;
    int   nBeepFIncNew;
    int   nOnCntInit;
    int   nVarioCnt;
    int   nVarioOffReq;
    int   nSinIndex;        ///< index into sinewave buffer
    int   nSndVarioSineLen; ///< length of sine in samples
    float flHDiffFilt;      ///< filtered vertical velocity in feet per second
    float flHOld;           ///< old height in feet

};



#endif  // CRRC_SOUND_H
