/*
    libzint - the open source barcode library
    Copyright (C) 2020 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* vim: set ts=4 sw=4 et : */

#include "testcommon.h"

static void test_encode(int index, int generate, int debug) {

    testStart("");

    int ret;
    struct item {
        int symbology;
        int input_mode;
        int option_1;
        int option_2;
        unsigned char *data;
        int ret;

        int expected_rows;
        int expected_width;
        char *comment;
        char *expected;
    };
    struct item data[] = {
        /*  0*/ { BARCODE_AZTEC, UNICODE_MODE, -1, -1, "123456789012", 0, 15, 15, "ISO/IEC 24778:2008 Figure 1 (left)",
                    "000111000011100"
                    "110111001110010"
                    "111100001000100"
                    "001111111111100"
                    "010100000001000"
                    "100101111101010"
                    "100101000101110"
                    "001101010101100"
                    "101101000101111"
                    "101101111101010"
                    "110100000001101"
                    "000111111111111"
                    "110001010010001"
                    "101011110101010"
                    "100010001000101"
                },
        /*  1*/ { BARCODE_AZTEC, UNICODE_MODE, -1, -1, "Aztec Code is a public domain 2D matrix barcode symbology of nominally square symbols built on a square grid with a distinctive square bullseye pattern at their center.", 0, 41, 41, "ISO/IEC 24778:2008 Figure 1 (right) NOTE: Not the same but down to single encoding mode difference (UPPER space rather than LOWER space after 2D)",
                    "00001100110010010010111000010100001011000"
                    "01000110010110110001000000100101101000001"
                    "01011100101011001110101100000001100011001"
                    "11100000010100001000010010010100110001010"
                    "10101010101010101010101010101010101010101"
                    "00110101011100000001000100011001101100010"
                    "11001000100011110101100010110100011011010"
                    "01000101001101011011011111101100000101011"
                    "11011010111101111111101010001101111011100"
                    "01000101001000110011010101001101110100111"
                    "01011101100011100001100000101010111011001"
                    "00100101100111100101001111011011011100101"
                    "11101000000100001111101011010101101010101"
                    "00000100011101100101000001011110110000000"
                    "01001101010001111111111111110101101011100"
                    "00110000000010100000000000110100010000111"
                    "00001011011111101111111110100110000011110"
                    "00000111010010101000000010110011101100000"
                    "11001001010101101011111010111111100010111"
                    "11110001000101101010001010100111001100011"
                    "10101010101010101010101010101010101010101"
                    "00100100010000101010001010100101100000000"
                    "00001100111110101011111010111001111010000"
                    "10010101000010101000000010110110111001010"
                    "01011101011011101111111110110000110011110"
                    "00110101011100100000000000111011010100100"
                    "00111101000010111111111111110100011011011"
                    "00000010101100011111001110001000111000010"
                    "10001000010111101001100110011000001010001"
                    "10010001111100101111011100100011110101101"
                    "01111100000001001110100011011011110011000"
                    "10000001000010110110010010010101111000000"
                    "00101100010010001011111101101111111110101"
                    "00100001101000100101010001001110010001111"
                    "10011001111110000001110100000001010010011"
                    "10110010000000111110011001111111100101011"
                    "10101010101010101010101010101010101010101"
                    "10000010011110010010000100001010001101110"
                    "10101101110100001100111100110101001010001"
                    "00010010010011001011011010000110001000101"
                    "10001000001010100110100000001001001110000"
                },
        /*  2*/ { BARCODE_AZTEC, UNICODE_MODE, -1, -1, "Code 2D!", 0, 15, 15, "ISO/IEC 24778:2008 Figure G.2",
                    "000110001100000"
                    "000000110000010"
                    "101100001000101"
                    "011111111111100"
                    "111100000001101"
                    "000101111101100"
                    "100101000101111"
                    "001101010101001"
                    "001101000101010"
                    "010101111101001"
                    "100100000001011"
                    "100111111111101"
                    "010001100010010"
                    "011000011011010"
                    "111001101100000"
                },
        /*  3*/ { BARCODE_AZTEC, UNICODE_MODE, -1, 1, "Code 2D!", 0, 15, 15, "ISO/IEC 24778:2008 Figure G.2; specify size",
                    "000110001100000"
                    "000000110000010"
                    "101100001000101"
                    "011111111111100"
                    "111100000001101"
                    "000101111101100"
                    "100101000101111"
                    "001101010101001"
                    "001101000101010"
                    "010101111101001"
                    "100100000001011"
                    "100111111111101"
                    "010001100010010"
                    "011000011011010"
                    "111001101100000"
                },
        /*  4*/ { BARCODE_AZTEC, UNICODE_MODE, -1, -1, "1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111", 0, 53, 53, "**NOT SAME** ISO/IEC 24778:2008 Figure I.1 (left) TODO: investigate",
                    "00010101011010101010101010110101010101010110101010101"
                    "00101010100101010101010101001010101010101001010101010"
                    "11100101011010101010101010110101010101010110101010110"
                    "11011010100101010101010101001010101010101001010101001"
                    "00101001011010101010101010110101010101010110101011010"
                    "01010110100101010101010101001010101010101001010100101"
                    "10101010011010101010101010110101010101010110101101010"
                    "01010101100101010101010101001010101010101001010010101"
                    "10101010101010101010101010110101010101010110110101010"
                    "01010101010101010101010101001010101010101001001010101"
                    "10101010101010101010101010101010101010101010101010101"
                    "10101010100101111100100010001100110110010001010101010"
                    "01010101011011100000011011111100100111010010101010101"
                    "10101010100101000000011011011101100100000101010101010"
                    "01010101011010110110100001100000100100100010101010101"
                    "10101010100111101100101000001111010001001001010101010"
                    "01010101011110000110010111110110001101010010101010101"
                    "10101010100111001001110011011111100001000101010101010"
                    "01010101011110101000100001101101111001000010101010101"
                    "10101010100010011011101000000010011011000101010101010"
                    "01010101011010000001111111111111110100110110101010101"
                    "10101010100110000001100000000000110000010001010101010"
                    "01010101011000001000101111111110100001100110101010101"
                    "10101010100011000011101000000010101100001001010101010"
                    "01010101011101111101101011111010101000011010101010101"
                    "10101010100001100111101010001010101010110001010101010"
                    "10101010101010101010101010101010101010101010101010101"
                    "01010101010001101100101010001010100110010100101010101"
                    "10101010101101100011101011111010110001100011010101010"
                    "01010101010010110110101000000010100110110100101010101"
                    "10101010101111000100101111111110111110000011010101010"
                    "01010101010110011110100000000000110001100000101010101"
                    "10101010101011001000111111111111110100001111010101010"
                    "01010101010011101000001000001111000110001100101010101"
                    "10101010101001010100110111101001011010101111010101010"
                    "01010101010010100011010011000101101111011000101010101"
                    "10101010101111000100000111111101101110111111010101010"
                    "01010101010110100110110110000011100010000000101010101"
                    "10101010101101110111010001101100001000011111010101010"
                    "01010101010110001101001100010100101010110000101010101"
                    "10101010101000101001111010110101100111100111010101010"
                    "01010101010010010000111100011100110101100000101010101"
                    "10101010101010101010101010101010101010101010101010101"
                    "10101010010010101010101010010101010101010101010101010"
                    "01010101101101010101010101101010101010101010101010101"
                    "10101001010010101010101010010101010101010100110101010"
                    "01010110101101010101010101101010101010101011001010101"
                    "10100101010010101010101010010101010101010100101101010"
                    "01011010101101010101010101101010101010101011010010101"
                    "10010101010010101010101010010101010101010100101011010"
                    "01101010101101010101010101101010101010101011010100101"
                    "01010101010010101010101010010101010101010100101010110"
                    "10101010101101010101010101101010101010101011010101001"
                },
        /*  5*/ { BARCODE_AZTEC, UNICODE_MODE, -1, -1, "3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333", 0, 53, 53, "**NOT SAME** ISO/IEC 24778:2008 Figure I.1 (right) TODO: investigate",
                    "00111111111111111111111111111111111111111111111111111"
                    "00000000000000000000000000000000000000000000000000000"
                    "11101111111111111111111111111111111111111111111111101"
                    "11100000000000000000000000000000000000000000000000001"
                    "00101011111111111111111111111111111111111111111110101"
                    "10101000000000000000000000000000000000000000000000101"
                    "10101010111111111111111111111111111111111111111010101"
                    "10101010000000000000000000000000000000000000000010101"
                    "10101010101111111111111111111111111111111111101010101"
                    "10101010100000000000000000000000000000000000001010101"
                    "10101010101010101010101010101010101010101010101010101"
                    "10101010100100010001110111001110000101000000101010101"
                    "10101010101101011100101111110011101011101010101010101"
                    "10101010100100111100011011000001110011010000101010101"
                    "10101010101100110110001101111010001010101010101010101"
                    "10101010100110111110011111000100011001111100101010101"
                    "10101010101110110100111001111011110011111110101010101"
                    "10101010100110011110100000010101000011010000101010101"
                    "10101010101011001010101110101110001010011110101010101"
                    "10101010100110010101101000000010010101100000101010101"
                    "10101010101101111111111111111111111010111110101010101"
                    "10101010100111110111100000000000110001101000101010101"
                    "10101010101101101100101111111110100000000110101010101"
                    "10101010100101100001101000000010101110011000101010101"
                    "10101010101100110101101011111010101001000010101010101"
                    "10101010100001100111101010001010101100100000101010101"
                    "10101010101010101010101010101010101010101010101010101"
                    "10101010100000111110101010001010100010011000101010101"
                    "10101010101011011101101011111010111011110110101010101"
                    "10101010100100010000101000000010101001010000101010101"
                    "10101010101011010100101111111110110110000010101010101"
                    "10101010100011111000100000000000110110101000101010101"
                    "10101010101101000110111111111111111100100010101010101"
                    "10101010100100100000001000001111000100010100101010101"
                    "10101010101000111110011001100110101101100110101010101"
                    "10101010100101100010110011011100001010110100101010101"
                    "10101010101101111011000111100110101010011110101010101"
                    "10101010100001101011111001001000010001001000101010101"
                    "10101010101010100111010011111110000111100010101010101"
                    "10101010100000001011111101010011011010000000101010101"
                    "10101010101100100000011011100010000000101010101010101"
                    "10101010100001101101001111001100000001000100101010101"
                    "10101010101010101010101010101010101010101010101010101"
                    "10101010000000000000000000000000000000000000101010101"
                    "10101010111111111111111111111111111111111110101010101"
                    "10101000000000000000000000000000000000000000001010101"
                    "10101011111111111111111111111111111111111111101010101"
                    "10100000000000000000000000000000000000000000000010101"
                    "10101111111111111111111111111111111111111111111010101"
                    "10000000000000000000000000000000000000000000000000101"
                    "10111111111111111111111111111111111111111111111110101"
                    "00000000000000000000000000000000000000000000000000001"
                    "11111111111111111111111111111111111111111111111111101"
                },
        /*  6*/ { BARCODE_AZTEC, GS1_MODE, -1, -1, "[01]03453120000011[17]120508[10]ABCD1234[410]9501101020917", 0, 23, 23, "#189 Follow embedded FLG(n) with FLG(0)",
                    "00100000101111000100100"
                    "00011101100110001010000"
                    "00000111000111101011011"
                    "01000001010100001100010"
                    "00001001100110000111110"
                    "00011011111000110101101"
                    "00001011100111101111110"
                    "11000111111111111010000"
                    "11001101000000010111110"
                    "00111011011111010111110"
                    "01000001010001011110010"
                    "00011001010101011111101"
                    "01101111010001011110011"
                    "10011001011111011011100"
                    "01011011000000010010110"
                    "00011101111111111001100"
                    "10101100110111100001101"
                    "10110101111100111010001"
                    "11101001010000011001110"
                    "00101010101010001111001"
                    "11000101000100100000100"
                    "00010001010101010101011"
                    "11101100000000000010110"
                },
        /*  7*/ { BARCODE_AZTEC, GS1_MODE, -1, -1, "[01]95012345678903[3103]000123", 0, 19, 19, "#189 Follow embedded FLG(n) with FLG(0)",
                    "0000000100001010101"
                    "0001101111011000000"
                    "0111100100010110100"
                    "0011000110100100110"
                    "0010110101010010110"
                    "0101111111111110101"
                    "0011110000000101000"
                    "0101010111110100011"
                    "1101110100010110100"
                    "1000010101010111010"
                    "0011110100010100110"
                    "1000110111110111110"
                    "0010110000000101010"
                    "1111011111111111101"
                    "1001001111001000100"
                    "1111111110110100011"
                    "0111000111101011001"
                    "1000110111011000101"
                    "1010100000101101001"
                },
        /*  8*/ { BARCODE_AZTEC, GS1_MODE, -1, -1, "[01]04610044273252[21]LRFX)k<C7ApWJ[91]003A[92]K8rNAqdvjmdxsmCVuj3FhaoNzQuq7Uff0sHXfz1TT/doiMaGQqNF+VPwMvwVbm1fxjzuDt6jxLCcc8o/tqbEDA==", 0, 45, 45, "#189 Follow embedded FLG(n) with FLG(0)",
                    "000000101110011010101010010110011000001010111"
                    "000110001111011100111101101110110000000000011"
                    "001000101100101011000011111111101110111010101"
                    "010010001000111100001101110011100111110100100"
                    "001001100110001000011011011011000101001010010"
                    "001101011110100001001101001101110111100111100"
                    "101010101010101010101010101010101010101010101"
                    "000111001111100010000000100000111001000111110"
                    "111100111010000100100111100011001110001100001"
                    "110101001000001101011100110101001110000101010"
                    "000111111010001101111110100110001011001110001"
                    "011000000000100001101100000000011000100011110"
                    "001100110100000101000111111101110100011001010"
                    "011010000111011010100100001101001111010000111"
                    "100011100111010101011110000001010111001110111"
                    "011010000001010110011000000101101011110100010"
                    "001100100001010111111111111111110010101111100"
                    "111010010111010010000000000011101100100011000"
                    "010100110000110110111111111011010010011111100"
                    "001011001110010110100000001010101111010011000"
                    "001010101101100110101111101011100111111111000"
                    "010001001100111010101000101011111100000100000"
                    "101010101010101010101010101010101010101010101"
                    "101100010101111010101000101010100000110101111"
                    "011010101100110010101111101010000001001101101"
                    "001110000010110010100000001011101111010011100"
                    "011001100001111010111111111010001010001111000"
                    "000001011101101010000000000011101100010110111"
                    "110111111111001011111111111111110101101001101"
                    "000010011011101001110000111000101101000000011"
                    "111111101001101111011111110110101010001110001"
                    "001010011011011100100101100100000100100101111"
                    "100101100101110100010111010111001100001111011"
                    "010111001001000111010101110101010111010111100"
                    "001110110000001110001010000100110011101100101"
                    "101001000110001000010101001110110010110000100"
                    "101111100000010000110111010000010111111111111"
                    "100000011010001000010001110100010100010111010"
                    "101010101010101010101010101010101010101010101"
                    "101001000011011010010001001100101110100010110"
                    "001100100111001101011011111110110010011110100"
                    "111100011011100010110001001111110011100011101"
                    "101101111001110100000011101110110100011101011"
                    "000011010110101110000101110100000111000011010"
                    "101001110101010110100011010010001111001101101"
                },
        /*  9*/ { BARCODE_HIBC_AZTEC, UNICODE_MODE, -1, -1, "H123ABC01234567890", 0, 19, 19, "**NOT SAME** ANSI/HIBC 2.6 - 2016 Figure C1 TODO: investigate",
                    "0001100111110111011"
                    "0010111100111110001"
                    "0111011000010101001"
                    "1010010100011110000"
                    "0010110101001010011"
                    "0110111111111110001"
                    "0110110000000101001"
                    "0011110111110111001"
                    "1101010100010101010"
                    "1111010101010100101"
                    "1011110100010111000"
                    "0011110111110100101"
                    "1101010000000101011"
                    "0101011111111110110"
                    "0000000000010001101"
                    "0111000100111000010"
                    "0101000000100011010"
                    "1001001001100001110"
                    "1100001010110101000"
                },
        /* 10*/ { BARCODE_AZRUNE, UNICODE_MODE, -1, -1, "0", 0, 11, 11, "ISO/IEC 24778:2008 Figure A.1 (1st)",
                    "11101010101"
                    "11111111111"
                    "01000000010"
                    "11011111011"
                    "01010001010"
                    "11010101011"
                    "01010001010"
                    "11011111011"
                    "01000000010"
                    "01111111111"
                    "00101010100"
                },
        /* 11*/ { BARCODE_AZRUNE, UNICODE_MODE, -1, -1, "25", 0, 11, 11, "ISO/IEC 24778:2008 Figure A.1 (2nd)",
                    "11101100101"
                    "11111111111"
                    "01000000011"
                    "01011111011"
                    "01010001010"
                    "11010101011"
                    "11010001011"
                    "11011111010"
                    "11000000011"
                    "01111111111"
                    "00100100000"
                },
        /* 12*/ { BARCODE_AZRUNE, UNICODE_MODE, -1, -1, "125", 0, 11, 11, "ISO/IEC 24778:2008 Figure A.1 (3rd)",
                    "11110101101"
                    "11111111111"
                    "11000000011"
                    "11011111011"
                    "01010001010"
                    "01010101010"
                    "01010001011"
                    "01011111011"
                    "11000000010"
                    "01111111111"
                    "00111101000"
                },
        /* 13*/ { BARCODE_AZRUNE, UNICODE_MODE, -1, -1, "255", 0, 11, 11, "ISO/IEC 24778:2008 Figure A.1 (4th)",
                    "11110101001"
                    "11111111111"
                    "01000000011"
                    "11011111011"
                    "11010001011"
                    "01010101011"
                    "01010001010"
                    "11011111011"
                    "11000000010"
                    "01111111111"
                    "00110011100"
                },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    char escaped[1024];

    for (int i = 0; i < data_size; i++) {

        if (index != -1 && i != index) continue;

        struct zint_symbol *symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        symbol->input_mode = data[i].input_mode;
        if (data[i].option_1 != -1) {
            symbol->option_1 = data[i].option_1;
        }
        if (data[i].option_2 != -1) {
            symbol->option_2 = data[i].option_2;
        }
        symbol->debug |= debug;

        int length = strlen(data[i].data);

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (generate) {
            printf("        /*%3d*/ { %s, %s, %d, %d, \"%s\", %s, %d, %d, \"%s\",\n",
                    i, testUtilBarcodeName(data[i].symbology), testUtilInputModeName(data[i].input_mode), data[i].option_1, data[i].option_2,
                    testUtilEscape(data[i].data, length, escaped, sizeof(escaped)), testUtilErrorName(data[i].ret), symbol->rows, symbol->width, data[i].comment);
            testUtilModulesDump(symbol, "                    ", "\n");
            printf("                },\n");
        } else {
            if (ret < 5) {
                assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, data[i].data);
                assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, data[i].data);

                if (ret == 0) {
                    int width, row;
                    ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
                    assert_zero(ret, "i:%d testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, ret, width, row, data[i].data);
                }
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

// #181 Nico Gunkel OSS-Fuzz
static void test_fuzz(int index, int debug) {

    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char *data;
        int length;
        int input_mode;
        int option_1;
        int ret;
    };
    // s/\/\*[ 0-9]*\*\//\=printf("\/*%2d*\/", line(".") - line("'<"))
    struct item data[] = {
        /* 0*/ { BARCODE_AZTEC,
                    "\133\060\060\060\135\060\125\125\125\125\140\060\125\125\125\125\060\060\060\271\060\060\125\103\164\125\125\125\377\377\125\125"
                    "\125\125\125\125\125\133\060\076\060\135\261\177\261\261\261\236\261\261\261\040\261\261\261\261\261\261\261\020\261\261\261\261"
                    "\261\261\265\261\261\261\261\261\261\261\261\261\261\261\261\040\224\261\261\261\261\261\000\000\004\000\031\060\031\031\031\031"
                    "\031\031\031\031\261\261\040\261\261\261\261\261\261\261\020\261\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\376\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
                    "\377\377\377\377\377\377\261\261\261\261\261\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\135\135\135\135\135\135"
                    "\135\335\135\060\060\010\010\010\010\010\060",
                    2251, UNICODE_MODE, -1, ZINT_ERROR_TOO_LONG }, // Original OSS-Fuzz triggering data for malloc leak
        /* 1*/ { BARCODE_AZTEC,
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\000\060\060\060\060\000\060\060\000\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\005\060\060\060\060\060\060\060\005\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\000\000\060\350\060\060\060\060\060\350\060\345\060\343\060\060\060\005\060\060\060"
                    "\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\005\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060"
                    "\060\060\060\060\060\175\175\175\175\060\175\175\175\175\175\060\060\060\060\175\060\175\175\175\175\060\060\060\060\005\060\060"
                    "\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\005\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\060\000\060\060\060\060\060\005\060\060\060\060\060\005\005\060\005\060\060\060\060\060\060\005"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\353\060\060\353"
                    "\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060"
                    "\060\060\060\060\353\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060"
                    "\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353"
                    "\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060"
                    "\060\060\353\060\060\060\060\060\353\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060"
                    "\060\060\353\060\060\060\060\353\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060"
                    "\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060"
                    "\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060"
                    "\353\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353"
                    "\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\264\060"
                    "\060\060\060\060\264\060\060\060\060\060\264\060\060\060\060\264\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060"
                    "\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060"
                    "\353\060\060\060\060\060\353\060\060\060\060\060\353\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353"
                    "\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060"
                    "\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060"
                    "\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060"
                    "\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060"
                    "\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060"
                    "\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060"
                    "\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353"
                    "\060\060\060\060\353\060\060\060\060\060\353\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060"
                    "\060\060\060\060\353\060\060\060\060\060\353\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060"
                    "\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\060"
                    "\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\353\060\060\060\060\353\060\060\060\060\060\353\060\060\060\060\353"
                    "\060\060\060\060\060\353\060\060\060\060\060\353\060\060\353\060\060\060\343\060\060\060\060\060\060\060\060\060\005\060\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\377\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\005\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\361\060\060\060\060\060\361\060\060\060\060\361\060\060\060\060\060\361\060\060\060\060\060\361"
                    "\060\060\060\361\060\060\060\005\060\060\060\060\060\060\060\005\060\060\060\060\060\005\363\060\362\060\060\060\060\377\060\060"
                    "\060\177\060\060\060\060\060\263\060\060\060\060\060\060\060\060\060\000\060\060\060\060\377\177\060\060\060\362\060\362\060\363"
                    "\060\363\060\177\060\377\060\060\060\060\060\060\060\377\177\060\355\060\057\060\060\060\377\060\377\060\060\060\177\000\060\000"
                    "\060\377\060\060\060\060\060\060\005\060\060\000\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\005\060\060"
                    "\060\060\060\060\060\371\060\060\060\363\060\060\060\060\060\060\362\060\060\060\060\060\362\060\363\060\362\060\060\060\060\377"
                    "\060\177\060\060\060\060\060\263\060\060\060\060\060\060\060\060\060\060\000\060\060\060\377\177\060\060\060\362\060\362\060\363"
                    "\060\363\060\060\377\057\060\060\060\060\060\060\377\177\060\355\060\060\060\060\060\377\060\377\060\060\060\177\000\060\000\060"
                    "\377\060\353\060\060\000\060\060\060\060\060\060\060\060\060\256\060\060\060\362\060\060\060\060\060\060\060\060\005\362\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\057\060\060\060\060\060\060\060\000\060\000\060\060\060\000\060\060"
                    "\057\060\060\060\060\060\377\060\377\060\060\060\060\060\060\060\060\060\060\060\060\000\060\060\347\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\000\060\377\060\060\060\377\060\363\060\363\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                    "\362\060\060\060\060\060\060\362\347\060\354\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\005\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\005\060\060\060\000\060\000\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\344\005\060\060\005\060\342\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\005\060\000\060\060\060\060\060\060\060\060\060\060\377\060\060\060\060\005\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\005\060\060\060\060"
                    "\060\060\060\060\060\060\060\005\060\060\060\005\060\060\060\060\060\060\060\060\060\060\005\060\351\060\060\060\060\060\060\005"
                    "\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005"
                    "\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\060\005\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\005\060\060\060\005\060\060\060\060\005"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\005\060\060\060\175\175\060\175\060\060\060"
                    "\060\175\060\060\060\060\175\060\060\060\060\175\060\175\175\175\005\060\060\060\060\060\060\060\060\377\060\060\060\060\060\060"
                    "\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\005\060\342\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\005\060\060\060\345\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\005\060\371\060\060\060\060\060\060\060\060\060\060\060\361\060\060\060\060\060\005\361\060\060\060\060"
                    "\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\005\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\005\060\351\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\005\363\060\060\005\060\060\005\060\060\060\060\060\060\060\060\060\005\060\000\060"
                    "\060\060\060\060\060\060\362\060\060\060\060\060\060\060\060\060\060\060\005\060\005\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\000\060\060\060\060\060\060\060\060\060\005\060\377\060\005"
                    "\060\371\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\005\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060"
                    "\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\175\175\175\175\060\175\175\060\060\060\060\060\175\060\175\175"
                    "\175\005\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\005\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\005\060\060\060\060\060\060\060\060"
                    "\060\060\005\060\060\060\060\060\005\060\000\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060"
                    "\060\060\005\060\060\060\060\060\005\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060"
                    "\060\060\005\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\005\060\351\060\060\060\060\060\060\060"
                    "\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060\060\060\060\060\060\060\060\060\060\060\060\060\005\060\060\060"
                    "\060\005\060\060\060\005\060\362\060\060\060\060\060\060\060\060\060\060\060\060\377\060\060\000\177\060\060\000\060\377\060\353"
                    "\060\060\060\363\060\060\060\060\060\060\060\060\060\060\060\060\362\060\060\060\060\060\000\060\060\377\060\060\060\175\175\175"
                    "\175\060\060\060\175\175\175\175\060\060\005\060\005\060\005\060\060\060\060\000\000\060\060\060\060\060\060\377\060\060\060\060"
                    "\377\060\377\377\060\060\057\060\060\057\060\060\060\000\000\060\060",
                    2801, UNICODE_MODE, -1, ZINT_ERROR_TOO_LONG }, // Original OSS-Fuzz triggering data for binary_string buffer overrun
        /* 2*/ { BARCODE_AZTEC,
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123",
                    4483, -1, 1, 0 }, // 4483 = (1664 (Max codewords) - 169 (ECC codewords) - 2 (overhead)) * 3 (3 4-bit digits per 12-bit wordcode)
        /* 3*/ { BARCODE_AZTEC,
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123",
                    4484, -1, 1, ZINT_ERROR_TOO_LONG },
        /* 4*/ { BARCODE_AZTEC,
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXY",
                    3587, -1, 1, 0 },
        /* 5*/ { BARCODE_AZTEC,
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ",
                    3588, -1, 1, ZINT_ERROR_TOO_LONG },
        /* 6*/ { BARCODE_AZTEC,
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240",
                    2079, -1, 1, 0 },
        /* 7*/ { BARCODE_AZTEC,
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240",
                    2080, -1, 1, ZINT_ERROR_TOO_LONG },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        if (index != -1 && i != index) continue;

        struct zint_symbol *symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        int length = data[i].length;
        if (length == -1) {
            length = strlen(data[i].data);
        }
        if (data[i].input_mode != -1) {
            symbol->input_mode = data[i].input_mode;
        }
        if (data[i].option_1 != -1) {
            symbol->option_1 = data[i].option_1;
        }
        symbol->debug |= debug;

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

int main(int argc, char *argv[]) {

    testFunction funcs[] = { /* name, func, has_index, has_generate, has_debug */
        { "test_encode", test_encode, 1, 1, 1 },
        { "test_fuzz", test_fuzz, 1, 0, 1 },
    };

    testRun(argc, argv, funcs, ARRAY_SIZE(funcs));

    testReport();

    return 0;
}
