
/*
#    Sfront, a SAOL to C translator    
#    This file: Tokenizes special idents in SAOL
#
# Copyright (c) 1999-2006, Regents of the University of California
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#  Redistributions of source code must retain the above copyright
#  notice, this list of conditions and the following disclaimer.
#
#  Redistributions in binary form must reproduce the above copyright
#  notice, this list of conditions and the following disclaimer in the
#  documentation and/or other materials provided with the distribution.
#
#  Neither the name of the University of California, Berkeley nor the
#  names of its contributors may be used to endorse or promote products
#  derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#    Maintainer: John Lazzaro, lazzaro@cs.berkeley.edu
*/


#include "tree.h"
#include "parser.tab.h"

/*********************************************************/
/*        returns MP4 Code for Special Tokens            */
/*********************************************************/

int identtoken (tnode * ident)

{

 switch (ident->val[0]) {
  case 'a':
    if (!(strcmp(ident->val,"abs")))
      return S_ABS;
    if (!(strcmp(ident->val,"acos")))
      return S_ACOS;
    if (!(strcmp(ident->val,"aexpon")))
      return S_AEXPON;
    if (!(strcmp(ident->val,"aexprand")))
      return S_AEXPRAND;
    if (!(strcmp(ident->val,"agaussrand")))
      return S_AGAUSSRAND;
    if (!(strcmp(ident->val,"aline")))
      return S_ALINE;
    if (!(strcmp(ident->val,"alinrand")))
      return S_ALINRAND;
    if (!(strcmp(ident->val,"allpass")))
      return S_ALLPASS;
    if (!(strcmp(ident->val,"ampdb")))
      return S_AMPDB;
    if (!(strcmp(ident->val,"aphasor")))
      return S_APHASOR;
    if (!(strcmp(ident->val,"apoissonrand")))
      return S_APOISSONRAND;
    if (!(strcmp(ident->val,"arand")))
      return S_ARAND;
    if (!(strcmp(ident->val,"asin")))
      return S_ASIN;
    if (!(strcmp(ident->val,"atan")))
      return S_ATAN;
    return 0;
  case 'b':
    if (!(strcmp(ident->val,"balance")))
      return S_BALANCE;
    if (!(strcmp(ident->val,"bandpass")))
      return S_BANDPASS;
    if (!(strcmp(ident->val,"bandstop")))
      return S_BANDSTOP;
    if (!(strcmp(ident->val,"biquad")))
      return S_BIQUAD;
    if (!(strcmp(ident->val,"buzz")))
      {
	if ((ident->next != NULL) 
	    && (ident->next->ttype == S_COM))
	  return S_BUZZWAVE;
	else
	  return S_BUZZOPCODE;
      }
    return 0;
  case 'c':
    if (!(strcmp(ident->val,"ceil")))
      return S_CEIL;
    if (!(strcmp(ident->val,"channel")))
      return S_CHANNEL;
    if (!(strcmp(ident->val,"chorus")))
      return S_CHORUS;
    if (!(strcmp(ident->val,"comb")))
      return S_COMB;
    if (!(strcmp(ident->val,"compressor")))
      return S_COMPRESSOR;
    if (!(strcmp(ident->val,"concat")))
      return S_CONCAT;
    if (!(strcmp(ident->val,"cos")))
      return S_COS;
    if (!(strcmp(ident->val,"cpsmidi")))
      return S_CPSMIDI;
    if (!(strcmp(ident->val,"cpsoct")))
      return S_CPSOCT;
    if (!(strcmp(ident->val,"cpspch")))
      return S_CPSPCH;
    if (!(strcmp(ident->val,"cpuload")))
      return S_CPULOAD;
    if (!(strcmp(ident->val,"cubicseg")))
      return S_CUBICSEG;
    return 0;
  case 'd':
    if (!(strcmp(ident->val,"data")))
      return S_DATA;
    if (!(strcmp(ident->val,"dbamp")))
      return S_DBAMP;
    if (!(strcmp(ident->val,"decimate")))
      return S_DECIMATE;
    if (!(strcmp(ident->val,"delay")))
      return S_DELAY;
    if (!(strcmp(ident->val,"delay1")))
      return S_DELAY1;
    if (!(strcmp(ident->val,"direction")))
      return S_DIRECTION;
    if (!(strcmp(ident->val,"doscil")))
      return S_DOSCIL;
    if (!(strcmp(ident->val,"downsamp")))
      return S_DOWNSAMP;
    if (!(strcmp(ident->val,"dur")))
      return S_DUR;
    return 0;
  case 'e':
    if (!(strcmp(ident->val,"empty")))
      return S_EMPTY;
    if (!(strcmp(ident->val,"exp")))
      return S_EXP;
    if (!(strcmp(ident->val,"expseg")))
      return S_EXPSEG;
    return 0;
  case 'f':
    if (!(strcmp(ident->val,"fft")))
      return S_FFT;
    if (!(strcmp(ident->val,"fir")))
      return S_FIR;
    if (!(strcmp(ident->val,"firt")))
      return S_FIRT;
    if (!(strcmp(ident->val,"flange")))
      return S_FLANGE;
    if (!(strcmp(ident->val,"floor")))
      return S_FLOOR;
    if (!(strcmp(ident->val,"frac")))
      return S_FRAC;
    if (!(strcmp(ident->val,"fracdelay")))
      return S_FRACDELAY;
    if (!(strcmp(ident->val,"ftbasecps")))
      return S_FTBASECPS;
    if (!(strcmp(ident->val,"ftlen")))
      return S_FTLEN;
    if (!(strcmp(ident->val,"ftloop")))
      return S_FTLOOP;
    if (!(strcmp(ident->val,"ftloopend")))
      return S_FTLOOPEND;
    if (!(strcmp(ident->val,"ftsetbase")))
      return S_FTSETBASE;
    if (!(strcmp(ident->val,"ftsetend")))
      return S_FTSETEND;
    if (!(strcmp(ident->val,"ftsetloop")))
      return S_FTSETLOOP;
    if (!(strcmp(ident->val,"ftsetsr")))
      return S_FTSETSR;
    if (!(strcmp(ident->val,"ftsr")))
      return S_FTSR;
    return 0;
  case 'g':
    if (!(strcmp(ident->val,"gain")))
      return S_GAIN;
    if (!(strcmp(ident->val,"gettempo")))
      return S_GETTEMPO;
    if (!(strcmp(ident->val,"gettune")))
      return S_GETTUNE;
    if (!(strcmp(ident->val,"grain")))
      return S_GRAIN;
    return 0;
  case 'h':
    if (!(strcmp(ident->val,"harm")))
      return S_HARM;
    if (!(strcmp(ident->val,"harm_phase")))
      return S_HARM_PHASE;
    if (!(strcmp(ident->val,"hipass")))
      return S_HIPASS;
    return 0;
  case 'i':
    if (!(strcmp(ident->val,"iexprand")))
      return S_IEXPRAND;
    if (!(strcmp(ident->val,"ifft")))
      return S_IFFT;
    if (!(strcmp(ident->val,"igaussrand")))
      return S_IGAUSSRAND;
    if (!(strcmp(ident->val,"iir")))
      return S_IIR;
    if (!(strcmp(ident->val,"iirt")))
      return S_IIRT;
    if (!(strcmp(ident->val,"ilinrand")))
      return S_ILINRAND;
    if (!(strcmp(ident->val,"inchan")))
      return S_INCHAN;
    if (!(strcmp(ident->val,"inGroup")))
      return S_INGROUP;
    if (!(strcmp(ident->val,"input")))
      return S_INPUT;
    if (!(strcmp(ident->val,"input_bus")))
      return S_INPUT_BUS;
    if (!(strcmp(ident->val,"int")))
      return S_INT;
    if (!(strcmp(ident->val,"irand")))
      return S_IRAND;
    if (!(strcmp(ident->val,"itime")))
      return S_ITIME;
    return 0;
  case 'j':
    return 0;
  case 'k':
    if (!(strcmp(ident->val,"k_rate")))
      return S_K_RATE;
    if (!(strcmp(ident->val,"kexpon")))
      return S_KEXPON;
    if (!(strcmp(ident->val,"kexprand")))
      return S_KEXPRAND;
    if (!(strcmp(ident->val,"kgaussrand")))
      return S_KGAUSSRAND;
    if (!(strcmp(ident->val,"kline")))
      return S_KLINE;
    if (!(strcmp(ident->val,"klinrand")))
      return S_KLINRAND;
    if (!(strcmp(ident->val,"koscil")))
      return S_KOSCIL;
    if (!(strcmp(ident->val,"kphasor")))
      return S_KPHASOR;
    if (!(strcmp(ident->val,"kpoissonrand")))
      return S_KPOISSONRAND;
    if (!(strcmp(ident->val,"krand")))
      return S_KRAND;
    return 0;
  case 'l':
    if (!(strcmp(ident->val,"lineseg")))
      return S_LINESEG;
    if (!(strcmp(ident->val,"listenerPosition")))
      return S_LISTENERPOSITION;
    if (!(strcmp(ident->val,"listenerDirection")))
      return S_LISTENERDIRECTION;
    if (!(strcmp(ident->val,"log")))
      return S_LOG;
    if (!(strcmp(ident->val,"log10")))
      return S_LOG10;
    if (!(strcmp(ident->val,"lopass")))
      return S_LOPASS;
    if (!(strcmp(ident->val,"loscil")))
      return S_LOSCIL;
    return 0;
  case 'm':
    if (!(strcmp(ident->val,"max")))
      return S_MAX;
    if (!(strcmp(ident->val,"maxBack")))
      return S_MAXBACK;
    if (!(strcmp(ident->val,"maxFront")))
      return S_MAXFRONT;
    if (!(strcmp(ident->val,"midicps")))
      return S_MIDICPS;
    if (!(strcmp(ident->val,"midioct")))
      return S_MIDIOCT;
    if (!(strcmp(ident->val,"midipch")))
      return S_MIDIPCH;
    if (!(strcmp(ident->val,"min")))
      return S_MIN;
    if (!(strcmp(ident->val,"minFront")))
      return S_MINFRONT;
    return 0;
  case 'M':
    if (!(strcmp(ident->val,"MIDIctrl")))
      return S_MIDICTRL;
    if (!(strcmp(ident->val,"MIDItouch")))
      return S_MIDITOUCH;
    if (!(strcmp(ident->val,"MIDIbend")))
      return S_MIDIBEND;
    return 0;
  case 'n':
    return 0;
  case 'o':
    if (!(strcmp(ident->val,"octcps")))
      return S_OCTCPS;
    if (!(strcmp(ident->val,"octmidi")))
      return S_OCTMIDI;
    if (!(strcmp(ident->val,"octpch")))
      return S_OCTPCH;
    if (!(strcmp(ident->val,"oscil")))
      return S_OSCIL;
    if (!(strcmp(ident->val,"outchan")))
      return S_OUTCHAN;
    if (!(strcmp(ident->val,"output_bus")))
      return S_OUTPUT_BUS;
    return 0;
  case 'p':
    if (!(strcmp(ident->val,"params")))
      return S_PARAMS;
    if (!(strcmp(ident->val,"periodic")))
      return S_PERIODIC;
    if (!(strcmp(ident->val,"pchcps")))
      return S_PCHCPS;
    if (!(strcmp(ident->val,"pchmidi")))
      return S_PCHMIDI;
    if (!(strcmp(ident->val,"pchoct")))
      return S_PCHOCT;
    if (!(strcmp(ident->val,"pluck")))
      return S_PLUCK;
    if (!(strcmp(ident->val,"polynomial")))
      return S_POLYNOMIAL;
    if (!(strcmp(ident->val,"port")))
      return S_PORT;
    if (!(strcmp(ident->val,"position")))
      return S_POSITION;
    if (!(strcmp(ident->val,"pow")))
      return S_POW;
    if (!(strcmp(ident->val,"preset")))
      return S_PRESET;
    return 0;
  case 'q':
    return 0;
  case 'r':
    if (!(strcmp(ident->val,"random")))
      return S_RANDOM;
    if (!(strcmp(ident->val,"released")))
      return S_RELEASED;
    if (!(strcmp(ident->val,"reverb")))
      return S_REVERB;
    if (!(strcmp(ident->val,"rms")))
      return S_RMS;
    return 0;
  case 's':
    if (!(strcmp(ident->val,"s_rate")))
      return S_S_RATE;
    if (!(strcmp(ident->val,"sample")))
      return S_SAMPLE;
    if (!(strcmp(ident->val,"samphold")))
      return S_SAMPHOLD;
    if (!(strcmp(ident->val,"sblock")))
      return S_SBLOCK;
    if (!(strcmp(ident->val,"settempo")))
      return S_SETTEMPO;
    if (!(strcmp(ident->val,"settune")))
      return S_SETTUNE;
    if (!(strcmp(ident->val,"sgn")))
      return S_SGN;
    if (!(strcmp(ident->val,"sin")))
      return S_SIN;
    if (!(strcmp(ident->val,"spatialize")))
      return S_SPATIALIZE;
    if (!(strcmp(ident->val,"speedt")))
      return S_SPEEDT;
    if (!(strcmp(ident->val,"spline")))
      return S_SPLINE;
    if (!(strcmp(ident->val,"sqrt")))
      return S_SQRT;
    if (!(strcmp(ident->val,"startup")))
      return S_STARTUP;
    if (!(strcmp(ident->val,"step")))
      return S_STEP;
    return 0;
  case 't':
    if (!(strcmp(ident->val,"tableread")))
      return S_TABLEREAD;
    if (!(strcmp(ident->val,"tablewrite")))
      return S_TABLEWRITE;
    if (!(strcmp(ident->val,"time")))
      return S_TIME;
    return 0;
  case 'u':
    if (!(strcmp(ident->val,"upsamp")))
      return S_UPSAMP;
    return 0;
  case 'v':
    return 0;
  case 'w':
    if (!(strcmp(ident->val,"window")))
      return S_WINDOW;
    return 0;
  case 'x':
    return 0;
  case 'y':
    return 0;
  case 'z':
    return 0;
 }
 return 0; /* should never happen */
}

/*********************************************************/
/*     converts MP4 token to sfront-compatabile token    */
/*********************************************************/

int tokenmap(int token)

{
  switch(token) {

  case S_RESERVED0:
    return S_IDENT;
  case S_AOPCODE:
    return S_AOPCODE;
  case S_ASIG:
    return S_ASIG;
  case S_ELSE:
    return S_ELSE;
  case S_EXPORTS:
    return S_EXPORTS;
  case S_EXTEND:
    return S_EXTEND;
  case S_GLOBAL:     
    return S_GLOBAL;
  case S_IF:
    return S_IF;
  case S_IMPORTS:
    return S_IMPORTS;
  case S_INCHANNELS:
    return S_INCHANNELS;
  case S_INSTR:
    return S_INSTR;
  case S_IOPCODE:
    return S_IOPCODE;
  case S_IVAR:
    return S_IVAR;
  case S_KOPCODE:
    return S_KOPCODE;
  case S_KRATE:
    return S_KRATE;
  case S_KSIG:
    return S_KSIG;
  case S_MAP:
    return S_MAP;
  case S_OPARRAY:
    return S_OPARRAY;
  case S_OPCODE:
    return S_OPCODE;
  case S_OUTBUS:
    return S_OUTBUS;
  case S_OUTCHANNELS:
    return S_OUTCHANNELS;
  case S_OUTPUT:
    return S_OUTPUT;
  case S_RETURN:
    return S_RETURN;
  case S_ROUTE:
    return S_ROUTE;
  case S_SEND:
    return S_SEND;
  case S_SEQUENCE:
    return S_SEQUENCE;
  case S_SASBF:
    return S_SASBF;
  case S_SPATIALIZE:
    return S_SPATIALIZE;
  case S_SRATE:
    return S_SRATE;
  case S_TABLE:
    return S_TABLE;
  case S_TABLEMAP:
    return S_TABLEMAP;
  case S_TEMPLATE:
    return S_TEMPLATE;
  case S_TURNOFF:
    return S_TURNOFF;
  case S_WHILE:
    return S_WHILE;
  case S_WITH:
    return S_WITH;
  case S_XSIG:
    return S_XSIG;
  case S_INTERP:
    return S_INTERP;
  case S_PRESET:        /* PRESET not in grammer */
    return S_IDENT;
  case S_RESERVED1:
    return S_IDENT;
  case S_RESERVED2:
    return S_IDENT;
  case S_RESERVED3:
    return S_IDENT;
  case S_RESERVED4:
    return S_IDENT;
  case S_RESERVED5:
    return S_IDENT;
  case S_RESERVED6:
    return S_IDENT;
  case S_RESERVED7:
    return S_IDENT;
  case S_RESERVED8:
    return S_IDENT;
  case S_RESERVED9:
    return S_IDENT;
  case S_RESERVED10:
    return S_IDENT;
  case S_K_RATE:
    return S_IDENT;
  case S_S_RATE:
    return S_IDENT;
  case S_INCHAN:
    return S_IDENT;
  case S_OUTCHAN:
    return S_IDENT;
  case S_TIME:
    return S_IDENT;
  case S_DUR:
    return S_IDENT;
  case S_MIDICTRL:
    return S_IDENT;
  case S_MIDITOUCH:
    return S_IDENT;
  case S_MIDIBEND:
    return S_IDENT;
  case S_INPUT:
    return S_IDENT;
  case S_INGROUP:
    return S_IDENT;
  case S_RELEASED:
    return S_IDENT;
  case S_CPULOAD:
    return S_IDENT;
  case S_POSITION:
    return S_IDENT;
  case S_DIRECTION:
    return S_IDENT;
  case S_LISTENERPOSITION:
    return S_IDENT;
  case S_LISTENERDIRECTION:
    return S_IDENT;
  case S_MINFRONT:
    return S_IDENT;
  case S_MINBACK:
    return S_IDENT;
  case S_MAXFRONT:
    return S_IDENT;
  case S_MAXBACK:
    return S_IDENT;
  case S_PARAMS:
    return S_IDENT;
  case S_ITIME:
    return S_IDENT;
  case S_RESERVED11:
    return S_IDENT;
  case S_CHANNEL:
    return S_IDENT;
  case S_INPUT_BUS:
    return S_IDENT;
  case S_OUTPUT_BUS:
    return S_IDENT;
  case S_STARTUP:
    return S_IDENT;
  case S_RESERVED15:
    return S_IDENT;
  case S_RESERVED16:
    return S_IDENT;
  case S_RESERVED17:
    return S_IDENT;
  case S_RESERVED18:
    return S_IDENT;
  case S_AND:
    return S_AND;
  case S_OR:
    return S_OR;
  case S_GEQ:
    return S_GEQ;
  case S_LEQ:
    return S_LEQ;
  case S_NEQ:
    return S_NEQ;
  case S_EQEQ:
    return S_EQEQ;
  case S_MINUS:
    return S_MINUS;
  case S_STAR:
    return S_STAR;
  case S_SLASH:
    return S_SLASH;
  case S_PLUS:
    return S_PLUS;
  case S_GT:
    return S_GT;
  case S_LT:
    return S_LT;
  case S_Q:
    return S_Q;
  case S_COL:
    return S_COL;
  case S_LP:
    return S_LP;
  case S_RP:
    return S_RP;
  case S_LC:
    return S_LC;
  case S_RC:
    return S_RC;
  case S_LB:
    return S_LB;
  case S_RB:
    return S_RB;
  case S_SEM:
    return S_SEM;
  case S_COM:
    return S_COM;
  case S_EQ:
    return S_EQ;
  case S_NOT:
    return S_NOT;
  case S_RESERVED19:
    return S_IDENT;
  case S_RESERVED20:
    return S_IDENT;
  case S_RESERVED21:
    return S_IDENT;
  case S_RESERVED22:
    return S_IDENT;
  case S_RESERVED23:
    return S_IDENT;
  case S_RESERVED24:
    return S_IDENT;
  case S_RESERVED25:
    return S_IDENT;
  case S_SAMPLE:
    return S_IDENT;
  case S_DATA:
    return S_IDENT;
  case S_RANDOM:
    return S_IDENT;
  case S_STEP:
    return S_IDENT;
  case S_LINESEG:
    return S_IDENT;
  case S_EXPSEG:
    return S_IDENT;
  case S_CUBICSEG:
    return S_IDENT;
  case S_POLYNOMIAL:
    return S_IDENT;
  case S_SPLINE:
    return S_IDENT;
  case S_WINDOW:
    return S_IDENT;
  case S_HARM:
    return S_IDENT;
  case S_HARM_PHASE:
    return S_IDENT;
  case S_PERIODIC:
    return S_IDENT;
  case S_BUZZWAVE:
    return S_IDENT;
  case S_CONCAT:
    return S_IDENT;
  case S_EMPTY:
    return S_IDENT;
  case S_DESTROY:
    return S_IDENT;
  case S_INT:
    return S_IDENT;
  case S_FRAC:
    return S_IDENT;
  case S_DBAMP:
    return S_IDENT;
  case S_AMPDB:
    return S_IDENT;
  case S_ABS:
    return S_IDENT;
  case S_EXP:
    return S_IDENT;
  case S_LOG:
    return S_IDENT;
  case S_SQRT:
    return S_IDENT;
  case S_SIN:
    return S_IDENT;
  case S_COS:
    return S_IDENT;
  case S_ATAN:
    return S_IDENT;
  case S_POW:
    return S_IDENT;
  case S_LOG10:
    return S_IDENT;
  case S_ASIN:
    return S_IDENT;
  case S_ACOS:
    return S_IDENT;
  case S_FLOOR:
    return S_IDENT;
  case S_CEIL:
    return S_IDENT;
  case S_MIN:
    return S_IDENT;
  case S_MAX:
    return S_IDENT;
  case S_PCHOCT:
    return S_IDENT;
  case S_OCTPCH:
    return S_IDENT;
  case S_CPSPCH:
    return S_IDENT;
  case S_PCHCPS:
    return S_IDENT;
  case S_CPSOCT:
    return S_IDENT;
  case S_OCTCPS:
    return S_IDENT;
  case S_PCHMIDI:
    return S_IDENT;
  case S_MIDIPCH:
    return S_IDENT;
  case S_OCTMIDI:
    return S_IDENT;
  case S_MIDIOCT:
    return S_IDENT;
  case S_CPSMIDI:
    return S_IDENT;
  case S_MIDICPS:
    return S_IDENT;
  case S_SGN:
    return S_IDENT;
  case S_FTLEN:
    return S_IDENT;
  case S_FTLOOP:
    return S_IDENT;
  case S_FTLOOPEND:
    return S_IDENT;
  case S_FTSETLOOP:
    return S_IDENT;
  case S_FTSETEND:
    return S_IDENT;
  case S_FTBASECPS:
    return S_IDENT;
  case S_FTSETBASE:
    return S_IDENT;
  case S_TABLEREAD:
    return S_IDENT;
  case S_TABLEWRITE:
    return S_IDENT;
  case S_OSCIL:
    return S_IDENT;
  case S_LOSCIL:
    return S_IDENT;
  case S_DOSCIL:
    return S_IDENT;
  case S_KOSCIL:
    return S_IDENT;
  case S_KLINE:
    return S_IDENT;
  case S_ALINE:
    return S_IDENT;
  case S_SBLOCK:
    return S_IDENT;
  case S_KEXPON:
    return S_IDENT;
  case S_AEXPON:
    return S_IDENT;
  case S_KPHASOR:
    return S_IDENT;
  case S_APHASOR:
    return S_IDENT;
  case S_PLUCK:
    return S_IDENT;
  case S_BUZZOPCODE:
    return S_IDENT;
  case S_GRAIN:
    return S_IDENT;
  case S_IRAND:
    return S_IDENT;
  case S_KRAND:
    return S_IDENT;
  case S_ARAND:
    return S_IDENT;
  case S_ILINRAND:
    return S_IDENT;
  case S_KLINRAND:
    return S_IDENT;
  case S_ALINRAND:
    return S_IDENT;
  case S_IEXPRAND:
    return S_IDENT;
  case S_KEXPRAND:
    return S_IDENT;
  case S_AEXPRAND:
    return S_IDENT;
  case S_KPOISSONRAND:
    return S_IDENT;
  case S_APOISSONRAND:
    return S_IDENT;
  case S_IGAUSSRAND:
    return S_IDENT;
  case S_KGAUSSRAND:
    return S_IDENT;
  case S_AGAUSSRAND:
    return S_IDENT;
  case S_PORT:
    return S_IDENT;
  case S_HIPASS:
    return S_IDENT;
  case S_LOPASS:
    return S_IDENT;
  case S_BANDPASS:
    return S_IDENT;
  case S_BANDSTOP:
    return S_IDENT;
  case S_FIR:
    return S_IDENT;
  case S_IIR:
    return S_IDENT;
  case S_FIRT:
    return S_IDENT;
  case S_IIRT:
    return S_IDENT;
  case S_BIQUAD:
    return S_IDENT;
  case S_FFT:
    return S_IDENT;
  case S_IFFT:
    return S_IDENT;
  case S_RMS:
    return S_IDENT;
  case S_GAIN:
    return S_IDENT;
  case S_BALANCE:
    return S_IDENT;
  case S_DECIMATE:
    return S_IDENT;
  case S_UPSAMP:
    return S_IDENT;
  case S_DOWNSAMP:
    return S_IDENT;
  case S_SAMPHOLD:
    return S_IDENT;
  case S_DELAY:
    return S_IDENT;
  case S_DELAY1:
    return S_IDENT;
  case S_FRACDELAY:
    return S_IDENT;
  case S_COMB:
    return S_IDENT;
  case S_ALLPASS:
    return S_IDENT;
  case S_CHORUS:
    return S_IDENT;
  case S_FLANGE:
    return S_IDENT;
  case S_REVERB:
    return S_IDENT;
  case S_COMPRESSOR:
    return S_IDENT;
  case S_GETTUNE:
    return S_IDENT;
  case S_SETTUNE:
    return S_IDENT;
  case S_FTSR:
    return S_IDENT;
  case S_FTSETSR:
    return S_IDENT;
  case S_GETTEMPO:
    return S_IDENT;
  case S_SETTEMPO:
    return S_IDENT;
  case S_FX_SPEEDC:
    return S_IDENT;
  case S_SPEEDT:
    return S_IDENT;
  case S_RESERVED26:
    return S_IDENT;
  case S_RESERVED27:
    return S_IDENT;
  case S_RESERVED28:
    return S_IDENT;
  case S_RESERVED29:
    return S_IDENT;
  case S_RESERVED30:
    return S_IDENT;
  case S_RESERVED31:
    return S_IDENT;
  case S_RESERVED32:
    return S_IDENT;
  case S_IDENT:
    return S_IDENT;
  case S_INTGR:
    return S_INTGR;
  case S_NUMBER:
    return S_NUMBER;
  case S_STRCONST:
    return S_STRCONST;
  case S_BYTE:
    return S_INTGR;
  case S_FREE1:
    return S_IDENT;
  case S_FREE2:
    return S_IDENT;
  case S_FREE3:
    return S_IDENT;
  case S_FREE4:
    return S_IDENT;
  case S_FREE5:
    return S_IDENT;
  case S_FREE6:
    return S_IDENT;
  case S_FREE7:
    return S_IDENT;
  case S_FREE8:
    return S_IDENT;
  case S_FREE9:
    return S_IDENT;
  case S_FREE10:
    return S_IDENT;
  case S_EOO:
    return S_EOO;
  }
 return S_EOO; /* should never happen */
}


/*********************************************************/
/*     converts MP4 token to bison-compatabile token    */
/*********************************************************/


int parsetokenmap(int token)

{
  switch(token) {

  case S_RESERVED0:
    return IDENT;
  case S_AOPCODE:
    return AOPCODE;
  case S_ASIG:
    return ASIG;
  case S_ELSE:
    return ELSE;
  case S_EXPORTS:
    return EXPORTS;
  case S_EXTEND:
    return EXTEND;
  case S_GLOBAL:     
    return GLOBAL;
  case S_IF:
    return IF;
  case S_IMPORTS:
    return IMPORTS;
  case S_INCHANNELS:
    return INCHANNELS;
  case S_INSTR:
    return INSTR;
  case S_IOPCODE:
    return IOPCODE;
  case S_IVAR:
    return IVAR;
  case S_KOPCODE:
    return KOPCODE;
  case S_KRATE:
    return KRATE;
  case S_KSIG:
    return KSIG;
  case S_MAP:
    return MAP;
  case S_OPARRAY:
    return OPARRAY;
  case S_OPCODE:
    return OPCODE;
  case S_OUTBUS:
    return OUTBUS;
  case S_OUTCHANNELS:
    return OUTCHANNELS;
  case S_OUTPUT:
    return OUTPUT;
  case S_RETURN:
    return RETURN;
  case S_ROUTE:
    return ROUTE;
  case S_SEND:
    return SEND;
  case S_SEQUENCE:
    return SEQUENCE;
  case S_SASBF:
    return SASBF;
  case S_SPATIALIZE:
    return SPATIALIZE;
  case S_SRATE:
    return SRATE;
  case S_TABLE:
    return TABLE;
  case S_TABLEMAP:
    return TABLEMAP;
  case S_TEMPLATE:
    return TEMPLATE;
  case S_TURNOFF:
    return TURNOFF;
  case S_WHILE:
    return WHILE;
  case S_WITH:
    return WITH;
  case S_XSIG:
    return XSIG;
  case S_INTERP:
    return INTERP;
  case S_PRESET:        /* PRESET not in grammer */
    return IDENT;
  case S_RESERVED1:
    return IDENT;
  case S_RESERVED2:
    return IDENT;
  case S_RESERVED3:
    return IDENT;
  case S_RESERVED4:
    return IDENT;
  case S_RESERVED5:
    return IDENT;
  case S_RESERVED6:
    return IDENT;
  case S_RESERVED7:
    return IDENT;
  case S_RESERVED8:
    return IDENT;
  case S_RESERVED9:
    return IDENT;
  case S_RESERVED10:
    return IDENT;
  case S_K_RATE:
    return IDENT;
  case S_S_RATE:
    return IDENT;
  case S_INCHAN:
    return IDENT;
  case S_OUTCHAN:
    return IDENT;
  case S_TIME:
    return IDENT;
  case S_DUR:
    return IDENT;
  case S_MIDICTRL:
    return IDENT;
  case S_MIDITOUCH:
    return IDENT;
  case S_MIDIBEND:
    return IDENT;
  case S_INPUT:
    return IDENT;
  case S_INGROUP:
    return IDENT;
  case S_RELEASED:
    return IDENT;
  case S_CPULOAD:
    return IDENT;
  case S_POSITION:
    return IDENT;
  case S_DIRECTION:
    return IDENT;
  case S_LISTENERPOSITION:
    return IDENT;
  case S_LISTENERDIRECTION:
    return IDENT;
  case S_MINFRONT:
    return IDENT;
  case S_MINBACK:
    return IDENT;
  case S_MAXFRONT:
    return IDENT;
  case S_MAXBACK:
    return IDENT;
  case S_PARAMS:
    return IDENT;
  case S_ITIME:
    return IDENT;
  case S_RESERVED11:
    return IDENT;
  case S_CHANNEL:
    return IDENT;
  case S_INPUT_BUS:
    return IDENT;
  case S_OUTPUT_BUS:
    return IDENT;
  case S_STARTUP:
    return IDENT;
  case S_RESERVED15:
    return IDENT;
  case S_RESERVED16:
    return IDENT;
  case S_RESERVED17:
    return IDENT;
  case S_RESERVED18:
    return IDENT;
  case S_AND:
    return AND;
  case S_OR:
    return OR;
  case S_GEQ:
    return GEQ;
  case S_LEQ:
    return LEQ;
  case S_NEQ:
    return NEQ;
  case S_EQEQ:
    return EQEQ;
  case S_MINUS:
    return MINUS;
  case S_STAR:
    return STAR;
  case S_SLASH:
    return SLASH;
  case S_PLUS:
    return PLUS;
  case S_GT:
    return GT;
  case S_LT:
    return LT;
  case S_Q:
    return Q;
  case S_COL:
    return COL;
  case S_LP:
    return LP;
  case S_RP:
    return RP;
  case S_LC:
    return LC;
  case S_RC:
    return RC;
  case S_LB:
    return LB;
  case S_RB:
    return RB;
  case S_SEM:
    return SEM;
  case S_COM:
    return COM;
  case S_EQ:
    return EQ;
  case S_NOT:
    return NOT;
  case S_RESERVED19:
    return IDENT;
  case S_RESERVED20:
    return IDENT;
  case S_RESERVED21:
    return IDENT;
  case S_RESERVED22:
    return IDENT;
  case S_RESERVED23:
    return IDENT;
  case S_RESERVED24:
    return IDENT;
  case S_RESERVED25:
    return IDENT;
  case S_SAMPLE:
    return IDENT;
  case S_DATA:
    return IDENT;
  case S_RANDOM:
    return IDENT;
  case S_STEP:
    return IDENT;
  case S_LINESEG:
    return IDENT;
  case S_EXPSEG:
    return IDENT;
  case S_CUBICSEG:
    return IDENT;
  case S_POLYNOMIAL:
    return IDENT;
  case S_SPLINE:
    return IDENT;
  case S_WINDOW:
    return IDENT;
  case S_HARM:
    return IDENT;
  case S_HARM_PHASE:
    return IDENT;
  case S_PERIODIC:
    return IDENT;
  case S_BUZZWAVE:
    return IDENT;
  case S_CONCAT:
    return IDENT;
  case S_EMPTY:
    return IDENT;
  case S_DESTROY:
    return IDENT;
  case S_INT:
    return IDENT;
  case S_FRAC:
    return IDENT;
  case S_DBAMP:
    return IDENT;
  case S_AMPDB:
    return IDENT;
  case S_ABS:
    return IDENT;
  case S_EXP:
    return IDENT;
  case S_LOG:
    return IDENT;
  case S_SQRT:
    return IDENT;
  case S_SIN:
    return IDENT;
  case S_COS:
    return IDENT;
  case S_ATAN:
    return IDENT;
  case S_POW:
    return IDENT;
  case S_LOG10:
    return IDENT;
  case S_ASIN:
    return IDENT;
  case S_ACOS:
    return IDENT;
  case S_FLOOR:
    return IDENT;
  case S_CEIL:
    return IDENT;
  case S_MIN:
    return IDENT;
  case S_MAX:
    return IDENT;
  case S_PCHOCT:
    return IDENT;
  case S_OCTPCH:
    return IDENT;
  case S_CPSPCH:
    return IDENT;
  case S_PCHCPS:
    return IDENT;
  case S_CPSOCT:
    return IDENT;
  case S_OCTCPS:
    return IDENT;
  case S_PCHMIDI:
    return IDENT;
  case S_MIDIPCH:
    return IDENT;
  case S_OCTMIDI:
    return IDENT;
  case S_MIDIOCT:
    return IDENT;
  case S_CPSMIDI:
    return IDENT;
  case S_MIDICPS:
    return IDENT;
  case S_SGN:
    return IDENT;
  case S_FTLEN:
    return IDENT;
  case S_FTLOOP:
    return IDENT;
  case S_FTLOOPEND:
    return IDENT;
  case S_FTSETLOOP:
    return IDENT;
  case S_FTSETEND:
    return IDENT;
  case S_FTBASECPS:
    return IDENT;
  case S_FTSETBASE:
    return IDENT;
  case S_TABLEREAD:
    return IDENT;
  case S_TABLEWRITE:
    return IDENT;
  case S_OSCIL:
    return IDENT;
  case S_LOSCIL:
    return IDENT;
  case S_DOSCIL:
    return IDENT;
  case S_KOSCIL:
    return IDENT;
  case S_KLINE:
    return IDENT;
  case S_ALINE:
    return IDENT;
  case S_SBLOCK:
    return IDENT;
  case S_KEXPON:
    return IDENT;
  case S_AEXPON:
    return IDENT;
  case S_KPHASOR:
    return IDENT;
  case S_APHASOR:
    return IDENT;
  case S_PLUCK:
    return IDENT;
  case S_BUZZOPCODE:
    return IDENT;
  case S_GRAIN:
    return IDENT;
  case S_IRAND:
    return IDENT;
  case S_KRAND:
    return IDENT;
  case S_ARAND:
    return IDENT;
  case S_ILINRAND:
    return IDENT;
  case S_KLINRAND:
    return IDENT;
  case S_ALINRAND:
    return IDENT;
  case S_IEXPRAND:
    return IDENT;
  case S_KEXPRAND:
    return IDENT;
  case S_AEXPRAND:
    return IDENT;
  case S_KPOISSONRAND:
    return IDENT;
  case S_APOISSONRAND:
    return IDENT;
  case S_IGAUSSRAND:
    return IDENT;
  case S_KGAUSSRAND:
    return IDENT;
  case S_AGAUSSRAND:
    return IDENT;
  case S_PORT:
    return IDENT;
  case S_HIPASS:
    return IDENT;
  case S_LOPASS:
    return IDENT;
  case S_BANDPASS:
    return IDENT;
  case S_BANDSTOP:
    return IDENT;
  case S_FIR:
    return IDENT;
  case S_IIR:
    return IDENT;
  case S_FIRT:
    return IDENT;
  case S_IIRT:
    return IDENT;
  case S_BIQUAD:
    return IDENT;
  case S_FFT:
    return IDENT;
  case S_IFFT:
    return IDENT;
  case S_RMS:
    return IDENT;
  case S_GAIN:
    return IDENT;
  case S_BALANCE:
    return IDENT;
  case S_DECIMATE:
    return IDENT;
  case S_UPSAMP:
    return IDENT;
  case S_DOWNSAMP:
    return IDENT;
  case S_SAMPHOLD:
    return IDENT;
  case S_DELAY:
    return IDENT;
  case S_DELAY1:
    return IDENT;
  case S_FRACDELAY:
    return IDENT;
  case S_COMB:
    return IDENT;
  case S_ALLPASS:
    return IDENT;
  case S_CHORUS:
    return IDENT;
  case S_FLANGE:
    return IDENT;
  case S_REVERB:
    return IDENT;
  case S_COMPRESSOR:
    return IDENT;
  case S_GETTUNE:
    return IDENT;
  case S_SETTUNE:
    return IDENT;
  case S_FTSR:
    return IDENT;
  case S_FTSETSR:
    return IDENT;
  case S_GETTEMPO:
    return IDENT;
  case S_SETTEMPO:
    return IDENT;
  case S_FX_SPEEDC:
    return IDENT;
  case S_SPEEDT:
    return IDENT;
  case S_RESERVED26:
    return IDENT;
  case S_RESERVED27:
    return IDENT;
  case S_RESERVED28:
    return IDENT;
  case S_RESERVED29:
    return IDENT;
  case S_RESERVED30:
    return IDENT;
  case S_RESERVED31:
    return IDENT;
  case S_RESERVED32:
    return IDENT;
  case S_IDENT:
    return IDENT;
  case S_INTGR:
    return INTGR;
  case S_NUMBER:
    return NUMBER;
  case S_STRCONST:
    return STRCONST;
  case S_BYTE:
    return INTGR;
  case S_FREE1:
    return IDENT;
  case S_FREE2:
    return IDENT;
  case S_FREE3:
    return IDENT;
  case S_FREE4:
    return IDENT;
  case S_FREE5:
    return IDENT;
  case S_FREE6:
    return IDENT;
  case S_FREE7:
    return IDENT;
  case S_FREE8:
    return IDENT;
  case S_FREE9:
    return IDENT;
  case S_FREE10:
    return IDENT;
  case S_EOO:
    return 0;
  }
  return 0; /* should never happen */
}


/*********************************************************/
/*             returns string for token                  */
/*********************************************************/

char * strfortoken(char * name, int token)

{
  switch(token) {

  case S_RESERVED0:
    return strcpy(name,"reserved0");
  case S_AOPCODE:
    return strcpy(name,"aopcode");
  case S_ASIG:
    return strcpy(name,"asig");
  case S_ELSE:
    return strcpy(name,"else");
  case S_EXPORTS:
    return strcpy(name,"exports");
  case S_EXTEND:
    return strcpy(name,"extend");
  case S_GLOBAL:     
    return strcpy(name,"global");
  case S_IF:
    return strcpy(name,"if");
  case S_IMPORTS:
    return strcpy(name,"imports");
  case S_INCHANNELS:
    return strcpy(name,"inchannels");
  case S_INSTR:
    return strcpy(name,"instr");
  case S_IOPCODE:
    return strcpy(name,"iopcode");
  case S_IVAR:
    return strcpy(name,"ivar");
  case S_KOPCODE:
    return strcpy(name,"kopcode");
  case S_KRATE:
    return strcpy(name,"krate");
  case S_KSIG:
    return strcpy(name,"ksig");
  case S_MAP:
    return strcpy(name,"map");
  case S_OPARRAY:
    return strcpy(name,"oparray");
  case S_OPCODE:
    return strcpy(name,"opcode");
  case S_OUTBUS:
    return strcpy(name,"outbus");
  case S_OUTCHANNELS:
    return strcpy(name,"outchannels");
  case S_OUTPUT:
    return strcpy(name,"output");
  case S_RETURN:
    return strcpy(name,"return");
  case S_ROUTE:
    return strcpy(name,"route");
  case S_SEND:
    return strcpy(name,"send");
  case S_SEQUENCE:
    return strcpy(name,"sequence");
  case S_SASBF:
    return strcpy(name,"sasbf");
  case S_SPATIALIZE:
    return strcpy(name,"spatialize");
  case S_SRATE:
    return strcpy(name,"srate");
  case S_TABLE:
    return strcpy(name,"table");
  case S_TABLEMAP:
    return strcpy(name,"tablemap");
  case S_TEMPLATE:
    return strcpy(name,"template");
  case S_TURNOFF:
    return strcpy(name,"turnoff");
  case S_WHILE:
    return strcpy(name,"while");
  case S_WITH:
    return strcpy(name,"with");
  case S_XSIG:
    return strcpy(name,"xsig");
  case S_INTERP:
    return strcpy(name,"interp");
  case S_PRESET:
    return strcpy(name,"preset");
  case S_RESERVED1:
    return strcpy(name,"reserved1");
  case S_RESERVED2:
    return strcpy(name,"reserved2");
  case S_RESERVED3:
    return strcpy(name,"reserved3");
  case S_RESERVED4:
    return strcpy(name,"reserved4");
  case S_RESERVED5:
    return strcpy(name,"reserved5");
  case S_RESERVED6:
    return strcpy(name,"reserved6");
  case S_RESERVED7:
    return strcpy(name,"reserved7");
  case S_RESERVED8:
    return strcpy(name,"reserved8");
  case S_RESERVED9:
    return strcpy(name,"reserved9");
  case S_RESERVED10:
    return strcpy(name,"reserved10");
  case S_K_RATE:
    return strcpy(name,"k_rate");
  case S_S_RATE:
    return strcpy(name,"s_rate");
  case S_INCHAN:
    return strcpy(name,"inchan");
  case S_OUTCHAN:
    return strcpy(name,"outchan");
  case S_TIME:
    return strcpy(name,"time");
  case S_DUR:
    return strcpy(name,"dur");
  case S_MIDICTRL:
    return strcpy(name,"MIDIctrl");
  case S_MIDITOUCH:
    return strcpy(name,"MIDItouch");
  case S_MIDIBEND:
    return strcpy(name,"MIDIbend");
  case S_INPUT:
    return strcpy(name,"input");
  case S_INGROUP:
    return strcpy(name,"ingroup");
  case S_RELEASED:
    return strcpy(name,"released");
  case S_CPULOAD:
    return strcpy(name,"cpuload");
  case S_POSITION:
    return strcpy(name,"position");
  case S_DIRECTION:
    return strcpy(name,"direction");
  case S_LISTENERPOSITION:
    return strcpy(name,"listenerPosition");
  case S_LISTENERDIRECTION:
    return strcpy(name,"listenerDirection");
  case S_MINFRONT:
    return strcpy(name,"minFront");
  case S_MINBACK:
    return strcpy(name,"minBack");
  case S_MAXFRONT:
    return strcpy(name,"maxFront");
  case S_MAXBACK:
    return strcpy(name,"maxBack");
  case S_PARAMS:
    return strcpy(name,"params");
  case S_ITIME:
    return strcpy(name,"itime");
  case S_RESERVED11:
    return strcpy(name,"reserved11");
  case S_CHANNEL:
    return strcpy(name,"channel");
  case S_INPUT_BUS:
    return strcpy(name,"input_bus");
  case S_OUTPUT_BUS:
    return strcpy(name,"output_bus");
  case S_STARTUP:
    return strcpy(name,"startup");
  case S_RESERVED15:
    return strcpy(name,"reserved15");
  case S_RESERVED16:
    return strcpy(name,"reserved16");
  case S_RESERVED17:
    return strcpy(name,"reserved17");
  case S_RESERVED18:
    return strcpy(name,"reserved18");
  case S_AND:
    return strcpy(name,"&&");
  case S_OR:
    return strcpy(name,"||");
  case S_GEQ:
    return strcpy(name,">=");
  case S_LEQ:
    return strcpy(name,"<=");
  case S_NEQ:
    return strcpy(name,"!=");
  case S_EQEQ:
    return strcpy(name,"==");
  case S_MINUS:
    return strcpy(name,"-");
  case S_STAR:
    return strcpy(name,"*");
  case S_SLASH:
    return strcpy(name,"/");
  case S_PLUS:
    return strcpy(name,"+");
  case S_GT:
    return strcpy(name,">");
  case S_LT:
    return strcpy(name,"<");
  case S_Q:
    return strcpy(name,"?");
  case S_COL:
    return strcpy(name,":");
  case S_LP:
    return strcpy(name,"(");
  case S_RP:
    return strcpy(name,")");
  case S_LC:
    return strcpy(name,"{");
  case S_RC:
    return strcpy(name,"}");
  case S_LB:
    return strcpy(name,"[");
  case S_RB:
    return strcpy(name,"]");
  case S_SEM:
    return strcpy(name,";");
  case S_COM:
    return strcpy(name,",");
  case S_EQ:
    return strcpy(name,"=");
  case S_NOT:
    return strcpy(name,"!");
  case S_RESERVED19:
    return strcpy(name,"reserved19");
  case S_RESERVED20:
    return strcpy(name,"reserved20");
  case S_RESERVED21:
    return strcpy(name,"reserved21");
  case S_RESERVED22:
    return strcpy(name,"reserved22");
  case S_RESERVED23:
    return strcpy(name,"reserved23");
  case S_RESERVED24:
    return strcpy(name,"reserved24");
  case S_RESERVED25:
    return strcpy(name,"reserved25");
  case S_SAMPLE:
    return strcpy(name,"sample");
  case S_DATA:
    return strcpy(name,"data");
  case S_RANDOM:
    return strcpy(name,"random");
  case S_STEP:
    return strcpy(name,"step");
  case S_LINESEG:
    return strcpy(name,"lineseg");
  case S_EXPSEG:
    return strcpy(name,"expseg");
  case S_CUBICSEG:
    return strcpy(name,"cubicseg");
  case S_POLYNOMIAL:
    return strcpy(name,"polynomial");
  case S_SPLINE:
    return strcpy(name,"spline");
  case S_WINDOW:
    return strcpy(name,"window");
  case S_HARM:
    return strcpy(name,"harm");
  case S_HARM_PHASE:
    return strcpy(name,"harm_phase");
  case S_PERIODIC:
    return strcpy(name,"periodic");
  case S_BUZZWAVE:
    return strcpy(name,"buzz");
  case S_CONCAT:
    return strcpy(name,"concat");
  case S_EMPTY:
    return strcpy(name,"empty");
  case S_DESTROY:
    return strcpy(name,"destroy");
  case S_INT:
    return strcpy(name,"int");
  case S_FRAC:
    return strcpy(name,"frac");
  case S_DBAMP:
    return strcpy(name,"dbamp");
  case S_AMPDB:
    return strcpy(name,"ampdb");
  case S_ABS:
    return strcpy(name,"abs");
  case S_EXP:
    return strcpy(name,"exp");
  case S_LOG:
    return strcpy(name,"log");
  case S_SQRT:
    return strcpy(name,"sqrt");
  case S_SIN:
    return strcpy(name,"sin");
  case S_COS:
    return strcpy(name,"cos");
  case S_ATAN:
    return strcpy(name,"atan");
  case S_POW:
    return strcpy(name,"pow");
  case S_LOG10:
    return strcpy(name,"log10");
  case S_ASIN:
    return strcpy(name,"asin");
  case S_ACOS:
    return strcpy(name,"acos");
  case S_FLOOR:
    return strcpy(name,"floor");
  case S_CEIL:
    return strcpy(name,"ceil");
  case S_MIN:
    return strcpy(name,"min");
  case S_MAX:
    return strcpy(name,"max");
  case S_PCHOCT:
    return strcpy(name,"pchoct");
  case S_OCTPCH:
    return strcpy(name,"octpch");
  case S_CPSPCH:
    return strcpy(name,"cpspch");
  case S_PCHCPS:
    return strcpy(name,"pchcps");
  case S_CPSOCT:
    return strcpy(name,"cpsoct");
  case S_OCTCPS:
    return strcpy(name,"octcps");
  case S_PCHMIDI:
    return strcpy(name,"pchmidi");
  case S_MIDIPCH:
    return strcpy(name,"midipch");
  case S_OCTMIDI:
    return strcpy(name,"octmidi");
  case S_MIDIOCT:
    return strcpy(name,"midioct");
  case S_CPSMIDI:
    return strcpy(name,"cpsmidi");
  case S_MIDICPS:
    return strcpy(name,"midicps");
  case S_SGN:
    return strcpy(name,"sgn");
  case S_FTLEN:
    return strcpy(name,"ftlen");
  case S_FTLOOP:
    return strcpy(name,"ftloop");
  case S_FTLOOPEND:
    return strcpy(name,"ftloopend");
  case S_FTSETLOOP:
    return strcpy(name,"ftsetloop");
  case S_FTSETEND:
    return strcpy(name,"ftsetend");
  case S_FTBASECPS:
    return strcpy(name,"ftbasecps");
  case S_FTSETBASE:
    return strcpy(name,"ftsetbase");
  case S_TABLEREAD:
    return strcpy(name,"tableread");
  case S_TABLEWRITE:
    return strcpy(name,"tablewrite");
  case S_OSCIL:
    return strcpy(name,"oscil");
  case S_LOSCIL:
    return strcpy(name,"loscil");
  case S_DOSCIL:
    return strcpy(name,"doscil");
  case S_KOSCIL:
    return strcpy(name,"koscil");
  case S_KLINE:
    return strcpy(name,"kline");
  case S_ALINE:
    return strcpy(name,"aline");
  case S_SBLOCK:
    return strcpy(name,"sblock");
  case S_KEXPON:
    return strcpy(name,"kexpon");
  case S_AEXPON:
    return strcpy(name,"aexpon");
  case S_KPHASOR:
    return strcpy(name,"kphasor");
  case S_APHASOR:
    return strcpy(name,"aphasor");
  case S_PLUCK:
    return strcpy(name,"pluck");
  case S_BUZZOPCODE:
    return strcpy(name,"buzz");
  case S_GRAIN:
    return strcpy(name,"grain");
  case S_IRAND:
    return strcpy(name,"irand");
  case S_KRAND:
    return strcpy(name,"krand");
  case S_ARAND:
    return strcpy(name,"arand");
  case S_ILINRAND:
    return strcpy(name,"ilinrand");
  case S_KLINRAND:
    return strcpy(name,"klinrand");
  case S_ALINRAND:
    return strcpy(name,"alinrand");
  case S_IEXPRAND:
    return strcpy(name,"iexprand");
  case S_KEXPRAND:
    return strcpy(name,"kexprand");
  case S_AEXPRAND:
    return strcpy(name,"aexprand");
  case S_KPOISSONRAND:
    return strcpy(name,"kpoissonrand");
  case S_APOISSONRAND:
    return strcpy(name,"apoissonrand");
  case S_IGAUSSRAND:
    return strcpy(name,"igaussrand");
  case S_KGAUSSRAND:
    return strcpy(name,"kgaussrand");
  case S_AGAUSSRAND:
    return strcpy(name,"agaussrand");
  case S_PORT:
    return strcpy(name,"port");
  case S_HIPASS:
    return strcpy(name,"hipass");
  case S_LOPASS:
    return strcpy(name,"lopass");
  case S_BANDPASS:
    return strcpy(name,"bandpass");
  case S_BANDSTOP:
    return strcpy(name,"bandstop");
  case S_FIR:
    return strcpy(name,"fir");
  case S_IIR:
    return strcpy(name,"iir");
  case S_FIRT:
    return strcpy(name,"firt");
  case S_IIRT:
    return strcpy(name,"iirt");
  case S_BIQUAD:
    return strcpy(name,"biquad");
  case S_FFT:
    return strcpy(name,"fft");
  case S_IFFT:
    return strcpy(name,"ifft");
  case S_RMS:
    return strcpy(name,"rms");
  case S_GAIN:
    return strcpy(name,"gain");
  case S_BALANCE:
    return strcpy(name,"balance");
  case S_DECIMATE:
    return strcpy(name,"decimate");
  case S_UPSAMP:
    return strcpy(name,"upsamp");
  case S_DOWNSAMP:
    return strcpy(name,"downsamp");
  case S_SAMPHOLD:
    return strcpy(name,"samphold");
  case S_DELAY:
    return strcpy(name,"delay");
  case S_DELAY1:
    return strcpy(name,"delay1");
  case S_FRACDELAY:
    return strcpy(name,"fracdelay");
  case S_COMB:
    return strcpy(name,"comb");
  case S_ALLPASS:
    return strcpy(name,"allpass");
  case S_CHORUS:
    return strcpy(name,"chorus");
  case S_FLANGE:
    return strcpy(name,"flange");
  case S_REVERB:
    return strcpy(name,"reverb");
  case S_COMPRESSOR:
    return strcpy(name,"compressor");
  case S_GETTUNE:
    return strcpy(name,"gettune");
  case S_SETTUNE:
    return strcpy(name,"settune");
  case S_FTSR:
    return strcpy(name,"ftsr");
  case S_FTSETSR:
    return strcpy(name,"ftsetsr");
  case S_GETTEMPO:
    return strcpy(name,"gettempo");
  case S_SETTEMPO:
    return strcpy(name,"settempo");
  case S_FX_SPEEDC:
    return strcpy(name,"fx_speedc");
  case S_SPEEDT:
    return strcpy(name,"speedt");
  case S_RESERVED26:
    return strcpy(name,"reserved26");
  case S_RESERVED27:
    return strcpy(name,"reserved27");
  case S_RESERVED28:
    return strcpy(name,"reserved28");
  case S_RESERVED29:
    return strcpy(name,"reserved29");
  case S_RESERVED30:
    return strcpy(name,"reserved30");
  case S_RESERVED31:
    return strcpy(name,"reserved31");
  case S_RESERVED32:
    return strcpy(name,"reserved32");
  case S_IDENT:
    return strcpy(name,"");
  case S_INTGR:
    return strcpy(name,"");
  case S_NUMBER:
    return strcpy(name,"");
  case S_STRCONST:
    return strcpy(name,"");
  case S_BYTE:
    return strcpy(name,"");
  case S_FREE1:
    return strcpy(name,"free1");
  case S_FREE2:
    return strcpy(name,"free2");
  case S_FREE3:
    return strcpy(name,"free3");
  case S_FREE4:
    return strcpy(name,"free4");
  case S_FREE5:
    return strcpy(name,"free5");
  case S_FREE6:
    return strcpy(name,"free6");
  case S_FREE7:
    return strcpy(name,"free7");
  case S_FREE8:
    return strcpy(name,"free8");
  case S_FREE9:
    return strcpy(name,"free9");
  case S_FREE10:
    return strcpy(name,"free10");
  case S_EOO:
    return strcpy(name,"");
  }

  return strcpy(name,""); /* should never happen */
}
