/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcas_test.c
    \brief  Program to test the LCAS and its plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This program has elements of the edg-gatekeeper to be able to test the
    LCAS and its plugins without having the edg-gatekeeper installed.
    To run it : just run ./lcas-test

    compile:
	gcc -o lcas_test \
	    -I<globus-prefix>/include/gcc64dbg -L<globus-prefix>/lib \
	    -lglobus_gss_assist_gcc64dbg -ldl lcas_test.c
*/

#ifndef ALLOW_EMPTY_CREDENTIALS
#define ALLOW_EMPTY_CREDENTIALS	1
#endif

#include "lcas_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if HAVE_MALLOC_H
#include <malloc.h>
#endif

#include <gssapi.h>
#include <globus_gss_assist.h>

/* Programming interface to dynamic linking loader */
#if HAVE_DLFCN_H
#include <dlfcn.h>
#endif

#if LINKED_LCAS
#include "lcas.h"                                                                             
#endif

static void failure(short failure_type, char *s);
static void notice(int, char *s);

#define FAILED_AUTHORIZATION        1
#define FAILED_SERVICELOOKUP        2
#define FAILED_SERVER               3
#define FAILED_NOLOGIN              4
#define FAILED_AUTHENTICATION       5
#define FAILED_PING                 6

static char     tmpbuf[1024];
#define notice2(i,a,b) {sprintf(tmpbuf, a,b); notice(i,tmpbuf);}
#define notice3(i,a,b,c) {sprintf(tmpbuf, a,b,c); notice(i,tmpbuf);}
#define notice4(i,a,b,c,d) {sprintf(tmpbuf, a,b,c,d); notice(i,tmpbuf);}
#define failure2(t,a,b) {sprintf(tmpbuf, a,b); failure(t,tmpbuf);}
#define failure3(t,a,b,c) {sprintf(tmpbuf, a,b,c); failure(t,tmpbuf);}
#define failure4(t,a,b,c,d) {sprintf(tmpbuf, a,b,c,d); failure(t,tmpbuf);}

static char *   lcasmod_name= NULL;
static FILE *   usrlog_fp=NULL;


gss_cred_id_t make_cred(char *proxyname)	{
	static gss_cred_id_t delegated_cred_handle = GSS_C_NO_CREDENTIAL;
        OM_uint32 major_status;
        OM_uint32 minor_status;
    
        setenv("X509_USER_PROXY",proxyname,1);
        major_status = globus_gss_assist_acquire_cred(&minor_status,
                                                      GSS_C_INITIATE, /* or GSS_C_ACCEPT */
                                                      &delegated_cred_handle);

        if (major_status != GSS_S_COMPLETE)
        {
            globus_gss_assist_display_status(stderr,
                                             "Some failure message here",
                                             major_status,
                                             minor_status,
                                             0);
            exit(1);
        }
	return delegated_cred_handle;
}



int main(void)
{
    gss_cred_id_t delegated_cred_handle;
    char * proxyname = "/tmp/x509up_u1000";
    char * lcas_request="Test lcas_request";
    char * client_name="/O=dutchgrid/O=users/O=nikhef/CN=Mischa Salle";
    int    retval=0;

    delegated_cred_handle=make_cred(proxyname);

    usrlog_fp=stderr;
#if LINKED_LCAS
        {
            int retval;

            notice(0,"Using linked in version of LCAS");
            /* Initialize, send authorization request to and terminate the LCAS */
            retval=lcas_init(usrlog_fp);
            if (retval)
            {
                failure(FAILED_SERVER, "LCAS initialization failure.");
            }
#if ALLOW_EMPTY_CREDENTIALS
            retval=lcas_get_fabric_authorization(client_name,delegated_cred_handle, lcas_request);
#else
            retval=lcas_get_fabric_authorization(delegated_cred_handle, lcas_request);
#endif /* ALLOW_EMPTY_CREDENTIALS */
            if (retval)
            {
                failure(FAILED_AUTHORIZATION, "LCAS failed authorization.");
            }
            retval=lcas_term();
            if (retval)
            {
                failure(FAILED_SERVER, "LCAS termination failure.");
            }
        }
#else /* LINKED_LCAS */
	lcasmod_name="liblcas.so";
        {
            void *handle;
            char *error;
            int retval;
            int (*LcasInit)(FILE *);
            int (*LcasTerm)();
#if ALLOW_EMPTY_CREDENTIALS
            int (*LcasGetFabricAuthorization)(char*, gss_cred_id_t, char*);
            notice(0,"temporarily ALLOW empty credentials");
#else
            int (*LcasGetFabricAuthorization)(gss_cred_id_t, char*);
#endif

            notice2(0,"lcasmod_name = %s",lcasmod_name);
            handle = dlopen(lcasmod_name,RTLD_NOW|RTLD_GLOBAL);
            if (!handle)
            {
                notice2(0,"dlopen error: %s",dlerror());
                failure2(FAILED_SERVER,"Cannot open LCAS module of %s",lcasmod_name);
            }
            else
            {
                /* Check the symbols */
                LcasInit=dlsym(handle,"lcas_init");
                if ((error = dlerror()) != NULL)
                {
                    notice2(0,"dlsym error: %s",error);
                    failure(FAILED_SERVER,"LCAS module not compliant.");
                }
                LcasGetFabricAuthorization=dlsym(handle,"lcas_get_fabric_authorization");
                if ((error = dlerror()) != NULL)
                {
                    notice2(0,"dlsym error: %s",error);
                    failure(FAILED_SERVER,"LCAS module not compliant.");
                }
                LcasTerm=dlsym(handle,"lcas_term");
                if ((error = dlerror()) != NULL)
                {
                    notice2(0,"dlsym error: %s",error);
                    failure(FAILED_SERVER,"LCAS module not compliant.");
                }

                /* Initialize, send authorization request to and terminate the LCAS */
                retval=(*LcasInit)(usrlog_fp);
                if (retval)
                {
                    failure(FAILED_SERVER, "LCAS initialization failure.");
                }
#if ALLOW_EMPTY_CREDENTIALS
                retval=(*LcasGetFabricAuthorization)(client_name, delegated_cred_handle, lcas_request);
#else
                retval=(*LcasGetFabricAuthorization)(delegated_cred_handle, lcas_request);
#endif
                if (retval)
                {
                    failure(FAILED_AUTHORIZATION, "LCAS failed authorization.");
                }
                retval=(*LcasTerm)();
                if (retval)
                {
                    failure(FAILED_SERVER, "LCAS termination failure.");
                }
                dlclose(handle);
            }
        }
#endif /* LINKED_LCAS */
    fprintf(usrlog_fp,"return value= %d\n",retval);
    return 0;
}
/******************************************************************************
Function:       notice()
Description:    
Parameters: prty is the syslog priority, but if = 0, then dont syslog. 
Returns:
******************************************************************************/
static void 
notice(int prty, char * s)
{
    {
        fprintf(usrlog_fp, "Notice: %d: %s\n", prty, s);
    }
} /* notice() */

/******************************************************************************
Function:       failure()
Description:    
Parameters:
Returns:
******************************************************************************/
static void 
failure(short failure_type, char * s)
{
    fprintf(stderr,"Failure: %s\n", s);
    {
        fprintf(usrlog_fp, "Failure: %s\n", s);
    }
    exit(1);
} /* failure() */
/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/src/lcas_test.c,v $
    $Date: 2010-05-03 10:42:49 $
    $Revision: 2.8 $
    $Author: okoeroo $
******************************************************************************/
