"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createCspRulesPreResponseHandler = createCspRulesPreResponseHandler;
var _csp_header_utils = require("./csp_header_utils");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const FRAME_ANCESTORS_DIRECTIVE = 'frame-ancestors';
const CSP_RULES_FRAME_ANCESTORS_CONFIG_KEY = 'csp.rules.frame-ancestors';

// add new directives to this Map when onboarding.
const SUPPORTED_DIRECTIVES = new Map([[CSP_RULES_FRAME_ANCESTORS_CONFIG_KEY, {
  directiveName: FRAME_ANCESTORS_DIRECTIVE,
  defaultValue: ["'self'"]
}]]);

/**
 * This function creates a pre-response handler to dynamically set the CSP rules.
 * It give precedence to the rules from application config plugin over those from YML.
 * In case no value from application config, it will ensure a default frame-ancestors is set.
 *
 * @param core Context passed to the plugins `setup` method
 * @param cspHeader The CSP header from YML
 * @param getConfigurationClient The function provided by application config plugin to retrieve configurations
 * @param logger The logger
 * @returns The pre-response handler
 */
function createCspRulesPreResponseHandler(core, cspHeader, getConfigurationClient, logger) {
  return async (request, response, toolkit) => {
    const parsedCspHeader = (0, _csp_header_utils.parseCspHeader)(cspHeader);
    try {
      const shouldCheckDest = ['document', 'frame', 'iframe', 'embed', 'object'];
      const currentDest = request.headers['sec-fetch-dest'];
      if (!shouldCheckDest.includes(currentDest)) {
        return toolkit.next({});
      }
      const client = getConfigurationClient(request);
      await updateDirectivesFromConfigurationClient(parsedCspHeader, client, request, logger);
      return updateNext(parsedCspHeader, toolkit);
    } catch (e) {
      logger.error(`Failure happened in CSP rules pre response handler due to ${e}`);
      updateDirectivesFromDefault(parsedCspHeader);
      return updateNext(parsedCspHeader, toolkit);
    }
  };
}
async function updateDirectivesFromConfigurationClient(parsedCspHeader, client, request, logger) {
  for (const [configKey, directive] of SUPPORTED_DIRECTIVES) {
    try {
      const value = await client.getEntityConfig(configKey, {
        headers: request.headers
      });
      if (!value || !value.trim()) {
        return addDirectiveWhenMissing(parsedCspHeader, directive);
      }
      parsedCspHeader.set(directive.directiveName, value.trim().split(' '));
    } catch (e) {
      logger.error(`Failure happened when handling CSP directive ${directive.directiveName} due to ${e}`);
      addDirectiveWhenMissing(parsedCspHeader, directive);
    }
  }
}
function updateDirectivesFromDefault(parsedCspHeader) {
  SUPPORTED_DIRECTIVES.forEach(async directive => {
    addDirectiveWhenMissing(parsedCspHeader, directive);
  });
}
function addDirectiveWhenMissing(parsedCspHeader, directive) {
  if (parsedCspHeader.has(directive.directiveName)) {
    return;
  }
  parsedCspHeader.set(directive.directiveName, directive.defaultValue);
}
function updateNext(parsedCspHeader, toolkit) {
  const additionalHeaders = {
    'content-security-policy': (0, _csp_header_utils.stringifyCspHeader)(parsedCspHeader)
  };
  return toolkit.next({
    headers: additionalHeaders
  });
}