"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CustomPanelsAdaptor = void 0;
var _uuid = require("uuid");
var _custom_panels = require("../../../common/constants/custom_panels");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class CustomPanelsAdaptor {
  constructor() {
    // index a panel
    _defineProperty(this, "indexPanel", async function (client, panelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            operationalPanel: panelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Panel Error:' + error);
      }
    });
    // update a panel
    _defineProperty(this, "updatePanel", async function (client, panelId, updatePanelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: panelId,
          body: {
            operationalPanel: updatePanelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Panel Error:' + error);
      }
    });
    // fetch a panel by id
    _defineProperty(this, "getPanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Panel Error:' + error);
      }
    });
    // gets list of panels stored in index
    _defineProperty(this, "viewPanelList", async function (client) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'operationalPanel',
          maxItems: 10000
        });
        return response.observabilityObjectList.filter(panel => !panel.operationalPanel.applicationId).map(panel => ({
          name: panel.operationalPanel.name,
          id: panel.objectId,
          dateCreated: panel.createdTimeMs,
          dateModified: panel.lastUpdatedTimeMs
        }));
      } catch (error) {
        throw new Error('View Panel List Error:' + error);
      }
    });
    // Delete a panel by Id
    _defineProperty(this, "deletePanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectById', {
          objectId: panelId
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel Error:' + error);
      }
    });
    // Delete a panel by Id
    _defineProperty(this, "deletePanelList", async function (client, panelIdList) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: panelIdList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel List Error:' + error);
      }
    });
    // Create a new Panel
    _defineProperty(this, "createNewPanel", async (client, panelName, appId) => {
      const panelBody = {
        name: panelName,
        visualizations: [],
        timeRange: {
          to: 'now',
          from: 'now-1d'
        },
        queryFilter: {
          query: '',
          language: 'ppl'
        }
      };
      if (appId) {
        panelBody.applicationId = appId;
        panelBody.timeRange = {
          to: 'now',
          from: 'now-24h'
        };
      }
      try {
        const response = await this.indexPanel(client, panelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
    // Rename an existing panel
    _defineProperty(this, "renamePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };
      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Rename Panel Error:' + error);
      }
    });
    // Clone an existing panel
    _defineProperty(this, "clonePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };
      try {
        const getPanel = await this.getPanel(client, panelId);
        const clonePanelBody = {
          name: panelName,
          visualizations: getPanel.operationalPanel.visualizations,
          timeRange: getPanel.operationalPanel.timeRange,
          queryFilter: getPanel.operationalPanel.queryFilter
        };
        const indexResponse = await this.indexPanel(client, clonePanelBody);
        const getClonedPanel = await this.getPanel(client, indexResponse.objectId);
        return {
          clonePanelId: getClonedPanel.objectId,
          dateCreated: getClonedPanel.createdTimeMs,
          dateModified: getClonedPanel.lastUpdatedTimeMs
        };
      } catch (error) {
        throw new Error('Clone Panel Error:' + error);
      }
    });
    // Add filters to an existing panel
    _defineProperty(this, "addPanelFilter", async (client, panelId, query, language, to, from) => {
      const updatePanelBody = {
        timeRange: {
          to,
          from
        },
        queryFilter: {
          query,
          language
        }
      };
      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Add Panel Filter Error:' + error);
      }
    });
    // parses fetched saved visualization
    _defineProperty(this, "parseSavedVisualizations", visualization => {
      return {
        id: visualization.objectId,
        name: visualization.savedVisualization.name,
        query: visualization.savedVisualization.query,
        type: visualization.savedVisualization.type,
        timeField: visualization.savedVisualization.selected_timestamp.name,
        selected_date_range: visualization.savedVisualization.selected_date_range,
        selected_fields: visualization.savedVisualization.selected_fields,
        userConfigs: visualization.savedVisualization.hasOwnProperty('userConfigs') || visualization.savedVisualization.hasOwnProperty('user_configs') ? JSON.parse(visualization.savedVisualization.user_configs || visualization.savedVisualization.userConfigs) : {},
        subType: visualization.savedVisualization.hasOwnProperty('subType') ? visualization.savedVisualization.subType : '',
        metricType: visualization.savedVisualization.hasOwnProperty('metricType') ? visualization.savedVisualization.metricType : '',
        units_of_measure: visualization.savedVisualization.hasOwnProperty('units_of_measure') ? visualization.savedVisualization.units_of_measure : '',
        ...(visualization.savedVisualization.application_id ? {
          application_id: visualization.savedVisualization.application_id
        } : {})
      };
    });
    // gets all saved visualizations
    _defineProperty(this, "getAllSavedVisualizations", async client => {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'savedVisualization'
        });
        return response.observabilityObjectList.map(visualization => this.parseSavedVisualizations(visualization));
      } catch (error) {
        throw new Error('View Saved Visualizations Error:' + error);
      }
    });
    // gets list of savedVisualizations by Id
    _defineProperty(this, "getSavedVisualizationById", async (client, savedVisualizationId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: savedVisualizationId
        });
        const visualization = response.observabilityObjectList[0];
        return this.parseSavedVisualizations(visualization);
      } catch (error) {
        throw new Error('Fetch Saved Visualizations By Id Error:' + error);
      }
    });
    // Get All Visualizations from a Panel
    _defineProperty(this, "getVisualizations", async (client, panelId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0].operationalPanel.visualizations;
      } catch (error) {
        throw new Error('Get Visualizations Error:' + error);
      }
    });
    _defineProperty(this, "calculatOverlapArea", (bb1, bb2) => {
      const xLeft = Math.max(bb1.x1, bb2.x1);
      const yTop = Math.max(bb1.y1, bb2.y1);
      const xRight = Math.min(bb1.x2, bb2.x2);
      const yBottom = Math.min(bb1.y2, bb2.y2);
      if (xRight < xLeft || yBottom < yTop) return 0;
      return (xRight - xLeft) * (yBottom - yTop);
    });
    _defineProperty(this, "getTotalOverlapArea", panelVisualizations => {
      const newVizBox = {
        x1: 0,
        y1: 0,
        x2: 6,
        y2: 4
      };
      const currentVizBoxes = panelVisualizations.map(visualization => {
        return {
          x1: visualization.x,
          y1: visualization.y,
          x2: visualization.x + visualization.w,
          y2: visualization.y + visualization.h
        };
      });
      let isOverlapping = 0;
      currentVizBoxes.map(viz => {
        isOverlapping += this.calculatOverlapArea(viz, newVizBox);
      });
      return isOverlapping;
    });
    // We want to check if the new visualization being added, can be placed at { x: 0, y: 0, w: 6, h: 4 };
    // To check this we try to calculate overlap between all the current visualizations and new visualization
    // if there is no overalap (i.e Total Overlap Area is 0), we place the new viz. in default position
    // else, we add it to the bottom of the panel
    _defineProperty(this, "getNewVizDimensions", panelVisualizations => {
      let maxY = 0;
      let maxYH = 0;

      // check if we can place the new visualization at default location
      if (this.getTotalOverlapArea(panelVisualizations) === 0) {
        return {
          x: 0,
          y: 0,
          w: 6,
          h: 4
        };
      }

      // else place the new visualization at the bottom of the panel
      panelVisualizations.map(panelVisualization => {
        if (panelVisualization.y >= maxY) {
          maxY = panelVisualization.y;
          maxYH = panelVisualization.h;
        }
      });
      return {
        x: 0,
        y: maxY + maxYH,
        w: 6,
        h: 4
      };
    });
    // Add Visualization in the  Panel
    _defineProperty(this, "addVisualization", async (client, panelId, savedVisualizationId, oldVisualizationId) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [];
        if (oldVisualizationId === undefined) {
          newDimensions = this.getNewVizDimensions(allPanelVisualizations);
          visualizationsList = allPanelVisualizations;
        } else {
          allPanelVisualizations.map(visualization => {
            if (visualization.id !== oldVisualizationId) {
              visualizationsList.push(visualization);
            } else {
              newDimensions = {
                x: visualization.x,
                y: visualization.y,
                w: visualization.w,
                h: visualization.h
              };
            }
          });
        }
        const newPanelVisualizations = [...visualizationsList, {
          id: 'panel_viz_' + (0, _uuid.v4)(),
          savedVisualizationId,
          ...newDimensions
        }];
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: newPanelVisualizations
        });
        return newPanelVisualizations;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });
    // Add Multiple visualizations in a Panel
    _defineProperty(this, "addMultipleVisualizations", async (client, panelId, savedVisualizationIds) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [...allPanelVisualizations];
        savedVisualizationIds.map(savedVisualizationId => {
          newDimensions = this.getNewVizDimensions(visualizationsList);
          visualizationsList = [...visualizationsList, {
            id: 'panel_viz_' + (0, _uuid.v4)(),
            savedVisualizationId,
            ...newDimensions
          }];
        });
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: visualizationsList
        });
        return visualizationsList;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });
    // Edits all Visualizations in the Panel
    _defineProperty(this, "editVisualization", async (client, panelId, visualizationParams) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        const filteredPanelVisualizations = [];
        for (let i = 0; i < allPanelVisualizations.length; i++) {
          for (let j = 0; j < visualizationParams.length; j++) {
            if (allPanelVisualizations[i].id === visualizationParams[j].i) {
              filteredPanelVisualizations.push({
                ...allPanelVisualizations[i],
                x: visualizationParams[j].x,
                y: visualizationParams[j].y,
                w: visualizationParams[j].w,
                h: visualizationParams[j].h
              });
            }
          }
        }
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: filteredPanelVisualizations
        });
        return filteredPanelVisualizations;
      } catch (error) {
        throw new Error('Edit Visualizations Error:' + error);
      }
    });
    // Create Sample Panels
    _defineProperty(this, "addSamplePanels", async (client, savedVisualizationIds) => {
      try {
        const panelBody = (0, _custom_panels.createDemoPanel)(savedVisualizationIds);
        const indexResponse = await this.indexPanel(client, panelBody);
        const fetchPanel = await this.getPanel(client, indexResponse.objectId);
        const fetchResponse = {
          name: fetchPanel.operationalPanel.name,
          id: fetchPanel.objectId,
          dateCreated: fetchPanel.createdTimeMs,
          dateModified: fetchPanel.lastUpdatedTimeMs
        };
        return [fetchResponse];
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
  }
}
exports.CustomPanelsAdaptor = CustomPanelsAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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